/*-----------------------------------------------------------------------------
	Set_up_buffers - 
			Allocates and initializes structures to store sounds
			2-Jul-94

			This module allocates SNDHDR and SOUND structures that
			are used in some of the demonstration programs.  These
			structures will hold sounds that have been recorded by
			the sound toolkit and/or sounds that are to be
			played-back by the sound toolkit.

			For demonstration purposes, this routine allocates as
			many sets of SNDHDR and SOUND structures as it can,
			using all available memory.  Because of this, the
			buffers that are used internally by the sound toolkit
			must be allocated first.  In the demonstration
			programs, the sound toolkit buffers are allocated by
			set_up_sound().

			Programs that use the file-play or file-record
			functions usually do not require any SNDHDR or SOUND
			structures.

	DGETBUF is part of the PSSJ Digital Sound Toolkit.
	Copyright 1994 Frank Durda IV, all rights reserved.
	Commercial use is restricted.  See intro(PSSJ) for more information.
-----------------------------------------------------------------------------*/

#include <stdio.h>
#include <malloc.h>
#include "sound.h"
#include "dgetbuf.h"

/*	Declare arrays of pointers to SNDHDR and SOUND structures.  Note
	that we can never have more than 200 4000-byte buffers since 
	200 x 4000 > 768K, and that does not include MS-DOS or the application.
	On a 640K system, the program usually only allocates around 60 buffers.
	The larger the buffers (to a point), the more efficient the use of
	memory.  Extremely small buffers waste memory.  Also, the smaller
	the buffer, the more resolution the elapsed-time operations will be
	able to provide.  

	Note that these two structures are not required by the sound
	toolkit.  They are used by the demonstration programs so that
	memory that has been allocated can be located and released when
	it is no longer needed.
*/

SNDHDR	far * sndheaders[MAXBUFFERS];	/*List of SNDHDR structures*/
SOUND	far * sndstructs[MAXBUFFERS];	/*List of SOUND structures*/

int	numbuffers = 0;			/*Count of buffers actually allocated*/

#ifdef __STDC__
void set_up_buffers(void)
#else
void set_up_buffers()
#endif
{
	SNDHDR far *shp;
	SOUND far *stp;
	char far *bp;

/*	Allocate as many BUFFERSIZE-byte buffers as possible.
	At the same time, allocate the matching SNDHDR and SOUND structs.
*/

	for (numbuffers = 0; numbuffers < MAXBUFFERS; numbuffers++)
	{
		bp = _fmalloc(BUFFERSIZE);
		shp = (SNDHDR far *) _fmalloc(sizeof(SNDHDR));
		stp = (SOUND far *) _fmalloc(sizeof(SOUND));

		if (bp == 0 || shp == 0 || stp == 0)	/*Check if any*/
		{					/*allocations failed*/

/*	When an allocation finally fails, release any buffers that were
	allocated that can't be used.  Remember that we need equal numbers
	of buffers, SNDHDR and SOUND structures.
*/

			if (bp != 0)
				_ffree(bp);
			if (shp != 0)
				_ffree(shp);
			if (stp != 0)
				_ffree(stp);
			printf("Allocated %d %dK buffers\n", 
				numbuffers, (BUFFERSIZE+512)/1024);
			return;
		}

/*	Fill in the SNDHDR structure */

		shp->sndp = stp;	/*Fill in pointer to SOUND*/
		shp->start = 0;		/*Fill in start/end*/
		shp->end = 0;		/*Zero for both indicates the entire*/
					/*sound should be played.*/
		shp->rate = R11000;	/*Sampling rate is one of*/
					/*R5500, R11000, or R22000*/

/*	Fill in SOUND structure */

		stp->buffer = bp;	/*Fill in buffer address */
		stp->sndlen = 0;	/*Initially no sound recorded */

/*	Here we set initialize the buffer size but subtract two bytes
	so that snd_decompress_part can decompress a full buffer back
	into the same-sized buffer without overflowing.
*/
		stp->buflen = BUFFERSIZE - 2;	/*Fill in buffer length*/

/*	Save addresses so they can be deallocated later on.*/

		sndheaders[numbuffers] = shp;
		sndstructs[numbuffers] = stp;
	}
}
