;********************* SUBROUTINES
;
; GENERAL NOTE:  Subroutines should make no assumptions about registers such
; as DS and BP.  They should not modify any word register not used to return
; values.  (They can modify flags, however.)
;
; This routine takes three parameters:  a handle in DX; the number of pages
; currently allocated to the handle in AX; and the number of additional pages
; to allocate to this handle in BX.  It is assumed that DX has been allocated
; and that the requested number of pages has been reserved by reducing
; UNALLOCPAGES.  This routine does no error checking.  
;
; On return, additional logical pages are allocated to the handle as
; specified, beginning with the first unused logical page number.  Returns
; nothing.
;
ALLOCPAGES:	PUSH	AX
		PUSH	BX
		PUSH	CX
		PUSH	DX
		PUSH	SI
		PUSH	DI
		PUSH	BP
		PUSH	DS
		PUSH	CS
		POP	DS
		;
		; Special case:  request to allocate 0 pages.
		;
		OR	BX,BX
		JZ	>L7
		;
		; BP is the logical page number.
		;
		MOV	BP,AX
		;
		; Loop over physical pages.
		;
		MOV	CX,512
		XOR	SI,SI
		XOR	DI,DI
		OR	DH,80h
L5:		MOV	AL,0
		XCHG	AL,PAGEALLOC[SI]	; test and set
		OR	AL,AL
		JZ	>L6
		MOV	TRANSLATEARRAY[DI],DX	; free page found, allocate
		MOV	LOGPAGES[DI],BP
		INC	BP
		DEC	BX
L6:		INC	SI
		INC	DI
		INC	DI
		OR	BX,BX
		LOOPNZ	L5
L7:		POP	DS
		POP	BP
		POP	DI
		POP	SI
		POP	DX
		POP	CX
		POP	BX
		POP	AX
		RET
;
; This routine takes two parameters:  a handle in DX; and the number of pages
; that should remain allocated to that handle in BX.  If the number is greater
; than what is currently allocated, there is no effect.  If BX is zero, all
; pages are deallocated.  It is assumed that DX is an allocated handle; there
; is no error checking.
;
; On return, logical pages have been deallocated from the handle, starting
; at the highest logical page number and proceeding downward.  Returns
; nothing.
;
DEALLOCPAGES:	PUSH	BX
		PUSH	CX
		PUSH	DX
		PUSH	SI
		PUSH	DI
		PUSH	DS
		PUSH	CS
		POP	DS
		;
		; BX is the highest logical page number that will be 
		; retained (-1 if all to be deallocated).
		;
		DEC	BX
		;
		; Loop over physical pages.
		;
		MOV	CX,512
		XOR	SI,SI
		XOR	DI,DI
		OR	DH,80h
L5:		CMP	PAGEALLOC[SI],0		; page bad or allocated?
		JNE	>L6
		CMP	TRANSLATEARRAY[DI],DX	; allocated to this handle?
		JNE	>L6
		CMP	LOGPAGES[DI],BX		; one of the pages we're after?
		JLE	>L6
		;
		; Logical page found to deallocate.
		;
		MOV	LOGPAGES[DI],0
		MOV	TRANSLATEARRAY[DI],0
		MOV	PAGEALLOC[SI],0FFh
L6:		INC	SI
		INC	DI
		INC	DI
		LOOP	L5
L7:		POP	DS
		POP	DI
		POP	SI
		POP	DX
		POP	CX
		POP	BX
		RET
;
; This routine takes a handle in DX and returns the number of pages allocated
; to the handle in BX.  Handle assumed valid.
;
COUNTPAGES:	PUSH	AX
		PUSH	CX
		PUSH	DX
		PUSH	SI
		PUSH	DI
		PUSH	DS
		PUSH	CS
		POP	DS
		;
		; Find last page allocated to handle.
		;
		XOR	SI,SI
		XOR	DI,DI
		MOV	BX,-1		; BX is highest page number found
		OR	DH,80h
		MOV	CX,512
L1:		CMP	PAGEALLOC[SI],0
		JNE	>L2
		CMP	TRANSLATEARRAY[DI],DX
		JNE	>L2
		CMP	LOGPAGES[DI],BX
		JNG	>L2
		MOV	BX,LOGPAGES[DI]
L2:		INC	SI
		INC	DI
		INC	DI
		LOOP	L1
		;
		; Increment highest page to get number of pages.
		;
		INC	BX
		POP	DS
		POP	DI
		POP	SI
		POP	DX
		POP	CX
		POP	AX
		RET
;
; This routine takes the linear address of the first byte of a conventional
; memory region in BX:CX and the linear address of the last byte of the 
; region in DX:AX.  It returns with carry set if the specified region 
; overlaps the 64k EMS page frame at 0D0000h.
;
; Is high address < 0D0000h?  If so, no overlap.
;
VERIFYPGFRAME:	PUSH	BX
		PUSH	CX
		MOV	BX,0Dh
		XOR	CX,CX
		CALL	CMP32
		POP	CX
		POP	BX
		JB	VERIFYPGFRAME_GOOD
		;
		; Is low address > 0DFFFFh?  If so, no overlap.
		;
		PUSH	AX
		PUSH	DX
		MOV	DX,0Dh
		MOV	AX,-1
		CALL	CMP32
		POP	DX
		POP	AX
		JB	VERIFYPGFRAME_GOOD
		;
		; Otherwise, overlap.
		;
VERIFYPGFRAME_BAD:
		STC
		RET
VERIFYPGFRAME_GOOD:
		CLC
		RET
;
; This routine calls CONVSEG2LIN or CONVEXP2LIN, depending on DI.  It permits
; code to be shared for the conventional and expanded cases.
;
CONVBOTH:	OR	DI,DI
		JZ	CONVSEG2LIN
		JMP	CONVEXP2LIN
;
; This routine takes a segment in DX and an offset in AX.  It adjusts the
; address so that the offset is less than 16k.  Sets carry if address >= 1M; 
; in this case value returned in DX:AX is undefined.
;
CONV16K:	CMP	AX,16384
		CMC
		JNC	CONV16K_END
		CALL	CONVSEG2LIN
		CALL	CONVLIN2SEG
CONV16K_END:	RET
;
; This routine converts a segment in DX and an offset in AX into a linear
; address in DX:AX.  No error checking.
;
CONVSEG2LIN:	PUSH	CX
		MOV	CL,4
		ROL	DX,CL
		MOV	CX,DX
		AND	DX,0Fh
		AND	CX,0FFF0h
		ADD	AX,CX
		ADC	DX,0
		POP	CX
		RET
;
; This routine converts a linear address in DX:AX into a segment in DX and
; an offset in AX.  Sets carry if the input address is >= 1M; in this case
; the value returned is undefined.  The offset is less than 16.
;
CONVLIN2SEG:	PUSH	CX
		CMP	DX,10h		; > 1M?
		JAE	CONVLIN2SEG_BAD
		MOV	CL,4
		MOV	CH,AL
		SHR	AX,CL
		ROR	DX,CL
		OR	DX,AX
		MOV	AL,CH
		AND	AX,0Fh
		CLC
		JMP	CONVLIN2SEG_EXIT
CONVLIN2SEG_BAD: STC
CONVLIN2SEG_EXIT:
		POP	CX
		RET
;
; This routine verifies the handle passed in DX.  Sets carry if the handle
; is invalid or inactive.
;
VERIFYHNDL:	PUSH	BX
		CMP	DX,255
		JAE	VERIFYHNDL_BAD
		MOV	BX,DX
		CMP	CS:HNDLALLOC[BX],0
		JNE	VERIFYHNDL_BAD
		CLC
		JMP	VERIFYHNDL_EXIT
VERIFYHNDL_BAD:	STC
VERIFYHNDL_EXIT: POP	BX 
		RET
;
; This routine verifies that the logical page passed in BX is in fact 
; allocated to the handle passed in DX.  Assumes that the handle is itself 
; valid.  Clears carry if logical page is valid for handle, sets carry
; otherwise.
;
VERIFYLOGICAL:	PUSH	SI
		CALL	LOG2HARD
		POP	SI
		RET
;
; This routine converts the logical page passed in BX, for the handle passed
; in DX, into a hardware page number, returned in SI, provided that BX is
; indeed a logical page for DX.  Otherwise, this routine returns with carry
; set, and SI is undefined.
;
LOG2HARD:	PUSH	CX
		PUSH	DX
		PUSH	DI
		PUSH	DS
		OR	DX,8000h
		MOV	CX,512
		PUSH	CS
		POP	DS
		XOR	SI,SI
		XOR	DI,DI
LOG2HARD_LOOP:	CMP	PAGEALLOC[SI],0
		JNE	LOG2HARD_NEXT
		CMP	TRANSLATEARRAY[DI],DX
		JNE	LOG2HARD_NEXT
		CMP	LOGPAGES[DI],BX
		JE	LOG2HARD_GOOD
LOG2HARD_NEXT:	INC	SI
		INC	DI
		INC	DI
		LOOP	LOG2HARD_LOOP
		STC
		JMP	LOG2HARD_EXIT
LOG2HARD_GOOD:	CLC
LOG2HARD_EXIT:	POP	DS
		POP	DI
		POP	DX
		POP	CX
		RET
;
; This routine converts the logical page in DX and offset in AX into a
; linear address in DX:AX.  No error checking.
;
CONVEXP2LIN:	PUSH	BX
		XOR	BX,BX
		SHR	DX,1
		RCR	BX,1
		SHR	DX,1
		RCR	BX,1
		ADD	AX,BX
		POP	BX
		RET
;
; This routine converts the linear address in DX:AX into a logical page in
; DX and an offset in AX.  No error checking.
;
CONVLIN2EXP:	SHL	AX,1
		RCL	DX,1
		SHL	AX,1
		RCL	DX,1
		SHR	AX,1
		SHR	AX,1
		RET
;
; This routine compares the 32-bit value in DX:AX to the value in BX:CX
; and sets the zero and carry flags as if a CMP (DX:AX),(BX:CX) had been
; done (i.e., CF set if DX:AX < BX:CX, ZF set if DX:AX = BX:CX).
;
CMP32:		PUSH	DX
		PUSH	BX
		PUSH	AX
		SUB	AX,CX
		SBB	DX,BX		; carry set now
		MOV	BX,AX
		LAHF			; save the carry in AH
		AND	AH,0BFh		; clear ZF
		OR	BX,BX
		JNZ	CMP32_EXIT
		OR	DX,DX
		JNZ	CMP32_EXIT
		OR	AH,40h		; set ZF, result is zero
CMP32_EXIT:	SAHF
		POP	AX
		POP	BX
		POP	DX
		RET
;
; This routine converts the segment address passed in AX to a physical page
; number, returned in AL.  Sets carry if the segment address does not
; correspond to a physical page.
;
SEG2PHYSPAGE:	PUSH	BX
		PUSH	CX
		PUSH	DX
		MOV	DX,0D000h
		MOV	CX,4
		MOV	BL,0
L0:		CMP	AX,DX
		JE	>L1
		INC	BL
		ADD	DX,400h
		LOOP	L0
		STC
		JMP	>L2
L1:		MOV	AL,BL
		CLC
L2:		POP	DX
		POP	CX
		POP	BX
		RET
;
; This routine returns the hardware page currently mapped into the physical
; page passed in AL, or 0FFFFh if none.  Hardware page returned in AX.
;
GETPHYSPAGE:	PUSH	BX
		PUSH	CX
		PUSH	DX
		CLI		; disable interrupts while reading registers
		XOR	AH,AH
		ROR	AX,1
		ROR	AX,1
		ADD	AX,208h
		MOV	DX,AX
		;
		; Loop over the cards.
		;
		MOV	CX,4
		MOV	BX,OFFSET INSTALLED
		;
		; Skip this card if not installed.
		;
L0:		CMP	BYTE PTR CS:[BX],0	
		JNE	>L00
		IN	AL,DX
		TEST	AL,80h
		JNZ	>L1
		MOV	AL,0		; fix invalid values, if any
		OUT	DX,AL
L00:		INC	DX
		INC	DX
		INC	BX
		LOOP	L0
		MOV	AX,0FFFFh	; no card has this physical page mapped
		JMP	>L2
		;
		; Some card has the requested physical page mapped.  Low 7
		; bits of AL are the low 7 bits of the hardware page.  Bits
		; 1 and 2 of DX are the card number, which is also the high
		; 2 bits of the hardware page.
		;
L1:		XOR	AH,AH
		SHL	AL,1
		SHR	DX,1
		SHR	DX,1
		RCR	AL,1
		SHR	DX,1
		RCL	AH,1
L2:		STI
		POP	DX
		POP	CX
		POP	BX
		RET
;
; This routine sets the physical page passed in AL to the hardware page
; passed in SI.  If SI = 0FFFFh on entry, the physical page is unmapped.
;
SETPHYSPAGE:	PUSH	AX
		PUSH	CX
		PUSH	DX
		CLI		; disable interrupts while writing registers
		;
		; Loop over all cards, unmapping the physical page.
		;
		MOV	DL,AL
		XOR	DH,DH
		ROR	DX,1
		ROR	DX,1
		ADD	DX,208h
		MOV	CX,4
		XOR	AL,AL
L0:		OUT	DX,AL
		INC	DX
		INC	DX
		LOOP	L0
		;
		; Request to unmap?
		;
		CMP	SI,-1
		JE	>L1
		;
		; Map the requested page in.  DH is unchanged from before:
		; xx00_0010, where xx = physical page number.
		;
		MOV	AX,SI
		SHL	AX,1
		SHR	AL,1		; value to write uses low 7 bits
		OR	AL,80h
		MOV	DL,AH		; high 2 bits are card number
		SHL	DL,1
		ADD	DL,8
		OUT	DX,AL
L1:		STI
		POP	DX
		POP	CX
		POP	AX
		RET
;
; Subroutine, sets the mapping registers according to the saved context
; addressed by DS:SI.  The context is assumed valid.  Modifies no registers.
;
SETCONTEXT:	PUSH	AX
		PUSH	CX
		PUSH	DX
		PUSH	SI
		CLD		; set to increment
		CLI		; disable interrupts while changing registers
		;
		; Set all registers to zero, unmapping all pages.
		;
		MOV	DX,208h
		MOV	CX,4
		MOV	AL,0
L0:		PUSH	CX
		MOV	CX,4
L1:		OUT	DX,AL
		ADD	DX,4000h
		LOOP	L1
		SUB	DX,0BFFEh
		POP	CX
		LOOP	L0
		;
		; Set registers according to saved context.
		;
		MOV	DX,208h
		MOV	CX,4
L2:		PUSH	CX
		MOV	CX,4
L3:		LODSB
		OUT	DX,AL
		ADD	DX,4000h
		LOOP	L3
		SUB	DX,0BFFEh
		POP	CX
		LOOP	L2
		STI
		POP	SI
		POP	DX
		POP	CX
		POP	AX
		RET
;
; Subroutine, stores the contents of the mapping registers into the buffer
; addressed by ES:DI.  Modifies no registers.
;
GETCONTEXT:	PUSH	AX
		PUSH	BX
		PUSH	CX
		PUSH	DX
		PUSH	DI
		CLD
		CLI
		MOV	DX,208h
		MOV	CX,4
L0:		PUSH	CX
		MOV	CX,4
L1:		IN	AL,DX
		TEST	AL,80h		; fix invalid values, if any
		JNZ	>L2
		MOV	AL,0
		OUT	DX,AL
L2:		STOSB
		ADD	DX,4000h
		LOOP	L1
		SUB	DX,0BFFEh
		POP	CX
		LOOP	L0
		STI
		POP	DI
		POP	DX
		POP	CX
		POP	BX
		POP	AX
		RET
;
; Subroutine, checks the saved context addressed by DS:SI for validity.
; Sets carry if invalid.  Modifies no registers.
;
VERIFYCONTEXT:	PUSH	AX
		PUSH	BX
		PUSH	CX
		PUSH	SI
		PUSH	DI
		;
		; Loop over physical pages:
		;
		; DS:SI addresses the saved register value for card 0, current
		; page.
		;
		MOV	CX,4
VERIFYCONTEXT_PAGELOOP:
		PUSH	CX
		;
		; CS:DI addresses the byte indicating whether the current
		; card is installed.  AH counts the number of installed
		; cards with the current physical page mapped.  DS:SI+BX
		; addresses the saved register value for the current card,
		; current page.
		;
		MOV	DI,OFFSET INSTALLED
		MOV	AH,0
		XOR	BX,BX
		MOV	CX,4
		;
		; Skip this card if not installed.
		;
VERIFYCONTEXT_CARDLOOP:
		CMP	BYTE PTR CS:[DI],0
		JNE	>L0
		;
		; If not mapped, continue.
		;
		MOV	AL,[BX+SI]
		OR	AL,AL
		JZ	>L0
		;
		; If the register is nonzero, but the most significant bit
		; is not set, the register value is invalid, and the context
		; is bad.
		;
		JNS	VERIFYCONTEXT_BAD
		;
		; Otherwise, this physical page is mapped on the card.
		;
		INC	AH		; add 1 to count
L0:		INC	DI
		ADD	BX,4
		LOOP	VERIFYCONTEXT_CARDLOOP
		;
		; Context bad if more than one installed card is indicated
		; to have the same physical page mapped.
		;
		CMP	AH,1
		JA	VERIFYCONTEXT_BAD
		INC	SI
		POP	CX
		LOOP	VERIFYCONTEXT_PAGELOOP
		;
		; No errors found, report good.
		;
		CLC
		JMP	VERIFYCONTEXT_END
		;
		; Error in context, report bad.
		;
VERIFYCONTEXT_BAD:
		POP	CX
		STC
VERIFYCONTEXT_END:
		POP	DI
		POP	SI
		POP	CX
		POP	BX
		POP	AX
		RET
;
; Subroutine, searches the HANDLENAMES array for the name addressed by DS:SI.
; If found, returns the handle for the name in AX.  Otherwise, carry is set,
; and AX is undefined.
;
FINDNAME:	PUSH	BX
		PUSH	CX
		PUSH	SI
		PUSH	DI
		PUSH	ES
		;
		; Register use:
		;   ES:DI -> HANDLENAMES array, current entry.
		;   ES:BX -> HNDLALLOC array, current entry.
		;   (BX = handle number)
		;   DS:SI -> name to search for
		;   AX used to save SI
		;   CX = loop counter
		;
		PUSH	CS
		POP	ES
		MOV	DI,OFFSET HANDLENAMES
		XOR	BX,BX
		MOV	CX,255
L0:		CMP	ES:HNDLALLOC[BX],0	; skip if handle inactive
		JNE	>L1
		CALL	CMPNAME
		JE	FINDNAME_FOUND
L1:		INC	BX
		ADD	DI,8
		LOOP	L0
		STC
		JMP	FINDNAME_END
FINDNAME_FOUND:	CLC
		MOV	AX,BX		; copy handle into AX for return
FINDNAME_END:	POP	ES
		POP	DI
		POP	SI
		POP	CX
		POP	BX
		RET
;
; Subroutine, sets carry if the name addressed by DS:SI is null.
;
CHKNULLNAME:	PUSH	AX
		PUSH	CX
		PUSH	DI
		PUSH	ES
		PUSH	DS
		POP	ES
		MOV	DI,SI
		CLD
		MOV	CX,4
		XOR	AX,AX
		REPZ	SCASW
		LAHF
		SHL	AH,1
		SHL	AH,1
		POP	ES
		POP	DI
		POP	CX
		POP	AX
		RET
;
; Subroutine, sets zero flag if the name addressed by DS:SI is the same as
; that addressed by ES:DI.
;
CMPNAME:	PUSH	CX
		PUSH	SI
		PUSH	DI
		CLD
		MOV	CX,4
		REPE	CMPSW
		POP	DI
		POP	SI
		POP	CX
		RET