; MODSTUF2.ASM
;
; This is the second part of the assembler "half" of the player module.
; Here we handle time-critical things like actual resampling and mixing of
; mod samples.  This file is broken into four parts to get around the
; assembler's 64k limit on source files.
;
; EFFECT TICK ROUTINES ***********************************************
;
; These routines should modify no registers.  On entry, DS is the default
; data segment, and BX addresses the channel data structure.
;
; Tick procedure for effects 0 (arpeggio) and L (arpeggio, ScreamTracker
; version).
;
	EVEN
EF0TICK:
	PUSH	AX
	MOV	AL,[BX].CHANARPCOUNT	; get arpeggio counter in AL
	INC	AL			; increment
	CMP	AL,2			; wrap back to 0 if 3
	JBE	EF0TICK_SAVECOUNT
	XOR	AL,AL
EF0TICK_SAVECOUNT:
	MOV	[BX].CHANARPCOUNT,AL	; save back counter
	CMP	AL,1			; is it 0, 1 or 2?
	JE	EF0TICK_STEP1
	JA	EF0TICK_STEP2
	MOV	AX,[BX].CHANASTEP0FRAC	; counter is 0 - set sample step 0
	MOV	[BX].CHANSTEPFRAC,AX
	MOV	AX,[BX].CHANASTEP0INT
	MOV	[BX].CHANSTEPINT,AX
	JMP	SHORT EF0TICK_DONE
	EVEN
EF0TICK_STEP1:
	MOV	AX,[BX].CHANASTEP1FRAC	; counter is 1 - set sample step 1
	MOV	[BX].CHANSTEPFRAC,AX
	MOV	AX,[BX].CHANASTEP1INT
	MOV	[BX].CHANSTEPINT,AX
	JMP	SHORT EF0TICK_DONE
	EVEN
EF0TICK_STEP2:
	MOV	AX,[BX].CHANASTEP2FRAC	; counter is 2 - set sample step 2
	MOV	[BX].CHANSTEPFRAC,AX
	MOV	AX,[BX].CHANASTEP2INT
	MOV	[BX].CHANSTEPINT,AX
EF0TICK_DONE:
	POP	AX
	RET
;
; Tick procedure for effects 1 (slide up), 2 (slide down), I (slide/fineslide
; down), and J (slide/fineslide up).
;
	EVEN
EF1TICK:
	PUSH	AX
	PUSH	CX
	PUSH	DX
	PUSH	DI
	MOV	AX,[BX].CHANEFFPERIOD	; get effective period
	ADD	AX,[BX].CHANSLIDEINCR	; add signed slide increment
	JG	EF1TICK_HIGHENUF	; if less than 1, set to 1
	MOV	AX,1
	JMP	SHORT EF1TICK_SETPERIOD
	EVEN
EF1TICK_HIGHENUF:
	CMP	AX,27392		; if greater than 27392, set to 27392
	JLE	EF1TICK_SETPERIOD
	MOV	AX,27392
EF1TICK_SETPERIOD:
	MOV	[BX].CHANEFFPERIOD,AX	; save new effective period
	PUSH	BX			; save pointer to channel data
	MOV	CX,AX			; get period in CX
	MOV	BX,[BX].CHANC4SPD	; get C4SPD in BX
	CALL	GETSTEP			; get sample step (BX, CX, DI gone)
	POP	BX			; get back channel data pointer
	MOV	[BX].CHANSTEPFRAC,AX	; set sample step
	MOV	[BX].CHANSTEPINT,DX
	POP	DI
	POP	DX
	POP	CX
	POP	AX
	RET
;
; Tick procedure for effect 3 (slide to note).
;
	EVEN
EF3TICK:
#IF M_I286
	PUSHA
#ELSE
	PUSH	AX
	PUSH	CX
	PUSH	DX
	PUSH	SI
	PUSH	DI
#ENDIF
	MOV	AX,[BX].CHANEFFPERIOD	; get effective period in AX
	CMP	AX,[BX].CHANPERIODGOAL	; are we at the goal?
	JE	EF3TICK_DONE		; if so, we're done
	ADD	AX,[BX].CHANSLIDEINCR	; not done - add slide increment
	JG	EF3TICK_NOTNEGATIVE	; force result into the range 1-27392
	MOV	AX,1
	JMP	SHORT EF3TICK_BOUNDED
	EVEN
EF3TICK_NOTNEGATIVE:
	CMP	AX,27392
	JLE	EF3TICK_BOUNDED
	MOV	AX,27392
EF3TICK_BOUNDED:
	MOV	[BX].CHANEFFPERIOD,AX	; save new effective period
	CMP	[BX].CHANGLISS,0	; is the glissando flag set?
	JE	EF3TICK_NOGLISS
	PUSH	BX			; glissando set, save BX
	CALL	NEARHALF		; destroys BX, DX, SI, DI
	POP	BX			; restore BX
EF3TICK_NOGLISS:
	CMP	[BX].CHANSLIDEINCR,0	; is the slide increment negative?
	JL	EF3TICK_SLIDEDOWN
	CMP	AX,[BX].CHANPERIODGOAL	; increment positive:  test >= goal?
	JGE	EF3TICK_ATGOAL
	JMP	SHORT EF3TICK_SETSTEP
	EVEN
EF3TICK_SLIDEDOWN:
	CMP	AX,[BX].CHANPERIODGOAL	; increment negative:  test <= goal?
	JG	EF3TICK_SETSTEP
EF3TICK_ATGOAL:
	MOV	AX,[BX].CHANPERIODGOAL	; set test = goal
	MOV	[BX].CHANEFFPERIOD,AX	; set effective period = goal
EF3TICK_SETSTEP:
	PUSH	BX			; save BX
	MOV	CX,AX			; period in CX
	MOV	BX,[BX].CHANC4SPD	; C4SPD in BX
	CALL	GETSTEP			; compute sample step (BX, CX, DI gone)
	POP	BX			; restore BX
	MOV	[BX].CHANSTEPFRAC,AX	; store it
	MOV	[BX].CHANSTEPINT,DX
EF3TICK_DONE:
#IF M_I286
	POPA
#ELSE
	POP	DI
	POP	SI
	POP	DX
	POP	CX
	POP	AX
#ENDIF
	RET
;
; Tick procedure for effects 4 (vibrato) and Q (fine vibrato).
;
	EVEN
EF4TICK:
	PUSH	AX
	PUSH	CX
	PUSH	DX
	PUSH	SI
	MOV	SI,[BX].CHANVIBPOS	; get position in vibrato table
	ADD	SI,[BX].CHANVIBINCR	; compute new position
	AND	SI,3Fh			; (make 0-63)
	MOV	[BX].CHANVIBPOS,SI	; save it back
	CMP	SI,33			; position < 33? use it directly then
	JB	EF4TICK_LOOKUP
	NEG	SI			; position >= 33, use 64-position
	ADD	SI,64
EF4TICK_LOOKUP:
	SHL	SI,1			; get waveform table index
	ADD	SI,[BX].CHANVIBWAVE	; SI addresses period adjustment
	MOV	AX,[SI]			; AX is fixed-point adjustment
	MUL	[BX].CHANVIBDEPTH	; multiply by vibrato depth
	SHL	AX,1			; round to nearest
	ADC	DX,0			; DX is (positive) adjustment
	CMP	[BX].CHANVIBPOS,32	; position < 32? adjust up
	JB	EF4TICK_SETPERIOD
	NEG	DX			; position >= 32, adjust down
EF4TICK_SETPERIOD:
	ADD	DX,[BX].CHANEFFPERIOD	; DX is period for first tick
	JG	EF4TICK_HIGHENUF	; if < 1, set to 1
	MOV	DX,1
	JMP	SHORT EF4TICK_SETSTEP
	EVEN
EF4TICK_HIGHENUF:
	CMP	DX,27392		; if > 27392, set to 27392
	JLE	EF4TICK_SETSTEP
	MOV	DX,27392
EF4TICK_SETSTEP:
	PUSH	BX			; save pointer to channel data
	MOV	CX,DX			; get period in CX
	MOV	BX,[BX].CHANC4SPD	; get C4SPD in BX
	CALL	GETSTEP			; compute sample step (BX, CX, DI gone)
	POP	BX			; get back channel data pointer
	MOV	[BX].CHANSTEPFRAC,AX	; set sample step
	MOV	[BX].CHANSTEPINT,DX
	POP	SI
	POP	DX
	POP	CX
	POP	AX
	RET
;
; Tick procedure for effects 5 (slide to note plus volume slide) and N
; (slide to note plus volume slide, ScreamTracker version).
;
	EVEN
EF5TICK:
	CALL	EF3TICK			; do slide to note
	JMP	EFATICK			; and volume slide
;
; Tick procedure for effects 6 (vibrato plus volume slide) and M (vibrato
; plus volume slide, ScreamTracker version).
;
	EVEN
EF6TICK:
	CALL	EF4TICK			; do vibrato
	JMP	EFATICK			; and volume slide
;
; Tick procedure for effect 7 (tremolo).
;
	EVEN
EF7TICK:
	PUSH	AX
	PUSH	DX
	PUSH	SI
	MOV	SI,[BX].CHANTREMPOS	; get position in tremolo table
	ADD	SI,[BX].CHANTREMINCR	; compute new position
	AND	SI,3Fh			; (make 0-63)
	MOV	[BX].CHANTREMPOS,SI	; save it back
	CMP	SI,33			; position < 33? use it directly then
	JB	EF7TICK_LOOKUP
	NEG	SI			; position >= 33, use 64-position
	ADD	SI,64
EF7TICK_LOOKUP:
	SHL	SI,1			; get waveform table index
	ADD	SI,[BX].CHANTREMWAVE	; SI addresses period adjustment
	MOV	AX,[SI]			; AX is fixed-point adjustment
	MUL	[BX].CHANTREMDEPTH	; multiply by tremolo depth
	SHL	AX,1			; round to nearest
	ADC	DX,0			; DX is (positive) adjustment
	CMP	[BX].CHANTREMPOS,32	; position < 32? adjust up
	JB	EF7TICK_SETVOL
	NEG	DX			; position >= 32, adjust down
EF7TICK_SETVOL:
	ADD	DL,[BX].CHANEFFVOL	; add effective volume
	JGE	EF7TICK_HIGHENUF	; if < 0, set to 0
	XOR	DL,DL
	JMP	SHORT EF7TICK_SETMIX
	EVEN
EF7TICK_HIGHENUF:
	CMP	DL,64			; if > 64, set to 64
	JLE	EF7TICK_SETMIX
	MOV	DL,64
EF7TICK_SETMIX:
	MOV	[BX].CHANMIXVOL,DL
	POP	SI
	POP	DX
	POP	AX
	RET
;
; Tick procedure for effect A (volume slide) and H (volume slide/fineslide).
;
	EVEN
EFATICK:
	PUSH	AX
	MOV	AL,[BX].CHANEFFVOL	; get effective volume
	ADD	AL,[BX].CHANVOLINCR	; add signed volume increment
	JGE	EFATICK_POSITIVE	; result < 0?
	XOR	AL,AL			; set to 0 if so
	JMP	SHORT EFATICK_SETVOL
	EVEN
EFATICK_POSITIVE:
	CMP	AL,64			; result > 64?
	JLE	EFATICK_SETVOL
	MOV	AL,64			; set to 64 if so
EFATICK_SETVOL:
	MOV	[BX].CHANEFFVOL,AL	; set effective volume
	MOV	[BX].CHANMIXVOL,AL	; ... and mixing volume
	POP	AX
	RET
;
; Tick procedure for effect E9 (retrigger sample).
;
	EVEN
EFE9TICK:
	DEC	[BX].CHANRETRIGCOUNT	; decrement the retrigger count
	JNZ	EFE9TICK_DONE		; if not zero, done
	PUSH	AX
	PUSH	SI
	MOV	AL,[BX].CHANRETRIGSPEED	; set retrigger count to speed
	MOV	[BX].CHANRETRIGCOUNT,AL
	MOV	SI,[BX].CHANSAMPPTR	; set sample position to start
	MOV	AX,[SI].SAMPSEG
	MOV	[BX].CHANSAMPSEG,AX
	XOR	AX,AX
	MOV	[BX].CHANSAMPFRAC,AX
	MOV	[BX].CHANSAMPOFFS,AX
	MOV	[BX].CHANSAMPHALF,AL
	POP	SI
	POP	AX
EFE9TICK_DONE:
	RET
;
; Tick procedure for effect EC (cut sample).
;
	EVEN
EFECTICK:
	DEC	[BX].CHANCUTCOUNT	; decrement the cut count
	JNZ	EFECTICK_DONE		; if not zero, done
	MOV	[BX].CHANEFFVOL,0	; cut sample (set volume to 0)
	MOV	[BX].CHANMIXVOL,0
	MOV	[BX].CHANEFFTICKPROC,OFFSET DONOTHING ; no need to do it again
EFECTICK_DONE:
	RET
;
; Tick procedure for effect ED (delay sample).
;
	EVEN
EFEDTICK:
	DEC	[BX].CHANDELAYCOUNT	; decrement the delay count
	JNZ	EFEDTICK_DONE		; if not zero, done
	MOV	[BX].CHANEFFTICKPROC,OFFSET DONOTHING ; no need to do it again
EFEDTICK_DONE:
	RET
;
; Tick procedure for effect K (tremor).
;
	EVEN
EFKTICK:
	PUSH	AX
	MOV	AL,[BX].CHANTREMORCOUNT	; get tremor count
	INC	AL			; increment it
	CMP	AL,[BX].CHANTREMOREND	; is it "on" time?
	JE	EFKTICK_ONTIME
	MOV	[BX].CHANTREMORCOUNT,AL	; save count for next tick
	CMP	AL,[BX].CHANTREMOROFF	; is it "off" time?
	JE	EFKTICK_OFFTIME
EFKTICK_DONE:
	POP	AX
	RET
	EVEN
EFKTICK_ONTIME:
	MOV	[BX].CHANTREMORCOUNT,0	; "on" time - set count back to 0
	MOV	AL,[BX].CHANEFFVOL	; set mixing volume
	MOV	[BX].CHANMIXVOL,AL
	JMP	SHORT EFKTICK_DONE
	EVEN
EFKTICK_OFFTIME:
	MOV	[BX].CHANMIXVOL,0
	JMP	SHORT EFKTICK_DONE
;
; Tick procedure for effect O (retrigger plus volume slide).
;
	EVEN
EFOTICK:
	DEC	[BX].CHANRETRIGCOUNT	; decrement the retrigger count
	JZ	EFOTICK_DORETRIG	; if not zero, nothing to do
	RET
	EVEN
EFOTICK_DORETRIG:
	PUSH	AX
	PUSH	CX
	MOV	AL,[BX].CHANRETRIGSPEED	; set retrigger count to speed
	MOV	[BX].CHANRETRIGCOUNT,AL
	MOV	SI,[BX].CHANSAMPPTR	; set sample position to start
	MOV	AX,[SI].SAMPSEG
	MOV	[BX].CHANSAMPSEG,AX
	XOR	AX,AX
	MOV	[BX].CHANSAMPFRAC,AX
	MOV	[BX].CHANSAMPOFFS,AX
	MOV	[BX].CHANSAMPHALF,AL
	MOV	AL,[BX].CHANINFOBYTE	; get volume change
#IF M_I286
	SHR	AL,4
#ELSE
	SHR	AL,1
	SHR	AL,1
	SHR	AL,1
	SHR	AL,1
#ENDIF
	JZ	EFOTICK_DONE		; done if no volume change
	CMP	AL,6			; if 1 <= x <= 5, subtract a value
	JB	EFOTICK_SUBTRACT
	JE	EFOTICK_TWOTHIRDS	; if 6, multiply by 2/3
	CMP	AL,8			; if 7, multiply by 1/2
	JB	EFOTICK_ONEHALF
	JE	EFOTICK_DONE		; 8 is invalid for the volume slide
	CMP	AL,0Eh			; if 9 <= x <= 13, add a value
	JB	EFOTICK_ADD
	JE	EFOTICK_THREEHALFS	; if 14, multiply by 3/2
	MOV	AL,[BX].CHANEFFVOL	; 15:  multiply by 2
	SHL	AL,1
	JMP	SHORT EFOTICK_HIENUF	; guaranteed not negative at least
	EVEN
EFOTICK_SUBTRACT:
	DEC	AL			; 0 <= AL <= 4
	MOV	CL,AL			; CL is shift count
	MOV	AL,1			; AL = 1, 2, 4, 8 or 16
	SHL	AL,CL
	NEG	AL			; AL = -1, -2, -4, -8 or -16
	JMP	SHORT EFOTICK_DOADD
	EVEN
EFOTICK_ADD:
	SUB	AL,9			; 0 <= AL <= 4
	MOV	CL,AL			; CL is shift count
	MOV	AL,1			; AL = 1, 2, 4, 8 or 16
	SHL	AL,CL
EFOTICK_DOADD:
	ADD	AL,[BX].CHANEFFVOL	; add (or subtract) value
	JMP	SHORT EFOTICK_CONSTRAIN
	EVEN
EFOTICK_TWOTHIRDS:
	MOV	AL,[BX].CHANEFFVOL	; multiply by 2/3
	CBW
	SHL	AX,1
	MOV	CL,3
	DIV	CL
	SHR	CL,1
	CMP	CL,AH
	ADC	AL,0
	JMP	SHORT EFOTICK_SETVOL	; need not constrain - in range already
	EVEN
EFOTICK_ONEHALF:
	MOV	AL,[BX].CHANEFFVOL	; multiply by 1/2
	SHR	AL,1
	JMP	SHORT EFOTICK_SETVOL	; need not constrain - in range already
	EVEN
EFOTICK_THREEHALFS:
	MOV	AL,[BX].CHANEFFVOL	; multiply by 3/2
	MOV	AH,AL
	SHL	AL,1
	ADD	AL,AH
	SHR	AL,1
	JMP	SHORT EFOTICK_HIENUF	; guaranteed not negative at least
	EVEN
EFOTICK_CONSTRAIN:
	OR	AL,AL			; constrain:  if < 0, set to 0
	JNS	EFOTICK_HIENUF
	XOR	AL,AL
	JMP	SHORT EFOTICK_SETVOL
	EVEN
EFOTICK_HIENUF:
	CMP	AL,64			; if > 64, set to 64
	JBE	EFOTICK_SETVOL
	MOV	AL,64
EFOTICK_SETVOL:
	MOV	[BX].CHANEFFVOL,AL	; save new effective volume
	MOV	[BX].CHANMIXVOL,AL	; ... and new mixing volume
EFOTICK_DONE:
	POP	CX
	POP	AX
	RET
;
; EFFECT END ROUTINES ************************************************
;
; These routines should modify no registers.  On entry, DS is the default
; data segment, BX addresses the channel data structure, and ES:DI addresses
; the note structure for the current note.
;
; End procedure for effects 0 (arpeggio) and L (arpeggio, ScreamTracker
; version).
;
	EVEN
EF0END:
	PUSH	AX
	MOV	AX,[BX].CHANASTEP0FRAC	; set sample step back to default
	MOV	[BX].CHANSTEPFRAC,AX
	MOV	AX,[BX].CHANASTEP0INT
	MOV	[BX].CHANSTEPINT,AX
	POP	AX
	RET
;
; End procedure for effects 3 (slide to note), 5 (slide to note plus volume
; slide), and N (slide to note plus volume slide, ScreamTracker version).
;
	EVEN
EF3END:
	PUSH	AX
	PUSH	CX
	PUSH	DX
	PUSH	DI
	MOV	AL,ES:[DI].NOTEEFFNUM	; check the new effect number
	CMP	AL,3			; if 3, 5, or 23, this effect con-
	JE	EF3END_DONE		;   tinues, so do nothing
	CMP	AL,5
	JE	EF3END_DONE
	CMP	AL,23
	JE	EF3END_DONE
	CMP	[BX].CHANGLISS,0	; if no glissando slide, sample step
	JE	EF3END_DONE		;   already corresponds, so do nothing
	PUSH	BX			; save BX
	MOV	CX,[BX].CHANEFFPERIOD	; period in CX
	MOV	BX,[BX].CHANC4SPD	; C4SPD in BX
	CALL	GETSTEP			; compute sample step (BX, CX, DI gone)
	POP	BX			; restore BX
	MOV	[BX].CHANSTEPFRAC,AX	; store it
	MOV	[BX].CHANSTEPINT,DX
EF3END_DONE:
	POP	DI
	POP	DX
	POP	CX
	POP	AX
	RET
;
; End procedure for effects 4 (vibrato), 6 (vibrato plus volume slide), M
; (vibrato plus volume slide, ScreamTracker version), and Q (fine vibrato).
;
	EVEN
EF4END:
	PUSH	AX
	PUSH	CX
	PUSH	DX
	PUSH	DI
	MOV	AL,ES:[DI].NOTEEFFNUM	; check new effect number
	CMP	AL,4			; if 4, 6, 22, or 26, this effect
	JE	EF4END_DONE		;   continues, so do nothing
	CMP	AL,6
	JE	EF4END_DONE
	CMP	AL,22
	JE	EF4END_DONE
	CMP	AL,26
	JE	EF4END_DONE
	PUSH	BX			; save BX
	MOV	CX,[BX].CHANEFFPERIOD	; period in CX
	MOV	BX,[BX].CHANC4SPD	; C4SPD in BX
	CALL	GETSTEP			; compute sample step (BX, CX, DI gone)
	POP	BX			; restore BX
	MOV	[BX].CHANSTEPFRAC,AX	; store it
	MOV	[BX].CHANSTEPINT,DX
EF4END_DONE:
	POP	DI
	POP	DX
	POP	CX
	POP	AX
	RET
;
; End procedure for effects 7 (tremolo) and K (tremor).
;
	EVEN
EF7END:
	PUSH	AX
	MOV	AL,ES:[DI].NOTEEFFNUM	; if new effect is 7 or K (20), do
	CMP	AL,7			;   nothing
	JE	EF7END_DONE
	CMP	AL,20
	JE	EF7END_DONE
	MOV	AL,[BX].CHANEFFVOL	; set mixing volume = effective volume
	MOV	[BX].CHANMIXVOL,AL
EF7END_DONE:
	POP	AX
	RET
;
; RESAMPLERS *********************************************************
;
; These routines may modify any register, including DS, but should leave
; DF clear.  On entry, DS addresses the default data segment, and BX
; addresses the channel data structure.  They return nothing.
;
; RSAZ routine, "resamples" a "first" channel on the left or right.  This
; is a special-case resampler for a zero-volume sample.  For any size sample.
;
	EVEN
RSAZ:
	;
	; First fill the partial-mix buffer with zeros, since this is a
	; "first" channel.
	;
	MOV	CX,_samplespertick
	LES	DI,DWORD PTR [BX].CHANPARTOFFS
	XOR	AX,AX
RSAZ_FILLLOOP:
	STOSW
	ADD	DI,2
	LOOP	RSAZ_FILLLOOP
	;
	; Figure out where the sample pointer is after "generating" the
	; above silence samples.
	;
	MOV	AX,[BX].CHANSTEPFRAC	; multiply sample step by # samples
	MUL	_samplespertick
	MOV	BP,AX			; (accumulate in DX:DI.BP)
	MOV	DI,DX
	MOV	AX,[BX].CHANSTEPINT
	MUL	_samplespertick
	ADD	DI,AX
	ADC	DX,0
	ADD	BP,[BX].CHANSAMPFRAC	; add in current position in sample
	ADC	DI,[BX].CHANSAMPOFFS
	ADC	DX,0
	MOV	SI,[BX].CHANSAMPPTR	; SI addresses sample data
	CMP	[SI].SAMPEMSFLAG,1	; sample in EMS?
	JE	RSAZ_INEMS
	MOV	AX,[SI].SAMPSEG		; set CF if over 64k
	CMP	AX,[BX].CHANSAMPSEG
	JMP	SHORT RSAZ_ADDSEG
	EVEN
RSAZ_INEMS:
	CMP	[BX].CHANSAMPHALF,1	; set CF if 2nd 64k
	CMC
RSAZ_ADDSEG:
	ADC	DX,0			; add 1 if past 64k
	SHR	DX,1			; convert to words
	RCR	DI,1
	RCR	BP,1
	;
	; The current position in words relative to the start of the sample
	; is now in DX:DI.BP.  See if we're past the end of the sample.
	;
	OR	DX,DX
	JNZ	RSAZ_PASTEND
	CMP	DI,[SI].SAMPLEN
	JAE	RSAZ_PASTEND
	;
	; We're not past the end.  Save the new position.
	;
	CMP	[SI].SAMPEMSFLAG,1	; sample in EMS?
	JE	RSAZ_INEMS2
	SHL	BP,1			; convert back to bytes
	RCL	DI,1
#IF M_I286
	RCR	DX,4
#ELSE
	RCR	DX,1
	SHR	DX,1
	SHR	DX,1
	SHR	DX,1
#ENDIF
	ADD	DX,[SI].SAMPSEG
	MOV	[BX].CHANSAMPSEG,DX
	JMP	SHORT RSAZ_SETOFFSET
	EVEN
RSAZ_INEMS2:
	SHL	BP,1			; convert back to bytes
	RCL	DI,1
	RCL	DL,1
	MOV	[BX].CHANSAMPHALF,DL
RSAZ_SETOFFSET:
	MOV	[BX].CHANSAMPFRAC,BP
	MOV	[BX].CHANSAMPOFFS,DI
	RET
	;
	; We're past the end.  If the sample is not looped, mark it done
	; and set the last sample to zero.
	;
	EVEN
RSAZ_PASTEND:
	CMP	[SI].SAMPLLEN,0
	JNE	RSAZ_LOOPED
	MOV	[BX].CHANSAMPDONE,1
	MOV	[BX].CHANLASTSAMP,0
	RET
	;
	; Sample is looped.  Determine new position (inside the loop) and
	; save it.
	;
	EVEN
RSAZ_LOOPED:
	SUB	DI,[SI].SAMPLSTART	; subtract loop start from position
	SBB	DX,0
	MOV	AX,DX			; divide high word of offset by loop
	XOR	DX,DX			;   length
	DIV	[SI].SAMPLLEN		; keep remainder, discard quotient
	MOV	AX,DI			; divide remainder:low word
	DIV	[SI].SAMPLLEN
	ADD	DX,[SI].SAMPLSTART	; add loop start to remainder
	XOR	AX,AX
	CMP	[SI].SAMPEMSFLAG,1	; sample in EMS?
	JE	RSAZ_INEMS3
	SHL	BP,1			; convert to bytes
	RCL	DX,1
#IF M_I286
	RCR	AX,4
#ELSE
	RCR	AX,1
	SHR	AX,1
	SHR	AX,1
	SHR	AX,1
#ENDIF
	ADD	AX,[SI].SAMPSEG
	MOV	[BX].CHANSAMPSEG,AX
	JMP	SHORT RSAZ_SETOFFSET2
	EVEN
RSAZ_INEMS3:
	SHL	BP,1			; convert back to bytes
	RCL	DX,1
	RCL	AL,1
	MOV	[BX].CHANSAMPHALF,AL
RSAZ_SETOFFSET2:
	MOV	[BX].CHANSAMPFRAC,BP
	MOV	[BX].CHANSAMPOFFS,DX
	RET
;
; RS64AN routine, resamples the "first" channel on the left or right.  This
; resampler assumes that the sample is shorter than 64k and that we will
; not get to the end of it during the tick - a good but fairly common case.
;
	EVEN
RS64AN:
#IF M_I286
	; (286 version)
	;
	; Save the channel data pointer on the stack and put a copy in BP.
	;
	PUSH	BX
	MOV	BP,BX
	;
	; We know that the sample will not be played through.  The limit in
	; BX is the offset from the start of the partial-mix buffer of the
	; end of the output for the tick on this channel.  DI is a negative
	; number which added to that limit gives the offset where the
	; current sample goes.
	;
	MOV	DI,_samplespertick
	SHL	DI,1
	SHL	DI,1
	LES	BX,DWORD PTR [BP].CHANPARTOFFS
	ADD	BX,DI
	NEG	DI
	;
	; Patch in the volume.  If we have more than 8 channels, we need
	; to adjust the volume downward to avoid overflowing 16 bits in the
	; partial-mix buffer.
	;
	MOV	AL,[BP].CHANMIXVOL
	MUL	_globalvol
	SHL	AX,1
	SHL	AX,1
	MOV	CL,BYTE PTR _nchannels
	CMP	CL,8
	JBE	RS64AN_SETVOL
	SHR	AH,1
	CMP	CL,16
	JBE	RS64AN_SETVOL
	SHR	AH,1
RS64AN_SETVOL:
	MOV	BYTE PTR CS:RS64AN_VOLPT1,AH
	;
	; Now set up remaining register values.  We have:
	;
	;    ES:BX+DI      addresses buffer where samples go (already set)
	;    DS            current segment in sample
	;    SI.CX         current byte offset in sample
	;    BP.DX         sample step in bytes
	;
	MOV	CX,[BP].CHANSAMPFRAC	; BX = fraction of byte offset
	LDS	SI,DWORD PTR [BP].CHANSAMPOFFS ; DS:SI = segment:offset
	MOV	DX,[BP].CHANSTEPFRAC	; DX = fraction of sample step
	MOV	BP,[BP].CHANSTEPINT	; BP = integer part of sample step
	EVEN
RS64AN_MAINLOOP:
	DB	0B0h			; MOV AL,immed MIXVOL
RS64AN_VOLPT1:
	DB	0
	IMUL	BYTE PTR [SI]
	MOV	ES:[BX+DI],AX
	ADD	CX,DX
	ADC	SI,BP
	ADD	DI,4
	JNZ	RS64AN_MAINLOOP
	;
	; We got all the samples needed.  Save back the current segment
	; and offset.  There is at least one more sample to be output on
	; this channel, so we're not at the end yet.
	;
	POP	BP		; get back pointer to channel data
	MOV	[BP].CHANSAMPFRAC,CX
	MOV	[BP].CHANSAMPOFFS,SI
	RET
#ELSE
	; (8086 version)
	;
	; Save the channel data pointer on the stack.
	;
	PUSH	BX
	;
	; We know that the sample will not be played through.  The limit in
	; BP is the offset from the start of the partial-mix buffer of the
	; end of the output for the tick on this channel.  DI is a negative
	; number which added to that limit gives the offset where the
	; current sample goes.
	;
	MOV	DI,_samplespertick
	SHL	DI,1
	SHL	DI,1
	LES	BP,DWORD PTR [BX].CHANPARTOFFS
	ADD	BP,DI
	NEG	DI
	;
	; Patch in the volume.  In the 8086 version, we use the offset in
	; the precomputed multiplication table.  If there are more than 8
	; channels in the mod, we have to adjust the volume downward to
	; avoid overflowing 16 bits in the partial-mix buffer.
	;
	MOV	AH,[BX].CHANMIXVOL	; "multiply" mixing volume by global
	MOV	AL,_globalvol		;   volume
	SHL	AX,1
	MOV	SI,AX
	MOV	AX,MULTBL[SI]
	SHL	AX,1			; "divide" by 64
	SHL	AX,1
	MOV	CL,BYTE PTR _nchannels
	CMP	CL,8
	JBE	RS64AN_SETVOL
	SHR	AH,1
	CMP	CL,16
	JBE	RS64AN_SETVOL
	SHR	AH,1
RS64AN_SETVOL:
	XOR	AL,AL
	SHL	AX,1
	ADD	AX,OFFSET MULTBL
	MOV	WORD PTR CS:RS64AN_VOLPT1,AX
	;
	; Now set up remaining register values.  We have:
	;
	;    ES:BP+DI      addresses buffer where samples go (already set)
	;    DS            current segment in sample
	;    SI.CX         current byte offset in sample
	;    AX.DX         sample step in bytes
	;
	MOV	DX,[BX].CHANSTEPFRAC	; DX = fraction of sample step
	MOV	AX,[BX].CHANSTEPINT	; BP = integer part of sample step
	MOV	CX,[BX].CHANSAMPFRAC	; BX = fraction of byte offset
	LDS	SI,DWORD PTR [BX].CHANSAMPOFFS ; DS:SI = segment:offset
	EVEN
RS64AN_MAINLOOP:
	MOV	BL,[SI]
	XOR	BH,BH
	SHL	BX,1
	DB	36h,8Bh,9Fh		; MOV BX,SS:[BX+immed]
RS64AN_VOLPT1:
	DW	0
	MOV	ES:[BP+DI],BX
	ADD	CX,DX
	ADC	SI,AX
	ADD	DI,4
	JNZ	RS64AN_MAINLOOP
	;
	; We got all the samples needed.  Save back the current segment
	; and offset.  There is at least one more sample to be output on
	; this channel, so we're not at the end yet.
	;
	POP	BP		; get back pointer to channel data
	MOV	[BP].CHANSAMPFRAC,CX
	MOV	[BP].CHANSAMPOFFS,SI
	RET
#ENDIF
;
; RS64A routine, resamples the "first" channel on the left or right.  This
; resampler assumes that the sample is shorter than 64k, though we may get
; to the end of it during the tick.
;
	EVEN
RS64A:
#IF M_I286
	; (286 version)
	;
	; Patch in the volume.  If we have more than 8 channels, we need
	; to adjust the volume downward to avoid overflowing 16 bits in the
	; partial-mix buffer.  Do these patches first since we will need
	; them even if the sample has played through.
	;
	MOV	AL,[BX].CHANMIXVOL
	MUL	_globalvol
	SHL	AX,1
	SHL	AX,1
	MOV	CL,BYTE PTR _nchannels
	CMP	CL,8
	JBE	RS64A_SETVOL
	SHR	AH,1
	CMP	CL,16
	JBE	RS64A_SETVOL
	SHR	AH,1
RS64A_SETVOL:
	MOV	BYTE PTR CS:RS64A_VOLPT1,AH
	MOV	BYTE PTR CS:RS64A_VOLPT2,AH
	;
	; First check if the sample has played through.
	;
	MOV	CX,_samplespertick
	MOV	AL,[BX].CHANLASTSAMP
	LES	DI,DWORD PTR [BX].CHANPARTOFFS
	MOV	BP,BX
	CMP	[BX].CHANSAMPDONE,0
	JE	RS64A_DOPATCH
	JMP	RS64A_DORAMP
	;
	; The sample is not played through.  Patch the code (not enough
	; registers).  The limit is the offset from the start of the partial-
	; mix buffer of the end of the output for the tick on this channel.
	; DI is a negative number which added to that limit gives the offset
	; where the current sample goes.
	;
	EVEN
RS64A_DOPATCH:
	SHL	CX,1
	SHL	CX,1
	ADD	DI,CX
	MOV	WORD PTR CS:RS64A_LIMITPT1,DI
	MOV	WORD PTR CS:RS64A_LIMITPT2,DI
	NEG	CX
	MOV	DI,CX
	;
	; Get the sample step in BP.DX and patch in the fractional part.
	;
	MOV	BP,[BX].CHANSTEPINT	; BP = integer part of sample step
	MOV	DX,[BX].CHANSTEPFRAC	; DX = fraction of sample step
	MOV	WORD PTR CS:RS64A_STFRACPT1,DX ; patch in fraction
	;
	; Patch in the sample length, loop start, and loop length, leaving
	; the loop length in AX.  Convert these to bytes.
	;
	MOV	SI,[BX].CHANSAMPPTR	; address sample data
	MOV	CX,[SI].SAMPLEN		; get sample length in CX
	SHL	CX,1
	MOV	AX,[SI].SAMPLSTART	; patch in start of sample loop
	SHL	AX,1
	MOV	WORD PTR CS:RS64A_LSTARTPT1,AX
	MOV	WORD PTR CS:RS64A_LSTARTPT2,AX
	MOV	AX,[SI].SAMPLLEN	; patch in loop length
	SHL	AX,1
	MOV	WORD PTR CS:RS64A_LLENPT1,AX
	;
	; If the sample is looped, patch the code so we won't jump over the
	; part where we go back to the start of the loop.  Otherwise, patch
	; it so that we will.
	;
	MOV	WORD PTR CS:RS64A_SAMPEND,9090h
	OR	AX,AX			; loop length still in AX from above
	JNZ	RS64A_LOOPED
	MOV	WORD PTR CS:RS64A_SAMPEND,RS64A_JUMPINSTR
	;
	; Now set up remaining register values.  We have:
	;
	;    ES:DI+limit   addresses buffer where samples go (already set)
	;    DS            current segment in sample
	;    SI.BX         current byte offset in sample
	;    BP.DX         sample step in bytes (already set)
	;    CX            sample length in bytes (already set)
	;
RS64A_LOOPED:
	PUSH	BX			; save pointer to channel data
	LDS	SI,DWORD PTR [BX].CHANSAMPOFFS ; DS:SI = segment:offset
	MOV	BX,SS:[BX].CHANSAMPFRAC	; BX = fraction of byte offset
	EVEN
RS64A_MAINLOOP:
	DB	0B0h			; MOV AL,immed MIXVOL
RS64A_VOLPT1:
	DB	0
	IMUL	BYTE PTR [SI]
	DB	26h,89h,85h		; MOV ES:[DI+immed],AX
RS64A_LIMITPT1:
	DW	0
	ADD	BX,DX
	ADC	SI,BP
	JC	RS64A_SAMPEND
	CMP	SI,CX
	JAE	RS64A_SAMPEND		; (JAE = JNC)
RS64A_MAINLPEND:
	ADD	DI,4
	JNZ	RS64A_MAINLOOP
	;
	; We got all the samples needed.  Save back the current segment
	; and offset.  There is at least one more sample to be output on
	; this channel, so we're not at the end yet.
	;
	POP	BP		; get back pointer to channel data
	MOV	[BP].CHANSAMPFRAC,BX
	MOV	[BP].CHANSAMPOFFS,SI
	RET
	;
	; We got to the end of the sample.  CF:SI.BX is the current offset
	; in bytes from the start of the sample.  If the sample is looped,
	; go back to the beginning of the loop.
	;
	EVEN
RS64A_SAMPEND:
	DB	0EBh		; JMP SHORT below (or NOP, NOP)
	DB	RS64A_JUMPSIZE
	MOV	DX,0		; LSB of DX = CF
	RCL	DX,1
	MOV	AX,SI		; DX:AX = integer of offset in bytes
	DB	2Dh		; SUB AX,immed SAMPLSTART (in bytes)
RS64A_LSTARTPT1:
	DW	0
	SBB	DX,0
	DB	0BEh		; MOV SI,immed SAMPLLEN (in bytes)
RS64A_LLENPT1:
	DW	0
	DIV	SI
	MOV	SI,DX
	DB	81h,0C6h	; ADD SI,immed SAMPLSTART (in bytes)
RS64A_LSTARTPT2:
	DW	0
	DB	0BAh		; MOV DX,immed STEPFRAC
RS64A_STFRACPT1:
	DW	0
	JMP	SHORT RS64A_MAINLPEND
	;
	; We got to the end of the sample, and it is not looped.  CF:SI.BX
	; is the current offset in bytes from the start of the sample.  BP.DX
	; is the sample step in bytes.
	;
	EVEN
RS64A_JUMPSIZE		EQU	$ - (OFFSET RS64A_SAMPEND + 2)
RS64A_JUMPINSTR		EQU	(RS64A_JUMPSIZE*256) + 0EBh
	SUB	BX,DX			; back up to last sample
	SBB	SI,BP			; guaranteed to get back under 64k
	POP	BP			; BP addresses channel data
	MOV	AL,[SI]			; get back sample
	MOV	[BP].CHANSAMPDONE,1	; mark the sample finished
	ADD	DI,4			; DI didn't get incremented last time
	JZ	RS64A_DONE		; are we done?
	MOV	CX,DI			; CX = number of samples left to do
	NEG	CX
	SHR	CX,1
	SHR	CX,1
	DB	81h,0C7h		; ADD DI,immed (make DI a real offset)
RS64A_LIMITPT2:
	DW	0
	;
	; Code for ramping to the baseline.  When we get here, the sample is
	; not looped and has been played through.  CX is the number of output
	; samples left to generate this tick.  ES:DI addresses the buffer
	; where output samples go.  AL is the next input sample, moving
	; toward the baseline.  BP addresses the channel data structure.
	;
RS64A_DORAMP:
	OR	AL,AL			; see whether we're ramping up or down
	JZ	RS64A_FILL		; no ramping if already at baseline
	DB	0B7h			; MOV BH,immed MIXVOL
RS64A_VOLPT2:
	DB	0
	JS	RS64A_BELOW		; ramp up if negative
        ;
        ; Current sample is above the baseline.  Decrement the sample till
        ; it gets to 0 or until we have enough for this tick.
        ;
RS64A_ABOVE:
	MOV	BL,AL		; save copy in BL
	IMUL	BH		; multiply by the volume
	STOSW			; store it in the partial-mix buffer
	ADD	DI,2		; go to next in partial-mix buffer
	MOV	AL,BL		; get back sample in AL
	DEC	AL		; decrement sample
	LOOPNZ	RS64A_ABOVE	; go again if more and not at baseline
	JCXZ	RS64A_DONE	; done if partial-mix buffer full
	JMP	SHORT RS64A_FILL ; ... otherwise fill with silence
        ;
        ; Current sample is below the baseline.  Increment the sample till
        ; it gets to 0 or until we have enough for this tick.
        ;
	EVEN
RS64A_BELOW:
	MOV	BL,AL		; save copy in BL
	IMUL	BH		; multiply by the volume
	STOSW			; store it in the partial-mix buffer
	ADD	DI,2		; go to next in partial-mix buffer
	MOV	AL,BL		; get back sample in AL
	INC	AL		; increment sample
	LOOPNZ	RS64A_BELOW	; go again if more and not at baseline
	JCXZ	RS64A_DONE	; done if partial-mix buffer full
	;
	; Current sample is at the baseline.  Since this is the first channel
	; on the left or right, we still need to fill with zeros in this case.
	;
RS64A_FILL:
	XOR	AX,AX
RS64A_FILLLOOP:
	STOSW
	ADD	DI,2
	LOOP	RS64A_FILLLOOP
	;
	; All done for now.
	;
RS64A_DONE:
	MOV	[BP].CHANLASTSAMP,AL	; save the last output sample
	RET
#ELSE
	; (8086 version)
	;
	; Patch in the volume.  In the 8086 version, we use the offset in
	; the precomputed multiplication table.  These patches need to be
	; done even if the sample has been played through, so do them here.
	; If there are more than 8 channels in the mod, the volume needs to
	; be adjusted downward to avoid overflowing 16 bits in the partial-
	; mix buffer.
	;
	MOV	AH,[BX].CHANMIXVOL	; "multiply" mixing volume by global
	MOV	AL,_globalvol		;   volume
	SHL	AX,1
	MOV	SI,AX
	MOV	AX,MULTBL[SI]
	SHL	AX,1			; "divide" by 64
	SHL	AX,1
	MOV	CL,BYTE PTR _nchannels
	CMP	CL,8
	JBE	RS64A_SETVOL
	SHR	AH,1
	CMP	CL,16
	JBE	RS64A_SETVOL
	SHR	AH,1
RS64A_SETVOL:
	XOR	AL,AL
	SHL	AX,1
	ADD	AX,OFFSET MULTBL
	MOV	WORD PTR CS:RS64A_VOLPT1,AX
	MOV	WORD PTR CS:RS64A_VOLPT2,AX
	MOV	WORD PTR CS:RS64A_VOLPT3,AX
	;
	; Check if the sample has played through.
	;
	MOV	CX,_samplespertick
	MOV	AL,[BX].CHANLASTSAMP
	LES	DI,DWORD PTR [BX].CHANPARTOFFS
	MOV	BP,BX
	CMP	[BX].CHANSAMPDONE,0
	JE	RS64A_DOPATCH
	JMP	RS64A_DORAMP
	;
	; The sample is not played through.  Patch the code (not enough
	; registers).  The limit is the offset from the start of the partial-
	; mix buffer of the end of the output for the tick on this channel.
	; DI is a negative number which added to that limit gives the offset
	; where the current sample goes.
	;
	EVEN
RS64A_DOPATCH:
	SHL	CX,1
	SHL	CX,1
	ADD	DI,CX
	MOV	WORD PTR CS:RS64A_LIMITPT1,DI
	MOV	WORD PTR CS:RS64A_LIMITPT2,DI
	NEG	CX
	MOV	DI,CX
	;
	; Patch in the sample length and loop length, leaving the loop
	; length in AX.  Convert these to bytes.
	;
	MOV	SI,[BX].CHANSAMPPTR	; address sample data
	MOV	CX,[SI].SAMPLEN		; get sample length in CX
	SHL	CX,1
	MOV	AX,[SI].SAMPLLEN	; patch in loop length
	SHL	AX,1
	MOV	WORD PTR CS:RS64A_LLENPT1,AX
	;
	; If the sample is looped, patch the code so we won't jump over the
	; part where we go back to the start of the loop.  Otherwise, patch
	; it so that we will.
	;
	MOV	WORD PTR CS:RS64A_SAMPEND,9090h
	OR	AX,AX			; loop length still in AX from above
	JNZ	RS64A_LOOPED
	MOV	WORD PTR CS:RS64A_SAMPEND,RS64A_JUMPINSTR
	;
	; Now set up register values.  We have:
	;
	;    ES:DI+limit   addresses buffer where samples go (already set)
	;    DS            current segment in sample
	;    SI.AX         current byte offset in sample
	;    BP.DX         sample step in bytes
	;    CX            sample length in bytes (already set)
	;
RS64A_LOOPED:
	PUSH	BX			; save pointer to channel data
	MOV	BP,[BX].CHANSTEPINT	; BP = integer part of sample step
	MOV	DX,[BX].CHANSTEPFRAC	; DX = fraction of sample step
	MOV	AX,[BX].CHANSAMPFRAC	; AX = fraction of byte offset
	LDS	SI,DWORD PTR [BX].CHANSAMPOFFS ; DS:SI = segment:offset
	EVEN
RS64A_MAINLOOP:
	MOV	BL,[SI]
	XOR	BH,BH
	SHL	BX,1
	DB	36h,8Bh,9Fh		; MOV BX,SS:[BX+immed]
RS64A_VOLPT1:
	DW	0
	DB	26h,89h,9Dh		; MOV ES:[DI+immed],BX
RS64A_LIMITPT1:
	DW	0
	ADD	AX,DX
	ADC	SI,BP
	JC	RS64A_SAMPEND
	CMP	SI,CX
	JAE	RS64A_SAMPEND		; (JAE = JNC)
RS64A_MAINLPEND:
	ADD	DI,4
	JNZ	RS64A_MAINLOOP
	;
	; We got all the samples needed.  Save back the current segment
	; and offset.  There is at least one more sample to be output on
	; this channel, so we're not at the end yet.
	;
	POP	BP		; get back pointer to channel data
	MOV	[BP].CHANSAMPFRAC,AX
	MOV	[BP].CHANSAMPOFFS,SI
	RET
	;
	; We got to the end of the sample.  CF:SI.AX is the current offset
	; in bytes from the start of the sample.  If the sample is looped,
	; go back to the beginning of the loop.
	;
	EVEN
RS64A_SAMPEND:
	DB	0EBh		; JMP SHORT below (or NOP, NOP)
	DB	RS64A_JUMPSIZE
	MOV	BL,0		; LSB of BL = CF
	RCL	BL,1		; BL:SI = integer of offset in bytes
RS64A_ADJLOOP:
	DB	81h,0EEh	; SUB SI,immed SAMPLLEN (in bytes)
RS64A_LLENPT1:
	DW	0
	SBB	BL,0
        JNZ     RS64A_ADJLOOP
	CMP	SI,CX
        JAE     RS64A_ADJLOOP
	JMP	SHORT RS64A_MAINLPEND
	;
	; We got to the end of the sample, and it is not looped.  CF:SI.AX
	; is the current offset in bytes from the start of the sample.  BP.DX
	; is the sample step in bytes.
	;
	EVEN
RS64A_JUMPSIZE		EQU	$ - (OFFSET RS64A_SAMPEND + 2)
RS64A_JUMPINSTR		EQU	(RS64A_JUMPSIZE*256) + 0EBh
	SUB	AX,DX			; back up to last sample
	SBB	SI,BP			; guaranteed to get back under 64k
	POP	BP			; BP addresses channel data
	MOV	AL,[SI]			; get back sample
	MOV	[BP].CHANSAMPDONE,1	; mark the sample finished
	ADD	DI,4			; DI didn't get incremented last time
	JZ	RS64A_DONE		; are we done?
	MOV	CX,DI			; CX = number of samples left to do
	NEG	CX
	SHR	CX,1
	SHR	CX,1
	DB	81h,0C7h		; ADD DI,immed (make DI a real offset)
RS64A_LIMITPT2:
	DW	0
	MOV	BX,DGROUP		; DS addresses local data again
	MOV	DS,BX
	;
	; Code for ramping to the baseline.  When we get here, the sample is
	; not looped and has been played through.  CX is the number of output
	; samples left to generate this tick.  ES:DI addresses the buffer
	; where output samples go.  AL is the next input sample, moving
	; toward the baseline.  BP addresses the channel data structure.
	; DS addresses the default data segment.
	;
RS64A_DORAMP:
	OR	AL,AL			; see whether we're ramping up or down
	JZ	RS64A_FILL		; no ramping if already at baseline
	JS	RS64A_BELOW		; ramp up if negative
        ;
        ; Current sample is above the baseline.  Decrement the sample till
        ; it gets to 0 or until we have enough for this tick.
        ;
RS64A_ABOVE:
	MOV	BL,AL		; get sample in BL
	XOR	BH,BH		; convert to word offset
	SHL	BX,1
	DB	8Bh,9Fh		; MOV BX,[BX+immed] (multiply by volume)
RS64A_VOLPT2:
	DW	0
	MOV	ES:[DI],BX	; store it in the partial-mix buffer
	ADD	DI,4		; go to next in partial-mix buffer
	DEC	AL		; decrement sample
	LOOPNZ	RS64A_ABOVE	; go again if more and not at baseline
	JCXZ	RS64A_DONE	; done if partial-mix buffer full
	JMP	SHORT RS64A_FILL ; ... otherwise fill with silence
        ;
        ; Current sample is below the baseline.  Increment the sample till
        ; it gets to 0 or until we have enough for this tick.
        ;
	EVEN
RS64A_BELOW:
	MOV	BL,AL		; save copy in BL
	XOR	BH,BH		; convert to word offset
	SHL	BX,1
	DB	8Bh,9Fh		; MOV BX,[BX+immed] (multiply by volume)
RS64A_VOLPT3:
	DW	0
	MOV	ES:[DI],BX	; store it in the partial-mix buffer
	ADD	DI,4		; go to next in partial-mix buffer
	INC	AL		; increment sample
	LOOPNZ	RS64A_BELOW	; go again if more and not at baseline
	JCXZ	RS64A_DONE	; done if partial-mix buffer full
	;
	; Current sample is at the baseline.  Since this is the first channel
	; on the left or right, we still need to fill with zeros in this case.
	;
RS64A_FILL:
	XOR	AX,AX
RS64A_FILLLOOP:
	STOSW
	ADD	DI,2
	LOOP	RS64A_FILLLOOP
	;
	; All done for now.
	;
RS64A_DONE:
	MOV	[BP].CHANLASTSAMP,AL	; save the last output sample
	RET
#ENDIF
;
; RS128A routine, resamples the "first" channel on the left or right.  This
; is the worst-case resampler for the "first" channel.  It can handle samples
; longer than 64k where we may get to the end of the sample during the tick.
;
	EVEN
RS128A:
#IF M_I286
	; (286 version)
	;
	; Patch in the volume.  If we have more than 8 channels, we need
	; to adjust the volume downward to avoid overflowing 16 bits in the
	; partial-mix buffer.  Do these patches first since we will need
	; them even if the sample has played through.
	;
	MOV	AL,[BX].CHANMIXVOL
	MUL	_globalvol
	SHL	AX,1
	SHL	AX,1
	MOV	CL,BYTE PTR _nchannels
	CMP	CL,8
	JBE	RS128A_SETVOL
	SHR	AH,1
	CMP	CL,16
	JBE	RS128A_SETVOL
	SHR	AH,1
RS128A_SETVOL:
	MOV	BYTE PTR CS:RS128A_VOLPT1,AH
	MOV	BYTE PTR CS:RS128A_VOLPT2,AH
	;
	; Check if the sample has played through.
	;
	MOV	CX,_samplespertick
	MOV	AL,[BX].CHANLASTSAMP
	LES	DI,DWORD PTR [BX].CHANPARTOFFS
	MOV	BP,BX
	CMP	[BX].CHANSAMPDONE,0
	JE	RS128A_DOPATCH
	JMP	RS128A_DORAMP
	;
	; The sample is not played through.  Patch the code (not enough
	; registers).  The limit is the offset from the start of the partial-
	; mix buffer of the end of the output for the tick on this channel.
	; DI is a negative number which added to that limit gives the offset
	; where the current sample goes.
	;
	EVEN
RS128A_DOPATCH:
	SHL	CX,1
	SHL	CX,1
	ADD	DI,CX
	MOV	WORD PTR CS:RS128A_LIMITPT1,DI
	MOV	WORD PTR CS:RS128A_LIMITPT2,DI
	NEG	CX
	MOV	DI,CX
	;
	; Get the sample step in BP.DX, convert it to words, and patch in
	; the fractional part.
	;
	MOV	BP,[BX].CHANSTEPINT	; BP = integer part of sample step
	MOV	DX,[BX].CHANSTEPFRAC	; DX = fraction of sample step
	SHR	BP,1			; convert to words
	RCR	DX,1
	MOV	WORD PTR CS:RS128A_STFRACPT1,DX ; patch in fraction
	;
	; Patch in the sample length, loop start, and loop length, leaving
	; the loop length in AX.  These are word quantities already.
	;
	MOV	SI,[BX].CHANSAMPPTR	; SI addresses sample data
	MOV	AX,[SI].SAMPLEN		; patch in sample length
	MOV	WORD PTR CS:RS128A_LENPT1,AX
	MOV	AX,[SI].SAMPLSTART	; patch in start of sample loop
	MOV	WORD PTR CS:RS128A_LSTARTPT1,AX
	MOV	WORD PTR CS:RS128A_LSTARTPT2,AX
	MOV	AX,[SI].SAMPLLEN	; patch in loop length
	MOV	WORD PTR CS:RS128A_LLENPT1,AX
	;
	; If the sample is looped, patch the code so we won't jump over the
	; part where we go back to the start of the loop.  Otherwise, patch
	; it so that we will.
	;
	MOV	WORD PTR CS:RS128A_SAMPEND,9090h
	OR	AX,AX			; loop length still in AX from above
	JNZ	RS128A_LOOPED
	MOV	WORD PTR CS:RS128A_SAMPEND,RS128A_JUMPINSTR
	;
	; Now set up register values.  We have:
	;
	;    ES:DI+limit   addresses buffer where samples go (already set)
	;    DS            current segment in sample
	;    SI.BX         current word offset in sample
	;    BP.DX         sample step in words (already set)
	;    CX            8000h if 64k or more past sample start, 0 otherwise
	;
RS128A_LOOPED:
	PUSH	BX			; save pointer to channel data
	XOR	CX,CX			; assume first 64k
	MOV	AX,[BX].CHANSAMPSEG	; get segment in AX
	CMP	[SI].SAMPEMSFLAG,1	; is the sample in EMS?
	JE	RS128A_INEMS
	MOV	BYTE PTR CS:RS128A_SEGJMP,RS128A_SEGJMP1
	CMP	[SI].SAMPSEG,AX		; set CF if 2nd 64k
	JMP	SHORT RS128A_SETCX
	EVEN
RS128A_INEMS:
	MOV	BYTE PTR CS:RS128A_SEGJMP,RS128A_SEGJMP2
	MOV	SI,[SI].SAMPEMSHANDLE	; get EMS handle and patch into code
	MOV	WORD PTR CS:RS128A_EMSHANDLE,SI
	CMP	[BX].CHANSAMPHALF,1	; set CF if in upper 64k
	CMC
RS128A_SETCX:				; CF set if 2nd 64k
	RCR	CX,1
	MOV	SI,[BX].CHANSAMPOFFS	; SI = integer part of byte offset
	MOV	BX,[BX].CHANSAMPFRAC	; BX = fraction of byte offset
	SHR	SI,1			; convert offset to words
	RCR	BX,1
	MOV	DS,AX			; DS = current segment
	EVEN
RS128A_MAINLOOP:
	DB	0B0h			; MOV AL,immed MIXVOL
RS128A_VOLPT1:
	DB	0
	SHL	BX,1
	RCL	SI,1
	IMUL	BYTE PTR [SI]
	SHR	SI,1
	RCR	BX,1
	DB	26h,89h,85h		; MOV ES:[DI+immed],AX
RS128A_LIMITPT1:
	DW	0
	OR	SI,CX
	ADD	BX,DX
	ADC	SI,BP
	JC	RS128A_SAMPEND
	DB	81h,0FEh		; CMP SI,immed SAMPLEN
RS128A_LENPT1:
	DW	0
	JAE	RS128A_SAMPEND		; (JAE = JNC)
RS128A_SEGTEST:
	XOR	SI,CX
	DB	78h			; JS segchange_code
RS128A_SEGJMP:
	DB	0
RS128A_MAINLPEND:
	ADD	DI,4
	JNZ	RS128A_MAINLOOP
	;
	; We got all the samples needed.  Save back the current segment
	; and offset and the "over 64k" flag.  There is at least one more
	; sample to be output on this channel, so we're not at the end yet.
	;
	POP	BP		; get back pointer to channel data
	SHL	BX,1		; SI.BX is position in bytes relative to
	RCL	SI,1		;   current segment
	MOV	[BP].CHANSAMPFRAC,BX
	MOV	[BP].CHANSAMPOFFS,SI
	MOV	[BP].CHANSAMPSEG,DS
	ROL	CH,1
	MOV	[BP].CHANSAMPHALF,CH
	RET
	;
	; Different 64k segment now.  Adjust DS, CX, and SI.  The most
	; significant bit of SI is set when we get here and needs to be
	; cleared in any case.  If CX is 0, we're going from the lower 64k
	; to the upper 64k, and we need to set DS to SAMPSEG+1000h and CX
	; to 8000h.  If CX is 8000h, we're going from the upper 64k to the
	; lower 64k (a looped sample longer than 64k, and we just looped).
	; In that case, we need to set DS to SAMPSEG and CX to 0.  This is
	; for a sample loaded in conventional RAM.
	;
	EVEN
RS128A_SEGJMP1		EQU	$ - (OFFSET RS128A_MAINLPEND)
	MOV	AX,8000h
	XOR	SI,AX			; clear MSB of SI
	XOR	CX,AX			; invert MSB of CX
	MOV	AX,DS			; DS value in AX for adjustment
	JZ	RS128A_SEGLOW		; if CX = 0, CX was 8000h, so low 64k
	ADD	AH,10h			; low to high, add 1000h to DS
	MOV	DS,AX
	JMP	SHORT RS128A_MAINLPEND
	EVEN
RS128A_SEGLOW:
	SUB	AH,10h			; high to low, subtract 1000h from DS
	MOV	DS,AX
	JMP	SHORT RS128A_MAINLPEND
	;
	; Different 64k segment now.  Adjust CX and SI, and map in the
	; correct half of the EMS sample.  The most significant bit of SI
	; is set when we get here and needs to be cleared in any case.  If
	; CX is 0, we're going from the lower 64k to the upper 64k, and we
	; need to set CX to 8000h and map in the upper 64k.  If CX is 8000h,
	; we're going from the upper 64k to the lower 64k (a looped sample
	; longer than 64k, and we just looped).  In that case, we need to
	; set CX to 0 and map in the lower 64k.  This is for a sample loaded
	; in EMS RAM.
	;
	EVEN
RS128A_SEGJMP2		EQU	$ - (OFFSET RS128A_MAINLPEND)
	MOV	AX,8000h
	XOR	SI,AX			; clear MSB of SI
	XOR	CX,AX			; invert MSB of CX
	DB	68h			; PUSH immed SAMPEMSHANDLE
RS128A_EMSHANDLE:
	DW	0
	JZ	RS128A_SEGLOW2		; if CX = 0, CX was 8000h, so low 64k
	CALL	MAPEMS2ND		; low to high, map in the second 64k
	JMP	SHORT RS128A_MAINLPEND
	EVEN
RS128A_SEGLOW2:
	CALL	MAPEMS1ST		; high to low, map in the first 64k
	JMP	SHORT RS128A_MAINLPEND
	;
	; We got to the end of the sample.  CF:SI.BX is the current offset
	; in words from the start of the sample.  If the sample is looped,
	; go back to the beginning of the loop.
	;
	EVEN
RS128A_SAMPEND:
	DB	0EBh		; JMP SHORT below (or NOP, NOP)
	DB	RS128A_JUMPSIZE
	MOV	DX,0		; LSB of DX = CF
	RCL	DX,1
	MOV	AX,SI		; DX:AX = integer of offset in words
	DB	2Dh		; SUB AX,immed SAMPLSTART
RS128A_LSTARTPT1:
	DW	0
	SBB	DX,0
	DB	0BEh		; MOV SI,immed SAMPLLEN
RS128A_LLENPT1:
	DW	0
	DIV	SI
	MOV	SI,DX
	DB	81h,0C6h	; ADD SI,immed SAMPLSTART
RS128A_LSTARTPT2:
	DW	0
	DB	0BAh		; MOV DX,immed STEPFRAC
RS128A_STFRACPT1:
	DW	0
	JMP	SHORT RS128A_SEGTEST
	;
	; We got to the end of the sample, and it is not looped.  CF:SI.BX
	; is the current offset in words from the start of the sample.  BP.DX
	; is the sample step in words.
	;
	EVEN
RS128A_JUMPSIZE		EQU	$ - (OFFSET RS128A_SAMPEND + 2)
RS128A_JUMPINSTR	EQU	(RS128A_JUMPSIZE*256) + 0EBh
	SUB	BX,DX			; back up to last sample
	SBB	SI,BP			; guaranteed to get back under 128k
	POP	BP			; BP addresses channel data
	MOV	AL,[SI]			; get back sample
	MOV	[BP].CHANSAMPDONE,1	; mark the sample finished
	ADD	DI,4			; DI didn't get incremented last time
	JZ	RS128A_DONE		; are we done?
	MOV	CX,DI			; CX = number of samples left to do
	NEG	CX
	SHR	CX,1
	SHR	CX,1
	DB	81h,0C7h		; ADD DI,immed (make DI a real offset)
RS128A_LIMITPT2:
	DW	0
	;
	; Code for ramping to the baseline.  When we get here, the sample is
	; not looped and has been played through.  CX is the number of output
	; samples left to generate this tick.  ES:DI addresses the buffer
	; where output samples go.  AL is the next input sample, moving
	; toward the baseline.  BP addresses the channel data structure.
	;
RS128A_DORAMP:
	OR	AL,AL			; see whether we're ramping up or down
	JZ	RS128A_FILL		; no ramping if already at baseline
	DB	0B7h			; MOV BH,immed MIXVOL
RS128A_VOLPT2:
	DB	0
	JS	RS128A_BELOW		; ramp up if negative
        ;
        ; Current sample is above the baseline.  Decrement the sample till
        ; it gets to 0 or until we have enough for this tick.
        ;
RS128A_ABOVE:
	MOV	BL,AL		; save copy in BL
	IMUL	BH		; multiply by the volume
	STOSW			; store it in the partial-mix buffer
	ADD	DI,2		; go to next in partial-mix buffer
	MOV	AL,BL		; get back sample in AL
	DEC	AL		; decrement sample
	LOOPNZ	RS128A_ABOVE	; go again if more and not at baseline
	JCXZ	RS128A_DONE	; done if partial-mix buffer full
	JMP	SHORT RS128A_FILL ; ... otherwise fill with silence
        ;
        ; Current sample is below the baseline.  Increment the sample till
        ; it gets to 0 or until we have enough for this tick.
        ;
	EVEN
RS128A_BELOW:
	MOV	BL,AL		; save copy in BL
	IMUL	BH		; multiply by the volume
	STOSW			; store it in the partial-mix buffer
	ADD	DI,2		; go to next in partial-mix buffer
	MOV	AL,BL		; get back sample in AL
	INC	AL		; increment sample
	LOOPNZ	RS128A_BELOW	; go again if more and not at baseline
	JCXZ	RS128A_DONE	; done if partial-mix buffer full
	;
	; Current sample is at the baseline.  Since this is the first channel
	; on the left or right, we still need to fill with zeros in this case.
	;
RS128A_FILL:
	XOR	AX,AX
RS128A_FILLLOOP:
	STOSW
	ADD	DI,2
	LOOP	RS128A_FILLLOOP
	;
	; All done for now.
	;
RS128A_DONE:
	MOV	[BP].CHANLASTSAMP,AL	; save the last output sample
	RET
#ELSE
	; (8086 version)
	;
	; Patch in the volume.  In the 8086 version, we use the offset in
	; the precomputed multiplication table.  These patches need to be
	; done even if the sample has been played through, so do them here.
	; If there are more than 8 channels in the mod, the volume needs to
	; be adjusted downward to avoid overflowing 16 bits in the partial-
	; mix buffer.
	;
	MOV	AH,[BX].CHANMIXVOL	; "multiply" mixing volume by global
	MOV	AL,_globalvol		;   volume
	SHL	AX,1
	MOV	SI,AX
	MOV	AX,MULTBL[SI]
	SHL	AX,1			; "divide" by 64
	SHL	AX,1
	MOV	CL,BYTE PTR _nchannels
	CMP	CL,8
	JBE	RS128A_SETVOL
	SHR	AH,1
	CMP	CL,16
	JBE	RS128A_SETVOL
	SHR	AH,1
RS128A_SETVOL:
	XOR	AL,AL
	SHL	AX,1
	ADD	AX,OFFSET MULTBL
	MOV	WORD PTR CS:RS128A_VOLPT1,AX
	MOV	WORD PTR CS:RS128A_VOLPT2,AX
	MOV	WORD PTR CS:RS128A_VOLPT3,AX
	;
	; Check if the sample has played through.
	;
	MOV	CX,_samplespertick
	MOV	AL,[BX].CHANLASTSAMP
	LES	DI,DWORD PTR [BX].CHANPARTOFFS
	MOV	BP,BX
	CMP	[BX].CHANSAMPDONE,0
	JE	RS128A_DOPATCH
	JMP	RS128A_DORAMP
	;
	; The sample is not played through.  Patch the code (not enough
	; registers).  The limit is the offset from the start of the partial-
	; mix buffer of the end of the output for the tick on this channel.
	; DI is a negative number which added to that limit gives the offset
	; where the current sample goes.
	;
	EVEN
RS128A_DOPATCH:
	SHL	CX,1
	SHL	CX,1
	ADD	DI,CX
	MOV	WORD PTR CS:RS128A_LIMITPT1,DI
	MOV	WORD PTR CS:RS128A_LIMITPT2,DI
	NEG	CX
	MOV	DI,CX
	;
	; Patch in the sample length and loop length, leaving the loop
	; length in AX.  These are word quantities already.
	;
	MOV	SI,[BX].CHANSAMPPTR	; address sample data
	MOV	AX,[SI].SAMPLEN		; patch in sample length
	MOV	WORD PTR CS:RS128A_LENPT1,AX
	MOV	WORD PTR CS:RS128A_LENPT2,AX
	MOV	AX,[SI].SAMPLLEN	; patch in loop length
	MOV	WORD PTR CS:RS128A_LLENPT1,AX
	;
	; If the sample is looped, patch the code so we won't jump over the
	; part where we go back to the start of the loop.  Otherwise, patch
	; it so that we will.
	;
	MOV	WORD PTR CS:RS128A_SAMPEND,9090h
	OR	AX,AX			; loop length still in AX from above
	JNZ	RS128A_LOOPED
	MOV	WORD PTR CS:RS128A_SAMPEND,RS128A_JUMPINSTR
	;
	; Now set up register values.  We have:
	;
	;    ES:DI+limit   addresses buffer where samples go (already set)
	;    DS            current segment in sample
	;    SI.AX         current word offset in sample
	;    BP.DX         sample step in words
	;    CX            8000h if 64k or more past sample start, 0 otherwise
	;
RS128A_LOOPED:
	PUSH	BX			; save pointer to channel data
	XOR	CX,CX			; assume first 64k
	MOV	AX,[BX].CHANSAMPSEG	; get segment in AX
	CMP	[SI].SAMPEMSFLAG,1	; is the sample in EMS?
	JE	RS128A_INEMS
	MOV	BYTE PTR CS:RS128A_SEGJMP,RS128A_SEGJMP1
	CMP	[SI].SAMPSEG,AX		; set CF if 2nd 64k
	JMP	SHORT RS128A_SETCX
	EVEN
RS128A_INEMS:
	MOV	BYTE PTR CS:RS128A_SEGJMP,RS128A_SEGJMP2
	MOV	SI,[SI].SAMPEMSHANDLE	; get EMS handle and patch into code
	MOV	WORD PTR CS:RS128A_EMSHANDLE,SI
	CMP	[BX].CHANSAMPHALF,1	; set CF if in upper 64k
	CMC
RS128A_SETCX:				; CF set if 2nd 64k
	RCR	CX,1
	MOV	BP,[BX].CHANSTEPINT	; BP = integer part of sample step
	MOV	DX,[BX].CHANSTEPFRAC	; DX = fraction of sample step
	SHR	BP,1			; convert to words
	RCR	DX,1
	MOV	SI,[BX].CHANSAMPOFFS	; SI = integer part of byte offset
	MOV	BX,[BX].CHANSAMPFRAC	; BX = fraction of byte offset
	SHR	SI,1			; convert offset to words
	RCR	BX,1
	MOV	DS,AX			; DS = current segment
	MOV	AX,BX			; AX = fraction of word offset
	EVEN
RS128A_MAINLOOP:
	SHL	AX,1
	RCL	SI,1
	MOV	BL,[SI]
	SHR	SI,1
	RCR	AX,1
	XOR	BH,BH
	SHL	BX,1
	DB	36h,8Bh,9Fh		; MOV BX,SS:[BX+immed]
RS128A_VOLPT1:
	DW	0
	DB	26h,89h,9Dh		; MOV ES:[DI+immed],BX
RS128A_LIMITPT1:
	DW	0
	OR	SI,CX
	ADD	AX,DX
	ADC	SI,BP
	JC	RS128A_SAMPEND
	DB	81h,0FEh		; CMP SI,immed SAMPLEN
RS128A_LENPT1:
	DW	0
	JAE	RS128A_SAMPEND		; (JAE = JNC)
RS128A_SEGTEST:
	XOR	SI,CX
	DB	78h			; JS segchange_code
RS128A_SEGJMP:
	DB	0
RS128A_MAINLPEND:
	ADD	DI,4
	JNZ	RS128A_MAINLOOP
	;
	; We got all the samples needed.  Save back the current segment
	; and offset and the "over 64k" flag.  There is at least one more
	; sample to be output on this channel, so we're not at the end yet.
	;
	POP	BP		; get back pointer to channel data
	SHL	AX,1		; SI.AX is position in bytes relative to
	RCL	SI,1		;   current segment
	MOV	[BP].CHANSAMPFRAC,AX
	MOV	[BP].CHANSAMPOFFS,SI
	MOV	[BP].CHANSAMPSEG,DS
	ROL	CH,1
	MOV	[BP].CHANSAMPHALF,CH
	RET
	;
	; Different 64k segment now.  Adjust DS, CX, and SI.  The most
	; significant bit of SI is set when we get here and needs to be
	; cleared in any case.  If CX is 0, we're going from the lower 64k
	; to the upper 64k, and we need to set DS to SAMPSEG+1000h and CX
	; to 8000h.  If CX is 8000h, we're going from the upper 64k to the
	; lower 64k (a looped sample longer than 64k, and we just looped).
	; In that case, we need to set DS to SAMPSEG and CX to 0.  This is
	; for a sample loaded in conventional RAM.
	;
	EVEN
RS128A_SEGJMP1		EQU	$ - (OFFSET RS128A_MAINLPEND)
	MOV	BX,8000h
	XOR	SI,BX			; clear MSB of SI
	XOR	CX,BX			; invert MSB of CX
	MOV	BX,DS			; DS value in BX for adjustment
	JZ	RS128A_SEGLOW		; if CX = 0, CX was 8000h, so low 64k
	ADD	BH,10h			; low to high, add 1000h to DS
	MOV	DS,BX
	JMP	SHORT RS128A_MAINLPEND
	EVEN
RS128A_SEGLOW:
	SUB	BH,10h			; high to low, subtract 1000h from DS
	MOV	DS,BX
	JMP	SHORT RS128A_MAINLPEND
	;
	; Different 64k segment now.  Adjust CX and SI, and map in the
	; correct half of the EMS sample.  The most significant bit of SI
	; is set when we get here and needs to be cleared in any case.  If
	; CX is 0, we're going from the lower 64k to the upper 64k, and we
	; need to set CX to 8000h and map in the upper 64k.  If CX is 8000h,
	; we're going from the upper 64k to the lower 64k (a looped sample
	; longer than 64k, and we just looped).  In that case, we need to
	; set CX to 0 and map in the lower 64k.  This is for a sample loaded
	; in EMS RAM.
	;
	EVEN
RS128A_SEGJMP2		EQU	$ - (OFFSET RS128A_MAINLPEND)
	MOV	BX,8000h
	XOR	SI,BX			; clear MSB of SI
	XOR	CX,BX			; invert MSB of CX
	DB	0BBh			; MOV BX,immed SAMPEMSHANDLE
RS128A_EMSHANDLE:
	DW	0
	PUSH	BX
	JZ	RS128A_SEGLOW2		; if CX = 0, CX was 8000h, so low 64k
	CALL	MAPEMS2ND		; low to high, map in the second 64k
	JMP	SHORT RS128A_MAINLPEND
	EVEN
RS128A_SEGLOW2:
	CALL	MAPEMS1ST		; high to low, map in the first 64k
	JMP	SHORT RS128A_MAINLPEND
	;
	; We got to the end of the sample.  CF:SI.AX is the current offset
	; in words from the start of the sample.  If the sample is looped,
	; go back to the beginning of the loop.
	;
	EVEN
RS128A_SAMPEND:
	DB	0EBh		; JMP SHORT below (or NOP, NOP)
	DB	RS128A_JUMPSIZE
	MOV	BL,0		; LSB of BL = CF
	RCL	BL,1		; BL:SI = integer of offset in words
RS128A_ADJLOOP:
	DB	81h,0EEh	; SUB SI,immed SAMPLLEN
RS128A_LLENPT1:
	DW	0
	SBB	BL,0
        JNZ     RS128A_ADJLOOP
	DB	81h,0FEh	; CMP SI,immed SAMPLEN
RS128A_LENPT2:
	DW	0
        JAE     RS128A_ADJLOOP
	JMP	SHORT RS128A_SEGTEST
	;
	; We got to the end of the sample, and it is not looped.  CF:SI.AX
	; is the current offset in words from the start of the sample.  BP.DX
	; is the sample step in words.
	;
	EVEN
RS128A_JUMPSIZE		EQU	$ - (OFFSET RS128A_SAMPEND + 2)
RS128A_JUMPINSTR	EQU	(RS128A_JUMPSIZE*256) + 0EBh
	SUB	AX,DX			; back up to last sample
	SBB	SI,BP			; guaranteed to get back under 128k
	POP	BP			; BP addresses channel data
	MOV	AL,[SI]			; get back sample
	MOV	[BP].CHANSAMPDONE,1	; mark the sample finished
	ADD	DI,4			; DI didn't get incremented last time
	JZ	RS128A_DONE		; are we done?
	MOV	CX,DI			; CX = number of samples left to do
	NEG	CX
	SHR	CX,1
	SHR	CX,1
	DB	81h,0C7h		; ADD DI,immed (make DI a real offset)
RS128A_LIMITPT2:
	DW	0
	MOV	BX,DGROUP		; DS addresses local data again
	MOV	DS,BX
	;
	; Code for ramping to the baseline.  When we get here, the sample is
	; not looped and has been played through.  CX is the number of output
	; samples left to generate this tick.  ES:DI addresses the buffer
	; where output samples go.  AL is the next input sample, moving
	; toward the baseline.  BP addresses the channel data structure.
	; DS addresses the default data segment.
	;
RS128A_DORAMP:
	OR	AL,AL			; see whether we're ramping up or down
	JZ	RS128A_FILL		; no ramping if already at baseline
	JS	RS128A_BELOW		; ramp up if negative
        ;
        ; Current sample is above the baseline.  Decrement the sample till
        ; it gets to 0 or until we have enough for this tick.
        ;
RS128A_ABOVE:
	MOV	BL,AL		; get sample in BL
	XOR	BH,BH		; convert to word offset
	SHL	BX,1
	DB	8Bh,9Fh		; MOV BX,[BX+immed] (multiply by volume)
RS128A_VOLPT2:
	DW	0
	MOV	ES:[DI],BX	; store it in the partial-mix buffer
	ADD	DI,4		; go to next in partial-mix buffer
	DEC	AL		; decrement sample
	LOOPNZ	RS128A_ABOVE	; go again if more and not at baseline
	JCXZ	RS128A_DONE	; done if partial-mix buffer full
	JMP	SHORT RS128A_FILL ; ... otherwise fill with silence
        ;
        ; Current sample is below the baseline.  Increment the sample till
        ; it gets to 0 or until we have enough for this tick.
        ;
	EVEN
RS128A_BELOW:
	MOV	BL,AL		; save copy in BL
	XOR	BH,BH		; convert to word offset
	SHL	BX,1
	DB	8Bh,9Fh		; MOV BX,[BX+immed] (multiply by volume)
RS128A_VOLPT3:
	DW	0
	MOV	ES:[DI],BX	; store it in the partial-mix buffer
	ADD	DI,4		; go to next in partial-mix buffer
	INC	AL		; increment sample
	LOOPNZ	RS128A_BELOW	; go again if more and not at baseline
	JCXZ	RS128A_DONE	; done if partial-mix buffer full
	;
	; Current sample is at the baseline.  Since this is the first channel
	; on the left or right, we still need to fill with zeros in this case.
	;
RS128A_FILL:
	XOR	AX,AX
RS128A_FILLLOOP:
	STOSW
	ADD	DI,2
	LOOP	RS128A_FILLLOOP
	;
	; All done for now.
	;
RS128A_DONE:
	MOV	[BP].CHANLASTSAMP,AL	; save the last output sample
	RET
#ENDIF
;
; RESAMPLEA routine, resamples the "first" channel on the left or right
; (i.e., it uses MOV rather than ADD to put samples in the partial-mix
; buffer).  This is a "shell" routine that selects the fastest available
; subroutine to resample the channel.  If the sample is in EMS, this
; routine maps in the current 64k of it.
;
	EVEN
RESAMPLEA:
	MOV	SI,[BX].CHANSAMPPTR	; SI addresses sample data
	CMP	[SI].SAMPLEN,0		; is the sample less than 64k?
	JS	RESAMPLEA_OVER64
	;
	; Sample less than 64k.  See if the sample is already played through.
	; The normal resampler handles that as a special case.
	;
	CMP	[BX].CHANSAMPDONE,1	; is the sample played through?
	JE	RESAMPLEA_64NORMAL
	;
	; If the sample has zero volume, we have a special resampler for
	; that.
	;
	CMP	[BX].CHANMIXVOL,0
	JE	RESAMPLEA_ZEROVOL
	;
	; Sample has nonzero volume and hasn't been played through.  Map
	; it in if it's EMS.
	;
	CMP	[SI].SAMPEMSFLAG,0
	JE	RESAMPLEA_64CHKEND
	PUSH	[SI].SAMPEMSHANDLE
	CALL	MAPEMS1ST
	;
	; See if we will get to the end of it this time.
	;
RESAMPLEA_64CHKEND:
	MOV	AX,[BX].CHANSTEPFRAC	; multiply sample step by # samples
	MUL	_samplespertick
	MOV	CX,AX			; (accumulate in DI.CX)
	MOV	DI,DX
	MOV	AX,[BX].CHANSTEPINT
	MUL	_samplespertick
	JC	RESAMPLEA_64NORMAL
	ADD	DI,AX
	JC	RESAMPLEA_64NORMAL
	ADD	CX,[BX].CHANSAMPFRAC	; add in current position in sample
	ADC	DI,[BX].CHANSAMPOFFS
	JC	RESAMPLEA_64NORMAL
	MOV	AX,[SI].SAMPLEN		; AX is length of sample in bytes
	SHL	AX,1
	CMP	DI,AX
	JAE	RESAMPLEA_64NORMAL
	JMP	RS64AN
	EVEN
RESAMPLEA_64NORMAL:
	JMP	RS64A
	EVEN
RESAMPLEA_ZEROVOL:
	JMP	RSAZ
	;
	; Sample 64k or longer.  We won't make any special resamplers for
	; this rare case, but maybe we can use a 64k resampler.  First
	; check whether the sample has already played through.  The normal
	; resampler handles that as a special case.
	;
	EVEN
RESAMPLEA_OVER64:
	CMP	[BX].CHANSAMPDONE,1	; is the sample played through?
	JE	RESAMPLEA_128NORMAL
	;
	; If the sample has zero volume, we have a special resampler for
	; that.
	;
	CMP	[BX].CHANMIXVOL,0
	JE	RESAMPLEA_ZEROVOL
	;
	; Sample has nonzero volume and has not played through.  Map it in
	; if it's in EMS.
	;
	CMP	[SI].SAMPEMSFLAG,0
	JE	RESAMPLEA_128CHKEND
	PUSH	[SI].SAMPEMSHANDLE
	CMP	[BX].CHANSAMPHALF,1
	JE	RESAMPLEA_2NDHALF
	CALL	MAPEMS1ST
	JMP	SHORT RESAMPLEA_GETSPOT
	EVEN
RESAMPLEA_2NDHALF:
	CALL	MAPEMS2ND
	JMP	SHORT RESAMPLEA_128NORMAL	; already past 64k
	;
	; We can use a 64k resampler if we're guaranteed to stay within 64k
	; of the sample start, so try that.
	;
	EVEN
RESAMPLEA_128CHKEND:
	MOV	AX,[SI].SAMPSEG		; if already above 64k, forget it
	CMP	AX,[BX].CHANSAMPSEG
	JB	RESAMPLEA_128NORMAL
RESAMPLEA_GETSPOT:
	MOV	AX,[BX].CHANSTEPFRAC	; multiply sample step by # samples
	MUL	_samplespertick
	MOV	CX,AX			; (accumulate in DI.CX)
	MOV	DI,DX
	MOV	AX,[BX].CHANSTEPINT
	MUL	_samplespertick
	JC	RESAMPLEA_128NORMAL
	ADD	DI,AX
	JC	RESAMPLEA_128NORMAL
	ADD	CX,[BX].CHANSAMPFRAC	; add in current position in sample
	ADC	DI,[BX].CHANSAMPOFFS
	JC	RESAMPLEA_128NORMAL
	JMP	RS64AN
	EVEN
RESAMPLEA_128NORMAL:
	JMP	RS128A
