; MEMMGR.ASM
;
; This assembler module contains routines for allocating and deallocating
; blocks of memory.  All blocks are paragraph-aligned.  The DMA buffer
; must be allocated before any other block.
;

;
; Globally-visible routines and variables.
;
PUBLIC	_emsframeseg	; EMS page frame segment
PUBLIC	_emsenable	; 1 if EMS usage enabled
PUBLIC	_initmem	; initialization routine for this module
PUBLIC	_allocdma	; allocates a DMA buffer
PUBLIC	_getblock	; gets a block of memory
PUBLIC	_freeblock	; releases a block of memory
PUBLIC	_movedown	; moves a block as low in memory as possible
PUBLIC	_checkmem	; checks memory integrity
PUBLIC	_endmem		; finalization routine for this module
PUBLIC	_getemsblock	; gets a block of expanded memory
PUBLIC	_freeemsblock	; releases a block of expanded memory
PUBLIC	MAPEMS1ST	; maps in the 1st 64k of an EMS block
PUBLIC	MAPEMS2ND	; maps in the 2nd 64k of an EMS block

;
; Establish the segment ordering.  Code comes first.  This is the C
; standard segment ordering for Small model.
;
_TEXT	SEGMENT WORD PUBLIC 'CODE'
_TEXT	ENDS

_DATA	SEGMENT PARA PUBLIC 'DATA'
_DATA	ENDS

CONST	SEGMENT WORD PUBLIC 'CONST'
CONST	ENDS

_BSS	SEGMENT WORD PUBLIC 'BSS'
_BSS	ENDS

STACK	SEGMENT PARA STACK 'STACK'
STACK	ENDS

DGROUP	GROUP	_DATA,CONST,_BSS,STACK

_DATA	SEGMENT PARA PUBLIC 'DATA'
		;
		; Segment address of memory we allocate from DOS.  We get
		; the largest available block.
		;
		EVEN
OURSEG		DW	0FFFFh
		;
		; Segment address of the end of the memory we manage.  If
		; OURSEG >= ENDOURS, we haven't allocated any memory from
		; DOS.
		;
		EVEN
ENDOURS		DW	0
		;
		; Segment address of the first free memory block, i.e., the
		; start of the free list.  The free list consists of a linked
		; list of blocks with records in them like this:
		;
		;	NEXTBLOCK	word	segment of next free block
		;				  (0 if last)
		;	BLOCKSIZE	word	size of this free block
		;
		; Note that we don't keep track of allocated blocks at all.
		; Instead, we require the caller of _freeblock to specify
		; the size of the block when deallocating it.  That way, we
		; don't waste 16 bytes of RAM at the beginning of each block
		; on a header.
		;
		EVEN
FREESTART	DW	0
BLOCKSIZE	EQU	WORD PTR [0]
NEXTBLOCK	EQU	WORD PTR [2]
		;
		; Expanded memory manager device name.
		;
EMSNAME		DB	"EMMXXXX0",0
		;
		; Flag, 1 if EMS is present in the system.
		;
HAVEEMS		DB	0
		;
		; EMS enable flag:  1 if EMS usage enabled.
		;
_emsenable	DB	1
		;
		; Expanded memory manager version number.
		;
EMSVERSION	DB	0
		;
		; Segment address of the EMS page frame.
		;
		EVEN
_emsframeseg	DW	0
		;
		; Table of EMS handles allocated by this package.
		;
		EVEN
MAXHANDLES	EQU	99
EMSHANDLES	DW	MAXHANDLES DUP (0)
		;
		; Number of logical pages for each EMS handle.
		;
		EVEN
EMSSIZES	DW	MAXHANDLES DUP (0)
		;
		; Miserable hack:  the expanded memory manager I wrote for
		; the Micro Mainframe 5150T EMS card can't map logical pages
		; fast enough for Tantrakr.  If I detect my EMM, I'm going
		; to use the Set Page Map function instead.
		;
MYEMM		DB	1	; 1 if my expanded memory manager is present
		;
		; Structure array for EMM Map Multiple Pages function.
		;
		EVEN
MULTISTRUCS	DW	0,0,0,1,0,2,0,3
_DATA	ENDS

HACKDATA SEGMENT
		;
		; Page map save areas for my miserable EMM hack.
		;
PAGEMAPS	DB	MAXHANDLES DUP (32 DUP (0))
HACKDATA ENDS

_TEXT	SEGMENT WORD PUBLIC 'CODE'
;
; DETECTEMS routine, to detect the presence of an expanded memory manager,
; saving the EMS page frame address.  Takes no parameters.
;
; Returns nothing.  Destroys nothing.
;
	EVEN
DETECTEMS:
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	DX
	PUSH	DI
	PUSH	ES
	MOV	MYEMM,1			; assume my driver
	MOV	DX,OFFSET EMSNAME
	MOV	AX,3D02h		; (open read/write)
	INT	21h			; modifies AX
	JNC	DETECTEMS_DEVICEOPEN	; "No EMS" if can't open device
	JMP	DETECTEMS_NOEMS
	EVEN
DETECTEMS_DEVICEOPEN:
	MOV	BX,AX			; Get Device Data
	MOV	AX,4400h
	INT	21h			; modifies AX, DX
	JC	DETECTEMS_NOTDEVICE
	TEST	DL,80h
	JNZ	DETECTEMS_ISDEVICE
DETECTEMS_NOTDEVICE:
	MOV	AX,3E00h	; If not device, close file and "No EMS"
	INT	21h			; modifies AX
	JMP	DETECTEMS_NOEMS
	EVEN
DETECTEMS_ISDEVICE:
	MOV	AX,4407h		; Check if device is ready
	INT	21h			; modifies AX
	JC	DETECTEMS_NOTDEVICE	; "No EMS" if not ready
	CMP	AL,0FFh
	JNE	DETECTEMS_NOTDEVICE
	MOV	AH,40h			; write one byte to device
	MOV	CX,1
	INT	21h			; modifies AX
	JC	DETECTEMS_NOTMINE
	CMP	AX,CX
	JNE	DETECTEMS_NOTMINE
	PUSH	DS
	MOV	AX,SEG HACKDATA
	MOV	DS,AX
	MOV	ES,AX
	MOV	AH,3Fh			; read 10 bytes from device
	MOV	CX,10
	MOV	DX,OFFSET PAGEMAPS
	INT	21h			; modifies AX
	POP	DS
	JC	DETECTEMS_NOTMINE
	CMP	AX,CX
	JNE	DETECTEMS_NOTMINE
	MOV	DI,DX			; check last 2 bytes read
	CMP	BYTE PTR ES:[DI+8],'B'
	JNE	DETECTEMS_NOTMINE
	CMP	BYTE PTR ES:[DI+9],'W'
	JE	DETECTEMS_CLOSEDEV
DETECTEMS_NOTMINE:
	MOV	MYEMM,0
DETECTEMS_CLOSEDEV:
	MOV	AH,3Eh			; Close device handle
	INT	21h			; modifies AX
	MOV	AH,40h			; Get EMM status
	INT	67h			; modifies AX
	OR	AH,AH
	JNZ	DETECTEMS_NOEMS		; "No EMS" if not working
	MOV	AH,41h			; EMM working - get frame address
	INT	67h			; modifies AX, BX
	OR	AH,AH
	JNZ	DETECTEMS_NOEMS
	MOV	_emsframeseg,BX
	MOV	AH,46h			; get EMM version
	INT	67h			; modifies AX
	OR	AH,AH
	JNZ	DETECTEMS_NOEMS
	MOV	EMSVERSION,AL
	CMP	AL,41h			; version 4.1?
	JE	DETECTEMS_SETFLAG
	MOV	MYEMM,0
DETECTEMS_SETFLAG:
	MOV	HAVEEMS,1
	MOV	DI,OFFSET EMSHANDLES	; clear list of allocated handles
	MOV	CX,MAXHANDLES
	MOV	AX,DS
	MOV	ES,AX
	XOR	AX,AX
	CLD
	REP	STOSW
	MOV	DI,OFFSET EMSSIZES	; and set all sizes to 0
	MOV	CX,MAXHANDLES
	REP	STOSW
	JMP	SHORT DETECTEMS_DONE
	EVEN
DETECTEMS_NOEMS:
	MOV	HAVEEMS,0
	MOV	MYEMM,0
DETECTEMS_DONE:
	POP	ES
	POP	DI
	POP	DX
	POP	CX
	POP	BX
	POP	AX
	RET
;
; FREEALLEMS routine, to release all allocated expanded memory.  Takes no
; parameters.
;
; Returns nothing.  Destroys nothing.
;
	EVEN
FREEALLEMS:
	PUSH	AX
	PUSH	CX
	PUSH	DX
	PUSH	SI
	CMP	HAVEEMS,0		; if no EMS, nothing to deallocate
	JE	FREEALLEMS_DONE
	MOV	SI,OFFSET EMSHANDLES	; release all nonzero EMS handles
	MOV	CX,MAXHANDLES
	CLD
FREEALLEMS_LOOP:
	LODSW
	OR	AX,AX
	JZ	FREEALLEMS_LOOPEND
	MOV	DX,AX
	MOV	AH,45h
	INT	67h			; modifies AX
	MOV	WORD PTR [SI-2],0	; mark not allocated
FREEALLEMS_LOOPEND:
	LOOP	FREEALLEMS_LOOP
FREEALLEMS_DONE:
	POP	SI
	POP	DX
	POP	CX
	POP	AX
	RET
;
; _getemsblock routine, to allocate a block of expanded memory.  Takes one
; parameter, the size of the desired block in paragraphs.  Uses C calling
; conventions.
;
; Returns the EMS handle of the allocated block in AX if successful, 0 if
; not.  Destroys AX, BX, CX, DX, ES.
;
_getemsblock_NPARAS EQU	WORD PTR [BP+4]
	EVEN
_getemsblock:
	PUSH	BP
	MOV	BP,SP
	PUSH	SI			; save SI, DI for C
	PUSH	DI
	MOV	AL,HAVEEMS		; if no EMS, can't allocate
	AND	AL,_emsenable		; ... nor if EMS usage disabled
	JZ	_getemsblock_FAILED
	MOV	AX,DS			; find a free entry in the handle array
	MOV	ES,AX
	MOV	DI,OFFSET EMSHANDLES
	MOV	CX,MAXHANDLES
	XOR	AX,AX
	CLD
	REPNE	SCASW
	JNE	_getemsblock_FAILED	; failed if none
	SUB	DI,2			; DI addresses entry in handle array
	MOV	SI,DI			; SI addresses entry in sizes array
	ADD	SI,OFFSET EMSSIZES - OFFSET EMSHANDLES
	MOV	BX,_getemsblock_NPARAS	; convert # of paragraphs to 16k pages
	OR	BX,BX			; allocating 0 paragraphs is invalid
	JZ	_getemsblock_FAILED
	ADD	BX,1023			; (round up)
	RCR	BX,1			; (divide by 1024)
	SHR	BX,1
	MOV	BL,BH
	XOR	BH,BH
	MOV	AH,43h
	INT	67h			; modifies AX, DX
	OR	AH,AH			; see if call succeeded
	JNZ	_getemsblock_FAILED
	MOV	[DI],DX			; save handle
	MOV	[SI],BX			; ... and number of pages allocated
	CMP	MYEMM,1			; if my EMS driver, map in the pages
	JNE	_getemsblock_NOTMINE	;   and save the page maps
	MOV	MYEMM,0
	PUSH	DX
	CALL	MAPEMS1ST		; modifies nothing
	MOV	DI,DX
	MOV	CL,5
	SHL	DI,CL
	ADD	DI,OFFSET PAGEMAPS
	PUSH	ES
	MOV	AX,SEG HACKDATA
	MOV	ES,AX
	MOV	AX,4E00h
	INT	67h			; modifies AX
	PUSH	DX
	CALL	MAPEMS2ND		; modifies nothing
	ADD	DI,16
	MOV	AX,4E00h
	INT	67h			; modifies AX
	POP	ES
	MOV	MYEMM,1
_getemsblock_NOTMINE:
	MOV	AX,DX			; return handle to caller
	JMP	SHORT _getemsblock_DONE
	EVEN
_getemsblock_FAILED:
	XOR	AX,AX
_getemsblock_DONE:
	POP	DI			; restore DI, SI
	POP	SI
	POP	BP
	RET
;
; _freeemsblock routine, to release a block of expanded memory.  Takes one
; parameter, the handle of the block to release.  Uses C calling conventions.
;
; Returns 0 in AX if successful, -1 if not.  Modifies AX, CX, DX, ES.
;
_freeemsblock_HANDLE	EQU	WORD PTR [BP+4]
	EVEN
_freeemsblock:
	PUSH	BP
	MOV	BP,SP
	PUSH	SI			; save SI, DI for C
	PUSH	DI
	CMP	HAVEEMS,0		; if no EMS, can't release
	JE	_freeemsblock_FAILED
	MOV	AX,DS			; find the block in the handle array
	MOV	ES,AX
	MOV	DI,OFFSET EMSHANDLES
	MOV	CX,MAXHANDLES
	MOV	AX,_freeemsblock_HANDLE
	OR	AX,AX			; invalid if handle is 0
	JZ	_freeemsblock_FAILED
	CLD
	REPNE	SCASW
	JNE	_freeemsblock_FAILED	; failed if not found
	SUB	DI,2			; DI addresses entry in handle array
	MOV	SI,DI			; SI addresses entry in sizes array
	ADD	SI,OFFSET EMSSIZES - OFFSET EMSHANDLES
	MOV	DX,AX			; release the handle
	MOV	AH,45h
	INT	67h			; modifies AX
	OR	AH,AH			; see if call succeeded
	JNZ	_freeemsblock_FAILED
	XOR	AX,AX			; mark array entries unused
	MOV	[DI],AX
	MOV	[SI],AX
	JMP	SHORT _freeemsblock_DONE ; (return success)
	EVEN
_freeemsblock_FAILED:
	MOV	AX,-1
_freeemsblock_DONE:
	POP	DI			; restore DI, SI
	POP	SI
	POP	BP
	RET
;
; MAPEMS1ST routine, to map in the first 64k of a block of expanded memory.
; This routine may be called when DS does not address the default data
; segment and may not modify any registers.  It may assume that DF is clear
; for incrementing.  Takes one parameter, the handle of the EMS block.  Uses
; Pascal calling conventions.
;
; Returns nothing.  Destroys nothing.
;
#IF M_I286
MAPEMS1ST_HANDLE	EQU	WORD PTR [BP+18]
#ELSE
MAPEMS1ST_HANDLE	EQU	WORD PTR [BP+4]
#ENDIF
	EVEN
MAPEMS1ST:
#IF M_I286
	PUSHA
	MOV	BP,SP
#ELSE
	PUSH	BP
	MOV	BP,SP
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	DX
	PUSH	SI
	PUSH	DI
#ENDIF
	PUSH	DS
	PUSH	ES
	MOV	AX,DGROUP		; get default data segment in DS
	MOV	DS,AX
	MOV	ES,AX			; ... and in ES
	CMP	HAVEEMS,0		; if no EMS, skip it (invalid)
	JE	MAPEMS1ST_DONE
	MOV	DI,OFFSET EMSHANDLES	; find the block in the handle array
	MOV	CX,MAXHANDLES
	MOV	AX,MAPEMS1ST_HANDLE
	OR	AX,AX			; invalid if handle is 0
	JZ	MAPEMS1ST_DONE
	REPNE	SCASW
	JNE	MAPEMS1ST_DONE		; skip it if not found
	CMP	MYEMM,1			; if my EMM, use Set Page Map
	JE	MAPEMS1ST_MYEMM
					; DI addresses entry in sizes array
	ADD	DI,(OFFSET EMSSIZES - OFFSET EMSHANDLES) - 2
	MOV	DX,AX			; DX is EMS handle
	XOR	BX,BX			; start with logical page 0
	MOV	CX,[DI]			; CX is number of pages to map
	CMP	CX,4			; (don't map more than 4 pages)
	JBE	MAPEMS1ST_DOMAP
	MOV	CX,4
	;
	; Here we have DX = handle; CX = number of logical pages to map,
	; starting with 0; BX = first logical page to map (0).
	;
MAPEMS1ST_DOMAP:
	CMP	EMSVERSION,40h		; EMM version 4.0 or higher?
	JAE	MAPEMS1ST_MAPMULTI	; (use Map Multiple function if so)
	;
	; Expanded memory manager is LIM 3.x.  Map pages one at a time.
	;
	MOV	AX,4400h		; start with physical page 0
MAPEMS1ST_MAPLOOP:
	MOV	DI,AX			; save AX in DI
	INT	67h			; map the page (modifies AX)
	OR	AH,AH			; did the call succeed?
	JNZ	MAPEMS1ST_DONE
	MOV	AX,DI			; get AX back
	INC	AL			; go to next physical page
	INC	BX			; go to next logical page
	LOOP	MAPEMS1ST_MAPLOOP
	JMP	SHORT MAPEMS1ST_DONE
	;
	; Special case (read "hack"):  my expanded memory manager.
	;
	EVEN
MAPEMS1ST_MYEMM:
	MOV	SI,AX
#IF M_I286
	SHL	SI,5
#ELSE
	SHL	SI,1
	SHL	SI,1
	SHL	SI,1
	SHL	SI,1
	SHL	SI,1
#ENDIF
	ADD	SI,OFFSET PAGEMAPS
	PUSH	DS
	MOV	AX,SEG HACKDATA
	MOV	DS,AX
	MOV	AX,4E01h
	INT	67h
	POP	DS
	JMP	SHORT MAPEMS1ST_DONE
	;
	; Expanded memory manager is LIM 4.x.  Map multiple pages in one call.
	;
	EVEN
MAPEMS1ST_MAPMULTI:
	MOV	SI,OFFSET MULTISTRUCS	; DS:SI, ES:DI -> map structure table
	MOV	DI,SI
	MOV	AX,BX			; AX = logical page number
	MOV	BX,CX			; BX = number of pages to map
MAPEMS1ST_MULTILP:
	STOSW				; fill in logical page numbers to map
	INC	AX
	ADD	DI,2
	LOOP	MAPEMS1ST_MULTILP
	MOV	CX,BX			; get back number of pages to map
	MOV	AX,5000h		; Map Multiple Handle Pages
	INT	67h			; modifies AX (ignore errors)
MAPEMS1ST_DONE:
	POP	ES
	POP	DS
#IF M_I286
	POPA
#ELSE
	POP	DI
	POP	SI
	POP	DX
	POP	CX
	POP	BX
	POP	AX
	POP	BP
#ENDIF
	RET	2
;
; MAPEMS2ND routine, to map in the second 64k of a block of expanded memory.
; This routine may be called when DS does not address the default data
; segment and may not modify any registers.  It may assume that DF is clear
; for incrementing.  Takes one parameter, the handle of the EMS block.  Uses
; Pascal calling conventions.
;
; Returns nothing.  Destroys nothing.
;
#IF M_I286
MAPEMS2ND_HANDLE	EQU	WORD PTR [BP+18]
#ELSE
MAPEMS2ND_HANDLE	EQU	WORD PTR [BP+4]
#ENDIF
	EVEN
MAPEMS2ND:
#IF M_I286
	PUSHA
	MOV	BP,SP
#ELSE
	PUSH	BP
	MOV	BP,SP
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	DX
	PUSH	SI
	PUSH	DI
#ENDIF
	PUSH	DS
	PUSH	ES
	MOV	AX,DGROUP		; get default data segment in DS
	MOV	DS,AX
	MOV	ES,AX			; ... and in ES
	CMP	HAVEEMS,0		; if no EMS, skip it (invalid)
	JE	MAPEMS2ND_DONE
	MOV	DI,OFFSET EMSHANDLES	; find the block in the handle array
	MOV	CX,MAXHANDLES
	MOV	AX,MAPEMS2ND_HANDLE
	OR	AX,AX			; invalid if handle is 0
	JZ	MAPEMS2ND_DONE
	REPNE	SCASW
	JNE	MAPEMS2ND_DONE		; skip it if not found
	CMP	MYEMM,1			; if my EMM, use Set Page Map
	JE	MAPEMS2ND_MYEMM
					; DI addresses entry in sizes array
	ADD	DI,(OFFSET EMSSIZES - OFFSET EMSHANDLES) - 2
	MOV	DX,AX			; DX is EMS handle
	MOV	BX,4			; start with logical page 4
	MOV	CX,[DI]			; CX is number of pages to map
	SUB	CX,BX
	JBE	MAPEMS2ND_DONE		; if less than 5 pages, invalid
	CMP	CX,BX			; (don't map more than 4 pages)
	JBE	MAPEMS2ND_DOMAP
	MOV	CX,BX
	;
	; Here we have DX = handle; CX = number of logical pages to map,
	; starting with 4; BX = first logical page to map (4).
	;
MAPEMS2ND_DOMAP:
	CMP	EMSVERSION,40h		; EMM version 4.0 or higher?
	JAE	MAPEMS2ND_MAPMULTI	; (use Map Multiple function if so)
	;
	; Expanded memory manager is LIM 3.x.  Map pages one at a time.
	;
	MOV	AX,4400h		; start with physical page 0
MAPEMS2ND_MAPLOOP:
	MOV	DI,AX			; save AX in DI
	INT	67h			; map the page (modifies AX)
	OR	AH,AH			; did the call succeed?
	JNZ	MAPEMS2ND_DONE
	MOV	AX,DI			; get AX back
	INC	AL			; go to next physical page
	INC	BX			; go to next logical page
	LOOP	MAPEMS2ND_MAPLOOP
	JMP	SHORT MAPEMS2ND_DONE
	;
	; Special case (read "hack"):  my expanded memory manager.
	;
	EVEN
MAPEMS2ND_MYEMM:
	MOV	SI,AX
#IF M_I286
	SHL	SI,5
#ELSE
	SHL	SI,1
	SHL	SI,1
	SHL	SI,1
	SHL	SI,1
	SHL	SI,1
#ENDIF
	ADD	SI,OFFSET PAGEMAPS+16
	PUSH	DS
	MOV	AX,SEG HACKDATA
	MOV	DS,AX
	MOV	AX,4E01h
	INT	67h
	POP	DS
	JMP	SHORT MAPEMS2ND_DONE
	;
	; Expanded memory manager is LIM 4.x.  Map multiple pages in one call.
	;
	EVEN
MAPEMS2ND_MAPMULTI:
	MOV	SI,OFFSET MULTISTRUCS	; DS:SI, ES:DI -> map structure table
	MOV	DI,SI
	MOV	AX,BX			; AX = logical page number
	MOV	BX,CX			; BX = number of pages to map
MAPEMS2ND_MULTILP:
	STOSW				; fill in logical page numbers to map
	INC	AX
	ADD	DI,2
	LOOP	MAPEMS2ND_MULTILP
	MOV	CX,BX			; get back number of pages to map
	MOV	AX,5000h		; Map Multiple Handle Pages
	INT	67h			; modifies AX (ignore errors)
MAPEMS2ND_DONE:
	POP	ES
	POP	DS
#IF M_I286
	POPA
#ELSE
	POP	DI
	POP	SI
	POP	DX
	POP	CX
	POP	BX
	POP	AX
	POP	BP
#ENDIF
	RET	2
;
; _initmem routine, to initialize the memory manager.  Uses C calling
; conventions.  Takes no parameters.
;
; Returns 0 in AX if successful, -1 if not.  Destroys BX, ES.
;
	EVEN
_initmem:
	CMP	ENDOURS,0	; check if already initialized
	JNE	_initmem_FAILED
	CMP	OURSEG,0FFFFh
	JNE	_initmem_FAILED
	MOV	BX,0FFFFh	; request an impossibly large block
	MOV	AH,48h
	INT	21h
	CMP	AX,8		; call should return "insufficient memory"
	JNE	_initmem_FAILED
	OR	BX,BX		; no memory available at all? we failed then
	JZ	_initmem_FAILED
	MOV	AH,48h		; BX is size of largest available block
	INT	21h		; allocate it
	JC	_initmem_FAILED	; this should work
	MOV	OURSEG,AX	; save the block address
	MOV	FREESTART,AX	; put the block in the free list
	MOV	ES,AX
	MOV	ES:NEXTBLOCK,0
	MOV	ES:BLOCKSIZE,BX
	ADD	AX,BX		; save end of our memory
	MOV	ENDOURS,AX
	XOR	AX,AX		; return success
	CALL	DETECTEMS	; check for EMS (modifies no registers)
	JMP	SHORT _initmem_DONE
	EVEN
_initmem_FAILED:
	MOV	AX,-1		; return failure
_initmem_DONE:
	RET
;
; _allocdma routine, to allocate a 64k DMA buffer aligned on a 64k boundary.
; Uses C calling conventions.  Takes no parameters.  This routine must be
; called after _initmem, but before any blocks are allocated with _getblock.
;
; Returns the segment address of the DMA buffer in AX if successful, 0 if
; not.  Destroys BX, CX, DX, ES.
;
	EVEN
_allocdma:
	XOR	AX,AX			; start out pessimistic
	MOV	BX,FREESTART		; get the memory block address
	OR	BX,BX			; error if no free block
	JZ	_allocdma_DONE
	MOV	ES,BX			; ES addresses the block
	MOV	DX,BX			; DX is end of allocated block
	ADD	DX,ES:BLOCKSIZE
	;
	; Determine a candidate for DMA buffer.
	;
	ADD	BX,0FFFh	; round block address up to a 64k boundary
	AND	BX,0F000h
	MOV	CX,BX		; save its address in CX
	;
	; Verify the DMA buffer.
	;
	CMP	BX,DX			; if past end of our block, failed
	JAE	_allocdma_DONE
	ADD	BX,1000h		; BX is end of DMA buffer
	SUB	DX,BX			; make sure we own the whole thing
	JB	_allocdma_DONE		; (DX is size of what's after)
	MOV	AX,CX		; get segment of DMA buffer in AX for caller
	;
	; Create a new free block after the DMA buffer.
	;
	JE	_allocdma_NONEAFTER
	MOV	ES,BX		; ES addresses new free block
	MOV	ES:NEXTBLOCK,0	; no free blocks after this one
	MOV	ES:BLOCKSIZE,DX	; set size of new block
	MOV	ES,FREESTART	; fix up the next block pointer of previous
	MOV	ES:NEXTBLOCK,BX	;   block
	;
	; Adjust the free block before the DMA buffer.
	;
_allocdma_NONEAFTER:
	CMP	AX,FREESTART	; DMA exactly at start of our block?
	JE	_allocdma_NONEBEFORE
	MOV	BX,AX		; no - set new block size
	SUB	BX,FREESTART
	MOV	ES:BLOCKSIZE,BX
	JMP	SHORT _allocdma_DONE	; all done
	;
	; Special case:  no free block before the DMA buffer.  Set FREESTART
	; to the block after.
	;
	EVEN
_allocdma_NONEBEFORE:
	MOV	BX,ES:NEXTBLOCK
	MOV	FREESTART,BX
	;
	; One way or another, we're done.
	;
_allocdma_DONE:
	RET
;
; _getblock routine, to allocate a paragraph-aligned block of memory.  Uses
; C calling conventions.  Takes one parameter, the size of the desired block
; in paragraphs.
;
; Returns the segment address of the allocated block in AX if successful,
; 0 if not.  Destroys BX, CX, DX, ES.
;
_getblock_NPARAS EQU	WORD PTR [BP+4]
	EVEN
_getblock:
	PUSH	BP
	MOV	BP,SP
	PUSH	DS
	XOR	AX,AX			; start out pessimistic
	CMP	_getblock_NPARAS,AX	; requesting 0 paragraphs is invalid
	JE	_getblock_DONE
	MOV	BX,FREESTART		; get the start of the free list
	OR	BX,BX			; failed if no free block
	JZ	_getblock_DONE
	MOV	ES,BX			; ES addresses the first free block
	;
	; See if the first block in the free list will work.
	;
	MOV	BX,ES:BLOCKSIZE		; size of first block <= requested?
	SUB	BX,_getblock_NPARAS	; (BX is what's left of the block)
	JB	_getblock_NOTFIRST	; try another block if not
	MOV	AX,ES			; save block address in AX
	JE	_getblock_ALLFIRST	; check if we return the entire block
	;
	; Special case:  we're returning part of the first block.
	;
	MOV	CX,ES:NEXTBLOCK		; CX is pointer to next block
	MOV	DX,ES			; create a new block
	ADD	DX,_getblock_NPARAS
	MOV	ES,DX
	MOV	ES:NEXTBLOCK,CX		; set next block pointer
	MOV	ES:BLOCKSIZE,BX		; set block size
	MOV	FREESTART,ES		; new block is first in list
	JMP	SHORT _getblock_DONE	; all done
	;
	; Special case:  we're returning the entire first block.
	;
	EVEN
_getblock_ALLFIRST:
	MOV	BX,ES:NEXTBLOCK
	MOV	FREESTART,BX
	JMP	SHORT _getblock_DONE	; all done
	;
	; First block in the free list would not work.  Try later ones.
	;
	EVEN
_getblock_NOTFIRST:
	MOV	DS,FREESTART		; DS is pointer to previous
	;
	; Loop over the free blocks until we find one we can use, or until
	; we get to the end of the list.
	;
_getblock_FINDLOOP:
	MOV	BX,ES:NEXTBLOCK		; failed if no more blocks
	OR	BX,BX
	JE	_getblock_DONE
	MOV	ES,BX			; current = current->next
	MOV	BX,ES:BLOCKSIZE		; size of block <= requested?
	SUB	BX,_getblock_NPARAS	; (BX is what's left of the block)
	JAE	_getblock_FOUND
	MOV	BX,ES			; previous = current
	MOV	DS,BX
	JMP	SHORT _getblock_FINDLOOP ; find next block
	;
	; One way or another, we're done (we put this in the middle to
	; avoid out-of-range errors on conditional jumps).
	;
	EVEN
_getblock_DONE:
	POP	DS
	POP	BP
	RET
	;
	; We found a block, now fix up the list.
	;
	EVEN
_getblock_FOUND:
	MOV	AX,ES			; save block address in AX
	JE	_getblock_WHOLE		; check if returning entire block
	MOV	CX,ES:NEXTBLOCK		; CX is pointer to next block
	MOV	DX,ES			; create a new block
	ADD	DX,_getblock_NPARAS
	MOV	ES,DX
	MOV	ES:NEXTBLOCK,CX		; set next block pointer
	MOV	ES:BLOCKSIZE,BX		; set block size
	MOV	NEXTBLOCK,DX		; previous->next = new block
	JMP	SHORT _getblock_DONE	; all done
	;
	; Special case:  allocating an entire block.
	;
	EVEN
_getblock_WHOLE:
	MOV	BX,ES:NEXTBLOCK		; previous->next = current->next
	MOV	NEXTBLOCK,BX
	JMP	SHORT _getblock_DONE	; all done
;
; _freeblock routine, to release a paragraph-aligned block of memory
; previously allocated with _getblock.  Uses C calling conventions.  Takes
; two parameters, the segment address of the block and its size in paragraphs.
;
; Note that we allow part of a block to be released.  Don't know how useful
; that will be.
;
; Returns 0 in AX if successful, -1 if not.  Destroys BX, CX, DX, ES.
;
_freeblock_SEGADDR EQU	WORD PTR [BP+4]
_freeblock_NPARAS EQU	WORD PTR [BP+6]
	EVEN
_freeblock:
	PUSH	BP
	MOV	BP,SP
	PUSH	DS
	CMP	_freeblock_NPARAS,0	; don't free it if length is 0
	JE	_freeblock_ERROR
	MOV	CX,_freeblock_SEGADDR	; CX is start of block
	CMP	CX,OURSEG		; segment out of range?
	JB	_freeblock_ERROR
	MOV	DX,CX			; DX is end of block
	ADD	DX,_freeblock_NPARAS
	JC	_freeblock_ERROR	; wrap past 1M boundary - bad!
	CMP	DX,ENDOURS		; end out of range?
	JA	_freeblock_ERROR
	MOV	AX,FREESTART		; AX is start of first free block
	OR	AX,AX			; is the free list empty?
	JZ	_freeblock_ADDFIRST	; add unconditionally if so
	CMP	DX,AX			; end of block <= first free?
	JA	_freeblock_NOTFIRST
	;
	; The block being freed is below the first existing free block.
	; Put it at the front of the list.
	;
_freeblock_ADDFIRST:
	MOV	ES,CX			; ES addresses the block to free
	MOV	ES:NEXTBLOCK,AX		; current front of list is next
	MOV	BX,_freeblock_NPARAS	; set block size
	MOV	ES:BLOCKSIZE,BX
	MOV	FREESTART,ES		; put at front of the list
	OR	AX,AX			; no next block?
	JZ	_freeblock_SUCCESS	; no need to coalesce if so
	CMP	DX,AX			; next block continguous?
	JNE	_freeblock_SUCCESS	; don't coalesce if not
	MOV	DS,AX			; ES -> current, DS -> next
	MOV	AX,NEXTBLOCK		; current->next = next->next
	MOV	ES:NEXTBLOCK,AX
	ADD	BX,BLOCKSIZE		; current->size += next->size
	MOV	ES:BLOCKSIZE,BX
	JMP	SHORT _freeblock_SUCCESS
	;
	; The block is not below the first one.  Loop over the remaining
	; blocks.  At top of loop, AX = start of previous block in free
	; list, CX = block to free, DX = end of block to free.
	;
	EVEN
_freeblock_NOTFIRST:
	MOV	ES,AX		; ES addresses previous block in free list
	ADD	AX,ES:BLOCKSIZE	; AX is end of block
	CMP	CX,AX		; does block to free overlap this block?
	JB	_freeblock_ERROR ; error if so
	MOV	BX,ES:NEXTBLOCK	; BX addresses start of next block
	OR	BX,BX		; if no more, we found the insertion point
	JZ	_freeblock_ADDHERE
	CMP	DX,BX		; end of block to free <= start of next?
	JBE	_freeblock_ADDHERE ; we found insertion point if so
	MOV	AX,BX		; insertion point not found, go to next
	JMP	SHORT _freeblock_NOTFIRST
	;
	; Error:  block was invalid or not allocated (we put this here to
	; avoid out-of-range errors on conditional jumps).
	;
	EVEN
_freeblock_ERROR:
	MOV	AX,-1
	JMP	SHORT _freeblock_DONE
	;
	; Block successfully freed.
	;
	EVEN
_freeblock_SUCCESS:
	XOR	AX,AX
_freeblock_DONE:
	POP	DS
	POP	BP
	RET
	;
	; The place to add this block to the free list has been found.
	; ES addresses the previous free block.  BX is the next free block,
	; if there is one, 0 otherwise.  CX is the start of the block to
	; be freed.  DX is the end of the block to be freed.  AX is the end
	; of the previous free block.
	;
	EVEN
_freeblock_ADDHERE:
	MOV	DS,CX			; DS addresses new free block
	PUSH	_freeblock_NPARAS	; set block size of new block
	POP	BLOCKSIZE
	MOV	NEXTBLOCK,BX		; set next pointer of new block
	OR	BX,BX			; is there a next block?
	JZ	_freeblock_DOPRIOR	; if not, don't coalesce with it
	CMP	DX,BX			; contiguous w/next block?
	JNE	_freeblock_DOPRIOR	; if not, don't coalesce
	MOV	DX,ES			; coalesce w/next - save previous
	MOV	ES,BX			; ES addresses next
	MOV	BX,ES:NEXTBLOCK		; current->next = next->next
	MOV	NEXTBLOCK,BX
	MOV	BX,ES:BLOCKSIZE		; current->size += next->size
	ADD	BLOCKSIZE,BX
	MOV	ES,DX			; ES addresses previous again
	;
	; Adjust the previous block's next pointer to put the new block in
	; the list.  ES addresses previous block, DS addresses new block,
	; AX is end of previous block, CX is start of new block.
	;
_freeblock_DOPRIOR:
	MOV	ES:NEXTBLOCK,CX		; previous->next = current
	CMP	CX,AX			; contiguous w/previous block?
	JNE	_freeblock_SUCCESS	; all done if so
	MOV	AX,NEXTBLOCK		; previous->next = current->next
	MOV	ES:NEXTBLOCK,AX
	MOV	AX,BLOCKSIZE		; previous->size += current->size
	ADD	ES:BLOCKSIZE,AX
	JMP	SHORT _freeblock_SUCCESS
;
; _movedown routine, to move a block of allocated memory down as far as
; possible in memory.  Uses C calling conventions.  Takes two parameters,
; the segment address of the block to move and its size in paragraphs.  The
; block must be less than 64k in size.
;
; Returns the segment address of the relocated block in AX.  Destroys BX,
; CX, DX, ES.
;
_movedown_SEGADDR EQU	WORD PTR [BP+4]
_movedown_NPARAS EQU	WORD PTR [BP+6]
_movedown_SAVESEG EQU	WORD PTR [BP-2]
_movedown_NEWSEG EQU	WORD PTR [BP-4]
	EVEN
_movedown:
#IF M_I286
	ENTER	4,0
#ELSE
	PUSH	BP	; address parameters, and allocate 4 bytes on the
	MOV	BP,SP	;   stack
	SUB	SP,4
#ENDIF
	PUSH	SI			; save SI, DI for C
	PUSH	DI
	;
	; Initially assume that the block will not be moved.
	;
	MOV	AX,_movedown_SEGADDR
	MOV	_movedown_NEWSEG,AX
	;
	; Temporarily allocate another block of the same size as this one.
	;
	PUSH	_movedown_NPARAS
	CALL	_getblock		; destroys BX, CX, DX, ES
	ADD	SP,2
	OR	AX,AX			; if unable to allocate, don't move
	JZ	_movedown_DONE
	MOV	_movedown_SAVESEG,AX	; save segment address
	;
	; Copy data from this block into the temporary block.
	;
	MOV	ES,AX
	XOR	DI,DI
	MOV	DS,_movedown_SEGADDR
	MOV	SI,DI
	MOV	CX,_movedown_NPARAS
#IF M_I286
	SHL	CX,3
#ELSE
	SHL	CX,1
	SHL	CX,1
	SHL	CX,1
#ENDIF
	CLD
	REP	MOVSW
	MOV	AX,SS			; (get back DS)
	MOV	DS,AX
	;
	; Free the original block.
	;
	PUSH	_movedown_NPARAS
	PUSH	_movedown_SEGADDR
	CALL	_freeblock		; destroys AX, BX, CX, DX, ES
	ADD	SP,4
	;
	; Special case:  if the temporary segment is below the original
	; segment, make it the (permanent) new segment.
	;
	MOV	AX,_movedown_SAVESEG
	CMP	AX,_movedown_SEGADDR
	JAE	_movedown_GETNEW
	MOV	_movedown_NEWSEG,AX
	JMP	_movedown_DONE
	;
	; The temporary segment is above the original segment.  Since the
	; original segment has been freed, we're guaranteed to be able to
	; allocate a segment lower in memory.  Do so.
	;
	EVEN
_movedown_GETNEW:
	PUSH	_movedown_NPARAS
	CALL	_getblock		; destroys BX, CX, DX, ES
	ADD	SP,2
	MOV	_movedown_NEWSEG,AX
	;
	; Copy the data from the temporary block to the new block.
	;
	MOV	ES,AX
	XOR	DI,DI
	MOV	DS,_movedown_SAVESEG
	MOV	SI,DI
	MOV	CX,_movedown_NPARAS
#IF M_I286
	SHL	CX,3
#ELSE
	SHL	CX,1
	SHL	CX,1
	SHL	CX,1
#ENDIF
	CLD
	REP	MOVSW
	MOV	AX,SS			; (get back DS)
	MOV	DS,AX
	;
	; Release the temporary block.
	;
	PUSH	_movedown_NPARAS
	PUSH	_movedown_SAVESEG
	CALL	_freeblock		; destroys AX, BX, CX, DX, ES
	ADD	SP,4
	;
	; All done.
	;
_movedown_DONE:
	MOV	AX,_movedown_NEWSEG
	POP	DI			; restore DI, SI
	POP	SI
#IF M_I286
	LEAVE
#ELSE
	MOV	SP,BP
	POP	BP
#ENDIF
	RET
;
; _checkmem routine, to verify that the free list has not been corrupted,
; for example, by writing past the end of an allocated block.  Uses C calling
; conventions.  Takes no parameters.
;
; Returns 0 in AX if the memory manager is in a valid state, -1 if not.
; Destroys BX, ES.
;
	EVEN
_checkmem:
	CMP	OURSEG,0FFFFh	; are we initialized?
	JNE	_checkmem_INITDONE
	XOR	AX,AX		; not initialized - correct for this state?
	CMP	ENDOURS,AX
	JNE	_checkmem_BAD
	CMP	FREESTART,AX
	JNE	_checkmem_BAD
	JMP	SHORT _checkmem_GOOD
	;
	; We're initialized.
	;
	EVEN
_checkmem_INITDONE:
	MOV	AX,OURSEG		; OURSEG < ENDOURS? error if not
	CMP	AX,ENDOURS
	JNB	_checkmem_BAD
	MOV	AX,FREESTART		; start at front of list
	OR	AX,AX			; all done if list empty
	JZ	_checkmem_GOOD
	CMP	AX,OURSEG		; FREESTART < OURSEG? error if so
	JB	_checkmem_BAD
	;
	; Loop over the blocks, checking them.
	;
_checkmem_BLOCKLOOP:
	CMP	AX,ENDOURS		; current >= ENDOURS? error if so
	JAE	_checkmem_BAD
	MOV	ES,AX			; ES addresses current
	MOV	BX,ES:BLOCKSIZE		; BX is block size
	OR	BX,BX			; block size = 0? error if so
	JZ	_checkmem_BAD
	ADD	BX,AX			; BX is end of this block
	JC	_checkmem_BAD		; wrap past 1M is an error
	CMP	BX,ENDOURS		; end of block > ENDOURS? error if so
	JA	_checkmem_BAD
	MOV	AX,ES:NEXTBLOCK		; AX = next
	OR	AX,AX			; all done if end of list
	JZ	_checkmem_GOOD
	CMP	BX,AX			; end of block >= next? error if so
	JAE	_checkmem_BAD
	JMP	SHORT _checkmem_BLOCKLOOP ; check next block
	;
	; Everything checks out.
	;
	EVEN
_checkmem_GOOD:
	XOR	AX,AX
	JMP	SHORT _checkmem_DONE
	;
	; Invalid state or freelist corrupt.
	;
	EVEN
_checkmem_BAD:
	MOV	AX,-1
_checkmem_DONE:
	RET
;
; _endmem routine, to finalize the memory manager.  Uses C calling
; conventions.  Takes no parameters.
;
; Returns nothing.  Destroys AX, ES.
;
	EVEN
_endmem:
	MOV	ES,OURSEG	; release our memory block back to DOS
	MOV	AH,49h
	INT	21h
	MOV	FREESTART,0	; mark our free list empty so we won't try
				;   to allocate anything
	MOV	OURSEG,0FFFFh	; make the start of our memory greater than
	MOV	ENDOURS,0	;   the end so we won't try to free anything
	CALL	FREEALLEMS	; release all allocated expanded memory
	RET			;   (modifies no registers)
_TEXT	ENDS
	END
