; DACSTUF.ASM
;
; This assembler module contains routines for programming the DAC, DMA
; controller, and interrupt controller.
;

;
; Globally-visible stuff.
;
PUBLIC	_clockspersec	; number of DAC clocks per second
PUBLIC	_clockspersam	; number of clocks per output sample
PUBLIC	_samprate	; output sampling rate
PUBLIC	_volume		; DAC volume setting
PUBLIC	_partbufs	; table of segment addresses of 2k buffers
PUBLIC	DACDETECT	; DAC detection routine
PUBLIC	INITDAC		; sound hardware initialization routine
PUBLIC	ENDDAC		; sound hardware finalization routine
PUBLIC	DACSTART	; routine to start DMA
PUBLIC	DACSTOP		; routine to stop or pause DMA
PUBLIC	DACRESUME	; routine to resume DMA when paused
PUBLIC	DACSETVOL	; routine to change volume
PUBLIC	DACSETSPEED	; routine to change sampling rate
PUBLIC	DACGETPART	; routine to get the current 2k buffer number
PUBLIC	DACGETLAST	; routine to get the last output sample

;
; Establish the segment ordering.  Code comes first.  This is the C
; standard segment ordering for Small model.
;
_TEXT	SEGMENT WORD PUBLIC 'CODE'
_TEXT	ENDS

_DATA	SEGMENT PARA PUBLIC 'DATA'
_DATA	ENDS

CONST	SEGMENT WORD PUBLIC 'CONST'
CONST	ENDS

_BSS	SEGMENT WORD PUBLIC 'BSS'
_BSS	ENDS

STACK	SEGMENT PARA STACK 'STACK'
STACK	ENDS

DGROUP	GROUP	_DATA,CONST,_BSS,STACK

_DATA	SEGMENT PARA PUBLIC 'DATA'
		;
		; Base I/O port address of the Tandy DAC.
		;
		EVEN
DACBASE		DW	0
		;
		; Clocks per second - our main time measurement.
		;
		EVEN
_clockspersec	DD	3579545
		;
		; Output sampling rate, word 4000-44100.  The default here
		; is 22000 for the 286 version, 10000 for the 8086 version.
		;
		EVEN
#IF M_I286
_samprate	DW	22000
#ELSE
_samprate	DW	10000
#ENDIF
		;
		; Number of clocks per output sample (DAC divider value),
		; word 81-895.
		;
		EVEN
_clockspersam	DW	0
		;
		; Volume of DAC output (0 = silence, 7 = maximum).
		;
_volume		DB	7
		;
		; Flag, indicates whether the sound chip has been initialized.
		;
INITIALIZED	DB	0
		;
		; Segment address of 64k DMA buffer.
		;
		EVEN
DMASEG		DW	0
		;
		; Segment addresses of the 32 2k partial buffers.
		;
		EVEN
_partbufs	DW	32 DUP (0)
		;
		; DMA page register value for our buffer.
		;
DMAPAGE		DB	0
		;
		; Flag for whether the DAC is a newer double-buffered model
		; or not.
		;
NEWCHIP		DB	0
		;
		; Default vector for sound chip interrupt.
		;
		EVEN
INT0FDEFAULT	DD	0
		;
		; Values to write to the DAC to ramp it up, with
		; appropriate delays between, for the old DAC, for each
		; final volume 0-7.  These are records of the following
		; format:
		;    word           number of volume records
		;    3 bytes each   volume records
		; Volume records are in this format:
		;    byte           DAC volume
		;    byte           initial output sample
		;    byte           final output sample
		; The table comes from the PSSJ Digital Sound Toolkit.
		;
OGAIN0		DW	1
		DB	0,80h,80h
OGAIN1		DW	1
		DB	20h,0,80h
OGAIN2		DW	2
		DB	20h,0,2Ch
		DB	40h,0,80h
OGAIN3		DW	2
		DB	20h,0,67h
		DB	60h,0,80h
OGAIN4		DW	2
		DB	20h,0,0BCh
		DB	80h,0,80h
OGAIN5		DW	3
		DB	20h,0,0BCh
		DB	80h,0,02Bh
		DB	0A0h,0,80h
OGAIN6		DW	3
		DB	20h,0,0BCh
		DB	80h,0,67h
		DB	0C0h,0,80h
OGAIN7		DW	3
		DB	20h,0,0BCh
		DB	80h,0,0BDh
		DB	0E0h,0,80h
		;
		; Values to write to the DAC to ramp it up, with
		; appropriate delays between, for the new DAC, for each
		; final volume 1-7.  See above for the format.  The table
		; comes from the PSSJ Digital Sound Toolkit.
		;
NGAIN1		DW	2
		DB	80h,0DCh,0
		DB	20h,0BDh,80h
NGAIN2		DW	2
		DB	80h,0DCh,0Dh
		DB	40h,80h,80h
NGAIN3		DW	2
		DB	80h,0DCh,3Dh
		DB	60h,80h,80h
NGAIN4		DW	1
		DB	80h,0DCh,80h
NGAIN5		DW	1
		DB	0A0h,7Dh,80h
NGAIN6		DW	1
		DB	0C0h,3Ah,80h
NGAIN7		DW	1
		DB	0E0h,0Ah,80h
		;
		; Table of pointers to instruction records to ramp up the
		; old DAC.  The desired output volume is an index into this
		; table.
		;
		EVEN
OGAINTBL	DW	OFFSET OGAIN0
		DW	OFFSET OGAIN1
		DW	OFFSET OGAIN2
		DW	OFFSET OGAIN3
		DW	OFFSET OGAIN4
		DW	OFFSET OGAIN5
		DW	OFFSET OGAIN6
		DW	OFFSET OGAIN7
		;
		; Table of pointers to instruction records to ramp up the
		; new DAC.  The desired output volume is an index into this
		; table.
		;
		EVEN
NGAINTBL	DW	OFFSET OGAIN0
		DW	OFFSET NGAIN1
		DW	OFFSET NGAIN2
		DW	OFFSET NGAIN3
		DW	OFFSET NGAIN4
		DW	OFFSET NGAIN5
		DW	OFFSET NGAIN6
		DW	OFFSET NGAIN7
_DATA	ENDS

_TEXT	SEGMENT WORD PUBLIC 'CODE'
;
; DACDETECT routine, to detect a Tandy DAC and obtain its base I/O port
; address.  Takes no parameters.  Uses Pascal calling conventions.
;
; Returns 0 in AX if the DAC is present, -1 if not.  Destroys AX, CX, DX.
;
	EVEN
DACDETECT:
	;
	; Check for PCMCIA Socket Services.
	;
	MOV	AX,8003h
	XOR	CX,CX
	INT	1Ah
	CMP	CX,5353h
	JE	DACDETECT_NODAC
	;
	; Check for Tandy DAC.
	;
	MOV	AX,8100h
	INT	1Ah
	CMP	AX,8000h
	JAE	DACDETECT_NODAC
	MOV	DACBASE,AX	; save base port candidate
	MOV	DX,AX		; DX = base DAC port + 2
	ADD	DX,2
	CLI
	IN	AL,DX		; get current value and save in CL
	MOV	CL,AL
	MOV	CH,-1		; set CH=-1 (assume no DAC)
	MOV	AL,CH		; set all the bits
	OUT	DX,AL
	XOR	AL,AL		; now clear them all
	OUT	DX,AL
	IN	AL,DX		; read them back
	OR	AL,AL			; all clear?
	JNZ	DACDETECT_DACCHKDONE	; if not, no DAC
	NOT	AL			; set all the bits
	OUT	DX,AL
	IN	AL,DX			; read them back
	NOT	AL			; all set?
	JNZ	DACDETECT_DACCHKDONE	; if not, no DAC
	XOR	CH,CH		; CH=0 (definitely a DAC here)
DACDETECT_DACCHKDONE:
	MOV	AL,CL		; restore previous value at DAC base + 2
	OUT	DX,AL
	STI
	MOV	AL,CH		; set AX=0 if DAC was there, AX=-1 if not
	CBW
	JMP	SHORT DACDETECT_END
	EVEN
DACDETECT_NODAC:
	MOV	AX,-1
DACDETECT_END:
	RET
;
; Routine to delay a little bit.  Used when ramping up and down.
;
; Returns nothing.  Destroys nothing.
;
	EVEN
DELAY:
	PUSH	CX
	MOV	CX,100
	LOOP	$
	POP	CX
	RET
;
; RAMPUP routine, to ramp the sound chip up from the speaker idling position
; to the baseline sample at the desired output volume.  Assumes that the DAC
; is in direct write mode, no DMA, and that the volume is initially set to
; zero.
;
; Returns nothing.  Destroys AX, CX, DX, SI, DI.
;
	EVEN
RAMPUP:
	;
	; Get the DAC data port in DI, the volume port in DX.
	;
	MOV	DI,DACBASE
	INC	DI
	MOV	DX,DI
	INC	DX
	INC	DX
	;
	; DS:SI addresses the table of records for the chip.
	;
	MOV	SI,OFFSET OGAINTBL
	CMP	NEWCHIP,1
	JNE	RAMPUP_FINDGOAL
	MOV	SI,OFFSET NGAINTBL
	;
	; DS:SI addresses the correct record for the desired volume level.
	;
RAMPUP_FINDGOAL:
	MOV	AL,_volume
	CBW
	SHL	AX,1
	ADD	SI,AX
	MOV	SI,[SI]
	;
	; Clear the direction flag.  CX is the number of times we need to
	; change the volume.
	;
	CLD
	LODSW
	MOV	CX,AX
	;
	; Loop over the volume settings.
	;
RAMPUP_VOLLOOP:
	LODSW			; volume in AL, initial sample in AH
	;
	; Set the volume and the initial sample for that volume.
	;
	CLI			; do this as fast as possible
	OUT	DX,AL		; set the volume
	XCHG	DI,DX		; DX = data port, DI = volume port
	MOV	AL,AH		; initial sample in AL, copy in AH
	OUT	DX,AL		; write the initial sample
	STI
	;
	; Wait for a little bit.
	;
	CALL	DELAY
	;
	; Figure out whether we're going up or down.
	;
	LODSB			; final sample in AL
	XCHG	AL,AH		; current sample in AL, final sample in AH
	CMP	AL,AH
	JE	RAMPUP_VOLLPEND	; if current = final, go to next volume
	JA	RAMPUP_DOWNLOOP	; else if current > final, going down
	;
	; Here we slowly increment the sample value until we get to the
	; final one for this volume setting.
	;
RAMPUP_UPLOOP:
	INC	AL		; increment the sample value
	OUT	DX,AL		; send it to the data port
	CALL	DELAY		; wait
	CMP	AL,AH		; go again if not yet at final value
	JNE	RAMPUP_UPLOOP
	JMP	SHORT RAMPUP_VOLLPEND ; if no more, go to next volume setting
	;
	; Here we slowly decrement the sample value until we get to the
	; final one for this volume setting.
	;
	EVEN
RAMPUP_DOWNLOOP:
	DEC	AL		; decrement the sample value
	OUT	DX,AL		; send it to the data port
	CALL	DELAY		; wait
	CMP	AL,AH		; go again if not yet at final value
	JNE	RAMPUP_DOWNLOOP
	;
	; Get the volume port back in DX, data port in DI, and go to the
	; next volume setting.
	;
RAMPUP_VOLLPEND:
	XCHG	DI,DX		; DX = volume port, DI = data port
	LOOP	RAMPUP_VOLLOOP
	RET
;
; Interrupt handler for IRQ 7, the sound chip interrupt.  Basically all we
; want to do here is to acknowledge the interrupt so that DMA will continue.
; This program does its actual work by polling the DMA current address
; registers.
;
	EVEN
INT0FHDLR:
	CLI
	PUSH	AX	; just save AX here, we might not need DX or DS
        ;
        ; Read the interrupt controller's in-service register to see if an
        ; IRQ 7 has in fact occurred.  If not (electrical noise), just restore
        ; registers and return.
        ;
        MOV	AL,0Bh
        OUT	20h,AL
        JMP	$+2
        IN	AL,20h
        TEST	AL,80h
        JZ	INT0FHDLR_RESTOREAX	; only register to restore is AX
	;
	; Check if it was a DAC interrupt.  If not, just issue EOI and
	; return.
	;
	PUSH	DX		; guess we do need 'em
	PUSH	DS
	MOV	AX,CS		; DS addresses local data
	MOV	DS,AX
	MOV	DX,DACBASE
	IN	AL,DX
	TEST	AL,8
	JZ	INT0FHDLR_EOI
	;
	; It was our interrupt.  Clear the DMA interrupt at the sound chip
	; so we will get another interrupt.
	;
	AND	AL,0F7h
	OUT	DX,AL
	OR	AL,8
	OUT	DX,AL
	;
	; Issue EOI to the interrupt controller.
	;
INT0FHDLR_EOI:
	MOV	AL,20h
	OUT	20h,AL
	POP	DS
	POP	DX
INT0FHDLR_RESTOREAX:
	POP	AX
	IRET			; will set the interrupt bit
;
; INITDAC routine, to initialize the sound hardware for playback.  Places
; the DAC in playback mode and ramps up to the baseline, hooks the sound
; chip interrupt (0Fh), allocates a DMA buffer and programs the DMA
; controller, and enables the sound chip interrupt at the interrupt
; controller.  Takes no parameters.  Uses Pascal calling conventions.
;
; Returns 0 in AX if successful, -1 if a DMA buffer could not be allocated.
; Destroys BX, CX, DX, ES.
;
	EVEN
INITDAC:
	;
	; Allocate a DMA buffer.  If unable to do so, return -1.
	;
	CALL	_allocdma		; destroys BX, CX, DX, ES
	OR	AX,AX
	JNZ	INITDAC_GOTDMA
	DEC	AX			; AX = -1
	RET
	EVEN
INITDAC_GOTDMA:
	PUSH	SI			; save SI,DI for C
	PUSH	DI
	MOV	DMASEG,AX		; save DMA segment address
#IF M_I286
	SHR	AH,4			; compute DMA page
#ELSE
	SHR	AH,1
	SHR	AH,1
	SHR	AH,1
	SHR	AH,1
#ENDIF
	MOV	DMAPAGE,AH
	MOV	AX,DS			; compute the 2k buffer segments
	MOV	ES,AX
	MOV	DI,OFFSET _partbufs
	MOV	CX,32
	MOV	AX,DMASEG
	CLD
INITDAC_PARTBUFLP:
	STOSW
	ADD	AX,128
	LOOP	INITDAC_PARTBUFLP
	;
	; Determine whether the DAC is a new model so we can ramp it up
	; correctly.
	;
	CLI
	MOV	NEWCHIP,1	; initially assume it's a new chip
	MOV	DX,DACBASE	; address amplitude/frequency MSB register
	ADD	DX,3
	IN	AL,DX		; clear bit 4
	AND	AL,0EFh
	OUT	DX,AL
	OR	AL,10h		; set bit 4
	OUT	DX,AL
	IN	AL,DX		; is it set?
	TEST	AL,10h
	JZ	INITDAC_NEWCHIP	; if not, it's a new version
	AND	AL,0EFh		; clear bit 4
	OUT	DX,AL
	IN	AL,DX		; is it clear?
	TEST	AL,10h
	JNZ	INITDAC_NEWCHIP	; if not, it's a new version
	MOV	NEWCHIP,0	; old chip detected
INITDAC_NEWCHIP:
	STI
	;
	; Disable IRQ 7 at the interrupt controller.
	;
	CLI
	IN	AL,21h
	JMP	$+2
	OR	AL,80h
	OUT	21h,AL
	;
	; Disable DMA channel 1.
	;
	MOV	AL,5
	OUT	0Ah,AL
	;
	; Program the sound chip for playback mode, no DMA yet.
	;
	MOV	DX,DACBASE	; get DAC base port
	MOV	AL,10h		; enable DMA interrupt, still joystick mode
	OUT	DX,AL
	ADD	DX,3		; set sound volume to zero
	XOR	AL,AL
	OUT	DX,AL
	MOV	DX,DACBASE	; direct write to DAC, DMA disabled, DMA
	MOV	AL,13h		;   interrupt enabled, DMA interrupt clear
	OUT	DX,AL
	STI
	;
	; Ramp up the chip to the baseline.  It takes a while to do it, so
	; it should be done with interrupts enabled.
	;
	CALL	RAMPUP		; destroys AX, CX, DX, SI, DI
	;
	; Program the sound chip for playback with DMA.
	;
	CLI
	MOV	DX,DACBASE	; direct write to DAC, DMA enabled, DMA
	MOV	AL,17h		;   interrupt enabled, DMA interrupt clear
	OUT	DX,AL
	MOV	AL,1Fh		; direct write to DAC, DMA enabled, DMA
	OUT	DX,AL		;   interrupt enabled, DMA interrupt allowed
	;
	; Compute an initial value for the number of clocks per output sample.
	; We'll continue to use that value until the user changes the sampling
	; rate.
	;
	MOV	AX,WORD PTR _clockspersec
	MOV	DX,WORD PTR _clockspersec+2
	MOV	BX,_samprate
	DIV	BX
	SHR	BX,1
	CMP	BX,DX
	ADC	AX,0
	MOV	_clockspersam,AX
	;
	; Program the sound chip speed.  Sound chip is ready to go now.
	;
	MOV	DX,DACBASE	; program low byte of speed
	ADD	DX,2
	OUT	DX,AL
	INC	DX		; program high byte of speed
	IN	AL,DX		; (keep current volume from RAMPUP)
	AND	AL,0E0h
	OR	AL,AH
	OUT	DX,AL
	;
	; Set up the DMA controller for autoinitialize-mode playback on
	; DMA channel 1, using the 64k DMA buffer we allocated above.  Leave
	; DMA channel 1 disabled.
	;
	MOV	AL,59h		; select channel 1, read transfer from memory,
	OUT	0Bh,AL		;   autoinitialization enabled, address incre-
	JMP	$+2		;   ment, single mode
	MOV	AL,DMAPAGE	; set DMA channel 1 page register
	OUT	83h,AL
	JMP	$+2
	MOV	AL,0FFh		; clear byte pointer flip/flop
	OUT	0Ch,AL
	JMP	$+2
	OUT	03h,AL		; program low word of count - count programmed
	JMP	$+2		;   is 0FFFFh for a 64k autotinitialize buffer
	OUT	03h,AL		; program high word
	JMP	$+2
	INC	AL		; set base address = 0
	OUT	02h,AL		;   program low word
	JMP	$+2
	OUT	02h,AL		;   program high word
	;
	; Hook Int 0Fh.
	;
	XOR	AX,AX
	MOV	ES,AX
	MOV	BX,4*0Fh
	MOV	AX,ES:[BX]
	MOV	WORD PTR INT0FDEFAULT,AX
	MOV	AX,ES:[BX+2]
	MOV	WORD PTR INT0FDEFAULT+2,AX
	MOV	WORD PTR ES:[BX],OFFSET INT0FHDLR
	MOV	ES:[BX+2],CS
	;
	; Enable IRQ7 at the interrupt controller.  At this point, we're all
	; set to go.  From now on, we can start/stop playback and alter the
	; rate or volume by programming the DAC amplitude/frequency registers
	; and the DMA single mask and channel 1 address registers.  We can
	; poll the channel 1 address registers to determine where we are in
	; the DMA buffer.
	;
	IN	AL,21h
	JMP	$+2
	AND	AL,7Fh
	OUT	21h,AL
	STI
	MOV	INITIALIZED,1		; mark the sound hardware initialized
	XOR	AX,AX			; return success
	POP	DI			; restore DI,SI for C
	POP	SI
	RET
;
; RAMPDOWN routine, to ramp the sound chip down from the baseline sample
; at the current output volume to the speaker idling position.  Assumes
; that the DAC is in direct write mode, no DMA.
;
; Returns nothing.  Destroys AX, BX, CX, DX, SI, DI.
;
	EVEN
RAMPDOWN:
	;
	; Get the DAC data port in DX, the volume port in DI.
	;
	MOV	DX,DACBASE
	INC	DX
	MOV	DI,DX
	INC	DI
	INC	DI
	;
	; DS:SI addresses the table of records for the chip.
	;
	MOV	SI,OFFSET OGAINTBL
	CMP	NEWCHIP,1
	JNE	RAMPDOWN_FINDGOAL
	MOV	SI,OFFSET NGAINTBL
	;
	; DS:SI addresses the correct record for the current volume level.
	;
RAMPDOWN_FINDGOAL:
	MOV	AL,_volume
	CBW
	SHL	AX,1
	ADD	SI,AX
	MOV	SI,[SI]
	;
	; CX is the number of times we need to change the volume.  DS:SI
	; points to the second byte of the record for the output volume.
	;
	MOV	CX,[SI]
	ADD	SI,CX
	ADD	SI,CX
	ADD	SI,CX
	;
	; Set DF since we're going backwards.
	;
	STD
	;
	; Get the current output sample.
	;
	IN	AL,DX
	MOV	AH,AL
	XCHG	DI,DX		; DX = volume port, DI = data port
	JMP	SHORT RAMPDOWN_VOLLPSTART
	;
	; Loop over the volume settings.
	;
	EVEN
RAMPDOWN_VOLLOOP:
	LODSB			; initial sample in AL
	MOV	AH,AL		; initial sample in AH
RAMPDOWN_VOLLPSTART:
	LODSB			; final sample in AL
	MOV	BL,AL		; final sample in BL
	LODSB			; volume in AL, initial sample in AH
	;
	; Set the volume and the initial sample for that volume.
	;
	CLI			; do this as fast as possible
	OUT	DX,AL		; set the volume
	XCHG	DI,DX		; DX = data port, DI = volume port
	MOV	AL,AH		; initial sample in AL
	OUT	DX,AL		; write the initial sample
	STI
	;
	; Wait for a little bit.
	;
	CALL	DELAY
	;
	; Figure out whether we're going up or down.
	;
	CMP	AL,BL
	JE	RAMPDOWN_VOLLPEND ; if current = final, go to next volume
	JA	RAMPDOWN_DOWNLOOP ; else if current > final, going down
	;
	; Here we slowly increment the sample value until we get to the
	; final one for this volume setting.
	;
RAMPDOWN_UPLOOP:
	INC	AL		; increment the sample value
	OUT	DX,AL		; send it to the data port
	CALL	DELAY		; wait
	CMP	AL,BL		; go again if not yet at final value
	JNE	RAMPDOWN_UPLOOP
	JMP	SHORT RAMPDOWN_VOLLPEND ; if no more, go to next volume setting
	;
	; Here we slowly decrement the sample value until we get to the
	; final one for this volume setting.
	;
	EVEN
RAMPDOWN_DOWNLOOP:
	DEC	AL		; decrement the sample value
	OUT	DX,AL		; send it to the data port
	CALL	DELAY		; wait
	CMP	AL,BL		; go again if not yet at final value
	JNE	RAMPDOWN_DOWNLOOP
	;
	; Get the volume port back in DX, data port in DI, and go to the
	; next volume setting.
	;
RAMPDOWN_VOLLPEND:
	XCHG	DI,DX		; DX = volume port, DI = data port
	LOOP	RAMPDOWN_VOLLOOP
	;
	; All ramped down; now clear the direction flag.  It is an
	; "undocumented feature" of QuickC 2.0 that the string library
	; will malfunction if DF is set when we get back to the main
	; program.
	;
	CLD
	RET
;
; ENDDAC routine, to finalize the sound hardware before program termination.
; Disables DMA if in progress, ramps the sound chip down to the speaker
; rest position, puts the sound chip in joystick mode, and unhooks IRQ 7.
; Takes no parameters.  Uses Pascal calling conventions.
;
; Returns nothing.  Destroys AX, BX, CX, DX, ES.
;
	EVEN
ENDDAC:
	CMP	INITIALIZED,1		; are we initialized?
	JE	ENDDAC_INITDONE
	RET				; just return if not
	EVEN
ENDDAC_INITDONE:
	PUSH	SI			; save SI,DI for C
	PUSH	DI
	;
	; Disable DMA channel 1, if not disabled already.
	;
	CLI
	MOV	AL,5
	OUT	0Ah,AL
	;
	; Stop DMA at the sound chip.
	;
	MOV	DX,DACBASE
	IN	AL,DX
	AND	AL,0F3h		; clear DMA enable and DMA interrupt clear
	OUT	DX,AL
	STI
	;
	; Ramp the chip down to avoid the click at the end.  It needs to
	; be done with interrupts enabled since it takes a while.
	;
	CALL	RAMPDOWN	; destroys AX, BX, CX, DX, SI, DI
	;
	; Okay, now we're going to clear the DMA interrupt enable bit.
	; That may cause an interrupt, which we need to wait for.
	;
	CLI
	MOV	DX,DACBASE
	XOR	AL,AL		; set sound chip for joystick mode, DMA
	OUT	DX,AL		;   interrupt enable bit clear
	INC	DX		; read a byte from the data port (probably
	IN	AL,DX		;   not needed for playing)
	STI
	;
	; Wait for the interrupt, but not forever.  There doesn't seem to
	; be one generated on the 1000TL; maybe there is on the new chip?
	;
	MOV	CX,2000
	LOOP	$
	;
	; The DAC is in joystick mode.  Disable further interrupts on IRQ 7.
	;
	CLI
	IN	AL,21h
	JMP	$+2
	OR	AL,80h
	OUT	21h,AL
	;
	; Unhook Int 0Fh.
	;
	XOR	AX,AX
	MOV	ES,AX
	MOV	BX,4*0Fh
	MOV	AX,WORD PTR INT0FDEFAULT
	MOV	ES:[BX],AX
	MOV	AX,WORD PTR INT0FDEFAULT+2
	MOV	ES:[BX+2],AX
	STI
	MOV	INITIALIZED,0		; mark not initialized
	POP	DI			; restore DI,SI for C
	POP	SI
	RET
;
; DACSTART routine, to start playback from the beginning of the DMA buffer.
; Sets the DMA base and current address to 0 and enables DMA channel 1.
; Takes no parameters.  Uses Pascal calling conventions.
;
; Returns nothing.  Destroys AX.
;
	EVEN
DACSTART:
	CLI
	MOV	AL,5		; disable DMA channel 1
	OUT	0Ah,AL
	JMP	$+2
	MOV	AL,0FFh		; clear byte pointer flip/flop
	OUT	0Ch,AL
	JMP	$+2
	OUT	03h,AL		; program low word of count - count programmed
	JMP	$+2		;   is 0FFFFh for a 64k autotinitialize buffer
	OUT	03h,AL		; program high word
	JMP	$+2
	INC	AL		; set base address = 0
	OUT	02h,AL		;   program low word
	JMP	$+2
	OUT	02h,AL		;   program high word
	JMP	$+2
	INC	AL		; AL = 1
	OUT	0Ah,AL		; enable DMA channel 1
	STI
	RET
;
; DACSTOP routine, to stop or pause playback.  Disables DMA channel 1.  Takes
; no parameters.  Uses Pascal calling conventions.
;
; Returns nothing.  Destroys AX.
;
	EVEN
DACSTOP:
	MOV	AL,5		; disable DMA channel 1
	OUT	0Ah,AL
	RET
;
; DACRESUME routine, to resume playback at the current DMA buffer position.
; Enables DMA channel 1.  Takes no parameters.  Uses Pascal calling
; conventions.
;
; Returns nothing.  Destroys AX.
;
	EVEN
DACRESUME:
	MOV	AL,1		; enable DMA channel 1
	OUT	0Ah,AL
	RET
;
; DACSETVOL routine, to set the volume on the DAC.  Takes one parameter,
; the new volume, which is saved in the _volume variable.  Uses Pascal
; calling conventions.
;
; Returns nothing.  Destroys AX, CX, DX.
;
	EVEN
DACSETVOL:
	POP	CX		; CX is return address
	MOV	DX,DACBASE	; DX is volume port
	ADD	DX,3
	POP	AX		; AL = new volume
	MOV	_volume,AL	; save it
#IF M_I286
	ROR	AL,3		; put in most significant 3 bits
#ELSE
	ROR	AL,1
	ROR	AL,1
	ROR	AL,1
#ENDIF
	MOV	AH,AL		; most significant 3 bits of AH = volume
	CLI
	IN	AL,DX		; get DAC amplitude/frequency MSB
	AND	AL,1Fh		; mask out old volume
	OR	AL,AH		; put in new volume
	OUT	DX,AL		; write back DAC amplitude/frequency MSB
	STI
	JMP	CX		; jump to return address
;
; DACSETSPEED routine, to change the sampling rate.  Takes one parameter,
; the new sampling rate in Hertz, which is saved in the _samprate variable.
; Recomputes _clockspersam for the new rate.  Note:  This routine should
; only be called when sound playback is stopped, and if any sound has been
; queued for playback at the old rate, it needs to be discarded and recomputed
; at the new rate before playback resumes.  Uses Pascal calling conventions.
;
; Returns nothing.  Destroys AX, BX, CX, DX.
;
	EVEN
DACSETSPEED:
	POP	CX		; CX is return address
	POP	BX		; BX is the new sampling rate
	MOV	_samprate,BX	; save it
	;
	; Compute new DAC divider value.
	;
	MOV	AX,WORD PTR _clockspersec
	MOV	DX,WORD PTR _clockspersec+2
	DIV	BX
	SHR	BX,1
	CMP	BX,DX
	ADC	AX,0
	MOV	_clockspersam,AX ; save it
	MOV	DX,DACBASE	; get DAC amplitude/frequency port
	ADD	DX,2
	CLI
	OUT	DX,AL		; set low 8 bits of divider
	INC	DX		; get most significant byte
	IN	AL,DX
	AND	AL,0E0h		; keep the volume
	OR	AL,AH		; set high 4 bits of divider
	OUT	DX,AL
	STI
	JMP	CX		; jump to return address
;
; DACGETPART routine, to get the number of the 2k buffer segment where DMA
; is currently located.  Takes no parameters.  Uses Pascal calling
; conventions.
;
; Returns the segment number (0-31) in AX.  Destroys nothing.
;
	EVEN
DACGETPART:
	CLI
	MOV	AL,0FFh		; clear byte pointer flip/flop
	OUT	0Ch,AL
	JMP	$+2
	IN	AL,2		; read low byte of address (throw it away)
	JMP	$+2
	IN	AL,2		; read high byte of address
	STI
#IF M_I286
	SHR	AL,3
#ELSE
	SHR	AL,1
	SHR	AL,1
	SHR	AL,1
#ENDIF
	CBW
	RET
;
; DACGETLAST routine, to get the last output sample.  Takes no parameters.
; Uses Pascal calling conventions.
;
; Returns the sample (0-255) in AX.  Destroys DX.
;
	EVEN
DACGETLAST:
	MOV	DX,DACBASE	; DX addresses data port
	INC	DX
	IN	AL,DX		; get sample
	XOR	AH,AH
	RET
_TEXT	ENDS
	END
