; SETDAC.ASM
;
; Version 1.4 - August 29, 1996.  Modified to allow patching the BIOS
; parallel port list and enabling the joystick.  Added return codes.
; Changed the chip detection routine.
;
; Version 1.3 - July 25, 1994.  Modified to improve detection of the DAC,
; to make the base I/O port of the DAC a variable, and to display the port
; address to the user.
;
; This program can (1) set up the DAC to take direct writes of PCM sound
; data; (2) set up the DAC to enable the joystick; (3) modify the BIOS
; parallel port list in low memory to put the DAC in the list, and save the
; old parallel port value from the list in either the master environment
; or the BIOS inter-process communication area; or (4) restore the old
; parallel port value from the location where it was saved to the BIOS
; parallel port list.
;
; The command syntax is:
;
;     SETDAC [/Pn] [/U] [/J] [/I]
;
; If Setdac is invoked without parameters, it sets the DAC for direct
; (byte-by-byte) writes and displays the hexadecimal port address that
; programs should be set up to use to write to the DAC.
;
; If /Pn is specified, where n is 1, 2 or 3, Setdac will additionally
; patch the BIOS parallel port list at 40:8h to make the base port address
; for LPTn: be the DAC's Direct Read/Write Register (the port displayed
; to the user, and where sound data should go).  The old base port address
; for LPTn: will be stored as a variable in the master environment, or if
; there is no room there (or if /I is specified), in the BIOS inter-process
; communication area at 40:F0h.  /Pn is incompatible with /U or /J.
;
; If /U is specified, Setdac will recover the saved LPTn: port address
; from the master environment (unsetting the variable) or from the BIOS
; inter-process communication area, where it was stored by a previous
; invocation of SETDAC /Pn, and will restore the BIOS parallel port list
; to its default state.  /U is incompatible with /Pn.
;
; If /J is specified, Setdac will set up the DAC to enable the joystick
; rather than to enable direct writes.  /J is incompatible with /Pn.
;
; If /I is specified, the behavior of Setdac with the /Pn and /U options
; is modified, so that the program will not attempt to locate the master
; environment, but will instead (with /Pn) save the default LPTn: port
; address in the inter-process communication area or (with /U) attempt to
; restore the default LPTn: port from the inter-process communication area.
; /I requires /Pn or /U.
;
; Setdac returns the following codes:
;     0   if successful
;     1   if Tandy DAC not found
;     2   if command-line syntax is invalid
;     3   if /U failed to restore the parallel port list
;

	JMP	START

;
; Data.
;
SOUNDHALTED	DB	0	; used when initializing the sound chip
INT15DEFAULT	DD	0	; default Int 15h vector, for initialization
INT0FDEFAULT	DD	0	; default Int 0Fh vector, for finalization
PORTBASE	DW	0	; base I/O port of DAC
OLDPORT		DW	0	; default LPTn: port - used by /U routines
OLDPORTNUM	DB	0	; n in LPTn:, 0-2 == LPT1: to LPT3:
		;
		; Error message if no DAC.
		;
NODACMSG	DB	"Tandy DAC not detected.",0Dh,0Ah,"$"
		;
		; Message to tell the user the DAC port.
		;
PORTMSG		DB	"Tandy DAC at port "
PORTSTR		DB	"xxxxh",0Dh,0Ah,"$"
SETDIRECT	DB	1	; 1 if setting DAC for direct mode
SETLPT		DB	0	; 1 if /Pn specified
SETJOY		DB	0	; 1 if /J specified
UNSETLPT	DB	0	; 1 if /U specified
NOMASTER	DB	0	; 1 if /I specified (do not use master env)
		;
		; Environment string for default LPTn: port.
		;
ENVSTRING	DB	"_P"
ENVDIGIT	DB	"n="
ENVPORT		DB	"xxxx"
ENVSTRINGLEN	EQU	$-OFFSET ENVSTRING
		;
		; Command-line syntax message.
		;
SYNTAXMSG	DB	"Syntax:",0Dh,0Ah,0Dh,0Ah
		DB	"    SETDAC [/Pn] [/U] [/J] [/I]",0Dh,0Ah,0Dh,0Ah
		DB	"where:",0Dh,0Ah
		DB	"    /Pn    set Tandy DAC for direct mode and",0Dh,0Ah
		DB	"           set port LPTn: to be the DAC",0Dh,0Ah
		DB	"    /U     restore port LPTn: after previous /Pn"
		DB	0Dh,0Ah
		DB	"    /J     set Tandy DAC to enable joystick",0Dh,0Ah
		DB	"    /I     use IPC area with /Pn and /U",0Dh,0Ah
		DB	"See TSPAK.DOC for details.",0Dh,0Ah,"$"
		;
		; Message displayed if enabling the joystick.
		;
JOYMSG		DB	"Tandy joystick enabled.",0Dh,0Ah,"$"
		;
		; Message displayed if the LPTn: port previously saved with
		; the /Pn option was successfully restored to the BIOS port
		; list.
		;
UNSETMSG_GOOD	DB	"LPT: port restored.",0Dh,0Ah,"$"
UNSETMSG_BAD	DB	"Saved LPT: port not found, couldn't restore.",
		DB	0Dh,0Ah,"$"
		;
		; Message displayed if the LPTn: port was successfully set
		; to the DAC in the BIOS port list.
		;
SETLPTMSG	DB	"LPT: port set for DAC.",0Dh,0Ah,"$"

;
; Replacement Int 15h handler, for BIOS callout.
;
INT15HDLR:
	CMP	AX,91FBh
	JE	>L0
	JMP	DWORD PTR CS:INT15DEFAULT
L0:	MOV	CS:SOUNDHALTED,1
	IRET

;
; Routine to check whether there is BIOS support for Tandy sound functions.  
; Sets carry if no Tandy DAC found; otherwise, carry is cleared and the base
; I/O port of the DAC is returned in AX.  (Modified, 7/13/1994.)
;
CHKDAC:
	PUSH	CX
	PUSH	DX
	;
	; Check for PCMCIA Socket Services.
	;
	MOV	AX,8003h
	XOR	CX,CX
	INT	1Ah
	CMP	CX,5353h
	JE	CHKDAC_NODAC
	;
	; Check for Tandy DAC.
	;
	MOV	AX,8100h
	INT	1Ah
	CMP	AX,8000h
	JAE	CHKDAC_NODAC
	;
	; DAC found.  Base port returned in AX.  Verify the presence of a
	; read/write port at (base+2).
	;
	MOV	DX,AX		; DX addresses (base+2)
	INC	DX
	INC	DX
	MOV	CL,40h		; set bit 6 of CL (assume port present)
	CLI			; disable interrupts
	IN	AL,DX		; save value from port in CH
	MOV	CH,AL
	MOV	AL,0FFh		; set all bits of port
	OUT	DX,AL
	MOV	AL,0		; clear all bits
	OUT	DX,AL
	IN	AL,DX		; read back and test
	OR	AL,AL
	LAHF
	AND	CL,AH		; bit 6 of CL still set if successful
	MOV	AL,0FFh		; set all bits
	OUT	DX,AL
	IN	AL,DX		; read back and test
	NOT	AL
	OR	AL,AL
	LAHF
	AND	CL,AH		; bit 6 of CL still set if successful
	MOV	AL,CH		; write back saved value
	OUT	DX,AL
	STI			; enable interrupts
	TEST	CL,40h		; if bit 6 of CL still set, port found
	JZ	CHKDAC_NODAC
	;
	; Read/write port found.  Return port base in AX.
	;
	MOV	AX,DX
	DEC	AX		; decrement twice to get base port back
	DEC	AX
	CLC
	JMP	CHKDAC_END
CHKDAC_NODAC:
	STC
CHKDAC_END:
	POP	DX
	POP	CX
	RET

;
; Routine to convert a number 0-15 to a hex digit.  The number is in AL;
; the digit is returned in AL.
;
HEXDIGIT:
	CMP	AL,9
	JA	HEXDIGIT_LETTER
	ADD	AL,'0'
	JMP	HEXDIGIT_DONE
HEXDIGIT_LETTER:
	ADD	AL,'A'-10
HEXDIGIT_DONE:
	RET

;
; Routine to convert a word integer to a hex string.  The integer is passed
; in AX.  ES:DI addresses the location where the string should be stored.
;
HEXWORD:
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	DX
	PUSH	DI
	MOV	DX,AX
	MOV	BL,12
	MOV	CX,4
	CLD
HEXWORD_LOOP:
	PUSH	CX
	MOV	AX,DX
	MOV	CL,BL
	SHR	AX,CL
	AND	AL,0Fh
	CALL	HEXDIGIT
	STOSB
	SUB	BL,4
	POP	CX
	LOOP	HEXWORD_LOOP
	POP	DI
	POP	DX
	POP	CX
	POP	BX
	POP	AX
	RET

;
; Subroutine, takes pointer to string in DS:SI and length of string in CX,
; skips over blanks and tabs, returns pointer to first nonblank character
; in the string in DS:SI, length of remaining string in CX.  If end of
; string is reached, return pointer to end of string in DS:SI, zero in CX.
; Assumes the direction flag is clear.
;
SKIPBLANKS:
	PUSH	AX
SKIPBLANKS_LOOP:
	JCXZ	SKIPBLANKS_END
	LODSB
	DEC	CX
	CMP	AL,9
	JE	SKIPBLANKS_LOOP
	CMP	AL,20h
	JE	SKIPBLANKS_LOOP
	DEC	SI
	INC	CX
SKIPBLANKS_END:
	POP	AX
	RET

;
; Routine to save the current LPTn: base port address as a hex string in
; the string to be placed in the environment.  AL is the port, 0 for LPT1:,
; 1 for LPT2:, 2 for LPT3:.
;
SAVELPT:
	PUSH	AX
	PUSH	DI
	PUSH	ES
	MOV	DI,40h		; ES:DI addresses LPT: port list
	MOV	ES,DI
	MOV	DI,8
	MOV	AH,0		; ES:DI addresses current base port
	SHL	AX,1
	ADD	DI,AX
	MOV	AX,ES:[DI]	; current base port in AX
	MOV	DI,DS		; ES:DI addresses hex string
	MOV	ES,DI
	MOV	DI,OFFSET ENVPORT
	CALL	HEXWORD		; convert and save the port
	POP	ES
	POP	DI
	POP	AX
	RET

;
; Routine to parse the command line.  This routine checks for the command
; line switches /Pn, /U, /J, and /I, setting variables SETDIRECT, SETLPT,
; SETJOY, UNSETLPT, and NOMASTER according to what it finds.  Returns with
; carry clear if the command line is valid, set if not.
;
PARSECMD:
	PUSH	AX
	PUSH	CX
	PUSH	SI
	;
	; Get pointer to the command line in DS:SI, length in CX.  Clear
	; the direction flag.
	;
	MOV	CL,[80h]
	MOV	CH,0
	MOV	SI,81h
	CLD
	;
	; Loop over the command-line parameters.
	;
PARSECMD_LOOP:
	CALL	SKIPBLANKS
	JCXZ	PARSECMD_CHECK		; exit loop if no more
	;
	; Command-line argument found.  Check if it's a switch.
	;
	LODSB
	CMP	AL,'/'
	JNE	PARSECMD_INVALID
	;
	; Go to the letter following the slash.  It's an error if there is
	; none.
	;
	DEC	CX
	JZ	PARSECMD_INVALID
	;
	; Get the letter and convert it to uppercase.
	;
	LODSB
	DEC	CX
	AND	AL,0DFh
	;
	; Is it a "U"?
	;
	CMP	AL,'U'
	JNE	PARSECMD_NOTU
	MOV	UNSETLPT,1
	MOV	SETDIRECT,0
	JMP	PARSECMD_LOOP
	;
	; Is it a "J"?
	;
PARSECMD_NOTU:
	CMP	AL,'J'
	JNE	PARSECMD_NOTJ
	MOV	SETJOY,1
	MOV	SETDIRECT,0
	JMP	PARSECMD_LOOP
	;
	; Is it an "I"?
	;
PARSECMD_NOTJ:
	CMP	AL,'I'
	JNE	PARSECMD_NOTI
	MOV	NOMASTER,1
	JMP	PARSECMD_LOOP
	;
	; Is it a "P"?  If not, it's invalid.
	;
PARSECMD_NOTI:
	CMP	AL,'P'
	JNE	PARSECMD_INVALID
	;
	; Get the next character (there better be one) and save it in the
	; string to be placed in the environment.
	;
	JCXZ	PARSECMD_INVALID
	LODSB
	DEC	CX
	MOV	ENVDIGIT,AL
	;
	; Convert the digit 1-3 to a number 0-2.  Save the current LPTn:
	; port in the string to be placed in the environment.
	;
	SUB	AL,'1'
	JB	PARSECMD_INVALID
	CMP	AL,2
	JA	PARSECMD_INVALID
	CALL	SAVELPT		; this clears DF again, which is harmless
	MOV	SETLPT,1
	JMP	PARSECMD_LOOP
	;
	; We got the command-line switches, and they seem OK.  Verify that
	; no incompatible switches were chosen (see top of file).  To
	; simplify this, I take the various switches selected and combine
	; them into an integer.
	;
PARSECMD_CHECK:
	MOV	AL,SETLPT
	SHL	AL,1
	OR	AL,UNSETLPT
	SHL	AL,1
	OR	AL,SETJOY
	SHL	AL,1
	OR	AL,NOMASTER
	CMP	AL,9
	JA	PARSECMD_INVALID
	CMP	AL,3
	JE	PARSECMD_INVALID
	CMP	AL,1
	JE	PARSECMD_INVALID
	;
	; The command line is valid.  Clear carry and return.
	;
	CLC
	JMP	PARSECMD_DONE
	;
	; The command line is invalid.  Set carry and return.
	;
PARSECMD_INVALID:
	STC
PARSECMD_DONE:
	POP	SI
	POP	CX
	POP	AX
	RET

;
; Routine to initialize/finalize the sound chip.
;
INITDAC:
	PUSH	AX
	PUSH	BX
	PUSH	DX
	PUSH	ES
	;
	; Use the replacement Int 15h handler now.
	;
	MOV	AX,3515h
	INT	21h
	MOV	WORD PTR INT15DEFAULT,BX
	MOV	WORD PTR INT15DEFAULT+2,ES
	MOV	DX,OFFSET INT15HDLR
	MOV	AX,2515h
	INT	21h
	;
	; Initialize/finalize the sound chip.
	;
	MOV	SOUNDHALTED,0
L0:	MOV	AH,84h
	INT	1Ah
L1:	MOV	AH,81h
	INT	1Ah
	JC	L1
	CMP	SOUNDHALTED,0
	JE	L0
	;
	; Restore Int 15h vector to default.
	;
	PUSH	DS
	MOV	AX,2515h
	MOV	DX,WORD PTR INT15DEFAULT
	MOV	DS,WORD PTR INT15DEFAULT+2
	INT	21h
	POP	DS
	POP	ES
	POP	DX
	POP	BX
	POP	AX
	RET

;
; Interrupt 0Fh (DMA EOP) interrupt handler.  This routine just clears the
; interrupt bit at the sound chip and returns.
;
INT0FHDLR:
	CLI
	PUSH	AX
	PUSH	DX
        ;
        ; Read the interrupt controller's in-service register to see if an
        ; IRQ 7 has in fact occurred.  If not (electrical noise), just restore
        ; registers and return.
        ;
        MOV	AL,0Bh
        OUT	20h,AL
        JMP	$+2
        IN	AL,20h
        TEST	AL,80h
        JZ	INT0FHDLR_RESTORE
	;
	; Check if it was a DAC interrupt.  If not, just issue EOI and
	; return.
	;
	MOV	DX,CS:PORTBASE
	IN	AL,DX
	TEST	AL,8
	JZ	INT0FHDLR_EOI
	;
	; It was our interrupt.  Clear the DMA interrupt at the sound chip.
	;
	AND	AL,0F7h
	OUT	DX,AL
	OR	AL,8
	OUT	DX,AL
INT0FHDLR_EOI:
	MOV	AL,20h
	OUT	20h,AL
INT0FHDLR_RESTORE:
	POP	DX
	POP	AX
	IRET			; will set the interrupt bit

;
; Routine to set the DAC for joystick mode.  We're going to try not to make
; any assumptions about what mode we're in to begin with.
;
SETJOYPROC:
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	DX
	PUSH	ES
	;
	; Disable DMA channel 1.
	;
	CLI
	MOV	AL,5
	OUT	0Ah,AL
	;
	; Stop recording at the sound chip (we assume the DAC is in record
	; mode, since that is the hardest one to get out of).
	;
	MOV	DX,PORTBASE
	IN	AL,DX
	AND	AL,0FCh		; switch to playback mode temporarily
	OR	AL,3
	OUT	DX,AL
	INC	DX		; read a sample to clear the approximator
	IN	AL,DX
	DEC	DX		; set DAC for joystick mode, DMA interrupt
	MOV	AL,10h		;   enabled still, DMA interrupt cleared
	OUT	DX,AL		;   DMA disabled
	MOV	AL,0		; set DAC for joystick mode, DMA interrupt
	OUT	DX,AL		;   disabled, DMA interrupt cleared, DMA
	INC	DX		;   disabled
	IN	AL,DX		; read another sample just to make sure
	;
	; Hook Int 0Fh.
	;
	XOR	AX,AX
	MOV	ES,AX
	MOV	BX,4*0Fh
	MOV	AX,ES:[BX]
	MOV	WORD PTR INT0FDEFAULT,AX
	MOV	AX,ES:[BX+2]
	MOV	WORD PTR INT0FDEFAULT+2,AX
	MOV	WORD PTR ES:[BX],OFFSET INT0FHDLR
	MOV	ES:[BX+2],CS
	;
	; Enable IRQ7 at the interrupt controller.
	;
	IN	AL,21h
	JMP	$+2
	AND	AL,7Fh
	OUT	21h,AL
	STI
	;
	; Disabling DMA interrupts at the sound chip may cause one.  Wait
	; for the interrupt, but not forever, since the old version of the
	; chip won't produce one.
	;
	MOV	CX,2000
	LOOP	$
	;
	; The DAC is in joystick mode.  Disable further interrupts on IRQ 7.
	;
	CLI
	IN	AL,21h
	JMP	$+2
	OR	AL,80h
	OUT	21h,AL
	JMP	$+2
	;
	; Unhook Int 0Fh.  ES:BX -> Int 0Fh vector from above.
	;
	MOV	AX,WORD PTR INT0FDEFAULT
	MOV	ES:[BX],AX
	MOV	AX,WORD PTR INT0FDEFAULT+2
	MOV	ES:[BX+2],AX
	STI
	POP	ES
	POP	DX
	POP	CX
	POP	BX
	POP	AX
	RET

;
; Routine to set the DAC for direct mode.
;
DIRECTSET:
	PUSH	AX
	PUSH	DX
	MOV	DX,PORTBASE
	CLI
	IN	AL,DX		; set base port (DAC Control Register)
	JMP	$+2
	AND	AL,0E3h
	OR	AL,3
	OUT	DX,AL
	JMP	$+2
	INC	DX		; set volume at (base+2), (base+3)
	INC	DX
	MOV	AL,0
	OUT	DX,AL
	JMP	$+2
	INC	DX
	MOV	AL,0E0h
	OUT	DX,AL
	JMP	$+2
	STI
	POP	DX
	POP	AX
	RET

;
; Routine to get the segment address of the master environment, from the
; book, _Undocumented DOS_, 2nd edition, by Shulman et al.  This routine
; can fail under some circumstances - there is no completely reliable way
; to do this.  The master environment segment is returned in AX, its size
; in bytes in BX.
;
GETMASTER:
	PUSH	ES
	MOV	AX,352Eh	; get pointer to COMMAND.COM
	INT	21h
	MOV	AX,ES:[2Ch]	; get its environment segment in AX
	MOV	BX,AX		; ES -> MCB of master environment
	DEC	BX
	MOV	ES,BX
	MOV	BX,ES:[3]	; get size of master environment in BX
	SHL	BX,1		; convert to bytes
	SHL	BX,1
	SHL	BX,1
	SHL	BX,1
	POP	ES
	RET

;
; Routine to set the LPTn: address in the BIOS port list and save the old
; value - somewhere, hopefully somewhere safe.
;
SETLPTPROC:
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	SI
	PUSH	DI
	PUSH	ES
	CLD		; we're going to need this whatever we do
	;
	; If /I was set on the command line, skip trying to use the master
	; environment.
	;
	CMP	NOMASTER,1
	JE	SETLPTPROC_IPC
	;
	; ES:DI -> master environment, BX = number of bytes allocated to it.
	;
	CALL	GETMASTER
	MOV	ES,AX
	XOR	DI,DI
	;
	; Find the end of the master environment.
	;
	MOV	AL,0
	MOV	CX,8000h		; max environment size is 32k
SETLPTPROC_SCANLP:
	CMP	BYTE PTR ES:[DI],0
	JE	SETLPTPROC_ENDVARS
	REPNE	SCASB
	JCXZ	SETLPTPROC_IPC		; should never happen
	JMP	SETLPTPROC_SCANLP
	;
	; ES:DI is pointing to where our string should go; the only
	; question is whether there is enough room.  Figure out how big
	; the master environment would become if we added the string.
	;
SETLPTPROC_ENDVARS:
	MOV	AX,DI
	ADD	AX,ENVSTRINGLEN
	INC	AX		; for null byte at end of string
	INC	AX		; for second null at end of environment
	CMP	AX,BX		; too big? use the IPC
	JA	SETLPTPROC_IPC
	;
	; All right, then, we should be OK - so copy our string.
	;
	MOV	SI,OFFSET ENVSTRING	; DS:SI -> string to save
	MOV	CX,ENVSTRINGLEN		; CX = length of string
	REP	MOVSB			; zap!
	MOV	AL,0			; store the two nulls too
	STOSB
	STOSB
	JMP	SETLPTPROC_SAVED
	;
	; The master environment could not or should not be used.  Use the
	; IPC area.  This is simpler, but less safe.
	;
SETLPTPROC_IPC:
	MOV	AX,40h			; ES:DI -> IPC area
	MOV	ES,AX
	MOV	DI,0F0h
	MOV	SI,OFFSET ENVSTRING	; DS:SI -> string to save
	MOV	CX,ENVSTRINGLEN		; CX = length of string
	REP	MOVSB
	MOV	AL,0			; append a null for convenience
	STOSB
	;
	; One way or another, the old LPTn: port address has been saved.
	; Pray for its continued good health :).  Time to do the dirty
	; deed with the BIOS parallel port list.
	;
SETLPTPROC_SAVED:
	MOV	AX,40h
	MOV	ES,AX
	MOV	AL,ENVDIGIT
	SUB	AL,'1'
	MOV	AH,0
	SHL	AX,1
	MOV	BX,AX
	MOV	AX,PORTBASE
	INC	AX
	MOV	ES:[BX+8],AX
	POP	ES
	POP	DI
	POP	SI
	POP	CX
	POP	BX
	POP	AX
	RET

;
; Routine to determine whether a character is a hex digit or not.  We
; only allow uppercase letters here, since the point is to determine whether
; a hex number was saved by this program or not.  Returns carry clear if
; the character is a numeral 0-9 or a letter A-F.  The character is passed
; in AL.
;
ISHEXDIGIT:
	CMP	AL,'0'
	JB	ISHEXDIGIT_NO
	CMP	AL,'9'
	JBE	ISHEXDIGIT_YES
	CMP	AL,'A'
	JB	ISHEXDIGIT_NO
	CMP	AL,'F'
	JBE	ISHEXDIGIT_YES
ISHEXDIGIT_NO:
	STC
	RET
ISHEXDIGIT_YES:
	CLC
	RET

;
; Routine to determine whether a given environment string is a parallel
; port address saved by SETDAC /Pn.  ES:DI -> string.  Assumes DF clear
; for incrementing.  Returns carry clear if it is, set if not.
;
ISITOURS:
	PUSH	AX
	PUSH	CX
	PUSH	SI
	PUSH	DS
	MOV	SI,ES		; DS:SI -> string
	MOV	DS,SI
	MOV	SI,DI
	;
	; Check whether this is a variable of the form "_Pn=", where
	; n=1,2,3.
	;
	LODSB
	CMP	AL,'_'
	JNE	ISITOURS_NO
	LODSB
	CMP	AL,'P'
	JNE	ISITOURS_NO
	LODSB
	CMP	AL,'1'
	JB	ISITOURS_NO
	CMP	AL,'3'
	JA	ISITOURS_NO
	LODSB
	CMP	AL,'='
	JNE	ISITOURS_NO
	;
	; Check whether the next four characters are valid uppercase
	; hexadecimal digits.
	;
	MOV	CX,4
ISITOURS_DIGLOOP:
	LODSB
	CALL	ISHEXDIGIT
	JC	ISITOURS_NO
	LOOP	ISITOURS_DIGLOOP
	;
	; Check for the terminating null.
	;
	LODSB
	OR	AL,AL
	JNZ	ISITOURS_NO
	;
	; It's our variable, all right.
	;
	CLC
	JMP	ISITOURS_DONE
	;
	; Nope, not our variable.
	;
ISITOURS_NO:
	STC
ISITOURS_DONE:
	POP	DS
	POP	SI
	POP	CX
	POP	AX
	RET

;
; Routine to convert a hex digit '0'-'9', 'A'-'F' to a number 0-15.  The
; digit is passed in AL, and the number is returned there.
;
DIGITHEX:
	CMP	AL,'A'
	JAE	DIGITHEX_LETTER
	SUB	AL,'0'
	RET
DIGITHEX_LETTER:
	SUB	AL,'A'-10
	RET

;
; Routine to convert a string of 4 hex digits to a word integer value.
; Digits '0'-'9' and 'A'-'F' are allowed, no lower case.  The string is
; assumed to be a valid hex number exactly 4 digits long.  The address of
; the string is passed in ES:DI, and the integer is returned in AX.
;
HEXTOWORD:
	PUSH	BX
	PUSH	CX
	MOV	CL,4
	MOV	AL,ES:[DI]
	CALL	DIGITHEX
	MOV	BH,AL
	SHL	BH,CL
	MOV	AL,ES:[DI+1]
	CALL	DIGITHEX
	OR	BH,AL
	MOV	AL,ES:[DI+2]
	CALL	DIGITHEX
	MOV	BL,AL
	SHL	BL,CL
	MOV	AL,ES:[DI+3]
	CALL	DIGITHEX
	OR	BL,AL
	MOV	AX,BX
	POP	CX
	POP	BX
	RET

;
; Routine to retrieve the default LPTn: port from wherever the previous
; call to SETDAC /Pn put it and restore it to the BIOS parallel port list.
; Returns carry clear if successful, set if not.
;
UNSETPROC:
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	SI
	PUSH	DI
	PUSH	ES
	CLD		; we're going to need this whatever we do
	;
	; If /I was set on the command line, skip trying to use the master
	; environment.  There really isn't any need for /I with /U, but
	; I'm allowing it for symmetry's sake with /Pn.
	;
	CMP	NOMASTER,1
	JE	UNSETPROC_IPC
	;
	; ES:DI -> master environment, BX = number of bytes allocated to it.
	;
	CALL	GETMASTER
	MOV	ES,AX
	XOR	DI,DI
	;
	; Search the master environment for the string.
	;
	MOV	AL,0
	MOV	CX,8000h		; max environment size is 32k
	CLD
UNSETPROC_SCANLP:
	CMP	BYTE PTR ES:[DI],0
	JE	UNSETPROC_IPC
	CALL	ISITOURS
	JNC	UNSETPROC_FOUND
	REPNE	SCASB
	JCXZ	UNSETPROC_IPC		; should never happen
	JMP	UNSETPROC_SCANLP
	;
	; Our variable was found in the master environment, and ES:DI
	; addresses it.  Get the port address out of it - and which LPT:
	; port was it again?
	;
UNSETPROC_FOUND:
	ADD	DI,4		; point ES:DI at the hex port address
	CALL	HEXTOWORD
	SUB	DI,4		; get DI back to where it was
	MOV	OLDPORT,AX
	MOV	AL,ES:[DI+2]
	SUB	AL,'1'
	MOV	OLDPORTNUM,AL
	;
	; Now we want to remove our variable.  That is a simple (?) matter
	; of moving down all the stuff that was above it in the environment.
	;
	MOV	SI,DI	; DS:SI -> data to move
	ADD	SI,9	; delete 8 bytes for the var, plus the null byte
	MOV	AX,ES	; DS doesn't address local data any more!
	MOV	DS,AX
	MOV	CX,BX	; CX = number of bytes in env segment above the var
	SUB	CX,DI
	SUB	CX,9
	REP	MOVSB	; var is gone
	MOV	AX,CS	; get DS back, we need it!
	MOV	DS,AX
	JMP	UNSETPROC_PATCH
	;
	; Either we didn't find it in the master environment, or we're not
	; supposed to look there.  Check the IPC for the needed variable.
	;
UNSETPROC_IPC:
	MOV	AX,40h
	MOV	ES,AX
	MOV	DI,0F0h
	CALL	ISITOURS
	JC	UNSETPROC_NOTFOUND
	;
	; We found it here in the IPC.  Again, we need to get the port
	; number and address out of the string.
	;
	ADD	DI,4		; point ES:DI at the hex port address
	CALL	HEXTOWORD
	SUB	DI,4		; get DI back to where it was
	MOV	OLDPORT,AX
	MOV	AL,ES:[DI+2]
	SUB	AL,'1'
	MOV	OLDPORTNUM,AL
	;
	; Let's wipe the IPC for safety's sake, so we don't find the
	; variable again in a later invocation.
	;
	XOR	AX,AX
	MOV	CX,8
	REP	STOSW
	JMP	UNSETPROC_PATCH
	;
	; We looked everywhere and didn't find it ... :(
	;
UNSETPROC_NOTFOUND:
	STC
	JMP	UNSETPROC_DONE
	;
	; The old LPTn: port is in OLDPORT, and n is OLDPORTNUM.  Fix the
	; BIOS parallel port list to put it back the way you found it.
	;
UNSETPROC_PATCH:
	MOV	AX,40h		; ES:BX -> place to restore in BIOS list
	MOV	ES,AX
	MOV	BL,OLDPORTNUM
	SHL	BL,1
	MOV	BH,0
	MOV	AX,OLDPORT
	MOV	ES:[BX+8],AX
	CLC
UNSETPROC_DONE:
	POP	ES
	POP	DI
	POP	SI
	POP	CX
	POP	BX
	POP	AX
	RET

;
; Main program.
;
START:
	;
	; Check whether the needed sound circuitry is present; display error
	; message and halt if not.
	;
	CALL	CHKDAC
	JNC	DACFOUND
	MOV	DX,OFFSET NODACMSG
	MOV	AH,9
	INT	21h
	MOV	AL,1
	JMP	TERMINATE
	;
	; Save the base I/O port returned.  Get the port address of the
	; direct read/write register and convert it to a string for display
	; to the user.
	;
DACFOUND:
	MOV	PORTBASE,AX
	INC	AX
	MOV	DI,OFFSET PORTSTR
	CALL	HEXWORD
	;
	; Parse the command line.  Display error message and halt if
	; invalid syntax.
	;
	CALL	PARSECMD
	JNC	COMMANDOK
	MOV	DX,OFFSET SYNTAXMSG
	MOV	AH,9
	INT	21h
	MOV	AL,2
	JMP	TERMINATE
	;
	; If setting the DAC to enable the joystick, do so.
	;
COMMANDOK:
	CMP	SETJOY,1
	JNE	DOUNSET
	CALL	SETJOYPROC
	;
	; Tell the user what you did.
	;
	MOV	DX,OFFSET JOYMSG
	MOV	AH,9
	INT	21h
	;
	; If restoring the LPTn: address in the BIOS port list, do so.
	;
DOUNSET:
	CMP	UNSETLPT,1
	JNE	DOSETLPT
	CALL	UNSETPROC
	JC	DOUNSET_BAD
	MOV	DX,OFFSET UNSETMSG_GOOD
	MOV	BL,0			; return code 0 = OK
	JMP	DOUNSET_MSG
DOUNSET_BAD:
	MOV	DX,OFFSET UNSETMSG_BAD
	MOV	BL,3			; return code 3 = can't unset
	;
	; Tell the user whether it worked or not.
	;
DOUNSET_MSG:
	MOV	AH,9
	INT	21h
	MOV	AL,BL			; get return code in AL
	JMP	TERMINATE
	;
	; If setting the LPTn: address to the DAC address, do so.
	;
DOSETLPT:
	CMP	SETLPT,1
	JNE	DOSETDIRECT
	CALL	SETLPTPROC
	;
	; Tell the user what you did.
	;
	MOV	DX,OFFSET SETLPTMSG
	MOV	AH,9
	INT	21h
	;
	; If setting the DAC for direct mode output, do so.
	;
DOSETDIRECT:
	CMP	SETDIRECT,1
	JNE	ALLDONE
	CALL	INITDAC
	CALL	DIRECTSET
	;
	; Display the output port (base+1) to the user.
	;
	MOV	DX,OFFSET PORTMSG
	MOV	AH,9
	INT	21h
	;
	; Return code 0 = success.
	;
ALLDONE:
	MOV	AL,0
	;
	; Terminate.
	;
TERMINATE:
	MOV	AH,4Ch
	INT	21h
