/*
    Wav3voi.c

    Routines for reading in a .wav file, for program Play3voi.exe.
*/

#include <stdio.h>      /* Standard I/O functions header file */
#include <string.h>     /* Standard string functions header file */
#include <dos.h>        /* Standard DOS functions header file */
#include "play3voi.h"   /* Global constants and variables */
#include "wav3voi.h"    /* Routines for reading in a .wav file */


/*************************************************************************/
/*                                                                       */
/* Macros.                                                               */
/*                                                                       */
/*************************************************************************/

    /* Format tag for Microsoft PCM .wav files. */
#define WAVE_FORMAT_PCM         (0x0001)

    /* Size of the file read buffer. */
#define READBUFSIZ 4096

    /* TRUE and FALSE */
#ifndef TRUE
#define TRUE    1
#endif
#ifndef FALSE
#define FALSE   0
#endif


/*************************************************************************/
/*                                                                       */
/* Types.                                                                */
/*                                                                       */
/*************************************************************************/

    /* RIFF WAVE header structure. */
struct riffhdr
{
    char rifftag[4];        /* four letters "RIFF" */
    unsigned long rifflen;  /* length of RIFF file - 8 */
    char wavetag[4];        /* four letters "WAVE" */
};

    /* Chunk header structure. */
struct chunkhdr
{
    char tag[4];            /* four-letter tag for the chunk */
    unsigned long chunklen; /* length of the chunk in bytes */
};

    /* Format chunk for PCM WAVE file. */
struct pcmfmt
{
        /* Format type tag.  For Microsoft PCM .wav files, this is equal 
           to WAVE_FORMAT_PCM (see above). */
    unsigned wFormatTag;
    unsigned nChannels;             /* number of channels */
    unsigned long nSamplesPerSec;   /* sampling rate in Hz */
    unsigned long nAvgBytesPerSec;  /* bytes per second in file */
        /* Data block alignment.  For PCM files, this is not important, and 
           in some files it is set to 0.  Normally, it will be equal to the 
           number of bytes per sample. */
    unsigned nBlockAlign;
        /* Bits per sample of mono data.  Normally, 8 or 16. */
    unsigned wBitsPerSample;
};


/*************************************************************************/
/*                                                                       */
/* Global variables.                                                     */
/*                                                                       */
/*************************************************************************/

    /* Buffer for reading from the file. */
static char ReadBuffer[READBUFSIZ];

static unsigned long FileSize;  /* size of the .wav file in bytes */
static unsigned NChannels;      /* number of channels, 1 or 2 */
static unsigned BitsPerSample;  /* number of bits per sample, 8 or 16 */
    /* Length of the data chunk. */
static unsigned long DataLen;
    /* Pointer to function used to convert file samples to 8-bit mono. */
static void (*Convert)( unsigned nsamples );
    /* The ratio of samples read from the file to samples played, after 
       sampling rate adjustment. */
static unsigned Multiple;
    /* The number of bytes read from the file per sample played, after 
       sampling rate adjustment, and taking into account stereo and 16-bit 
       samples. */
static unsigned BytesPerSam;
    /* The total number of playable samples in the file, after sampling 
       rate adjustment, and the total number remaining to be loaded. */
static unsigned long TotalSamples;
static unsigned long TotalLeft;
    /* The maximum number of playable samples that will fit in the file 
       read buffer, based on BytesPerSam (above) and READBUFSIZ. */
static unsigned ReadMax;
    /* Pointer to function used to discard samples not played, when the 
       sampling rate has been adjusted downward. */
static void (*Multiple_Adjust)( unsigned nsamples );


/*************************************************************************/
/*                                                                       */
/* Function prototypes.                                                  */
/*                                                                       */
/*************************************************************************/

static int Read_Wave_Header( void );
static void BUM_2_BUM( unsigned nsamples );
static void BUS_2_BUM( unsigned nsamples );
static void WSM_2_BUM( unsigned nsamples );
static void WSS_2_BUM( unsigned nsamples );
static void huge *Add_To_Huge( void huge *hptr, unsigned increment );
static void Adjust_Multiple_8bit( unsigned samples );
static void Adjust_Multiple_16bit( unsigned nsamples );
static void Adjust_Multiple_32bit( unsigned nsamples );
static int Setup_For_Read( void );


/*************************************************************************/
/*                                                                       */
/* Read_Wave_Header() function.  This function reads in the .wav header  */
/* and sets global variables according to what it finds.  If successful, */
/* it leaves the file pointer at the start of the sound data and returns */
/* 0.  If the header is invalid, it returns -1.                          */
/*                                                                       */
/*************************************************************************/

static int Read_Wave_Header( void )
{
    struct riffhdr fileheader;      /* RIFF WAVE header */
    struct chunkhdr chunkheader;    /* WAVE chunk header */
    struct pcmfmt format;           /* format chunk */
    int fmtfound;       /* TRUE when "fmt " chunk found */
    int datafound;      /* TRUE when "data" chunk found */
    int i;              /* for looping over characters */

    /* Read the RIFF WAVE header. */
    if ( fread( &fileheader, 1, sizeof( struct riffhdr ), FileHandle )
            != sizeof( struct riffhdr ) )
        return( -1 );

    /* Verify the RIFF WAVE header. */
    if ( strncmp( fileheader.rifftag, "RIFF", 4 ) != 0 ||
            strncmp( fileheader.wavetag, "WAVE", 4 ) != 0 )
        return( -1 );

    /* Save the file size from the header, in case it turns out to be 
       useful. */
    FileSize = fileheader.rifflen + 8L;

    /* Read chunks from the file until we get to the data chunk, or until 
       an error is detected. */
    fmtfound = FALSE;
    datafound = FALSE;
    while ( datafound == FALSE )
    {
        /* Get the chunk header. */
        if ( fread( &chunkheader, 1, sizeof( struct chunkhdr ), FileHandle )
                != sizeof( struct chunkhdr ) )
            return( -1 );

        /* If it's a format chunk, get the format. */
        if ( strncmp( chunkheader.tag, "fmt ", 4 ) == 0 )
        {
            /* If we already have a format chunk, the file is invalid. */
            if ( fmtfound == TRUE )
                return( -1 );

            /* We have a format chunk. */
            fmtfound = TRUE;

            /* If the format chunk is not long enough to be a PCM format 
               chunk, return FALSE. */
            if ( chunkheader.chunklen != sizeof( struct pcmfmt ) )
                return( -1 );

            /* Read the format chunk. */
            if ( fread( &format, 1, sizeof( struct pcmfmt ), FileHandle )
                    != sizeof( struct pcmfmt ) )
                return( -1 );

            /* Verify the format chunk. */
            if ( format.wFormatTag != WAVE_FORMAT_PCM ||
                    format.nChannels > 2 || format.nChannels == 0 ||
                    format.nSamplesPerSec > 65535L ||
                    format.nSamplesPerSec < (long) MIN_SAMPRATE ||
                    /* format.nAvgBytesPerSec not used */
                    /* format.nBlockAlign not used */
                    format.wBitsPerSample == 0 ||
                    format.wBitsPerSample > 16 )
                return( -1 );

            /* Get the number of bits per sample. */
            BitsPerSample = format.wBitsPerSample;
            if ( BitsPerSample <= 8 )
                BitsPerSample = 8;
            else
                BitsPerSample = 16;

            /* Save the number of channels (mono or stereo) and the 
               sampling rate in Hz. */
            NChannels = format.nChannels;
            SampRate = format.nSamplesPerSec;
        }

        /* If it's a data chunk, verify that we have a format chunk, and 
           leave the file pointer at the start of the PCM data. */
        else if ( strncmp( chunkheader.tag, "data", 4 ) == 0 )
        {
            /* If we don't already have a format chunk, the file is 
               invalid. */
            if ( fmtfound == FALSE )
                return( -1 );

            /* We have a data chunk. */
            datafound = TRUE;

            /* Save the length of the data chunk. */
            DataLen = chunkheader.chunklen;
        }

        /* If it's some other kind of chunk, skip over it. */
        else
        {
            /* Verify that it's a valid chunk tag. */
            for ( i=0; i<4; i++ )
                if ( (chunkheader.tag)[i] < ' ' ||
                        (chunkheader.tag)[i] > '~' )
                    return( -1 );

            /* Skip the chunk. */
            if ( chunkheader.chunklen > 0L )
                if ( fseek( FileHandle, chunkheader.chunklen, SEEK_CUR ) != 0 )
                    return( -1 );
        }
    }

    /* Return success. */
    return( 0 );
}


/*************************************************************************/
/*                                                                       */
/* BUM_2_BUM() function.  This is the sample conversion function to      */
/* convert 8-bit mono samples to 8-bit mono samples.  It does nothing.   */
/*                                                                       */
/*************************************************************************/

static void BUM_2_BUM( unsigned nsamples )
{
}


/*************************************************************************/
/*                                                                       */
/* BUS_2_BUM() function.  This is the sample conversion function to      */
/* convert 8-bit stereo samples to 8-bit mono samples.  The samples in   */
/* ReadBuffer are converted in place.                                    */
/*                                                                       */
/*************************************************************************/

static void BUS_2_BUM( unsigned nsamples )
{
    /* Convert the buffer. */
    _asm {
        push si
        push di
        push es
        mov ax,ds
        mov es,ax
        lea si,word ptr ReadBuffer
        mov di,si
        mov cx,word ptr nsamples
        cld
    bus_2_bum_loop:
        lodsw
        add al,ah
        rcr al,1
        stosb
        loop bus_2_bum_loop
        pop es
        pop di
        pop si
    }
}


/*************************************************************************/
/*                                                                       */
/* WSM_2_BUM() function.  This is the sample conversion function to      */
/* convert 16-bit mono samples to 8-bit mono samples.  The samples in    */
/* ReadBuffer are converted in place.                                    */
/*                                                                       */
/*************************************************************************/

static void WSM_2_BUM( unsigned nsamples )
{
    _asm {
        push si
        push di
        push es
        mov ax,ds
        mov es,ax
        lea si,word ptr ReadBuffer
        mov di,si
        mov cx,word ptr nsamples
        cld
    wsm_2_bum_loop:
        lodsw
        mov al,ah
        add al,128
        stosb
        loop wsm_2_bum_loop
        pop es
        pop di
        pop si
    }
}


/*************************************************************************/
/*                                                                       */
/* WSS_2_BUM() function.  This is the sample conversion function to      */
/* convert 16-bit stereo samples to 8-bit mono samples.  The samples in  */
/* ReadBuffer are converted in place.                                    */
/*                                                                       */
/*************************************************************************/

static void WSS_2_BUM( unsigned nsamples )
{
    _asm {
        push si
        push di
        push es
        mov ax,ds
        mov es,ax
        lea si,word ptr ReadBuffer
        mov di,si
        mov cx,word ptr nsamples
        cld
    wss_2_bum_loop:
        lodsw
        mov dx,ax
        lodsw
        add dh,128
        add ah,128
        add ax,dx
        rcr ax,1
        mov al,ah
        stosb
        loop wss_2_bum_loop
        pop es
        pop di
        pop si
    }
}


/*************************************************************************/
/*                                                                       */
/* Add_To_Huge() function.  This function adds increment to hptr and     */
/* returns the result.  Unlike normal huge pointer incrementation, this  */
/* function normalizes the pointer.                                      */
/*                                                                       */
/*************************************************************************/

static void huge *Add_To_Huge( void huge *hptr, unsigned increment )
{
    _asm {
        ;
        ; Get pointer.
        ;
        mov ax,word ptr hptr
        mov dx,word ptr hptr+2
        ;
        ; Convert to linear address.
        ;
        mov cl,4
        rol dx,cl
        mov bx,dx
        and bx,0fff0h
        and dx,0fh
        add ax,bx
        adc dx,0
        ;
        ; Add the increment.
        ;
        add ax,word ptr increment
        adc dx,0
        ;
        ; Convert to normalized pointer.
        ;
        mov bx,ax
        shr bx,cl
        ror dx,cl
        add dx,bx
        and ax,0fh
    }
}


/*************************************************************************/
/*                                                                       */
/* Adjust_Multiple_8bit() function.  This function takes nsamples *      */
/* Multiple 8-bit samples in ReadBuffer, discards samples not to be      */
/* played, and leaves the remaining samples in the buffer.               */
/*                                                                       */
/*************************************************************************/

static void Adjust_Multiple_8bit( unsigned nsamples )
{
    /* If Multiple is 1, or there are no samples in the buffer, there is 
       nothing to do. */
    if ( Multiple == 1 || !nsamples )
        return;

    _asm {
        push si
        push di
        push es
        mov ax,ds
        mov es,ax
        lea si,word ptr ReadBuffer
        mov di,si
        mov cx,word ptr nsamples
        mov bx,word ptr Multiple
        dec bx
        cld
    multiple_8_loop:
        lodsb
        add si,bx
        stosb
        loop multiple_8_loop
        pop es
        pop di
        pop si
    }
}


/*************************************************************************/
/*                                                                       */
/* Adjust_Multiple_16bit() function.  This function takes nsamples *     */
/* Multiple 16-bit samples in ReadBuffer, discards samples not to be     */
/* played, and leaves the remaining samples in the buffer.               */
/*                                                                       */
/*************************************************************************/

static void Adjust_Multiple_16bit( unsigned nsamples )
{
    /* If Multiple is 1, or there are no samples in the buffer, there is 
       nothing to do. */
    if ( Multiple == 1 || !nsamples )
        return;

    _asm {
        push si
        push di
        push es
        mov ax,ds
        mov es,ax
        lea si,word ptr ReadBuffer
        mov di,si
        mov cx,word ptr nsamples
        mov bx,word ptr Multiple
        dec bx
        shl bx,1
        cld
    multiple_16_loop:
        lodsw
        add si,bx
        stosw
        loop multiple_16_loop
        pop es
        pop di
        pop si
    }
}


/*************************************************************************/
/*                                                                       */
/* Adjust_Multiple_32bit() function.  This function takes nsamples *     */
/* Multiple 32-bit samples in ReadBuffer, discards samples not to be     */
/* played, and leaves the remaining samples in the buffer.               */
/*                                                                       */
/*************************************************************************/

static void Adjust_Multiple_32bit( unsigned nsamples )
{
    /* If Multiple is 1, or there are no samples in the buffer, there is 
       nothing to do. */
    if ( Multiple == 1 || !nsamples )
        return;

    _asm {
        push si
        push di
        push es
        mov ax,ds
        mov es,ax
        lea si,word ptr ReadBuffer
        mov di,si
        mov cx,word ptr nsamples
        mov bx,word ptr Multiple
        dec bx
        shl bx,1
        shl bx,1
        cld
    multiple_32_loop:
        lodsw               ; get first 16 bits
        mov dx,ax           ; save in DX
        lodsw               ; get second 16 bits
        xchg ax,dx          ; sample in DX:AX
        add si,bx           ; skip to next
        stosw               ; save low 16 bits
        mov ax,dx           ; get high 16 bits
        stosw               ; save them too
        loop multiple_32_loop
        pop es
        pop di
        pop si
    }
}


/*************************************************************************/
/*                                                                       */
/* Setup_For_Read() function.  This function sets up global variables    */
/* for reading in the file, including the following:                     */
/*     Convert is the function used to convert file samples, which may   */
/* be stereo and/or 16-bit, to 8-bit mono.                               */
/*     The system may be unable to play at the rate specified in the     */
/* file header.  In that case, SampRate is adjusted to an integer        */
/* fraction of the rate specified in the file (i.e., 1/2, 1/3, etc.).    */
/* Multiple is the ratio of samples loaded to samples played; e.g., if   */
/* SampRate is adjusted to 1/3 of the file rate, Multiple is 3.          */
/*     BytesPerSam is the number of bytes to read from the file per      */
/* playable sample.  It is based on Multiple, NChannels, and             */
/* BitsPerSample.                                                        */
/*     TotalSamples is the total number of playable samples in the file  */
/* at the adjusted rate, based on the values of DataLen and BytesPerSam. */
/* TotalLeft is initialized to TotalSamples.                             */
/*     ReadMax is the maximum number of playable samples that will fit   */
/* in the file read buffer.  It is based on READBUFSIZ and BytesPerSam.  */
/*     Multiple_Adjust is the function used to discard unplayed samples  */
/* read from the file, when the sampling rate has been adjusted          */
/* downward.                                                             */
/*     This function returns 0 if successful, -1 if the system is too    */
/* slow to play the file at a rate of at least MIN_SAMPRATE.             */
/*                                                                       */
/*************************************************************************/

static int Setup_For_Read( void )
{
    unsigned testrate;      /* sampling rate for speed testing */

    /* Set the pointer to the function used to convert file samples to 8- 
       bit mono.  (The 3-voice chip will use 8-bit samples only.) */
    if ( NChannels == 1 )
    {
        if ( BitsPerSample == 8 )
            Convert = BUM_2_BUM;
        else /* BitsPerSample == 16 */
            Convert = WSM_2_BUM;
    }
    else /* NChannels == 2 */
    {
        if ( BitsPerSample == 8 )
            Convert = BUS_2_BUM;
        else /* BitsPerSample == 16 */
            Convert = WSS_2_BUM;
    }

    /* Adjust the sampling rate downward as necessary, setting Multiple 
       accordingly. */
    Multiple = 1;
    testrate = SampRate;
    while ( testrate >= MIN_SAMPRATE )
    {
        /* If the rate is playable on the system, exit the loop. */
        if ( Test_Speed( testrate ) )
            break;

        /* Try a lower rate. */
        Multiple++;
        testrate = SampRate / Multiple;
    }

    /* If no playable rate was found, return failure.  Otherwise, set 
       SampRate to the adjusted rate. */
    if ( testrate < MIN_SAMPRATE )
        return( -1 );
    SampRate = testrate;

    /* Compute the number of bytes to read from the file per playable 
       sample. */
    BytesPerSam = Multiple * NChannels;
    if ( BitsPerSample == 16 )
        BytesPerSam <<= 1;

    /* Set the total number of samples to be played at the new rate.  Set 
       the total remaining to the total number. */
    TotalLeft = TotalSamples = DataLen / (unsigned long) BytesPerSam;

    /* Determine the maximum number of samples that will fit in a read 
       buffer, given BytesPerSam. */
    ReadMax = READBUFSIZ / BytesPerSam;

    /* Set the pointer to the function used to discard samples not played, 
       when the sampling rate has been adjusted downward. */
    if ( NChannels == 1 )
    {
        if ( BitsPerSample == 8 )
            Multiple_Adjust = Adjust_Multiple_8bit;
        else /* BitsPerSample == 16 */
            Multiple_Adjust = Adjust_Multiple_16bit;
    }
    else /* NChannels == 2 */
    {
        if ( BitsPerSample == 8 )
            Multiple_Adjust = Adjust_Multiple_16bit;
        else /* BitsPerSample == 16 */
            Multiple_Adjust = Adjust_Multiple_32bit;
    }

    /* Return success. */
    return( 0 );
}


/*************************************************************************/
/*                                                                       */
/* Read_Wave_File() function.  This function reads in the open .wav file */
/* whose handle is FileHandle, into the buffer SoundBuf, where the size  */
/* of the buffer is given by SoundBufSize.  If successful, it sets       */
/* SoundLen to the length of the sound in the buffer and SampRate to the */
/* sampling rate in Hertz, and returns 0.  If the file will not entirely */
/* fit, as much as will go is placed in the buffer, Soundlen is set to   */
/* the length of the data that was actually loaded, SampRate is set      */
/* according to the header, TooLong is set to 1, and 0 is returned.  If  */
/* the file is not an uncompressed PCM .wav file or is corrupt, -1 is    */
/* returned, and other variables are not set.                            */
/*     If the file is a 16-bit or stereo file, it is converted to mono   */
/* 8-bit during loading.                                                 */
/*                                                                       */
/*************************************************************************/

int Read_Wave_File( void )
{
    unsigned thistime;      /* number of bytes to read this time */
    unsigned samplesgot;    /* number of samples actually read */
    char huge *bufptr;      /* pointer into sound buffer */
    unsigned dseg;          /* DS value */

    /* Read the .wav header.  Return error if found to be invalid. */
    if ( Read_Wave_Header() == -1 )
    {
        fputs( "\nCorrupt file or unsupported file type.\n", stderr );
        return( -1 );
    }

    /* Set up variables for reading. */
    if ( Setup_For_Read() == -1 )
    {
        fputs( "\nSystem too slow - can't set sampling rate.\n", stderr );
        return( -1 );
    }

    /* Determine the number of samples that will actually fit in the 
       buffer.  If less than the number of playable samples in the file, 
       set the TooLong flag and adjust the number of samples to be loaded. 
       */
    if ( TotalSamples > SoundBufSize )
    {
        TooLong = 1;
        TotalLeft = TotalSamples = SoundBufSize;
    }

    /* Start at the beginning of the sound buffer.  Get data segment value. 
       */
    bufptr = SoundBuf;
    _asm{ mov dseg,ds }

    /* Loop while there are more samples to read. */
    while ( TotalLeft != 0L )
    {
        /* Determine the number of samples to get this time. */
        if ( TotalLeft < (unsigned long) ReadMax )
            thistime = (unsigned) TotalLeft;
        else
            thistime = ReadMax;

        /* Read some samples. */
        samplesgot = fread( ReadBuffer, BytesPerSam, thistime, FileHandle );

        /* If we got some samples, discard the ones we're skipping at the 
           adjusted sampling rate, convert the remaining ones to 8-bit mono 
           unsigned, and place them in the sample buffer.  Increment and 
           normalize the buffer pointer for next time. */
        if ( samplesgot )
        {
            (*Multiple_Adjust)( samplesgot );
            (*Convert)( samplesgot );
            movedata( dseg, (unsigned) ReadBuffer, FP_SEG( bufptr ),
                FP_OFF( bufptr ), samplesgot );
            bufptr = Add_To_Huge( bufptr, samplesgot );
        }

        /* Reduce the number of samples remaining. */
        TotalLeft -= (unsigned long) samplesgot;

        /* If we didn't get the expected number of samples, exit the loop. */
        if ( samplesgot != thistime )
            break;
    }

    /* If we successfully loaded the entire file, write "OK" to stderr.  
       Otherwise, determine the cause and display that. */
    if ( TotalLeft != 0L )
    {
        if ( feof( FileHandle ) )
            fputs( " premature end-of-file.\n", stderr );
        else
            fputs( " file or disk error.\n", stderr );
    }
    else
        fputs( " OK.\n", stderr );

    /* Determine the number of samples actually in the buffer.  If none, 
       return failure. */
    SoundLen = TotalSamples - TotalLeft;
    if ( SoundLen == 0L )
    {
        fputs( "No samples loaded - halting.\n", stderr );
        return( -1 );
    }

    /* Return success. */
    return( 0 );
}
