/* ----------------------------------------------------------------------------	*/
/*	Originators:																*/
/*  	Tom Arnold,	Quark, Inc.,	300 S. Jackson, 	Denver, 	CO 80209	*/
/*		Ron Perry, 	Atex, Inc.,		165 Lexington Road,	Billerica, 	MA 01821	*/
/* ----------------------------------------------------------------------------	*/
/*	The source code in this file is provided by Quark and Atex for use in the	*/
/*	development of a Publishing Interchange Language. The code provided herein	*/
/*	is non-proprietary, non-copyrighted, and is for use in the public domain.	*/
/*	The source code contained herein is provided "as is".						*/
/* ----------------------------------------------------------------------------	*/
/*	QUARK AND ATEX DISCLAIM ALL WARRANTIES, EXPRESSED OR IMPLIED, WITH REGARD	*/ 
/*	TO THE ENCLOSED SOURCE CODE. QUARK AND ATEX DISCLAIM ANY IMPLIED WARRANTY,	*/
/*	INCLUDING, BUT NOT LIMITED TO ANY IMPLIED WARRANTY OF FITNESS FOR A			*/
/*	PARTICULAR PURPOSE OR MERCHANTABILITY.										*/
/* ----------------------------------------------------------------------------	*/


/* ------------------------------------------------------------------------	*/
/*		P I L  P A R S E R  M O D U L E										*/
/* ------------------------------------------------------------------------	*/

/*$au ------------------------------ audits -----------------------------------
    Author: Ron Perry
            Atex, Inc.
            165 Lexington Road
			Billerica, MA   01821
			508-670-7259


    Ver     Date       Who  Etc
    v01.01  24-apr-91  rp	Update for PIL 5.0
    v01.00  12-feb-91  rp	Initial version
--------------------------------------------------------------------------- */

/* ------------- */
/* Include Files */
/* ------------- */


#include	"pildefs.h"


/* ------- */
/* Globals */
/* ------- */


#ifdef PIL_ANSI_PROTOTYPES						/* If using ANSI prototypes */
extern	int	(FPTRTYPE *pil_getchar)(PIL_VOID);	/* Appl GetChar Func Ptr	*/
#else											/* else use K&R func decl's	*/
extern	int	(FPTRTYPE *pil_getchar)();			/* Appl GetChar Func Ptr	*/	
#endif											/* end of prototype ifdef	*/


extern		PIL_UINT16			pil_char_class_table [PIL_CHAR_CLASS_TSIZE];
extern		PIL_PARSER_STATE	pil_parser_state;
extern		PIL_UINT32			pil_cnt_buf_size;
extern		char PTRPTR			pil_asa_list;
extern		PIL_INT32 			pil_asa_count;
extern		char PTRTYPE 		*pil_kws[];


/* ------------- */
/* Parser Macros */
/* ------------- */


/* Fetch the next token and return a parse error if not a Begin Block 		*/
/* ------------------------------------------------------------------------ */
#define		PIL_FORCE_BEG()		{											\
									pil_get_token ();						\
									if (!pil_is_token_beg_block()) 			\
									return (PIL_BAD_SYNTAX);				\
								}


/* Fetch the next token and return a parse error if not an End Block 		*/
/* ------------------------------------------------------------------------ */
#define		PIL_FORCE_END()		{											\
									pil_get_token ();						\
									if (!pil_is_token_end_block()) 			\
									return (PIL_BAD_SYNTAX);				\
								}


/* Fetch the next token and return a parse error if not a String 			*/
/* ------------------------------------------------------------------------ */
#define		PIL_FORCE_STR()		{											\
									pil_get_token ();						\
									if (!pil_is_token_string()) 			\
									return (PIL_BAD_SYNTAX);				\
								}


/* Fetch the next token and return a parse error if not a Symbol 			*/
/* ------------------------------------------------------------------------ */
#define		PIL_FORCE_SYM()		{											\
									pil_get_token ();						\
									if (!pil_is_token_symbol()) 			\
									return (PIL_BAD_SYNTAX);				\
								}


/* Fetch the next token and return a parse error if not an int32. 			*/
/* Set the given argument to the token's numerical value.					*/
/* ------------------------------------------------------------------------ */
#define		PIL_FORCE_I32(A)	{											\
									pil_get_token ();						\
									if (!pil_is_token_int32()) 				\
									return (PIL_BAD_SYNTAX);				\
									else A = (PIL_INT32)					\
									pil_parser_state.token_constant;		\
								}


/* Fetch the next token and return a parse error if not a uint32. 			*/
/* Set the given argument to the token's numerical value.					*/
/* ------------------------------------------------------------------------ */
#define		PIL_FORCE_U32(A)	{											\
									pil_get_token ();						\
									if (!pil_is_token_uint32()) 			\
									return (PIL_BAD_SYNTAX);				\
									else A = (PIL_UINT32)					\
									pil_parser_state.token_constant;		\
								}


/* Fetch the next token and return a parse error if not a double. 			*/
/* Set the given argument to the token's numerical value.					*/
/* ------------------------------------------------------------------------ */
#define		PIL_FORCE_DBL(A)	{											\
									pil_get_token ();						\
									if (!pil_is_token_double()) 			\
									return (PIL_BAD_SYNTAX);				\
									else A = (PIL_DOUBLE)					\
									pil_parser_state.token_constant;		\
								}


/* Perform range checking on color intensity values 						*/
/* ------------------------------------------------------------------------ */
#define		PIL_RANGE_COL(A)	{											\
									if ((A < 0) || (A > 1))					\
									return (PIL_BAD_SYNTAX);				\
								}


/* ----------------------- */
/* Parser code begins here */
/* ----------------------- */


PIL_VOID pil_init_parser ()
{
	pil_init_char_class_table ();
	pil_init_parser_state ();
}


PIL_VOID pil_init_char_class_table ()
{
	FAST PIL_UINT16 i;
	FAST PIL_UINT16 size;
	size = PIL_CHAR_CLASS_TSIZE-1;
	
	
	for (i =  0 ; i <= size; ++i) 	pil_char_class_table [i] = PIL_C_GARBAGE;
	for (i = '0'; i <=  '9'; ++i) 	pil_char_class_table [i] = PIL_C_DIGIT;
	for (i = 'A'; i <=  'Z'; ++i) 	pil_char_class_table [i] = PIL_C_LETTER;
	for (i = 'a'; i <=  'z'; ++i) 	pil_char_class_table [i] = PIL_C_LETTER;
	
	
	pil_char_class_table [9]  	= PIL_C_WHITESPACE;
	pil_char_class_table [10] 	= PIL_C_WHITESPACE;
	pil_char_class_table [11] 	= PIL_C_WHITESPACE;
	pil_char_class_table [12] 	= PIL_C_WHITESPACE;
	pil_char_class_table [13] 	= PIL_C_WHITESPACE;
	pil_char_class_table [32] 	= PIL_C_WHITESPACE;
	pil_char_class_table ['*'] 	= PIL_C_STAR;
	pil_char_class_table ['"'] 	= PIL_C_STRING;
	pil_char_class_table ['+'] 	= PIL_C_PLUS;
	pil_char_class_table ['-'] 	= PIL_C_MINUS;
	pil_char_class_table ['.'] 	= PIL_C_DOT;
	pil_char_class_table ['/'] 	= PIL_C_COMMENT;
	pil_char_class_table ['_'] 	= PIL_C_UNDERSCORE;
	pil_char_class_table ['{'] 	= PIL_C_BEG_BLOCK;
	pil_char_class_table ['}'] 	= PIL_C_END_BLOCK;
}


PIL_VOID pil_init_parser_state ()
{
	pil_parser_state.initial_fetch = PIL_TRUE;
	pil_parser_state.line_number = 1;
	pil_parser_state.state = PIL_PARSER_S_START;
}


PIL_ERROR pil_get_component (pil_comp)
pil_component PTRPTR pil_comp;
{
	PIL_UINT16 				next_state;
	PIL_ERROR 				ret_code;
	PIL_PTR_PARSER_STATE 	pps;
	
	
	if (pil_getchar == NULL) return (pil_set_error(PIL_NO_R_FUNC));
	pps = &pil_parser_state;
	ret_code = PIL_PREV_ERR;
	
	
	while (1)
	{
		switch (pps->state)
		{
			case PIL_PARSER_S_START:
			{
				if (!pil_is_pil_file ()) 
				{
					ret_code = PIL_BAD_FILE_TYPE;
					pps->state = PIL_PARSER_S_ERROR;
					continue;
				}
				else
				{
					pps->state = PIL_PARSER_S_ENTITY;
					continue;
				}
			}

			
			case PIL_PARSER_S_ENTITY:
			{
				pil_get_token ();
				switch (pps->token_type)
				{
					case PIL_TOKEN_SYMBOL:
					{
						if (pil_is_kw_pil_layout())
						{
							pps->state = PIL_PARSER_S_LAYOUT_START;
							continue;
						}
						if (pil_is_kw_pil_content())
						{
							pps->state = PIL_PARSER_S_CNT_START;
							continue;
						}
						ret_code = PIL_BAD_SYNTAX;
						pps->state = PIL_PARSER_S_ERROR;
						continue;
					}
					
					case PIL_TOKEN_EOF:
					{
						pps->state = PIL_PARSER_S_EOF;
						continue;
					}
					
					default:
					{
						ret_code = PIL_BAD_SYNTAX;
						pps->state = PIL_PARSER_S_ERROR;
						continue;
					}
				}
			}

			
			case PIL_PARSER_S_LAYOUT_START:
			{
				ret_code = pil_parse_layout_start (pil_comp);
				if (ret_code == PIL_OK)
				{
					pps->state = PIL_PARSER_S_NAMETABLE;
					return (PIL_OK);
				}
				else
				{
					pps->state = PIL_PARSER_S_ERROR;
					continue;
				}
			}

			
			case PIL_PARSER_S_NAMETABLE:
			{
				ret_code = pil_parse_nametable (pil_comp);
				if (ret_code == PIL_OK)
				{
					pps->state = PIL_PARSER_S_CANVAS;
					return (PIL_OK);
				}
				else
				{
					pps->state = PIL_PARSER_S_ERROR;
					continue;
				}
			}

			
			case PIL_PARSER_S_CANVAS:
			{
				ret_code = pil_parse_canvas (pil_comp);
				if (ret_code == PIL_OK)
				{
					pps->state = PIL_PARSER_S_OBJECT;
					return (PIL_OK);
				}
				else
				{
					pps->state = PIL_PARSER_S_ERROR;
					continue;
				}
			}

			
			case PIL_PARSER_S_OBJECT:
			{
				pil_get_token ();
				switch (pps->token_type)
				{
					case PIL_TOKEN_SYMBOL:
					{
						if (pil_is_kw_object()) 
						{
							ret_code = pil_parse_object (pil_comp);
							next_state = PIL_PARSER_S_OBJECT;
						}
						else if (pil_is_kw_text_flow())
						{
							ret_code = pil_parse_text_flow (pil_comp);
							next_state = PIL_PARSER_S_TEXTFLOW;
						}
						else if (pil_is_kw_group())
						{
							ret_code = pil_parse_group (pil_comp);
							next_state = PIL_PARSER_S_GROUP;
						}
						else
						{
							ret_code = PIL_BAD_SYNTAX;
						}

						if (ret_code == PIL_OK)
						{
							pps->state = next_state;
							return (PIL_OK);
						}
						else
						{
							pps->state = PIL_PARSER_S_ERROR;
							continue;
						}
					}
					
					case PIL_TOKEN_END_BLOCK:
					{
						pil_make_layout_end_comp (pil_comp);
						pps->state = PIL_PARSER_S_ENTITY;
						return (PIL_OK);
					}
					
					default:
					{
						ret_code = PIL_BAD_SYNTAX;
						pps->state = PIL_PARSER_S_ERROR;
						continue;
					}
				}
			}

			
			case PIL_PARSER_S_TEXTFLOW:
			{
				pil_get_token ();
				switch (pps->token_type)
				{
					case PIL_TOKEN_SYMBOL:
					{
						if (pil_is_kw_text_flow())
						{
							ret_code = pil_parse_text_flow (pil_comp);
							next_state = PIL_PARSER_S_TEXTFLOW;
						}
						else if (pil_is_kw_group())
						{
							ret_code = pil_parse_group (pil_comp);
							next_state = PIL_PARSER_S_GROUP;
						}
						else
						{
							ret_code = PIL_BAD_SYNTAX;
						}

						if (ret_code == PIL_OK)
						{
							pps->state = next_state;
							return (PIL_OK);
						}
						else
						{
							pps->state = PIL_PARSER_S_ERROR;
							continue;
						}
					}
					
					case PIL_TOKEN_END_BLOCK:
					{
						pil_make_layout_end_comp (pil_comp);
						pps->state = PIL_PARSER_S_ENTITY;
						return (PIL_OK);
					}
					
					default:
					{
						ret_code = PIL_BAD_SYNTAX;
						pps->state = PIL_PARSER_S_ERROR;
						continue;
					}
				}
			}

			
			case PIL_PARSER_S_GROUP:
			{
				pil_get_token ();
				switch (pps->token_type)
				{
					case PIL_TOKEN_SYMBOL:
					{
						if (pil_is_kw_group())
						{
							ret_code = pil_parse_group (pil_comp);
							next_state = PIL_PARSER_S_GROUP;
						}
						else
						{
							ret_code = PIL_BAD_SYNTAX;
						}

						if (ret_code == PIL_OK)
						{
							pps->state = next_state;
							return (PIL_OK);
						}
						else
						{
							pps->state = PIL_PARSER_S_ERROR;
							continue;
						}
					}
					
					case PIL_TOKEN_END_BLOCK:
					{
						pil_make_layout_end_comp (pil_comp);
						pps->state = PIL_PARSER_S_ENTITY;
						return (PIL_OK);
					}
					
					default:
					{
						ret_code = PIL_BAD_SYNTAX;
						pps->state = PIL_PARSER_S_ERROR;
						continue;
					}
				}
			}

			
			case PIL_PARSER_S_CNT_START:
			{
				ret_code = pil_parse_cnt_start (pil_comp);
				if (ret_code == PIL_OK)
				{
					pps->state = PIL_PARSER_S_CNT_HEADER;
					return (PIL_OK);
				}
				else
				{
					pps->state = PIL_PARSER_S_ERROR;
					continue;
				}
			}

			
			case PIL_PARSER_S_CNT_HEADER:
			{
				pil_get_token ();
				switch (pps->token_type)
				{
					case PIL_TOKEN_SYMBOL:
					{
						ret_code = pil_parse_cnt_header (pil_comp);
						if (ret_code == PIL_OK)
						{
							pps->cnt_start_state = PIL_TRUE;
							pps->cnt_all_done = PIL_FALSE;
							pps->state = PIL_PARSER_S_CNT_DATA;
							return (PIL_OK);
						}
						else
						{
							pps->state = PIL_PARSER_S_ERROR;
							continue;
						}
					}
					
					case PIL_TOKEN_END_BLOCK:
					{
						pps->state = PIL_PARSER_S_CNT_END;
						continue;
					}
					
					default:
					{
						ret_code = PIL_BAD_SYNTAX;
						pps->state = PIL_PARSER_S_ERROR;
						continue;
					}
				}
			}


			case PIL_PARSER_S_CNT_DATA:
			{
				switch (pps->cnt_encoding)
				{
					case PIL_ASCII_CONTENT:
					{
						ret_code = pil_parse_ascii_cnt (pil_comp);
						break;
					}
					case PIL_BINARY_CONTENT:
					{
						ret_code = pil_parse_binary_cnt (pil_comp);
						break;
					}
				}
				if (ret_code == PIL_OK)
				{
					if (pps->cnt_all_done) pps->state = PIL_PARSER_S_CNT_HEADER;
					else pps->state = PIL_PARSER_S_CNT_DATA;
					return (PIL_OK);
				}
				else
				{
					pps->state = PIL_PARSER_S_ERROR;
					continue;
				}
			}


			case PIL_PARSER_S_CNT_END:
			{
				ret_code = pil_make_cnt_end_comp (pil_comp);
				if (ret_code == PIL_OK)
				{
					pps->state = PIL_PARSER_S_ENTITY;
					return (PIL_OK);
				}
				else
				{
					pps->state = PIL_PARSER_S_ERROR;
					continue;
				}
			}


			case PIL_PARSER_S_EOF:
			{
				return (PIL_END_OF_FILE);
			}
			
			
			case PIL_PARSER_S_ERROR:
			{
				return (ret_code);
			}
		}
	}
}


PIL_BOOL pil_is_pil_file ()
{
	/* Always return true for now */
	return (PIL_TRUE);
}


/* ------------------------------------------- */
/* PIL component parsing functions begins here */
/* ------------------------------------------- */


PIL_ERROR pil_parse_layout_start (pil_comp)
pil_component PTRPTR pil_comp;
{
	pil_copy_to_buf1();
	PIL_FORCE_STR();
	pil_copy_to_buf2();
	PIL_FORCE_BEG();
	return (pil_make_layout_start_comp (pil_comp));
}


PIL_ERROR pil_parse_nametable (pil_comp)
pil_component PTRPTR pil_comp;
{
	PIL_ERROR 				ret_code;
	PIL_UINT16 				type;
	pil_name_table			PTRTYPE *name_table;
	pil_name_table_entry	PTRTYPE *entry;
	pil_name_table_entry	PTRPTR next;
	
	
	PIL_FORCE_SYM();
	if (!pil_is_kw_name_table()) return (PIL_BAD_SYNTAX);
	PIL_FORCE_BEG();
	
	
	ret_code = pil_make_name_table_comp (pil_comp);
	if (ret_code != PIL_OK) return (ret_code);
	name_table = (pil_name_table PTRTYPE *) *pil_comp;
	next = &(name_table->entries);
	
	
	while (1)
	{
		type = pil_get_token ();
		if (type == PIL_TOKEN_END_BLOCK) return (PIL_OK);
		if (type != PIL_TOKEN_STRING) return (PIL_BAD_SYNTAX);
		pil_copy_to_buf1 ();
		
		
		type = pil_get_token ();
		switch (type)
		{
			case PIL_TOKEN_SYMBOL:
			{
				if (!pil_is_kw_nt_type()) return (PIL_BAD_SYNTAX);
				break;
			}
			case PIL_TOKEN_STRING:
			{
				pil_parser_state.val2 = PIL_CNT_OTHER;
				pil_copy_to_buf2 ();
				break;
			}
			default:
			{
				return (PIL_BAD_SYNTAX);
			}
		}
		
		
		type = pil_get_token ();
		switch (type)
		{
			case PIL_TOKEN_SYMBOL:
			{
				if (!pil_is_kw_nt_domain()) return (PIL_BAD_SYNTAX);
				break;
			}
			case PIL_TOKEN_STRING:
			{
				pil_parser_state.val3 = PIL_DMN_OTHER;
				pil_copy_to_buf3 ();
				break;
			}
			default:
			{
				return (PIL_BAD_SYNTAX);
			}
		}
		
		
		type = pil_get_token ();
		switch (type)
		{
			case PIL_TOKEN_SYMBOL:
			{
				if (!pil_is_kw_nt_value()) return (PIL_BAD_SYNTAX);
				break;
			}
			case PIL_TOKEN_STRING:
			{
				pil_parser_state.val4 = PIL_VAL_OTHER;
				pil_copy_to_buf4 ();
				break;
			}
			default:
			{
				return (PIL_BAD_SYNTAX);
			}
		}
		
		
		ret_code = pil_make_name_table_entry (&entry);
		if (ret_code != PIL_OK) return (ret_code);
		*next = entry;
		next = &(entry->next);
	}
}


PIL_ERROR pil_parse_canvas (pil_comp)
pil_component PTRPTR pil_comp;
{
	PIL_ERROR 	ret_code;
	PIL_UINT16 	type;
	PIL_UINT16	num_dimensions;
	PIL_UINT16	num_units;
	PIL_UINT16	num_user_name;
	PIL_UINT16	num_type;
	PIL_UINT16	num_clipper;
	pil_canvas	PTRTYPE *canvas;
	pil_asi		PTRTYPE *asi;
	pil_asi		PTRPTR next;
	
	
	PIL_FORCE_SYM();
	if (!pil_is_kw_canvas()) return (PIL_BAD_SYNTAX);
	PIL_FORCE_BEG();
	
	
	ret_code = pil_make_canvas_comp (pil_comp);
	if (ret_code != PIL_OK) return (ret_code);
	canvas = (pil_canvas PTRTYPE *) *pil_comp;
	next = &(canvas->asi);


	num_dimensions = 0;
	num_units = 0;
	num_user_name = 0;
	num_type = 0;
	num_clipper = 0;
	
	
	while (1)
	{
		type = pil_get_token ();
		switch (type)
		{
			case PIL_TOKEN_SYMBOL:
			{
				if (pil_is_kw_dimensions())
				{
					if (num_dimensions > 0) return (PIL_BAD_SYNTAX);
					++num_dimensions;
					ret_code = pil_parse_dimensions (pil_comp);
					if (ret_code != PIL_OK) return (ret_code);
					continue;
				}
				
				
				if (pil_is_kw_units())
				{
					if (num_units > 0) return (PIL_BAD_SYNTAX);
					++num_units;
					ret_code = pil_parse_units (pil_comp);
					if (ret_code != PIL_OK) return (ret_code);
					continue;
				}
				
				
				if (pil_is_kw_user_name())
				{
					if (num_user_name > 0) return (PIL_BAD_SYNTAX);
					++num_user_name;
					ret_code = pil_parse_user_name (pil_comp);
					if (ret_code != PIL_OK) return (ret_code);
					continue;
				}
				
				
				if (pil_is_kw_type())
				{
					if (num_type > 0) return (PIL_BAD_SYNTAX);
					++num_type;
					ret_code = pil_parse_type (pil_comp);
					if (ret_code != PIL_OK) return (ret_code);
					continue;
				}
				
				
				if (pil_is_kw_clipper())
				{
					if (num_clipper > 0) return (PIL_BAD_SYNTAX);
					++num_clipper;
					ret_code = pil_parse_simp_closed_prim (pil_comp, PIL_CLIPPER_DEST);
					if (ret_code != PIL_OK) return (ret_code);
					continue;
				}
				
				
				if (pil_is_kw_application())
				{
					ret_code = pil_parse_asi (&asi);
					if (ret_code != PIL_OK) return (ret_code);
					if (asi != NULL)
					{
						*next = asi;
						next = &(asi->next);
					}
					continue;
				}
				return (PIL_BAD_SYNTAX);
			}
			
			
			case PIL_TOKEN_END_BLOCK:
			{
				if (num_dimensions == 1 && num_units == 1) return (PIL_OK);
				else return (PIL_BAD_SYNTAX);
			}
			
			
			default:
			{
				return (PIL_BAD_SYNTAX);
			}
		}
	}
}


PIL_ERROR pil_parse_dimensions (pil_comp)
pil_component PTRPTR pil_comp;
{
	PIL_UINT32 	width;
	PIL_UINT32 	height;
	pil_canvas	PTRTYPE *canvas;
	pil_object	PTRTYPE *object;
	
	
	PIL_FORCE_BEG();
	PIL_FORCE_U32(width);	if (width  == 0) return (PIL_BAD_SYNTAX);
	PIL_FORCE_U32(height);	if (height == 0) return (PIL_BAD_SYNTAX);
	PIL_FORCE_END();
		
	
	switch ((*pil_comp)->type)
	{
		case PIL_CANVAS_C:
		{
			canvas = (pil_canvas PTRTYPE *) *pil_comp;
			canvas->width = width;
			canvas->height = height;
			return (PIL_OK);
		}
		case PIL_OBJECT_C:
		{
			object = (pil_object PTRTYPE *) *pil_comp;
			object->dest_rect.width = width;
			object->dest_rect.height = height;
			return (PIL_OK);
		}
	}
}


PIL_ERROR pil_parse_units (pil_comp)
pil_component PTRPTR pil_comp;
{
	PIL_DOUBLE 	units;
	pil_canvas	PTRTYPE *canvas;
	pil_object	PTRTYPE *object;


	PIL_FORCE_DBL(units);
	if (units <= 0) return (PIL_BAD_SYNTAX);
	
	
	switch ((*pil_comp)->type)
	{
		case PIL_CANVAS_C:
		{
			canvas = (pil_canvas PTRTYPE *) *pil_comp;
			canvas->units = units;
			return (PIL_OK);
		}
		case PIL_OBJECT_C:
		{
			object = (pil_object PTRTYPE *) *pil_comp;
			object->units = units;
			return (PIL_OK);
		}
	}
}


PIL_ERROR pil_parse_user_name (pil_comp)
pil_component PTRPTR pil_comp;
{
	pil_canvas	PTRTYPE *canvas;
	pil_object	PTRTYPE *object;
	
	
	PIL_FORCE_STR();
	switch ((*pil_comp)->type)
	{
		case PIL_CANVAS_C:
		{
			canvas = (pil_canvas PTRTYPE *) *pil_comp;
			return (pil_append_string ((char PTRTYPE *) pil_parser_state.token, 
			&(canvas->username)));
		}
		case PIL_OBJECT_C:
		{
			object = (pil_object PTRTYPE *) *pil_comp;
			return (pil_append_string ((char PTRTYPE *) pil_parser_state.token, 
			&(object->username)));
		}
	}
}


PIL_ERROR pil_parse_type (pil_comp)
pil_component PTRPTR pil_comp;
{
	PIL_UINT16 	token_type;
	pil_canvas	PTRTYPE *canvas;
	pil_object	PTRTYPE *object;
	
	
	switch ((*pil_comp)->type)
	{
		case PIL_CANVAS_C:
		{
			canvas = (pil_canvas PTRTYPE *) *pil_comp;
			PIL_FORCE_STR();
			return (pil_append_string ((char PTRTYPE *) pil_parser_state.token, 
			&(canvas->cantype)));
		}
		case PIL_OBJECT_C:
		{
			object = (pil_object PTRTYPE *) *pil_comp;
			token_type = pil_get_token ();
			switch (token_type)
			{
				case PIL_TOKEN_SYMBOL:
				{
					if 		(pil_is_kw_text())		object->objtype = PIL_OBJ_TEXT;
					else if	(pil_is_kw_picture())	object->objtype = PIL_OBJ_PICTURE;
					else if	(pil_is_kw_primitive())	object->objtype = PIL_OBJ_PRIMITIVE;
					else							return (PIL_BAD_SYNTAX);
					return (PIL_OK);
				}
				default:
				{
					return (PIL_BAD_SYNTAX);
				}
			}
		}
	}
}


PIL_ERROR pil_parse_simp_closed_prim (pil_comp, dest)
pil_component PTRPTR pil_comp;
PIL_UINT16 dest;
{
	PIL_UINT16 				ret_code;
	pil_rectangle 			PTRTYPE *rect;
	pil_circle				PTRTYPE *circle;
	pil_ellipse				PTRTYPE *ellipse;
	pil_polygon				PTRTYPE *poly;
	pil_closed_path			PTRTYPE *path;
	pil_canvas				PTRTYPE *canvas;
	pil_object				PTRTYPE *object;
	pil_closed_primitive	PTRTYPE *cprim;
	
	
	if (dest == PIL_CLIPPER_DEST) 
	{
		/* Both canvases and objects contain clippers */
		/* ------------------------------------------ */
		if ((*pil_comp)->type == PIL_CANVAS_C)
		{
			canvas = (pil_canvas PTRTYPE *) *pil_comp;
			cprim = (pil_closed_primitive PTRTYPE *) &(canvas->clipshape);
		}
		else
		{
			object = (pil_object PTRTYPE *) *pil_comp;
			cprim = (pil_closed_primitive PTRTYPE *) &(object->clipshape);
		}
	}
	else 
	{
		/* Only objects have containers */
		/* ---------------------------- */
		object = (pil_object PTRTYPE *) *pil_comp;
		cprim = (pil_closed_primitive PTRTYPE *) &(object->container);
	}
	
	
	PIL_FORCE_BEG();
	PIL_FORCE_SYM();
	if (!pil_is_graphic_prim()) return (PIL_BAD_SYNTAX);
	cprim->type = (PIL_PRIMITIVE) pil_parser_state.val1;
	

	switch (cprim->type)
	{
		case PIL_RECTANGLE_SHAPE:
		{
			rect = &(cprim->shape.rect);
			ret_code = pil_parse_rectangle (rect);
			break;
		}
		case PIL_CIRCLE_SHAPE:
		{
			circle = &(cprim->shape.circle);
			ret_code = pil_parse_circle (circle);
			break;
		}
		case PIL_ELLIPSE_SHAPE:
		{
			ellipse = &(cprim->shape.ellipse);
			ret_code = pil_parse_ellipse (ellipse);
			break;
		}
		case PIL_POLYGON_SHAPE:
		{
			poly = &(cprim->shape.polygon);
			ret_code = pil_parse_polygon (poly, PIL_SHAPE_SIMPLE);
			break;
		}
		case PIL_CLOSEDPATH_SHAPE:
		{
			path = &(cprim->shape.closedpath);
			ret_code = pil_parse_closed_path (path, PIL_SHAPE_SIMPLE);
			break;
		}
		default:
		{
			ret_code = PIL_BAD_SYNTAX;
			break;
		}
	}
	

	if (ret_code != PIL_OK) return (ret_code);
	PIL_FORCE_END();
	return (PIL_OK);
}


PIL_ERROR pil_parse_rectangle (rect)
pil_rectangle PTRTYPE *rect;
{
	PIL_FORCE_BEG();
	PIL_FORCE_I32(rect->ul.x);
	PIL_FORCE_I32(rect->ul.y);
	PIL_FORCE_U32(rect->width);
	PIL_FORCE_U32(rect->height);
	PIL_FORCE_END();
	return (PIL_OK);
}


PIL_ERROR pil_parse_circle (circle)
pil_circle PTRTYPE *circle;
{
	PIL_FORCE_BEG();
	PIL_FORCE_I32(circle->ul.x);
	PIL_FORCE_I32(circle->ul.y);
	PIL_FORCE_U32(circle->length);
	PIL_FORCE_END();
	return (PIL_OK);
}


PIL_ERROR pil_parse_ellipse (ellipse)
pil_ellipse PTRTYPE *ellipse;
{
	PIL_FORCE_BEG();
	PIL_FORCE_I32(ellipse->ul.x);
	PIL_FORCE_I32(ellipse->ul.y);
	PIL_FORCE_U32(ellipse->width);
	PIL_FORCE_U32(ellipse->height);
	PIL_FORCE_END();
	return (PIL_OK);
}


PIL_ERROR pil_parse_polygon (poly, complexity)
pil_polygon PTRTYPE *poly;
PIL_UINT16 complexity;
{
	pil_point		p;
	PIL_ERROR		rc;
	PIL_UINT16		token_type;
	pil_poly_pt		PTRTYPE *poly_pt;
	pil_poly_pt		PTRPTR next;
	
	
	PIL_FORCE_BEG();
	rc = pil_parse_point (&p); 
	if (rc != PIL_OK) return (rc);
	poly->p.x = p.x;
	poly->p.y = p.y;
	poly->next = NULL;
	next = &(poly->next);
		
	
	rc = pil_parse_point (&p); 
	if (rc != PIL_OK) return (rc);
	rc = pil_create_polypt (&poly_pt, p.x, p.y);
	if (rc != PIL_OK) return (rc);
	*next = poly_pt;
	next = &(poly_pt->next);
	
	
	rc = pil_parse_point (&p); 
	if (rc != PIL_OK) return (rc);
	rc = pil_create_polypt (&poly_pt, p.x, p.y);
	if (rc != PIL_OK) return (rc);
	*next = poly_pt;
	next = &(poly_pt->next);
	
	
	while (1)
	{
		token_type = pil_get_token ();
		switch (token_type)
		{
			case PIL_TOKEN_BEG_BLOCK:
			{
				PIL_FORCE_I32(p.x);
				PIL_FORCE_I32(p.y);
				PIL_FORCE_END();
				rc = pil_create_polypt (&poly_pt, p.x, p.y);
				if (rc != PIL_OK) return (rc);
				*next = poly_pt;
				next = &(poly_pt->next);
				continue;
			}
			case PIL_TOKEN_END_BLOCK:	return (PIL_OK);
			default:					return (PIL_BAD_SYNTAX);
		}
	}
}


PIL_ERROR pil_parse_point (p)
pil_point PTRTYPE *p;
{
	PIL_FORCE_BEG();
	PIL_FORCE_I32(p->x);
	PIL_FORCE_I32(p->y);
	PIL_FORCE_END();
	return (PIL_OK);
}


PIL_ERROR pil_parse_closed_path (path, complexity)
pil_closed_path PTRTYPE *path;
PIL_UINT16 complexity;
{
	pil_point		p;
	pil_point		cnt_pt1;
	pil_point		cnt_pt2;
	pil_point		end_pt;
	PIL_INT32		radius;
	PIL_ERROR		rc;
	PIL_UINT16		token_type;
	pil_path_pt		PTRTYPE *pathpt;
	pil_path_pt		PTRPTR next;
	PIL_INT32		zero;
	
	
	PIL_FORCE_BEG();
	PIL_FORCE_SYM();
	if (!pil_is_kw_move_to()) return (PIL_BAD_SYNTAX);
	rc = pil_parse_move_to (&p); 
	if (rc != PIL_OK) return (rc);
	
	
	path->type = PIL_MOVE_TO;
	path->p.mt.pt.x = p.x;
	path->p.mt.pt.y = p.y;
	path->next = NULL;
	next = &(path->next);
	zero = 0;
	
	
	while (1)
	{
		token_type = pil_get_token ();
		switch (token_type)
		{
			case PIL_TOKEN_SYMBOL:
			{
				if (complexity == PIL_SHAPE_COMPLEX && pil_is_kw_move_to())
				{
					rc = pil_parse_move_to (&p); 
					if (rc != PIL_OK) return (rc);
					rc = pil_create_pathpt (&pathpt, PIL_MOVE_TO,
					p.x, p.y, zero, zero, zero, zero, zero);
					if (rc != PIL_OK) return (rc);
					*next = pathpt;
					next = &(pathpt->next);
					continue;
				}
				
				
				if (pil_is_kw_line_to())
				{
					rc = pil_parse_line_to (&p); 
					if (rc != PIL_OK) return (rc);
					rc = pil_create_pathpt (&pathpt, PIL_LINE_TO,
					p.x, p.y, zero, zero, zero, zero, zero);
					if (rc != PIL_OK) return (rc);
					*next = pathpt;
					next = &(pathpt->next);
					continue;
				}
				
				
				if (pil_is_kw_curve_to())
				{
					rc = pil_parse_curve_to (&cnt_pt1, &cnt_pt2, &end_pt); 
					if (rc != PIL_OK) return (rc);
					rc = pil_create_pathpt (&pathpt, PIL_CURVE_TO,
					cnt_pt1.x, cnt_pt1.y, cnt_pt2.x, cnt_pt2.y, end_pt.x, 
					end_pt.y, zero);
					if (rc != PIL_OK) return (rc);
					*next = pathpt;
					next = &(pathpt->next);
					continue;
				}
				
				
				if (pil_is_kw_arc_to())
				{
					rc = pil_parse_arc_to (&cnt_pt1, &cnt_pt2, &radius);
					if (rc != PIL_OK) return (rc);
					rc = pil_create_pathpt (&pathpt, PIL_ARC_TO,
					cnt_pt1.x, cnt_pt1.y, cnt_pt2.x, cnt_pt2.y, zero, 
					zero, radius);
					if (rc != PIL_OK) return (rc);
					*next = pathpt;
					next = &(pathpt->next);
					continue;
				}
				return (PIL_BAD_SYNTAX);
			}
			
			
			case PIL_TOKEN_END_BLOCK:	return (PIL_OK);
			default:					return (PIL_BAD_SYNTAX);
		}
	}
}


PIL_ERROR pil_parse_move_to (p)
pil_point PTRTYPE *p;
{
	return (pil_parse_point (p));
}


PIL_ERROR pil_parse_line_to (p)
pil_point PTRTYPE *p;
{
	return (pil_parse_point (p));
}


PIL_ERROR pil_parse_curve_to (cnt_pt1, cnt_pt2, end_pt)
pil_point PTRTYPE *cnt_pt1;
pil_point PTRTYPE *cnt_pt2;
pil_point PTRTYPE *end_pt;
{
	PIL_ERROR rc;
	PIL_FORCE_BEG();
	rc = pil_parse_point (cnt_pt1);	if (rc != PIL_OK) return (rc);
	rc = pil_parse_point (cnt_pt2);	if (rc != PIL_OK) return (rc);
	rc = pil_parse_point (end_pt);	if (rc != PIL_OK) return (rc);
	PIL_FORCE_END();
	return (PIL_OK);
}


PIL_ERROR pil_parse_arc_to (cnt_pt1, cnt_pt2, radius)
pil_point PTRTYPE *cnt_pt1;
pil_point PTRTYPE *cnt_pt2;
PIL_INT32 PTRTYPE *radius;
{
	PIL_ERROR rc;
	PIL_FORCE_BEG();
	rc = pil_parse_point (cnt_pt1);	if (rc != PIL_OK) return (rc);
	rc = pil_parse_point (cnt_pt2);	if (rc != PIL_OK) return (rc);
	PIL_FORCE_I32(*radius);
	PIL_FORCE_END();
	return (PIL_OK);
}


PIL_ERROR pil_parse_asi (asi)
pil_asi PTRPTR asi;
{
	PIL_ERROR 	ret_code;
	PIL_UINT16 	type;
	PIL_UINT32	block_level;
	pil_asa		PTRTYPE *asa;
	pil_asa		PTRPTR next;
	
	
	PIL_FORCE_BEG();
	PIL_FORCE_SYM();
	if (!pil_is_kw_app_name()) return (PIL_BAD_SYNTAX);
	PIL_FORCE_STR();
	
	
	if (!pil_is_desired_asi())
	{
		block_level = 1;
		while (1)
		{
			type = pil_get_token ();
			switch (type)
			{
				case PIL_TOKEN_BEG_BLOCK:
				{
					++block_level;
					continue;
				}
				case PIL_TOKEN_END_BLOCK:
				{
					--block_level;
					if (block_level == 0) 
					{
						*asi = NULL;
						return (PIL_OK);
					}
					continue;
				}
				case PIL_TOKEN_ERROR:	return (PIL_BAD_SYNTAX);
				case PIL_TOKEN_EOF:		return (PIL_BAD_SYNTAX);
				default: 				continue;
			}
		}
	}
	
	
	ret_code = pil_create_asi (asi, (char PTRTYPE *) pil_parser_state.token);
	if (ret_code != PIL_OK) return (ret_code);
	next = &((**asi).asa);
	
	
	while (1)
	{
		type = pil_get_token ();
		switch (type)
		{
			case PIL_TOKEN_SYMBOL:
			{
				ret_code = pil_parse_app_attr (&asa);
				if (ret_code != PIL_OK) return (ret_code);
				*next = asa;
				next = &(asa->next);
				continue;
			}
			case PIL_TOKEN_END_BLOCK:	return (PIL_OK);
			default:					return (PIL_BAD_SYNTAX);
		}
	}
}


PIL_ERROR pil_parse_app_attr (asa)
pil_asa PTRPTR asa;
{
	PIL_ASA_VAL_TYPE 	value_type;
	PIL_UINT16 			token_type;
	PIL_INT8			val_int8;
	PIL_INT16			val_int16;
	PIL_INT32			val_int32;
	PIL_UINT8			val_uint8;
	PIL_UINT16			val_uint16;
	PIL_UINT32			val_uint32;
	PIL_DOUBLE			val_double;
	PIL_UINT32			val_count;
	pil_value			PTRTYPE *pvalue;
	pil_value			PTRPTR next;
	PIL_ERROR			rc;
	
	
	if (!pil_is_kw_app_type()) return (PIL_BAD_SYNTAX);
	value_type = (PIL_ASA_VAL_TYPE) pil_parser_state.val1;
	PIL_FORCE_BEG();
	PIL_FORCE_SYM();
	
	
	rc = pil_create_asa (asa, (char PTRTYPE *) pil_parser_state.token,
	(PIL_UINT32) 0L, value_type, (char PTRTYPE *) NULL);
	if (rc != PIL_OK) return (rc);
	(**asa).count = 1;
	next = &((**asa).value);
	
	
	token_type = pil_get_token ();
	switch (value_type)
	{
		case PIL_INT8_CODE:
		{
			if (pil_is_token_int8())
			{
				val_int8 = (PIL_INT8) pil_parser_state.token_constant;
				rc = pil_create_pvalue (&pvalue, value_type, 
				(char PTRTYPE *) &val_int8);
				if (rc != PIL_OK) return (rc);
				*next = pvalue;
				break;
			}
			else if (pil_is_token_beg_block())
			{
				val_count = 0;
				while (1)
				{
					pil_get_token ();
					if (pil_is_token_int8())
					{
						++val_count;
						val_int8 = (PIL_INT8) pil_parser_state.token_constant;
						rc = pil_create_pvalue (&pvalue, value_type, 
						(char PTRTYPE *) &val_int8);
						if (rc != PIL_OK) return (rc);
						*next = pvalue;
						next = &(pvalue->next);
					}
					else if (pil_is_token_end_block())
					{
						(**asa).type += PIL_LIST_CODE_OFFSET;
						(**asa).count = val_count;
						break;
					}
					else return (PIL_BAD_SYNTAX);
				}
				break;
			}
			else return (PIL_BAD_SYNTAX);
		}


		case PIL_INT16_CODE:
		{
			if (pil_is_token_int16())
			{
				val_int16 = (PIL_INT16) pil_parser_state.token_constant;
				rc = pil_create_pvalue (&pvalue, value_type, 
				(char PTRTYPE *) &val_int16);
				if (rc != PIL_OK) return (rc);
				*next = pvalue;
				break;
			}
			else if (pil_is_token_beg_block())
			{
				val_count = 0;
				while (1)
				{
					pil_get_token ();
					if (pil_is_token_int16())
					{
						++val_count;
						val_int16 = (PIL_INT16) pil_parser_state.token_constant;
						rc = pil_create_pvalue (&pvalue, value_type, 
						(char PTRTYPE *) &val_int16);
						if (rc != PIL_OK) return (rc);
						*next = pvalue;
						next = &(pvalue->next);
					}
					else if (pil_is_token_end_block())
					{
						(**asa).type += PIL_LIST_CODE_OFFSET;
						(**asa).count = val_count;
						break;
					}
					else return (PIL_BAD_SYNTAX);
				}
				break;
			}
			else return (PIL_BAD_SYNTAX);
		}


		case PIL_INT32_CODE:
		{
			if (pil_is_token_int32())
			{
				val_int32 = (PIL_INT32) pil_parser_state.token_constant;
				rc = pil_create_pvalue (&pvalue, value_type, 
				(char PTRTYPE *) &val_int32);
				if (rc != PIL_OK) return (rc);
				*next = pvalue;
				break;
			}
			else if (pil_is_token_beg_block())
			{
				val_count = 0;
				while (1)
				{
					pil_get_token ();
					if (pil_is_token_int32())
					{
						++val_count;
						val_int32 = (PIL_INT32) pil_parser_state.token_constant;
						rc = pil_create_pvalue (&pvalue, value_type, 
						(char PTRTYPE *) &val_int32);
						if (rc != PIL_OK) return (rc);
						*next = pvalue;
						next = &(pvalue->next);
					}
					else if (pil_is_token_end_block())
					{
						(**asa).type += PIL_LIST_CODE_OFFSET;
						(**asa).count = val_count;
						break;
					}
					else return (PIL_BAD_SYNTAX);
				}
				break;
			}
			else return (PIL_BAD_SYNTAX);
		}


		case PIL_UINT8_CODE:
		{
			if (pil_is_token_uint8())
			{
				val_uint8 = (PIL_UINT8) pil_parser_state.token_constant;
				rc = pil_create_pvalue (&pvalue, value_type, 
				(char PTRTYPE *) &val_uint8);
				if (rc != PIL_OK) return (rc);
				*next = pvalue;
				break;
			}
			else if (pil_is_token_beg_block())
			{
				val_count = 0;
				while (1)
				{
					pil_get_token ();
					if (pil_is_token_uint8())
					{
						++val_count;
						val_uint8 = (PIL_UINT8) pil_parser_state.token_constant;
						rc = pil_create_pvalue (&pvalue, value_type, 
						(char PTRTYPE *) &val_uint8);
						if (rc != PIL_OK) return (rc);
						*next = pvalue;
						next = &(pvalue->next);
					}
					else if (pil_is_token_end_block())
					{
						(**asa).type += PIL_LIST_CODE_OFFSET;
						(**asa).count = val_count;
						break;
					}
					else return (PIL_BAD_SYNTAX);
				}
				break;
			}
			else return (PIL_BAD_SYNTAX);
		}


		case PIL_UINT16_CODE:
		{
			if (pil_is_token_uint16())
			{
				val_uint16 = (PIL_UINT16) pil_parser_state.token_constant;
				rc = pil_create_pvalue (&pvalue, value_type, 
				(char PTRTYPE *) &val_uint16);
				if (rc != PIL_OK) return (rc);
				*next = pvalue;
				break;
			}
			else if (pil_is_token_beg_block())
			{
				val_count = 0;
				while (1)
				{
					pil_get_token ();
					if (pil_is_token_uint16())
					{
						++val_count;
						val_uint16 = (PIL_UINT16) pil_parser_state.token_constant;
						rc = pil_create_pvalue (&pvalue, value_type, 
						(char PTRTYPE *) &val_uint16);
						if (rc != PIL_OK) return (rc);
						*next = pvalue;
						next = &(pvalue->next);
					}
					else if (pil_is_token_end_block())
					{
						(**asa).type += PIL_LIST_CODE_OFFSET;
						(**asa).count = val_count;
						break;
					}
					else return (PIL_BAD_SYNTAX);
				}
				break;
			}
			else return (PIL_BAD_SYNTAX);
		}


		case PIL_UINT32_CODE:
		{
			if (pil_is_token_uint32())
			{
				val_uint32 = (PIL_UINT32) pil_parser_state.token_constant;
				rc = pil_create_pvalue (&pvalue, value_type, 
				(char PTRTYPE *) &val_uint32);
				if (rc != PIL_OK) return (rc);
				*next = pvalue;
				break;
			}
			else if (pil_is_token_beg_block())
			{
				val_count = 0;
				while (1)
				{
					pil_get_token ();
					if (pil_is_token_uint32())
					{
						++val_count;
						val_uint32 = (PIL_UINT32) pil_parser_state.token_constant;
						rc = pil_create_pvalue (&pvalue, value_type, 
						(char PTRTYPE *) &val_uint32);
						if (rc != PIL_OK) return (rc);
						*next = pvalue;
						next = &(pvalue->next);
					}
					else if (pil_is_token_end_block())
					{
						(**asa).type += PIL_LIST_CODE_OFFSET;
						(**asa).count = val_count;
						break;
					}
					else return (PIL_BAD_SYNTAX);
				}
				break;
			}
			else return (PIL_BAD_SYNTAX);
		}


		case PIL_DOUBLE_CODE:
		{
			if (pil_is_token_double())
			{
				val_double = (PIL_DOUBLE) pil_parser_state.token_constant;
				rc = pil_create_pvalue (&pvalue, value_type, 
				(char PTRTYPE *) &val_double);
				if (rc != PIL_OK) return (rc);
				*next = pvalue;
				break;
			}
			else if (pil_is_token_beg_block())
			{
				val_count = 0;
				while (1)
				{
					pil_get_token ();
					if (pil_is_token_double())
					{
						++val_count;
						val_double = (PIL_DOUBLE) pil_parser_state.token_constant;
						rc = pil_create_pvalue (&pvalue, value_type, 
						(char PTRTYPE *) &val_double);
						if (rc != PIL_OK) return (rc);
						*next = pvalue;
						next = &(pvalue->next);
					}
					else if (pil_is_token_end_block())
					{
						(**asa).type += PIL_LIST_CODE_OFFSET;
						(**asa).count = val_count;
						break;
					}
					else return (PIL_BAD_SYNTAX);
				}
				break;
			}
			else return (PIL_BAD_SYNTAX);
		}


		case PIL_STRING_CODE:
		{
			if (pil_is_token_string())
			{
				rc = pil_create_pvalue (&pvalue, value_type, 
				(char PTRTYPE *) pil_parser_state.token);
				if (rc != PIL_OK) return (rc);
				*next = pvalue;
				break;
			}
			else if (pil_is_token_beg_block())
			{
				val_count = 0;
				while (1)
				{
					pil_get_token ();
					if (pil_is_token_string())
					{
						++val_count;
						rc = pil_create_pvalue (&pvalue, value_type, 
						(char PTRTYPE *) pil_parser_state.token);
						if (rc != PIL_OK) return (rc);
						*next = pvalue;
						next = &(pvalue->next);
					}
					else if (pil_is_token_end_block())
					{
						(**asa).type += PIL_LIST_CODE_OFFSET;
						(**asa).count = val_count;
						break;
					}
					else return (PIL_BAD_SYNTAX);
				}
				break;
			}
			else return (PIL_BAD_SYNTAX);
		}
	}
	
	
	PIL_FORCE_END();
	return (PIL_OK);
}


PIL_ERROR pil_parse_object (pil_comp)
pil_component PTRPTR pil_comp;
{
	PIL_ERROR 	ret_code;
	PIL_UINT16 	type;
	PIL_UINT16	num_origin;
	PIL_UINT16	num_dimensions;
	PIL_UINT16	num_id;
	PIL_UINT16	num_units;
	PIL_UINT16	num_user_name;
	PIL_UINT16	num_type;
	PIL_UINT16	num_rot_angle;
	PIL_UINT16	num_rot_point;
	PIL_UINT16	num_clipper;
	PIL_UINT16	num_container;
	PIL_UINT16	num_graphic;
	PIL_UINT16	num_rc_type;
	PIL_UINT16	num_rc_name;
	PIL_UINT16	num_bbox;
	pil_object	PTRTYPE *object;
	pil_asi		PTRTYPE *asi;
	pil_asi		PTRPTR next;
	
	
	PIL_FORCE_BEG();
	ret_code = pil_make_object_comp (pil_comp);
	if (ret_code != PIL_OK) return (ret_code);
	object = (pil_object PTRTYPE *) *pil_comp;
	next = &(object->asi);

	
	num_origin = 0;
	num_dimensions = 0;
	num_id = 0;
	num_units = 0;
	num_user_name = 0;
	num_type = 0;
	num_rot_angle = 0;
	num_rot_point = 0;
	num_clipper = 0;
	num_container = 0;
	num_graphic = 0;
	num_rc_type = 0;
	num_rc_name = 0;
	num_bbox = 0;
	
	
	while (1)
	{
		type = pil_get_token ();
		switch (type)
		{
			case PIL_TOKEN_SYMBOL:
			{
				if (pil_is_kw_origin())
				{
					if (num_origin > 0) return (PIL_BAD_SYNTAX);
					++num_origin;
					ret_code = pil_parse_origin (pil_comp);
					if (ret_code != PIL_OK) return (ret_code);
					continue;
				}
				
				
				if (pil_is_kw_dimensions())
				{
					if (num_dimensions > 0) return (PIL_BAD_SYNTAX);
					++num_dimensions;
					ret_code = pil_parse_dimensions (pil_comp);
					if (ret_code != PIL_OK) return (ret_code);
					continue;
				}
				
				
				if (pil_is_kw_id())
				{
					if (num_id > 0) return (PIL_BAD_SYNTAX);
					++num_id;
					ret_code = pil_parse_id (pil_comp);
					if (ret_code != PIL_OK) return (ret_code);
					continue;
				}
				
				
				if (pil_is_kw_units())
				{
					if (num_units > 0) return (PIL_BAD_SYNTAX);
					++num_units;
					ret_code = pil_parse_units (pil_comp);
					if (ret_code != PIL_OK) return (ret_code);
					continue;
				}
				
				
				if (pil_is_kw_user_name())
				{
					if (num_user_name > 0) return (PIL_BAD_SYNTAX);
					++num_user_name;
					ret_code = pil_parse_user_name (pil_comp);
					if (ret_code != PIL_OK) return (ret_code);
					continue;
				}
				
				
				if (pil_is_kw_type())
				{
					if (num_type > 0) return (PIL_BAD_SYNTAX);
					++num_type;
					ret_code = pil_parse_type (pil_comp);
					if (ret_code != PIL_OK) return (ret_code);
					continue;
				}
				
				
				if (pil_is_kw_rot_angle())
				{
					if (num_rot_angle > 0) return (PIL_BAD_SYNTAX);
					++num_rot_angle;
					ret_code = pil_parse_rot_angle (pil_comp);
					if (ret_code != PIL_OK) return (ret_code);
					continue;
				}
				
				
				if (pil_is_kw_rot_point())
				{
					if (num_rot_point > 0) return (PIL_BAD_SYNTAX);
					++num_rot_point;
					ret_code = pil_parse_rot_point (pil_comp);
					if (ret_code != PIL_OK) return (ret_code);
					continue;
				}
				
				
				if (pil_is_kw_clipper())
				{
					if (num_clipper > 0) return (PIL_BAD_SYNTAX);
					++num_clipper;
					ret_code = pil_parse_simp_closed_prim (pil_comp, PIL_CLIPPER_DEST);
					if (ret_code != PIL_OK) return (ret_code);
					continue;
				}
				
				
				if (pil_is_kw_container())
				{
					if (num_container > 0) return (PIL_BAD_SYNTAX);
					++num_container;
					ret_code = pil_parse_simp_closed_prim (pil_comp, PIL_CONTAINER_DEST);
					if (ret_code != PIL_OK) return (ret_code);
					continue;
				}
				
				
				if (pil_is_kw_graphic())
				{
					if (num_graphic > 0) return (PIL_BAD_SYNTAX);
					++num_graphic;
					ret_code = pil_parse_graphic (pil_comp);
					if (ret_code != PIL_OK) return (ret_code);
					continue;
				}
				
				
				if (pil_is_kw_rc_type())
				{
					if (num_rc_type > 0) return (PIL_BAD_SYNTAX);
					++num_rc_type;
					ret_code = pil_parse_rc_type (pil_comp);
					if (ret_code != PIL_OK) return (ret_code);
					continue;
				}
				
				
				if (pil_is_kw_rc_name())
				{
					if (num_rc_name > 0) return (PIL_BAD_SYNTAX);
					++num_rc_name;
					ret_code = pil_parse_rc_name (pil_comp);
					if (ret_code != PIL_OK) return (ret_code);
					continue;
				}
				
				
				if (pil_is_kw_bbox())
				{
					if (num_bbox > 0) return (PIL_BAD_SYNTAX);
					++num_bbox;
					ret_code = pil_parse_bbox (pil_comp);
					if (ret_code != PIL_OK) return (ret_code);
					continue;
				}
				
				
				if (pil_is_kw_application())
				{
					ret_code = pil_parse_asi (&asi);
					if (ret_code != PIL_OK) return (ret_code);
					if (asi != NULL)
					{
						*next = asi;
						next = &(asi->next);
					}
					continue;
				}
				return (PIL_BAD_SYNTAX);
			}
			
			
			case PIL_TOKEN_END_BLOCK:
			{
				if (num_dimensions == 1 && num_origin == 1) return (PIL_OK);
				else return (PIL_BAD_SYNTAX);
			}
			
			
			default:
			{
				return (PIL_BAD_SYNTAX);
			}
		}
	}
}


PIL_ERROR pil_parse_origin (pil_comp)
pil_component PTRPTR pil_comp;
{
	pil_point 	p;
	pil_object	PTRTYPE *object;
	PIL_ERROR 	ret_code;
	
	
	ret_code = pil_parse_point (&p);
	if (ret_code != PIL_OK) return (ret_code);
	object = (pil_object PTRTYPE *) *pil_comp;
	object->dest_rect.ul.x = p.x;
	object->dest_rect.ul.y = p.y;
	return (PIL_OK);
}


PIL_ERROR pil_parse_id (pil_comp)
pil_component PTRPTR pil_comp;
{
	pil_object PTRTYPE *object;
	PIL_FORCE_STR();
	object = (pil_object PTRTYPE *) *pil_comp;
	return (pil_append_string ((char PTRTYPE *) pil_parser_state.token, 
	&(object->id)));
}


PIL_ERROR pil_parse_rot_angle (pil_comp)
pil_component PTRPTR pil_comp;
{
	PIL_DOUBLE	rot_angle;
	pil_object	PTRTYPE *object;
	
	
	PIL_FORCE_DBL(rot_angle);
	object = (pil_object PTRTYPE *) *pil_comp;
	object->rotation_angle = rot_angle;
	return (PIL_OK);
}


PIL_ERROR pil_parse_rot_point (pil_comp)
pil_component PTRPTR pil_comp;
{
	pil_point 	p;
	pil_object	PTRTYPE *object;
	PIL_ERROR 	ret_code;
	
	
	ret_code = pil_parse_point (&p);
	if (ret_code != PIL_OK) return (ret_code);
	object = (pil_object PTRTYPE *) *pil_comp;
	object->rotation_point.x = p.x;
	object->rotation_point.y = p.y;
	return (PIL_OK);
}


PIL_ERROR pil_parse_graphic (pil_comp)
pil_component PTRPTR pil_comp;
{
	PIL_ERROR 		ret_code;
	PIL_UINT16 		type;
	PIL_UINT16		num_stroke_color;
	PIL_UINT16		num_position;
	PIL_UINT16		num_width;
	PIL_UINT16		num_cap;
	PIL_UINT16		num_join;
	PIL_UINT16		num_miter_limit;
	PIL_UINT16		num_fill_color;
	PIL_UINT16		num_fill_rule;
	pil_graphic		PTRTYPE *graphic;
	pil_object		PTRTYPE *object;
	PIL_BOOL		keep_parsing;
	
	
	object = (pil_object PTRTYPE *) *pil_comp;
	graphic = &(object->graphic);
	PIL_FORCE_BEG();
	
	
	PIL_FORCE_SYM(); if (!pil_is_kw_color_model()) 	return (PIL_BAD_SYNTAX);
	PIL_FORCE_SYM(); if (!pil_is_kw_cm_value()) 	return (PIL_BAD_SYNTAX);
	graphic->color_model = (PIL_COLOR_MODEL) pil_parser_state.val1;
	
	
	PIL_FORCE_SYM(); if (!pil_is_kw_render_op()) return (PIL_BAD_SYNTAX);
	PIL_FORCE_SYM(); if (!pil_is_kw_rop_value()) return (PIL_BAD_SYNTAX);
	graphic->renderop = (PIL_RENDER_OP) pil_parser_state.val1;
	
	
	PIL_FORCE_SYM();
	if (!pil_is_kw_render_attr()) return (PIL_BAD_SYNTAX);
	PIL_FORCE_BEG();
	
	
	num_stroke_color = 0;
	num_position = 0;
	num_width = 0;
	num_cap = 0;
	num_join = 0;
	num_miter_limit = 0;
	num_fill_color = 0;
	num_fill_rule = 0;
	keep_parsing = PIL_TRUE;
	
	
	while (keep_parsing)
	{
		type = pil_get_token ();
		switch (type)
		{
			case PIL_TOKEN_SYMBOL:
			{
				if (pil_is_kw_stroke_color() && PIL_IS_STROKE_OP(graphic->renderop))
				{
					if (num_stroke_color > 0) return (PIL_BAD_SYNTAX);
					++num_stroke_color;
					ret_code = pil_parse_color_value (&graphic->stroke_color, 
					graphic->color_model);
					if (ret_code != PIL_OK) return (ret_code);
					continue;
				}
				
				
				if (pil_is_kw_position() && PIL_IS_STROKE_OP(graphic->renderop))
				{
					if (num_position > 0) return (PIL_BAD_SYNTAX);
					++num_position;
					ret_code = pil_parse_stroke_position (graphic);
					if (ret_code != PIL_OK) return (ret_code);
					continue;
				}
				
				
				if (pil_is_kw_width() && PIL_IS_STROKE_OP(graphic->renderop))
				{
					if (num_width > 0) return (PIL_BAD_SYNTAX);
					++num_width;
					ret_code = pil_parse_stroke_width (graphic);
					if (ret_code != PIL_OK) return (ret_code);
					continue;
				}
				
				
				if (pil_is_kw_cap() && PIL_IS_STROKE_OP(graphic->renderop))
				{
					if (num_cap > 0) return (PIL_BAD_SYNTAX);
					++num_cap;
					ret_code = pil_parse_stroke_cap (graphic);
					if (ret_code != PIL_OK) return (ret_code);
					continue;
				}
				
				
				if (pil_is_kw_join() && PIL_IS_STROKE_OP(graphic->renderop))
				{
					if (num_join > 0) return (PIL_BAD_SYNTAX);
					++num_join;
					ret_code = pil_parse_stroke_join (graphic);
					if (ret_code != PIL_OK) return (ret_code);
					continue;
				}
				
				
				if (pil_is_kw_miter_limit() && PIL_IS_STROKE_OP(graphic->renderop))
				{
					if (num_miter_limit > 0) return (PIL_BAD_SYNTAX);
					++num_miter_limit;
					ret_code = pil_parse_stroke_mlimit (graphic);
					if (ret_code != PIL_OK) return (ret_code);
					continue;
				}
				
				
				if (pil_is_kw_fill_color() && PIL_IS_FILL_OP(graphic->renderop))
				{
					if (num_fill_color > 0) return (PIL_BAD_SYNTAX);
					++num_fill_color;
					ret_code = pil_parse_color_value (&graphic->fill_color, 
					graphic->color_model);
					if (ret_code != PIL_OK) return (ret_code);
					continue;
				}
				
				
				if (pil_is_kw_fill_rule() && PIL_IS_FILL_OP(graphic->renderop))
				{
					if (num_fill_rule > 0) return (PIL_BAD_SYNTAX);
					++num_fill_rule;
					ret_code = pil_parse_fill_rule (graphic);
					if (ret_code != PIL_OK) return (ret_code);
					continue;
				}
				return (PIL_BAD_SYNTAX);
			}
			
			
			case PIL_TOKEN_END_BLOCK:	keep_parsing = PIL_FALSE; break;
			default:					return (PIL_BAD_SYNTAX);
		}
	}


	PIL_FORCE_SYM();
	if (!pil_is_graphic_prim()) return (PIL_BAD_SYNTAX);
	graphic->shape.type = (PIL_PRIMITIVE) pil_parser_state.val1;
	

	switch (graphic->shape.type)
	{
		case PIL_LINE_SHAPE:
		{
			if (graphic->renderop != PIL_STROKE_RENDER_OP) return (PIL_BAD_SYNTAX);
			ret_code = pil_parse_line (&(graphic->shape.shape.line));
			break;
		}
		case PIL_ARC_SHAPE:
		{
			if (graphic->renderop != PIL_STROKE_RENDER_OP) return (PIL_BAD_SYNTAX);
			ret_code = pil_parse_arc (&(graphic->shape.shape.arc));
			break;
		}
		case PIL_BEZIER_SHAPE:
		{
			if (graphic->renderop != PIL_STROKE_RENDER_OP) return (PIL_BAD_SYNTAX);
			ret_code = pil_parse_bezier (&(graphic->shape.shape.bezier));
			break;
		}
		case PIL_POLYLINE_SHAPE:
		{
			if (graphic->renderop != PIL_STROKE_RENDER_OP) return (PIL_BAD_SYNTAX);
			ret_code = pil_parse_poly_line (&(graphic->shape.shape.polyline));
			break;
		}
		case PIL_OPENPATH_SHAPE:
		{
			if (graphic->renderop != PIL_STROKE_RENDER_OP) return (PIL_BAD_SYNTAX);
			ret_code = pil_parse_open_path (&(graphic->shape.shape.openpath));
			break;
		}
		case PIL_RECTANGLE_SHAPE:
		{
			ret_code = pil_parse_rectangle (&(graphic->shape.shape.rect));
			break;
		}
		case PIL_CIRCLE_SHAPE:
		{
			ret_code = pil_parse_circle (&(graphic->shape.shape.circle));
			break;
		}
		case PIL_ELLIPSE_SHAPE:
		{
			ret_code = pil_parse_ellipse (&(graphic->shape.shape.ellipse));
			break;
		}
		case PIL_POLYGON_SHAPE:
		{
			ret_code = pil_parse_polygon (&(graphic->shape.shape.polygon), 
			PIL_SHAPE_COMPLEX);
			break;
		}
		case PIL_CLOSEDPATH_SHAPE:
		{
			ret_code = pil_parse_closed_path (&(graphic->shape.shape.closedpath), 
			PIL_SHAPE_COMPLEX);
			break;
		}
		case PIL_CLIPPER_SHAPE:
		{
			ret_code = PIL_OK;
			break;
		}
		case PIL_CONTAINER_SHAPE:
		{
			ret_code = PIL_OK;
			break;
		}
	}
	

	if (ret_code != PIL_OK) return (ret_code);	
	PIL_FORCE_END();
	if (num_stroke_color == 0) graphic->renderop &= ~PIL_STROKE_RENDER_OP;
	if (num_fill_color == 0) graphic->renderop &= ~PIL_FILL_RENDER_OP;
	return (PIL_OK);
}


PIL_ERROR pil_parse_color_value (color_value, color_model)
pil_color PTRTYPE *color_value;
PIL_COLOR_MODEL color_model;
{
	PIL_DOUBLE 	gray;
	PIL_DOUBLE 	r, g, b;
	PIL_DOUBLE 	c, m, y, k;
	PIL_ERROR	rc;
	

	switch (color_model)
	{
		case PIL_DEV_GRAY_CM:
		case PIL_CAL_GRAY_CM:
		{
			PIL_FORCE_DBL(gray);
			PIL_RANGE_COL(gray);
			if (color_model == PIL_DEV_GRAY_CM) color_value->devgray.value = gray;
			else color_value->calgray.value = gray;
			return (PIL_OK);
		}
		
		
		case PIL_DEV_RGB_CM:
		case PIL_CAL_RGB_CM:
		{
			PIL_FORCE_BEG();
			PIL_FORCE_DBL(r);
			PIL_RANGE_COL(r);
			PIL_FORCE_DBL(g);
			PIL_RANGE_COL(g);
			PIL_FORCE_DBL(b);
			PIL_RANGE_COL(b);
			PIL_FORCE_END();
			
			
			if (color_model == PIL_DEV_RGB_CM) 
			{
				color_value->devrgb.r = r;
				color_value->devrgb.g = g;
				color_value->devrgb.b = b;
			}
			else 
			{
				color_value->calrgb.r = r;
				color_value->calrgb.g = g;
				color_value->calrgb.b = b;
			}
			return (PIL_OK);
		}
		
		
		case PIL_PANTONE_CM:
		case PIL_SPOT_CM:
		{
			PIL_FORCE_STR();
			if (color_model == PIL_PANTONE_CM) 
			{
				rc = pil_append_string ((char PTRTYPE *) pil_parser_state.token,
				&(color_value->pantone.name));
			}
			else 
			{
				rc = pil_append_string ((char PTRTYPE *) pil_parser_state.token,
				&(color_value->spot.name));
			}
			return (rc);
		}
		
		
		case PIL_DEV_CMYK_CM:
		{
			PIL_FORCE_BEG();
			PIL_FORCE_DBL(c);
			PIL_RANGE_COL(c);
			PIL_FORCE_DBL(m);
			PIL_RANGE_COL(m);
			PIL_FORCE_DBL(y);
			PIL_RANGE_COL(y);
			PIL_FORCE_DBL(k);
			PIL_RANGE_COL(k);
			PIL_FORCE_END();
			
			
			color_value->devcmyk.c = c;
			color_value->devcmyk.m = m;
			color_value->devcmyk.y = y;
			color_value->devcmyk.k = k;
			return (PIL_OK);
		}
	}
}


PIL_ERROR pil_parse_stroke_position	(graphic)
pil_graphic PTRTYPE *graphic;
{
	PIL_FORCE_SYM();
	if (pil_is_kw_inside())		{ graphic->lineposition = PIL_INSIDE;	return (PIL_OK); }
	if (pil_is_kw_outside())  	{ graphic->lineposition = PIL_OUTSIDE;	return (PIL_OK); }
	if (pil_is_kw_centered()) 	{ graphic->lineposition = PIL_CENTERED;	return (PIL_OK); }
	return (PIL_BAD_SYNTAX);
}


PIL_ERROR pil_parse_stroke_width (graphic)
pil_graphic PTRTYPE *graphic;
{
	PIL_UINT32 width;
	PIL_FORCE_U32(width);
	graphic->linewidth = width;
	return (PIL_OK);
}


PIL_ERROR pil_parse_stroke_cap (graphic)
pil_graphic PTRTYPE *graphic;
{
	PIL_FORCE_SYM();
	if (pil_is_kw_butt_cap())		
	{ 
		graphic->linecap = PIL_BUTT_CAP;		
		return (PIL_OK); 
	}
	if (pil_is_kw_round_cap())		
	{ 
		graphic->linecap = PIL_ROUND_CAP;		
		return (PIL_OK); 
	}
	if (pil_is_kw_projecting_cap())	
	{ 
		graphic->linecap = PIL_PROJECTING_CAP;
		return (PIL_OK); 
	}
	return (PIL_BAD_SYNTAX);
}


PIL_ERROR pil_parse_stroke_join (graphic)
pil_graphic PTRTYPE *graphic;
{
	PIL_FORCE_SYM();
	if (pil_is_kw_miter_join())	{ graphic->linejoin = PIL_MITER_JOIN;	return (PIL_OK); }
	if (pil_is_kw_bevel_join())	{ graphic->linejoin = PIL_BEVEL_JOIN;	return (PIL_OK); }
	if (pil_is_kw_round_join())	{ graphic->linejoin = PIL_ROUND_JOIN;	return (PIL_OK); }
	return (PIL_BAD_SYNTAX);
}


PIL_ERROR pil_parse_stroke_mlimit (graphic)
pil_graphic PTRTYPE *graphic;
{
	PIL_DOUBLE miter_limit;
	PIL_FORCE_DBL(miter_limit);
	if (miter_limit < 1) return (PIL_BAD_SYNTAX);
	graphic->miter_limit = miter_limit;
	return (PIL_OK);
}


PIL_ERROR pil_parse_fill_rule (graphic)
pil_graphic PTRTYPE *graphic;
{
	PIL_FORCE_SYM();
	if (pil_is_kw_even_odd())	
	{ 
		graphic->fill_rule = PIL_EVEN_ODD; 		
		return (PIL_OK); 
	}
	if (pil_is_kw_nz_winding())	
	{ 
		graphic->fill_rule = PIL_NON_ZERO_WINDING; 
		return (PIL_OK); 
	}	
	return (PIL_BAD_SYNTAX);
}


PIL_ERROR pil_parse_line (line)
pil_line PTRTYPE *line;
{
	PIL_FORCE_BEG();
	PIL_FORCE_I32(line->start.x);
	PIL_FORCE_I32(line->start.y);
	PIL_FORCE_I32(line->end.x);
	PIL_FORCE_I32(line->end.y);
	PIL_FORCE_END();
	return (PIL_OK);
}


PIL_ERROR pil_parse_arc (arc)
pil_arc PTRTYPE *arc;
{
	PIL_FORCE_BEG();
	PIL_FORCE_I32(arc->arc_square.ul.x);
	PIL_FORCE_I32(arc->arc_square.ul.y);
	PIL_FORCE_U32(arc->arc_square.length);
	PIL_FORCE_DBL(arc->start_angle);
	PIL_FORCE_DBL(arc->end_angle);
	PIL_FORCE_END();
	return (PIL_OK);
}


PIL_ERROR pil_parse_bezier (bezier)
pil_bezier PTRTYPE *bezier;
{
	PIL_ERROR rc;
	PIL_FORCE_BEG();
	if ((rc = pil_parse_point (&bezier->start    		)) != PIL_OK) return (rc);
	if ((rc = pil_parse_point (&bezier->start_ctrl_pt	)) != PIL_OK) return (rc);
	if ((rc = pil_parse_point (&bezier->end_ctrl_pt  	)) != PIL_OK) return (rc);
	if ((rc = pil_parse_point (&bezier->end      		)) != PIL_OK) return (rc);
	PIL_FORCE_END();
	return (PIL_OK);
}


PIL_ERROR pil_parse_poly_line (poly_line)
pil_polyline PTRTYPE *poly_line;
{
	pil_point		p;
	PIL_ERROR		rc;
	PIL_UINT16		token_type;
	pil_poly_pt		PTRTYPE *poly_pt;
	pil_poly_pt		PTRPTR next;

	
	PIL_FORCE_BEG();
	rc = pil_parse_point (&p); 
	if (rc != PIL_OK) return (rc);
	poly_line->p.x = p.x;
	poly_line->p.y = p.y;
	poly_line->next = NULL;
	next = &(poly_line->next);
		
	
	rc = pil_parse_point (&p); 
	if (rc != PIL_OK) return (rc);
	rc = pil_create_polypt (&poly_pt, p.x, p.y);
	if (rc != PIL_OK) return (rc);
	*next = poly_pt;
	next = &(poly_pt->next);
	
	
	while (1)
	{
		token_type = pil_get_token ();
		switch (token_type)
		{
			case PIL_TOKEN_BEG_BLOCK:
			{
				PIL_FORCE_I32(p.x);
				PIL_FORCE_I32(p.y);
				PIL_FORCE_END();
				rc = pil_create_polypt (&poly_pt, p.x, p.y);
				if (rc != PIL_OK) return (rc);
				*next = poly_pt;
				next = &(poly_pt->next);
				continue;
			}
			case PIL_TOKEN_END_BLOCK:	return (PIL_OK);
			default:					return (PIL_BAD_SYNTAX);
		}
	}
}


PIL_ERROR pil_parse_open_path (open_path)
pil_open_path PTRTYPE *open_path;
{
	pil_point		p;
	pil_point		cnt_pt1;
	pil_point		cnt_pt2;
	pil_point		end_pt;
	PIL_INT32		radius;
	PIL_ERROR		rc;
	PIL_UINT16		token_type;
	pil_path_pt		PTRTYPE *pathpt;
	pil_path_pt		PTRPTR next;
	PIL_INT32		zero;
	
	
	PIL_FORCE_BEG();
	PIL_FORCE_SYM();
	if (!pil_is_kw_move_to()) return (PIL_BAD_SYNTAX);
	rc = pil_parse_move_to (&p); 
	if (rc != PIL_OK) return (rc);
	
	
	open_path->type = PIL_MOVE_TO;
	open_path->p.mt.pt.x = p.x;
	open_path->p.mt.pt.y = p.y;
	open_path->next = NULL;
	next = &(open_path->next);
	zero = 0;
	
	
	while (1)
	{
		token_type = pil_get_token ();
		switch (token_type)
		{
			case PIL_TOKEN_SYMBOL:
			{
				if (pil_is_kw_move_to())
				{
					rc = pil_parse_move_to (&p);
					if (rc != PIL_OK) return (rc);
					rc = pil_create_pathpt (&pathpt, PIL_MOVE_TO,
					p.x, p.y, zero, zero, zero, zero, zero);
					if (rc != PIL_OK) return (rc);
					*next = pathpt;
					next = &(pathpt->next);
					continue;
				}
				
				
				if (pil_is_kw_line_to())
				{
					rc = pil_parse_line_to (&p);
					if (rc != PIL_OK) return (rc);
					rc = pil_create_pathpt (&pathpt, PIL_LINE_TO,
					p.x, p.y, zero, zero, zero, zero, zero);
					if (rc != PIL_OK) return (rc);
					*next = pathpt;
					next = &(pathpt->next);
					continue;
				}
				
				
				if (pil_is_kw_curve_to())
				{
					rc = pil_parse_curve_to (&cnt_pt1, &cnt_pt2, &end_pt);
					if (rc != PIL_OK) return (rc);
					rc = pil_create_pathpt (&pathpt, PIL_CURVE_TO,
					cnt_pt1.x, cnt_pt1.y, cnt_pt2.x, cnt_pt2.y, end_pt.x, 
					end_pt.y, zero);
					if (rc != PIL_OK) return (rc);
					*next = pathpt;
					next = &(pathpt->next);
					continue;
				}
				
				
				if (pil_is_kw_arc_to())
				{
					rc = pil_parse_arc_to (&cnt_pt1, &cnt_pt2, &radius);
					if (rc != PIL_OK) return (rc);
					rc = pil_create_pathpt (&pathpt, PIL_ARC_TO,
					cnt_pt1.x, cnt_pt1.y, cnt_pt2.x, cnt_pt2.y, 
					zero, zero, radius);
					if (rc != PIL_OK) return (rc);
					*next = pathpt;
					next = &(pathpt->next);
					continue;
				}
				return (PIL_BAD_SYNTAX);
			}
			
			
			case PIL_TOKEN_END_BLOCK:	return (PIL_OK);
			default:					return (PIL_BAD_SYNTAX);
		}
	}
}


PIL_ERROR pil_parse_rc_type (pil_comp)
pil_component PTRPTR pil_comp;
{
	PIL_UINT16 	token_type;
	PIL_RC_TYPE	rc_type;
	pil_object	PTRTYPE *object;


	PIL_FORCE_BEG();
	rc_type = PIL_RC_UNKNOWN;
	while (1)
	{
		token_type = pil_get_token ();
		switch (token_type)
		{
			case PIL_TOKEN_SYMBOL:
			{
				if (pil_is_kw_rc_type_text())
				{
					if (rc_type & PIL_RC_TEXT) return (PIL_BAD_SYNTAX);
					rc_type |= PIL_RC_TEXT;
					continue;
				}
				
				
				if (pil_is_kw_rc_type_geometry())
				{
					if (rc_type & PIL_RC_GEOMETRY) return (PIL_BAD_SYNTAX);
					rc_type |= PIL_RC_GEOMETRY;
					continue;
				}
				
				
				if (pil_is_kw_rc_type_lineart())
				{
					if (rc_type & PIL_RC_LINEART) return (PIL_BAD_SYNTAX);
					rc_type |= PIL_RC_LINEART;
					continue;
				}
				
				
				if (pil_is_kw_rc_type_image())
				{
					if (rc_type & PIL_RC_IMAGE) return (PIL_BAD_SYNTAX);
					rc_type |= PIL_RC_IMAGE;
					continue;
				}
				return (PIL_BAD_SYNTAX);
			}
			
			
			case PIL_TOKEN_END_BLOCK:
			{
				object = (pil_object PTRTYPE *) *pil_comp;
				object->rctype = rc_type;
				return (PIL_OK);
			}
			
			
			default:
			{
				return (PIL_BAD_SYNTAX);
			}
		}
	}
}


PIL_ERROR pil_parse_rc_name (pil_comp)
pil_component PTRPTR pil_comp;
{
	pil_object PTRTYPE *object;
	PIL_FORCE_STR();
	object = (pil_object PTRTYPE *) *pil_comp;
	return (pil_append_string ((char PTRTYPE *) pil_parser_state.token, 
	&(object->rcname)));
}


PIL_ERROR pil_parse_bbox (pil_comp)
pil_component PTRPTR pil_comp;
{
	PIL_DOUBLE 	x;
	PIL_DOUBLE 	y;
	PIL_DOUBLE 	w;
	PIL_DOUBLE 	h;
	pil_object	PTRTYPE *object;
	
	
	PIL_FORCE_BEG();
	PIL_FORCE_DBL(x);
	PIL_FORCE_DBL(y);
	PIL_FORCE_DBL(w); if (w <= 0) return (PIL_BAD_SYNTAX);
	PIL_FORCE_DBL(h); if (h <= 0) return (PIL_BAD_SYNTAX);
	PIL_FORCE_END();


	object = (pil_object PTRTYPE *) *pil_comp;
	object->src_rect.ul.x 	= x;
	object->src_rect.ul.y 	= y;
	object->src_rect.width 	= w;
	object->src_rect.height	= h;
	return (PIL_OK);
}


PIL_ERROR pil_parse_text_flow (pil_comp)
pil_component PTRPTR pil_comp;
{
	PIL_ERROR 			ret_code;
	PIL_UINT16 			type;
	PIL_UINT16			num_flow_label;
	PIL_UINT16			num_flow_objects;
	PIL_UINT16			num_flow_from;
	PIL_UINT16			num_flow_to;
	PIL_UINT16			num_flow_content;
	
	
	PIL_FORCE_BEG();
	ret_code = pil_make_text_flow_comp (pil_comp);
	if (ret_code != PIL_OK) return (ret_code);
	
	
	num_flow_label = 0;
	num_flow_objects = 0;
	num_flow_from = 0;
	num_flow_to = 0;
	num_flow_content = 0;
	
	
	while (1)
	{
		type = pil_get_token ();
		switch (type)
		{
			case PIL_TOKEN_SYMBOL:
			{
				if (pil_is_kw_flow_label())
				{
					if (num_flow_label > 0) return (PIL_BAD_SYNTAX);
					++num_flow_label;
					ret_code = pil_parse_flow_label (pil_comp);
					if (ret_code != PIL_OK) return (ret_code);
					continue;
				}
				
				
				if (pil_is_kw_flow_objects())
				{
					if (num_flow_objects > 0) return (PIL_BAD_SYNTAX);
					++num_flow_objects;
					ret_code = pil_parse_flow_objects (pil_comp);
					if (ret_code != PIL_OK) return (ret_code);
					continue;
				}
				
				
				if (pil_is_kw_flow_from())
				{
					if (num_flow_from > 0) return (PIL_BAD_SYNTAX);
					++num_flow_from;
					ret_code = pil_parse_flow_from (pil_comp);
					if (ret_code != PIL_OK) return (ret_code);
					continue;
				}
				
				
				if (pil_is_kw_flow_to())
				{
					if (num_flow_to > 0) return (PIL_BAD_SYNTAX);
					++num_flow_to;
					ret_code = pil_parse_flow_to (pil_comp);
					if (ret_code != PIL_OK) return (ret_code);
					continue;
				}
				
				
				if (pil_is_kw_flow_content())
				{
					if (num_flow_content > 0) return (PIL_BAD_SYNTAX);
					++num_flow_content;
					ret_code = pil_parse_flow_content (pil_comp);
					if (ret_code != PIL_OK) return (ret_code);
					continue;
				}
				return (PIL_BAD_SYNTAX);
			}
			
			
			case PIL_TOKEN_END_BLOCK:	return (PIL_OK);
			default:					return (PIL_BAD_SYNTAX);
		}
	}
}


PIL_ERROR pil_parse_flow_label (pil_comp)
pil_component PTRPTR pil_comp;
{
	pil_text_flow PTRTYPE *text_flow;
	PIL_FORCE_STR();
	text_flow = (pil_text_flow PTRTYPE *) *pil_comp;
	return (pil_append_string ((char PTRTYPE *) pil_parser_state.token, 
	&(text_flow->label)));
}


PIL_ERROR pil_parse_flow_from (pil_comp)
pil_component PTRPTR pil_comp;
{
	PIL_UINT16		token_type;
	pil_text_flow 	PTRTYPE *text_flow;
	PIL_ERROR 		rc;


	text_flow = (pil_text_flow PTRTYPE *) *pil_comp;
	PIL_FORCE_BEG();
	PIL_FORCE_STR();
	rc = pil_append_string ((char PTRTYPE *) pil_parser_state.token, 
	&(text_flow->from_layout));
	if (rc != PIL_OK) return (rc);


	token_type = pil_get_token ();
	switch (token_type)
	{
		case PIL_TOKEN_STRING:
		{
			rc = pil_append_string ((char PTRTYPE *) pil_parser_state.token, 
			&(text_flow->from_label));
			if (rc != PIL_OK) return (rc);
			break;
		}
		case PIL_TOKEN_END_BLOCK:	return (PIL_OK);
		default:					return (PIL_BAD_SYNTAX);
	}
	
	
	PIL_FORCE_END();
	return (PIL_OK);
}


PIL_ERROR pil_parse_flow_to (pil_comp)
pil_component PTRPTR pil_comp;
{
	PIL_UINT16		token_type;
	pil_text_flow 	PTRTYPE *text_flow;
	PIL_ERROR 		rc;
	
	
	text_flow = (pil_text_flow PTRTYPE *) *pil_comp;
	PIL_FORCE_BEG();
	PIL_FORCE_STR();
	rc = pil_append_string ((char PTRTYPE *) pil_parser_state.token, 
	&(text_flow->to_layout));
	if (rc != PIL_OK) return (rc);
	
	
	token_type = pil_get_token ();
	switch (token_type)
	{
		case PIL_TOKEN_STRING:
		{
			rc = pil_append_string ((char PTRTYPE *) pil_parser_state.token, 
			&(text_flow->to_label));
			if (rc != PIL_OK) return (rc);
			break;
		}
		case PIL_TOKEN_END_BLOCK:	return (PIL_OK);
		default:					return (PIL_BAD_SYNTAX);
	}
	
	
	PIL_FORCE_END();
	return (PIL_OK);
}


PIL_ERROR pil_parse_flow_objects (pil_comp)
pil_component PTRPTR pil_comp;
{
	PIL_UINT16			token_type;
	pil_text_flow		PTRTYPE *text_flow;
	pil_object_id_list	PTRTYPE *obj_list;
	pil_object_id_list	PTRPTR next;
	PIL_ERROR			rc;
	
	
	PIL_FORCE_BEG();
	text_flow = (pil_text_flow PTRTYPE *) *pil_comp;
	next = &(text_flow->objects);
	
	
	while (1)
	{
		token_type = pil_get_token ();
		switch (token_type)
		{
			case PIL_TOKEN_STRING:
			{
				rc = pil_create_objid (&obj_list, (char PTRTYPE *) 
				pil_parser_state.token);
				if (rc != PIL_OK) return (rc);
				*next = obj_list;
				next = &(obj_list->next);
				continue;
			}
			case PIL_TOKEN_END_BLOCK:	return (PIL_OK);
			default:					return (PIL_BAD_SYNTAX);
		}
	}
}


PIL_ERROR pil_parse_flow_content (pil_comp)
pil_component PTRPTR pil_comp;
{
	PIL_UINT16 			token_type;
	pil_text_flow		PTRTYPE *text_flow;
	pil_object_id_list	PTRTYPE *cnt_list;
	pil_object_id_list	PTRPTR next;
	PIL_ERROR			rc;


	PIL_FORCE_BEG();
	text_flow = (pil_text_flow PTRTYPE *) *pil_comp;
	next = &(text_flow->content);
	
	
	while (1)
	{
		token_type = pil_get_token ();
		switch (token_type)
		{
			case PIL_TOKEN_STRING:
			{
				rc = pil_create_objid (&cnt_list, (char PTRTYPE *) 
				pil_parser_state.token);
				if (rc != PIL_OK) return (rc);
				*next = cnt_list;
				next = &(cnt_list->next);
				continue;
			}
			case PIL_TOKEN_END_BLOCK:	return (PIL_OK);
			default:					return (PIL_BAD_SYNTAX);
		}
	}
}


PIL_ERROR pil_parse_group (pil_comp)
pil_component PTRPTR pil_comp;
{
	PIL_ERROR 	ret_code;
	PIL_UINT16 	type;
	PIL_UINT16	num_group_id;
	PIL_UINT16	num_group_objects;
	
	
	PIL_FORCE_BEG();
	ret_code = pil_make_group_comp (pil_comp);
	if (ret_code != PIL_OK) return (ret_code);
	num_group_id = 0;
	num_group_objects = 0;
	
	
	while (1)
	{
		type = pil_get_token ();
		switch (type)
		{
			case PIL_TOKEN_SYMBOL:
			{
				if (pil_is_kw_id())
				{
					if (num_group_id > 0) return (PIL_BAD_SYNTAX);
					++num_group_id;
					ret_code = pil_parse_group_id (pil_comp);
					if (ret_code != PIL_OK) return (ret_code);
					continue;
				}
				
				
				if (pil_is_kw_group_objects())
				{
					if (num_group_objects > 0) return (PIL_BAD_SYNTAX);
					++num_group_objects;
					ret_code = pil_parse_group_objects (pil_comp);
					if (ret_code != PIL_OK) return (ret_code);
					continue;
				}
				return (PIL_BAD_SYNTAX);
			}
			
			
			case PIL_TOKEN_END_BLOCK:	return (PIL_OK);
			default:					return (PIL_BAD_SYNTAX);
		}
	}
}


PIL_ERROR pil_parse_group_id (pil_comp)
pil_component PTRPTR pil_comp;
{
	pil_group PTRTYPE *group;
	PIL_FORCE_STR();
	group = (pil_group PTRTYPE *) *pil_comp;
	return (pil_append_string ((char PTRTYPE *) pil_parser_state.token, 
	&(group->name)));
}


PIL_ERROR pil_parse_group_objects (pil_comp)
pil_component PTRPTR pil_comp;
{
	PIL_UINT16 			token_type;
	pil_group			PTRTYPE *group;
	pil_object_id_list	PTRTYPE *obj_list;
	pil_object_id_list	PTRPTR next;
	PIL_ERROR			rc;


	PIL_FORCE_BEG();
	group = (pil_group PTRTYPE *) *pil_comp;
	next = &(group->objects);
	
	
	while (1)
	{
		token_type = pil_get_token ();
		switch (token_type)
		{
			case PIL_TOKEN_STRING:
			{
				rc = pil_create_objid (&obj_list, (char PTRTYPE *) 
				pil_parser_state.token);
				if (rc != PIL_OK) return (rc);
				*next = obj_list;
				next = &(obj_list->next);
				continue;
			}
			case PIL_TOKEN_END_BLOCK:	return (PIL_OK);
			default:					return (PIL_BAD_SYNTAX);
		}
	}
}


PIL_ERROR pil_parse_cnt_start (pil_comp)
pil_component PTRPTR pil_comp;
{
	pil_copy_to_buf1();
	PIL_FORCE_STR();
	pil_copy_to_buf2();
	PIL_FORCE_BEG();
	return (pil_make_cnt_start_comp (pil_comp));
}


PIL_ERROR pil_parse_cnt_header (pil_comp)
pil_component PTRPTR pil_comp;
{
	if (!pil_is_kw_header()) return (PIL_BAD_SYNTAX);
	PIL_FORCE_BEG();
	
	
	PIL_FORCE_SYM();
	if (pil_is_kw_little_endian()) pil_parser_state.cnt_ordering = PIL_LITTLE_ENDIAN;
	else if (pil_is_kw_big_endian()) pil_parser_state.cnt_ordering = PIL_BIG_ENDIAN;
	else return (PIL_BAD_SYNTAX);
	
	
	PIL_FORCE_SYM();
	if (pil_is_kw_ascii()) pil_parser_state.cnt_encoding = PIL_ASCII_CONTENT;
	else if (pil_is_kw_binary()) pil_parser_state.cnt_encoding = PIL_BINARY_CONTENT;
	else return (PIL_BAD_SYNTAX);
	
	
	pil_get_token ();
	if (!pil_is_token_uint8()) return (PIL_BAD_SYNTAX);
	pil_parser_state.cnt_escape = (PIL_UINT8) pil_parser_state.token_constant;


	pil_get_token ();
	if (!pil_is_token_uint8()) return (PIL_BAD_SYNTAX);
	pil_parser_state.cnt_beg = (PIL_UINT8) pil_parser_state.token_constant;


	pil_get_token ();
	if (!pil_is_token_uint8()) return (PIL_BAD_SYNTAX);
	pil_parser_state.cnt_end = (PIL_UINT8) pil_parser_state.token_constant;
	
	
	PIL_FORCE_END();
	if (pil_parser_state.cnt_escape == pil_parser_state.cnt_end) return (PIL_BAD_SYNTAX);
	return (pil_make_cnt_header_comp (pil_comp));
}


PIL_ERROR pil_parse_ascii_cnt (pil_comp)
pil_component PTRPTR pil_comp;
{
	int					char_code;
	PIL_UINT8			esc_ch;
	PIL_UINT8			beg_ch;
	PIL_UINT8			end_ch;
	PIL_UINT8			cur_ch;
	PIL_UINT16			state;
	PIL_UINT8			hex_digit1;
	PIL_UINT8			hex_digit2;
	PIL_UINT32			buf_count;
	pil_content_data	PTRPTR content_data;
	char				PTRTYPE *pdata;
	PIL_ERROR			rc;
	
	
	if (pil_parser_state.cnt_start_state)
	{
		PIL_FORCE_SYM();
		if (!pil_is_kw_data()) return (PIL_BAD_SYNTAX);
		state = PIL_ASC_S_BEG_SEARCH;
		buf_count = pil_cnt_buf_size;
		pil_parser_state.cnt_start_state = PIL_FALSE;
	}
	else
	{
		state = PIL_ASC_S_FETCH_DATA;
		buf_count = pil_cnt_buf_size;
	}
	
	
	esc_ch = pil_parser_state.cnt_escape;
	beg_ch = pil_parser_state.cnt_beg;
	end_ch = pil_parser_state.cnt_end;
	
	
	content_data = (pil_content_data PTRPTR) pil_comp;
	rc = pil_create_content_data (content_data);
	if (rc != PIL_OK) return (rc);
	rc = pil_alloc (buf_count, (PIL_VOID PTRPTR) &pdata);
	if (rc != PIL_OK) return (rc);
	(**content_data).data = pdata;
	
	
	while (1)
	{
		switch (state)
		{
			case PIL_ASC_S_BEG_SEARCH:
			{
				if (pil_parser_state.initial_fetch) char_code = (*pil_getchar)();
				else char_code = pil_parser_state.char_code;
				pil_parser_state.initial_fetch = PIL_TRUE;
				if (char_code == PIL_REOF) { state = PIL_ASC_S_ERROR; continue; }
				cur_ch = (PIL_UINT8) char_code;
				if (cur_ch == PIL_END_OF_LINE) ++pil_parser_state.line_number;
				if (cur_ch == beg_ch) { state = PIL_ASC_S_FETCH_DATA; continue; }
				continue;
			}
			
			
			case PIL_ASC_S_FETCH_DATA:
			{
				char_code = (*pil_getchar)();
				if (char_code == PIL_REOF) { state = PIL_ASC_S_ERROR; continue; }
				cur_ch = (PIL_UINT8) char_code;
				if (cur_ch == PIL_END_OF_LINE) ++pil_parser_state.line_number;
				if (cur_ch == esc_ch) { state = PIL_ASC_S_ESCAPE; continue; }
				if (cur_ch == end_ch) { state = PIL_ASC_S_END; continue; }
				state = PIL_ASC_S_PUT_DATA;
				continue;
			}
			
			
			case PIL_ASC_S_ESCAPE:
			{
				char_code = (*pil_getchar)();
				if (char_code == PIL_REOF) { state = PIL_ASC_S_ERROR; continue; }
				cur_ch = (PIL_UINT8) char_code;
				if (pil_is_hex_digit (cur_ch))
				{
					hex_digit1 = cur_ch;
					char_code = (*pil_getchar)();
					if (char_code == PIL_REOF) { state = PIL_ASC_S_ERROR; continue; }
					cur_ch = (PIL_UINT8) char_code;
					if (pil_is_hex_digit (cur_ch))
					{
						hex_digit2 = cur_ch;
						cur_ch = pil_combine_hex_digits (hex_digit1, hex_digit2);
						state = PIL_ASC_S_PUT_DATA;
						continue;
					}
					else
					{
						state = PIL_ASC_S_ERROR; 
						continue;
					}
				}
				else
				{
					state = PIL_ASC_S_ERROR; 
					continue;
				}
			}
			
			
			case PIL_ASC_S_PUT_DATA:
			{
				*pdata++ = cur_ch;
				--buf_count;
				if (buf_count == 0)
				{
					(**content_data).length = pil_cnt_buf_size;
					return (PIL_OK);
				}
				else
				{
					state = PIL_ASC_S_FETCH_DATA;
					continue;
				}
			}
			
			
			case PIL_ASC_S_ERROR:
			{
				return (PIL_BAD_SYNTAX);
			}
			
			
			case PIL_ASC_S_END:
			{
				(**content_data).length = pil_cnt_buf_size - buf_count;
				pil_parser_state.cnt_all_done = PIL_TRUE;
				return (PIL_OK);
			}
		}
	}
}


PIL_ERROR pil_parse_binary_cnt (pil_comp)
pil_component PTRPTR pil_comp;
{
	int					char_code;
	PIL_UINT8			cur_ch;
	PIL_UINT16			state;
	PIL_UINT8			hex_digit1;
	PIL_UINT8			hex_digit2;
	PIL_UINT16			hex_count;
	PIL_UINT32			buf_count;
	pil_content_data	PTRPTR content_data;
	char				PTRTYPE *pdata;
	PIL_ERROR			rc;

	
	if (pil_parser_state.cnt_start_state)
	{
		PIL_FORCE_SYM();
		if (!pil_is_kw_data()) return (PIL_BAD_SYNTAX);
		PIL_FORCE_BEG();
		pil_parser_state.cnt_start_state = PIL_FALSE;
	}
	
	
	state = PIL_BIN_S_SPIN;
	hex_count = 0;
	buf_count = pil_cnt_buf_size;
	
	
	content_data = (pil_content_data PTRPTR) pil_comp;
	rc = pil_create_content_data (content_data);
	if (rc != PIL_OK) return (rc);
	rc = pil_alloc (buf_count, (PIL_VOID PTRPTR) &pdata);
	if (rc != PIL_OK) return (rc);
	(**content_data).data = pdata;


	while (1)
	{
		switch (state)
		{
			case PIL_BIN_S_SPIN:
			{
				if (pil_parser_state.initial_fetch) char_code = (*pil_getchar)();
				else char_code = pil_parser_state.char_code;
				pil_parser_state.initial_fetch = PIL_TRUE;
				if (char_code == PIL_REOF) { state = PIL_BIN_S_ERROR; continue; }
				cur_ch = (PIL_UINT8) char_code;
				if (cur_ch == PIL_END_OF_LINE) ++pil_parser_state.line_number;
				if (pil_is_white_space (cur_ch)) continue;
				if (cur_ch == '/') { state = PIL_BIN_S_COMMENT_BEG; continue; }
				if (cur_ch == '}') { state = PIL_BIN_S_END; continue; }
				if (pil_is_hex_digit (cur_ch))
				{
					++hex_count;
					if (hex_count == 1)
					{
						hex_digit1 = cur_ch;
						state = PIL_BIN_S_SPIN;
						continue;
					}
					else
					{
						hex_count = 0;
						hex_digit2 = cur_ch;
						cur_ch = pil_combine_hex_digits (hex_digit1, hex_digit2);
						state = PIL_BIN_S_PUT_DATA;
						continue;
					}
				}
				state = PIL_BIN_S_ERROR;
				continue;
			}
			
			
			case PIL_BIN_S_COMMENT_BEG:
			{
				char_code = (*pil_getchar)();
				if (char_code == PIL_REOF) { state = PIL_BIN_S_ERROR; continue; }
				cur_ch = (PIL_UINT8) char_code;
				if (cur_ch == '/') { state = PIL_BIN_S_COMMENT_1; continue; }
				if (cur_ch == '*') { state = PIL_BIN_S_COMMENT_2; continue; }
				state = PIL_BIN_S_ERROR;
				continue;
			}
			
			
			case PIL_BIN_S_COMMENT_1:
			{
				char_code = (*pil_getchar)();
				if (char_code == PIL_REOF) { state = PIL_BIN_S_ERROR; continue; }
				cur_ch = (PIL_UINT8) char_code;
				if (cur_ch == PIL_END_OF_LINE) 
				{
					++pil_parser_state.line_number;
					state = PIL_BIN_S_SPIN;
				}
				continue;
			}
			
			
			case PIL_BIN_S_COMMENT_2:
			{
				char_code = (*pil_getchar)();
				if (char_code == PIL_REOF) { state = PIL_BIN_S_ERROR; continue; }
				cur_ch = (PIL_UINT8) char_code;
				if (cur_ch == '*')
				{
					char_code = (*pil_getchar)();
					if (char_code == PIL_REOF) { state = PIL_BIN_S_ERROR; continue; }
					cur_ch = (PIL_UINT8) char_code;
					if (cur_ch == '/') { state = PIL_BIN_S_SPIN; continue; }
					if (cur_ch == PIL_END_OF_LINE) ++pil_parser_state.line_number;
					continue;
				}
				else
				{
					if (cur_ch == PIL_END_OF_LINE) ++pil_parser_state.line_number;
					continue;
				}
			}
			
			
			case PIL_BIN_S_PUT_DATA:
			{
				*pdata++ = cur_ch;
				--buf_count;
				if (buf_count == 0)
				{
					(**content_data).length = pil_cnt_buf_size;
					return (PIL_OK);
				}
				else
				{
					state = PIL_BIN_S_SPIN;
					continue;
				}
			}
			
			
			case PIL_BIN_S_ERROR:
			{
				return (PIL_BAD_SYNTAX);
			}
			
			
			case PIL_BIN_S_END:
			{
				if (hex_count == 1) { state = PIL_BIN_S_ERROR; continue; }
				(**content_data).length = pil_cnt_buf_size - buf_count;
				pil_parser_state.cnt_all_done = PIL_TRUE;
				return (PIL_OK);
			}
		}
	}
}


/* ------------------------------------------- */
/* PIL component creation functions begin here */
/* ------------------------------------------- */


PIL_ERROR pil_make_layout_start_comp (pil_comp)
pil_component PTRPTR pil_comp;
{
	PIL_ERROR 			rc;
	pil_layout_start	PTRPTR pls;
	char				PTRTYPE *name_p;

	
	pls = (pil_layout_start PTRPTR) pil_comp;
	rc = pil_create_layout_start (pls);
	if (rc != PIL_OK) return (rc);
	name_p = (char PTRTYPE *) &pil_parser_state.buf2[0];
	rc = pil_append_string (name_p, &((*pls)->name));
	return (rc);
}


PIL_ERROR pil_make_layout_end_comp (pil_comp)
pil_component PTRPTR pil_comp;
{
	pil_layout_end PTRPTR layout_end;
	layout_end = (pil_layout_end PTRPTR) pil_comp;
	return (pil_create_layout_end (layout_end));
}


PIL_ERROR pil_make_name_table_comp (pil_comp)
pil_component PTRPTR pil_comp;
{
	pil_name_table PTRPTR name_table;
	name_table = (pil_name_table PTRPTR) pil_comp;
	return (pil_create_name_table (name_table));
}


PIL_ERROR pil_make_name_table_entry (entry)
pil_name_table_entry PTRPTR entry;
{
	pil_name_table_entry 	PTRTYPE *nt_entry;
	char PTRTYPE 			*name;
	PIL_NT_TYPE				tcode;
	char PTRTYPE 			*tstring;
	PIL_NT_DMN				dcode;
	char PTRTYPE 			*dstring;
	PIL_NT_VAL				vcode;
	char PTRTYPE 			*vstring;
	PIL_ERROR				rc;
	
	
	tstring = NULL;
	dstring = NULL;
	vstring = NULL;
	
	
	tcode = (PIL_NT_TYPE) pil_parser_state.val2;
	if (tcode == PIL_CNT_OTHER) tstring = (char PTRTYPE *) 
	pil_parser_state.buf2;
	
	
	dcode = (PIL_NT_DMN) pil_parser_state.val3;
	if (dcode == PIL_DMN_OTHER) dstring = (char PTRTYPE *) 
	pil_parser_state.buf3;
	
	
	vcode = (PIL_NT_VAL) pil_parser_state.val4;
	if (vcode == PIL_VAL_OTHER) vstring = (char PTRTYPE *) 
	pil_parser_state.buf4;
	

	name = (char PTRTYPE *) pil_parser_state.buf1;
	rc = pil_create_nametbleentry (&nt_entry, name, tcode, tstring,
	dcode, dstring, vcode, vstring);
	if (rc != PIL_OK) return (rc);
	*entry = nt_entry;
	return (PIL_OK);
}


PIL_ERROR pil_make_canvas_comp (pil_comp)
pil_component PTRPTR pil_comp;
{
	pil_canvas PTRPTR canvas;
	canvas = (pil_canvas PTRPTR) pil_comp;
	return (pil_create_canvas (canvas));
}


PIL_ERROR pil_make_object_comp (pil_comp)
pil_component PTRPTR pil_comp;
{
	pil_object PTRPTR object;
	object = (pil_object PTRPTR) pil_comp;
	return (pil_create_object (object));
}


PIL_ERROR pil_make_text_flow_comp (pil_comp)
pil_component PTRPTR pil_comp;
{
	pil_text_flow PTRPTR text_flow;
	text_flow = (pil_text_flow PTRPTR) pil_comp;
	return (pil_create_text_flow (text_flow));
}


PIL_ERROR pil_make_group_comp (pil_comp)
pil_component PTRPTR pil_comp;
{
	pil_group PTRPTR group;
	group = (pil_group PTRPTR) pil_comp;
	return (pil_create_group (group));
}


PIL_ERROR pil_make_cnt_start_comp (pil_comp)
pil_component PTRPTR pil_comp;
{
	PIL_ERROR 			rc;
	pil_content_start 	PTRPTR pcs;
	char				PTRTYPE *name_p;


	pcs = (pil_content_start PTRPTR) pil_comp;
	rc = pil_create_content_start (pcs);
	if (rc != PIL_OK) return (rc);
	name_p = (char PTRTYPE *) &pil_parser_state.buf2[0];
	rc = pil_append_string (name_p, &((*pcs)->name));
	return (rc);
}


PIL_ERROR pil_make_cnt_end_comp (pil_comp)
pil_component PTRPTR pil_comp;
{
	pil_content_end PTRPTR content_end;
	content_end = (pil_content_end PTRPTR) pil_comp;
	return (pil_create_content_end (content_end));
}


PIL_ERROR pil_make_cnt_header_comp (pil_comp)
pil_component PTRPTR pil_comp;
{
	PIL_ERROR 		rc;
	pil_content_hdr	PTRPTR content_hdr;
	
	
	content_hdr = (pil_content_hdr PTRPTR) pil_comp;
	rc = pil_create_content_hdr (content_hdr);
	if (rc != PIL_OK) return (rc);
	
	
	(**content_hdr).byteorder 	= pil_parser_state.cnt_ordering;
	(**content_hdr).encoding  	= pil_parser_state.cnt_encoding;
	(**content_hdr).escapechar	= pil_parser_state.cnt_escape;
	(**content_hdr).begblkchar	= pil_parser_state.cnt_beg;
	(**content_hdr).endblkchar	= pil_parser_state.cnt_end;
	return (PIL_OK);
}


/* ---------------------------- */
/* Utility functions begin here */
/* ---------------------------- */


PIL_ERROR pil_spinner ()
{
	PIL_UINT32 block_level;
	PIL_UINT16 type;
	block_level = 1;
	PIL_FORCE_BEG();
	
	
	while (1)
	{
		type = pil_get_token ();
		switch (type)
		{
			case PIL_TOKEN_BEG_BLOCK:
			{
				++block_level;
				continue;
			}
			case PIL_TOKEN_END_BLOCK:
			{
				--block_level;
				if (block_level == 0) return (PIL_OK);
				continue;
			}
			case PIL_TOKEN_ERROR:	return (PIL_BAD_SYNTAX);
			case PIL_TOKEN_EOF:		return (PIL_BAD_SYNTAX);
			default: 				continue;
		}
	}
}


PIL_BOOL pil_is_desired_asi ()
{
	FAST PIL_INT32 i;
	if (pil_asa_count  < 0)	return (PIL_TRUE);
	if (pil_asa_count == 0)	return (PIL_FALSE);
	
	
	for (i = 0; i < pil_asa_count; ++i)
	{
		if (pil_asa_list[i] == NULL) continue;
		if (pil_str_eq (pil_asa_list[i])) return (PIL_TRUE);
	}
	return (PIL_FALSE);
}


#ifdef PIL_ANSI_PROTOTYPES						/* If using ANSI prototypes */
PIL_BOOL pil_is_hex_digit (FAST PIL_UINT8 char_code)
#else
PIL_BOOL pil_is_hex_digit (char_code)
FAST PIL_UINT8 char_code;
#endif
{
	if ((char_code >= '0' && char_code <= '9') ||
		(char_code >= 'A' && char_code <= 'F') ||
		(char_code >= 'a' && char_code <= 'f'))		return (PIL_TRUE);
	else											return (PIL_FALSE);
}


#ifdef PIL_ANSI_PROTOTYPES						/* If using ANSI prototypes */
PIL_UINT8 pil_combine_hex_digits (FAST PIL_UINT8 hex1, FAST PIL_UINT8 hex2)
#else
PIL_UINT8 pil_combine_hex_digits (hex1, hex2)
FAST PIL_UINT8 hex1;
FAST PIL_UINT8 hex2;
#endif
{
	FAST PIL_UINT8 result;
	
	
	if (hex1 >= (PIL_UINT8)'0' && (PIL_UINT8)hex1 <= '9')	
		result = hex1 - (PIL_UINT8)'0';
	else if (hex1 >= (PIL_UINT8)'A' && hex1 <= (PIL_UINT8)'F')	
		result = (PIL_UINT8)(hex1 - (PIL_UINT8)'A' + 10);
	else
		result = (PIL_UINT8)(hex1 - (PIL_UINT8)'a' + 10);
	result <<= 4;
	
	
	if (hex2 >= (PIL_UINT8)'0' && hex2 <= (PIL_UINT8)'9')	
		result += hex2 - (PIL_UINT8)'0';
	else if (hex2 >= (PIL_UINT8)'A' && hex2 <= (PIL_UINT8)'F')	
		result += hex2 - (PIL_UINT8)'A' + 10;
	else
		result += hex2 - (PIL_UINT8)'a' + 10;
	return (result);
}


#ifdef PIL_ANSI_PROTOTYPES						/* If using ANSI prototypes */
PIL_BOOL pil_is_white_space (FAST PIL_UINT8 char_code)
#else
PIL_BOOL pil_is_white_space (char_code)
FAST PIL_UINT8 char_code;
#endif
{
	if ((char_code >= 9 && char_code <= 13) || (char_code == 32)) return (PIL_TRUE);
	else return (PIL_FALSE);
}


PIL_UINT32 pil_get_line_num ()
{
	return ((PIL_UINT32) pil_parser_state.line_number);
}


/* ------------------------------ */
/* Buffering functions begin here */
/* ------------------------------ */


PIL_VOID pil_copy_to_buf1 ()
{
	pil_strcpy ((char PTRTYPE *) pil_parser_state.buf1, 
	(char PTRTYPE *) pil_parser_state.token);
}


PIL_VOID pil_copy_to_buf2 ()
{
	pil_strcpy ((char PTRTYPE *) pil_parser_state.buf2, 
	(char PTRTYPE *) pil_parser_state.token);
}


PIL_VOID pil_copy_to_buf3 ()
{
	pil_strcpy ((char PTRTYPE *) pil_parser_state.buf3, 
	(char PTRTYPE *) pil_parser_state.token);
}


PIL_VOID pil_copy_to_buf4 ()
{
	pil_strcpy ((char PTRTYPE *) pil_parser_state.buf4, 
	(char PTRTYPE *) pil_parser_state.token);
}


/* --------------------------------------- */
/* Keyword comparison functions begin here */
/* --------------------------------------- */


#define	KEYW_TEST(A) if (pil_str_eq (A)) return (PIL_TRUE);else return (PIL_FALSE);


PIL_BOOL pil_str_eq (string)
char PTRTYPE *string;
{
	if (!pil_strcmp ((char PTRTYPE *) pil_parser_state.token, string)) return (PIL_TRUE);
	else return (PIL_FALSE);
}


PIL_BOOL pil_is_kw_pil_layout ()
{
	if (pil_str_eq (pil_kws[PIL_KW_LAYOUT_VS1]))	return (PIL_TRUE);
	if (pil_str_eq (pil_kws[PIL_KW_LAYOUT_VS2])) 	return (PIL_TRUE);
	if (pil_str_eq (pil_kws[PIL_KW_LAYOUT_VS3])) 	return (PIL_TRUE);
	if (pil_str_eq (pil_kws[PIL_KW_LAYOUT_VS4])) 	return (PIL_TRUE);
	return (PIL_FALSE);
}


PIL_BOOL pil_is_kw_pil_content ()
{
	if (pil_str_eq (pil_kws[PIL_KW_CONTENT_VS1]))	return (PIL_TRUE);
	if (pil_str_eq (pil_kws[PIL_KW_CONTENT_VS2])) 	return (PIL_TRUE);
	if (pil_str_eq (pil_kws[PIL_KW_CONTENT_VS3])) 	return (PIL_TRUE);
	if (pil_str_eq (pil_kws[PIL_KW_CONTENT_VS4])) 	return (PIL_TRUE);
	return (PIL_FALSE);
}


PIL_BOOL pil_is_kw_object	()				{ KEYW_TEST (pil_kws[PIL_KW_OBJECT]) 			}
PIL_BOOL pil_is_kw_text_flow ()				{ KEYW_TEST (pil_kws[PIL_KW_TEXT_FLOW]) 		}
PIL_BOOL pil_is_kw_group ()					{ KEYW_TEST (pil_kws[PIL_KW_GROUP]) 			}
PIL_BOOL pil_is_kw_unknown ()				{ KEYW_TEST (pil_kws[PIL_KW_UNKNOWN]) 			}
PIL_BOOL pil_is_kw_ascii_text ()			{ KEYW_TEST (pil_kws[PIL_KW_ASCII_TEXT]) 		}
PIL_BOOL pil_is_kw_atex_aris ()				{ KEYW_TEST (pil_kws[PIL_KW_ATEX_ARIS]) 		}
PIL_BOOL pil_is_kw_atex_itf ()				{ KEYW_TEST (pil_kws[PIL_KW_ATEX_ITF]) 			}
PIL_BOOL pil_is_kw_eps ()					{ KEYW_TEST (pil_kws[PIL_KW_EPS]) 				}
PIL_BOOL pil_is_kw_icl ()					{ KEYW_TEST (pil_kws[PIL_KW_ICL]) 				}
PIL_BOOL pil_is_kw_it8_1 ()					{ KEYW_TEST (pil_kws[PIL_KW_IT8_1]) 			}
PIL_BOOL pil_is_kw_it8_2 ()					{ KEYW_TEST (pil_kws[PIL_KW_IT8_2]) 			}
PIL_BOOL pil_is_kw_ms_rtf ()				{ KEYW_TEST (pil_kws[PIL_KW_MS_RTF]) 			}
PIL_BOOL pil_is_kw_ms_word_3 ()				{ KEYW_TEST (pil_kws[PIL_KW_MS_WORD_3]) 		}
PIL_BOOL pil_is_kw_ms_word_4 ()				{ KEYW_TEST (pil_kws[PIL_KW_MS_WORD_4]) 		}
PIL_BOOL pil_is_kw_mif ()					{ KEYW_TEST (pil_kws[PIL_KW_MIF]) 				}
PIL_BOOL pil_is_kw_pict ()					{ KEYW_TEST (pil_kws[PIL_KW_PICT]) 				}
PIL_BOOL pil_is_kw_pil_layout_ntet ()		{ KEYW_TEST (pil_kws[PIL_KW_PIL_LAYOUT_NTET]) 	}
PIL_BOOL pil_is_kw_postscript ()			{ KEYW_TEST (pil_kws[PIL_KW_POSTSCRIPT]) 		}
PIL_BOOL pil_is_kw_quark_xpresstags ()		{ KEYW_TEST (pil_kws[PIL_KW_QUARK_XPRESSTAGS]) 	}
PIL_BOOL pil_is_kw_scitex_ct ()				{ KEYW_TEST (pil_kws[PIL_KW_SCITEX_CT]) 		}
PIL_BOOL pil_is_kw_scitex_handshake ()		{ KEYW_TEST (pil_kws[PIL_KW_SCITEX_HANDSHAKE]) 	}
PIL_BOOL pil_is_kw_scitex_lw ()				{ KEYW_TEST (pil_kws[PIL_KW_SCITEX_LW]) 		}
PIL_BOOL pil_is_kw_sgml ()					{ KEYW_TEST (pil_kws[PIL_KW_SGML]) 				}
PIL_BOOL pil_is_kw_tiff ()					{ KEYW_TEST (pil_kws[PIL_KW_TIFF]) 				}
PIL_BOOL pil_is_kw_wp_100 ()				{ KEYW_TEST (pil_kws[PIL_KW_WP_100]) 			}
PIL_BOOL pil_is_kw_wp_102 ()				{ KEYW_TEST (pil_kws[PIL_KW_WP_102]) 			}
PIL_BOOL pil_is_kw_xyquest_xywrite ()		{ KEYW_TEST (pil_kws[PIL_KW_XYQUEST_XYWRITE]) 	}
PIL_BOOL pil_is_kw_mac_filename ()			{ KEYW_TEST (pil_kws[PIL_KW_MAC_FILENAME]) 		}
PIL_BOOL pil_is_kw_unix_filename ()			{ KEYW_TEST (pil_kws[PIL_KW_UNIX_FILENAME]) 	}
PIL_BOOL pil_is_kw_msdos_filename ()		{ KEYW_TEST (pil_kws[PIL_KW_MSDOS_FILENAME]) 	}
PIL_BOOL pil_is_kw_os2_filename ()			{ KEYW_TEST (pil_kws[PIL_KW_OS2_FILENAME]) 		}
PIL_BOOL pil_is_kw_vms_filename ()			{ KEYW_TEST (pil_kws[PIL_KW_VMS_FILENAME]) 		}
PIL_BOOL pil_is_kw_vm_filename ()			{ KEYW_TEST (pil_kws[PIL_KW_VM_FILENAME]) 		}
PIL_BOOL pil_is_kw_lcd_filename ()			{ KEYW_TEST (pil_kws[PIL_KW_LCD_FILENAME]) 		}
PIL_BOOL pil_is_kw_inline ()				{ KEYW_TEST (pil_kws[PIL_KW_INLINE]) 			}
PIL_BOOL pil_is_kw_name_table ()			{ KEYW_TEST (pil_kws[PIL_KW_NAME_TABLE]) 		}
PIL_BOOL pil_is_kw_canvas ()				{ KEYW_TEST (pil_kws[PIL_KW_CANVAS]) 			}
PIL_BOOL pil_is_kw_dimensions ()			{ KEYW_TEST (pil_kws[PIL_KW_DIMENSIONS]) 		}	
PIL_BOOL pil_is_kw_units ()					{ KEYW_TEST (pil_kws[PIL_KW_UNITS]) 			}
PIL_BOOL pil_is_kw_user_name ()				{ KEYW_TEST (pil_kws[PIL_KW_USER_NAME]) 		}
PIL_BOOL pil_is_kw_type ()					{ KEYW_TEST (pil_kws[PIL_KW_TYPE]) 				}
PIL_BOOL pil_is_kw_clipper ()				{ KEYW_TEST (pil_kws[PIL_KW_CLIPPER]) 			}
PIL_BOOL pil_is_kw_application ()			{ KEYW_TEST (pil_kws[PIL_KW_APPLICATION]) 		}
PIL_BOOL pil_is_kw_app_name ()				{ KEYW_TEST (pil_kws[PIL_KW_APP_NAME]) 			}
PIL_BOOL pil_is_kw_app_int8 ()				{ KEYW_TEST (pil_kws[PIL_KW_APP_INT8]) 			}		
PIL_BOOL pil_is_kw_app_int16 ()				{ KEYW_TEST (pil_kws[PIL_KW_APP_INT16]) 		}
PIL_BOOL pil_is_kw_app_int32 ()				{ KEYW_TEST (pil_kws[PIL_KW_APP_INT32]) 		}
PIL_BOOL pil_is_kw_app_uint8 ()				{ KEYW_TEST (pil_kws[PIL_KW_APP_UINT8]) 		}
PIL_BOOL pil_is_kw_app_uint16 ()			{ KEYW_TEST (pil_kws[PIL_KW_APP_UINT16]) 		}
PIL_BOOL pil_is_kw_app_uint32 ()			{ KEYW_TEST (pil_kws[PIL_KW_APP_UINT32]) 		}
PIL_BOOL pil_is_kw_app_double ()			{ KEYW_TEST (pil_kws[PIL_KW_APP_DOUBLE]) 		}
PIL_BOOL pil_is_kw_app_string ()			{ KEYW_TEST (pil_kws[PIL_KW_APP_STRING]) 		}
PIL_BOOL pil_is_kw_fill_rule ()				{ KEYW_TEST (pil_kws[PIL_KW_FILL_RULE]) 		}
PIL_BOOL pil_is_kw_rectangle ()				{ KEYW_TEST (pil_kws[PIL_KW_RECTANGLE]) 		}
PIL_BOOL pil_is_kw_circle ()				{ KEYW_TEST (pil_kws[PIL_KW_CIRCLE]) 			}
PIL_BOOL pil_is_kw_ellipse ()				{ KEYW_TEST (pil_kws[PIL_KW_ELLIPSE]) 			}
PIL_BOOL pil_is_kw_polygon ()				{ KEYW_TEST (pil_kws[PIL_KW_POLYGON]) 			}
PIL_BOOL pil_is_kw_closed_path ()			{ KEYW_TEST (pil_kws[PIL_KW_CLOSED_PATH]) 		}
PIL_BOOL pil_is_kw_even_odd ()				{ KEYW_TEST (pil_kws[PIL_KW_EVEN_ODD]) 			}
PIL_BOOL pil_is_kw_nz_winding ()			{ KEYW_TEST (pil_kws[PIL_KW_NZ_WINDING]) 		}
PIL_BOOL pil_is_kw_move_to ()				{ KEYW_TEST (pil_kws[PIL_KW_MOVE_TO]) 			}
PIL_BOOL pil_is_kw_line_to ()				{ KEYW_TEST (pil_kws[PIL_KW_LINE_TO]) 			}
PIL_BOOL pil_is_kw_curve_to ()				{ KEYW_TEST (pil_kws[PIL_KW_CURVE_TO]) 			}
PIL_BOOL pil_is_kw_arc_to ()				{ KEYW_TEST (pil_kws[PIL_KW_ARC_TO]) 			}
PIL_BOOL pil_is_kw_flow_label ()			{ KEYW_TEST (pil_kws[PIL_KW_FLOW_LABEL]) 		}
PIL_BOOL pil_is_kw_flow_objects ()			{ KEYW_TEST (pil_kws[PIL_KW_FLOW_OBJECTS]) 		}
PIL_BOOL pil_is_kw_flow_from ()				{ KEYW_TEST (pil_kws[PIL_KW_FLOW_FROM]) 		}
PIL_BOOL pil_is_kw_flow_to ()				{ KEYW_TEST (pil_kws[PIL_KW_FLOW_TO]) 			}
PIL_BOOL pil_is_kw_flow_content ()			{ KEYW_TEST (pil_kws[PIL_KW_FLOW_CONTENT]) 		}
PIL_BOOL pil_is_kw_id ()					{ KEYW_TEST (pil_kws[PIL_KW_ID]) 				}
PIL_BOOL pil_is_kw_group_objects ()			{ KEYW_TEST (pil_kws[PIL_KW_GROUP_OBJECTS]) 	}
PIL_BOOL pil_is_kw_origin ()				{ KEYW_TEST (pil_kws[PIL_KW_ORIGIN]) 			}
PIL_BOOL pil_is_kw_rot_angle ()				{ KEYW_TEST (pil_kws[PIL_KW_ROT_ANGLE]) 		}
PIL_BOOL pil_is_kw_rot_point ()				{ KEYW_TEST (pil_kws[PIL_KW_ROT_POINT]) 		}
PIL_BOOL pil_is_kw_container ()				{ KEYW_TEST (pil_kws[PIL_KW_CONTAINER]) 		}
PIL_BOOL pil_is_kw_graphic ()				{ KEYW_TEST (pil_kws[PIL_KW_GRAPHIC]) 			}
PIL_BOOL pil_is_kw_rc_type ()				{ KEYW_TEST (pil_kws[PIL_KW_RC_TYPE]) 			}
PIL_BOOL pil_is_kw_rc_name ()				{ KEYW_TEST (pil_kws[PIL_KW_RC_NAME]) 			}
PIL_BOOL pil_is_kw_bbox ()					{ KEYW_TEST (pil_kws[PIL_KW_BBOX]) 				}
PIL_BOOL pil_is_kw_rc_type_text ()			{ KEYW_TEST (pil_kws[PIL_KW_RC_TYPE_TEXT]) 		}	
PIL_BOOL pil_is_kw_rc_type_geometry ()		{ KEYW_TEST (pil_kws[PIL_KW_RC_TYPE_GEOMETRY])	}
PIL_BOOL pil_is_kw_rc_type_lineart ()		{ KEYW_TEST (pil_kws[PIL_KW_RC_TYPE_LINEART]) 	}
PIL_BOOL pil_is_kw_rc_type_image ()			{ KEYW_TEST (pil_kws[PIL_KW_RC_TYPE_IMAGE]) 	}
PIL_BOOL pil_is_kw_color_model ()			{ KEYW_TEST (pil_kws[PIL_KW_COLOR_MODEL]) 		}
PIL_BOOL pil_is_kw_dev_gray ()				{ KEYW_TEST (pil_kws[PIL_KW_DEV_GRAY]) 			}
PIL_BOOL pil_is_kw_cal_gray ()				{ KEYW_TEST (pil_kws[PIL_KW_CAL_GRAY]) 			}
PIL_BOOL pil_is_kw_dev_rgb ()				{ KEYW_TEST (pil_kws[PIL_KW_DEV_RGB]) 			}
PIL_BOOL pil_is_kw_cal_rgb ()				{ KEYW_TEST (pil_kws[PIL_KW_CAL_RGB]) 			}
PIL_BOOL pil_is_kw_dev_cmyk ()				{ KEYW_TEST (pil_kws[PIL_KW_DEV_CMYK]) 			}
PIL_BOOL pil_is_kw_spot ()					{ KEYW_TEST (pil_kws[PIL_KW_SPOT]) 				}
PIL_BOOL pil_is_kw_pantone ()				{ KEYW_TEST (pil_kws[PIL_KW_PANTONE]) 			}
PIL_BOOL pil_is_kw_render_op ()				{ KEYW_TEST (pil_kws[PIL_KW_RENDER_OP]) 		}
PIL_BOOL pil_is_kw_stroke ()				{ KEYW_TEST (pil_kws[PIL_KW_STROKE]) 			}
PIL_BOOL pil_is_kw_fill ()					{ KEYW_TEST (pil_kws[PIL_KW_FILL]) 				}
PIL_BOOL pil_is_kw_fill_stroke ()			{ KEYW_TEST (pil_kws[PIL_KW_FILL_STROKE]) 		}
PIL_BOOL pil_is_kw_render_attr ()			{ KEYW_TEST (pil_kws[PIL_KW_RENDER_ATTR]) 		}
PIL_BOOL pil_is_kw_stroke_color ()			{ KEYW_TEST (pil_kws[PIL_KW_STROKE_COLOR]) 		}
PIL_BOOL pil_is_kw_position ()				{ KEYW_TEST (pil_kws[PIL_KW_POSITION]) 			}
PIL_BOOL pil_is_kw_width ()					{ KEYW_TEST (pil_kws[PIL_KW_WIDTH]) 			}
PIL_BOOL pil_is_kw_cap ()					{ KEYW_TEST (pil_kws[PIL_KW_CAP]) 				}
PIL_BOOL pil_is_kw_join ()					{ KEYW_TEST (pil_kws[PIL_KW_JOIN]) 				}
PIL_BOOL pil_is_kw_miter_limit ()			{ KEYW_TEST (pil_kws[PIL_KW_MITER_LIMIT]) 		}
PIL_BOOL pil_is_kw_fill_color ()			{ KEYW_TEST (pil_kws[PIL_KW_FILL_COLOR]) 		}
PIL_BOOL pil_is_kw_inside ()				{ KEYW_TEST (pil_kws[PIL_KW_INSIDE]) 			}
PIL_BOOL pil_is_kw_outside ()				{ KEYW_TEST (pil_kws[PIL_KW_OUTSIDE]) 			}
PIL_BOOL pil_is_kw_centered ()				{ KEYW_TEST (pil_kws[PIL_KW_CENTERED]) 			}
PIL_BOOL pil_is_kw_butt_cap ()				{ KEYW_TEST (pil_kws[PIL_KW_BUTT_CAP]) 			}
PIL_BOOL pil_is_kw_round_cap ()				{ KEYW_TEST (pil_kws[PIL_KW_ROUND_CAP]) 		}
PIL_BOOL pil_is_kw_projecting_cap ()		{ KEYW_TEST (pil_kws[PIL_KW_PROJECTING_CAP]) 	}
PIL_BOOL pil_is_kw_miter_join ()			{ KEYW_TEST (pil_kws[PIL_KW_MITER_JOIN]) 		}
PIL_BOOL pil_is_kw_bevel_join ()			{ KEYW_TEST (pil_kws[PIL_KW_BEVEL_JOIN]) 		}
PIL_BOOL pil_is_kw_round_join ()			{ KEYW_TEST (pil_kws[PIL_KW_ROUND_JOIN]) 		}
PIL_BOOL pil_is_kw_line ()					{ KEYW_TEST (pil_kws[PIL_KW_LINE]) 				}
PIL_BOOL pil_is_kw_arc ()					{ KEYW_TEST (pil_kws[PIL_KW_ARC]) 				}
PIL_BOOL pil_is_kw_bezier ()				{ KEYW_TEST (pil_kws[PIL_KW_BEZIER]) 			}
PIL_BOOL pil_is_kw_poly_line ()				{ KEYW_TEST (pil_kws[PIL_KW_POLY_LINE]) 		}
PIL_BOOL pil_is_kw_open_path ()				{ KEYW_TEST (pil_kws[PIL_KW_OPEN_PATH]) 		}
PIL_BOOL pil_is_kw_data ()					{ KEYW_TEST (pil_kws[PIL_KW_DATA]) 				}
PIL_BOOL pil_is_kw_header ()				{ KEYW_TEST (pil_kws[PIL_KW_HEADER]) 			}
PIL_BOOL pil_is_kw_little_endian ()			{ KEYW_TEST (pil_kws[PIL_KW_LITTLE_ENDIAN]) 	}
PIL_BOOL pil_is_kw_big_endian ()			{ KEYW_TEST (pil_kws[PIL_KW_BIG_ENDIAN]) 		}
PIL_BOOL pil_is_kw_ascii ()					{ KEYW_TEST (pil_kws[PIL_KW_ASCII]) 			}
PIL_BOOL pil_is_kw_binary ()				{ KEYW_TEST (pil_kws[PIL_KW_BINARY]) 			}
PIL_BOOL pil_is_kw_text ()					{ KEYW_TEST (pil_kws[PIL_KW_OBJ_TEXT]) 			}
PIL_BOOL pil_is_kw_picture ()				{ KEYW_TEST (pil_kws[PIL_KW_OBJ_PICTURE]) 		}
PIL_BOOL pil_is_kw_primitive ()				{ KEYW_TEST (pil_kws[PIL_KW_OBJ_PRIMITIVE]) 	}
PIL_BOOL pil_is_kw_clipper_shape ()			{ KEYW_TEST (pil_kws[PIL_KW_CLIPPER_SHAPE]) 	}
PIL_BOOL pil_is_kw_container_shape ()		{ KEYW_TEST (pil_kws[PIL_KW_CONTAINER_SHAPE]) 	}


PIL_BOOL pil_is_kw_nt_type ()
{
	PIL_BOOL rc;
	PIL_PTR_PARSER_STATE pps;
	pps = &pil_parser_state;
	rc = PIL_TRUE;
	
	
	if (pil_is_kw_unknown())			{ pps->val2 = PIL_CNT_UNKNOWN; 			return (rc); }	
	if (pil_is_kw_ascii_text())			{ pps->val2 = PIL_CNT_ASCII_TEXT; 		return (rc); }
	if (pil_is_kw_atex_aris())			{ pps->val2 = PIL_CNT_ATEX_ARIS; 		return (rc); }
	if (pil_is_kw_atex_itf())			{ pps->val2 = PIL_CNT_ATEX_ITF; 		return (rc); }
	if (pil_is_kw_eps())				{ pps->val2 = PIL_CNT_EPS; 				return (rc); }
	if (pil_is_kw_icl())				{ pps->val2 = PIL_CNT_ICL; 				return (rc); }
	if (pil_is_kw_it8_1())				{ pps->val2 = PIL_CNT_IT8_1; 			return (rc); }
	if (pil_is_kw_it8_2())				{ pps->val2 = PIL_CNT_IT8_2; 			return (rc); }
	if (pil_is_kw_ms_rtf())				{ pps->val2 = PIL_CNT_MS_RTF; 			return (rc); }
	if (pil_is_kw_ms_word_3())			{ pps->val2 = PIL_CNT_MS_WORD_3_0; 		return (rc); }
	if (pil_is_kw_ms_word_4())			{ pps->val2 = PIL_CNT_MS_WORD_4_0; 		return (rc); }
	if (pil_is_kw_mif())				{ pps->val2 = PIL_CNT_MIF; 				return (rc); }
	if (pil_is_kw_pict())				{ pps->val2 = PIL_CNT_PICT; 			return (rc); }
	if (pil_is_kw_pil_layout_ntet())	{ pps->val2 = PIL_CNT_PIL_LAYOUT; 		return (rc); }
	if (pil_is_kw_postscript())			{ pps->val2 = PIL_CNT_POSTSCRIPT; 		return (rc); }
	if (pil_is_kw_quark_xpresstags())	{ pps->val2 = PIL_CNT_QUARK_XP_TAGS; 	return (rc); }
	if (pil_is_kw_scitex_ct())			{ pps->val2 = PIL_CNT_SCITEX_CT; 		return (rc); }
	if (pil_is_kw_scitex_handshake())	{ pps->val2 = PIL_CNT_SCITEX_HS; 		return (rc); }
	if (pil_is_kw_scitex_lw())			{ pps->val2 = PIL_CNT_SCITEX_LW; 		return (rc); }
	if (pil_is_kw_sgml())				{ pps->val2 = PIL_CNT_SGML; 			return (rc); }
	if (pil_is_kw_tiff())				{ pps->val2 = PIL_CNT_TIFF; 			return (rc); }
	if (pil_is_kw_wp_100())				{ pps->val2 = PIL_CNT_WP_1_00; 			return (rc); }
	if (pil_is_kw_wp_102())				{ pps->val2 = PIL_CNT_WP_1_02; 			return (rc); }
	if (pil_is_kw_xyquest_xywrite())	{ pps->val2 = PIL_CNT_XYQUEST_XYWRITE;	return (rc); }
	return (PIL_FALSE);
}


PIL_BOOL pil_is_kw_nt_domain ()
{
	PIL_BOOL rc;
	PIL_PTR_PARSER_STATE pps;
	pps = &pil_parser_state;
	rc = PIL_TRUE;
	
	
	if (pil_is_kw_mac_filename())	{ pps->val3 = PIL_DMN_MAC_FILE; 	return (rc); }	
	if (pil_is_kw_unix_filename())	{ pps->val3 = PIL_DMN_UNIX_FILE; 	return (rc); }
	if (pil_is_kw_msdos_filename())	{ pps->val3 = PIL_DMN_MSDOS_FILE; 	return (rc); }
	if (pil_is_kw_os2_filename())	{ pps->val3 = PIL_DMN_OS2_FILE; 	return (rc); }
	if (pil_is_kw_vms_filename())	{ pps->val3 = PIL_DMN_VMS_FILE; 	return (rc); }
	if (pil_is_kw_vm_filename())	{ pps->val3 = PIL_DMN_VM_FILE; 		return (rc); }
	if (pil_is_kw_lcd_filename())	{ pps->val3 = PIL_DMN_LCD_FILE; 	return (rc); }
	if (pil_is_kw_inline())			{ pps->val3 = PIL_DMN_INLINE; 		return (rc); }
	if (pil_is_kw_unknown())		{ pps->val3 = PIL_DMN_UNKNOWN; 		return (rc); }
	return (PIL_FALSE);
}


PIL_BOOL pil_is_kw_nt_value ()
{
	PIL_BOOL rc;
	PIL_PTR_PARSER_STATE pps;
	pps = &pil_parser_state;
	rc = PIL_TRUE;
	
	
	if (pil_is_kw_inline())			{ pps->val4 = PIL_VAL_INLINE; 		return (rc); }
	if (pil_is_kw_unknown())		{ pps->val4 = PIL_VAL_UNKNOWN; 		return (rc); }
	return (PIL_FALSE);
}


PIL_BOOL pil_is_kw_app_type ()
{
	PIL_BOOL rc;
	PIL_PTR_PARSER_STATE pps;
	pps = &pil_parser_state;
	rc = PIL_TRUE;
	
	
	if (pil_is_kw_app_int8())		{ pps->val1 = PIL_INT8_CODE; 		return (rc); }	
	if (pil_is_kw_app_int16())		{ pps->val1 = PIL_INT16_CODE; 		return (rc); }
	if (pil_is_kw_app_int32())		{ pps->val1 = PIL_INT32_CODE; 		return (rc); }
	if (pil_is_kw_app_uint8())		{ pps->val1 = PIL_UINT8_CODE; 		return (rc); }
	if (pil_is_kw_app_uint16())		{ pps->val1 = PIL_UINT16_CODE; 		return (rc); }
	if (pil_is_kw_app_uint32())		{ pps->val1 = PIL_UINT32_CODE; 		return (rc); }
	if (pil_is_kw_app_double())		{ pps->val1 = PIL_DOUBLE_CODE; 		return (rc); }
	if (pil_is_kw_app_string())		{ pps->val1 = PIL_STRING_CODE; 		return (rc); }
	return (PIL_FALSE);
}


PIL_BOOL pil_is_kw_cm_value ()
{
	PIL_BOOL rc;
	PIL_PTR_PARSER_STATE pps;
	pps = &pil_parser_state;
	rc = PIL_TRUE;
	
	
	if (pil_is_kw_dev_gray())		{ pps->val1 = PIL_DEV_GRAY_CM; 	return (rc); }	
	if (pil_is_kw_cal_gray())		{ pps->val1 = PIL_CAL_GRAY_CM; 	return (rc); }
	if (pil_is_kw_dev_rgb())		{ pps->val1 = PIL_DEV_RGB_CM; 	return (rc); }
	if (pil_is_kw_cal_rgb())		{ pps->val1 = PIL_CAL_RGB_CM; 	return (rc); }
	if (pil_is_kw_dev_cmyk())		{ pps->val1 = PIL_DEV_CMYK_CM; 	return (rc); }
	if (pil_is_kw_pantone())		{ pps->val1 = PIL_PANTONE_CM; 	return (rc); }
	if (pil_is_kw_spot())			{ pps->val1 = PIL_SPOT_CM; 		return (rc); }
	return (PIL_FALSE);
}


PIL_BOOL pil_is_kw_rop_value ()
{
	PIL_BOOL rc;
	PIL_PTR_PARSER_STATE pps;
	pps = &pil_parser_state;
	rc = PIL_TRUE;
	
	
	if (pil_is_kw_stroke())			{ pps->val1 = PIL_STROKE_RENDER_OP; 		return (rc); }	
	if (pil_is_kw_fill())			{ pps->val1 = PIL_FILL_RENDER_OP; 			return (rc); }
	if (pil_is_kw_fill_stroke())	{ pps->val1 = PIL_FILL_STROKE_RENDER_OP; 	return (rc); }
	return (PIL_FALSE);
}


PIL_BOOL pil_is_graphic_prim ()
{
	PIL_BOOL rc;
	PIL_PTR_PARSER_STATE pps;
	pps = &pil_parser_state;
	rc = PIL_TRUE;
	
	
	if (pil_is_kw_line())			{ pps->val1 = PIL_LINE_SHAPE; 			return (rc); }	
	if (pil_is_kw_arc())			{ pps->val1 = PIL_ARC_SHAPE; 			return (rc); }
	if (pil_is_kw_bezier())			{ pps->val1 = PIL_BEZIER_SHAPE; 		return (rc); }
	if (pil_is_kw_poly_line())		{ pps->val1 = PIL_POLYLINE_SHAPE; 		return (rc); }
	if (pil_is_kw_open_path())		{ pps->val1 = PIL_OPENPATH_SHAPE; 		return (rc); }
	if (pil_is_kw_rectangle())		{ pps->val1 = PIL_RECTANGLE_SHAPE; 		return (rc); }
	if (pil_is_kw_circle())			{ pps->val1 = PIL_CIRCLE_SHAPE; 		return (rc); }
	if (pil_is_kw_ellipse())		{ pps->val1 = PIL_ELLIPSE_SHAPE; 		return (rc); }
	if (pil_is_kw_polygon())		{ pps->val1 = PIL_POLYGON_SHAPE; 		return (rc); }
	if (pil_is_kw_closed_path())	{ pps->val1 = PIL_CLOSEDPATH_SHAPE; 	return (rc); }
	if (pil_is_kw_clipper_shape())	{ pps->val1 = PIL_CLIPPER_SHAPE; 		return (rc); }
	if (pil_is_kw_container_shape()){ pps->val1 = PIL_CONTAINER_SHAPE; 		return (rc); }
	return (PIL_FALSE);
}


/* ------------------------------------------------------------------------	*/
/*		P I L  T O K E N I Z E R  M O D U L E								*/
/* ------------------------------------------------------------------------	*/


/* ------------------------------ */
/* Macros used by pil_get_token() */
/* ------------------------------ */


/* Stuff end of string character into current token and check for overflow */
/* ----------------------------------------------------------------------- */
#define		PIL_END_TOKEN(A)	if (token_offset < PIL_MAX_TOKEN_SIZE) 				\
								{ 													\
									pps->initial_fetch = A; 						\
									pps->token [token_offset] = PIL_END_OF_STRING;	\
								}													\
								else 												\
								{ 													\
									pps->initial_fetch = A; 						\
									pps->token_error = PIL_TOKEN_ERR_TOKEN_SIZE; 	\
									return (pps->token_type = PIL_TOKEN_ERROR); 	\
								}
								

/* Stuff last character into current token and check for overflow */
/* -------------------------------------------------------------- */
#define		PIL_STUFF_TOKEN()	if (token_offset < PIL_MAX_TOKEN_SIZE) 				\
								{ 													\
									pps->token [token_offset++] 					\
									= (PIL_UINT8)pps->char_code;					\
								}													\
								else 												\
								{ 													\
									pps->initial_fetch = PIL_FALSE; 				\
									pps->token_error = PIL_TOKEN_ERR_TOKEN_SIZE; 	\
									return (pps->token_type = PIL_TOKEN_ERROR); 	\
								}
			

PIL_UINT16 pil_get_token ()
{
	FAST	PIL_UINT16 				state;
	FAST	PIL_UINT16 				char_class;
	FAST	PIL_UINT16 				fstate;
	FAST	PIL_UINT16				token_offset;
	FAST	PIL_PTR_PARSER_STATE 	pps;
	
	
	pps = &pil_parser_state;
	state = PIL_S_START;
	token_offset = 0;
	
	
	while (1)
	{
		switch (state)
		{
			case PIL_S_START:
			{
				if (pps->initial_fetch) pps->char_code = (*pil_getchar)();
				pps->initial_fetch = PIL_TRUE;

				if (pps->char_code == PIL_REOF)
				{
					pps->initial_fetch = PIL_FALSE;
					return (pps->token_type = PIL_TOKEN_EOF);
				}
								
				char_class = pil_char_class_table [pps->char_code];
				switch (char_class)
				{
					case PIL_C_LETTER: 		state = PIL_S_SYMBOL; 		continue;
					case PIL_C_DIGIT: 		state = PIL_S_CONSTANT; 	continue;
					case PIL_C_DOT: 		state = PIL_S_CONSTANT; 	continue;
					case PIL_C_COMMENT: 	state = PIL_S_COMMENT; 		continue;
					case PIL_C_STRING: 		state = PIL_S_STRING; 		continue;					
					case PIL_C_WHITESPACE:	state = PIL_S_WHITESPACE;	continue;						
					case PIL_C_BEG_BLOCK: 	state = PIL_S_BEGBLOCK;		continue;
					case PIL_C_END_BLOCK: 	state = PIL_S_ENDBLOCK;		continue;
					case PIL_C_PLUS:		state = PIL_S_POSSCONST;	continue;
					case PIL_C_MINUS:		state = PIL_S_POSSCONST;	continue;
					default:				state = PIL_S_ERROR;		continue;
				}
				break;
			}

				
			case PIL_S_WHITESPACE:
			{
				if (pps->char_code == PIL_END_OF_LINE) ++pps->line_number;
				state = PIL_S_START;
				continue;
			}

						
			case PIL_S_BEGBLOCK:	return (pps->token_type = PIL_TOKEN_BEG_BLOCK);
			case PIL_S_ENDBLOCK: 	return (pps->token_type = PIL_TOKEN_END_BLOCK);
				

			case PIL_S_POSSCONST:
			{
				PIL_STUFF_TOKEN ();
				pps->char_code = (*pil_getchar)();
				if (pps->char_code == PIL_REOF)
				{
					pps->initial_fetch = PIL_FALSE;
					pps->token_error = PIL_TOKEN_ERR_BAD_NUM;
					return (pps->token_type = PIL_TOKEN_ERROR);
				}
				char_class = pil_char_class_table [pps->char_code];
				switch (char_class)
				{
					case PIL_C_DOT:
					case PIL_C_DIGIT:
						state = PIL_S_CONSTANT;
						continue;
						
					default:
						pps->initial_fetch = PIL_FALSE;
						pps->token_error = PIL_TOKEN_ERR_BAD_NUM;
						return (pps->token_type = PIL_TOKEN_ERROR);
				}
				break;
			}

				
			case PIL_S_ERROR:
			{
				pps->token_error = PIL_TOKEN_ERR_SYNTAX;
				return (pps->token_type = PIL_TOKEN_ERROR);
			}

											
			case PIL_S_SYMBOL:
			{
				PIL_STUFF_TOKEN ();
				while (1)
				{
					pps->char_code = (*pil_getchar)();
					if (pps->char_code == PIL_REOF)
					{
						PIL_END_TOKEN (PIL_FALSE);
						return (pps->token_type = PIL_TOKEN_SYMBOL);
					}
					
					
					char_class = pil_char_class_table [pps->char_code];
					if (char_class == PIL_C_LETTER	||
					    char_class == PIL_C_DIGIT	||
					    char_class == PIL_C_MINUS	||
					    char_class == PIL_C_DOT		||
					    char_class == PIL_C_UNDERSCORE)
					{
						PIL_STUFF_TOKEN ();
						continue;
					}
					else
					{
						PIL_END_TOKEN (PIL_FALSE);
						return (pps->token_type = PIL_TOKEN_SYMBOL);
					}
				}
				break;
			}

	
			case PIL_S_COMMENT:
			{
				pps->char_code = (*pil_getchar)();
				if (pps->char_code == PIL_REOF)
				{
					pps->initial_fetch = PIL_FALSE;
					pps->token_error = PIL_TOKEN_ERR_BAD_COMMENT;
					return (pps->token_type = PIL_TOKEN_ERROR);
				}
				char_class = pil_char_class_table [pps->char_code];
				switch (char_class)
				{
					case PIL_C_COMMENT:
						while (1)
						{
							pps->char_code = (*pil_getchar)();
							if (pps->char_code == PIL_REOF) 
							{
								pps->initial_fetch = PIL_FALSE;
								return (pps->token_type = PIL_TOKEN_EOF);
							}
							if (pps->char_code == PIL_END_OF_LINE)
							{
								++pps->line_number;
								state = PIL_S_START;
								break;
							}
						}
						break;
					
					case PIL_C_STAR:
						while (1)
						{
							pps->char_code = (*pil_getchar)();
							if (pps->char_code == PIL_REOF) 
							{
								pps->initial_fetch = PIL_FALSE;
								pps->token_error = PIL_TOKEN_ERR_BAD_COMMENT;
								return (pps->token_type = PIL_TOKEN_ERROR);
							}
							char_class = pil_char_class_table [pps->char_code];
							if (char_class == PIL_C_STAR)
							{
								pps->char_code = (*pil_getchar)();
								if (pps->char_code == PIL_REOF) 
								{
									pps->initial_fetch = PIL_FALSE;
									pps->token_error = PIL_TOKEN_ERR_BAD_COMMENT;
									return (pps->token_type = PIL_TOKEN_ERROR);
								}
								char_class = pil_char_class_table [pps->char_code];
								if (char_class == PIL_C_COMMENT)
								{
									state = PIL_S_START;
									break;
								}
							}
							if (pps->char_code == PIL_END_OF_LINE) ++pps->line_number;
							continue;
						}
						break;

					default:
						pps->initial_fetch = PIL_FALSE;
						pps->token_error = PIL_TOKEN_ERR_BAD_COMMENT;
						return (pps->token_type = PIL_TOKEN_ERROR);
				}
				break;
			}
				

			case PIL_S_CONSTANT:
			{
				fstate = 0;
				while (1)
				{
					switch (fstate)
					{
						case 0:
						
							PIL_STUFF_TOKEN ();
							pps->char_code = (*pil_getchar)();
							if (char_class == PIL_C_DIGIT) fstate = 1;
							else fstate = 3;
							continue;
							
						case 1:
						
							if (pps->char_code == PIL_REOF)
							{
								PIL_END_TOKEN (PIL_FALSE);
								pps->token_constant = pil_atof ((char PTRTYPE *) pps->token);
								return (pps->token_type = PIL_TOKEN_ICONSTANT);
							}
							char_class = pil_char_class_table [pps->char_code];
							if (char_class == PIL_C_DIGIT)
							{
								PIL_STUFF_TOKEN ();
								pps->char_code = (*pil_getchar)();
								continue;
							}
							if (char_class == PIL_C_DOT)
							{
								PIL_STUFF_TOKEN ();
								pps->char_code = (*pil_getchar)();
								fstate = 2;
								continue;
							}
							if (pps->char_code == 'e' || pps->char_code == 'E')
							{
								PIL_STUFF_TOKEN ();
								pps->char_code = (*pil_getchar)();
								fstate = 5;
								continue;
							}
							PIL_END_TOKEN (PIL_FALSE);
							pps->token_constant = pil_atof ((char PTRTYPE *) pps->token);
							return (pps->token_type = PIL_TOKEN_ICONSTANT);
							
						case 2:
						
							if (pps->char_code == PIL_REOF)
							{
								PIL_END_TOKEN (PIL_FALSE);
								pps->token_constant = pil_atof ((char PTRTYPE *) pps->token);
								return (pps->token_type = PIL_TOKEN_FCONSTANT);
							}
							char_class = pil_char_class_table [pps->char_code];
							if (char_class == PIL_C_DIGIT)
							{
								PIL_STUFF_TOKEN ();
								pps->char_code = (*pil_getchar)();
								fstate = 4;
								continue;
							}
							PIL_END_TOKEN (PIL_FALSE);
							pps->token_constant = pil_atof ((char PTRTYPE *) pps->token);
							return (pps->token_type = PIL_TOKEN_FCONSTANT);
							
						case 3:
						
							if (pps->char_code == PIL_REOF)
							{
								pps->initial_fetch = PIL_FALSE;
								pps->token_error = PIL_TOKEN_ERR_BAD_NUM;
								return (pps->token_type = PIL_TOKEN_ERROR);
							}
							char_class = pil_char_class_table [pps->char_code];
							if (char_class == PIL_C_DIGIT)
							{
								PIL_STUFF_TOKEN ();
								pps->char_code = (*pil_getchar)();
								fstate = 4;
								continue;
							}
							pps->initial_fetch = PIL_FALSE;
							pps->token_error = PIL_TOKEN_ERR_BAD_NUM;
							return (pps->token_type = PIL_TOKEN_ERROR);
							
						case 4:
						
							if (pps->char_code == PIL_REOF)
							{
								PIL_END_TOKEN (PIL_FALSE);
								pps->token_constant = pil_atof ((char PTRTYPE *) pps->token);
								return (pps->token_type = PIL_TOKEN_FCONSTANT);
							}
							char_class = pil_char_class_table [pps->char_code];
							if (char_class == PIL_C_DIGIT)
							{
								PIL_STUFF_TOKEN ();
								pps->char_code = (*pil_getchar)();
								continue;
							}
							if (pps->char_code == 'e' || pps->char_code == 'E')
							{
								PIL_STUFF_TOKEN ();
								pps->char_code = (*pil_getchar)();
								fstate = 5;
								continue;
							}
							PIL_END_TOKEN (PIL_FALSE);
							pps->token_constant = pil_atof ((char PTRTYPE *) pps->token);
							return (pps->token_type = PIL_TOKEN_FCONSTANT);
							
						case 5:
						
							if (pps->char_code == PIL_REOF)
							{
								pps->initial_fetch = PIL_FALSE;
								pps->token_error = PIL_TOKEN_ERR_BAD_NUM;
								return (pps->token_type = PIL_TOKEN_ERROR);
							}
							char_class = pil_char_class_table [pps->char_code];
							if (char_class == PIL_C_PLUS || char_class == PIL_C_MINUS)
							{
								PIL_STUFF_TOKEN ();
								pps->char_code = (*pil_getchar)();
								fstate = 6;
								continue;
							}
							if (char_class == PIL_C_DIGIT)
							{
								PIL_STUFF_TOKEN ();
								pps->char_code = (*pil_getchar)();
								fstate = 7;
								continue;
							}
							pps->initial_fetch = PIL_FALSE;
							pps->token_error = PIL_TOKEN_ERR_BAD_NUM;
							return (pps->token_type = PIL_TOKEN_ERROR);
							
						case 6:
						
							if (pps->char_code == PIL_REOF)
							{
								pps->initial_fetch = PIL_FALSE;
								pps->token_error = PIL_TOKEN_ERR_BAD_NUM;
								return (pps->token_type = PIL_TOKEN_ERROR);
							}
							char_class = pil_char_class_table [pps->char_code];
							if (char_class == PIL_C_DIGIT)
							{
								PIL_STUFF_TOKEN ();
								pps->char_code = (*pil_getchar)();
								fstate = 7;
								continue;
							}
							pps->initial_fetch = PIL_FALSE;
							pps->token_error = PIL_TOKEN_ERR_BAD_NUM;
							return (pps->token_type = PIL_TOKEN_ERROR);
							
						case 7:
						
							if (pps->char_code == PIL_REOF)
							{
								PIL_END_TOKEN (PIL_FALSE);
								pps->token_constant = pil_atof ((char PTRTYPE *) pps->token);
								return (pps->token_type = PIL_TOKEN_FCONSTANT);
							}
							char_class = pil_char_class_table [pps->char_code];
							if (char_class == PIL_C_DIGIT)
							{
								PIL_STUFF_TOKEN ();
								pps->char_code = (*pil_getchar)();
								continue;
							}
							PIL_END_TOKEN (PIL_FALSE);
							pps->token_constant = pil_atof ((char PTRTYPE *) pps->token);
							return (pps->token_type = PIL_TOKEN_FCONSTANT);
					}
				}
				break;
			}
				

			case PIL_S_STRING:
			{
				while (1)
				{
					pps->char_code = (*pil_getchar)();
					if (pps->char_code == PIL_REOF || pps->char_code == PIL_END_OF_LINE)
					{
						pps->initial_fetch = PIL_FALSE;
						pps->token_error = PIL_TOKEN_ERR_NO_STREND;
						return (pps->token_type = PIL_TOKEN_ERROR);
					}
					char_class = pil_char_class_table [pps->char_code];
					if (char_class == PIL_C_STRING)
					{
						PIL_END_TOKEN (PIL_TRUE);
						return (pps->token_type = PIL_TOKEN_STRING);
					}
					PIL_STUFF_TOKEN ();
				}
				break;
			}
		}
	}
}


PIL_BOOL pil_is_token_int8 ()
{
	if 	
	(
		pil_parser_state.token_type 	== PIL_TOKEN_ICONSTANT	&&
		pil_parser_state.token_constant >= PIL_INT8_MIN			&&
		pil_parser_state.token_constant <= PIL_INT8_MAX
	)
			return (PIL_TRUE);
	else	return (PIL_FALSE);
}


PIL_BOOL pil_is_token_int16 ()
{
	if 	
	(
		pil_parser_state.token_type 	== PIL_TOKEN_ICONSTANT	&&
		pil_parser_state.token_constant >= PIL_INT16_MIN		&&
		pil_parser_state.token_constant <= PIL_INT16_MAX
	)
			return (PIL_TRUE);
	else	return (PIL_FALSE);
}


PIL_BOOL pil_is_token_int32 ()
{
	if 	
	(
		pil_parser_state.token_type 	== PIL_TOKEN_ICONSTANT	&&
		pil_parser_state.token_constant >= PIL_INT32_MIN		&&
		pil_parser_state.token_constant <= PIL_INT32_MAX
	)
			return (PIL_TRUE);
	else	return (PIL_FALSE);
}


PIL_BOOL pil_is_token_uint8 ()
{
	if 	
	(
		pil_parser_state.token_type 	== PIL_TOKEN_ICONSTANT	&&
		pil_parser_state.token_constant >= PIL_UINT_MIN			&&
		pil_parser_state.token_constant <= PIL_UINT8_MAX
	)
			return (PIL_TRUE);
	else	return (PIL_FALSE);
}


PIL_BOOL pil_is_token_uint16 ()
{
	if 	
	(
		pil_parser_state.token_type 	== PIL_TOKEN_ICONSTANT	&&
		pil_parser_state.token_constant >= PIL_UINT_MIN			&&
		pil_parser_state.token_constant <= PIL_UINT16_MAX
	)
			return (PIL_TRUE);
	else	return (PIL_FALSE);
}


PIL_BOOL pil_is_token_uint32 ()
{
	if 	
	(
		pil_parser_state.token_type 	== PIL_TOKEN_ICONSTANT	&&
		pil_parser_state.token_constant >= PIL_UINT_MIN			&&
		pil_parser_state.token_constant <= PIL_UINT32_MAX
	)
			return (PIL_TRUE);
	else	return (PIL_FALSE);
}


PIL_BOOL pil_is_token_double ()
{
	if 	
	(
		pil_parser_state.token_type == PIL_TOKEN_ICONSTANT	||
		pil_parser_state.token_type == PIL_TOKEN_FCONSTANT
	)
			return (PIL_TRUE);
	else	return (PIL_FALSE);
}


PIL_BOOL pil_is_token_string ()
{
	if (pil_parser_state.token_type == PIL_TOKEN_STRING) return (PIL_TRUE);
	else return (PIL_FALSE);
}


PIL_BOOL pil_is_token_symbol ()
{
	if (pil_parser_state.token_type == PIL_TOKEN_SYMBOL) return (PIL_TRUE);
	else return (PIL_FALSE);
}


PIL_BOOL pil_is_token_beg_block ()
{
	if (pil_parser_state.token_type == PIL_TOKEN_BEG_BLOCK) return (PIL_TRUE);
	else return (PIL_FALSE);
}


PIL_BOOL pil_is_token_end_block ()
{
	if (pil_parser_state.token_type == PIL_TOKEN_END_BLOCK) return (PIL_TRUE);
	else return (PIL_FALSE);
}


PIL_BOOL pil_is_token_error ()
{
	if (pil_parser_state.token_type == PIL_TOKEN_ERROR) return (PIL_TRUE);
	else return (PIL_FALSE);
}


PIL_BOOL pil_is_token_eof ()
{
	if (pil_parser_state.token_type == PIL_TOKEN_EOF) return (PIL_TRUE);
	else return (PIL_FALSE);
}
