/* asc2word ("ASCII-to-Word")  Process a normal ASCII text file for use
   with the Microsoft "Word" word processing program.

This program takes much of the manual work out of the job of
converting a conventional test file into a file easily editable with
Microsoft Word, by converting carriage return/linefeed combinations
(which are taken to be paragraph boundaries by Word) into either
Word's end-of-line marker (shift-CR in Word), thus preserving line
boundaries, or spaces, thus allowing paragraph filling.
	   
The -n option indicates that the document is to be processed
(initially, at least - see ".nf" and ".fi" commands below) with the
"No-fill" option; CR's are converted to end-of-line markers (shift-CR
in Word).  This has the overall effect of preserving the line
boundaries in the original document, but not converting each line to
a paragraph, which adds extra blank lines with many paragraph styles.

Without the "-n" option, single CR's are (initially) converted to
spaces, which allows Word to fill all lines in a paragraph according
to the style sheet. 

Regardless of the -n option setting, two successive CR's are always
interpreted as a paragraph boundary.

The fill/nofill option can changed as needed throughout the document
(to accomodate tables, for example, in the middle of otherwise
fillable text).  To switch to no-fill, insert a line containing only
".nf" (no quotes) in the document at the appropriate place; the
following text will then be processed by converting CR's to Word
end-of-line markers.  To switch to fill mode, insert a line
containing only ".fi" (no quotes) in the document; the following text
will be processed by converting CR's to spaces.

There can be as many ".nf" and ".fi" lines in the document as needed,
in any order, but each must be alone on its own line.

Compiled with Microsoft C 3.0.

George Marshall  11-10-86

	Usage: asc2word [-n] input_file output_file
*/

#include <stdio.h>
#include <ctype.h>

FILE *istream, *ostream;

#define	NEWLINE	0x0a
#define TRUE 1
#define FALSE 0

#define MAXLINE	512
char	line[MAXLINE]={'A'};		/* the input buffer */

int	lastnull=TRUE;
					/* end-of-line for word (shift-CR) */
#define WORDEOL 0x0b
#define SPACE ' '

char	nofill[]={".nf"};
char	yesfill[]={".fi"};
int	fill_sw = TRUE;			/* switch: default is to fill */
int	lineno=0;
int	argn = 0;

main (argc, argv)
int argc;
char *argv[];
{
	int	i, maxchar;
					/* first, get the options and
					filenames. */
	if (argc == 1) {
		printf ("Usage: asc2word [-n] input_file output_file\n");
		printf ("	-n (optional) means start out as No-fill\n");
		printf ("	(i.e., don't let Word fill in or justify lines\n");
		printf ("	.fi on its own line means start filling\n");
		printf ("	.nf on its own line means start no-fill\n");
		return;
	}

	argn = 1;			/* first arg to be examined */

	if (*argv[argn] == '-') {
		if (toupper( *(argv[argn]+1) ) == 'N') {
					/* arg is no-fill option */
			fill_sw = FALSE;
			argn++;
		}
		else {
			printf("Sorry - I don't understand option %s\n",
				argv[argn]);
			exit(-1);
		}
	}
			
	if ((istream = freopen (argv[argn], "rt", stdin)) == NULL) {
		printf ("I can't find the input file %s", argv[argn]);
		return;
	}
	argn++;
	
	if ((ostream = fopen (argv[argn], "wt")) == NULL) {
		printf ("I can't open the output file %s", argv[argn]);
		fclose (istream);
		return;
	}

	while ((fgets (line, MAXLINE, istream)) != (char *)NULL) {
	    lineno++;
					    /* get length of line */
	    maxchar = strlen(line) - 1;

	    if (line[maxchar] == NEWLINE) line[maxchar] = 0;

	    if (strcmp(line, nofill) == 0) {
					        /* found no-fill marker: cease
						removing CR's in this region:
						just convert to EOL's */
						/* were in fill mode - force
						new paragraph if needed. */
		    if (fill_sw == TRUE && !lastnull) putc(NEWLINE, ostream);
		    fill_sw = FALSE;
		    lastnull = TRUE;
		    continue;			/* next line, please */
	    }
	    if (strcmp(line, yesfill) == 0) {
					        /* found fill marker - turn
						on filling again, which means
						all CRs get converted to
						spaces, unless after another
						CR to signal end of parag.*/
						/* were in nofill mode - force
						new paragraph if needed. */
		    if (fill_sw == FALSE && !lastnull) putc(NEWLINE, ostream);
		    fill_sw = TRUE;
		    lastnull = TRUE;
		    continue;
	    }
	    if ( (maxchar == 0) || (whiteline(line, maxchar) )) {
					        /* terminate last line with 
						cr/lf if this one is a null
						line, so Word doc will have 
						a paragraph marker wherever
					        the original doc had blank
						lines. */
		    putc(NEWLINE, ostream);
		    lastnull = TRUE;
	    }
	    else {
						/* normal line - write out the
						line terminator for the last
						line, and the current line
						(with no terminator) */
		    if (lastnull) 
			    putc(NEWLINE, ostream);
		    else {
			    if (fill_sw) putc(SPACE,   ostream);
			    else         putc(WORDEOL, ostream);
		    }
						/* if we are in fill mode,
						get rid of leading blanks
						etc for better appearance.*/
		    i = 0;
		    if (fill_sw) i = first_char(line);
		    fprintf(ostream, "%s", &line[i]);
		    lastnull = FALSE;
	    }
    }
		    putc(NEWLINE, ostream);	/* final line terminator */

}
whiteline (line, num)
char *line;
int num;
{
					/* this function returns true if 
					the line consists only of whitespace:
					spaces and tabs */
	int i;
	for (i=0; i < num; i++) if (!isspace(line[i])) return(FALSE);
	return (TRUE);
}

first_char(line)
char *line;
					/* return index of first (non-white)
					printing character */
{
	int i;
	for (i=0 ; line[i] != 0 ; i++) if (!isspace(line[i])) return(i);
	return strlen(line);
}
