'--------------------------------------------------------------------
' PROGRAM: SIMCOM -- a SIMple COMmunications program in
'  Turbo Basic.
'
' AUTHOR: Reid Collins
' WRITTEN: April 25, 1987
'
' DESCRIPTION:
'  SIMCOM is a simple program that provides a basic level of
'  full-duplex asynchronous communications support with a few
'  amenities to enhance its usefulness.
'
'--------------------------------------------------------------------

'--------------------------------------------------------------------
' Initialization -- default values and possible override of the
' communications parameters from a disk init file.
'--------------------------------------------------------------------
CLEAR		' initialize memory; clear event traps
CLOSE
DEFINT A-Z	' all vars are ints unless noted otherwise

'--- Boolean values ---
False = 0
True = NOT False

'--- set up an error handler ---
ON ERROR GOTO ErrorHandler

'--- define screen colors ---
Fgnd = 2
Bkgnd = 0

'--- default communications parameters ---
Portnum = 1
Port$ = "COM1"
Rate$ = "1200"	' specify as bits per second
Parity$ = "E"   ' even parity
Databits$ = "7" ' ASCII characters only
Stopbits$ = "1" ' 1 or 2 stop bits

'--- set screen and cursor ---
SCREEN 0, 1, 0, 0
COLOR FGND, BKGND
CLS
LOCATE 1, 1, 1

'--- flow control variables and constants ---
Paused = False
PausedCount = 0
WarningLevel = 128
Waiting = 0	' comm input queue size (used for testing)

'--- read init file, if any ---
10 OPEN "SIMCOM.INI" FOR INPUT AS #2
   INPUT #2, Portnum, Port$, Rate$, Parity$, Databits$, Stopbits$
   CLOSE #2	

'--- enable communications event trapping ---
20 IF Portnum < 0 OR Portnum > 2 THEN
     PRINT "Bad port specification: COM"; Portnum
     END
   END IF
   COM(Portnum) ON
   ON COM(Portnum) GOSUB GetComInput

'--- define values for a BREAK signal ---
IF Portnum = 1 THEN
  PortAddr = &H3FB	' COM1 control port address
ELSE
  PortAddr = &H2FB      ' COM2 control port address
END IF
BreakBit = &H40         ' bit pattern for BREAK on/off control
BreakPeriod! = 0.5	' seconds to hold a SPACING signal

'--- open the needed data streams ---
COMPARM$ = Port$+":"+Rate$+","+Parity$+","+Databits$+","+Stopbits$
PRINT "SIMCOM Parameters: " + COMPARM$
OPEN COMPARM$ AS #1    ' serial port default setup

'--------------------------------------------------------------------
' Keyboard dialing reoutine -- get a number from the user and call
' the remote system (configured for Hayes-compatible tone dialing).
'--------------------------------------------------------------------
INPUT "Number to call: ", Number$  ' get number for the host
PRINT #1, "ATDT" + Number$  ' call using tone dialing

'--------------------------------------------------------------------
' Main loop -- enter an interactive session with the remote system.
' The program spends most of its time waiting for the user to type
' something.  When data arrives at the com port, it is read and acted
' upon immediately, interrupting anything the keyboard routines may
' be doing at the time.
'--------------------------------------------------------------------
MainLoop:
  WHILE True
    ' process keyboard input, if any
    IF INSTAT THEN
      ' got something -- check for special keys
      K$ = INKEY$
      IF LEN(K$) = 2 THEN
        GOSUB ExtendedCode
      ELSE  ' send anything else to the remote host
        PRINT #1, K$;
      END IF
    END IF
  WEND

'--------------------------------------------------------------------
' Extended code subroutine -- process selected extended key codes.
'--------------------------------------------------------------------
ExtendedCode:	
  SELECT CASE ASC(RIGHT$(K$, 1))  ' read the scan code
  CASE 59  ' F1 key pressed -- send a hardware BREAK signal
    GOSUB SendBreak
  CASE 83  ' Del key pressed -- send a real ASCII DEL code
    PRINT #1, CHR$(127);
  CASE 117  ' Ctrl-end pressed -- return to BASIC (or DOS)
            ' after reporting flow control data
    CLS
    PRINT PausedCount; "XOFF(s) sent to the remote system"
    PRINT "Longest waiting data stream ="; Waiting; "Character(s)"
    PRINT "Bye from SIMCOM"
    END
  END SELECT
  RETURN

'--------------------------------------------------------------------
' Minimal error-handler routine -- a disk error means there is no
' init file.  Anything else causes an immediate return to the
' keyboard input routine (so the user can try to recover)
'--------------------------------------------------------------------
ErrorHandler:
  IF 10 = ERL THEN  ' no initialization file found
    PRINT "Using default communication parameters"
    RESUME 20
  ELSE  ' go back to reading the keyboard
    RESUME MainLoop
  END IF

'--------------------------------------------------------------------
' Received data subroutine -- process communications line data
' coming in from the host.
'--------------------------------------------------------------------
GetComInput:
  IF EOF(1) THEN
    IF Paused THEN
      Paused = False		' OK for host to send again
      PRINT #1, CHR$(17);	' send an XON to the host
    END IF
    RETURN
  END IF
  ' next two lines are for testing the communications input buffer
  InputBufLen = LOC(1)
  IF InputBufLen > WarningLevel AND not Paused THEN
    Paused = TRUE		' input buffer filling up
    PRINT #1, CHR$(19);		' send an XOFF to the host
    PausedCount = PausedCount + 1
  END IF
  IF InputBufLen > Waiting THEN Waiting = InputBufLen
  ' read from the communications buffer
  I$ = INPUT$(1, #1)
  IF I$ = CHR$(8) THEN
    ' simulate a non-destructive backspace
      IF POS(0) > 1 THEN LOCATE , POS(0) - 1
  ELSEIF I$ = CHR$(13) THEN
    LOCATE , 1	' simulate a lone carriage return
  ELSE
    PRINT I$;	' display anything else unchanged
  END IF
  GOTO GetComInput	        ' see whether there's any more input

'--------------------------------------------------------------------
' BREAK signal subroutine -- holds the communication line at logical
' zero for a period of time that exceeds one normal character
' transmission period (~1/2 second is commonly used).
'--------------------------------------------------------------------
SendBreak:
  OUT PortAddr, (INP(PortAddr) OR BreakBit)	 ' set BREAK bit
  DELAY BreakPeriod!
  OUT PortAddr, (INP(PortAddr) AND NOT BreakBit) ' clear BREAK bit
  RETURN
