program testdiv;

(*  This program tests recording and playback dividers by recording and
    playing back a short snippet of sound at specified rates.  The sampling
    rate in Hz is displayed.  The sample can also be saved to a file as
    an 8-bit unsigned raw PCM file.  *)

uses dos, crt;

(*********************************************************************)
(***************************** types *********************************)
(*********************************************************************)

type
  offseg = record            (* for isolating parts of a pointer *)
      offset,                (* pointer offset *)
      segment:               (* pointer segment *)
        word;
    end; (* record *)

(*********************************************************************)
(************************ global variables ***************************)
(*********************************************************************)

var
  int1bdefault,              (* default Int 1Bh vector *)
  int23default,              (* default Int 23h vector *)
  nextexit,                  (* pointer to next exit procedure in chain *)
  buffer:                    (* pointer to sound buffer on the heap *)
    pointer;
  recdivider,                (* divider for recording *)
  playdivider,               (* divider for playback *)
  bufsize,                   (* size of sound buffer *)
  menuchoice:                (* user function selection *)
    word;
  sounddone:                 (* true when sound I/O is complete *)
    boolean;

(*********************************************************************)
(********************* interrupt service routines ********************)
(*********************************************************************)

procedure int1b; interrupt;
  (*  This replacement Int 1Bh handler does nothing and returns.  This
      disables <control>-<break>.  *)

begin (* int1b *)
end; (* int1b *)

(*********************************************************************)

procedure int23; interrupt;
  (*  This replacement Int 23h handler does nothing and returns.  This
      disables <control>-C.  *)

begin (* int23 *)
end; (* int23 *)

(*********************************************************************)
(*************************** subroutines *****************************)
(*********************************************************************)

function dac_found:
    boolean;
  (*  This function returns true if a Tandy DAC is detected.  *)

var
  regs:                      (* registers for Int 1Ah *)
    registers;

begin (* dac_found *)
  regs.AX := $8003;
  regs.CX := 0;
  intr( $1A, regs );
  if regs.CX = $5353 then
    begin
    dac_found := false;
    exit;
    end;
  regs.AX := $8100;
  intr( $1A, regs );
  dac_found := regs.AX <> $8100;
end; (* dac_found *)

(*********************************************************************)

{$L TESTDIV}
procedure set_i15(
  sounddone:                 (* true if sound I/O is complete *)
    pointer;
  hook:                      (* flag:  0 = hook Int 15h; 1 = unhook Int 15h *)
    integer ); external;
  (*  This external subroutine hooks or unhooks Int 15h to intercept the
      BIOS callout when sound I/O is complete.  The replacement interrupt
      handler, part of Testdiv.obj, sets the boolean flag at flagptr to
      true.  *)

(*********************************************************************)

procedure dac_init;
  (*  This procedure initializes the DAC, preparing it for sound I/O.  Uses
      global variable sounddone.  *)

var
  regs:                      (* registers for Int 1Ah *)
    registers;

begin (* dac_init *)
  sounddone := false;
  repeat
    regs.AH := $84;
    intr( $1A, regs );
    repeat
      regs.AH := $81;
      intr( $1A, regs );
    until not odd( regs.flags );
  until sounddone;
end; (* dac_init *)

(*********************************************************************)

procedure show_menu(
  recdivider,                (* recording divider to display *)
  playdivider,               (* playback divider to display *)
  bufsize:                   (* sound buffer size to display *)
    word );
  (*  This procedure displays a menu to the user.  *)

begin (* show_menu *)
  clrscr;
  writeln( 'Testdiv - DAC divider test program' );
  write( 'Buffer size:  ', bufsize:5 );
  write( '    Dividers:  ', playdivider, ' (play), ' );
  writeln( recdivider, ' (record)' );
  writeln( 'Select one:' );
  writeln( '  1  Record' );
  writeln( '  2  Play' );
  writeln( '  3  Set buffer size' );
  writeln( '  4  Set playback divider' );
  writeln( '  5  Set recording divider' );
  writeln( '  6  Write buffer to disk' );
  writeln( '  7  Increment playback divider and play' );
  writeln( '  8  Decrement playback divider and play' );
  writeln( '  9  Halt program' );
  writeln;
  write( 'Enter your selection:  ' );
end; (* show_menu *)

(*********************************************************************)

procedure err(
  st:                        (* string to display *)
    string );
  (*  This procedure displays a string, then waits for the user to hit
      <return>.  (For debugging.)  *)

begin (* err *)
  write( st, '  Press <enter>' );
  readln;
end; (* err *)

(*********************************************************************)

procedure get_word(
  var entry:                 (* integer entered by user, returned *)
    word;
  lolimit,                   (* lowest acceptable value *)
  hilimit:                   (* highest acceptable value *)
    word );
  (*  This procedure gets an unsigned integer in the indicated range from
      the user.  *)

var
  instring:                  (* string input from user *)
    string;
  valid:                     (* true if number is valid *)
    boolean;
  code:                      (* return code from val() procedure *)
    integer;

begin (* get_word *)
  repeat
    readln( instring );
    val( instring, entry, code );
    valid := (code = 0) and (entry >= lolimit) and (entry <= hilimit);
    if not valid then
      write( 'Enter a number between ', lolimit, ' and ', hilimit, ':  ' );
  until valid;
end; (* get_word *)

(*********************************************************************)

procedure do_record(
  buffer:                    (* pointer to sound buffer *)
    pointer;
  bufsize,                   (* number of bytes to record *)
  recdivider:                (* divider for recording *)
    word );
  (*  This procedure records a buffer full of sound data.  Uses global
      variable sounddone to detect when done recording.  *)

var
  regs:                      (* registers for Int 1Ah *)
    registers;

begin (* do_record *)
  regs.AH := $82;
  regs.BX := offseg( buffer ).offset;
  regs.ES := offseg( buffer ).segment;
  regs.CX := bufsize;
  regs.DX := recdivider;
  sounddone := false;
  intr( $1A, regs );
  while not sounddone do;
end; (* do_record *)

(*********************************************************************)

procedure do_play(
  buffer:                    (* pointer to sound buffer *)
    pointer;
  bufsize,                   (* number of bytes to play *)
  playdivider:               (* divider for playing *)
    word );
  (*  This procedure plays a buffer full of sound data.  Uses global
      variable sounddone to detect when done playing.  *)

var
  regs:                      (* registers for Int 1Ah *)
    registers;

begin (* do_play *)
  regs.AH := $83;
  regs.AL := 7;
  regs.BX := offseg( buffer ).offset;
  regs.ES := offseg( buffer ).segment;
  regs.CX := bufsize;
  regs.DX := playdivider;
  sounddone := false;
  intr( $1A, regs );
  while not sounddone do;
end; (* do_play *)

(*********************************************************************)

procedure write_buffer(
  buffer:                    (* pointer to sound buffer *)
    pointer;
  bufsize:                   (* number of bytes to write to disk *)
    word );
  (*  This procedure writes the contents of the sound buffer to disk,
      using file TESTSND.UB in the current directory.  *)

var
  outfile:                   (* output file *)
    file;

begin (* write_buffer *)
  assign( outfile, 'TESTSND.UB' );
  {$I-} rewrite( outfile, 1 ); {$I+}
  if IOResult <> 0 then
    begin
    err( 'Unable to create output file.' );
    exit;
    end;
  {$I-} blockwrite( outfile, buffer, bufsize ); {$I+}
  if IOResult <> 0 then
    err( 'Error writing file.' );
  close( outfile );
end; (* write_buffer *)

(*********************************************************************)
(************************** exit procedure ***************************)
(*********************************************************************)

{$F+} procedure mainexit; {$F-}
  (*  This procedure, which is executed when the program halts for any
      reason, unhooks interrupts 15h, 1Bh and 23h and finalizes the
      DAC.  *)

begin (* mainexit *)
  dac_init;
  set_i15( nil, 1 );
  setintvec( $1B, int1bdefault );
  setintvec( $23, int23default );
  exitproc := nextexit;
end; (* mainexit *)

(*********************************************************************)
(************************** main program *****************************)
(*********************************************************************)

begin (* testdiv *)
    (* hook interrupts, disabling <control>-C and <control>-<break>, and
       set exit procedure *)
  getintvec( $1B, int1bdefault );
  getintvec( $23, int23default );
  set_i15( @sounddone, 0 );
  setintvec( $1B, @int1b );
  setintvec( $23, @int23 );
  nextexit := exitproc;
  exitproc := @mainexit;

    (* check for Tandy DAC *)
  if not dac_found then
    begin
    writeln( 'This program requires a Tandy DAC.' );
    halt;
    end;

    (* initialize the DAC *)
  dac_init;

    (* initialize dividers (set for 5020Hz) and buffer size to default
       values *)
  playdivider := 713;
  recdivider := 70;
  bufsize := 65526;

    (* allocate sound DMA buffer *)
  if maxavail < bufsize then
    begin
    writeln( 'Insufficient memory.' );
    halt;
    end;
  getmem( buffer, bufsize );

    (* main menu loop *)
  repeat
    show_menu( recdivider, playdivider, bufsize );
    get_word( menuchoice, 1, 9 );
    case menuchoice of
      1:    (* record sound *)
          begin
          write( 'Press return to begin recording.' );
          readln;
          do_record( buffer, bufsize, recdivider );
          end;
      2:    (* play sound *)
          do_play( buffer, bufsize, playdivider );
      3:    (* set buffer size *)
          begin
          writeln( 'Enter sound buffer size:' );
          get_word( bufsize, 1, 65526 );
          end;
      4:    (* set playback divider *)
          begin
          writeln( 'Enter new playback divider:' );
          get_word( playdivider, 1, 4095 );
          end;
      5:    (* set recording divider *)
          begin
          writeln( 'Enter new recording divider:' );
          get_word( recdivider, 1, 4095 );
          end;
      6:    (* write sound buffer to disk *)
          begin
          writeln( 'Writing raw sound to file:  TESTSND.UB' );
          write_buffer( buffer, bufsize );
          end;
      7:    (* increment divider and play *)
          begin
          if playdivider < 4095 then
            playdivider := playdivider + 1;
          do_play( buffer, bufsize, playdivider );
          end;
      8:    (* decrement divider and play *)
          begin
          if playdivider > 1 then
            playdivider := playdivider - 1;
          do_play( buffer, bufsize, playdivider );
          end;
      9:    (* exit program *)
    end; (* case *)
  until menuchoice = 9;
end. (* testdiv *)