/*
 *  Routines for sampling from Soundblaster 8-bit soundcards
 *  These routines require the SB functions and DMA functions written
 *  by Heath I. Hunnicutt.  The required functions are included here.
 *  For the full copy of his routines, refer to:
 *  ftp://ftp.inf.tu-dresden.de/pub/ms-dos/sound/program/sb_dsp.zip
 *  and, at the same location, dma_code.zip
 *
 *  Copyright (C) 1995  Philip VanBaren
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "specgram.h"

#ifdef SC_SB8

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <dos.h>
#include <alloc.h>
#include <mem.h>

#include "display.h"
#include "sb.h"
#include "extern.h"

/* Function prototypes */
void reset_sb8(void);
void halt_sb8(void);
void cleanup_sb8(void);
void recordblock_sb8(void *);
void set_mixer_sb8(int mix,int level);

unsigned int sbpro_get_line_level(void)
{
   unsigned int val;
   outportb(sb_addr+4,0x2E);
   val=inportb(sb_addr+5);
   return(((val&0x0F)+(val/16))*100/32);
}

unsigned int sbpro_get_cd_level(void)
{
   unsigned int val;
   outportb(sb_addr+4,0x28);
   val=inportb(sb_addr+5);
   return(((val&0x0F)+(val/16))*100/32);
}

unsigned int sbpro_get_mic_level(void)
{
   unsigned int val;
   outportb(sb_addr+4,0x0A);
   val=inportb(sb_addr+5);
   return((val&0x07)*100/8);
}

volatile int sb_record_buffer;
unsigned char far *sb_buffer[2];

int atox(char *ptr)
{
   /* Convert ascii hex values to integer */
   int v=0;

   while( ((*ptr>='0') && (*ptr<='9')) || ( ((*ptr|0x20)>='a') && ((*ptr|0x20) <= 'f') ) )
   {
      v=v*16;
      if(*ptr<='9')
         v=v+*ptr-'0';
      else
         v=v+(*ptr|0x20)-'a'+10;
      ptr++;
   }
   return v;
}

#define is_blaster(var) ((((var)[0]=='B')||((var)[0]=='b')) && \
                         (((var)[1]=='L')||((var)[1]=='l')) && \
                         (((var)[2]=='A')||((var)[2]=='a')) && \
                         (((var)[3]=='S')||((var)[3]=='s')) && \
                         (((var)[4]=='T')||((var)[4]=='t')) && \
                         (((var)[5]=='E')||((var)[5]=='e')) && \
                         (((var)[6]=='R')||((var)[6]=='r')))

void far *aligned_malloc(long len)
{
   void far *ptr;
   unsigned int far *orig_ptr;
   unsigned seg,orig_seg,orig_off;
   long lin_addr;

   /* Allocate double the required space */
   ptr=farmalloc(len*2+32);
   if(ptr!=NULL)
   {
      /* Compute a new pointer to the buffer such that the next "len" bytes */
      /* do not cross a page boundary, and the offset is zero */
      /* (as required by DMA transfers) */
      orig_off=FP_OFF(ptr);
      orig_seg=FP_SEG(ptr);
      /* reserve 4 bytes for the original pointer */
      lin_addr=(orig_seg*16L)+orig_off+4L;
      if((lin_addr&0xF0000L) != ((lin_addr+len-1)&0xF0000L))
         lin_addr=(lin_addr+len-1)&0xF0000L;
      else
         lin_addr=(lin_addr+15)&0xFFFF0L;

      seg=(int)(lin_addr/16);
      orig_ptr=(unsigned far *)MK_FP(seg-1,0x000C);
      orig_ptr[0]=orig_off;
      orig_ptr[1]=orig_seg;
      ptr=MK_FP(seg,0);
      #ifdef DEBUG_OUTPUT
         printf("Original: %04x:%04x, New: %04x:%04x, Linear: %05lx\n",
         orig_seg,orig_off,FP_SEG(ptr),FP_OFF(ptr),lin_addr);
      #endif
   }
   return ptr;
}

void aligned_free(void far *ptr)
{
   if(ptr!=NULL)
   {
      unsigned far *old_ptr=(unsigned far *)MK_FP(FP_SEG(ptr)-1,0x000c);
      #ifdef DEBUG_OUTPUT
         printf("Freeing: %04x:%04x, Ptr: %04x:%04x\n",
         FP_SEG(old_ptr),FP_OFF(old_ptr),FP_SEG(ptr),FP_OFF(ptr));
      #endif
      farfree(MK_FP(old_ptr[1],old_ptr[0]));
   }
}

/* Interrupt handler for DMA complete IRQ from Soundblaster */
void far interrupt (*OldIRQ)();

int error_in_interrupt=0;

void far interrupt SBHandler()
{
   buffer_full=1;
   sb_record_buffer=1-sb_record_buffer;

   /* Start recording a new buffer. */
   if(dma_setup(sb_dma,(char far *)sb_buffer[sb_record_buffer],fftlen,0))
      error_in_interrupt=1;
   else if(dma_errno!=0)
      error_in_interrupt=1;
   else
      dsp_dma_prepare(0,fftlen);

   inportb(DSP_DATA_AVAIL);
   outportb(0x20,0x20);
}

void init_sb8(char **environ)
{
   int i;
   unsigned char tm,im;
   long timeout;
   // Scan the environment variables for BLASTER=Axxx Ix Dx
   for(i=0;environ[i]!=NULL;i++)
   {
      if(is_blaster(environ[i]))
      {
         int j;
         DOUT("SB8: Found the BLASTER environment variable:");
         for(j=8;environ[i][j]!=0;j++)
         {
            if((environ[i][j]=='A')||(environ[i][j]=='a'))
            {
               DOUT("SB8: Axxx found");
               sb_addr=atox(&environ[i][j+1]);
            }
            if((environ[i][j]=='D')||(environ[i][j]=='d'))
            {
               DOUT("SB8: Dx found");
               sb_dma=atoi(&environ[i][j+1]);
            }
            if((environ[i][j]=='I')||(environ[i][j]=='i'))
            {
               DOUT("SB8: Ix found");
               sb_irq=atoi(&environ[i][j+1]);
            }
            // Skip to the next parameter
            while((environ[i][j] != ' ') && (environ[i][j+1] != 0))
	            j++;
         }
         break;
      }
   }
   #ifdef DEBUG_OUTPUT
   {
      char message[100];
      sprintf(message,"SB8: Address=0x%03x, DMA=%d, IRQ=%d",sb_addr,sb_dma,sb_irq);
      DOUT(message);
   }
   #endif
   if((sb_dma<0)||(sb_dma>3))
   {
      puts("Only SB DMA channels up to 3 are supported.");
      exit(1);
   }
   if((sb_irq<0)||(sb_irq>7))
   {
      puts("Only SB IRQs up to 7 are supported.");
      exit(1);
   }
   /* Allocate the sampling buffers */
   DOUT("SB8: Allocating the buffers");
   sb_buffer[0]=aligned_malloc(MAX_LEN);
   sb_buffer[1]=aligned_malloc(MAX_LEN);
   if((sb_buffer[0]==NULL) || (sb_buffer[1]==NULL))
   {
      puts("Unable to allocate enough memory for the sampling buffers!");
      exit(1);
   }
   _fmemset(sb_buffer[0],0,MAX_LEN);
   _fmemset(sb_buffer[1],0,MAX_LEN);
   reset_soundcard=reset_sb8;
   halt_soundcard=halt_sb8;
   cleanup_soundcard=cleanup_sb8;
   recordblock=recordblock_sb8;
   sample_size=8;
   mixers=0;
   stereo=0;   // Force mono mode

   DOUT("SB8: Initializing the sound card");
   set_SB_address(sb_addr);
   if(!dsp_reset())
   {
      draw_cleanup_graphics();
      printf("Soundblaster not found at 0x%04x\n",sb_addr);
      exit(1);
   }
   dsp_voice(0);

   /* Check if we can do mixers, and enable them, if so */
   timeout=1000000L;
   /* Wait for bit 7 to be cleared, or for a timeout */
   while(  (inportb(sb_addr+0x0E) & 0x80)  && (--timeout));
   outportb(sb_addr+0x0C,0xE1);    /* Get DSP version number */
   timeout=1000000L;
   /* Wait for bit 7 to be set, or for a timeout */
   while((!(inportb(sb_addr+0x0E) & 0x80)) && (--timeout));
   i=inportb(sb_addr+0x0A);     /* Get the major version number */
   while((!(inportb(sb_addr+0x0E) & 0x80)) && (--timeout));
   inportb(sb_addr+0x0A);       /* Get the major version number */

   if(i>2)
   {
      DOUT("SB8: Found an SBPro or greater, mixers are available");
      mixers=1;
      set_mixer=set_mixer_sb8;
      mic_level=sbpro_get_mic_level();
      ext_level=sbpro_get_line_level();
      int_level=sbpro_get_cd_level();
   }

   /*
    *  Add the SB DMA interrupt handler in the interrupt chain
    */
   disable();
   OldIRQ = getvect(0x08 + sb_irq);
   setvect(0x08 + sb_irq,SBHandler);
   im=inportb(0x21);
   tm=~(1 << sb_irq);
   outportb(0x21,im & tm);
   enable();
}

void reset_sb8(void)
{
   /* Initialize Soundblaster stuff */
   int dsp_tc=floor(256.5-1000000.0/(double)SampleRate);
   if(dsp_tc>255) dsp_tc=255;
   SampleRate=floor(1000000.0/(256-dsp_tc)+0.5);
   dsp_time(dsp_tc);

   /*
    *  Initialize the SB DMA channel
    */
   if(dma_reset(sb_dma))
   {
      draw_cleanup_graphics();
      puts("Error resetting SB DMA channel.");
      puts(dma_errlist[dma_errno]);
      cleanup_sb8();
      exit(1);
   }
   // Reset the buffer pointers
   sb_record_buffer=0;
   buffer_full=0;

   /*
    *  Start recording a new buffer.
    */
   if(dma_setup(sb_dma,(char far *)sb_buffer[sb_record_buffer],fftlen,0))
   {
      draw_cleanup_graphics();
      printf("Error in dma_setup(): %d\n",dma_errno);
      puts(dma_errlist[dma_errno]);
      cleanup_sb8();
      exit(1);
   }
   if(dma_errno!=0)
   {
      draw_cleanup_graphics();
      puts("DMA error");
      puts(dma_errlist[dma_errno]);
      cleanup_sb8();
      exit(1);
   }
   dsp_dma_prepare(0,fftlen);
   inportb(DSP_DATA_AVAIL);
}


void halt_sb8(void)
{
   /* Shut down the DMA transfers */
   disable();
   dma_reset(sb_dma);
   dsp_reset();
   enable();
}


void cleanup_sb8(void)
{
   /* Clean up the soundcard routines */

   unsigned char im,tm;

   disable();
   dma_reset(sb_dma);
   dsp_reset();
   /* Restore old IRQ vector */
   setvect(0x08 + sb_irq,OldIRQ);
   im=inportb(0x21);
   tm=1 << sb_irq;
   outportb(0x21,im | tm);
   enable();
   if(sb_buffer[0]) aligned_free(sb_buffer[0]);
   if(sb_buffer[1]) aligned_free(sb_buffer[1]);
}

void recordblock_sb8(void *buffer)
{
   /* Wait for DMA to complete */
   while(!buffer_full && !done);
   if(error_in_interrupt)
   {
      draw_cleanup_graphics();
      printf("DMA error occurred: %d\n",dma_errno);
      puts(dma_errlist[dma_errno]);
      cleanup_sb8();
      exit(1);
   }
   _fmemcpy(buffer,sb_buffer[1-sb_record_buffer],fftlen);
   buffer_full=0;
}

void set_mixer_sb8(int mix,int level)
{
   /*
    *  Set a mixer level on the PAS16 card
    */
   level=level*16/100;
   if(level>15) level=15;

   if(mix==MIXER_EXT)
   {
      DOUT("SB8: Setting the line mixer level");
      level=level+level*16;
      outportb(sb_addr+4,0x2e);
      outportb(sb_addr+5,level);
      outportb(sb_addr+4,0x0C);  /* Select the line input */
      outportb(sb_addr+5,0x27);
   }
   else if(mix==MIXER_INT)
   {
      DOUT("SB8: Setting the CD mixer level");
      level=level+level*16;
      outportb(sb_addr+4,0x28);
      outportb(sb_addr+5,level);
      outportb(sb_addr+4,0x0C);  /* Select the CD input */
      outportb(sb_addr+5,0x23);
   }
   else if(mix==MIXER_MIC)
   {
      DOUT("SB8: Setting the microphone mixer level");
      outportb(sb_addr+4,0x0a);
      outportb(sb_addr+5,level/2);
      outportb(sb_addr+4,0x0C);  /* Select the mic input */
      outportb(sb_addr+5,0x21);
   }
}

#endif

