/***************************************************************************
FIXANSI 2.23
	(c) Paul J. Bodin, August, 1993.  All rights reserved.
	E-mail address: pjb3@columbia.edu

	This code may not be distributed for profit, nor used
	in programs intended for resale or commercial use without
	express permission.

Description:

	FIXANSI converts files formatted on a screen-oriented basis by
	ANSI escapes to text files formatted for use by Nota Bene (t)
	and XyWrite (t).

	Output may be customized to other formats by adjusting the output
	format strings noted among the global variable declarations.

Command line syntax:

	FIXANSI [d:][path]infile [[d:][path]outfile]

Compiled with Turbo C++ 3.0
***************************************************************************/

#include <io.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#define ESC 0x1b
#define BACKSPACE 0x08
#define CR 0x0d
#define LF 0x0a
#define TAB 0x09
#define ROWS 24
#define COLUMNS 91

FILE *infile, *outfile;

long length, index;
int xcoord, ycoord, count, writeflag;
char this_char, modeno;
char name1[80], name2[80];
char title[] = "\nFIXANSI 2.23, (c) Paul J. Bodin -- all rights reserved\n";
char success[] = "File successfully processed";

/* Output can be customized for other apps by adjusting the following. */

char format[] = "OF10,10TW61FD84PL78,80,72FF0PT17";
char normal[] = "MDNM";
char bold[] = "MDBO";
char ScreenEnd[] = "\r\nMDBO_______________MDNM\r\nPG\r\n";

struct {
	char text[COLUMNS + 1], mode[COLUMNS + 1];    /* Define an apparent screen structure. */
	int linelength;
} line[ROWS + 1];

char *make_bak(char *fname);
int pars_ansi(void);
int change_position(char *yparm, char *xparm);
int write_screen(void);
int clear_screen(void);
int help(int problem);

/***************************************************************************
Function make_bak():

	char *make_bak(char *fname)

Description:

	Receives filespec string pointer (*fname) from calling function,
	creates backup filename and renames the actual file as the backup 
	file.  Checks for errors and returns a pointer (*backup) to the
	backup filespec string.
***************************************************************************/

char *make_bak(char *fname)
{
	char filename[80];
	char extension[] = ".bak";
	char *dot, *backup;

	/* Allocate memory for backup[80] */

	if ((backup = malloc(80)) == NULL)
		help(11);

	/* Assign *fname to both backup and filename. */

	strcpy(backup, fname);
	strcpy(filename, backup);

	/* Check backup for an extension--if none exists, add ".bak" to 
	backup; if one already exists, remove it and substitute ".bak". */

	if((dot = strchr(backup, '.')) == NULL)
		strcat(backup, extension);
	else {
		sprintf(dot, "\0");
		strcat(backup, extension);
	}

	/* We want to rename the actual file to *.bak and use it as the 
	source file.  If its extension is already ".bak", call help(). */

	if(!strcmp(backup, filename))
		help(7);

	/* Does actual source file really exist? 
		If so, does ANOTHER file "*.bak" already exist? 
			If so, delete it (call help() upon error). */

	if(!access(filename, 0)) {
		if(!access(backup, 0)) {
			if((unlink(backup)) != 0)
				help(3);
		}
	}
	else help(4);    /* If no source file, call help() & leave *.bak. */

	/* Rename actual file to desired source filename.
	Check for file handling error. */

	if((rename(filename, backup)) != 0)
		help(4);

	return(backup);
}

/***************************************************************************
Function pars_ansi():

	int pars_ansi(void)

Description:

	Called by the filtering function when it encounters an escape,
	this function parses the ensuing ANSI sequence from infile.
	At the end of the sequence, it selects and calls the function
	that emulates the ANSI call indicated by the last character of
	the escape sequence.  Passes the necessary parameters to the 
	emulation function called, and returns 0 upon receiving the 
	successful return from the emulation function.
***************************************************************************/

int pars_ansi(void)
{
	int stop, digit, row, column;
	char temp[] = "0";
	struct {
		char element[3];
	} parm[3];

	digit = 0;
	stop = 1;


	/* Initialize parameter fields. */

	strcpy(parm[0].element, "\0");
	strcpy(parm[1].element, "\0");
	strcpy(parm[2].element, "\0");


	/* Sort data from the escape sequence
	and call the emulation function. */

	do {
		this_char = getc(infile);
		index++;    /* Maintain record of infile pointer position. */

		switch(this_char) {

			case '[' :    /* Discard "begin" character. */
				break;

			case 'J' :    /* Clear screen. */
				clear_screen();
				stop = 0;
				break;

			case 'K' :    /* Clear line from cursor. */
				for(column = (xcoord - 1); column < line[ycoord - 1].linelength; column++) {
					line[ycoord - 1].text[column] = '\0';
					line[ycoord - 1].mode[column] = '\0';
				}
				line[ycoord - 1].linelength = xcoord - 1;
				stop = 0;
				break;

			case 'h' :    /* Discard video mode string. */
				stop = 0;
				break;

			case 'l' :    /* Discard video mode string. */
				stop = 0;
				break;

			case '=' :    /* Discard irrelevant VT-100 string. */
				stop = 0;
				break;

			case '<' :    /* Discard irrelevant VT-100 string. */
				stop = 0;
				break;

			case '?' :    /* Discard irrelevant character. */
				break;

			case 'q' :    /* Discard irrelevant VT-100 string. */
				stop = 0;
				break;

			case ';' :
				digit++;    /* Shift to next field. */
				break;    /* Discard separator. */

			case 'm' :    /* Adjust display mode flag. */
				sscanf(parm[0].element, "%c", &modeno);
				stop = 0;
				break;

			case 'H' :    /* Call position change function. */
				change_position(parm[0].element, parm[1].element);
				stop = 0;
				break;

			default :    /* Add digit to parameter field. */
				sprintf(temp, "%c", this_char);
				strcat(parm[digit].element, temp);
				break;

		}

	} while(stop != 0);

	return(0);
}

/***************************************************************************
Function change_position():

	int change_position(char *yparm, char *xparm,)

Description:

	Emulates ANSI cursor position change command in the screen data 
	structure (line), updates the apparent cursor, and writes screen 
	data to outfile as necessary.  Receives screen location parameters 
	from the calling function, updates xcoord and ycoord, and 
	returns 0 upon success.
***************************************************************************/

int change_position(char *yparm, char *xparm)
{
	int newx, newy, times;

	/* Emulate a new screen call by writing the screen buffer 
	and re-initialize the apparent cursor. */

	if((strlen(xparm) == 0) && (strlen(yparm) == 0)) {
		write_screen();
		xcoord = ycoord = 1;
	}

	else {

		/* Convert parameter strings to decimal numbers. */

		sscanf(xparm, "%d", &newx);
		sscanf(yparm, "%d", &newy);

		/* If cursor jumps up significantly, 
		treat it as a screen overwrite. */

		if(newy < (ycoord - 1)) {
			write_screen();
			xcoord = newx;
			ycoord = newy;
		}

		/* Pad line with spaces, if necessary. */

		else {
			if((times = line[newy - 1].linelength) < newx) {
				for (times += 1; times < newx; times++)
					if(line[newy - 1].linelength < COLUMNS) {
						line[newy - 1].text[times - 1] = ' ';
						line[newy - 1].mode[times - 1] = modeno;
						line[newy - 1].linelength++;
					}
			}

		/* Adjust apparent cursor. */

		xcoord = newx;
		ycoord = newy;

		}
	}

	return(0);
}

/***************************************************************************
Function write_screen():

	int write_screen(void)

Description:

	Writes the data from the screen data structure (line) to 
	outfile.  Receives nothing and returns 0 upon success.
***************************************************************************/

int write_screen(void)
{
	int row, column;
	char temp, newmode, oldmode;

	oldmode = '0';

	/* Process each line of text in turn. */

	for(row = 0; row < ROWS; row++) {
		for(column = 0; column < line[row].linelength; column++) {

			/* Check for screen attribute change. */

			sscanf(&line[row].mode[column], "%c", &newmode);

			if(newmode != oldmode) {

				switch(newmode) {
					case '0' :
						fputs(normal, outfile);
						break;
					case '1' :
						fputs(bold, outfile);
				}

				oldmode = newmode;
			}

			/* Write the character. */

			sscanf(&line[row].text[column], "%c", &temp);
			fputc(temp, outfile);
		}

		/* End line with CRLF. */

		fputc(CR, outfile);
		fputc(LF, outfile);
	}

	/* Write end of screen marker. */

	fputs(ScreenEnd, outfile);

	/* Clear writeflag. */

        writeflag = 0;

	/* Update patience message... */

	switch(count % 4) {
		case 1 :
			printf("\b/");
			break;
		case 2 :
			printf("\b-");
			break;
		case 3 :
			printf("\b\\");
			break;
		default :
			printf("\b|");
	}
	count++;

	return(0);
}

/***************************************************************************
Function clear_screen():

	int clear_screen(void)

Description:

	Clears the apparent screen data structure (line) and initializes 
	the apparent cursor.  Receives nothing and returns 0 upon success.
***************************************************************************/

int clear_screen(void)
{
	int row, column;

	/* Initialize entire screen data structure. */

	memset(line, '\0', sizeof(line) - 1);

	xcoord = ycoord = 1;    /* Initialize apparent cursor. */

	writeflag = 0;    /* Clear writeflag. */

	return(0);
}

/***************************************************************************
Function help():

	int help(int problem)

Description:

	Writes an error message to stdout and terminates the program,
	sending an exit code to DOS.  Receives a numerical problem code
	from the calling function, returns a message to stdout and an
	error code to DOS.
***************************************************************************/

int help(int problem)
{

	/* Error diagnostic messages. */

	char diag1[] = "Unable to open source file";
	char diag2[] = "Unable to open destination file";
	char diag3[] = "Unable to create backup file";
	char diag4[] = "Source file not found";
	char diag5[] = "WARNING: error in closing file(s) -- check data integrity!";
	char diag6[] = "Syntax error -- please use FIXANSI [d:][path]infile [[d:][path]outfile]";
	char diag7[] = "Syntax error -- cannot backup *.BAK -- please specify destination file";
	char diag9[] = "Cannot write a file to itself -- check filespecs!";
	char diag11[] = "Out of Memory.";
	char diag12[] = "Filespec too long -- exceeds 79 characters!";

	/* Select appropriate error message and write it. */

	switch(problem) {
		case 1 :
			puts(diag1);
			break;
		case 2 :
			puts(diag2);
			break;
		case 3 :
			puts(diag3);
			break;
		case 4 :
			puts(diag4);
			break;
		case 5 :
			puts(diag5);
			break;
		case 6 :
			puts(diag6);
			break;
		case 7 :
			puts(diag7);
			break;
		case 9 :
			puts(diag9);
			break;
		case 11 :
			puts(diag11);
			puts(diag3);
			break;
		case 12 :
			puts(diag12);
			break;
	}

	/* Terminate program and return appropriate error code. */

	exit(problem);

	return(0);
}

/***************************************************************************
Function main():

	int main(int argc, char *argv[])

Description:

	Coordinates other functions and performs the bulk of the
	filtering.  Initializes and maintains an "apparent cursor"
	to support conversion from screen oriented format to text
	oriented format.
***************************************************************************/

int main(int argc, char *argv[])
{
	int tab_diff, pad;

	/* Write program title to screen. */

	puts(title);

	/* Check command line and open appropriate files. */

	switch(argc) {

		case 2 :
			if (strlen(argv[1]) > 79)  /* Check length of filespec. */
				help(12);
			strcpy(name2, argv[1]);
			strcpy(name1, make_bak(argv[1]));
			break;

		case 3 :
			if (strlen(argv[1]) > 79)  /* Check length of first filespec. */
				help(12);
			if (strlen(argv[2]) > 79)  /* Check length of second filespec. */
				help(12);
			strcpy(name1, argv[1]);
			strcpy(name2, argv[2]);

			/* Check that names are not the same. */

			if(!strcmp(name1, name2))
				help(9);
			break;

		default :
			help(6);
	}

	/* Call proper help() routine upon error opening files. */

	if ((infile = fopen((name1), "rb")) == NULL)
		help(1);
	if ((outfile = fopen((name2), "wb")) == NULL)
		help(2);

	fputs(format, outfile);    /* Write text file format header. */

	/* Initialize global data structures. */

	index = 1;
	modeno = '0';
	clear_screen();

	/* Write patience signal... */

	printf("%s ==> %s\n", name1, name2);
	printf("\nWorking |");
	count = 1;

	/* Filter data, parsing any escape sequences. */

	length = filelength(fileno(infile));
	while(index <= length) {
		this_char = getc(infile);
		index++;

		/* Set writeflag. */

		if(!writeflag)
			writeflag = 1;

		switch(this_char) {

			case ESC :    /* Process ANSI codes. */
				pars_ansi();
				break;

			/* The following three cases are included mainly
			as precautions, to insure accurate preservation of
			the apparent cursor if the file has ASCII cursor
			movement characters interspersed with the ANSI
			formatting commands. */

			/* Convert backspace char to destructive backspace,
			adjusting outfile file pointer and apparent cursor. */

			case BACKSPACE :
				xcoord--;
				line[ycoord - 1].text[xcoord - 1] = '\0';
				line[ycoord - 1].linelength = xcoord - 1;
				break;

			case CR :
				break;    /* Discard carriage returns. */

			case LF :
				ycoord++;
				xcoord = 1;    /* Adjust apparent cursor. */
				break;

			/* Change tabs to spaces and adjust the apparent
			cursor, assuming regular eight column tabs. */

			case TAB :
				if((xcoord % 8) == 0)
					tab_diff = 1;
				else
					tab_diff = (9 - (xcoord % 8));

				for(pad = 0; pad < tab_diff; pad++) {
					if(xcoord <= COLUMNS) {
						line[ycoord - 1].text[xcoord - 1] = ' ';
						line[ycoord - 1].mode[xcoord - 1] = modeno;
						if(line[ycoord - 1].linelength <= xcoord)
							line[ycoord - 1].linelength++;
					}
					xcoord++;
				}

				break;

			default :    /* Pass all other characters. */
				if(xcoord <= COLUMNS) {
					line[ycoord - 1].text[xcoord - 1] = this_char;
					line[ycoord - 1].mode[xcoord - 1] = modeno;
					if(line[ycoord - 1].linelength < xcoord)
						line[ycoord - 1].linelength++;
				}
				xcoord++;    /* Adjust apparent cursor. */
		}

		/* Check for truncated line. */

		if(xcoord > COLUMNS)
			printf("Warning: line length exceeded in file %s\n", name1);

		/* Check for sections of the file not formatted with
		ANSI command sequences. */

		if(ycoord > ROWS) {
			write_screen();
			clear_screen();
		}
	}

	/* Check for unwritten data. */

	if(writeflag)
		write_screen();

	if (fcloseall() != 2)    /* Close files, check for error. */
		help(5);

	/* Write program title and success notice upon completion. */

	printf("\b\b\b\b\b\b\b\b\bDone     \n");
	puts(success);

	return(0);
}

/* End of Code, program FIXANSI. */