/* -----------------------------------------------------------------------
   rpmspec.r - PresentationManager specific junk

   ----------------------------------------------------------------------- */

#ifdef PresentationManager

/* ----------------------------- Globals --------------------------------- */
/* handle to the interpreter thread anchor block */
extern HAB HInterpAnchorBlock;

/* the color table */
colorEntry *ColorTable;
colorEntry *FreeCEntries;
colorEntry *UsedCEntries;

/* the local id table */
lclIdentifier *LocalIds;
lclIdentifier *FreeIdEntries;
lclIdentifier *FontIdEntries;
lclIdentifier *PatIdEntries;

/* the binding for the console window - FILE * for simplicity */
FILE *ConsoleBinding = NULL;

/* the resolutions used to select image fonts */
extern LONG FontResX;
extern LONG FontResY;
/* the device max index for logical color table */
extern LONG MaxPSColors;
/* window state list */
extern wstate *wstates;
/* the window that current focus */
extern wstate *CurrentFocus;

/* flags for context setting, querying */
static ULONG lineAttrs = LBB_COLOR | LBB_MIX_MODE | LBB_WIDTH |
                         LBB_GEOM_WIDTH | LBB_TYPE | LBB_END | LBB_JOIN;
static ULONG areaAttrs = ABB_COLOR | ABB_MIX_MODE | ABB_BACK_COLOR |
                         ABB_BACK_MIX_MODE | ABB_SET | ABB_SYMBOL | 
                         ABB_REF_POINT;
static ULONG charAttrs = CBB_COLOR | CBB_BACK_COLOR | CBB_MIX_MODE |
                         CBB_BACK_MIX_MODE | CBB_SET | CBB_MODE | CBB_BOX |
                         CBB_ANGLE | CBB_SHEAR | CBB_DIRECTION;
static ULONG imageAttrs = IBB_COLOR | IBB_MIX_MODE | IBB_BACK_COLOR |
                          IBB_BACK_MIX_MODE;

/* the system color table - made to match hardware - may not work with
   all drivers - index into the table is implied by order  */
LONG defClrTable[16] = {
        RGB_BLACK,
        RGB_BLUE,
        RGB_RED,
        RGB_PINK,
        RGB_GREEN,
        RGB_CYAN,
        RGB_YELLOW,
        MAKERGB(190, 190, 190),  /* pale gray */
        MAKERGB(128, 128, 128),  /* dark gray */
        MAKERGB(0, 0, 128),      /* dark blue */
        MAKERGB(128, 0, 0),      /* dark red */
        MAKERGB(128, 0, 128),    /* dark pink */
        MAKERGB(0, 128, 0),      /* dark green */
        MAKERGB(0, 128, 128),    /* dark cyan */
        MAKERGB(128, 128, 0),    /* brown */
        RGB_WHITE
        };

/* the color name <-> rgb table. You can add more color names and appropriate
   RGB values here */
stringint siColorNames[] = {
  {0, 31},
  {"beige",             MAKERGB(245, 245, 220)},
  {"black",             RGB_BLACK},
  {"blue",              RGB_BLUE},
  {"brown",             MAKERGB(128, 128, 0)},
  {"chocolate",         MAKERGB(210, 105, 30)},
  {"cyan",              RGB_CYAN},
  {"darkblue",          MAKERGB(0, 0, 128)},
  {"darkcyan",          MAKERGB(0, 128, 128)},
  {"darkgray",          MAKERGB(128, 128, 128)},
  {"darkgreen",         MAKERGB(0, 128, 0)},
  {"darkpink",          MAKERGB(128, 0, 128)},
  {"darkred",           MAKERGB(128, 0, 0)},
  {"firebrick",         MAKERGB(178, 34, 34)},
  {"gold",              MAKERGB(255, 215, 0)},
  {"green",             RGB_GREEN},
  {"ivory",             MAKERGB(255, 255, 240)},
  {"magenta",           MAKERGB(255, 0, 255)},
  {"maroon",            MAKERGB(176, 48, 96)},
  {"navy",              MAKERGB(0, 0, 128)},
  {"orange",            MAKERGB(255, 165, 0)},
  {"orchid",            MAKERGB(218, 112, 214)},
  {"palegray",          MAKERGB(190, 190, 190)},
  {"pink",              RGB_PINK},
  {"plum",              MAKERGB(221, 160, 221)},
  {"purple",            MAKERGB(160, 32, 240)},
  {"red",               RGB_RED},
  {"tan",               MAKERGB(210, 180, 140)},
  {"violet",            MAKERGB(238, 130, 238)},
  {"wheat",             MAKERGB(245, 222, 179)},
  {"white",             RGB_WHITE},
  {"yellow",            RGB_YELLOW}
  };

/* string to int pattern symbols */
stringint siPatternSyms[] = {
  {0, 16},
  {"blank",             PATSYM_BLANK},
  {"diagonal1",         PATSYM_DIAG1},
  {"diagonal2",         PATSYM_DIAG2},
  {"diagonal3",         PATSYM_DIAG3},
  {"diagonal4",         PATSYM_DIAG4},
  {"gray1",             PATSYM_DENSE8},
  {"gray2",             PATSYM_DENSE7},
  {"gray3",             PATSYM_DENSE6},
  {"gray4",             PATSYM_DENSE5},
  {"gray5",             PATSYM_DENSE4},
  {"gray6",             PATSYM_DENSE3},
  {"gray7",             PATSYM_DENSE2},
  {"gray8",             PATSYM_DENSE1},
  {"horizontalLines",   PATSYM_HORIZ},
  {"solid",             PATSYM_SOLID}, 
  {"verticalLines",     PATSYM_VERT}};

/* string to int mix modes */
stringint siMixModes[] = {
  {0, 17},
  {"and",               ROP_SRCAND},
  {"andInverted",       ROP_SRCERASE},
  {"andReverse",        ROP_SRCERASE},
  {"clear",             ROP_ZERO},
  {"copy",              ROP_SRCCOPY},
  {"copyInverted",      ROP_NOTSRCCOPY},
  {"equiv",             ROP_SRCCOPY},
  {"invert",            ROP_DSTINVERT},
  {"nand",              ROP_SRCERASE},
  {"noop",              ROP_ONE},
  {"nor",               ROP_MERGEPAINT},
  {"or",                ROP_SRCPAINT},
  {"orInverted",        ROP_MERGEPAINT},
  {"orReverse",         0xDD},
  {"reverse",           ROP_USER1},
  {"set",               ROP_ONE},
  {"xor",               ROP_SRCINVERT}};
 
/* string to int line types */
stringint siLineTypes[] = {
  {0, 9},
  {"alternate",         LINETYPE_ALTERNATE},
  {"dashdot",           LINETYPE_DASHDOT},
  {"dashdoubledot",     LINETYPE_DASHDOUBLEDOT},
  {"dotted",            LINETYPE_DOT},
  {"doubledot",         LINETYPE_DOUBLEDOT},
  {"longdash",          LINETYPE_LONGDASH},
  {"onoff",             LINETYPE_ALTERNATE},
  {"shortdash",         LINETYPE_SHORTDASH},
  {"solid",             LINETYPE_SOLID}};


#define NUMCURSORSYMS	4
/* string to int cursor/pointer symbols */
stringint siCursorSyms[] = {
  { 0, NUMCURSORSYMS},
  {"arrow",		SPTR_ARROW},
  {"clock",		SPTR_WAIT},
  {"ibeam", 		SPTR_TEXT},
  {"xterm",		SPTR_TEXT}};

/* ------------------------ Local Prototypes ----------------------------- */
static lclIdentifier *AllocateLocalID(void);
static void RippleColorAddition(LONG aindx);
static void RippleLocalIdAddition(wcontext *wc, LONG id);
static void RippleLocalIdRemoval(LONG id);

/* ---------------------------- Functions -------------------------------- */
/* ------------------------ Context Functions ---------------------------- */
/* -----------------------------------------------------------------------
   Name - UnsetAllContext - XXX make this a macro
   Purpose - unsets all windows dependant on this context causing a
             reload of all bundles
   Parameters -
   Returns -
   Side Effects -
   ----------------------------------------------------------------------- */
void UnsetAllContext(wcontext *wc)
{
  /* these are macros.. */
  UnsetLineContext(wc);
  UnsetCharContext(wc);
  UnsetAreaContext(wc);
  return;
} /* End of UnsetAllContext */



/* -----------------------------------------------------------------------
   Name - UnsetCharContext
   Purpose - disassociates all dependent windows from the char context -
             thus forcing them to 'reload' upon the next use.  This is
             a way of enforcing a lazy updating mechanism
   Parameters -
   Returns -
   Side Effects -
   ----------------------------------------------------------------------- */
void UnsetCharContext(wcontext *wc)
{
  int i, num;
  wstate *ws;
 
  if (wc) {
    num = wc->numDeps;
    for (i = 0; i < wc->maxDeps && num > 0; i++)
      if (ws = wc->depWindows[i]) {
        num--;
        if (ws->charContext == wc) ws->charContext = NULL;
        } /* End of if - found one */
    } /* End of if - sanity check */
  return;
} /* End of UnsetCharContext */


/* -----------------------------------------------------------------------
   Name - UnsetLineContext
   Purpose - disassociates all dependent windows from the line context -
             thus forcing them to 'reload' upon the next use.  This is
             a way of enforcing a lazy updating mechanism
   Parameters -
   Returns -
   Side Effects -
   ----------------------------------------------------------------------- */
void UnsetLineContext(wcontext *wc)
{
  int i, num;
  wstate *ws;

  if (wc) {
    num = wc->numDeps;
    for (i = 0; i < wc->maxDeps && num > 0; i++)
       if (ws = wc->depWindows[i]) {
         num--;
         if (ws->lineContext == wc) ws->lineContext = NULL;
         } /* End of if - found one */
    } /* End of if - sanity check */
  return;
} /* End of UnsetLineContext */


/* -----------------------------------------------------------------------
   Name - UnsetAreaContext
   Purpose - disassociates all dependent windows from the area context -
             thus forcing them to 'reload' upon the next use.  This is
             a way of enforcing a lazy updating mechanism
   Parameters -
   Returns -
   Side Effects -
   ----------------------------------------------------------------------- */
void UnsetAreaContext(wcontext *wc)
{
  int i, num;
  wstate *ws;

  if (wc) {
    num = wc->numDeps;
    for (i = 0; i < wc->maxDeps && num > 0; i++)
      if (ws = wc->depWindows[i]) {
        num--;
        if (ws->areaContext == wc) ws->areaContext = NULL;
        } /* End of if - found one */
    } /* End of if - sanity check */
  return;
} /* End of UnsetCharContext */


/* -----------------------------------------------------------------------
   Name - UnsetImageContext
   Purpose -
   Parameters -
   Returns -
   Side Effects -
   ----------------------------------------------------------------------- */
void UnsetImageContext(wcontext *wc)
{
  int i, num;
  wstate *ws;

  if (wc) {
    num = wc->numDeps;
    for (i = 0; i < wc->maxDeps && num > 0; i++)
      if (ws = wc->depWindows[i]) {
        num--;
        if (ws->imageContext == wc) ws->imageContext = NULL;
        } /* End of if - found one */
     } /* End of if - sanity check */
  return;
} /* End of UnsetImageContext */


/* -----------------------------------------------------------------------
   Name - UnsetClipContext
   Purpose -
   Parameters -
   Returns -
   Side Effects -
   ----------------------------------------------------------------------- */
void UnsetClipContext(wcontext *wc)
{
  int i, num;
  wstate *ws;

  if (wc) {
    num = wc->numDeps;
    for (i = 0; i < wc->maxDeps && num > 0; i++)
      if (ws = wc->depWindows[i]) {
        num--;
        if (ws->clipContext == wc) ws->clipContext = NULL;
        } /* End of if - found one */
     } /* End of if - sanity check */
  return;
} /* End of UnsetClipContext */


/* -----------------------------------------------------------------------
   Name - SetImageContext
   Purpose -
   Parameters -
   Returns -
   Side Effects -
   ----------------------------------------------------------------------- */
void SetImageContext(wbinding *wb, wstate *ws, wcontext *wc)
{
  /* have to always make sure our clip is loaded */
  SetClipContext(wb, ws, wc);
  if (ws->imageContext != wc) {
    /* have to set the attributes from the line bundle in wc */
    if (ws->hpsWin)
      GpiSetAttrs(ws->hpsWin, PRIM_IMAGE, imageAttrs, 0UL, &(wc->imageBundle));
    GpiSetAttrs(ws->hpsBitmap, PRIM_IMAGE, imageAttrs, 0UL, &(wc->imageBundle));
    ws->imageContext = wc;
    } /* End of if - check the context out */
  return;
} /* End of SetImageContext */


/* -----------------------------------------------------------------------
   Name - SetLineContext
   Purpose -
   Parameters -
   Returns -
   Side Effects -
   ----------------------------------------------------------------------- */
void SetLineContext(wbinding *wb, wstate *ws, wcontext *wc)
{
  SetClipContext(wb, ws, wc);
  if (ws->lineContext != wc) {
    /* have to set the attributes from the line bundle in wc */
    if (ws->hpsWin)
      GpiSetAttrs(ws->hpsWin, PRIM_LINE, lineAttrs, 0UL, &(wc->lineBundle));
    GpiSetAttrs(ws->hpsBitmap, PRIM_LINE, lineAttrs, 0UL, &(wc->lineBundle));
    ws->lineContext = wc;
    } /* End of if - check the context out */
  return;
} /* End of SetLineContext */


/* -----------------------------------------------------------------------
   Name - SetCharContext
   Purpose -
   Parameters -
   Returns -
   Side Effects -
   ----------------------------------------------------------------------- */
void SetCharContext(wbinding *wb, wstate *ws, wcontext *wc)
{
  SetClipContext(wb, ws, wc); 
  if (ws->charContext != wc) {
    /* have to set the attributes from the char bundle in wc */
    if (ws->hpsWin)
      GpiSetAttrs(ws->hpsWin, PRIM_CHAR, charAttrs, 0UL, &(wc->charBundle));
    GpiSetAttrs(ws->hpsBitmap, PRIM_CHAR, charAttrs, 0UL, &(wc->charBundle));
    /* set the pointer to the context */
    ws->charContext = wc;
    /* set the cursor info */
    UpdateCursorConfig(ws, wc);
    UpdateCursorPos(ws, wc);
    /* set the font descender value used when the window is resized */
    ws->lastDescender = wc->font->metrics.lMaxDescender;
    } /* End of if - check the context out */
  return;
} /* End of SetCharContext */


/* -----------------------------------------------------------------------
   Name - SetAreaContext
   Purpose -
   Parameters -
   Returns -
   Side Effects -
   ----------------------------------------------------------------------- */
void SetAreaContext(wbinding *wb, wstate *ws, wcontext *wc)
{
  SetClipContext(wb, ws, wc);
  if (ws->areaContext != wc) {
    if (ws->hpsWin)
      GpiSetAttrs(ws->hpsWin, PRIM_AREA, areaAttrs, 0UL, &(wc->areaBundle));
    GpiSetAttrs(ws->hpsBitmap, PRIM_AREA, areaAttrs, 0UL, &(wc->areaBundle));
    ws->areaContext = wc;
    } /* End of if - check the context out */
  return;
} /* End of SetAreaContext */


/* -----------------------------------------------------------------------
   Name - SetClipContext
   Purpose - 
   Parameters -
   Returns -
   Side Effects -
   ----------------------------------------------------------------------- */
void SetClipContext(wbinding *wb, wstate *ws, wcontext *wc)
{
  HRGN hscrap;
  RECTL rect;
  int diff;

  if (ws->clipContext != wc) {
    ws->hClipBitmap = ws->hClipWindow = NULLHANDLE;
    /* if clip is even set */
    if (wc->cliph || wc->clipw || wc->clipx || wc->clipy) {
      rect.xLeft = wc->clipx;
      rect.xRight = (wc->clipw) ? wc->clipx + wc->clipw : ws->width;
      rect.yTop = ws->bitHeight - wc->clipy;
      rect.yBottom = (wc->cliph) ? rect.yTop - wc->cliph : 0;
      ws->hClipBitmap = GpiCreateRegion(ws->hpsBitmap, 1, &rect);
      if (ws->hpsWin) {
        diff = ws->bitHeight - ws->height;
        rect.yTop -= diff;
        rect.yBottom -= diff;
        ws->hClipWindow = GpiCreateRegion(ws->hpsWin, 1, &rect);
        } /* End of if - window too */
      } /* End of if - new context is clipping */
    GpiSetClipRegion(ws->hpsBitmap, ws->hClipBitmap, &hscrap);
    GpiDestroyRegion(ws->hpsBitmap, hscrap);
    if (ws->hpsWin) {
      GpiSetClipRegion(ws->hpsWin, ws->hClipWindow, &hscrap);
      GpiDestroyRegion(ws->hpsWin, hscrap);
      } /* End of if - window there */
    /* we are loaded now */
    ws->clipContext = wc;
    } /* End of if - this context not already loaded */

  return;
} /* End of SetClipContext */


/* -------------------------- Dependancies ------------------------------- */
/* -----------------------------------------------------------------------
   Name - AddWindowDep
   Purpose - adds a window dependency to a context
   Parameters -
   Returns -
   Side Effects -
   ----------------------------------------------------------------------- */
int AddWindowDep(wstate *ws, wcontext *wc)
{
  int i, first = -1;

  /* make sure dependency does not already exist */
  for (i = 0; i < wc->maxDeps; i++) {
    /* already there */
    if (wc->depWindows[i] == ws) return 1;
    /* first open slot */
    if (first < 0 && !wc->depWindows[i]) first = i;
    } /* End of for - look for the dependency and find a free slot */
  /* see if we have a free slot */
  if (first < 0) {
    ULONG newsize;
    wstate **newtbl;
    newsize = (wc->maxDeps << 1) * sizeof(wstate *);
    /* blow the table up a little bigger, if possible */
    if (!(newtbl = realloc(wc->depWindows, newsize))) return 0;
    /* zero out the upper pointers */
    memset(&newtbl[wc->maxDeps], 0, sizeof(wstate *) * wc->maxDeps);
    wc->depWindows = newtbl;
    first = wc->maxDeps;
    wc->maxDeps <<= 1;
    } /* End of if - no free slots, make more */
  wc->depWindows[first] = ws;
  wc->numDeps++;
  return 1;
} /* End of AddWindowDep */


/* -----------------------------------------------------------------------
   Name - AddContextDep
   Purpose -
   Parameters -
   Returns -
   Side Effects -
   ----------------------------------------------------------------------- */
int AddContextDep(wstate *ws, wcontext *wc)
{
  int i, first = -1;

  /* make sure dependency does not already exist */
  for (i = 0; i < ws->maxDeps; i++) {
    if (ws->depContexts[i] == wc) return 1;
    if (first < 0 && !ws->depContexts[i]) first = i;
    } /* End of for - look for the dependency and find a free slot */
  /* wasn't already mapped ... */
  /* see if we have a free slot */
  if (first < 0) {
    ULONG newsize;
    wcontext **newtbl;
    newsize = (ws->maxDeps << 1) * sizeof(wcontext *);
    /* blow the table up a little bigger, if possible */
    if (!(newtbl = realloc(ws->depContexts, newsize))) return 0;
    /* zero out the upper pointers */
    memset(&newtbl[ws->maxDeps], 0, sizeof(wcontext *) * ws->maxDeps);
    ws->depContexts = newtbl;
    first = ws->maxDeps;
    ws->maxDeps <<= 1;
    } /* End of if - no free slots, make more */
  /* fill in the slot */
  ws->depContexts[first] = wc;
  ws->numDeps++;
  return 1;
} /* End of AddContextDep */


/* ------------------------ Local ID Functions --------------------------- */
/* -----------------------------------------------------------------------
   Name - InitializeIdTable
   Purpose -
   Parameters -
   Returns - 
   Side Effects -
   ----------------------------------------------------------------------- */
void InitializeIdTable()
{
  ULONG i;

  /* allocate the space for the table */
  LocalIds = (lclIdentifier *)calloc(sizeof(lclIdentifier), MAXLOCALS);
  /* link all the entries together on the free list */
  FreeIdEntries = LocalIds;
  for (i = 0; i < MAXLOCALS - 1; i++) {
    LocalIds[i].next = &LocalIds[i + 1];
    LocalIds[i + 1].previous = &LocalIds[i];
    } /* End of for - link them all up */
  /* initilize font and pattern list pointers */
  FontIdEntries = PatIdEntries = NULL;
  return;
} /* End of InitializeIdTable */


/* -----------------------------------------------------------------------
   Name - SetNewPattern
   Purpose -
   Parameters -
   Returns - id of the newly created pattern or -1 on error
   Side Effects -
   ----------------------------------------------------------------------- */
int SetNewBitPattern(wcontext *wc, PBYTE bits)
{
  lclIdentifier *ptr;
  int i;
  SHORT id;
  wstate *ws;
  HBITMAP hbm;
  HDC hdcMem;
  HPS hpsMem;
  SIZEL size = {0, 0};
  BITMAPINFOHEADER2 bmp;
  PBITMAPINFO2 pbmi;
  PRGB2 prgb;

/* XXX for now, each pattern will be separate */
  /* try to find it first */

  /* grab a new local id */
  if (ptr = AllocateLocalID()) {
    /* initialize some data */
    ptr->idtype = IS_PATTERN;
    /* if there was a different stipple before destroy it */
    ReleasePattern(wc->currPattern);

    /* make the bitmap for the pattern */
    /* grab a memory DC */
    hdcMem = DevOpenDC(HInterpAnchorBlock, OD_MEMORY, "*", 0, NULL,
                       NULLHANDLE);
    /* make a presentation space */
    hpsMem = GpiCreatePS(HInterpAnchorBlock, hdcMem, &size, GPIT_MICRO |
                         GPIA_ASSOC | PU_PELS);
    /* initialize the bitmap structure */
    memset(&bmp, 0, sizeof(BITMAPINFOHEADER2));
    bmp.cbFix = sizeof(BITMAPINFOHEADER2);
    bmp.cPlanes = 1;
    bmp.cBitCount = 1;
    bmp.cx = bmp.cy = 8;

    pbmi = (PBITMAPINFO2)calloc(1, sizeof(BITMAPINFO2) + (sizeof(RGB2) << 1));
    pbmi->cbFix = bmp.cbFix; 
    pbmi->cx = pbmi->cy = 8;
    pbmi->cPlanes = 1;
    pbmi->cBitCount = 1;
    /* set up the color info */
    prgb = (PRGB2)(pbmi->argbColor);
    prgb[1].bBlue = prgb[1].bRed = prgb[1].bGreen = 255;  /* white */

    /* build the bitmap */
    hbm = GpiCreateBitmap(hpsMem, &bmp, CBM_INIT, (PBYTE)bits, pbmi);
    /* stor the handle to the pattern bitmap */
    ptr->u.hpat = hbm;
    /* shouldn't need this anymore ... */
    free(pbmi);

    /* dump the hps */
    GpiAssociate(hpsMem, NULLHANDLE);
    GpiDestroyPS(hpsMem);
    /* get rid of the dc */
    DevCloseDC(hdcMem);
 
    /* set the current stipple pattern */
    id = ((LONG)ptr - (LONG)LocalIds) / sizeof(lclIdentifier) + 1;
    wc->currPattern = id;
    /* if we are currently 'stippled', update the area bundle */
    if (!(wc->fillstyle & FS_SOLID)) {
      wc->areaBundle.usSet = id;
      wc->areaBundle.usSymbol = PATSYM_DEFAULT;
      /* unset the newly update area context */
      UnsetAreaContext(wc);  
      } /* End of if - currently stippled */
    /* update all the PSes */
    RippleLocalIdAddition(wc, id);
    /* add the pattern to the pattern list */
    ptr->next = PatIdEntries;
    if (PatIdEntries) PatIdEntries->previous = ptr;
    ptr->previous = NULL;
    PatIdEntries = ptr;
    return 1;
    } /* End of if - obtained a new local id */
  return 0;
} /* End of SetNewBitPattern */



/* -----------------------------------------------------------------------
   Name - SetPattern
   Purpose -
   Parameters -
   Returns -
   Side Effects -
   ----------------------------------------------------------------------- */
int SetPattern(wcontext *wc, char *name, int len)
{
  char buf[64];
  int symbol;

  /* find the symbol id */
  mystrncpy(buf, name, len);
  if ((symbol = si_s2i(siPatternSyms, buf)) >= 0) {
    /* release the previous pattern */
    ReleasePattern(wc->currPattern);
    wc->currPattern = -symbol;
    /* if we are currently stippled, change the bundle */
    if (!(wc->fillstyle & FS_SOLID)) {
      wc->areaBundle.usSet = LCID_DEFAULT;
      wc->areaBundle.usSymbol = symbol;
      UnsetAreaContext(wc);
      } /* End of if - had to set bundle */
    return 1;
    } /* End of if - symbol name found */
  return 0;
} /* End of SetPattern */


/* -----------------------------------------------------------------------
   Name - AddLocalIdToWindow
   Purpose -
   Parameters - assume id is presentation space relative
   Returns -
   Side Effects -
   ----------------------------------------------------------------------- */
void AddLocalIdToWindow(wstate *ws, LONG id)
{
  LONG newid;
  lclIdentifier *ptr;
  FATTRS fat;
  LONG ret;

  newid = id - 1;
  if (newid >= 0 && newid < MAXLOCALS) {
    MutexOn(ws);
    ptr = &LocalIds[newid];
    if (ptr->idtype & IS_PATTERN) {
      GpiSetBitmapId(ws->hpsBitmap, ptr->u.hpat, id);
      if (ws->hpsWin) 
        GpiSetBitmapId(ws->hpsWin, ptr->u.hpat, id);
      } /* End of if - is a pattern */
    else {
      memset(&fat, 0, sizeof(FATTRS));
      fat.usRecordLength = sizeof(FATTRS);
      fat.lMaxBaselineExt = ptr->u.font.metrics.lMaxBaselineExt;
      fat.lAveCharWidth = ptr->u.font.metrics.lAveCharWidth;
      strcpy(fat.szFacename, ptr->u.font.metrics.szFacename);
      fat.lMatch = ptr->u.font.metrics.lMatch;
      ret = GpiCreateLogFont(ws->hpsBitmap, NULL, id, &fat);
      if (ws->hpsWin)
        ret = GpiCreateLogFont(ws->hpsWin, NULL, id, &fat);
      } /* End of else - is a font */
    MutexOff(ws);
    } /* End of if - id valid */
  return;
} /* End of AddLocaIdToWindow */


/* -----------------------------------------------------------------------
   Name - RippleLocalIdAddition
   Purpose - 
   Parameters - assume id is presentation space relative id and valid
   Returns -
   Side Effects -
   ----------------------------------------------------------------------- */
static void RippleLocalIdAddition(wcontext *wc, LONG id)
{
  wstate *ws;
  int i;

  for (i = 0; i < wc->maxDeps; i++)
    if (ws = wc->depWindows[i]) 
      AddLocalIdToWindow(ws, id);
  return;
} /* End of RippleLocalIdAddition */


/* -----------------------------------------------------------------------
   Name - RippleLocalIdRemoval
   Purpose -
   Parameters - assumes id is index with respect to presentation space
   Returns -
   Side Effects -
   ----------------------------------------------------------------------- */
static void RippleLocalIdRemoval(LONG id)
{
  wstate *ws;

  for (ws = wstates; ws; ws = ws->next) {
    MutexOn(ws);
    if (ws->hpsWin)
      GpiDeleteSetId(ws->hpsWin, id);
    GpiDeleteSetId(ws->hpsBitmap, id);
    MutexOff(ws);
    } /* End of for - go through all window states */
  return;
} /* End of RippleLocalIdRemoval */



/* -----------------------------------------------------------------------
   Name - ReleaseLocalId
   Purpose - like a lazy release
   Parameters - id is presentation space relative
   Returns -
   Side Effects -
   ----------------------------------------------------------------------- */
void ReleaseLocalId(LONG id)
{
  LONG newid;
  lclIdentifier *ptr;

  newid = id - 1;
  if (newid >= 0 && newid < MAXLOCALS && --LocalIds[newid].refcount < 1) {
    /* go change all the PSes */
    RippleLocalIdRemoval(id);
    /* quick access */
    ptr = &LocalIds[newid];
    /* free info  and begin removal from list */
    if (ptr->idtype & IS_PATTERN) {
      GpiDeleteBitmap(ptr->u.hpat);
      if (ptr->previous == NULL) PatIdEntries = ptr->next;
      } /* End of if - was a pattern */
    else if (ptr->idtype & IS_FONT)
      if (ptr->previous == NULL) FontIdEntries = ptr->next;
    /* finish removal */
    if (ptr->previous) ptr->previous->next = ptr->next;
    if (ptr->next) ptr->next->previous = ptr->previous;
    /* put on the free list */
    if (FreeIdEntries) FreeIdEntries->previous = ptr;
    ptr->next = FreeIdEntries;
    FreeIdEntries = ptr;
    ptr->previous = NULL;
    } /* End of if - id within range and element needs to be removed */
  return;
} /* End of ReleaseLocalId */


/* -----------------------------------------------------------------------
   Name - AddLocalIdDependant
   Purpose - 
   Parameters -
   Returns -
   Side Effects -
   ----------------------------------------------------------------------- */
void AddLocalIdDependant(LONG id)
{
  LONG newid;

  /* we don't map 0 cause that is LCID_DEFAULT */
  newid = id - 1;
  if (newid >= 0 && newid < MAXLOCALS) LocalIds[newid].refcount++;
  return;
} /* End of AddLocalIdDependant */


/* -----------------------------------------------------------------------
   Name - GetFontName
   Purpose -
   Parameters - the index of the font, PS relative
   Returns - 
   Side Effects -
   ----------------------------------------------------------------------- */
int GetFontName(LONG id, char *buf, int maxlen)
{
  LONG newid;
  lclIdentifier *ptr;

  /* sanity check */
  if ((newid = id - 1) >= 0 && newid < MAXLOCALS &&
      (LocalIds[newid].idtype & IS_FONT)) {
    ptr = &LocalIds[newid];
/* XXX notice, we are ignoring the maxlen here - oooh dangerous :) */
/* XXX may have to get the 'long' facename by following the facename atom */
/* XXX what if we are simulating italic or bold? - may need flags for that */
    sprintf(buf, "%s %d", ptr->u.font.metrics.szFacename,
            ptr->u.font.metrics.lEmHeight);
    return strlen(buf);
    } /* End of if - should be valid */
  /* error */
  return 0;
} /* End of GetFontName */


/* -----------------------------------------------------------------------
   Name - SetFont
   Purpose -
   Parameters -
   Returns - the index of the font, PS relative
   Side Effects -
   ----------------------------------------------------------------------- */
int SetFont(wcontext *wc, char *family, LONG attrs, LONG fsize)
{
  lclIdentifier *ptr;
  LONG newid;
  SIZEL size = { 0, 0};
  SIZEF sizef;
  HPS hps;
  HDC hdc;
  char name1[128];
  char name2[128];
  char attstr[128];
  FONTMETRICS afm[20];
  FATTRS fat;
  LONG cfonts;
  LONG i;

  /* break out the selection attributes */
  attstr[0] = '\0';
  if (attrs & FATTR_SEL_BOLD) strcat(attstr, " Bold");
  if (attrs & FATTR_SEL_ITALIC) strcat(attstr, " Italic");
  /* build the name to search for */
  sprintf(name1, "%s%s %d", family, attstr, fsize);
  sizef.cx = MAKEFIXED(fsize, 0);
  sizef.cy = MAKEFIXED(fsize, 0);
  /* search through font entries.. */
  for (ptr = FontIdEntries; ptr; ptr = ptr->next) {
    newid = ((LONG)ptr - (LONG)LocalIds) / sizeof(lclIdentifier) + 1;
    GetFontName(newid, name2, 128);
    if (!strcmp(name1, name2)) break;
    } /* End of for - try to find matching name */
  /* if we didn't find the font, make a new one, else increment refcount */
  if (ptr)
    AddFontDependant(newid);
  else {
    /* open a memory DC compat with the screen */
    hdc = DevOpenDC(HInterpAnchorBlock, OD_MEMORY, "*", 0, NULL, NULLHANDLE);
    /* create a PS and associate with DC */
    hps = GpiCreatePS(HInterpAnchorBlock, hdc, &size,
                      GPIT_MICRO | GPIA_ASSOC | PU_PELS);

    /* try to create an outline font first */
    /* build the font request */
    memset(&fat, 0, sizeof(FATTRS));
    fat.usRecordLength = sizeof(FATTRS);
    fat.fsFontUse = FATTR_FONTUSE_OUTLINE | FATTR_FONTUSE_TRANSFORMABLE;
    /* make the facename to look for */
    sprintf(fat.szFacename, "%s%s", family, attstr);
    /* request the font */
    if (GpiCreateLogFont(hps, NULL, 1, &fat) != FONT_MATCH) {
      /* couldn't get an outline font, try for an image */
      cfonts = 20;  /* maximum fonts to get info on */
      GpiQueryFonts(hps, QF_PUBLIC, family, &cfonts,
                    sizeof(FONTMETRICS), afm);
      /* try to match one with the exact face name */
      for (i = 0; i < cfonts && (afm[i].sXDeviceRes != FontResX ||
           afm[i].sYDeviceRes != FontResY || afm[i].lEmHeight < fsize ||
           !strcmp(afm[i].szFacename, fat.szFacename)); i++);
      /* see if we got one */
      if (i >= cfonts) {
        /* ok, didn't find one, this time search for just the res and
           the size -> hoping that the attribs messed us up - we can
           fake those in an image font */
        for (i = 0; i < cfonts && (afm[i].sXDeviceRes != FontResX ||
             afm[i].sYDeviceRes != FontResY || afm[i].lEmHeight < fsize); i++);
        /* couldn't find an appropriate font - sorry charlie */
        if (i >= cfonts) return 0;
        /* else - we made it */
        fat.fsSelection = attrs;
        } /* End of if - failed on that try */

      fat.lMaxBaselineExt = afm[i].lMaxBaselineExt;
      fat.lAveCharWidth = afm[i].lAveCharWidth;
      fat.fsFontUse = 0;
      strcpy(fat.szFacename, afm[i].szFacename);
      fat.lMatch = afm[i].lMatch;
      /* try once again to make the font */
      if (GpiCreateLogFont(hps, NULL, 1, &fat) != FONT_MATCH) return 0;
      } /* End of else - failed on finding outline font */

    /* grab a new font node */
    if (!(ptr = AllocateLocalID())) return 0;
    ptr->idtype = IS_FONT;
    newid = ((LONG)ptr - (LONG)LocalIds) / sizeof(lclIdentifier) + 1;

    /* set the box on the presentation space - in case we have an outline */
    GpiSetCharBox(hps, &sizef);
    /* set the new set */
    GpiSetCharSet(hps, 1);
    GpiQueryFontMetrics(hps, sizeof(FONTMETRICS), &(ptr->u.font.metrics));
    /* disassocitate the scratch PS and destory it */
    GpiAssociate(hps, NULLHANDLE);
    GpiDestroyPS(hps);
    /* release the DC */
    DevCloseDC(hdc);
    /* add this font to the font list */
    ptr->next = FontIdEntries;
    if (FontIdEntries) FontIdEntries->previous = ptr;
    ptr->previous = NULL;
    FontIdEntries = ptr;
    } /* End of if - have to get a new font */

  /* before we release the font, subtract out the leading */
  if (wc->font) wc->fntLeading -= wc->font->metrics.lExternalLeading;
  /* release the previous font */
  ReleaseFont(wc->charBundle.usSet);
  /* set the cell in case we got an outline font */
  wc->charBundle.sizfxCell = sizef;
  /* the new character set */
  wc->charBundle.usSet = newid;
  /* set the new font */
  wc->font = &(ptr->u.font);
  /* our new leading */
  wc->fntLeading += wc->font->metrics.lExternalLeading;
  /* ripple the change -> put this log font on all our wstate PSes */
  RippleLocalIdAddition(wc, newid);
  /* unset us from everyone */
  UnsetCharContext(wc);
  return 1;
} /* End of SetFont */


/* -----------------------------------------------------------------------
   Name - ParseFontSpec
   Purpose -
   Parameters -
   Returns -
   Side Effects -
   ----------------------------------------------------------------------- */
int ParseFontSpec(char *rawdata, char *family, LONG *flags, LONG *size)
{
  char *sptr, *eptr;
  char savech;
  LONG len;

  /* set up the defaults */
  strcpy(family, "System Monospaced");
  *flags = 0;
  *size = 10;
  /* now, scan through the raw and break out pieces */
  sptr = eptr = rawdata;
  while (isspace(*sptr) || *sptr == ',') sptr++;
  while (*sptr) {
    eptr = sptr;
    if (isdigit(*eptr)) {
      while (isdigit(*eptr)) eptr++;
      savech = *eptr;
      *eptr = '\0';
      *size = atoi(sptr);
      *eptr = savech;
      } /* End of if - start of size */
    else if (isalpha(*eptr)) {
      while (isspace(*eptr) || isalnum(*eptr)) eptr++;
      while (isspace(*(eptr - 1)) && eptr != sptr) eptr--;
      len = (LONG)eptr - (LONG)sptr;
      if (!strnicmp("italic", sptr, len))
        *flags |= FATTR_SEL_ITALIC;
      else if (!strnicmp("bold", sptr, len))
        *flags |= FATTR_SEL_BOLD;
      else if (!strnicmp("normal", sptr, len))
        *flags = 0;
      else
        mystrncpy(family, sptr, (LONG)eptr - (LONG)sptr);
      }
    else return 0;
    while (isspace(*eptr) || *eptr == ',') eptr++;
    sptr = eptr;
    } /* End of while */
  /* expand out family name aliases */
  if (!strcmp(family, "fixed"))
    strcpy(family, "System Monospaced");
  else if (!strcmp(family, "times"))
    strcpy(family, "Times New Roman");
  else if (!strcmp(family, "helv"))
    strcpy(family, "Helvetica");
  else if (!strcpy(family, "proportional"))
    strcpy(family, "System Proportional");
  return 1;
} /* End of ParseFontSpec */



/* -----------------------------------------------------------------------
   Name - AllocateLocalID
   Purpose -
   Parameters -
   Returns - 
   Side Effects -
   ----------------------------------------------------------------------- */
static lclIdentifier *AllocateLocalID()
{
  lclIdentifier *tmp;

  if (tmp = FreeIdEntries){
    FreeIdEntries = tmp->next;
    if (FreeIdEntries) FreeIdEntries->previous = NULL;
    memset(tmp, 0, sizeof(lclIdentifier));
    tmp->refcount = 1;
    } /* End of if - was a free node */
  return tmp;
} /* End of AllocateLocalID */


/* ---------------------- Color Table Functions -------------------------- */
/* -----------------------------------------------------------------------
   Name - InitializeColorTable
   Purpose -
   Parameters -
   Returns -
   Side Effects -
   ----------------------------------------------------------------------- */
void InitializeColorTable()
{
  ULONG i;

  /* grab space for the color table (minus the default colors) */
  ColorTable = (colorEntry *)calloc(sizeof(colorEntry), MaxPSColors);
  /* fill in the default colors (system colors) */
  UsedCEntries = &ColorTable[0];
  for (i = 0; i < 15; i++) {
    /* lock the color so it cannot be release */
    ColorTable[i].bits = CLR_LOCKED | CLR_USED | CLR_BASE;
    /* set the color data */
    ColorTable[i].rgb = defClrTable[i];
    /* link to next color */
    ColorTable[i].next = &ColorTable[i + 1];
    ColorTable[i + 1].previous = &ColorTable[i];
    } /* End of for - link all the default colors together */
  /* finish up color 16 */
  ColorTable[15].bits = CLR_LOCKED | CLR_USED | CLR_BASE;
  ColorTable[15].rgb = defClrTable[15];
  /* link all the remaining entries together for the free list */
  FreeCEntries = &ColorTable[16];
  for (i = 16; i < MaxPSColors - 1; i++) {
    ColorTable[i].next = &ColorTable[i + 1];
    ColorTable[i + 1].previous = &ColorTable[i];
    } /* End of for - link them all up */
  return;
} /* End of InitializeColorTable */


/* -----------------------------------------------------------------------
   Name - GetRGBColorIndex
   Purpose -
   Parameters -
   Returns -
   Side Effects -
   ----------------------------------------------------------------------- */
LONG GetRGBColorIndex(ULONG rgb)
{
  colorEntry *ce;
  LONG indx;

  /* let's see if we can find it already allocated and placed */
  for (ce = UsedCEntries; ce; ce = ce->next)
    if (ce->rgb == rgb) {
       ce->refcount++;
       return ((ULONG)ce - (ULONG)ColorTable) / sizeof(colorEntry);
       } /* End of if - found the match */
  /* must not have found it..so allocate a new color and ripple effect */
  if (FreeCEntries) {
    /* remove the node from the list */
    ce = FreeCEntries;
    FreeCEntries = ce->next;
    if (FreeCEntries) FreeCEntries->previous = NULL;
    /* put it in the used list */
    ce->next = UsedCEntries;
    if (UsedCEntries) UsedCEntries->previous = ce;
    UsedCEntries = ce;
    /* fill the entry in */
    ce->refcount = 1;
    ce->rgb = rgb;
    ce->bits = CLR_USED;
    /* update all the presentation spaces */
    indx = ((ULONG)ce - (ULONG)ColorTable) / sizeof(colorEntry);
    RippleColorAddition(indx);
    return indx;
    } /* End of if - a free entry was available */
  return -1;
} /* End of GetRGBColorIndex */


/* -----------------------------------------------------------------------
   Name - RippleColorAddition
   Purpose - adds a color to all presentation spaces
   Parameters - the index of the color to 'ripple'
   Returns -
   Side Effects -
   ----------------------------------------------------------------------- */
static void RippleColorAddition(LONG indx)
{
  wstate *ws;
  LONG map[2];

  map[0] = indx;
  map[1] = (LONG)ColorTable[indx].rgb;
  for (ws = wstates; ws; ws = ws->next) {
    MutexOn(ws);
    /* add the rgb-index mapping (color entry) to each PS */
    if (ws->hpsWin)
      GpiCreateLogColorTable(ws->hpsWin, 0, LCOLF_INDRGB, 0, 2, map);
    GpiCreateLogColorTable(ws->hpsBitmap, 0, LCOLF_INDRGB, 0, 2, map);
    MutexOff(ws);
    } /* End of for - go through all window states */
  return;
} /* End of RipppleColorAddition*/


/* -----------------------------------------------------------------------
   Name - AddColorDependant
   Purpose - 
   Parameters -
   Returns -
   Side Effects -
   ----------------------------------------------------------------------- */
void AddColorDependant(LONG indx)
{
  if (indx >= 0 && indx < MaxPSColors && (ColorTable[indx].bits & CLR_USED))
    ColorTable[indx].refcount++;
  return;
} /* End of AddColorDependant */


/* -----------------------------------------------------------------------
   Name - ReleaseColor
   Purpose - decrements the reference count of the color, if last one,
             the color entry is returned to the free pool
   Parameters -
   Returns -
   Side Effects -
   ----------------------------------------------------------------------- */
void ReleaseColor(LONG indx)
{
  colorEntry *ptr;

  if (indx >= 0 && indx < MaxPSColors && (ColorTable[indx].bits & CLR_USED) &&
      !(ColorTable[indx].bits & CLR_LOCKED) && --ColorTable[indx].refcount < 1) {
    ptr = &ColorTable[indx];
    /* knock out bits and count */
    ptr->bits = ptr->refcount = 0;
    /* remove from used list */
    if (ptr->next) ptr->next->previous = ptr->previous;
    if (ptr->previous) ptr->previous->next = ptr->next;
    else UsedCEntries = ptr->next;
    /* place on the free list */
    if (FreeCEntries) FreeCEntries->previous = ptr;
    ptr->next = FreeCEntries;
    ptr->previous = NULL;
    FreeCEntries = ptr;
    } /* End of if - last reference removed */
  return;
} /* End of ReleaseColor */


/* -----------------------------------------------------------------------
   Name - ColorInitPS
   Purpose - initializes a new wstate will all the colors in the global
             'rainbow'
   Parameters -
   Returns -
   Side Effects -
   ----------------------------------------------------------------------- */
void ColorInitPS(wbinding *wb)
{
  wstate *ws = wb->window;
  colorEntry *ce;
  LONG map[2];

  /* load the base color tables */
  if (ws->hpsWin)
    GpiCreateLogColorTable(ws->hpsWin, 0, LCOLF_CONSECRGB, 0, 16, defClrTable);
  GpiCreateLogColorTable(ws->hpsBitmap, 0, LCOLF_CONSECRGB, 0, 16, defClrTable);
  for (ce = UsedCEntries; ce; ce = ce->next) {
    if (!(ce->bits & CLR_BASE)) {
      /* figure out next entry */
      map[0] = ((ULONG)ce - (ULONG)ColorTable) / sizeof(colorEntry);
      map[1] = (LONG)ce->rgb;
      /* add the rgb-index mapping (color entry) to PS */
      if (ws->hpsWin)
        GpiCreateLogColorTable(ws->hpsWin, 0, LCOLF_INDRGB, 0, 2, map);
      GpiCreateLogColorTable(ws->hpsBitmap, 0, LCOLF_INDRGB, 0, 2, map);
      } /* End of if - not a base color */
    } /* End of for - go through all color entries */
  return;
} /* End of ColorInitPS */



/* -----------------------------------------------------------------------
   Name - GetColorName
   Purpose - 
   Parameters -
   Returns -
   Side Effects -
   ----------------------------------------------------------------------- */
void GetColorName(LONG indx, char *buf, int len)
{
  char *ptr;
  PRGB2 prgb;

  if (indx >= 0 && indx < MaxPSColors && (ColorTable[indx].bits & CLR_USED)) {
    if (!(ptr = si_i2s(siColorNames, ColorTable[indx].rgb)))  {
      prgb = (PRGB2)&ColorTable[indx].rgb;
      sprintf(buf, "rgb:%d,%d,%d", prgb->bRed, prgb->bGreen, prgb->bBlue);
      } /* End of if - rgb color */
    else mystrncpy(buf, ptr, len);
    } /* End of if - color is allocated */
  else strcpy(buf, "color not allocated");
  return;
} /* End of GetColorName */


/* -----------------------------------------------------------------------
   Name - ParseColor
   Purpose -
   Parameters -
   Returns -
   Side Effects -
   ----------------------------------------------------------------------- */
LONG ParseRGBValue(char *buf)
{
  LONG rgb = -1;
  int len, sublen;
  int r, g, b;
  char save;

  r = g = b = 0;
  while (isspace(*buf)) buf++;
  if (*buf == '#') {
    buf++;
    for (len = 0; isalnum(buf[len]); len++);
    buf[len] = '\0';
    if (len % 3 != 0) return -1;
    sublen = len / 3;
    save = buf[sublen];
    buf[sublen] = '\0';
    if (!sscanf(buf, "%x", &r)) return -1;
    buf += sublen;
    *buf = save;
    save = buf[sublen];
    buf[sublen] = '\0';
    if (!sscanf(buf, "%x", &g)) return -1;
    buf += sublen;
    *buf = save;
    if (!sscanf(buf, "%x", &b)) return -1;
    } /* End if - #rgb format */
  else if (sscanf(buf, "%d,%d,%d", &r, &g, &b) != 3)
    return -1;
  RGB16TO8(r); RGB16TO8(g); RGB16TO8(b);
  return MAKERGB(r, g, b);
} /* End of ParseColor */


/* -----------------------------------------------------------------------
   Name - ParseRGBValues
   Purpose -
   Parameters -
   Returns -
   Side Effects -
   ----------------------------------------------------------------------- */
void ParseRGBValues(char *buf, int *r, int *g, int *b)
{
  *r = *g = *b = 0;
  sscanf(buf, "%d, %d, %d", r, g, b);
  return;
} /* End of ParseRGBValues */


/* -----------------------------------------------------------------------
   Name - EnsureColorAvailable
   Purpose -
   Parameters -
   Returns -
   Side Effects -
   ----------------------------------------------------------------------- */
void EnsureColorAvailable(LONG indx)
{
  wstate *ws;
  LONG map[2];

  if (!(ColorTable[indx].bits & CLR_USED)) {
    map[0] = indx;
    /* make up some color to put in the slot */
    map[1] = MAKERGB(0, 0, 0);
    for (ws = wstates; ws; ws = ws->next) {
      MutexOn(ws);
      /* add the rgb-index mapping (color entry) to each PS */
      if (ws->hpsWin)
        GpiCreateLogColorTable(ws->hpsWin, 0, LCOLF_INDRGB, 0, 2, map);
      GpiCreateLogColorTable(ws->hpsBitmap, 0, LCOLF_INDRGB, 0, 2, map);
      MutexOff(ws);
      } /* End of for - go through all window states */
     } /* End of if - not already there */
  return;
} /* End of EnsureColorAvailable */


/* ------------------------ Cursor Functions ----------------------------- */
/* -----------------------------------------------------------------------
   Name - UpdateCursorConfig
   Purpose -
   Parameters -
   Returns -
   Side Effects -
   ----------------------------------------------------------------------- */
void UpdateCursorConfig(wstate *ws, wcontext *wc)
{
  PCURSORINFO ptr;

  if (ws->hpsWin) {
    ptr = &(ws->cursInfo);
    ptr->cx = wc->font->metrics.lAveCharWidth;
    if (ISCURSORONW(ws) && ws->hwnd == WinQueryFocus(HWND_DESKTOP))
      WinCreateCursor(ws->hwnd, ptr->x, ptr->y, ptr->cx, ptr->cy,
                      CURSOR_FLASH, NULL);
    }
  return;
}

/* -----------------------------------------------------------------------
   Name - UpdateCursorPos
   Purpose -
   Parameters -
   Returns -
   Side Effects -
   ----------------------------------------------------------------------- */
void UpdateCursorPos(wstate *ws, wcontext *wc)
{
  PCURSORINFO ptr;

  if (ws->hpsWin) {
    ptr = &(ws->cursInfo);
    ptr->x = ws->x + wc->dx;
    ptr->y = ws->height - (ws->y + wc->dy);
    if (ISCURSORONW(ws) && ws->hwnd == WinQueryFocus(HWND_DESKTOP))
      WinCreateCursor(ws->hwnd, ptr->x, ptr->y, 0, 0, CURSOR_SETPOS, NULL);
    }
  return;
}


/* ------------------------- Console Functions -------------------------- */
/* -----------------------------------------------------------------------
   Name - OpenConsole
   Purpose -
   Parameters -
   Returns -
   Side Effects -
   ----------------------------------------------------------------------- */
FILE *OpenConsole()
{
  struct descrip attrs[4];

  if (!ConsoleBinding) {
    tended struct b_list *hp;
    tended struct b_lelem *bp;

    /* allocate an empty event queue */
    hp = alclist(0);
    bp = alclstb(MinListSlots, (word)0, 0);
    hp->listhead = hp->listtail = (union block *)bp;
    /* build the attribute list */
    StrLoc(attrs[0]) = "cursor=on";
    StrLen(attrs[0]) = strlen("cursor=on");
    StrLoc(attrs[1]) = "reverse=on";
    StrLen(attrs[1]) = strlen("reverse=on");
    StrLoc(attrs[2]) = "rows=24";
    StrLen(attrs[2]) = strlen("rows=24");
    StrLoc(attrs[3]) = "columns=80";
    StrLen(attrs[3]) = strlen("columns=80");
    ConsoleBinding = WinOpen("Console", hp, attrs, 4);
    } /* End of if - have to open the console */
  return ConsoleBinding;
} /* End of OpenConsole */


/* ---------------------- Miscellaneous Functions ------------------------ */
/* -----------------------------------------------------------------------
   Name - ParseGeometry
   Purpose - parse a string of the form: intxint[+-]int[+-]int
   Parameters -
   Returns -
   Side Effects -
   ----------------------------------------------------------------------- */
ULONG ParseGeometry(char *buf, SHORT *x, SHORT *y, SHORT *width, SHORT *height)
{
  ULONG flags = 0;

  if ((*width = atoi(buf)) > 0) flags |= GEOM_WIDTH;
  while (isspace(*buf)) buf++;
  while (isdigit(*buf)) buf++;
  /* step over the separator */
  if ((*height = atoi(++buf)) > 0) flags |= GEOM_HEIGHT;
  while (isspace(*buf)) buf++;
  while (isdigit(*buf)) buf++;
  if ((*x = atoi(buf)) != 0) flags |= GEOM_POSX;
  while (isspace(*buf) || *buf == '+' || *buf == '-') buf++;
  while (isdigit(*buf)) buf++;
  if ((*y = atoi(buf)) != 0) flags |= GEOM_POSY;
  return flags;
} /* End of ParseGeometry */


/* -----------------------------------------------------------------------
   Name - GetTextWidth
   Purpose -
   Parameters -
   Returns -
   Side Effects -
   ----------------------------------------------------------------------- */
int GetTextWidth(wbinding *wb, char *text, int len)
{
  POINTL pts[TXTBOX_COUNT];
  STDLOCALS(wb);

  /* make sure our text context is loaded */
  SetCharContext(wb, ws, wc);
  /* calculate the text box */
  GpiQueryTextBox(stdbit, len, text, TXTBOX_COUNT, pts);
  return pts[TXTBOX_TOPRIGHT].x - pts[TXTBOX_BOTTOMLEFT].x;
} /* End of GetTextWidth */


/* -----------------------------------------------------------------------
   Name - PMfprintf
   Purpose -
   Parameters -
   Returns -
   Side Effects -
   ----------------------------------------------------------------------- */
int PMfprintf(FILE *file, char *format, ...)
{
  va_list list;
  int retval = -1;
  char buf[1024];
  wbinding *console;

  va_start(list, format);
  if ((file == stdout && !StdOutRedirect) ||
      (file == stderr && !StdErrRedirect)) {
    console = (wbinding *)OpenConsole();
    if ((retval = vsprintf(buf, format, list)) >= 0);
      wputstr(console, buf, strlen(buf));
    } /* End of if - output stdio and not redirected */
  else 
    retval = vfprintf(file, format, list);
  va_end(list);
  return retval;
} /* End of PMfprintf */


/* -----------------------------------------------------------------------
   Name - PMputc
   Purpose -
   Parameters -
   Returns -
   Side Effects -
   ----------------------------------------------------------------------- */
int PMputc(int c, FILE *file)
{
  if ((file == stdout && !StdOutRedirect) ||
      (file == stderr && !StdErrRedirect)) 
    return wputc(c, (wbinding *)OpenConsole());
  return fputc(c, file);
} /* End of PMputc */
#else					/* PresentationManager */
static char cxc;			/* avoid empty module */
#endif					/* PresentationManager */
