/*
 * MultiMail offline mail reader
 * resource class

 Copyright (c) 1996 Toth Istvan <stoty@vma.bme.hu>
 Copyright (c) 1998 William McBrine <wmcbrine@clark.net>

 Distributed under the GNU General Public License.
 For details, see the file COPYING in the parent directory. */

#include "mmail.h"
#include "../interfac/error.h"
#include "../interfac/mysystem.h"

#include <sys/stat.h>

#ifdef __MSDOS__
# define DEFEDIT "edit"
#else
# ifdef __EMX__
#  define DEFEDIT "tedit"
# else
#  define DEFEDIT "vi"
# endif
#endif

#if defined (__MSDOS__) || defined (__EMX__)
# define RCNAME "mmail.rc"
# define ADDRBOOK "address.bk"
#else
# define RCNAME ".mmailrc"
# define ADDRBOOK "addressbook"
#endif

extern ErrorType error;

resource::resource()
{
	FILE *configFile;
	char configFileName[256], inp;
	const char *envhome, *greeting = "\nWelcome new user!\n\n"
		"A new (empty) " RCNAME " has been created. "
		"If you continue now, " MM_NAME " will\nuse the default "
		"values. If you wish to edit your " RCNAME " first, say "
		"'N' at\nthe prompt.\n\nContinue? (y/n) ";
	int c;

	for (c = 0; c < noOfResources; c++)
		resourceData[c].value = NULL;

	envhome = getenv("MMAIL");
	if (!envhome)
		envhome = getenv("HOME");
	if (!envhome)
		envhome = error.getOrigDir();

	set(homeDir, fixPath(envhome));
	sprintf(configFileName, "%.243s/" RCNAME, get(homeDir));

	if (!(configFile = fopen(configFileName, "rt"))) {
		if (!createNewConfig(configFileName)) {
			printf(greeting);
			inp = getchar();
			if (toupper(inp) == 'N')
				exit(1);
			configFile = fopen(configFileName, "rt");
		} else
			fatalError("Error opening " RCNAME "!");
	}

	mytmpnam(basedir);
	checkPath(basedir, 0);
	initinit();
	parseConfig(configFile);
	fclose(configFile);
	mmHomeInit();

	c = verifyPaths();
	if (!c)
		c = taglineCheck();
	if (c)
		fatalError("Unable to access data directories");
}

resource::~resource()
{
	clearDirectory(resourceData[WorkDir].value);
	clearDirectory(resourceData[UpWorkDir].value);
	mychdir("..");
	rmdir(resourceData[WorkDir].value);
	rmdir(resourceData[UpWorkDir].value);
	mychdir("..");
	rmdir(basedir);
	for (int c = 0; c < noOfResources; c++)
		delete resourceData[c].value;
}

// For consistency, no path should end in a slash:
const char *resource::fixPath(const char *path)
{
	static char tmp[256];
	char d = path[strlen(path) - 1];

	if ((d == '/') || (d == '\\')) {
		sprintf(tmp, "%.254s.", path); 
		return tmp;
	} else
		return path;
}

int resource::checkPath(const char *onepath, int show)
{
	if (mychdir(onepath)) {
		if (show)
			printf("Creating %s...\n", onepath);
		if (mkdir(onepath, S_IRWXU))
			return 1;
	}
	return 0;
}

// Create tagline file if it doesn't exist.
int resource::taglineCheck()
{
	// Default taglines:
#include "taglines.h"

	FILE *tagx;
	const char **p;
	int flag = 1;

	if ((tagx = fopen(resourceData[TaglineFile].value, "rt")))
		flag = 0;
	else {
		printf("Creating %s...\n", resourceData[TaglineFile].value);
		if ((tagx = fopen(resourceData[TaglineFile].value, "wt"))) {
			for (p = defaultTags; **p; p++) {
				fputs(*p, tagx);
				fputc('\n', tagx);
			}
			flag = 0;
		}
	}
	if (!flag)
		fclose(tagx);
	return flag;
}

int resource::verifyPaths()
{
	if (!checkPath(resourceData[mmHomeDir].value, 1))
	    if (!checkPath(resourceData[PacketDir].value, 1))
		if (!checkPath(resourceData[ReplyDir].value, 1))
		    if (!checkPath(resourceData[SaveDir].value, 1))
			return 0;
	return 1;
}

int resource::createNewConfig(const char *configname)
{
	// Default .mmailrc:
#include "mmailrc.h"

	FILE *mmrc;
	const char **p;

	printf("Creating %s...\n", configname);
	if ((mmrc = fopen(configname, "wt"))) {
		for (p = defaultMmailrc; **p; p++) {
			fputs(*p, mmrc);
			fputc('\n', mmrc);
		}
		fclose(mmrc);
		return 0;
	} else
		return 1;
}

void resource::parseConfig(FILE *configFile)
{
	const int startUpLen = 17;

	static const char *const names[startUpLen] =
	{
		"homeDir", "mmHomeDir", "PacketDir", "ReplyDir",
		"SaveDir", "editor", "AddressBook", "TaglineFile",
		"arjUncompressCommand", "zipUncompressCommand",
		"lhaUncompressCommand", "unknownUncompressCommand",
		"arjCompressCommand", "zipCompressCommand",
		"lhaCompressCommand", "unknownCompressCommand", "signature"
	};

	static const int startUp[startUpLen] =
	{
		homeDir, mmHomeDir, PacketDir, ReplyDir,
		SaveDir, editor, AddressFile, TaglineFile,
		arjUncompressCommand, zipUncompressCommand,
		lhaUncompressCommand, unknownUncompressCommand,
		arjCompressCommand, zipCompressCommand,
		lhaCompressCommand, unknownCompressCommand, sigFile
	};

	char buffer[256], *resName, *resValue, *pos, *end;
	int c;

	while (fgets(buffer, 255, configFile)) {
		if ((buffer[0] != '#') && (buffer[0] != '\n')) {
			pos = buffer;
			end = pos + 255;

			//leading spaces
			while (*pos == ' ' || *pos == '\t' && pos < end)
				pos++;

			//skip "bw" -- for backwards compatiblity
			if (*pos == 'b' && pos[1] == 'w')
				pos += 2;

			//resName
			resName = pos;
			while (*pos != ':' && *pos != ' ' &&
				*pos != '\t' && pos < end)
					pos++;
			*pos++ = '\0';

			//chars between strings
			while (*pos == ' ' || *pos == '\t' ||
				*pos == ':' && pos < end)
					pos++;

			//resValue
			resValue = pos;
			while (*pos != '\n' && pos < end)
				pos++;
			*pos++ = '\0';

			for (c = 0; c < startUpLen; c++)
				if (!strcasecmp(names[c], resName)) {
					set(startUp[c], fixPath(resValue));
					break;
				}
		}
	}
}

const char *resource::get(int ID)
{
	return (((ID < 0) || (ID >= noOfResources)) ? 0 :
		resourceData[ID].value);
}

void resource::set(int ID, const char *newValue)
{
	if ((ID >= 0) && (ID < noOfResources)) {
		delete resourceData[ID].value;
		resourceData[ID].value = strdupplus(newValue);
	}
}

int resource::getValue(int ID)
{
	return (((ID < 0) || (ID >= noOfResources)) ? 0 :
		resourceData[ID].val);
}

void resource::setValue(int ID, int newValue)
{
	if ((ID >= 0) && (ID < noOfResources))
		resourceData[ID].val = newValue;
}

//------------------------------------------------------------------------
// The resource initializer functions
//------------------------------------------------------------------------

void resource::mmEachInit(int index, const char *dirname)
{
	char tmp[256];

	if (!resourceData[index].value) {
		sprintf(tmp, "%.243s/%s", resourceData[mmHomeDir].value,
			dirname);
		set(index, tmp);
	}
}

void resource::subPath(int index, const char *dirname)
{
	char tmp[256];

	sprintf(tmp, "%.243s/%s", basedir, dirname);
	set(index, tmp);
	if (checkPath(tmp, 0))
		fatalError("tmp Dir could not be created");
}

void resource::initinit()
{
	char tmp[256], *p;

	sprintf(tmp, "%.243s/mmail", resourceData[homeDir].value);
	set(mmHomeDir, tmp);

	set(UncompressCommand, DEFUNZIP);
	set(CompressCommand, DEFZIP);

	subPath(WorkDir, "work");
	subPath(UpWorkDir, "upwork");

	p = getenv("EDITOR");
	set(editor, (p ? p : DEFEDIT));
}

void resource::mmHomeInit()
{
	mmEachInit(PacketDir, "down");
	mmEachInit(ReplyDir, "up");
	mmEachInit(SaveDir, "save");
	mmEachInit(AddressFile, ADDRBOOK);
	mmEachInit(TaglineFile, "taglines");
}
