/*
 * MultiMail offline mail reader
 * file_header and file_list

 Copyright (c) 1996 Toth Istvan <stoty@vma.bme.hu>
 Copyright (c) 1998 William McBrine <wmcbrine@clark.net>

 Distributed under the GNU General Public License.
 For details, see the file COPYING in the parent directory. */

#include "mmail.h"

#include <dirent.h>
#include <sys/stat.h>

// ------------------------------------------------------------------------
// file_list functions
// ------------------------------------------------------------------------

file_list::file_list(const char *FileDir)
{
	initFiles(FileDir);
	if (noOfFiles > 1)
		fileqsort(0, noOfFiles - 1);
}

file_list::~file_list()
{
	while (noOfFiles)
		delete files[--noOfFiles];
	delete files;
}

void file_list::initFiles(const char *DirName)
{
	DIR *Dir;
	struct dirent *entry;
	struct stat fileStat;

	if (!(Dir = opendir(DirName)))
		fatalError("There is no Packet Dir!");

	mychdir(DirName);

	noOfFiles = 0;

	file_header head("", 0, 0);
	file_header *filept = &head;

	while ((entry = readdir(Dir)))
		if (entry->d_name[0] != '.')
			if (!(stat(entry->d_name, &fileStat)))
			    if (!access(entry->d_name, R_OK | W_OK)) {
				filept->next =
				    new file_header(entry->d_name,
					fileStat.st_mtime, fileStat.st_size);
				filept = filept->next;
				noOfFiles++;
			}

	files = new file_header *[noOfFiles];
	int c = 0;
	filept = head.next;
	while (filept) {
		files[c++] = filept;
		filept = filept->next;
	}

	closedir(Dir);
}

void file_list::fileqsort(int left, int right)
{
	int lower, upper, d, t;
	file_header *pivot, *phtmp;
	const char *p, *q;

	lower = left;
	upper = right;
	pivot = files[((left + right) >> 1)];
	p = pivot->getName();
	do {
		do {
			q = files[lower]->getName();
			d = strcasecmp(q, p);
			if (!d)
				d = strcmp(p, q);
			t = (d < 0);
			if (t)
				lower++;
		} while (t);
		do {
			q = files[upper]->getName();
			d = strcasecmp(q, p);
			if (!d)
				d = strcmp(p, q);
			t = (d > 0);
			if (t)
				upper--;
		} while (t);
		if (lower <= upper) {
			if (lower != upper) {
				phtmp = files[lower];
				files[lower] = files[upper];
				files[upper] = phtmp;
			}
			lower++;
			upper--;
		}
	} while (lower < upper);
	if (left < upper)
		fileqsort(left, upper);
	if (lower < right)
		fileqsort(lower, right);
}

int file_list::getNoOfFiles()
{
	return noOfFiles;
}

void file_list::gotoFile(int fileNo)
{
	if ((fileNo >= 0) && (fileNo < noOfFiles))
		activeFile = fileNo;
}

const char *file_list::getName()
{
	return files[activeFile]->getName();
}

const char *file_list::exists(const char *fname)
{
	for (int c = 0; c < noOfFiles; c++)
		if (!strcasecmp(files[c]->getName(), fname)) {
			activeFile = c;
			return getName();
		}
	return NULL;
}

const char *file_list::getNext(const char *extent)
{
	int c, len;
	const char *p, *q;

	for (c = activeFile; c < noOfFiles; c++) {
		q = files[c]->getName();
		len = strlen(q);
		if (len > 5) {
			p = q + len - 4;
			if (!strcasecmp(p, extent)) {
				activeFile = c + 1;
				return q;
			}
		}
        }
        return NULL;
}

FILE *file_list::ftryopen(const char *fname, const char *mode)
{
	const char *p = exists(fname);
	return (p ? fopen(p, mode) : 0);
}

time_t file_list::getDate()
{
	return files[activeFile]->getDate();
}

off_t file_list::getSize()
{
	return files[activeFile]->getSize();
}

// --------------------------------------------------------------------------
// file_header methods
// --------------------------------------------------------------------------
file_header::file_header(char *nameA, time_t dateA, off_t sizeA)
{
	name = strdupplus(nameA);
	date = dateA;
	size = sizeA;
	next = NULL;
}

file_header::~file_header()
{
	delete name;
}

const char *file_header::getName()
{
	return name;
}

time_t file_header::getDate()
{
	return date;
}

off_t file_header::getSize()
{
	return size;
}
