/*****************************************************************************
*   "Irit" - the 3d polygonal solid modeller.				     *
*									     *
* Written by:  Gershon Elber				Ver 0.2, Mar. 1990   *
******************************************************************************
* Setting attributes for geometric objects.				     *
*****************************************************************************/

#include <string.h>
#include <stdio.h>
#include <math.h>
#include "program.h"
#include "allocate.h"
#include "attribut.h"
#include "windows.h"
#include "graphgen.h"

/*****************************************************************************
*   Routine to return the color of the geometric object.		     *
*****************************************************************************/
int GetObjectColor(ObjectStruct *PObj)
{
    if (!IS_GEOM_OBJ(PObj))
	FatalError("Attributes are for geometry objects only");

    return (int) (PObj -> U.Attr.Color);
}

/*****************************************************************************
*   Routine to set the color of the geometric object.			     *
*****************************************************************************/
void SetObjectColor(ObjectStruct *PObj, int Color)
{
    if (!IS_GEOM_OBJ(PObj))
	FatalError("Attributes are for geometry objects only");

    PObj -> U.Attr.Color = (ByteType) Color;
}

/*****************************************************************************
*   Routine to set a string attribute. A string attribute consists of an     *
* attribute name (string) and data (also string).			     *
*   If Name = "", all attributes of given object are printed.		     *
*   If Data = "", the attribute with name Name is been freed.		     *
*   If attribute by the given name already exists, it is replaced.	     *
*****************************************************************************/
void SetObjectStrAttrib(ObjectStruct *PObj, char *Name, char *Data)
{
    int i;
    char Line[LINE_LEN_LONG];
    AttributeStruct *Attr;

    if (!IS_GEOM_OBJ(PObj))
	FatalError("Attributes are for geometry objects only");

    Attr = &(PObj -> U.Attr);

    /* If Name is an empty string - print all attributes. */
    if (strlen(Name) == 0) {
	sprintf(Line, "[COLOR %d]", Attr -> Color);
	WndwInputWindowPutStr(Line);
	for (i = 0; i < Attr -> NumStrAttribs; i++) {
	    sprintf(Line, "[%s %s]",
		    Attr -> StrAttrName[i], Attr -> StrAttrData[i]);
	    WndwInputWindowPutStr(Line);
	}
	return;
    }

    for (i = 0; i < Attr -> NumStrAttribs; i++) {
	if (strcmp(Name, Attr -> StrAttrName[i]) == 0) {
	    /* If Data is an empty string, remove this entry. */
	    if (strlen(Data) == 0) {
		MyFree((VoidPtr) Attr -> StrAttrName[i], ALLOC_OTHER);
		MyFree((VoidPtr) Attr -> StrAttrData[i], ALLOC_OTHER);
		for ( ; i < (int) Attr -> NumStrAttribs - 2; i++) {
		    Attr -> StrAttrName[i] = Attr -> StrAttrName[i + 1];
		    Attr -> StrAttrData[i] = Attr -> StrAttrData[i + 1];
		}
		Attr -> NumStrAttribs--;
	    }
	    else {
		MyFree((VoidPtr) Attr -> StrAttrData[i], ALLOC_OTHER);
		Attr -> StrAttrData[i] = strdup(Data);
	    }

	    return;
	}
    }

    /* O.k. it is a new attribute. */
    if (Attr -> NumStrAttribs >= MAX_NUM_ATTRS) {
	WndwInputWindowPutStr("Maximum num. of attributes exceeded");
	return;
    }
    Attr -> StrAttrName[Attr -> NumStrAttribs] = strdup(Name);
    Attr -> StrAttrData[Attr -> NumStrAttribs++] = strdup(Data);
}

/*****************************************************************************
*   Routine to get a string attribute. A string attribute consists of an     *
* attribute name (string) and data (also string).			     *
*   Returns a pointer to data if string name is found, NULL otherwise.	     *
*****************************************************************************/
char *GetObjectStrAttrib(ObjectStruct *PObj, char *Name)
{
    int i;
    AttributeStruct *Attr;

    if (!IS_GEOM_OBJ(PObj))
	FatalError("Attributes are for geometry objects only");

    Attr = &(PObj -> U.Attr);

    /* If Name is an empty string - print all attributes. */
    if (Name == NULL || strlen(Name) == 0) return NULL;

    for (i = 0; i < Attr -> NumStrAttribs; i++)
	if (strcmp(Name, Attr -> StrAttrName[i]) == 0)
    	    return Attr -> StrAttrData[i];

    return NULL;
}

/*****************************************************************************
*   Routine to release all attributes of the given Object.		     *
*****************************************************************************/
void ReleaseStrAttrib(ObjectStruct *PObj)
{
    int i;
    AttributeStruct *Attr;

    if (!IS_GEOM_OBJ(PObj))
	FatalError("Attributes are for geometry objects only");

    Attr = &(PObj -> U.Attr);
    for (i = 0; i < Attr -> NumStrAttribs; i++) {
	MyFree((VoidPtr) Attr -> StrAttrName[i], ALLOC_OTHER);
	MyFree((VoidPtr) Attr -> StrAttrData[i], ALLOC_OTHER);
    }
}

/*****************************************************************************
*   Routine to initialize the Attributes of the given Object.		     *
*****************************************************************************/
void ResetObjectAttribs(ObjectStruct *PObj)
{
    SetObjectColor(PObj, WHITE);		 /* Default color attribute. */
    PObj -> U.Attr.NumStrAttribs = 0;
}

/*****************************************************************************
*   Routine to copy the attributes from one object to another.		     *
*****************************************************************************/
void CopyGeomAttrib(ObjectStruct *Dest, ObjectStruct *Src)
{
    int i;
    if (!IS_GEOM_OBJ(Dest) || !IS_GEOM_OBJ(Src))
	FatalError("Attributes are for geometry objects only");

    Dest -> U.Attr.Color = Src -> U.Attr.Color;
    Dest -> U.Attr.NumStrAttribs = Src -> U.Attr.NumStrAttribs;

    for (i = 0; i < (Dest -> U.Attr.NumStrAttribs); i++) {
	Dest -> U.Attr.StrAttrName[i] = strdup(Src -> U.Attr.StrAttrName[i]);
	Dest -> U.Attr.StrAttrData[i] = strdup(Src -> U.Attr.StrAttrData[i]);
    }
}
