/* ---------------------------------------------------------------------- */
/*   BASTOPCX.c (C) Version 2.0 Copyright Bill Buckels 1989-1999          */
/*   All Rights Reserved.                                                 */
/*                                                                        */
/*   BASTOPCX is distributed as ShareWare.                                */
/*   It is part of the STIX(C) Children's Sticker Draw Package.           */
/*   Suggested Registration for STIX(C) is $10.00 per family.             */
/*                                                                        */
/*   You are expected to register with the Author if you use STIX         */
/*   or any of its associated files and programs including BASTOPCX       */
/*   beyond a 30-day evaluation period. Send registration in the form     */
/*   of cheque, or money order to:                                        */
/*                                                                        */
/*   Bill Buckels                                                         */
/*   589 Oxford Street                                                    */
/*   Winnipeg, Manitoba, Canada R3M 3J2                                   */
/*                                                                        */
/*   Email: bbuckels@escape.ca                                            */
/*   WebSite: http://www.escape.ca/~bbuckels                              */
/*                                                                        */
/*   Registered users of STIX have a royalty-free right to use, modify,   */
/*   reproduce and distribute this source code (and/or any modified       */
/*   version) in way you find useful, provided that you do not compete    */
/*   with Bill Buckels or his agents, and that you agree that Bill        */
/*   Buckels has no warranty obligations or liability whatsoever          */
/*   resulting from any associated loss or damage.                        */
/*                                                                        */
/*   If you do not agree with these terms, remove this source and         */
/*   all associated files from your computer now.                         */
/*                                                                        */
/*   Description                                                          */
/*   -----------                                                          */
/*   Supported Screen Modes are CGA Graphics Modes 4 and 6 only.          */
/*                                                                        */
/*   1. converts a BASIC BSAVED IMAGE FILE (full screen CGA graphics)     */
/*   to a ZSOFT .PCX format PCPAINTBRUSH compatible image file            */
/*   per the VERSION 2.8 STANDARD without Color Map.                      */
/*                                                                        */
/*   2. converts from ZSOFT .PCX CGA graphics file to BASIC BSAVED        */
/*   image file. Assumes VERSION 2.8 STANDARD w/o Color Map.              */
/*                                                                        */
/*   This program has been tested with PCPAINTBRUSH Version 3             */
/*   and LOGITECH PAINTSHOW PLUS Version 2.1 and its PCX2TIF and          */
/*   TIF2PCX import/export utilities and with GWBASIC 3.22 and            */
/*   QUICKBASIC versions 2.0 and 4.5.                                     */
/*                                                                        */
/*   Additionally, Should be compatible with VENTURA PUBLISHING 1.01      */
/*   and any other program accepting either .PCX or BSAVED IMAGES.        */
/*                                                                        */
/*   BASTOPCX is written in Large Model Microsoft C Version 6.00a         */
/*                                                                        */
/* ---------------------------------------------------------------------- */

#include <stdio.h>
#include <fcntl.h>
#include <dos.h>
#include <bios.h>
#include <io.h>
#include <string.h>
#include <malloc.h>

#define SUCCESS  0
#define INVALID -1
#define TRUE     1
#define FALSE    0

#define CGA_320  4
#define HI_RES   6

#define S_IWRITE    0000200     /* write permission, owner   */

FILE *pcxfile;

int screen_mode=CGA_320;       /* default */

/* type conversion functions */
unsigned int byteword(unsigned char a, unsigned char b){return b<<8|a;}
unsigned char lsb(unsigned int word){ return word &0xff;}
unsigned char msb(unsigned int word){ return word >>8;}
unsigned char pcxheader[128];

/* a microsoft compatible bsaved image format descriptor */
char MEDRES_header[7]={
    '\xfd','\x00','\xb8','\x00','\x00','\x00','\x40'};

/* ---------------------------------------------------------------------- */
/* check for a BASIC BSAVED CGA screen image                              */
/* ---------------------------------------------------------------------- */
int checkforbas(char *name)
{
    FILE *fp;
    char c;
    int i,status=SUCCESS;

    if ((fp = fopen(name,"rb")) == NULL)return INVALID;
    for(i=0;i!=4;i++) {
      if((c=fgetc(fp))!=MEDRES_header[i]) {
        status=INVALID;
        break;
      }
    }
    fclose(fp);
    return status;
}

/* ---------------------------------------------------------------------- */
/* check for a CGA Mode Full Screen PCX Image                             */
/* ---------------------------------------------------------------------- */
int checkforpcx(char *name)
{
    FILE *fp; /* reads a ZSOFT .PCX header but ignores the color map */
    int i;    /* we only want CGA compatible full screens. */

    unsigned int zsoft,version,codetype,pixbits;
    unsigned int xmin, ymin, xmax, ymax;
    unsigned int hres, vres;
    unsigned int no_planes, bytesperline;
    int status=SUCCESS;

    /* read the file header */
    if((fp=fopen(name,"rb"))==NULL)return INVALID;
    for(i=0;i!=128;i++)pcxheader[i]=fgetc(fp);
    fclose(fp);

    zsoft   =pcxheader[0];
    version =pcxheader[1];
    codetype=pcxheader[2];
    pixbits =pcxheader[3];

    if(zsoft!=10)        status = INVALID;
    if(codetype!=1)      status = INVALID;
    if(pixbits != 2 )
       if(pixbits != 1)  status = INVALID;

    xmin=byteword(pcxheader[4],pcxheader[5]);
    ymin=byteword(pcxheader[6],pcxheader[7]);
    xmax=byteword(pcxheader[8],pcxheader[9]);
    ymax=byteword(pcxheader[10],pcxheader[11]);

    hres=byteword(pcxheader[12],pcxheader[13]);
    vres=byteword(pcxheader[14],pcxheader[15]);

    no_planes   =pcxheader[65];
    bytesperline=byteword(pcxheader[66],pcxheader[67]);

    if(xmin != 0  )      status = INVALID;
    if(ymin != 0  )      status = INVALID;
    if(xmax != 319)
       if(xmax!=639)     status = INVALID;
    if(ymax != 199)      status = INVALID;

    if(no_planes!=1)     status = INVALID;
    if(bytesperline !=80)status = INVALID;
    /* we can ignore the color map since we        */
    /* are limiting ourselves to CGA modes         */
    /* so we will not handle over 2-bits per pixel */
    return status;

}

/* ---------------------------------------------------------------------- */
/* make a header for a CGA Mode Full Screen PCX Image                     */
/* ---------------------------------------------------------------------- */
int makeheader()
{
    int i;
    unsigned char zsoft=10,version=3,codetype=1,pixbits=2;
    unsigned int  xmin=0, ymin=0, xmax=319, ymax=199;
    unsigned int  hres=320, vres=200;
    unsigned char no_planes=1;
    unsigned int  bytesperline=80;

    for(i=0;i!=128;i++)pcxheader[i]=0;/* pad the header with nulls */

    if(screen_mode==HI_RES){
      pixbits=pixbits/2;
      xmax=((xmax+1)*2)-1;
      hres=hres*2;
    }

    pcxheader[0]=zsoft;
    pcxheader[1]=version;
    pcxheader[2]=codetype;
    if(screen_mode==HI_RES)pixbits=1;
    pcxheader[3]=pixbits;
    pcxheader[4] =lsb(xmin);
    pcxheader[5] =msb(xmin);
    pcxheader[6] =lsb(ymin);
    pcxheader[7] =msb(ymin);
    pcxheader[8] =lsb(xmax);
    pcxheader[9] =msb(xmax);
    pcxheader[10]=lsb(ymax);
    pcxheader[11]=msb(ymax);
    pcxheader[12]=lsb(hres);
    pcxheader[13]=msb(hres);
    pcxheader[14]=lsb(vres);
    pcxheader[15]=msb(vres);
    pcxheader[65]=no_planes;
    pcxheader[66]=lsb(bytesperline);
    pcxheader[67]=msb(bytesperline);
    return 0;
}

/* ---------------------------------------------------------------------- */
/* make a CGA Mode Full Screen PCX Image from a BASIC BSAVED Image        */
/* ---------------------------------------------------------------------- */
int bastopcx(char *name1, char *name2)
{
    char far *bigbuffer;

    unsigned char crtbuf[80];
    unsigned char leafbuf[80];
    unsigned int  offset=0;
    unsigned int  inleaf=8192;
    int packet = 80,i;

    FILE *fp;
    int fh;
    char headbuf[7];

    if((fh = open(name1,O_RDONLY|O_BINARY)) == INVALID)return INVALID;

    printf("BASTOPCX(C) Copyright 1989-1999 Bill Buckels\n");
    printf("All Rights Reserved.\n");
    printf("Input  File: %s\n",name1);
    printf("Output File: %s\n",name2);

    bigbuffer= _fmalloc(16385);
    read(fh,headbuf,7);
    read(fh,bigbuffer,16385);
    close(fh);

    pcxfile=fopen(name2,"wb");
    for(i=0;i!=128;i++)fputc(pcxheader[i],pcxfile);/* write the header */
    for(offset=0;offset<8000;offset+=packet){
      memcpy(crtbuf,bigbuffer+offset,packet);
      memcpy(leafbuf,bigbuffer+inleaf,packet);
      encline(crtbuf,packet);
      encline(leafbuf,packet);
      inleaf+=80;
    }
    fclose(pcxfile);
    return SUCCESS;
}

/* ---------------------------------------------------------------------- */
/* make a CGA Mode BASIC BSAVED IMAGE from a Full Screen CGA PCX Image    */
/* ---------------------------------------------------------------------- */
int pcxtobas(char *name1,char *name2)
{
    char far *bigbuffer;

    unsigned int byteoff=0,inleaf=8192,packet,width=0;
    FILE *fp;
    unsigned char byte,bytecount;
    unsigned wordcount,target;
    int fh;

    if ((fp = fopen(name1,"rb")) == NULL)return INVALID;
    printf("PCXTOBAS(C) Copyright by Bill Buckels\n\n");
    printf("Input  File: %s\n",name1);
    printf("Output File: %s\n",name2);

    target = (unsigned)filelength(fileno(fp));
    for(wordcount=0;wordcount!=128;wordcount++)pcxheader[wordcount]=fgetc(fp);
    bigbuffer= _fmalloc(16385);

    do{ bytecount=1;                          /* start with a seed count */
        byte=fgetc(fp);
        wordcount++;
                                              /* check to see if its raw */
        if(0xC0 == (0xC0 &byte)){             /* if its not, run encoded */
          bytecount= 0x3f &byte;
          byte=fgetc(fp);
          wordcount++;
        }
        for(packet=0;packet<bytecount;packet++){
          if(width<80){
            bigbuffer[byteoff]=byte;
            width++;
            byteoff++;
          }
          else{
            bigbuffer[inleaf]=byte;
            inleaf++;
            width++;
            if(width>159)width=0;
          }
        }
    }while(wordcount<target);
    fclose(fp);

    if((fh =
        open(name2,O_CREAT|O_TRUNC|O_WRONLY|O_BINARY,S_IWRITE)) == INVALID)
        return INVALID;
    write(fh,MEDRES_header,7);
    write(fh,bigbuffer,16385);
    if(close(fh) == INVALID)
    return SUCCESS;
}

char separators[]=" .\n";
/* ---------------------------------------------------------------------- */
/* main program                                                           */
/* ---------------------------------------------------------------------- */
main(int argc,char *argv[])
{
   char buffer[128],
        name1[128],
        name2[128],
        *wordptr;

   int status;

   switch(argc)
   {
     case 3: if(atoi(argv[2])==1)screen_mode=HI_RES;

     case 2: strcpy(buffer,argv[1]);
             wordptr=strtok(buffer,separators);
             strcpy(name1,buffer);
             strcat(name1,".BAS");
             strcpy(name2,buffer);
             strcat(name2,".PCX");

             if((status=checkforpcx(name2))==0){
               status=pcxtobas(name2,name1);
               exit((status!=0));
             }
             if((status=checkforbas(name1))==0){
               makeheader();
               status=bastopcx(name1,name2);
               exit((status!=0));
             }

     default:
          printf("BASTOPCX(C) Version 2.0 Copyright Bill Buckels 1989-1999\n");
          printf("All Rights Reserved.\n");
          printf("Usage : \"BASTOPCX [filename] [bits-per-pixel(optional 1 or2)]\"\n");
          printf("Converts .PCX GRAPHICS to BSAVED IMAGES and VICE-VERSA.\n");
          printf("CGA MED_RES and HI_RES modes, screen dumps only.\n");
   }
   exit(1);
}

/* ---------------------------------------------------------------------- */
/* these next 2-routines are samples implemented from the ZSOFT technical */
/* reference manual virtually without change.                             */
/*                                                                        */
/* The manual is tiny for sure, and extremely conscise.                   */
/* There was no charge for the manual.                                    */
/*                                                                        */
/* Phone or Write-              ZSOFT CORPORATION                         */
/*                         450 FRANKLIN RD., SUITE 100                    */
/*                             MARIETTA, GA. 30067                        */
/*                               (404)428-0008                            */
/*                                                                        */
/* There is a developer's kit available for the .PCX format as well, but  */
/* when I phoned the authors (GENUS MICROPROGRAMMING) both the            */
/* RIDICULOUS PRICE and their INDIFFERENCE launched me into self-help     */
/* mode.                                                                  */
/*                                                                        */
/* I had originally planned to use the .TIF format (ALDUS/MICROSOFT) but  */
/* after purchasing the developer's kit, I ran into incompatability       */
/* between the sample programs in the kit, and in the FILES that my       */
/* SCANNER and LOGITECH's PSPLUS produces. Now, after working with the    */
/* .PCX format, I will likely choose .PCX as my preferred SCREEN IMAGE    */
/* FORMAT for future endeavors.                                           */
/* ---------------------------------------------------------------------- */

/* ---------------------------------------------------------------------- */
/* encodes a raw line and writes it out to disk.                          */
/* required to write the PCX files saved by this program.                 */
/* ---------------------------------------------------------------------- */
int encline(unsigned char *inbuff, int inlen)
{
    unsigned char current,last;
    int srcindex,i;
    register int total;
    register unsigned char runcount;

    total=0;
    last = *(inbuff);
    runcount=1;

    for(srcindex=1;srcindex!=inlen;srcindex++){
      current= *(++inbuff);
      if(current==last){
        runcount++;
        if(runcount==63){
            if(!(i=encput(last,runcount)))
            return(0);
            total+=i;
            runcount=0;
        }
      }
      else {
        if(runcount){
            if(!(i=encput(last,runcount)))
            return(0);
            total+=i;
        }
        last=current;
        runcount=1;
      }
   }

  if (runcount) {
    if (!(i=encput(last,runcount)))
      return(0);
    return(total+i);
  }
  return (total);

}

/* ---------------------------------------------------------------------- */
/* the writer for the encline function                                    */
/* ---------------------------------------------------------------------- */
int encput(unsigned char byt, unsigned char cnt)
{
  if (cnt){
    if((cnt==1)&& (0xc0 != (0xc0 &byt))){
      if (EOF == fputc((int)byt,pcxfile))
         return(0);
      return(1);
    }
    else {
      if (EOF==fputc((int)0xc0|cnt,pcxfile))
        return(0);

      if(EOF==fputc((int)byt,pcxfile))
        return(0);

      return(2);
    }
  }
  return(0);
}
