#include "syncdir.h"

#include <string.h>
#include <assert.h>
#include <errno.h>
#include <time.h>

static char     *copyBuf   = 0;
static int       copyBufSz = 16384;
static EXECMODE  execMode  = EXECMODE_NORMAL;

static time_t        startTime;
static time_t        pauseStart;
static int           pauseCt;
static unsigned long bytesOut;

void pause(BOOL set)
{
  if (set)
  {
    if (!pauseCt++)
      pauseStart = time(0);
  } else if (pauseCt)
    pauseCt--;
}

unsigned long elapsed(void)
{
  return (startTime)
    ? (unsigned long ) difftime(time(0), startTime)
    : 0;
}

unsigned long BPS(void)
{
  unsigned long secs = elapsed();
  if (pauseCt)
    secs -= (unsigned long ) difftime(time(0), pauseStart);
  return (secs)
    ? bytesOut / secs
    : 0;
}

void execModeSet(EXECMODE mode)
{
  execMode = mode; 
}

EXECMODE execModeGet(void)
{
  return execMode; 
}

/*
   determine whether [act] is acceptable
*/
static BOOL prompt(const char *src, const char *dst, ACTION act)
{
  UINT32 ackFlag; /* the force flag */
  UINT32 nakFlag; /* the never flag */
  BOOL ret = TRUE;

  pause(TRUE);

  switch (act)
  {
    case ACTION_NONE:
      ret = FALSE;
    case ACTION_ADD:
      ackFlag = OPT_FORCE_ADD;
      nakFlag = OPT_NEVER_ADD;
      break;
    case ACTION_COPY:
    case ACTION_UPDATE:
      ackFlag = OPT_FORCE_COPY;
      nakFlag = OPT_NEVER_COPY;
      break;
    case ACTION_DELETE:
      ackFlag = OPT_FORCE_DELETE;
      nakFlag = OPT_NEVER_DELETE;
      break;
    case ACTION_ERROR:
      ret = FALSE;
      break;
    default:
      assert(0);
      ret = FALSE;
  }

  if (ret)
  {
    if (optionTest(ackFlag))
      ret = TRUE;
    else if (optionTest(nakFlag))
      ret = FALSE;
    else
      ret = (optionTest(OPT_NOACTION)) ? TRUE : queryAction(src, dst, act);
  }
  if (!optionTest(nakFlag))
    logExecuteAction(src, dst, act, ret);
  pause(FALSE);
  return (optionTest(OPT_NOACTION)) ? FALSE : ret;
}

/*
   any read/write error is fatal & causes removal of [dst]
   being unable to get or set file stats (access time, mode) is not fatal
*/
static BOOL fileCopy(const char *src, const char *dst)
{
  FILEHANDLE srcHandle;
  BOOL err = FALSE;
  FILEATTR fa;
  BOOL     faValid;

  srcHandle = fileOpen(src);
  if (fileHandleIsValid(srcHandle))
  {
    if (fileDelete(dst) || (sysError() == ENOENT))
    {
      UINT32 toCopy        = fileLength(srcHandle);
      FILEHANDLE dstHandle = fileCreate(dst);
      faValid = fileAttrGet(src, &fa);
      if (!faValid)
        logExecuteError(src, LOG_FN_GET_FS, LOG_CANNOT_GET_FILE_INFO);
      if (fileHandleIsValid(dstHandle))
      {
        int in;
        UINT32 amtCopied = 0;

        if (!optionTest(OPT_QUIET))
          showCopyBegin(src, dst, toCopy);
        do
        {
          in = fileRead(srcHandle, copyBuf, copyBufSz);
          if (in > 0)
          {
            int out = fileWrite(dstHandle, copyBuf, in);
            if ((out < 0) || (out != in))
            {
              logExecuteError(dst, LOG_FN_WRITE, strerror(sysError()));
              err = TRUE;
            }  else
            {
              bytesOut += in;
              amtCopied += in;
              showCopyUpdate(src, dst, toCopy, amtCopied);
            }
          } else if (in < 0)
          {
            logExecuteError(src, LOG_FN_READ, strerror(sysError()));
            err = TRUE;
          } 
        } while ((in == copyBufSz) && (execModeGet() != EXECMODE_ABORT));
        fileClose(dstHandle);
        if (err || (execModeGet() == EXECMODE_ABORT))
        {
          if (!fileDelete(dst))
            logExecuteError(dst, LOG_FN_DELETE, strerror(sysError()));
        } else if (faValid && !fileAttrSet(dst, &fa))
        {
          logExecuteError(dst, LOG_FN_SET_FS, LOG_CANNOT_SET_FILE_INFO);
          err = TRUE;
        }
        if (!optionTest(OPT_QUIET))
          showCopyComplete(src, dst, toCopy);
      } else
      {
        logExecuteError(dst, LOG_FN_CREATE, strerror(sysError()));
        err = TRUE;
      }
    } else
    {
      logExecuteError(dst, LOG_FN_DELETE, strerror(sysError())); /* error deleting this file */
      err = TRUE;
    }
    fileClose(srcHandle);
    if (optionTest(OPT_RESET_ACCESS) && faValid && !fileAttrSet(src, &fa))
    {
      logExecuteError(src, LOG_FN_SET_FS, LOG_CANNOT_SET_FILE_INFO);
      err = TRUE;
    }
  } else
  {
    logExecuteError(src, LOG_FN_OPEN, strerror(sysError()));    
    err = TRUE;
  }
  return err;
}

static BOOL exec(FILETABLE *src, FILETABLE *dst, ACTION act)
{
  char *srcPtr = src->name + strlen(src->name);
  char *dstPtr = dst->name + strlen(dst->name);
  FILEINFO *srcInf;

  if (!optionTest(OPT_QUIET))
    showExecuteActionBegin(src->name, dst->name, act);
  for (srcInf = fileTableGetFirst(src); srcInf && (execModeGet() == EXECMODE_NORMAL); srcInf = fileTableGetNext(src))
  {
    if ((srcInf->action == act) || ((act == ACTION_COPY) && (srcInf->action == ACTION_UPDATE)))
    {
      strcpy(srcPtr, srcInf->name);
      strcpy(dstPtr, srcInf->name);
      if (prompt(src->name, dst->name, srcInf->action))
      {
        switch (srcInf->action)
        {
          case ACTION_ADD:
            if (fileInfoIsDirectory(srcInf))
            {
              if (!directoryCreate(dst->name))
                logExecuteError(dst->name, LOG_FN_CREATE, strerror(sysError()));
              break;
            }
          case ACTION_COPY:
          case ACTION_UPDATE:
            fileCopy(src->name, dst->name);
            break;
          case ACTION_DELETE:
            if (fileInfoIsDirectory(srcInf))
            {
              if (!directoryDelete(src->name))
                logExecuteError(src->name, LOG_FN_DELETE, strerror(sysError()));
            } else if (!fileDelete(src->name))
              logExecuteError(src->name, LOG_FN_DELETE, strerror(sysError()));
            break;
          default:
            assert(0);
        }
      }
      {
        ACTIONSTATS *act;
        switch (srcInf->action)
        {
          case ACTION_ADD:    act = &src->stAdd; break;
          case ACTION_COPY:  
          case ACTION_UPDATE: act = &src->stCopy; break;
          case ACTION_DELETE: act = &src->stDelete; break;
          default:
            act = 0;
        }
        if (act)
        {
          act->cCt++;
          act->cSz += srcInf->size;
        }
      }
    }
  }
  if (!optionTest(OPT_QUIET))
    showExecuteActionComplete(src->name, dst->name, act);
  *srcPtr = 0;
  *dstPtr = 0;
  return TRUE;
}

BOOL execute(void)
{
  copyBuf = malloc(copyBufSz);
  if (!copyBuf)
    longjmp(jmpbuf, SYNCDIR_ERROR_MEMORY);
 
  startTime = time(0);
  pause(TRUE); /* start ``paused'' */

  if (!optionTest(OPT_QUIET))
    showExecuteBegin();
  exec(source, dest,   ACTION_ERROR);   /* alert user to any errors */
  exec(dest,   source, ACTION_ERROR);
  exec(source, dest,   ACTION_DELETE);  /* delete first */
  exec(dest,   source, ACTION_DELETE);
  pause(FALSE);
  exec(source, dest,   ACTION_COPY);    /* copy & update */
  exec(dest,   source, ACTION_COPY);
  exec(source, dest,   ACTION_ADD);     /* finally, add! */
  exec(dest,   source, ACTION_ADD);
  if (execModeGet() != EXECMODE_NORMAL)
    logExecuteError(0, LOG_FN_USER, LOG_USER_CANCEL);
  if (!optionTest(OPT_QUIET))
    showExecuteComplete();
  free(copyBuf);
  return TRUE;
}
