*=================================================================
* nB (nano Base)
* Copyright (c) 1996-1997   Daniele Giacomini
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License as
* published by the Free Software Foundation; either version 2 of
* the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public
* License along with this program; if not, write to the Free
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139,
* USA.
*
*=================================================================
* NB.PRG SOURCE FILE ORGANISATION AND SEQUENCE.
*=================================================================
*
* - NB.CH files inclusion.
*
* - global static variables definition inside this source.
*
* - static functions:
*
*   - nB()       Main function.
*
*   - dot*()     "Dot" command line.
*
*   - frm*()     Report forms creation and editing.
*
*   - htf*()     Help Text File generation and browse.
*
*   - idb*()     Interactive DB functions.
*
*   - lbl*()     Label forms creation and editing.
*
*   - macro*()   Macro functions.
*
*   - mnu*()     Menu system support functions and
*                vertical/pop-up menu.
*
*   - set*()     Set functions.
*
*   - status*()  Status information functions.
*
*
* - public functions:
*
*   - assist()   nB menu driven system.
*
*   - doc()     Text editor.
*
*
*=================================================================
* FILE INCLUSION
*=================================================================
#include "nb.ch"

*=================================================================
* FUNCTIONS NAME
*=================================================================

#define DOC_FUNCTION_NAME       "Doc()"


*=================================================================
* STATIC GLOBAL
*=================================================================

* Help Text File location.
static nbHelp


* used by doc()
#ifndef RUNTIME

    #define DOC_OLD      1
    #define DOC_NAME     2
    #define DOC_TEXT     3
    #define DOC_WRAP     4
    #define DOC_INS      5
    #define DOC_ROW      6
    #define DOC_COL      7
    #define DOC_RELROW   8
    #define DOC_RELCOL   9
    #define DOC_CHANGED  10
    #define DOC_LASTKEY  11
    #define DOC_EXIT     12
    #define DOC_NEW      13
    #define DOC_LENGTH   13  // array length

    static docArray

#endif //RUNTIME


#ifndef LINK
*===============================================================
* nB() - Main function
*===============================================================
function nB(;
        cPar0,;
        cPar1,;
        cPar2,;
        cPar3,;
        cPar4,;
        cPar5,;
        cPar6,;
        cPar7,;
        cPar8,;
        cPar9,;
        cPar10,;
        cPar11;
    )
*
* nB( [<cPar0>], [<cPar1>], [<cPar2>], [<cPar3>],
*     [<cPar4>], [<cPar5>], [<cPar6>], [<cPar7>],
*     [<cPar8>], [<cPar9>], [<cPar10>], [<cPar11>] ) --> NIL
*
* This is the main program function.
*
* <cPar0>      May contains the name of a macro program (&)
*              to be executed. If no paramenter is given,
*              nB() starts with the Dot() function.
*
* <cPar1> ... <cPar11> Optional parameters to send to a macro
*                      program via public variables:
*                      c_Par1 ... c_Par9.
*

    local cOldScreen
    local cOldColor         := setcolor()
    local bOldErrorHandler  := errorblock( {|e| errorHandler(e)} )
    local nOldRow           := row()
    local nOldCol           := col()

    local Result

#ifndef RUNTIME
    local lCopyright          := .T.
#else
    local lCopyright          := .F.
#endif // RUNTIME

    default( @cPar0, "" )
    default( @cPar1, "" )
    default( @cPar2, "" )
    default( @cPar3, "" )
    default( @cPar4, "" )
    default( @cPar5, "" )
    default( @cPar6, "" )
    default( @cPar7, "" )
    default( @cPar8, "" )
    default( @cPar9, "" )
    default( @cPar10, "" )
    default( @cPar11, "" )

    *-------------------------------------------------------------
    * Translate the calling parameters into public variables
    * for macro (&) use.
    *-------------------------------------------------------------

    public _MEMVAR_PAR0
    public _MEMVAR_PAR1
    public _MEMVAR_PAR2
    public _MEMVAR_PAR3
    public _MEMVAR_PAR4
    public _MEMVAR_PAR5
    public _MEMVAR_PAR6
    public _MEMVAR_PAR7
    public _MEMVAR_PAR8
    public _MEMVAR_PAR9

    *-------------------------------------------------------------
    * Start the main sequence.
    *-------------------------------------------------------------

    begin sequence

        while .T.                                       // FOREVER

            *-----------------------------------------------------
            * Analise the starting parameters.
            *-----------------------------------------------------

            do case
            case;
                upper(cPar0) == "-C" .or.;
                upper(cPar0) == "/C"
                
                *-------------------------------------------------
                * Do NOT display copyright informations.
                *-------------------------------------------------

                lCopyright := .F.

            case;
                upper(cPar0) == "-W" .or.;
                upper(cPar0) == "/W"
                
                *-------------------------------------------------
                * Do NOT show the wait wheel.
                *-------------------------------------------------

                waitWheel(.F.)

            case;
                cPar0 == "-?" .or.;
                cPar0 == "/?"

                *-------------------------------------------------
                * Show a little help on standard output.
                *-------------------------------------------------

                qqout( PROGRAM_COPYRIGHT_START )
                qqout( PROGRAM_COPYRIGHT_HELP )

                *-------------------------------------------------
                * Save actual cursor position bacause it will
                * be restored at the end of program.
                *-------------------------------------------------

                nOldRow := row()
                nOldCol := 0

                *-------------------------------------------------
                * Terminate execution.
                *-------------------------------------------------

                break                                   // BREAK

            case upper(cPar0) == "-M"

                *-------------------------------------------------
                * Macro "compilation".
                * Show the copyright at the standard output.
                *-------------------------------------------------

                qqout( PROGRAM_COPYRIGHT_START )

                *-------------------------------------------------
                * Save actual cursor position bacause it will
                * be restored at the end of program.
                *-------------------------------------------------

                nOldRow := row()
                nOldCol := 0

                *-------------------------------------------------
                * Start the macro "compilation" using the
                * next two parameters.
                * Note that there is no mouse support during
                * the compilation.
                *-------------------------------------------------

                cm( cPar1, cPar2 )

                *-------------------------------------------------
                * Termiante program execution.
                *-------------------------------------------------

                break                                   // BREAK

            otherwise

                *-------------------------------------------------
                * If cPar0 is not empty, it should be a macro
                * call; if cPar0 is empty, the program is
                * started as a dot command line.
                *-------------------------------------------------

                exit                                    // EXIT

            end

            *-----------------------------------------------------
            * If the first parameter (cPar0) was correctly
            * read, we are here. The other parameters are
            * shifted down one position, because, there
            * may be another parameter to be read at the
            * new first position.
            *-----------------------------------------------------

            cPar0 := cPar1
            cPar1 := cPar2
            cPar2 := cPar3
            cPar3 := cPar4
            cPar4 := cPar5
            cPar5 := cPar6
            cPar6 := cPar7
            cPar7 := cPar8
            cPar8 := cPar9
            cPar9 := cPar10
            cPar10 := cPar11
            cPar11 := ""

        end

#ifdef RUNTIME

        if empty( cPar0 )

            *-----------------------------------------------------
            * There is no macro call but it should be
            * as this is the code for the runtime.
            * The copyright and help will be shown.
            *-----------------------------------------------------

            qqout( PROGRAM_COPYRIGHT_START )
            qqout( PROGRAM_COPYRIGHT_HELP )

            *-----------------------------------------------------
            * Save actual cursor position bacause it will
            * be restored at the end of program.
            *-----------------------------------------------------

            nOldRow := row()
            nOldCol := 0

            *-----------------------------------------------------
            * Terminate runtime execution.
            *-----------------------------------------------------

            break                                       // BREAK

        end

#endif // RUNTIME

        *---------------------------------------------------------
        * The parameters (after shift down) are saved into
        * Memvars available for macroes.
        *---------------------------------------------------------

        Memvar->_MEMVAR_PAR0 := cPar0
        Memvar->_MEMVAR_PAR1 := cPar1
        Memvar->_MEMVAR_PAR2 := cPar2
        Memvar->_MEMVAR_PAR3 := cPar3
        Memvar->_MEMVAR_PAR4 := cPar4
        Memvar->_MEMVAR_PAR5 := cPar5
        Memvar->_MEMVAR_PAR6 := cPar6
        Memvar->_MEMVAR_PAR7 := cPar7
        Memvar->_MEMVAR_PAR8 := cPar8
        Memvar->_MEMVAR_PAR9 := cPar9

        *---------------------------------------------------------
        * Show the copyright.
        *---------------------------------------------------------

        if lCopyright

            qqout( PROGRAM_COPYRIGHT_START )

            *-----------------------------------------------------
            * Save actual cursor position bacause it will
            * be restored at the end of program.
            *-----------------------------------------------------

            nOldRow := row()
            nOldCol := 0

        end

        *---------------------------------------------------------
        * Copyright notice is saved with savescreen().
        *---------------------------------------------------------

        cOldScreen      := savescreen()

        *---------------------------------------------------------
        * nB default SET.
        *---------------------------------------------------------

        *---------------------------------------------------------
        * Activate Mouse support, but hide the cursor.
        *---------------------------------------------------------

        mInit()
        setMouse(.F.)

        *---------------------------------------------------------
        * Find the NB.HLP file and save it into the static
        * variable <nBHelp>.
        *---------------------------------------------------------

        nBHelp :=;
            strFileFind(;
                "NB.HLP", "\" + curdir() + ";" + getenv( "PATH" );
            )

        *---------------------------------------------------------
        * Date for the next millenium.
        *---------------------------------------------------------

        set( _SET_DATEFORMAT, "dd/mm/yyyy" )

        *---------------------------------------------------------
        * Only one output device at a time.
        *---------------------------------------------------------

        setOutput( "CON" )
        * set( _SET_CONSOLE, "ON" )
        * set( _SET_PRINTER, "OFF" )
        * set( _SET_ALTERNATE, .F.)
        * set( _SET_ALTFILE, "CON", .T. )

        *---------------------------------------------------------
        * No default directory path.
        *---------------------------------------------------------

        set( _SET_DEFAULT, "")

        *---------------------------------------------------------
        * Show deleted records.
        *---------------------------------------------------------

        set( _SET_DELETED, "OFF" )

        *---------------------------------------------------------
        * Do not exit automatically form a get.
        *---------------------------------------------------------

        set( _SET_CONFIRM, "ON" )

        *---------------------------------------------------------
        * No scoreborad (other infoline is supplied).
        *---------------------------------------------------------

        set( _SET_SCOREBOARD, "OFF" )

        *---------------------------------------------------------
        * No exit auto exit with up/down arrow.
        *---------------------------------------------------------

        set( _SET_EXIT, "OFF" )

        *---------------------------------------------------------
        * Insert mode is normal.
        *---------------------------------------------------------

        set( _SET_INSERT, "ON" )

        *---------------------------------------------------------
        * Displayed decimal digits.
        * It is necessary also when _SET_FIXED is Off.
        *---------------------------------------------------------

        set( _SET_DECIMALS, 8 )

        *---------------------------------------------------------
        * Color.
        *---------------------------------------------------------

        setcolor( COLOR_BASE )

        *---------------------------------------------------------
        * Replaceable database driver.
        *---------------------------------------------------------

        rddsetdefault( _DEFAULT_RDD )

        *---------------------------------------------------------
        * Rpt() do not Ejects sheets
        * (contiuous paper is supposed to be there).
        *---------------------------------------------------------
        
        setRptEject( .F. )

        *---------------------------------------------------------
        * nB default SET terminated.
        *---------------------------------------------------------

        *---------------------------------------------------------
        * Macro execution or dot command line activation
        *---------------------------------------------------------

        if !( empty( cPar0 ) )

            *-----------------------------------------------------
            * The function ex() will check the file
            * type and existance.
            *-----------------------------------------------------

            ex( cPar0 )

        else

            *-----------------------------------------------------
            * Dot command line.
            *-----------------------------------------------------

            dot()

        end

        *---------------------------------------------------------
        * End of program.
        *---------------------------------------------------------

        dbcloseall()
        dbcommitall()
        scroll()
        setpos(0,0)

    recover

        *---------------------------------------------------------
        * Nothing will be made to recover errors.
        *---------------------------------------------------------

    end //sequence

    *-------------------------------------------------------------
    * Restore old values.
    *-------------------------------------------------------------

    errorblock(bOldErrorHandler)
    setcolor( cOldColor )
    restscreen( NIL, NIL, NIL, NIL, cOldScreen )
    setpos( nOldRow, nOldCol )

    *-------------------------------------------------------------
    * Terminate mouse support.
    *-------------------------------------------------------------
    
    mEnd()

    return NIL

#endif //LINK

#ifndef RUNTIME
*-----------------------------------------------------------------
* DOT()
*-----------------------------------------------------------------

#define DOT_MESSAGE_ERROR               "Error"

#define DOT_PROMPT_EXIT                 "Exit nB ?"

#define DOT_LIST_WINDOW_TOP;
   "Dot line history"

*-----------------------------------------------------------------
static function dot()
*&
*
* Dot() --> NIL
*
* Interactive "dot" command line. Only functions and calculations
* allowed.
*

    local aoGet         := {}
    local aButtons      := {}

    local bOld_F1       :=;
        setkey( K_F1, {|| nbHelp( "<THE DOT LINE>", , chr(K_ENTER) ) } )
    local bOld_F3       :=;
        setkey( K_F3, { || Text( dbiAllStatus() ) } )
    local bOld_F5       :=;
        setkey( K_F5, { || setAlternate()} )
    local bOld_F8       :=;
        setkey( K_F8, { || dotLine()} )

    local bOldErrorHandler
    local cOldScreen    := mouseScrSave()
    local lOldReadexit  := set( _SET_EXIT, .T. )

    local lMore         := .T.
    local cCommand      := space(_MAX_STRING_LEN)
    local lError
    local acCommand     := {}
    local nMax          := 0
    local nInd          := 0

    *-------------------------------------------------------------
    * clear screen
    *-------------------------------------------------------------

    scroll()

    *-------------------------------------------------------------
    * Dot loop.
    *-------------------------------------------------------------

    while .T.                                           // FOREVER

        *---------------------------------------------------------
        * Show the status line.
        *---------------------------------------------------------

        statusLine()

        *---------------------------------------------------------
        * Clear last 2 lines on the screen.
        *---------------------------------------------------------

        scroll( maxrow()-1,00,maxrow(),maxcol() )
        setpos( maxrow(), 00 )

        get(;
            @aoGet, maxrow()-1, 00,;
            { |x| iif( pcount() > 0, cCommand := x, cCommand ) },;
            "@s"+ltrim(str(maxcol()+1))+"@",;
            ,;
            {||;
                truesetkey(;
                    K_F10,;
                    { || set( _SET_EXIT, .F. ), assist(), statusLine(), set( _SET_EXIT, .T. ) };
                    );
                .and.;
                truesetkey(;
                    K_ALT_M,;
                    { || set( _SET_EXIT, .F. ), assist(), statusLine(), set( _SET_EXIT, .T. ) };
                    );
                .and.;
                truesetkey(;
                    K_F2,;
                    { || cCommand :=;
                        padr(;
                            dotList( acCommand ),;
                            _MAX_STRING_LEN;
                            );
                        };
                    );
                };
            )

        button(;
            @aButtons, maxrow(), 00,;
            _BUTTON_ESC_EXIT,;
            ,;
            {|| keyboard( chr(K_ESC) ) };
            )

        button(;
            @aButtons, maxrow(), col()+1,;
            _BUTTON_F1_HELP,;
            ,;
            {|| eval( setkey(K_F1) ) };
            )

        button(;
            @aButtons, maxrow(), col()+1,;
            _BUTTON_F2_LIST,;
            ,;
            {|| eval( setkey(K_F2) ) };
            )

        button(;
            @aButtons, maxrow(), col()+1,;
            _BUTTON_ENTER_EXEC,;
            ,;
            {|| keyboard( chr(K_ENTER) ) };
            )

        button(;
            @aButtons, maxrow(), col()+1,;
            _BUTTON_UP_PREVIOUS,;
            ,;
            {|| keyboard( chr(K_UP) ) };
            )

        button(;
            @aButtons, maxrow(), col()+1,;
            _BUTTON_DOWN_NEXT,;
            ,;
            {|| keyboard( chr(K_DOWN) ) };
            )

        button(;
            @aButtons, maxrow(), col()+1,;
            _BUTTON_F10_MENU,;
            ,;
            {|| eval( setkey(K_F10) ) };
            )

        *---------------------------------------------------------
        * Read.
        *---------------------------------------------------------

        read( aoGet, , aButtons )
        aoGet    := {}
        aButtons := {}

        *---------------------------------------------------------
        * Test exit key.
        *---------------------------------------------------------

        do case
        case lastkey() = K_UP

            *-----------------------------------------------------
            * Previous.
            *-----------------------------------------------------

            if nInd > 1

                if nInd > nMax

                    nInd := nMax

                else

                    nInd--

                end

                cCommand := padr( acCommand[nInd], _MAX_STRING_LEN )

            end

        case lastkey() = K_DOWN

            *-----------------------------------------------------
            * Next.
            *-----------------------------------------------------

            do case
            case nInd < nMax

                nInd++

                cCommand := padr( acCommand[nInd], _MAX_STRING_LEN )

            case nInd >= nMax

                cCommand := space(_MAX_STRING_LEN)

                nInd := nMax +1

            end

        case lastkey() = K_ESC

            *-----------------------------------------------------
            * Exit.
            *-----------------------------------------------------

            if  alertBox(;
                    DOT_PROMPT_EXIT,;
                    { _MENU_YES, _MENU_NO };
                ) == 1

                exit                                    // EXIT

            end

        case lastkey() = K_ENTER

            *-----------------------------------------------------
            * Prepare <lError> False.
            *-----------------------------------------------------

            lError := .F.
            
            *-----------------------------------------------------
            * Execute.
            *-----------------------------------------------------

            cCommand := alltrim( cCommand )

            if  !empty( cCommand )

                *-------------------------------------------------
                * Scroll Screen up 2 lines.
                *-------------------------------------------------

                scroll(00,00,maxrow()-2,maxcol(), +2)

                setpos( maxrow()-3, 00 )

                *-------------------------------------------------
                * Display the command to execute.
                *-------------------------------------------------

                dispout( cCommand )

                *-------------------------------------------------
                * Prepare to recover a possible error.
                *-------------------------------------------------

                bOldErrorHandler :=;
                    errorblock( {|e| ErrorHandler(e)} )
                begin sequence

                    *---------------------------------------------
                    * Prepare the cursor position where the
                    * result should be displayed.
                    *---------------------------------------------

                    setpos( maxrow()-2, 00 )

                    *---------------------------------------------
                    * Set exit with [Up] and [Down] OFF (normal)
                    *---------------------------------------------

                    set( _SET_EXIT, .F. )

                    *---------------------------------------------
                    * Execute the command displaying the result.
                    *---------------------------------------------

                    dispout( &(cCommand) )

                    *---------------------------------------------
                    * Restore the set exit for the normal use
                    * with Dot().
                    *---------------------------------------------

                    set( _SET_EXIT, .T. )

                    *---------------------------------------------
                    * Macro recorder.
                    *---------------------------------------------

                    if  isMemvar( _MEMVAR_MACRO_RECORDING_NAME )

                        Memvar->_MEMVAR_MACRO_RECORDER +=;
                            NL(1) +;
                            alltrim( cCommand ) +;
                            NL(1)

                    end

                    *---------------------------------------------
                    * Array increment or swap.
                    *---------------------------------------------

                    nInd := ascan( acCommand, cCommand )

                    if nInd == 0

                        aadd( acCommand, cCommand )

                    else

                        adel( acCommand, nInd )

                        acCommand[nMax] := cCommand

                    end

                recover

                    *---------------------------------------------
                    * Error recovery.
                    *---------------------------------------------

                    setpos( maxrow()-2, 00 )

                    dispout( DOT_MESSAGE_ERROR )

                    lError := .T.

                end
                errorblock(bOldErrorHandler)

                *-------------------------------------------------
                * Array pointer update.
                *-------------------------------------------------

                nMax := len( acCommand )

                nInd := nMax +1

                *-------------------------------------------------
                * cCommand prepare.
                *-------------------------------------------------

                if lError
                
                    cCommand := padr( cCommand, _MAX_STRING_LEN )
                    
                else
                
                    cCommand := space(_MAX_STRING_LEN)
                    
                end

            else

                cCommand := space(_MAX_STRING_LEN)

            end

        otherwise

            *-----------------------------------------------------
            * Exit with another key is not allowed: loop again.
            *-----------------------------------------------------

        end

    end

    *-------------------------------------------------------------
    * Restore old data.
    *-------------------------------------------------------------

    setkey( K_F1, bOld_F1 )
    setkey( K_F3, bOld_F3 )
    setkey( K_F5, bOld_F5 )
    setkey( K_F8, bOld_F8 )
    set( _SET_EXIT, lOldReadexit )

    mouseScrRestore( NIL, NIL, NIL, NIL, cOldScreen )

    return NIL

*-----------------------------------------------------------------
static function dotList( acCommand )
*

    local cOldScreen
    local cOldColor     := setcolor()
    local nOldCursor    := setcursor( SETCURSOR_NONE )
    local nOldRow       := row()
    local nOldCol       := col()
    local bOld_F1       :=;
        setkey( K_F1, {|| nbHelp( "<THE DOT LINE>", , chr(K_ENTER) ) } )
    local bOld_F2       := setkey( K_F2, NIL )
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth
    local nHeight

    local nI            := 0

    local cReturn       := ""

    begin sequence

        *---------------------------------------------------------
        * Create a kind of window.
        *---------------------------------------------------------

        nTop    := 1
        nLeft   := 0
        nBottom := maxrow()
        nRight  := maxcol()
        nWidth  := nRight - nLeft +1
        nHeight := nBottom - nTop +1

        if nHeight > len( acCommand ) +5

            nTop := nBottom - len( acCommand ) -4

        end

        cOldScreen  := mouseScrSave( nTop, nLeft, nBottom, nRight )

        setcolor( COLOR_BODY )

        scroll( nTop, nLeft, nBottom, nRight )

        dispBoxShadow(;
            nTop, nLeft, nBottom, nRight,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )

        say(;
            nTop+1, nLeft+1,;
            padc( DOT_LIST_WINDOW_TOP, nWidth-2 ),;
            ,;
            COLOR_HEAD;
        )

        *---------------------------------------------------------
        * achoice.
        *---------------------------------------------------------

        * aChoice(;
        *    [<nTop>], [<nLeft>], [<nBottom>], [<nRight>],;
        *    <acMenuItems>, [<alSelectable>], [<nFirst>],;
        *    [<lButtons> | <aButtons>];
        * ) --> nPosition

        nI :=;
            achoice(;
                nTop+2, nLeft+1, nBottom-1, nRight-1,;
                acCommand, , 1,;
                .T.;
            )

        do case
        case nI == 0

            *-----------------------------------------------------
            * [Esc] - do nothing.
            *-----------------------------------------------------

        case nI > 0

            *-----------------------------------------------------
            * [Enter] - select the command.
            *-----------------------------------------------------

            cReturn := acCommand[nI]

        otherwise

            *-----------------------------------------------------
            * Exit with any other key is equivalent to [Esc].
            *-----------------------------------------------------

        end

        *---------------------------------------------------------
        * Restore the screen.
        *---------------------------------------------------------

        mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )


    end //sequence

    *-------------------------------------------------------------
    * Restore old data.
    *-------------------------------------------------------------

    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )
    setcursor( nOldCursor )
    setcolor( cOldColor )

    setpos( nOldRow, nOldCol )

    *-------------------------------------------------------------
    * Return the selected string.
    *-------------------------------------------------------------

    return cReturn

#endif  // RUNTIME

#ifndef RUNTIME
*-----------------------------------------------------------------
* FRM - FORM REPORT CREATION AND EDITING
*-----------------------------------------------------------------


#define FRM_DIALOG_BOX_TOP_REPORT;
    "REPORT FORM ..."

#define FRM_PROMPT_COMPRESSED_PRINT;
    "Compressed Print ? (Y/T/N/F) "
    
#define FRM_PROMPT_OPEN_FILE_REPORT;
    "Report file name:"
    
#define FRM_PROMPT_WHILE_EXPRESSION;
    "WHILE expression condition:"
    
#define FRM_PROMPT_FOR_EXPRESSION;
    "FOR expression condition:"
    
#define FRM_PROMPT_NEW_FILE_NAME;
    "Insert new name."
    
*-----------------------------------------------------------------
static function frmPrint()
*&
*
* frmPrint() --> NIL
*
* This function asks for the label file name, the
* FOR and WHILE condition.
*

    local aoGet                 := {}
    local aButtons              := {}
    local aTab                  := TAB_DEFAULT

    local bOldErrorHandler
    local cOldScreen
    local cOldColor             := setcolor()
    local nOldCursor            := setcursor( SETCURSOR_NORMAL )
    local bOld_F1               :=;
        setkey( K_F1, {|| nbHelp( "<REPORT - REPORT FORM>", , chr(K_ENTER) ) } )
    local bOld_F2               := setkey( K_F2, NIL )
    local nOldRow               := row()
    local nOldCol               := col()
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth

    local lGoPrint              := .T.
    local cReportFile           :=;
        padr( "*." + _EXTENTION_FORM, _MAX_STRING_LEN )
    local cWhileCondition       := padr( ".T.", _MAX_STRING_LEN )
    local cForCondition         := padr( ".T.", _MAX_STRING_LEN )

    local bWhileCondition
    local bForCondition

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Verify if an active Alias is present.
        *---------------------------------------------------------

        if ( alias() == "" )

            alertBox( _ERROR_NO_ALIAS )

            break                                       // BREAK

        end

        *---------------------------------------------------------
        * Create a kind of window.
        *---------------------------------------------------------

        nBottom             := maxrow()
        nTop                := nBottom - 8
        nLeft               := 0
        nRight              := maxcol()
        nWidth              := nRight - nLeft +1

        aTab[TAB_LEFT]      := nLeft
        aTab[TAB_RIGHT]     := nRight
        aTab[TAB_TAB_ARRAY] := { 32, 4 }
        
        cOldScreen          :=;
            mouseScrSave( nTop, nLeft, nBottom, nRight )

        setcolor( COLOR_BODY )

        scroll( nTop, nLeft, nBottom, nRight )

        dispBoxShadow(;
            nTop, nLeft, nBottom, nRight,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )

        say(;
            nTop+1, nLeft+1,;
            padc(;
                FRM_DIALOG_BOX_TOP_REPORT,;
                nWidth-2;
                ),;
            ,;
            COLOR_HEAD;
        )

        *---------------------------------------------------------
        * Do editing.
        *---------------------------------------------------------

        while .t.                                   // FOREVER

            say(;
                nTop+3, nLeft+1,;
                FRM_PROMPT_OPEN_FILE_REPORT;
            )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cReportFile := x, cReportFile ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { || trueSetkey( K_F2, {|| gvFileDir( @cReportFile ) } ) },;
                { || gvFileExist( @cReportFile, _EXTENTION_FORM ) };
            )

            say(;
                row()+1, nLeft+1,;
                FRM_PROMPT_WHILE_EXPRESSION;
            )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cWhileCondition := x, cWhileCondition ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { || trueSetkey( K_F2, {|| gvAdd( @cWhileCondition, fldNormal() ) } ) },;
                { || gvDefault( @cWhileCondition, ".T." ) };
            )

            say(;
                row()+1, nLeft+1,;
                FRM_PROMPT_FOR_EXPRESSION;
            )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cForCondition := x, cForCondition ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { || trueSetkey( K_F2, {|| gvAdd( @cForCondition, fldNormal() ) } ) },;
                { || gvDefault( @cForCondition, ".T." ) };
            )

            button( @aButtons, row()+2, nLeft+1,;
                _BUTTON_ESC_CANCEL, , {||keyboard(chr(K_ESC))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F1_HELP, , {|| eval(setkey(K_F1))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F2_LIST, , {|| iif( valtype(setkey(K_F2)) == "B", eval(setkey(K_F2)), NIL ) } )
            button( @aButtons, row(), col()+1,;
                _BUTTON_PGDN_CONFIRM, , {||keyboard(chr(K_PGDN))} )

            *-----------------------------------------------------
            * Read.
            *-----------------------------------------------------

            read( aoGet, , aButtons )
            aoGet   := {}
            aButtons := {}

            *-----------------------------------------------------
            * Check what was read.
            *-----------------------------------------------------

            do case
            case lastkey() = K_ESC

                *-------------------------------------------------
                * [Esc] means leave.
                *-------------------------------------------------

                lGoPrint := .F.

                exit                                    // EXIT

            case lastkey() = K_PGDN

                *-------------------------------------------------
                * [PgDn] means confirm.
                * Before, check for valid data.
                *-------------------------------------------------

                if  isFile(;
                        strAddExtention(;
                            cReportFile,;
                            _EXTENTION_FORM;
                        );
                    )                               .and.;
                    !empty( cWhileCondition )       .and.;
                    !empty( cForCondition )

                    *---------------------------------------------
                    * Data is valid: continue.
                    *---------------------------------------------

                    lGoPrint := .T.

                    *---------------------------------------------
                    * Transform character conditions into code
                    * blocks.
                    *---------------------------------------------

                    bWhileCondition     :=;
                        { || &(cWhileCondition) }

                    bForCondition :=;
                        { || &(cForCondition) .and. waitFileEval() }

                    exit                                // EXIT

                else

                    *---------------------------------------------
                    * As data is not valid, loop again.
                    *---------------------------------------------

                end

            otherwise

                *-------------------------------------------------
                * Exit with different keys is not allowed.
                * Loop again.
                *-------------------------------------------------

            end

        end

        *---------------------------------------------------------
        * The editing is terminated.
        * Close window.
        *---------------------------------------------------------

        mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

        *---------------------------------------------------------
        * Continue if so instructed.
        *---------------------------------------------------------

        if lGoPrint

            *-----------------------------------------------------
            * As a file process is going to take place, prepare
            * to handle the possible error.
            *-----------------------------------------------------

            bOldErrorHandler := errorblock( {|e| ErrorHandler(e)} )
            begin sequence

                *-------------------------------------------------
                * Check if the print goes on console.
                *-------------------------------------------------

                if  isConsoleOn()

                    *---------------------------------------------
                    * Print goes on the screen: clear it.
                    *---------------------------------------------

                    setcolor( COLOR_BODY )

                    scroll()

                end

                *-------------------------------------------------
                * Print
                *-------------------------------------------------

                rf(;
                    alltrim(cReportFile),;
                    bForCondition,;
                    bWhileCondition;
                )

                *-------------------------------------------------
                * Close the wait bar if still present.
                *-------------------------------------------------

                waitFileEval( .T. )

                *-------------------------------------------------
                * Wait if console is ON.
                *-------------------------------------------------

                if  isConsoleOn()

                    *---------------------------------------------
                    * Wait for a key press.
                    *---------------------------------------------

                    alertbuttons( maxrow(), 0, {"[Esc]"} )

                    scroll()

                end

                *-------------------------------------------------
                * The record pointer is maybe at bottom.
                * No action is taken to move the record pointer.
                * dbgotop()
                *-------------------------------------------------

                *-------------------------------------------------
                * Macro recorder.
                *-------------------------------------------------

                if  isMemvar( _MEMVAR_MACRO_RECORDING_NAME )

                    Memvar->_MEMVAR_MACRO_RECORDER +=;
                        NL(1) +;
                        '// Report print.' +;
                        NL(1) +;
                        '// This is not a CA-Clipper ' +;
                        'standard function.' +;
                        NL(1) +;
                        'rf( "' +;
                        alltrim(cReportFile) +;
                        '", {|| ' +;
                        alltrim(cForCondition) +;
                        ' }, {|| ' +;
                        alltrim(cWhileCondition) +;
                        '} )' +;
                        NL(1)

                end

            recover

                *-------------------------------------------------
                * Try to recover.
                *-------------------------------------------------

                dbgotop()

            end //sequence
            errorblock(bOldErrorHandler)

            waitFileEval( .T. )

        end

    end //sequence

    *-------------------------------------------------------------
    * Restore old data.
    *-------------------------------------------------------------

    setcursor( nOldCursor )
    setcolor( cOldColor )
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )

    return NIL

*----------------------------------------------------------------
static function frmModify()
*
* frmModify() --> NIL
*
* It executes the frm() function asking for the file name to
* modify.
*

    local aoGet                 := {}
    local aButtons              := {}
    local aTab                  := TAB_DEFAULT

    local bOldErrorHandler
    local cOldScreen
    local cOldColor             := setcolor()
    local nOldCursor            := setcursor( SETCURSOR_NORMAL )
    local bOld_F1               :=;
        setkey( K_F1, {|| nbHelp( "<REPORT - MODIFY REPORT>", , chr(K_ENTER) ) } )
    local bOld_F2               := setkey( K_F2, NIL )
    local nOldRow               := row()
    local nOldCol               := col()
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth

    local lGoOn                 := .F.
    local cReportFile           :=;
        padr( "*." + _EXTENTION_FORM, _MAX_STRING_LEN )

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Create a kind of window.
        *---------------------------------------------------------

        nBottom             := maxrow()
        nTop                := nBottom - 6
        nLeft               := 0
        nRight              := maxcol()
        nWidth              := nRight - nLeft +1

        aTab[TAB_LEFT]      := nLeft
        aTab[TAB_RIGHT]     := nRight
        aTab[TAB_TAB_ARRAY] := { 16, 4 }
        
        cOldScreen          :=;
            mouseScrSave( nTop, nLeft, nBottom, nRight )

        setcolor( COLOR_BODY )

        scroll( nTop, nLeft, nBottom, nRight )

        dispBoxShadow(;
            nTop, nLeft, nBottom, nRight,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )

        say(;
            nTop+1, nLeft+1,;
            padc(;
                FRM_DIALOG_BOX_TOP_REPORT,;
                nWidth-2;
                ),;
            ,;
            COLOR_HEAD;
        )

        *---------------------------------------------------------
        * Do editing.
        *---------------------------------------------------------

        while .t.                                       // FOREVER

            say(;
                nTop+3, nLeft+1,;
                FRM_PROMPT_OPEN_FILE_REPORT;
            )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cReportFile := x, cReportFile ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { || trueSetkey( K_F2, {|| gvFileDir( @cReportFile ) } ) },;
                { || gvFileExist( @cReportFile, _EXTENTION_FORM ) };
            )

            button( @aButtons, row()+2, nLeft+1,;
                _BUTTON_ESC_CANCEL, , {||keyboard(chr(K_ESC))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F1_HELP, , {|| eval(setkey(K_F1))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F2_LIST, , {|| iif( valtype(setkey(K_F2)) == "B", eval(setkey(K_F2)), NIL ) } )
            button( @aButtons, row(), col()+1,;
                _BUTTON_PGDN_CONFIRM, , {||keyboard(chr(K_PGDN))} )

            *-----------------------------------------------------
            * Read.
            *-----------------------------------------------------

            read( aoGet, , aButtons )
            aoGet   := {}
            aButtons := {}

            *-----------------------------------------------------
            * Check what was read.
            *-----------------------------------------------------

            do case
            case lastkey() = K_ESC

                *-------------------------------------------------
                * [Esc] means leave.
                *-------------------------------------------------

                lGoOn := .F.

                exit                                    // EXIT

            case lastkey() = K_PGDN
            
                *-------------------------------------------------
                * [PgDn] means confirm.
                * Before, check for valid data.
                *-------------------------------------------------
            
                if  isFile(;
                        strAddExtention(;
                            cReportFile,;
                            _EXTENTION_FORM;
                        );
                    )

                    lGoOn := .T.

                    exit                                // EXIT

                else

                    *---------------------------------------------
                    * As data is not valid, loop again.
                    *---------------------------------------------

                end

            otherwise

                *-------------------------------------------------
                * Exit with different keys is not allowed.
                * Loop again.
                *-------------------------------------------------

            end

        end

        *---------------------------------------------------------
        * The editing is terminated.
        * Close window.
        *---------------------------------------------------------

        mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

        *---------------------------------------------------------
        * Continue if so instructed.
        *---------------------------------------------------------

        if lGoOn

            frm( cReportFile )

        end

    end //sequence

    *-------------------------------------------------------------
    * Restore old data.
    *-------------------------------------------------------------

    setcursor( nOldCursor )
    setcolor( cOldColor )
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )

    return NIL

#endif  // RUNTIME

#ifndef RUNTIME
*-----------------------------------------------------------------
* IDB - INTERACTIVE DB FUNCTIONS
*-----------------------------------------------------------------
#define IDB_DIALOG_BOX_TOP_NTX_NEW;
    "New Index"
#define IDB_DIALOG_BOX_TOP_OPEN_DBF;
    "Open ." + _EXTENTION_DBF
#define IDB_DIALOG_BOX_TOP_OPEN_NTX;
    "Open ." + _EXTENTION_INDEXBAG
#define IDB_DIALOG_BOX_TOP_REPLACE;
    "Replace"
#define IDB_DIALOG_BOX_TOP_RECALL;
    "Recall"
#define IDB_DIALOG_BOX_TOP_DELETE;
    "Delete"
#define IDB_DIALOG_BOX_TOP_RELATION_DEFINITION;
    "Relation definition"
#define IDB_DIALOG_BOX_DBF_FILE;
    "." + _EXTENTION_DBF + " File"

#define IDB_WINDOW_TOP_DBF_STRUCTURE;
    "Database file structure"

#define IDB_WAIT_PACK;
    "Removing deleted records..."

#define IDB_PROMPT_MODIFY_STRUCTURE;
    "Inproper structure modification;" +;
    "may result in a loss of data."
#define IDB_PROMPT_OPEN_FILE_TO_MODIFY;
    "." + _EXTENTION_DBF + " file to be modifyed:"
#define IDB_PROMPT_DBF_CREATE_LAST_STRUCTURE;
    "Do you want to use last structure?"
#define IDB_PROMPT_DBF_MODIFY_STRUCTURE;
    "Confirm structure modification?"
#define IDB_PROMPT_DBF_REPLACE;
    "Confirm replacement on all records?"
#define IDB_PROMPT_DBF_RECALL;
    "Confirm deleted record recall?"
#define IDB_PROMPT_DBF_DELETE;
    "Confirm record deletion?"
#define IDB_PROMPT_DBF_PACK;
    "Confirm file pack?"
#define IDB_PROMPT_NEW_NTX_FILENAME;
    "Index file name to create:"
#define IDB_PROMPT_TAG_NAME;
    "Order name (Tag) to be created:"
#define IDB_PROMPT_NEW_NTX_KEYEXPR;
    "Key expression:"
#define IDB_PROMPT_OPEN_NTX_FILENAME;
    "Insert the index file name to open:"
#define IDB_PROMPT_WHILE_EXPRESSION;
    "WHILE expression:"
#define IDB_PROMPT_FOR_EXPRESSION;
    "FOR expression:"
#define IDB_PROMPT_OPEN_DBF_FILENAME;
    "Filename:"
#define IDB_PROMPT_OPEN_DBF_ALIASNAME;
    "Alias:"
#define IDB_PROMPT_OPEN_DBF_RDDNAME;
    "RDD:"
#define IDB_PROMPT_OPEN_DBF_SHARED;
    "Shared? (T/F):"
#define IDB_PROMPT_OPEN_DBF_READ_ONLY;
    "Read Only? (T/F):"
#define IDB_PROMPT_REPLACE_FIELD;
    "Field to replace:"
#define IDB_PROMPT_REPLACE_NEW_VALUE;
    "New value expression:"
#define IDB_PROMPT_TEMP_DELETE;
    "Do you want to delete the temporary file " + ;
    "corresponding to the old one before the modification?"
#define IDB_PROMPT_RELATION_DEFINITION_CHILDNAME;
    "Child Alias:"
#define IDB_PROMPT_RELATION_DEFINITION_EXPRESSION;
    "Relation expression:"
#define IDB_PROMPT_NEW_DBF_SAVE;
    "." + _EXTENTION_DBF + " file save; insert the name:"
#define IDB_PROMPT_NEW_RDD_SAVE;
    "Replaceable Database Driver (rdd):"

#define IDB_ERROR_FILE_NOT_EXIST;
    "The file do not exists."
#define IDB_ERROR_RELATION;
    "Relation creation error!"
#define IDB_ERROR_FIELD_TYPE;
    "Valid values are C, N, D, L and M"
#define IDB_ERROR_FIELD_NUMERIC_TOO_LITTE;
    "A numeric field must have a lenght greater then 0"
#define IDB_ERROR_FIELD_DECIMAL_TOO_BIG;
    "Decimal lenght may be 0 or less than integer lenght -1"

#define IDB_HELP_OPEN_FILE_TO_MODIFY;
    "OPEN A FILE AND MODIFY THE STRUCTURE." +;
    NL(3) +;
    "Insert the file name that you intend to modify."

#define IDB_HELP_SAVE_CREATED_MODIFYED_FILE;
    "SAVE A " + _EXTENTION_DBF + " FILE." +;
    NL(3) +;
    "Insert the name and the database driver " +;
    "to use to save the file."

#define IDB_HELP_DBF_STRUCTURE_BODY;
    "ORDER - The column order may be changed modifying the " +;
    "serial number on the left. This information is " +;
    "temporary, that is, it is not part of the database file." +;
    NL(2) +;
    "FIELD NAME - The column name may be long " +;
    "10 character max, it " +;
    "must begin with a letter and may contains also " +;
    "numbers and '_'; the space ' ' is not allowed." +;
    NL(2) +;
    "TYPE - The column type may be 'C' for character, 'N' for " +;
    "numeric, 'D' for data, 'L' for logic and 'M' for " +;
    "memo." +;
    NL(2) +;
    "LENGTH - The length is the column width in character." +;
    NL(2) +;
    "DECIMAL - Is the decimal portion of LENGTH when the " +;
    "column type identifies a number. The maximum decimal " +;
    "dimention is LENGTH -2 as the decimal point takes one " +;
    "place." +;
    NL(3) +;
    "The maximum column (field) width may be different " +;
    "depending on the active database driver." + NL(1) +;
    "For DBFNTX, the standard, the maximum dimentions are " +;
    "as follows:" +;
    NL(2) +;
    "C - Character - max length  = 255" + NL(1) +;
    "                max decimal = 250" + NL(1) +;
    "    Width = LENGTH + (DECIMAL * 256)" + NL(1) +;
    "    Max Width = 64255" +;
    NL(2) +;
    "N - Numeric - max length  = 16" + NL(1) +;
    "              max decimal = LENGTH-2" + NL(1) +;
    "    Width = LENGTH" +;
    NL(2) +;
    "D - Data - length  = leave 0" + NL(1) +;
    "           decimal = leave 0" +;
    NL(2) +;
    "L - Logic - length  = leave 0" + NL(1) +;
    "            decimal = leave 0" +;
    NL(2) +;
    "M - Memo - length  = leave 0" + NL(1) +;
    "           decimal = leave 0" +;
    NL(3) +;
    "After completing the column (field) description, " +;
    "press [Esc] to exit: the name to use to save " +;
    "the new structure will be asked."

#define IDB_HELP_DBF_CREATE;
    "." + _EXTENTION_DBF + " CREATION" +;
    NL(2) +;
    "This function permits to create a new database file." +;
    NL(3) +;
    IDB_HELP_DBF_STRUCTURE_BODY

#define IDB_HELP_DBF_MODIFY_STRUCTURE;
    "." + _EXTENTION_DBF + " STRUCTURE MODIFICATION" +;
    NL(2) +;
    "Only the content of columns with the same name " +;
    "are saved; the others are new empty columns." +;
    NL(3) +;
    IDB_HELP_DBF_STRUCTURE_BODY

#define IDB_HELP_NTX_CREATE;
    "INDEX/TAG CREATION" +;
    NL(2) +;
    "Create an index file (Order Bag) or add a Tag to " +;
    "an Order Bag." +;
    NL(2) +;
    "Depending on the active Database Driver, the following " +;
    "may be asked." +;
    NL(2) +;
    "- The index file (Order Bag) name." + NL(1) +;
    "- The key expression." + NL(1) +;
    "- The FOR condition."  + NL(1) +;
    "- The Tag name." +;
    _STDHLP_START +;
    _STDHLP_EVER_TRUE +;
    _STDHLP_FOR_CONDITION

#define IDB_HELP_SET_INDEX;
    "." + _EXTENTION_INDEXBAG + " FILE OPEN" +;
    NL(2) +;
    "It activates an index previously created."

#define IDB_HELP_DBF_OPEN;
    "." + _EXTENTION_DBF + " FILE OPEN" +;
    NL(2) +;
    "Open (use) another ." +;
    _EXTENTION_DBF + " file. " +;
    NL(2) +;
    "FILENAME - The complete file " +;
    "pathname of what is going to be opened." +;
    NL(2) +;
    "ALIAS - The name to be used as Alias." +;
    NL(2) +;
    "RDD - The Replaceable Database Driver name to be used." +;
    NL(2) +;
    "SHARED - When 'Shared' is set to true (T), the file is " +;
    "opened in shared mode." +;
    NL(2) +;
    "READ ONLY - When true, " +;
    "it will not be possible any modification to " +;
    "the file." +;
    NL(2) +;
    "To open a file in EXCLUSIVE mode, set both SHARED " +;
    "and READ ONLY to false (.F.)."

#define IDB_HELP_DBF_REPLACE;
    "FIELD REPLACE" +;
    NL(2) +;
    "It permits to modify the content of a " +;
    "column. The replace can be controlled by the " +;
    "WHILE and FOR conditions." +;
    _STDHLP_START +;
    _STDHLP_EVER_TRUE +;
    _STDHLP_WHILE_CONDITION +;
    _STDHLP_FOR_CONDITION

#define IDB_HELP_DBF_RECALL;
    "DELETED RECORD RECALL" +;
    NL(2) +;
    "It permits to recall as many records as " +;
    "defined by the WHILE and FOR conditions." +;
    _STDHLP_START +;
    _STDHLP_EVER_TRUE +;
    _STDHLP_WHILE_CONDITION +;
    _STDHLP_FOR_CONDITION

#define IDB_HELP_DBF_DELETE;
    "DELETE RECORDS" +;
    NL(2) +;
    "It permits to delete as many records as " +;
    "defined by the WHILE and FOR conditions." +;
    _STDHLP_START +;
    _STDHLP_EVER_TRUE +;
    _STDHLP_WHILE_CONDITION +;
    _STDHLP_FOR_CONDITION

#define IDB_HELP_RELATION_DEFINITION;
    "RELATION DEFINITION" +;
    NL(2) +;
    "It makes a link between the active Alias and a 'child' " +;
    "one." +;
    NL(2) +;
    "Usually, the 'child' Alias is already associated " +;
    "to an index file." +;
    NL(2) +;
    "The expression defines the relation using elements " +;
    "from the active Alias."

*-----------------------------------------------------------------
static function idbModStructure()
*
* idbModStructure() --> NIL
*
* This function modifies a .dbf file structure asking for the name
* of the file to modify.
*

    local aoGet             := {}
    local aButtons          := {}
    local aTab              := TAB_DEFAULT

    local bOldErrorHandler
    local cOldScreen
    local cOldColor         := setcolor( COLOR_BASE )
    local nOldCursor        := setcursor( SETCURSOR_NORMAL )
    local bOld_F1           :=;
        setkey( K_F1, {|| nbHelp( "<FILE - MODIFY .DBF STRUCTURE>", , chr(K_ENTER) ) } )
    local bOld_F2           := setkey( K_F2, NIL )
    local nOldRow           := row()
    local nOldCol           := col()
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth

    local cFileName := padr( "*." + _EXTENTION_DBF, _MAX_STRING_LEN )
    local lGoOn := .F.

    *-------------------------------------------------------------
    * Start a sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Create a kind of window.
        *---------------------------------------------------------

        nBottom     := maxrow()
        nTop        := nBottom - 6
        nLeft       := 0
        nRight      := maxcol()
        nWidth      := nRight - nLeft +1
        
        aTab[TAB_LEFT]      := nLeft
        aTab[TAB_RIGHT]     := nRight
        aTab[TAB_TAB_ARRAY] := { 16, 4 }
        
        cOldScreen  := mouseScrSave( nTop, nLeft, nBottom, nRight )
        
        setcolor( COLOR_BODY )
        
        scroll( nTop, nLeft, nBottom, nRight )
        
        dispBoxShadow(;
            nTop, nLeft, nBottom, nRight,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )
        
        say(;
            nTop+1, nLeft+1,;
            padc( IDB_DIALOG_BOX_DBF_FILE, nWidth-2 ),;
            ,;
            COLOR_HEAD;
        )

        *---------------------------------------------------------
        * Do editing.
        *---------------------------------------------------------

        while .t.                                       // FOREVER

            say( nTop+3, nLeft+1, IDB_PROMPT_OPEN_FILE_TO_MODIFY )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cFileName := x, cFileName ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { ||;
                    trueSetkey( K_F2,;
                        { || gvFileDir( @cFileName ) };
                    );
                },;
                { || gvFileExist( @cFileName, _EXTENTION_DBF ) };
            )

            button( @aButtons, row()+2, nLeft+1,;
                _BUTTON_ESC_CANCEL, , {||keyboard(chr(K_ESC))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F1_HELP, , {|| eval(setkey(K_F1))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F2_LIST, , {|| iif( valtype(setkey(K_F2)) == "B", eval(setkey(K_F2)), NIL ) } )
            button( @aButtons, row(), col()+1,;
                _BUTTON_PGDN_CONFIRM, , {||keyboard(chr(K_PGDN))} )

            *-----------------------------------------------------
            * Read.
            *-----------------------------------------------------

            read( aoGet, , aButtons )
            aoGet   := {}
            aButtons := {}

            *-----------------------------------------------------
            * Check what was read.
            *-----------------------------------------------------

            do case
            case lastkey() = K_ESC

                *-------------------------------------------------
                * [Esc] means leave.
                *-------------------------------------------------

                lGoOn := .F.
                exit                                    // EXIT

            case lastkey() = K_PGDN

                *-------------------------------------------------
                * [PgDn] means confirm.
                * Before, check for valid data.
                *-------------------------------------------------

                if file(;
                        strAddExtention( cFileName,;
                        _EXTENTION_DBF );
                        )

                    lGoOn := .T.

                    exit                                // EXIT

                else

                    *---------------------------------------------
                    * The file don't exists. Loop again.
                    *---------------------------------------------

                end

            otherwise

                *-------------------------------------------------
                * Exit with different keys is not allowed.
                * Loop again.
                *-------------------------------------------------

            end

        end

        *---------------------------------------------------------
        * The editing is terminated.
        * Close the window and do as instructed.
        *---------------------------------------------------------

        mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

        if lGoOn

            *-----------------------------------------------------
            * The work is made outside.
            *-----------------------------------------------------

            idbStructure( cFileName )

        end

    end //sequence

    *-------------------------------------------------------------
    * Restore old values.
    *-------------------------------------------------------------

    setcursor( nOldCursor )
    setcolor( cOldColor )
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )

    return NIL

*-----------------------------------------------------------------
static function idbStructure(cName)
*&
*
* idbStructure(<cName>) --> NIL
*
* <cName>  database file name.
*
* It modifies or creates the file (.dbf) <cName>
*

    local bOld_F1             := setkey( K_F1 )
    local aComplex
    local aStruct
    local cNewName
    local cNewRdd
    local nI

    *-------------------------------------------------------------
    * Determinate what kind of help is needed.
    *-------------------------------------------------------------

    if empty( cName )
        setkey( K_F1, {|| nbHelp( "<FILE - NEW .DBF>", , chr(K_ENTER) ) } )
    else
        setkey( K_F1, {|| nbHelp( "<FILE - MODIFY .DBF STRUCTURE>", , chr(K_ENTER) ) } )
    end

    *-------------------------------------------------------------
    * The work is made from another function that returns all
    * informations for the macro recorder.
    *-------------------------------------------------------------

    aComplex := dbModifyStructure( cName )

    *-------------------------------------------------------------
    * If somethig was returned, the macro may be recorded.
    *-------------------------------------------------------------

    if !empty( aComplex )

        cNewName := aComplex[1]
        cNewRdd  := aComplex[2]
        aStruct  := aComplex[3]

        if cName == NIL

            *-----------------------------------------------------
            * It may be recorded only a file creation and not
            * a structure modification.
            * If <cName> contains NIL, a file was created.
            *-----------------------------------------------------

            if isMemvar( _MEMVAR_MACRO_RECORDING_NAME )

                Memvar->_MEMVAR_MACRO_RECORDER +=;
                NL(1) +;
                '// Create a db file.' +;
                NL(1) +;
                '// dbcreate( <cDatabase>, <aStruct>, [<cDriver>] ) --> NIL' +;
                NL(1) +;
                'dbcreate( "' + strCutExtention(cNewName) + '", ;' +;
                    NL(1) +;
                    '   { ;' +;
                    NL(1)

                for nI := 1 to len( aStruct )
                    Memvar->_MEMVAR_MACRO_RECORDER +=;
                        '      {"' +;
                        aStruct[nI][1] +;
                        '","' +;
                        aStruct[nI][2] +;
                        '",' +;
                        ltrim(str(aStruct[nI][3])) +;
                        ',' +;
                        ltrim(str(aStruct[nI][4])) +;
                        '}' +;
                        iif( nI == len(aStruct), '', ',' ) +;
                        ';' +;
                        NL(1)
                next

                Memvar->_MEMVAR_MACRO_RECORDER +=;
                    [   }, "] + alltrim(cNewRdd) + [" )] +;
                    NL(1)

            end

        end

    end

    *-------------------------------------------------------------
    * Restore old values and return.
    *-------------------------------------------------------------

    setkey( K_F1, bOld_F1 )

    return NIL

*-----------------------------------------------------------------
static function idbUseArea()
*&
*
* idbUseArea() --> NIL
*
* It Opens a (.dbf) file.
*

    local aoGet             := {}
    local aButtons          := {}
    local aTab              := TAB_DEFAULT
    local bOldErrorHandler
    local cOldScreen
    local cOldColor         := setcolor( COLOR_BASE )
    local nOldCursor        := setcursor( SETCURSOR_NORMAL )
    local nOldSelect        := select()
    local cOldRddDefault    := rddsetdefault()
    local bOld_F1           :=;
        setkey( K_F1, {|| nbHelp( "<FILE - OPEN .DBF>", , chr(K_ENTER) ) } )
    local bOld_F2           := setkey( K_F2, NIL )
    local nOldRow           := row()
    local nOldCol           := col()
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth

    local cDbfName          :=;
        padr( "*." + _EXTENTION_DBF, _MAX_STRING_LEN )
    local cAliasName        := space(10) // max alias name length
    local lShared           := .T.
    local lReadOnly         := .T.
    local cRdd
    local lGoOn

    *-------------------------------------------------------------
    * Start a sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Prepare the RDD name.
        *---------------------------------------------------------

        cRdd := padr( cOldRddDefault, _MAX_STRING_LEN )

        *---------------------------------------------------------
        * Create a kind of window.
        *---------------------------------------------------------

        nBottom             := maxrow()
        nTop                := nBottom - 10
        nLeft               := 0
        nRight              := maxcol()
        nWidth              := nRight - nLeft +1
        
        aTab[TAB_LEFT]      := nLeft
        aTab[TAB_RIGHT]     := nRight
        aTab[TAB_TAB_ARRAY] := { 20, 4 }

        cOldScreen          := mouseScrSave( nTop, nLeft, nBottom, nRight )

        setcolor( COLOR_BODY )

        scroll( nTop, nLeft, nBottom, nRight )

        dispBoxShadow(;
            nTop, nLeft, nBottom, nRight,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )

        say(;
            nTop+1, nLeft+1,;
            padc( IDB_DIALOG_BOX_TOP_OPEN_DBF, nWidth-2 ),,;
            COLOR_HEAD;
        )

        *---------------------------------------------------------
        * Do editing.
        *---------------------------------------------------------

        while .t.                                       // FOREVER

            setpos( nTop+2, nLeft+1 )

            say( row()+1, nLeft+1, IDB_PROMPT_OPEN_DBF_FILENAME )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cDbfName := x, cDbfName ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { ||; 
                    setkey( K_F2, {|| gvFileDir( @cDbfName ) } ),;
                    .T.; 
                },;
                { || gvFileExist( @cDbfName, _EXTENTION_DBF ) },;
            )
            
            say( row()+1, nLeft+1, IDB_PROMPT_OPEN_DBF_ALIASNAME )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cAliasName := x, cAliasName ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { ||;
                    setkey( K_F2, NIL ),;
                    gvDefault( @cAliasName, strCutExtention( strFile(cDbfName) ) ),;
                    .T.;
                },;
                { ||; 
                    !empty(cAliasName)                          .and.; 
                    select( alltrim( cAliasName ) ) == 0;
                };
            )
            
            say( row()+1, nLeft+1, IDB_PROMPT_OPEN_DBF_RDDNAME )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cRdd := x, cRdd ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { ||;
                    trueSetkey( K_F2, {|| gvSubst( @cRdd, dbRddSelect() ) } );
                },;
                { || !empty( cRdd ) };
            )
            
            say( row()+1, nLeft+1, IDB_PROMPT_OPEN_DBF_SHARED )
            tab( aTab )
            get( @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, lShared := x, lShared ) },;
                "@L",;
                ,;
                {|| truesetkey( K_F2, NIL ) };
            )
            tab( aTab )
            button( @aButtons, row(), col(),;
                _BUTTON_CHANGE, , {|| (lShared := !lShared), aoGet[4]:display() } )
            
            say( row()+1, nLeft+1, IDB_PROMPT_OPEN_DBF_READ_ONLY )
            tab( aTab )
            get( @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, lReadOnly := x, lReadOnly ) },;
                "@L",;
                ,;
                {|| truesetkey( K_F2, NIL ) };
            )
            tab( aTab )
            button( @aButtons, row(), col(),;
                _BUTTON_CHANGE, , {|| (lReadOnly := !lReadOnly), aoGet[5]:display() } )

            button( @aButtons, row()+2, nLeft+1,;
                _BUTTON_ESC_CANCEL, , {||keyboard(chr(K_ESC))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F1_HELP, , {|| eval(setkey(K_F1))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F2_LIST, , {|| iif( valtype(setkey(K_F2)) == "B", eval(setkey(K_F2)), NIL ) } )
            button( @aButtons, row(), col()+1,;
                _BUTTON_PGDN_CONFIRM, , {||keyboard(chr(K_PGDN))} )

            *-----------------------------------------------------
            * Read.
            *-----------------------------------------------------

            read( aoGet, 1, aButtons )
            aoGet    := {}
            aButtons := {}

            *-----------------------------------------------------
            * Test exit key.
            *-----------------------------------------------------

            do case
            case lastkey() = K_ESC

                *-------------------------------------------------
                * [Esc] means leave.
                *-------------------------------------------------

                lGoOn := .F.

                exit                                    // EXIT

            case lastkey() = K_PGDN

                *-------------------------------------------------
                * [PgDn] means confirm.
                * Before:
                * set default data;
                * check for valid data.
                *-------------------------------------------------
                
                gvDefault( @cAliasName, strCutExtention( strFile(cDbfName) ) )
                
                if  isFile(; 
                        strAddExtention( cDbfName,_EXTENTION_DBF ); 
                    );
                    .and.; 
                    !empty(cAliasName)          .and.;
                    select( alltrim(cAliasName) ) == 0

                    *---------------------------------------------
                    * Data is valid. Continue.
                    *---------------------------------------------

                    lGoOn := .T.

                    exit                                // EXIT

                else

                    *---------------------------------------------
                    * Data is not valid. Loop again.
                    *---------------------------------------------

                end

            otherwise

                *-------------------------------------------------
                * Exit with different keys is not allowed.
                * Loop again.
                *-------------------------------------------------

            end

        end

        *---------------------------------------------------------
        * Close window.
        *---------------------------------------------------------

        mouseScrRestore(;
            nTop, nLeft, nBottom, nRight, cOldScreen;
        )

        *---------------------------------------------------------
        * If so was requested, open the file.
        *---------------------------------------------------------

        if lGoOn

            *-----------------------------------------------------
            * Prepare for error handling, before file opening.
            *-----------------------------------------------------

            bOldErrorHandler :=;
                errorblock( {|e| ErrorHandler(e)} )

            begin sequence

                *-------------------------------------------------
                * The file is opened.
                *-------------------------------------------------

                dbusearea( .T., cRdd,;
                    alltrim(cDbfName), alltrim(cAliasName),;
                    lShared, lReadOnly )

                *-------------------------------------------------
                * Macro recorder.
                *-------------------------------------------------

                if isMemvar( _MEMVAR_MACRO_RECORDING_NAME )
                    Memvar->_MEMVAR_MACRO_RECORDER +=;
                        NL(1) +;
                        '// Open a db file.'                    +;
                        NL(1)                                   +;
                        '// dbusearea( [<lNewArea>], ' +;
                        '[<cDriver>], <cName>, [<xcAlias>], ' +;
                        '[<lShared>], [<lReadOnly>] ) --> NIL' +;
                        NL(1) +;
                        'dbusearea( .T., "' +;
                        alltrim(cRdd) + '", "' +;
                        alltrim(cDbfName) +;
                        '", "' +;
                        rtrim(cAliasName) +;
                        '", .' +;
                        transform( lShared, 'L' ) +;
                        '., .' +;
                        transform( lReadOnly, 'L' ) +;
                        '. ) ' +;
                        NL(1)
                end

            recover

                *-------------------------------------------------
                * No recovery is made.
                *-------------------------------------------------
            
            end //sequence
            errorblock(bOldErrorHandler)

        end

    end //sequence

    *-------------------------------------------------------------
    * Restore old data.
    *-------------------------------------------------------------

    rddsetdefault( cOldRddDefault )
    setcursor( nOldCursor )
    setcolor( cOldColor )
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )

    return NIL

*-----------------------------------------------------------------
static function idbReplace()
*&
*

    local aoGet             := {}
    local aButtons          := {}
    local aTab              := TAB_DEFAULT
    local bOldErrorHandler
    local cOldScreen
    local cOldColor         := setcolor()
    local nOldCursor        := setcursor( SETCURSOR_NORMAL )
    local nOldSelect        := select()
    local bOld_F1           :=;
        setkey( K_F1, {|| nbHelp( "<EDIT - REPLACE>", , chr(K_ENTER) ) } )
    local bOld_F2           := setkey( K_F2, NIL )
    local nOldRow           := row()
    local nOldCol           := col()
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth

    local lGoOn             := .F.

    local cField            := space(10)
    local cExpr             := space(_MAX_STRING_LEN)
    local cWhileCondition   := space(_MAX_STRING_LEN)
    local cForCondition     := space(_MAX_STRING_LEN)

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * This function acts on the current work area
        * (the active Alias). If it don't exists... break.
        *---------------------------------------------------------

        if alias() == ""

            alertBox( _ERROR_NO_ALIAS )

            break                                       // BREAK

        end

        *---------------------------------------------------------
        * As the function should modify record on all the Alias,
        * try to lock. If it is not possible, break.
        *---------------------------------------------------------

        if !flock()

            alertBox( _ERROR_FLOCK_FAILURE )

            break                                       // BREAK

        end

        *---------------------------------------------------------
        * Create a kind of window.
        *---------------------------------------------------------

        nBottom             := maxrow()
        nTop                := nBottom - 9
        nLeft               := 0
        nRight              := maxcol()
        nWidth              := nRight - nLeft +1

        aTab[TAB_LEFT]      := nLeft
        aTab[TAB_RIGHT]     := nRight
        aTab[TAB_TAB_ARRAY] := { 16, 8 }
        
        cOldScreen          := mouseScrSave( nTop, nLeft, nBottom, nRight )
        
        setcolor( COLOR_BODY )
        
        scroll( nTop, nLeft, nBottom, nRight )
        
        dispBoxShadow(;
            nTop, nLeft, nBottom, nRight,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )
        
        say(;
            nTop+1, nLeft+1,;
            padc( IDB_DIALOG_BOX_TOP_REPLACE, nWidth-2 ),;
            ,;
            COLOR_HEAD;
        )

        *---------------------------------------------------------
        * Do editing.
        *---------------------------------------------------------

        while .t.                                   // FOREVER

            say( nTop+3, nLeft+1, IDB_PROMPT_REPLACE_FIELD )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cField := x, cField ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                {||; 
                    setkey(; 
                        K_F2,; 
                        {|| cField := padr( fldNormal(), 10 ) }; 
                    ),;
                    .T.; 
                };
            )

            say( row()+1, nLeft+1, IDB_PROMPT_REPLACE_NEW_VALUE )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cExpr := x, cExpr ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                {||; 
                    setkey(; 
                        K_F2,; 
                        {|| gvAdd( @cExpr,  fldNormal() ) }; 
                    ),;
                    .T.;
                };
            )

            say ( row()+1, nLeft+1, IDB_PROMPT_WHILE_EXPRESSION )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cWhileCondition := x, cWhileCondition ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                {||;
                    gvDefault( @cWhileCondition, ".T." ),;
                    setkey(; 
                        K_F2,; 
                        {|| gvAdd( @cWhileCondition,  fldNormal() ) }; 
                    ),;
                    .T.;
                },;
                {|| gvDefault( @cWhileCondition, ".T." ) };
            )

            say( row()+1, nLeft+1, IDB_PROMPT_FOR_EXPRESSION )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cForCondition := x, cForCondition ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                {||;
                    gvDefault( @cForCondition, ".T." ),;
                    setkey(; 
                        K_F2,; 
                        {|| gvAdd( @cForCondition, fldNormal() ) }; 
                    ),;
                    .T.;
                },;
                {|| gvDefault( @cForCondition, ".T." )};
            )

            button( @aButtons, row()+2, nLeft+1,;
                _BUTTON_ESC_CANCEL, , {||keyboard(chr(K_ESC))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F1_HELP, , {|| eval(setkey(K_F1))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F2_LIST, , {|| iif( valtype(setkey(K_F2)) == "B", eval(setkey(K_F2)), NIL ) } )
            button( @aButtons, row(), col()+1,;
                _BUTTON_PGDN_CONFIRM, , {||keyboard(chr(K_PGDN))} )

            *-----------------------------------------------------
            * Read.
            *-----------------------------------------------------

            read( aoGet, , aButtons )
            aoGet   := {}
            aButtons := {}

            *-----------------------------------------------------
            * Check what was read.
            *-----------------------------------------------------

            do case
            case lastkey() = K_ESC

                *-------------------------------------------------
                * [Esc] means leave.
                *-------------------------------------------------

                lGoOn := .F.

                exit                                    // EXIT

            case lastkey() = K_PGDN

                *-------------------------------------------------
                * [PgDn] means confirm.
                * Check for data.
                * Ask for another confirmation before proceeding.
                *-------------------------------------------------

                if  empty(cField)               .or.;
                    empty(cExpr)                .or.;
                    empty(cWhileCondition)      .or.;
                    empty(cForCondition)
                    
                    loop                                // LOOP
                
                end

                if  alertBox(; 
                        IDB_PROMPT_DBF_REPLACE,;
                        { _MENU_NO, _MENU_YES }; 
                    ) == 2
                    
                    lGoOn := .T.
                    
                    exit                                // EXIT
                
                else

                    *---------------------------------------------
                    * If not confirmed loop again.
                    *---------------------------------------------

                end

            otherwise

                *-------------------------------------------------
                * Exit with different keys is not allowed.
                * Loop again.
                *-------------------------------------------------

            end

        end

        *---------------------------------------------------------
        * The editing is terminated, close the window.
        *---------------------------------------------------------

        mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

        *---------------------------------------------------------
        * If so requested, proceed.
        *---------------------------------------------------------

        if lGoOn

            *-----------------------------------------------------
            * As a file process is going to start, prepare
            * to handle errors.
            *-----------------------------------------------------

            bOldErrorHandler :=;
                errorblock( {|e| ErrorHandler(e)} )
            begin sequence

                cField := padr( alltrim(cField), 10 )

                dbgotop()

                dbeval(; 
                    {|| _FIELD->&(cField) := &(rtrim(cExpr))},;
                    {||; 
                        &( alltrim( cForCondition ) )   .and.;
                        !deleted()                      .and.;
                        waitFileEval(); 
                    },;
                    {|| &(alltrim(cWhileCondition))},;
                    ,, .F.; 
                )

                *-------------------------------------------------
                * Close the possible wait bar.
                *-------------------------------------------------
                
                waitFileEval( .T. )

                *-------------------------------------------------
                * Don't let the record pointer on eof().
                *-------------------------------------------------
                
                dbgotop()

                *-------------------------------------------------
                * If all was done, take care of the
                * macro recorder.
                *-------------------------------------------------

                if isMemvar( _MEMVAR_MACRO_RECORDING_NAME )
                    Memvar->_MEMVAR_MACRO_RECORDER +=;
                        NL(1) +;
                        [// Replace.] +;
                        NL(1) +;
                        [flock()] +;
                        NL(1) +;
                        [dbeval( { || (Field->] +;
                        alltrim(cField) +;
                        [:=] +;
                        rtrim(cExpr) +;
                        [) }, { || (!deleted() .and. ] +;
                        alltrim(cForCondition) +;
                        [ ) }, { || ] +;
                        alltrim(cWhileCondition) +;
                        [ },,, .F. )] +;
                        NL(1) +;
                        [dbgotop()] +;
                        NL(1) +;
                        [dbunlock()] +;
                        NL(1)
                end

            recover

                *-------------------------------------------------
                * Try to remedy if error occurred.
                *-------------------------------------------------

                dbgotop()

            end //sequence
            errorblock(bOldErrorHandler)
            
            waitFileEval( .T. )

        end

        *---------------------------------------------------------
        * Finally, release the lock.
        *---------------------------------------------------------

        dbunlock()

    end //sequence

    *-------------------------------------------------------------
    * Restore old data and return.
    *-------------------------------------------------------------

    setcursor( nOldCursor )
    setcolor( cOldColor )
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )

    return NIL

*-----------------------------------------------------------------
static function idbRecall()
*&
*

    local aoGet             := {}
    local aButtons          := {}
    local aTab              := TAB_DEFAULT

    local bOldErrorHandler
    local cOldScreen
    local cOldColor         := setcolor()
    local nOldCursor        := setcursor( SETCURSOR_NORMAL )
    local nOldSelect        := select()
    local bOld_F1           :=;
        setkey( K_F1, {|| nbHelp( "<EDIT - RECALL>", , chr(K_ENTER) ) } )
    local bOld_F2           := setkey( K_F2, NIL )
    local nOldRow           := row()
    local nOldCol           := col()
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth

    local lGoOn             := .F.

    local cWhileCondition   := space(_MAX_STRING_LEN)
    local cForCondition     := space(_MAX_STRING_LEN)

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * This function acts on the current work area
        * (the active Alias). If it don't exists... break.
        *---------------------------------------------------------

        if alias() == ""

            alertBox( _ERROR_NO_ALIAS )

            break                                   // BREAK

        end

        *---------------------------------------------------------
        * As the function should modify record on all the Alias,
        * try to lock. If it is not possible, break.
        *---------------------------------------------------------

        if flock()

            // Ok.

        else

            alertBox( _ERROR_FLOCK_FAILURE )

            break                                   // BREAK

        end

        *---------------------------------------------------------
        * Create a kind of window.
        *---------------------------------------------------------

        nBottom             := maxrow()
        nTop                := nBottom - 7
        nLeft               := 0
        nRight              := maxcol()
        nWidth              := nRight - nLeft +1

        aTab[TAB_LEFT]      := nLeft
        aTab[TAB_RIGHT]     := nRight
        aTab[TAB_TAB_ARRAY] := { 16, 8 }

        cOldScreen          := mouseScrSave( nTop, nLeft, nBottom, nRight )

        setcolor( COLOR_BODY )

        scroll( nTop, nLeft, nBottom, nRight )

        dispBoxShadow(;
            nTop, nLeft, nBottom, nRight,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )

        say(;
            nTop+1, nLeft+1,;
            padc( IDB_DIALOG_BOX_TOP_RECALL, nWidth-2 ),;
            ,;
            COLOR_HEAD;
        )

        *---------------------------------------------------------
        * Do editing.
        *---------------------------------------------------------

        while .t.                                   // FOREVER

            say( nTop+3, nLeft+1, IDB_PROMPT_WHILE_EXPRESSION )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cWhileCondition := x, cWhileCondition ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                {||;
                    gvDefault( @cWhileCondition, ".T." ),;
                    trueSetkey( K_F2, {|| gvAdd( @cWhileCondition,  fldNormal() ) } );
                },;
                {|| gvDefault( @cWhileCondition, ".T." ) };
            )

            say( row()+1, nLeft+1, IDB_PROMPT_FOR_EXPRESSION )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cForCondition := x, cForCondition ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                {||;
                    gvDefault( @cForCondition, ".T." ),;
                    trueSetkey( K_F2, {|| gvAdd( @cForCondition, fldNormal() ) } );
                },;
                {|| gvDefault( @cForCondition, ".T." )};
            )

            button( @aButtons, row()+2, nLeft+1,;
                _BUTTON_ESC_CANCEL, , {||keyboard(chr(K_ESC))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F1_HELP, , {|| eval(setkey(K_F1))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F2_LIST, , {|| iif( valtype(setkey(K_F2)) == "B", eval(setkey(K_F2)), NIL ) } )
            button( @aButtons, row(), col()+1,;
                _BUTTON_PGDN_CONFIRM, , {||keyboard(chr(K_PGDN))} )

            *-----------------------------------------------------
            * Read.
            *-----------------------------------------------------

            read( aoGet, , aButtons )
            aoGet   := {}
            aButtons := {}

            *-----------------------------------------------------
            * Check what was read.
            *-----------------------------------------------------

            do case
            case lastkey() = K_ESC

                *-------------------------------------------------
                * [Esc] means leave.
                *-------------------------------------------------

                lGoOn := .F.

                exit                                    // EXIT

            case lastkey() = K_PGDN

                *-------------------------------------------------
                * [PgDn] means confirm.
                * Check for data.
                * Ask for another confirmation before proceeding.
                *-------------------------------------------------

                if  empty(cWhileCondition) .or.;
                    empty(cForCondition);

                    loop                                // LOOP

                end

                if  alertBox(;
                        IDB_PROMPT_DBF_RECALL,;
                        { _MENU_NO, _MENU_YES };
                    ) == 2

                    lGoOn := .T.

                    exit                                // EXIT

                else

                    *---------------------------------------------
                    * If not confirmed loop again.
                    *---------------------------------------------

                end

            otherwise

                *-------------------------------------------------
                * Exit with different keys is not allowed.
                * Loop again.
                *-------------------------------------------------

            end

        end

        *---------------------------------------------------------
        * The editing is terminated, close the window.
        *---------------------------------------------------------

        mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

        *---------------------------------------------------------
        * If so requested, proceed.
        *---------------------------------------------------------

        if lGoOn

            *-----------------------------------------------------
            * As a file process is going to start, prepare
            * to handle errors.
            *-----------------------------------------------------

            bOldErrorHandler :=;
                errorblock( {|e| ErrorHandler(e)} )
            begin sequence

                dbgotop()

                dbeval(;
                    {|| dbrecall() },;
                    {||;
                        deleted()                   .and.;
                        &(alltrim(cForCondition))   .and.;
                        waitFileEval();
                    },;
                    {|| &(alltrim(cWhileCondition))};
                    ,,, .F.;
                )

                *-------------------------------------------------
                * Close the possible wait bar.
                *-------------------------------------------------

                waitFileEval( .T. )

                *-------------------------------------------------
                * Return to the top of file.
                *-------------------------------------------------

                dbgotop()

                *-------------------------------------------------
                * If all was done, take care of the
                * macro recorder.
                *-------------------------------------------------

                if isMemvar( _MEMVAR_MACRO_RECORDING_NAME )

                    Memvar->_MEMVAR_MACRO_RECORDER +=;
                        NL(1) +;
                        '// Recall records marked for deletion.' +;
                        NL(1) +;
                        'flock()' +;
                        NL(1) +;
                        'dbeval( { || ( dbrecall() ) }, ' +;
                            '{ || (deleted() .and. ' +;
                            alltrim(cForCondition) +;
                            ' ) }, { || ' +;
                            alltrim(cWhileCondition) +;
                            ' },,, .F. )' +;
                        NL(1) +;
                        'dbgotop()' +;
                        NL(1) +;
                        'dbunlock()' +;
                        NL(1)

                end

            recover

                *-------------------------------------------------
                * Try to remedy if error occurred.
                *-------------------------------------------------

                dbgotop()

            end //sequence
            waitFileEval( .T. )
            errorblock(bOldErrorHandler)

        end

        *---------------------------------------------------------
        * Finally, release the lock.
        *---------------------------------------------------------

        dbunlock()

    end //sequence

    *-------------------------------------------------------------
    * Restore old data and return.
    *-------------------------------------------------------------

    setcursor( nOldCursor )
    setcolor( cOldColor )
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )

    return NIL

*-----------------------------------------------------------------
static function idbDelete()
*&
*
    local aoGet             := {}
    local aButtons          := {}
    local aTab              := TAB_DEFAULT

    local bOldErrorHandler
    local cOldScreen
    local cOldColor         := setcolor( COLOR_BASE )
    local nOldCursor        := setcursor( SETCURSOR_NORMAL )
    local nOldSelect        := select()
    local bOld_F1           :=;
        setkey( K_F1, {|| nbHelp( "<EDIT - DELETE>", , chr(K_ENTER) ) } )
    local bOld_F2           := setkey( K_F2, NIL )
    local nOldRow           := row()
    local nOldCol           := col()
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth

    local lGoOn := .F.

    local cWhileCondition   := padr( ".F.", _MAX_STRING_LEN )
    local cForCondition     := padr( ".F.", _MAX_STRING_LEN )

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * For this operation a Alias must be active.
        *---------------------------------------------------------

        if alias() == ""

            alertBox( _ERROR_NO_ALIAS )

            break                                       // BREAK

        end

        *---------------------------------------------------------
        * Try to lock the active Alias.
        *---------------------------------------------------------

        if !flock()

            alertBox( _ERROR_FLOCK_FAILURE )

            break                                       // BREAK

        end

        *---------------------------------------------------------
        * Create a kind of window.
        *---------------------------------------------------------

        nBottom             := maxrow()
        nTop                := nBottom - 7
        nLeft               := 0
        nRight              := maxcol()
        nWidth              := nRight - nLeft +1

        aTab[TAB_LEFT]      := nLeft
        aTab[TAB_RIGHT]     := nRight
        aTab[TAB_TAB_ARRAY] := { 16, 8 }

        cOldScreen          :=;
            mouseScrSave( nTop, nLeft, nBottom, nRight )

        setcolor( COLOR_BODY )

        scroll( nTop, nLeft, nBottom, nRight )

        dispBoxShadow(;
            nTop, nLeft, nBottom, nRight,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )

        say(;
            nTop+1, nLeft+1,;
            padc( IDB_DIALOG_BOX_TOP_DELETE, nWidth-2 ),;
            ,;
            COLOR_HEAD;
        )

        *---------------------------------------------------------
        * Do editing.
        *---------------------------------------------------------

        while .t.                                   // FOREVER

            say( nTop+3, nLeft+1, IDB_PROMPT_WHILE_EXPRESSION )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cWhileCondition := x, cWhileCondition ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                {||;
                    gvDefault( @cWhileCondition, ".F." ),;
                    trueSetkey( K_F2, {|| gvAdd( @cWhileCondition,  fldNormal() ) } );
                },;
                {|| gvDefault( @cWhileCondition, ".F." ) };
            )

            say( row()+1, nLeft+1, IDB_PROMPT_FOR_EXPRESSION )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cForCondition := x, cForCondition ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                {||;
                    gvDefault( @cForCondition, ".F." ),;
                    trueSetkey( K_F2, {|| gvAdd( @cForCondition, fldNormal() ) } );
                },;
                {|| gvDefault( @cForCondition, ".F." )};
            )

            button( @aButtons, row()+2, nLeft+1,;
                _BUTTON_ESC_CANCEL, , {||keyboard(chr(K_ESC))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F1_HELP, , {|| eval(setkey(K_F1))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F2_LIST, , {|| iif( valtype(setkey(K_F2)) == "B", eval(setkey(K_F2)), NIL ) } )
            button( @aButtons, row(), col()+1,;
                _BUTTON_PGDN_CONFIRM, , {||keyboard(chr(K_PGDN))} )

            *-----------------------------------------------------
            * Read.
            *-----------------------------------------------------

            read( aoGet, , aButtons )
            aoGet   := {}
            aButtons := {}

            *-----------------------------------------------------
            * Check what was read.
            *-----------------------------------------------------

            do case
            case lastkey() = K_ESC

                *-------------------------------------------------
                * [Esc] means leave.
                *-------------------------------------------------

                lGoOn := .F.

                exit                                    // EXIT

            case lastkey() = K_PGDN

                *-------------------------------------------------
                * [PgDn] means confirm.
                * Check for data.
                * Ask for another confirmation before proceeding.
                *-------------------------------------------------

                if  empty(cWhileCondition) .or.;
                    empty(cForCondition);

                    loop                                // LOOP

                end

                if  alertBox(;
                        IDB_PROMPT_DBF_DELETE,;
                        { _MENU_NO, _MENU_YES };
                    ) == 2

                    lGoOn := .T.

                    exit                                // EXIT

                else

                    *---------------------------------------------
                    * If not confirmed loop again.
                    *---------------------------------------------

                end

            otherwise

                *-------------------------------------------------
                * Exit with different keys is not allowed.
                * Loop again.
                *-------------------------------------------------

            end

        end

        *---------------------------------------------------------
        * The editing is terminated, close the window.
        *---------------------------------------------------------

        mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

        *---------------------------------------------------------
        * If so requested, proceed.
        *---------------------------------------------------------

        if lGoOn

            *-----------------------------------------------------
            * As a file process is going to start, prepare
            * to handle errors.
            *-----------------------------------------------------

            bOldErrorHandler :=;
                errorblock( {|e| ErrorHandler(e)} )
            begin sequence

                dbgotop()

                dbeval(;
                    {|| dbdelete() },;
                    {||;
                        !deleted()                  .and.;
                        &(alltrim(cForCondition))   .and.;
                        waitFileEval();
                    },;
                    {|| &(alltrim(cWhileCondition)) };
                    ,,, .F.;
                )
                
                *-------------------------------------------------
                * Close the possible wait bar.
                *-------------------------------------------------

                waitFileEval( .T. )

                dbgotop()

                *-------------------------------------------------
                * Macro recorder
                *-------------------------------------------------

                if isMemvar( _MEMVAR_MACRO_RECORDING_NAME )

                    Memvar->_MEMVAR_MACRO_RECORDER +=;
                        NL(1) +;
                        '// Delete records.'                +;
                        NL(1) +;
                        'flock()'                           +;
                        NL(1) +;
                        'dbeval( { || ( dbdelete() ) }, '   +;
                            '{ || (!deleted() .and. '       +;
                            alltrim(cForCondition)          +;
                            ' ) }, { || '                   +;
                            alltrim(cWhileCondition)        +;
                            ' },,, .F. )' +;
                        NL(1) +;
                        'dbgotop()'                         +;
                        NL(1) +;
                        'dbunlock()'                        +;
                        NL(1)

                end

            recover

                *-------------------------------------------------
                * Try to remedy if error occurred.
                *-------------------------------------------------

                dbgotop()

            end //sequence
            waitFileEval( .T. )
            errorblock(bOldErrorHandler)

        end

        *---------------------------------------------------------
        * Finally, release the lock.
        *---------------------------------------------------------

        dbunlock()

    end //sequence

    *-------------------------------------------------------------
    * Restore old data.
    *-------------------------------------------------------------

    setcursor( nOldCursor )
    setcolor( cOldColor )
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )

    return NIL

*-----------------------------------------------------------------
static function idbPack()
*&
*
* It tries to make a PACK, and if it fails, there is no program
* interruption.
*

    local bOldErrorHandler

    local bOld_F1               :=;
        setkey( K_F1, {|| nbHelp( "<EDIT - PACK>", , chr(K_ENTER) ) } )

    *-------------------------------------------------------------
    * For this operation a Alias must be active.
    *-------------------------------------------------------------

    if alias() == ""  // no alias

        alertBox( _ERROR_NO_ALIAS )

        return NIL                                      // RETURN

    end

    *-------------------------------------------------------------
    * Do you really want it?
    *-------------------------------------------------------------

    if  alertBox(;
            IDB_PROMPT_DBF_PACK,;
            { _MENU_NO, _MENU_YES };
        ) == 1

        return NIL                                      // RETURN

    end

    *-------------------------------------------------------------
    * A little time to wait is needed.
    *-------------------------------------------------------------

    waitFor( IDB_WAIT_PACK )

    *-------------------------------------------------------------
    * The operation may generate an error.
    *-------------------------------------------------------------

    bOldErrorHandler := errorblock( {|e| ErrorHandler(e)} )
    begin sequence

        dbPack()

        *---------------------------------------------------------
        * Macro recorder
        *---------------------------------------------------------

        if isMemvar( _MEMVAR_MACRO_RECORDING_NAME )

            Memvar->_MEMVAR_MACRO_RECORDER +=;
                NL(1) +;
                '// Pack.'                              +;
                NL(1) +;
                '__dbpack()'                            +;
                NL(1)

        end

    recover
    
        *---------------------------------------------------------
        * Nothing to do for error recovery.
        *---------------------------------------------------------

    end //sequence
    errorblock(bOldErrorHandler)

    *-------------------------------------------------------------
    * Close the wait box.
    *-------------------------------------------------------------

    waitFor()


    *-------------------------------------------------------------
    * Restore old data.
    *-------------------------------------------------------------

    setkey( K_F1, bOld_F1 )

    return NIL

*------------------------------------------------------------------
static function idbOrdCreate()
*&
*
* idbOrdCreate() --> NIL
*
* Create an index for the active Alias.
*

    local aoGet             := {}
    local aButtons          := {}
    local aTab              := TAB_DEFAULT

    local bOldErrorHandler
    local cOldScreen
    local cOldColor         := setcolor( COLOR_BASE )
    local nOldCursor        := setcursor( SETCURSOR_NORMAL )
    local nOldSelect        := select()
    local bOld_F1           :=;
        setkey( K_F1, {|| nbHelp( "<FILE - NEW .NTX / NEW TAG>", , chr(K_ENTER) ) } )
    local bOld_F2           := setkey( K_F2, NIL )
    local nOldRow           := row()
    local nOldCol           := col()
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth
    local lOldDeleted       := set( _SET_DELETED, .T. )

    local cFileName         := space(_MAX_STRING_LEN)
    local lExit             := .F.
    local lGoOn             := .T.

    static cTagName
    static cForCondition
    static cOrdBagName
    static cOrdExpr

    default( @cForCondition,    padr( ".T.", _MAX_STRING_LEN ) )

    default( @cTagName,         space( _MAX_STRING_LEN ) )

    default(;
        @cOrdBagName,;
        padr( "*." + _EXTENTION_INDEXBAG, _MAX_STRING_LEN );
    )

    default( @cOrdExpr,         space( _MAX_STRING_LEN ) )

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * For this operation a Alias must be active.
        *---------------------------------------------------------

        if alias() == "" // no alias

            alertBox( _ERROR_NO_ALIAS )

            break                                       // BREAK

        end

        *---------------------------------------------------------
        * Create a kind of window.
        *---------------------------------------------------------

        nBottom     := maxrow()

        do case
        case rddname() == "DBFNTX"
            nTop    := nBottom - 8
        case rddname() == "DBFNDX"
            nTop    := nBottom - (8 -1)
        case rddname() == "DBPX"
            nTop    := nBottom - (8 -1)
        case rddname() == "DBFCDX"
            nTop    := nBottom - (8 +1)
        case rddname() == "DBFMDX"
            nTop    := nBottom - (8 +1)
        otherwise
            nTop    := nBottom - (8 -1)
        end

        nLeft       := 0
        nRight      := maxcol()
        nWidth      := nRight - nLeft +1

        aTab[TAB_LEFT]      := nLeft
        aTab[TAB_RIGHT]     := nRight
        aTab[TAB_TAB_ARRAY] := { 32, 4 }

        cOldScreen          :=;
            mouseScrSave( nTop, nLeft, nBottom, nRight )

        setcolor( COLOR_BODY )

        scroll( nTop, nLeft, nBottom, nRight )

        dispBoxShadow(;
            nTop, nLeft, nBottom, nRight,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )

        say(;
            nTop+1, nLeft+1,;
            padc( IDB_DIALOG_BOX_TOP_NTX_NEW, nWidth-2 ),;
            ,;
            COLOR_HEAD;
        )

        *---------------------------------------------------------
        * Do editing.
        *---------------------------------------------------------

        while .t.                                   // FOREVER

            say( nTop+3, nLeft+1, IDB_PROMPT_NEW_NTX_FILENAME )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cOrdBagName := x, cOrdBagName ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { ||;
                    setkey( K_F2, {|| gvFileDir( @cOrdBagName ) } ),;
                    .T.;
                },;
                { ||;
                    gvFileExtention( @cOrdBagName, _EXTENTION_INDEXBAG );
                };
            )

            say( row()+1, nLeft+1, IDB_PROMPT_NEW_NTX_KEYEXPR )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cOrdExpr := x, cOrdExpr ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { ||;
                    setkey( K_F2, {|| gvAdd( @cOrdExpr, fldIndex() ) } ),;
                    .T.;
                },;
                { ||;
                    !empty( cOrdExpr );
                };
            )

            if  rddname() == "DBFMDX"       .or.;
                rddname() == "DBFCDX"

                say( row()+1, nLeft+1, IDB_PROMPT_TAG_NAME )
                tab( aTab )
                get(;
                    @aoGet, row(), col(),;
                    { |x| iif( pcount() > 0, cTagName := x, cTagName ) },;
                    picChrMax( col(), nRight-1 ),;
                    ,;
                    { ||;
                        setkey( K_F2, NIL ),;
                        .T.;
                    },;
                    { ||;
                        gvDefault( @cTagName, cOrdBagName );
                    };
                )

            end

            if  rddname() == "DBFNDX"           .or.;
                rddname() == "DBPX"
                
                *-------------------------------------------------
                * There is no FOR condition.
                *-------------------------------------------------

            else

                say( row()+1, nLeft+1, IDB_PROMPT_FOR_EXPRESSION )
                tab( aTab )
                get(;
                    @aoGet, row(), col(),;
                    { |x| iif( pcount() > 0, cForCondition := x, cForCondition ) },;
                    picChrMax( col(), nRight-1 ),;
                    ,;
                    { ||;
                        setkey( K_F2, {|| gvAdd( @cForCondition, fldNormal() ) } ),;
                        .T.;
                    },;
                    { ||;
                        gvDefault( @cForCondition, ".T." );
                    };
                )

            end


            button( @aButtons, row()+2, nLeft+1,;
                _BUTTON_ESC_CANCEL, , {||keyboard(chr(K_ESC))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F1_HELP, , {|| eval(setkey(K_F1))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F2_LIST, , {|| iif( valtype(setkey(K_F2)) == "B", eval(setkey(K_F2)), NIL ) } )
            button( @aButtons, row(), col()+1,;
                _BUTTON_PGDN_CONFIRM, , {||keyboard(chr(K_PGDN))} )

            *-----------------------------------------------------
            * Read.
            *-----------------------------------------------------

            read( aoGet, , aButtons )
            aoGet   := {}
            aButtons := {}

            *-----------------------------------------------------
            * Check what was read.
            *-----------------------------------------------------

            do case
            case lastkey() = K_ESC

                *-------------------------------------------------
                * [Esc] means leave.
                *-------------------------------------------------

                lGoOn := .F.

                exit                                    // EXIT

            case lastkey() = K_PGDN

                *-------------------------------------------------
                * [PgDn] means confirm.
                * Check for data.
                *-------------------------------------------------

                if  isFile(; 
                        strAddExtention(;
                            cOrdBagName,;
                            _EXTENTION_INDEXBAG;
                        ); 
                    )

                    *---------------------------------------------
                    * The file already exist: ask for overwrite.
                    *---------------------------------------------

                    if  alertBox(;
                            strAddExtention(;
                                cOrdBagName,;
                                _EXTENTION_INDEXBAG;
                            ) + NL(1) +;
                                _ERROR_PROMPT_ALREADY_EXIST_OVERWRITE,;
                            { _MENU_NO, _MENU_YES };
                        ) == 2

                        *-----------------------------------------
                        * Ok, overwrite.
                        *-----------------------------------------

                        lGoOn := .T.

                        exit                            // EXIT

                    else

                        *-----------------------------------------
                        * Don't overwrite: loop again.
                        *-----------------------------------------

                    end

                else

                    *---------------------------------------------
                    * Wild characters inside the index name are
                    * not permitted.
                    *---------------------------------------------

                    if !isWild( cOrdBagName )

                        lGoOn := .T.

                        exit                            // EXIT

                    else

                        *-----------------------------------------
                        * Loop.
                        *-----------------------------------------

                    end

                end

            otherwise

                *-------------------------------------------------
                * Exit with different keys is not allowed.
                * Loop again.
                *-------------------------------------------------

            end

        end

        *---------------------------------------------------------
        * The editing is terminated, close the window.
        *---------------------------------------------------------

        mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

        *---------------------------------------------------------
        * If so requested, proceed.
        *---------------------------------------------------------

        if lGoOn

            *-----------------------------------------------------
            * As a file process is going to start, prepare
            * to handle errors.
            *-----------------------------------------------------

            bOldErrorHandler :=;
                errorblock( {|e| ErrorHandler(e)} )
            begin sequence

                lExit := .T.

                do case
                case rddname() == "DBFCDX"

                    ordCondSet(;
                        (cForCondition),;
                        {|| &(cForCondition)},;
                        ,,;
                        {|| WaitFileEval()},;
                        , RECNO(),,,, .F.;
                    )

                    ordCreate(;
                        (alltrim(cOrdBagName)),;
                        (alltrim(cTagName)),;
                        (alltrim(cOrdExpr)),;
                        {|| &(alltrim(cOrdExpr))},;
                        .F.;
                    )

                    waitFileEval( .T. )

                    *---------------------------------------------
                    * Macro recorder.
                    *---------------------------------------------

                    if  isMemvar( _MEMVAR_MACRO_RECORDING_NAME )

                        Memvar->_MEMVAR_MACRO_RECORDER +=;
                            NL(1) +;
                            '// Create index.'                  +;
                            NL(1) +;
                            'ordCondSet( '                      +;
                                alltrim(cForCondition)          +;
                                ' , {|| '                       +;
                                alltrim(cForCondition)          +;
                                '},,,,, RECNO(),,,, .F. )'      +;
                            NL(1) +;
                            'ordCreate( "'                      +;
                                alltrim(cOrdBagName)            +;
                                '", "'                          +;
                                alltrim(cTagName)               +;
                                '", "'                          +;
                                alltrim(cOrdExpr)               +;
                                '", {|| '                       +;
                                alltrim(cOrdExpr)               +;
                                ' }, .F. )'                     +;
                            NL(1)

                    end

                case rddname() == "DBFMDX"

                    ordCondSet(;
                        (cForCondition),;
                        {|| &(cForCondition)},;
                        ,,;
                        {|| WaitFileEval()},;
                        , RECNO(),,,, .F.;
                    )

                    ordCreate(;
                        (alltrim(cOrdBagName)),;
                        (alltrim(cTagName)),;
                        (alltrim(cOrdExpr)),;
                        {|| &(alltrim(cOrdExpr))},;
                        .F.;
                    )

                    waitFileEval( .T. )

                    *---------------------------------------------
                    * Macro recorder.
                    *---------------------------------------------

                    if  isMemvar( _MEMVAR_MACRO_RECORDING_NAME )

                        Memvar->_MEMVAR_MACRO_RECORDER +=;
                            NL(1) +;
                            '// Create index.'              +;
                            NL(1) +;
                            'ordCondSet( '                  +;
                                alltrim(cForCondition)      +;
                                ' , {|| '                   +;
                                alltrim(cForCondition)      +;
                                '},,,,, RECNO(),,,, .F. )'  +;
                            NL(1) +;
                            'ordCreate( "'                  +;
                                alltrim(cOrdBagName)        +;
                                '", "'                      +;
                                alltrim(cTagName)           +;
                                '", "'                      +;
                                alltrim(cOrdExpr)           +;
                                '", {|| '                   +;
                                alltrim(cOrdExpr)           +;
                                ' }, .F. )'                 +;
                            NL(1)

                    end

                case rddname() == "DBFNTX"

                    ordCondSet(;
                        (cForCondition),;
                        {|| &(cForCondition) },;
                        ,,;
                        {|| waitFileEval() },;
                        , RECNO(),,,, .F.;
                    )

                    ordCreate(;
                        (alltrim(cOrdBagName)),;
                        ,;
                        (alltrim(cOrdExpr)),;
                        {|| &(alltrim(cOrdExpr)) },;
                        .F.;
                    )

                    waitFileEval( .T. )

                    *---------------------------------------------
                    * Macro recorder.
                    *---------------------------------------------

                    if  isMemvar( _MEMVAR_MACRO_RECORDING_NAME )

                        Memvar->_MEMVAR_MACRO_RECORDER +=;
                            NL(1) +;
                            '// Create index.'              +;
                            NL(1) +;
                            'ordCondSet( '                  +;
                                alltrim(cForCondition)      +;
                                ' , {|| '                   +;
                                alltrim(cForCondition)      +;
                                '},,,,, RECNO(),,,, .F. )'  +;
                            NL(1) +;
                            'ordCreate( "'                  +;
                                alltrim(cOrdBagName)        +;
                                '",, "'                     +;
                                alltrim(cOrdExpr)           +;
                                '", {|| '                   +;
                                alltrim(cOrdExpr)           +;
                                ' }, .F. )'                 +;
                            NL(1)

                    end

                case rddname() == "DBFNDX"

                    ordCreate(;
                        (alltrim(cOrdBagName)),;
                        ,;
                        (alltrim(cOrdExpr)),;
                        {|| &(alltrim(cOrdExpr)) },;
                        .F.;
                    )

                    *---------------------------------------------
                    * Macro recorder.
                    *---------------------------------------------

                    if  isMemvar( _MEMVAR_MACRO_RECORDING_NAME )

                        Memvar->_MEMVAR_MACRO_RECORDER +=;
                            NL(1) +;
                            '// Create index.'              +;
                            NL(1) +;
                            'ordCondSet( ,,,, '             +;
                                '{|| waitFileEval()},, '    +;
                                'RECNO(),,,, .F. )'         +;
                            NL(1) +;
                            'ordCreate( "'                  +;
                                alltrim(cOrdBagName)        +;
                                '",, "'                     +;
                                alltrim(cOrdExpr)           +;
                                '", {|| '                   +;
                                alltrim(cOrdExpr)           +;
                                ' }, .F. )'                 +;
                            NL(1)

                    end

                end

                dbgotop()

            recover

                *-------------------------------------------------
                * Try to recover the error.
                *-------------------------------------------------

                ordListClear()

                dbgotop()

                lExit := .F.

            end //sequence
            errorblock(bOldErrorHandler)
            waitFileEval( .T. )

        end

    end //sequence

    *-------------------------------------------------------------
    * Restore old data.
    *-------------------------------------------------------------

    setcursor( nOldCursor )
    setcolor( cOldColor )
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )
    set( _SET_DELETED, lOldDeleted )

    return NIL

*-----------------------------------------------------------------
static function idbSetindex()
*&
*
* idbNtxOpen() --> NIL
*
* It Opens an index bag file.
*

    local aoGet             := {}
    local aButtons          := {}
    local aTab              := TAB_DEFAULT

    local bOldErrorHandler
    local cOldScreen
    local cOldColor         := setcolor( COLOR_BASE )
    local nOldCursor        := setcursor( SETCURSOR_NORMAL )
    local nOldSelect        := select()
    local bOld_F1           :=;
        setkey( K_F1, {|| nbHelp( "<FILE - OPEN INDEX>", , chr(K_ENTER) ) } )
    local bOld_F2           := setkey( K_F2, NIL )
    local nOldRow           := row()
    local nOldCol           := col()
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth

    local cOrdBagName       :=;
        padr( "*." + _EXTENTION_INDEXBAG, _MAX_STRING_LEN )

    local lGoOn

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * For this operation a Alias must be active.
        *---------------------------------------------------------

        if alias() == ""

            alertBox( _ERROR_NO_ALIAS )

            break                                       // BREAK

        end

        *---------------------------------------------------------
        * Create a kind of window.
        *---------------------------------------------------------

        nBottom     := maxrow()
        nTop        := nBottom - 6
        nLeft       := 0
        nRight      := maxcol()
        nWidth      := nRight - nLeft +1

        nLeft       := 0
        nRight      := maxcol()
        nWidth      := nRight - nLeft +1

        aTab[TAB_LEFT]      := nLeft
        aTab[TAB_RIGHT]     := nRight
        aTab[TAB_TAB_ARRAY] := { 24, 4 }

        cOldScreen          :=;
            mouseScrSave( nTop, nLeft, nBottom, nRight )

        setcolor( COLOR_BODY )

        scroll( nTop, nLeft, nBottom, nRight )

        dispBoxShadow(;
            nTop, nLeft, nBottom, nRight,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )

        say(;
            nTop+1, nLeft+1,;
            padc( IDB_DIALOG_BOX_TOP_OPEN_NTX, nWidth-2 ),;
            ,;
            COLOR_HEAD;
        )

        *---------------------------------------------------------
        * Do editing.
        *---------------------------------------------------------

        while .t.                                   // FOREVER

            say( nTop+3, nLeft+1, IDB_PROMPT_OPEN_NTX_FILENAME )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cOrdBagName := x, cOrdBagName ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { ||;
                    setkey( K_F2, {|| gvFileDir( @cOrdBagName ) } ),;
                    .T.;
                },;
                { ||;
                    gvFileExist( @cOrdBagName, _EXTENTION_INDEXBAG );
                };
            )

            button( @aButtons, row()+2, nLeft+1,;
                _BUTTON_ESC_CANCEL, , {||keyboard(chr(K_ESC))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F1_HELP, , {|| eval(setkey(K_F1))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F2_LIST, , {|| iif( valtype(setkey(K_F2)) == "B", eval(setkey(K_F2)), NIL ) } )
            button( @aButtons, row(), col()+1,;
                _BUTTON_PGDN_CONFIRM, , {||keyboard(chr(K_PGDN))} )

            *-----------------------------------------------------
            * Read.
            *-----------------------------------------------------

            read( aoGet, , aButtons )
            aoGet   := {}
            aButtons := {}

            *-----------------------------------------------------
            * Check what was read.
            *-----------------------------------------------------

            do case
            case lastkey() = K_ESC

                *-------------------------------------------------
                * [Esc] means leave.
                *-------------------------------------------------

                lGoOn := .F.

                exit                                    // EXIT

            case lastkey() = K_PGDN

                *-------------------------------------------------
                * [PgDn] means confirm.
                * Check for data.
                *-------------------------------------------------

                if  gvFileExist(;
                        cOrdBagName,;
                        _EXTENTION_INDEXBAG;
                    );
                    .and.;
                    !empty( cOrdBagName )

                    lGoOn := .T.

                    exit                                // EXIT

                else

                    *---------------------------------------------
                    * The file don't exist or the name was not
                    * given. Loop again
                    *---------------------------------------------

                end

            otherwise

                *-------------------------------------------------
                * Exit with different keys is not allowed.
                * Loop again.
                *-------------------------------------------------

            end

        end

        *---------------------------------------------------------
        * The editing is terminated, close the window.
        *---------------------------------------------------------

        mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

        *---------------------------------------------------------
        * If so requested, proceed.
        *---------------------------------------------------------

        if lGoOn

            *-----------------------------------------------------
            * As a file process is going to start, prepare
            * to handle errors.
            *-----------------------------------------------------

            bOldErrorHandler := errorblock( {|e| ErrorHandler(e)} )
            begin sequence

                dbsetindex( alltrim( cOrdBagName ) )

                *-------------------------------------------------
                * Macro recorder.
                *-------------------------------------------------

                if isMemvar( _MEMVAR_MACRO_RECORDING_NAME )

                    Memvar->_MEMVAR_MACRO_RECORDER +=;
                        NL(1)                               +;
                        '// Open an index.'                 +;
                        NL(1)                               +;
                        '// dbsetindex(<cOrdBagName>)'      +;
                        NL(1)                               +;
                        'dbsetindex( "'                     +;
                            alltrim(cOrdBagName)            +;
                            '" ) '                          +;
                        NL(1)

                end

            recover

                *-------------------------------------------------
                * Try to remedy if error occurred.
                *-------------------------------------------------

                ordListClear()

            end //sequence
            errorblock(bOldErrorHandler)

        end

    end //sequence

    *-------------------------------------------------------------
    * Restore old data.
    *-------------------------------------------------------------

    setcursor( nOldCursor )
    setcolor( cOldColor )
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )

    return NIL

*-----------------------------------------------------------------
static function idbClearIndex()
*&
*

    *-------------------------------------------------------------
    * For this operation a Alias must be active.
    *-------------------------------------------------------------

    if alias() == ""

        return NIL                                      // RETURN

    else

        *---------------------------------------------------------
        * Do it.
        *---------------------------------------------------------

        dbclearindex()

        *---------------------------------------------------------
        * Macro Recorder.
        *---------------------------------------------------------

        if  isMemvar( _MEMVAR_MACRO_RECORDING_NAME )

            Memvar->_MEMVAR_MACRO_RECORDER +=;
                NL(1)                                       +;
                '// Close all indexes '                     +;
                    'for the current work area.'            +;
                NL(1)                                       +;
                'dbclearindex()'                            +;
                NL(1)

        end

    end

    return NIL

*-----------------------------------------------------------------
static function idbReindex()
*&
*
    local bOldErrorHandler

    do case
    case ( alias() == "" )

        *---------------------------------------------------------
        * There is no active Alias.
        *---------------------------------------------------------

        alertBox( _ERROR_NO_ALIAS )

        return NIL                                      // RETURN

    case ordbagname( ordsetfocus() ) == ""

        *---------------------------------------------------------
        * There is no active index.
        *---------------------------------------------------------

        alertBox( _ERROR_NO_INDEX )

    otherwise

        *---------------------------------------------------------
        * File operation may generate errors.
        *---------------------------------------------------------

        bOldErrorHandler := errorblock( {|e| ErrorHandler(e)} )
        begin sequence

            *-----------------------------------------------------
            * Try to flock, but it works only on NTX files.
            * With other index bag types an exclusive
            * open is required!!!
            *-----------------------------------------------------

            if !flock()

                *-------------------------------------------------
                * File lock failed.
                *-------------------------------------------------

                alertBox( _ERROR_FLOCK_FAILURE )

                break                                   // BREAK

            end

            *-----------------------------------------------------
            * Start a nested sequence.
            *-----------------------------------------------------

            begin sequence

                ordCondSet(,,,, {||waitFileEval()},,,,,,,,)
                ordListRebuild()

                waitFileEval( .T. )

                *-------------------------------------------------
                * Macro Recorder.
                *-------------------------------------------------

                if  isMemvar( _MEMVAR_MACRO_RECORDING_NAME )

                    Memvar->_MEMVAR_MACRO_RECORDER +=;
                        NL(1)                               +;
                        '// Reindex.'                       +;
                        NL(1)                               +;
                        'flock()'                           +;
                        NL(1)                               +;
                        'dbreindex()'                       +;
                        NL(1)                               +;
                        'dbunlock()'                        +;
                        NL(1)

                end

            recover

                *-------------------------------------------------
                * If an error is recovered here, it is mostly
                * probable that a different RDD form DBFNTX
                * was used. In this case, a reindex requires
                * an exclusive open.
                *-------------------------------------------------

                alertBox( _ERROR_EXCLUSIVE_REQUIRED )

            end //sequence

        end //sequence
        errorblock(bOldErrorHandler)
        dbunlock()

    end

    return NIL

*-----------------------------------------------------------------
static function idbRelation()
*&
*
* idbRelation() --> NIL
*
* Establish a relation in a guided form.
*

    local aoGet             := {}
    local aButtons          := {}
    local aTab              := TAB_DEFAULT

    local bOldErrorHandler
    local cOldScreen
    local cOldColor           := setcolor( COLOR_BASE )
    local nOldCursor          := setcursor( SETCURSOR_NORMAL )
    local nOldSelect          := select()
    local bOld_F1             :=;
        setkey( K_F1, {|| nbHelp( "<FILE - SET RELATION>", , chr(K_ENTER) ) } )
    local bOld_F2             := setkey( K_F2, NIL )
    local nOldRow             := row()
    local nOldCol             := col()
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth

    local cChildName := space(10)
    local lGoOn := .F.

    static cRelExpression

    default( @cRelExpression,   space(_MAX_STRING_LEN) )

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        if alias() == ""

            alertBox( _ERROR_NO_ALIAS )

            break                                       // BREAK

        end

        *---------------------------------------------------------
        * Create a kind of window.
        *---------------------------------------------------------

        nBottom     := maxrow()
        nTop        := nBottom - 7
        nLeft       := 0
        nRight      := maxcol()
        nWidth      := nRight - nLeft +1

        aTab[TAB_LEFT]      := nLeft
        aTab[TAB_RIGHT]     := nRight
        aTab[TAB_TAB_ARRAY] := { 24, 4 }

        cOldScreen          :=;
            mouseScrSave( nTop, nLeft, nBottom, nRight )

        setcolor( COLOR_BODY )

        scroll( nTop, nLeft, nBottom, nRight )

        dispBoxShadow(;
            nTop, nLeft, nBottom, nRight,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )

        say(;
            nTop+1, nLeft+1,;
            padc( IDB_DIALOG_BOX_TOP_RELATION_DEFINITION, nWidth-2 ),;
            ,;
            COLOR_HEAD;
        )

        *---------------------------------------------------------
        * Do editing.
        *---------------------------------------------------------

        while .t.                                   // FOREVER

            say( nTop+3, nLeft+1, IDB_PROMPT_RELATION_DEFINITION_CHILDNAME )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cChildName := x, cChildName ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { ||;
                    setkey(;
                        K_F2,;
                        { ||;
                            cChildName := padr( selectAlias(), 10 );
                         };
                    ),;
                    .T.;
                },;
                { || .T. };
            )

            say( row()+1, nLeft+1, IDB_PROMPT_RELATION_DEFINITION_EXPRESSION )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cRelExpression := x, cRelExpression ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { ||;
                    setkey(;
                        K_F2,;
                        {|| gvAdd( @cRelExpression, fldRelation() ) };
                    ),;
                    .T.;
                },;
                { || .T. };
            )

            button( @aButtons, row()+2, nLeft+1,;
                _BUTTON_ESC_CANCEL, , {||keyboard(chr(K_ESC))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F1_HELP, , {|| eval(setkey(K_F1))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F2_LIST, , {|| iif( valtype(setkey(K_F2)) == "B", eval(setkey(K_F2)), NIL ) } )
            button( @aButtons, row(), col()+1,;
                _BUTTON_PGDN_CONFIRM, , {||keyboard(chr(K_PGDN))} )

            *-----------------------------------------------------
            * Read.
            *-----------------------------------------------------

            read( aoGet, , aButtons )
            aoGet   := {}
            aButtons := {}

            *-----------------------------------------------------
            * Check what was read.
            *-----------------------------------------------------

            do case
            case lastkey() = K_ESC

                *-------------------------------------------------
                * [Esc] means leave.
                *-------------------------------------------------

                lGoOn := .F.

                exit                                    // EXIT

            case lastkey() = K_PGDN

                *-------------------------------------------------
                * [PgDn] means confirm.
                *-------------------------------------------------

                lGoOn := .T.

                exit                                    // EXIT

            otherwise

                *-------------------------------------------------
                * Exit with different keys is not allowed.
                * Loop again.
                *-------------------------------------------------

            end

        end

        *---------------------------------------------------------
        * The editing is terminated, close the window.
        *---------------------------------------------------------

        mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

        *---------------------------------------------------------
        * If so requested, proceed.
        *---------------------------------------------------------

        if lGoOn

            bOldErrorHandler :=;
                errorblock( {|e| ErrorHandler(e)} )

            *-----------------------------------------------------
            * As a file process is going to start, prepare
            * to handle errors.
            *-----------------------------------------------------

            begin sequence

                *-------------------------------------------------
                * Do it.
                *-------------------------------------------------

                dbSetRelation(;
                    alltrim(cChildName),;
                    {|| &(alltrim(cRelExpression))},;
                    (alltrim(cRelExpression));
                )

                *-------------------------------------------------
                * Macro Recorder.
                *-------------------------------------------------

                if  isMemvar( _MEMVAR_MACRO_RECORDING_NAME )

                    Memvar->_MEMVAR_MACRO_RECORDER +=;
                        NL(1)                               +;
                        '// Relate two work areas.'         +;
                        NL(1)                               +;
                        '// dbsetrelation( '                +;
                            '<nArea>|<cAlias>, '            +;
                            '<bExpr>, [<cExpr>] ) '         +;
                            '--> NIL'                       +;
                        NL(1)                               +;
                        'dbsetrelation( "'                  +;
                            alltrim(cChildName)             +;
                            '", { || ('                     +;
                            alltrim(cRelExpression)         +;
                            ') }, "'                        +;
                            alltrim(cRelExpression)         +;
                            '" ) '                          +;
                        NL(1)

                end

            recover

                *-------------------------------------------------
                * Try to remedy if error occurred.
                *-------------------------------------------------

                alertBox( IDB_ERROR_RELATION )

            end //sequence
            errorblock(bOldErrorHandler)

        end

    end //sequence

    *-------------------------------------------------------------
    * Restore old data.
    *-------------------------------------------------------------

    setcursor( nOldCursor )
    setcolor( cOldColor )
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )

    return NIL

#endif  //RUNTIME

#ifndef RUNTIME
*-----------------------------------------------------------------
* LBL - LABELS CREATION / EDITING
*-----------------------------------------------------------------

#define LBL_DIALOG_BOX_LABEL_FORM_TOP;
    "LABEL FORM ..."

#define LBL_WINDOW_LABEL;
    "LABEL"

#define LBL_PROMPT_COMPRESSED_PRINT;
    "Compressed Print ? (Y/T/N/F) "
#define LBL_PROMPT_WHILE_EXPRESSION;
    "WHILE expression condition:"
#define LBL_PROMPT_FOR_EXPRESSION;
    "FOR expression condition:"

#define LBL_PROMPT_OPEN_FILE_LABEL;
    "Label file name:"

*-----------------------------------------------------------------
static function lblPrint()
*&
*
* lblPrint() --> NIL
*
* This function asks for label file name,
* the FOR and WHILE condition before print.
*

    local aoGet             := {}
    local aButtons          := {}
    local aTab              := TAB_DEFAULT

    local bOldErrorHandler
    local cOldScreen
    local cOldColor         := setcolor()
    local nOldCursor        := setcursor( SETCURSOR_NORMAL )
    local bOld_F1           :=;
        setkey( K_F1, {|| nbHelp( "<REPORT - LABEL FORM>", , chr(K_ENTER) ) } )

    local bOld_F2           := setkey( K_F2, NIL )
    local nOldRow           := row()
    local nOldCol           := col()
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth

    local lGoOn             := .T.
    local cLabelFile        :=;
        padr( "*." + _EXTENTION_LABEL, _MAX_STRING_LEN )
    local cWhileCondition   := padr( ".T.", _MAX_STRING_LEN )
    local cForCondition     := padr( ".T.", _MAX_STRING_LEN )

    local bWhileCondition
    local bForCondition

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * The labels can be printed only if there is an active
        * Alias.
        *---------------------------------------------------------

        if alias() == ""

            alertBox( _ERROR_NO_ALIAS )

            break                                       // BREAK

        end

        *---------------------------------------------------------
        * Create a kind of window.
        *---------------------------------------------------------

        nBottom             := maxrow()
        nTop                := nBottom - 8
        nLeft               := 0
        nRight              := maxcol()
        nWidth              := nRight - nLeft +1

    	aTab[TAB_LEFT]      := nLeft
	    aTab[TAB_RIGHT]     := nRight
    	aTab[TAB_TAB_ARRAY] := { 32, 4 }
	
        cOldScreen  := mouseScrSave( nTop, nLeft, nBottom, nRight )
	
        setcolor( COLOR_BODY )
	
        scroll( nTop, nLeft, nBottom, nRight )
	
        dispBoxShadow(;
            nTop, nLeft, nBottom, nRight,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )
	
        say(;
            nTop+1, nLeft+1,;
            padc( LBL_DIALOG_BOX_LABEL_FORM_TOP, nWidth-2 ),;
            ,;
            COLOR_HEAD;
        )

        *---------------------------------------------------------
        * Do editing.
        *---------------------------------------------------------

        while .T.                                       // FOREVER

            say( nTop+3, nLeft+1, LBL_PROMPT_OPEN_FILE_LABEL )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cLabelFile := x, cLabelFile ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { ||;
                    setkey(;
                        K_F2,;
                        { || gvFileDir( @cLabelFile ) };
                    ),;
                    .T.;
                },;
                { || gvFileExist( @cLabelFile, _EXTENTION_LABEL ) };
            )

            say( row()+1, nLeft+1, LBL_PROMPT_WHILE_EXPRESSION )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cWhileCondition := x, cWhileCondition ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                {||;
                    gvDefault( @cWhileCondition, ".T." ),;
                    setkey(; 
                        K_F2,; 
                        {|| gvAdd( @cWhileCondition,  fldNormal() ) }; 
                    ),;
                    .T.;
                },;
                {|| gvDefault( @cWhileCondition, ".T." ) };
            )

            say( row()+1, nLeft+1, LBL_PROMPT_FOR_EXPRESSION )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cForCondition := x, cForCondition ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                {||;
                    gvDefault( @cForCondition, ".T." ),;
                    setkey(; 
                        K_F2,; 
                        {|| gvAdd( @cForCondition, fldNormal() ) }; 
                    ),;
                    .T.;
                },;
                {|| gvDefault( @cForCondition, ".T." )};
            )

            button( @aButtons, row()+2, nLeft+1,;
                _BUTTON_ESC_CANCEL, , {||keyboard(chr(K_ESC))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F1_HELP, , {|| eval(setkey(K_F1))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F2_LIST, , {|| iif( valtype(setkey(K_F2)) == "B", eval(setkey(K_F2)), NIL ) } )
            button( @aButtons, row(), col()+1,;
                _BUTTON_PGDN_CONFIRM, , {||keyboard(chr(K_PGDN))} )

            *-----------------------------------------------------
            * Read.
            *-----------------------------------------------------

            read( aoGet, , aButtons )
            aoGet   := {}
            aButtons := {}

            *-----------------------------------------------------
            * Check what was read.
            *-----------------------------------------------------

            do case
            case lastkey() = K_ESC

                *-------------------------------------------------
                * [Esc] means leave.
                *-------------------------------------------------

                lGoOn := .F.

                exit                                    // EXIT

            case lastkey() = K_PGDN

                *-------------------------------------------------
                * [PgDn] means confirm.
                * Check for valid data.
                * Ask for another confirmation before proceeding.
                *-------------------------------------------------

                if  empty(cLabelFile)                   .or.;
                    !isFile(;
                        strAddExtention(;
                            cLabelFile,;
                            _EXTENTION_LABEL );
                    )                                   .or.;
                    empty(cWhileCondition)              .or.;
                    empty(cForCondition)
                    
                    loop                                // LOOP
                
                end

                *-------------------------------------------------
                * Data seems to be correct.
                *-------------------------------------------------

                lGoOn := .T.

                *-------------------------------------------------
                * Transform character conditions into codeblocks.
                *-------------------------------------------------

                bWhileCondition :=;
                    bCompile( alltrim(cWhileCondition) )

                bForCondition :=;
                    bCompile(;
                        alltrim(cForCondition) +;
                        " .and. waitFileEval()";
                    )

                exit                                    // EXIT

            otherwise

                *-------------------------------------------------
                * Exit with different keys is not allowed.
                * Loop again.
                *-------------------------------------------------

            end

        end

        *---------------------------------------------------------
        * The editing is terminated, close the window.
        *---------------------------------------------------------

        mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

        *---------------------------------------------------------
        * If so requested, proceed.
        *---------------------------------------------------------

        if lGoOn

            *-----------------------------------------------------
            * As a file process is going to start, prepare
            * to handle errors.
            *-----------------------------------------------------

            bOldErrorHandler := errorblock( {|e| ErrorHandler(e)} )
            begin sequence

                *-------------------------------------------------
                * Check if the print goes on console.
                *-------------------------------------------------

                if isConsoleOn()

                    *---------------------------------------------
                    * Before a console print, change colors and
                    * prepare an empy screen.
                    *---------------------------------------------

                    setcolor( COLOR_BODY )

                    scroll()

                end

                *-------------------------------------------------
                * Print.
                *-------------------------------------------------

                __LabelForm(;
                    cLabelFile, .f., , .f.,;
                    bForCondition, bWhileCondition, , , .f., .f.;
                )

                *-------------------------------------------------
                * Close waitProgress().
                *-------------------------------------------------

                waitFileEval( .T. )

                *-------------------------------------------------
                * If output is on the console, wait for a key.
                *-------------------------------------------------

                if isConsoleOn()

                    *---------------------------------------------
                    * Wait for a key press.
                    *---------------------------------------------

                    alertbuttons( maxrow(), 0, {"[Esc]"} )
                    &&&&inkey(0)

                    scroll()

                end

                *-------------------------------------------------
                * The record pointer is maybe at bottom.
                * No action is taken to move the record pointer.
                * dbgotop()
                *-------------------------------------------------

                *-------------------------------------------------
                * Macro recorder
                *-------------------------------------------------

                if isMemvar( _MEMVAR_MACRO_RECORDING_NAME )

                    Memvar->_MEMVAR_MACRO_RECORDER +=;
                        NL(1) +;
                        '// Label print.' +;
                        NL(1) +;
                        '__LabelForm( "' +;
                        alltrim(cLabelFile) +;
                        '", .F., NIL, .F., {|| ' +;
                        alltrim(cForCondition) +;
                        ' }, {|| ' +;
                        alltrim(cWhileCondition) +;
                        '},,, .F., .F. )' +;
                        NL(1)

                end

            recover

                *-------------------------------------------------
                * Try to remedy if error occurred.
                *-------------------------------------------------

                dbgotop()

            end //sequence
            errorblock(bOldErrorHandler)

            waitFileEval( .T. )

        end

    end //sequence

    *-------------------------------------------------------------
    * Restore old data and return.
    *-------------------------------------------------------------

    setcursor( nOldCursor )
    setcolor( cOldColor )
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )

    return NIL

*-----------------------------------------------------------------
static function lblModify()
*
* lblModify() --> NIL
*
* It executes the lbl() function asking for the file name to
* modify.
*

    local aoGet             := {}
    local aButtons          := {}
    local aTab              := TAB_DEFAULT

    local bOldErrorHandler
    local cOldScreen
    local cOldColor         := setcolor()
    local nOldCursor        := setcursor( SETCURSOR_NORMAL )
    local bOld_F1           :=;
        setkey( K_F1, {|| nbHelp( "<REPORT - MODIFY LABEL>", , chr(K_ENTER) ) } )
    local bOld_F2           := setkey( K_F2, NIL )
    local nOldRow           := row()
    local nOldCol           := col()
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth

    local lGoOn             := .F.
    local cLabelFile        :=;
        padr( "*." + _EXTENTION_LABEL, _MAX_STRING_LEN )

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Create a kind of window.
        *---------------------------------------------------------

        nBottom     := maxrow()
        nTop        := nBottom - 6
        nLeft       := 0
        nRight      := maxcol()
        nWidth      := nRight - nLeft +1

    	aTab[TAB_LEFT]      := nLeft
	    aTab[TAB_RIGHT]     := nRight
    	aTab[TAB_TAB_ARRAY] := { 16, 4 }
	
        cOldScreen  := mouseScrSave( nTop, nLeft, nBottom, nRight )
	
        setcolor( COLOR_BODY )
	
        scroll( nTop, nLeft, nBottom, nRight )
	
        dispBoxShadow(;
            nTop, nLeft, nBottom, nRight,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )
	
        say(;
            nTop+1, nLeft+1,;
            padc( LBL_WINDOW_LABEL, nWidth-2 ),;
            ,;
            COLOR_HEAD;
        )

        *---------------------------------------------------------
        * Do editing.
        *---------------------------------------------------------

        while .T.                                       // FOREVER

            say( nTop+3, nLeft+1, LBL_PROMPT_OPEN_FILE_LABEL )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cLabelFile := x, cLabelFile ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { ||;
                    setkey(;
                        K_F2,;
                        { || gvFileDir( @cLabelFile ) };
                    ),;
                    .T.;
                },;
                { || gvFileExist( @cLabelFile, _EXTENTION_LABEL ) };
            )

            button( @aButtons, row()+2, nLeft+1,;
                _BUTTON_ESC_CANCEL, , {||keyboard(chr(K_ESC))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F1_HELP, , {|| eval(setkey(K_F1))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F2_LIST, , {|| iif( valtype(setkey(K_F2)) == "B", eval(setkey(K_F2)), NIL ) } )
            button( @aButtons, row(), col()+1,;
                _BUTTON_PGDN_CONFIRM, , {||keyboard(chr(K_PGDN))} )

            *-----------------------------------------------------
            * Read.
            *-----------------------------------------------------

            read( aoGet, , aButtons )
            aoGet   := {}
            aButtons := {}

            *-----------------------------------------------------
            * Check what was read.
            *-----------------------------------------------------

            do case
            case lastkey() = K_ESC

                *-------------------------------------------------
                * [Esc] means leave.
                *-------------------------------------------------

                lGoOn := .F.

                exit                                    // EXIT

            case lastkey() = K_PGDN

                *-------------------------------------------------
                * [PgDn] means confirm.
                * Check for valid data.
                *-------------------------------------------------

                if  file(;
                        strAddExtention(;
                            cLabelFile,;
                            _EXTENTION_LABEL;
                        );
                    )

                    lGoOn := .T.

                    exit                                // EXIT

                else

                    *---------------------------------------------
                    * Data is not valid: loop again.
                    *---------------------------------------------

                end

            otherwise

                *-------------------------------------------------
                * Exit with different keys is not allowed.
                * Loop again.
                *-------------------------------------------------

            end

        end

        *---------------------------------------------------------
        * Close window.
        *---------------------------------------------------------

        mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

        *---------------------------------------------------------
        * Modify the label if so instructed.
        *---------------------------------------------------------

        if lGoOn

            lbl( cLabelFile )

        end

    end //sequence

    *-------------------------------------------------------------
    * Restore old data.
    *-------------------------------------------------------------

    setcursor( nOldCursor )
    setcolor( cOldColor )
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )

    return NIL


#endif  // RUNTIME

#ifndef RUNTIME
*-----------------------------------------------------------------
* MACRO - MACRO FUNCTIONS
*-----------------------------------------------------------------

#define MACRO_DIALOG_BOX_TOP_COMPILE;
    "Macro Compilation"

#define MACRO_DIALOG_BOX_TOP_OPEN;
    "Macro Load and execute"

#define MACRO_DIALOG_BOX_TOP_EDIT;
    "Macro Edit"

#define MACRO_DIALOG_BOX_TOP_SAVE;
    "Save Macro"

#define MACRO_PROMPT_SOURCE_FILE;
    "Source macro filename:"
#define MACRO_PROMPT_DESTINATION_FILE;
    "Destination macro filename:"

#define MACRO_ALERT_ERASE;
    "Do you want to erase the macro recording?"

#define MACRO_PROMPT_LOAD;
    "Insert the macro file name:"

#define MACRO_PROMPT_PARAMETERS;
    "Parameters to send to the macro (Par1 .. Par9)"
#define MACRO_PROMPT_LOAD_PAR1;
    "Par1"
#define MACRO_PROMPT_LOAD_PAR2;
    "Par2"
#define MACRO_PROMPT_LOAD_PAR3;
    "Par3"
#define MACRO_PROMPT_LOAD_PAR4;
    "Par4"
#define MACRO_PROMPT_LOAD_PAR5;
    "Par5"
#define MACRO_PROMPT_LOAD_PAR6;
    "Par6"
#define MACRO_PROMPT_LOAD_PAR7;
    "Par7"
#define MACRO_PROMPT_LOAD_PAR8;
    "Par8"
#define MACRO_PROMPT_LOAD_PAR9;
    "Par9"

#define MACRO_PROMPT_SAVE_MACRO;
    "Insert the macro name:"

*-----------------------------------------------------------------
static function macroCompile()
*
* It asks for the name of a macro to compile.
*

    local aoGet             := {}
    local aButtons          := {}
    local aTab              := TAB_DEFAULT
    
    local bOldErrorHandler
    local nOldSelect        := select()
    local cOldScreen
    local cOldColor         := setcolor()
    local nOldCursor        := setcursor( SETCURSOR_NORMAL )
    local bOld_F1           :=;
        setkey( K_F1, {|| nbHelp( "<MACRO - MACRO COMPILATION>", , chr(K_ENTER) ) } )

    local bOld_F2           := setkey( K_F2, NIL )
    local nOldRow           := row()
    local nOldCol           := col()
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth

    local lGoOn             := .T.

    local cMacroName        :=;
        padr( "*." + _EXTENTION_MACRO_ASCII, _MAX_STRING_LEN )
    local cCompiledName     := space(_MAX_STRING_LEN)

    *-------------------------------------------------------------
    * Start a sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Create a kind of window right aligned.
        *---------------------------------------------------------

        nBottom     := maxrow()
        nTop        := nBottom - 7
        nLeft       := 0
        nRight      := maxcol()
        nWidth      := nRight - nLeft +1

        aTab[TAB_LEFT]      := nLeft
        aTab[TAB_RIGHT]     := nRight
        aTab[TAB_TAB_ARRAY] := { 30, 4 }

        cOldScreen  := mouseScrSave( nTop, nLeft, nBottom, nRight )

        setcolor( COLOR_BODY )

        scroll( nTop, nLeft, nBottom, nRight )

        dispBoxShadow(;
            nTop, nLeft, nBottom, nRight,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )

        say(;
            nTop+1, nLeft+1,;
            padc(;
                MACRO_DIALOG_BOX_TOP_COMPILE,;
                nWidth-2;
                ),;
            ,;
            COLOR_HEAD;
        )

        *---------------------------------------------------------
        * Do editing.
        *---------------------------------------------------------

        while .T.                                       // FOREVER

            say( nTop+3, nLeft+1, MACRO_PROMPT_SOURCE_FILE )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cMacroName := x, cMacroName ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { ||; 
                    setkey( K_F2, {|| gvFileDir( @cMacroName ) } ),;
                    .T.; 
                },;
                { || gvFileExist( @cMacroName, _EXTENTION_MACRO_ASCII ) },;
            )

            say( row()+1, nLeft+1, MACRO_PROMPT_DESTINATION_FILE )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cCompiledName := x, cCompiledName ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { ||; 
                    setkey( K_F2, {|| gvFileDir( @cCompiledName ) } ),;
                    gvDefault ( @cCompiledName, strCutExtention( cMacroName ) + "." + _EXTENTION_MACRO_COMPILED ),;
                    .T.; 
                },;
                { ||;
                    gvDefault ( @cCompiledName, strCutExtention( cMacroName ) + "." + _EXTENTION_MACRO_COMPILED ),;
                    gvFileExtention( @cCompiledName, _EXTENTION_MACRO_COMPILED );
                };
            )

            button( @aButtons, row()+2, nLeft+1,;
                _BUTTON_ESC_CANCEL, , {||keyboard(chr(K_ESC))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F1_HELP, , {|| eval(setkey(K_F1))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F2_LIST, , {|| iif( valtype(setkey(K_F2)) == "B", eval(setkey(K_F2)), NIL ) } )
            button( @aButtons, row(), col()+1,;
                _BUTTON_PGDN_CONFIRM, , {||keyboard(chr(K_PGDN))} )

            *-----------------------------------------------------
            * Read.
            *-----------------------------------------------------

            read( aoGet, 1, aButtons )
            aoGet    := {}
            aButtons := {}

            *-----------------------------------------------------
            * Test exit key.
            *-----------------------------------------------------

            do case
            case lastkey() = K_ESC

                *-------------------------------------------------
                * [Esc] means leave.
                *-------------------------------------------------

                lGoOn := .F.

                exit                                    // EXIT

            case lastkey() = K_PGDN // choice confirmation

                *-------------------------------------------------
                * Verify to have correct data.
                *-------------------------------------------------

                if  isFile( alltrim(cMacroName) )       .and.;
                    !empty( strCutExtention( cCompiledName ) );
                    .and.;
                    !isWild( alltrim( cCompiledName ) )

                    lGoOn := .T.

                    exit                                // EXIT

                else

                    *---------------------------------------------
                    * Data is not valid: loop.
                    *---------------------------------------------

                end

            otherwise

                *-------------------------------------------------
                * Exit with different keys is not allowed.
                * Loop again.
                *-------------------------------------------------

            end

        end

        *---------------------------------------------------------
        * Delete window.
        *---------------------------------------------------------

        mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

        *---------------------------------------------------------
        * Restore Colors.
        *---------------------------------------------------------

        setcolor( cOldColor )

        *---------------------------------------------------------
        * Proceed if so was decided.
        *---------------------------------------------------------

        if lGoOn

            *-----------------------------------------------------
            * Start macro compilation.
            *-----------------------------------------------------

            cm( cMacroName, cCompiledName )

        end

    end //sequence

    *-------------------------------------------------------------
    * Restore old data and return.
    *-------------------------------------------------------------

    select( nOldSelect )
    setcursor( nOldCursor )
    setcolor( cOldColor )
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )

    return NIL

*-----------------------------------------------------------------
static function macroLoad()
*
* It asks for the name of a macro to load and execute.
*

    local aoGet             := {}
    local aButtons          := {}
    local aTab              := TAB_DEFAULT

    local bOldErrorHandler
    local nOldSelect        := select()
    local cOldScreen
    local cOldColor         := setcolor()
    local nOldCursor        := setcursor( SETCURSOR_NORMAL )
    local bOld_F1           :=;
        setkey( K_F1, {|| nbHelp( "<MACRO - LOAD + EXECUTE MACRO>", , chr(K_ENTER) ) } )

    local bOld_F2           := setkey( K_F2, NIL )
    local nOldRow           := row()
    local nOldCol           := col()
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth

    local lGoOn             := .T.

    local cMacroName        :=;
        padr( "*.*", _MAX_STRING_LEN )
    local cPar1             := space(_MAX_STRING_LEN)
    local cPar2             := space(_MAX_STRING_LEN)
    local cPar3             := space(_MAX_STRING_LEN)
    local cPar4             := space(_MAX_STRING_LEN)
    local cPar5             := space(_MAX_STRING_LEN)
    local cPar6             := space(_MAX_STRING_LEN)
    local cPar7             := space(_MAX_STRING_LEN)
    local cPar8             := space(_MAX_STRING_LEN)
    local cPar9             := space(_MAX_STRING_LEN)

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Create a kind of window.
        *---------------------------------------------------------

        nBottom     := maxrow()
        nTop        := nBottom - 11
        nLeft       := 0
        nRight      := maxcol()
        nWidth      := nRight - nLeft +1

        aTab[TAB_LEFT]      := nLeft
        aTab[TAB_RIGHT]     := nRight
        aTab[TAB_TAB_ARRAY] := { 8, 20, 8, 20, 8, 20, 8 }
        
        cOldScreen  := mouseScrSave( nTop, nLeft, nBottom, nRight )
        
        setcolor( COLOR_BODY )
        
        scroll( nTop, nLeft, nBottom, nRight )
        
        dispBoxShadow(;
            nTop, nLeft, nBottom, nRight,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )
        
        say(;
            nTop+1, nLeft+1,;
            padc( MACRO_DIALOG_BOX_TOP_OPEN, nWidth-2 ),;
            ,;
            COLOR_HEAD;
        )

        *---------------------------------------------------------
        * Do editing.
        *---------------------------------------------------------

        while .t.                                       // FOREVER

            say( nTop+3, nLeft+1, MACRO_PROMPT_LOAD )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cMacroName := x, cMacroName ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { ||; 
                    setkey( K_F2, {|| gvFileDir( @cMacroName ) } ),;
                    .T.; 
                },;
                { || gvFileExist( @cMacroName, _EXTENTION_MACRO_COMPILED ) },;
            )

            say( row()+2, nLeft+1, MACRO_PROMPT_PARAMETERS )

            say( row()+1, nLeft+1, MACRO_PROMPT_LOAD_PAR1 )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cPar1 := x, cPar1 ) },;
                "@s16@",;
                ,;
                { ||; 
                    setkey( K_F2, {|| gvFileDir( @cPar1 ) } ),;
                    .T.; 
                },;
                { || .T. },;
            )
            tab( aTab )
            say( row(), col(), MACRO_PROMPT_LOAD_PAR2 )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cPar2 := x, cPar2 ) },;
                "@s16@",;
                ,;
                { ||; 
                    setkey( K_F2, {|| gvFileDir( @cPar2 ) } ),;
                    .T.; 
                },;
                { || .T. },;
            )
            tab( aTab )
            say( row(), col(), MACRO_PROMPT_LOAD_PAR3 )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cPar3 := x, cPar3 ) },;
                "@s16@",;
                ,;
                { ||; 
                    setkey( K_F2, {|| gvFileDir( @cPar3 ) } ),;
                    .T.; 
                },;
                { || .T. },;
            )

            say( row()+1, nLeft+1, MACRO_PROMPT_LOAD_PAR4 )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cPar4 := x, cPar4 ) },;
                "@s16@",;
                ,;
                { ||; 
                    setkey( K_F2, {|| gvFileDir( @cPar4 ) } ),;
                    .T.; 
                },;
                { || .T. },;
            )
            tab( aTab )
            say( row(), col(), MACRO_PROMPT_LOAD_PAR5 )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cPar5 := x, cPar5 ) },;
                "@s16@",;
                ,;
                { ||; 
                    setkey( K_F2, {|| gvFileDir( @cPar5 ) } ),;
                    .T.; 
                },;
                { || .T. },;
            )
            tab( aTab )
            say( row(), col(), MACRO_PROMPT_LOAD_PAR6 )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cPar6 := x, cPar6 ) },;
                "@s16@",;
                ,;
                { ||; 
                    setkey( K_F2, {|| gvFileDir( @cPar6 ) } ),;
                    .T.; 
                },;
                { || .T. },;
            )

            say( row()+1, nLeft+1, MACRO_PROMPT_LOAD_PAR7 )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cPar7 := x, cPar7 ) },;
                "@s16@",;
                ,;
                { ||; 
                    setkey( K_F2, {|| gvFileDir( @cPar7 ) } ),;
                    .T.; 
                },;
                { || .T. },;
            )
            tab( aTab )
            say( row(), col(), MACRO_PROMPT_LOAD_PAR8 )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cPar8 := x, cPar8 ) },;
                "@s16@",;
                ,;
                { ||; 
                    setkey( K_F2, {|| gvFileDir( @cPar8 ) } ),;
                    .T.; 
                },;
                { || .T. },;
            )
            tab( aTab )
            say( row(), col(), MACRO_PROMPT_LOAD_PAR9 )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cPar9 := x, cPar9 ) },;
                "@s16@",;
                ,;
                { ||; 
                    setkey( K_F2, {|| gvFileDir( @cPar9 ) } ),;
                    .T.; 
                },;
                { || .T. },;
            )

            button( @aButtons, row()+2, nLeft+1,;
                _BUTTON_ESC_CANCEL, , {||keyboard(chr(K_ESC))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F1_HELP, , {|| eval(setkey(K_F1))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F2_LIST, , {|| iif( valtype(setkey(K_F2)) == "B", eval(setkey(K_F2)), NIL ) } )
            button( @aButtons, row(), col()+1,;
                _BUTTON_PGDN_CONFIRM, , {||keyboard(chr(K_PGDN))} )

            *-----------------------------------------------------
            * Read.
            *-----------------------------------------------------

            read( aoGet, , aButtons )
            aoGet   := {}
            aButtons := {}

            *-----------------------------------------------------
            * Check what was read.
            *-----------------------------------------------------

            do case
            case lastkey() = K_ESC

                *-------------------------------------------------
                * [Esc] means leave.
                *-------------------------------------------------

                lGoOn := .F.

                exit                                    // EXIT

            case lastkey() = K_PGDN

                *-------------------------------------------------
                * [PgDn] means confirm.
                * Verify to have correct data.
                *-------------------------------------------------

                if  gvFileExist( @cMacroName, _EXTENTION_MACRO_COMPILED )

                    lGoOn := .T.

                    exit                                // EXIT

                else

                    *---------------------------------------------
                    * As the file don't exist: loop.
                    *---------------------------------------------

                end

            otherwise

                *-------------------------------------------------
                * Exit with different keys is not allowed.
                * Loop again.
                *-------------------------------------------------

            end

        end

        *---------------------------------------------------------
        * Delete window.
        *---------------------------------------------------------

        mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

        *---------------------------------------------------------
        * Restore Colors.
        *---------------------------------------------------------

        setcolor( cOldColor )

        *---------------------------------------------------------
        * If so was selected: proceed.
        *---------------------------------------------------------

        if lGoOn

            *-----------------------------------------------------
            * As a macro is going to be executed: turn off the
            * nB HELP.
            *-----------------------------------------------------

            setkey( K_F1, NIL )

            *-----------------------------------------------------
            * Run the macro executor, but before, transfer
            */ public variables.
            *-----------------------------------------------------

            Memvar->_MEMVAR_PAR0 := alltrim( cMacroName )
            Memvar->_MEMVAR_PAR1 := alltrim( cPar1 )
            Memvar->_MEMVAR_PAR2 := alltrim( cPar2 )
            Memvar->_MEMVAR_PAR3 := alltrim( cPar3 )
            Memvar->_MEMVAR_PAR4 := alltrim( cPar4 )
            Memvar->_MEMVAR_PAR5 := alltrim( cPar5 )
            Memvar->_MEMVAR_PAR6 := alltrim( cPar6 )
            Memvar->_MEMVAR_PAR7 := alltrim( cPar7 )
            Memvar->_MEMVAR_PAR8 := alltrim( cPar8 )
            Memvar->_MEMVAR_PAR9 := alltrim( cPar9 )

            ex( alltrim(cMacroName) )

        end

    end //sequence

    *-------------------------------------------------------------
    * Restore old data and return.
    *-------------------------------------------------------------

    select( nOldSelect )
    setcursor( nOldCursor )
    setcolor( cOldColor )
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )

    return NIL

*-----------------------------------------------------------------
static function macroEdit()
*
* Edit memvar->c_MacroRec
*

    local bOldErrorHandler
    local cOldScreen
    local cOldColor           := setcolor()
    local nOldCursor          := setcursor( SETCURSOR_NORMAL )
    local bOld_F1             :=;
        setkey( K_F1, {|| nbHelp( "<MACRO - EDIT RECORDING>", , chr(K_ENTER) ) } )

    local bOld_F2             := setkey( K_F2, NIL )
    local nOldRow             := row()
    local nOldCol             := col()
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Create a kind of window.
        *---------------------------------------------------------

        nBottom     := maxrow()
        nTop        := 0
        nLeft       := 0
        nRight      := maxcol()
        nWidth      := nRight - nLeft +1
        
        cOldScreen  := mouseScrSave( nTop, nLeft, nBottom, nRight )
        
        setcolor( COLOR_BODY )
        
        scroll( nTop, nLeft, nBottom, nRight )
        
        dispBoxShadow(;
            nTop, nLeft, nBottom, nRight,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )
        
        say(;
            nTop+1, nLeft+1,;
            padc( MACRO_DIALOG_BOX_TOP_EDIT, nWidth-2 ),;
            ,;
            COLOR_HEAD;
        )

        say(;
             nBottom-2, nLeft+1,;
             padc( _KEY_ESC_CANCEL_F1, nWidth-2 ),;
            ,;
            COLOR_HEAD;
        )

        say(;
             nBottom-1, nLeft+1,;
             padc( _KEY_CTRLW, nWidth-2 ),;
            ,;
            COLOR_HEAD;
        )

        *---------------------------------------------------------
        * Do editing.
        *---------------------------------------------------------

        memvar->_MEMVAR_MACRO_RECORDER :=;
            memoedit(;
                memvar->_MEMVAR_MACRO_RECORDER,;
                2, 1, maxrow()-3, maxcol()-1,;
                .T., NIL, _MAX_STRING_LEN;
            )

        *---------------------------------------------------------
        * Restore screen deleting window.
        *---------------------------------------------------------

        mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

    end //sequence

    *-------------------------------------------------------------
    * Restore old data and return.
    *-------------------------------------------------------------

    setcursor( nOldCursor )
    setcolor( cOldColor )
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )

    return NIL

*-----------------------------------------------------------------
static function macroSave()


    local aoGet             := {}
    local aButtons          := {}
    local aTab              := TAB_DEFAULT

    local bOldErrorHandler
    local cOldScreen
    local cOldColor         := setcolor()
    local nOldCursor        := setcursor( SETCURSOR_NORMAL )
    local bOld_F1           :=;
        setkey( K_F1, {|| nbHelp( "<MACRO - SAVE RECORDING>", , chr(K_ENTER) ) } )

    local bOld_F2           := setkey( K_F2, NIL )
    local nOldRow           := row()
    local nOldCol           := col()
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth

    local lGoOn             := .T.

    local cMacroName        :=;
        padr( "*." + _EXTENTION_MACRO_ASCII, _MAX_STRING_LEN )

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Create a kind of window.
        *---------------------------------------------------------

        nBottom     := maxrow()
        nTop        := nBottom - 6
        nLeft       := 0
        nRight      := maxcol()
        nWidth      := nRight - nLeft +1

        aTab[TAB_LEFT]      := nLeft
        aTab[TAB_RIGHT]     := nRight
        aTab[TAB_TAB_ARRAY] := { 24, 4 }
        
        cOldScreen  := mouseScrSave( nTop, nLeft, nBottom, nRight )
        
        setcolor( COLOR_BODY )
        
        scroll( nTop, nLeft, nBottom, nRight )
        
        dispBoxShadow(;
            nTop, nLeft, nBottom, nRight,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )
        
        say(;
            nTop+1, nLeft+1,;
            padc( MACRO_DIALOG_BOX_TOP_SAVE, nWidth-2 ),;
            ,;
            COLOR_HEAD;
        )

        *---------------------------------------------------------
        * Do editing.
        *---------------------------------------------------------

        while .t.                                       // FOREVER

            say( nTop+3, nLeft+1, MACRO_PROMPT_SAVE_MACRO )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cMacroName := x, cMacroName ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { ||;
                    setkey( K_F2, {|| gvFileDir( @cMacroName ) } ),;
                    .T.;
                },;
                { || gvFileExtention( @cMacroName, _EXTENTION_MACRO_ASCII ) };
            )

            button( @aButtons, row()+2, nLeft+1,;
                _BUTTON_ESC_CANCEL, , {||keyboard(chr(K_ESC))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F1_HELP, , {|| eval(setkey(K_F1))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F2_LIST, , {|| iif( valtype(setkey(K_F2)) == "B", eval(setkey(K_F2)), NIL ) } )
            button( @aButtons, row(), col()+1,;
                _BUTTON_PGDN_CONFIRM, , {||keyboard(chr(K_PGDN))} )

            *-----------------------------------------------------
            * Read.
            *-----------------------------------------------------

            read( aoGet, , aButtons )
            aoGet   := {}
            aButtons := {}

            *-----------------------------------------------------
            * Check what was read.
            *-----------------------------------------------------

            do case
            case lastkey() = K_ESC

                *-------------------------------------------------
                * [Esc] means leave.
                *-------------------------------------------------

                lGoOn := .F.

                exit                                    // EXIT

            case lastkey() = K_PGDN

                *-------------------------------------------------
                * [PgDn] means confirm.
                *-------------------------------------------------

                if  empty( strCutExtention( cMacroName) );
                    .or.;
                    isWild( cMacroName );
                    .or.;
                    (;
                        file( cMacroName );
                        .and.;
                        (;
                            alertBox(;
                                alltrim(cMacroName) +;
                                NL(1) +;
                                _ERROR_PROMPT_ALREADY_EXIST_OVERWRITE,;
                                { _MENU_NO, _MENU_YES };
                            ) == 1;
                        );
                    )

                    *---------------------------------------------
                    * There is no macro name or the file already
                    * exist and no overwrite was selected.
                    * Loop
                    *---------------------------------------------

                else

                    *---------------------------------------------
                    * The data seems to be ok.
                    *---------------------------------------------

                    lGoOn := .T.

                    exit                                // EXIT

                end

            otherwise

                *-------------------------------------------------
                * Exit with different keys is not allowed.
                * Loop again.
                *-------------------------------------------------

            end

        end

        *---------------------------------------------------------
        * Restore screen deleting window.
        *---------------------------------------------------------

        mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

        *---------------------------------------------------------
        * If so was selected: proceed.
        *---------------------------------------------------------

        if lGoOn

            *-----------------------------------------------------
            * ok save.
            *-----------------------------------------------------

            memowrit(;
                alltrim(cMacroName),;
                Memvar->_MEMVAR_MACRO_RECORDER;
            )

        else

            break                                       // BREAK

        end

    end //sequence

    *-------------------------------------------------------------
    * Restore old data and return.
    *-------------------------------------------------------------

    setcursor( nOldCursor )
    setcolor( cOldColor )
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )

    return NIL

#endif  // RUNTIME

#ifndef RUNTIME
*------------------------------------------------------------------
* MNU - MENU SYSTEM
*------------------------------------------------------------------

#define MNU_MENU_FILE_CHOICE        "File"
#define MNU_MENU_EDIT_CHOICE        "Edit"
#define MNU_MENU_REPORT_CHOICE      "Report"
#define MNU_MENU_HTF_CHOICE         "Htf"
#define MNU_MENU_MACRO_CHOICE       "Macro"
#define MNU_MENU_INFO_CHOICE        "Info"
#define MNU_MENU_DOC_CHOICE         "Doc"

#define MNU_MENU_FILE_MSG;
    "File menu"
#define MNU_MENU_EDIT_MSG;
    "Editing menu"
#define MNU_MENU_REPORT_MSG;
    "Report menu"
#define MNU_MENU_HTF_MSG;
    "Help Text File menu"
#define MNU_MENU_MACRO_MSG;
    "Macro menu"
#define MNU_MENU_INFO_MSG;
    "Information menu"
#define MNU_MENU_DOC_MSG;
    "Text/Document menu"

#define MNU_MENU_FILE_NEWDIR_CHOICE;
    padr( "Change Directory",;
    _MENU_PULLDOWN_WIDTH-6 ) + "... "
#define MNU_MENU_FILE_DBF_CHOICE;
    padr( "File ." + _EXTENTION_DBF,;
    _MENU_PULLDOWN_WIDTH-3 ) + ""
#define MNU_MENU_FILE_NTX_CHOICE;
    padr( "File ." + _EXTENTION_INDEXBAG,;
    _MENU_PULLDOWN_WIDTH-3 ) + ""
#define MNU_MENU_FILE_ALIAS_CHOICE;
    padr( "Alias",;
    _MENU_PULLDOWN_WIDTH-3 ) + ""
#define MNU_MENU_FILE_INDEX_CHOICE;
    padr( "Order",;
    _MENU_PULLDOWN_WIDTH-3 ) + ""
#define MNU_MENU_FILE_RELATION_CHOICE;
    padr( "Relation",;
    _MENU_PULLDOWN_WIDTH-3 ) + ""
#define MNU_MENU_FILE_RDD_DEFAULT_CHOICE;
    padr( "RDD Default",;
    _MENU_PULLDOWN_WIDTH-3 ) + ""

#define MNU_MENU_FILE_NEWDIR_MSG;
    "Change the actual directory"
#define MNU_MENU_FILE_DBF_MSG;
    "." + _EXTENTION_DBF + " file operations"
#define MNU_MENU_FILE_NTX_MSG;
    "." + _EXTENTION_INDEXBAG + " file operations"
#define MNU_MENU_FILE_ALIAS_MSG;
    "Alias operations"
#define MNU_MENU_FILE_INDEX_MSG;
    "Index/Order operations"
#define MNU_MENU_FILE_RELATION_MSG;
    "Relation operations"
#define MNU_MENU_FILE_RDD_DEFAULT_MSG;
   "Replaceable Database Driver"

#define MNU_MENU_FILE_DBF_NEW_CHOICE;
    padr( "New ." + _EXTENTION_DBF,;
    _MENU_PULLDOWN_WIDTH-6 ) + "..." + "&"
#define MNU_MENU_FILE_DBF_MODSTRU_CHOICE;
    padr( "Modify ." + _EXTENTION_DBF + " structure",;
    _MENU_PULLDOWN_WIDTH-6 ) + "... "
#define MNU_MENU_FILE_DBF_OPEN_CHOICE;
    padr( "Open ." + _EXTENTION_DBF,;
    _MENU_PULLDOWN_WIDTH-6 ) + "..." + "&"

#define MNU_MENU_FILE_DBF_NEW_MSG;
    "Create a new ." + _EXTENTION_DBF + " file"
#define MNU_MENU_FILE_DBF_OPEN_MSG;
    "Open a ." + _EXTENTION_DBF + " file"
#define MNU_MENU_FILE_DBF_MODSTRU_MSG;
    "Modify the structure of a ." + _EXTENTION_DBF + " file"

#define MNU_MENU_FILE_NTX_NEW_CHOICE;
    padr( "New ." + _EXTENTION_INDEXBAG + " / New Tag",;
    _MENU_PULLDOWN_WIDTH-6 ) + "..." + "&"
#define MNU_MENU_FILE_NTX_OPEN_CHOICE;
    padr( "Open Index",;
    _MENU_PULLDOWN_WIDTH-6 ) + "..." + "&"

#define MNU_MENU_FILE_NTX_NEW_MSG;
    "Create a new index for the actual Alias or add a Tag"
#define MNU_MENU_FILE_NTX_OPEN_MSG;
    "Open and index for the actual Alias"

#define MNU_MENU_FILE_ALIAS_SELECT_CHOICE;
    padr( "Select",;
    _MENU_PULLDOWN_WIDTH-6 ) + "..." + "&"
#define MNU_MENU_FILE_ALIAS_DISPSTRU_CHOICE;
    padr( "Display structure",;
    _MENU_PULLDOWN_WIDTH-2 )
#define MNU_MENU_FILE_ALIAS_CLOSE_CHOICE;
    padr( "Close active Alias",;
    _MENU_PULLDOWN_WIDTH-3 ) + "&"
#define MNU_MENU_FILE_ALIAS_CLOSEALL_CHOICE;
    padr( "Close all Aliases",;
    _MENU_PULLDOWN_WIDTH-3 ) + "&"

#define MNU_MENU_FILE_ALIAS_SELECT_MSG;
    "Select another Alias or a new area"
#define MNU_MENU_FILE_ALIAS_DISPSTRU_MSG;
    "Show the actual Alias scructure"
#define MNU_MENU_FILE_ALIAS_CLOSE_MSG;
    "Close the actual Alias"
#define MNU_MENU_FILE_ALIAS_CLOSEALL_MSG;
    "Close all areas"

#define MNU_MENU_FILE_INDEX_REINDEX_CHOICE;
    padr( "Order List Rebuild",;
    _MENU_PULLDOWN_WIDTH-3 ) + "&"
#define MNU_MENU_FILE_INDEX_SET_ORDER_CHOICE;
    padr( "Order Set Focus",;
    _MENU_PULLDOWN_WIDTH-6 ) + "..." + "&"
#define MNU_MENU_FILE_INDEX_CLOSENTX_CHOICE;
    padr( "Order List Clear",;
    _MENU_PULLDOWN_WIDTH-3 ) + "&"

#define MNU_MENU_FILE_INDEX_SET_ORDER_MSG;
    "Select another index order for the actual Alias"
#define MNU_MENU_FILE_INDEX_REINDEX_MSG;
    "Index rebuild"
#define MNU_MENU_FILE_INDEX_CLOSENTX_MSG;
    "Close indexes for the actual Alias"

#define MNU_MENU_FILE_RELATION_SET_CHOICE;
    padr( "Set Relation",;
    _MENU_PULLDOWN_WIDTH-6 ) + "..." + "&"
#define MNU_MENU_FILE_RELATION_CLOSE_CHOICE;
    padr( "Clear Relation",;
    _MENU_PULLDOWN_WIDTH-3 ) + "&"

#define MNU_MENU_FILE_RELATION_SET_MSG;
    "Define a relation"
#define MNU_MENU_FILE_RELATION_CLOSE_MSG;
    "Relise all relations for the actual Alias"

#define MNU_MENU_FILE_RDD_SHOW_CHOICE;
    padr( "Show actual RDD default",;
    _MENU_PULLDOWN_WIDTH-2 )
#define MNU_MENU_FILE_RDD_SET_DEFAULT_CHOICE;
    padr( "Set default RDD",;
    _MENU_PULLDOWN_WIDTH-6 ) + "..." + "&"

#define MNU_MENU_FILE_RDD_SHOW_MSG;
    "Show the default RDD"
#define MNU_MENU_FILE_RDD_SET_DEFAULT_MSG;
    "Define a new default RDD"

#define MNU_MENU_EDIT_VIEW_CHOICE;
    padr( "View",;
    _MENU_PULLDOWN_WIDTH-3 ) + "&"
#define MNU_MENU_EDIT_BROWSE_CHOICE;
    padr( "Edit/Browse",;
    _MENU_PULLDOWN_WIDTH-3 ) + "&"
#define MNU_MENU_EDIT_REPLACE_CHOICE;
    padr( "Replace",;
    _MENU_PULLDOWN_WIDTH-6 ) + "..." + "&"
#define MNU_MENU_EDIT_RECALL_CHOICE;
    padr( "Recall",;
    _MENU_PULLDOWN_WIDTH-6 ) + "..." + "&"
#define MNU_MENU_EDIT_DELETE_CHOICE;
    padr( "Delete",;
    _MENU_PULLDOWN_WIDTH-6 ) + "..." + "&"
#define MNU_MENU_EDIT_PACK_CHOICE;
    padr( "Pack",;
    _MENU_PULLDOWN_WIDTH-3 ) + "&"

#define MNU_MENU_EDIT_BROWSE_MSG;
    "Browse the actual Alias"
#define MNU_MENU_EDIT_VIEW_MSG;
    "View the actual Alias"
#define MNU_MENU_EDIT_REPLACE_MSG;
    "Data replacement"
#define MNU_MENU_EDIT_RECALL_MSG;
    "Record undelete"
#define MNU_MENU_EDIT_DELETE_MSG;
    "Record delete"
#define MNU_MENU_EDIT_PACK_MSG;
    "Deleted record elimination"

#define MNU_MENU_REPORT_DBGOTOP_CHOICE;
    padr( "dbgotop()",;
    _MENU_PULLDOWN_WIDTH-6 ) + "   &"
#define MNU_MENU_REPORT_LABEL_NEW_CHOICE;
    padr( "New Label",;
    _MENU_PULLDOWN_WIDTH-6 ) + "... "
#define MNU_MENU_REPORT_LABEL_MODIFY_CHOICE;
    padr( "Modify Label",;
    _MENU_PULLDOWN_WIDTH-6 ) + "... "
#define MNU_MENU_REPORT_LABEL_FORM_CHOICE;
    padr( "Label Form",;
    _MENU_PULLDOWN_WIDTH-6 ) + "..." + "&"
#define MNU_MENU_REPORT_REPORT_NEW_CHOICE;
    padr( "New Report",;
    _MENU_PULLDOWN_WIDTH-6 ) + "... "
#define MNU_MENU_REPORT_REPORT_MODIFY_CHOICE;
    padr( "Modify Report",;
    _MENU_PULLDOWN_WIDTH-6 ) + "... "
#define MNU_MENU_REPORT_REPORT_FORM_CHOICE;
    padr( "Report Form",;
    _MENU_PULLDOWN_WIDTH-6 ) + "..." + "&"
#define MNU_MENU_REPORT_DOCUMENT_NEW_CHOICE;
    padr( "Create/Modify/Print Text",;
    _MENU_PULLDOWN_WIDTH-2 )

#define MNU_MENU_REPORT_DBGOTOP_MSG;
    "Go top record on the active Alias"
#define MNU_MENU_REPORT_LABEL_NEW_MSG;
    "Create a label file '.LBL'"
#define MNU_MENU_REPORT_LABEL_MODIFY_MSG;
    "Modify a label file '.LBL'"
#define MNU_MENU_REPORT_LABEL_FORM_MSG;
    "Print labels"
#define MNU_MENU_REPORT_REPORT_NEW_MSG;
    "Create a report file ." + _EXTENTION_FORM
#define MNU_MENU_REPORT_REPORT_MODIFY_MSG;
    "Modify a report file ." +_EXTENTION_FORM
#define MNU_MENU_REPORT_REPORT_FORM_MSG;
    "Print a report"
#define MNU_MENU_REPORT_DOCUMENT_NEW_MSG;
    "DOC() - Create / Modify / Print a text or document file"

#define MNU_MENU_HTF_GENERATE_CHOICE;
    padr( "New Help Text File",;
    _MENU_PULLDOWN_WIDTH-6 ) + "... "
#define MNU_MENU_HTML_GENERATE_CHOICE;
    padr( "New HTML file",;
    _MENU_PULLDOWN_WIDTH-6 ) + "... "
#define MNU_MENU_HTF_BROWSE_CHOICE;
    padr( "Open Help Text File",;
    _MENU_PULLDOWN_WIDTH-6 ) + "... "

#define MNU_MENU_HTF_GENERATE_MSG;
    "Generate a Help Text File from a Ascii text file"
#define MNU_MENU_HTML_GENERATE_MSG;
    "Generate a HTML File from a Ascii text file"
#define MNU_MENU_HTF_BROWSE_MSG;
    "Open and browse a Help Text File"

#define MNU_MENU_MACRO_START_STOP_CHOICE;
    padr( iif( isMemvar( _MEMVAR_MACRO_RECORDING_NAME ),;
    "Pause Recording",;
    "Start Recording" ),;
    _MENU_PULLDOWN_WIDTH-2 )
#define MNU_MENU_MACRO_SAVE_RECORD_CHOICE;
    padr( "Save Recording",;
    _MENU_PULLDOWN_WIDTH-6 ) + "... "
#define MNU_MENU_MACRO_DELETE_RECORD_CHOICE;
    padr( "Erase Recording",;
    _MENU_PULLDOWN_WIDTH-2 )
#define MNU_MENU_MACRO_EDIT_RECORD_CHOICE;
    padr( "Edit Recording",;
    _MENU_PULLDOWN_WIDTH-2 )
#define MNU_MENU_MACRO_COMPILE_CHOICE;
    padr( 'Macro "compilation"',;
    _MENU_PULLDOWN_WIDTH-6 ) + "... "
#define MNU_MENU_MACRO_LOAD_CHOICE;
    padr( "Load + Execute Macro",;
    _MENU_PULLDOWN_WIDTH-6 ) + "... "

#define MNU_MENU_MACRO_START_STOP_MSG;
    "Start/Stop macro recording"
#define MNU_MENU_MACRO_SAVE_RECORD_MSG;
    "Save macro to file"
#define MNU_MENU_MACRO_DELETE_RECORD_MSG;
    "Delete macro recording"
#define MNU_MENU_MACRO_EDIT_RECORD_MSG;
    "Edit macro recording"
#define MNU_MENU_MACRO_COMPILE_MSG;
    'Macro "compilation"'
#define MNU_MENU_MACRO_LOAD_MSG;
    "Load and execute a macro file"

#define MNU_MENU_INFO_ABOUT_CHOICE;
    padr( "About",;
    _MENU_PULLDOWN_WIDTH-2 )
#define MNU_MENU_INFO_MANUAL_CHOICE;
    padr( "Manual browse",;
    _MENU_PULLDOWN_WIDTH-2 )
#define MNU_MENU_INFO_F1_CHOICE;
    padr( "[F1] help",;
    _MENU_PULLDOWN_WIDTH-2 )
#define MNU_MENU_INFO_F2_CHOICE;
    padr( "[F2] list",;
    _MENU_PULLDOWN_WIDTH-2 )
#define MNU_MENU_INFO_F3_CHOICE;
    padr( "[F3] Alias info",;
    _MENU_PULLDOWN_WIDTH-2 )
#define MNU_MENU_INFO_F5_CHOICE;
    padr( "[F5] Set output to",;
    _MENU_PULLDOWN_WIDTH-6 ) + "..." + "&"
#define MNU_MENU_INFO_F8_CHOICE;
    padr( "[F8] Dot Line Calculator",;
    _MENU_PULLDOWN_WIDTH-2 )

#define MNU_MENU_INFO_ABOUT_MSG;
    "Copyright information"
#define MNU_MENU_INFO_MANUAL_MSG;
    "nB Manual browse"

#define MNU_MENU_DOC_NEW_CHOICE;
    padr( "New",;
    _MENU_PULLDOWN_WIDTH-2 )
#define MNU_MENU_DOC_OPEN_CHOICE;
    padr( "Open",;
    _MENU_PULLDOWN_WIDTH-6 ) + "... "
#define MNU_MENU_DOC_SAVE_CHOICE;
    padr( "Save",;
    _MENU_PULLDOWN_WIDTH-2 )
#define MNU_MENU_DOC_SAVE_AS_CHOICE;
    padr( "Save as",;
    _MENU_PULLDOWN_WIDTH-6 ) + "... "
#define MNU_MENU_DOC_SET_DEFAULT_TO_CHOICE;
    padr( "Set output to",;
    _MENU_PULLDOWN_WIDTH-6 ) + "..." + "&"
#define MNU_MENU_DOC_PRINT_AS_IS_CHOICE;
    padr( "Print as it is",;
    _MENU_PULLDOWN_WIDTH-3 ) + "&"
#define MNU_MENU_DOC_PRINT_ONCE_CHOICE;
    padr( "Print with RPT() once",;
    _MENU_PULLDOWN_WIDTH-3 ) + "&"
#define MNU_MENU_DOC_PRINT_CHOICE;
    padr( "Print with RPT() std",;
    _MENU_PULLDOWN_WIDTH-6 ) +"..." + "&"
#define MNU_MENU_DOC_EXIT_CHOICE;
    padr( "Exit DOC()",;
    _MENU_PULLDOWN_WIDTH-2 )

#define MNU_MENU_DOC_NEW_MSG;
    "Removes currently loaded file from memory"
#define MNU_MENU_DOC_OPEN_MSG;
    "Loads new file into memory"
#define MNU_MENU_DOC_SAVE_MSG;
    "Saves current file"
#define MNU_MENU_DOC_SAVE_AS_MSG;
    "Saves current file with specified name"
#define MNU_MENU_DOC_SET_DEFAULT_TO_MSG;
    "Determinate the output peripheral or file"
#define MNU_MENU_DOC_PRINT_AS_IS_MSG;
    "Prints the actual text/document as it appears on the screen"
#define MNU_MENU_DOC_PRINT_ONCE_MSG;
    "Prints once the actual text/document " +;
    "with the RPT() function"
#define MNU_MENU_DOC_PRINT_MSG;
    "Prints for every logical record " +;
    "the actual text/document with the RPT() function"
#define MNU_MENU_DOC_EXIT_MSG;
    "Exit " + DOC_FUNCTION_NAME

*------------------------------------------------------------------
static procedure mnuLeft()

    *--------------------------------------------------------------
    * Redirect left key for menu.
    *--------------------------------------------------------------

    keyboard( chr(K_ESC)+chr(K_LEFT)+chr(K_ENTER) )

    return

*-----------------------------------------------------------------
static procedure mnuRight()

    *-------------------------------------------------------------
    * Redirect right key for menu.
    *-------------------------------------------------------------

    keyboard( chr(K_ESC)+chr(K_RIGHT)+chr(K_ENTER) )

    return

*-----------------------------------------------------------------
static function mnuvFile( nHLeft, nHRight )

    local aMenu             := {}
    local cOldScreen        := mouseScrSave()
    local bOld_F1           :=;
        setkey( K_F1, {|| nbHelp( "<MENU FILE>", , chr(K_ENTER) )} )
    local nTB               := 2            // top border
    local nLB               :=;
        iif( nHLeft+_MENU_PULLDOWN_WIDTH > maxcol(),;
        nHRight - _MENU_PULLDOWN_WIDTH,;
        nHLeft )                            // left border
    local nBB               := nTB +9       // bottom border
    local nRB               :=;
        nLB +_MENU_PULLDOWN_WIDTH -1         // right border
    local n

    local cSaveScreen
    local bSaveF1

    local nMnuChoice := 0

    local dummy

    local lRetBreak := .F.

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        while .T.                                       // FOREVER

            *-----------------------------------------------------
            * Show the updated status line.
            *-----------------------------------------------------

            statusLine()

            *-----------------------------------------------------
            * Prepare colors for menu.
            *-----------------------------------------------------

            setcolor( COLOR_MENU )

            *-----------------------------------------------------
            * Show the box around the menu.
            *-----------------------------------------------------

            dispBoxShadow(;
                nTB, nLB, nBB, nRB,;
                1,;
                dispBoxColor( 1 ),;
                dispBoxColor( 2 );
            )

            *-----------------------------------------------------
            * Display key reminder.
            *-----------------------------------------------------

            say( maxrow(), 00, padc( _KEY_MENU_V, maxcol()+1 ) )

            *-----------------------------------------------------
            * <n> contains the menu choice. At the beguinning it
            * starts from zero.
            *-----------------------------------------------------

            n := 0

            *-----------------------------------------------------
            * Start the sequence of menu items and separation
            * lines.
            *-----------------------------------------------------

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_FILE_NEWDIR_CHOICE,;
                {||;
                    say(;
                        maxrow()-1, 0,;
                        padc( MNU_MENU_FILE_NEWDIR_MSG, maxcol()+1 );
                    );
                };
            )

            say(;
                nTB+(++n), nLB+1,;
                replicate("", _MENU_PULLDOWN_WIDTH-2);
            )

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_FILE_DBF_CHOICE,;
                {||;
                    say(;
                        maxrow()-1, 0,;
                        padc( MNU_MENU_FILE_DBF_MSG, maxcol()+1 );
                    );
                };
            )

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_FILE_NTX_CHOICE,;
                {||;
                    say(;
                        maxrow()-1, 0,;
                        padc( MNU_MENU_FILE_NTX_MSG, maxcol()+1 );
                    );
                };
            )

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_FILE_ALIAS_CHOICE,;
                {||;
                    say(;
                        maxrow()-1, 0,;
                        padc( MNU_MENU_FILE_ALIAS_MSG, maxcol()+1 );
                    );
                };
            )

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_FILE_INDEX_CHOICE,;
                {||;
                    say(;
                        maxrow()-1, 0,;
                        padc( MNU_MENU_FILE_INDEX_MSG, maxcol()+1 );
                    );
                };
            )

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_FILE_RELATION_CHOICE,;
                {||;
                    say(;
                        maxrow()-1, 0,;
                        padc( MNU_MENU_FILE_RELATION_MSG, maxcol()+1 );
                    );
                };
            )

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_FILE_RDD_DEFAULT_CHOICE,;
                {||;
                    say(;
                        maxrow()-1, 0,;
                        padc( MNU_MENU_FILE_RDD_DEFAULT_MSG, maxcol()+1 );
                    );
                };
            )

            *-----------------------------------------------------
            * Read the menu choice.
            *-----------------------------------------------------

            nMnuChoice := menuTo( @aMenu, nMnuChoice )

            *-----------------------------------------------------
            * Save the screen before any action is taken.
            *-----------------------------------------------------

            cSaveScreen := mouseScrSave()

            *-----------------------------------------------------
            * Arrow key redirections are released.
            *-----------------------------------------------------

            setkey( K_LEFT, NIL )
            setkey( K_RIGHT, NIL )

            *-----------------------------------------------------
            * The basic color is prepared.
            *-----------------------------------------------------

            setcolor( COLOR_BASE )

            *-----------------------------------------------------
            * The help is not mainteined.
            *-----------------------------------------------------

            bSaveF1 := setkey( K_F1, NIL )

            *-----------------------------------------------------
            * The choice is analised.
            *-----------------------------------------------------

            do case
            case nMnuChoice == 0

                *-------------------------------------------------
                * [Esc] means leave.
                *-------------------------------------------------

                exit                                    // EXIT

            case nMnuChoice == 1

                dummy := Dir( "", .T., .T., .F. )

                if right( dummy, 1 ) == "\"

                    *---------------------------------------------
                    * The last backslash must be removed
                    * or it will not work on most MS-DOS
                    * version.
                    *---------------------------------------------

                    dummy := left( dummy, len(dummy) -1 )

                end

                run( "cd " + dummy )

                dummy := strDrive( dummy )

                *-------------------------------------------------
                * If the string contains a Drive information.
                *-------------------------------------------------

                if !(dummy == "")

                    *---------------------------------------------
                    * Change drive.
                    *---------------------------------------------

                    run( dummy )

                end

            case nMnuChoice == 2

                mnupDbf(5,nRB)

            case nMnuChoice == 3

                mnupNtx(6,nRB)

            case nMnuChoice == 4

                mnupAlias(7,nRB)

            case nMnuChoice == 5

                mnupIndex(8,nRB)

            case nMnuChoice == 6

                mnupRelation(9,nRB)

            case nMnuChoice == 7

                mnupRDD(10,nRB)

            end

            *-----------------------------------------------------
            * Restore arrow redirection.
            *-----------------------------------------------------

            setkey( K_LEFT, { || mnuLeft() } )
            setkey( K_RIGHT, { || mnuRight() } )

            *-----------------------------------------------------
            * Restore the screen containing the menu.
            *-----------------------------------------------------

            mouseScrRestore(,,,,cSaveScreen)

            *-----------------------------------------------------
            * Restore the help.
            *-----------------------------------------------------

            setkey( K_F1, bSaveF1 )

        end

    end //sequence

    *-------------------------------------------------------------
    * Restore old data and screen before this menu.
    *-------------------------------------------------------------

    mouseScrRestore( ,,,, cOldScreen )
    setkey( K_F1, bOld_F1 )

    *-------------------------------------------------------------
    * <lBreak> is used to know if after the selection of a
    * particular item, the menu should remain or disappear.
    *-------------------------------------------------------------

    return lRetBreak

*-----------------------------------------------------------------
static function mnupDbf(nTB,nLB)

    local aMenu             := {}
    local cOldScreen        := mouseScrSave()
    local bOld_F1           :=;
        setkey( K_F1, {|| nbHelp( "<MENU FILE>", , chr(K_ENTER) )} )
    local nMenuHeight       := 5    // menu lines +2
    local nBB                       // bottom border
    local nRB                       // right border
    local n

    local nMnuChoice := 0

    local bSaveF1

    local lRetBreak := .F.

    if (nTB + nMenuHeight -1) > maxrow()
        nBB := maxrow()
        nTB := maxrow() - nMenuHeight +1
    else
        nBB := nTB + nMenuHeight -1
    end

    if (nLB + _MENU_PULLDOWN_WIDTH -1) > maxcol()
        nLB := nLB - 2*_MENU_PULLDOWN_WIDTH +1
    end

    nRB := nLB + _MENU_PULLDOWN_WIDTH -1

    begin sequence

        setcolor( COLOR_MENU )
        dispBoxShadow(;
            nTB, nLB, nBB, nRB,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
            )
        say( maxrow(), 00,;
            padc( _KEY_MENU_P,;
                maxcol()+1 );
            )
        n := 0
        menuPrompt( @aMenu, nTB+(++n), nLB+1,;
            MNU_MENU_FILE_DBF_NEW_CHOICE,;
            {|| say( maxrow()-1, 0,;
                padc( MNU_MENU_FILE_DBF_NEW_MSG, maxcol()+1 ) );
                };
            )
        menuPrompt( @aMenu, nTB+(++n), nLB+1,;
            MNU_MENU_FILE_DBF_MODSTRU_CHOICE,;
            {|| say( maxrow()-1, 0,;
                padc( MNU_MENU_FILE_DBF_MODSTRU_MSG, maxcol()+1 ) );
                };
            )
        menuPrompt( @aMenu, nTB+(++n), nLB+1,;
            MNU_MENU_FILE_DBF_OPEN_CHOICE,;
            {|| say( maxrow()-1, 0,;
                padc( MNU_MENU_FILE_DBF_OPEN_MSG, maxcol()+1 ) );
                };
            )

        nMnuChoice := menuTo( @aMenu, nMnuChoice )

        bSaveF1 := setkey( K_F1, NIL )

        do case
        case nMnuChoice == 1
            idbStructure()
        case nMnuChoice == 2
            idbModStructure()
        case nMnuChoice == 3
            idbUseArea()
        end

        setkey( K_F1, bSaveF1 )

    end //sequence

    // restore
    mouseScrRestore( ,,,, cOldScreen )
    setkey( K_F1, bOld_F1 )

    return lRetBreak

*-----------------------------------------------------------------
static function mnupNtx(nTB,nLB)

    local aMenu            := {}
    local cOldScreen        := mouseScrSave()
    local bOld_F1           :=;
        setkey( K_F1, {|| nbHelp( "<MENU FILE>", , chr(K_ENTER) )} )

    local nMenuHeight       := 4    // menu lines +2
    local nBB                       // bottom border
    local nRB                       // right border
    local n

    local nMnuChoice := 0

    local bSaveF1

    local lRetBreak := .F.

    if (nTB + nMenuHeight -1) > maxrow()
        nBB := maxrow()
        nTB := maxrow() - nMenuHeight +1
    else
        nBB := nTB + nMenuHeight -1
    end
    if (nLB + _MENU_PULLDOWN_WIDTH -1) > maxcol()
        nLB := nLB - 2*_MENU_PULLDOWN_WIDTH +1
    end
    nRB := nLB + _MENU_PULLDOWN_WIDTH -1

    begin sequence

        setcolor( COLOR_MENU )
        dispBoxShadow(;
            nTB, nLB, nBB, nRB,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
            )
        say( maxrow(), 00, padc( _KEY_MENU_P, maxcol()+1 ) )
        n := 0

        menuPrompt( @aMenu, nTB+(++n), nLB+1,;
            MNU_MENU_FILE_NTX_NEW_CHOICE,;
            {||;
                say(;
                    maxrow()-1, 0,;
                    padc(;
                        MNU_MENU_FILE_NTX_NEW_MSG,;
                        maxcol()+1;
                        );
                    );
                };
            )
        menuPrompt( @aMenu, nTB+(++n), nLB+1,;
            MNU_MENU_FILE_NTX_OPEN_CHOICE,;
            {||;
                say(;
                    maxrow()-1, 0,;
                    padc(;
                        MNU_MENU_FILE_NTX_OPEN_MSG,;
                        maxcol()+1;
                        );
                    );
                };
            )

        nMnuChoice := menuTo( @aMenu, nMnuChoice )

        bSaveF1 := setkey( K_F1, NIL )

        do case
        case (nMnuChoice == 1)
            idbOrdCreate()
        case (nMnuChoice == 2)
            idbSetIndex()
        end

        setkey( K_F1, bSaveF1 )

    end //sequence

    // restore
    mouseScrRestore( ,,,, cOldScreen )
    setkey( K_F1, bOld_F1 )

    return lRetBreak

*------------------------------------------------------------------
static function mnupAlias(nTB,nLB)

    local aMenu            := {}
    local cOldScreen        := mouseScrSave()
    local bOld_F1           :=;
        setkey( K_F1, {|| nbHelp( "<MENU FILE>", , chr(K_ENTER) )} )

    local nMenuHeight       := 6    // menu lines +2
    local nBB                       // bottom border
    local nRB                       // right border
    local n

    local dummy

    local nMnuChoice := 0

    local bSaveF1

    local lRetBreak := .F.

    if (nTB + nMenuHeight -1) > maxrow()
        nBB := maxrow()
        nTB := maxrow() - nMenuHeight +1
    else
        nBB := nTB + nMenuHeight -1
    end
    if (nLB + _MENU_PULLDOWN_WIDTH -1) > maxcol()
        nLB := nLB - 2*_MENU_PULLDOWN_WIDTH +1
    end
    nRB := nLB + _MENU_PULLDOWN_WIDTH -1

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Prepare colors for menu.
        *---------------------------------------------------------

        setcolor( COLOR_MENU )

        *---------------------------------------------------------
        * Show the box around the menu.
        *---------------------------------------------------------

        dispBoxShadow(;
            nTB, nLB, nBB, nRB,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )

        *---------------------------------------------------------
        * Display key reminder.
        *---------------------------------------------------------

        say( maxrow(), 00, padc( _KEY_MENU_P, maxcol()+1 ) )

        *---------------------------------------------------------
        * <n> contains the menu choice. At the beguinning it
        * starts from zero.
        *---------------------------------------------------------

        n := 0

        *---------------------------------------------------------
        * Start the sequence of menu items and separation
        * lines.
        *---------------------------------------------------------

        menuPrompt(;
            @aMenu, nTB+(++n), nLB+1,;
            MNU_MENU_FILE_ALIAS_SELECT_CHOICE,;
            {||;
                say(;
                    maxrow()-1, 0,;
                    padc(;
                        MNU_MENU_FILE_ALIAS_SELECT_MSG,;
                        maxcol()+1;
                    );
                );
            };
        )

        menuPrompt(;
            @aMenu, nTB+(++n), nLB+1,;
            MNU_MENU_FILE_ALIAS_DISPSTRU_CHOICE,;
            {||;
                say(;
                    maxrow()-1, 0,;
                    padc(;
                        MNU_MENU_FILE_ALIAS_DISPSTRU_MSG,;
                        maxcol()+1;
                    );
                );
            };
        )

        menuPrompt(;
            @aMenu, nTB+(++n), nLB+1,;
            MNU_MENU_FILE_ALIAS_CLOSE_CHOICE,;
            {||;
                say(;
                    maxrow()-1, 0,;
                    padc(;
                        MNU_MENU_FILE_ALIAS_CLOSE_MSG,;
                        maxcol()+1;
                    );
                );
            };
        )

        menuPrompt(;
            @aMenu, nTB+(++n), nLB+1,;
            MNU_MENU_FILE_ALIAS_CLOSEALL_CHOICE,;
            {||;
                say(;
                    maxrow()-1, 0,;
                    padc(;
                        MNU_MENU_FILE_ALIAS_CLOSEALL_MSG,;
                        maxcol()+1;
                    );
                );
            };
        )

        *---------------------------------------------------------
        * Read the menu choice.
        *---------------------------------------------------------

        nMnuChoice := menuTo( @aMenu, nMnuChoice )

        *---------------------------------------------------------
        * The help is not mainteined.
        *---------------------------------------------------------

        bSaveF1 := setkey( K_F1, NIL )

        *---------------------------------------------------------
        * The choice is analised.
        *---------------------------------------------------------

        do case
        case nMnuChoice == 1

            setkey( K_F1, {|| nbHelp( "<FILE - SELECT>", , chr(K_ENTER) ) } )

            dummy := selectArray()[1]

            if valtype(dummy) == "N"

                dbselectarea(dummy)

                *-------------------------------------------------
                * Macro Recorder.
                *-------------------------------------------------

                if  isMemvar( _MEMVAR_MACRO_RECORDING_NAME )

                    Memvar->_MEMVAR_MACRO_RECORDER +=;
                        NL(1)                               +;
                        '// Select Alias.'                  +;
                        NL(1)                               +;
                        'dbselectarea( "' + alias() + '" )' +;
                        NL(1)

                end

            end

            setkey( K_F1, bSaveF1 )

        case nMnuChoice == 2

            setkey( K_F1, {|| nbHelp( "<FILE - DISPLAY STRUCTURE>", , chr(K_ENTER) ) } )

            dummy := dbiStructure()

            if valtype(dummy) == "C"

                text( dummy )

            end

            setkey( K_F1, bSaveF1 )

        case nMnuChoice == 3

            dbclosearea()

            *-----------------------------------------------------
            * Macro Recorder.
            *-----------------------------------------------------

            if  isMemvar( _MEMVAR_MACRO_RECORDING_NAME )

                Memvar->_MEMVAR_MACRO_RECORDER +=;
                    NL(1)                                   +;
                    '// Close the active work area.'        +;
                    NL(1)                                   +;
                    'dbclosearea()'                         +;
                    NL(1)

            end

        case nMnuChoice == 4

            dbcloseall()

            *-----------------------------------------------------
            * Macro Recorder.
            *-----------------------------------------------------

            if  isMemvar( _MEMVAR_MACRO_RECORDING_NAME )

                Memvar->_MEMVAR_MACRO_RECORDER +=;
                    NL(1)                                   +;
                    '// Close all work areas.'              +;
                    NL(1)                                   +;
                    '// dbcloseall()'                       +;
                    NL(1)                                   +;
                    'dbclose()'                             +;
                    NL(1)

            end

        end

        *---------------------------------------------------------
        * Restore the help.
        *---------------------------------------------------------

        setkey( K_F1, bSaveF1 )

    end //sequence

    *-------------------------------------------------------------
    * Restore previous data.
    *-------------------------------------------------------------

    mouseScrRestore( ,,,, cOldScreen )
    setkey( K_F1, bOld_F1 )

    return lRetBreak

*-----------------------------------------------------------------
static function mnupIndex(nTB,nLB)

    local aMenu            := {}
    local cOldScreen        := mouseScrSave()
    local bOld_F1           :=;
        setkey( K_F1, {|| nbHelp( "<MENU FILE>", , chr(K_ENTER) )} )

    local nMenuHeight       := 5    // menu lines +2
    local nBB                       // bottom border
    local nRB                       // right border
    local n

    local dummy

    local nMnuChoice := 0

    local bSaveF1

    local lRetBreak := .F.

    if (nTB + nMenuHeight -1) > maxrow()
        nBB := maxrow()
        nTB := maxrow() - nMenuHeight +1
    else
        nBB := nTB + nMenuHeight -1
    end

    if (nLB + _MENU_PULLDOWN_WIDTH -1) > maxcol()
        nLB := nLB - 2*_MENU_PULLDOWN_WIDTH +1
    end

    nRB := nLB + _MENU_PULLDOWN_WIDTH -1

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Prepare colors for menu.
        *---------------------------------------------------------

        setcolor( COLOR_MENU )

        *---------------------------------------------------------
        * Show the box around the menu.
        *---------------------------------------------------------

        dispBoxShadow(;
            nTB, nLB, nBB, nRB,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )

        *---------------------------------------------------------
        * Display key reminder.
        *---------------------------------------------------------

        say( maxrow(), 00, padc( _KEY_MENU_P, maxcol()+1 ) )

        *---------------------------------------------------------
        * <n> contains the menu choice. At the beguinning it
        * starts from zero.
        *---------------------------------------------------------

        n := 0

        *---------------------------------------------------------
        * Start the sequence of menu items and separation
        * lines.
        *---------------------------------------------------------

        menuPrompt(;
            @aMenu, nTB+(++n), nLB+1,;
            MNU_MENU_FILE_INDEX_REINDEX_CHOICE,;
            {||;
                say(;
                    maxrow()-1, 0,;
                    padc(;
                        MNU_MENU_FILE_INDEX_REINDEX_MSG,;
                        maxcol()+1;
                    );
                );
            };
        )

        menuPrompt(;
            @aMenu, nTB+(++n), nLB+1,;
            MNU_MENU_FILE_INDEX_SET_ORDER_CHOICE,;
            {||;
                say(;
                    maxrow()-1, 0,;
                    padc(;
                        MNU_MENU_FILE_INDEX_SET_ORDER_MSG,;
                        maxcol()+1;
                    );
                );
            };
        )

        menuPrompt(;
            @aMenu, nTB+(++n), nLB+1,;
            MNU_MENU_FILE_INDEX_CLOSENTX_CHOICE,;
            {||;
                say(;
                    maxrow()-1, 0,;
                    padc(;
                        MNU_MENU_FILE_INDEX_CLOSENTX_MSG,;
                        maxcol()+1;
                    );
                );
            };
        )

        *---------------------------------------------------------
        * Read the menu choice.
        *---------------------------------------------------------

        nMnuChoice := menuTo( @aMenu, nMnuChoice )

        *---------------------------------------------------------
        * The help is not mainteined.
        *---------------------------------------------------------

        bSaveF1 := setkey( K_F1, NIL )

        *---------------------------------------------------------
        * The choice is analised.
        *---------------------------------------------------------

        do case
        case nMnuChoice == 1

            idbReindex()

        case nMnuChoice == 2

            setkey( K_F1, {|| nbHelp( "<FILE - ORDER SET FOCUS>", , chr(K_ENTER) ) } )

            dummy := selectOrder()

            if valtype( dummy ) == "N"

                ordsetfocus( dummy )

                *-------------------------------------------------
                * Macro Recorder.
                *-------------------------------------------------

                if  isMemvar( _MEMVAR_MACRO_RECORDING_NAME )

                    Memvar->_MEMVAR_MACRO_RECORDER +=;
                        NL(1)                               +;
                        '// ordSetFocus.'                   +;
                        NL(1)

                    if dummy == 0
                        Memvar->_MEMVAR_MACRO_RECORDER +=;
                            'ordsetfocus( 0 )'
                    else
                        Memvar->_MEMVAR_MACRO_RECORDER +=;
                            'ordsetfocus( "' + ordname( dummy ) + '" )'
                    end

                    Memvar->_MEMVAR_MACRO_RECORDER +=;
                        NL(1)

                end

            end

            setkey( K_F1, bOld_F1 )

        case nMnuChoice == 3

            idbClearIndex()

        end

        *---------------------------------------------------------
        * Restore the help.
        *---------------------------------------------------------

        setkey( K_F1, bSaveF1 )

    end //sequence

    *-------------------------------------------------------------
    * Restore previous data.
    *-------------------------------------------------------------

    mouseScrRestore( ,,,, cOldScreen )
    setkey( K_F1, bOld_F1 )

    return lRetBreak

*-----------------------------------------------------------------
static function mnupRelation(nTB,nLB)

    local aMenu             := {}
    local cOldScreen        := mouseScrSave()
    local bOld_F1           :=;
        setkey( K_F1, {|| nbHelp( "<MENU FILE>", , chr(K_ENTER) )} )

    local nMenuHeight       := 4    // menu lines +2
    local nBB                       // bottom border
    local nRB                       // right border
    local n

    local dummy

    local nMnuChoice := 0

    local bSaveF1

    local lRetBreak := .F.

    if (nTB + nMenuHeight -1) > maxrow()
        nBB := maxrow()
        nTB := maxrow() - nMenuHeight +1
    else
        nBB := nTB + nMenuHeight -1
    end

    if (nLB + _MENU_PULLDOWN_WIDTH -1) > maxcol()
        nLB := nLB - 2*_MENU_PULLDOWN_WIDTH +1
    end

    nRB := nLB + _MENU_PULLDOWN_WIDTH -1

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Prepare colors for menu.
        *---------------------------------------------------------

        setcolor( COLOR_MENU )

        *---------------------------------------------------------
        * Show the box around the menu.
        *---------------------------------------------------------

        dispBoxShadow(;
            nTB, nLB, nBB, nRB,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )

        *---------------------------------------------------------
        * Display key reminder.
        *---------------------------------------------------------

        say( maxrow(), 00, padc( _KEY_MENU_P, maxcol()+1 ) )

        *---------------------------------------------------------
        * <n> contains the menu choice. At the beguinning it
        * starts from zero.
        *---------------------------------------------------------

        n := 0

        *---------------------------------------------------------
        * Start the sequence of menu items and separation
        * lines.
        *---------------------------------------------------------

        menuPrompt(;
            @aMenu, nTB+(++n), nLB+1,;
            MNU_MENU_FILE_RELATION_SET_CHOICE,;
            {||;
                say(;
                    maxrow()-1, 0,;
                    padc(;
                        MNU_MENU_FILE_RELATION_SET_MSG,;
                        maxcol()+1;
                    );
                );
            };
        )

        menuPrompt(;
         @aMenu, nTB+(++n), nLB+1,;
            MNU_MENU_FILE_RELATION_CLOSE_CHOICE,;
            {||;
                say(;
                    maxrow()-1, 0,;
                    padc(;
                        MNU_MENU_FILE_RELATION_CLOSE_MSG,;
                        maxcol()+1;
                    );
                );
            };
        )

        *---------------------------------------------------------
        * Read the menu choice.
        *---------------------------------------------------------

        nMnuChoice := menuTo( @aMenu, nMnuChoice )

        *---------------------------------------------------------
        * The help is not mainteined.
        *---------------------------------------------------------

        bSaveF1 := setkey( K_F1, NIL )

        *---------------------------------------------------------
        * The choice is analised.
        *---------------------------------------------------------

        do case
        case nMnuChoice == 1

            idbRelation()

        case nMnuChoice == 2

            dbclearrelation()

            *-----------------------------------------------------
            * Macro Recorder.
            *-----------------------------------------------------

            if  isMemvar( _MEMVAR_MACRO_RECORDING_NAME )

                Memvar->_MEMVAR_MACRO_RECORDER +=;
                    NL(1)                                   +;
                    '// Clear active relation.'             +;
                    NL(1)                                   +;
                    'dbclearrelation()'                     +;
                    NL(1)

            end

        end

        *---------------------------------------------------------
        * Restore the help.
        *---------------------------------------------------------

        setkey( K_F1, bSaveF1 )

    end //sequence

    *-------------------------------------------------------------
    * Restore previous data.
    *-------------------------------------------------------------

    mouseScrRestore( ,,,, cOldScreen )
    setkey( K_F1, bOld_F1 )

    return lRetBreak

*-----------------------------------------------------------------
static function mnupRDD(nTB,nLB)

    local aMenu             := {}
    local cOldScreen        := mouseScrSave()
    local lOldSetWrap       := set(_SET_WRAP, .T.)
    local nOldSetMessage    := set(_SET_MESSAGE, maxrow()-1)
    local lOldSetCenter     := set(_SET_MCENTER, .T.)
    local bOld_F1           :=;
        setkey( K_F1, {|| nbHelp( "<MENU FILE>", , chr(K_ENTER) )} )

    local nMenuHeight       := 4    // menu lines +2
    local nBB                       // bottom border
    local nRB                       // right border
    local n

    local dummy

    local nMnuChoice := 0

    local bSaveF1

    local lRetBreak := .F.

    if (nTB + nMenuHeight -1) > maxrow()
        nBB := maxrow()
        nTB := maxrow() - nMenuHeight +1
    else
        nBB := nTB + nMenuHeight -1
    end

    if (nLB + _MENU_PULLDOWN_WIDTH -1) > maxcol()
        nLB := nLB - 2*_MENU_PULLDOWN_WIDTH +1
    end

    nRB := nLB + _MENU_PULLDOWN_WIDTH -1

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Prepare colors for menu.
        *---------------------------------------------------------

        setcolor( COLOR_MENU )

        *---------------------------------------------------------
        * Show the box around the menu.
        *---------------------------------------------------------

        dispBoxShadow(;
            nTB, nLB, nBB, nRB,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )

        *---------------------------------------------------------
        * Display key reminder.
        *---------------------------------------------------------

        say( maxrow(), 00, padc( _KEY_MENU_P, maxcol()+1 ) )

        *---------------------------------------------------------
        * <n> contains the menu choice. At the beguinning it
        * starts from zero.
        *---------------------------------------------------------

        n := 0

        *---------------------------------------------------------
        * Start the sequence of menu items and separation
        * lines.
        *---------------------------------------------------------

        menuPrompt(;
            @aMenu, nTB+(++n), nLB+1,;
            MNU_MENU_FILE_RDD_SHOW_CHOICE,;
            {||;
                say(;
                    maxrow()-1, 0,;
                    padc(;
                        MNU_MENU_FILE_RDD_SHOW_MSG,;
                        maxcol()+1;
                    );
                );
            };
        )

        menuPrompt(;
            @aMenu, nTB+(++n), nLB+1,;
            MNU_MENU_FILE_RDD_SET_DEFAULT_CHOICE,;
            {||;
                say(;
                    maxrow()-1, 0,;
                    padc(;
                        MNU_MENU_FILE_RDD_SET_DEFAULT_MSG,;
                        maxcol()+1;
                    );
                );
            };
        )

        *---------------------------------------------------------
        * Read the menu choice.
        *---------------------------------------------------------

        nMnuChoice := menuTo( @aMenu, nMnuChoice )

        *---------------------------------------------------------
        * The help is not mainteined.
        *---------------------------------------------------------

        bSaveF1 := setkey( K_F1, NIL )

        *---------------------------------------------------------
        * The choice is analised.
        *---------------------------------------------------------

        do case
        case nMnuChoice == 1

            alertBox( rddsetdefault() )

        case nMnuChoice == 2

            *-----------------------------------------------------
            * Prepare the right help.
            *-----------------------------------------------------

            setkey( K_F1, {|| nbHelp( "<FILE - SET DEFAULT RDD>", , chr(K_ENTER) ) } )

            dummy := dbRddSelect()

            if valtype(dummy) == "C"

                rddsetdefault( dummy )

                *-----------------------------------------------------
                * Macro recorder.
                *-----------------------------------------------------

                if isMemvar( _MEMVAR_MACRO_RECORDING_NAME )

                    Memvar->_MEMVAR_MACRO_RECORDER +=;
                        NL(1)                                   +;
                        '// Change the current default RDD.'    +;
                        NL(1)                                   +;
                        'rddsetdefault( "' + dummy + '" )'      +;
                        NL(1)

                end

            end

        end

        *---------------------------------------------------------
        * Restore the help.
        *---------------------------------------------------------

        setkey( K_F1, bSaveF1 )

    end //sequence

    *-------------------------------------------------------------
    * Restore previous data.
    *-------------------------------------------------------------

    mouseScrRestore( ,,,, cOldScreen )
    setkey( K_F1, bOld_F1 )

    return lRetBreak

*-----------------------------------------------------------------
static function mnuvEdit( nHLeft, nHRight )

    local aMenu             := {}
    local cOldScreen        := mouseScrSave()
    local lOldSetWrap       := set(_SET_WRAP, .T.)
    local nOldSetMessage    := set(_SET_MESSAGE, maxrow()-1)
    local lOldSetCenter     := set(_SET_MCENTER, .T.)
    local bOld_F1           :=;
        setkey( K_F1, {|| nbHelp( "<MENU EDIT>", , chr(K_ENTER) )} )

    local nTB               := 2            // top border
    local nLB               :=;
        iif( nHLeft+_MENU_PULLDOWN_WIDTH > maxcol(),;
        nHRight - _MENU_PULLDOWN_WIDTH,;
        nHLeft )                            // left border
    local nBB               := nTB +8       // bottom border
    local nRB               :=;
        nLB +_MENU_PULLDOWN_WIDTH -1         // right border
    local n

    local cSaveScreen
    local bSaveF1

    local nMnuChoice := 0

    local lRetBreak := .F.

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        while .T.                                       // FOREVER

            *-----------------------------------------------------
            * Show the updated status line.
            *-----------------------------------------------------

            statusLine()

            *-----------------------------------------------------
            * Prepare colors for menu.
            *-----------------------------------------------------

            setcolor( COLOR_MENU )

            *-----------------------------------------------------
            * Show the box around the menu.
            *-----------------------------------------------------

            dispBoxShadow(;
                nTB, nLB, nBB, nRB,;
                1,;
                dispBoxColor( 1 ),;
                dispBoxColor( 2 );
            )

            *-----------------------------------------------------
            * Display key reminder.
            *-----------------------------------------------------

            say( maxrow(), 00, padc( _KEY_MENU_V, maxcol()+1 ) )

            *-----------------------------------------------------
            * <n> contains the menu choice. At the beguinning it
            * starts from zero.
            *-----------------------------------------------------

            n := 0

            *-----------------------------------------------------
            * Start the sequence of menu items and separation
            * lines.
            *-----------------------------------------------------

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_EDIT_VIEW_CHOICE,;
                {||;
                    say(;
                        maxrow()-1, 0,;
                        padc( MNU_MENU_EDIT_VIEW_MSG, maxcol()+1 );
                    );
                };
            )

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_EDIT_BROWSE_CHOICE,;
                {||;
                    say(;
                        maxrow()-1, 0,;
                        padc( MNU_MENU_EDIT_BROWSE_MSG, maxcol()+1 );
                    );
                };
            )

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_EDIT_REPLACE_CHOICE,;
                {||;
                    say(;
                        maxrow()-1, 0,;
                        padc( MNU_MENU_EDIT_REPLACE_MSG, maxcol()+1 );
                    );
                };
            )

            say(;
                nTB+(++n), nLB+1,;
                replicate("", _MENU_PULLDOWN_WIDTH-2);
            )

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_EDIT_RECALL_CHOICE,;
                {||;
                    say(;
                        maxrow()-1, 0,;
                        padc( MNU_MENU_EDIT_RECALL_MSG, maxcol()+1 );
                    );
                };
            )

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_EDIT_DELETE_CHOICE,;
                {||;
                    say(;
                        maxrow()-1, 0,;
                        padc( MNU_MENU_EDIT_DELETE_MSG, maxcol()+1 );
                    );
                };
            )

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_EDIT_PACK_CHOICE,;
                {||;
                    say(;
                        maxrow()-1, 0,;
                        padc( MNU_MENU_EDIT_PACK_MSG, maxcol()+1 );
                    );
                };
            )

            *-----------------------------------------------------
            * Read the menu choice.
            *-----------------------------------------------------

            nMnuChoice := menuTo( @aMenu, nMnuChoice )

            *-----------------------------------------------------
            * Save the screen before any action is taken.
            *-----------------------------------------------------

            cSaveScreen := mouseScrSave()

            *-----------------------------------------------------
            * Arrow key redirections are released.
            *-----------------------------------------------------

            setkey( K_LEFT, NIL )
            setkey( K_RIGHT, NIL )

            *-----------------------------------------------------
            * The basic color is prepared.
            *-----------------------------------------------------

            setcolor( COLOR_BASE )

            *-----------------------------------------------------
            * The help is not mainteined.
            *-----------------------------------------------------

            bSaveF1 := setkey( K_F1, NIL )

            *-----------------------------------------------------
            * The choice is analised.
            *-----------------------------------------------------

            do case
            case nMnuChoice == 0

                *-------------------------------------------------
                * [Esc] means leave.
                *-------------------------------------------------

                exit                                    // EXIT

            case nMnuChoice == 1

                setkey( K_F1, {|| nbHelp( "<EDIT - VIEW>", , chr(K_ENTER) ) } )

                * TB( <nTop>, <nLeft>, <nBottom>, <nRight>,
                *      <acCol>, [<acColSayPic>],
                *      [<acColTopSep>], [<acColBodySep>], [<acColBotSep>],;
                *      [<acColHead>], [<acColFoot>],
                *      [<alColCalc>],
                *      [<abColValid>],
                *      [<abColMsg>],
                *      [<cColor>], [<abColColors>],;
                *      [<nFreeze>],
                *      [<lModify>], [<lAppend>], [<lDelete>],
                *      [lButtons|aButtons] )  --> NIL
                
                scroll()

                tb(;
                    ,,,,;
                    ,,;
                    ,,,;
                    ,,;
                    ,;
                    ,;
                    ,;
                    ,,;
                    ,;
                    .F.,,,;
                    .T.;
                )

                *-------------------------------------------------
                * Macro recorder.
                *-------------------------------------------------

                if  isMemvar( _MEMVAR_MACRO_RECORDING_NAME )

                    Memvar->_MEMVAR_MACRO_RECORDER +=;
                        NL(1)                               +;
                        '// View the active Alias.'         +;
                        NL(1)                               +;
                        '// This is not a CA-Clipper '      +;
                            'standard function.'            +;
                        NL(1)                               +;
                        'scroll()'                          +;
                        NL(1)                               +;
                        'tb(,,,,,,,,,,,,,,,,, .F.,,, .T.)'  +;
                        NL(1)

                end

            case nMnuChoice == 2

                setkey( K_F1, {|| nbHelp( "<EDIT - EDIT/BROWSE>", , chr(K_ENTER) ) } )

                * TB( <nTop>, <nLeft>, <nBottom>, <nRight>,
                *      <acCol>, [<acColSayPic>],
                *      [<acColTopSep>], [<acColBodySep>], [<acColBotSep>],;
                *      [<acColHead>], [<acColFoot>],
                *      [<alColCalc>],
                *      [<abColValid>],
                *      [<abColMsg>],
                *      [<cColor>], [<abColColors>],;
                *      [<nFreeze>],
                *      [<lModify>], [<lAppend>], [<lDelete>],
                *      [lButtons|aButtons] )  --> NIL

                scroll()

                tb(;
                    ,,,,;
                    ,,;
                    ,,,;
                    ,,;
                    ,;
                    ,;
                    ,;
                    ,,;
                    ,;
                    ,,,;
                    .T.;
                )

                *-------------------------------------------------
                * Macro recorder.
                *-------------------------------------------------

                if  isMemvar( _MEMVAR_MACRO_RECORDING_NAME )

                    Memvar->_MEMVAR_MACRO_RECORDER +=;
                        NL(1)                               +;
                        '// Edit the active Alias.'         +;
                        NL(1)                               +;
                        '// This is not a CA-Clipper '      +;
                        'standard function.'                +;
                        NL(1)                               +;
                        'scroll()'                          +;
                        NL(1)                               +;
                        'tb(,,,,,,,,,,,,,,,,,,,, .T.)'      +;
                        NL(1)
                end

            case nMnuChoice == 3

                idbReplace()

            case nMnuChoice == 4

                idbRecall()

            case nMnuChoice == 5

                idbDelete()

            case nMnuChoice == 6

                idbPack()

            end

            *-----------------------------------------------------
            * Restore arrow redirection.
            *-----------------------------------------------------

            setkey( K_LEFT, { || mnuLeft() } )
            setkey( K_RIGHT, { || mnuRight() } )

            *-----------------------------------------------------
            * Restore the screen containing the menu.
            *-----------------------------------------------------

            mouseScrRestore(,,,,cSaveScreen)

            *-----------------------------------------------------
            * Restore the help.
            *-----------------------------------------------------

            setkey( K_F1, bSaveF1 )

        end

    end //sequence

    *-------------------------------------------------------------
    * Restore old data and screen before this menu.
    *-------------------------------------------------------------

    mouseScrRestore( ,,,, cOldScreen )
    set(_SET_WRAP, lOldSetWrap )
    set(_SET_MESSAGE, nOldSetMessage )
    set(_SET_MCENTER, lOldSetCenter )
    setkey( K_F1, bOld_F1 )

    return lRetBreak

*-----------------------------------------------------------------
static function mnuvReport( nHLeft, nHRight )

    local aMenu             := {}
    local cOldScreen        := mouseScrSave()
    local lOldSetWrap       := set(_SET_WRAP, .T.)
    local nOldSetMessage    := set(_SET_MESSAGE, maxrow()-1)
    local lOldSetCenter     := set(_SET_MCENTER, .T.)
    local bOld_F1           :=;
        setkey( K_F1, {|| nbHelp( "<MENU REPORT>", , chr(K_ENTER) )} )

    local nTB               := 2            // top border
    local nLB               :=;
        iif( nHLeft+_MENU_PULLDOWN_WIDTH > maxcol(),;
        nHRight - _MENU_PULLDOWN_WIDTH,;
        nHLeft )                            // left border
    local nBB               := nTB +12      // bottom border
    local nRB               :=;
        nLB +_MENU_PULLDOWN_WIDTH -1         // right border
    local n

    local cSaveScreen
    local bSaveF1

    local nMnuChoice := 0

    local lRetBreak := .F.

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        while .T.                                       // FOREVER

            *-----------------------------------------------------
            * Show the updated status line.
            *-----------------------------------------------------

            statusLine()

            *-----------------------------------------------------
            * Prepare colors for menu.
            *-----------------------------------------------------

            setcolor( COLOR_MENU )

            *-----------------------------------------------------
            * Show the box around the menu.
            *-----------------------------------------------------

            dispBoxShadow(;
                nTB, nLB, nBB, nRB,;
                1,;
                dispBoxColor( 1 ),;
                dispBoxColor( 2 );
            )

            *-----------------------------------------------------
            * Display key reminder.
            *-----------------------------------------------------

            say( maxrow(), 00, padc( _KEY_MENU_V, maxcol()+1 ) )

            *-----------------------------------------------------
            * <n> contains the menu choice. At the beguinning it
            * starts from zero.
            *-----------------------------------------------------

            n := 0

            *-----------------------------------------------------
            * Start the sequence of menu items and separation
            * lines.
            *-----------------------------------------------------

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_REPORT_DBGOTOP_CHOICE,;
                {||;
                    say(;
                        maxrow()-1, 0,;
                        padc( MNU_MENU_REPORT_DBGOTOP_MSG, maxcol()+1 );
                    );
                };
            )

            say(;
                nTB+(++n), nLB+1,;
                replicate("", _MENU_PULLDOWN_WIDTH-2);
            )

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_REPORT_LABEL_NEW_CHOICE,;
                {||;
                    say(;
                        maxrow()-1, 0,;
                        padc( MNU_MENU_REPORT_LABEL_NEW_MSG, maxcol()+1 );
                    );
                };
            )

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_REPORT_LABEL_MODIFY_CHOICE,;
                {||;
                    say(;
                        maxrow()-1, 0,;
                        padc( MNU_MENU_REPORT_LABEL_MODIFY_MSG, maxcol()+1 );
                    );
                };
            )

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_REPORT_LABEL_FORM_CHOICE,;
                {||;
                    say(;
                        maxrow()-1, 0,;
                        padc( MNU_MENU_REPORT_LABEL_FORM_MSG, maxcol()+1 );
                    );
                };
            )

            say(;
                nTB+(++n), nLB+1,;
                replicate("", _MENU_PULLDOWN_WIDTH-2);
            )

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_REPORT_REPORT_NEW_CHOICE,;
                {||;
                    say(;
                        maxrow()-1, 0,;
                        padc( MNU_MENU_REPORT_REPORT_NEW_MSG, maxcol()+1 );
                    );
                };
            )

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_REPORT_REPORT_MODIFY_CHOICE,;
                {||;
                    say(;
                        maxrow()-1, 0,;
                        padc( MNU_MENU_REPORT_REPORT_MODIFY_MSG, maxcol()+1 );
                    );
                };
            )

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_REPORT_REPORT_FORM_CHOICE,;
                {||;
                    say(;
                        maxrow()-1, 0,;
                        padc( MNU_MENU_REPORT_REPORT_FORM_MSG, maxcol()+1 );
                    );
                };
            )

            say(;
                nTB+(++n), nLB+1,;
                replicate("", _MENU_PULLDOWN_WIDTH-2);
            )

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_REPORT_DOCUMENT_NEW_CHOICE,;
                {||;
                    say(;
                        maxrow()-1, 0,;
                        padc( MNU_MENU_REPORT_DOCUMENT_NEW_MSG, maxcol()+1 );
                    );
                };
            )

            *-----------------------------------------------------
            * Read the menu choice.
            *-----------------------------------------------------

            nMnuChoice := menuTo( @aMenu, nMnuChoice )

            *-----------------------------------------------------
            * Save the screen before any action is taken.
            *-----------------------------------------------------

            cSaveScreen := mouseScrSave()

            *-----------------------------------------------------
            * Arrow key redirections are released.
            *-----------------------------------------------------

            setkey( K_LEFT, NIL )
            setkey( K_RIGHT, NIL )

            *-----------------------------------------------------
            * The basic color is prepared.
            *-----------------------------------------------------

            setcolor( COLOR_BASE )

            *-----------------------------------------------------
            * The help is not mainteined.
            *-----------------------------------------------------

            bSaveF1 := setkey( K_F1, NIL )

            *-----------------------------------------------------
            * The choice is analised.
            *-----------------------------------------------------

            do case
            case nMnuChoice == 0

                *-------------------------------------------------
                * [Esc] means leave.
                *-------------------------------------------------

                exit                                    // EXIT

            case nMnuChoice == 1

                if Alias() == ""

                    alertBox( _ERROR_NO_ALIAS )

                else

                    dbgotop()

                    *---------------------------------------------
                    * Macro Recorder.
                    *---------------------------------------------

                    if  isMemvar( _MEMVAR_MACRO_RECORDING_NAME )

                        Memvar->_MEMVAR_MACRO_RECORDER +=;
                            NL(1)                           +;
                            '// Go Top Record.'             +;
                            NL(1)                           +;
                            'dbgotop()'                     +;
                            NL(1)

                    end

                end

            case nMnuChoice == 2

                setkey( K_F1, {|| nbHelp( "<REPORT - NEW LABEL>", , chr(K_ENTER) ) } )

                lbl()

            case nMnuChoice == 3

                lblModify()

            case nMnuChoice == 4
            
                lblPrint()
                
            case nMnuChoice == 5

                setkey( K_F1, {|| nbHelp( "<REPORT - NEW REPORT>", , chr(K_ENTER) ) } )

                frm()

            case nMnuChoice == 6

                frmModify()

            case nMnuChoice == 7

                frmPrint()

            case nMnuChoice == 8

                doc()

            end

            *-----------------------------------------------------
            * Restore arrow redirection.
            *-----------------------------------------------------

            setkey( K_LEFT, { || mnuLeft() } )
            setkey( K_RIGHT, { || mnuRight() } )

            *-----------------------------------------------------
            * Restore the screen containing the menu.
            *-----------------------------------------------------

            mouseScrRestore(,,,,cSaveScreen)

            *-----------------------------------------------------
            * Restore the help.
            *-----------------------------------------------------

            setkey( K_F1, bSaveF1 )

        end

    end //sequence

    *-------------------------------------------------------------
    * Restore old data and screen before this menu.
    *-------------------------------------------------------------

    mouseScrRestore( ,,,, cOldScreen )
    set(_SET_WRAP, lOldSetWrap )
    set(_SET_MESSAGE, nOldSetMessage )
    set(_SET_MCENTER, lOldSetCenter )
    setkey( K_F1, bOld_F1 )

    return lRetBreak

*-----------------------------------------------------------------
static function mnuvHtf( nHLeft, nHRight )

    local aMenu             := {}
    local cOldScreen        := mouseScrSave()
    local lOldSetWrap       := set(_SET_WRAP, .T.)
    local nOldSetMessage    := set(_SET_MESSAGE, maxrow()-1)
    local lOldSetCenter     := set(_SET_MCENTER, .T.)
    local bOld_F1           :=;
        setkey( K_F1, {|| nbHelp( "<MENU HTF>", , chr(K_ENTER) )} )

    local nTB               := 2            // top border
    local nLB               :=;
        iif(;
            nHLeft+_MENU_PULLDOWN_WIDTH > maxcol(),;
            nHRight - _MENU_PULLDOWN_WIDTH,;
            nHLeft;
        )                                   // left border
    local nBB               := nTB +4       // bottom border
    local nRB               :=;
        nLB +_MENU_PULLDOWN_WIDTH -1        // right border
    local n

    local cSaveScreen
    local bSaveF1

    local nMnuChoice := 0

    local lRetBreak := .F.

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        while .T.                                       // FOREVER

            *-----------------------------------------------------
            * Show the updated status line.
            *-----------------------------------------------------

            statusLine()

            *-----------------------------------------------------
            * Prepare colors for menu.
            *-----------------------------------------------------

            setcolor( COLOR_MENU )

            *-----------------------------------------------------
            * Show the box around the menu.
            *-----------------------------------------------------

            dispBoxShadow(;
                nTB, nLB, nBB, nRB,;
                1,;
                dispBoxColor( 1 ),;
                dispBoxColor( 2 );
            )

            *-----------------------------------------------------
            * Display key reminder.
            *-----------------------------------------------------

            say( maxrow(), 00, padc( _KEY_MENU_V, maxcol()+1 ) )

            *-----------------------------------------------------
            * <n> contains the menu choice. At the beguinning it
            * starts from zero.
            *-----------------------------------------------------

            n := 0

            *-----------------------------------------------------
            * Start the sequence of menu items and separation
            * lines.
            *-----------------------------------------------------

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_HTF_BROWSE_CHOICE,;
                {|| say( maxrow()-1, 0,;
                    padc( MNU_MENU_HTF_BROWSE_MSG, maxcol()+1 ) );
                };
            )

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_HTF_GENERATE_CHOICE,;
                {||;
                    say( maxrow()-1, 0,;
                    padc( MNU_MENU_HTF_GENERATE_MSG, maxcol()+1 ) );
                };
            )

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_HTML_GENERATE_CHOICE,;&&&&&&&&&
                {||;
                    say( maxrow()-1, 0,;
                    padc( MNU_MENU_HTML_GENERATE_MSG, maxcol()+1 ) );
                };
            )

            *-----------------------------------------------------
            * Read the menu choice.
            *-----------------------------------------------------

            nMnuChoice := menuTo( @aMenu, nMnuChoice )

            *-----------------------------------------------------
            * Save the screen before any action is taken.
            *-----------------------------------------------------

            cSaveScreen := mouseScrSave()

            *-----------------------------------------------------
            * Arrow key redirections are released.
            *-----------------------------------------------------

            setkey( K_LEFT, NIL )
            setkey( K_RIGHT, NIL )

            *-----------------------------------------------------
            * The basic color is prepared.
            *-----------------------------------------------------

            setcolor( COLOR_BASE )

            *-----------------------------------------------------
            * The help is not mainteined.
            *-----------------------------------------------------

            bSaveF1 := setkey( K_F1, NIL )

            *-----------------------------------------------------
            * The choice is analised.
            *-----------------------------------------------------

            do case
            case nMnuChoice == 0

                *-------------------------------------------------
                * [Esc] means leave.
                *-------------------------------------------------

                exit                                    // EXIT

            case nMnuChoice == 1

                htfOpen()

            case nMnuChoice == 2

                htfGenDialog()

            case nMnuChoice == 3

                htf2HtmlDialog()

            end

            *-----------------------------------------------------
            * Restore arrow redirection.
            *-----------------------------------------------------

            setkey( K_LEFT, { || mnuLeft() } )
            setkey( K_RIGHT, { || mnuRight() } )

            *-----------------------------------------------------
            * Restore the screen containing the menu.
            *-----------------------------------------------------

            mouseScrRestore(,,,,cSaveScreen)

            *-----------------------------------------------------
            * Restore the help.
            *-----------------------------------------------------

            setkey( K_F1, bSaveF1 )

        end

    end //sequence

    *-------------------------------------------------------------
    * Restore old data and screen before this menu.
    *-------------------------------------------------------------

    mouseScrRestore( ,,,, cOldScreen )
    set(_SET_WRAP, lOldSetWrap )
    set(_SET_MESSAGE, nOldSetMessage )
    set(_SET_MCENTER, lOldSetCenter )
    setkey( K_F1, bOld_F1 )

    return lRetBreak

*-----------------------------------------------------------------
static function mnuvMacro( nHLeft, nHRight )

    local aMenu             := {}
    local cOldScreen        := mouseScrSave()
    local lOldSetWrap       := set(_SET_WRAP, .T.)
    local nOldSetMessage    := set(_SET_MESSAGE, maxrow()-1)
    local lOldSetCenter     := set(_SET_MCENTER, .T.)
    local bOld_F1           :=;
        setkey( K_F1, {|| nbHelp( "<MENU MACRO>", , chr(K_ENTER) )} )
    local nOldSelect

    local nTB               := 2            // top border
    local nLB               :=;
        iif(;
            nHLeft+_MENU_PULLDOWN_WIDTH > maxcol(),;
            nHRight - _MENU_PULLDOWN_WIDTH,;
            nHLeft;
        )                                    // left border
    local nBB               := nTB +10       // bottom border
    local nRB               :=;
        nLB +_MENU_PULLDOWN_WIDTH -1         // right border
    local n

    local cSaveScreen
    local bSaveF1

    local nMnuChoice := 0

    local lRetBreak := .F.

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        while .T.                                       // FOREVER

            *-----------------------------------------------------
            * Show the updated status line.
            *-----------------------------------------------------

            statusLine()

            *-----------------------------------------------------
            * Prepare colors for menu.
            *-----------------------------------------------------

            setcolor( COLOR_MENU )

            *-----------------------------------------------------
            * Show the box around the menu.
            *-----------------------------------------------------

            dispBoxShadow(;
                nTB, nLB, nBB, nRB,;
                1,;
                dispBoxColor( 1 ),;
                dispBoxColor( 2 );
            )

            *-----------------------------------------------------
            * Display key reminder.
            *-----------------------------------------------------

            say( maxrow(), 00, padc( _KEY_MENU_V, maxcol()+1 ) )

            *-----------------------------------------------------
            * <n> contains the menu choice. At the beguinning it
            * starts from zero.
            *-----------------------------------------------------

            n := 0

            *-----------------------------------------------------
            * Start the sequence of menu items and separation
            * lines.
            *-----------------------------------------------------

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_MACRO_START_STOP_CHOICE,;
                {||;
                    say(;
                        maxrow()-1, 0,;
                        padc( MNU_MENU_MACRO_START_STOP_MSG, maxcol()+1 );
                    );
                };
            )

            say(;
                nTB+(++n), nLB+1,;
                replicate("", _MENU_PULLDOWN_WIDTH-2);
            )

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_MACRO_SAVE_RECORD_CHOICE,;
                {||;
                    say(;
                        maxrow()-1, 0,;
                        padc( MNU_MENU_MACRO_SAVE_RECORD_MSG, maxcol()+1 );
                    );
                };
            )

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_MACRO_DELETE_RECORD_CHOICE,;
                {||;
                    say(;
                        maxrow()-1, 0,;
                        padc( MNU_MENU_MACRO_DELETE_RECORD_MSG, maxcol()+1 );
                    );
                };
            )

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_MACRO_EDIT_RECORD_CHOICE,;
                {|| say( maxrow()-1, 0,;
                    padc( MNU_MENU_MACRO_EDIT_RECORD_MSG, maxcol()+1 ) );
                    };
                )

            say(;
                nTB+(++n), nLB+1,;
                replicate("", _MENU_PULLDOWN_WIDTH-2);
            )

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_MACRO_COMPILE_CHOICE,;
                {||;
                    say(;
                        maxrow()-1, 0,;
                        padc( MNU_MENU_MACRO_COMPILE_MSG, maxcol()+1 );
                    );
                };
            )

            say(;
                nTB+(++n), nLB+1,;
                replicate("", _MENU_PULLDOWN_WIDTH-2);
            )

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_MACRO_LOAD_CHOICE,;
                {||;
                    say(;
                        maxrow()-1, 0,;
                        padc( MNU_MENU_MACRO_LOAD_MSG, maxcol()+1 );
                    );
                };
            )

            *-----------------------------------------------------
            * Read the menu choice.
            *-----------------------------------------------------

            nMnuChoice := menuTo( @aMenu, nMnuChoice )

            *-----------------------------------------------------
            * Save the screen before any action is taken.
            *-----------------------------------------------------

            cSaveScreen := mouseScrSave()

            *-----------------------------------------------------
            * Arrow key redirections are released.
            *-----------------------------------------------------

            setkey( K_LEFT, NIL )
            setkey( K_RIGHT, NIL )

            *-----------------------------------------------------
            * The basic color is prepared.
            *-----------------------------------------------------

            setcolor( COLOR_BASE )

            *-----------------------------------------------------
            * The help is not mainteined.
            *-----------------------------------------------------

            bSaveF1 := setkey( K_F1, NIL )

            *-----------------------------------------------------
            * The choice is analised.
            *-----------------------------------------------------

            do case
            case nMnuChoice == 0

                *-------------------------------------------------
                * [Esc] means leave.
                *-------------------------------------------------

                exit                                    // EXIT

            case nMnuChoice == 1

                *-------------------------------------------------
                * Start/Stop the macro recording.
                *-------------------------------------------------

                do case
                case;
                    isMemvar( _MEMVAR_MACRO_RECORDING_NAME );
                    .and.;
                    isMemvar( _MEMVAR_MACRO_RECORDER_NAME )

                    *---------------------------------------------
                    * Recording in on: pause recording.
                    *---------------------------------------------

                    memRelease( _MEMVAR_MACRO_RECORDING_NAME )

                case;
                    isMemvar( _MEMVAR_MACRO_RECORDING_NAME );
                    .and.;
                    !isMemvar( _MEMVAR_MACRO_RECORDER_NAME )

                    *---------------------------------------------
                    * It is an error: Correct error.
                    *---------------------------------------------

                    memRelease( _MEMVAR_MACRO_RECORDING_NAME )

                case;
                    !isMemvar( _MEMVAR_MACRO_RECORDING_NAME );
                    .and.;
                    isMemvar( _MEMVAR_MACRO_RECORDER_NAME )

                    *---------------------------------------------
                    * Recording is paused: resume recording.
                    *---------------------------------------------

                    public _MEMVAR_MACRO_RECORDING

                    memvar->_MEMVAR_MACRO_RECORDING := .T.

                case;
                    !isMemvar( _MEMVAR_MACRO_RECORDING_NAME );
                    .and.;
                    !isMemvar( _MEMVAR_MACRO_RECORDER_NAME )

                    *---------------------------------------------
                    * Recording is stopped: start recording.
                    *---------------------------------------------

                    public _MEMVAR_MACRO_RECORDING

                    memvar->_MEMVAR_MACRO_RECORDING := .T.

                    public _MEMVAR_MACRO_RECORDER

                    memvar->_MEMVAR_MACRO_RECORDER :=;
                        "//" + replicate( "=", 65 )         +;
                        NL(1)                               +;
                        "// nB " + PROGRAM_VERSION          +;
                        NL(1)                               +;
                        "// " + PROGRAM_C + "."             +;
                        NL(1)                               +;
                        "//"                                +;
                        NL(1)                               +;
                        "// Macro recorder."                +;
                        NL(1)                               +;
                        "// " + dtoc(date()) + " " + time() +;
                        NL(1)                               +;
                        "// " + replicate( "=", 65 )        +;
                        NL(2)

                end

            case nMnuChoice == 2

                if  isMemvar( _MEMVAR_MACRO_RECORDER_NAME )

                    macroSave()

                end

            case nMnuChoice == 3

                if  alertBox(;
                        MACRO_ALERT_ERASE,;
                        {;
                            _MENU_NO,;
                            _MENU_YES;
                        };
                    ) == 2

                    *---------------------------------------------
                    * Erase recording OK.
                    *---------------------------------------------

                    if  isMemvar( _MEMVAR_MACRO_RECORDING_NAME )
                        memRelease( _MEMVAR_MACRO_RECORDING_NAME )
                    end

                    if  isMemvar( _MEMVAR_MACRO_RECORDER_NAME )
                        memRelease( _MEMVAR_MACRO_RECORDER_NAME )
                    end

                else

                    *---------------------------------------------
                    * Do not erase recording.
                    *---------------------------------------------

                end

            case nMnuChoice == 4

                if isMemvar( _MEMVAR_MACRO_RECORDER_NAME )

                    macroEdit()

                end

            case nMnuChoice == 5

                macroCompile()

            case nMnuChoice == 6

                macroLoad()

            end

            *-----------------------------------------------------
            * Restore arrow redirection.
            *-----------------------------------------------------

            setkey( K_LEFT, { || mnuLeft() } )
            setkey( K_RIGHT, { || mnuRight() } )

            *-----------------------------------------------------
            * Restore the screen containing the menu.
            *-----------------------------------------------------

            mouseScrRestore(,,,,cSaveScreen)

            *-----------------------------------------------------
            * Restore the help.
            *-----------------------------------------------------

            setkey( K_F1, bSaveF1 )

        end

    end //sequence

    *-------------------------------------------------------------
    * Restore old data and screen before this menu.
    *-------------------------------------------------------------

    mouseScrRestore( ,,,, cOldScreen )
    set(_SET_WRAP, lOldSetWrap )
    set(_SET_MESSAGE, nOldSetMessage )
    set(_SET_MCENTER, lOldSetCenter )
    setkey( K_F1, bOld_F1 )

    return lRetBreak

*-----------------------------------------------------------------
static function mnuvInfo( nHLeft, nHRight )

    local aMenu             := {}
    local cOldScreen        := mouseScrSave()
    local lOldSetWrap       := set(_SET_WRAP, .T.)
    local nOldSetMessage    := set(_SET_MESSAGE, maxrow()-1)
    local lOldSetCenter     := set(_SET_MCENTER, .T.)
    local bOld_F1           :=;
        setkey( K_F1, {|| nbHelp( "<MENU INFO>", , chr(K_ENTER) )} )
    local nOldSelect

    local nTB               := 2            // top border
    local nLB               :=;
        iif( nHLeft+_MENU_PULLDOWN_WIDTH > maxcol(),;
        nHRight - _MENU_PULLDOWN_WIDTH,;
        nHLeft )                            // left border
    local nBB               := nTB +8       // bottom border
    local nRB               :=;
        nLB +_MENU_PULLDOWN_WIDTH -1         // right border
    local n

    local cSaveScreen
    local bSaveF1

    local nMnuChoice := 0

    local lRetBreak := .F.

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        while .T.                                       // FOREVER

            *-----------------------------------------------------
            * Show the updated status line.
            *-----------------------------------------------------

            statusLine()

            *-----------------------------------------------------
            * Prepare colors for menu.
            *-----------------------------------------------------

            setcolor( COLOR_MENU )

            *-----------------------------------------------------
            * Show the box around the menu.
            *-----------------------------------------------------

            dispBoxShadow(;
                nTB, nLB, nBB, nRB,;
                1,;
                dispBoxColor( 1 ),;
                dispBoxColor( 2 );
            )

            *-----------------------------------------------------
            * Display key reminder.
            *-----------------------------------------------------

            say( maxrow(), 00, padc( _KEY_MENU_V, maxcol()+1 ) )

            *-----------------------------------------------------
            * <n> contains the menu choice. At the beguinning it
            * starts from zero.
            *-----------------------------------------------------

            n := 0

            *-----------------------------------------------------
            * Start the sequence of menu items and separation
            * lines.
            *-----------------------------------------------------

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_INFO_ABOUT_CHOICE,;
                {||;
                    say( maxrow()-1, 0,;
                        padc( MNU_MENU_INFO_ABOUT_MSG, maxcol()+1 );
                    );
                };
            )

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_INFO_MANUAL_CHOICE,;
                {||;
                    say( maxrow()-1, 0,;
                        padc( MNU_MENU_INFO_MANUAL_MSG, maxcol()+1 );
                     );
                };
            )

            say(;
                nTB+(++n), nLB+1,;
                replicate("", _MENU_PULLDOWN_WIDTH-2);
            )

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_INFO_F1_CHOICE,;
                {|| say( maxrow()-1, 0, padc( "", maxcol()+1 ) ) };
            )

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_INFO_F3_CHOICE,;
                {|| say( maxrow()-1, 0, padc( "", maxcol()+1 ) ) };
            )

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_INFO_F5_CHOICE,;
                {|| say( maxrow()-1, 0, padc( "", maxcol()+1 ) ) };
            )

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_INFO_F8_CHOICE,;
                {|| say( maxrow()-1, 0, padc( "", maxcol()+1 ) ) };
            )

            *-----------------------------------------------------
            * Read the menu choice.
            *-----------------------------------------------------

            nMnuChoice := menuTo( @aMenu, nMnuChoice )

            *-----------------------------------------------------
            * Save the screen before any action is taken.
            *-----------------------------------------------------

            cSaveScreen := mouseScrSave()

            *-----------------------------------------------------
            * Arrow key redirections are released.
            *-----------------------------------------------------

            setkey( K_LEFT, NIL )
            setkey( K_RIGHT, NIL )

            *-----------------------------------------------------
            * The basic color is prepared.
            *-----------------------------------------------------

            setcolor( COLOR_BASE )

            *-----------------------------------------------------
            * The help is not mainteined.
            *-----------------------------------------------------

            bSaveF1 := setkey( K_F1, NIL )

            *-----------------------------------------------------
            * The choice is analised.
            *-----------------------------------------------------

            do case
            case nMnuChoice == 0

                *-------------------------------------------------
                * [Esc] means leave.
                *-------------------------------------------------

                exit                                    // EXIT

            case nMnuChoice == 1

                alertBox( PROGRAM_COPYRIGHT_INFO )

            case nMnuChoice == 2

                nbHelp( "<INDEX>", , chr(K_ENTER) )

            case nMnuChoice == 3

                nbHelp( "<THE MENU SYSTEM>", , chr(K_ENTER) )

            case nMnuChoice == 4

                Text( dbiAllStatus() )

            case nMnuChoice == 5

                setAlternate()

            case nMnuChoice == 6

                dotLine()

            end

            *-----------------------------------------------------
            * Restore arrow redirection.
            *-----------------------------------------------------

            setkey( K_LEFT, { || mnuLeft() } )
            setkey( K_RIGHT, { || mnuRight() } )

            *-----------------------------------------------------
            * Restore the screen containing the menu.
            *-----------------------------------------------------

            mouseScrRestore(,,,,cSaveScreen)

            *-----------------------------------------------------
            * Restore the help.
            *-----------------------------------------------------

            setkey( K_F1, bSaveF1 )

        end

    end //sequence

    *-------------------------------------------------------------
    * Restore old data and screen before this menu.
    *-------------------------------------------------------------

    mouseScrRestore( ,,,, cOldScreen )
    set(_SET_WRAP, lOldSetWrap )
    set(_SET_MESSAGE, nOldSetMessage )
    set(_SET_MCENTER, lOldSetCenter )
    setkey( K_F1, bOld_F1 )

    return lRetBreak

*-----------------------------------------------------------------
static function mnuvDoc( nHLeft, nHRight )

    local aMenu               := {}
    local cOldScreen          := mouseScrSave()
    local lOldSetWrap         := set(_SET_WRAP, .T.)
    local nOldSetMessage      := set(_SET_MESSAGE, maxrow()-1)
    local lOldSetCenter       := set(_SET_MCENTER, .T.)
    local bOld_F1             :=;
        setkey( K_F1, {|| nbHelp( "<MENU DOC>", , chr(K_ENTER) )} )

    local nTB                 := 2      // top border
    local nLB                 :=;
        iif( nHLeft+_MENU_PULLDOWN_WIDTH > maxcol(),;
        nHRight - _MENU_PULLDOWN_WIDTH,;
        nHLeft )                        // left border
    local nBB                 := nTB +13       // bottom border
    local nRB                 :=;
        nLB +_MENU_PULLDOWN_WIDTH -1            // right border
    local n

    local cSaveScreen
    local bSaveF1

    local nMnuChoice := 0

    local lRetBreak := .F.

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        while .T.                                       // FOREVER

            *-----------------------------------------------------
            * Show the updated status line.
            *-----------------------------------------------------

            statusLine()

            *-----------------------------------------------------
            * Prepare colors for menu.
            *-----------------------------------------------------

            setcolor( COLOR_MENU )

            *-----------------------------------------------------
            * Show the box around the menu.
            *-----------------------------------------------------

            dispBoxShadow(;
                nTB, nLB, nBB, nRB,;
                1,;
                dispBoxColor( 1 ),;
                dispBoxColor( 2 );
            )

            *-----------------------------------------------------
            * Display key reminder.
            *-----------------------------------------------------

            say( maxrow(), 00, padc( _KEY_MENU_V, maxcol()+1 ) )

            *-----------------------------------------------------
            * <n> contains the menu choice. At the beguinning it
            * starts from zero.
            *-----------------------------------------------------

            n := 0

            *-----------------------------------------------------
            * Start the sequence of menu items and separation
            * lines.
            *-----------------------------------------------------

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_DOC_NEW_CHOICE,;
                {||;
                    say( maxrow()-1, 0,;
                        padc( MNU_MENU_DOC_NEW_MSG, maxcol()+1 );
                    );
                };
            )

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_DOC_OPEN_CHOICE,;
                {||;
                    say( maxrow()-1, 0,;
                        padc( MNU_MENU_DOC_OPEN_MSG, maxcol()+1 );
                    );
                };
            )

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_DOC_SAVE_CHOICE,;
                {||;
                    say( maxrow()-1, 0,;
                        padc( MNU_MENU_DOC_SAVE_MSG, maxcol()+1 );
                    );
                };
            )

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_DOC_SAVE_AS_CHOICE,;
                {||;
                    say( maxrow()-1, 0,;
                        padc( MNU_MENU_DOC_SAVE_AS_MSG, maxcol()+1 );
                    );
                };
            )

            say(;
                nTB+(++n), nLB+1,;
                replicate("", _MENU_PULLDOWN_WIDTH-2);
            )

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_DOC_SET_DEFAULT_TO_CHOICE,;
                {||;
                    say( maxrow()-1, 0,;
                        padc( MNU_MENU_DOC_SET_DEFAULT_TO_MSG, maxcol()+1 );
                    );
                };
            )

            say(;
                nTB+(++n), nLB+1,;
                replicate("", _MENU_PULLDOWN_WIDTH-2);
            )

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_DOC_PRINT_AS_IS_CHOICE,;
                {||;
                    say( maxrow()-1, 0,;
                        padc( MNU_MENU_DOC_PRINT_AS_IS_MSG, maxcol()+1 );
                    );
                };
            )

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_DOC_PRINT_ONCE_CHOICE,;
                {||;
                    say( maxrow()-1, 0,;
                        padc( MNU_MENU_DOC_PRINT_ONCE_MSG, maxcol()+1 );
                    );
                };
            )

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_DOC_PRINT_CHOICE,;
                {||;
                    say( maxrow()-1, 0,;
                        padc( MNU_MENU_DOC_PRINT_MSG, maxcol()+1 );
                    );
                };
            )

            say(;
                nTB+(++n), nLB+1,;
                replicate("", _MENU_PULLDOWN_WIDTH-2);
            )

            menuPrompt(;
                @aMenu, nTB+(++n), nLB+1,;
                MNU_MENU_DOC_EXIT_CHOICE,;
                {||;
                    say( maxrow()-1, 0,;
                        padc( MNU_MENU_DOC_EXIT_MSG, maxcol()+1 );
                    );
                };
            )

            *-----------------------------------------------------
            * Read the menu choice.
            *-----------------------------------------------------

            nMnuChoice := menuTo( @aMenu, nMnuChoice )

            *-----------------------------------------------------
            * Save the screen before any action is taken.
            *-----------------------------------------------------

            cSaveScreen := mouseScrSave()

            *-----------------------------------------------------
            * Arrow key redirections are released.
            *-----------------------------------------------------

            setkey( K_LEFT, NIL )
            setkey( K_RIGHT, NIL )

            *-----------------------------------------------------
            * The basic color is prepared.
            *-----------------------------------------------------

            setcolor( COLOR_BASE )

            *-----------------------------------------------------
            * The help is not mainteined.
            *-----------------------------------------------------

            bSaveF1 := setkey( K_F1, NIL )

            *-----------------------------------------------------
            * The choice is analised.
            *-----------------------------------------------------

            do case
            case nMnuChoice == 0

                *-------------------------------------------------
                * [Esc] means leave.
                *-------------------------------------------------

                exit                                    // EXIT

            case (nMnuChoice == 1)

                lRetBreak := .T.

                docNew()

                exit                                    // EXIT

            case (nMnuChoice == 2)

                lRetBreak := .T.

                docOpen()

                exit                                    // EXIT

            case (nMnuChoice == 3)

                lRetBreak := .T.

                docSave()

                exit                                    // EXIT

            case (nMnuChoice == 4)

                lRetBreak := .T.

                docSaveAs()

                exit                                    // EXIT

            case (nMnuChoice == 5)

                setAlternate()

            case (nMnuChoice == 6)

                lRetBreak := .T.

                if  isConsoleOn()

                    setcolor( COLOR_BODY )

                    scroll()

                end

                QOut( docArray[DOC_TEXT] )

                if  isConsoleOn()

                    *---------------------------------------------
                    * Wait for a key press.
                    *---------------------------------------------

                    alertbuttons( maxrow(), 0, {"[Esc]"} )
                    &&&&inkey(0)

                    scroll()

                end

                *-------------------------------------------------
                * Macro recorder.
                *-------------------------------------------------

                if  isMemvar( _MEMVAR_MACRO_RECORDING_NAME )

                    Memvar->_MEMVAR_MACRO_RECORDER +=;
                        NL(1)                               +;
                        '// Print text.'                    +;
                        NL(1)                               +;
                        '// qout( <text> )'                 +;
                        NL(1)                               +;
                        'qout( memoread( "'                 +;
                            docArray[DOC_NAME]              +;
                            '" ) )'                         +;
                        NL(1)

                end

                exit                                    // EXIT

            case (nMnuChoice == 7)

                lRetBreak := .T.

                if  isConsoleOn()

                    setcolor( COLOR_BODY )

                    scroll()

                end
                
                rpt( docArray[DOC_TEXT] )

                if isConsoleOn()

                    *---------------------------------------------
                    * Wait for a key press.
                    *---------------------------------------------

                    alertbuttons( maxrow(), 0, {"[Esc]"} )
                    &&&&inkey(0)

                    scroll()

                end

                *-------------------------------------------------
                * Macro recorder.
                *-------------------------------------------------

                if  isMemvar( _MEMVAR_MACRO_RECORDING_NAME )

                    Memvar->_MEMVAR_MACRO_RECORDER +=;
                        NL(1)                               +;
                        '// Print Text once with the '      +;
                            'active Alias.'                 +;
                        NL(1)                               +;
                        '// This is not a CA-Clipper '      +;
                            'standard function.'            +;
                        NL(1)                               +;
                        '// rpt( <cText> )'                 +;
                        NL(1)                               +;
                        'rpt( memoread("'                   +;
                            docArray[DOC_NAME]              +;
                            '") )'                          +;
                        NL(1)

                end
                    
                exit                                    // EXIT

            case (nMnuChoice == 8)

                lRetBreak := .T.

                if isConsoleOn()

                    setcolor( COLOR_BODY )

                    scroll()

                end

                docPrintCondition( docArray[DOC_TEXT] )

                if alias() == ""

                    *---------------------------------------------
                    * Can't print without alias.
                    *---------------------------------------------
                
                    
                else

                    if  isConsoleOn()

                        *-----------------------------------------
                        * Wait for a key press.
                        *-----------------------------------------

                        alertbuttons( maxrow(), 0, {"[Esc]"} )
                        &&&&inkey(0)

                        scroll()

                    end

                    *---------------------------------------------
                    * Macro recorder.
                    *---------------------------------------------

                    if  isMemvar( _MEMVAR_MACRO_RECORDING_NAME )
    
                        Memvar->_MEMVAR_MACRO_RECORDER +=;
                            NL(1)                               +;
                            '// Print Text with the active '    +;
                                'Alias.'                        +;
                            NL(1)                               +;
                            '// This is not a CA-Clipper '      +;
                                'standard function.'            +;
                            NL(1)                               +;
                            '// rptMany( <cText> )'             +;
                            NL(1)                               +;
                            'rptMany( memoread("'               +;
                                docArray[DOC_NAME]              +;
                                '") )'                          +;
                            NL(1)

                    end
                    
                end

                exit                                    // EXIT

            case (nMnuChoice == 9)

                lRetBreak := .T.

                docArray[DOC_EXIT] := .T.

                exit                                    // EXIT

            end

            *-----------------------------------------------------
            * Restore arrow redirection.
            *-----------------------------------------------------

            setkey( K_LEFT, { || mnuLeft() } )
            setkey( K_RIGHT, { || mnuRight() } )

            *-----------------------------------------------------
            * Restore the screen containing the menu.
            *-----------------------------------------------------

            mouseScrRestore(,,,,cSaveScreen)

            *-----------------------------------------------------
            * Restore the help.
            *-----------------------------------------------------

            setkey( K_F1, bSaveF1 )

        end

    end //sequence

    *-------------------------------------------------------------
    * Restore old data and screen before this menu.
    *-------------------------------------------------------------

    mouseScrRestore( ,,,, cOldScreen )
    set(_SET_WRAP, lOldSetWrap )
    set(_SET_MESSAGE, nOldSetMessage )
    set(_SET_MCENTER, lOldSetCenter )
    setkey( K_F1, bOld_F1 )

    *-------------------------------------------------------------
    * If <lRetBreak> is True, the menu will terminate.
    *-------------------------------------------------------------

    return lRetBreak

#endif  // RUNTIME

*-----------------------------------------------------------------
* SET - SET FUNCTIONS
*-----------------------------------------------------------------

#define SET_DIALOG_BOX_TOP_ALTERNATE_FILE;
    "SET OUTPUT TO ..."

#define SET_PROMPT_ALTERNATE_FILE;
    "Set output to (define output device):"

#define SET_HELP_ALTERNATE_FILE;
    "SET OUTPUT TO ..." +;
    NL(3) +;
    "A output filename or peripheral name must be specified." +;
    NL(3) +;
    "Examples:" +;
    NL(2) +;
    "CON       Console/Display." + NL(1) +;
    "PRN       Printer." + NL(1) +;
    "LPT1      Printer connected at LPT1." + NL(1) +;
    "LPT2      Printer connected at LPT2." + NL(1) +;
    "LPT3      Printer connected at LPT3." + NL(1) +;
    "LPT4      Printer connected at LPT4." + NL(1) +;
    "COM1      Printer connected at COM1." + NL(1) +;
    "COM2      Printer connected at COM2." + NL(1) +;
    "COM3      Printer connected at COM3." + NL(1) +;
    "COM4      Printer connected at COM4." + NL(1) +;
    NL(2) +;
    "If it is used a name not corresponding to any present " +;
    "peripheral, then, a text file with the same name and .TXT " +;
    "extention will be created."

*-----------------------------------------------------------------
static function setAlternate()
*&
*

    local aoGet             := {}
    local aButtons          := {}
    local aTab              := TAB_DEFAULT

    local bOldErrorHandler
    local cOldScreen
    local cOldColor         := setcolor()
    local nOldCursor        := setcursor( SETCURSOR_NORMAL )
    local bOld_F1           :=;
        setkey( K_F1, {|| Text( SET_HELP_ALTERNATE_FILE ) } )
    local bOld_F2           := setkey( K_F2, NIL )
    local nOldRow           := row()
    local nOldCol           := col()
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth

    local cAltFile          := ;
        padr( set( _SET_ALTFILE ), _MAX_STRING_LEN )

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Create a kind of window.
        *---------------------------------------------------------

        nBottom     := maxrow()
        nTop        := nBottom - 6
        nLeft       := 0
        nRight      := maxcol()
        nWidth      := nRight - nLeft +1

        aTab[TAB_LEFT]      := nLeft
        aTab[TAB_RIGHT]     := nRight
        aTab[TAB_TAB_ARRAY] := { 16, 8 }

        cOldScreen          :=; 
            mouseScrSave( nTop, nLeft, nBottom, nRight )

        setcolor( COLOR_BODY )

        scroll( nTop, nLeft, nBottom, nRight )

        dispBoxShadow(;
            nTop, nLeft, nBottom, nRight,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )

        say(;
            nTop+1, nLeft+1,;
            padc( SET_DIALOG_BOX_TOP_ALTERNATE_FILE, nWidth-2 ),;
            ,;
            COLOR_HEAD;
        )

        *---------------------------------------------------------
        * Do editing.
        *---------------------------------------------------------

        while .t.                                   // FOREVER

            say(;
                nTop+3, nLeft+1,;
                SET_PROMPT_ALTERNATE_FILE;
            )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cAltFile := x, cAltFile ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { || trueSetkey( K_F2, NIL ) },;
                { || .T. };
            )

            button( @aButtons, row()+2, nLeft+1,;
                _BUTTON_ESC_CANCEL, , {||keyboard(chr(K_ESC))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F1_HELP, , {|| eval(setkey(K_F1))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_PGDN_CONFIRM, , {||keyboard(chr(K_PGDN))} )

            *-----------------------------------------------------
            * Read.
            *-----------------------------------------------------

            read( aoGet, , aButtons )
            aoGet   := {}
            aButtons := {}

            *-----------------------------------------------------
            * Check what was read.
            *-----------------------------------------------------
                                        
            do case
            case lastkey() = K_ESC

                *-------------------------------------------------
                * [Esc] means leave.
                *-------------------------------------------------

                exit                                    // EXIT

            case lastkey() = K_PGDN

                *-------------------------------------------------
                * [PgDn] means confirm.
                *-------------------------------------------------

                setOutput( alltrim( cAltFile ) )

                *-------------------------------------------------
                * Macro recorder.
                *-------------------------------------------------

                if  isMemvar( _MEMVAR_MACRO_RECORDING_NAME )

                    Memvar->_MEMVAR_MACRO_RECORDER +=;
                        NL(1)                               +;
                        '// Set Output for qout() and '     +;
                                'qqout()'                   +;
                        NL(1)                               +;
                        '// This is not a CA-Clipper '      +;
                            'standard function.'            +;
                        NL(1)                               +;
                        'setOutput( "'                      +;
                            alltrim( cAltFile )             +;
                            '" )'                           +;
                        NL(1)

                end

                exit                                    // EXIT

            otherwise

                *-------------------------------------------------
                * Exit with different keys is not allowed.
                * Loop again.
                *-------------------------------------------------

            end

        end

        *---------------------------------------------------------
        * Restore screen deleting the window.
        *---------------------------------------------------------

        mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

    end //sequence

    *-------------------------------------------------------------
    * Restore values.
    *-------------------------------------------------------------

    setcursor( nOldCursor )
    setcolor( cOldColor )
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )

    return NIL

*-----------------------------------------------------------------
* STATUS - STATUS LINE
*-----------------------------------------------------------------
static function statusLine()
*
* statusLine() --> NIL
*
* It shows a status line on the first line of the screen.
*

    local bOldErrorHandler
    local cOldColor           := setcolor( COLOR_BASE )
    local lOldSetBlink        := setblink()
    local nOldCursor          := setcursor( SETCURSOR_NONE )
    local nOldRow             := row()
    local nOldCol             := col()

    *-------------------------------------------------------------
    * The status line cleans the space automatically.
    *-------------------------------------------------------------

    say( 0, 0, "" )

    *-------------------------------------------------------------
    * Prepare for a possible blinking.
    *-------------------------------------------------------------

    setblink( .T. )

    if  isMemvar(_MEMVAR_MACRO_RECORDER_NAME)

        *---------------------------------------------------------
        * The macro exists.
        *---------------------------------------------------------

        if  isMemvar(_MEMVAR_MACRO_RECORDING_NAME)

            *-----------------------------------------------------
            * The recorder is active: blink.
            *-----------------------------------------------------

            say( row(), col(), "&", , "*" + setcolor() )

        else

            *-----------------------------------------------------
            * The recorder is paused: no blink.
            *-----------------------------------------------------

            say( row(), col(), "&" )

        end
    else

        *---------------------------------------------------------
        * The memvar macro is not present.
        *---------------------------------------------------------

        say( row(), col(), " " )

    end

    *-------------------------------------------------------------
    * Restore previous blink status.
    *-------------------------------------------------------------

    setblink( lOldSetBlink )

    say( row(), col(), "" )

    *-------------------------------------------------------------
    * Display the default RDD.
    *-------------------------------------------------------------

    say( row(), col(), padr(rddsetdefault(), 10) )

    say( row(), col(), "" )

    *-------------------------------------------------------------
    * Display "*" is set deleted is OFF and the deleted record
    * are shown.
    *-------------------------------------------------------------

    say( row(), col(), iif( set( _SET_DELETED ), " ", "*" ) )

    say( row(), col(), "" )

    *-------------------------------------------------------------
    * Show the current area number.
    *-------------------------------------------------------------

    say( row(), col(), str(select(), 3) )

    say( row(), col(), "" )

    *-------------------------------------------------------------
    * Display the active Alias.
    *-------------------------------------------------------------

    say( row(), col(), padr( alias(), 10 ) )

    say( row(), col(), "" )

    *-------------------------------------------------------------
    * Display the current record number.
    *-------------------------------------------------------------

    do case
    case alias() == ""

        *---------------------------------------------------------
        * There is no active Alias.
        *---------------------------------------------------------

        say( row(), col(), space( 13 ) )


    case eof() .and. bof()

        say( row(), col(), " bof()/eof() " )

    case bof()
    
        say( row(), col(), "    bof()    " )

    case eof()
    
        say( row(), col(), "    eof()    " )

    otherwise

        say(;
            row(), col(),;
            str( recno(), 6 ) + "/" + str( reccount(), 6 );
        )

    end

    say( row(), col(), "" )

    *-------------------------------------------------------------
    * Display the current order bag name (the filename).
    *-------------------------------------------------------------

    say( row(), col(),;
        padr(;
            iif(;
                alias() == "",;
                "",;
                strFile( ordbagname( ordsetfocus() ) );
            ),;
            12;
        );
    )

    say( row(), col(), "" )

    *-------------------------------------------------------------
    * Display the current order number.
    *-------------------------------------------------------------

    say( row(), col(),;
        iif(;
            alias() == "",;
            "  0",;
            str( ordnumber( ordsetfocus() ), 3 );
        );
    )

    say( row(), col(), "" )

    *-------------------------------------------------------------
    * Display the current order name.
    *-------------------------------------------------------------

    say(;
        row(),col(),;
        padr( iif( alias() == "", "", ordsetfocus() ), 10 );
    )

    say( row(), col(), "" )

    *-------------------------------------------------------------
    * Clear the rest of the screen line.
    *-------------------------------------------------------------

    scroll( row(),col(),row(),maxcol() )

    *-------------------------------------------------------------
    * Restore old data.
    *-------------------------------------------------------------

    setcursor( nOldCursor )
    setcolor( cOldColor )
    setpos( nOldRow, nOldCol )

    return NIL

#ifndef RUNTIME
*=================================================================
* DOC - TEXT EDITOR
*=================================================================


#define DOC_TOP      0
#define DOC_LEFT     0
#define DOC_BOTTOM   maxrow()
#define DOC_RIGHT    maxcol()
#define DOC_WIDTH    _MAX_STRING_LEN
#define DOC_TABSIZE  4


#define DOC_WRITING     "Writing..."
#define DOC_WRITE_OK    "Write OK."
#define DOC_WRITE_ERROR "Write error."

#define DOC_PROMPT_OPEN_FILE_DOC;
    "Text/Document filename:"
#define DOC_PROMPT_EXIT;
    "Exit " + DOC_FUNCTION_NAME + " ?"
#define DOC_PROMPT_ABANDON_CHANGES;
    "Abandon changes?"
#define DOC_PROMPT_OPEN_FILE_TEXT_DOCUMENT;
    "Text/Document filename:"
#define DOC_PROMPT_WHILE_EXPRESSION;
    "WHILE expression condition:"
#define DOC_PROMPT_FOR_EXPRESSION;
    "FOR expression condition:"
#define DOC_OPEN_DOCUMENT;
    "Open Document for editing."
#define DOC_INSERT_NEW_DOCUMENT_NAME;
    "Insert new Document name."
#define DOC_FILE_NOT_CHANGED;
    "File has not been modified."
#define DOC_NEW_NAME;
    "Enter new filename."

#define DOC_DIALOG_BOX_TOP_SAVEAS;
    "Text/document Save As"

#define DOC_DIALOG_BOX_TOP_OPEN;
    "Open a Text/Document"
#define DOC_DIALOG_BOX_TOP_PRINT;
    "Print a Text/Document"

#define DOC_ERROR_FILE_CREATION;
    "File creation error!"

*=================================================================
function doc( cFileName )
*
* doc( [<cFileName>] ) --> NIL
*
* Text editor for small files, less than 64K.
*
* <cFileName>  the text file name to modify.
*

    local bOldErrorHandler
    local cOldScreen            := mouseScrSave()
    local cOldColor             := setcolor()
    local nOldCursor            := setcursor( SETCURSOR_NORMAL )
    local bOld_F1               := setkey( K_F1, NIL )
    local bOld_F2               := setkey( K_F2, {|| keyboard( chr(174) + fldAlign() + chr(175) ) } )
    local lOldBell              := set(_SET_BELL, .f.)
    local lOldScoreboard        := set(_SET_SCOREBOARD, .f.)

    *-------------------------------------------------------------
    * [Alt]+[M] and [F10] must be resetted, otherwise the menu
    * will be the ASSIST()!!
    *-------------------------------------------------------------

    local bOld_ALT_M    := setkey( K_ALT_M, NIL )
    local bOld_F10      := setkey( K_F10, NIL )

    local nOldRow       := row()
    local nOldCol       := col()

    local cOldName
    local lNew

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Check input data.
        *---------------------------------------------------------

        if valtype( cFileName ) <> "C"

            *-----------------------------------------------------
            * Define a default name.
            *-----------------------------------------------------

            cFileName :=;
                strAddExtention( _UNTITLED, _EXTENTION_TEXT )

            *-----------------------------------------------------
            * <cOldName> and <lNew> are used to determinate
            * duriing the save procedure if an overwrite is
            * dangerous or is normal.
            *-----------------------------------------------------

            cOldName := NIL

            lNew := .T.

        else

            *-----------------------------------------------------
            * Correct the extention if it was not given.
            *-----------------------------------------------------

            cFileName :=;
                upper(;
                    strAddExtention(;
                        cFileName,;
                        _EXTENTION_TEXT;
                    );
                )

            *-----------------------------------------------------
            * If the file exists, it will be modifyed, else
            * it is a new one.
            *-----------------------------------------------------

            if file( cFileName )

                cOldName := cFileName

            else

                cOldName := NIL

                lNew := .T.

            end

        end

        *---------------------------------------------------------
        * Start the editing.
        *---------------------------------------------------------

        docEditor( cFileName, cOldName, lNew )

    end //sequence

    *-------------------------------------------------------------
    * Restore old data.
    *-------------------------------------------------------------

    setcursor( nOldCursor )
    setcolor( cOldColor )
    mouseScrRestore( NIL, NIL, NIL, NIL, cOldScreen )
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )
    setkey( K_ALT_M, bOld_ALT_M )
    setkey( K_F10, bOld_F10 )
    set(_SET_BELL, lOldBell)
    set(_SET_SCOREBOARD, lOldScoreboard)

    return NIL

*-----------------------------------------------------------------
static function docEditor( cFile, cOld, lNew )

    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth
    local nHeight

    local nKey

    setkey( K_F1, {|| nbHelp( "<THE TEXT EDITOR DOC()>", , chr(K_ENTER) )} )

    *-------------------------------------------------------------
    * <docArray> prepare.
    *-------------------------------------------------------------

    docArray                := Array(DOC_LENGTH)

    docArray[DOC_NEW]       := lNew

    docArray[DOC_OLD]       := cOld
        // eventually upper and alltrim

    docArray[DOC_NAME]      := cFile
        // eventually upper and alltrim

    do case
    case docArray[DOC_NEW]

        docArray[DOC_TEXT]      := ""

    case file( docArray[DOC_NAME] )

        docArray[DOC_TEXT]      := memoRead(cFile)

    otherwise

        docArray[DOC_TEXT]      := ""

    end

    docArray[DOC_WRAP]      := .T.

    docArray[DOC_INS]       := Set(_SET_INSERT)

    docArray[DOC_ROW]       := 1

    docArray[DOC_COL]       := 0

    docArray[DOC_RELROW]    := 0

    docArray[DOC_RELCOL]    := 0

    docArray[DOC_CHANGED]   := .F.

    docArray[DOC_LASTKEY]   := 0

    docArray[DOC_EXIT]      := .F.

    *-------------------------------------------------------------
    * Create a kind of window without saveing the screen.
    *-------------------------------------------------------------

    nTop    := DOC_TOP
    nLeft   := DOC_LEFT
    nBottom := DOC_BOTTOM
    nRight  := DOC_RIGHT
    nWidth  := nRight - nLeft +1
    nHeight := nBottom - nTop +1

    setcolor( COLOR_BODY )

    scroll( nTop, nLeft, nBottom, nRight )

    dispBoxShadow(;
        nTop, nLeft, nBottom, nRight,;
        1,;
        dispBoxColor( 1 ),;
        dispBoxColor( 2 );
    )

    *-------------------------------------------------------------
    * Here is displayed only the footer and not the header
    * as this is a job of DOCUDF()
    *-------------------------------------------------------------

    say(;
        nBottom-1, nLeft+1,;
        padc( _KEY_ESC_EXIT_F1_F10, nWidth-2 ),;
        ,;
        COLOR_HEAD;
    )

    *-------------------------------------------------------------
    * Editing.
    *-------------------------------------------------------------

    while !docArray[DOC_EXIT]

        *---------------------------------------------------------
        * Do normal editing.
        *---------------------------------------------------------

        docDoEditing()

        *---------------------------------------------------------
        * Analise the exit key
        *---------------------------------------------------------

        nKey := docArray[DOC_LASTKEY]

        do case
        case;
            nKey == K_F10                       .or.;
            nKey == K_ALT_M

            *-----------------------------------------------------
            * Start the menu doc() version.
            *-----------------------------------------------------

            docMenu()

            *-----------------------------------------------------
            * Show a new header (it may be changed).
            *-----------------------------------------------------

            docInfoLine()

        case nKey == K_ESC

            *-----------------------------------------------------
            * Termiante editing.
            *-----------------------------------------------------

            docExit()

        otherwise

            *-----------------------------------------------------
            * No other exit key is allowed: loop.
            *-----------------------------------------------------

        end

    end

    *-------------------------------------------------------------
    * Release data inside the static array.
    *-------------------------------------------------------------

    docArray := {}

    return NIL

*-----------------------------------------------------------------
static function  docDoEditing()

    docArray[DOC_WRAP] := .T.

    docArray[DOC_TEXT] :=;
        memoedit(;
            docArray[DOC_TEXT],;
            DOC_TOP+2,;
            DOC_LEFT+1,;
            DOC_BOTTOM-2,;
            DOC_RIGHT-1,;
            .t.,;
            "docUDF",;
            DOC_WIDTH,;
            DOC_TABSIZE,;
            docArray[DOC_ROW],;
            docArray[DOC_COL],;
            docArray[DOC_RELROW],;
            docArray[DOC_RELCOL];
        )

    return NIL

*-----------------------------------------------------------------
function docUDF(nMode, nLine, nCol)
*
* This is the UDF function for memoedit().
* The call to docUDF() is made with macro interpretation at
* runtime inside memoedit(). This is the reason why it can't
* be static.
*

    local nKey

    docArray[DOC_LASTKEY]        := nKey := LastKey()
    docArray[DOC_ROW]            := nLine
    docArray[DOC_COL]            := nCol
    docArray[DOC_RELROW]         := Row() - DOC_TOP
    docArray[DOC_RELCOL]         := Col() - DOC_LEFT

    if  nMode == MEMOEDIT_INIT

        if  docArray[DOC_WRAP]

            *-----------------------------------------------------
            * Turn off word wrap.
            *-----------------------------------------------------

            docArray[DOC_WRAP] := .f.

            return (MEMOEDIT_TOGGLEWRAP)                // RETURN

        end

        setcursor( if(docArray[DOC_INS], SETCURSOR_INSERT, SETCURSOR_NORMAL) )

    elseif nMode == MEMOEDIT_IDLE

        *---------------------------------------------------------
        * Show information line.
        *---------------------------------------------------------

        docInfoLine()

    else

        *---------------------------------------------------------
        * Keystroke exception.
        *---------------------------------------------------------

        if  nMode == MEMOEDIT_UNKEYX

            docArray[DOC_CHANGED] := .t.

        end

        do case
        case (nKey == K_F1)

            nbHelp( "<THE TEXT EDITOR DOC()>", , chr(K_ENTER) )

        case;
            nKey == K_F10                               .or.;
            nKey == K_ALT_M

            *-----------------------------------------------------
            * Call the menu.
            *-----------------------------------------------------

            return (K_CTRL_W)                           // RETURN

        case nKey == K_INS

            *-----------------------------------------------------
            * Insert/Overwrite.
            *-----------------------------------------------------

            docArray[DOC_INS] := !Set(_SET_INSERT)

            setcursor( if(docArray[DOC_INS], SETCURSOR_INSERT, SETCURSOR_NORMAL) )

            docInfoLine()

            return (nKey)

        case (nKey == K_ESC)

            *-----------------------------------------------------
            * Esc.
            *-----------------------------------------------------

            return (K_CTRL_W)                           // RETURN

        otherwise

            *-----------------------------------------------------
            * No other exit key is taken into consideration.
            *-----------------------------------------------------

        end

    end

    return 0

*-----------------------------------------------------------------
static function docInfoLine()

    local nRow          := row()
    local nCol          := col()
    local nOldCursor    := setcursor( SETCURSOR_NONE )
    local cInfoLine

    cInfoLine := ""
    cInfoLine += DOC_FUNCTION_NAME
    cInfoLine += space(2)
    cInfoLine += alltrim( docArray[DOC_NAME] )
    cInfoLine += space(2)
    cInfoLine += padl( alltrim( str( docArray[DOC_ROW] ) ), 6 )
    cInfoLine += ":"
    cInfoLine += padr( alltrim( str( docArray[DOC_COL]+1 ) ), 4 )
    cInfoLine += space(2)
    cInfoLine += iif( docArray[DOC_INS], "INS", "OVR" )
    cInfoLine += space(5)
    cInfoLine += "(No mouse, sorry :-)"

    say(;
        DOC_TOP+1, DOC_LEFT+1,;
        padr( cInfoLine, DOC_RIGHT-DOC_LEFT-1 ),;
        ,;
        COLOR_HEAD;
    )

    *-------------------------------------------------------------
    * Restore old data.
    *-------------------------------------------------------------

    setpos(nRow, nCol)

    setCursor( nOldCursor)

    return NIL

*-----------------------------------------------------------------
static function docNew()

    local lOkDoIt := .F.

    if  docArray[DOC_CHANGED]                           .and.;
        alertBox(;
                DOC_PROMPT_ABANDON_CHANGES,;
                { _MENU_NO, _MENU_YES };
        ) == 2;
        .or.;
        !docArray[DOC_CHANGED]

        *-------------------------------------------------------------
        * New file.
        *-------------------------------------------------------------

        docArray[DOC_OLD]       := NIL
        docArray[DOC_NAME]      := _UNTITLED + "." + _EXTENTION_TEXT
        docArray[DOC_TEXT]      := ""
        docArray[DOC_CHANGED]   := .F.
        docArray[DOC_ROW]       := 1
        docArray[DOC_COL]       := 0
        docArray[DOC_RELROW]    := 0
        docArray[DOC_RELCOL]    := 0

    end

    return NIL

*-----------------------------------------------------------------
static function docOpen()

    local aoGet                 := {}
    local aButtons              := {}
    local aTab                  := TAB_DEFAULT

    local bOldErrorHandler
    local cOldScreen
    local cOldColor           := setcolor( COLOR_BASE )
    local nOldCursor          := setcursor( SETCURSOR_NORMAL )
    local bOld_F1               :=;
        setkey( K_F1, {|| nbHelp( "<DOC - OPEN>", , chr(K_ENTER) )} )
    local bOld_F2             := setkey( K_F2, NIL )
    local nOldRow             := row()
    local nOldCol             := col()
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth

    local lGoOn               := .F.
    local cFileName           :=;
        padr( "*." + _EXTENTION_TEXT, _MAX_STRING_LEN )

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        if  docArray[DOC_CHANGED]                       .and.;
            alertBox(;
                DOC_PROMPT_ABANDON_CHANGES,;
                { _MENU_NO, _MENU_YES };
            ) == 2;
            .or.;
            !docArray[DOC_CHANGED]

            *-----------------------------------------------------
            * Create a kind of window.
            *-----------------------------------------------------

            nBottom             := maxrow()
            nTop                := nBottom - 6
            nLeft               := 0
            nRight              := maxcol()
            nWidth              := nRight - nLeft +1
            
            aTab[TAB_LEFT]      := nLeft
            aTab[TAB_RIGHT]     := nRight
            aTab[TAB_TAB_ARRAY] := { 16, 4 }

            cOldScreen          :=; 
                mouseScrSave( nTop, nLeft, nBottom, nRight )
            
            setcolor( COLOR_BODY )
        
            scroll( nTop, nLeft, nBottom, nRight )
            
            dispBoxShadow(;
                nTop, nLeft, nBottom, nRight,;
                1,;
                dispBoxColor( 1 ),;
                dispBoxColor( 2 );
            )

            say(;
                nTop+1, nLeft+1,;
                padc(;
                    DOC_DIALOG_BOX_TOP_OPEN,;
                    nWidth-2;
                    ),;
                ,;
                COLOR_HEAD;
            )

            *-----------------------------------------------------
            * Do editing.
            *-----------------------------------------------------
            
            while .t.                                   // FOREVER

                say( nTop+3, nLeft+1, DOC_PROMPT_OPEN_FILE_DOC )
                tab( aTab )
                get(;
                    @aoGet, row(), col(),;
                    { |x| iif( pcount() > 0, cFileName := x, cFileName ) },;
                    picChrMax( col(), nRight-1 ),;
                    ,;
                    { ||; 
                        setkey( K_F2, {|| gvFileDir( @cFileName ) } ),; 
                        .T.;
                    },;
                    { ||; 
                        gvFileExist( @cFileName, _EXTENTION_TEXT ); 
                    };
                )
                
                button( @aButtons, row()+2, nLeft+1,;
                    _BUTTON_ESC_CANCEL, , {||keyboard(chr(K_ESC))} )
                button( @aButtons, row(), col()+1,;
                    _BUTTON_F1_HELP, , {|| eval(setkey(K_F1))} )
                button( @aButtons, row(), col()+1,;
                    _BUTTON_F2_LIST, , {|| iif( valtype(setkey(K_F2)) == "B", eval(setkey(K_F2)), NIL ) } )
                button( @aButtons, row(), col()+1,;
                    _BUTTON_PGDN_CONFIRM, , {||keyboard(chr(K_PGDN))} )

                *-------------------------------------------------
                * Read.
                *-------------------------------------------------

                read( aoGet, , aButtons )
                aoGet   := {}
                aButtons := {}

                *-------------------------------------------------
                * Check what was read.
                *-------------------------------------------------

                do case
                case lastkey() = K_ESC  // exit

                    *---------------------------------------------
                    * [Esc] means leave.
                    *---------------------------------------------

                    lGoOn := .F.

                    exit                                // EXIT

                case lastkey() = K_PGDN
                    
                    *---------------------------------------------
                    * [PgDn] means confirm.
                    * Before, check for valid data.
                    *---------------------------------------------

                    if  gvFileExist( @cFileName, _EXTENTION_TEXT )

                        lGoOn := .T.

                        exit                            // EXIT

                    else

                        *-----------------------------------------
                        * The data is not valid: loop.
                        *-----------------------------------------

                    end

                otherwise

                    *---------------------------------------------
                    * Exit with different keys is not allowed.
                    * Loop again.
                    *---------------------------------------------

                end

            end

            *-----------------------------------------------------
            * Delete window.
            *-----------------------------------------------------

            mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

            *-----------------------------------------------------
            * Proceed if so was choosen.
            *-----------------------------------------------------

            if lGoOn

                docArray[DOC_NEW]       := .F.

                docArray[DOC_OLD]       := upper(alltrim( cFileName ))

                docArray[DOC_NAME]      := upper(alltrim( cFileName ))

                docArray[DOC_TEXT]      := MemoRead(alltrim(cFileName))

                docArray[DOC_CHANGED]   := .F.

                docArray[DOC_ROW]       := 1

                docArray[DOC_COL]       := 0

                docArray[DOC_RELROW]    := 0

                docArray[DOC_RELCOL]    := 0

            end

        end

    end //sequence

    *-------------------------------------------------------------
    * Restore old data.
    *-------------------------------------------------------------

    setcursor( nOldCursor )
    setcolor( cOldColor )
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )

    return NIL

*-----------------------------------------------------------------
static function docExit()

    local lRet  := .T.

    if  docArray[DOC_CHANGED]

        if  alertBox(;
                DOC_PROMPT_ABANDON_CHANGES,;
                { _MENU_NO, _MENU_YES };
            ) == 2

            lRet := .T.

        else

            lRet := .f.

        end

    else

        if  alertBox(;
                DOC_PROMPT_EXIT,;
                { _MENU_YES, _MENU_NO };
            ) == 1

            lRet := .T.

        else

            lRet := .f.

        end

    end

    docArray[DOC_EXIT] := lRet

    return (lRet)

*-----------------------------------------------------------------
static function docSave()

    local lRet

    lRet := .F.

    WaitFor( DOC_WRITING + docArray[DOC_NAME] )

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        do case
        case docArray[DOC_NEW]

            *-----------------------------------------------------
            * New file
            *-----------------------------------------------------

            lRet := docSaveAs()   // recursive call

            break                                       // BREAK

        case docArray[DOC_NAME] <> docArray[DOC_OLD]

            *-----------------------------------------------------
            * These names are allready upper and alltrim.
            *-----------------------------------------------------

            if file( docArray[DOC_NAME] )

                *-------------------------------------------------
                * Overwrite?
                *-------------------------------------------------

                if  alertBox(;
                        docArray[DOC_NAME] + NL(1) +;
                        _ERROR_PROMPT_ALREADY_EXIST_OVERWRITE,;
                        { _MENU_NO, _MENU_YES };
                    ) == 1

                    *---------------------------------------------
                    * Do not overwrite and do not save.
                    *---------------------------------------------

                    lRet := .F.

                    break                               // BREAK

                else

                    *---------------------------------------------
                    * Ok, save.
                    *---------------------------------------------

                end

            else

                *-------------------------------------------------
                * Ok, go on.
                *-------------------------------------------------

            end

        end

        *---------------------------------------------------------
        * If here, go on.
        *---------------------------------------------------------

        *---------------------------------------------------------
        * Update the old name.
        *---------------------------------------------------------

        docArray[DOC_OLD] := docArray[DOC_NAME]

        *---------------------------------------------------------
        * Try to save the file.
        *---------------------------------------------------------

        if  memoWrit( docArray[DOC_NAME], docArray[DOC_TEXT] )

            waitFor( DOC_WRITE_OK )

            docArray[DOC_CHANGED] := .F.

            lRet := .T.

        else

            alertBox( DOC_WRITE_ERROR )

            lRet := .F.

        end

    end //sequence

    *-------------------------------------------------------------
    * Close a wait box if it is present.
    *-------------------------------------------------------------

    waitFor()

    *-------------------------------------------------------------
    * Return the saving result.
    *-------------------------------------------------------------

    return lRet

*-----------------------------------------------------------------
static function docSaveAs()


    local aoGet                 := {}
    local aButtons              := {}
    local aTab                  := TAB_DEFAULT

    local cOldScreen
    local cOldColor             := setcolor( COLOR_BASE )
    local bOld_F1               :=;
        setkey( K_F1, {|| nbHelp( "<DOC - SAVE AS>", , chr(K_ENTER) )} )
    local bOld_F2               := setkey( K_F2, NIL )
    local nOldRow               := row()
    local nOldCol               := col()
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth

    local cNewName              :=;
         padr( docArray[DOC_NAME], _MAX_STRING_LEN )

    local lRet                  := .F.

    *-------------------------------------------------------------
    * Start a sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Create a kind of window right aligned.
        *---------------------------------------------------------

        nBottom     := maxrow()
        nTop        := nBottom - 6
        nRight      := maxcol()
        nLeft       := 0
        nWidth      := nRight - nLeft +1
        
        cOldScreen  := mouseScrSave( nTop, nLeft, nBottom, nRight )

        setcolor( COLOR_BODY )

        scroll( nTop, nLeft, nBottom, nRight )

        dispBoxShadow(;
            nTop, nLeft, nBottom, nRight,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )

        say(;
            nTop+1, nLeft+1,;
            padc(;
                DOC_DIALOG_BOX_TOP_SAVEAS,;
                nWidth-2;
                ),;
            ,;
            COLOR_HEAD;
        )

        *---------------------------------------------------------
        * Do editing.
        *---------------------------------------------------------

        while .t.                                   // FOREVER

            say(;
                nTop+3, nLeft+1,;
                DOC_NEW_NAME;
            )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cNewName := x, cNewName ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { || trueSetkey( K_F2, {|| gvFileDir( @cNewName ) } ) },;
                { || gvFileExtention( @cNewName, _EXTENTION_TEXT ) };
            )

            button( @aButtons, row()+2, nLeft+1,;
                _BUTTON_ESC_CANCEL, , {||keyboard(chr(K_ESC))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F1_HELP, , {|| eval(setkey(K_F1))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F2_LIST, , {|| iif( valtype(setkey(K_F2)) == "B", eval(setkey(K_F2)), NIL ) } )
            button( @aButtons, row(), col()+1,;
                _BUTTON_PGDN_CONFIRM, , {||keyboard(chr(K_PGDN))} )

            *-----------------------------------------------------
            * Read.
            *-----------------------------------------------------

            read( aoGet, , aButtons )
            aoGet   := {}
            aButtons := {}

            *-----------------------------------------------------
            * Check what was read.
            *-----------------------------------------------------
                                        
            do case
            case lastkey() = K_ESC

                *-------------------------------------------------
                * [Esc] means leave.
                *-------------------------------------------------

                lRet := .F.

                exit                                    // EXIT

            case lastkey() = K_PGDN
                                         
                *-------------------------------------------------
                * [PgDn] means confirm.
                * Before, check for valid data.
                *-------------------------------------------------

                cNewName := strAddExtention( cNewName, _EXTENTION_TEXT )

                docArray[DOC_NAME] := upper( alltrim(cNewName) )

                docArray[DOC_NEW] := .F.

                lRet := docSave()

                exit                                    // EXIT

            otherwise

                *-------------------------------------------------
                * Exit with different keys is not allowed.
                * Loop again.
                *-------------------------------------------------

            end

        end

        *---------------------------------------------------------
        * The editing is terminated: restore screen window.
        *---------------------------------------------------------

        mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

    end //sequence

    *-------------------------------------------------------------
    * Restore old values.
    *-------------------------------------------------------------

    setcolor( cOldColor )
    setpos( nOldRow, nOldCol )
    setkey( K_F2, bOld_F2 )
    setkey( K_F1, bOld_F1 )

    return lRet

*-----------------------------------------------------------------&&&&
static function docMenu()
*
    local aMenu               := {}
    local cOldScreen          := mouseScrSave()
    local cOldColor           := setcolor()
    local nOldCursor          := setcursor( SETCURSOR_NORMAL )
    local lOldSetWrap         := set(_SET_WRAP, .T.)
    local nOldSetMessage      := set(_SET_MESSAGE, maxrow()-1)
    local lOldSetCenter       := set(_SET_MCENTER, .T.)
    local bOld_F1             :=;
        setkey( K_F1, {|| nbHelp( "<THE MENU SYSTEM>", , chr(K_ENTER) )} )
    local bOld_F2             := setkey( K_F2, NIL )
    local bOld_LEFT           := setkey( K_LEFT, NIL )
    local bOld_RIGHT          := setkey( K_RIGHT, NIL )
    local nOldRow             := row()
    local nOldCol             := col()

    local nMnuChoice := 1

    begin sequence

        while .T.
            statusLine()
            setcolor( COLOR_MENU )
            scroll( 01, 00, 01, maxcol() )
            scroll( maxrow()-1, 00, maxrow()-1, maxcol() )
            say( maxrow(), 00, padc( _KEY_MENU_H, maxcol()+1 ) )
            // The horizontal menu starts.
            menuPrompt( @aMenu, 01, 01,;
                MNU_MENU_DOC_CHOICE,;
                {|| say( maxrow()-1, 0,;
                    padc( MNU_MENU_DOC_MSG, maxcol()+1 ) ) };
                )
            menuPrompt( @aMenu, 01, col()+1,;
                MNU_MENU_FILE_CHOICE,;
                {|| say( maxrow()-1, 0,;
                    padc( MNU_MENU_FILE_MSG, maxcol()+1 ) ) };
                )
            menuPrompt( @aMenu, row(), col()+1,;
                MNU_MENU_EDIT_CHOICE,;
                {|| say( maxrow()-1, 0,;
                    padc( MNU_MENU_EDIT_MSG, maxcol()+1 ) ) };
                )
            menuPrompt( @aMenu, row(), col()+1,;
                MNU_MENU_MACRO_CHOICE,;
                {|| say( maxrow()-1, 0,;
                    padc( MNU_MENU_MACRO_MSG, maxcol()+1 ) ) };
                )
            menuPrompt( @aMenu, row(), col()+1,;
                MNU_MENU_INFO_CHOICE,;
                {|| say( maxrow()-1, 0,;
                    padc( MNU_MENU_INFO_MSG, maxcol()+1 ) ) };
                )

            nMnuChoice := menuTo( @aMenu, nMnuChoice )

            // [Left]/[Right] key redirection.
            setkey( K_LEFT, { || mnuLeft() } )
            setkey( K_RIGHT, { || mnuRight() } )

            setcolor( COLOR_BASE )

            // Menu analilsys.
            do case
            case nMnuChoice == 0 // ESC
                break
            case nMnuChoice == 1
                if mnuvDoc( 0, 0+len(MNU_MENU_DOC_CHOICE)+1 )
                    break
                end
            case nMnuChoice == 2
                mnuvFile( 4, 4+len(MNU_MENU_DOC_CHOICE)+1 )
            case nMnuChoice == 3
                mnuvEdit( 9, 9+len(MNU_MENU_DOC_CHOICE)+1 )
            case nMnuChoice == 4
                mnuvMacro( 14, 14+len(MNU_MENU_DOC_CHOICE)+1 )
            case nMnuChoice == 5
                mnuvInfo( 20, 20+len(MNU_MENU_DOC_CHOICE)+1 )
            end

            // Arrow keys are released.
            setkey( K_LEFT, NIL )
            setkey( K_RIGHT, NIL )

        end

    end //sequence

    // Restore.
    setcolor( cOldColor )
    setcursor( nOldCursor )
    mouseScrRestore( ,,,, cOldScreen )
    set(_SET_WRAP, lOldSetWrap )
    set(_SET_MESSAGE, nOldSetMessage )
    set(_SET_MCENTER, lOldSetCenter )
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )
    setkey( K_LEFT, bOld_LEFT )
    setkey( K_RIGHT, bOld_RIGHT )

    return NIL

*-----------------------------------------------------------------
static function docPrintCondition( cText )
*
* It asks for conditions FOR and WHILE before printing with the
* RPT() function.
*
* <cText>      the text to print.
*

    local aoGet                 := {}
    local aButtons              := {}
    local aTab                  := TAB_DEFAULT

    local bOldErrorHandler
    local cOldScreen
    local cOldColor             := setcolor()
    local nOldCursor            := setcursor( SETCURSOR_NORMAL )
    local bOld_F1               :=;
        setkey( K_F1, {|| nbHelp( "<DOC - PRINT WITH RPT() STD>", , chr(K_ENTER) )} )

    local bOld_F2               := setkey( K_F2, NIL )
    local nOldRow               := row()
    local nOldCol               := col()
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth

    local lGoOn                 := .T.

    local cWhileCondition       := padr( ".T.", _MAX_STRING_LEN )
    local cForCondition         := padr( ".T.", _MAX_STRING_LEN )

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Parameters check.
        *---------------------------------------------------------

        if valtype( cText ) <> "C"

            *-----------------------------------------------------
            * There is no text to print.
            *-----------------------------------------------------

            break                                       // BREAK

        end

        *---------------------------------------------------------
        * An active Alias must be present.
        *---------------------------------------------------------

        if alias() == "";

            *-----------------------------------------------------
            * Null string means no active alias.
            *-----------------------------------------------------

            alertBox( _ERROR_NO_ALIAS )

            break                                       // BREAK

        end

        *---------------------------------------------------------
        * Create a window.
        *---------------------------------------------------------

        nBottom             := maxrow()
        nTop                := nBottom - 6
        nLeft               := 0
        nRight              := maxcol()
        nWidth              := nRight - nLeft +1

        aTab[TAB_LEFT]      := nLeft
        aTab[TAB_RIGHT]     := nRight
        aTab[TAB_TAB_ARRAY] := { 16, 8 }

        cOldScreen  :=;
            mouseScrSave( nTop, nLeft, nBottom, nRight )

        setcolor( COLOR_BODY )

        scroll( nTop, nLeft, nBottom, nRight )

        dispBoxShadow(;
            nTop, nLeft, nBottom, nRight,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )

        say(;
            nTop+1, nLeft+1,;
            padc(;
                DOC_DIALOG_BOX_TOP_PRINT,;
                nWidth-2;
                ),;
            ,;
            COLOR_HEAD;
        )

        *---------------------------------------------------------
        * Start editing loop.
        *---------------------------------------------------------

        while .t.                                       // FOREVER

            say( nTop+3, nLeft+1, DOC_PROMPT_WHILE_EXPRESSION )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cWhileCondition := x, cWhileCondition ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { ||; 
                    setkey( K_F2, NIL ),;
                    .T.; 
                },;
                { || gvDefault( @cWhileCondition, ".T." ) },;
            )

            say( row()+1, nLeft+1, DOC_PROMPT_FOR_EXPRESSION )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cForCondition := x, cForCondition ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { ||; 
                    setkey( K_F2, NIL ),;
                    .T.; 
                },;
                { || gvDefault( @cForCondition, ".T." ) },;
            )
            
            button( @aButtons, row()+2, nLeft+1,;
                _BUTTON_ESC_CANCEL, , {||keyboard(chr(K_ESC))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F1_HELP, , {|| eval(setkey(K_F1))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F2_LIST, , {|| iif( valtype(setkey(K_F2)) == "B", eval(setkey(K_F2)), NIL ) } )
            button( @aButtons, row(), col()+1,;
                _BUTTON_PGDN_CONFIRM, , {||keyboard(chr(K_PGDN))} )

            *-----------------------------------------------------
            * Read.
            *-----------------------------------------------------

            read( aoGet, , aButtons )
            aoGet   := {}
            aButtons := {}

            *-----------------------------------------------------
            * Check what was read.
            *-----------------------------------------------------

            do case
            case lastkey() = K_ESC

                *-------------------------------------------------
                * [Esc] means leave.
                *-------------------------------------------------

                lGoOn := .F.

                exit                                    // EXIT

            case lastkey() = K_PGDN

                *-------------------------------------------------
                * [PgDn] means confirm.
                * Check for valid data.
                *-------------------------------------------------

                if !empty( cWhileCondition )            .and.;
                   !empty( cForCondition )

                    lGoOn := .T.

                    exit                                // EXIT

                else

                    *---------------------------------------------
                    * Data is not correct: Loop.
                    *---------------------------------------------

                end

            otherwise

                *-------------------------------------------------
                * Exit with different keys is not allowed.
                * Loop again.
                *-------------------------------------------------

            end

        end

        *---------------------------------------------------------
        * Delete window.
        *---------------------------------------------------------

        mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

        *---------------------------------------------------------
        * Proceed if so selected.
        *---------------------------------------------------------

        if lGoOn

            rptMany(;
                cText,;
                { || &(alltrim(cWhileCondition)) },;
                { || &(alltrim(cForCondition)) }, NIL;
            )

        end

        waitFileEval( .T. )

    end //sequence

    *-------------------------------------------------------------
    * Restore old data and return.
    *-------------------------------------------------------------

    setcursor( nOldCursor )
    setcolor( cOldColor )
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )

    return NIL

#endif  //RUNTIME

#ifndef RUNTIME
*=================================================================
* ASSIST - nB MENU
*=================================================================
function Assist()
*
* This is the nB menu.
* Note that [F10] is inhibited as this is the key used to call
* the menu from the dot line.
*

    local aMenu            := {}
    local cOldScreen        := mouseScrSave()
    local cOldColor         := setcolor()
    local nOldCursor        := setcursor( SETCURSOR_NORMAL )
    local bOld_F1           :=;
        setkey( K_F1, {|| nbHelp( "<THE MENU SYSTEM>", , chr(K_ENTER) )} )
    local bOld_F2           := setkey( K_F2, NIL )
    local bOld_F3           :=;
        setkey( K_F3, { || Text( dbiAllStatus() ) } )
    local bOld_F5           :=;
        setkey( K_F5, { || setAlternate()} )
    local bOld_F10          := setkey( K_F10, NIL )
    local bOld_LEFT         := setkey( K_LEFT, NIL )
    local bOld_RIGHT        := setkey( K_RIGHT, NIL )
    local nOldRow           := row()
    local nOldCol           := col()

    local nMnuChoice := 1

    begin sequence

        while .T.
            statusLine()
            setcolor( COLOR_MENU )
            scroll( 01, 00, 01, maxcol() )
            scroll( maxrow()-1, 00, maxrow()-1, maxcol() )
            say( maxrow(), 00, padc( _KEY_MENU_H, maxcol()+1 ) )

            // The horizontal menu starts
            menuPrompt( @aMenu, 01, 01,;
                MNU_MENU_FILE_CHOICE,;
                {|| say( maxrow()-1, 0,;
                    padc( MNU_MENU_FILE_MSG, maxcol()+1 ) ) };
                )
            menuPrompt( @aMenu, row(), col()+1,;
                MNU_MENU_EDIT_CHOICE,;
                {|| say( maxrow()-1, 0,;
                    padc( MNU_MENU_EDIT_MSG, maxcol()+1 ) ) };
                )
            menuPrompt( @aMenu, row(), col()+1,;
                MNU_MENU_REPORT_CHOICE,;
                {|| say( maxrow()-1, 0,;
                    padc( MNU_MENU_REPORT_MSG, maxcol()+1 ) ) };
                )
            menuPrompt( @aMenu, row(), col()+1,;
                MNU_MENU_HTF_CHOICE,;
                {|| say( maxrow()-1, 0,;
                    padc( MNU_MENU_HTF_MSG, maxcol()+1 ) ) };
                )
            menuPrompt( @aMenu, row(), col()+1,;
                MNU_MENU_MACRO_CHOICE,;
                {|| say( maxrow()-1, 0,;
                    padc( MNU_MENU_MACRO_MSG, maxcol()+1 ) ) };
                )
            menuPrompt( @aMenu, row(), col()+1,;
                MNU_MENU_INFO_CHOICE,;
                {|| say( maxrow()-1, 0,;
                    padc( MNU_MENU_INFO_MSG, maxcol()+1 ) ) };
                )

            nMnuChoice := menuTo( @aMenu, nMnuChoice )

            // [Left]/[Right] key redirection
            setkey( K_LEFT, { || mnuLeft() } )
            setkey( K_RIGHT, { || mnuRight() } )

            setcolor( COLOR_BASE )

            // analisi del menu
            do case
            case nMnuChoice == 0 // ESC
                break
            case nMnuChoice == 1
                mnuvFile( 0, 0+len(MNU_MENU_FILE_CHOICE)+1 )
            case nMnuChoice == 2
                mnuvEdit( 5, 5+len(MNU_MENU_FILE_CHOICE)+1 )
            case nMnuChoice == 3
                mnuvReport( 10, 10+len(MNU_MENU_FILE_CHOICE)+1 )
            case nMnuChoice == 4
                mnuvHtf( 17, 17+len(MNU_MENU_FILE_CHOICE)+1 )
            case nMnuChoice == 5
                mnuvMacro( 21, 21+len(MNU_MENU_FILE_CHOICE)+1 )
            case nMnuChoice == 6
                mnuvInfo( 27, 27+len(MNU_MENU_FILE_CHOICE)+1 )
            end

            // arrow keys are released
            setkey( K_LEFT, NIL )
            setkey( K_RIGHT, NIL )

        end

    end //sequence

    // restore
    setcolor( cOldColor )
    setcursor( nOldCursor )
    mouseScrRestore( ,,,, cOldScreen )
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )
    setkey( K_F3, bOld_F3 )
    setkey( K_F5, bOld_F5 )
    setkey( K_F10, bOld_F10 )
    setkey( K_LEFT, bOld_LEFT )
    setkey( K_RIGHT, bOld_RIGHT )

    return NIL

#endif  //RUNTIME

#ifndef RUNTIME
*=================================================================
* HTF - HELP TEXT FILE
*=================================================================

#define HTF_INDEX_START         "##"
#define HTF_INDEX_END           "##"
#define HTF_POINTER_START       "<"
#define HTF_POINTER_END         ">"

#define HTF_ERROR_FILE_TYPE;
    "This is not a Help Text File."

#define HTF_DIALOG_BOX_TOP_GENERATE;
    "Generate a file for the use with htf()."

#define HTF_HTML_DIALOG_BOX_TOP_GENERATE;
    "Generate a HTML from a HTF text source."

#define HTF_DIALOG_BOX_TOP_OPEN;
    "Help Text File open."

#define HTF_PROMPT_HTF_FILE_OPEN;
    "Insert the Help Text File to open:"

#define HTF_PROMPT_SOURCE_FILE;
    "Source text filename:"
#define HTF_PROMPT_DESTINATION_FILE;
    "Destination filename:"
#define HTF_PROMPT_INDEX_START;
    "INDEX start code:"
#define HTF_PROMPT_INDEX_END;
    "INDEX end code:"
#define HTF_PROMPT_POINTER_START;
    "POINTER start code:"
#define HTF_PROMPT_POINTER_END;
    "POINTER end code:"

#define HTF_PROMPT_HTML_TITLE;
    "The HTML title:"

#define HTF_WAIT_APPENDING;
    "Loading "
#define HTF_WAIT_SEARCHING_INDEXES;
    "Searching for indexes..."
#define HTF_WAIT_SEARCHING_POINTERS;
    "Searching for pointers..."

#define HTF_HELP_OPEN;
    "OPEN AND BROWSE A HELP TEXT FILE" +;
    NL(3) +;
    "Insert the Help Text File name to be browsed."

#define HTF_HELP_GENERATE;
    "HELP TEXT FILE GENERATION." +;
    NL(3) +;
    "It generates a file readable from the Help Text File browser." +;
    NL(2) +;
    "The source is an Ascii file where three kind of information " +;
    "are available: Normal text, Indexes and pointers." +;
    NL(2) +;
    "Indexes and Pointers are word or phrases delimited with " +;
    "user defined delimiters; indexes are placed inside the " +;
    "text to indicate an argument, pointers are placed inside " +;
    "the text to indicate a reference to indexes." +;
    NL(2) +;
    "Only one index per line is allowed, only one pointer per " +;
    "line is allowed." +;
    NL(2) +;
    "The Delimiters used do identify indexes and pointers are " +;
    "user defined; the start identifyer symbol can be equal " +;
    "to the end identifyer symbol. The symbols used for " +;
    "indexes cannot be used for the pointers."

*-----------------------------------------------------------------
static function htfOpen( cHtf )
*
* htfOpen( [<cHtf>] ) --> NIL
*
* <cHtf>       is the name of the file to search for; it can
*              contain wildcards.
*
* It asks for the name of the HTF file to browse if <cHtf> is not
* found or it contains wildcards.
*

    local aoGet             := {}
    local aButtons          := {}
    local aTab              := TAB_DEFAULT

    local bOldErrorHandler
    local nOldSelect        := select()
    local cOldScreen
    local cOldColor         := setcolor()
    local nOldCursor        := setcursor( SETCURSOR_NORMAL )
    local bOld_F1           :=;
        setkey( K_F1, {|| nbHelp( "<HTF - OPEN HELP TEXT FILE>", , chr(K_ENTER) ) } )
    local bOld_F2           := setkey( K_F2, NIL )
    local nOldRow           := row()
    local nOldCol           := col()
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth

    local lGoOn             := .T.

    local cSearchPath       :=;
        "\" + curdir() + ";" + set( _SET_PATH ) + ";" + getenv( "PATH" )

    *-------------------------------------------------------------
    * The first problem is to locate the HTF file.
    *-------------------------------------------------------------

    if valtype( cHtf ) == "C"

        *---------------------------------------------------------
        * Data type is correct: find it if you can.
        *---------------------------------------------------------

        cHtf := strFileFind( cHtf, cHtf + ";" + cSearchPath )

    else

        cHtf := "*." + _EXTENTION_HTF

    end

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        do case
        case "*" $ cHtf .or. "?" $ cHtf

            *-----------------------------------------------------
            * Wildcards means continue.
            *-----------------------------------------------------

        case file( alltrim( cHtf ) )

            *-----------------------------------------------------
            * The file is present:
            * just open it.
            * Open the selected htf file.
            *-----------------------------------------------------

            dbusearea(;
                .T.,;
                _DEFAULT_RDD,;
                alltrim(cHtf), "HlpTxtFile",;
                .T., .T.;
            )

            *-----------------------------------------------------
            * Run htf.
            *-----------------------------------------------------

            htf( NIL, alltrim(cHtf) )

            *-----------------------------------------------------
            * Close htf file.
            *-----------------------------------------------------

            HlpTxtFile->(dbclosearea())

            *-----------------------------------------------------
            * Exit.
            *-----------------------------------------------------

            break                                       // BREAK

        otherwise

            *-----------------------------------------------------
            * Announce that there is no file.
            *-----------------------------------------------------

            alertBox( alltrim( cHtf ) + NL(1) + _ERROR_NO_FILE_HERE )

        end

        *---------------------------------------------------------
        * Ask for the name to load.
        *---------------------------------------------------------

        *---------------------------------------------------------
        * Prepare data before editing.
        *---------------------------------------------------------

        cHtf := padr( cHtf, _MAX_STRING_LEN )

        *---------------------------------------------------------
        * Create a window.
        *---------------------------------------------------------

        nBottom             := maxrow()
        nTop                := nBottom - 6
        nLeft               := 0
        nRight              := maxcol()
        nWidth              := nRight - nLeft +1

        aTab[TAB_LEFT]      := nLeft
        aTab[TAB_RIGHT]     := nRight
        aTab[TAB_TAB_ARRAY] := { 16, 8 }

        cOldScreen          :=; 
            mouseScrSave( nTop, nLeft, nBottom, nRight )

        setcolor( COLOR_BODY )

        scroll( nTop, nLeft, nBottom, nRight )

        dispBoxShadow(;
            nTop, nLeft, nBottom, nRight,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )

        say(;
            nTop+1, nLeft+1,;
            padc( HTF_DIALOG_BOX_TOP_OPEN, nWidth-2 ),;
            ,;
            COLOR_HEAD;
        )

        *---------------------------------------------------------
        * Do editing.
        *---------------------------------------------------------

        while .t.                                   // FOREVER

            say(;
                nTop+3, nLeft+1,;
                HTF_PROMPT_HTF_FILE_OPEN;
            )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cHtf := x, cHtf ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { || trueSetkey( K_F2, {|| gvFileDir( @cHtf ) } ) },;
                { || gvFileExist( @cHtf, _EXTENTION_HTF ) };
            )

            button( @aButtons, row()+2, nLeft+1,;
                _BUTTON_ESC_CANCEL, , {||keyboard(chr(K_ESC))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F1_HELP, , {|| eval(setkey(K_F1))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F2_LIST, , {|| iif( valtype(setkey(K_F2)) == "B", eval(setkey(K_F2)), NIL ) } )
            button( @aButtons, row(), col()+1,;
                _BUTTON_PGDN_CONFIRM, , {||keyboard(chr(K_PGDN))} )

            *-----------------------------------------------------
            * Read.
            *-----------------------------------------------------

            read( aoGet, , aButtons )
            aoGet   := {}
            aButtons := {}

            *-----------------------------------------------------
            * Check what was read.
            *-----------------------------------------------------
                                        
            do case
            case lastkey() = K_ESC

                *-------------------------------------------------
                * [Esc] means leave.
                *-------------------------------------------------

                lGoOn := .F.

                exit                                    // EXIT

            case lastkey() = K_PGDN

                *-------------------------------------------------
                * [PgDn] means confirm.
                * Before, check for valid data.
                *-------------------------------------------------

                if gvFileExist( @cHtf, _EXTENTION_HTF )

                    *---------------------------------------------
                    * The file was found, continue.
                    *---------------------------------------------

                    lGoOn := .T.

                    exit                                // EXIT

                else

                    *---------------------------------------------
                    * The file wasn't found: loop.
                    *---------------------------------------------

                end

            otherwise

                *-------------------------------------------------
                * Exit with different keys is not allowed.
                * Loop again.
                *-------------------------------------------------

            end

        end

        *---------------------------------------------------------
        * The editing is terminated: delete window.
        *---------------------------------------------------------

        mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

        *---------------------------------------------------------
        * Restore Colors: it is necessary to not interfere with
        * htf().
        *---------------------------------------------------------

        setcolor( cOldColor )

        *---------------------------------------------------------
        * Proceed if it was so required.
        *---------------------------------------------------------

        if lGoOn

            *-----------------------------------------------------
            * Open the selected htf file.
            *-----------------------------------------------------

            dbusearea(;
                .T.,;
                _DEFAULT_RDD,;
                alltrim(cHtf), "HlpTxtFile",;
                .T., .T.;
            )

            *-----------------------------------------------------
            * Run htf()
            *-----------------------------------------------------

            htf( NIL, alltrim(cHtf) )

            *-----------------------------------------------------
            * Close htf file.
            *-----------------------------------------------------

            HlpTxtFile->(dbclosearea())

        end

    end //sequence

    *-------------------------------------------------------------
    * Restore old data.
    *-------------------------------------------------------------

    dbselectarea( nOldSelect )
    setcursor( nOldCursor )
    setcolor( cOldColor )
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )

    return NIL

*-----------------------------------------------------------------
static function htfGenDialog()
*
* It asks for parameters needed to run htfGenerate().
*

    local aoGet             := {}
    local aButtons          := {}
    local aTab              := TAB_DEFAULT

    local bOldErrorHandler
    local cOldScreen
    local cOldColor         := setcolor()
    local nOldCursor        := setcursor( SETCURSOR_NORMAL )
    local bOld_F1           :=;
        setkey( K_F1, {|| nbHelp( "<HTF - NEW HELP TEXT FILE>", , chr(K_ENTER) ) } )
    local bOld_F2           := setkey( K_F2, NIL )
    local nOldRow           := row()
    local nOldCol           := col()
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth

    local lGoOn             := .T.

    local cHtfSource        := padr( "*.*", _MAX_STRING_LEN )
    local cHtfDestination   := padr( "*." + _EXTENTION_HTF, _MAX_STRING_LEN )

    static cIndexStart
    static cIndexEnd
    static cPointerStart
    static cPointerEnd

    default(;
        @cIndexStart,;
        padr( HTF_INDEX_START, _MAX_STRING_LEN );
    )

    default(;
        @cIndexEnd,;
        padr( HTF_INDEX_END, _MAX_STRING_LEN );
    )

    default(;
        @cPointerStart,;
        padr( HTF_POINTER_START, _MAX_STRING_LEN );
    )

    default(;
        @cPointerEnd,;
        padr( HTF_POINTER_END, _MAX_STRING_LEN );
    )

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Create a window.
        *---------------------------------------------------------

        nBottom             := maxrow()
        nTop                := nBottom - 9
        nLeft               := 0
        nRight              := maxcol()
        nWidth              := nRight - nLeft +1

        aTab[TAB_LEFT]      := nLeft
        aTab[TAB_RIGHT]     := nRight
        aTab[TAB_TAB_ARRAY] := { 24, 16, 24, 16, 4 }

        cOldScreen          :=; 
            mouseScrSave( nTop, nLeft, nBottom, nRight )

        setcolor( COLOR_BODY )

        scroll( nTop, nLeft, nBottom, nRight )

        dispBoxShadow(;
            nTop, nLeft, nBottom, nRight,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )

        say(;
            nTop+1, nLeft+1,;
            padc( HTF_DIALOG_BOX_TOP_GENERATE, nWidth-2 ),;
            ,;
            COLOR_HEAD;
        )

        *---------------------------------------------------------
        * Do editing.
        *---------------------------------------------------------

        while .t.                                   // FOREVER

            say(;
                nTop+3, nLeft+1,;
                HTF_PROMPT_SOURCE_FILE;
            )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cHtfSource := x, cHtfSource ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { || trueSetkey( K_F2, {|| gvFileDir( @cHtfSource ) } ) },;
                { || gvFileExist( @cHtfSource, _EXTENTION_TEXT ) };
            )

            say(;
                row()+1, nLeft+1,;
                HTF_PROMPT_DESTINATION_FILE;
            )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cHtfDestination := x, cHtfDestination ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { || trueSetkey( K_F2, {|| gvFileDir( @cHtfDestination ) } ) },;
                { || gvFileExtention( @cHtfDestination, _EXTENTION_HTF ) };
            )

            say(;
                row()+1, nLeft+1,;
                HTF_PROMPT_INDEX_START;
            )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cIndexStart := x, cIndexStart ) },;
                "@s8@",;
                ,;
                { || trueSetkey( K_F2, NIL ) },;
                { || gvDefault( @cIndexStart, HTF_INDEX_START ) };
            )
            tab( aTab )
            say(;
                row(), col(),;
                HTF_PROMPT_INDEX_END;
            )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cIndexEnd := x, cIndexEnd ) },;
                "@s8@",;
                ,;
                { || trueSetkey( K_F2, NIL ) },;
                { || gvDefault( @cIndexEnd, HTF_INDEX_END ) };
            )

            say(;
                row()+1, nLeft+1,;
                HTF_PROMPT_POINTER_START;
            )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cPointerStart := x, cPointerStart ) },;
                "@s8@",;
                ,;
                { || trueSetkey( K_F2, NIL ) },;
                { || gvDefault( @cPointerStart, HTF_POINTER_START ) };
            )
            tab( aTab )
            say(;
                row(), col(),;
                HTF_PROMPT_POINTER_END;
            )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cPointerEnd := x, cPointerEnd ) },;
                "@s8@",;
                ,;
                { || trueSetkey( K_F2, NIL ) },;
                { || gvDefault( @cPointerEnd, HTF_POINTER_END ) };
            )

            button( @aButtons, row()+2, nLeft+1,;
                _BUTTON_ESC_CANCEL, , {||keyboard(chr(K_ESC))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F1_HELP, , {|| eval(setkey(K_F1))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F2_LIST, , {|| iif( valtype(setkey(K_F2)) == "B", eval(setkey(K_F2)), NIL ) } )
            button( @aButtons, row(), col()+1,;
                _BUTTON_PGDN_CONFIRM, , {||keyboard(chr(K_PGDN))} )

            *-----------------------------------------------------
            * Read.
            *-----------------------------------------------------

            read( aoGet, , aButtons )
            aoGet   := {}
            aButtons := {}

            *-----------------------------------------------------
            * Check what was read.
            *-----------------------------------------------------
                                        
            do case
            case lastkey() = K_ESC

                *-------------------------------------------------
                * [Esc] means leave.
                *-------------------------------------------------

                lGoOn := .F.

                exit

            case lastkey() = K_PGDN

                *-------------------------------------------------
                * [PgDn] means confirm.
                * Before, check for valid data.
                *-------------------------------------------------

                if  gvFileExist( @cHtfSource, _EXTENTION_HTF );
                    .and.;
                    !isWild( cHtfDestination );
                    .and.;
                    !empty( cIndexStart );
                    .and.;
                    !empty( cIndexEnd );
                    .and.;
                    !empty( cPointerStart );
                    .and.;
                    !empty( cPointerEnd )

                    lGoOn := .T.

                    exit                                // EXIT

                else

                    *---------------------------------------------
                    * Data is not valid: loop.
                    *---------------------------------------------

                end

            otherwise

                *-------------------------------------------------
                * Exit with different keys is not allowed.
                * Loop again.
                *-------------------------------------------------

            end

        end

        *---------------------------------------------------------
        * The editing is terminated: delete window.
        *---------------------------------------------------------

        mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

        *---------------------------------------------------------
        * Restore Colors.
        *---------------------------------------------------------

        setcolor( cOldColor )

        *---------------------------------------------------------
        * Proceed if it was so required.
        *---------------------------------------------------------

        if lGoOn

            htfGenerate(;
                alltrim(cHtfSource),;
                alltrim(cHtfDestination),;
                alltrim(cIndexStart),;
                alltrim(cIndexEnd),;
                alltrim(cPointerStart),;
                alltrim(cPointerEnd);
            )

        end

    end //sequence

    *-------------------------------------------------------------
    * Restore old data.
    *-------------------------------------------------------------

    setcursor( nOldCursor )
    setcolor( cOldColor )
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )

    return NIL

*-----------------------------------------------------------------
static function htf2HtmlDialog()
*
* It asks for parameters needed to run htf2Html().
*

    local aoGet             := {}
    local aButtons          := {}
    local aTab              := TAB_DEFAULT

    local bOldErrorHandler
    local cOldScreen
    local cOldColor         := setcolor()
    local nOldCursor        := setcursor( SETCURSOR_NORMAL )
    local bOld_F1           :=;
        setkey( K_F1, {|| nbHelp( "<HTF - NEW HELP TEXT FILE>", , chr(K_ENTER) ) } ) &&&&
    local bOld_F2           := setkey( K_F2, NIL )
    local nOldRow           := row()
    local nOldCol           := col()
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth

    local lGoOn             := .T.

    local cHtfSource        := padr( "*.*", _MAX_STRING_LEN )
    local cHtfDestination   := padr( "*." + _EXTENTION_HTM, _MAX_STRING_LEN )
    local cHtfTitle         := space( _MAX_STRING_LEN )

    static cIndexStart
    static cIndexEnd
    static cPointerStart
    static cPointerEnd

    default(;
        @cIndexStart,;
        padr( HTF_INDEX_START, _MAX_STRING_LEN );
    )

    default(;
        @cIndexEnd,;
        padr( HTF_INDEX_END, _MAX_STRING_LEN );
    )

    default(;
        @cPointerStart,;
        padr( HTF_POINTER_START, _MAX_STRING_LEN );
    )

    default(;
        @cPointerEnd,;
        padr( HTF_POINTER_END, _MAX_STRING_LEN );
    )

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Create a window.
        *---------------------------------------------------------

        nBottom             := maxrow()
        nTop                := nBottom - 10
        nLeft               := 0
        nRight              := maxcol()
        nWidth              := nRight - nLeft +1

        aTab[TAB_LEFT]      := nLeft
        aTab[TAB_RIGHT]     := nRight
        aTab[TAB_TAB_ARRAY] := { 24, 16, 24, 16, 4 }

        cOldScreen          :=; 
            mouseScrSave( nTop, nLeft, nBottom, nRight )

        setcolor( COLOR_BODY )

        scroll( nTop, nLeft, nBottom, nRight )

        dispBoxShadow(;
            nTop, nLeft, nBottom, nRight,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )

        say(;
            nTop+1, nLeft+1,;
            padc( HTF_HTML_DIALOG_BOX_TOP_GENERATE, nWidth-2 ),;
            ,;
            COLOR_HEAD;
        )

        *---------------------------------------------------------
        * Do editing.
        *---------------------------------------------------------

        while .t.                                   // FOREVER

            say(;
                nTop+3, nLeft+1,;
                HTF_PROMPT_SOURCE_FILE;
            )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cHtfSource := x, cHtfSource ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { || trueSetkey( K_F2, {|| gvFileDir( @cHtfSource ) } ) },;
                { || gvFileExist( @cHtfSource, _EXTENTION_TEXT ) };
            )

            say(;
                row()+1, nLeft+1,;
                HTF_PROMPT_DESTINATION_FILE;
            )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cHtfDestination := x, cHtfDestination ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { || trueSetkey( K_F2, {|| gvFileDir( @cHtfDestination ) } ) },;
                { || gvFileExtention( @cHtfDestination, _EXTENTION_HTM ) };
            )

            say(;
                row()+1, nLeft+1,;
                HTF_PROMPT_INDEX_START;
            )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cIndexStart := x, cIndexStart ) },;
                "@s8@",;
                ,;
                { || trueSetkey( K_F2, NIL ) },;
                { || gvDefault( @cIndexStart, HTF_INDEX_START ) };
            )
            tab( aTab )
            say(;
                row(), col(),;
                HTF_PROMPT_INDEX_END;
            )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cIndexEnd := x, cIndexEnd ) },;
                "@s8@",;
                ,;
                { || trueSetkey( K_F2, NIL ) },;
                { || gvDefault( @cIndexEnd, HTF_INDEX_END ) };
            )

            say(;
                row()+1, nLeft+1,;
                HTF_PROMPT_POINTER_START;
            )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cPointerStart := x, cPointerStart ) },;
                "@s8@",;
                ,;
                { || trueSetkey( K_F2, NIL ) },;
                { || gvDefault( @cPointerStart, HTF_POINTER_START ) };
            )
            tab( aTab )
            say(;
                row(), col(),;
                HTF_PROMPT_POINTER_END;
            )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cPointerEnd := x, cPointerEnd ) },;
                "@s8@",;
                ,;
                { || trueSetkey( K_F2, NIL ) },;
                { || gvDefault( @cPointerEnd, HTF_POINTER_END ) };
            )
            
            say(;
                row()+1, nLeft+1,;
                HTF_PROMPT_HTML_TITLE;
            )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cHtfTitle := x, cHtfTitle ) },;
                picChrMax( col(), nRight-1 );
            )

            button( @aButtons, row()+2, nLeft+1,;
                _BUTTON_ESC_CANCEL, , {||keyboard(chr(K_ESC))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F1_HELP, , {|| eval(setkey(K_F1))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F2_LIST, , {|| iif( valtype(setkey(K_F2)) == "B", eval(setkey(K_F2)), NIL ) } )
            button( @aButtons, row(), col()+1,;
                _BUTTON_PGDN_CONFIRM, , {||keyboard(chr(K_PGDN))} )

            *-----------------------------------------------------
            * Read.
            *-----------------------------------------------------

            read( aoGet, , aButtons )
            aoGet   := {}
            aButtons := {}

            *-----------------------------------------------------
            * Check what was read.
            *-----------------------------------------------------
                                        
            do case
            case lastkey() = K_ESC

                *-------------------------------------------------
                * [Esc] means leave.
                *-------------------------------------------------

                lGoOn := .F.

                exit

            case lastkey() = K_PGDN

                *-------------------------------------------------
                * [PgDn] means confirm.
                * Before, check for valid data.
                *-------------------------------------------------

                if  gvFileExist( @cHtfSource, _EXTENTION_HTF );
                    .and.;
                    !isWild( cHtfDestination );
                    .and.;
                    !empty( cIndexStart );
                    .and.;
                    !empty( cIndexEnd );
                    .and.;
                    !empty( cPointerStart );
                    .and.;
                    !empty( cPointerEnd )

                    lGoOn := .T.

                    exit                                // EXIT

                else

                    *---------------------------------------------
                    * Data is not valid: loop.
                    *---------------------------------------------

                end

            otherwise

                *-------------------------------------------------
                * Exit with different keys is not allowed.
                * Loop again.
                *-------------------------------------------------

            end

        end

        *---------------------------------------------------------
        * The editing is terminated: delete window.
        *---------------------------------------------------------

        mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

        *---------------------------------------------------------
        * Restore Colors.
        *---------------------------------------------------------

        setcolor( cOldColor )

        *---------------------------------------------------------
        * Proceed if it was so required.
        *---------------------------------------------------------

        if lGoOn

            htf2Html(;
                alltrim(cHtfSource),;
                alltrim(cHtfDestination),;
                alltrim(cIndexStart),;
                alltrim(cIndexEnd),;
                alltrim(cPointerStart),;
                alltrim(cPointerEnd),;
                alltrim(cHtfTitle);
            )

        end

    end //sequence

    *-------------------------------------------------------------
    * Restore old data.
    *-------------------------------------------------------------

    setcursor( nOldCursor )
    setcolor( cOldColor )
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )

    return NIL

#endif  //RUNTIME

*-----------------------------------------------------------------
* NB HELP
*-----------------------------------------------------------------

#define NBHELP_FILE_NOT_FOUND;
    "Help file not found."

*-----------------------------------------------------------------
static function nbHelp( xInitial, cHead, cKey )

    local nOldSelect          := select()
    local bOld_F1             := setkey( K_F1, NIL )

    default( @cKey, "" )

    *-------------------------------------------------------------
    * Do the help file exists?.
    *-------------------------------------------------------------
    
    if !file( nbHelp )

        *---------------------------------------------------------
        * The help file don't exists: tell it and return.
        *---------------------------------------------------------

        alertbox( NBHELP_FILE_NOT_FOUND )

        return NIL                                      // RETURN

    end

    *-------------------------------------------------------------
    * Open the file.
    *-------------------------------------------------------------
    
    dbusearea( .T., _DEFAULT_RDD, nbHelp, "HlpTxtFile", .T., .T.)

    *-------------------------------------------------------------
    * If the help file was successfully opened, run htf().
    *-------------------------------------------------------------
    
    if !neterr()

        *---------------------------------------------------------
        * Set up some defaults.
        *---------------------------------------------------------
        
        default( @cHead,        nbHelp )
        default( @xInitial,     1 )

        *---------------------------------------------------------
        * Stuff <ckey> into keyboard.
        *---------------------------------------------------------
        
        keyboard( cKey )

        *---------------------------------------------------------
        * Run htf().
        *---------------------------------------------------------
        
        htf( xInitial, cHead )

        *---------------------------------------------------------
        * Close the help file.
        *---------------------------------------------------------
        
        HlpTxtFile->(dbclosearea())

    end

    *-------------------------------------------------------------
    * Restore old data.
    *-------------------------------------------------------------
    
    dbselectarea( nOldSelect )
    setkey( K_F1, bOld_F1 )

    return NIL



