/* Copyright 1995-96 Jon Griffiths.  See the file "jlib.doc" for details.  */
#include <jlib.h>


/*+------------------------------------------------------------------------+*/
/*|Draw a sprite in a buffer without clipping                              |*/
/*+------------------------------------------------------------------------+*/
void buff_draw_spriteNC (sprite_system * ssys, int snum, buffer_rec * obuff)
{
   int frame      = ssys->sprites[snum]->frame,
       x          = ssys->sprites[snum]->x,
       y          = ssys->sprites[snum]->y,
       bwidth     = B_X_SIZE(obuff) - ssys->sprite_data[frame]->width,
       height     = ssys->sprite_data[frame]->height;
   UBYTE *pattern = ssys->sprite_data[frame]->pattern,
         *data    = ssys->sprite_data[frame]->data,
         *out     = B_OFFSET(obuff,y) + x;
   
   JLIB_ENTER("buff_draw_spriteNC");

   while(height--){
      int width = *pattern++;
      while(width--){
         UBYTE datum = *pattern++;
         SPR_STENCIL_COPY(out,data,datum);
         out += (datum & 15);
         data += (datum & 15);
      }
      out +=  bwidth;
   }

   JLIB_LEAVE;
}


/*+----------------------------------------------------------------------+ */
/*|draw_sprite - draw a sprite on screen with clipping                   | */
/*+----------------------------------------------------------------------+ */
void buff_draw_sprite (sprite_system * ssys, int snum, buffer_rec * obuff)
{
   int width, height, bwidth, bheight, clip=0, x, y, frame;
   UBYTE *data = (UBYTE *)0;

   JLIB_ENTER("buff_draw_sprite");

#ifndef JLIB_PRODUCTION
   jlib_check_buffer(obuff);
   jlib_check_sprite_system(ssys);
#endif

   frame   = ssys->sprites[snum]->frame;

#ifndef JLIB_PRODUCTION
   if((ssys->sprite_data[frame] == NULL)||(frame >= SPR_NUM_LOADED(ssys))){
      printf("%d frame is %d\n",frame,SPR_NUM_LOADED(ssys));
      jlib_exit("Invalid Frame");
   }
#endif

   x       = ssys->sprites[snum]->x;
   y       = ssys->sprites[snum]->y;
   width   = ssys->sprite_data[frame]->width;
   height  = ssys->sprite_data[frame]->height;
   bwidth  = B_MAX_X(obuff);
   bheight = B_MAX_Y(obuff);

   /* x clipping */
   if(x + width > bwidth){
      width = bwidth - x + 1;
      clip |= C_RIGHT;
   }
   else{
      if(x + width < 0){
        JLIB_LEAVE;
        return;
      }
   }

   if(x < 0){
      width += x;
      data  -= x;
      x = 0;
      clip |= C_LEFT;
   }
   else{
      if((x > bwidth)){
        JLIB_LEAVE;
        return;
      }
   }


   if (clip) {
       /* worst case: x is clipped */
       UBYTE *out;
       int sprwidth = ssys->sprite_data[frame]->width;
       data += (unsigned int)ssys->sprite_data[frame]->data;

       bwidth++;

       /* y clipping */
       if(y + height > bheight){
          height = bheight - y + 1;
          clip |= C_DOWN;
       }
       else{
          if(y + height < 0){
            JLIB_LEAVE;
            return;
          }
       }
    
       if(y < 0){
          height += y;
          do{
             data += sprwidth;
          }while(++y != 0);
          clip |= C_UP;
       }
       else{
          if(y > bheight){
            JLIB_LEAVE;
            return;
          }
       }

       out = B_OFFSET(obuff,y) + x;
   
       while(height--){
          int w = width;
          while(w--){
             if(data[w]){
                out[w] = data[w];
             }
          }
          out   += bwidth;
          data  += sprwidth;
       }
       JLIB_LEAVE;
       return;
   }
   else{
       UBYTE *pattern = ssys->sprite_data[frame]->pattern;
       data = ssys->sprite_data[frame]->data;

       /* y clipping */
       if(y + height > bheight){
          height = bheight - y + 1;
          clip |= C_DOWN;
       }
       else{
          if(y + height < 0){
            JLIB_LEAVE;
            return;
          }
       }
    
       if(y < 0){
          height += y;
          do{
             pattern += *pattern + 1; /* skip through RLE pattern */
             data += width;
          }while(++y != 0);
          clip |= C_UP;
       }
       else{
          if(y > bheight){
            JLIB_LEAVE;
            return;
          }
       }
    
       if(clip){
           /* average case: y is clipped */
           UBYTE *out = B_OFFSET(obuff,y) + x;
           bwidth = bwidth - width + 1;

           while(height--){
              int w = *pattern++;
              while(w--){
                 UBYTE datum = *pattern++;
                 SPR_STENCIL_COPY(out,data,datum);
                 out  += (datum & 15);
                 data += (datum & 15);
              }
              out +=  bwidth;
           }
       }
       else{
           /* best case: no clipping */
           buff_draw_spriteNC(ssys,snum,obuff);
       }
    }
    JLIB_LEAVE;
}


/*+-----------------------------------------------------------------------+*/
/*|Save the overwritten buffer under a sprite without clipping.           |*/
/*+-----------------------------------------------------------------------+*/
void buff_save_spriteNC (sprite_system * ssys, int snum, buffer_rec * obuff)
{
   int frame      = ssys->sprites[snum]->frame,
       x          = ssys->sprites[snum]->x,
       y          = ssys->sprites[snum]->y,
       bwidth     = B_X_SIZE(obuff) - ssys->sprite_data[frame]->width,
       height     = ssys->sprite_data[frame]->height;
   UBYTE *pattern = ssys->sprite_data[frame]->pattern,
         *buffer  = ssys->sprites[snum]->buffer,
         *out     = B_OFFSET(obuff,y) + x;
   
   JLIB_ENTER("buff_save_spriteNC");

   while(height--){
      int width = *pattern++;
      while(width--){
         UBYTE datum = *pattern++;
         SPR_STENCIL_COPY(buffer,out,datum);
         if(datum & 16)
                buffer += (datum & 15);
         out += (datum & 15);
      }
      out +=  bwidth;
   }

   JLIB_LEAVE;
}


/*+-----------------------------------------------------------------------+*/
/*|Save the overwritten buffer under a sprite with clipping               |*/
/*+-----------------------------------------------------------------------+*/
void buff_save_sprite (sprite_system * ssys, int snum, buffer_rec * obuff)
{
   int width, height, bwidth, bheight, clip=0, x, y, frame;
   UBYTE *buffer;

   JLIB_ENTER("buff_save_sprite");

#ifndef JLIB_PRODUCTION
   jlib_check_buffer(obuff);
   jlib_check_sprite_system(ssys);
#endif

   frame   = ssys->sprites[snum]->frame;

#ifndef JLIB_PRODUCTION
   if((ssys->sprite_data[frame] == NULL) || (frame >= SPR_NUM_LOADED(ssys)) || (frame < 0)){
      jlib_exit("Invalid Frame");
   }
#endif

   x       = ssys->sprites[snum]->x;
   y       = ssys->sprites[snum]->y;
   buffer  = ssys->sprites[snum]->buffer;
   width   = ssys->sprite_data[frame]->width;
   height  = ssys->sprite_data[frame]->height;
   bwidth  = B_MAX_X(obuff);
   bheight = B_MAX_Y(obuff);

   /* x clipping */
   if(x + width > bwidth){
      width = bwidth - x + 1;
      clip |= C_RIGHT;
   }
   else{
      if(x + width < 0){
        JLIB_LEAVE;
        return;
      }
   }

   if(x < 0){
      width += x;
      x = 0;
      clip |= C_LEFT;
   }
   else{
      if((x > bwidth)){
        JLIB_LEAVE;
        return;
      }
   }


   if (clip) {
       /* worst case: x is clipped */
       UBYTE *out;

       bwidth++;

       /* y clipping */
       if(y + height > bheight){
          height = bheight - y + 1;
          clip |= C_DOWN;
       }
       else{
          if(y + height < 0){
            JLIB_LEAVE;
            return;
          }
       }
    
       if(y < 0){
          height += y;
          y = 0;
          clip |= C_UP;
       }
       else{
          if(y > bheight){
            JLIB_LEAVE;
            return;
          }
       }

       out = B_OFFSET(obuff,y) + x;
   
       while(height--){
          int w = width;
          while(w--){
             buffer[w] = out[w]; /* copy all bytes to save buffer */
          }
          out    += bwidth;
          buffer += width;
       }
       JLIB_LEAVE;
       return;
   }
   else{
       UBYTE *pattern = ssys->sprite_data[frame]->pattern;

       /* y clipping */
       if(y + height > bheight){
          height = bheight - y + 1;
          clip |= C_DOWN;
       }
       else{
          if(y + height < 0){
            JLIB_LEAVE;
            return;
          }
       }
    
       if(y < 0){
          height += y;
          do{
             pattern += *pattern + 1; /* skip through RLE pattern */
          }while(++y != 0);
          clip |= C_UP;
       }
       else{
          if(y > bheight){
            JLIB_LEAVE;
            return;
          }
       }
    
       if(clip){
           /* average case: y is clipped */
           UBYTE *out = B_OFFSET(obuff,y) + x;
           bwidth = bwidth - width + 1;

           while(height--){
              int w = *pattern++;
              while(w--){
                 UBYTE datum = *pattern++;
                 SPR_STENCIL_COPY(buffer,out,datum);
                 if (datum & 16)
                        buffer += (datum & 15);
                 out  += (datum & 15);
              }
              out +=  bwidth;
           }
       }
       else{
           /* best case: no clipping */
           buff_save_spriteNC(ssys,snum,obuff);
       }
    }
    JLIB_LEAVE;
}


/*+-----------------------------------------------------------------------+*/
/*|Restore the buffer under a sprite without clipping                     |*/
/*+-----------------------------------------------------------------------+*/
void buff_rest_spriteNC (sprite_system * ssys, int snum, buffer_rec * obuff)
{
   int frame      = ssys->sprites[snum]->frame,
       x          = ssys->sprites[snum]->x,
       y          = ssys->sprites[snum]->y,
       bwidth     = B_X_SIZE(obuff) - ssys->sprite_data[frame]->width,
       height     = ssys->sprite_data[frame]->height;
   UBYTE *pattern = ssys->sprite_data[frame]->pattern,
         *buffer  = ssys->sprites[snum]->buffer,
         *out     = B_OFFSET(obuff,y) + x;
   
   JLIB_ENTER("buff_rest_spriteNC");

   while(height--){
      int width = *pattern++;
      while(width--){
         UBYTE datum = *pattern++;
         SPR_STENCIL_COPY(out,buffer,datum);
         if(datum & 16)
                buffer += (datum & 15);
         out += (datum & 15);
      }
      out +=  bwidth;
   }

   JLIB_LEAVE;
}


/*+-----------------------------------------------------------------------+*/
/*|Restore the overwritten buffer under a sprite with clipping            |*/
/*+-----------------------------------------------------------------------+*/
void buff_rest_sprite (sprite_system * ssys, int snum, buffer_rec * obuff)
{
   int width, height, bwidth, bheight, clip=0, x, y, frame;
   UBYTE *buffer;

   JLIB_ENTER("buff_rest_sprite");

#ifndef JLIB_PRODUCTION
   jlib_check_buffer(obuff);
   jlib_check_sprite_system(ssys);
#endif

   frame   = ssys->sprites[snum]->frame;

#ifndef JLIB_PRODUCTION
   if((ssys->sprite_data[frame] == NULL) || (frame >= SPR_NUM_LOADED(ssys)) || (frame < 0)){
      jlib_exit("Invalid Frame");
   }
#endif

   x       = ssys->sprites[snum]->x;
   y       = ssys->sprites[snum]->y;
   buffer  = ssys->sprites[snum]->buffer;
   width   = ssys->sprite_data[frame]->width;
   height  = ssys->sprite_data[frame]->height;
   bwidth  = B_MAX_X(obuff);
   bheight = B_MAX_Y(obuff);

   /* x clipping */
   if(x + width > bwidth){
      width = bwidth - x + 1;
      clip |= C_RIGHT;
   }
   else{
      if(x + width < 0){
        JLIB_LEAVE;
        return;
      }
   }

   if(x < 0){
      width += x;
      x = 0;
      clip |= C_LEFT;
   }
   else{
      if((x > bwidth)){
        JLIB_LEAVE;
        return;
      }
   }


   if (clip) {
       /* worst case: x is clipped */
       UBYTE *out;

       bwidth++;

       /* y clipping */
       if(y + height > bheight){
          height = bheight - y + 1;
          clip |= C_DOWN;
       }
       else{
          if(y + height < 0){
            JLIB_LEAVE;
            return;
          }
       }
    
       if(y < 0){
          height += y;
          y = 0;
          clip |= C_UP;
       }
       else{
          if(y > bheight){
            JLIB_LEAVE;
            return;
          }
       }

       out = B_OFFSET(obuff,y) + x;
   
       while(height--){
          int w = width;
          while(w--){
             out[w] = buffer[w]; /* copy all bytes to save buffer */
          }
          out    += bwidth;
          buffer += width;
       }
       JLIB_LEAVE;
       return;
   }
   else{
       UBYTE *pattern = ssys->sprite_data[frame]->pattern;

       /* y clipping */
       if(y + height > bheight){
          height = bheight - y + 1;
          clip |= C_DOWN;
       }
       else{
          if(y + height < 0){
            JLIB_LEAVE;
            return;
          }
       }
    
       if(y < 0){
          height += y;
          do{
             pattern += *pattern + 1; /* skip through RLE pattern */
          }while(++y != 0);
          clip |= C_UP;
       }
       else{
          if(y > bheight){
            JLIB_LEAVE;
            return;
          }
       }
    
       if(clip){
           /* average case: y is clipped */
           UBYTE *out = B_OFFSET(obuff,y) + x;
           bwidth = bwidth - width + 1;

           while(height--){
              int w = *pattern++;
              while(w--){
                 UBYTE datum = *pattern++;
                 SPR_STENCIL_COPY(out,buffer,datum);
                 if (datum & 16)
                        buffer += (datum & 15);
                 out  += (datum & 15);
              }
              out +=  bwidth;
           }
       }
       else{
           /* best case: no clipping */
           buff_rest_spriteNC(ssys,snum,obuff);
       }
    }
    JLIB_LEAVE;
}


/*+-----------------------------------------------------------------------+*/
/*|Stamp a sprite frame on buff without clipping                          |*/
/*+-----------------------------------------------------------------------+*/
void buff_stamp_spriteNC(sprite_system *ssys,int frame, buffer_rec * obuff,int x, int y)
{
   int bwidth     = B_X_SIZE(obuff) - ssys->sprite_data[frame]->width,
       height     = ssys->sprite_data[frame]->height;
   UBYTE *pattern = ssys->sprite_data[frame]->pattern,
         *data    = ssys->sprite_data[frame]->data,
         *out     = B_OFFSET(obuff,y) + x;
   
   JLIB_ENTER("buff_stamp_spriteNC");

   while(height--){
      int width = *pattern++;
      while(width--){
         UBYTE datum = *pattern++;
         SPR_STAMP_COPY(out,data,datum);
         out += (datum & 15);
         data += (datum & 15);
      }
      out +=  bwidth;
   }

   JLIB_LEAVE;
}


/*+-----------------------------------------------------------------------+*/
/*|Stamp a sprite frame on buff with clipping                             |*/
/*+-----------------------------------------------------------------------+*/
void buff_stamp_sprite(sprite_system *ssys,int frame, buffer_rec * obuff,int x, int y)
{
   int width, height, bwidth, bheight, clip=0;
   UBYTE *data = (UBYTE *)0;

   JLIB_ENTER("buff_stamp_sprite");

#ifndef JLIB_PRODUCTION
   jlib_check_buffer(obuff);
   jlib_check_sprite_system(ssys);
#endif

   width   = ssys->sprite_data[frame]->width;
   height  = ssys->sprite_data[frame]->height;
   bwidth  = B_MAX_X(obuff);
   bheight = B_MAX_Y(obuff);

   /* x clipping */
   if(x + width > bwidth){
      width = bwidth - x + 1;
      clip |= C_RIGHT;
   }
   else{
      if(x + width < 0){
        JLIB_LEAVE;
        return;
      }
   }

   if(x < 0){
      width += x;
      data  -= x;
      x = 0;
      clip |= C_LEFT;
   }
   else{
      if((x > bwidth)){
        JLIB_LEAVE;
        return;
      }
   }


   if (clip) {
       /* worst case: x is clipped */
       UBYTE *out;
       int sprwidth = ssys->sprite_data[frame]->width;
       data += (unsigned int)ssys->sprite_data[frame]->data;

       bwidth++;

       /* y clipping */
       if(y + height > bheight){
          height = bheight - y + 1;
          clip |= C_DOWN;
       }
       else{
          if(y + height < 0){
            JLIB_LEAVE;
            return;
          }
       }
    
       if(y < 0){
          height += y;
          do{
             data += sprwidth;
          }while(++y != 0);
          clip |= C_UP;
       }
       else{
          if(y > bheight){
            JLIB_LEAVE;
            return;
          }
       }

       out = B_OFFSET(obuff,y) + x;
   
       while(height--){
          int w = width;
          while(w--){
             out[w] = data[w];
          }
          out   += bwidth;
          data  += sprwidth;
       }
       JLIB_LEAVE;
       return;
   }
   else{
       UBYTE *pattern = ssys->sprite_data[frame]->pattern;
       data = ssys->sprite_data[frame]->data;

       /* y clipping */
       if(y + height > bheight){
          height = bheight - y + 1;
          clip |= C_DOWN;
       }
       else{
          if(y + height < 0){
            JLIB_LEAVE;
            return;
          }
       }
    
       if(y < 0){
          height += y;
          do{
             pattern += *pattern + 1; /* skip through RLE pattern */
             data += width;
          }while(++y != 0);
          clip |= C_UP;
       }
       else{
          if(y > bheight){
            JLIB_LEAVE;
            return;
          }
       }
    
       if(clip){
           /* average case: y is clipped */
           UBYTE *out = B_OFFSET(obuff,y) + x;
           bwidth = bwidth - width + 1;

           while(height--){
              int w = *pattern++;
              while(w--){
                 UBYTE datum = *pattern++;
                 SPR_STAMP_COPY(out,data,datum);
                 out  += (datum & 15);
                 data += (datum & 15);
              }
              out +=  bwidth;
           }
       }
       else{
           /* best case: no clipping */
           buff_stamp_spriteNC(ssys,frame,obuff,x,y);
       }
    }
    JLIB_LEAVE;
}


/*+-----------------------------------------------------------------------+*/
/*|Stencil a sprite frame on buff without clipping                        |*/
/*+-----------------------------------------------------------------------+*/
void buff_stencil_spriteNC(sprite_system *ssys,int frame, buffer_rec * obuff,int x, int y)
{
   int bwidth     = B_X_SIZE(obuff) - ssys->sprite_data[frame]->width,
       height     = ssys->sprite_data[frame]->height;
   UBYTE *pattern = ssys->sprite_data[frame]->pattern,
         *data    = ssys->sprite_data[frame]->data,
         *out     = B_OFFSET(obuff,y) + x;
   
   JLIB_ENTER("buff_stencil_spriteNC");

   while(height--){
      int width = *pattern++;
      while(width--){
         UBYTE datum = *pattern++;
         SPR_STENCIL_COPY(out,data,datum);
         out += (datum & 15);
         data += (datum & 15);
      }
      out +=  bwidth;
   }

   JLIB_LEAVE;
}

/*+-----------------------------------------------------------------------+*/
/*|Stencil a sprite frame on buff with clipping                           |*/
/*+-----------------------------------------------------------------------+*/
void buff_stencil_sprite(sprite_system *ssys,int frame, buffer_rec * obuff,int x, int y)
{
   int width, height, bwidth, bheight, clip=0;
   UBYTE *data = (UBYTE *)0;

   JLIB_ENTER("buff_stencil_sprite");

#ifndef JLIB_PRODUCTION
   jlib_check_buffer(obuff);
   jlib_check_sprite_system(ssys);
#endif

   width   = ssys->sprite_data[frame]->width;
   height  = ssys->sprite_data[frame]->height;
   bwidth  = B_MAX_X(obuff);
   bheight = B_MAX_Y(obuff);

   /* x clipping */
   if(x + width > bwidth){
      width = bwidth - x + 1;
      clip |= C_RIGHT;
   }
   else{
      if(x + width < 0){
        JLIB_LEAVE;
        return;
      }
   }

   if(x < 0){
      width += x;
      data  -= x;
      x = 0;
      clip |= C_LEFT;
   }
   else{
      if((x > bwidth)){
        JLIB_LEAVE;
        return;
      }
   }


   if (clip) {
       /* worst case: x is clipped */
       UBYTE *out;
       int sprwidth = ssys->sprite_data[frame]->width;
       data += (unsigned int)ssys->sprite_data[frame]->data;

       bwidth++;

       /* y clipping */
       if(y + height > bheight){
          height = bheight - y + 1;
          clip |= C_DOWN;
       }
       else{
          if(y + height < 0){
            JLIB_LEAVE;
            return;
          }
       }
    
       if(y < 0){
          height += y;
          do{
             data += sprwidth;
          }while(++y != 0);
          clip |= C_UP;
       }
       else{
          if(y > bheight){
            JLIB_LEAVE;
            return;
          }
       }

       out = B_OFFSET(obuff,y) + x;
   
       while(height--){
          int w = width;
          while(w--){
             if(data[w])
                out[w] = data[w];
          }
          out   += bwidth;
          data  += sprwidth;
       }
       JLIB_LEAVE;
       return;
   }
   else{
       UBYTE *pattern = ssys->sprite_data[frame]->pattern;
       data = ssys->sprite_data[frame]->data;

       /* y clipping */
       if(y + height > bheight){
          height = bheight - y + 1;
          clip |= C_DOWN;
       }
       else{
          if(y + height < 0){
            JLIB_LEAVE;
            return;
          }
       }
    
       if(y < 0){
          height += y;
          do{
             pattern += *pattern + 1; /* skip through RLE pattern */
             data += width;
          }while(++y != 0);
          clip |= C_UP;
       }
       else{
          if(y > bheight){
            JLIB_LEAVE;
            return;
          }
       }
    
       if(clip){
           /* average case: y is clipped */
           UBYTE *out = B_OFFSET(obuff,y) + x;
           bwidth = bwidth - width + 1;

           while(height--){
              int w = *pattern++;
              while(w--){
                 UBYTE datum = *pattern++;
                 SPR_STENCIL_COPY(out,data,datum);
                 out  += (datum & 15);
                 data += (datum & 15);
              }
              out +=  bwidth;
           }
       }
       else{
           /* best case: no clipping */
           buff_stencil_spriteNC(ssys,frame,obuff,x,y);
       }
    }
    JLIB_LEAVE;
}


/*+-----------------------------------------------------------------------+*/
/*|Draw every active sprite                                               |*/
/*+-----------------------------------------------------------------------+*/
JINLINE void buff_draw_all_sprites (sprite_system * ssys, buffer_rec * obuff)
{
   int count;

   JLIB_ENTER("buff_draw_all_sprites");

#ifndef JLIB_PRODUCTION
   jlib_check_buffer(obuff);
   jlib_check_sprite_system(ssys);
#endif

   for (count = 0; count < ssys->no_sprites; count++) {
      if (SPR_IS_ON (ssys, count)) {
         buff_draw_sprite (ssys, count, obuff);
      }
   }

   JLIB_LEAVE;
}


/*+-----------------------------------------------------------------------+*/
/*|Save every active sprite                                               |*/
/*+-----------------------------------------------------------------------+*/
JINLINE void buff_save_all_sprites (sprite_system * ssys, buffer_rec * obuff)
{
   int count;

   JLIB_ENTER("buff_save_all_sprites");

#ifndef JLIB_PRODUCTION
   jlib_check_buffer(obuff);
   jlib_check_sprite_system(ssys);
#endif

   for (count = 0; count < ssys->no_sprites; count++) {
      if (SPR_IS_ON (ssys, count)) {
         buff_save_sprite (ssys, count, obuff);
      }
   }

   JLIB_LEAVE;
}


/*+-----------------------------------------------------------------------+*/
/*|Restore every active sprite                                            |*/
/*+-----------------------------------------------------------------------+*/
JINLINE void buff_rest_all_sprites (sprite_system * ssys, buffer_rec * obuff)
{
   int count;

   JLIB_ENTER("buff_rest_all_sprites");

#ifndef JLIB_PRODUCTION
   jlib_check_buffer(obuff);
   jlib_check_sprite_system(ssys);
#endif

   for (count = 0; count < ssys->no_sprites; count++) {
      if (SPR_IS_ON (ssys, count)) {
          buff_rest_sprite (ssys, count, obuff);
      }
   }

   JLIB_LEAVE;
}
