/*
   Module: send7221.c
   Date:   3/9/92
   Version: 1.0b
   Author:  Dave Lutz
   Email:   lutz@psych.rochester.edu
   Copyright: 1992 University of Rochester, Psychology Dept.

   Disclaimer:  This software is distributed free of charge.  As such, it
                comes with ABSOLUTELY NO WARRANTY.  The user of the software
                assumes ALL RISKS associated with its use.

                Your rights to modify and/or distribute this software are
                outlined in the file SEND7221.DOC.

   Purpose: This module provides a method to send hp7221 codes to an hp7221
            plotter for plotting.  It reads the codes from a file, and sends
            them out one of the serial ports.  The program handles buffer
            control to avoid buffer overflow on large jobs.
*/

#include <time.h>
#include <stdio.h>
#include "input.h"
#include "comio.h"
#include "hpcodes.h"
#include "retcodes.h"


/* local funtion prototypes */
void usage(void);
int sendbuff (PLTFILE *infile, COMPORT *com);


int main (argc, argv)
   int argc;
   char *argv[];
{
   int i, readcnt;
   char *infilename;
   int baudrate;
   unsigned commask, baudmask, datamask, stopmask, paritymask, initmask;
   time_t timeout;
   PLTFILE *infile = (PLTFILE *) NULL;
   COMPORT com;

   /* print welcome message */
   printf ("SEND7221 1.0b.  University of Rochester, Psychology Dept.\n\n");

   /* set defaults */
   commask    = COM1;
   baudmask   = BAUD_2400;
   datamask   = DATA_7;
   stopmask   = STOP_1;
   paritymask = PAR_E;
   timeout    = 120;  /* seconds */
   infilename = (char *)NULL;

   /* process command line args */
   i = 0;
   while (++i < argc) {
      if (argv[i][0] == '/') {
         if (argv[i][2] != ':') {
            fprintf (stderr, "Error: Invalid option: %s\n",argv[i]);
            usage();
            exit (BADARGS);
         } else {
            switch (argv[i][1]) {
               case 'C':
               case 'c':
                  /* set com port */
                  switch (argv[i][3]) {
                     case '1':
                        commask = COM1;
                        break;
                     case '2':
                        commask = COM2;
                        break;
                     case '3':
                        commask = COM3;
                        break;
                     case '4':
                        commask = COM4;
                        break;
                     default:
                        fprintf (stderr,"Error: Invalid com port: %s\n",
                                 argv[i]);
                        usage();
                        exit (BADARGS);
                  }
                  break;
               case 'B':
               case 'b':
                  /* set baud rate */
                  baudrate = atoi(&argv[i][3]);
                  switch (baudrate) {
                     case 300:
                        baudmask = BAUD_300;
                        break;
                     case 1200:
                        baudmask = BAUD_1200;
                        break;
                     case 2400:
                        baudmask = BAUD_2400;
                        break;
                     default:
                        fprintf (stderr, "Error: Invalid baud: %s\n",
                                 argv[i]);
                        usage();
                        exit (BADARGS);
                  }
                  break;
               case 'D':
               case 'd':
                  /* set data bits */
                  switch (argv[i][3]) {
                     case '7':
                        datamask = DATA_7;
                        break;
                     case '8':
                        datamask = DATA_8;
                        break;
                     case 'D':
                     case 'd':
                        datamask = DATA_D; /* don't care */
                        break;
                     default:
                        fprintf (stderr, "Error: Invalid data bits: %s\n",
                                 argv[i]);
                        usage();
                        exit (BADARGS);
                  }
                  break;
               case 'S':
               case 's':
                  /* set stop bits */
                  switch (argv[i][3]) {
                     case '1':
                        stopmask = STOP_1;
                        break;
                     case '2':
                        stopmask = STOP_2;
                        break;
                     default:
                        fprintf (stderr, "Error: Invalid stop bits: %s\n",
                                 argv[i]);
                        usage();
                        exit (BADARGS);
                  }
                  break;
               case 'P':
               case 'p':
                  /* set parity */
                  switch (argv[i][3]) {
                     case 'E':
                     case 'e':
                        paritymask = PAR_E;
                        break;
                     case 'O':
                     case 'o':
                        paritymask = PAR_O;
                        break;
                     case 'N':
                     case 'n':
                        paritymask = PAR_N;
                        break;
                     case 'D':
                     case 'd':
                        paritymask = PAR_D;  /* don't care */
                        break;
                     default:
                        fprintf (stderr, "Error: Invalid parity: %s\n",
                                 argv[i]);
                        usage();
                        exit (BADARGS);
                  }
                  break;
               case 'T':
               case 't':
                  /* set timeout in seconds */
                  timeout = (time_t) atoi(&argv[i][3]);
                  break;
               default:
                  fprintf (stderr, "Error: Invalid option: %s\n", argv[i]);
                  usage();
                  exit (BADARGS);
            }
         } 
      } else {
         /* argv[i][0] != '/' */
         if (infilename != (char *)NULL) {
            fprintf (stderr, "Error: Too many input files specified.\n");
            usage();
            exit (BADARGS);
         } else {
            infilename = argv[i];
         }
      }
   }
   if (infilename == (char *) NULL) {
      fprintf (stderr, "Error: Input file not specified.\n");
      usage();
      exit (BADARGS);
   }

   switch (openin(infilename, &infile)) {
      case TRUE:
         break;
      case NOBUFF:
         fprintf (stderr, "Error: can't allocate input buffer.\n");
         exit (NOBUFF);
      case BADOPEN:
         fprintf (stderr, "Error: can't open input file for reading.\n");
         exit (BADOPEN);
   }
   
   initmask = (datamask | stopmask | paritymask | baudmask);
   (void) cominit (&com, commask, initmask, timeout);

   if (comout (&com, HP_INIT, HP_INIT_BYTES) != HP_INIT_BYTES) {
      fprintf (stderr, "Error: I/O error sending init string to plotter.\n");
      (void) closein (&infile);
      exit (BADCIO);
   }

   while ((readcnt = fillbuff(infile)) > 0) {
      if (!sendbuff (infile, &com)) {
         fprintf (stderr, "Error: I/O error sending data to plotter.\n");
         (void) closein (&infile);
         exit (BADCIO);
      }
   }
   if (readcnt == -1) {
       fprintf (stderr, "Error: I/O error reading input file.\n");
       (void) comout (&com, HP_ABORT, HP_ABORT_BYTES);
       (void) closein (&infile);
       exit (BADFIO);
   }
   if (!closein (&infile)) {
      fprintf (stderr, "Error: I/O error closing input file.\n");
      exit (BADFIO);
   }
   if (comout (&com, HP_END, HP_END_BYTES) != HP_END_BYTES) {
      fprintf (stderr, "Error: I/O error sending HP_END command to plotter.\n");
      exit (BADCIO);
   }
   exit (ALLOK);
}



/*
   Function: usage
   Purpose:  Print out a usage message for the program.

   Pre: none

   Post:  A usage message is printed to stderr.
*/

void usage(void)
{
   fprintf (stderr, "Usage: SEND7221 [/C:?] [/D:?] [/S:?] [/P:?] [/B:????] infile\n\n");
   fprintf (stderr, "\t/C:? = Com port number (1-4)        Default=1\n");
   fprintf (stderr, "\t/D:? = Number of data bits (7/8/D)  Default=7\n");
   fprintf (stderr, "\t/S:? = Number of stop bits (1/2)    Default=1\n");
   fprintf (stderr, "\t/P:? = Parity (E/O/N/D)             Default=E\n");
   fprintf (stderr, "\t/B:???? = Baud rate (300/1200/2400) Default=2400\n");
   fprintf (stderr, "\t/T:???? = read Timeout in seconds   Default=120\n");
}


/*
   Function: sendbuff
   Purpose:  Send the contents of the infile buffer to the HP7221.

   Pre: infile is a pointer to a PLTFILE that has been opened for reading.
        The buffer in infile has been filled by a call to fillbuff.
        comp is a pointer to a COMPORT.
        comp has been set up by a call to cominit.

   Post: An attempt is made to send the contents of the buffer to the HP7221.
         If all of the bytes in the buffer can't be sent, FALSE is returned.
         Otherwise, TRUE is returned.
*/

int sendbuff (infile, comp)
   PLTFILE *infile;
   COMPORT *comp;
{
   char cts,  /* storage for clear_to_send character */
        *buf_ptr;

   int fullblocks, /* number of HP_BLOCKSIZE blocks needed to send the data */
       lbsize;     /* size of last (short) block, if needed */

   fullblocks = infile->buf_ctr / HP_BLOCKSIZE;
   lbsize     = infile->buf_ctr % HP_BLOCKSIZE;

   buf_ptr = infile->buf;

   while (fullblocks-- > 0) {
      if (comout (comp, HP_RTS, HP_RTS_BYTES) != HP_RTS_BYTES)
         return (FALSE);
      if (comin (comp, &cts, 1, -1) < 1)
         return (FALSE);
      if (comout (comp, buf_ptr, HP_BLOCKSIZE) != HP_BLOCKSIZE)
         return (FALSE);
      buf_ptr += HP_BLOCKSIZE;
   }
   if (lbsize != 0) {
      if (comout (comp, HP_RTS, HP_RTS_BYTES) != HP_RTS_BYTES)
         return (FALSE);
      if (comin (comp, &cts, 1, -1) < 1)
         return (FALSE);
      if (comout (comp, buf_ptr, lbsize) != lbsize)
         return (FALSE);
      buf_ptr += lbsize;
   }
   return (TRUE);
}
