/*
   Module:  hp7221.c
   Date:    3/9/92
   Version: 1.0b
   Author:  Dave Lutz
   Email:   lutz@psych.rochester.edu
   Copyright: 1992 University of Rochester, Psychology Dept.

   Disclaimer:  This software is distributed free of charge.  As such, it
                comes with ABSOLUTELY NO WARRANTY.  The user of the software
                assumes ALL RISKS associated with its use.

                Your rights to modify and/or distribute this software are
                outlined in the file ADI7221.DOC.

   Purpose: This module provides functions for assembling the codes that
            are to be sent to the HP 7221 plotter.  Each function assembles
            a char array of codes that are used to represent the
            desired plotter functions.  The resulting arrays are then
            output via the putarr function that is provided by the
            generic ADI module.

   Functions provided:

        dev_beginplot
        dev_endplot
        dev_move
        dev_draw
        dev_newpen      (automatic pen selection)
        dev_setspeed
        dev_linetype
        dev_penchange   (pen up for manual pen change)
        dev_abortplot

        Note: When a dev_penchange is called, the main driver is expected
        to pause while the operator manually changes the pen.  If this is
        not possible (ie the driver is not a direct feed to the plotter),
        dev_penchange should not be called.

   Functions required:

        putarr       (output an char array)
        uinttosbn    (unsigned int to Single Byte Number)
        uiptombp     (unsigned int pair to Multiple Byte Pair)
*/

#include "inout.h"
#include "hpparms.h"
#include "hpcodes.h"
#include "hp7221.h"
#include "retcodes.h"

static drawing=FALSE;  /* global used to indicate if we are currently in
                          drawing mode.
                       */

/*
   Function: dev_beginplot
   Purpose:  Produces the code used to initialize the plotter for a new
             plot.

   Pre: pltfp is a pointer to a PLTFILE that has been opened for writing.

   Post: The char array of codes necessary to initialize the plotter is 
         output via putarr().  The array contains the HP_BEGINPLOT code.
         If putarr() returns an error, BADIO is returned.
         Otherwise TRUE is returned.
*/

int dev_beginplot (pltfp)
   PLTFILE *pltfp;
{
   if (!putarr(pltfp,HPBP_BYTES,HP_BEGINPLOT))
      return (BADIO);
   return (TRUE);
}



/*
   Function: dev_endplot
   Purpose:  Produces the code used to signal the plotter that the current
             plot is complete.

   Pre: The global variable 'drawing' indicates whether a draw is currently
        in progess.
        pltfp is a pointer to a PLTFILE that has been opened for writing.

   Post: The char array of codes necessary to signal the end of a plot is 
         output via putarr().
         If the global variable 'drawing' is TRUE, it is set FALSE, and the 
         array begins with the HP_END_DRAW code. The array always contains 
         the HP_ENDPLOT code.
         If putarr() returns an error, BADIO is returned.
         Otherwise TRUE is returned.
*/

int dev_endplot (pltfp)
   PLTFILE *pltfp;
{
   extern drawing;

   if (drawing) {
      drawing = FALSE;
      if (!putarr (pltfp,HPED_BYTES,HP_END_DRAW))
         return (BADIO);
   }
   if (!putarr (pltfp,HPEP_BYTES,HP_ENDPLOT))
      return (BADIO);
   return (TRUE);
}



/*
   Function: dev_move
   Purpose:  Produces the code used to cause the plotter to perform a pen 
             up move.

   Pre: xcoord contains the X coordinate of the ending location of the move.
        ycoord contains the Y coordinate of the ending location of the move.
        pltfp is a pointer to a PLTFILE that has been opened for writing.
        The global variable 'drawing' indicates if a draw is currently in
        progress.

   Post: The char array of codes necessary to perform a pen up move is 
         output via putarr().
         If the global variable 'drawing' is TRUE, it is set FALSE and the
         array begins with the HP_END_DRAW code.  
         The array always includes: the HP_MOVE code, the MBP formatted 
         representation of xcoord and ycoord, and the HP_END_MOVE code.
         If xcoord or ycoord are out of range, BADFMT is returned.
         If putarr returns an error, BADIO is returned.
         Otherwise TRUE is returned.
*/

int dev_move (xcoord, ycoord, pltfp)
   unsigned xcoord,ycoord;
   PLTFILE *pltfp;
{
   extern drawing;
   char mbp[MAXMBPLEN];
   int cnt;

   if (drawing) {
      drawing = FALSE;
      if (!putarr (pltfp,HPED_BYTES,HP_END_DRAW))
         return (BADIO);
   }
   if (!putarr (pltfp,HPM_BYTES,HP_MOVE))
      return (BADIO);
   if (!uiptombp (xcoord, ycoord, mbp, &cnt))
      return (BADFMT);
   if (!putarr (pltfp,cnt,mbp))
      return (BADIO);
   if (!putarr (pltfp,HPEM_BYTES,HP_END_MOVE))
      return (BADIO);
   return (TRUE);
}



/*
   Function: dev_draw
   Purpose:  Produces the code used to cause the plotter to perform a pen
             down draw.

   Pre: xcoord contains the X coordinate of the ending location of the draw.
        ycoord contains the Y coordinate of the ending location of the draw.
        pltfp is a pointer to a PLTFILE that has been opened for writing.
        The global variable 'drawing' indicates whether or not a draw is 
        currently in progress.

   Post: The char array of codes necessary to perform a pen down draw is 
         output via putarr().
         If the global variable 'drawing' is initially TRUE, the array
         contains only the MBP formatted representation of xcoord and ycoord.
         Otherwise, 'drawing' is set TRUE, and the array contains the
         HP_DRAW code, followed by the MBP formatted representation of
         xcoord and ycoord.
         If xcoord or ycoord are out of range, BADFMT is returned. 
         If putarr() returns an error, BADIO is returned.
         Otherwise TRUE is returned.
*/

int dev_draw (xcoord, ycoord, pltfp)
   unsigned xcoord, ycoord;
   PLTFILE *pltfp;
{
   extern drawing;
   char mbp[MAXMBPLEN];
   int cnt;

   if (!drawing) {
      drawing = TRUE;
      if (!putarr (pltfp,HPD_BYTES,HP_DRAW))
         return (BADIO);
   }
   if (!uiptombp (xcoord, ycoord, mbp, &cnt))
      return (BADFMT);
   if (!putarr (pltfp,cnt,mbp))
      return (BADIO);
   return (TRUE);
}



/*
   Function: dev_newpen
   Purpose:  Produces the code used to cause the plotter to automatically
             select a new pen.

   Pre: pennum contains the number of the pen to be selected.
        pltfp is a pointer to a PLTFILE that has been opened for writing.
        The global variable 'drawing' indicates whether a draw is currently
        in progress.

   Post: The char array of codes necessary to perform an automatic pen 
         select is output via putarr().
         If the global variable 'drawing' is TRUE, it is set FALSE and the
         array begins with the HP_END_DRAW code.  
         The array always includes the HP_NEWPEN code followed by the SBN
         formatted representation of pennum.
         If pennum is out of range, BADFMT is returned. 
         If putarr() returns an error, BADIO is returned.
         Otherwise TRUE is returned.
*/

int dev_newpen (pennum, pltfp)
   unsigned pennum;
   PLTFILE *pltfp;
{
   extern drawing;
   char sbn;

   if (drawing) {
      drawing = FALSE;
      if (!putarr (pltfp,HPED_BYTES,HP_END_DRAW))
         return (BADIO);
   }
   if (!putarr (pltfp,HPNP_BYTES,HP_NEWPEN))
      return (BADIO);
   if (!uinttosbn(pennum,&sbn))
      return (BADFMT);
   if (!putarr (pltfp,1,&sbn))
      return (BADIO);
   return (TRUE);
}



/*
   Function: dev_setspeed
   Purpose:  Produces the code used to cause the plotter to select a new
             pen speed.

   Pre: speed contains the code for the new speed.
        pltfp is a pointer to a PLTFILE that has been opened for writing.
        The global variable 'drawing' indicates whether a draw is currently
        in progress.

   Post: The char array of codes necessary to cause the plotter to select 
         a new speed is output via putarr().
         If speed is out of range, BADFMT is returned. 
         If the global variable 'drawing' is TRUE, it is set FALSE and the
         array begins with the HP_END_DRAW code.  
         The array always includes the HP_SETSPEED code followed by the SBN
         fomatted representation of speed.
         If putarr() returns an error, BADIO is returned.
         Otherwise TRUE is returned.
*/

int dev_setspeed (speed, pltfp)
   unsigned speed;
   PLTFILE *pltfp;
{
   extern drawing;
   char sbn;

   if (speed > HP_MAXSPEED)
      return (BADFMT);

   if (drawing) {
      drawing = FALSE;
      if (!putarr (pltfp,HPED_BYTES,HP_END_DRAW))
         return (BADIO);
   }
   if (!putarr (pltfp,HPSS_BYTES,HP_SETSPEED))
      return (BADIO);
   if (!uinttosbn(speed,&sbn))
      return (BADFMT);
   if (!putarr (pltfp,1,&sbn))
      return (BADIO);
   return (TRUE);
}



/*
   Function: dev_linetype
   Purpose:  Produces the code needed to cause the plotter to beging using a
             different line type for subesequent draw commands.

             Currently supported line types are:

                (0)  ----------------------------------- (solid)
                (1)  -- -- -- -- -- -- -- -- -- -- -- -- (dashed)
                (2)  - - - - - - - - - - - - - - - - - - (hidden)
                (3)  ---- - ---- - ---- - ---- - ---- -  (center)
                (4)  ----- - - ----- - - ----- - - ----- (phantom)
                (5)  ................................... (dot)
                (6)  -- . -- . -- . -- . -- . -- . -- .  (dashdot)
                (7)  -- -- . -- -- . -- -- . -- -- . --  (border)
                (8)  -- .. -- .. -- .. -- .. -- .. -- .. (divide)
        
   Pre: linecode contains the code for the desired line type.
        pltfp is a pointer to a PLTFILE that has been opened for writing.
        The global variable 'drawing' indicates whether a draw is currently
        in progress.

   Post: The char array of codes necessary to cause the plotter to begin 
         using a different line type is output via putarr().
         If linecode is out of range, BADFMT is returned. 
         If the global variable 'drawing' is TRUE, it is set FALSE and the
         array begins with the HP_END_DRAW code.
         The array always contains the HP_LINETYPE code followed by the
         SBN and MBN representations of the code required to set the desired 
         line type.
         If putarr() returns an error, BADIO is returned.
         Otherwise TRUE is returned.

         If the module is compiled with VARIABLE_DASHES defined, patterns
         in the lines will be varied so that an integral number of patterns
         will be included in each line segment.  If VARIABLE_DASHES is not
         defined, fixed length patterns will be used without regard for
         the completeness of each pattern. (ie short line segments may not
         include a complete pattern)
*/

int dev_linetype (linecode, pltfp)
   unsigned linecode;
   PLTFILE *pltfp;
{
   extern drawing;
   char *codestr;
   int codelen;

   switch (linecode) {
      case 0:
         codestr=LT_SOLID;
         codelen=LTS_BYTES;
         break;
      case 1:
         codestr=LT_DASHED;
         codelen=LTDA_BYTES;
         break;
      case 2:
         codestr=LT_HIDDEN;
         codelen=LTH_BYTES;
         break;
      case 3:
         codestr=LT_CENTER;
         codelen=LTC_BYTES;
         break;
      case 4:
         codestr=LT_PHANTOM;
         codelen=LTP_BYTES;
         break;
      case 5:
         codestr=LT_DOT;
         codelen=LTDO_BYTES;
         break;
      case 6:
         codestr=LT_DASHDOT;
         codelen=LTDD_BYTES;
         break;
      case 7:
         codestr=LT_BORDER;
         codelen=LTB_BYTES;
         break;
      case 8:
         codestr=LT_DIVIDE;
         codelen=LTDI_BYTES;
         break;
      default:
         return (BADFMT);
         break;
   }

   if (drawing) {
      drawing = FALSE;
      if (!putarr (pltfp,HPED_BYTES,HP_END_DRAW))
         return (BADIO);
   }
   if (!putarr(pltfp,HPLT_BYTES,HP_LINETYPE))
      return (BADIO);
   if (!putarr (pltfp,codelen,codestr))
      return (BADIO);
   return (TRUE);
}



/*
   Function: dev_penchange
   Purpose:  Produces the code needed to raise the pen so that the main
             program can pause and allow the operator to manually change
             pens.

   Pre: pltfp is a pointer to a PLTFILE that has been opened for writing.
        The global variable 'drawing' indicates whether a draw is currently
        in progress.

   Post: The char array of codes needed to cause the plotter to raise the 
         pen is output via putarr().
         If the global variable 'drawing' is TRUE, it is set FALSE and the
         array begins with the HP_END_DRAW code.  
         The array always contains the HP_PENCHANGE code.
         If putarr() returns an error, BADIO is returned.
         Otherwise, TRUE is returned.
*/

int dev_penchange (pltfp)
   PLTFILE *pltfp;
{
   extern drawing;

   if (drawing) {
      drawing = FALSE;
      if (!putarr (pltfp,HPED_BYTES,HP_END_DRAW))
         return (BADIO);
   }
   if (!putarr (pltfp,HPPC_BYTES,HP_PENCHANGE))
      return (BADIO);
   return (TRUE);
}



/*
   Function: dev_abortplot
   Purpose:  Produces the code needed to cause the plotter to abort all
             actions related to the current plot.

   Pre: pltfp is a pointer to a PLTFILE that has been opened for writing.

   Post: The char array of codes needed to cause the plotter to abort the 
         current plot is output via putarr().
         The array contains the HP_ABORTPLOT code.
         If putarr() returns an error, BADIO is returned.
         Otherwise, TRUE is returned.
*/

int dev_abortplot (pltfp)
   PLTFILE *pltfp;
{
   if (!putarr(pltfp,HPAP_BYTES,HP_ABORTPLOT))
      return (BADIO);
   return (TRUE);
}
