WITH Ada.Text_IO;
WITH Ada.Integer_Text_IO;
WITH Screen;
PROCEDURE Histogram IS
------------------------------------------------------------------
--| Plots a histogram on the screen consisting of vertical bars.
--| Each bar represents the frequency of occurrence of a given
--| alphabet letter in the input file.
--| The input file is assumed to be Standard_Input; 
--| use input redirection if you wish to use a disk file instead.
--| Author: Michael B. Feldman, The George Washington University 
--| Last Modified: November 1995                                     
------------------------------------------------------------------
 
  SUBTYPE UpperCase IS Character RANGE 'A'..'Z';
  SUBTYPE LowerCase IS Character RANGE 'a'..'z';
  TYPE Occurrences IS ARRAY(Character RANGE <>) OF integer;
  Uppers   : Occurrences(UpperCase);
  Lowers   : Occurrences(LowerCase);

  NextCh   : Character;
  Scale    : Natural;
  MaxCount : Natural := 0;
  WhichCol : Screen.Width;
 

  PROCEDURE Plot(WhichCol  : Screen.Width; 
                 BottomRow : Screen.Depth;
                 HowMany   : Screen.Depth; 
                 WhichChar : Character) IS

  -- draws one vertical bar on the screen
  -- Pre: WhichCol, BottomRow, HowMany, and WhichChar are defined
  -- Post: draws a bar in column WhichCol, using character WhichChar
  --       to do the plotting. The bottom of the bar is given by
  --       BottomRow; the bar contains HowMany characters.

  BEGIN -- Plot

    FOR Count IN 0 .. Howmany - 1 LOOP

      Screen.MoveCursor(Column => WhichCol, Row => BottomRow - Count);
      Ada.Text_IO.Put(Item => WhichChar);

    END LOOP;

  END Plot;
 
BEGIN -- Histogram

  -- initialize letter-counter arrays
  Uppers := (OTHERS => 0);
  Lowers := (OTHERS => 0);

  -- read each character in the file; update letter counters
  WHILE NOT Ada.Text_IO.End_Of_File LOOP
    WHILE NOT Ada.Text_IO.End_Of_Line LOOP

      Ada.Text_IO.Get(NextCh);
      CASE NextCh IS
	WHEN UpperCase =>
	  Uppers(NextCh) := Uppers(NextCh) + 1;
          IF Uppers(NextCh) > MaxCount THEN
            MaxCount := Uppers(NextCh);
          END IF;
	WHEN LowerCase =>
	  Lowers(NextCh) := Lowers(NextCh) + 1;
          IF Lowers(NextCh) > MaxCount THEN
            MaxCount := Lowers(NextCh);
          END IF;
	WHEN OTHERS =>
	  NULL;
      END CASE;

    END LOOP;
    Ada.Text_IO.Skip_Line;
  END LOOP;
 
  Scale := MaxCount / 20 + 1;
 
  Screen.ClearScreen;
  Screen.MoveCursor(Row => 1, Column => 15);
  Ada.Text_IO.Put(Item => "Scale: 1 star = ");
  Ada.Integer_Text_IO.Put(Item => Scale, Width => 1);
  Ada.Text_IO.Put(Item => " occurrences");
  Screen.MoveCursor(Row => 22, Column => 4);
  Ada.Text_IO.Put
    (Item => "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ");
  WhichCol := 4;

  FOR C IN LowerCase LOOP

    IF Lowers(C) /= 0 THEN
      Plot(WhichCol, 21, Lowers(C) / scale + 1, '*');
    END IF;
    WhichCol := WhichCol + 1;

  END LOOP;

  FOR C IN UpperCase LOOP

    IF Uppers(C) /= 0 THEN
      Plot(WhichCol, 21, Uppers(C) / scale + 1, '*');
    END IF;
    WhichCol := WhichCol + 1;

  END LOOP;

  Screen.MoveCursor(Row => 24, Column => 1);
  
END Histogram;
