%TITLE "Memory-Mapped Video -- Copyright (c) 1989,1995 by Tom Swan"


;-----  NOTE: You must call ScInit before calling other routines
;             in this package!


        IDEAL
        MODEL   small

MaxRow          EQU     25      ; Maximum number of display rows
MaxCol          EQU     80      ; Maximum number of display columns
MonoBASE        EQU     0b000h  ; Monochrome RAM segment address
DefaultBASE     EQU     0b800h  ; Other mode RAM segment address


;-----  Character attribute byte & AND masks

RECORD attrByte Blink:1, Background:3, Intensity:1, Foreground:3

BlinkMask       EQU     MASK    Blink
BackMask        EQU     MASK    Background
IntensityMask   EQU     MASK    Intensity
ForeMask        EQU     MASK    Foreground



        DATASEG

attribute       attrByte <0,0,7>        ; Attribute, default values
vBASE           DW      DefaultBASE     ; Video RAM buffer address


;-----  ScRow: Array of offsets (from vBASE) in video RAM buffer

BytesPerRow = MaxCol * 2
row = 0
LABEL   ScRow   Word
REPT    MaxRow
 DW ( row * BytesPerRow )
 row = row + 1
ENDM

        CODESEG

        PUBLIC  ScGotoXY, ScReadXY, ScPokeChar, ScPokeStr, ScClrRect
        PUBLIC  ScSetBack, ScSetFore, ScBright, ScDim, ScBlink
        PUBLIC  ScNoBlink, ScGetAttribute, ScSetAttribute, ScInit

%NEWPAGE
;---------------------------------------------------------------
; SetVidAddr    Prepare video-RAM address
;---------------------------------------------------------------
; Note:
;       Private subroutine for ScPokeChar and ScPokeStr
; Input:
;       dh = row (0 is top line)
;       dl = column (0 is at far left)
; Output:
;       es:di = video RAM buffer address for (row, column)
;       Note: dh and dl are not checked!!
; Registers:
;       bx, dx, di, es changed
;---------------------------------------------------------------
PROC    SetVidAddr
        mov     es, [vBASE]     ; Set es to video segment address
        xor     bh, bh          ; Zero upper half of bx
        mov     bl, dh          ; Assign row to bl
        shl     bx, 1           ; Multiply row (bx) times 2
        mov     di, [scRow+bx]  ; Set di to video buffer row address
        xor     dh, dh          ; Convert column to 16-bit word
        shl     dx, 1           ; Multiply column (dx) times 2
        add     di, dx          ; Add column offset to row address
        ret                     ; Return to caller
ENDP    SetVidAddr
%NEWPAGE
;---------------------------------------------------------------
; ScGotoXY      Set cursor position
;---------------------------------------------------------------
; Input:
;       dh = row (0 is top line)
;       dl = column (0 is at far left)
; Output:
;       Cursor in current page repositioned to (row, column)
; Registers:
;       none
;---------------------------------------------------------------
PROC    ScGotoXY
        push    ax              ; Save modified registers
        push    bx
        mov     ah, 15          ; Get display page number into bh
        int     10h             ; Call BIOS video service
        mov     ah, 2           ; BIOS function number
        int     10h             ; Call BIOS--set cursor position
        pop     bx              ; Restore registers
        pop     ax
        ret                     ; Return to caller
ENDP    ScGotoXY
%NEWPAGE
;---------------------------------------------------------------
; ScReadXY      Get cursor position
;---------------------------------------------------------------
; Input:
;       none
; Output:
;       dh = row (0 is top line)
;       dl = column (0 is at far left)
; Registers:
;       dx changed
;---------------------------------------------------------------
PROC    ScReadXY
        push    ax              ; Save modified registers
        push    bx
        push    cx
        mov     ah, 15          ; Get display page number into bh
        int     10h             ; Call BIOS video service
        mov     ah, 3           ; BIOS function number
        int     10h             ; Call BIOS--get cursor position
        pop     cx              ; Restore registers
        pop     bx
        pop     ax
        ret                     ; Return to caller
ENDP    ScReadXY
%NEWPAGE
;---------------------------------------------------------------
; ScPokeChar    Poke a character into the display
;---------------------------------------------------------------
; Input:
;       al = ASCII character code
;       dh = row (0 is top line) *
;       dl = column (0 is at far left) *
; Output:
;       Character in al displayed at position (row, column)
;     * Note: Row and Column values not checked!!
; Registers:
;       ax, bx, dx, di changed
;---------------------------------------------------------------
PROC    ScPokeChar
        push    es              ; Save es segment register
        call    SetVidAddr      ; Prepare es:di
        mov     ah, [attribute] ; Assign attribute to ah
        stosw                   ; Display attribute and char
        pop     es              ; Restore es register
        ret                     ; Return to caller
ENDP    ScPokeChar
%NEWPAGE
;---------------------------------------------------------------
; ScPokeStr     Poke a string into the display
;---------------------------------------------------------------
; Input:
;       cx = number of characters to write
;       dh = row (0 is top line) *
;       dl = column (0 is at far left) *
;       ds:si = address of ASCII string (any format)
; Output:
;     * Note: Row and Column values not checked!!
;       Note: Any string terminator is ignored
; Registers:
;       ax, bx, cx, dx, di, si changed
;---------------------------------------------------------------
PROC    ScPokeStr
        push    es              ; Save es segment address
        call    SetVidAddr      ; Prepare es:di
        mov     ah, [attribute] ; Assign attribute to ah
        cld                     ; Auto-increment si, di
@@10:
        lodsb                   ; Get next char into al
        stosw                   ; Display attribute and char
        loop    @@10            ; Loop on cx
        pop     es              ; Restore es segment address
        ret                     ; Return to caller
ENDP    ScPokeStr
%NEWPAGE
;---------------------------------------------------------------
; ScClrRect     Clear rectangular area on display
;---------------------------------------------------------------
; Input:
;       ch, cl = row & column of upper left corner
;       dh, dl = row & column of lower left corner
; Output:
;       Rectangle defined by ch,cl & dh,dl cleared
;        to current attributes
; Registers:
;       ax
;---------------------------------------------------------------
PROC    ScClrRect
        mov     ah, 6                   ; Select BIOS scroll routine
        mov     al, 0                   ; Tells routine to clear area
        mov     bh, [attribute]         ; Get attribute to use
        int     10h                     ; Call BIOS video service
        ret                             ; Return to caller
ENDP    ScClrRect
%NEWPAGE
;---------------------------------------------------------------
; ScSetBack     Set background color (attribute)
;---------------------------------------------------------------
; Input:
;       al = background color
; Output:
;       Background color set for ScPokeChar and ScPokeStr
; Registers:
;       al
;---------------------------------------------------------------
PROC    ScSetBack
IF Background GT 0
        push    cx                      ; If background not in lsbs
        mov     cl, Background          ;  then shift bits into 
        shl     al, cl                  ;  position for ORing into
        pop     cx                      ;  attribute byte
ENDIF
        and     al,BackMask             ; Isolate bits in al
        and     [attribute], NOT BackMask ; Zero background bits
        or      [attribute], al         ; Add background to attribute
        ret                             ; Return to caller
ENDP    ScSetBack
%NEWPAGE
;---------------------------------------------------------------
; ScSetFore     Set foreground color
;---------------------------------------------------------------
; Input:
;       al = foreground color
; Output:
;       Foreground color set for ScPokeChar and ScPokeStr
; Registers:
;       al
;---------------------------------------------------------------
PROC    ScSetFore
IF Foreground GT 0
        push    cx                      ; If foreground not in lsbs
        mov     cl, Foreground          ;  then shift bits into
        shl     al, cl                  ;  position for ORing into
        pop     cx                      ;  attribute byte
ENDIF
        and     al, ForeMask            ; Isolate bits in al
        and     [attribute], NOT ForeMask ; Zero foreground bits
        or      [attribute], al         ; Add foreground to attribute
        ret                             ; Return to caller
ENDP    ScSetFore
%NEWPAGE
;---------------------------------------------------------------
; ScBright      Turn on intensity bit
; ScDim         Turn off intensity bit
; ScBlink       Turn on blink bit
; ScNoBlink     Turn off blink bit
;---------------------------------------------------------------
; Input:
;       none
; Output:
;       Attribute's intensity & blink bits modified
; Registers:
;       none
;---------------------------------------------------------------
PROC    ScBright
        or      [attribute], IntensityMask
        ret
ENDP    ScBright

PROC    ScDim
        and     [attribute], NOT IntensityMask
        ret
ENDP    ScDim

PROC    ScBlink
        or      [attribute], BlinkMask
        ret
ENDP    ScBlink

PROC    ScNoBlink
        and     [attribute], NOT BlinkMask
        ret
ENDP    ScNoBlink
%NEWPAGE
;---------------------------------------------------------------
; ScGetAttribute        Get current attribute value
;---------------------------------------------------------------
; Input:
;       none
; Output:
;       dl = current attribute value
; Registers:
;       dl
;---------------------------------------------------------------
PROC    ScGetAttribute
        mov     dl, [attribute]         ; Get attribute byte
        ret                             ; Return to caller
ENDP    ScGetAttribute
%NEWPAGE
;---------------------------------------------------------------
; ScSetAttribute        Change attribuyte value
;---------------------------------------------------------------
; Input:
;       al = new attribute value
; Output:
;       none: attribute stored for later use
; Registers:
;       none
;---------------------------------------------------------------
PROC    ScSetAttribute
        mov     [attribute], al         ; Set attribute byte
        ret                             ; Return to caller
ENDP    ScSetAttribute
%NEWPAGE
;---------------------------------------------------------------
; ScInit        Initialize SCREEN package
;---------------------------------------------------------------
; Input:
;       none
; Output:
;       vBASE initialized
; Registers:
;       none
;---------------------------------------------------------------
PROC    ScInit
        push    ax                      ; Save modified registers
        push    bx
        mov     ah, 15                  ; BIOS function number
        int     10h                     ; Get video mode in al
        cmp     al, 7                   ; Is mode monochrome?
        jne     @@10                    ; If no, jump
        mov     [vBASE], MonoBASE       ; Assign monochrome address
@@10:
        pop     bx                      ; Restore registers
        pop     ax
        ret                             ; Return to caller
ENDP    ScInit

        END                     ; End of module
