%TITLE "Binary Coded Decimals (BCD) -- by Tom Swan"

        IDEAL

        MODEL   small


;-----  Equates

ASCIINull       EQU     0       ; ASCII end-of-string null character
PackedSize      EQU     10      ; Bytes in a packed BCD value
UnpackedSize    EQU     20      ; Bytes in an unpacked BCD value

;-----  note: PackedSize must be even!


        UDATASEG

TempUPBCD       DT      ?, ?    ; Unpacked BCD word space (20 bytes)


        CODESEG

        PUBLIC  BCDAdd, BCDSubtract, PackedToUnpacked
        PUBLIC  UnpackedToPacked, BCDToASCII, BCDCopy

%NEWPAGE
;---------------------------------------------------------------
; BCDAdd                Add two packed BCD numbers
;---------------------------------------------------------------
; Input:
;       si = address of source BCD value (10 bytes)
;       di = address of destination BCD value (10 bytes)
; Output:
;       destinationBCD <- destinationBCD + sourceBCD
;       cf = 0 : No error
;       cf = 1 : Overflow error occurred
; Registers:
;       none
;---------------------------------------------------------------
PROC    BCDAdd
        push    ax                      ; Save modified registers
        push    cx
        push    di
        push    si

        cld                             ; Auto-increment si & di
        clc                             ; Clear carry for 1st adc
        mov     cx, PackedSize          ; Assign loop count to cx
@@10:
        lodsb                           ; Get two digits of source
        adc     al, [byte di]           ; Add two digits of dest + cf
        daa                             ; Adjust to packed BCD format
        stosb                           ; Store result in destination
        loop    @@10                    ; Loop until done (cx = 0)

        pop     si                      ; Restore saved registers
        pop     di
        pop     cx
        pop     ax
        ret                             ; Return to caller
ENDP    BCDAdd
%NEWPAGE
;---------------------------------------------------------------
; BCDSubtract           Subtract two packed BCD numbers
;---------------------------------------------------------------
; Input:
;       si = address of source BCD value (10 bytes)
;       di = address of destination BCD value (10 bytes)
; Output:
;       destinationBCD <- destinationBCD - sourceBCD
;       cf = 0 : No error
;       cf = 1 : Underflow error occurred
; Registers:
;       none
;---------------------------------------------------------------
PROC    BCDSubtract
        push    ax                      ; Save modified registers
        push    cx
        push    di
        push    si

        cld                             ; Auto-increment si & di
        clc                             ; Clear carry for 1st sbb
        mov     cx, PackedSize          ; Assign loop count to cx
@@10:
        lodsb                           ; Get two digits of source
        sbb     [byte di], al           ; dest <- dest - source bytes
        mov     al, [byte di]           ; Load binary result into al
        das                             ; Adjust to packed BCD format
        stosb                           ; Store result in destination
        loop    @@10                    ; Loop until done (cx = 0)

        pop     si                      ; Restore saved registers
        pop     di
        pop     cx
        pop     ax
        ret                             ; Return to caller
ENDP    BCDSubtract
%NEWPAGE
;---------------------------------------------------------------
; PackedToUnpacked      Convert packed BCD to unpacked BCD
;---------------------------------------------------------------
; Input:
;       si = address of source packed BCD value (10 bytes)
;       di = address of destination unpacked BCD value (20 bytes)
; Output:
;       destinationBCD <- unpacked( sourceBCD )
; Registers:
;       none
;---------------------------------------------------------------
PROC    PackedToUnpacked
        push    ax                      ; Save modified registers
        push    cx
        push    di
        push    si

        cld                             ; Auto-increment si & di
        mov     cx, PackedSize          ; Assign loop count to cx
@@10:
        lodsb                           ; Get two digits of source
        mov     ah, al                  ; Copy digits from al to ah
        shr     ah, 1                   ; Shift upper digit to 
        shr     ah, 1                   ;  lower 4 bits of ah
        shr     ah, 1
        shr     ah, 1
        and     al, 0Fh                 ; Mask upper digit from al
        stosw                           ; Store ax to destination
        loop    @@10                    ; Loop until done (cx = 0)

        pop     si                      ; Restore saved registers
        pop     di
        pop     cx
        pop     ax
        ret                             ; Return to caller
ENDP    PackedToUnpacked
%NEWPAGE
;---------------------------------------------------------------
; UnpackedToPacked      Convert unpacked BCD to packed BCD
;---------------------------------------------------------------
; Input:
;       si = address of source unpacked BCD value (20 bytes)
;       di = address of destination packed BCD value (10 bytes)
; Output:
;       destinationBCD <- packed( sourceBCD )
; Registers:
;---------------------------------------------------------------
PROC    UnpackedToPacked
        push    ax                      ; Save modified registers
        push    cx
        push    di
        push    si

        cld                             ; Auto-increment si & di
        mov     cx, PackedSize          ; Assign loop count to cx
@@10:
        lodsw                           ; Get two digits of source
        shl     ah, 1                   ; Shift digit to
        shl     ah, 1                   ;  upper 4 bits of ah
        shl     ah, 1
        shl     ah, 1
        or      al, ah                  ; Pack 2 digits into al
        stosb                           ; Store al to destination
        loop    @@10                    ; Loop until done (cx = 0)

        pop     si                      ; Restore saved registers
        pop     di
        pop     cx
        pop     ax
        ret                             ; Return to caller
ENDP    UnpackedToPacked
%NEWPAGE
;---------------------------------------------------------------
; BCDToASCII            Convert packed BCD value to ASCII
;---------------------------------------------------------------
; Input:
;       si = address of source packed BCD value (10 bytes)
;       di = address of destination ASCIIZ string (21 bytes)
; Output:
;       ASCIIZ <- ASCII( sourceBCD) + null character
; Registers:
;       none
;---------------------------------------------------------------
PROC    BCDToASCII
        push    ax                      ; Save modified registers
        push    cx
        push    di
        push    si

        push    di                      ; Save destination address
        mov     di, offset TempUPBCD    ; Use temporary work area
        call    PackedToUnpacked        ; Unpack source to temp
        pop     di                      ; Restore destination address

;-----  Address last word of temporary work space
        mov     si, offset TempUPBCD + UnpackedSize - 2

        mov     cx, PackedSize          ; Assign loop count to cx
@@10:
        std                             ; Auto-decrement si
        lodsw                           ; Get two digits into ax
        or      ax, 03030h              ; Convert to ASCII
        xchg    ah, al                  ; Swap characters
        cld                             ; Auto-increment di
        stosw                           ; Store chars in destination
        loop    @@10                    ; Loop until done (cx = 0)
        mov     [byte di], ASCIINull    ; Store end-of-string marker

        pop     si                      ; Restore saved registers
        pop     di
        pop     cx
        pop     ax
        ret                             ; Return to caller
ENDP    BCDToASCII
%NEWPAGE
;---------------------------------------------------------------
; BCDCopy               Copy a packed BCD value
;---------------------------------------------------------------
; Input:
;       si = address of source BCD value (10 bytes)
;       di = address of destination BCD value (10 bytes)
; Output:
;       destinationBCD <- sourceBCD
; Registers:
;       none
;---------------------------------------------------------------
PROC    BCDCopy
        push    cx                      ; Save modified registers
        push    di
        push    si

        cld                             ; Auto-increment si & di
        mov     cx, PackedSize/2        ; Assign loop count to cx
        rep     movsw                   ; Copy using word moves

        pop     si                      ; Restore saved registers
        pop     di
        pop     cx
        ret                             ; Return to caller
ENDP    BCDCopy

        END                     ; End of BCD module
