%TITLE "Asynch Serial Comm Module -- by Tom Swan"

        IDEAL

        MODEL   small

        PUBLIC  ComPort

ComPort = 0             ; 0 = COM1:, 1 = COM2:

IF ComPort EQ 0
        Port            EQU     03F8h   ; 8250 base address
        VectorNum       EQU     0Ch     ; Interrupt vector number
        EnableIRQ       EQU     0EFh    ; Mask to enable 8259 IRQ
        DisableIRQ      EQU     10h     ; Mask to disable 8259 IRQ

ELSEIF ComPort EQ 1
        Port            EQU     02F8h   ; same comments as above
        VectorNum       EQU     0Bh
        EnableIRQ       EQU     0F7h
        DisableIRQ      EQU     08h
ELSE
        DISPLAY "ComPort must be 0 or 1"
        ERR
ENDIF

;-----  Adapter register addresses

TxRegister      =       Port + 0        ; Transmit Register
RxRegister      =       Port + 0        ; Receive Register
IntEnable       =       Port + 1        ; Interrupt Enable Register
IntIdent        =       Port + 2        ; Interrupt Identification
LineControl     =       Port + 3        ; Line Control Register
ModemControl    =       Port + 4        ; Modem Control Register
LineStatus      =       Port + 5        ; Line Status Register
ModemStatus     =       Port + 6        ; Modem Status Register

;-----  Other equates

Ctrl8259_0      EQU     020h            ; 8259 port
Ctrl8259_1      EQU     021h            ; 8259 port (masks)
EOI             EQU     020h            ; 8259 end-of-interrupt
BufSize         EQU     2048            ; Size of input buffer


        DATASEG

vectorSeg       DW      ?               ; Old vector segment
vectorOfs       DW      ?               ; Old vector offset
bufHead         DW      ?               ; Buffer head pointer
bufTail         DW      ?               ; Buffer tail pointer
buffer          DB      BufSize DUP (?) ; Input buffer


        CODESEG

        PUBLIC  AsynchInit, AsynchStop, AsynchStat
        PUBLIC  AsynchOut, AsynchIn, AsynchInStat

%NEWPAGE
;---------------------------------------------------------------
; EmptyBuffer           Empty the input buffer
;---------------------------------------------------------------
; Note:
;       Private to module
; Input:
;       none
; Output:
;       none
; Registers:
;       none
;---------------------------------------------------------------
PROC    EmptyBuffer
        cli                             ; Prevent interrupts
        push    ax                      ; Save ax
        mov     ax, offset buffer       ; Buffer is empty when
        mov     [bufHead], ax           ;  the head and tail pointers
        mov     [bufTail], ax           ;  are equal
        pop     ax                      ; Restore ax
        sti                             ; Enable interrupts
        ret                             ; Return to caller
ENDP    EmptyBuffer
%NEWPAGE
;---------------------------------------------------------------
; AsynchInit            Initialize serial port and install ISR
;---------------------------------------------------------------
; Input:
;       none
; Output:
;       none
;
;       NOTE: Precede (usually) with call to int 14h to
;       set baud rate
;
;       NOTE: Interrupt-driven input begins immediately
;       upon exit from this routine.
;
;       WARNING: You must call AsynchStop before your
;       program ends to avoid a system crash!
;
; Registers:
;       ax, bx, dx
;---------------------------------------------------------------
PROC    AsynchInit

        call    EmptyBuffer             ; Initialize buffer

;-----  Save and reassign interrupt vector

        push    ds                      ; Save segment registers
        push    es
        mov     ax, 3500h + VectorNum   ; Get vector address
        int     21h                     ; Call DOS
        mov     [vectorSeg], es         ; Save segment address
        mov     [vectorOfs], bx         ; Save offset address
        push    cs                      ; Address AsynchISR
        pop     ds                      ;  with ds:dx, and call
        mov     dx, offset AsynchISR    ;  DOS function 25h to 
        mov     ax, 2500h + VectorNum   ;  set the new vector
        int     21h                     ;  address.
        pop     es                      ; Restore saved registers
        pop     ds

;-----  Enable 8259 interrupt (IRQ) line for this asynch adapter

        in      al, Ctrl8259_1          ; Read 8259 enable masks
        and     al, EnableIRQ           ; Clear masked bit
        out     Ctrl8259_1, al          ; Write new 8259 masks

;-----  Enable 8250 interrupt-on-data-ready

        mov     dx, LineControl         ; First, read the line control
        in      al, dx                  ;  register, and clear bit
        and     al, 07Fh                ;  7, the Divisor Latch Access
        out     dx, al                  ;  Bit, or DLAB
        mov     dx, IntEnable           ; With DLAB=0, set bit 0 of
        mov     al, 1                   ;  interrupt enable register
        out     dx, al                  ;  to 1, enabling interrupt

;-----  Clear 8250 status and data registers

@@10:
        mov     dx, RxRegister          ; Clear data register
        in      al, dx                  ;  by reading port
        mov     dx, LineStatus          ; Clear line status
        in      al, dx                  ;  by reading port
        mov     dx, ModemStatus         ; Clear modem status
        in      al, dx                  ;  by reading port
        mov     dx, IntIdent            ; Check interrupt ident
        in      al, dx                  ;  register
        test    al, 1                   ; Bit 1 should be 1
        jz      @@10                    ; Jump if interrupt pending

;-----  Set bit 3 of modem control register

        mov     dx, ModemControl        ; Interrupts will be
        in      al, dx                  ;  acknowledged as soon as
        or      al, 08h                 ;  this bit is set to 1
        out     dx, al                  ; Done!

;-----  Empty input buffer again, just in case a stray character
;       managed to squeak in

        call    EmptyBuffer             ; Empty buffer again

        ret                             ; Return to caller
ENDP    AsynchInit
%NEWPAGE
;---------------------------------------------------------------
; AsynchStop            Uninstall Asynch ISR
;---------------------------------------------------------------
; Input:
;       none
; Output:
;       none
;
;       WARNING: Always call AsynchStop before your program
;       ends or a system crash is inevitable!
;
; Registers:
;       al, dx
;---------------------------------------------------------------
PROC    AsynchStop

;-----  Mask (disable) 8259 IRQ interrupt

        in      al, Ctrl8259_1          ; Read 8259 masks
        or      al, DisableIRQ          ; Mask IRQ bit
        out     Ctrl8259_1, al          ; Write new masks

;-----  Disable 8250 interrupt

        mov     dx, LineControl         ; First, read the line control
        in      al, dx                  ;  register, and clear bit
        and     al, 07Fh                ;  7, the Divisor Latch Access
        out     dx, al                  ;  Bit, or DLAB
        mov     dx, IntEnable           ; With DLAB=0, clear all bits
        xor     al, al                  ;  to disable interrupts
        out     dx, al                  ; Write new register value

;-----  Set bit 3 in modem control register to 0

        mov     dx, ModemControl        ; Assign port address
        in      al, dx                  ; Get current register
        and     al, 0F7h                ; Clear bit 3
        out     dx, al                  ; Output new register value

;-----  Interrupts are disabled.  Restore saved interrupt vector.

        push    ds                      ; Save segment register
        mov     ax, 2500h + VectorNum   ; Set interrupt vector
        mov     dx, [vectorOfs]         ; Get saved offset
        mov     ds, [vectorSeg]         ; Get saved segment
        int     21h                     ; Set interrupt vector
        pop     ds                      ; Restore saved register

        ret                             ; Return to caller
ENDP    AsynchStop
%NEWPAGE
;---------------------------------------------------------------
; AsynchStat            Get status for output
;---------------------------------------------------------------
; Input:
;       none
; Output:
;       ah = line status
;       al = modem status
; Registers:
;       ax, dx
;---------------------------------------------------------------
PROC    AsynchStat
        mov     ah, 3                   ; Get-status function number
        mov     dx, ComPort             ; 0=COM1:, 1=COM2:
        int     14h                     ; Call BIOS RS232_IO service
        ret                             ; Return to caller
ENDP    AsynchStat
%NEWPAGE
;---------------------------------------------------------------
; AsynchOut             Output a byte (to output port)
;---------------------------------------------------------------
; Input:
;       al = character (or byte) to output
; Output:
;       none
; Registers:
;       none
;---------------------------------------------------------------
PROC    AsynchOut
        push    dx                      ; Save modified dx
        push    ax                      ; Save char in al
@@10:
        mov     dx, LineStatus          ; Address Line Status Register
        in      al, dx                  ; Get line status
        and     al, 020h                ; Isolate Trasmit Holding Reg.
        jz      @@10                    ; Jump if THRE is not empty
        pop     ax                      ; Restore character
        mov     dx, TxRegister          ; Address transmit register
        out     dx, al                  ; Output char in al
        pop     dx                      ; Restore saved dx
        ret                             ; Return to caller
ENDP    AsynchOut
%NEWPAGE
;---------------------------------------------------------------
; AsynchIn              Input a byte (from buffer)
;---------------------------------------------------------------
; Input:
;       none
; Output:
;       al = char from buffer
;
;       Note: if buffer is empty, al will be zero, with
;       no indication that this is not an input value.
;       Precede with call to AsynchInStat to avoid reads
;       from an empty buffer.
;
; Registers:
;       al, bx
;---------------------------------------------------------------
PROC    AsynchIn
        xor     al, al                  ; Preset result to null
        mov     bx, [bufTail]           ; Get tail pointer
        cmp     bx, [bufHead]           ; Test if buffer is empty
        je      @@99                    ; Exit if empty (al=0)
        mov     al, [byte ptr bx]       ; Else read char from buffer
        inc     [bufTail]               ; Advance tail pointer
        cmp     [word ptr bufTail], offset buffer + BufSize  ; At end?
        jb      @@99                    ; Jump if not so
        mov     [bufTail], offset buffer ; Else reset tail pointer
@@99:
        ret                             ; Return to caller
ENDP    AsynchIn
%NEWPAGE
;---------------------------------------------------------------
; AsynchInStat          Get status of input buffer
;---------------------------------------------------------------
; Input:
;       none
; Output:
;       dx = number of bytes (or chars) in buffer
; Registers:
;       dx
;---------------------------------------------------------------
PROC    AsynchInStat
        mov     dx, [bufHead]           ; Get head pointer
        sub     dx, [bufTail]           ; Subtract tail from head
        jge     @@99                    ; Jump if result >= 0
        add     dx, BufSize             ; Handle negative result
@@99:
        ret                             ; Return to caller
ENDP    AsynchInStat
%NEWPAGE
;---------------------------------------------------------------
; AsynchISR     Asynchronous input interrupt service routine
;---------------------------------------------------------------
; Input:
;       none
; Output:
;       none (char read and deposited in buffer)
;
;       NOTE: This version ignores buffer overflows
;
; Registers:
;       none
;---------------------------------------------------------------
PROC    AsynchISR
        push    ax                      ; Save modified registers
        push    bx
        push    ds
        push    dx

        mov     ax, @data               ; Address local data with ds
        mov     ds, ax
        mov     dx, RxRegister          ; dx = Receive Register
        in      al, dx                  ; Read byte from port
        mov     bx, [bufHead]           ; Get head pointer
        mov     [byte ptr bx], al       ; Store byte in buffer
        inc     bx                      ; Advance head pointer
        cmp     bx, offset buffer + BufSize  ; Is ptr at end?
        jb      @@10                    ; Jump if not
        mov     bx, offset buffer       ; Else reset to beginning
@@10:
        cmp     bx, [bufTail]           ; Check for overflow
        jne     @@20                    ; Jump if no overflow
        mov     bx, [bufHead]           ; Cancel pointer advance
@@20:
        mov     [bufHead], bx           ; Save new head pointer
        mov     al, EOI                 ; Issue end-of-interrupt to
        out     Ctrl8259_0, al          ;  8259 port

        pop     dx                      ; Restore saved registers
        pop     ds
        pop     bx
        pop     ax
        iret                            ; Return from interrupt
ENDP    AsynchISR

        END                     ; End of module
