;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; UnIMG v0.01 - Unpack FAT12 disk image file - Open Source
;
; Usage:
;   unimg disk.img
;
; It will create subdir (=filename without extension) and unpack
; full directory structure with all files and subdirs to HD.
;
; Notes:
;   - only FAT12 with DOS like 8.3 names supported
;   - image file need file extension
;     (filename first 8 chars is used by dir name)
;   - use only normal DOS file functions
;
; ToDo:
;   - FAT16
;
; Compile with fasm 1.56+
;
; Version history:
;   0.01 - 2.Feb.2006 by ATV            email: askovuori(at)hotmail.com
;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
use16
org 0100h

struc BOOT_TYPE 			;Boot sector data
{
	.JMP_Bytes		rb 3
	.OEM_ID 		rb 8
	.BytesPerSector 	rw 1
	.SectorsPerCluster	rb 1
	.ReservedSectors	rw 1
	.TotalFATs		rb 1
	.MaxRootEntries 	rw 1
	.TotalSectorsSmall	rw 1
	.MediaDescriptor	rb 1
	.SectorsPerFAT		rw 1
	.SectorsPerTrack	rw 1
	.NumHeads		rw 1
	.HiddenSectors		rw 2
	.TotalSectorsLarge	rw 2
	.DriveNumber		rb 1
	.Flags			rb 1
	.Signature		rb 1
	.VolumeID		rw 2
	.VolumeLabel		rb 11
	.SystemID		rb 8
	.filler 		rb 510-($-.JMP_Bytes)
	.BootSectorID		rw 1
	.SIZEOF = $-.JMP_Bytes
}
virtual at 0
boot_type BOOT_TYPE
end virtual

struc DIR_TYPE				;32 bytes
{
	.Name			rb 11
	.Attribute		rb 1
	.Reserved1		rb 10
	.Time			rw 1
	.Date			rw 1
	.FirstCluster		rw 1
	.Size			rd 1
	.SIZEOF = $-.Name
}
virtual at 0
DIR_TYPE DIR_TYPE
end virtual

start:
	cld
	call	get_parameters
	mov	dx,txt_usage
	jz	.show_error

	mov	dx,img_name
	mov	ax,3d00h		;DOS - Open file (readonly)
	int	21h
	mov	dx,error_open
	jc	.show_error
	mov	[img_handle],ax
	xchg	bx,ax

	call	check_img
	jc	.show_error_close

	mov	dx,entry_name
	mov	ax,0			;0 = rootdir
	call	copy_dir
	jc	.show_error_close

	mov	bx,[img_handle]
	mov	ah,3eh			;DOS - Close file
	int	21h
	mov	ax,4c00h		;DOS - Exit to DOS
	int	21h

    .show_error_close:
	mov	bx,[img_handle]
	mov	ah,3eh			;DOS - Close file
	int	21h

    .show_error:
	mov	ah,09h			;DOS - Display string
	int	21h
	mov	ax,4c01h		;DOS - Exit to DOS
	int	21h

;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; Procedure get_parameters
; Param:  none
; Return: ZF set if empty name
; Change: ax,bx,cx,di,si
;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
get_parameters:
	mov	si,0081h
	mov	di,img_name
	mov	bx,entry_name
	xor	cx,cx
	or	cl,[si-1]
	jz	.name_end

    .skip_spc:
	lodsb
	cmp	al,20h
	ja	.first_char
	loop	.skip_spc
	jmp	.name_end

    .new_char:
	lodsb
	cmp	al,20h
	jbe	.name_end

    .first_char:
	stosb
	cmp	al,'.'
	jne	.not_dir_end
	mov	al,0

    .not_dir_end:
	mov	[bx],al
	inc	bx
	cmp	al,':'
	jz	.is_path
	cmp	al,'\'
	jnz	.not_path

    .is_path:
	mov	bx,entry_name

    .not_path:
	loop	.new_char

    .name_end:
	mov	al,0
	stosb
	mov	[bx],al
	cmp	byte [entry_name],0
	ret

;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; Procedure check_img
; Param:  none
; Return: CF set on error (ds:dx -> error message)
; Change: ax,bx,cx,dx
;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
check_img:
	mov	di,bs
	mov	cx,1
	xor	ax,ax
	call	read_sectors
	jc	.img_error
	cmp	[bs.BootSectorID],0aa55h
	jnz	.img_error
	cmp	word [bs.SystemID+3],3231h ;is it FAT12
	jnz	.img_error

	mov	cx,[bs.SectorsPerFAT]
	xor	ax,ax
	mov	al,[bs.TotalFATs]	;number of FATs
	cmp	al,2			;error if more than 2 FATs
	ja	.img_error
	mul	cx			;sectors used by FATs
	test	ax,ax
	jz	.img_error
	mov	[FAT_sectors],ax

	mov	ax,0020h		;32 byte directory entry
	mul	[bs.MaxRootEntries]	;total size of directory
	div	[bs.BytesPerSector]	;sectors used by directory
	mov	[rootdir_size],ax

	mov	ax,[bs.ReservedSectors] ;adjust for bootsector
	mov	[FAT_start],ax

	add	ax,[FAT_sectors]
	mov	[rootdir_start],ax

	add	ax,[rootdir_size]
	mov	[data_start],ax

	mov	ax,[FAT_start]
	mov	di,FAT_table
	call	read_sectors
	ret

   .img_error:
	mov	dx,error_img
	stc
	ret

;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; Procedure copy_dir
; Param:  ds:dx -> ASCIIZ dir name
;         ax - first cluster (0 = rootdir)
; Return: CF set on error (ds:dx -> error message)
; Change: ax,bx,cx,dx,di,si
; Note:   recursive call
;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
copy_dir:
	mov	[dir_cluster],ax
	mov	ah,39h			;DOS - Create directory
	int	21h
	jnc	.md_ok
	mov	dx,error_md
	ret

    .md_ok:
	mov	ah,3bh			;DOS - Change directory
	int	21h
	jnc	.cd_ok
	mov	dx,error_cd
	ret

    .cd_ok:
	mov	ax,[bs.MaxRootEntries]
	cmp	[dir_cluster],0
	jz	.is_rootdir

    .new_cluster:
	mov	al,16			;16 entrys per sector
	mul	[bs.SectorsPerCluster]

    .is_rootdir:
	mov	[entry_count],ax
	mov	[entry_no],0

    .new_entry:
	mov	ax,[dir_cluster]
	mov	bx,[entry_no]
	shr	bx,4			;dir sector = entry_no / 16
	mov	di,dir_buffer
	call	read_data_sector
	jc	.dir_ret
	mov	si,[entry_no]
	and	si,15
	shl	si,5			;entry ofs = (entry_no % 16) * 32
	add	si,dir_buffer

	mov	al,[si]
	cmp	al,0e5h 		;deleted entry
	jz	.next_entry
	cmp	al,'.'			;current dir
	jz	.next_entry
	cmp	al,0			;not used
	jz	.done
	mov	al,[si+DIR_TYPE.Attribute]
	and	al,0fh
	cmp	al,0fh			;skip long filename
	jz	.next_entry
	test	al,08h			;volume label
	jnz	.next_entry

	mov	di,entry_name
	call	get_entry_name
	jz	.bad_name		;no name or unsupported chars
	test	byte [si+DIR_TYPE.Attribute],10h ;directory flag
	jnz	.is_dir

    .is_file:
	mov	dx,entry_name
	call	copy_file
	jnc	.next_entry

    .dir_ret:
	ret

    .is_dir:
	push	[dir_cluster]		;save dir arrays
	push	[entry_no]
	push	[entry_count]
	mov	dx,entry_name
	mov	ax,[si+DIR_TYPE.FirstCluster]
	call	copy_dir		;Note: recursive call
	pop	[entry_count]
	pop	[entry_no]
	pop	[dir_cluster]
	jnc	.next_entry
	ret

    .bad_name:
	mov	dx,error_bad_name
	mov	ah,09h			;DOS - Display string
	int	21h

    .next_entry:
	inc	[entry_no]
	mov	ax,[entry_no]
	cmp	ax,[entry_count]
	jnb	.next_cluster
	jmp	.new_entry

    .next_cluster:
	mov	ax,[dir_cluster]
	test	ax,ax			;is rootdir end?
	jz	.done
	call	get_FAT
	jc	.done			;cluster end
	mov	[dir_cluster],ax
	jmp	.new_cluster

    .done:
	mov	dx,dir_up
	mov	ah,3bh			;DOS - Change directory
	int	21h
	mov	dx,error_cd
	ret

;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; Procedure copy_file
; Param:  ds:dx -> ASCIIZ file name
;         ds:si -> 32 bytes file entry
; Return: CF set on error (ds:dx -> error message)
; Change: ax,bx,cx,dx,di
;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
copy_file:
	xor	cx,cx
	mov	cl,[si+DIR_TYPE.Attribute]
	mov	ah,3ch			;DOS - Create file
	int	21h
	jnc	.create_ok

	mov	dx,error_create
	mov	ah,09h			;DOS - Display string
	int	21h
	clc				;not critical error
	ret

    .create_ok:
	mov	[file_handle],ax
	mov	ax,[si+DIR_TYPE.FirstCluster]
	mov	[file_cluster],ax
	mov	ax,word [si+DIR_TYPE.Size]
	mov	dx,word [si+DIR_TYPE.Size+2]
	mov	word [file_size],ax
	mov	word [file_size+2],dx
	mov	[file_sector],0

    .read_more:
	mov	ax,[file_cluster]
	mov	bx,[file_sector]
	mov	di,file_buffer
	call	read_data_sector
	jc	.file_error

	mov	cx,512
	cmp	word [file_size+2],0
	jnz	.size_ok
	cmp	cx,word [file_size]
	jb	.size_ok
	mov	cx,word [file_size]

    .size_ok:
	mov	bx,[file_handle]
	mov	dx,file_buffer
	mov	ah,40h			;DOS - Write file
	int	21h
	mov	dx,error_write
	jc	.file_error
	cmp	ax,cx
	jnz	.file_error

	inc	[file_sector]
	sub	word [file_size],ax
	sbb	word [file_size+2],0
	mov	ax,word [file_size]
	or	ax,word [file_size+2]
	jnz	.read_more

	mov	bx,[file_handle]
	mov	cx,[si+DIR_TYPE.Time]
	mov	dx,[si+DIR_TYPE.Date]
	mov	ax,5701h		;DOS - Set file date and time
	int	21h

	mov	bx,[file_handle]
	mov	ah,3eh			;DOS - Close file
	int	21h
	clc
	ret

    .file_error:
	mov	bx,[file_handle]
	mov	ah,3eh			;DOS - Close file
	int	21h
	stc
	ret

;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; Porcedure get_entry_name
; Param:  ds:si -> 8.3 space padded entry name
;         es:di -> buffer for ASCIIZ name
; Return: ZF set if empty name
; Note:   skip over unsupported chars
;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
get_entry_name:
	pusha
	mov	cx,0208h

    .new_char:
	lodsb
	cmp	al,20h
	jbe	.skip
	cmp	al,'"'
	jz	.skip
	cmp	al,'*'
	jz	.skip
	cmp	al,'+'
	jz	.skip
	cmp	al,','
	jz	.skip
	cmp	al,'.'
	jz	.skip
	cmp	al,'/'
	jz	.skip
	cmp	al,':'
	jz	.skip
	cmp	al,';'
	jz	.skip
	cmp	al,'<'
	jz	.skip
	cmp	al,'>'
	jz	.skip
	cmp	al,'?'
	jz	.skip
	cmp	al,'\'
	jz	.skip
	cmp	al,7ch
	jz	.skip
	cmp	al,7fh
	jnb	.skip
	stosb

    .skip:
	dec	cl
	jnz	.new_char
	dec	ch
	jz	.name_end
	mov	cl,3
	mov	al,'.'
	stosb
	jmp	.new_char

    .name_end:
	cmp	byte [di-1],'.'
	jnz	.set_0
	dec	di

    .set_0:
	mov	al,0
	stosb
	popa
	cmp	byte [di],0
	ret

;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; Procedure get_FAT
; Param:  ax - cluster number
; Return: ax - cluster state
;              0000 - free
;              0ff7 - bad cluster
;              0ff8 - end of chain
; Return: CF set on EndOfFile
;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
get_FAT:
	push	bx
	mov	bx,ax
	add	ax,ax
	add	bx,ax
	shr	bx,1			;cluster ofs = cluster*3/2
	mov	ax,word [bx+FAT_table]
	pop	bx
	jnc	.low
	shr	ax,4			;use high 12 bits (=remove low 4 bits)

    .low:
	and	ax,0fffh		;use low 12 bits
	cmp	ax,2
	jb	.FAT_error
	cmp	ax,0ff0h		;test for end of file
	jnb	.FAT_error
	clc
	ret

    .FAT_error:
	stc
	ret

;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; Procedure read_data_sector
; Param:  ax - first cluster (0 = rootdir)
;         bx - file position (sector number)
;         ds:di -> buffer
; Return: CF set on error (ds:dx -> error message)
;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
read_data_sector:
	pusha
	test	ax,ax
	jnz	.not_rootdir
	mov	ax,[rootdir_start]
	cmp	bx,[rootdir_size]
	jb	.sector_ok

    .read_error:
	popa
	mov	dx,error_read
	stc
	ret

    .not_rootdir:
	xor	cx,cx
	mov	cl,[bs.SectorsPerCluster]

    .is_cluster_ok:
	cmp	bx,cx
	jb	.cluster_ok
	sub	bx,cx
	call	get_FAT
	jnc	.is_cluster_ok

	popa
	mov	dx,error_FAT
	ret

    .cluster_ok:
	; convert FAT cluster into LBA addressing scheme
	; LBA = (cluster - 2) * SectorsPerCluster + DataStart
	dec	ax			;zero base cluster number
	dec	ax
	mul	cx
	add	ax,[data_start] 	;base data sector

    .sector_ok:
	add	ax,bx
	mov	cx,1
	call	read_sectors
	jc	.read_error

	popa
	clc
	ret

;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; Procedure read_sectors
; reads sectors from disk into memory
; Param:  ax = absolute sector
;         cx = sector count
;         ds:di -> buffer
; Return: CF set on error (ds:dx -> error message)
;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
read_sectors:
	pusha
	push	cx
	mov	dx,512
	mul	dx
	mov	cx,dx
	mov	dx,ax
	mov	bx,[img_handle]
	mov	ax,4200h		;DOS - Seek file
	int	21h
	pop	cx
	jc	.read_error

	shl	cx,9			;sector count * 512
	mov	dx,di
	mov	bx,[img_handle]
	mov	ah,3fh			;DOS - Read file
	int	21h
	jc	.read_error
	cmp	ax,cx
	jnz	.read_error

	popa
	clc
	ret

    .read_error:
	popa
	mov	dx,error_read
	stc
	ret

;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
txt_usage	db "UnIMG v0.01 - Unpack FAT12 disk image file",0dh,0ah
		db 0dh,0ah
		db "Usage:",0dh,0ah
		db "  unimg disk.img",0dh,0ah
		db 0dh,0ah
		db "Notes:",0dh,0ah
		db "  - only FAT12 with DOS like 8.3 names supported",0dh,0ah
		db "  - image file need file extension",0dh,0ah
		db "    (filename first 8 chars is used by dir name)",0dh,0ah
		db "$"
error_img	db "Image boot sector error",0dh,0ah,"$"
error_open	db "Can't find image file",0dh,0ah,"$"
error_read	db "Can't read image file",0dh,0ah,"$"
error_FAT	db "Image FAT error",0dh,0ah,"$"
error_md	db "Can't create dir",0dh,0ah,"$"
error_cd	db "Can't change dir",0dh,0ah,"$"
error_create	db "Can't create file",0dh,0ah,"$"
error_write	db "Can't write file",0dh,0ah,"$"
error_bad_name	db "Bad entry name",0dh,0ah,"$"
dir_up		db "..",0

img_name	rb 256
entry_name	rb 256
img_handle	rw 1
file_handle	rw 1
FAT_sectors	rw 1
FAT_start	rw 1
rootdir_start	rw 1
rootdir_size	rw 1
data_start	rw 1
dir_cluster	rw 1
entry_no	rw 1
entry_count	rw 1
file_size	rd 1
file_cluster	rw 1
file_sector	rw 1
bs		BOOT_TYPE
dir_buffer	rb 512
file_buffer	rb 512
FAT_table	rb 12*512
