//------------------------------------------------------------------------------
// File: Graph.cpp
//
// Desc: Sample code for BDA graph building.
//
//
// Copyright (c) 2003, Conexant System, Inc. All rights reserved.
//------------------------------------------------------------------------------

#include "graph.h"
#include <stdlib.h>
#include <stdio.h>
#include <TCHAR.h>

// 
// NOTE: In this sample, text strings are hard-coded for 
// simplicity and for readability.  For product code, you should
// use string tables and LoadString().
//

//
// An application can advertise the existence of its filter graph
// by registering the graph with a global Running Object Table (ROT).
// The GraphEdit application can detect and remotely view the running
// filter graph, allowing you to 'spy' on the graph with GraphEdit.
//
// To enable registration in this sample, define REGISTER_FILTERGRAPH.
//
#define REGISTER_FILTERGRAPH

// Constructor, initializes member variables
// and calls InitializeGraphBuilder
CBDAFilterGraph::CBDAFilterGraph() :
    m_fGraphBuilt(FALSE),
    m_fGraphRunning(FALSE),

#if 0  //Asiasat2  (Dwtv)
    m_ulCarrierFrequency(4000000),      
	m_ulSymbolRate(28125),
	m_DISEQC(DISEQC_D),
	m_SignalPolarisation(BDA_POLARISATION_LINEAR_H),
	m_LNB_POWER(LNB_POWER_ON),
	m_LNB_TYPE(LNB_TYPE_5150_5750),
	m_LNB_LOF(5150000),
	m_LNB_HOF(5750000),
	m_K22_Tone(K22_Tone_OFF),
#endif

#if 1  //Apstart2R  (Playboy)
	m_ulCarrierFrequency(12615000),     
	m_ulSymbolRate(22425),
	m_DISEQC(DISEQC_D),
	m_SignalPolarisation(BDA_POLARISATION_LINEAR_V),
	m_LNB_POWER(LNB_POWER_ON),
	m_LNB_TYPE(LNB_TYPE_CUSTOM),
	m_LNB_LOF(0),
	m_LNB_HOF(11300),
	m_K22_Tone(K22_Tone_OFF),
#endif

#if 0  //Apstart2R  (ET)
	m_ulCarrierFrequency(12530000),     
	m_ulSymbolRate(30000),
	m_DISEQC(DISEQC_D),
	m_SignalPolarisation(BDA_POLARISATION_LINEAR_V),
	m_LNB_POWER(LNB_POWER_ON),
	m_LNB_TYPE(LNB_TYPE_CUSTOM),
	m_LNB_LOF(0),
	m_LNB_HOF(11300),
	m_K22_Tone(K22_Tone_OFF),
#endif

#if 0  //Apstart2R  (Afa)
	m_ulCarrierFrequency(12280000),      
	m_ulSymbolRate(22425),
	m_DISEQC(DISEQC_D),
	m_SignalPolarisation(BDA_POLARISATION_LINEAR_V),
	m_LNB_POWER(LNB_POWER_ON),
	m_LNB_TYPE(LNB_TYPE_CUSTOM),
	m_LNB_LOF(0),
	m_LNB_HOF(11300),
	m_K22_Tone(K22_Tone_OFF),
#endif

    m_KsTunerPropSet(NULL),
    m_KsDemodPropSet(NULL),
    m_pTunerPin(NULL),
    m_pDemodPin(NULL),
    m_ONID(0),                         // Original Network ID
    m_TSID(0),                         //
    m_SID(0),
    m_mappedVidPid(0),                 // Video PID
    m_mappedAudPid(0),                 // Audio PID
	m_mappedTHParPid(0),               // THParser PID
    m_dwGraphRegister(0)
{
    if(FAILED(InitializeGraphBuilder()))
        m_fGraphFailure = TRUE;
    else
        m_fGraphFailure = FALSE;
}


// Destructor
CBDAFilterGraph::~CBDAFilterGraph()
{
    StopGraph();

    if(m_fGraphBuilt || m_fGraphFailure)
    {
        TearDownGraph();
    }
}

void
CBDAFilterGraph::ReleaseInterfaces()
{
    OutputDebugString(TEXT("BDASample: CBDAFilterGraph::ReleaseInterfaces\n"));

    if (m_pITuningSpace)
    {
        m_pITuningSpace = NULL;
    }

    if (m_pTuningSpaceContainer)
    {
        m_pTuningSpaceContainer = NULL;
    }

    if (m_pITuner)
    {
        m_pITuner = NULL;
    }

    if (m_pFilterGraph)
    {
        m_pFilterGraph = NULL;
    }

    if (m_pIMediaControl)
    {
        m_pIMediaControl = NULL;
    }

    if (m_pVideoPin)
    {
        m_pVideoPin = NULL;
    }

    if (m_pAudioPin)
    {
        m_pAudioPin = NULL;
    }
	 
	if (m_pTHParserPin)
	{
		m_pTHParserPin = NULL;
	}

    if (m_pIVideoPIDMap)
    {
        m_pIVideoPIDMap = NULL;
    }

    if (m_pIAudioPIDMap)
    {
        m_pIAudioPIDMap = NULL;
    }

	if (m_pITHParserPIDMap)
	{
		m_pITHParserPIDMap = NULL;
	}

	if (m_pIMpeg2PsiParser)
	{
		m_pIMpeg2PsiParser = NULL;
	}

     if ( m_KsDemodPropSet )
     {
        m_KsDemodPropSet = NULL;
     }

     if ( m_KsTunerPropSet )
     {
        m_KsTunerPropSet = NULL;
     }

     if ( m_pTunerPin )
     {
        m_pTunerPin = NULL;
     }

     if ( m_pDemodPin )
     {
        m_pDemodPin = NULL;
     }
}

// Instantiate graph object for filter graph building
HRESULT CBDAFilterGraph::InitializeGraphBuilder()
{
    HRESULT hr = S_OK;
    
    OutputDebugString(TEXT("BDASample: CBDAFilterGraph::InitializeGraphBuilder\n"));

    // we have a graph already
    if (m_pFilterGraph)
        return S_OK;

    // create the filter graph
    if (FAILED(hr = m_pFilterGraph.CoCreateInstance(CLSID_FilterGraph)))
    {
        ErrorMessageBox(TEXT("Couldn't CoCreate IGraphBuilder\n"));
        m_fGraphFailure = true;
        return hr;
    }
    
    return hr;
}

// BuildGraph sets up devices, adds and connects filters
HRESULT CBDAFilterGraph::BuildGraph(
    CComBSTR pNetworkType)   // the registered Tuning Space name
{
    HRESULT hr = S_OK;

    OutputDebugString(TEXT("BDASample: CBDAFilterGraph::BuildGraph\n"));

    // if we have already have a filter graph, tear it down
    if (m_fGraphBuilt) {
        if(m_fGraphRunning) {
            hr = StopGraph();
        }

        hr = TearDownGraph();
    }

    // STEP 1: load network provider first so that it can configure other
    // filters, such as configuring the demux to sprout output pins.
    // We also need to submit a tune request to the Network Provider so it will
    // tune to a channel
    if (FAILED(hr = LoadNetworkProvider(pNetworkType)))
    {
        ErrorMessageBox(TEXT("Cannot load network provider\n"));
        BuildGraphError();
        return hr;
    }

    hr = m_pNetworkProvider->QueryInterface(__uuidof (ITuner), reinterpret_cast <void**> (&m_pITuner));
    if (FAILED(hr)) {
        ErrorMessageBox(TEXT("pNetworkProvider->QI: Can't QI for ITuner.\n"));
        BuildGraphError();
        return hr;
    }

    // create a tune request to initialize the network provider
    // before connecting other filters
    CComPtr <IDVBTuneRequest> pDVBSTuneRequest;
    if (FAILED(hr = CreateDVBSTuneRequest(&pDVBSTuneRequest)))
    {
        ErrorMessageBox(TEXT("Cannot create tune request\n"));
        BuildGraphError();
        return hr;
    }

    //submit the tune request to the network provider
    hr = m_pITuner->put_TuneRequest(pDVBSTuneRequest);
    if (FAILED(hr)) {
        ErrorMessageBox(TEXT("Cannot submit the tune request\n"));
        BuildGraphError();
        return hr;
    }

    // STEP2: Load tuner device and connect to network provider
    if (FAILED(hr = LoadFilter(KSCATEGORY_BDA_NETWORK_TUNER, 
                               &m_pTunerDemodDevice,
                               m_pNetworkProvider, 
                               TRUE)) || !m_pTunerDemodDevice)
	//if (FAILED(hr = LoadTunerFilter(KSCATEGORY_BDA_NETWORK_TUNER, 
     //                          m_pNetworkProvider, 
    //                           TRUE)))    
    {
        ErrorMessageBox(TEXT("Cannot load tuner device and connect network provider\n"));
        BuildGraphError();
        return hr;
    }

    // Step3: Load capture device and connect to tuner/demod device
    if (FAILED(hr = LoadFilter(KSCATEGORY_BDA_RECEIVER_COMPONENT,
                               &m_pCaptureDevice,
                               m_pTunerDemodDevice, 
                               TRUE)))
    {
        ErrorMessageBox(TEXT("Cannot load capture device and connect tuner/demod\n"));
        BuildGraphError();
        return hr;
    }

    
    //
    // this next call loads and connects filters associated with
    // the demultiplexor. if you want to manually load individual
    // filters such as audio and video decoders, use the code at
    // the bottom of this file
    //
#ifdef REGISTER_FILTERGRAPH
    hr = AddGraphToRot(m_pFilterGraph, &m_dwGraphRegister);
    if (FAILED(hr))
    {
      ///ErrorMessageBox(TEXT("Failed to register filter graph with ROT!  hr=0x%x"), hr);
        m_dwGraphRegister = 0;
    }
#endif

	// Step4: Load demux
    if (FAILED(hr = LoadDemux())) {
        ErrorMessageBox(TEXT("Cannot load demux\n"));
        BuildGraphError();
        return hr;
    }

    // Step5: Render demux pins
    //if (FAILED (hr = BuildAVSegment()))
    if (FAILED(hr = RenderDemux())) {
        ErrorMessageBox(TEXT("Cannot load demux\n"));
        BuildGraphError();
        return hr;
    }

    // Step6: get pointer to Tuner and Demod pins on Tuner/Demod filter
    // for setting the tuner and demod. property sets
    if (GetTunerDemodPropertySetInterfaces() == FALSE) {
        ErrorMessageBox(TEXT("Getting tuner/demod pin pointers failed\n"));
        BuildGraphError();
		return hr;
    }

	if (!CheckBDAInterface()) {
        ErrorMessageBox(TEXT("CheckBDAInterface failed\n"));
        BuildGraphError();
		return S_FALSE;
    }

	m_TH_Params.ulMCEFreqTranslate = 0;		// MCE frequency table translate	
    m_TH_Params.ulFixedBW;					// The fixed bandwidth in registry
    m_TH_Params.ulShiftFreqScan;			// Disable Lock Shift in registry
    
    m_TH_Params.ulLNBLOF = m_LNB_TYPE;		// Local Oscillator Frequency (LOF)
    m_TH_Params.ulLNBLOFLowBand = m_LNB_LOF;// LOF Low Band
    m_TH_Params.ulLNBLOFHighBand = m_LNB_HOF;// LOF High Band
    m_TH_Params.ulDiSEqC = m_DISEQC;		// DiSEqC Selection
    m_TH_Params.ulLNBPower = m_LNB_POWER;   // LNB Power ON/OFF
    m_TH_Params.ulTone = m_K22_Tone;		// Tone ON/OFF
    
	if (!THBDA_IOCTL_SET_REG_PARAMS_Fun(&m_TH_Params)) {
		ErrorMessageBox(TEXT("THBDA_IOCTL_SET_REG_PARAMS_Fun failed\n"));
		BuildGraphError();
		return S_FALSE;
	}

    m_fGraphBuilt = true;
    m_fGraphFailure = false;
    
    return S_OK;
}


// Get the IPin addresses to the Tuner (Input0) and the
// Demodulator (MPEG2 Transport) pins on the Tuner/Demod filter.
// Then gets the IKsPropertySet interfaces for the pins.
//
BOOL CBDAFilterGraph::GetTunerDemodPropertySetInterfaces()
{
    if (!m_pTunerDemodDevice) {
        return FALSE;
    }

    m_pTunerPin = FindPinOnFilter(m_pTunerDemodDevice, "Input0");
    if (!m_pTunerPin) {
        ErrorMessageBox(TEXT("Cannot find Input0 pin on BDA Tuner/Demod filter!\n"));
        return FALSE;
    }

    m_pDemodPin = FindPinOnFilter(m_pTunerDemodDevice, "MPEG2 Transport");
    if (!m_pDemodPin) {
        ErrorMessageBox(TEXT("Cannot find MPEG2 Transport pin on BDA Tuner/Demod filter!\n"));
        return FALSE;
    }

    HRESULT hr = m_pTunerPin->QueryInterface(IID_IKsPropertySet,
                                   reinterpret_cast<void**>(&m_KsTunerPropSet));
    if (FAILED(hr)) {
        ErrorMessageBox(TEXT("QI of IKsPropertySet failed\n"));
        m_pTunerPin = NULL;
		return FALSE;
	}

    hr = m_pDemodPin->QueryInterface(IID_IKsPropertySet,
                                   reinterpret_cast<void**>(&m_KsDemodPropSet));
    if (FAILED(hr)) {
        ErrorMessageBox(TEXT("QI of IKsPropertySet failed\n"));
        m_pDemodPin = NULL;
		return FALSE;
	}

    return TRUE;
}

// Makes call to tear down the graph and sets graph failure
// flag to true
VOID
CBDAFilterGraph::BuildGraphError()
{
    OutputDebugString(TEXT("BDASample: CBDAFilterGraph::BuildGraphError\n"));

    TearDownGraph();
    m_fGraphFailure = true;
}


// This creates a new DVBS Tuning Space entry in the registry.
//
HRESULT CBDAFilterGraph::CreateTuningSpace(
    CComBSTR pNetworkType)
{
    HRESULT hr = S_OK;

    OutputDebugString(TEXT("BDASample: CBDAFilterGraph::CreateTuningSpace\n"));

    CComPtr <IDVBSTuningSpace> pIDVBTuningSpace;
    hr = pIDVBTuningSpace.CoCreateInstance(CLSID_DVBSTuningSpace);
    if (FAILED(hr) || !pIDVBTuningSpace) {
        ErrorMessageBox(TEXT("Failed to create system tuning space."));
        return FALSE;
    }

    // set the Frequency mapping
    WCHAR szFreqMapping[ 3 ]=L"-1";
    BSTR bstrFreqMapping = SysAllocString(szFreqMapping);
    if (bstrFreqMapping) {
        hr = pIDVBTuningSpace->put_FrequencyMapping(bstrFreqMapping); // not used
        SysFreeString(bstrFreqMapping);
    }

    // set the Friendly Name of the network type, shown as Name in the registry
    hr = pIDVBTuningSpace->put_UniqueName(pNetworkType);
    if (FAILED(hr)) {
        OutputDebugString(TEXT("BDASample: put_FriendlyName failed\n"));
        return hr;
    }

    // set the System type to terrestrial
    hr = pIDVBTuningSpace->put_SystemType(DVB_Satellite);
    if (FAILED(hr)) {
        OutputDebugString(TEXT("BDASample: put_SystemType failed\n"));
        return hr;
    }

    // set the Network Type to DVBT
	//CComBSTR clsid_dvbt = ("{216C62DF-6D7F-4e9a-8571-05F14EDB766A}");
    hr = pIDVBTuningSpace->put__NetworkType(CLSID_DVBSNetworkProvider);
    if (FAILED(hr)) {
        OutputDebugString(TEXT("BDASample: put_NetworkType failed\n"));
        return hr;
    }

    // set the Friendly Name, shown as Description in the registry
    WCHAR szFriendlyName[ 32 ]=L"Twinhan DVBS";
    BSTR bstrFriendlyName = SysAllocString(szFriendlyName);
    if (bstrFriendlyName) {
        hr = pIDVBTuningSpace->put_FriendlyName(bstrFriendlyName);
        SysFreeString(bstrFriendlyName);
    }
/*
	// create IBDA_FrequencyFilter
	CComPtr <IBDA_FrequencyFilter> pIBDA_FrequencyFilter;
    hr = CoCreateInstance(CLSID_BDA_FrequencyFilter,
                          NULL,
                          CLSCTX_INPROC_SERVER,
                          IID_IBDA_FrequencyFilter,
                          reinterpret_cast<void**>(&pIBDA_FrequencyFilter));
    if (FAILED(hr) || !pIBDA_FrequencyFilter) {
        ErrorMessageBox(TEXT("BDASample: Failed to create pIBDA_FrequencyFilter."));
        return hr;
    }
	hr = pIBDA_FrequencyFilter->put_FrequencyMultiplier(1000);
*/
    // create DVBS Locator
    CComPtr <IDVBSLocator> pIDVBSLocator;
    hr = CoCreateInstance(CLSID_DVBSLocator,
                          NULL,
                          CLSCTX_INPROC_SERVER,
                          IID_IDVBSLocator,
                          reinterpret_cast<void**>(&pIDVBSLocator));
    if (FAILED(hr) || !pIDVBSLocator) {
        ErrorMessageBox(TEXT("BDASample: Failed to create system locator."));
        return hr;
    }

    // NOTE: The below parameter with the exception of
    //       setting the carrier frequency don't need to set.
    //       Thus they are set to -1.  The demodulator can
    //       handle these parameters without having to specifically
    //       set them in the hardware.

    // set the Carrier Frequency
    hr = pIDVBSLocator->put_CarrierFrequency(m_ulCarrierFrequency);
	//hr = pIDVBSLocator->put_SymbolRate(m_ulSymbolRate);
	//hr = pIDVBSLocator->put_SignalPolarisation(m_SignalPolarisation);
	
    
    // set the Bandwidth
    //hr = pIDVBSLocator->put_Bandwidth(-1);                            // not used 
    
    // set the Low Priority FEC type
    //hr = pIDVBSLocator->put_LPInnerFEC(BDA_FEC_METHOD_NOT_SET);       // not used
       
    // set the Low Priority code rate
    //hr = pIDVBSLocator->put_LPInnerFECRate(BDA_BCC_RATE_NOT_SET);     // not used
    
    // set the hieriarcy alpha
    //hr = pIDVBSLocator->put_HAlpha(BDA_HALPHA_NOT_SET);               // not used 
        
    // set the guard interval
    //hr = pIDVBSLocator->put_Guard(BDA_GUARD_NOT_SET);                 // not used 
        
    // set the transmission mode/FFT
    //hr = pIDVBSLocator->put_Mode(BDA_XMIT_MODE_NOT_SET);              // not used 
        
    // set whether the frequency is being used by another DVB-S broadcaster
    //hr = pIDVBSLocator->put_OtherFrequencyInUse(VARIANT_BOOL(FALSE)); // not used

    // set the inner FEC type
    hr = pIDVBSLocator->put_InnerFEC(BDA_FEC_METHOD_NOT_SET);         // not used
        
    // set the inner code rate
    hr = pIDVBSLocator->put_InnerFECRate(BDA_BCC_RATE_NOT_SET);       // not used 
        
    // set the outer FEC type
    hr = pIDVBSLocator->put_OuterFEC(BDA_FEC_METHOD_NOT_SET);         // not used 
        
    // set the outer code rate
    hr = pIDVBSLocator->put_OuterFECRate(BDA_BCC_RATE_NOT_SET);       // not used 
        
    // set the modulation type
    hr = pIDVBSLocator->put_Modulation(BDA_MOD_NOT_SET);              // not used 
       
    // set the symbol rate
    hr = pIDVBSLocator->put_SymbolRate(-1);                           // not used

    // set this a default locator
    hr = pIDVBTuningSpace->put_DefaultLocator(pIDVBSLocator);

    // create a tuning space container for the values we just set
    // so we can create in the registry.
    CComPtr <ITuningSpaceContainer> pTunerSpaceContainer;
    hr = CoCreateInstance(CLSID_SystemTuningSpaces,
                          NULL,
                          CLSCTX_INPROC_SERVER,
                          IID_ITuningSpaceContainer,
                          reinterpret_cast<void**>(&pTunerSpaceContainer));
    if (FAILED(hr) || !pTunerSpaceContainer) {
        ErrorMessageBox(TEXT("BDASample: Failed to create tuning space container"));
        return hr;
    }

    // get the number of tuning spaces that have been created
    LONG lCount;
    hr = pTunerSpaceContainer->get_Count(&lCount); 
    if (FAILED(hr)) {
        ErrorMessageBox(TEXT("BDASample: get_Count failed"));
        return hr;
    }

    // add the tuning spacing container values to the registry
    CComVariant varIndex(lCount+1);
    hr = pTunerSpaceContainer->Add(pIDVBTuningSpace, &varIndex);
    if (FAILED(hr)) {
        ErrorMessageBox(TEXT("BDASample: Failed to add new tuning space to registry"));
        return hr;
    }

    // create a copy of this tuning space for the 
    // class member variable.
    hr = pIDVBTuningSpace->Clone(&m_pITuningSpace);
    if (FAILED(hr)) {
        ErrorMessageBox(TEXT("BDASample: Clone failed"));
        m_pITuningSpace = NULL;
        return hr;
    }

    pIDVBTuningSpace     = NULL;
    pIDVBSLocator        = NULL;
	//pIBDA_FrequencyFilter  = NULL;
    pTunerSpaceContainer = NULL;

    return hr;
}


// Loads the correct tuning space based on NETWORK_TYPE that got
// passed into BuildGraph()
HRESULT CBDAFilterGraph::LoadTuningSpace(
    CComBSTR pNetworkType)
{   
	HRESULT hr;
	ITuningSpace* pTuningSpace           = NULL;
    IEnumTuningSpaces* pEnumTuningSpaces = NULL;
    ITuningSpace** pTuningSpaceArray     = NULL;
    long lCount                          = 0;
    ULONG ulNumFetched = 0, i = 0;
    CComBSTR bstrTemp;

    OutputDebugString(TEXT("BDASample: CBDAFilterGraph::LoadTuningSpace\n"));

	// Get the tuning space collection
    hr = CoCreateInstance(CLSID_SystemTuningSpaces, NULL, 
                          CLSCTX_INPROC_SERVER, IID_ITuningSpaceContainer,
                          reinterpret_cast<void**>(&m_pTuningSpaceContainer));
    if (FAILED(hr)) {
        ErrorMessageBox(TEXT("Failed to create system tuning space."));
        return hr;
    }

	// Find the tuning space in the collection
    hr = m_pTuningSpaceContainer->get_EnumTuningSpaces(&pEnumTuningSpaces);
    if (FAILED(hr)) {
        ErrorMessageBox(TEXT("Failed to get tuning space enumerator."));
        return hr;
    }
    hr = m_pTuningSpaceContainer->get_Count(&lCount);
    if (FAILED(hr)) {
        ErrorMessageBox(TEXT("Failed to enumerate tuning spaces."));
        return hr;
    }

    // Read tuning space info into allocated array
    pTuningSpaceArray = new ITuningSpace*[lCount];
    hr = pEnumTuningSpaces->Next(lCount, pTuningSpaceArray, &ulNumFetched);
    if (FAILED(hr))  {
        ErrorMessageBox(TEXT("Failed to read tuning spaces."));
        return hr;
    }

    pEnumTuningSpaces = NULL;

    // Find the tuning space
    for (i = 0; i < ulNumFetched; i++) {
        hr = pTuningSpaceArray[i]->get_UniqueName(&bstrTemp);
        if (FAILED(hr)) {
            ErrorMessageBox(TEXT("Failed to read tuning space unique name."));
            return hr;
        }

        // Is this the correct tuning space?
        if (bstrTemp == pNetworkType) {
            OutputDebugString(TEXT("BDASample: Found the MYDVBT tuning space.\n"));
            hr = pTuningSpaceArray[i]->Clone(&pTuningSpace);
            break;
        }
    }

    pTuningSpaceArray = NULL;

    if (pTuningSpace == NULL) {
        OutputDebugString(TEXT("BDASample: Could not find MYDVBT tuning space.\n"));
        return E_FAIL;
    }

	// QI for IDVBTuningSpace
    hr = pTuningSpace->QueryInterface(IID_IDVBSTuningSpace, reinterpret_cast<void**>(&m_pITuningSpace));
    if (FAILED(hr)) {
        ErrorMessageBox(TEXT("Failed to QI for IDVBTuningSpace."));
        return hr;
    }

    pTuningSpace = NULL;

    delete pTuningSpaceArray;
    
    return hr;
}


// Creates an DVBT Tune Request
HRESULT
CBDAFilterGraph::CreateDVBSTuneRequest(
    IDVBTuneRequest** pTuneRequest)
{
    HRESULT hr = S_OK;

    OutputDebugString(TEXT("BDASample: CBDAFilterGraph::CreateDVBSTuneRequest\n"));

    if (pTuneRequest == NULL)
    {
        ErrorMessageBox (TEXT("Invalid pointer\n"));
        return E_POINTER;
    }

    // Making sure we have a valid tuning space
    if (m_pITuningSpace == NULL)
    {
        ErrorMessageBox(TEXT("Tuning Space is NULL\n"));
        return E_FAIL;
    }

    //  Create an instance of the DVBS tuning space
    CComQIPtr <IDVBTuningSpace> pDVBTTuningSpace (m_pITuningSpace);
    if (!pDVBTTuningSpace)
    {
        ErrorMessageBox(TEXT("Cannot QI for an IDVBTTuningSpace\n"));
        return E_FAIL;
    }

    //  Create an empty tune request.
    CComPtr <ITuneRequest> pNewTuneRequest;
    hr = pDVBTTuningSpace->CreateTuneRequest(&pNewTuneRequest);
    if (FAILED (hr))
    {
        ErrorMessageBox(TEXT("CreateTuneRequest: Can't create tune request.\n"));
        return hr;
    }

    //query for an IDVBTuneRequest interface pointer
    CComQIPtr <IDVBTuneRequest> pDVBSTuneRequest (pNewTuneRequest);
    if (!pDVBSTuneRequest)
    {
        ErrorMessageBox(TEXT("CreateDVBSTuneRequest: Can't QI for IDVBTuneRequest.\n"));
        return E_FAIL;
    }

    // set the Original Network ID
    //hr = pDVBSTuneRequest->put_ONID( m_ONID ); 
    //if (FAILED(hr))
    //{
    //    ErrorMessageBox(TEXT("put_ONID failed\n"));
    //    return hr;
    //}

    // set the Transponder ID
    //hr = pDVBSTuneRequest->put_TSID( m_TSID ); 
    //if(FAILED(hr))
    //{
    //    ErrorMessageBox(TEXT("put_TSID failed\n"));
    //    return hr;
    //}

    // set the Service ID
    //hr = pDVBSTuneRequest->put_SID( m_SID );
    //if(FAILED(hr))
    //{
    //    ErrorMessageBox(TEXT("put_SID failed\n"));
    //    return hr;
    //}

    CComPtr <IDVBSLocator> pDVBSLocator;
    hr = pDVBSLocator.CoCreateInstance (CLSID_DVBSLocator);	
    if (FAILED(hr) || !pDVBSLocator) {
        ErrorMessageBox(TEXT("BDASample: Failed to create system locator."));
        return hr;
    }
    if (FAILED( hr))
    {
        ErrorMessageBox(TEXT("Cannot create the DVB-S locator failed\n"));
        return hr;
    }

    hr = pDVBSLocator->put_CarrierFrequency( m_ulCarrierFrequency );
    if (FAILED(hr))
    {
        ErrorMessageBox(TEXT("put_CarrierFrequency failed\n"));
        return hr;
    }
	hr = pDVBSLocator->put_SymbolRate(m_ulSymbolRate);
	hr = pDVBSLocator->put_SignalPolarisation(m_SignalPolarisation);

    hr = pDVBSTuneRequest->put_Locator (pDVBSLocator);
    if (FAILED (hr))
    {
        ErrorMessageBox(TEXT("Cannot put the locator\n"));
        return hr;
    }

    hr = pDVBSTuneRequest.QueryInterface (pTuneRequest);

    return hr;
}


// LoadNetworkProvider loads network provider
HRESULT CBDAFilterGraph::LoadNetworkProvider(
    CComBSTR pNetworkType)
{
    HRESULT  hr = S_OK;
    CComBSTR bstrNetworkType;
    CLSID    CLSIDNetworkType;

    OutputDebugString(TEXT("BDASample: CBDAFilterGraph::LoadNetworkProvider\n"));

    // obtain tuning space then load network provider
    if (m_pITuningSpace == NULL) {
        hr = LoadTuningSpace(pNetworkType);
        if (FAILED(hr)) {
            OutputDebugString(TEXT("BDASample: Cannot find MYDVB-S TuningSpace\n"));

            // so we create one.
            hr = CreateTuningSpace(pNetworkType);
            if (FAILED(hr)) {
                ErrorMessageBox(TEXT("Cannot load/create MYDVB-S tuning space\n"));
                return E_FAIL;
            }
        }
    }

    if (!m_pITuningSpace) {
        ErrorMessageBox(TEXT("Tuning space error!\n"));
        return E_FAIL;
    }

    // Get the current Network Type clsid
    hr = m_pITuningSpace->get_NetworkType(&bstrNetworkType);
    if (FAILED(hr)) {
        ErrorMessageBox(TEXT("ITuningSpace::Get Network Type failed\n"));
        return hr;
    }

    hr = CLSIDFromString(bstrNetworkType, &CLSIDNetworkType);
    if (FAILED(hr)) {
        ErrorMessageBox(TEXT("Couldn't get CLSIDFromString\n"));
        return hr;
    }

    // create the network provider based on the clsid obtained from the tuning space
    hr = CoCreateInstance(CLSIDNetworkType, NULL, CLSCTX_INPROC_SERVER,
                          IID_IBaseFilter, 
                          reinterpret_cast<void**>(&m_pNetworkProvider));
    if (FAILED(hr)) {
        ErrorMessageBox(TEXT("Couldn't CoCreate Network Provider\n"));
        return hr;
    }

    //add the Network Provider filter to the graph
    hr = m_pFilterGraph->AddFilter(m_pNetworkProvider, L"Network Provider");

    return hr;
}


// enumerates through registered filters
// instantiates the the filter object and adds it to the graph
// it checks to see if it connects to upstream filter
// if not,  on to the next enumerated filter
// used for tuner, capture, MPE Data Filters and decoders that
// could have more than one filter object
// if pUpstreamFilter is NULL don't bother connecting
HRESULT CBDAFilterGraph::LoadFilter(
    REFCLSID clsid, 
    IBaseFilter** ppFilter,
    IBaseFilter* pConnectFilter, 
    BOOL fIsUpstream)
{
    HRESULT                hr = S_OK;
    BOOL                   fFoundFilter = FALSE;
    CComPtr <IMoniker>     pIMoniker;
    CComPtr <IEnumMoniker> pIEnumMoniker;

    OutputDebugString(TEXT("BDASample: CBDAFilterGraph::LoadFilter\n"));

    if (!m_pICreateDevEnum) {
        hr = m_pICreateDevEnum.CoCreateInstance(CLSID_SystemDeviceEnum);
        if (FAILED(hr)) {
            ErrorMessageBox(TEXT("LoadFilter(): Cannot CoCreate ICreateDevEnum"));
            return hr;
        }
    }

    // obtain the enumerator
    hr = m_pICreateDevEnum->CreateClassEnumerator(clsid, &pIEnumMoniker, 0);
    // the call can return S_FALSE if no moniker exists, so explicitly check S_OK
    if (FAILED(hr)) {
        ErrorMessageBox(TEXT("LoadFilter(): Cannot CreateClassEnumerator"));
        return hr;
    }
    if (S_OK != hr) { // Class not found
        ErrorMessageBox(TEXT("LoadFilter(): Class not found, CreateClassEnumerator returned S_FALSE"));
        return E_UNEXPECTED;
    }

    // next filter
    while (pIEnumMoniker->Next(1, &pIMoniker, 0) == S_OK)
    {
        // obtain filter's friendly name
        CComPtr <IPropertyBag> pBag;

        hr = pIMoniker->BindToStorage(NULL, 
                                      NULL, 
                                      IID_IPropertyBag,
                                      reinterpret_cast<void**>(&pBag));
        if (FAILED(hr)) {
            OutputDebugString (TEXT("BDASample: LoadFilter(): Cannot BindToStorage"));
            return hr;
        }

        CComVariant varBSTR;
        hr = pBag->Read(L"FriendlyName", &varBSTR, NULL);
        if (FAILED(hr)) {
            OutputDebugString (TEXT("BDASample: LoadFilter(): IPropertyBag->Read method failed"));
            pIMoniker = NULL;
            continue;
        }

        pBag = NULL;

        // bind the filter
        CComPtr <IBaseFilter> pFilter;

        hr = pIMoniker->BindToObject(NULL, 
                                     NULL, 
                                     IID_IBaseFilter,
                                     reinterpret_cast<void**>(&pFilter));

        if (FAILED(hr)) {
            pIMoniker = NULL;
            pFilter = NULL;
            continue;
        }

        hr = m_pFilterGraph->AddFilter (pFilter, varBSTR.bstrVal);
        if (FAILED(hr)) {
            OutputDebugString (TEXT("BDASample: Cannot add filter\n"));
            return hr;
        }

        //MessageBox (NULL, _T(""), _T(""), MB_OK);
        // test connections
        // to upstream filter
        if (pConnectFilter) {
            if (fIsUpstream) {
                hr = ConnectFilters(pConnectFilter, pFilter);
            }
            else {
                hr = ConnectFilters(pFilter, pConnectFilter);
            }

            if (SUCCEEDED(hr)) {
                //that's the filter we want
                fFoundFilter = TRUE;
                pFilter.QueryInterface(ppFilter);
                break;
            }
            else {
                fFoundFilter = FALSE;
                // that wasn't the the filter we wanted
                // so unload and try the next one
                hr = m_pFilterGraph->RemoveFilter(pFilter);
                if (FAILED(hr)) {
                    OutputDebugString(TEXT("BDASample: Failed unloading Filter\n"));
                    return hr;
                }
            }
        }
        else {
            fFoundFilter = TRUE;
            pFilter.QueryInterface(ppFilter);
            break;
        }

        pIMoniker = NULL;
        pFilter = NULL;

    } // while

    pIEnumMoniker = NULL;

    return S_OK;
}

// Loads the demux into the FilterGraph
HRESULT CBDAFilterGraph::LoadDemux()
{
    HRESULT hr = S_OK;
    
    hr = CoCreateInstance(CLSID_MPEG2Demultiplexer, 
                          NULL, 
                          CLSCTX_INPROC_SERVER,
                          IID_IBaseFilter, 
                          reinterpret_cast<void**>(&m_pDemux));
    if (FAILED(hr)) {
        ErrorMessageBox(TEXT("Could not CoCreateInstance CLSID_MPEG2Demultiplexer\n"));
        return hr;
    }

    hr = m_pFilterGraph->AddFilter(m_pDemux, L"Demux");
    if (FAILED(hr)) {
        ErrorMessageBox(TEXT("Unable to add demux filter to graph\n"));
        return hr;
    }

    return hr;
}

// Renders demux output pins.
HRESULT CBDAFilterGraph::RenderDemux()
{
    HRESULT             hr = S_OK;
    CComPtr <IPin>      pIPin;
    CComPtr <IPin>      pDownstreamPin;
    CComPtr <IEnumPins> pIEnumPins;
    PIN_DIRECTION       direction;

    OutputDebugString(TEXT("BDASample: CBDAFilterGraph::RenderDemux\n"));

    assert(m_pDemux);

    // connect the demux to the capture device
    hr = ConnectFilters(m_pCaptureDevice, m_pDemux);
    if (FAILED(hr)) {
        ErrorMessageBox(TEXT("Cannot connect demux to capture filter\n"));
        return hr;
    }

    // load transform information filter and connect it to the demux
    hr = LoadFilter(KSCATEGORY_BDA_TRANSPORT_INFORMATION, 
                    &m_pTIF, 
                    m_pDemux, 
                    TRUE);	
	if (FAILED(hr)) {
        ErrorMessageBox(TEXT("Cannot load TIF\n"));
        return hr;
    }


	hr = CoCreateInstance(CLSID_THPsiParser, NULL,
            CLSCTX_INPROC, IID_IBaseFilter,
            reinterpret_cast<void**>(&m_pTHPsiParser));
	if(SUCCEEDED(hr) && m_pTHPsiParser)
    {
        hr = m_pFilterGraph->AddFilter(m_pTHPsiParser, L"THPsiParser");
        if(FAILED(hr))
        {
            ErrorMessageBox(TEXT("Adding THPsiParser to the FilterGraph Failed\n"));
            return hr;
        }
    }
    else
    {
        ErrorMessageBox(TEXT("Loading THPsiParser Failed\n"));
        return hr;
    }

	hr = m_pTHPsiParser->QueryInterface(IID_IMpeg2PsiParser, reinterpret_cast<void**>(&m_pIMpeg2PsiParser));
    if (FAILED(hr)) {
        ErrorMessageBox(TEXT("Failed to QI for IMpeg2PsiParser."));
        return hr;
    }

    // render/connect the rest of the demux pins
    hr = m_pDemux->EnumPins(&pIEnumPins);
    if (FAILED(hr)) {
        ErrorMessageBox(TEXT("Cannot get the enumpins\n"));
        return hr;
    }

    while(pIEnumPins->Next(1, &pIPin, 0) == S_OK)
    {
        hr = pIPin->QueryDirection(&direction);
    
        if(direction == PINDIR_OUTPUT) {
            pIPin->ConnectedTo(&pDownstreamPin);
   
            if (pDownstreamPin == NULL) {
                m_pFilterGraph->Render(pIPin);
            }
   
           pDownstreamPin = NULL;
        }

        pIPin = NULL;
    }

    pIEnumPins = NULL;

    return hr;
}


// Removes each filter from the graph and un-registers
// the filter.
HRESULT
CBDAFilterGraph::TearDownGraph()
{
    HRESULT hr = S_OK;
    CComPtr <IBaseFilter> pFilter;
    CComPtr <IEnumFilters> pIFilterEnum;

    OutputDebugString(TEXT("BDASample: CBDAFilterGraph::TearDownGraph\n"));

    if(m_fGraphBuilt || m_fGraphFailure)
    {
        // unload manually added filters
        if ( m_pIPSink )
        {
            m_pFilterGraph->RemoveFilter(m_pIPSink);
            m_pIPSink = NULL;
        }

        if ( m_pTHPsiParser )
        {
            m_pFilterGraph->RemoveFilter(m_pTHPsiParser);
            m_pTHPsiParser = NULL;
        }

        if ( m_pTIF )
        {
            m_pFilterGraph->RemoveFilter(m_pTIF);
            m_pTIF = NULL;
        }

        if ( m_pDemux )
        {
            m_pFilterGraph->RemoveFilter(m_pDemux);
            m_pDemux = NULL;
        }

        if ( m_pNetworkProvider )
        {
            m_pFilterGraph->RemoveFilter(m_pNetworkProvider);
            m_pNetworkProvider = NULL;
        }

        if ( m_pTunerDemodDevice )
        {
            m_pFilterGraph->RemoveFilter(m_pTunerDemodDevice);
            m_pTunerDemodDevice = NULL;
        }

        if ( m_pCaptureDevice )
        {
            m_pFilterGraph->RemoveFilter(m_pCaptureDevice);
            m_pCaptureDevice = NULL;
        }

        // now go unload rendered filters
        hr = m_pFilterGraph->EnumFilters(&pIFilterEnum);

        if(FAILED(hr))
        {
            ErrorMessageBox(TEXT("TearDownGraph: cannot EnumFilters\n"));
            return E_FAIL;
        }

        pIFilterEnum->Reset();
                                
        while(pIFilterEnum->Next(1, &pFilter, 0) == S_OK) // addrefs filter
        {
            hr = m_pFilterGraph->RemoveFilter(pFilter);

            if (FAILED (hr))
                return hr;

            pIFilterEnum->Reset();
            pFilter = NULL;
        }
    }

#ifdef REGISTER_FILTERGRAPH
    if (m_dwGraphRegister)
    {
        RemoveGraphFromRot(m_dwGraphRegister);
        m_dwGraphRegister = 0;
    }
#endif

    m_fGraphBuilt = FALSE;
    return S_OK;
}


// ConnectFilters is called from BuildGraph
// to enumerate and connect pins
HRESULT
CBDAFilterGraph::ConnectFilters(
    IBaseFilter* pFilterUpstream, 
    IBaseFilter* pFilterDownstream)
{
    HRESULT         hr = E_FAIL;

    CComPtr <IPin>  pIPinUpstream;

    PIN_INFO        PinInfoUpstream;
    PIN_INFO        PinInfoDownstream;

    OutputDebugString(TEXT("BDASample: CBDAFilterGraph::ConnectFilters\n"));
    
    // validate passed in filters
    assert(pFilterUpstream);

    assert(pFilterDownstream);

    // grab upstream filter's enumerator
    CComPtr <IEnumPins> pIEnumPinsUpstream;
    hr = pFilterUpstream->EnumPins(&pIEnumPinsUpstream);

    if(FAILED(hr))
    {
        ErrorMessageBox(TEXT("Cannot Enumerate Upstream Filter's Pins\n"));
        return hr;
    }

    // iterate through upstream filter's pins
    while (pIEnumPinsUpstream->Next (1, &pIPinUpstream, 0) == S_OK)
    {
        hr = pIPinUpstream->QueryPinInfo (&PinInfoUpstream);
        if(FAILED(hr))
        {
            ErrorMessageBox(TEXT("Cannot Obtain Upstream Filter's PIN_INFO\n"));
            return hr;
        }

        CComPtr <IPin> pPinDown;
        pIPinUpstream->ConnectedTo (&pPinDown);

        // bail if pins are connected
        // otherwise check direction and connect
        if ((PINDIR_OUTPUT == PinInfoUpstream.dir) && (pPinDown == NULL))
        {
            // grab downstream filter's enumerator
            CComPtr <IEnumPins> pIEnumPinsDownstream;
            hr = pFilterDownstream->EnumPins (&pIEnumPinsDownstream);
            if(FAILED(hr))
            {
                ErrorMessageBox(TEXT("Cannot enumerate pins on downstream filter!\n"));
                return hr;
            }

            // iterate through downstream filter's pins
            CComPtr <IPin>  pIPinDownstream;
            while (pIEnumPinsDownstream->Next (1, &pIPinDownstream, 0) == S_OK)
            {
                // make sure it is an input pin
                hr = pIPinDownstream->QueryPinInfo(&PinInfoDownstream);
                if(SUCCEEDED(hr))
                {
                    CComPtr <IPin>  pPinUp;

                    // Determine if the pin is already connected.  Note that 
                    // VFW_E_NOT_CONNECTED is expected if the pin isn't yet connected.
                    hr = pIPinDownstream->ConnectedTo (&pPinUp);
                    if(FAILED(hr) && hr != VFW_E_NOT_CONNECTED)
                    {
                        ErrorMessageBox(TEXT("Failed in pIPinDownstream->ConnectedTo()!\n"));
                        continue;
                    }

                    if ((PINDIR_INPUT == PinInfoDownstream.dir) && (pPinUp == NULL))
                    {
                        assert(pIPinUpstream);
                        assert(pIPinDownstream);

                        if (SUCCEEDED (m_pFilterGraph->Connect(pIPinUpstream, pIPinDownstream)))
                        {
                            PinInfoDownstream.pFilter->Release();
                            PinInfoUpstream.pFilter->Release();
                            return S_OK;
                        }
                    }
                }

                PinInfoDownstream.pFilter->Release();
                pIPinDownstream = NULL;

            } // while next downstream filter pin

            //We are now back into the upstream pin loop
        } // if output pin

        pIPinUpstream = NULL;
        PinInfoUpstream.pFilter->Release();

    } // while next upstream filter pin

    return E_FAIL;
}


// RunGraph checks to see if a graph has been built
// if not it calls BuildGraph
// RunGraph then calls MediaCtrl-Run
HRESULT
CBDAFilterGraph::RunGraph()
{
    // check to see if the graph is already running
    if(m_fGraphRunning)
        return S_OK;

    HRESULT hr = S_OK;

    OutputDebugString(TEXT("BDASample: CBDAFilterGraph::RunGraph\n"));

    if (m_pIMediaControl == NULL)
    {
        hr = m_pFilterGraph.QueryInterface (&m_pIMediaControl);
    }

    if (SUCCEEDED (hr))
    {
        // run the graph
        hr = m_pIMediaControl->Run();
        if(SUCCEEDED(hr))
        {
            m_fGraphRunning = true;
        }
        else
        {
            // stop parts of the graph that ran
            m_pIMediaControl->Stop();
            ErrorMessageBox(TEXT("Cannot run graph\n"));
        }
    }

    return hr;
}


// StopGraph calls MediaCtrl - Stop
HRESULT
CBDAFilterGraph::StopGraph()
{
    // check to see if the graph is already stopped
    if(m_fGraphRunning == false)
        return S_OK;

    HRESULT hr = S_OK;

    OutputDebugString(TEXT("BDASample: CBDAFilterGraph::StopGraph\n"));

    assert(m_pIMediaControl);

    // pause before stopping
    hr = m_pIMediaControl->Pause();

    // stop the graph
    hr = m_pIMediaControl->Stop();

    m_fGraphRunning = (FAILED (hr))?true:false;

    return hr;
}


// Set our client area for viewing
//
// Note, what you're not seeing here is a call to
// IAMSreamCconfig's GetFormat to obtain the video
// format properties that would enable us to set
// the viewing window's size
HRESULT CBDAFilterGraph::SetVideoWindow(
    HWND hwndMain)
{
    CComPtr <IVideoWindow> pVideoWindow;
    RECT                   rc;
    INT                    cyBorder;
    INT                    cy;
    HRESULT                hr = S_OK;

    // get IVideoWindow interface
    hr = m_pFilterGraph->QueryInterface(&pVideoWindow);
    if (FAILED(hr)) {
        ErrorMessageBox(TEXT("QueryInterface IVideoWindow Failed\n"));
        return hr;
    }

    hr = pVideoWindow->put_Owner(reinterpret_cast <LONG> (hwndMain));
    if (FAILED(hr)) {
        ErrorMessageBox(TEXT("Unable to set video window\n"));
        return hr;
    }

    hr = pVideoWindow->put_WindowStyle(WS_CHILD);
    if (FAILED(hr)) {
        ErrorMessageBox(TEXT("Unable to set the style for the video window\n"));
        return hr;
    }

    GetClientRect(hwndMain, &rc);
    cyBorder = GetSystemMetrics(SM_CYBORDER);
    cy = cyBorder;
    rc.bottom -= cy;
    hr = pVideoWindow->SetWindowPosition(0, 0, rc.right, rc.bottom);
    hr = pVideoWindow->put_Visible(OATRUE);

    pVideoWindow = NULL;

    return hr;
}

// Submit a tune request for the new
// channel.
HRESULT
CBDAFilterGraph::ChangeChannel()
{
    HRESULT hr = S_OK;

    OutputDebugString(TEXT("BDASample: CBDAFilterGraph::ChangeChannel\n"));

    if (!m_pNetworkProvider)
    {
        ErrorMessageBox(TEXT("The FilterGraph is not yet built.\n"));
        return E_FAIL;
    }

    // create tune request
    CComPtr <IDVBTuneRequest> pTuneRequest;
    hr = CreateDVBSTuneRequest(&pTuneRequest);
    if(SUCCEEDED(hr))
    {
        hr = m_pITuner->put_TuneRequest (pTuneRequest);
        if (FAILED (hr))
		{
            ErrorMessageBox(TEXT("Cannot submit tune request\n"));
		}
    }
    else
    {
        ErrorMessageBox(TEXT("Cannot Change Channels\n"));
    }

    return hr;
}


#ifdef REGISTER_FILTERGRAPH

// Adds a DirectShow filter graph to the Running Object Table,
// allowing GraphEdit to "spy" on a remote filter graph.
HRESULT CBDAFilterGraph::AddGraphToRot(
    IUnknown *pUnkGraph, 
    DWORD *pdwRegister) 
{
    CComPtr <IMoniker>            pMoniker;
    CComPtr <IRunningObjectTable> pROT;
    WCHAR wsz[128];
    HRESULT hr;

    if (FAILED(GetRunningObjectTable(0, &pROT)))
        return E_FAIL;

    wsprintfW(wsz, L"FilterGraph %08x pid %08x\0", (DWORD_PTR) pUnkGraph, 
              GetCurrentProcessId());

    hr = CreateItemMoniker(L"!", wsz, &pMoniker);
    if (SUCCEEDED(hr)) 
        hr = pROT->Register(ROTFLAGS_REGISTRATIONKEEPSALIVE, pUnkGraph, 
                            pMoniker, pdwRegister);
        
    return hr;
}

// Removes a filter graph from the Running Object Table
void CBDAFilterGraph::RemoveGraphFromRot(
    DWORD pdwRegister)
{
    CComPtr <IRunningObjectTable> pROT;

    if (SUCCEEDED(GetRunningObjectTable(0, &pROT))) 
    {
        pROT->Revoke(pdwRegister);
    }

}

#endif

// Set the audio and video PIDs on the
// demux. filter.
HRESULT
CBDAFilterGraph::SetVideoAndAudioPIDs(
	LONG lNewTHParserPID,
    LONG lNewVideoPID,
    LONG lNewAudioPID)
{
    HRESULT hr;

    OutputDebugString(TEXT("BDASample: CBDAFilterGraph::SetVideoAndAudioPIDs\n"));

    if (!m_pVideoPin)
    {
        m_pVideoPin = FindPinOnFilter(m_pDemux, "2");
        if (!m_pVideoPin)
            return E_FAIL;
    }

    if (!m_pAudioPin)
    {
        m_pAudioPin = FindPinOnFilter(m_pDemux, "3");
        if (!m_pAudioPin)
            return E_FAIL;
    }

	if (!m_pTHParserPin)
	{
        m_pTHParserPin = FindPinOnFilter(m_pDemux, "4");
        if (!m_pTHParserPin) {
			OutputDebugString(TEXT("BDASample: SetVideoAndAudioPIDs => Find m_pTHParserPin fail\n"));
            return E_FAIL;
		}
    }

    if (!m_pIVideoPIDMap)
    {
        hr = m_pVideoPin->QueryInterface(IID_IMPEG2PIDMap, (void **) &m_pIVideoPIDMap);
        if (FAILED(hr))
        {
            ErrorMessageBox(TEXT("QI of IMPEGPIDMap on video pin failed\n"));
            return E_FAIL;
        }

    }

    if (!m_pIAudioPIDMap)
    {
        hr = m_pAudioPin->QueryInterface(IID_IMPEG2PIDMap, (void **) &m_pIAudioPIDMap);
        if (FAILED(hr))
        {
            ErrorMessageBox(TEXT("QI of IMPEGPIDMap on audio pin failed\n"));
            return E_FAIL;
        }
    }

	if (!m_pITHParserPIDMap)
    {
        hr = m_pTHParserPin->QueryInterface(IID_IMPEG2PIDMap, (void **) &m_pITHParserPIDMap);
        if (FAILED(hr) || !m_pITHParserPIDMap)
        {
            ErrorMessageBox(TEXT("QI of IMPEGPIDMap on THParser pin failed\n"));
            return E_FAIL;
        }
    }


    ULONG ulVideoPID[1] = {(ULONG) lNewVideoPID};
    ULONG ulAudioPID[1] = {(ULONG) lNewAudioPID};
	ULONG ulTHParserPID[1] = {(ULONG) lNewTHParserPID};

    hr = m_pIVideoPIDMap->MapPID(1, ulVideoPID, MEDIA_ELEMENTARY_STREAM);
    if (FAILED(hr))
    {
        ErrorMessageBox(TEXT("mapping Video PID failed\n"));
        return E_FAIL;
    }

    hr = m_pIAudioPIDMap->MapPID(1, ulAudioPID, MEDIA_ELEMENTARY_STREAM);
    if (FAILED(hr))
    {
        ErrorMessageBox(TEXT("mapping Audio PID failed\n"));
        return E_FAIL;
    }

	MessagePrint("MapPID lNewVideoPID=%d, lNewAudioPID=%d, lNewTHParserPID=%d", ulVideoPID[0], ulAudioPID[0], ulTHParserPID[0]);
	hr = m_pITHParserPIDMap->MapPID(1, ulTHParserPID, MEDIA_TRANSPORT_PAYLOAD);
    if (FAILED(hr))
    {
        ErrorMessageBox(TEXT("mapping THParser PID failed\n"));
        return E_FAIL;
    }

	hr = m_pIMpeg2PsiParser->SetCAPMT(lNewVideoPID, lNewAudioPID, CAM_DEFAULT);
	if (FAILED(hr))
	{
		ErrorMessageBox(TEXT("SetCAPMT failed\n"));
		return E_FAIL;
	}
	/*
	//Get CAM module information
	THAppInfo AppInfo;
	GetAppInfo(&AppInfo);
	MessagePrint("AppInfo %d, %d  %d  %s", AppInfo.app_type, AppInfo.application_manufacture, 
					AppInfo.manufacture_code, AppInfo.application_info);
	*/

	Sleep(1000);  //Wait for get PMT data

	//Get PMT Data
	BYTE pBuff[4096];
	BYTE byBuffSize = 0;
	hr = m_pIMpeg2PsiParser->GetPMTData(pBuff, &byBuffSize);
	if (FAILED(hr))
    {
        ErrorMessageBox(TEXT("GetPMTData failed\n"));
        return E_FAIL;
    }	
	MessagePrint("GetPMTData BuffSize=%d, (%x)  (%x)  (%x)", byBuffSize, pBuff[0], pBuff[1], pBuff[2]);
	
	if (byBuffSize>0) {			
		SendCAPMT(pBuff, byBuffSize);
/*
		PARSERPMTINFO PMTInfo;
		PMTInfo.pPMTData = new BYTE [byBuffSize+1];
		memcpy(PMTInfo.pPMTData, pBuff, byBuffSize);
		PMTInfo.ulDataLen = byBuffSize;
		PMTInfo.VPid = lNewVideoPID;
		PMTInfo.APid = lNewAudioPID;		
		PMTInfo.CAM_Type = CAM_DEFAULT;	
		CI_Parser_PMT(&PMTInfo);
		delete []PMTInfo.pPMTData;
*/
	}

    m_pIVideoPIDMap = NULL;
    m_pIAudioPIDMap = NULL;
	m_pITHParserPIDMap = NULL;

    m_mappedVidPid = lNewVideoPID;
    m_mappedAudPid = lNewAudioPID;
	m_mappedTHParPid = lNewTHParserPID;
    
    return hr;
}

// Un-map the video and audio PIDs on the
// demux. filter.
HRESULT
CBDAFilterGraph::UnmapVideoAndAudioPIDs(
	LONG lOldParserPID,
    LONG lOldVideoPID,
    LONG lOldAudioPID)
{
    HRESULT hr = S_OK;

    OutputDebugString(TEXT("BDASample: CBDAFilterGraph::UnmapVideoAndAudioPIDs\n"));

    if (!m_pIVideoPIDMap || !m_pIAudioPIDMap)
        return E_FAIL;

    ULONG ulVideoPID[1] = {(ULONG) lOldVideoPID};
    ULONG ulAudioPID[1] = {(ULONG) lOldAudioPID};
	ULONG ulTHParserPID[1] = {(ULONG) lOldParserPID};

    hr = m_pIVideoPIDMap->UnmapPID(1, ulVideoPID);
    if (FAILED(hr))
    {
        ErrorMessageBox(TEXT("un-mapping Video PID failed\n"));
        return E_FAIL;
    }

    hr = m_pIAudioPIDMap->UnmapPID(1, ulAudioPID);
    if (FAILED(hr))
    {
        ErrorMessageBox(TEXT("un-mapping Audio PID failed\n"));
        return E_FAIL;
    }

	hr = m_pITHParserPIDMap->UnmapPID(1, ulTHParserPID);
    if (FAILED(hr))
    {
        ErrorMessageBox(TEXT("un-mapping THParser PID failed\n"));
        return E_FAIL;
    }

    m_pIVideoPIDMap = NULL;
    m_pIAudioPIDMap = NULL;
	m_pITHParserPIDMap = NULL;

    m_mappedVidPid = 0;
    m_mappedAudPid = 0;
	m_mappedTHParPid = 0;

    return hr;
}

// Returns a pointer address of the name of a Pin on a 
// filter.
IPin*
CBDAFilterGraph::FindPinOnFilter(
    IBaseFilter* pBaseFilter,
    char* pPinName )
{
	HRESULT hr;
	IEnumPins *pEnumPin = NULL;
	ULONG CountReceived = 0;
	IPin *pPin = NULL, *pThePin = NULL;
	char String[80];
	char* pString;
    PIN_INFO PinInfo;
	int length;

	// enumerate of pins on the filter 
	hr = pBaseFilter->EnumPins(&pEnumPin);
	if (hr == S_OK && pEnumPin)
	{
		pEnumPin->Reset();
		while (pEnumPin->Next( 1, &pPin, &CountReceived) == S_OK && pPin)
		{
			memset(String, NULL, sizeof(String));

			hr = pPin->QueryPinInfo(&PinInfo);
			if (hr == S_OK)
			{
				length = wcslen (PinInfo.achName) + 1;
				pString = new char [length];
		
				// get the pin name 
				WideCharToMultiByte(CP_ACP, 0, PinInfo.achName, -1, pString, length,
									NULL, NULL);
		
				strcat (String, pString);

				// is there a match
				if (!strcmp(String, pPinName))
					pThePin = pPin;	  // yes
				else
					pPin = NULL;	  // no

				delete pString;

			}
			else
			{
				// need to release this pin
				pPin->Release();
			}


		}	// end if have pin

		// need to release the enumerator
		pEnumPin->Release();
	}

	// return address of pin if found on the filter
	return pThePin;

}

// Gets the signal strength.
BOOL
CBDAFilterGraph::GetSignalStrength(LONG* pSignalStrength)
{
    LONG lSignalStrength;
    HRESULT hr;

    OutputDebugString(TEXT("BDASample: CBDAFilterGraph::GetSignalStrength\n"));

    hr = m_pITuner->get_SignalStrength(&lSignalStrength);
    if (FAILED(hr))
    {
        return FALSE;
    }

    *pSignalStrength = lSignalStrength;

    return TRUE;
}


// Gets the signal quality.
BOOL
CBDAFilterGraph::GetSignalQuality(LONG* pSignalQuality)
{
    OutputDebugString(TEXT("BDASample: CBDAFilterGraph::GetSignalQuality\n"));

    if ( !m_KsDemodPropSet )
    {
        ErrorMessageBox(TEXT("IKsPropertySet pointer error\n"));
        return FALSE;
    }
    
	HRESULT hr;
    ULONG   ulBytesReturned;
    LONG    lSignalQuality;
    DWORD   type_support = 0;

    // See if a demod. property set is supported
  	hr = m_KsDemodPropSet->QuerySupported(KSPROPSETID_BdaSignalStats,
                                         KSPROPERTY_BDA_SIGNAL_QUALITY, 
                                         &type_support);
    if (FAILED(hr))
    {
		OutputDebugString(TEXT("QuerySupported, KSPROPERTY_BDA_SIGNAL_QUALITY failed!!!!\n"));
	  	return FALSE;
	}
        
    if (type_support & KSPROPERTY_SUPPORT_GET)
	{
        KSPROPERTY instance_data;
    
        // make call into driver
	   	hr = m_KsDemodPropSet->Get(KSPROPSETID_BdaSignalStats, 
                              KSPROPERTY_BDA_SIGNAL_QUALITY, 
	  						  &instance_data, sizeof(instance_data),
                              &lSignalQuality, sizeof(lSignalQuality),
                              &ulBytesReturned);
        if (FAILED(hr))
        {
            ErrorMessageBox(TEXT("KSPROPERTY_BDA_SIGNAL_QUALITY, GET failed !!!!\n"));
            return FALSE;
        }
	}
	else
	{
        ErrorMessageBox(TEXT("KSPROPERTY_BDA_SIGNAL_QUALITY, GET not supported !!!!\n"));
        return FALSE;
	}

    *pSignalQuality = lSignalQuality;

    return TRUE;
}


//
// USE THE CODE BELOW IF YOU WANT TO MANUALLY LOAD AND
// CONNECT A/V DECODERS TO THE DEMUX OUTPUT PINS
//

//
// To use this code:
// 1) in LoadAudioDecoder() and LoadVideoDecoder(), fill in decoder specific information (clsid)
// 2) goto BuildGraph() and replace RenderDemux() with BuildAVSegment()
//


// Builds the Audio, Video segment of the digital TV graph.
// Demux -> AV Decoder -> OVMixer -> Video Renderer
/*HRESULT
CBDAFilterGraph::BuildAVSegment()
{
    HRESULT hr = E_FAIL;

    // connect the demux to the capture device
    hr = ConnectFilters(m_pCaptureDevice, m_pDemux);

    hr = LoadVideoDecoder();
    if(SUCCEEDED(hr) && m_pVideoDecoder)
    {
        // Connect the demux & video decoder
        hr = ConnectFilters(m_pDemux, m_pVideoDecoder);
        if(FAILED(hr))
        {
            ErrorMessageBox(TEXT("Connecting Demux & Video Decoder Failed\n"));
            goto err;
        }
    }
    else
    {
        ErrorMessageBox(TEXT("Unable to load Video Decoder\n"));
        goto err;
    }

    //Audio
    hr = LoadAudioDecoder();
    if(SUCCEEDED(hr) && m_pAudioDecoder)
    {
        hr = ConnectFilters(m_pDemux, m_pAudioDecoder);
        if(FAILED(hr))
        {
            ErrorMessageBox(TEXT("Connecting Deumx & Audio Decoder Failed\n"));
            goto err;
        }
    }
    else
    {
        ErrorMessageBox(TEXT("Unable to load Audio Decoder\n"));
        goto err;
    }

    // Create the OVMixer & Video Renderer for the video segment
    hr = CoCreateInstance(CLSID_OverlayMixer, NULL, CLSCTX_INPROC_SERVER,
            IID_IBaseFilter, reinterpret_cast<void**>(&m_pOVMixer));

    if(SUCCEEDED(hr) && m_pOVMixer)
    {
        hr = m_pFilterGraph->AddFilter(m_pOVMixer, L"OVMixer");
        if(FAILED(hr))
        {
            ErrorMessageBox(TEXT("Adding OVMixer to the FilterGraph Failed\n"));
            goto err;
        }
    }
    else
    {
        ErrorMessageBox(TEXT("Loading OVMixer Failed\n"));
        goto err;
    }

    hr = CoCreateInstance(CLSID_VideoRenderer, NULL, CLSCTX_INPROC_SERVER,
            IID_IBaseFilter, reinterpret_cast<void**>(&m_pVRenderer));
    if(SUCCEEDED(hr) && m_pVRenderer)
    {
        hr = m_pFilterGraph->AddFilter(m_pVRenderer, L"Video Renderer");
        if(FAILED(hr))
        {
            ErrorMessageBox(TEXT("Adding Video Renderer to the FilterGraph Failed\n"));
            goto err;
        }
    }
    else
    {
        ErrorMessageBox(TEXT("Loading Video Renderer Failed\n"));
        goto err;
    }

    // Split AV Decoder? Then add Default DirectSound Renderer to the filtergraph
    if(m_pVideoDecoder != m_pAudioDecoder)
    {
        hr = CoCreateInstance(CLSID_DSoundRender, NULL,
                        CLSCTX_INPROC_SERVER, IID_IBaseFilter,
                        reinterpret_cast<void**>(&m_pDDSRenderer));
        if(SUCCEEDED(hr) && m_pDDSRenderer)
        {
            hr = m_pFilterGraph->AddFilter(m_pDDSRenderer, L"Sound Renderer");
            if(FAILED(hr))
            {
                ErrorMessageBox(TEXT("Adding DirectSound Device to the FilterGraph Failed\n"));
                goto err;
            }
        }
        else
        {
            ErrorMessageBox(TEXT("Loading DirectSound Device Failed\n"));
            goto err;
        }
    }

    hr = ConnectFilters(m_pVideoDecoder, m_pOVMixer);
    if(FAILED(hr))
    {
        ErrorMessageBox(TEXT("Connecting Capture & OVMixer Failed\n"));
        goto err;
    }

    hr = ConnectFilters(m_pOVMixer, m_pVRenderer);
    if(FAILED(hr))
    {
        ErrorMessageBox(TEXT("Connecting OVMixer & Video Renderer Failed\n"));
        goto err;
    }

    // Split AV Decoder & if you need audio too ?? then connect Audio decoder to Sound Renderer
    if(m_pVideoDecoder != m_pAudioDecoder)
    {
        hr = ConnectFilters(m_pAudioDecoder, m_pDDSRenderer);

        if(FAILED(hr))
        {
            ErrorMessageBox(TEXT("Connecting AudioDecoder & DirectSound Device Failed\n"));
            goto err;
        }
    }

err:
    return hr;
}

// placeholders for real decoders
//DEFINE_GUID(CLSID_FILL_IN_NAME_AUDIO_DECODER, 0xFEEDFEED, 0x0000, 0x0000, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
//0x00);
DEFINE_GUIDSTRUCT("7E2E0DC1-31FD-11D2-9C21-00104B3801F6", CLSID_FILL_IN_NAME_AUDIO_DECODER);
#define CLSID_AudioDecoder DEFINE_GUIDNAMED(CLSID_FILL_IN_NAME_AUDIO_DECODER)
//DEFINE_GUID(CLSID_FILL_IN_NAME_AUDIO_DECODER, 0x7E2E0DC1, 0x31FD, 0x11D2, 0x9C, 0x21, 0x00, 0x10, 0x4B, 0x38, 0x01, 0xF6);

//DEFINE_GUID(CLSID_FILL_IN_NAME_VIDEO_DECODER, 0xFEEDFEED, 0x0000, 0x0000, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
//0x00);
DEFINE_GUIDSTRUCT("0246CA20-776D-11D2-8010-00104B9B8592", CLSID_FILL_IN_NAME_VIDEO_DECODER);
#define CLSID_VideoDecoder DEFINE_GUIDNAMED(CLSID_FILL_IN_NAME_VIDEO_DECODER)
//DEFINE_GUID(CLSID_FILL_IN_NAME_VIDEO_DECODER, 0x0246CA20, 0x77CD, 0x11D2, 0x80, 0x10, 0x00, 0x10, 0x4B, 0x9B, 0x85, 0x92);


HRESULT
CBDAFilterGraph::LoadVideoDecoder()
{
    HRESULT hr = E_FAIL;

    //hr = CoCreateInstance(CLSID_FILL_IN_NAME_VIDEO_DECODER, NULL,
    hr = CoCreateInstance(CLSID_VideoDecoder, NULL,
            CLSCTX_INPROC_SERVER, IID_IBaseFilter,
            reinterpret_cast<void**>(&m_pVideoDecoder));

    if(SUCCEEDED(hr) && m_pVideoDecoder)
    {
        hr = m_pFilterGraph->AddFilter(m_pVideoDecoder, L"Video Decoder");

        if(FAILED(hr))
        {
            ErrorMessageBox(TEXT("Unable to add Video Decoder filter to graph\n"));
        }
    }

    return hr;
}


HRESULT
CBDAFilterGraph::LoadAudioDecoder()
{
    HRESULT hr = E_FAIL;

    //hr = CoCreateInstance(CLSID_FILL_IN_NAME_AUDIO_DECODER, NULL,
    hr = CoCreateInstance(CLSID_AudioDecoder, NULL,
            CLSCTX_INPROC_SERVER, IID_IBaseFilter,
            reinterpret_cast<void**>(&m_pAudioDecoder));

    if(SUCCEEDED(hr) && m_pAudioDecoder)
    {
        hr = m_pFilterGraph->AddFilter(m_pAudioDecoder, L"Audio Decoder");

        if(FAILED(hr))
        {
            ErrorMessageBox(TEXT("Unable to add Audio filter to graph\n"));
        }
    }

    return hr;
}*/

BOOL CBDAFilterGraph::CheckBDAInterface()
{
    BOOL bStatus = FALSE;
    DWORD dwBytesReturned = 0;

    bStatus = BDAIOControl( THBDA_IOCTL_CHECK_INTERFACE,
											NULL,
											0,
											NULL,
											0,
											(LPDWORD)&dwBytesReturned);
    
    return bStatus;
}

BOOL CBDAFilterGraph::BDAIOControl( DWORD  dwIoControlCode,
									LPVOID lpInBuffer,
									DWORD  nInBufferSize,
									LPVOID lpOutBuffer,
									DWORD  nOutBufferSize,
									LPDWORD lpBytesReturned)
{
    if (!m_KsTunerPropSet)
        return FALSE;

    KSPROPERTY instance_data;

    ULONG    ulOutBuf = 0;
    ULONG    ulReturnBuf = 0;
    THBDACMD THBDACmd;

    THBDACmd.CmdGUID = GUID_THBDA_CMD;
    THBDACmd.dwIoControlCode = dwIoControlCode;
    THBDACmd.lpInBuffer = lpInBuffer;
    THBDACmd.nInBufferSize = nInBufferSize;
    THBDACmd.lpOutBuffer = lpOutBuffer;
    THBDACmd.nOutBufferSize = nOutBufferSize;
    THBDACmd.lpBytesReturned = lpBytesReturned;

    HRESULT hr = m_KsTunerPropSet->Set(GUID_THBDA_TUNER, 
                              NULL, 
	  						  &instance_data, sizeof(instance_data),
                              &THBDACmd, sizeof(THBDACmd));

    if (FAILED(hr))
        return FALSE;
    else
        return TRUE;
}

BOOL CBDAFilterGraph::THBDA_IOCTL_SET_REG_PARAMS_Fun(PTHBDAREGPARAMS pTH_Params)
{	
	BOOLEAN bResult	= FALSE;
	int     nBytes  = 0;	

	bResult = BDAIOControl(	THBDA_IOCTL_SET_REG_PARAMS,
							(LPVOID)pTH_Params, 
							sizeof(THBDAREGPARAMS),     
							NULL, 
							0,                    
							(unsigned long *)&nBytes);
	if (bResult==FALSE)	{
		OutputDebugString(TEXT("IOCTL Error: THBDA_IOCTL_SET_REG_PARAMS failed! \n"));
	}

	return bResult;
}

BOOL CBDAFilterGraph::GetAppInfo(PTHAppInfo pAppInfo)
{
    BOOL bStatus = FALSE;
    DWORD dwBytesReturned = 0;

    bStatus = BDAIOControl( THBDA_IOCTL_CI_GET_APP_INFO,
							NULL,
							0,
							(LPVOID)pAppInfo,
							sizeof(THAppInfo),
							(LPDWORD)&dwBytesReturned);
    
    return bStatus;
}

BOOL CBDAFilterGraph::CI_Parser_PMT(PPARSERPMTINFO pPMTInfo)
{
	BOOL bStatus = FALSE;
    DWORD dwBytesReturned = 0;

    bStatus = BDAIOControl( THBDA_IOCTL_CI_PARSER_PMT,
							(LPVOID)pPMTInfo,
							sizeof(PARSERPMTINFO),
							NULL,
							0,
							(LPDWORD)&dwBytesReturned);
    
    return bStatus;
}

BOOL CBDAFilterGraph::SendCAPMT(PBYTE pBuff, BYTE byBuffSize)
{
    BOOL bStatus = FALSE;
    DWORD dwBytesReturned = 0;

    bStatus = BDAIOControl( THBDA_IOCTL_CI_SEND_PMT,
							(LPVOID)pBuff,
							byBuffSize,
							NULL,
							0,
							(LPDWORD)&dwBytesReturned);
    
    return bStatus;
}

BOOL CBDAFilterGraph::ChangeSetting(void)
{
	HRESULT hr = S_OK;

	OutputDebugString(TEXT("BDASample: ChangeSetting\n"));
	
	// create a tune request to initialize the network provider
    // before connecting other filters
    CComPtr <IDVBTuneRequest> pDVBSTuneRequest;
    if (FAILED(hr = CreateDVBSTuneRequest(&pDVBSTuneRequest)))
    {
        ErrorMessageBox(TEXT("Cannot create tune request\n"));
        BuildGraphError();
        return hr;
    }

    //submit the tune request to the network provider
    hr = m_pITuner->put_TuneRequest(pDVBSTuneRequest);
    if (FAILED(hr)) {
        ErrorMessageBox(TEXT("Cannot submit the tune request\n"));
        BuildGraphError();
        return hr;
    }

	m_TH_Params.ulMCEFreqTranslate = 0;		// MCE frequency table translate	
    m_TH_Params.ulFixedBW;					// The fixed bandwidth in registry
    m_TH_Params.ulShiftFreqScan;			// Disable Lock Shift in registry
    
    m_TH_Params.ulLNBLOF = m_LNB_TYPE;		// Local Oscillator Frequency (LOF)
    m_TH_Params.ulLNBLOFLowBand = m_LNB_LOF;// LOF Low Band
    m_TH_Params.ulLNBLOFHighBand = m_LNB_HOF;// LOF High Band
    m_TH_Params.ulDiSEqC = m_DISEQC;		// DiSEqC Selection
    m_TH_Params.ulLNBPower = m_LNB_POWER;   // LNB Power ON/OFF
    m_TH_Params.ulTone = m_K22_Tone;		// Tone ON/OFF
    
	if (!THBDA_IOCTL_SET_REG_PARAMS_Fun(&m_TH_Params)) {
		ErrorMessageBox(TEXT("THBDA_IOCTL_SET_REG_PARAMS_Fun failed\n"));
		BuildGraphError();
		return S_FALSE;
	}

	OutputDebugString(TEXT("BDASample: ChangeSetting OK\n"));
	return hr;
}


VOID MessagePrint(LPTSTR szFormat, ...)
{
    static TCHAR szBuffer[2048];  // Large buffer
    const size_t NUMCHARS = sizeof(szBuffer) / sizeof(szBuffer[0]);
    const int LASTCHAR = NUMCHARS - 1;

    // Format the input string
    va_list pArgs;
    va_start(pArgs, szFormat);

    // Use a bounded buffer size to prevent buffer overruns.  Limit count to
    // character size minus one to allow for a NULL terminating character.
    _vsntprintf(szBuffer, NUMCHARS - 1, szFormat, pArgs);
    va_end(pArgs);

    // Ensure that the formatted string is NULL-terminated
    szBuffer[LASTCHAR] = TEXT('\0');

    // Display a message box with the formatted string
    OutputDebugString(szBuffer);
}