/*
   LCDCall -- C subroutines used to call LCDBios

   version 1.31
   written by Donald Sawdai, 11/95
   copyright 1995, 1996 Donald Sawdai

   This code may be freely used, distributed, and compiled royalty-free into
   applications as long as it is unmodified in any way.  Donald Sawdai
   gives no warantee of any sort for this code.  However, I am always
   interested in comments (send to dsawdai@engin.umich.edu).

   requirements:  AT class machine, 386 or higher
		  register-compatible VGA card
		  compatible 3D glasses

   note:	  This has been compiled and tested using 386 instruction set
		  and (1) real-mode small model on Borland C/C++ 3.1
		  and (2) protected-mode on Watcom C/C++ 10.5 with the
		  DOS/4GW extender.

   See documentation for details.

   Current limitations/bugs:
   (1) Any program interference with the AT Real-Time Clock (RTC) or with VGA
       registers (CRTC Offset register, CRTC Max Scan Line register,
       CRTC Vertical Retrace End register, or CRTC Start Address High/Low
       registers) while 3D mode is on may conflict with this program.  The
       most likely conflict is screen flipping for buffered screen draws.
       The screen flipping (multiple buffering) must be done through
       LCDBios functions.
*/


typedef enum {
/* These are the different modes that LCDBIOS can operate in. */
Off,           /* 3D mode off */
TwoBuffer,     /* 2 full separate buffers: one for each eye,
	       each at the same resolution as the screen. */
AlternateFull, /* 1 line-alternate buffer, twice as high as the
	       screen resolution.  The image for each eye is at
	       the same resolution as the screen. */
AlternateHalf, /* 1 line-alternate buffer, same resolution as the
	       screen.  The image for each eye is half the height
	       of the screen. */
} LCD3DMode;


typedef struct {
/* These are the VESA Start Addresses (location of the upper left-hand
corner of the left- and right-eye buffers, in X-Y pixel coordinates)
that would normally be sent to the VESA function call to set Start
Addresses.  For the AlternateFull and AlternateHalf 3D modes, only the
leftX and leftY fields are used.
*/
	unsigned short int leftX;
	unsigned short int leftY;
	unsigned short int rightX;
	unsigned short int rightY;
} VESAPageAddresses;


typedef struct {
/* These are the VGA Start Addresses (VGA memory addresses of the
beginning of the left- and right-eye buffers) that would normally be
sent to the VGA CRTC registers.  The high byte of each word is
analagous to the Start Address High register.  The low byte of each
word is analagous to the Start Address Low register.  The actual VGA
Start Address registers should not be altered by the application
program.  Note that the "pad" variables are reserved for future use.
For the AlternateFull and AlternateHalf 3D modes, only the
left field is used.
*/
	unsigned short int left;
	unsigned short int pad1;
	unsigned short int right;
	unsigned short int pad2;
} VGAPageAddresses;


typedef union {
/* This union contains the page addresses for one pair of buffers
(left- and right-eye).  The description may be in either VGA or in
VESA format, as defined above.
*/
	VESAPageAddresses        VESA;
	VGAPageAddresses         VGA;
} PageAddresses;


#pragma pack ( 2 );       	/* make sure data structure is word aligned */
typedef struct {
/* These are the data flags that can be read by the calling program to
determine when LCDBios events occur.
   pageAdvanced is a LCDBios flag that indicates when vertical retrace has
occurred.  The flag is set to 1 when every vertical retrace begins when 3D
mode is on and calibration has finished.  When LCDBiosAdvancePage is called,
this value is reset to 0.  When the page is actually advanced (at vertical
retrace), it is set to 1.  The application program may also set the flag to 0
(or any other value).  It will still be set to 1 when the next vertical
retrace begins (if 3D mode is on).  Note that when 3D mode is entered,
pageAdvanced is set to 2.  After the first retrace after the calibration
period (about 0.8 seconds later), pageAdvanced is set to 1.
   interruptCount is a LCDBios flag the is incremented every time that the
real-time clock (RTC) interrupt goes off.  This is the interrupt that
LCDBios monitors.  It normally goes off 1024 times/second.  However,
the user can increase or decrease the interrupt rate by 2, 4, or 8 times
using the /FastInt:x switch when running LCDBios.  Also note that LCDBios
occasionally misses a RTC tick, so that this value may not be very accurate.
The application program may reset this value to 0.  Then interruptCount
will contain the number of RTC interrupts since it was reset.
   delayCount is incremented every time that LCDBios incurs a significant
delay in the program execution.  Since LCDBios interrupts the main program
and waits for vertical retrace when /LockFlip is specified, this may affect
the timing of the application.  So this field is incremented to let the
application know that a delay occurred.  This is very useful in joystick
timing loops.
*/
  short int pageAdvanced;
  long int interruptCount;
  long int delayCount;
} LCDFlags;
#pragma pack;

/* These are the *pointers* to the actual flags structure, which is in
LCDBios program memory (in DOS 1MB memory).  They are only a valid pointer when
3D mode is active (LCDBiosInit3DMode).  At all other times, they point to NULL.
*/
extern volatile LCDFlags far *LCDFlagsR;
/* LCDFlagsR is a real-mode far pointer */
extern volatile LCDFlags *LCDFlagsP;
/* LCDFlagsP is a protected-mode flat pointer.  It is the absolute
address of the structure in the DOS 1MB memory. */


int isLCDBiosInstalled( int *majorVersion, int *minorVersion);
/* Return:     1 if LCDBIOS is installed.
	       0 if it is not.
   Output:     majorVersion = major version number
	       minorVersion = minor version number (0-99)

   Example:    Version 1.05 returns a majorVersion=1, minorVersion=5.
*/


int LCDBiosInit3DMode( LCD3DMode mode, int numberOfPages,
		       PageAddresses *pages );
/* Return:     1 on success.  0 on failure.
   Input:      mode = [ TwoBuffer, AlternateFull, AlternateHalf]
	       numberOfPages = number of stereo pages to use.
		    Each page consists of a left buffer and a
		    right buffer.  For example, in TwoBuffer
		    mode, if there are 2 left buffers and 2 right
		    buffers, it is still considered to be 2
		    pages.
	       pages = pointer to array of PageAddresses (one
		    per page, as specified by numberOfPages).
		    For TwoBuffer mode, both left- and right-
		    buffer addresses must be specified for each
		    page.  For Alternate... modes, only the left-
		    buffer addresses should be used.  The pages are
		    numbered sequentially starting at 0.
Description:
LCDBIOS puts the screen in 3D mode (as specified by "mode") and
activates the shutter glasses.  The current page is page 0.
LCDFlags... pointers are set to point to the LCDFlags structure in LCDBios.
pageAdvanced is set to 2 until the calibration period is completed.  Then, after
the first 3D vertical retrace, pageAdvanced is set to 1.  The video card must be
in a VGA or VESA graphics mode prior to calling
this function.
*/


int LCDBiosDeinit3DMode( void );
/* Return:     1 on success.  0 on failure.
Description:
LCDBIOS removes the screen from 3D mode.  LCDFlags... pointers are
set to NULL.
*/


int LCDBiosAdvancePage( void );
/* Return:     1 on success.  0 on failure.
Description:
Advances to next page in "pages" array specified by LCDBiosInit3DMode.
If the current page is the last page in the array, it is advanced
circularly to page 0.  This function also resets the pageAdvanced flag
to 0.  When the page is actually advanced, the pageAdvanced flag is
set to 1 by the timer interrupt.
*/


int LCDBiosSetPage( int page );
/* Return:     1 on success.  0 on failure.
Description:
Sets the current page to "page" in pages array specified by
LCDBiosInit3DMode.  Resets pageAdvanced flag to 0.  When the page is
actually advanced, pageAdvanced flag is set to 1 by the timer
interrupt.
*/

void LCDBiosPageWait( void );
/*
Description:
Waits until the current page is actually advanced (vertical retrace).  If
3D mode is not on, or if 3D mode is in its calibration period
(pageAdvanced = 2), then this function returns immediately.
*/
