// Copyright (C) 1998-1999 Scott Cutler
// Please see the "readme.txt" file for license details

#include "include.h"

LPDIRECTDRAWSURFACE4 getbuffer(uint32);


struct XGLLockTemp {
	LPDIRECTDRAWSURFACE4	DDS;
	DDSURFACEDESC2			DDSD;
	XGLLock					lock;
	uint32					lockemu;
	uchar					*tempsurface;
	uint32					lockedby[XGLLOCK_MAX];
	XGLLockTemp() { 
		tempsurface = 0;
		lockemu = 0;
		lockedby[XGLLOCK_READONLY] = 0;
		lockedby[XGLLOCK_WRITEONLY] = 0;
	}
	int locked() { 
		return (lockedby[XGLLOCK_READONLY] || lockedby[XGLLOCK_WRITEONLY]); 
	}
	int islastlock(uint32 type) {
		uint32 curlocks[XGLLOCK_MAX];

		curlocks[XGLLOCK_READONLY] = lockedby[XGLLOCK_READONLY];
		curlocks[XGLLOCK_WRITEONLY] = lockedby[XGLLOCK_WRITEONLY];

		curlocks[type] = 0;

		return (!curlocks[XGLLOCK_READONLY] && !curlocks[XGLLOCK_WRITEONLY]);
	}
};

XGLLockTemp locks[XGLBUF_MAX];




int unlock(uint32 buf, uint32 type) {
	vlog("Called unlock: type=" << type);

	// If both types are locked, then just clear the flag
	// We also don't want to unlock if neither flag is set
	if (locks[buf].locked()) {

		locks[buf].lockedby[type] = 0;

		// Clear the flag, and check.  If there are no more outstanding locks, then we can unlock
		if (!locks[buf].locked()) {
			if (locks[buf].DDS->Unlock(0) != D3D_OK) {
				log("  Error unlocking surface in unlock()");
				return XGLFALSE;
			} else {
				vlog("  Physically unlocked surface");
			}
		}
	} else {
		locks[buf].lockedby[type] = 0;
	}

	return XGLTRUE;
}


int getlock(uint32 buf, uint32 type) {
	vlog("Called getlock: buffer=" << buf << "  type=" << type);

	_exebuffer->Flush();

	if (!locks[buf].locked()) {
		vlog("  Surface not locked, trying to obtain...");

		locks[buf].DDS = getbuffer(buf);

		locks[buf].DDSD.dwSize = sizeof(DDSURFACEDESC2);

		if (locks[buf].DDS->Lock(0, &locks[buf].DDSD, DDLOCK_WRITEONLY|DDLOCK_READONLY|DDLOCK_WAIT, 0) != D3D_OK) {
			log("  Error locking surface in getlock()"); 
			return XGLFALSE;
		} else {
			vlog("  Obtained physical lock on surface");
		}
	}
	locks[buf].lockedby[type] = 1;

	vlog("  Finished getlock");

	return XGLTRUE;
}





XGLLINK uint32 XGLCALL grLfbLock(uint32 type, uint32 buf, uint32 mode, uint32 origin, uint32 temp, XGLLock *lock) {

	uint32 i, j;

	vlog("Called grLfbLock");
	vlog("  type=" << type);
	vlog("  buffer=" << buf);
	vlog("  mode=" << mode);
	vlog("  origin=" << origin);

	// We can do a normal, optimized lock
//	if (locks[type].lock.mode == XGLLFB_FMT_RGB565 || locks[type].lock.mode == XGLLFB_FMT_ANY) {
	if (mode == XGLLFB_FMT_RGB565 || mode == XGLLFB_FMT_ANY) {
//	if (0) {
		getlock(buf, type);

		locks[buf].lock.size = sizeof(XGLLock);
		locks[buf].lock.surface = locks[buf].DDSD.lpSurface;
		locks[buf].lock.pitch = locks[buf].DDSD.lPitch;
		locks[buf].lock.mode = mode;
		locks[buf].lock.origin = origin;
		*lock = locks[buf].lock;
		locks[buf].lockemu = 0;
	} else if (!locks[buf].locked()) {
		// Must emulate the 3dfx pixel pipeline.
		// Copy screen to a buffer and convert color format.
		getlock(buf, type);

		vlog("  Not locked, copying surface");

		int dbpp = getLfbBytesPerPixel(mode);
		int sbpp = 2;

		if (!locks[buf].tempsurface) {
			locks[buf].tempsurface = new uchar[Voodoo.width*Voodoo.height*2];
		}

		locks[buf].lock.size = sizeof(XGLLock);
		locks[buf].lock.surface = locks[buf].tempsurface;
		locks[buf].lock.pitch = Voodoo.width*dbpp;
		locks[buf].lock.mode = mode;
		locks[buf].lock.origin = origin;
		locks[buf].lockemu = 1;
		*lock = locks[buf].lock;


		void (*bufcolorconv)(uchar *, uchar *);

		switch (mode) {
		case XGLLFB_FMT_RGB565:		bufcolorconv = RGB565toRGB565;		break;
		case XGLLFB_FMT_RGB555:		bufcolorconv = RGB565toRGB555;		break;
		case XGLLFB_FMT_ARGB1555:	bufcolorconv = RGB565toARGB1555;	break;
		case XGLLFB_FMT_ANY:		bufcolorconv = RGB565toRGB565;		break; 
		default:
			bufcolorconv = RGB565toRGB565;
			log("   Unsupported write mode=" << mode << " in grLfbLock");
		}

		uchar *ds = (uchar *)lock->surface;
		uchar *ss = (uchar *)locks[buf].DDSD.lpSurface;
		uint32 dpitch = Voodoo.width * dbpp;
		uint32 spitch = locks[buf].DDSD.lPitch;

		// Needs to be optimized, currently does one pixel at a time.
		for (i=0; i<Voodoo.height; i++) {
			for (j=0; j<Voodoo.width; j++) {
				bufcolorconv(ds + (i*dpitch + j*dbpp), ss + (i*spitch + j*sbpp));
			}
		}

//		if (!alreadylocked) unlock(buf, XGLLOCK_READONLY);
	} else {
		*lock = locks[buf].lock;
	}

	return XGLTRUE;
}


XGLLINK uint32 XGLCALL grLfbUnlock(uint32 type, uint32 buf) {
	vlog("Called grLfbUnlock: type=" << type << " buffer=" << buf);

	uint32 i, j;

	_exebuffer->Flush();


	if (locks[buf].islastlock(type) && locks[buf].lockemu) {
		vlog("  Last lock, copy surface back");

		int sbpp = getLfbBytesPerPixel(locks[buf].lock.mode);
		int dbpp = 2;

		void (*bufcolorconv)(uchar *, uchar *);

		switch (locks[buf].lock.mode) {
		case XGLLFB_FMT_RGB565:		bufcolorconv = RGB565toRGB565;		break;
		case XGLLFB_FMT_RGB555:		bufcolorconv = RGB555toRGB565;		break;
		case XGLLFB_FMT_ARGB1555:	bufcolorconv = ARGB1555toRGB565;	break;
		case XGLLFB_FMT_ANY:		bufcolorconv = RGB565toRGB565;		break; 
		default:
			bufcolorconv = RGB565toRGB565;
			log("   Unsupported lock write mode=" << locks[buf].lock.mode << " in grLfbUnlock");
		}

		uchar *ds = (uchar *)locks[buf].DDSD.lpSurface;
		uchar *ss = (uchar *)locks[buf].lock.surface;
		uint32 dpitch = locks[buf].DDSD.lPitch;
		uint32 spitch = Voodoo.width * sbpp;

		// Needs to be optimized, currently does one pixel at a time.
		for (i=0; i<Voodoo.height; i++) {
			for (j=0; j<Voodoo.width; j++) {
				bufcolorconv(ds+(i*dpitch + j*dbpp), ss+(i*spitch + j*sbpp));
			}
		}

	}
	unlock(buf, type);

	return XGLTRUE;
}


XGLLINK uint32 XGLCALL grLfbWriteRegion(uint32 buf,
										 uint32 xoffset,
										 uint32 yoffset,
										 uint32 format,
										 uint32 width,
										 uint32 height,
										 uint32 pitch,
										 uchar *data ) {

	log("Called grLfbWriteRegion");
	log("  buffer=" << buf);
	log("  xoffset=" << xoffset);
	log("  yoffset=" << yoffset);
	log("  format=" << format);
	log("  width=" << width);
	log("  height=" << height);
	log("  pitch=" << pitch);

	int needslock = !locks[buf].locked();

	if (needslock) getlock(buf, XGLLOCK_WRITEONLY);

	uint32 i, j;

	void (*bufcolorconv)(uchar *, uchar *);

	switch (format) {
	case XGLLFB_FMT_RGB565:		bufcolorconv = RGB565toRGB565;		break;
	case XGLLFB_FMT_RGB555:		bufcolorconv = RGB555toRGB565;		break;
	case XGLLFB_FMT_ARGB1555:	bufcolorconv = ARGB1555toRGB565;	break;
	default:
		bufcolorconv = RGB565toRGB565;
		log("   Unsupported format=" << format << " in grLfbWriteRegion");
		break;
	}

	if (format == XGLLFB_FMT_RGB565 || format == XGLLFB_FMT_ANY) {
		// Program requested format the same as the color buffer

		uchar *ds = (uchar *)locks[buf].DDSD.lpSurface;
		uint32 dpitch = locks[buf].DDSD.lPitch;
		uint32 bpp = 2;

		for (i=0; i<height; i++) {
			memcpy(ds+(i+yoffset)*dpitch+xoffset*bpp, data+(i*pitch), width*bpp);
		}
	} else {
		// We have to emulate the 3dfx pixel pipeline, so convert the pixel while copying

		uchar *ds = (uchar *)locks[buf].DDSD.lpSurface;
		uint32 dpitch = locks[buf].DDSD.lPitch;

		for (i=0; i<height; i++) {
			for (j=0; j<width; j++) {
				bufcolorconv(ds+((i+yoffset)*dpitch+(xoffset+j)*2), data+(i*pitch)+j*2);
			}
		}
	}

	if (needslock) unlock(buf, XGLLOCK_WRITEONLY);

	return XGLTRUE;
}


XGLLINK uint32 XGLCALL grLfbReadRegion(uint32 buf,
										uint32 xoffset,
										uint32 yoffset,
										uint32 width,
										uint32 height,
										uint32 pitch,
										uchar *data) {

	log("  Called grLfbReadRegion");

	_exebuffer->Flush();

	int needslock = !locks[buf].locked();

	if (needslock) getlock(buf, XGLLOCK_READONLY);

	uint32 i;
	int bpp = 2;

	uchar *ds = (uchar *)locks[buf].DDSD.lpSurface;
	uint32 spitch = locks[buf].DDSD.lPitch;

	for (i=0; i<height; i++) {
		memcpy(data+i*pitch, ds+(i+yoffset)*spitch+xoffset*bpp, width*bpp);
	}

	if (needslock) unlock(buf, XGLLOCK_READONLY);

	return XGLTRUE;
}





LPDIRECTDRAWSURFACE4 getbuffer(uint32 buffer) {
	switch (buffer) {
	case XGLBUF_FRONT:		return DX.DDSPrimary;
	case XGLBUF_BACK:		return DX.DDSRender;
	case XGLBUF_AUX:
	case XGLBUF_DEPTH:
	case XGLBUF_ALPHA:
	case XGLBUF_TRIPLE:
	default:				return DX.DDSRender;
	}
}

void unlockall() {
		unlock(XGLBUF_FRONT, XGLLOCK_READONLY);
		unlock(XGLBUF_FRONT, XGLLOCK_WRITEONLY);
		unlock(XGLBUF_BACK, XGLLOCK_READONLY);
		unlock(XGLBUF_BACK, XGLLOCK_WRITEONLY);
		unlock(XGLBUF_AUX, XGLLOCK_READONLY);
		unlock(XGLBUF_AUX, XGLLOCK_WRITEONLY);
		unlock(XGLBUF_DEPTH, XGLLOCK_READONLY);
		unlock(XGLBUF_DEPTH, XGLLOCK_WRITEONLY);
		unlock(XGLBUF_ALPHA, XGLLOCK_READONLY);
		unlock(XGLBUF_ALPHA, XGLLOCK_WRITEONLY);
		unlock(XGLBUF_TRIPLE, XGLLOCK_READONLY);
		unlock(XGLBUF_TRIPLE, XGLLOCK_WRITEONLY);
}

