/* -*- mode: C; mode: fold; -*- */
/*  Copyright (c) 1997 John E. Davis (davis@space.mit.edu)
 *  All rights reserved.
 *
 * Most of the code in this file came from Bjoern Frantzen, bjoff@bgnett.no
 */

#include "config.h"
#include "slrnfeat.h"

#include <stdio.h>
#include <string.h>

#ifdef HAVE_STDLIB_H
# include <stdlib.h>
#endif

#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif

#ifdef VMS
# include "vms.h"
#endif

#include <slang.h>
#include "jdmacros.h"

#include "misc.h"
#include "util.h"
#include "chmap.h"
#include "group.h"
#include "art.h"

#if SLRN_HAS_CHARACTER_MAP
char *Slrn_Charset;

static char File_Error [] = "File error:";
static char Not_Posted [] = "--- message not posted.";

static unsigned char *ChMap_To_Iso_Map;
static unsigned char *ChMap_From_Iso_Map;

/* This include file contains static globals */
# include "charmaps.h"

static void chmap_map_string (char *str, unsigned char *map)
{
   unsigned char ch;

   if (map == NULL)
     return;

   while (0 != (ch = *str))
     *str++ = map[ch];
}

static void chmap_map_string_from_iso (char *str)
{
   chmap_map_string (str, ChMap_From_Iso_Map);
}

static void chmap_map_string_to_iso (char *str)
{
   chmap_map_string (str, ChMap_To_Iso_Map);
}

#endif

/* We fix the rest of the header later from hide_art_headers() */
void slrn_chmap_fix_headers (void)
{
#if SLRN_HAS_CHARACTER_MAP
   Slrn_Header_Type *h = Slrn_First_Header;

   while (h != NULL)
     {
	if ((h->flags & HEADER_CHMAP_PROCESSED) == 0)
	  {
	     chmap_map_string_from_iso (h->subject);
	     chmap_map_string_from_iso (h->from);
	     h->flags |= HEADER_CHMAP_PROCESSED;
	  }
        h = h->real_next;
     }
#endif
}

void slrn_chmap_fix_body (void)
{
#if SLRN_HAS_CHARACTER_MAP
   Slrn_Article_Line_Type *ptr = Slrn_Article_Lines;

   while (ptr != NULL)
     {
        chmap_map_string_from_iso (ptr->buf);
        ptr = ptr->next;
     }
#endif
}

int slrn_chmap_fix_file (char *file)
{
#if SLRN_HAS_CHARACTER_MAP
   FILE *fp, *tmpfp;
   char buf [4096];
   char tmp_file [SLRN_MAX_PATH_LEN];
   char *name;
   unsigned int len;
   int ret;

   name = slrn_basename (file);
   len = (unsigned int) (name - file);
   if (len != 0)
     {
	strncpy (tmp_file, file, len);
# ifndef VMS
	/* It appears that some non-unix systems cannot handle pathnames that
	 * end in a trailing slash.
	 */
	if (len > 1) len--;
# endif
     }
   else
     {
	tmp_file [0] = '.';
	len++;
     }
   tmp_file [len] = 0;

   if (NULL == (fp = fopen (file, "r")))
     {
        slrn_error ("%s %s %s", File_Error, file, Not_Posted);
        return -1;
     }

   if (NULL == (tmpfp = slrn_open_tmpfile_in_dir (tmp_file, tmp_file, "w")))
     {
	slrn_error ("%s %s %s", File_Error, tmp_file, Not_Posted);
	fclose (fp);
	return -1;
     }

   ret = 0;
   while (NULL != fgets (buf, sizeof (buf), fp))
     {
	chmap_map_string_to_iso (buf);
	if (EOF == fputs (buf, tmpfp))
	  {
	     slrn_error ("Write Error. Disk Full? %s", Not_Posted);
	     ret = -1;
	     break;
	  }
     }

   slrn_fclose (fp);
   if (-1 == slrn_fclose (tmpfp))
     ret = -1;

   if (ret == -1)
     {
	(void) slrn_delete_file (tmp_file);
	return -1;
     }

   ret = slrn_move_file (tmp_file, file);
   return ret;
#else
   (void) file;
   return 0;
#endif
}

#if SLRN_HAS_CHARACTER_MAP
void slrn_chmap_show_supported (void)
{
   CharMap_Type *map;
   unsigned int i;

   fprintf (stderr, "Supported character sets include:\n");
   fprintf (stderr, "\tISOLatin\n");

   for (i = 0; i < MAX_CHARMAPS; i++)
     {
	map = Char_Maps[i];
	if (map == NULL) continue;
	fprintf (stderr, "\t%s\n", map->map_name);
     }

   fprintf (stderr, "Default character set: %s\n", DEFAULT_CHARSET_NAME);
}
#endif

int slrn_set_charset (char *name)
{
#if SLRN_HAS_CHARACTER_MAP
   CharMap_Type *map;
   unsigned int i;

   if (name == NULL)
     name = DEFAULT_CHARSET_NAME;

   ChMap_To_Iso_Map = ChMap_From_Iso_Map = NULL;
   SLsmg_Display_Eight_Bit = 160;

   if (0 == slrn_case_strcmp ((unsigned char *)name, (unsigned char *)"isolatin"))
     return 0;

   for (i = 0; i < MAX_CHARMAPS; i++)
     {
	map = Char_Maps[i];
	if (map == NULL) continue;
	if (0 == slrn_case_strcmp ((unsigned char *)map->map_name, (unsigned char *)name))
	  {
	     SLsmg_Display_Eight_Bit = map->display_eight_bit;
	     ChMap_From_Iso_Map = map->from_iso_map;
	     ChMap_To_Iso_Map = map->to_iso_map;
	     return 0;
	  }
     }

   slrn_error ("Unsupport character set: %s", name);
   return -1;

#else
   (void) name;
   return -1;
#endif
}
