//***************************************************************************
//
//  ImgView.cpp
//
//  REQUIRED NOTICE: This utility contains LZW technology of the Unisys
//  Corporation covered by U.S. Patent No. 4,588,302 and foreign counter-
//  parts. No permission is granted by Unisys for the sale, lease, transfer,
//  or distribution of any products containing LZW capability including
//  products derived from this utility. Software or firmware derived from
//  this PC Magazine utility containing GIF and TIFF-LZW capability may be
//  used solely for the personal non-commercial and not-for-profit purposes
//  of the developer and may not be distributed to others for any purpose.
//  Information about LZW licensing may be obtained from:
//
//    Welch Licensing Department
//    Unisys Corporation
//    M/SC2SW1
//    Township Line & Union Meeting Roads
//    Blue Bell, PA 19424
//
//***************************************************************************
 
#include <windows.h>
#include <commctrl.h>
#include <initguid.h>
#include <shlobj.h>
#include <stdlib.h>
#include "\Lead\Include\L_bitmap.h"
#include "Password.h"
#include "Resource.h"
#include "ImgView.h"

/////////////////////////////////////////////////////////////////////////////
// Global variables

UINT            g_cRefThisDll = 0;          // Reference count for this DLL
HINSTANCE       g_hInstance = NULL;         // Instance handle for this DLL
BITMAPHANDLE    g_hBitmap;                  // Bitmap handle (LEAD format)
HPALETTE        g_hPalette = NULL;          // Palette handle

char g_szKey[] = "Software\\PC Magazine\\ImgView\\V1.0";
char g_szMainWndClassName[] = "MainWndClass";
char g_szViewWndClassName[] = "ViewWndClass";
char g_szValueName[] = "StatusBarPref";

/////////////////////////////////////////////////////////////////////////////
// DLL entry point

extern "C" BOOL WINAPI DllMain (HINSTANCE hInstance, ULONG ulReason,
    LPVOID lpReserved)
{
    if (ulReason == DLL_PROCESS_ATTACH) {
        g_hInstance = hInstance;
        g_hBitmap.Flags.Allocated = FALSE;
    }
    return TRUE;
}

/////////////////////////////////////////////////////////////////////////////
// Exported functions

STDAPI DllGetClassObject (REFCLSID rclsid, REFIID riid, LPVOID* ppv)
{
    *ppv = NULL; 
    if (rclsid != CLSID_FileViewer)
        return CLASS_E_CLASSNOTAVAILABLE;

    CClassFactory* pClassFactory = new CClassFactory;

    if (pClassFactory == NULL)
        return E_OUTOFMEMORY;

    HRESULT hr = pClassFactory->QueryInterface (riid, ppv);

    if (FAILED (hr))
        delete pClassFactory;

    return hr;
}

STDAPI DllCanUnloadNow ()
{
    return (g_cRefThisDll == 0) ? S_OK : S_FALSE;
}

/////////////////////////////////////////////////////////////////////////////
// CClassFactory member functions

CClassFactory::CClassFactory ()
{
    m_cRef = 0;
    g_cRefThisDll++;
}

CClassFactory::~CClassFactory ()
{
    g_cRefThisDll--;
}

HRESULT CClassFactory::QueryInterface (REFIID riid, LPVOID* ppv)
{
    *ppv = NULL;
    HRESULT hr = E_NOINTERFACE;

    if (riid == IID_IUnknown)
        *ppv = (LPUNKNOWN) this;
    else if (riid == IID_IClassFactory)
        *ppv = (LPCLASSFACTORY) this;

    if (*ppv != NULL) {
        ((LPUNKNOWN) *ppv)->AddRef ();
        hr = NOERROR;
    }
    return hr;
}

ULONG CClassFactory::AddRef ()
{
    return ++m_cRef;
}

ULONG CClassFactory::Release ()
{
    ULONG cRef = --m_cRef;
    if (m_cRef == 0)
        delete this;
    return cRef;
}

HRESULT CClassFactory::CreateInstance (LPUNKNOWN pUnkOuter, REFIID riid,
    LPVOID* ppv)
{
    *ppv = NULL;
    if (pUnkOuter != NULL)
        return CLASS_E_NOAGGREGATION;

    CFileViewer* pFileViewer = new CFileViewer;

    if (pFileViewer == NULL)
        return E_OUTOFMEMORY;

    HRESULT hr = pFileViewer->QueryInterface (riid, ppv);

    if (FAILED (hr))
        delete pFileViewer;

    return hr;
}

HRESULT CClassFactory::LockServer (BOOL fLock)
{
    return E_NOTIMPL;
}

/////////////////////////////////////////////////////////////////////////////
// CFileViewer member functions

CFileViewer::CFileViewer ()
{
    m_cRef = 0;
    g_cRefThisDll++;
    m_bShowInitializeCalled = FALSE;
    m_szFile[0] = 0;
}

CFileViewer::~CFileViewer ()
{
    g_cRefThisDll--;
}

HRESULT CFileViewer::QueryInterface (REFIID riid, LPVOID* ppv)
{
    *ppv = NULL;
    HRESULT hr = E_NOINTERFACE;

    if (riid == IID_IUnknown)
        *ppv = (LPUNKNOWN) this;
    else if (riid == IID_IPersistFile)
        *ppv = (LPPERSISTFILE) this;
    else if (riid == IID_IFileViewer)
        *ppv = (LPFILEVIEWER) this;

    if (*ppv != NULL) {
        ((LPUNKNOWN) *ppv)->AddRef ();
        hr = NOERROR;
    }
    return hr;
}

ULONG CFileViewer::AddRef ()
{
    return ++m_cRef;
}
 
ULONG CFileViewer::Release ()
{
    ULONG cRef = --m_cRef;
    if (m_cRef == 0)
        delete this;
    return cRef;
}

HRESULT CFileViewer::Load (LPCOLESTR pszFile, DWORD dwMode)
{
    wcstombs (m_szFile, pszFile, sizeof (m_szFile));
    return NOERROR;
}

HRESULT CFileViewer::GetCurFile (LPOLESTR* ppszFile)
{
    return E_NOTIMPL;
}

HRESULT CFileViewer::GetClassID (LPCLSID lpClsID)
{
    *lpClsID = CLSID_FileViewer;
    return NOERROR;
}

HRESULT CFileViewer::IsDirty ()
{
    return S_FALSE; // Can't get dirty if the file can't be edited
}

HRESULT CFileViewer::Save (LPCOLESTR pszFile, BOOL fRemember)
{
    return E_NOTIMPL;
}

HRESULT CFileViewer::SaveCompleted (LPCOLESTR pszFile)
{
    return E_NOTIMPL;
}

HRESULT CFileViewer::PrintTo (LPSTR pszDriver, BOOL fSuppressUI)
{
    return E_NOTIMPL;
}

HRESULT CFileViewer::ShowInitialize (LPFILEVIEWERSITE lpfsi)
{
    //
    // Unlock GIF and TIFF-LZW support in LEADTOOLS.
    //
    L_UnlockSupport (L_SUPPORT_GIFLZW, LEADTOOLS_GIFKEY);
    L_UnlockSupport (L_SUPPORT_TIFLZW, LEADTOOLS_TIFKEY);

    //
    // Load the image.
    //
    if (!LoadImage (m_szFile))
        return E_FAIL;

    //
    // Create the file viewer window.
    //
    WNDCLASS wc;
    wc.style = 0;
    wc.lpfnWndProc = (WNDPROC) MainWndProc;
    wc.cbClsExtra = 0;
    wc.cbWndExtra = 0;
    wc.hInstance = g_hInstance;
    wc.hIcon = LoadIcon (NULL, IDI_WINLOGO);
    wc.hCursor = LoadCursor (NULL, IDC_ARROW);
    wc.hbrBackground = (HBRUSH) (COLOR_WINDOW + 1);
    wc.lpszMenuName = MAKEINTRESOURCE (IDR_MENU);
    wc.lpszClassName = g_szMainWndClassName;

    RegisterClass (&wc);

    char szWndTitle[MAX_PATH + 16];
    GetFileTitle (m_szFile, szWndTitle, sizeof (szWndTitle));
    lstrcat (szWndTitle, " - Quick View");

    m_hWnd = CreateWindow (g_szMainWndClassName, szWndTitle,
        WS_OVERLAPPEDWINDOW, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT,
        CW_USEDEFAULT, HWND_DESKTOP, NULL, g_hInstance, NULL);

    if (m_hWnd == NULL) {
        UnregisterClass (g_szMainWndClassName, g_hInstance);
        L_FreeBitmap (&g_hBitmap);
        return E_OUTOFMEMORY;
    }

    SetWindowLong (m_hWnd, (int) this, GWL_USERDATA);

    //
    // Size the window to fit the image.
    //
    SizeWindowToImage (m_hWnd, &g_hBitmap);

    //
    // Set the flag indicating ShowInitialize was called, and exit.
    //
    m_bShowInitializeCalled = TRUE;
    return NOERROR;
}

HRESULT CFileViewer::Show (LPFVSHOWINFO pvsi)
{
    //
    // Fail the call if ShowInitialize wasn't called first.
    //
    if (!m_bShowInitializeCalled)
        return E_UNEXPECTED;

    //
    // Display the window on the screen.
    //
    ShowWindow (m_hWnd, pvsi->iShow);

    //
    // Forward messages to the window.
    //
    MSG msg;
    while (GetMessage (&msg, NULL, 0, 0)) {
        TranslateMessage (&msg);
        DispatchMessage (&msg);
    }

    //
    // Clean up and return.
    //
    UnregisterClass (g_szMainWndClassName, g_hInstance);
    UnregisterClass (g_szViewWndClassName, g_hInstance);

    L_FreeBitmap (&g_hBitmap);

    if (g_hPalette != NULL) {
        DeleteObject (g_hPalette);
        g_hPalette = NULL;
    }
    return NOERROR;
}

BOOL CFileViewer::LoadImage (LPSTR pszFile)
{
    FILEINFO fi;
    if (L_FileInfo (pszFile, &fi) != SUCCESS)
        return FALSE;

    if ((fi.Format != FILE_JFIF) &&
        (fi.Format != FILE_PCD)  &&
        (fi.Format != FILE_PCX)  &&
        (fi.Format != FILE_PNG)  &&
        (fi.Format != FILE_TGA)  &&
        (fi.Format != FILE_TIF)  &&
        (fi.Format != FILE_JTIF) &&
        (fi.Format != FILE_GIF))
        return FALSE; // Unsupported file format

    BITMAPHANDLE hBitmap;
    L_LoadBitmap (pszFile, &hBitmap, 0, ORDER_BGR);

    if (!hBitmap.Flags.Allocated)
        return FALSE;

    if (!g_hBitmap.Flags.Allocated)
        L_FreeBitmap (&g_hBitmap);
    g_hBitmap = hBitmap;

    if (g_hPalette != NULL)
        DeleteObject (g_hPalette);

    HDC hdc = GetDC (NULL);
    g_hPalette = L_CreatePaintPalette (hdc, &g_hBitmap);
    ReleaseDC (NULL, hdc);

    lstrcpy (m_szFile, pszFile);
    return TRUE;
}

/////////////////////////////////////////////////////////////////////////////
// Window procedures and helper functions

LRESULT CALLBACK MainWndProc (HWND hwnd, UINT msg, WPARAM wParam,
    LPARAM lParam)
{
    HDROP hDrop;
    char szFile[MAX_PATH];
    CFileViewer* pViewer;
    static HWND hWndStatusBar;
    static HWND hWndView;

    switch (msg) {

    case WM_CREATE:
        //
        // Create a status bar at the bottom of the window.
        //
        InitCommonControls ();
        hWndStatusBar = CreateStatusWindow (WS_CHILD | WS_VISIBLE, "Ready",
            hwnd, ID_STATUS_BAR);

        if (hWndStatusBar == NULL)
            return -1;

        //
        // Hide the status bar if the user prefers it hidden.
        //
        if (PreferStatusBarHidden ()) {
            ShowWindow (hWndStatusBar, SW_HIDE);
            CheckMenuItem (GetMenu (hwnd), IDM_STATUS, MF_UNCHECKED);
        }

        //
        // Fill the visible portion of the window's client area with a view.
        //
        WNDCLASS wc;
        wc.style = 0;
        wc.lpfnWndProc = (WNDPROC) ViewWndProc;
        wc.cbClsExtra = 0;
        wc.cbWndExtra = 0;
        wc.hInstance = g_hInstance;
        wc.hIcon = NULL;
        wc.hCursor = LoadCursor (NULL, IDC_ARROW);
        wc.hbrBackground = (HBRUSH) (COLOR_APPWORKSPACE + 1);
        wc.lpszMenuName = NULL;
        wc.lpszClassName = g_szViewWndClassName;

        RegisterClass (&wc);

        hWndView = CreateWindowEx (WS_EX_CLIENTEDGE, g_szViewWndClassName,
            NULL, WS_CHILD | WS_VISIBLE | WS_HSCROLL | WS_VSCROLL, 0, 0, 0, 0,
            hwnd, (HMENU) ID_VIEW, g_hInstance, NULL);

        if (hWndView == NULL) {
            UnregisterClass (g_szViewWndClassName, g_hInstance);
            return -1;
        }

        //
        // Register this window to accept dropped files.
        //
        DragAcceptFiles (hwnd, TRUE);
        return 0;

    case WM_SIZE:
        //
        // When the main window is resized, resize its children, too.
        //
        if ((wParam == SIZE_MAXIMIZED) || (wParam == SIZE_RESTORED)) {
            SendMessage (hWndStatusBar, WM_SIZE, wParam, lParam);
            ResizeView (hWndView, hWndStatusBar, (int) LOWORD (lParam),
                (int) HIWORD (lParam));
        }
        return 0;

    case WM_PALETTECHANGED:
        //
        // Realize the logical palette if g_hPalette is non-NULL and
        // the window handle in wParam is not our own.
        //
        if (hwnd == (HWND) wParam)
            return FALSE;

        // Fall through to the WM_QUERYNEWPALETTE handler...

    case WM_QUERYNEWPALETTE:
        //
        // Realize the logical palette if g_hPalette is non-NULL.
        //
        if (g_hPalette != NULL) {
            HDC hdc;
            hdc = GetDC (hwnd);
            SelectPalette (hdc, g_hPalette, FALSE);

            UINT nColors;
            if (nColors = RealizePalette (hdc))
                InvalidateRect (hwnd, NULL, FALSE);

            ReleaseDC (hwnd, hdc);
            return nColors ? TRUE : FALSE;
        }
        break;

    case WM_COMMAND:
        switch (LOWORD (wParam)) {

        case IDM_COPY:
            //
            // Copy the image to the clipboard.
            //
            L_CopyToClipboard (hwnd, &g_hBitmap);
            return 0;

        case IDM_STATUS:
            //
            // Toggle the status bar on or off.
            //
            if (IsWindowVisible (hWndStatusBar)) { // Hide it
                ShowWindow (hWndStatusBar, SW_HIDE);
                CheckMenuItem (GetMenu (hwnd), IDM_STATUS, MF_UNCHECKED);
                RecordStatusBarPreference (FALSE);
            }
            else { // Make it visible
                ShowWindow (hWndStatusBar, SW_SHOW);
                CheckMenuItem (GetMenu (hwnd), IDM_STATUS, MF_CHECKED);
                RecordStatusBarPreference (TRUE);
            }

            RECT rect;
            GetClientRect (hwnd, &rect);
            ResizeView (hWndView, hWndStatusBar, rect.right - rect.left,
                rect.bottom - rect.top);
            return 0;

        case IDM_ABOUT:
            //
            // Display the viewer's About box.
            //
            DialogBox (g_hInstance, MAKEINTRESOURCE (IDD_ABOUTDLG), hwnd,
                (DLGPROC) AboutDlgProc);
            return 0;

        case IDM_EXIT:
            //
            // Close the window.
            //
            PostMessage (hwnd, WM_CLOSE, 0, 0);
            return 0;
        }
        break;

    case WM_MENUSELECT:
        //
        // Display help text for menu commands.
        //
        UINT nItem, nFlags;
        nItem = LOWORD (wParam);
        nFlags = HIWORD (wParam);
        char szText[64];

        if ((nFlags == 0xFFFF) && (lParam == 0))
            SendMessage (hWndStatusBar, SB_SETTEXT, 0, (LPARAM) "Ready");
        else if (LoadString (g_hInstance, nItem, szText, sizeof (szText)))
            SendMessage (hWndStatusBar, SB_SETTEXT, 0, (LPARAM) szText);
        else
            SendMessage (hWndStatusBar, SB_SETTEXT, 0, (LPARAM) "");

        return 0;

    case WM_DROPFILES:
        //
        // Display an image dropped over the window.
        //
        hDrop = (HDROP) wParam;
        DragQueryFile (hDrop, 0, szFile, sizeof (szFile));

        pViewer = (CFileViewer*) GetWindowLong (hwnd, GWL_USERDATA);
        if (pViewer->LoadImage (szFile)) {
            RECT rect;
            GetClientRect (hWndView, &rect);
            SendMessage (hWndView, WM_SIZE, SIZE_RESTORED,
                MAKELPARAM (rect.right, rect.bottom));
            InvalidateRect (hWndView, NULL, TRUE);
        }

        DragFinish (hDrop);
        return 0;

    case WM_DESTROY:
        //
        // Post a WM_QUIT message to the window to end the message loop.
        //
        PostQuitMessage (0);
        return 0;
    }
    return DefWindowProc (hwnd, msg, wParam, lParam);
}

LRESULT CALLBACK ViewWndProc (HWND hwnd, UINT msg, WPARAM wParam,
    LPARAM lParam)
{
    SCROLLINFO si;
    int nMaxPos, nDelta;

    switch (msg) {

    case WM_SIZE:
        //
        // Adjust the scroll bars when the view size changes.
        //
        int cx, cy;
        cx = LOWORD (lParam);
        cy = HIWORD (lParam);

        BOOL bRepaintAll;
        bRepaintAll = FALSE;

        int nMax, nPos, nPage;
        nMax = nPos = nPage = 0;
        if (cx < g_hBitmap.Width) {
            nPage = cx;
            nMax = g_hBitmap.Width - 1;
            nPos = min (GetScrollPos (hwnd, SB_HORZ), nMax - nPage + 1);
        }

        if (nPos != GetScrollPos (hwnd, SB_HORZ))
            bRepaintAll = TRUE;

        si.cbSize = sizeof (si);
        si.fMask = SIF_PAGE | SIF_RANGE | SIF_POS | SIF_DISABLENOSCROLL;
        si.nMin = 0;
        si.nMax = nMax;
        si.nPos = nPos;
        si.nPage = nPage;

        SetScrollInfo (hwnd, SB_HORZ, &si, TRUE);

        nMax = nPos = nPage = 0;
        if (cy < g_hBitmap.Height) {
            nPage = cy;
            nMax = g_hBitmap.Height - 1;
            nPos = min (GetScrollPos (hwnd, SB_VERT), nMax - nPage + 1);
        }

        if (nPos != GetScrollPos (hwnd, SB_VERT))
            bRepaintAll = TRUE;

        si.cbSize = sizeof (si);
        si.fMask = SIF_PAGE | SIF_RANGE | SIF_POS | SIF_DISABLENOSCROLL;
        si.nMin = 0;
        si.nMax = nMax;
        si.nPos = nPos;
        si.nPage = nPage;

        SetScrollInfo (hwnd, SB_VERT, &si, TRUE);

        if (bRepaintAll)
            InvalidateRect (hwnd, NULL, FALSE);

        return 0;

    case WM_HSCROLL:
        //
        // Scroll the image horizontally.
        //
        si.cbSize = sizeof (si);
        si.fMask = SIF_PAGE | SIF_RANGE | SIF_POS;
        GetScrollInfo (hwnd, SB_HORZ, &si);

        nMaxPos = si.nMax - si.nPage + 1;
        nDelta = 0;

        switch (LOWORD (wParam)) {

        case SB_LINELEFT:
            if (si.nPos > 0)
                nDelta = -(min (4, si.nPos));
            break;

        case SB_PAGELEFT:
            if (si.nPos > 0)
                nDelta = -(min (((int) si.nPage * 10) / 9, si.nPos));
            break;

        case SB_THUMBTRACK:
            nDelta = (int) HIWORD (wParam) - si.nPos;
            break;

        case SB_PAGERIGHT:
            if (si.nPos < nMaxPos)
                nDelta = min (((int) si.nPage * 10) / 9, nMaxPos - si.nPos);
            break;

        case SB_LINERIGHT:
            if (si.nPos < nMaxPos)
                nDelta = min (4, nMaxPos - si.nPos);
            break;
        }

        if (nDelta) {
            SetScrollPos (hwnd, SB_HORZ, si.nPos + nDelta, TRUE);
            ScrollWindow (hwnd, -nDelta, 0, NULL, NULL);
            UpdateWindow (hwnd);
        }
        return 0;

    case WM_VSCROLL:
        //
        // Scroll the image vertically.
        //
        si.cbSize = sizeof (si);
        si.fMask = SIF_PAGE | SIF_RANGE | SIF_POS;
        GetScrollInfo (hwnd, SB_VERT, &si);

        nMaxPos = si.nMax - si.nPage + 1;
        nDelta = 0;

        switch (LOWORD (wParam)) {

        case SB_LINEUP:
            if (si.nPos > 0)
                nDelta = -(min (4, si.nPos));
            break;

        case SB_PAGEUP:
            if (si.nPos > 0)
                nDelta = -(min (((int) si.nPage * 10) / 9, si.nPos));
            break;

        case SB_THUMBTRACK:
            nDelta = (int) HIWORD (wParam) - si.nPos;
            break;

        case SB_PAGEDOWN:
            if (si.nPos < nMaxPos)
                nDelta = min (((int) si.nPage * 10) / 9, nMaxPos - si.nPos);
            break;

        case SB_LINEDOWN:
            if (si.nPos < nMaxPos)
                nDelta = min (4, nMaxPos - si.nPos);
            break;
        }

        if (nDelta) {
            SetScrollPos (hwnd, SB_VERT, si.nPos + nDelta, TRUE);
            ScrollWindow (hwnd, 0, -nDelta, NULL, NULL);
            UpdateWindow (hwnd);
        }
        return 0;

    case WM_PAINT:
        //
        // Draw the image.
        //
        int nHorzPos, nVertPos;
        nHorzPos = GetScrollPos (hwnd, SB_HORZ);
        nVertPos = GetScrollPos (hwnd, SB_VERT);

        RECT rect;
        SetRect (&rect, -nHorzPos, -nVertPos, g_hBitmap.Width - nHorzPos,
            g_hBitmap.Height - nVertPos);

        HDC hdc;
        PAINTSTRUCT ps;
        hdc = BeginPaint (hwnd, &ps);

        HPALETTE hOldPalette;
        if (g_hPalette != NULL) {
            hOldPalette = SelectPalette (hdc, g_hPalette, FALSE);
            RealizePalette (hdc);
        }

        L_PaintDC (hdc, &g_hBitmap, NULL, NULL, &rect, &ps.rcPaint, SRCCOPY);

        if (g_hPalette != NULL)
            SelectPalette (hdc, hOldPalette, FALSE);

        EndPaint (hwnd, &ps);
        return 0;
    }
    return DefWindowProc (hwnd, msg, wParam, lParam);
}

void SizeWindowToImage (HWND hWndMain, BITMAPHANDLE* pBitmap)
{
    HWND hWndView = GetDlgItem (hWndMain, ID_VIEW);
    HWND hWndStatusBar = GetDlgItem (hWndMain, ID_STATUS_BAR);

    //
    // Calculate the size of the view window.
    //
    RECT rect;
    SetRect (&rect, 0, 0,
        pBitmap->Width + GetSystemMetrics (SM_CXVSCROLL),
        pBitmap->Height + GetSystemMetrics (SM_CYHSCROLL));

    AdjustWindowRectEx (&rect, (DWORD) GetWindowLong (hWndView, GWL_STYLE),
        FALSE, (DWORD) GetWindowLong (hWndView, GWL_EXSTYLE));

    //
    // Then calculate the size of the main window.
    //
    AdjustWindowRectEx (&rect, (DWORD) GetWindowLong (hWndMain, GWL_STYLE),
        TRUE, (DWORD) GetWindowLong (hWndMain, GWL_EXSTYLE));

    MENUITEMINFO mii;
    mii.cbSize = sizeof (mii);
    mii.fMask = MIIM_STATE;

    GetMenuItemInfo (GetMenu (hWndMain), IDM_STATUS, FALSE, &mii);

    if (mii.fState & MFS_CHECKED) {
        RECT rectStatusBar;
        GetWindowRect (hWndStatusBar, &rectStatusBar);
        rect.bottom += (rectStatusBar.bottom - rectStatusBar.top);
    }

    //
    // Adjust the window size.
    //
    int nMaxWidth = (GetSystemMetrics (SM_CXSCREEN) * 9) / 10;
    int nMaxHeight = (GetSystemMetrics (SM_CYSCREEN) * 9) / 10;

    int nWidth = max (160, rect.right - rect.left);
    int nHeight = max (120, rect.bottom - rect.top);

    if ((nWidth <= nMaxWidth) && (nHeight <= nMaxHeight))
        SetWindowPos (hWndMain, NULL, 0, 0, nWidth, nHeight,
            SWP_NOZORDER | SWP_NOMOVE);
}

void ResizeView (HWND hWndView, HWND hWndStatusBar, int cx, int cy)
{
    if (hWndView != NULL) {
        if (IsWindowVisible (hWndStatusBar)) {
            RECT rect;
            GetWindowRect (hWndStatusBar, &rect);
            cy = cy - (rect.bottom - rect.top);
            if (cy < 0)
                cy = 0;
        }
        SetWindowPos (hWndView, NULL, 0, 0, cx, cy,
            SWP_NOMOVE | SWP_NOZORDER);
    }
}

void RecordStatusBarPreference (BOOL bVisible)
{
    HKEY hKey;

    if (RegCreateKeyEx (HKEY_CURRENT_USER, g_szKey, 0, NULL,
        REG_OPTION_NON_VOLATILE, KEY_WRITE, NULL, &hKey, NULL) ==
        ERROR_SUCCESS) {

        DWORD dwVal = bVisible ? 1 : 0;
        RegSetValueEx (hKey, g_szValueName, 0, REG_DWORD, (PBYTE) &dwVal,
            sizeof (dwVal));
        RegCloseKey (hKey);
    }
}

BOOL PreferStatusBarHidden ()
{
    HKEY hKey;
    BOOL bResult = FALSE;

    if (RegOpenKeyEx (HKEY_CURRENT_USER, g_szKey, 0, KEY_QUERY_VALUE,
        &hKey) == ERROR_SUCCESS) {

        DWORD dwVal;
        DWORD dwSize = sizeof (dwVal);
        if (RegQueryValueEx (hKey, g_szValueName, NULL, NULL, (PBYTE) &dwVal,
            &dwSize) == ERROR_SUCCESS)
            bResult = dwVal ? FALSE : TRUE;

        RegCloseKey (hKey);
    }
    return bResult;
}

/////////////////////////////////////////////////////////////////////////////
// Dialog procedures

BOOL CALLBACK AboutDlgProc (HWND hwnd, UINT uMessage, WPARAM wParam,
    LPARAM lParam)
{
    static HFONT hFont;
    static HBRUSH hBrush;

    switch (uMessage) {
    
    case WM_INITDIALOG:
        //
        // Enlarge the program title for visual effect and create a
        // custom background brush.
        //
        LOGFONT lf;
        GetObject ((HFONT) SendMessage (hwnd, WM_GETFONT, 0, 0),
            sizeof (lf), &lf);

        lf.lfHeight = lf.lfHeight * 3;
        lf.lfWidth = lf.lfWidth * 3;
        lf.lfWeight = FW_BOLD;
        lf.lfItalic = 1;
        lstrcpy (lf.lfFaceName, "Arial");

        hFont = CreateFontIndirect (&lf);
        if (hFont != NULL)
            SendDlgItemMessage (hwnd, IDC_TITLE, WM_SETFONT,
                (WPARAM) hFont, 0);

        hBrush = CreateSolidBrush (GetSysColor (COLOR_3DFACE));
        return TRUE;

    case WM_CTLCOLORSTATIC:
        //
        // Change the program title's text color to red.
        //
        SetBkColor ((HDC) wParam, GetSysColor (COLOR_3DFACE));
        if (GetDlgCtrlID ((HWND) lParam) == IDC_TITLE)
            SetTextColor ((HDC) wParam, RGB (255, 0, 0));
        return (BOOL) hBrush;

    case WM_COMMAND:
        //
        // Process messages from the controls.
        //
        UINT nID;
        nID = (UINT) LOWORD (wParam);

        switch (nID) {

        case IDOK:
            EndDialog (hwnd, IDOK);
            return TRUE;

        case IDCANCEL:
            EndDialog (hwnd, IDCANCEL);
            return TRUE;
        }
        break;

    case WM_DESTROY:
        //
        // Clean up and exit.
        //
        if (hFont != NULL)
            DeleteObject (hFont);
        DeleteObject (hBrush);
        return TRUE;
    }
    return FALSE;
}
