/* Basic type operations for S-Lang */
/* Copyright (c) 1992, 1996 John E. Davis
 * All rights reserved.
 *
 * You may distribute under the terms of either the GNU General Public
 * License or the Perl Artistic License.
 */

#include "config.h"
#include "sl-feat.h"

#include <stdio.h>

#include "slang.h"
#include "_slang.h"

#if SLANG_HAS_FLOAT
# include <math.h>
#endif

int SLang_pop_integer(int *i) /*{{{*/
{
   return _SLang_pop_i_val (SLANG_INT_TYPE, i);
}

/*}}}*/

#if SLANG_HAS_FLOAT
int SLang_pop_double(double *x, int *convertp, int *ip) /*{{{*/
{
   SLang_Object_Type obj;
   int i, convert;

   if (0 != SLang_pop (&obj))
     return -1;

   switch (obj.data_type)
     {
      case SLANG_DOUBLE_TYPE:
	*x = obj.v.f_val;
	i = 0;
	convert = 0;
	break;

      case SLANG_INT_TYPE:
	i = obj.v.i_val;
	*x = (double) obj.v.i_val;
	convert = 1;
	break;

      default:
	_SLclass_type_mismatch_error (SLANG_DOUBLE_TYPE, obj.data_type);
	SLang_free_object (&obj);
	return -1;
     }

   if (convertp != NULL) *convertp = convert;
   if (ip != NULL) *ip = i;

   return 0;
}

/*}}}*/
int SLang_push_double(double x) /*{{{*/
{
   SLang_Object_Type obj;

   obj.data_type = SLANG_DOUBLE_TYPE;
   obj.v.f_val = x;
   return SLang_push (&obj);
}

/*}}}*/

static int
double_typecast (unsigned char a_type, VOID_STAR ap, unsigned int na,
		 unsigned char b_type, VOID_STAR bp)
{
   double *b;
   unsigned int i;
   int *ia;

   (void) b_type;
   b = (double *) bp;
   switch (a_type)
     {
      default:
	return 0;

      case SLANG_INT_TYPE:
	ia = (int *) ap;
	for (i = 0; i < na; i++)
	  b[i] = (double) ia[i];
	break;
     }
   return 1;
}

#endif				       /* SLANG_HAS_FLOAT */

int SLpop_string (char **s) /*{{{*/
{
   char *sls;

   *s = NULL;

   if (-1 == SLang_pop_slstring (&sls))
     return -1;

   if (NULL == (*s = SLmake_string (sls)))
     {
	SLang_free_slstring (sls);
	return -1;
     }

   SLang_free_slstring (sls);
   return 0;
}

/*}}}*/

int SLang_pop_slstring (char **s) /*{{{*/
{
   SLang_Object_Type obj;

   if (-1 == _SLang_pop_object_of_type (SLANG_STRING_TYPE, &obj))
     {
	*s = NULL;
	return -1;
     }
   *s = obj.v.s_val;
   return 0;
}

/*}}}*/

/* if *data != 0, string should be freed upon use. */
int SLang_pop_string(char **s, int *data) /*{{{*/
{
   if (SLpop_string (s))
     return -1;

   *data = 1;
   return 0;
}

/*}}}*/

int SLang_push_integer(int i) /*{{{*/
{
   return _SLang_push_i_val (SLANG_INT_TYPE, i);
}

/*}}}*/

int _SLang_push_slstring (char *s)
{
   if (0 == _SLang_push_void_star (SLANG_STRING_TYPE, (VOID_STAR)s))
     return 0;
   
   SLang_free_slstring (s);
   return -1;
}


int SLang_push_string (char *t) /*{{{*/
{
   if (t == NULL)
     return _SLang_push_null ();

   if (NULL == (t = SLang_create_slstring (t)))
     return -1;
   
   return _SLang_push_slstring (t);
}

/*}}}*/

int SLang_push_malloced_string (char *c) /*{{{*/
{
   if (0 == SLang_push_string (c))
     {
	SLfree (c);
	return 0;
     }
   return -1;
}

/*}}}*/

static int int_int_power (int a, int b)
{
   int r, s;

   if (a == 0) return 0;
   if (b < 0) return 0;
   if (b == 0) return 1;

   s = 1;
   if (a < 0)
     {
	if ((b % 2) == 1) s = -1;
	a = -a;
     }

   /* FIXME!! This needs optimized */
   r = 1;
   while (b)
     {
	r = r * a;
	b--;
     }
   return r * s;
}

static int int_bin_op_result (int op, unsigned char a_type, unsigned char b_type,
			      unsigned char *c_type)
{
   (void) a_type;
   (void) b_type;
   (void) c_type;
   (void) op;
   *c_type = SLANG_INT_TYPE;
   return 1;
}

static int int_bin_op (int op,
		       unsigned char a_type, VOID_STAR ap, unsigned int na,
		       unsigned char b_type, VOID_STAR bp, unsigned int nb,
		       VOID_STAR cp)
{
   int *c, *a, *b;
   unsigned int n, n_max;
   unsigned int da, db;

   (void) a_type;
   (void) b_type;

   a = (int *) ap;
   b = (int *) bp;
   c = (int *) cp;

   if (na == 1) da = 0; else da = 1;
   if (nb == 1) db = 0; else db = 1;

   if (na > nb) n_max = na; else n_max = nb;

   switch (op)
     {
      default:
	return 0;

      case SLANG_POW:
	for (n = 0; n < n_max; n++)
	  {
	     c[n] = int_int_power (*a, *b); a += da; b += db;
	  }
	break;

      case SLANG_PLUS:
	for (n = 0; n < n_max; n++)
	  {
	     c[n] = (*a + *b); a += da; b += db;
	  }
	break;

      case SLANG_MINUS:
	for (n = 0; n < n_max; n++)
	  {
	     c[n] = (*a - *b); a += da; b += db;
	  }
	break;

      case SLANG_TIMES:
	for (n = 0; n < n_max; n++)
	  {
	     c[n] = (*a * *b); a += da; b += db;
	  }
	break;

      case SLANG_DIVIDE:
	for (n = 0; n < n_max; n++)
	  {
	     if (*b == 0)
	       {
		  SLang_Error = SL_DIVIDE_ERROR;
		  return -1;
	       }
	     c[n] = (*a / *b); a += da; b += db;
	  }
	break;

      case SLANG_EQ:
	for (n = 0; n < n_max; n++)
	  {
	     c[n] = (*a == *b); a += da; b += db;
	  }
	break;

      case SLANG_NE:
	for (n = 0; n < n_max; n++)
	  {
	     c[n] = (*a != *b); a += da; b += db;
	  }
	break;

      case SLANG_GT:
	for (n = 0; n < n_max; n++)
	  {
	     c[n] = (*a > *b); a += da; b += db;
	  }
	break;

      case SLANG_GE:
	for (n = 0; n < n_max; n++)
	  {
	     c[n] = (*a >= *b); a += da; b += db;
	  }
	break;

      case SLANG_LT:
	for (n = 0; n < n_max; n++)
	  {
	     c[n] = (*a < *b); a += da; b += db;
	  }
	break;

      case SLANG_LE:
	for (n = 0; n < n_max; n++)
	  {
	     c[n] = (*a <= *b); a += da; b += db;
	  }
	break;

      case SLANG_OR:
	for (n = 0; n < n_max; n++)
	  {
	     c[n] = (*a || *b); a += da; b += db;
	  }
	break;

      case SLANG_AND:
	for (n = 0; n < n_max; n++)
	  {
	     c[n] = (*a && *b); a += da; b += db;
	  }
	break;

      case SLANG_BAND:
	for (n = 0; n < n_max; n++)
	  {
	     c[n] = (*a & *b); a += da; b += db;
	  }
	break;

      case SLANG_BXOR:
	for (n = 0; n < n_max; n++)
	  {
	     c[n] = (*a ^ *b); a += da; b += db;
	  }
	break;

      case SLANG_MOD:
	for (n = 0; n < n_max; n++)
	  {
	     c[n] = (*a % *b); a += da; b += db;
	  }
	break;

      case SLANG_BOR:
	for (n = 0; n < n_max; n++)
	  {
	     c[n] = (*a | *b); a += da; b += db;
	  }
	break;

      case SLANG_SHL:
	for (n = 0; n < n_max; n++)
	  {
	     c[n] = (*a << *b); a += da; b += db;
	  }
	break;

      case SLANG_SHR:
	for (n = 0; n < n_max; n++)
	  {
	     c[n] = (*a >> *b); a += da; b += db;
	  }
	break;
     }
   return 1;
}

static int int_unary_op_result (int op, unsigned char a_type, unsigned char *b_type)
{
   (void) op;
   (void) a_type;
   *b_type = SLANG_INT_TYPE;
   return 1;
}

static int int_unary_op (int op,
			 unsigned char a_type, VOID_STAR ap, unsigned int na,
			 VOID_STAR bp)
{
   int *a, *b;
   unsigned int n;

   (void) a_type;

   a = (int *) ap;
   b = (int *) bp;

   switch (op)
     {
      default:
	return 0;

      case SLANG_PLUSPLUS:
	for (n = 0; n < na; n++) b[n] = (a[n] + 1);
	break;
      case SLANG_MINUSMINUS:
	for (n = 0; n < na; n++) b[n] = (a[n] - 1);
	break;
      case SLANG_CHS:
	for (n = 0; n < na; n++) b[n] = -(a[n]);
	break;
      case SLANG_SQR:
	for (n = 0; n < na; n++) b[n] = (a[n] * a[n]);
	break;
      case SLANG_MUL2:
	for (n = 0; n < na; n++) b[n] = (2 * a[n]);
	break;
      case SLANG_ABS:
	for (n = 0; n < na; n++) b[n] = abs (a[n]);
	break;
      case SLANG_SIGN:
	for (n = 0; n < na; n++)
	  {
	     if (a[n] < 0) b[n] = -1;
	     else if (a[n] > 0) b[n] = 1;
	     else b[n] = 0;
	  }
	break;

      case SLANG_NOT:
	for (n = 0; n < na; n++) b[n] = !(a[n]);
	break;
      case SLANG_BNOT:
	for (n = 0; n < na; n++) b[n] = ~(a[n]);
	break;
     }

   return 1;
}

#if SLANG_HAS_FLOAT
static int double_unary_op_result (int op, unsigned char a, unsigned char *b)
{
   (void) a;
   switch (op)
     {
      default:
	return 0;

      case SLANG_PLUSPLUS:
      case SLANG_MINUSMINUS:
      case SLANG_CHS:
      case SLANG_SQR:
      case SLANG_MUL2:
      case SLANG_ABS:
	*b = SLANG_DOUBLE_TYPE;
	break;

      case SLANG_SIGN:
	*b = SLANG_INT_TYPE;
	break;
     }
   return 1;
}

static int double_unary_op (int op,
			    unsigned char a_type, VOID_STAR ap, unsigned int na,
			    VOID_STAR bp)
{
   double *a, *b;
   int *c;
   unsigned int n;
   (void) a_type;

   a = (double *) ap;
   b = (double *) bp;
   c = (int *) bp;

   switch (op)
     {
      default:
	return 0;

      case SLANG_PLUSPLUS:
	for (n = 0; n < na; n++) b[n] = (a[n] + 1);
	break;
      case SLANG_MINUSMINUS:
	for (n = 0; n < na; n++) b[n] = (a[n] - 1);
	break;
      case SLANG_CHS:
	for (n = 0; n < na; n++) b[n] = -(a[n]);
	break;
      case SLANG_SQR:
	for (n = 0; n < na; n++) b[n] = (a[n] * a[n]);
	break;
      case SLANG_MUL2:
	for (n = 0; n < na; n++) b[n] = (2 * a[n]);
	break;
      case SLANG_ABS:
	for (n = 0; n < na; n++) b[n] = fabs (a[n]);
	break;
      case SLANG_SIGN:
	for (n = 0; n < na; n++)
	  {
	     if (a[n] < 0) c[n] = -1;
	     else if (a[n] > 0) c[n] = 1;
	     else c[n] = 0;
	  }
	break;
     }

   return 1;
}

static int double_bin_op_result (int op, unsigned char a, unsigned char b,
				 unsigned char *c)
{
   (void) a;
   (void) b;

   switch (op)
     {
      default:
	return 0;

      case SLANG_POW:
      case SLANG_PLUS:
      case SLANG_MINUS:
      case SLANG_TIMES:
      case SLANG_DIVIDE:
	*c = SLANG_DOUBLE_TYPE;
	break;

      case SLANG_EQ:
      case SLANG_NE:
      case SLANG_GT:
      case SLANG_LT:
      case SLANG_LE:
      case SLANG_GE:
	*c = SLANG_INT_TYPE;
	break;
     }
   return 1;
}

static int double_int_bin_op (int op,
			      unsigned char a_type, VOID_STAR ap, unsigned int na,
			      unsigned char b_type, VOID_STAR bp, unsigned int nb,
			      VOID_STAR cp)
{
   int *ic, *b;
   double *a, *c;
   unsigned int n, n_max;
   unsigned int da, db;

   (void) a_type;
   (void) b_type;

   a = (double *) ap;
   b = (int *) bp;
   c = (double *) cp;
   ic = (int *) cp;

   if (na == 1) da = 0; else da = 1;
   if (nb == 1) db = 0; else db = 1;

   if (na > nb) n_max = na; else n_max = nb;

   switch (op)
     {
      default:
	return 0;

      case SLANG_POW:
	for (n = 0; n < n_max; n++)
	  {
	     c[n] = pow (*a, (double) *b);   /* could return EDOM error */
	     a += da; b += db;
	  }
	break;

      case SLANG_PLUS:
	for (n = 0; n < n_max; n++)
	  {
	     c[n] = (*a + *b); a += da; b += db;
	  }
	break;

      case SLANG_MINUS:
	for (n = 0; n < n_max; n++)
	  {
	     c[n] = (*a - *b); a += da; b += db;
	  }
	break;

      case SLANG_TIMES:
	for (n = 0; n < n_max; n++)
	  {
	     c[n] = (*a * *b); a += da; b += db;
	  }
	break;

      case SLANG_DIVIDE:
	for (n = 0; n < n_max; n++)
	  {
	     if (*b == 0)
	       {
		  SLang_Error = SL_DIVIDE_ERROR;
		  return -1;
	       }
	     c[n] = (*a / *b); a += da; b += db;
	  }
	break;

      case SLANG_EQ:
	for (n = 0; n < n_max; n++)
	  {
	     ic[n] = (*a == (double) *b); a += da; b += db;
	  }
	break;

      case SLANG_NE:
	for (n = 0; n < n_max; n++)
	  {
	     ic[n] = (*a != (double) *b); a += da; b += db;
	  }
	break;

      case SLANG_GT:
	for (n = 0; n < n_max; n++)
	  {
	     ic[n] = (*a > (double) *b); a += da; b += db;
	  }
	break;

      case SLANG_GE:
	for (n = 0; n < n_max; n++)
	  {
	     ic[n] = (*a >= (double) *b); a += da; b += db;
	  }
	break;

      case SLANG_LT:
	for (n = 0; n < n_max; n++)
	  {
	     ic[n] = (*a < (double) *b); a += da; b += db;
	  }
	break;

      case SLANG_LE:
	for (n = 0; n < n_max; n++)
	  {
	     ic[n] = (*a <= (double) *b); a += da; b += db;
	  }
	break;
     }
   return 1;
}

static int int_double_bin_op (int op,
			      unsigned char a_type, VOID_STAR ap, unsigned int na,
			      unsigned char b_type, VOID_STAR bp, unsigned int nb,
			      VOID_STAR cp)
{
   int *ic, *a;
   double *b, *c;
   unsigned int n, n_max;
   unsigned int da, db;

   (void) a_type;
   (void) b_type;

   a = (int *) ap;
   b = (double *) bp;
   c = (double *) cp;
   ic = (int *) cp;

   if (na == 1) da = 0; else da = 1;
   if (nb == 1) db = 0; else db = 1;

   if (na > nb) n_max = na; else n_max = nb;

   switch (op)
     {
      default:
	return 0;

      case SLANG_POW:
	for (n = 0; n < n_max; n++)
	  {
	     c[n] = pow ((double) *a, *b);   /* could return EDOM error */
	     a += da; b += db;
	  }
	break;

      case SLANG_PLUS:
	for (n = 0; n < n_max; n++)
	  {
	     c[n] = (*a + *b); a += da; b += db;
	  }
	break;

      case SLANG_MINUS:
	for (n = 0; n < n_max; n++)
	  {
	     c[n] = (*a - *b); a += da; b += db;
	  }
	break;

      case SLANG_TIMES:
	for (n = 0; n < n_max; n++)
	  {
	     c[n] = (*a * *b); a += da; b += db;
	  }
	break;

      case SLANG_DIVIDE:
	for (n = 0; n < n_max; n++)
	  {
	     if (*b == 0.0)
	       {
		  SLang_Error = SL_DIVIDE_ERROR;
		  return -1;
	       }
	     c[n] = (*a / *b); a += da; b += db;
	  }
	break;

      case SLANG_EQ:
	for (n = 0; n < n_max; n++)
	  {
	     ic[n] = ((double) *a == *b); a += da; b += db;
	  }
	break;

      case SLANG_NE:
	for (n = 0; n < n_max; n++)
	  {
	     ic[n] = ((double)*a != *b); a += da; b += db;
	  }
	break;

      case SLANG_GT:
	for (n = 0; n < n_max; n++)
	  {
	     ic[n] = ((double)*a > *b); a += da; b += db;
	  }
	break;

      case SLANG_GE:
	for (n = 0; n < n_max; n++)
	  {
	     ic[n] = ((double)*a >= *b); a += da; b += db;
	  }
	break;

      case SLANG_LT:
	for (n = 0; n < n_max; n++)
	  {
	     ic[n] = ((double)*a < *b); a += da; b += db;
	  }
	break;

      case SLANG_LE:
	for (n = 0; n < n_max; n++)
	  {
	     ic[n] = ((double)*a <= *b); a += da; b += db;
	  }
	break;
     }
   return 1;
}

static int double_double_bin_op (int op,
				 unsigned char a_type, VOID_STAR ap, unsigned int na,
				 unsigned char b_type, VOID_STAR bp, unsigned int nb,
				 VOID_STAR cp)
{
   int *ic;
   double *a, *b, *c;
   unsigned int n, n_max;
   unsigned int da, db;

   (void) a_type;
   (void) b_type;

   a = (double *) ap;
   b = (double *) bp;
   c = (double *) cp;
   ic = (int *) cp;

   if (na == 1) da = 0; else da = 1;
   if (nb == 1) db = 0; else db = 1;

   if (na > nb) n_max = na; else n_max = nb;

   switch (op)
     {
      default:
	return 0;

      case SLANG_POW:
	for (n = 0; n < n_max; n++)
	  {
	     c[n] = pow (*a, *b);   /* could return EDOM error */
	     a += da; b += db;
	  }
	break;

      case SLANG_PLUS:
	for (n = 0; n < n_max; n++)
	  {
	     c[n] = (*a + *b); a += da; b += db;
	  }
	break;

      case SLANG_MINUS:
	for (n = 0; n < n_max; n++)
	  {
	     c[n] = (*a - *b); a += da; b += db;
	  }
	break;

      case SLANG_TIMES:
	for (n = 0; n < n_max; n++)
	  {
	     c[n] = (*a * *b); a += da; b += db;
	  }
	break;

      case SLANG_DIVIDE:
	for (n = 0; n < n_max; n++)
	  {
	     if (*b == 0.0)
	       {
		  SLang_Error = SL_DIVIDE_ERROR;
		  return -1;
	       }
	     c[n] = (*a / *b); a += da; b += db;
	  }
	break;

      case SLANG_EQ:
	for (n = 0; n < n_max; n++)
	  {
	     ic[n] = (*a == *b); a += da; b += db;
	  }
	break;

      case SLANG_NE:
	for (n = 0; n < n_max; n++)
	  {
	     ic[n] = (*a != *b); a += da; b += db;
	  }
	break;

      case SLANG_GT:
	for (n = 0; n < n_max; n++)
	  {
	     ic[n] = (*a > *b); a += da; b += db;
	  }
	break;

      case SLANG_GE:
	for (n = 0; n < n_max; n++)
	  {
	     ic[n] = (*a >= *b); a += da; b += db;
	  }
	break;

      case SLANG_LT:
	for (n = 0; n < n_max; n++)
	  {
	     ic[n] = (*a < *b); a += da; b += db;
	  }
	break;

      case SLANG_LE:
	for (n = 0; n < n_max; n++)
	  {
	     ic[n] = (*a <= *b); a += da; b += db;
	  }
	break;
     }
   return 1;
}

#endif				       /* SLANG_HAS_FLOAT */

static int
string_string_bin_op_result (int op, unsigned char a, unsigned char b,
			     unsigned char *c)
{
   (void) a;
   (void) b;
   switch (op)
     {
      default:
	return 0;

      case SLANG_PLUS:
	*c = SLANG_STRING_TYPE;
	break;

      case SLANG_GT:
      case SLANG_GE:
      case SLANG_LT:
      case SLANG_LE:
      case SLANG_EQ:
      case SLANG_NE:
	*c = SLANG_INT_TYPE;
	break;
     }
   return 1;
}

static int
string_string_bin_op (int op,
		      unsigned char a_type, VOID_STAR ap, unsigned int na,
		      unsigned char b_type, VOID_STAR bp, unsigned int nb,
		      VOID_STAR cp)
{
   int *ic;
   char **a, **b, **c;
   unsigned int n, n_max;
   unsigned int da, db;

   (void) a_type;
   (void) b_type;

   if (na == 1) da = 0; else da = 1;
   if (nb == 1) db = 0; else db = 1;

   if (na > nb) n_max = na; else n_max = nb;

   a = (char **) ap;
   b = (char **) bp;
   for (n = 0; n < n_max; n++)
     {
	if ((*a == NULL) || (*b == NULL))
	  {
	     SLang_verror (SL_VARIABLE_UNINITIALIZED, "String element not initialized for binary operation");
	     return -1;
	  }
	a += da; b += db;
     }

   a = (char **) ap;
   b = (char **) bp;
   ic = (int *) cp;
   c = NULL;

   switch (op)
     {
      case SLANG_DIVIDE:
      case SLANG_MINUS:
      default:
	return 0;

       case SLANG_PLUS:
	/* Concat */
	c = (char **) cp;
	for (n = 0; n < n_max; n++)
	  {
	     if (NULL == (c[n] = SLang_concat_slstrings (*a, *b)))
	       goto return_error;

	     a += da; b += db;
	  }
	break;

      case SLANG_NE:
	for (n = 0; n < n_max; n++)
	  {
	     ic [n] = (0 != strcmp (*a, *b));
	     a += da;
	     b += db;
	  }
	break;
      case SLANG_GT:
	for (n = 0; n < n_max; n++)
	  {
	     ic [n] = (strcmp (*a, *b) > 0);
	     a += da;
	     b += db;
	  }
	break;
      case SLANG_GE:
	for (n = 0; n < n_max; n++)
	  {
	     ic [n] = (strcmp (*a, *b) >= 0);
	     a += da;
	     b += db;
	  }
	break;
      case SLANG_LT:
	for (n = 0; n < n_max; n++)
	  {
	     ic [n] = (strcmp (*a, *b) < 0);
	     a += da;
	     b += db;
	  }
	break;
      case SLANG_LE:
	for (n = 0; n < n_max; n++)
	  {
	     ic [n] = (strcmp (*a, *b) <= 0);
	     a += da;
	     b += db;
	  }
	break;
      case SLANG_EQ:
	for (n = 0; n < n_max; n++)
	  {
	     ic [n] = (strcmp (*a, *b) == 0);
	     a += da;
	     b += db;
	  }
	break;
     }
   return 1;

   return_error:
   if (c != NULL)
     {
	unsigned int nn;
	for (nn = 0; nn < n; nn++)
	  {
	     SLang_free_slstring (c[nn]);
	     c[nn] = NULL;
	  }
	for (nn = n; nn < n_max; nn++)
	  c[nn] = NULL;
     }
   return -1;
}

static void string_destroy (unsigned char unused, VOID_STAR s)
{
   (void) unused;
   SLang_free_slstring (*(char **) s);
}

static int string_push (unsigned char unused, VOID_STAR sptr)
{
   (void) unused;
   SLang_push_string (*(char **) sptr);
   return 0;
}

/* Ref type */
int _SLang_pop_ref (_SLang_Ref_Type **ref)
{
   SLang_Object_Type obj;
   
   if (-1 == _SLang_pop_object_of_type (SLANG_REF_TYPE, &obj))
     return -1;
   
   *ref = (_SLang_Ref_Type *) obj.v.p_val;
   return 0;
}

int _SLang_push_ref (int is_global, VOID_STAR ptr)
{
   _SLang_Ref_Type *r;

   if (ptr == NULL)
     return _SLang_push_null ();

   r = (_SLang_Ref_Type *) SLmalloc (sizeof (_SLang_Ref_Type));
   if (r == NULL) return -1;
   
   r->is_global = is_global;
   r->v.nt = (SLang_Name_Type *) ptr;

   if (-1 == _SLang_push_void_star (SLANG_REF_TYPE, (VOID_STAR) r))
     {
	SLfree ((char *) r);
	return -1;
     }
   return 0;
}


static void ref_destroy (unsigned char type, VOID_STAR ptr)
{
   (void) type;
   SLfree ((char *) *(_SLang_Ref_Type **)ptr);
}

void _SLang_free_ref (_SLang_Ref_Type *ref)
{
   SLfree ((char *) ref);
}

static int ref_push (unsigned char type, VOID_STAR ptr)
{
   _SLang_Ref_Type *ref;
   
   (void) type;

   ref = *(_SLang_Ref_Type **) ptr;
   
   if (ref == NULL)
     return _SLang_push_null ();
   
   return _SLang_push_ref (ref->is_global, (VOID_STAR) ref->v.nt);
}

static char *ref_string (unsigned char type, VOID_STAR ptr)
{
   _SLang_Ref_Type *ref;
   
   (void) type;
   ref = *(_SLang_Ref_Type **) ptr;
   if (ref->is_global)
     return SLmake_string (ref->v.nt->name);
   
   return SLmake_string ("Local Variable Reference");
}



static int ref_dereference (unsigned char unused, VOID_STAR ptr)
{
   (void) unused;
   return _SLang_dereference_ref (*(_SLang_Ref_Type **) ptr);
}

/* NULL type */
int _SLang_push_null (void)
{
   return _SLang_push_void_star (SLANG_NULL_TYPE, NULL);
}

static int null_push (unsigned char unused, VOID_STAR ptr_unused)
{
   (void) unused; (void) ptr_unused;
   return _SLang_push_null ();
}

static int null_pop (unsigned char type, VOID_STAR ptr)
{
   SLang_Object_Type obj;

   if (-1 == _SLang_pop_object_of_type (type, &obj))
     return -1;

   *(char **) ptr = NULL;
   return 0;
}

/* Integer */
static int int_push (unsigned char type, VOID_STAR ptr)
{
   return _SLang_push_i_val (type, *(int *) ptr);
}

static int int_pop (unsigned char type, VOID_STAR ptr)
{
   return _SLang_pop_i_val (type, (int *) ptr);
}

static int
int_typecast (unsigned char a_type, VOID_STAR ap, unsigned int na,
	      unsigned char b_type, VOID_STAR bp)
{
   int *b;
   unsigned int i;
   char **s;
#if SLANG_HAS_FLOAT
   double *d;
#endif

   (void) b_type;
   b = (int *) bp;
   switch (a_type)
     {
      default:
	return 0;

      case SLANG_STRING_TYPE:
	s = (char **) ap;
	for (i = 0; i < na; i++)
	  {
	     if (s[i] == NULL) b[i] = 0;
	     else b[i] = s[i][0];
	  }
	break;

#if SLANG_HAS_FLOAT
      case SLANG_DOUBLE_TYPE:
	d = (double *) ap;
	for (i = 0; i < na; i++)
	  b[i] = (int) d[i];
	break;
#endif
     }
   return 1;
}

/* SLANG_CHAR_TYPE */
static int char_push (unsigned char unused, VOID_STAR ptr)
{
   (void) unused;
   return _SLang_push_i_val (SLANG_INT_TYPE, (int) *(unsigned char *) ptr);
}

static int char_pop (unsigned char unused, VOID_STAR ptr)
{
   int i;

   (void) unused;
   if (-1 == SLang_pop_integer (&i))
     return -1;
   *(unsigned char *) ptr = (unsigned char) i;
   return 0;
}

/* SLANG_INTP_TYPE */
static int intp_push (unsigned char unused, VOID_STAR ptr)
{
   (void) unused;
   return _SLang_push_i_val (SLANG_INT_TYPE, **(int **)ptr);
}

static int intp_pop (unsigned char unused, VOID_STAR ptr)
{
   (void) unused;
   return SLang_pop_integer (*(int **) ptr);
}

#if SLANG_HAS_FLOAT
/* Double */
static int double_push (unsigned char unused, VOID_STAR ptr)
{
   (void) unused;
   SLang_push_double (*(double *) ptr);
   return 0;
}

static int double_push_literal (unsigned char type, VOID_STAR ptr)
{
   (void) type;
   return SLang_push_double (**(double **)ptr);
}

static int double_pop (unsigned char unused, VOID_STAR ptr)
{
   (void) unused;
   return SLang_pop_double ((double *) ptr, NULL, NULL);
}

static void double_byte_code_destroy (unsigned char unused, VOID_STAR ptr)
{
   (void) unused;
   SLfree (*(char **) ptr);
}

#endif

static int undefined_push (unsigned char t, VOID_STAR p)
{
   (void) t; (void) p;
   if (SLang_Error == 0)
     SLang_Error = SL_VARIABLE_UNINITIALIZED;
   return -1;
}

int _SLregister_types (void)
{
   SLang_Class_Type *cl;

   if (-1 == _SLclass_init ())
     return -1;

   /* Undefined Type */
   if (NULL == (cl = SLclass_allocate_class ("Undefined_Type")))
     return -1;
   (void) SLclass_set_push_function (cl, undefined_push);
   (void) SLclass_set_pop_function (cl, undefined_push);
   if (-1 == SLclass_register_class (cl, SLANG_UNDEFINED_TYPE, sizeof (int),
				     SLANG_CLASS_TYPE_SCALAR))
     return -1;

   /* Integer Type */
   if (NULL == (cl = SLclass_allocate_class ("Integer_Type")))
     return -1;
   (void) SLclass_set_push_function (cl, int_push);
   (void) SLclass_set_pop_function (cl, int_pop);
   if (-1 == SLclass_register_class (cl, SLANG_INT_TYPE, sizeof (int),
				     SLANG_CLASS_TYPE_SCALAR))
     return -1;

   /* Char Type */
   if (NULL == (cl = SLclass_allocate_class ("Char_Type")))
     return -1;
   (void) SLclass_set_push_function (cl, char_push);
   (void) SLclass_set_pop_function (cl, char_pop);
   if (-1 == SLclass_register_class (cl, SLANG_CHAR_TYPE, sizeof (char),
				     SLANG_CLASS_TYPE_SCALAR))
     return -1;

   /* SLANG_INTP_TYPE */
   if (NULL == (cl = SLclass_allocate_class ("_IntegerP_Type")))
     return -1;
   (void) SLclass_set_push_function (cl, intp_push);
   (void) SLclass_set_pop_function (cl, intp_pop);
   if (-1 == SLclass_register_class (cl, SLANG_INTP_TYPE, sizeof (int),
				     SLANG_CLASS_TYPE_SCALAR))
     return -1;

   /* String Type */

   if (NULL == (cl = SLclass_allocate_class ("String_Type")))
     return -1;
   (void) SLclass_set_destroy_function (cl, string_destroy);
   (void) SLclass_set_push_function (cl, string_push);
   if (-1 == SLclass_register_class (cl, SLANG_STRING_TYPE, sizeof (char *),
				     SLANG_CLASS_TYPE_PTR))
     return -1;

   /* ref Type */
   if (NULL == (cl = SLclass_allocate_class ("Ref_Type")))
     return -1;
   cl->cl_dereference = ref_dereference;
   cl->cl_push = ref_push;
   cl->cl_destroy = ref_destroy;
   cl->cl_string = ref_string;
   if (-1 == SLclass_register_class (cl, SLANG_REF_TYPE,
				     sizeof (_SLang_Ref_Type *),
				     SLANG_CLASS_TYPE_PTR))
     return -1;

   /* NULL Type */

   if (NULL == (cl = SLclass_allocate_class ("Null_Type")))
     return -1;
   cl->cl_push = null_push;
   cl->cl_pop = null_pop;
   if (-1 == SLclass_register_class (cl, SLANG_NULL_TYPE, sizeof (char *),
				     SLANG_CLASS_TYPE_SCALAR))
     return -1;

#if SLANG_HAS_FLOAT
   if (NULL == (cl = SLclass_allocate_class ("Double_Type")))
     return -1;
   (void) SLclass_set_push_function (cl, double_push);
   (void) SLclass_set_pop_function (cl, double_pop);
   cl->cl_byte_code_destroy = double_byte_code_destroy;
   cl->cl_push_literal = double_push_literal;

   if (-1 == SLclass_register_class (cl, SLANG_DOUBLE_TYPE, sizeof (double),
				     SLANG_CLASS_TYPE_SCALAR))
     return -1;
#endif

   if ((-1 == SLclass_add_binary_op (SLANG_INT_TYPE, SLANG_INT_TYPE, int_bin_op, int_bin_op_result))
       || (-1 == SLclass_add_unary_op (SLANG_INT_TYPE, int_unary_op, int_unary_op_result))
#if SLANG_HAS_FLOAT
       || (-1 == SLclass_add_unary_op (SLANG_DOUBLE_TYPE, double_unary_op, double_unary_op_result))
       || (-1 == SLclass_add_binary_op (SLANG_DOUBLE_TYPE, SLANG_INT_TYPE, double_int_bin_op, double_bin_op_result))
       || (-1 == SLclass_add_binary_op (SLANG_INT_TYPE, SLANG_DOUBLE_TYPE, int_double_bin_op, double_bin_op_result))
       || (-1 == SLclass_add_binary_op (SLANG_DOUBLE_TYPE, SLANG_DOUBLE_TYPE, double_double_bin_op, double_bin_op_result))
#endif
       || (-1 == SLclass_add_typecast (SLANG_STRING_TYPE, SLANG_INT_TYPE, int_typecast, 0))
#if SLANG_HAS_FLOAT
       || (-1 == SLclass_add_typecast (SLANG_DOUBLE_TYPE, SLANG_INT_TYPE, int_typecast, 0))
       || (-1 == SLclass_add_typecast (SLANG_INT_TYPE, SLANG_DOUBLE_TYPE, double_typecast, 1))
#endif
       || (-1 == SLclass_add_binary_op (SLANG_STRING_TYPE, SLANG_STRING_TYPE, string_string_bin_op, string_string_bin_op_result))

       )
     return -1;

   return 0;
}

