/* Copyright (C) 1993-1996, Russell Lang.  All rights reserved.
  
  This file is part of GSview.
  
  This program is distributed with NO WARRANTY OF ANY KIND.  No author
  or distributor accepts any responsibility for the consequences of using it,
  or for whether it serves any particular purpose or works at all, unless he
  or she says so in writing.  Refer to the GSview Free Public Licence 
  (the "Licence") for full details.
  
  Every copy of GSview must include a copy of the Licence, normally in a 
  plain ASCII text file named LICENCE.  The Licence grants you the right 
  to copy, modify and redistribute GSview, but only under certain conditions 
  described in the Licence.  Among other things, the Licence requires that 
  the copyright notice and this notice be preserved on all copies.
*/

/* gvpm.c */
/* Main routines for PM GSview */
#define INCL_WINSTDDRAG
#include "gvpm.h"
#ifdef __EMX__
#include "sys/ioctl.h"
#endif
#include "gvphelp.h"   /* generated by doc2ipf.c */

#include <signal.h>
#include <setjmp.h>

char szAppName[MAXSTR] = GSVIEW_PRODUCT;  /* application name - for title bar */
char szHelpTopic[MAXSTR];
char szWait[MAXSTR];
char szExePath[MAXSTR];
char szHelpFile[MAXSTR];
char szFindText[MAXSTR];
char szIniFile[MAXSTR];
char szMMini[MAXSTR];
char previous_filename[MAXSTR];
char selectname[MAXSTR];
const char szScratch[] = "gv";	/* temporary filename prefix */
char *szSpoolPrefix = "\\\\spool\\";
HMTX hmutex_ps;
HAB hab;		/* Anchor Block */
HELPINIT hi_help;
ULONG os_version;
BOOL multithread;
HWND hwnd_frame;
HWND hwnd_bmp;
HWND hwnd_status;
HWND hwnd_button;
HWND hwnd_help;
HWND hwnd_modeless;	/* any modeless dialog box */
HWND hptr_crosshair;
HACCEL haccel;
POINTL buttonbar;
POINTL statusbar;
POINTL info_file;
POINTL info_page;
POINTL scroll_pos;
RECTL info_coord;
BOOL quitnow = FALSE;		/* Used to cause exit from nested message loops */

int percent_done;		/* percentage of document processed */
int percent_pending;		/* TRUE if WM_GSPERCENT is pending */

PSFILE psfile;		/* Postscript file structure */
BMAP bitmap;		/* Bitmap structure */
/* PROG gsprog;		/* Ghostscript program structure */
OPTIONS option;		/* GSview options (saved in INI file) */
DISPLAY display;	/* Display parameters */
PRINTER printer;	/* Printer GS parameters */

int page_skip = 5;		/* number of pages to skip in IDM_NEXTSKIP or IDM_PREVSKIP */
BOOL zoom = FALSE;		/* true if display zoomed */
BOOL debug = FALSE;		/* /D command line option used */
struct sound_s sound[NUMSOUND] = {
	{"SoundOutputPage", IDS_SNDPAGE, ""},
	{"SoundNoPage", IDS_SNDNOPAGE, BEEP},
	{"SoundNoNumbering", IDS_SNDNONUMBER, ""},
	{"SoundNotOpen", IDS_SNDNOTOPEN, ""},
	{"SoundError", IDS_SNDERROR, BEEP},
	{"SoundStart", IDS_SNDSTART, ""},
	{"SoundExit", IDS_SNDEXIT, ""},
	{"SoundBusy", IDS_SNDBUSY, BEEP},
};
PFN_MciPlayFile pfnMciPlayFile;

/* button bar */
struct button *buttonhead, *buttontail;
BOOL button_down;
struct button *button_current;
struct button *button_info;
POINTL button_shift;
POINTL button_size;

MRESULT EXPENTRY ClientWndProc(HWND, ULONG, MPARAM, MPARAM);
MRESULT EXPENTRY FrameWndProc(HWND, ULONG, MPARAM, MPARAM);
MRESULT EXPENTRY StatusWndProc(HWND, ULONG, MPARAM, MPARAM);
MRESULT EXPENTRY ButtonWndProc(HWND, ULONG, MPARAM, MPARAM);
PFNWP OldFrameWndProc;

BOOL scan_bitmap(BMAP *pbm);
HBITMAP make_bitmap(BMAP *, ULONG, ULONG, ULONG, ULONG, ULONG);
void cursorpos_paint(HPS hps);
void map_pt_to_pixel(float *x, float *y);
MRESULT DragOver(PDRAGINFO);
MRESULT Drop(PDRAGINFO);
void highlight_words(HPS ps, int first, int last);
BOOL text_marking = FALSE;
int text_mark_first = -1;
int text_mark_last = -1;

void
update_scroll_bars(void)
{
    /* Cause update of scroll bars etc. */
    SWP swp;
    WinQueryWindowPos(hwnd_bmp, &swp);
    WinSendMsg(hwnd_bmp, WM_SIZE, MPFROM2SHORT(swp.cx, swp.cy), MPFROM2SHORT(swp.cx, swp.cy));
}



/* Return TRUE if is OK to exit */
BOOL
query_close(void)
{
char buf[MAXSTR];
    /* tell GS DLL to unload */
    begin_crit_section();
    pending.unload = TRUE;
    quitnow = TRUE;
    end_crit_section();
    if (multithread)
        DosPostEventSem(display.event);	/* unblock display thread */
    return TRUE;
}

void
exit_func(void)
{
    psfile_free(&psfile);
    if (option.settings)
	write_profile();
}


/* code to protect GSview against Ghostscript memory violations */
jmp_buf sig_env;

void
sig_handler(int code)
{
	signal(code, SIG_ACK); 
	longjmp(sig_env, 1);
}

int
gs_sig_process()
{
    if (setjmp(sig_env) == 0) {
	/* got here on direct call */
	signal(SIGSEGV, sig_handler);
	signal(SIGFPE, sig_handler);
        gs_process();	/* start Ghostscript */
    }
    else {
	/* got here by calling longjmp */
	pending.now = FALSE;
	message_box("Segmentation violation or Floating Point Exception in Ghostscript or second GSview thread\rPlease exit GSview", 0);
    }
    signal(SIGFPE, SIG_DFL);	/* restore default handler */
    signal(SIGSEGV, SIG_DFL);	/* restore default handler */
}


/* Thread which loads Ghostscript DLL for display */
void 
gs_thread(void *arg)
{
HAB hab;
HMQ hmq;

    hab = WinInitialize(0);
/*
    hmq = WinCreateMsgQueue(hab, 0);
*/

    while (!quitnow) {
	if (!pending.now)
	    wait_event();
	if (!quitnow)
	    gs_sig_process();
    }

/*
    WinDestroyMsgQueue(hmq);
*/
    WinTerminate(hab);
    /* signal that we have finished */
    display.tid = 0;
}


int
main(int argc, char *argv[])
{
  HMQ hand_mq;		/* message queue */
  QMSG q_mess;		/* queue message */
  ULONG flFlags;	/* Window frame definition */
  APIRET rc = 0;
  SWP swp;

  hab = WinInitialize(0);	/* Get the Anchor Block */

  hand_mq = WinCreateMsgQueue(hab, 0); /* start a queue */

  if (argc > 4) {
	rc = 1;
	error_message("Usage: gvpm [/D] [/F] [/P] [/S[port]] [filename]");
  }

  rc = gsview_init(argc, argv);
/*
  atexit(exit_func);
*/

  if (!rc) {
        WinShowWindow(hwnd_frame, TRUE);
	haccel = WinQueryAccelTable(hab, hwnd_frame);
	/* create help window */
	hi_help.cb = sizeof(HELPINIT);
	hi_help.ulReturnCode = 0;
	hi_help.pszTutorialName = NULL;
	hi_help.phtHelpTable = NULL;
	hi_help.hmodAccelActionBarModule = 0;
	hi_help.idAccelTable = 0;
	hi_help.idActionBar = 0;
	hi_help.pszHelpWindowTitle=(PSZ)"GSview Help";
	hi_help.hmodHelpTableModule = 0;
	hi_help.fShowPanelId = 0;
	hi_help.pszHelpLibraryName = (PSZ)szHelpFile;
	hwnd_help = WinCreateHelpInstance(hab, &hi_help);
	if (!hwnd_help || hi_help.ulReturnCode) {
	    char buf[256];
	    sprintf(buf, "WinCreateHelpInstance handle=%d, rc=%d",  
		hwnd_help, hi_help.ulReturnCode);
	    message_box(buf, 0);
	}
	if (hwnd_help)
	    WinAssociateHelpInstance(hwnd_help, hwnd_frame);
  }
  if (multithread) {
      /* start thread for displaying */
/*
      display.tid = _beginthread(gs_thread, NULL, 16384, NULL);
*/
      display.tid = _beginthread(gs_thread, NULL, 32768, NULL);
  }

  play_sound(SOUND_START);
  info_wait(IDS_NOWAIT);
  if (gsview_changed())
      WinPostMsg(hwnd_bmp, WM_CLOSE, MPFROMLONG(0), MPFROMLONG(0));


  /* message loop */
  while (!rc && !quitnow && WinGetMsg(hab, &q_mess, 0L, 0, 0)) {
      WinDispatchMsg(hab, &q_mess);
      if (multithread) {
	  /* release other thread if needed */
	  if (pending.unload || pending.now || pending.next)
		DosPostEventSem(display.event);
      }
      else {
	  if (pending.now)
	    gs_sig_process();	/* start Ghostscript */
      }
  }

  play_sound(SOUND_EXIT);
  if (multithread && display.tid) {
      int i = 100;	/* 10 seconds */
      pending.unload = TRUE;
      quitnow = TRUE;
      DosPostEventSem(display.event);	/* unblock display thread */
      while (i && display.tid) {
	  /* wait for thread to close */
	  DosSleep(100);
          DosPostEventSem(display.event);	/* unblock display thread */
	  i--;
	  peek_message();
      }
  }

  /* Shut down the application window and queue */
  if (hwnd_help)
      WinDestroyHelpInstance(hwnd_help);
  DosCloseMutexSem(hmutex_ps);
  WinDestroyWindow(hwnd_frame);
  WinDestroyMsgQueue(hand_mq);
  WinTerminate(hab);

  exit_func();

  return rc;
}

#define MAX_PAL_SIZE 256
void
make_palette(BMAP *pbm)
{
ULONG tbl[MAX_PAL_SIZE];
PRGB2 palptr = (PRGB2) ((PBYTE)(pbm->pbmi) + pbm->pbmi->cbFix);
RGB *old_palptr = (RGB *)palptr;
int palcount = pbm->palimportant;
int i;
BOOL old_bmp = (pbm->pbmi->cbFix == sizeof(BITMAPINFOHEADER));
    if (old_bmp) {
	for (i=0; i<palcount; i++) {
	    tbl[i] = (old_palptr->bRed<<16) + (old_palptr->bGreen<<8) + (old_palptr->bBlue);
	    palptr++;
	}
    }
    else {
	for (i=0; i<palcount; i++) {
	    tbl[i] = (palptr->bRed<<16) + (palptr->bGreen<<8) + (palptr->bBlue);
	    palptr++;
	}
    }
    if (display.hpal_exists)
	GpiDeletePalette(display.hpal);
    display.hpal = GpiCreatePalette(hab, 0L, LCOLF_CONSECRGB, palcount, tbl);
    display.hpal_exists = TRUE;
}


/* scan bitmap */
/* update bitmap structure */
/* return value is TRUE if bitmap dimension has changed */
BOOL
scan_bitmap(BMAP *pbm)
{
PBITMAPINFO2 pbmi;
PBITMAPINFO old_pbmi;
BOOL old_bmp;

    if (!pbm->valid)
	return TRUE;
    if (!pbm->pbmi)
	return TRUE;

    pbmi = pbm->pbmi;
    old_pbmi = (PBITMAPINFO)pbmi;
    old_bmp = (pbmi->cbFix == sizeof(BITMAPINFOHEADER));

    if (old_bmp) {
  	/* it is a BITMAPINFO */
  	switch(old_pbmi->cBitCount) {
  	  case 24:
  	    pbm->palsize = 0;
  	    break;
  	  case 8:
  	    pbm->palsize = 256;
  	    break;
  	  case 4:
  	    pbm->palsize = 16;
  	    break;
  	  case 1:
  	    pbm->palsize = 2;
  	    break;
  	  default:
	    pbm->valid = FALSE;
  	    error_message("scan_bitmap: wrong number of bits"); /* panic */
  	    return FALSE;
	}
	pbm->palimportant = pbm->palsize;
	pbm->palsize = pbm->palsize * sizeof(RGB);
        pbm->bits   = (PBYTE)old_pbmi + old_pbmi->cbFix + pbm->palsize;
	pbm->width  = old_pbmi->cx;
	pbm->height = old_pbmi->cy;
	pbm->planes = old_pbmi->cPlanes;
	pbm->depth  = old_pbmi->cBitCount;
    }
    else {
 	/* it is a BITMAPINFO2 */
  	switch(pbmi->cBitCount) {
  	  case 24:
  	    pbm->palsize = 0;
  	    break;
  	  case 8:
  	    pbm->palsize = 256;
  	    break;
  	  case 4:
  	    pbm->palsize = 16;
  	    break;
  	  case 1:
  	    pbm->palsize = 2;
  	    break;
  	  default:
	    pbm->valid = FALSE;
  	    error_message("scan_bitmap: wrong number of bits"); /* panic */
  	    return FALSE;
	}
	if ( (pbmi->cbFix > (&(pbmi->cclrUsed) - &(pbmi->cbFix)))
		&& (pbmi->cclrUsed != 0) && (pbmi->cBitCount != 24) )
	    pbm->palsize = pbmi->cclrUsed;
	pbm->palimportant = pbm->palsize;
	if ( (pbmi->cbFix > (&(pbmi->cclrImportant) - &(pbmi->cbFix)))
		&& (pbmi->cclrImportant != 0) && (pbmi->cBitCount != 24) )
	    pbm->palimportant = pbmi->cclrImportant;
	pbm->palsize = pbm->palsize * sizeof(RGB2);
        pbm->bits   = (PBYTE)pbmi + pbmi->cbFix + pbm->palsize;
	pbm->width  = pbmi->cx;
	pbm->height = pbmi->cy;
	pbm->planes = pbmi->cPlanes;
	pbm->depth  = pbmi->cBitCount;
    }

    if ((pbm->palsize != pbm->old_palsize) || (pbm->palimportant != pbm->old_palimportant)) {
	if ( (pbm->depth == 8) && display.hasPalMan )
	    make_palette(pbm);
	pbm->old_palimportant = pbm->palimportant;
    }

    if ( (pbm->width   == pbm->old_width) && 
    	 (pbm->height  == pbm->old_height) &&
	 (pbm->planes  == pbm->old_planes) && 
	 (pbm->depth   == pbm->old_depth) &&
	 (pbm->palsize == pbm->old_palsize) &&
	 (pbm->old_bmp == old_bmp) )
	return FALSE;

    /* bitmap has changed */
    pbm->old_width   = pbm->width;
    pbm->old_height  = pbm->height;
    pbm->old_planes  = pbm->planes;
    pbm->old_depth   = pbm->depth;
    pbm->old_palsize = pbm->palsize;
    pbm->old_bmp     = old_bmp;
    return TRUE;
}

/* copy bitmap to the clipboard */
void
copy_clipboard(void)
{
HBITMAP hbmp;
    if (!gsdll.device || !bitmap.valid) {
	gserror(0, "Cannot copy to clipboard:\nNo Bitmap displayed", MB_ICONEXCLAMATION, SOUND_ERROR);
	return;
    }
    if ( text_index && (text_mark_first != -1) && (text_mark_last != -1)) {
	/* copy text, not image */
	int first, last, line;
	int length;
	int i;
	LPSTR data;
	PVOID mem;
	first = text_mark_first;
	last = text_mark_last;
	if (first > last) {
	    first = text_mark_last;
	    last = text_mark_first;
	}
	line = text_index[first].line;
	length = 1;
	for (i=first; i<=last; i++) {
	    if (text_index[i].line != line) {
	        line = text_index[i].line;
		length += 2;
	    }
	    length += strlen( text_words + text_index[i].word ) + 1;
	}
	if (DosAllocSharedMem(&mem, NULL, length, PAG_READ | PAG_WRITE | PAG_COMMIT | OBJ_GIVEABLE)) {
	    message_box("out of memory", 0);
	    return;
	}
	data = (LPSTR)mem;
	line = text_index[first].line;
	for (i=first; i<=last; i++) {
	    if (text_index[i].line != line) {
	        line = text_index[i].line;
		strcpy(data, "\r\n");
		data += strlen(data);
	    }
	    strcpy(data, text_words + text_index[i].word);
	    strcat(data, " ");
	    data += strlen(data);
	}
        if (WinOpenClipbrd(hab)) {
	    WinEmptyClipbrd(hab);
	    WinSetClipbrdData(hab, (ULONG)mem, CF_TEXT, CFI_POINTER);
	    WinCloseClipbrd(hab);
	}
	return;
    }

    if (WinOpenClipbrd(hab)) {
	/* get bmp mutex to stop gs.exe changing bitmap while we copy it */
	if (gsdll.lock_device && gsdll.device)
	    (*gsdll.lock_device)(gsdll.device, 1);
	if (scan_bitmap(&bitmap)) {
	    /* bitmap has changed */
	    update_scroll_bars();
	}
	hbmp = make_bitmap(&bitmap, 0, 0, bitmap.width, bitmap.height, bitmap.depth);
	if (hbmp) {
	    WinEmptyClipbrd(hab);
	    WinSetClipbrdData(hab, (ULONG)hbmp, CF_BITMAP, CFI_HANDLE);
	}
	
	if (gsdll.lock_device && gsdll.device)
	    (*gsdll.lock_device)(gsdll.device, 0);
	WinCloseClipbrd(hab);
    }
}

HBITMAP
make_bitmap(BMAP *pbm, ULONG left, ULONG bottom, ULONG right, ULONG top, ULONG depth)
{
HDC hdcMem = DEV_ERROR;
HPS hps = GPI_ERROR;
HBITMAP hbmp = GPI_ERROR, hbmr = HBM_ERROR;
SIZEL sizePS;
BITMAPINFOHEADER2 bmih;

	if ( (left == right) || (bottom == top) )
		return (HBITMAP)NULL;

	if (right > pbm->width)
		right = pbm->width;
	if (left > pbm->width)
		left = 0;
	if (top > pbm->height)
		top = pbm->height;
	if (bottom > pbm->height)
		bottom = 0;
		
	memset(&bmih, 0, sizeof(bmih));
	bmih.cbFix = sizeof(BITMAPINFOHEADER2);
	bmih.cx = right - left;
	bmih.cy = top - bottom;
	bmih.cPlanes = 1;
	bmih.cBitCount = depth;

	/* create memory DC compatible with screen */
	hdcMem = DevOpenDC(hab, OD_MEMORY, "*", 0L, NULL, NULLHANDLE);

	sizePS.cx = right - left;
	sizePS.cy = top - bottom;
	if (hdcMem != DEV_ERROR)
	    hps = GpiCreatePS(hab, hdcMem, &sizePS, 
		PU_PELS | GPIF_DEFAULT | GPIT_MICRO | GPIA_ASSOC );

	if (hps != GPI_ERROR)
		hbmp = GpiCreateBitmap(hps, &bmih, 0L, NULL, NULL);

	if (hbmp != GPI_ERROR)
		hbmr = GpiSetBitmap(hps, hbmp);


	if (hbmr != HBM_ERROR) {
		LONG rc;
		ERRORID eid;
  	    	POINTL apts[4];
		/* target is inclusive */
		apts[0].x = 0;
		apts[0].y = 0;
		apts[1].x = right - left - 1;
		apts[1].y = top - bottom - 1;
		/* source is not inclusive of top & right borders */
		apts[2].x = left;
		apts[2].y = bottom;
		apts[3].x = right;
		apts[3].y = top;

		rc = 0;
		eid = WinGetLastError(hab);
		if ((bitmap.depth == 8) && display.hasPalMan && display.hpal_exists)
		    GpiSelectPalette(hps, display.hpal);
	    	rc = GpiDrawBits(hps, pbm->bits, pbm->pbmi, 4, apts, 
			(bitmap.depth != 1) ? ROP_SRCCOPY : ROP_NOTSRCCOPY, 0);
		if ((bitmap.depth == 8) && display.hasPalMan && display.hpal_exists)
		    GpiSelectPalette(hps, (HPAL)NULL);
		if (rc==0) {
			char buf[MAXSTR];
			eid = WinGetLastError(hab);
			sprintf(buf,"make_bitmap: GpiDrawBits rc = %08x, eid = %08x",rc, eid);
			message_box(buf, 0);
		}
	}
	
	if (hbmr != HBM_ERROR)
		GpiSetBitmap(hps, (ULONG)0);
	if (hps != GPI_ERROR)
		GpiDestroyPS(hps);
	if (hdcMem != DEV_ERROR)
		DevCloseDC(hdcMem);

	if ( (hbmr == HBM_ERROR) || (hdcMem == DEV_ERROR) ||
		(hbmp == GPI_ERROR) || (hps == GPI_ERROR) ) {
		if (hbmp != GPI_ERROR)
			GpiDeleteBitmap(hbmp);
		return 0;
	}
	return hbmp;
}

MRESULT
paint_bitmap(HPS ps, PRECTL prect, int scrollx, int scrolly)
{
    POINTL apts[4];
    RECTL rect;
    int wx, wy;
    int ewx = 0;	/* empty area at right */
    int ewy = 0;	/* empty area at top */
    HRGN hrgn, hrgnold;
    HPOINTER hptr;
    if (WinIsRectEmpty(hab, prect))
	return 0;

/* KLUDGE BEGIN */
    /* remove pointer to cope with buggy display drivers */
    hptr = WinQueryPointer(HWND_DESKTOP);
    WinSetPointer(HWND_DESKTOP, 0);
/* KLUDGE END */

    /* source is not inclusive of top & right borders */
    wx = prect->xRight - prect->xLeft;   /* update width */
    wy = prect->yTop -   prect->yBottom; /* update height */
    if (prect->xLeft < display.offset.x)
	apts[2].x = 0;
    else
	apts[2].x = (prect->xLeft-display.offset.x)   + scrollx;
    if (prect->yBottom < display.offset.y)
        apts[2].y = 0;
    else
	apts[2].y = (prect->yBottom-display.offset.y) + scrolly;
    if (apts[2].x > bitmap.width)
	    apts[2].x = bitmap.width;
    if (apts[2].x + wx > bitmap.width) {
	    wx = bitmap.width - apts[2].x;
    }
    apts[3].x = apts[2].x + wx;
    if (apts[2].y > bitmap.height)
	    apts[2].y = bitmap.height;
    if (apts[2].y + wy > bitmap.height) {
	    wy = bitmap.height - apts[2].y;
    }
    apts[3].y = apts[2].y + wy;
    /* target is inclusive */
    if (prect->xLeft < display.offset.x)
	apts[0].x = display.offset.x;
    else 
	apts[0].x = prect->xLeft;
    if (prect->yBottom < display.offset.y)
 	apts[0].y = display.offset.y;
    else
	apts[0].y = prect->yBottom;
    apts[1].x = apts[0].x + wx - 1;
    apts[1].y = apts[0].y + wy - 1;

    if ( (option.drawmethod == IDM_DRAWWIN) || 
         ((option.drawmethod == IDM_DRAWDEF) && (display.bitcount == 4)) ) /* standard VGA is buggy */
        {
	/* slow code to dodge OS/2 bugs */
	/* this code double buffers the bitmap and works on a standard VGA
	 * but didn't work on an ATI Ultra Graphics Pro in 8514 emulation
	 */
	/* This won't work for OS/2 2.11, S3 or ATI GU, 8bit/pixel display, 8bit/pixel bitmap */
	HBITMAP hbmp;
	/* create a bitmap */
	hbmp = make_bitmap(&bitmap, apts[2].x, apts[2].y, apts[3].x, apts[3].y, bitmap.depth);
	/* Draw it to the display */
	if (hbmp) {
	    WinDrawBitmap(ps, hbmp, NULL, &apts[0], CLR_BLACK, CLR_WHITE, DBM_NORMAL);
	    GpiDeleteBitmap(hbmp);
	}
    }
    else {   /* (option.drawmethod == IDM_DRAWGPI) || (default and non buggy display) */
	/* fast code which doesn't always work */
	/* This code works on the Trident SVGA and 8514 in 256 color mode,
 	 * but GpiDrawBits fails with a SYS3175 on the standard VGA.
	 */
	/* This won't work for version 2.11, S3 or ATI GU, 8bit/pixel display, 1bit/pixel bitmap */
	GpiDrawBits(ps, bitmap.bits, bitmap.pbmi, 4, apts, 
		(bitmap.depth != 1) ? ROP_SRCCOPY : ROP_NOTSRCCOPY, 0);
    }
    /* Fill areas around page */
    if (prect->yBottom < display.offset.y) {	/* bottom centre */
	rect.yBottom = prect->yBottom;
	rect.yTop = display.offset.y;
	rect.xLeft = apts[0].x;
	rect.xRight = rect.xLeft + wx;
	WinFillRect(ps, &rect, SYSCLR_DIALOGBACKGROUND);
    }
    if (prect->yTop > bitmap.height + display.offset.y) { /* top centre */
	rect.yBottom = bitmap.height + display.offset.y;
	rect.yTop = prect->yTop;
	rect.xLeft = apts[0].x;
	rect.xRight = rect.xLeft + wx;
	WinFillRect(ps, &rect, SYSCLR_DIALOGBACKGROUND);
    }
    if (prect->xLeft < display.offset.x) { /* left */
	rect.yBottom = prect->yBottom;
	rect.yTop = prect->yTop;
	rect.xLeft = prect->xLeft;
	rect.xRight = display.offset.x;
	WinFillRect(ps, &rect, SYSCLR_DIALOGBACKGROUND);
    }
    if (prect->xRight > bitmap.width + display.offset.x) { /* right */
	rect.yBottom = prect->yBottom;
	rect.yTop = prect->yTop;
	rect.xLeft = bitmap.width + display.offset.x;
	rect.xRight = prect->xRight;
	WinFillRect(ps, &rect, SYSCLR_DIALOGBACKGROUND);
    }

    /* clip other drawing commands to update rectangle */
    hrgn = GpiCreateRegion(ps, 1, prect);
    GpiSetClipRegion(ps, hrgn, &hrgnold);
    if (hrgnold)
	GpiDestroyRegion(ps, hrgnold);

    /* draw bounding box */
    if ((psfile.doc != (PSDOC *)NULL) && option.show_bbox) {
        POINTL pt;
	float x, y;
	/* map bounding box to device coordinates */
	x = psfile.doc->boundingbox[LLX];
	y = psfile.doc->boundingbox[LLY];
	map_pt_to_pixel(&x, &y);
	rect.xLeft   = (int)x;
	rect.yBottom = (int)y;
	x = psfile.doc->boundingbox[URX];
	y = psfile.doc->boundingbox[URY];
	map_pt_to_pixel(&x, &y);
	rect.xRight  = (int)x;
	rect.yTop    = (int)y;

	/* draw it inverted */
	GpiSetColor(ps, CLR_TRUE);
	GpiSetLineType(ps, LINETYPE_SHORTDASH);
	GpiSetMix(ps, FM_XOR);
	pt.x = rect.xLeft; pt.y = rect.yBottom;
	GpiMove(ps, &pt);
	pt.x = rect.xRight; /* might be better to use GpiPolyLine */
	GpiLine(ps, &pt);
	pt.y = rect.yTop;
	GpiLine(ps, &pt);
	pt.x = rect.xLeft;
	GpiLine(ps, &pt);
	pt.y = rect.yBottom;
	GpiLine(ps, &pt);
	GpiSetLineType(ps, LINETYPE_DEFAULT);
	GpiSetMix(ps, FM_DEFAULT);
	GpiSetColor(ps, CLR_TRUE);
    }

    /* highlight found search word */
    if (gsdll.device && display.show_find) {
	float x, y;
	/* map bounding box to device coordinates */
	x = psfile.text_bbox.llx;
	y = psfile.text_bbox.lly;
	map_pt_to_pixel(&x, &y);
	rect.xLeft   = (int)x;
	rect.yBottom = (int)y;
	x = psfile.text_bbox.urx;
	y = psfile.text_bbox.ury;
	map_pt_to_pixel(&x, &y);
	rect.xRight  = (int)x;
	rect.yTop    = (int)y;
	if (rect.yTop < rect.yBottom) {
	    int temp = rect.yTop;
	    rect.yTop = rect.yBottom;
	    rect.yBottom = temp;
	}

	/* invert text */
	WinInvertRect(ps, &rect);
    }

    /* highlight marked words */
    highlight_words(ps, text_mark_first, text_mark_last);

    GpiSetClipRegion(ps, NULLHANDLE, &hrgnold);
    if (hrgnold)
	GpiDestroyRegion(ps, hrgnold);

/* KLUDGE BEGIN */
    /* restore pointer after we removed it */
    WinSetPointer(HWND_DESKTOP, hptr);
/* KLUDGE END */

    return 0;
}

/* enable or disable a menu item */
void
enable_menu_item(int menuid, int itemid, BOOL enabled)
{
HWND hwndMenu;
MENUITEM mi;
	hwndMenu = WinWindowFromID(hwnd_frame, FID_MENU);
	WinSendMsg(hwndMenu, MM_QUERYITEM, 
		MPFROM2SHORT(menuid, TRUE), MPFROMP(&mi));
	WinSendMsg(mi.hwndSubMenu, MM_SETITEMATTR, MPFROMLONG(itemid),
		MPFROM2SHORT(MIA_DISABLED, enabled ? 0 : MIA_DISABLED));
}


void
init_menu(int menuid)
{
BOOL idle;
BOOL addeps;
    idle = (gsdll.state != BUSY);
    switch (menuid) {
	case IDM_FILEMENU:
	    enable_menu_item(IDM_FILEMENU, IDM_PRINT, idle);
	    enable_menu_item(IDM_FILEMENU, IDM_PRINTTOFILE, idle);
	    enable_menu_item(IDM_FILEMENU, IDM_EXTRACT, idle);
	    enable_menu_item(IDM_FILEMENU, IDM_PSTOEPS, idle);
	    break;
	case IDM_EDITMENU:
	    request_mutex();
	    enable_menu_item(IDM_EDITMENU, IDM_COPYCLIP, bitmap.valid);
	    enable_menu_item(IDM_EDITMENU, IDM_PASTETO, bitmap.valid);
    	    addeps =  (psfile.doc != (PSDOC *)NULL) && psfile.doc->epsf && idle;
	    enable_menu_item(IDM_EDITMENU, IDM_ADDEPSMENU, addeps);
	    enable_menu_item(IDM_ADDEPSMENU, IDM_MAKEEPSU, addeps);
    	    addeps =  addeps && bitmap.valid;
	    enable_menu_item(IDM_ADDEPSMENU, IDM_MAKEEPSI, addeps);
	    enable_menu_item(IDM_ADDEPSMENU, IDM_MAKEEPST4, addeps);
	    enable_menu_item(IDM_ADDEPSMENU, IDM_MAKEEPST, addeps);
	    enable_menu_item(IDM_EDITMENU, IDM_EXTEPSMENU, 
		    ((psfile.preview == IDS_EPST) || (psfile.preview == IDS_EPSW)) && idle);
	    enable_menu_item(IDM_EDITMENU, IDM_TEXTEXTRACT, idle);
	    enable_menu_item(IDM_EDITMENU, IDM_TEXTFIND, idle);
	    enable_menu_item(IDM_EDITMENU, IDM_TEXTFINDNEXT, idle);
	    release_mutex();
	    break;
    }
}

/* This is the window function */
MRESULT EXPENTRY ClientWndProc(HWND hwnd, ULONG mess, 
			MPARAM mp1, MPARAM mp2)
{
  char buf[MAXSTR];

  static int cxClient, cyClient;
  static int cxAdjust, cyAdjust;
  static int nHscrollMax, nHscrollPos;
  static int nVscrollMax, nVscrollPos;
  int nHscrollInc;
  int nVscrollInc;
  HWND hwndScroll;
  HPS hps;
  RECTL rect;
  ULONG ulclr;

    switch(mess) {
	case WM_CREATE:
	    break;
	case WM_ERASEBACKGROUND:
	    /* by returning TRUE, the Presentation Manager automatically clears
	     * the window each time the window is resized or moved.
	     */
	    if (!gsdll.device || !bitmap.valid) {
		WinFillRect((HPS)mp1, (PRECTL)mp2, SYSCLR_DIALOGBACKGROUND);
		return (MRESULT)FALSE;	/* say we have done it */
	    }
	    return (MRESULT)TRUE;
	case WM_GSSYNC:
	    if (!WinInvalidateRect(hwnd_bmp, (PRECTL)NULL, TRUE))
		    error_message("error invalidating rect");
	    if (!WinUpdateWindow(hwnd_bmp))
		    error_message("error updating window");
	    return 0;
	case WM_GSPAGE:
	    play_sound(SOUND_PAGE);
	    if (!WinInvalidateRect(hwnd_bmp, (PRECTL)NULL, TRUE))
		    error_message("error invalidating rect");
	    if (!WinUpdateWindow(hwnd_bmp))
		    error_message("error updating window");
	    if (display.show_find)
		scroll_to_find();
	    return 0;
	case WM_GSDEVICE:
	    if (mp1 && gsdll.device) {
		memset(&bitmap, 0, sizeof(bitmap));
		/* if we don't know the bitmap pointer, get it now */
		(*gsdll.get_bitmap)(gsdll.device, (unsigned char **)&bitmap.pbmi);
		bitmap.valid = TRUE;
	    }
	    else {
		bitmap.valid = FALSE;
		update_scroll_bars();
	    }
	    return 0;
	case WM_GSSIZE:
	    if (scan_bitmap(&bitmap)) {
		/* bitmap has changed */
		update_scroll_bars();
	    }
	    return 0;
	case WM_GSMESSBOX:
	    /* delayed message box, usually from other thread */
	    load_string((int)mp1, buf, sizeof(buf));
	    message_box(buf, (int)mp2);
	    return 0;
	case WM_GSSHOWMESS:
	    /* delayed show message, usually from other thread */
	    gs_showmess();
	    return 0;
	case WM_GSREDISPLAY:
	    { PSFILE *tpsfile = gsview_openfile(psfile.name);
	      if (tpsfile) {
		tpsfile->pagenum = psfile.pagenum;
		request_mutex();
		pending.psfile = tpsfile;
		pending.now = TRUE;
		release_mutex();
	      }
	    }
	    return 0;
	case WM_GSTITLE:
	    /* update title */
	    request_mutex();
	    if (psfile.name[0] != '\0') {
		char buf[256];
		char *p;
		p = strrchr(psfile.name, '\\');
		if (p)
		    p++;
		else
		    p = psfile.name;
		sprintf(buf, "%s - %s", szAppName, p);
		WinSetWindowText(hwnd_frame, buf);
	    }
	    else
		WinSetWindowText(hwnd_frame, szAppName);
	    release_mutex();
	    return 0;
	case WM_GSPERCENT:
	    percent_pending = FALSE;
	    WinInvalidateRect(hwnd_status, (PRECTL)NULL, FALSE);
	    WinUpdateWindow(hwnd_status);
	    return 0;
	case WM_GSTEXTINDEX:
	    make_text_index();
	    text_marking = FALSE;
	    text_mark_first = text_mark_last = -1;
	    return 0;
	case WM_GSWAIT:
	    info_wait((int)mp1);
	    return 0;
	case WM_INITMENU:
	    init_menu((int)mp1);
	    WinSetAccelTable(hab, NULLHANDLE, hwnd_frame);	/* disable keyboard accelerators */
	    break;
	case WM_MENUEND:
	    WinSetAccelTable(hab, haccel, hwnd_frame);	/* reenable keyboard accelerators */
	    break;
	case WM_COMMAND:
		if (LONGFROMMP(mp1) == IDM_DROP) {
		    char *cmd = (char *)mp2;
		    if ((cmd[0] == '-') || (cmd[0] == '/')) {
			switch (toupper(cmd[1])) {
			    case 'P':
		              gsview_selectfile(cmd+2);
			      if (!dfreopen())
				  break;
			      if (psfile.name[0] != '\0')
				  gsview_print(FALSE);
			      dfclose();
			      break;
			    case 'F':
		              gsview_selectfile(cmd+2);
			      if (!dfreopen())
				  break;
			      if (psfile.name[0] != '\0')
				  gsview_print(TRUE);
			      dfclose();
			      break;
			    case 'S':
			      if (cmd[2] != ' ') {
				char *fname;
				/* skip over port name */
				for (fname=cmd+2; *fname && *fname!=' '; fname++)
				  /* nothing */ ;
			        /* skip blanks until file name */
			        if (*fname) {
			          *fname++ = '\0'; /* place null after port name */
			          for (; *fname==' '; fname++)
			            /* nothing */ ;
			        }
			        if (*fname) {
			            /* found both filename and port */
			            gsview_spool(fname, cmd+2);
			            break;
			        }
			      }
		              gsview_spool(cmd+2, (char *)NULL);
			      break;
			    default:
			      gserror(IDS_BADCLI, cmd, MB_ICONEXCLAMATION, SOUND_ERROR);
			}
		    }
		    else {
			FILE *f;
			/* only display file if it exists */
			/* because starting GSview from Desktop menu sets cmd to d:\Desktop */
			if ( (f=fopen(cmd,"rb")) != (FILE *)NULL ) {
			    fclose(f);
		            gsview_displayfile(cmd);
			}
		    }
		    free(cmd);
		}
		else {
		    if (LONGFROMMP(mp1) == IDM_PSTOEPS) {
			if (psfile.name[0] == '\0')	/* need open file */
			    WinSendMsg(hwnd, WM_COMMAND, MPFROMSHORT(IDM_OPEN), MPFROM2SHORT(CMDSRC_OTHER,FALSE));
		    }
		    gsview_command(LONGFROMMP(mp1));
		}
	    break;
	case WM_REALIZEPALETTE:
	    if ((bitmap.depth == 8) && display.hasPalMan && display.hpal_exists) {
		APIRET rc;
		hps = WinGetPS(hwnd);
		GpiSelectPalette(hps, display.hpal);
	        if ( (rc = WinRealizePalette(hwnd, hps, &ulclr)) > 0)
		    WinInvalidateRect(hwnd, NULL, FALSE);
		GpiSelectPalette(hps, (HPAL)NULL);
		WinReleasePS(hps);
	        return 0;
	    }
	    break;	/* use default processing */
	case WM_PAINT:
	    if (!gsdll.device || !bitmap.valid) {
	        hps = WinBeginPaint(hwnd, (ULONG)0, &rect);
		WinFillRect(hps, &rect, SYSCLR_DIALOGBACKGROUND);
		WinEndPaint(hwnd);
		return 0;
	    }

	    /* Refresh the window each time the WM_PAINT message is received */

	    /* get bmp mutex to stop gs.exe changing bitmap while we paint */
	    if (gsdll.device && gsdll.lock_device)
		(*gsdll.lock_device)(gsdll.device, 1);
	    if (gsdll.device && scan_bitmap(&bitmap)) {
		update_scroll_bars(); /* bitmap has changed */
	    }

	    if (!gsdll.device || !bitmap.valid) {
		if (gsdll.lock_device && gsdll.device)
		    (*gsdll.lock_device)(gsdll.device, 0);
	        hps = WinBeginPaint(hwnd, (ULONG)0, &rect);
		WinFillRect(hps, &rect, CLR_BACKGROUND);
		WinEndPaint(hwnd);
		return 0;
	    }

	    request_mutex();
	    hps = WinBeginPaint(hwnd, (HPS)NULL, &rect);
	    if ((bitmap.depth == 8) && display.hasPalMan && display.hpal_exists) {
	        GpiSelectPalette(hps, display.hpal);
                WinRealizePalette(hwnd, hps, &ulclr);
	        paint_bitmap(hps, &rect, nHscrollPos, nVscrollMax - nVscrollPos);
	        GpiSelectPalette(hps, (HPAL)NULL);
	    }
	    else
	        paint_bitmap(hps, &rect, nHscrollPos, nVscrollMax - nVscrollPos);
	    WinEndPaint(hwnd);
	    release_mutex();

	    if (gsdll.lock_device && gsdll.device)
	        (*gsdll.lock_device)(gsdll.device, 0);
	    return 0;
	case WM_MOVE:
	    /* don't interrogate the window location immediately since */
	    /* it causes the Diamond Stealth VL24 with IBM S3 drivers */
	    /* to corrupt the display */
	    DosSleep(50);
	    if (hwnd_frame) {
		SWP swp;
		WinQueryWindowPos(WinQueryWindow(hwnd, QW_PARENT), &swp);
		if (!(swp.fl & SWP_MINIMIZE)) {
		    if (!(swp.fl & SWP_MAXIMIZE)) {
		        option.img_origin.x = swp.x;
		        option.img_origin.y = swp.y;
		    }
		    option.img_max = ((swp.fl & SWP_MAXIMIZE) != 0);
		}
	    }
	    break;
	case WM_SIZE:
		cyClient = SHORT2FROMMP(mp2);
		cxClient = SHORT1FROMMP(mp2);

		cyAdjust = min(bitmap.height, cyClient) - cyClient;
		cyClient += cyAdjust;

		nVscrollMax = max(0, bitmap.height - cyClient);
		nVscrollPos = min(nVscrollPos, nVscrollMax);
		scroll_pos.y = nVscrollMax - nVscrollPos;

		if (!gsdll.device || !bitmap.valid)
			cyClient = cyAdjust = nVscrollMax = nVscrollPos = 0;

		hwndScroll = WinWindowFromID(WinQueryWindow(hwnd, QW_PARENT), FID_VERTSCROLL);
		WinSendMsg(hwndScroll, SBM_SETSCROLLBAR, MPFROMLONG(nVscrollPos), 
			MPFROM2SHORT(0, nVscrollMax));
		if (gsdll.device && bitmap.valid)
		    WinSendMsg(hwndScroll, SBM_SETTHUMBSIZE, MPFROM2SHORT(cyClient, bitmap.height),
			MPFROMLONG(0));
		else
		    WinSendMsg(hwndScroll, SBM_SETTHUMBSIZE, MPFROM2SHORT(1, 1),
			MPFROMLONG(0));

		cxAdjust = min(bitmap.width,  cxClient) - cxClient;
		cxClient += cxAdjust;

		nHscrollMax = max(0, bitmap.width - cxClient);
		nHscrollPos = min(nHscrollPos, nHscrollMax);
		scroll_pos.x = nHscrollPos;

		if (!gsdll.device || !bitmap.valid)
			cxClient = cxAdjust = nHscrollMax = nHscrollPos = 0;

		hwndScroll = WinWindowFromID(WinQueryWindow(hwnd, QW_PARENT), FID_HORZSCROLL);
		WinSendMsg(hwndScroll, SBM_SETSCROLLBAR, MPFROMLONG(nHscrollPos), 
			MPFROM2SHORT(0, nHscrollMax));
		if (gsdll.device && bitmap.valid)
		    WinSendMsg(hwndScroll, SBM_SETTHUMBSIZE, MPFROM2SHORT(cxClient, bitmap.width),
			MPFROMLONG(0));
		else
		    WinSendMsg(hwndScroll, SBM_SETTHUMBSIZE, MPFROM2SHORT(1, 1),
			MPFROMLONG(0));

		if ( option.fit_page && (cxAdjust!=0 || cyAdjust!=0) ) {
		        SWP swp;
			/* don't interrogate the window location immediately since */
			/* it causes the Diamond Stealth VL24 with IBM S3 drivers */
			/* to corrupt the display */
			DosSleep(50);
		        WinQueryWindowPos(WinQueryWindow(hwnd, QW_PARENT), &swp);
		        WinSetWindowPos(WinQueryWindow(hwnd, QW_PARENT), 0, 
			    swp.x, swp.y - cyAdjust,
			    swp.cx + cxAdjust, swp.cy + cyAdjust, SWP_SIZE | SWP_MOVE);
		        cxAdjust = cyAdjust = 0;
		}
		display.offset.x = -cxAdjust/2;
		display.offset.y = -cyAdjust/2;
	    	if (hwnd_frame) {
		    SWP swp;
		    WinQueryWindowPos(WinQueryWindow(hwnd, QW_PARENT), &swp);
		    if (!(swp.fl & SWP_MINIMIZE)) {
			if (!(swp.fl & SWP_MAXIMIZE)) {
		            option.img_size.x = swp.cx;
		            option.img_size.y = swp.cy;
			}
		        option.img_max = ((swp.fl & SWP_MAXIMIZE) != 0);
		    }
		}
		break;
	case WM_VSCROLL:
	    if (!gsdll.device && !bitmap.valid)
		break;
	    switch(SHORT2FROMMP(mp2)) {
		case SB_LINEUP:
			nVscrollInc = -cyClient/16;
			break;
		case SB_LINEDOWN:
			nVscrollInc = cyClient/16;
			break;
		case SB_PAGEUP:
			nVscrollInc = min(-1,-cyClient);
			break;
		case SB_PAGEDOWN:
			nVscrollInc = max(1,cyClient);
			break;
#ifdef NOTUSED
		case SB_SLIDERTRACK:
		    /* only do this for fast redraw modes */
		    /* these are 8bit/pixel and 1bit/pixel */
	            if ( (bitmap.depth == 1) ||
		       ((bitmap.depth == 8) && display.hasPalMan && display.hpal_exists) ) {
			nVscrollInc = SHORT1FROMMP(mp2) - nVscrollPos;
		    }
		    else
			nVscrollInc = 0;
#endif
		case SB_SLIDERPOSITION:
			nVscrollInc = SHORT1FROMMP(mp2) - nVscrollPos;
			break;
		case SB_TOP:
			nVscrollInc = -nVscrollPos;
			break;
		case SB_BOTTOM:
			nVscrollInc = nVscrollMax - nVscrollPos;
			break;
		case SB_FIND:
			nVscrollInc = (short)SHORT1FROMMP(mp2);
			break;
		default:
			nVscrollInc = 0;
	    }
	    if ((nVscrollInc = max(-nVscrollPos, 
		min(nVscrollInc, nVscrollMax - nVscrollPos)))!=0) {
		LONG lComplexity;
	        HPOINTER hptr;
		hwndScroll = WinWindowFromID(WinQueryWindow(hwnd, QW_PARENT), FID_VERTSCROLL);
		nVscrollPos += nVscrollInc;
		scroll_pos.y = nVscrollMax - nVscrollPos;
/* KLUDGE BEGIN */
		/* remove pointer to cope with buggy display drivers */
		hptr = WinQueryPointer(HWND_DESKTOP);
		WinSetPointer(HWND_DESKTOP, 0);
/* KLUDGE END */
		lComplexity = WinScrollWindow(hwnd, 0, nVscrollInc, (PRECTL)NULL, (PRECTL)NULL, 
			(HRGN)NULLHANDLE, (PRECTL)&rect, 0);
/* KLUDGE BEGIN */
		/* restore pointer after we removed it */
		WinSetPointer(HWND_DESKTOP, hptr);
/* KLUDGE END */
		WinSendMsg(hwndScroll, SBM_SETPOS, MPFROMLONG(nVscrollPos), 0);
		if (lComplexity != RGN_RECT) {
		    WinInvalidateRect(hwnd, (PRECTL)NULL, FALSE);
		    WinUpdateWindow(hwnd);
		}
		else {
		    /* redraw exposed area */
		    hps = WinGetPS(hwnd);
	            if ((bitmap.depth == 8) && display.hasPalMan && display.hpal_exists) {
	                GpiSelectPalette(hps, display.hpal);
                        WinRealizePalette(hwnd, hps, &ulclr);
		        paint_bitmap(hps, &rect, nHscrollPos, nVscrollMax - nVscrollPos);
	                GpiSelectPalette(hps, (HPAL)NULL);
	            }
		    else
		        paint_bitmap(hps, &rect, nHscrollPos, nVscrollMax - nVscrollPos);
		    WinValidateRect(hwnd, &rect, FALSE);
		    WinReleasePS(hps);
		}
	    }
	    break;
	case WM_HSCROLL:
	    if (!gsdll.device || !bitmap.valid)
		break;
	    switch(SHORT2FROMMP(mp2)) {
		case SB_LINELEFT:
			nHscrollInc = -cxClient/16;
			break;
		case SB_LINERIGHT:
			nHscrollInc = cyClient/16;
			break;
		case SB_PAGELEFT:
			nHscrollInc = min(-1,-cxClient);
			break;
		case SB_PAGERIGHT:
			nHscrollInc = max(1,cxClient);
			break;
#ifdef NOTUSED
		case SB_SLIDERTRACK:
		    /* only do this for fast redraw modes */
		    /* these are 8bit/pixel and 1bit/pixel */
	            if ( (bitmap.depth == 1) ||
		       ((bitmap.depth == 8) && display.hasPalMan && display.hpal_exists) ) {
			nHscrollInc = SHORT1FROMMP(mp2) - nHscrollPos;
		    }
		    else
			nHscrollInc = 0;
#endif
		case SB_SLIDERPOSITION:
			nHscrollInc = SHORT1FROMMP(mp2) - nHscrollPos;
			break;
		case SB_TOP:
			nHscrollInc = -nHscrollPos;
			break;
		case SB_BOTTOM:
			nHscrollInc = nHscrollMax - nHscrollPos;
			break;
		case SB_FIND:
			nHscrollInc = (short)SHORT1FROMMP(mp2);
			break;
		default:
			nHscrollInc = 0;
	    }
	    if ((nHscrollInc = max(-nHscrollPos, 
		min(nHscrollInc, nHscrollMax - nHscrollPos)))!=0) {
		LONG lComplexity;
		HPOINTER hptr;
		hwndScroll = WinWindowFromID(WinQueryWindow(hwnd, QW_PARENT), FID_HORZSCROLL);
		nHscrollPos += nHscrollInc;
		scroll_pos.x = nHscrollPos;
/* KLUDGE BEGIN */
		/* remove pointer to cope with buggy display drivers */
		hptr = WinQueryPointer(HWND_DESKTOP);
		WinSetPointer(HWND_DESKTOP, 0);
/* KLUDGE END */
		lComplexity = WinScrollWindow(hwnd, -nHscrollInc, 0, (PRECTL)NULL, (PRECTL)NULL, 
			(HRGN)NULLHANDLE, (PRECTL)&rect, 0);
/* KLUDGE BEGIN */
		/* restore pointer after we removed it */
		WinSetPointer(HWND_DESKTOP, hptr);
/* KLUDGE END */
		/* need to send next message BEFORE redrawing, otherwise S3 driver screws up */
		WinSendMsg(hwndScroll, SBM_SETPOS, MPFROMLONG(nHscrollPos), 0);
		if (lComplexity != RGN_RECT) {
		    WinInvalidateRect(hwnd, (PRECTL)NULL, FALSE);
		    WinUpdateWindow(hwnd);
		}
		else {
		    /* redraw exposed area */
		    hps = WinGetPS(hwnd);
	            if ((bitmap.depth == 8) && display.hasPalMan && display.hpal_exists) {
	                GpiSelectPalette(hps, display.hpal);
                        WinRealizePalette(hwnd, hps, &ulclr);
		        paint_bitmap(hps, &rect, nHscrollPos, nVscrollMax - nVscrollPos);
	                GpiSelectPalette(hps, (HPAL)NULL);
	            }
		    else
		        paint_bitmap(hps, &rect, nHscrollPos, nVscrollMax - nVscrollPos);
		    WinValidateRect(hwnd, &rect, FALSE);
		    WinReleasePS(hps);
		}
	    }
	    break;
	case WM_CHAR:	/* process keystrokes here */
	    /* Process only key presses, not key releases */
	    if (SHORT1FROMMP(mp1) & KC_KEYUP)
	        break;
	    if (SHORT1FROMMP(mp1) & KC_VIRTUALKEY) {
		USHORT vkey = SHORT2FROMMP(mp2);
		switch(vkey) {
		    case VK_HOME:
		    	WinSendMsg(hwnd, WM_VSCROLL, MPFROMLONG(0), MPFROM2SHORT(0, SB_TOP));
		    	break;
		    case VK_END:
		    	WinSendMsg(hwnd, WM_VSCROLL, MPFROMLONG(0), MPFROM2SHORT(0, SB_BOTTOM));
		    	break;
		    case VK_UP:
		    	WinSendMsg(hwnd, WM_VSCROLL, MPFROMLONG(0), MPFROM2SHORT(0, SB_LINEUP));
		    	break;
		    case VK_DOWN:
		    	WinSendMsg(hwnd, WM_VSCROLL, MPFROMLONG(0), MPFROM2SHORT(0, SB_LINEDOWN));
		    	break;
		    case VK_PAGEUP:
	    		if (SHORT1FROMMP(mp1) & KC_CTRL)
		    	    WinSendMsg(hwnd, WM_HSCROLL, MPFROMLONG(0), MPFROM2SHORT(0, SB_PAGELEFT));
		 	else
		    	    WinSendMsg(hwnd, WM_VSCROLL, MPFROMLONG(0), MPFROM2SHORT(0, SB_PAGEUP));
		    	break;
		    case VK_PAGEDOWN:
	    		if (SHORT1FROMMP(mp1) & KC_CTRL)
		    	    WinSendMsg(hwnd, WM_HSCROLL, MPFROMLONG(0), MPFROM2SHORT(0, SB_PAGERIGHT));
		 	else
		    	    WinSendMsg(hwnd, WM_VSCROLL, MPFROMLONG(0), MPFROM2SHORT(0, SB_PAGEDOWN));
		    	break;
		    case VK_LEFT:
	    		if (SHORT1FROMMP(mp1) & KC_CTRL)
		    	    WinSendMsg(hwnd, WM_HSCROLL, MPFROMLONG(0), MPFROM2SHORT(0, SB_PAGELEFT));
		  	else
		    	    WinSendMsg(hwnd, WM_HSCROLL, MPFROMLONG(0), MPFROM2SHORT(0, SB_LINELEFT));
		    	break;
		    case VK_RIGHT:
	    		if (SHORT1FROMMP(mp1) & KC_CTRL)
		    	    WinSendMsg(hwnd, WM_HSCROLL, MPFROMLONG(0), MPFROM2SHORT(0, SB_PAGERIGHT));
		    	else
		    	    WinSendMsg(hwnd, WM_HSCROLL, MPFROMLONG(0), MPFROM2SHORT(0, SB_LINERIGHT));
		    	break;
		}
	    }
	    break;
	case WM_BUTTON1DOWN:
		if (hwnd_modeless) 
		    WinPostMsg(hwnd_modeless, WM_COMMAND, MPFROMSHORT(BB_CLICK), MPFROMLONG(0));
		else {
		    int iword ;
		    float x, y;
		    if (get_cursorpos(&x, &y)) {
			HPS hps = WinGetPS(hwnd);
			if ( (iword = word_find((int)x, (int)y)) >= 0 ) {
			    /* remove any current selection */
			    highlight_words(hps, text_mark_first, text_mark_last);
			    /* mark new selection */
			    text_mark_first = text_mark_last = iword;
			    text_marking = TRUE;
			    highlight_words(hps, text_mark_first, text_mark_last);
			    WinSetCapture(HWND_DESKTOP, hwnd);
			}
			else {
			    /* remove selection */
			    highlight_words(hps, text_mark_first, text_mark_last);
			    text_mark_first = text_mark_last = -1;
			    WinSetCapture(HWND_DESKTOP, NULLHANDLE);
			}
			WinReleasePS(hps);
		    }
		}
		break;
	case WM_BUTTON2DOWN:
		{ float x, y;
		    request_mutex();
		    if (get_cursorpos(&x, &y)) {
		        zoom = !zoom;
		        display.zoom_xoffset = x;
		        display.zoom_yoffset = y;
			x = (scroll_pos.x+cxClient/2)*72.0/option.xdpi;
			y = (scroll_pos.y+cyClient/2)*72.0/option.ydpi;
			transform_point(&x, &y);
			x *= option.xdpi/72.0;
			y *= option.ydpi/72.0;
			display.zoom_xoffset -= (int)(x*72.0/option.zoom_xdpi);
			display.zoom_yoffset -= (int)(y*72.0/option.zoom_ydpi);
		    }
		    else {
		        zoom = FALSE;
		    }
		    release_mutex();
		    WinPostMsg(hwnd_bmp, WM_COMMAND, (MPARAM)IDM_ZOOM, MPFROMLONG(0));
		}
		break;
	case WM_MOUSEMOVE:
		/* track cursor and display coordinates */
		request_mutex();
		hps = WinGetPS(hwnd_status);
		WinFillRect(hps, &info_coord, SYSCLR_BUTTONMIDDLE);
		cursorpos_paint(hps);
		WinReleasePS(hps);
		release_mutex();
		/* extend marked text */
		if (text_marking) {
		    int iword ;
		    float x, y;
		    RECTL rect;
		    POINTL pt;
		    do { 
			if (!WinQueryPointerPos(HWND_DESKTOP, &pt))
			    return FALSE;

			WinMapWindowPoints(HWND_DESKTOP, hwnd_bmp, &pt, 1);
			WinQueryWindowRect(hwnd_bmp, &rect);
			if (pt.x > rect.xRight)
		    	    WinSendMsg(hwnd, WM_HSCROLL, MPFROMLONG(0), MPFROM2SHORT(0, SB_LINERIGHT));
			if (pt.x < rect.xLeft)
		    	    WinSendMsg(hwnd, WM_HSCROLL, MPFROMLONG(0), MPFROM2SHORT(0, SB_LINELEFT));
			if (pt.y < rect.yBottom)
		    	    WinSendMsg(hwnd, WM_VSCROLL, MPFROMLONG(0), MPFROM2SHORT(0, SB_LINEDOWN));
			if (pt.y > rect.yTop)
		    	    WinSendMsg(hwnd, WM_VSCROLL, MPFROMLONG(0), MPFROM2SHORT(0, SB_LINEUP));
			DosSleep(100);
		    } while ( (pt.x > rect.xRight) || (pt.x < rect.xLeft)
			   || (pt.y > rect.yTop)   || (pt.y < rect.yBottom) );
		    if (get_cursorpos(&x, &y)) {
			if ( (iword = word_find((int)x, (int)y)) >= 0 ) {
			    if (iword != text_mark_last) {
				HPS hps = WinGetPS(hwnd);
				int first, last;
				if ((text_mark_last-text_mark_first >= 0) != (iword-text_mark_first >= 0)) {
				    /* changing direction */
				    /* clear everything */
				    highlight_words(hps, text_mark_first, text_mark_last);
				    /* reinstate first word */
				    text_mark_last = text_mark_first;
				    highlight_words(hps, text_mark_first, text_mark_last);
				}
				if (iword != text_mark_last) {
				  if (iword >= text_mark_first) {
				    if (iword > text_mark_last)
					first=text_mark_last+1, last=iword;
				    else
					first=iword+1, last=text_mark_last;
				  }
				  else {
				    if (iword > text_mark_last)
					first=text_mark_last, last=iword-1;
				    else
					first=iword, last=text_mark_last-1;
				  }
				  highlight_words(hps, first, last);
				  text_mark_last = iword;
				}
				WinReleasePS(hps);
			    }
			}
		    }
		}
		/* update cursor */
	        if (szWait[0] != '\0')
    		    WinSetPointer(HWND_DESKTOP, WinQuerySysPointer(HWND_DESKTOP, SPTR_WAIT, FALSE));
		else {
		    if (!gsdll.device || !bitmap.valid)
    		        WinSetPointer(HWND_DESKTOP, WinQuerySysPointer(HWND_DESKTOP, SPTR_ARROW, FALSE));
		    else
    		        WinSetPointer(HWND_DESKTOP, hptr_crosshair);
		}
		return (MRESULT)TRUE;	/* cursor has been changed */
	    case WM_BUTTON1UP:
		WinSetCapture(HWND_DESKTOP, NULLHANDLE);
		text_marking = FALSE;
		break;
	    case WM_CLOSE:
		if (!query_close()) {
		     return (MRESULT)0;
		}
		break;
	    case WM_DESTROY:
		break;
	    case DM_DRAGOVER:
		return DragOver((PDRAGINFO)mp1);
	    case DM_DROP:
		return Drop((PDRAGINFO)mp1);
  }
  /* All messages not handled by the ClientWndProc must be passed */
  /* along to the Presentation Manager for default processing */
  return WinDefWindowProc(hwnd, mess, mp1, mp2);
}

/* map from a coordinate in points, to a coordinate in pixels */
/* This is the opposite of the transform part of get_cursorpos */
/* Used when showing bbox */
void
map_pt_to_pixel(float *x, float *y)
{
	if (zoom) {
	    /* WARNING - this doesn't cope with EPS Clip */
	    *x = (*x - display.zoom_xoffset) * option.zoom_xdpi / 72.0;
	    *y = (*y - display.zoom_yoffset) * option.zoom_ydpi / 72.0;
	    *x = (*x * 72.0 / option.xdpi);
	    *y = (*y * 72.0 / option.ydpi);
	    itransform_point(x, y);
	    *x = (*x * option.xdpi / 72.0) - scroll_pos.x + display.offset.x;
	    *y = (*y * option.ydpi / 72.0) - scroll_pos.y + display.offset.y;
	}
	else {
	    *x = *x - (display.epsf_clipped ? psfile.doc->boundingbox[LLX] : 0);
	    *y = *y - (display.epsf_clipped ? psfile.doc->boundingbox[LLY] : 0);
	    itransform_point(x, y);
	    *x = *x * option.xdpi/72.0
		  - scroll_pos.x + display.offset.x;
	    *y = *y * option.ydpi/72.0
		  - scroll_pos.y + display.offset.y;
	}
}


/* return coordinates in pts from origin of page */
/* For Portrait, this is the bottom left */
/* For rotated pages, this is NOT the bottom left */
BOOL
get_cursorpos(float *x, float *y)
{
RECTL rect;
POINTL pt;
	if (!gsdll.device || !bitmap.valid)
	    return FALSE;
	if (!WinQueryPointerPos(HWND_DESKTOP, &pt))
	    return FALSE;
	WinMapWindowPoints(HWND_DESKTOP, hwnd_bmp, &pt, 1);
	WinQueryWindowRect(hwnd_bmp, &rect);
	if (!WinPtInRect(hab, &rect, &pt))
	    return FALSE;
	*x = scroll_pos.x+pt.x-display.offset.x;
	*y = scroll_pos.y+pt.y-display.offset.y;
	transform_cursorpos(x, y);
	return TRUE;
}

void
cursorpos_paint(HPS hps)
{
float x, y;
POINTL pt;
char buf[64];
	request_mutex();
	if (get_cursorpos(&x, &y)) {
	    switch(option.unit) {
	       case IDM_UNITPT:   
	          sprintf(buf,"%.0f, %.0fpt", x, y);
		  break;
	       case IDM_UNITMM:   
	          sprintf(buf,"%.0f, %.0fmm", x/72*25.4, y/72*25.4);
		  break;
	       case IDM_UNITINCH:   
	          sprintf(buf,"%.1f, %.1fin", x/72, y/72);
		  break;
	    }
	    GpiSetTextAlignment(hps, TA_RIGHT, TA_NORMAL_VERT);
	    pt.x = info_coord.xRight - 1;
	    pt.y = info_page.y;
            GpiCharStringAt(hps, &pt, strlen(buf), buf);
	}
	release_mutex();
}

void
info_paint(HWND hwnd)
{
HPS ps;
RECTL rect;
POINTL pt;
int i;
char *p;
char buf[MAXSTR];
char fmt[MAXSTR];
	request_mutex();
        ps = WinBeginPaint(hwnd, (ULONG)0, &rect);
	/* draw background */
	WinFillRect(ps, &rect, SYSCLR_BUTTONMIDDLE);
	GpiSetColor(ps, SYSCLR_BUTTONDARK);
	pt.x = 0; pt.y = 0;
	GpiMove(ps, &pt);
	pt.x = statusbar.x - 1;
	GpiLine(ps, &pt);
	pt.y = statusbar.y - 1;
	GpiLine(ps, &pt);
	GpiSetColor(ps, SYSCLR_BUTTONLIGHT);
	pt.x = 0;
	GpiLine(ps, &pt);
	pt.y = 0;
	GpiLine(ps, &pt);
	GpiSetColor(ps, SYSCLR_BUTTONDEFAULT);

	/* write file information */
	if (psfile.name[0] != '\0') {
	    i = load_string(IDS_FILE, buf, sizeof(buf));
	    if ( (p = strrchr(psfile.name, '\\')) == (char *)NULL )
		p = psfile.name;
	    else
		p++;
	    strcat(buf, p);
	    GpiCharStringAt(ps, &info_file, strlen(buf), buf);
	    if (szWait[0] != '\0') {
		sprintf(buf, szWait, percent_done);
	        GpiCharStringAt(ps, &info_page, strlen(buf), buf);
	    }
	    else {
	      if (psfile.doc!=(PSDOC *)NULL) {
		int n = map_page(psfile.pagenum - 1);
		i = load_string(IDS_PAGEINFO, fmt, sizeof(fmt));
		if (psfile.doc->pages)
		    sprintf(buf, fmt, psfile.doc->pages[n].label ? psfile.doc->pages[n].label : " ",psfile.pagenum,  psfile.doc->numpages);
		else
		    sprintf(buf, fmt, " " ,psfile.pagenum,  psfile.doc->numpages);
		if (zoom)
		    strcat(buf, "  Zoomed");
	        GpiCharStringAt(ps, &info_page, strlen(buf), buf);
	      }
	      else {
		if (gsdll.state == IDLE)
		    i = load_string(IDS_NOMORE, buf, sizeof(buf));
		else {
		    i = load_string(IDS_PAGE, buf, sizeof(buf));
		    sprintf(buf+i, "%d", psfile.pagenum);
		}
	        GpiCharStringAt(ps, &info_page, strlen(buf), buf);
	      }
	      /* show coordinate */
	      cursorpos_paint(ps);
	    }
	}
	else {
	    i = load_string(IDS_NOFILE, buf, sizeof(buf));
	    GpiCharStringAt(ps, &info_file, strlen(buf), buf);
	    if (szWait[0] != '\0') {
		sprintf(buf, szWait, percent_done);
	        GpiCharStringAt(ps, &info_page, strlen(buf), buf);
	    }
	}
	WinEndPaint(hwnd);
	release_mutex();
}


/* Status Bar Window Proc */
MRESULT EXPENTRY StatusWndProc(HWND hwnd, ULONG mess, 
			MPARAM mp1, MPARAM mp2)
{
  HPS ps;
  RECTL rect;
  POINTL pt;
  switch (mess) {
    case WM_CREATE:
	{
	FONTMETRICS fm;
	POINTL char_size;
	ps = WinGetPS(hwnd);
	GpiQueryFontMetrics(ps, sizeof(FONTMETRICS), &fm);
	char_size.y = fm.lMaxAscender + fm.lMaxDescender + fm.lExternalLeading + 2;
	char_size.x = fm.lAveCharWidth;
	statusbar.y = char_size.y + 2;
	info_file.x = 2;
	info_file.y = fm.lMaxDescender + 3;
	info_coord.xLeft = 23 * char_size.x;
	info_coord.xRight = 38 * char_size.x;
	info_coord.yTop = char_size.y - 1;
	info_coord.yBottom = 1;
	info_page.x = 40 * char_size.x + 2;
	info_page.y = fm.lMaxDescender + 3;
/*
	WinFillRect(ps, SYSCLR_BUTTONMIDDLE);
*/
	WinReleasePS(ps);
	}
	break;
    case WM_PAINT:
	info_paint(hwnd);
	return 0;
  }
  return WinDefWindowProc(hwnd, mess, mp1, mp2);
}


void
shade_button(HPS hps, struct button *bp, BOOL down)
{
POINTL pt;
	/* draw button shading */
	if (down)
	    GpiSetColor(hps, SYSCLR_BUTTONLIGHT);
	else
	    GpiSetColor(hps, SYSCLR_BUTTONDARK);
	pt.x = bp->rect.xLeft; 
	pt.y = bp->rect.yBottom;
	GpiMove(hps, &pt);
	pt.x = bp->rect.xRight - 1;
	GpiLine(hps, &pt);
	pt.y = bp->rect.yTop - 1;
	GpiLine(hps, &pt);
	if (down)
	    GpiSetColor(hps, SYSCLR_BUTTONDARK);
	else
	    GpiSetColor(hps, SYSCLR_BUTTONLIGHT);
	pt.x = bp->rect.xLeft;
	GpiLine(hps, &pt);
	pt.y = bp->rect.yBottom;
	GpiLine(hps, &pt);
	GpiSetColor(hps, SYSCLR_BUTTONDEFAULT);
}

void
size_buttons(PPOINTL ppt)
{
  struct button *bp;
  int i = 0;
  if (option.button_show) {
    for (bp = buttonhead; bp != (struct button *)NULL; bp = bp->next) {
	bp->rect.xLeft = i * button_shift.x;
	bp->rect.yBottom = ppt->y - button_size.y - i * button_shift.y;
	bp->rect.xRight = bp->rect.xLeft + button_size.x;
	bp->rect.yTop = bp->rect.yBottom + button_size.y;
        i++;
    }
  }
}

/* Button Bar Window Proc */
MRESULT EXPENTRY ButtonWndProc(HWND hwnd, ULONG mess, 
			MPARAM mp1, MPARAM mp2)
{
  HPS hps;
  RECTL rect;
  POINTL pt;
  struct button *bp;
  switch (mess) {
    case WM_SIZE:
	/* reposition buttons */
	pt.x = SHORT1FROMMP(mp2);
	pt.y = SHORT2FROMMP(mp2);
        size_buttons(&pt);
	WinInvalidateRect(hwnd, (PRECTL)NULL, FALSE);
	break;
    case WM_BUTTON1DOWN:
	pt.x = SHORT1FROMMP(mp1);
	pt.y = SHORT2FROMMP(mp1);
	for (bp = buttonhead; bp != (struct button *)NULL; bp = bp->next) {
	    if (WinPtInRect(hab, &(bp->rect), &pt)) {
		button_current = bp;
		button_down = TRUE;
		hps = WinGetPS(hwnd);
		shade_button(hps, bp, TRUE);
		WinReleasePS(hps);
	        WinSetCapture(HWND_DESKTOP, hwnd);
		break;
	    }
	}
	break;
    case WM_BUTTON1UP:
	pt.x = SHORT1FROMMP(mp1);
	pt.y = SHORT2FROMMP(mp1);
	bp = button_current;
	if (bp != (struct button *)NULL) {
	    hps = WinGetPS(hwnd);
	    shade_button(hps, bp, FALSE);
	    WinReleasePS(hps);
	    if (WinPtInRect(hab, &(bp->rect), &pt)) {
	    	WinPostMsg(hwnd_bmp, WM_COMMAND, MPFROMSHORT(bp->id), MPFROMLONG(0));
	    }
	}
	WinSetCapture(HWND_DESKTOP, NULLHANDLE);
	WinInvalidateRect(hwnd_status, NULL, FALSE);
	WinUpdateWindow(hwnd_status);
	button_info = NULL;
	button_current = NULL;
	button_down = FALSE;
	break;
    case WM_MOUSEMOVE:
	pt.x = SHORT1FROMMP(mp1);
	pt.y = SHORT2FROMMP(mp1);
	bp = button_current;
	if (bp != (struct button *)NULL) {
	    if (button_down && !WinPtInRect(hab, &(bp->rect), &pt)) {
		button_down = FALSE;
		hps = WinGetPS(hwnd);
		shade_button(hps, bp, FALSE);
		WinReleasePS(hps);
	    }
	    else if (!button_down && WinPtInRect(hab, &(bp->rect), &pt)) {
		button_down = TRUE;
		hps = WinGetPS(hwnd);
		shade_button(hps, bp, TRUE);
		WinReleasePS(hps);
	    }
	}
	else {
	    /* track cursor and display button function */
	    char buf[64];
	    /* find button */
	    for (bp = buttonhead; bp != (struct button *)NULL; bp = bp->next) {
	      if (WinPtInRect(hab, &(bp->rect), &pt)) {
		break;
	      }
	    }
	    if (bp != button_info) {
	      button_info = bp;
	      if (bp != (struct button *)NULL) {
		if (WinQueryCapture(HWND_DESKTOP) != hwnd)
	            WinSetCapture(HWND_DESKTOP, hwnd); /* track all mouse movements */
	        /* erase page area of status bar */
	        hps = WinGetPS(hwnd_status);
	        rect.xLeft = info_file.x;
	        rect.xRight = statusbar.x - 1;
	        rect.yBottom = info_coord.yBottom;
	        rect.yTop = info_coord.yTop;
	        WinFillRect(hps, &rect, SYSCLR_BUTTONMIDDLE);
		load_string(bp->id, buf, sizeof(buf));
                GpiCharStringAt(hps, &info_file, strlen(buf), buf);
	        WinReleasePS(hps);
	      }
	      else {
	        WinSetCapture(HWND_DESKTOP, NULLHANDLE);  /* release capture */
		/* put statusbar back to normal */
	        WinInvalidateRect(hwnd_status, NULL, FALSE);
  		WinUpdateWindow(hwnd_status);
	      }
	    }
	}
	break;
    case WM_PAINT:
	hps = WinBeginPaint(hwnd, (ULONG)0, &rect);
	/* draw background */
	WinFillRect(hps, &rect, SYSCLR_BUTTONMIDDLE);
	GpiSetColor(hps, SYSCLR_BUTTONDARK);
	pt.x = buttonbar.x - 1; pt.y = 0;
	GpiMove(hps, &pt);
	pt.y = buttonbar.y - 1;
	GpiLine(hps, &pt);
	GpiSetColor(hps, SYSCLR_BUTTONLIGHT);
	pt.x = 0;
	GpiLine(hps, &pt);
	pt.y = 0;
	GpiLine(hps, &pt);
	GpiSetColor(hps, SYSCLR_BUTTONDEFAULT);
	for (bp = buttonhead; bp != (struct button *)NULL; bp = bp->next) {
	    if (bp->hbitmap) {
		/* draw bitmap */
		pt.x = bp->rect.xLeft;
		pt.y = bp->rect.yBottom;
		WinDrawBitmap(hps, bp->hbitmap, NULL, &pt, 0, 0, DBM_NORMAL);
	    }
	    else {
		/* draw button text */
		pt.x = (bp->rect.xLeft + bp->rect.xRight)/2;
		pt.y = (bp->rect.yBottom + bp->rect.yTop)/2;
	        GpiSetTextAlignment(hps, TA_CENTER, TA_HALF);
                GpiCharStringAt(hps, &pt, strlen(bp->str), bp->str);
	    }
	    shade_button(hps, bp, FALSE);
	}
	WinEndPaint(hwnd);
	return 0;
  }
  return WinDefWindowProc(hwnd, mess, mp1, mp2);
}

/* subclassed Frame Window Procedure */
MRESULT EXPENTRY FrameWndProc(HWND hwnd, ULONG mess, 
			MPARAM mp1, MPARAM mp2)
{
MRESULT mr;
int sCount;
PSWP pswpNew, pswpClient, pswp;

  switch(mess) {
    case WM_HELP:
	/* HM_DISPLAY_HELP is unreliable for HM_PANELNAME so we lookup */
	/* the id ourselves and use HM_RESOURCEID */
	{
	HELPIDX *hidx;
	  for (hidx = helpidx; hidx->id != 0; hidx++) {
	    if (strcmp(hidx->name, szHelpTopic)==0) {
		WinSendMsg(hwnd_help, HM_DISPLAY_HELP, MPFROMSHORT(hidx->id), MPFROMSHORT(HM_RESOURCEID));
		continue;
	    }
	  }
	}
	return 0;
    case WM_CALCFRAMERECT:
	mr =  (*OldFrameWndProc)(hwnd, mess, mp1, mp2);
	/* calculate position of client rectangle */
	if (mr && mp2) {
	    ((PRECTL)mp1)->yTop -= statusbar.y;
	    if (option.button_show)
	        ((PRECTL)mp1)->xLeft += buttonbar.x;
	}
	return mr;
    case WM_FORMATFRAME:
	/* reformat frame to make room for status bar */
	mr =  (*OldFrameWndProc)(hwnd, mess, mp1, mp2);
	sCount = (int)mr;
	pswp = (PSWP)mp1;
	pswpClient = (PSWP)mp1 + sCount - 1;
	pswpNew = pswpClient + 1;
	*pswpNew = *pswpClient;
	pswpNew->hwnd = WinWindowFromID(hwnd, ID_STATUSBAR);
	pswpNew->cy = statusbar.y;
	pswpClient->cy -= pswpNew->cy;
	pswpNew->y = pswpClient->y + pswpClient->cy;
	pswp[FID_VERTSCROLL - FID_SYSMENU].cy -= pswpNew->cy;
	pswpNew->x = pswp[FID_MENU - FID_SYSMENU].x;
	pswpNew->cx = pswp[FID_MENU - FID_SYSMENU].cx;
	statusbar.x = pswpNew->cx;
	sCount++;
	/* reformat frame to make room for button bar */
	pswpNew++;
	*pswpNew = *pswpClient;
	pswpNew->hwnd = WinWindowFromID(hwnd, ID_BUTTONBAR);
	pswpNew->x = pswpClient->x;
        pswpNew->cx = buttonbar.x;
	pswpNew->y = pswpClient->y;
	pswpNew->cy = pswpClient->cy;
	buttonbar.y = pswpNew->cy;
	pswpNew->fl &= ~(SWP_SHOW | SWP_HIDE);
	sCount++;
	if (option.button_show) {
	    pswpClient->x += buttonbar.x;
	    pswpClient->cx -= buttonbar.x;
	    pswpNew->fl |= SWP_SHOW;
	    /* following is a kludge because ButtonWndProc is receiving WM_SIZE
	       messages in reverse order to WM_FORMATFRAME here! */
	    WinPostMsg(hwnd_button, WM_SIZE, 
		MPFROM2SHORT(pswpNew->cx, pswpNew->cy), 
		MPFROM2SHORT(pswpNew->cx, pswpNew->cy));
	}
	else {
	    pswpNew->fl |= SWP_HIDE;
	}
	return (MRESULT)sCount;
    case WM_QUERYFRAMECTLCOUNT:
	mr =  (*OldFrameWndProc)(hwnd, mess, mp1, mp2);
    	sCount = (int)mr;
    	sCount+=2;
	return (MRESULT)sCount;
  }
  return (*OldFrameWndProc)(hwnd, mess, mp1, mp2);
}

MRESULT
DragOver(PDRAGINFO pDragInfo)
{
int cItems, i;
PDRAGITEM pditem;
USHORT usDrop, usDefaultOp;
char dtype[MAXSTR];
char drmf[MAXSTR];
	if (!DrgAccessDraginfo(pDragInfo))
	    return MRFROM2SHORT(DOR_NODROPOP, 0);
	switch(pDragInfo->usOperation) {
	    case DO_DEFAULT:
	    case DO_COPY:
		usDrop = DOR_DROP;
		usDefaultOp = DO_COPY;
		break;
	    default:
	    case DO_UNKNOWN:
	    case DO_MOVE:
		DrgFreeDraginfo(pDragInfo);
		return MRFROM2SHORT(DOR_NODROPOP, 0);
	}
	
	cItems = DrgQueryDragitemCount(pDragInfo);
	for (i=0; i<cItems; i++) {
	    pditem = DrgQueryDragitemPtr(pDragInfo, i);
	    DrgQueryStrName(pditem->hstrType, sizeof(dtype)-1, dtype);
	    DrgQueryStrName(pditem->hstrRMF, sizeof(drmf)-1, drmf);
	    if (pditem->fsSupportedOps & DO_COPYABLE)  {
	        if ( DrgVerifyRMF(pditem, "DRM_OS2FILE", NULL) )	/* changed from "DRF_TEXT" to NULL */
	            usDrop = DOR_DROP;
		else
	            usDrop = DOR_NEVERDROP;
	    }
	    else {
		usDrop = DOR_NODROPOP;
	    }
	}
	DrgFreeDraginfo(pDragInfo);
	return MRFROM2SHORT(usDrop, usDefaultOp);
}

MRESULT
Drop(PDRAGINFO pDragInfo)
{
int cItems, i;
DRAGITEM dragitem;
USHORT usDrop, usDefaultOp;
char dirname[MAXSTR];
char filename[MAXSTR];
char *dragname;
int length;
HPS hps;
POINTL pt;
	if (!DrgAccessDraginfo(pDragInfo))
	    return MRFROM2SHORT(DOR_NODROPOP, 0);
	cItems = DrgQueryDragitemCount(pDragInfo);
	for (i=0; i<cItems; i++) {
	    DrgQueryDragitem(pDragInfo, sizeof(dragitem), &dragitem, i);
	    length = DrgQueryStrName(dragitem.hstrContainerName, sizeof(dirname), dirname);
	    length += DrgQueryStrName(dragitem.hstrSourceName, sizeof(filename), filename);
	    dragname = malloc(length+1);
	    strcpy(dragname, dirname);
	    strcat(dragname, filename);
	    WinPostMsg(hwnd_bmp, WM_COMMAND, (MPARAM)IDM_DROP, MPFROMP(dragname));	/* mp2 is not strictly correct */
	}
	DrgDeleteDraginfoStrHandles(pDragInfo);
	DrgFreeDraginfo(pDragInfo);
	return (MRESULT)0;
}

/* if found word is not visible, scroll window to make it visible */
void
scroll_to_find(void)
{
    RECTL rect, rect_client;
    float x, y;

    request_mutex();
    /* first translate found box to window coordinates */
    x = psfile.text_bbox.llx;
    y = psfile.text_bbox.lly;
    map_pt_to_pixel(&x, &y);
    rect.xLeft   = (int)x;
    rect.yBottom = (int)y;
    x = psfile.text_bbox.urx;
    y = psfile.text_bbox.ury;
    map_pt_to_pixel(&x, &y);
    rect.xRight  = (int)x;
    rect.yTop    = (int)y;

    WinQueryWindowRect(hwnd_bmp, &rect_client);

    /* scroll to bring the bottom left to the centre of the window */
    if ((rect.xLeft < rect_client.xLeft) || (rect.xRight > rect_client.xRight))
	WinPostMsg(hwnd_bmp, WM_HSCROLL, 0, MPFROM2SHORT(((rect_client.xRight-rect_client.xLeft)/2) - rect.xLeft, SB_FIND));

    if ((rect.yTop > rect_client.yTop) || (rect.yBottom < rect_client.yBottom))
	WinPostMsg(hwnd_bmp, WM_VSCROLL, 0, MPFROM2SHORT(((rect_client.yTop-rect_client.yBottom)/2) - ((rect.yBottom+rect.yTop)/2) , SB_FIND));
    release_mutex();
}

/* highlight words from first to last inclusive */
/* first may be > last */
/* word = -1 means nothing to mark */
void
highlight_words(HPS ps, int first, int last)
{
    RECTL rect;
    float x, y;
    TEXTINDEX *text;
    int i;
    if ((first == -1) || (last == -1))
	return;


    if ((first > text_index_count) || (last > text_index_count)) {
	gs_addmess("\nhighlight_words called with invalid arguments\n");
	return;
    }
    if (first > last) {
        int temp = first;
	first = last;
	last = temp;
    }

    for (i = first; i<=last; i++) {
	text = &text_index[i];
	/* highlight found word */
	/* map bounding box to device coordinates */
	x = text->bbox.llx;
	y = text->bbox.lly;
	map_pt_to_pixel(&x, &y);
	rect.xLeft   = (int)x;
	rect.yBottom = (int)y;
	x = text->bbox.urx;
	y = text->bbox.ury;
	map_pt_to_pixel(&x, &y);
	rect.xRight  = (int)x;
	rect.yTop    = (int)y;
	if (rect.xRight < rect.xLeft) {
	    int temp = rect.xRight;
	    rect.xRight = rect.xLeft;
	    rect.xLeft = temp;
	}
	if (rect.yTop < rect.yBottom) {
	    int temp = rect.yTop;
	    rect.yTop = rect.yBottom;
	    rect.yBottom = temp;
	}


	/* invert text */
	WinInvertRect(ps, &rect);
    }
}
