/* Copyright (C) 1993-1996, Russell Lang.  All rights reserved.
  
  This file is part of GSview.
  
  This program is distributed with NO WARRANTY OF ANY KIND.  No author
  or distributor accepts any responsibility for the consequences of using it,
  or for whether it serves any particular purpose or works at all, unless he
  or she says so in writing.  Refer to the GSview Free Public Licence 
  (the "Licence") for full details.
  
  Every copy of GSview must include a copy of the Licence, normally in a 
  plain ASCII text file named LICENCE.  The Licence grants you the right 
  to copy, modify and redistribute GSview, but only under certain conditions 
  described in the Licence.  Among other things, the Licence requires that 
  the copyright notice and this notice be preserved on all copies.
*/

/* gvcdisp.c */
/* Display GSview routines common to Windows and PM */
#ifdef _Windows
#include "gvwin.h"
#else
#include "gvpm.h"
#endif

FILE *debug_file;

void
gs_puts(char *str, FILE *f)
{
    if (str != NULL) {
	fputs(str, f);
	if (debug_file != (FILE *)NULL)
	   fputs(str, debug_file);
    }
}

void 
gs_copy(FILE *from, FILE *to, long begin, long end)
{
	pscopyuntil(from, to, begin, end, NULL);
	if (debug_file != (FILE *)NULL)
	   pscopyuntil(from, debug_file, begin, end, NULL);
}

/* transform cursor position from device coordinates to points */
/* taking into account rotated pages */
void
transform_cursorpos(float *x, float *y)
{
	  if (zoom) {
            /* first figure out number of pixels to zoom origin point */
	    *x = *x * 72.0/option.xdpi;
	    *y = *y * 72.0/option.ydpi;
	    transform_point(x,y);
	    *x = *x * option.xdpi/72;
	    *y = *y * option.ydpi/72;
	    /* now convert to pts and offset it */
	    *x = *x * 72/option.zoom_xdpi + display.zoom_xoffset;
	    *y = *y * 72/option.zoom_ydpi + display.zoom_yoffset;
	  }
	  else {
	    *x = *x * 72.0/option.xdpi;
	    *y = *y * 72.0/option.ydpi;
	    transform_point(x,y);
	    *x = *x + (display.epsf_clipped ? psfile.doc->boundingbox[LLX] : 0);
	    *y = *y + (display.epsf_clipped ? psfile.doc->boundingbox[LLY] : 0);
	  }
}


/* transform point from coordinates relative to bottom left
 * corner of paper to bottom left corner of rotated coordinate
 * system
 */
void
transform_point(float *x, float *y)
{
float oldx, oldy;
int real_orientation;
int width, height;
	oldx = *x;
	oldy = *y;
	width  = (unsigned int)(display.width  * 72.0 / option.xdpi);
	height = (unsigned int)(display.height * 72.0 / option.ydpi);
	real_orientation = option.orientation;
	if (option.swap_landscape) {
	    if (option.orientation == IDM_LANDSCAPE)
		real_orientation = IDM_SEASCAPE;
	    else if (option.orientation == IDM_SEASCAPE)
		real_orientation = IDM_LANDSCAPE;
	}

	real_orientation -= IDM_PORTRAIT;
	if (psfile.ispdf)
	    real_orientation += (pdf_orientation() - IDM_PORTRAIT);
        real_orientation &= 3;  /* modulus 4 */
	real_orientation += IDM_PORTRAIT;

	switch (real_orientation) {
	    case IDM_PORTRAIT:
		break;
	    case IDM_LANDSCAPE:
	    	*x = width - oldy;	/* display.width = bitmap.height */
	    	*y = oldx;
	    	break;
	    case IDM_UPSIDEDOWN:
	    	*x = width - oldx;
	    	*y = height - oldy;
		break;
	    case IDM_SEASCAPE:
	    	*x = oldy;
	    	*y = height - oldx;
	    	break;
	}
	return;
}

/* inverse transform point from coordinates relative 
 * to bottom left corner of rotated coordinate system
 * to bottom left corner of paper 
 */
void
itransform_point(float *x, float *y)
{
float oldx, oldy;
int real_orientation;
int width, height;
	oldx = *x;
	oldy = *y;
	width  = (unsigned int)(display.width  * 72.0 / option.xdpi);
	height = (unsigned int)(display.height * 72.0 / option.ydpi);
	real_orientation = option.orientation;
	if (option.swap_landscape) {
	    if (option.orientation == IDM_LANDSCAPE)
		real_orientation = IDM_SEASCAPE;
	    else if (option.orientation == IDM_SEASCAPE)
		real_orientation = IDM_LANDSCAPE;
	}
	switch (real_orientation) {
	    case IDM_PORTRAIT:
		break;
	    case IDM_LANDSCAPE:
	    	*y = width - oldx;
	    	*x = oldy;
	    	break;
	    case IDM_UPSIDEDOWN:
	    	*x = width - oldx;
	    	*y = height - oldy;
		break;
	    case IDM_SEASCAPE:
	    	*y = oldx;
	    	*x = height - oldy;
	    	break;
	}
	return;
}

/* get current media index to paper_size[], or -1 if no match */
int
get_paper_size_index(void)
{
int i;
	for (i=0; papersizes[i].name != (char *)NULL; i++) {
	    if (!stricmp(papersizes[i].name, option.medianame))
		return i;
	}
	return -1;
}


/* change the size of the gs image if open */
void
gs_resize(void)
{
	pending.resize = TRUE;

	if ( gsdll.hmodule &&  (psfile.doc==(PSDOC *)NULL) && (gsdll.state != IDLE) )
	    /* don't know where we are so close and reopen */
	    pending.abort = TRUE;

	if (option.redisplay && (gsdll.state == PAGE)) {
	    if (psfile.doc != (PSDOC *)NULL)
	        pending.now = TRUE;
	    else {
		pending.abort = TRUE;	/* must restart from page 1 */
		pending.now = TRUE;
	    }
	}
}

void
gs_magnify(float scale)
{
int xtemp, ytemp;
	xtemp = (int)(option.xdpi * scale + 0.5);
	ytemp = (int)(option.ydpi * scale + 0.5);
	if ( (xtemp == option.xdpi) && (scale > 1.0) ) {
	    option.xdpi++;	/* force magnification if requested */
	    option.ydpi++;
	}
	else {
	    option.xdpi = xtemp;
	    option.ydpi = ytemp;
	}
	zoom = FALSE;
	gs_resize();
}

void
gsview_orientation(int new_orientation)
{
	if (new_orientation == option.orientation)
		return;
	if (new_orientation == IDM_SWAPLANDSCAPE) {
	    option.swap_landscape = !option.swap_landscape;
	    if (option.swap_landscape) 
	        check_menu_item(IDM_ORIENTMENU, IDM_SWAPLANDSCAPE, TRUE);
	    else
	        check_menu_item(IDM_ORIENTMENU, IDM_SWAPLANDSCAPE, FALSE);
	    if ((option.orientation != IDM_LANDSCAPE) && (option.orientation != IDM_SEASCAPE))
	        return;
	}
	else {
	    check_menu_item(IDM_ORIENTMENU, option.orientation, FALSE);
	    option.orientation = new_orientation;
	    check_menu_item(IDM_ORIENTMENU, option.orientation, TRUE);
	}
        zoom = FALSE;
	gs_resize();
	return;
}

void
gsview_media(int new_media)
{
	if ( (new_media == option.media) && (new_media != IDM_USERSIZE) )
		return;
	check_menu_item(IDM_MEDIAMENU, option.media, FALSE);
	option.media = new_media;
	check_menu_item(IDM_MEDIAMENU, option.media, TRUE);
	get_menu_string(IDM_MEDIAMENU, option.media, option.medianame, sizeof(option.medianame));
	gs_resize();
        zoom = FALSE;
	return;
}

void
gsview_unit(int new_unit)
{
	check_menu_item(IDM_UNITMENU, option.unit, FALSE);
	option.unit = new_unit;
	check_menu_item(IDM_UNITMENU, option.unit, TRUE);
	return;
}

/* free a PSFILE and contents */
/* Do NOT use this if you have just copied PSFILE to psfile */
void
e_free_psfile(PSFILE *ppsfile)
{
    psfile_free(ppsfile);
    free(ppsfile);
}


/* open a new document */
PSFILE *
gsview_openfile(char *filename)
{
int i;
PSFILE *tpsfile;
    tpsfile = (PSFILE *)malloc(sizeof(PSFILE));
    if (tpsfile == NULL)
	return NULL;
    memset((char *)tpsfile, 0, sizeof(PSFILE));

    strcpy(tpsfile->name, filename);
    tpsfile->pagenum = 1;
    info_wait(IDS_WAITREAD);
    if (dsc_scan(tpsfile)) {
        PSDOC *doc = tpsfile->doc;
	/* found DSC comments */
	if (doc->orientation == PORTRAIT)
	    gsview_orientation(IDM_PORTRAIT);
	if (doc->orientation == LANDSCAPE)
	    gsview_orientation(IDM_LANDSCAPE);
	if (doc->default_page_media) {
	    char thismedia[20];
	    for (i=IDM_LETTER; i<IDM_USERSIZE; i++) {
		get_menu_string(IDM_MEDIAMENU, i, thismedia, sizeof(thismedia));
		if (!stricmp(thismedia, doc->default_page_media->name)) {
		    gsview_media(i);
		    break;
		}
	    }
	    if (i == IDM_USERSIZE) {
		gsview_media(IDM_USERSIZE);
		option.user_width  = doc->default_page_media->width;
		option.user_height = doc->default_page_media->height;
		gsview_check_usersize();
	    }
	}
    }
    if (tpsfile->name[0] == '\0') {
	e_free_psfile(tpsfile);
	info_wait(IDS_NOWAIT);
	return NULL;
    }
    return tpsfile;
}


/* get filename then open new file for printing or extract */
void 
gsview_select()
{
char buf[MAXSTR];
	strcpy(buf, psfile.name);
	if (get_filename(buf, FALSE, FILTER_PSALL, 0, IDS_TOPICOPEN))
		gsview_selectfile(buf);
}

/* open new file for printing or extract */
void
gsview_selectfile(char *filename)
{
	while (*filename && *filename==' ')
	     filename++;

	if (gsdll.valid && (gsdll.state!=UNLOADED)) {
	    /* remember name for later */
	    strncpy(selectname, filename, sizeof(selectname));
	    /* close file and wait for notification */
	    post_img_message(WM_COMMAND, IDM_CLOSE);
	}
	else {
	    /* open it ourselves */
	    PSFILE *tpsfile = gsview_openfile(filename);
	    if (tpsfile) {
		psfile = *tpsfile;
		post_img_message(WM_GSTITLE, 0);
		free(tpsfile);	/* Do NOT free doc and page_list.select */
	    }
	}
	info_wait(IDS_NOWAIT);
}

/* get filename then open a new document and display it */
void 
gsview_display()
{
char buf[MAXSTR];
	strcpy(buf, psfile.name);
	if (get_filename(buf, FALSE, FILTER_PSALL, 0, IDS_TOPICOPEN))
		gsview_displayfile(buf);
}

/* open a new document and display it */
void
gsview_displayfile(char *filename)
{
PSFILE *tpsfile;
	tpsfile = gsview_openfile(filename);
	if (!tpsfile)
	    return;
	if (pending.psfile) {
	    message_box("pending.psfile is already set", 0);
	    e_free_psfile(tpsfile);
	    return;
	}
	pending.psfile = tpsfile;
	if ( gsdll.hmodule &&  (psfile.doc==(PSDOC *)NULL) && (gsdll.state != IDLE) )
	    /* don't know where we are so close and reopen */
	    pending.abort = TRUE;
	pending.now = TRUE;
}


/* Create and open a scratch file with a given name prefix. */
/* Write the actual file name at fname. */
FILE *
gp_open_scratch_file(const char *prefix, char *fname, const char *mode)
{	char *temp;
	if ( (temp = getenv("TEMP")) == NULL )
		gs_getcwd(fname, MAXSTR);
	else
		strcpy(fname, temp);

	/* Prevent X's in path from being converted by mktemp. */
	for ( temp = fname; *temp; temp++ ) {
		*temp = (char)tolower(*temp);
		if (*temp == '/')
		    *temp = '\\';
	}
	if ( strlen(fname) && (fname[strlen(fname)-1] != '\\') )
		strcat(fname, "\\");

	strcat(fname, prefix);
	strcat(fname, "XXXXXX");
#ifdef __IBMC__
	{ char *p;
	_tempnam(NULL, fname);
	strcpy(fname, p);
	free(p);
	}
#else
	mktemp(fname);
#endif
	return fopen(fname, mode);
}

/* reopen psfile */
/* psfile will then be locked until closed */
/* return TRUE if OK */
/* if psfile time/date or length has changed, return FALSE */
/* return FALSE if file can not be opened */
BOOL
dfreopen(void)
{
char *filename;
	if (psfile.ispdf)
	    return TRUE;	/* don't need to reopen */
	begin_crit_section();
	if (psfile.locked) {
	    end_crit_section();	/* someone else has it */
	    delayed_message_box(IDS_DEBUG_DFISLOCKED, 0);
	    return FALSE;
	}
	psfile.locked = TRUE;	/* stop others using it */
	end_crit_section();

	filename = psfile.name;

	if (psfile.file) {	/* should never happen */
	    fclose(psfile.file);
	    delayed_message_box(IDS_DEBUG_DFISOPEN, 0);
	}

	if (filename[0] == '\0') {
	    psfile.locked = FALSE;
	    delayed_message_box(IDS_NOTOPEN, 0);
	    return FALSE;
	}
	if ( (psfile.file = fopen(filename, "rb")) == (FILE *)NULL ) {
	    if (debug)
	        delayed_message_box(IDS_DEBUG_DFISMISSING, 0);
	    filename[0] = '\0';
	    psfile.locked = FALSE;
	    return FALSE;
	}
	if (psfile_changed()) {  /* doesn't cope with pdf file changing */
	    dfclose();
	    if (debug)
	        delayed_message_box(IDS_DEBUG_DFCHANGED, 0);
	    return FALSE;
	}
	return TRUE;
}

void
dfclose()
{
	if (debug) {
	    if (psfile.file == (FILE *)NULL)
		delayed_message_box(IDS_DEBUG_DFISCLOSED, 0);
	}
	if (psfile.file != (FILE *)NULL)
		fclose(psfile.file);
	psfile.file = (FILE *)NULL;
	psfile.locked = FALSE;
}


/* scan file for PostScript Document Structuring Conventions */
/* return TRUE if valid DSC comments found */
BOOL
dsc_scan(PSFILE *psf)
{
char line[MAXSTR];
PSDOC *doc;
	if (psf->file) {
	    message_box("dsc_scan: file is open but shouldn't be", 0);
	    fclose(psf->file);
	    psf->file = NULL;
	}

	if (psf->locked) {
	    char buf[MAXSTR];
	    load_string(IDS_DEBUG_DFISLOCKED, buf, sizeof(buf));
	    message_box(buf, 0);
	    return FALSE;
	}
	psf->locked = TRUE;	/* stop others using it */
	if ( (psf->file = fopen(psf->name, "rb")) == (FILE *)NULL ) {
	    char buf[MAXSTR+MAXSTR];
	    sprintf(buf, "File '%s' does not exist", psf->name);
	    message_box(buf, 0);
	    psf->name[0] = '\0';
	    psf->locked = FALSE;
	    return FALSE;
	}

	/* these shouldn't be needed */
	if (psf->page_list.select)
	    free(psf->page_list.select);
	psf->page_list.select = NULL;
	if (psf->doc)
		psfree(psf->doc);
	psf->preview = 0;
	
	/* check for PDF */
	psf->ispdf = FALSE;
	fgets(line, sizeof(line)-1, psf->file);
        rewind(psf->file);
	if ( strncmp("%PDF-", line, 5) == 0 ) {
	    fclose(psf->file);
	    psf->locked = FALSE;
	    psf->file = NULL;
	    psf->ispdf = TRUE;
	    return FALSE;	/* we don't know how many pages yet */
	}

	/* save file */
	psfile_savestat(psf);

	/* check for documents that start with Ctrl-D */
	psf->ctrld = (line[0] == '\004');
	if (option.ignore_dsc)
	    psf->doc = (PSDOC *)NULL;
	else 
	    psf->doc = psscan(psf->file);
	fclose(psf->file);
	psf->file = NULL;
	psf->locked = FALSE;
	/* check for DSC comments */
	doc = psf->doc;
	if (doc == (PSDOC *)NULL)
	    return FALSE;
	if (doc->doseps) {
	    /* check what sort of preview is present */
	    if (doc->doseps->tiff_begin)
		psf->preview = IDS_EPST;
	    if (doc->doseps->mf_begin)
		psf->preview = IDS_EPSW;
	}
	if (!psf->preview && (doc->beginpreview != doc->endpreview))
	    psf->preview = IDS_EPSI;
	if (doc->numpages) {
	    int i;
	    char *label;
	    for (i=0; i<doc->numpages; i++) {
		if ( (label = doc->pages[i].label) != NULL)  {
		    if (strlen(label)==0) {	/* remove old empty label */
			free(label);
			sprintf(line, "%d", i+1);
			label = malloc(strlen(line)+1);
			if (label)
			    strcpy(label, line);
			doc->pages[i].label = label;
		    }
		}
	    }
	    psf->page_list.select = (BOOL *)malloc( doc->numpages * sizeof(BOOL) );
	    if (psf->page_list.select)
	        memset(psf->page_list.select, 0, doc->numpages * sizeof(BOOL));
	}
	if (doc->epsf) {
	    /* warn if bounding box off the page */
	    int i = get_paper_size_index();
	    int width, height;
	    if (i < 0) {
	        width = option.user_width;
	        height = option.user_height;
	    }
	    else {
	        width = papersizes[i].width;
	        height = papersizes[i].height;
	    }
	    if ( !option.epsf_clip &&
	        ((doc->boundingbox[LLX] > width) || 
		 (doc->boundingbox[LLY] > height) ||
	         (doc->boundingbox[URX] < 0) || 
		 (doc->boundingbox[URY] < 0)) )
	    {
		char buf[MAXSTR];
	        load_string(IDS_EPS_OFF_PAGE, buf, sizeof(buf));
	        message_box(buf, 0);
	    }
	}
	return TRUE;
}



/* reverse zero based page number if needed */
int
map_page(int page)
{
    if (psfile.doc != (PSDOC *)NULL)
        if (psfile.doc->pageorder == DESCEND) 
	    return (psfile.doc->numpages - 1) - page;
    return page;
}

void
psfile_free(PSFILE *psf)
{
    if (psf == (PSFILE *)NULL)
	return;

    psf->name[0] = '\0';

    /* same as dfclose() */
    if (psf->file != (FILE *)NULL)
	fclose(psf->file);
    psf->file = (FILE *)NULL;
    psf->locked = FALSE;

    if (psf->page_list.select)
	free(psf->page_list.select);
    psf->page_list.select = NULL;
    if (psf->doc)
	psfree(psf->doc);
    psf->doc = (PSDOC *)NULL;
    if (psf->text_name) {
	if (!debug)
	    unlink(psf->text_name);
        psf->text_name[0] = '\0';
    }
}
