//===========================================================
// SCHED - A auto scheduler program.
// Copyright (C) 1994 Douglas Boling
//
// Revision History:
//
// 1.0   Initial Release
//
//===========================================================
// Returns no. of elements
#define dim(x) (sizeof(x) / sizeof(x[0]))   

#define STRICT


#define MAXCMDLINELEN     256
#define MAXENTRIES        32

#define ID_TIMER1         1
#define ID_TIMER2         2
//-----------------------------------------------------------
// Include files
//-----------------------------------------------------------
#include "windows.h"
#include "commdlg.h"
#include "stdlib.h"
#include "string.h"
#include "direct.h"
#include "time.h"
#include "sched.h"

//-----------------------------------------------------------
// Global data
//-----------------------------------------------------------
// Message dispatch table for MainWindowProc
struct decodeUINT MainMessages[] = {
	WM_CREATE, DoCreateMain,
	WM_SIZE, DoSizeMain,
	WM_COMMAND, DoCommandMain,
	WM_TIMER, DoTimerMain,
	WM_CLOSE, DoCloseMain,
	WM_DESTROY, DoDestroyMain,
};
// Command Message dispatch for MainWindowProc
struct decodeCMD MainMenuItems[] = {
	IDD_PROGLIST, DoMainCtlProgList,
	IDD_ADD, DoMainCtlAdd,
	IDD_DEL, DoMainCtlDel,
	IDD_EDIT, DoMainCtlEdit,
	IDD_ABOUT, DoMainCtlAbout,
	IDD_EXIT, DoMainCtlExit,
	IDOK, DoMainCtlExit,
	IDCANCEL, DoMainCtlExit,
};

HANDLE	hInst;
HWND		hMain;
UINT 		wVersion;
HLOCAL	hLocal = 0;
INT 		sEntryCnt = 0;
PSCHENTRY paArray;
UINT		hTimer = 0;
char	szCmd[MAXCMDLINELEN];

char	szTitleText[] = "Scheduler";      // Window title text
char	szAppName[] = "Sched";            // Application name
char	szIconName[] = "schedIcon";       // Icon name
char	szProfileName[] = "sched.ini";    // INI file name
//============================================================
// WinMain -- entry point for this application from Windows.
//============================================================
INT APIENTRY WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, 
                     LPSTR lpCmdLine, INT nCmdShow) {
	MSG	msg;
	INT	rc;

	hInst = hInstance;
	if (hPrevInstance == 0) {
		rc = InitApp(hInstance);
		if (rc) return rc;
	}		
	// Initialize this instance
	if((rc = InitInstance(hInstance, lpCmdLine, nCmdShow)) != 0)
		return rc;
	//
	// Application message loop
	//
	while (GetMessage (&msg, NULL, 0, 0)) {
		if (!IsDialogMessage (hMain, &msg)) {
			TranslateMessage(&msg);
			DispatchMessage(&msg);
		}	
	}
	// Instance cleanup
	return TermInstance(hInstance, msg.wParam);
}
  
//-----------------------------------------------------------
// InitApp - Global initialization code for this application.
//-----------------------------------------------------------
INT InitApp(HANDLE hInstance) {
	WNDCLASS 	wc;
	//
	// Register App Main Window class
	//
	wc.style = 0;                             // Window style
	wc.lpfnWndProc = MainWndProc;             // Callback function
	wc.cbClsExtra = 0;                        // Extra class data
	wc.cbWndExtra = DLGWINDOWEXTRA;           // Extra window data
	wc.hInstance = hInstance;                 // Owner handle
	wc.hIcon = LoadIcon(hInst, szIconName);   // Application icon
	wc.hCursor = LoadCursor(NULL, IDC_ARROW); // Default cursor
	wc.hbrBackground = GetStockObject(LTGRAY_BRUSH); // Background color 
	wc.lpszMenuName =  NULL;                  // Menu name
	wc.lpszClassName = szAppName;             // Window class name

	if (RegisterClass(&wc) == 0)
		return 1;
	return 0;
}
//-----------------------------------------------------------
// InitInstance - Instance initialization code for this app.
//-----------------------------------------------------------
INT InitInstance(HANDLE hInstance, LPSTR lpCmdLine, INT nCmdShow) {
	int	i, j;
	BOOL fGo = FALSE;
	char szNum[33];
	PSCHENTRY paPtr;
	LONG lTime;

	hLocal = LocalAlloc (GHND, sizeof (SCHENTRY) * MAXENTRIES);
	if (hLocal == 0)
		return 0x11;
	paArray = (PSCHENTRY) LocalLock (hLocal);
	
	if (*lpCmdLine != '\0') {
		lstrcpy (szCmd, lpCmdLine);
		i = -1;
		sEntryCnt = 0;
	} else {
		//
		// Read scheduler information from the INI file
		//
		sEntryCnt = GetPrivateProfileInt (szAppName, PRO_CNT, 0, 
		                                  szProfileName);
		if (sEntryCnt) {
			time (&lTime);
			paPtr = paArray;
			for (i = 0; i < sEntryCnt; i++) {
				itoa (i, szNum, 10);
				paPtr->fFlags = GetPrivateProfileInt (PRO_FLAGS, szNum, 
				                                      0, szProfileName);
				paPtr->lNextTime = MyGetPrivateProfileLong (PRO_NEXT, szNum, 
				                                      0, szProfileName);
				paPtr->lTime = MyGetPrivateProfileLong (PRO_TIME, szNum, 
				                                      0, szProfileName);
				paPtr->lData = MyGetPrivateProfileLong (PRO_DATA, szNum, 
				                                      0, szProfileName);
				GetPrivateProfileString (PRO_CMD, szNum, "", paPtr->szCmd,
				                         sizeof (paPtr->szCmd), 
				                         szProfileName);
				GetPrivateProfileString (PRO_DIR, szNum, "", paPtr->szDir,
				                         sizeof (paPtr->szDir), 
				                         szProfileName);

				if (!(paPtr->fFlags & SCH_ERROR)) {
					if (((paPtr->fFlags & 0x07) == SCH_ONCE) &&
					     (paPtr->lNextTime < lTime)) 
							paPtr->fFlags = SCH_EXPIRED;
					else {
						paPtr->lNextTime = ComputeNext (paPtr);
						fGo = TRUE;
					}
				}
				paPtr++;
			}	
		}
		i = GetPrivateProfileInt (szAppName, PRO_XPOS, 100,
		                          szProfileName);
		j = GetPrivateProfileInt (szAppName, PRO_YPOS, 100,
		                          szProfileName);
	}	
	// Create main window
	hMain = CreateDialog (hInstance, szAppName, 0, NULL);
	if(!hMain) return 0x10;

	// Start a timer if entries in the array
	if (fGo) 
		StartTimer();

	if (i != -1) {
		SetWindowPos (hMain, NULL, i, j, 0, 0, SWP_NOSIZE);
		ShowWindow(hMain, nCmdShow | SW_SHOW);
		UpdateWindow(hMain);           // force WM_PAINT message
	} else {	
		ShowWindow(hMain, nCmdShow | SW_SHOW);
		ShowWindow(hMain, SW_MINIMIZE);
	}	
	return 0;                         // return success flag
}
//------------------------------------------------------------
// TermInstance - Instance termination code for this app.
//------------------------------------------------------------
INT TermInstance(HANDLE hinstance, int sDefRC) {
	
	SavetoINI();
	return sDefRC;
}
//============================================================
// Message handling procedures for MainWindow
//============================================================
//------------------------------------------------------------
// MainWndProc - Callback function for application window
//------------------------------------------------------------
LONG CALLBACK MainWndProc(HWND hWnd, UINT wMsg, UINT wParam, 
                          LONG lParam) {
	INT i;
	//
	// Search message list to see if we need to handle this
	// message.  If in list, call procedure.
	//
	for(i = 0; i < dim(MainMessages); i++) {
		if(wMsg == MainMessages[i].Code)
			return (*MainMessages[i].Fxn)(hWnd, wMsg, wParam, lParam);
	}
	return DefDlgProc(hWnd, wMsg, wParam, lParam);
}
//------------------------------------------------------------
// DoCreateMain - process WM_CREATE message for frame window.
//------------------------------------------------------------ 
LONG DoCreateMain (HWND hWnd, UINT wMsg, UINT wParam, LONG lParam) {

	
	return 0;
}
//------------------------------------------------------------
// DoSizeMain - process WM_SIZE message for frame window.
//------------------------------------------------------------ 
LONG DoSizeMain (HWND hWnd, UINT wMsg, UINT wParam, LONG lParam) {

	SetButtons (hWnd);
	FillLB (hWnd);
	return DefDlgProc(hWnd, wMsg, wParam, lParam);
}
//------------------------------------------------------------
// DoTimerMain - process WM_TIMER message for frame window.
//------------------------------------------------------------ 
LONG DoTimerMain (HWND hWnd, UINT wMsg, UINT wParam, LONG lParam) {
	PSCHENTRY paPtr;
	UINT hExecInst;
	LONG lTime;
	INT i, sStartFlags;
	BOOL fRedraw = FALSE;

	//Switch to minute timer after first sync tick.		
	if (wParam = ID_TIMER2) {
		KillTimer (hWnd, ID_TIMER2);
		if (sEntryCnt)
			hTimer = SetTimer (hWnd, ID_TIMER1, (UINT)60000, NULL);
	} else if (sEntryCnt == 0) {
		KillTimer (hWnd, ID_TIMER1);
		hTimer = 0;
	}	
	time (&lTime);
	paPtr = paArray;
	for (i = 0; i < sEntryCnt; i++) {
		if (!(paPtr->fFlags & SCH_ERROR) && 
		    ((paPtr->fFlags & 0x07) != SCH_EXPIRED) && 
		    (paPtr->lNextTime <= lTime)) {

			if (strlen (paPtr->szDir) != 0) {
				if (paPtr->szDir[1] == ':') {
					//UCase drive letter & convert to num.
					_chdrive ((paPtr->szDir[0] & 0xdf) - 0x40);
					//Pt to dir after drive letter
					chdir ((char *)paPtr->szDir[2]);
				} else	
					chdir (paPtr->szDir);
			}
			if (paPtr->fFlags & SCH_ACTIVATE) {
				if (paPtr->fFlags & SCH_ICON)
					sStartFlags = SW_SHOWMINIMIZED;
				else
					sStartFlags = SW_SHOWNORMAL;
			} else {
				if (paPtr->fFlags & SCH_ICON)
					sStartFlags = SW_SHOWMINNOACTIVE;
				else
					sStartFlags = SW_SHOWNOACTIVATE;
			}		
			hExecInst = WinExec (paPtr->szCmd, sStartFlags);
			if (hExecInst < 32) {
				paPtr->fFlags |= SCH_ERROR;
				PrintError (hWnd, ERR_LAUNCH + hExecInst);
			} else {
				if ((paPtr->fFlags & 0x07) == SCH_ONCE) {
					paPtr->fFlags = SCH_EXPIRED;
					fRedraw = TRUE;
				} else	
					paPtr->lNextTime = ComputeNext (paPtr);
			}
		}
		paPtr++;
	}
	if (fRedraw)
		FillLB(hWnd);
	return 0;
}
//------------------------------------------------------------
// DoCommandMain - process WM_COMMAND message for frame window 
// by decoding the menubar item with the menuitems[] array, 
// then running the corresponding function to process the command.
//------------------------------------------------------------ 
LONG DoCommandMain (HWND hWnd, UINT wMsg, UINT wParam, LONG lParam) {
	INT	i;
	UINT	idItem, wNotifyCode;
	HWND	hwndCtl;

	idItem = (UINT) wParam;                      // Parse Parameters
	hwndCtl = (HWND) LOWORD(lParam);
	wNotifyCode = (UINT) HIWORD(lParam);
	//
	// Call routine to handle control message
	//
	for(i = 0; i < dim(MainMenuItems); i++) {
		if(idItem == MainMenuItems[i].Code)
			return (*MainMenuItems[i].Fxn)(hWnd, idItem, hwndCtl, 
			                               wNotifyCode);
	}
	return DefDlgProc(hWnd, wMsg, wParam, lParam);
}
//------------------------------------------------------------
// DoCloseMain - process WM_CLOSE message for frame window.
//------------------------------------------------------------ 
LONG DoCloseMain (HWND hWnd, UINT wMsg, UINT wParam, LONG lParam) {

	DestroyWindow (hMain);
	return TRUE;
}
//------------------------------------------------------------
// DoDestroyMain - process WM_DESTROY message for frame window.
//------------------------------------------------------------ 
LONG DoDestroyMain (HWND hWnd, UINT wMsg, UINT wParam, LONG lParam) {
   RECT	rect;

	//Save Window position
	if	(!IsIconic (hWnd) && !IsZoomed (hWnd)) {
		GetWindowRect (hWnd, &rect);
		MyWritePrivateProfileInt (szAppName, PRO_XPOS, rect.left,
		                          szProfileName);
		MyWritePrivateProfileInt (szAppName, PRO_YPOS, rect.top,
		                          szProfileName);
	}
	if (hTimer)
		KillTimer(hMain, ID_TIMER1);
	PostQuitMessage (0);
	return DefDlgProc(hWnd, wMsg, wParam, lParam);
}
//============================================================
// Control handling procedures for MainWindow
//============================================================
//------------------------------------------------------------
// DoMainCtlProgList - Process Listbox messages
//------------------------------------------------------------ 
LONG DoMainCtlProgList (HWND hWnd, UINT idItem, HWND hwndCtl, 
                        UINT wNotifyCode) {

	if (wNotifyCode == LBN_DBLCLK)
		SendMessage (hWnd, WM_COMMAND, IDD_EDIT, MAKELONG (0, BN_CLICKED));
	else if (wNotifyCode == LBN_SELCHANGE)
		SetButtons (hWnd);
	return 0;
}
//------------------------------------------------------------
// DoMainCtlAdd - Process Add button
//------------------------------------------------------------ 
LONG DoMainCtlAdd (HWND hWnd, UINT idItem, HWND hwndCtl, 
                   UINT wNotifyCode) {
	PSCHENTRY paEntry;
	SCHENTRY sch;

	if (wNotifyCode != BN_CLICKED)
		return 0;

	if (sEntryCnt + 1 >= MAXENTRIES) {
		PrintError (hWnd, ERR_MAXSCH);
		return 0;
	}
	memset (&sch, 0, sizeof sch);
	if (MyDisplayDialog(hInst, "AddBox", hWnd, (DLGPROC) AddDlgProc, 
	                    (LPARAM)MAKELONG (&sch, 0)) == 0) 
		return 0;
	paEntry = paArray + sEntryCnt;
	*paEntry = sch;
	sEntryCnt++;
	SavetoINI();                       //Save array to INI file
	FillLB (hWnd);
	return 0;
}
//------------------------------------------------------------
// DoMainCtlDel - Process Del button
//------------------------------------------------------------ 
LONG DoMainCtlDel (HWND hWnd, UINT idItem, HWND hwndCtl, 
                   UINT wNotifyCode) {
	PSCHENTRY paEntry;
	INT i, sSel, sRC;
								
	if (wNotifyCode != BN_CLICKED)
		return 0;
		
	sSel = (INT) SendDlgItemMessage (hWnd, IDD_PROGLIST, LB_GETCURSEL, 0, 0);
	if (sSel == LB_ERR) {
		PrintError (hWnd, ERR_NOSCHSEL);
		return 0;
	}
	sRC = MessageBox (hWnd, "Do you want to delete this item?", szTitleText,
	                  MB_YESNO);
	if (sRC != IDYES)
		return 0;
	//Delete item from list
	paEntry = paArray + sEntryCnt;
	for (i = sSel; i < sEntryCnt-1; i++)
		*paEntry = *(paEntry+1);
	sEntryCnt--;
	SavetoINI();                       //Save array to INI file
	FillLB (hWnd);
	return 0;
}
//------------------------------------------------------------
// DoMainCtlEdit - Process Edit button
//------------------------------------------------------------ 
LONG DoMainCtlEdit (HWND hWnd, UINT idItem, HWND hwndCtl, 
                    UINT wNotifyCode) {
	PSCHENTRY paEntry;
	SCHENTRY sch;
	INT sSel;
								
	if (wNotifyCode != BN_CLICKED)
		return 0;
		
	sSel = (INT) SendDlgItemMessage (hWnd, IDD_PROGLIST, LB_GETCURSEL, 0, 0);

	if (sSel == LB_ERR) {
		PrintError (hWnd, ERR_NOSCHSEL);
		return 0;
	}
	paEntry = paArray + sSel;
	sch = *paEntry;
	if (MyDisplayDialog(hInst, "AddBox", hWnd, (DLGPROC) AddDlgProc, 
	                (LPARAM)MAKELONG (&sch, 1)) == 0)
		return 0;

	*paEntry = sch;
	SavetoINI();                       //Save array to INI file
	FillLB (hWnd);
	return 0;
}
//------------------------------------------------------------
// DoMainCtlAbout - Process About button
//------------------------------------------------------------ 
LONG DoMainCtlAbout (HWND hWnd, UINT idItem, HWND hwndCtl, 
                     UINT wNotifyCode) {
								
	if (wNotifyCode == BN_CLICKED)
		MyDisplayDialog(hInst, "AboutBox", hWnd, 
   		             (DLGPROC) AboutDlgProc, (LONG) wVersion);
	return 0;
}
//------------------------------------------------------------
// DoMainCtlExit - Process Exit button
//------------------------------------------------------------ 
LONG DoMainCtlExit (HWND hWnd, UINT idItem, HWND hwndCtl, 
                     UINT wNotifyCode) {

	if (wNotifyCode == BN_CLICKED)
		SendMessage (hWnd, WM_CLOSE, 0, 0);
	return 0;
}
//============================================================
// AddDlgProc - Add dialog box dialog procedure
//============================================================
BOOL CALLBACK AddDlgProc (HWND hWnd, UINT wMsg, UINT wParam, 
                          LONG lParam) {
	static char *szFilter[] = {"Programs", "*.EXE;*.COM;*.PIF;*.BAT", 
	                           "All Files (*.*)", "*.*", 
										"" };
	static char *pszDays[] = {{"Sunday"}, {"Monday"}, {"Tuesday"}, {"Wednesday"}, 
	                          {"Thursday"}, {"Friday"}, {"Saturday"}};
	static PSCHENTRY paEntry;
	OFSTRUCT of;
	char szNum[33], szTemp[128], *pszTemp;
	INT i;
	LONG lValue;
                             
	switch (wMsg) {
		case WM_INITDIALOG:
			paEntry = (PSCHENTRY) LOWORD (lParam);
			if (HIWORD (lParam)) {
				if ((paEntry->fFlags & 0x0007) == SCH_EXPIRED)
					CheckRadioButton (hWnd, IDD_ONCE, IDD_MONTH, IDD_ONCE);
				else 	
					CheckRadioButton (hWnd, IDD_ONCE, IDD_MONTH, 
					                  IDD_ONCE + (paEntry->fFlags & 0x0007) - 1);
				TimeToAsc (paEntry->lTime, szTemp, sizeof (szTemp));
				SetDlgItemText (hWnd, IDD_STIME, szTemp);

				SetDlgItemText (hWnd, IDD_CMD, paEntry->szCmd);
				SetDlgItemText (hWnd, IDD_WDIR, paEntry->szDir);
				
				CheckDlgButton (hWnd, IDD_ICON, 
				                (paEntry->fFlags & SCH_ICON) ? 1 : 0);
				CheckDlgButton (hWnd, IDD_ACTIVATE, 
				                (paEntry->fFlags & SCH_ACTIVATE) ? 1 : 0);

				SendDlgItemMessage (hWnd, IDD_LIST1, LB_RESETCONTENT, 0, 0);
				
				switch (paEntry->fFlags & 0x0007) {
					case SCH_EXPIRED:
					case SCH_ONCE:
						DateToAsc (paEntry->lData, szTemp, sizeof (szTemp));
						SetDlgItemText (hWnd, IDD_EDIT1, szTemp);
						SetDlgItemText (hWnd, IDD_TEXT1, "Date (mm-dd-yy)");
						ShowWindow (GetDlgItem (hWnd, IDD_EDIT1), SW_SHOW);
						ShowWindow (GetDlgItem (hWnd, IDD_LIST1), SW_HIDE);
						break;

					case SCH_DAY:
						TimeToAsc (paEntry->lData, szTemp, sizeof (szTemp));
						if (paEntry->lData) {
							lValue = paEntry->lData / 60;
							ltoa (lValue / 60, szTemp, 10);
							strcat (szTemp, ":");
							ltoa (lValue % 60, &szTemp[strlen (szTemp)], 10);
						} else
							strcpy (szTemp, "Once/day");	
						SetDlgItemText (hWnd, IDD_EDIT1, szTemp);
						SetDlgItemText (hWnd, IDD_TEXT1, "Interval (hh:mm)");
						ShowWindow (GetDlgItem (hWnd, IDD_EDIT1), SW_SHOW);
						ShowWindow (GetDlgItem (hWnd, IDD_LIST1), SW_HIDE);
						break;

					case SCH_WEEK:
						lValue = paEntry->lData;
						for (i = 0; i < 7; i++) {
							SendDlgItemMessage (hWnd, IDD_LIST1, LB_ADDSTRING, 0, 
	   	                                (LPARAM)(LPSTR) pszDays[i]);
							if (lValue & 0x0080)
								SendDlgItemMessage (hWnd, IDD_LIST1, LB_SETSEL, 
								                    TRUE, MAKELONG (i, 0)); 
							lValue = lValue << 1;
						}
						SetDlgItemText (hWnd, IDD_TEXT1, "Days of the Week");
						ShowWindow (GetDlgItem (hWnd, IDD_EDIT1), SW_HIDE);
						ShowWindow (GetDlgItem (hWnd, IDD_LIST1), SW_SHOW);
						break;

					case SCH_MONTH:
						lValue = paEntry->lData;
						for (i = 0; i < 31; i++) {
							itoa (i+1, szNum, 10);							
							SendDlgItemMessage (hWnd, IDD_LIST1, LB_ADDSTRING, 0, 
	   	                                (LPARAM)(LPSTR) szNum);
							if (lValue & 0x80000000)
								SendDlgItemMessage (hWnd, IDD_LIST1, LB_SETSEL, 
								                    TRUE, MAKELONG (i, 0)); 
							lValue = lValue << 1;
						}
						SetDlgItemText (hWnd, IDD_TEXT1, "Days of the Month");
						ShowWindow (GetDlgItem (hWnd, IDD_EDIT1), SW_HIDE);
						ShowWindow (GetDlgItem (hWnd, IDD_LIST1), SW_SHOW);
						break;
				}
				SetWindowText (hWnd, "Edit Schedule");
				SetDlgItemText (hWnd, IDOK, "&Save");
			} else {
				SetWindowText (hWnd, "Schedule Program");
				CheckRadioButton (hWnd, IDD_ONCE, IDD_MONTH, IDD_ONCE);
				ShowWindow (GetDlgItem (hWnd, IDD_LIST1), SW_HIDE);
			}	
			return TRUE;

		case WM_COMMAND:
			switch (wParam) {
		   	case IDD_ADDBROWSE:
					if (MyGetFilename (hWnd, szTemp, 
					                   sizeof (szTemp), *szFilter, 1))
					SetDlgItemText (hWnd, IDD_CMD, szTemp);
					pszTemp = strrchr (szTemp, '\\');
					if (pszTemp) {
						*pszTemp = '\0';
						SetDlgItemText (hWnd, IDD_WDIR, szTemp);
					} else	
						SetDlgItemText (hWnd, IDD_WDIR, "");
					return TRUE;

				case IDD_ONCE:
					SetDlgItemText (hWnd, IDD_TIMETEXT, "Time (hh:mm)");
					SetDlgItemText (hWnd, IDD_TEXT1, "Date (mm-dd-yy)");
					ShowWindow (GetDlgItem (hWnd, IDD_EDIT1), SW_SHOW);
					ShowWindow (GetDlgItem (hWnd, IDD_LIST1), SW_HIDE);
					SetDlgItemText (hWnd, IDD_EDIT1, "");
					return TRUE;
					
				case IDD_DAY:
					SetDlgItemText (hWnd, IDD_TIMETEXT, "Starting Time");
					SetDlgItemText (hWnd, IDD_TEXT1, "Interval (hh:mm)");
					ShowWindow (GetDlgItem (hWnd, IDD_EDIT1), SW_SHOW);
					ShowWindow (GetDlgItem (hWnd, IDD_LIST1), SW_HIDE);
					SetDlgItemText (hWnd, IDD_EDIT1, "");
					return TRUE;
					
				case IDD_WEEK:
					SetDlgItemText (hWnd, IDD_TIMETEXT, "Time (hh:mm)");
					SetDlgItemText (hWnd, IDD_TEXT1, "Days of the Week");
					ShowWindow (GetDlgItem (hWnd, IDD_EDIT1), SW_HIDE);
					ShowWindow (GetDlgItem (hWnd, IDD_LIST1), SW_SHOW);
					SendDlgItemMessage (hWnd, IDD_LIST1, LB_RESETCONTENT, 0, 0);
					for (i = 0; i < 7; i++)
						SendDlgItemMessage (hWnd, IDD_LIST1, LB_ADDSTRING, 0, 
	                                   (LPARAM)(LPSTR) pszDays[i]);
					return TRUE;

				case IDD_MONTH:
					SetDlgItemText (hWnd, IDD_TIMETEXT, "Time (hh:mm)");
					SetDlgItemText (hWnd, IDD_TEXT1, "Days of the Month");
					ShowWindow (GetDlgItem (hWnd, IDD_EDIT1), SW_HIDE);
					ShowWindow (GetDlgItem (hWnd, IDD_LIST1), SW_SHOW);
					SendDlgItemMessage (hWnd, IDD_LIST1, LB_RESETCONTENT, 0, 0);
					for (i = 1; i < 32; i++) {
						itoa (i, szNum, 10);
						SendDlgItemMessage (hWnd, IDD_LIST1, LB_ADDSTRING, 0, 
	                                   (LPARAM)(LPSTR) szNum);
					}
					return TRUE;

				case IDOK:
					//Get time
					GetDlgItemText (hWnd, IDD_STIME, szTemp, sizeof (szTemp));
					paEntry->lTime = AscToTime (szTemp);
					if ((paEntry->lTime == -1) || (strlen (szTemp) == 0)) {
						PrintError (hWnd, ERR_BADTIME);
						return TRUE;
					}
					if (IsDlgButtonChecked (hWnd, IDD_ONCE)) {
						GetDlgItemText (hWnd, IDD_EDIT1, szTemp, sizeof (szTemp));
						// Get and verify date spec
						if (strlen (szTemp) == 0) {
							time(&lValue);
							DateToAsc (lValue, szTemp, sizeof (szTemp));
						}
						lValue = AscToDate (szTemp);
						if (lValue == -1) {
							PrintError (hWnd, ERR_BADDATE);
							return TRUE;
						}
						paEntry->lData = lValue + paEntry->lTime;
						time (&lValue);
						if (lValue > paEntry->lData) {
							PrintError (hWnd, ERR_TIMEPAST);
							return TRUE;
						}
						paEntry->fFlags = SCH_ONCE;

					// Day
					} else if (IsDlgButtonChecked (hWnd, IDD_DAY)) {
						GetDlgItemText (hWnd, IDD_EDIT1, szTemp, sizeof (szTemp));
						paEntry->lData = AscToTime (szTemp);
						if (paEntry->lData == -1) {
							PrintError (hWnd, ERR_BADINTERVAL);
							return TRUE;
						}
						paEntry->fFlags = SCH_DAY;
						
					// Week
					} else if (IsDlgButtonChecked (hWnd, IDD_WEEK)) {
						lValue = 0;
						for (i = 0; i < 7; i++) {
							if (SendDlgItemMessage (hWnd, IDD_LIST1, 
						                           LB_GETSEL, i, 0))
								lValue |= 1;
							lValue = lValue << 1;
						}
						if (lValue == 0) {
							PrintError (hWnd, ERR_NOSELECT);
							return TRUE;
						}
						paEntry->lData = lValue;
						paEntry->fFlags = SCH_WEEK;

					// Month
					} else if (IsDlgButtonChecked (hWnd, IDD_MONTH)) {
						lValue = 0;
						for (i = 0; i < 31; i++) {
							if (SendDlgItemMessage (hWnd, IDD_LIST1, 
						                           LB_GETSEL, i, 0))
								lValue |= 1;
							lValue = lValue << 1;
						}
						if (lValue == 0) {
							PrintError (hWnd, ERR_NOSELECT);
							return TRUE;
						}
						paEntry->lData = lValue;
						paEntry->fFlags = SCH_MONTH;
					}
					//Get and validate the command line
					GetDlgItemText (hWnd, IDD_CMD, szTemp, sizeof (szTemp));
					if (strlen (szTemp) == 0) {
						PrintError (hWnd, ERR_NOCMDLINE);
						return TRUE;
					}
					strcpy (paEntry->szCmd, szTemp);
					pszTemp = strpbrk (szTemp, " /\t");
					if (pszTemp) 
						*pszTemp = '\0';
					if (OpenFile (szTemp, &of, OF_EXIST) == HFILE_ERROR) {
						PrintError (hWnd, of.nErrCode);
						return TRUE;
					}
					//Get and validate the working directory
					GetDlgItemText (hWnd, IDD_WDIR, szTemp, sizeof (szTemp));
					strcpy (paEntry->szDir, szTemp);
					if (strlen (szTemp) != 0) {
						strcat (szTemp, "\\CON");
						if (OpenFile (szTemp, &of, OF_EXIST) == HFILE_ERROR) {
							PrintError (hWnd, -(ERR_DOS + of.nErrCode));
							return TRUE;
						}
					}
					//Get startup flags
					if (IsDlgButtonChecked (hWnd, IDD_ICON))
				      paEntry->fFlags |= SCH_ICON;
					else	
				      paEntry->fFlags &= ~SCH_ICON;
					if (IsDlgButtonChecked (hWnd, IDD_ACTIVATE))
				      paEntry->fFlags |= SCH_ACTIVATE;
					else	
				      paEntry->fFlags &= ~SCH_ACTIVATE;
						
					paEntry->lNextTime = ComputeNext (paEntry);
					EndDialog(hWnd, 1);
					return TRUE;
					
				case IDCANCEL:
					EndDialog(hWnd, 0);
					return TRUE;
			}	
			return FALSE;

		case WM_DESTROY:
			return TRUE;

	}
	return FALSE;
}
//============================================================
// AboutDlgProc - About dialog box dialog procedure
//============================================================
BOOL CALLBACK AboutDlgProc (HWND hWnd, UINT wMsg, UINT wParam, 
                            LONG lParam) {
	HDC hdc;
	HPEN hDPen, hLPen, hOldPen;
	PAINTSTRUCT ps;
	RECT rectAbout;
                             
	switch (wMsg) {
		case WM_PAINT:
			//
			// Draw the rectangle surrounding the dialog box text.
			//
			hdc = BeginPaint (hWnd, &ps);
			//Get text rectangle defined in RC file
			GetClientRect (GetDlgItem (hWnd, IDD_FRAME), &rectAbout);
			ClientToScreen (GetDlgItem (hWnd, IDD_FRAME), (LPPOINT)&rectAbout);
			ScreenToClient (hWnd, (LPPOINT)&rectAbout);
			rectAbout.left -= 2;
			rectAbout.top -= 2;
			rectAbout.right += rectAbout.left + 4;
			rectAbout.bottom += rectAbout.top + 4;
			hDPen = CreatePen (PS_SOLID, 1, GetSysColor (COLOR_BTNSHADOW));
			hLPen = CreatePen (PS_SOLID, 1, GetSysColor (COLOR_BTNHIGHLIGHT));

			//Start at bottom left, draw dark pen up and over.
			hOldPen = SelectObject (hdc, hDPen);
			MoveTo (hdc, rectAbout.left, rectAbout.bottom);
			LineTo (hdc, rectAbout.left, rectAbout.top);
			LineTo (hdc, rectAbout.right+1, rectAbout.top);
			//Start at bottom left, draw light pen over and up.
			SelectObject (hdc, hLPen);
			MoveTo (hdc, rectAbout.left+1, rectAbout.bottom);
			LineTo (hdc, rectAbout.right, rectAbout.bottom);
			LineTo (hdc, rectAbout.right, rectAbout.top);

			SelectObject (hdc, hOldPen);
			DeleteObject (hDPen);
			DeleteObject (hLPen);
			EndPaint (hWnd, &ps);
			return TRUE;

		case WM_COMMAND:
	   	if ((wParam == IDOK) || (wParam == IDCANCEL)) {
				EndDialog(hWnd, 0);
			}	
			return TRUE;
	}
	return FALSE;
}

//------------------------------------------------------------
// SavetoINI - Writes the data to the ini file.
//------------------------------------------------------------ 
void SavetoINI (void) {
	INT i;
	PSCHENTRY paPtr;
	char	szNum[8];
	
	WritePrivateProfileString (PRO_FLAGS, NULL, NULL, szProfileName);
	WritePrivateProfileString (PRO_NEXT, NULL, NULL, szProfileName);
	WritePrivateProfileString (PRO_TIME, NULL, NULL, szProfileName);
	WritePrivateProfileString (PRO_DATA, NULL, NULL, szProfileName);
	WritePrivateProfileString (PRO_CMD, NULL, NULL, szProfileName);
	WritePrivateProfileString (PRO_DIR, NULL, NULL, szProfileName);

	paPtr = paArray;
	for (i = 0; i < sEntryCnt; i++) {
		itoa (i, szNum, 10);
		MyWritePrivateProfileInt (PRO_FLAGS, szNum, paPtr->fFlags, szProfileName);
		MyWritePrivateProfileLong (PRO_NEXT, szNum, paPtr->lNextTime, szProfileName);
		MyWritePrivateProfileLong (PRO_TIME, szNum, paPtr->lTime, szProfileName);
		MyWritePrivateProfileLong (PRO_DATA, szNum, paPtr->lData, szProfileName);
		WritePrivateProfileString (PRO_CMD, szNum, paPtr->szCmd, szProfileName);
		WritePrivateProfileString (PRO_DIR, szNum, paPtr->szDir, szProfileName);
		paPtr++;
	}
	MyWritePrivateProfileInt (szAppName, PRO_CNT, sEntryCnt, szProfileName);
	return;
}	
//------------------------------------------------------------
// DispTimeText - Converts a time value into text
//------------------------------------------------------------ 
void DispTimeText (UINT fFlag, char *pszOut) {

	*pszOut = '0';
	if (fFlag & SCH_ERROR) {
		strcpy (pszOut, "Launch Failure");
		return;
	}	
	switch (fFlag & 0x07) {
		case SCH_EXPIRED:
			strcpy (pszOut, "Expired");
			break;
			
		case SCH_ONCE:
			strcpy (pszOut, "Once");
			break;
			
		case SCH_DAY:
			strcpy (pszOut, "Reoccurring");
			break;
			
		case SCH_WEEK:
			strcpy (pszOut, "Weekly");
			break;
			
		case SCH_MONTH:
			strcpy (pszOut, "Monthly");
			break;
	}			
	return;
}							 
//------------------------------------------------------------
// DateToAsc - Converts date to string format
//------------------------------------------------------------
char *DateToAsc (LONG lDate, char *pszDString, INT sMax) {
	TM *ptime;
	
	ptime = localtime (&lDate);
	strftime (pszDString, sMax, "%m-%d-%y", ptime);	
	return pszDString;
}		
//------------------------------------------------------------
// TimeToAsc - Converts time to string format
//------------------------------------------------------------
char *TimeToAsc (LONG lDate, char *pszDString, INT sMax) {
	TM *ptime;
	LONG lToday;
	
	//Init time structure.
	time (&lToday);
	ptime = localtime (&lToday);
	ptime->tm_hour = 0;
	ptime->tm_min = 0;
	ptime->tm_sec = 0;
	lDate += mktime (ptime);
	
	ptime = localtime (&lDate);
	strftime (pszDString, sMax, "%I:%M %p", ptime);	
	return pszDString;
}		
//------------------------------------------------------------
// AscToDate - Converts ASCII date string into Date format
// ASCII date should be in mm-dd-yy format.
//------------------------------------------------------------
LONG AscToDate (char *pszDString) {
	TM *ptime;
	INT sDay;
	char ch;
	LONG lToday;
	
	//Init time structure.
	time (&lToday);
	ptime = localtime (&lToday);
	ptime->tm_hour = 0;
	ptime->tm_min = 0;
	ptime->tm_sec = 0;
	lToday = mktime (ptime);
	//Figure out the sep char.	
	if (strchr (pszDString, '-') == 0) {
		if (strchr (pszDString, '/') == 0)
			return -1;
		ch = '/';
	} else 
		ch = '-';	

	//Convert month
	ptime->tm_mon = atoi (pszDString) - 1;
	if ((ptime->tm_mon < 0) || (ptime->tm_mon > 11))
		return -1;
	pszDString = strchr (pszDString, ch);
	//Convert day
	if (pszDString)
		ptime->tm_mday = atoi (pszDString+1);

	pszDString = strchr (pszDString+1, ch);
	//Convert year
	if (pszDString)
		ptime->tm_year = atoi (pszDString+1);
		if (ptime->tm_year > 1900)
			ptime->tm_year -= 1900;
		else if (ptime->tm_year < 80)
			ptime->tm_year += 100;
	//Compute last day of month and verify day.  
	// This fails for years 1900, 2100, 2200...
	if (ptime->tm_mon == 1) {
		if (ptime->tm_year % 4)
			sDay = 28;
		else
			sDay = 29;
	} else if (((ptime->tm_mon < 7) && (ptime->tm_mon % 2)) ||
	          ((ptime->tm_mon > 7) && (ptime->tm_mon % 2 == 0)))
		sDay = 30;
	else	
		sDay = 31;
	if ((ptime->tm_mday < 1) || (ptime->tm_mday > sDay))
		return -1;

	return mktime (ptime);
}		
//------------------------------------------------------------
// AscToTime - Converts ASCII date string into Date format
// ASCII time should be in hh:mm:ss [{AM|PM}] format.
//------------------------------------------------------------
LONG AscToTime (char *pszDString) {
	TM *ptime;
	LONG lToday;
	
	//Init time structure.
	time (&lToday);

	ptime = localtime (&lToday);
	ptime->tm_hour = 0;
	ptime->tm_min = 0;
	ptime->tm_sec = 0;
	lToday = mktime (ptime);
		
	//Convert hours val and save.
	ptime->tm_hour = atoi (pszDString);
	if ((ptime->tm_hour < 0) || (ptime->tm_hour > 23))
		return -1;

	//Convert minutes
	pszDString = strchr (pszDString, ':');
	if (pszDString)
		ptime->tm_min = atoi (pszDString+1);
	if ((ptime->tm_min < 0) || (ptime->tm_min > 59))
		return -1;
	//Check for AM/PM.
	pszDString = strpbrk (pszDString+1, "aApP");
	if (pszDString) {
		if (ptime->tm_hour > 12)
			return -1;
		strupr (pszDString);
		if ((strcmp (pszDString, "PM") == 0) ||
		    (strcmp (pszDString, "P") == 0)) {
			if (ptime->tm_hour == 12)
				;
			if (ptime->tm_hour < 12)
				ptime->tm_hour += 12;
		} else if ((strcmp (pszDString, "AM") == 0) ||
		           (strcmp (pszDString, "A") == 0))
			if (ptime->tm_hour == 12)
				ptime->tm_hour -= 12;
		else 
			return -1;
	}
	return mktime (ptime) - lToday;
}
//------------------------------------------------------------
// WeekMonthHelp - Common code between week and month 
// computation.
//------------------------------------------------------------ 
LONG WeekMonthHelp (LONG lValue, LONG lStartTime, LONG lToday, LONG lMask,
                    LONG lShift, INT sSize) {
	INT i = 0;
	LONG lNextTime;

	//If time past today, start looking at tomorrow.
	if ((lToday > lStartTime) && (lValue & lMask)) {
		if (lShift & lValue)
			lValue |= 1;
		lValue = lValue << 1;	
		i++;
	}
	//Shift through bit map to find next day.
	for (; i < sSize; i++) {
		if (lValue & lMask)
			break;
		if (lShift & lValue)
			lValue |= 1;
		lValue = lValue << 1;	
	}
	if (i)
		//Time = remainder of today + sec/day*days + Time of event
		lNextTime = (i * 86400) + lStartTime;
	else	
		lNextTime = lStartTime;
	return lNextTime;		
}
//------------------------------------------------------------
// ComputeNext - Computes the next expartion time for an event.
//------------------------------------------------------------ 
LONG ComputeNext (PSCHENTRY paEntry) {
	LONG lMidnight, lTime, lToday, lNextTime, lValue; 
	TM *ptime;
	
	//Get current time/day
	time (&lTime);
	ptime = localtime (&lTime);
	ptime->tm_hour = 0;
	ptime->tm_min = 0;
	ptime->tm_sec = 0;
	lMidnight = mktime (ptime);
	//Set lToday to elapsed seconds in this day.
	lToday = lTime - lMidnight;
	
	lValue = paEntry->lData;
	
	switch (paEntry->fFlags & 0x0007) {
		case SCH_ONCE:
			lNextTime = lValue;
			break;
			
		case SCH_DAY:
			//Assume today's starting time
			if (paEntry->lTime > lToday)
				lNextTime = lMidnight + paEntry->lTime;
			//If time past, and no interval, make it tomorrow
			else if (paEntry->lData == 0)
				lNextTime = lMidnight + 86400 + paEntry->lTime;
			//If time past, compute next interval by computing next interval
			// between now and start time, then add start time and day.
			else {
				lNextTime = (((lToday - paEntry->lTime) / paEntry->lData) + 1) 
				            * paEntry->lData;
				lNextTime += lMidnight + paEntry->lTime;
				//If interval past end of day, set to next day
				if (lNextTime > lMidnight + 86400)
					lNextTime = lMidnight + 86400 + paEntry->lTime;
			}
			break;
			
		case SCH_WEEK:
			//Compute mask to determine starting day of search
			//then call common code
			lNextTime = lMidnight + WeekMonthHelp (lValue, paEntry->lTime, lToday, 
			                                   0x80 >> ptime->tm_wday, 0x80, 7);
			break;
			
		case SCH_MONTH:
			//Compute mask to determine starting day of search
			//then call common code
			lNextTime = lMidnight + WeekMonthHelp (lValue, paEntry->lTime, lToday, 
			                                   0x80000000 >> ptime->tm_mday - 1,
			                                   0x80000000, 31);
			break;
	}
	return lNextTime;
}	
//------------------------------------------------------------
// StartTimer - Starts a timer and syncs it to the next minute
//------------------------------------------------------------
void StartTimer () {
	UINT i;
	LONG lTime;
	TM *ptime;

 	//Compute the number of seconds to the next minute
 	time(&lTime);
 	ptime = localtime (&lTime);
 	i = (UINT)(60 - ptime->tm_sec) * 1000;
 	hTimer = SetTimer (hMain, ID_TIMER2, i, NULL);
 	if (hTimer == 0)
 		MessageBox (hMain, szTitleText, "No Timers Free", MB_OK | MB_ICONSTOP);
	return;		
}
//------------------------------------------------------------
// SetButtons - Enables and disables the buttons depending on
// the current state of the window.
//------------------------------------------------------------ 
void SetButtons (HWND hWnd) {
	if (SendDlgItemMessage (hWnd, IDD_PROGLIST, LB_GETCURSEL, 0, 0)
	    == LB_ERR) {
		EnableWindow (GetDlgItem (hWnd, IDD_DEL), FALSE);
		EnableWindow (GetDlgItem (hWnd, IDD_EDIT), FALSE);
	} else {
		EnableWindow (GetDlgItem (hWnd, IDD_DEL), TRUE);
		EnableWindow (GetDlgItem (hWnd, IDD_EDIT), TRUE);
	}
	if (sEntryCnt == MAXENTRIES)
		EnableWindow (GetDlgItem (hWnd, IDD_ADD), FALSE);
	else		
		EnableWindow (GetDlgItem (hWnd, IDD_ADD), TRUE);
	return;		
}		
//------------------------------------------------------------
// AddLBEntry - Adds an item to the listbox
//------------------------------------------------------------ 
void AddLBEntry (HWND hWnd, PSCHENTRY paPtr) {
	char szTemp[MAXCMDLINELEN+33];
	char *pszTemp;

	DispTimeText (paPtr->fFlags, szTemp);
	strcat (szTemp, "\t");
	pszTemp = strrchr (paPtr->szCmd, '\\');
	if (pszTemp == 0)
		pszTemp = paPtr->szCmd;
	else	
		pszTemp++;
	strcat (szTemp, pszTemp);

	SendDlgItemMessage (hWnd, IDD_PROGLIST, LB_ADDSTRING, 0, 
	                    (LPARAM) (LPSTR) szTemp);
	return;
}	
//------------------------------------------------------------
// FillLB - Clears, then refills the hot key listbox
//------------------------------------------------------------ 
void FillLB (HWND hWnd) {
	INT i, sTab[3];
	PSCHENTRY paPtr;
	
	SendDlgItemMessage (hWnd, IDD_PROGLIST, LB_RESETCONTENT, 0, 0);
	SendDlgItemMessage (hWnd, IDD_PROGLIST, LB_SETHORIZONTALEXTENT, 
  		                 1000, 0L);
	sTab[0] = 50;
	sTab[1] = 175;
	sTab[2] = 0;
	SendDlgItemMessage (hWnd, IDD_PROGLIST, LB_SETTABSTOPS, 2, 
	                    (LPARAM)(LPINT)sTab);
	paPtr = paArray;
	for (i = 0; i < sEntryCnt; i++) {
		AddLBEntry (hWnd, paPtr);
		paPtr++;
	}
	SetButtons (hWnd);
	//Start a timer if necessary.
	if ((hTimer == 0) && sEntryCnt)
		StartTimer();
	return;
}	
//============================================================  
// General Helper Routines 
//============================================================  
//------------------------------------------------------------
// PrintError - Displays a message box with an error code
//------------------------------------------------------------ 
void PrintError (HWND hWnd, INT rc) {
	char szErrStr[80];
	char szTemp[12];
	
	if (rc > 0)
	   rc += ERR_LAUNCH;
	else 
	   rc = abs (rc);
	if (LoadString (hInst, rc, szErrStr, sizeof (szErrStr)) == 0) {
		itoa (rc, szTemp, 10);
		strcpy (szErrStr, "Error number: ");
		strcat (szErrStr, szTemp);
	}
	MessageBox (hWnd, szErrStr, szTitleText, MB_OK | MB_ICONHAND);
	return;
}
//------------------------------------------------------------
// MyDisplayDialog - Display a dialog box
//------------------------------------------------------------ 
INT MyDisplayDialog (HINSTANCE hInstance, LPCSTR szDlgName,
                     HWND hWnd, DLGPROC lpDialogProc, 
                     LPARAM lParam) {
    DLGPROC lpDlgProcInst;
    INT		rc;

    lpDlgProcInst = MakeProcInstance((FARPROC)lpDialogProc, hInstance);
    rc = DialogBoxParam (hInstance, szDlgName, hWnd, 
                         lpDlgProcInst, lParam);
    FreeProcInstance((FARPROC)lpDlgProcInst);
    return rc;                              
}
//------------------------------------------------------------
// MyWritePrivateProfileInt - Writes an integer to the profile
//------------------------------------------------------------
BOOL MyWritePrivateProfileInt (char *szSec, char *szEntry, 
                               int Num, char *szProfile) {
	char	szStr[33];
	                           
	itoa (Num, szStr, 10);
	return WritePrivateProfileString (szSec, szEntry, szStr, 
	                                  szProfile);
}
//------------------------------------------------------------
// MyWritePrivateProfileLong - Writes a long to the profile
//------------------------------------------------------------
BOOL MyWritePrivateProfileLong (char *szSec, char *szEntry, 
                                LONG Num,char *szProfile) {
	char	szStr[33];
	                           
	ltoa (Num, szStr, 10);
	return WritePrivateProfileString (szSec, szEntry, szStr, 
	                                  szProfile);
}
//------------------------------------------------------------
// MyGetPrivateProfileLong - Reads an long value from the profile
//------------------------------------------------------------
LONG MyGetPrivateProfileLong (char *szSec, char *szEntry, LONG Default,
                              char *szProfile) {
	char	szStr[33];
	 
	GetPrivateProfileString (szSec, szEntry, "----", szStr, 
	                         sizeof (szStr), szProfile);
	if (strcmp (szStr, "----") == 0)
		return Default;
	                           
	return atol (szStr);
}
//------------------------------------------------------------
// MyGetFilename - Uses the common File Open dialog box to
// query the user for a filename
//------------------------------------------------------------
BOOL MyGetFilename (HWND hWnd, char *szFilename, INT sMaxLen, char *szFilter,
                    INT sFilterIndex) {

   OPENFILENAME ofn;

   *szFilename = '\0';

	ofn.lStructSize = sizeof (OPENFILENAME);
	ofn.hwndOwner = hWnd;
	ofn.hInstance = hInst;
	ofn.lpstrFilter = szFilter;
	ofn.lpstrCustomFilter = 0;
	ofn.nMaxCustFilter = 0;
	ofn.nFilterIndex = sFilterIndex;
	ofn.lpstrFile = szFilename;
	ofn.nMaxFile = sMaxLen;
	ofn.lpstrFileTitle = 0;
	ofn.nMaxFileTitle = 0;
	ofn.lpstrInitialDir = 0;
	ofn.lpstrTitle = 0;
	ofn.Flags = OFN_FILEMUSTEXIST;
	ofn.nFileOffset = 0;
	ofn.nFileExtension = 0;
	ofn.lpstrDefExt = "WBT";
	ofn.lCustData = 0;
	ofn.lpfnHook = 0;
	ofn.lpTemplateName = 0;

 	return (GetOpenFileName (&ofn));
}
//------------------------------------------------------------
// MySubClassWindow - Subclasses a window 
//------------------------------------------------------------
WNDPROC MySubClassWindow (HWND hWnd, WNDPROC lpfnNewProc) {
   WNDPROC lpfnOldProc;

	lpfnOldProc = (WNDPROC) GetWindowLong (hWnd, GWL_WNDPROC);
	SetWindowLong (hWnd, GWL_WNDPROC, (LONG) lpfnNewProc);
	return lpfnOldProc;				               
}				               
