// ******************************************************************** //
//                                                                      //
//  QuikCopy 1.00  Copyright 1994, Michael Holmes and Bob Flanders    //
//  First published in PC Magazine, August 1994  1 Pass Disk Copy      //
//                                                                      //
//  QuikCopy is a one pass disk copy program.  QuikCopy utilizes EMS    //
//  or XMS memory to hold a copy of the disk image while making from    //
//  one to n disk copies.                                               //
//                                                                      //
//  usage:  QuikCopy  source_drive:  [target_drive:]  /1 /A             //
//                                                                      //
//  switches: /1  make only one copy                                    //
//            /A  copy all sectors                                      //
//                                                                      //
// ******************************************************************** //
//                                                                      //
//  Compile QuikCopy using the following vendor specific commands.      //
//                                                                      //
//    Borland C++ version 3.1                                           //
//      BCC -O2-i -mc QuikCopy.cpp                                      //
//                                                                      //
//    Microsoft C/C++ version 7.0                                       //
//                                                                      //
//                                                                      //
// ******************************************************************** //
//                                                                      //
//  Modifications Log                                                   //
//                                                                      //
//  Versn    Date    Author           Description                       //
//  ----- ---------  ---------------  --------------------------------  //
//   0.90 18 May 94  Holmes/Flanders  Initial release beta release      //
//   1.00 12 Jul 94  Holmes/Flanders  Released version                  //
//                                                                      //
// ******************************************************************** //

// ******************************************************************** //
// $lgb$
// 0.1      01/21/94      MDH      working
// $lge$
// ******************************************************************** //
// $nokeywords$


#pragma  pack(1)                            // pack to byte alignment
#include <stdio.h>                          // standard i/o library
#include <stdarg.h>                         // variable argument list
#include <string.h>                         // string functions
#include <stdlib.h>                         // ANSI compatibility
#include <dos.h>                            // DOS rtn definitions
#include <conio.h>                          // console i/o routines
#include <bios.h>                           // bios functions
#include <io.h>                             // i/o routine headers
#include <direct.h>                         // directory functions
#include <fcntl.h>                          // file control header
#include <malloc.h>                         // memory declarations
#include <alloc.h>                          // allocation declarations
#include <memory.h>                         // memory functions
#include <stdarg.h>                         // argument functions

#if defined(__TURBOC__) || defined(__BORLANDC__)    // BC++ 3.1
#define  _BC_VER                            // Borland compiler shorthand
#include <ctype.h>                          // character routines
#include <dir.h>                            // directory routines
#include <sys\stat.h>                       // ..attribute bits
#define INT_PARMS UINT bp, UINT di,         /* interrupt calling conv  */\
            UINT si, UINT ds, UINT es, UINT dx, UINT cx,                 \
            UINT bx, UINT ax, UINT cs, UINT ip, UINT flags
#endif


#if defined(_MSC_VER)                       // MSC C/C++ 7
#include <sys\types.h>                      // system types
#include <sys\stat.h>                       // ..attribute bits
#define findfirst(b,s,a)    _dos_findfirst(b,a,s)   // call compatibility
#define farmalloc(x)        _fmalloc(x)
#define farfree(x)          _ffree(x)
#define fnsplit(a,b,c,d,e)  _splitpath(a,b,c,d,e)
#define ctrlbrk(c)                          // fnc not available
#define INT_PARMS UINT es, UINT ds,         /* interrupt calling conv  */\
            UINT di, UINT si, UINT bp, UINT sp, UINT bx,                 \
            UINT dx, UINT cx, UINT ax, UINT ip, UINT cs, UINT flags
#define MK_FP(s,o)                                                       \
          (void far *)                                                   \
              (((long) s << 16) | (long) o) /* make a far pointer       */
#endif


#define COUNT(x) (sizeof(x) / sizeof(x[0])) // item count
#define SS(x)       sizeof(struct x)        // sizeof structure
#define NOT         !                       // shorthand logical
#define CLEAR(s,c)  memset(s,c,sizeof(s))   // string clear
#define UINT        unsigned int            // unsigned integer
#define BYTE        unsigned char           // ..and unsigned character
#define ULONG       unsigned long           // ..and unsigned long
#define HUGE        char huge               // huge character type
#define UHUGE       unsigned char huge      // ..and unsigned huge
#define MAX_PATH    79                      // maximum path length
#define LAST(s)     s[strlen(s) - 1]        // last character in string
#define TRUE        1                       // true value
#define FALSE       0                       // false value
#define DELAY_CNT   36                      // 2 second timer tick delay
#define min(x,y)    ((x) < (y)) ? (x) : (y) // minimum function
#define FLAT(x)     (ULONG)((FP_SEG(x)<<4)+FP_OFF(x))  // flat address
#define ADJUST(x)   x+=(((FLAT(x)+511)&(-512))-FLAT(x))// i/o boundary


/* ******************************************************************** *
 *
 *  Structures
 *
 * ******************************************************************** */

struct boot_rec                         // boot record
    {
    char jmp[3],                            // jump instruction
         oem_name[8];                       // OEM name
    UINT bytes_per_cluster;                 // bytes per sector
    char sector_per_cluster;                // sectors per cluster
    UINT res_sectors;                       // reserved sectors
    char fats;                              // number of fats
    UINT root_dir_ents,                     // nbr of root dir entries
         sectors;                           // total sectors
    char media;                             // media descriptor byte
    UINT fat_sectors,                       // sectors per fat
         sectors_per_track,                 // sectors per track
         heads;                             // number of heads
    long hidden_sectors,                    // hidden sectors
         huge_sectors;                      // sectors if above 32Mb
    };

#define BOOT    struct boot_rec             // shorthand


struct drive_info                       // all drive information
    {
    UINT   drive,                           // DOS drive number (1 = A:)
           heads,                           // disk heads
           tracks,                          // ..cylinders or tracks
           sectors,                         // ..sectors per track
           nclusters,                       // number of clusters
           bad_cluster,                     // bad cluster number
           max_cluster,                     // max cluster number
           bytes_sector,                    // bytes per sector
           sec_cluster,                     // sectors per cluster
           dir_sector,                      // dir entries per sector
           root_length,                     // root dir sectors
           root_sector,                     // 1st root dir sector nbr
           data_sector,                     // 1st data sector number
           avail_clusters,                  // available clusters
           fat_sectors,                     // FAT size in sectors
           fats;                            // nbr file allocation tables
    long   nsectors,                        // number of sectors on drv
           root_size,                       // root dir bytes
           cluster_size,                    // bytes per cluster
           avail_disk,                      // available bytes on disk
           raw_disk,                        // raw disk size in bytes
           fat_size;                        // size of FAT in bytes
    char   fat_16,                          // 16 bit FAT entry flag
           drive_type,                      // diskette drive type
                                            //  1 = 360k
                                            //  2 = 1.2mb
                                            //  3 = 1.44mb
                                            //  4 = 2.88mb ??
           disk_size,                       // diskette size type
                                            //  1 = 320k
                                            //  2 = 360k
                                            //  3 = 720k
                                            //  4 = 1.2mb
                                            //  5 = 1.44mb
                                            //  6 = 2.88mb ??
           dos_valid;                       // valid DOS diskette flag
                                            //  -1 = not formatted
                                            //   0 = non-DOS
                                            //   1 = DOS
    HUGE   *i_fat;                          // start of in-core FAT
    HUGE   *i_fatx;                         // ptr to start of allocation
    struct boot_rec huge *i_boot;           // base boot record
    HUGE   *i_bootx;                        // ptr to start of allocation
    } huge *d,                              // source drive info block
      huge *t;                              // target drive info block

#define IDRV    struct drive_info huge      // shorthand


struct device_parms                     // device parms
    {
    char spec_func,                         // special functions flag
         dev_type;                          // device type
    UINT dev_attr,                          // ..and attributes
         cylinders;                         // max tracks
    char media_type;                        // media type
    UINT bytes_per_cluster;                 // bytes per sector
    char sector_per_cluster;                // sectors per cluster
    UINT res_sectors;                       // reserved sectors
    char fats;                              // number of fats
    UINT root_dir_ents,                     // nbr of root dir entries
         sectors;                           // total sectors
    char media;                             // media descriptor byte
    UINT fat_sectors,                       // sectors per fat
         sectors_per_track,                 // sectors per track
         heads;                             // number of heads
    long hidden_sectors,                    // hidden sectors
         huge_sectors;                      // sectors if above 32Mb
    char disk_layout[10*18];                // track layout
    } dp;


struct  format_block                        // format track structure
    {
    BYTE track,                                 // track  (0 based)
         head,                                  // head   (0 based)
         sector,                                // sector (1 based)
         size;                                  // size (2 = 512 bytes)
    } fmt[18];


struct  drive_parms
    {
    BYTE   heads,                           // disk heads
           tracks,                          // ..cylinders or tracks
           sectors;                         // ..sectors per track
    } drv_parms[] =                         // test cases
        { { 1, 79, 18 },                        // 3.5"  1.44mb
          { 1, 79, 15 },                        // 5.25" 1.2mb
          { 1, 79,  9 },                        // either 720kb
          { 1, 39,  9 },                        // 5.25" 360kb
          { 1, 39,  8 },                        // 5.25" 320kb
          { 0, 39,  9 },                        // 5.25" 180kb
          { 0, 39,  8 }                         // 5.25" 160kb
        };


struct dos_i25                          // dos interrupt 25/26 block
    {
    long sector;                            // sector to read
    int  num_secs;                          // number of sectors to read
    HUGE *read_addr;                        // address of input area
    };


struct ioctl_read                       // IOCTL read/write control block
    {
    char reserved;                          // set to zero
    UINT head,                              // head number
         track,                             // ..track
         sector,                            // ..sector
         count;                             // copy sector count
    HUGE *buf;                              // disk buffer
    } io_rw;


struct media_id                         // media id block
    {
    int  info_level,                        // information level
         serial1,                           // serial number, part I
         serial2;                           // ..part II
    char vol_id[11],                        // volume label
         file_sys[8];                       // file system id
    };


struct dir_entry                        // directory entry
    {
    char d_efile[8],                        // filename
         d_eext[3],                         // ..and extension
         d_eattr,                           // attribute
         d_eres[10];                        // reserved
    UINT d_etime,                           // file time
         d_edate,                           // ..and date
         d_ecluster;                        // 1st cluster number
    long d_esize;                           // file size
    };


struct copy_entry                       // sectors to copy entry
    {
    long c_sector,                          // starting sector nbr
         c_length,                          // ..length in bytes
         c_ems;                             // EMS record nbr or offset
    } huge *c_table;


struct copy_block                       // XMS copy block
    {
    long count;                             // copy length
    int  from_handle;                       // from handle (0 = seg:off)
    long from_offset;                       // ..and offset
    int  to_handle;                         // to handle   (0 = seg:off)
    long to_offset;                         // ..and offset
    };



/* ******************************************************************** *
 *
 *  Command Line Parms and Switches
 *
 * ******************************************************************** */

char    sw_one,                             // one copy switch
        sw_all;                             // all sector copy

struct  cmd_parm                            // command line parms
    {
    char cp_ltr,                            // switch letter
         cp_flag,                           // entry type
         *cp_entry;                         // pointer to data
    } parm_table[] =
        {
        { '1', 0, &sw_one },                // 1 - one copy switch
        { 'A', 0, &sw_all }                 // A - copy all sectors
        };

#define PARM_TABLE_CNT  COUNT(parm_table)   // nbr of table entries



/* ******************************************************************** *
 *
 *  Globals
 *
 * ******************************************************************** */

int     pos_found,                          // number positionals found
        rc = 1,                             // errorlevel return code
        enabled,                            // update msg enable flag
        (far *xms_driver)(),                // XMS driver address
        break_request;                      // ^break in critical rtn

UINT    copy_entries = 1,                   // c_table entries available
        copied,                             // diskette copies made
        retries,                            // diskette i/o reties
        abort_error,                        // abort on error
        ems_segment,                        // EMS page frame segment
        ems_handle;                         // EMS handle

ULONG   i_remain,                           // initial..
        remaining,                          // remaining to be moved
        moved_count,                        // moved sector count
        format_count,                       // formatted sector cound
        read_time,                          // reading time in ticks
        move_time,                          // ..coping time
        format_time,                        // ..format time
        a_mem,                              // avail memory for buffer
        ems_required;                       // amt of extended memory

char  **pos_parms,                          // positional parms array
        s_drv,                              // source drive nbr (1 = A)
        t_drv,                              // target drive nbr (1 = A)
        same_drive,                         // source and target are same
        ems_flag,                           // EMS memory available/in-use
                                            //  ...1 XMS available
                                            //  ..1. EMS available
        huge *disk_buf,                     // disk buffer
       *update_msg,                         // current phase description
        block_1[17],                        // string of completed blks
        block_2[17],                        // ..and not done blocks
        block_3[5] =                        // ..and almost done blocks
            { 178, 177, 176, 250, 219 },
        bits[] =                            // bits definitions
            { '\x80', '\x40', '\x20', '\x10',
              '\x08', '\x04', '\x02', '\x01' };

void    interrupt (*old_1e)(...),           // old and new int 1E pointer
        interrupt (*new_1e)(...);           // .. disk parameter table

struct  SREGS s;                            // segment registers
union   REGS  r,                            // ..and general registers
              ro;                           // ..and general output regs

void    ems_free(void);                     // routine definition


/* ******************************************************************** *
 *      messages and strings                                            *
 * ******************************************************************** */

char    copyright[]    = "QuikCopy 1.00  Copyright (c) 1994, "
                         "Michael Holmes and Bob Flanders\n"
                         "First published in PC Magazine, "
                         "August 1994  1 Pass Disk Copy\n\n",
        too_many[]     = "Too many arguments\n",
        bad_op[]       = "Invalid argument %s\n",
        stop_here[]    = "\nStopping at user's request\n",
        net_drive[]    = "Cannot use a network, remote, SUBST'd "
                         "or ASSIGN'd drive\n",
        drv_wont[]     = "Target drive won't handle this format\n",
        disk_wont[]    = "Target diskette won't format\n",
        not_ready_s[]  = "Insert source diskette in drive %c: or press ESC\r",
        no_mem[]       = "Insufficient memory available for processing\n",
        no_ems[]       = "No EMS/XMS memory available\n",
        no_ems_space[] = "Insufficient EMS/XMS memory "
                         "available for processing\n",
        not_available[]= "Can't copy: Mismatched drive capacities\n",
        dos_error[]    = "Must run with DOS 3.3 or greater\n",
        non_removable[]= "Drive %c: does not use removable media\n",
        bad_drive[]    = "Invalid drive specified: %s\n",
        fat_error[]    = "Error reading the File Allocation Table\n",
        fat_w_error[]  = "\nError writing the File Allocation Table\n",
        unreadable[]   = "Source diskette unreadble\n",
        bad_target1[]  = "Target diskette is not same size as source\n",
        bad_target2[]  = "Target diskette unusable due to bad clusters\n",
        boot_error[]   = "\nError reading the boot record\n",
        boot_w_error[] = "\nError writing the boot record\n",
        ems_error[]    = "\nError accessing EMS memory\n",
        read_error[]   = "\nError reading the source disk\n",
        write_error[]  = "Error writing the target disk\n",
        fat_write_err[]= "\nError writing the FAT\n",
        checking_disk[]= "Determining diskette type ..\r",
        serial_nbr[]   = "Serial: %04X-%04X",
        disk_label[]   = "Label: %s  ",
        drive_info[]   = "     Drive %c: [DOS] Copying used clusters\n"
                         "   Volume Id: %s\n"
                         "   Disk Info: %lu Sectors, %u Clusters, "
                         "%lu Clustersize\n"
                         "    Capacity: %s, %s used, %s available\n",
        disk_info[]    = "     Drive %c: [Non-DOS] Copying all sectors\n"
                         "   Disk Info: %u Heads, %u Tracks, "
                         "%u Sectors/Track (%s)\n",
        requirements[] = "     Storing: %lu sectors using %s of %s memory\n",
        working[]      = "\r      Status: %s  Cyl %d, Head %d  %4.1f%% %s\r",
        clean_up[]     = "%79.79s\r",
        change_disk[]  = "%79.79s\rRemove disk from drive %c: "
                         "(%u copies made)\r",
        next_disk[]    = "%79.79s\rInsert next target diskette "
                         "in drive %c: or press ESC to cancel\r",
        read_source[]  = "Reading",
        write_target[] = "Writing",
        format_target[]= "Formatting",
        format_timing[]= ", %s/sec formatting",
        all_done[]     = "    Finished: %lu total sectors written\n"
                         "      Timing: %s/sec reading, %s/sec writing%s\n"
                         "      Copies: %u made using drive %c: \n",
        nothing[]      = "    Finished: Nothing copied\n",
        help[]         = "     Usage:  QuikCopy  source-drive [target-drive] [/1] [/A]\n\n"
                         "   Options:  source-drive is drive used to read source diskette\n"
                         "             target-drive is drive used to write diskette copies\n"
                         "             /1           bypasses next diskette prompt\n"
                         "             /A           is used to force a copy of all sectors\n";



/* ******************************************************************** *
 *
 *  get_key() -- get a key (including function keys)
 *
 * ******************************************************************** */

int     get_key(void)
{
int     k = 0;                              // local key variable


if (kbhit())                                // q. key available?
    {                                       // a. yes .. process it
    if (NOT (k = getch()))                  // q. function key?
        k = 0x100 + getch();                // a. yes .. special key
    }

return(k);                                  // return w/key if available

}



/* ******************************************************************** *
 *
 *  quit_with() -- give an error message, then return to DOS
 *
 * ******************************************************************** */

void    quit_with(char *msg, ...)           // string to print
{
va_list list;                               // variable list


if (ems_handle)                             // q. ems/xms memory allocated?
    ems_free();                             // a. yes .. release it

r.h.ah = 0xd;                               // ah = reset drive
int86(0x21, &r, &r);                        // issue DOS call

va_start(list, msg);                        // set up variable list
vprintf(msg, list);                         // give error message ..
exit(rc);                                   // ..and then quit

}



/* ******************************************************************** *
 *
 *  drive_check() -- check version and drive being local
 *
 * ******************************************************************** */

int     drive_check(char *d)                // drive string
{
int     dn;                                 // drive number
char    s[5];                               // work string


if ((strlen(d) != 2) || d[1] != ':')        // q. valid size and shape?
    quit_with(bad_drive, d);                // else .. give syntax err

*d = toupper(*d);                           // uppercase drive letter
sprintf(s, "%c:\\", *d);                    // format temp string
dn = *d - '@';                              // ..convert to drive number

r.x.ax = 0x4409;                            // ah = ioctl, local test
r.h.bl = dn;                                // bl = drive to test
int86(0x21, &r, &r);                        // test drive

if (r.x.cflag)                              // q. bad drive?
    quit_with(bad_drive, s);                // a. yes .. error

if (r.x.dx & 0x9200)                        // q. network/remote/subst?
    quit_with(net_drive);                   // a. yes .. error

r.x.ax = 0x4408;                            // ax = ioctl, removable media
r.x.bx = dn;                                // bx = drive number
int86(0x21, &r, &r);                        // issue DOS call

if (r.x.ax)                                 // q. removable media?
    quit_with(non_removable, *d);           // a. no .. quit w/error msg

return(dn);                                 // return with drive number

}



/* ******************************************************************** *
 *
 *  drive_ready() -- check if drive ready
 *
 *  return: 0 = drive ready
 *          x = error code (0x80 = drive not ready)
 *
 * ******************************************************************** */

int     drive_ready(int d,                  // drive nbr to check
                    int loop)               // loop counter
{
char    buffer[512];                        // work buffer
int     i;                                  // loop control


r.x.ax = 0x0201;                            // read one sector
r.x.dx = d - 1;                             // head 0, drive x
r.x.cx = 1;                                 // track 0, sector 1
s.es = FP_SEG(buffer);                      // ..buffer segment
r.x.bx = FP_OFF(buffer);                    // ..and offset addresses

for (i = loop; i--;)                        // loop a little
    {
    int86x(0x13, &r, &ro, &s);              // ..doing disk i/o

    if (NOT ro.x.cflag)                     // q. read ok?
        return(0);                          // a. yes .. return w/ok code

    if (ro.h.ah == 6)                       // q. disk change?
        {
        i++;                                // a. yes .. try again
        continue;                           // ..at the top to loop
        }
    }

return(ro.h.ah);                            // return w/completion code

}



/* ******************************************************************** *
 *
 *  check_source_available() -- make source diskette available
 *
 * ******************************************************************** */

void    check_source_available(int d)       // drive info block
{
long    far *timer = (long far *)           // BIOS timer tick counter
                MK_FP(0x40, 0x6c),
        wait_til;                           // wait until tick counter


if (drive_ready(d, 2) == 0x80)              // q. drive ready?
    {
    printf(not_ready_s, d + '@');           // a. no .. give warning msg
    while(drive_ready(d, 1) == 0x80)        // ..and wait for user
        {
        wait_til = *timer + 18;             // set up for 1 sec delay

        while (wait_til > *timer)           // q. time elapsed?
            if (get_key() == '\x1b')        // q. escape key pressed?
                {
                printf(clean_up, "");       // clean up messages and
                quit_with(stop_here);       // a. yes .. exit
                }
        }

    printf(clean_up, "");                   // clean up after prompt
    }
}



/* ******************************************************************** *
 *
 *  disk_change() -- handle disk change logic
 *
 *  return: TRUE if user pressed ESC key
 *
 * ******************************************************************** */

int     disk_change(void)
{
int     rc;                                 // return code from drive_ready
long    far *timer = (long far *)           // BIOS timer tick counter
                MK_FP(0x40, 0x6c),
        wait_til;                           // wait until tick counter


if (same_drive)                             // q. using same drive again?
    {                                       // a. yes .. handle disk change
    printf(change_disk, "",                 // prompt for new target
             t_drv + '@', copied);

    while (drive_ready(t_drv, 1) != 0x80)   // wait until user pulls diskette
        if (get_key() == '\x1b')            // q. escape key pressed?
            return(TRUE);                   // a. yes .. exit

    printf(next_disk, "", t_drv + '@');     // prompt for new target

    for (;;)                                // wait for a new diskette
        {
        if ((rc = drive_ready(t_drv, 1))    // q. drive not ready?
                != 0x80 && rc != 0x06)      // ..and not changed?
            break;                          // a. yes .. exit loop

        wait_til = *timer + 18;             // set up for 1 sec delay

        while (wait_til > *timer)           // q. time elapsed?
            if (get_key() == '\x1b')        // q. escape key pressed?
                {
                printf(clean_up, "");       // clean up messages and
                return(TRUE);               // a. yes .. exit
                }
        }
    }
 else
    same_drive = 1;                         // set flag for next time

printf(clean_up, "");                       // clean up messages and
return(FALSE);                              // ..finally, return all ok

}



/* ******************************************************************** *
 *
 *  clear_memory() -- clear an area of memory
 *
 * ******************************************************************** */

void    clear_memory(HUGE *p,               // address of memory block
                     long n)                // length of block
{
UINT    clr_size = 0;                       // working chunk size


for (; n; n -= clr_size)                    // clear in big chunks
    {
    if (n > (65536L - 16))                  // q. more than 64k to do?
        clr_size = (UINT) 65536L - 16;      // a. yes .. just do some
     else
        clr_size = (UINT) n;                // else .. do what's left

    _fmemset(p, 0, (UINT) clr_size);        // clear to block to null
    p += clr_size;                          // point to next block
    }
}



/* ******************************************************************** *
 *
 *  malloc_chk() -- allocate memory with error processing
 *
 * ******************************************************************** */

void   *malloc_chk(UINT n)                  // size of block
{
void   *s;                                  // temporary pointers


if (NOT (s = malloc(n)))                    // q. enough memory?
    quit_with(no_mem);                      // a. no .. give error msg

clear_memory((HUGE *) s, (long) n);         // else .. clear to nulls
return(s);                                  // ..and return w/address

}



/* ******************************************************************** *
 *
 *  huge_malloc() -- local malloc w/error handling
 *
 * ******************************************************************** */

void    *huge_malloc(long size)             // amount of memory to get
{
void    *p;                                 // temporary pointer


if (NOT (p = (void *) _fmalloc(size)))      // q. enough memory?
    quit_with(no_mem, size);                // a. no .. give error msg

return(p);                                  // else .. return w/address

}



/* ******************************************************************** *
 *
 *  read_label() -- get the volume 's label, if available
 *
 * ******************************************************************** */

char    *read_label(int drive)              // drive to check
{
char    *p, *q;                             // work pointers
struct  find_t dir;                         // directory entry
struct  media_id media;                     // media id block
char    drv[12];                            // drive string
static
char    work[43];                           // return string

sprintf(drv, "%c:\\*.*",                    // set up for search string
        (char) (drive + '@'));

if (_osmajor == 2)                          // q. DOS 2.x?
   return("");                              // a. yes .. just return

if (_dos_findfirst(drv, _A_VOLID, &dir))    // q. error on label get?
    strcpy(drv, "None");                  // a. yes .. then no label

 else
    {
    for(p = drv, q = dir.name; *q; q++)     // copy name w/o middle dot
        if (*q != '.')                      // q. is this char a dot?
            *p++ = *q;                      // a. no .. copy it

    *p = 0;                                 // make null terminated and
    }

sprintf(work, disk_label, drv);             // set up label information

if (_osmajor >= 4)                          // q. serial nbr available?
    {
    media.info_level = 0;                   // a. maybe .. set up call
    r.x.bx = drive;                         // set up drive number..
    r.x.cx = 0x0866;                        // ..sub function code
    s.ds   = FP_SEG(&media);                // ..work area segment
    r.x.dx = FP_OFF(&media);                // ..and offset pointers
    r.x.ax = 0x440d;                        // ..lastly function code
    int86x(0x21, &r, &r, &s);               // issue dos call

    if (NOT r.x.cflag)                      // q. complete ok?
        sprintf(&work[strlen(work)],        // a. yes .. format into msg
            serial_nbr, media.serial2,
            media.serial1);
    }

return(work);                               // ..return string

}



/* ******************************************************************** *
 *
 *  large_fmt() -- handle formatting of disk storage numbers
 *
 * ******************************************************************** */

char    *large_fmt(ULONG nbr,               // number to be converted
                   char *s)                 // destination string or 0
{
int     g_flag = 0;                         // gigabyte range flag
static
char    work[10];                           // return string


if (s == 0)                                 // q. sent a null pointer?
    s = work;                               // a. yes .. use our own

if (nbr < 1000)                             // q. less than 1k?
    sprintf(s, "%u bytes", nbr);            // a. yes .. then use bytes

 else
    {
    nbr /= 1024;                            // make number in kilobytes

    if (nbr < 1000)                         // q. only in the kb range?
        sprintf(s, "%ukb", nbr);            // a. yes .. format as kb

     else
        {
        nbr /= 10;                          // get mb or gb w/2 decimals

        if (nbr > 100000L)                  // q. gigabyte range?
            {
            g_flag++;                       // a. yes .. show as gb
            nbr /= 1000;                    // ..and make in gigabytes
            }

        sprintf(s, "%lu.%lu%s",             // format number
            nbr / 100, nbr % 100,
            g_flag ? "gb" : "mb");
        }
    }

return(s);                                  // then give user the result

}



/* ******************************************************************** *
 *
 *  graph_string() -- build graph string to show progress
 *
 * ******************************************************************** */

char   *graph_string(long w,                // number to show
                     long max)              // max to graph n against
{
UINT    n,                                  // not done count of blocks
        d;                                  // done count of blocks
char    p;                                  // work pointer
static
char    mask[] = "%-*.*s%c%*.*s",           // format string
        s[17];                              // return work area


w <<= 6;                                    // scale number up by 64
d = (int) (w / max);                        // determine 64ths not done
p = block_3[w ? d & 3 : 4];                 // pick up last block type
d >>= 2;                                    // scale back to 16ths
d = min(d, 15);                             // limit length of string
n = 15 - d;                                 // determine len of not done

sprintf(s, mask, d, d, block_1,             // build crude graph to
            p, n, n, block_2);              // ..show relative progress

return(s);                                  // finally return the string

}



/* ******************************************************************** *
 *
 *  update_progress() -- update progress of run
 *
 * ******************************************************************** */

void    update_progress(void)
{
long    moved;                              // total moved
float   pct;                                // percentage complete
static
UINT    b_call = 0,                         // raw counter
        far *timer = (UINT far *)           // BIOS timer tick counter
                MK_FP(0x40, 0x6c);

if (NOT enabled)                            // q. update msg enabled?
    return;                                 // a. no .. just return

b_call = *timer + 9;                        // set next timeout value
moved = i_remain - remaining;               // total clusters moved
pct = ((float) moved / i_remain) * 100.;    // calculate percentage done

printf(working, update_msg,                 // display status message
        io_rw.track, io_rw.head,            // ..with cyl, head, %complete,
        pct, graph_string(moved, i_remain));// ..and a little graph

}


/* ******************************************************************** *
 *
 *  get_drive_parms() - get current drive parms
 *
 * ******************************************************************** */

void    get_drive_parms(void)               // drive info block
{
int     i;                                  // working tracks
union   REGS ro;                            // output registers

dp.spec_func = 1;                           // get current status
r.x.ax = 0x440d;                            // ax = ioctl function
r.x.bx = s_drv;                             // bx = drive
r.x.cx = 0x0860;                            // cx = get device parms
r.x.dx = FP_OFF(&dp);                       // dx -> device parm block
s.ds = FP_SEG(&dp);                         // ds:dx -> ...
int86x(0x21, &r, &ro, &s);                  // get device parm block

dp.spec_func = 4;                           // set up sectors same size
r.x.cx = 0x0840;                            // cx = get device parms
int86x(0x21, &r, &r, &s);                   // set device parm block

                                            // check target capacity
old_1e = _dos_getvect(0x1e);                // save old dpt pointer

r.h.ah = 0x18;                              // get disk parameter entry
r.h.ch = i = dp.cylinders - 1;              // low 8 bits of track count
r.h.cl = dp.sectors_per_track               // sector and track counts
        + (((i - 1) >> 8) << 6);
r.x.dx = t_drv - 1;                         // target drive
int86x(0x13, &r, &ro, &s);                  // get dpt entry from BIOS

if (ro.x.cflag)                             // q. supported on device?
    quit_with(not_available);               // a. no .. give error/quit

(void *) new_1e = MK_FP(s.es, ro.x.di);     // save the dpt address then

}



/* ******************************************************************** *
 *
 *  disk_parms() -- set up interrupt 13 registers from sector nbr
 *
 * ******************************************************************** */

void    disk_parms(long sector,             // sector number
                   IDRV *d)                 // drive info block
{

r.h.dl = d->drive - 1;                      // drive number

r.h.ch = io_rw.track = (UINT)(sector /      // cylinder or track
        (d->sectors * d->heads));

r.h.cl =                                    // sector number
        (UINT)(sector % d->sectors) + 1;

io_rw.sector = r.h.cl - 1;                  // ..for IOCTL

r.h.dh = io_rw.head =                       // head number
        (UINT)((sector / d->sectors) % d->heads);

}



/* ******************************************************************** *
 *
 *  disk_io() -- absolute disk read/write by sector number
 *
 * ******************************************************************** */

int     disk_io(int  rw_flag,               // read/write flag,
                                            //   0=read, 1=write
                long start,                 // starting sector number
                UINT count,                 // count of sectors
                HUGE *buffer,               // disk buffer
                IDRV *d)                    // drive info block
{
UINT    i;                                  // loop control
ULONG   seg_max;                            // max left in segment


while (count)                               // loop until all read|written
    {
    disk_parms(start, d);                   // set up head/cyl/sector nbrs
    update_progress();                      // update user with status
    io_rw.buf = buffer;                     // ..and buffer address

    r.x.ax = 0x440d;                        // ax = IOCTL function
    r.x.bx = d->drive;                      // bx = drive number
    r.x.cx = rw_flag ? 0x0841 : 0x0861;     // cx = read/write function

    io_rw.count = (count > d->sectors)      // sector count
            ? d->sectors : count;

    if ((io_rw.count + io_rw.sector)        // q. more than one track?
            > d->sectors)
        io_rw.count = d->sectors -          // a. yes .. limit to one track
                io_rw.sector;

    seg_max = 0x10000L - (FLAT(buffer)      // determine the max i/o
            & 0xffff);                      // ..amount left in 64k segment

    if (((long) io_rw.count *               // q. going to span
            d->bytes_sector) > seg_max)     // ..the 64k segment line?
        io_rw.count = (UINT) (seg_max /     // a. yes .. set to max
                d->bytes_sector);

    r.x.dx = FP_OFF(&io_rw);                // dx -> io control block
    s.ds = FP_SEG(&io_rw);                  // ds:dx -> io block

    int86x(0x21, &r, &ro, &s);              // issue DOS call

    if (ro.x.cflag)                         // q. copy anything?
        break;                              // a. no .. exit loop

    i = io_rw.count;                        // get transfer count
    count -= i;                             // decrement remaining count
    remaining -= i;                         // update remaining counter
    start += i;                             // ..adjust next starting point
    buffer += (long) i * d->bytes_sector;   // ..and increment buffer ptr
    }

return(ro.x.cflag);                         // rtn last completion status

}



/* ******************************************************************** *
 *
 *  check_media_byte() -- check for valid media descriptor byte
 *
 *  This routine fills in the physical parameters to the drive info
 *  block, if the diskette is formatted as a DOS diskette.
 *
 *  Returns: 0 = non-DOS diskette
 *           1 = DOS formatted diskette
 *
 * ******************************************************************** */

int     check_media_byte(BYTE media,        // media descriptor byte
                         IDRV *d)           // drive info block
{

d->heads = 2;                               // default heads
d->tracks = 40;                             // ..cylinders or tracks
d->sectors = 9;                             // ..sectors per track
d->dos_valid = 1;                           // set initial state

switch(media)                               // using media id byte
    {
    case 0xf0:                              // 3.5" 1.44mb
        d->sectors = 18;                    // ..set up sectors/track
        d->tracks = 80;                     // ..and overall cylinders
        break;                              // ..and continue

    case 0xf9:                              // 5.25" 1.2mb or 3.5" 720kb
        r.h.ah = 0x36;                      // ah = get freespace
        r.h.dl = d->drive;                  // dl = drive number
        int86(0x21, &r, &r);                // get drive parameters

        if ((ULONG)((UINT) r.x.cx *         // q. bigger than ..
                (ULONG) r.h.al *            // ..the 3.5" 720kb ..
                (UINT) r.x.dx) > 737280L)   // ..floppy?
            d->sectors = 15;                // a. yes .. set up sectors

        d->tracks = 80;                     // but both use 80 cylinders
        break;                              // ..and continue

    case 0xfd:                              // 5.25" 360kb
        break;                              // ..and continue

    case 0xff:                              // 5.25" 320kb
        d->sectors = 8;                     // set up sectors per track
        break;                              // ..and continue

    case 0xfe:                              // 5.25" 160kb
        d->sectors = 8;                     // set up lowest density
                                            // ..then fall thru

    case 0xfc:                              // 5.25" 180kb
        d->heads = 1;                       // set up single sided
        break;                              // ..and continue

    default:                                // unknown type
        d->dos_valid = 0;                   // clear valid flag
    }

return(d->dos_valid);                       // return w/DOS validity flag

}



/* ******************************************************************** *
 *
 *  find_disk_maximums() -- find max head/track/sector
 *
 *  This routine fills in the physical parameters to the drive info
 *  block.
 *
 * ******************************************************************** */

int     find_disk_maximums(IDRV *d)         // drive info block
{
char    buf[512];                           // one sector buffer
int     i, j;                               // loop controls
struct  drive_parms *p = drv_parms;         // work pointer


r.h.dl = d->drive - 1;                      // drive number
s.es = FP_SEG(buf);                         // ..buffer segment
r.x.bx = FP_OFF(buf);                       // ..and offset addresses
r.h.ah = 2;                                 // ah = read function
r.h.al = 1;                                 // al = sector count

for (i = COUNT(drv_parms); i; i--, p++)     // while there is something..
    {
    r.h.dh = p->heads;                      // head number
    r.h.ch = p->tracks;                     // cylinder or track
    r.h.cl = p->sectors;                    // sector number

    for (j = 5, ro.h.ah = 1; j-- && ro.h.ah;)   // loop ..
        int86x(0x13, &r, &ro, &s);              // ..trying to read

    if (ro.x.ax == 1)                       // q. work ok?
        {                                   // a. yes .. save current
        d->tracks = p->tracks + 1;          // save tracks
        d->sectors = p->sectors;            // ..sectors per track
        d->heads = p->heads + 1;            // ..and heads
        break;                              // then exit loop
        }
    }

return(i >= 0);                             // return w/final status

}



/* ******************************************************************** *
 *
 *  get_disk_type() -- get disk physical information
 *
 *  This routine fills in the physical parameters to the drive info
 *  block if the disk has been formatted with DOS.
 *
 *  Returns:
 *    This routine fills in the dos_valid flag in the drive_info block. The
 *    drive_info block contains information essential to QuikCopy about the
 *    geometry of the source and target disks. Included in this block is the
 *    dos_valid flag which this routine sets to one of the following values:
 *
 *        -1 = disk is unformatted
 *         0 = disk is not DOS formatted
 *         1 = disk is DOS formatted
 *
 * ******************************************************************** */

IDRV   *get_disk_type(int drive)            // drive number
{
char    buf[512];                           // one sector buffer
struct  boot_rec b;                         // temp boot record
IDRV   *d;                                  // drive info block


printf(checking_disk);                      // give user some info

d = (IDRV *) huge_malloc(SS(drive_info));   // get memory for drive info
clear_memory((HUGE *) d, SS(drive_info));   // ..and clear to nulls

d->drive = drive;                           // set up bare necessities
d->heads = 2;                               // default heads
d->tracks = 40;                             // ..cylinders or tracks
d->sectors = 9;                             // ..sectors per track
d->bytes_sector = 512;                      // ..temp sector size

r.h.ah = 0;                                 // ah = reset diskette fnc
r.h.dl = d->drive - 1;                      // dl = drive number
int86(0x13, &r, &r);                        // reset disk system first

if (disk_io(0, 0L, 1, (HUGE *) &b, d))      // q. read boot sector ok?
    d->dos_valid = -1;                      // a. no .. format the disk

 else if (sw_all)                           // q. using /All switch?
    {                                       // a. yes .. check hard way
    if (NOT find_disk_maximums(d))          // q. find maximums?
        d->dos_valid = -1;                  // a. no .. needs formatting
    }

 else if (NOT check_media_byte(b.media, d)) // q. valid DOS diskette?
    {}                                      // a. no .. rtn w/flag set

 else if (disk_io(0, b.res_sectors,         // q. read 1st FAT sector ok?
        1, buf, d))
    d->dos_valid = 0;                       // a. no ..show diskette status

 else if (buf[0] != b.media)                // q. same media descriptor?
    d->dos_valid = 0;                       // a. no ..show diskette status

 else if (b.res_sectors != 1)               // q. right reserved sectors?
    d->dos_valid = 0;                       // a. no ..show diskette status

 else if (b.heads < 1 || b.heads > 2)       // q. right nbr of heads?
    d->dos_valid = 0;                       // a. no ..show diskette status

printf(clean_up, "");                       // clean up user info
d->raw_disk = (long) d->heads * d->tracks   // compute max disk size
        * d->sectors * d->bytes_sector;     // ..in bytes

return(d);                                  // then return w/disk info

}



/* ******************************************************************** *
 *
 *  get_fat() -- read boot record and FAT into memory
 *
 *  This routine gathers information about a DOS file system and
 *  returns it in the drive_info block.
 *
 * ******************************************************************** */

void    get_fat(IDRV *d)                    // drive information block
{

r.h.ah = 0x36;                              // ah = get freespace
r.h.dl = d->drive;                          // dl = drive to use
int86(0x21, &r, &r);                        // r.x.cx = bytes/sector

d->avail_clusters = r.x.bx;                 // save nbr of avail clusters
d->i_boot =                                 // get memory for boot rec
        (BOOT huge *) (d->i_bootx =         // ..and room for adjustments
        (HUGE *) huge_malloc((              // ..saving address for freeing
        d->bytes_sector = r.x.cx) + 511));
ADJUST((HUGE *) d->i_boot);                 // adjust for i/o operations

r.x.cx = 1;                                 // cx = number of sectors
r.x.dx = 0;                                 // dx = starting sector
r.x.bx = FP_OFF(d->i_boot);                 // bx = offset of buffer
s.ds   = FP_SEG(d->i_boot);                 // ds = segment of buffer
r.h.al = d->drive - 1;                      // al = drive number
int86x(0x25, &r, &r, &s);                   // read boot sector

if (r.x.cflag)                              // q. read boot sector ok?
    quit_with(boot_error);                  // a. no .. give err msg

d->fat_sectors = d->i_boot->fat_sectors;    // set up FAT size in sectors
d->fats = d->i_boot->fats;                  // ..nbr of FATs
d->fat_size = d->fat_sectors                // ..and FAT size in bytes
        * d->bytes_sector;

d->nsectors = (d->i_boot->sectors ?         // compute ..
        (long) d->i_boot->sectors :         // ..nbr of sectors on
        d->i_boot->huge_sectors);           // ..logical DOS drive
d->sec_cluster =                            // ..sectors per cluster
         d->i_boot->sector_per_cluster;
d->nclusters = (unsigned)                   // ..nbr of DOS clusters
        ((long)((d->nsectors
        - (d->i_boot->res_sectors
        + (d->fat_sectors * d->fats)
        + ((d->i_boot->root_dir_ents * 32)
        / d->bytes_sector)))
        / d->sec_cluster));

d->fat_16 = d->nclusters > 4086;            // set if 16bit FAT table
d->max_cluster = d->nclusters + 2;          // ..max cluster number
d->bad_cluster = d->fat_16 ? 0xfff7 : 0xff7;// ..and the bad cluster nbr

d->cluster_size = d->sec_cluster *          // ..bytes per cluster
        d->bytes_sector;
d->avail_disk = d->avail_clusters *         // ..available bytes
        d->cluster_size;
d->dir_sector = (UINT) (d->bytes_sector /   // ..dir entries per sector
        SS(dir_entry));

d->root_sector = (UINT)                     // get the sector number
        d->i_boot->res_sectors +            // ..of the root
        (d->fat_sectors * d->fats);         // ..directory sector

d->root_length = d->i_boot->root_dir_ents / // ..and length in sectors
        (d->bytes_sector / SS(dir_entry));
d->root_size = d->root_length *             // ..and length in bytes
        d->bytes_sector;

d->data_sector = d->root_sector +           // sector nbr of cluster two
        d->root_length;

d->i_fat = d->i_fatx = (HUGE *) huge_malloc(// get some memory for
        d->fat_size + 511);                 // ..the file allocation table
ADJUST(d->i_fat);                           // adjust for i/o operations

if (disk_io(0, (long)d->i_boot->res_sectors,// q. read FAT in?
        d->fat_sectors, d->i_fat, d))
    quit_with(fat_error);                   // a. no .. give error msg

}



/* ******************************************************************** *
 *
 *  free_drive_block() -- free drive info block
 *
 * ******************************************************************** */

void    free_drive_block(IDRV *d)           // drive information block
{

if (d)                                      // q. currently alloc'd block?
    {                                       // a. yes .. process it
    if (d->i_bootx)                         // q. boot record available?
        farfree(d->i_bootx);                // a. yes .. free it

    if (d->i_fatx)                          // q. FAT entry available?
        farfree(d->i_fatx);                 // a. yes .. free it

    farfree(d);                             // then delete main block
    }

}



/* ******************************************************************** *
 *
 *  check_cluster() -- check for valid cluster number
 *
 *  return:  0 = invalid cluster nbr
 *           2 = free
 *           3 = allocated
 *           4 = last in chain
 *           5 = bad cluster
 *
 * ******************************************************************** */

int     check_cluster(UINT nc,              // cluster number
                      IDRV *d)              // drive info block
{

if (nc == 1)                                // q. invalid cluster?
    return(0);                              // a. yes .. show in rtn cd

 else if (nc == 0)                          // q. free cluster?
    return(2);                              // a. yes .. mark free

 else if (nc <= d->max_cluster)             // q. allocated cluster?
    return(3);                              // a. yes .. show allocated

 else if (nc == d->bad_cluster)             // q. bad cluster?
    return(5);                              // a. yes .. show status

 else if (nc > d->bad_cluster)              // q. EOF cluster?
    return(4);                              // else .. set up as EOF

 else                                       // else ..
    return(0);                              // ..cluster nbr is invalid

}



/* ******************************************************************** *
 *
 *  get_fat_entry() -- get the FAT entry
 *
 * ******************************************************************** */

UINT    get_fat_entry(UINT n,               // cluster number
                      IDRV *d)              // drive info block
{
UINT    nc;                                 // next cluster number


if (d->fat_16)                              // q. 16 bit FAT entries?
    nc = ((UINT huge *) d->i_fat)[n];       // a. yes .. get FAT entry

 else
    {
    nc = *(UINT huge *)                     // get raw next cluster nbr
            &d->i_fat[((n << 1) + n) >> 1];

    if (n & 1)                              // q. need to do shift?
        nc >>= 4;                           // a. yes .. shift by 4 bits
     else
        nc &= 0x0fff;                       // else .. strip upper bits

    }

return(nc);                                 // return w/next cluster nbr

}



/* ******************************************************************** *
 *
 *  update_fat_entry() -- update a FAT entry
 *
 * ******************************************************************** */

void    update_fat_entry(UINT n,            // cluster number
                         UINT next,         // new entry number
                         IDRV *d)           // drive info block
{
UINT    huge *p;                            // work pointer


if (d->fat_16)                              // q. 16 bit FAT entries?
    {
    p = &((UINT huge *) d->i_fat)[n];       // a. yes .. get FAT addr
    *p = next;                              // store new next pointer
    }

 else
    {
    p = (UINT huge *) &d->i_fat[((n << 1)   // get address of cluster
            + n) >> 1];                     // ..entry requested

    if (n & 1)                              // q. need to do shift?
        {
        *p &= 0x000f;                       // a. yes .. clear old entry
        *p |= (next << 4);                  // ..shift and "or" in value
        }
     else
        {
        *p &= 0xf000;                       // just clear old entry
        *p |= next;                         // ..and "or" in new value
        }
    }
}



/* ******************************************************************** *
 *
 *  cluster_status() -- check for allocated clusters
 *
 *  return: FALSE = non-allocated
 *           TRUE = allocated
 *
 * ******************************************************************** */

int     cluster_status(UINT nc,             // cluster number
                       IDRV *d)             // drive info block
{
int     status;                             // work status


nc = get_fat_entry(nc, d);                  // get contents of FAT entry
return((status = check_cluster(nc, d)) == 3 // return TRUE if allocated
        || status == 4);                    // ..or EOF cluster

}



/* ******************************************************************** *
 *
 *  media_same_check() -- check that both diskettes are same size
 *
 *  returns: FALSE if both are same
 *            TRUE if error condition
 *
 * ******************************************************************** */

int     media_same_check(void)
{

return(((d->heads == t->heads) &&           // same number of heads
        (d->tracks == t->tracks) &&         // ..tracks or cylinders
        (d->sectors == t->sectors))         // ..and sectors per track
        ? 0 : 1);                           // yes .. return zero

}



/* ******************************************************************** *
 *
 *  do_FAT_check() -- check for bad clusters in target diskette
 *
 *  returns: FALSE if target is usable
 *            TRUE if target has bad clusters
 *
 * ******************************************************************** */

int     do_FAT_check(void)
{
UINT    i, j;                               // loop and work variables


for(i = 2; i < d->nclusters; i++)           // loop thru FAT
    {
    j = check_cluster(                      // get source cluster status
            get_fat_entry(i, d), d);

    if (j == 5)                             // q. source entry "bad"?
        update_fat_entry(i, 0, d);          // a. yes .. clear it

    if (check_cluster(                      // q. target entry "bad"?
            get_fat_entry(i, t), t) == 5)   // a. yes .. check source
        {
        if (j == 3 || j == 4)               // q. source cluster used?
            return(TRUE);                   // a. yes .. immediate death

        update_fat_entry(i,                 // else .. make source have
           d->bad_cluster, d);              // ..target's bad spots
        }
    }

return(FALSE);                              // finally, return all ok

}


/* ******************************************************************** *
 *
 *  clean_source_FAT() -- clear bad clusters from source FAT
 *
 * ******************************************************************** */

void    clean_source_FAT(void)
{
UINT    i;                                  // loop variable


for(i = 2; i < d->nclusters; i++)           // loop thru FAT
    {
    if (check_cluster(                      // q. source entry "bad"?
            get_fat_entry(i, d), d) == 5)
        update_fat_entry(i, 0, d);          // a. yes .. clear it
    }
}



/* ******************************************************************** *
 *
 *  parse_parms() -- parse command line parms
 *
 * ******************************************************************** */

int     parse_parms(int  ac,                // argument count
                    char *av[],             // command line arguments
                    int  n,                 // parse table entries
                    struct cmd_parm *t,     // cmd line parse table
                    char ***parms_array)    // positional parms array
{
int     i, j,                               // loop counter
        parms_fnd = 0,                      // positional parms found
        slash_fnd = 0;                      // slash found in token
char    *p, *q,                             // character pointer
        c;                                  // work character


*parms_array = (char **) huge_malloc(       // set up for max nbr tokens
            sizeof(char *) * ac);

for (i = 1; i < ac; i++)                    // for each cmd line token
    {
    p = av[i];                              // set up pointer to token

    while (*p)                              // process token
        {
        if (*p == '/' || slash_fnd)         // q. option?
            {
            if (NOT slash_fnd)              // q. embedded slash?
                p++;                        // a. no .. bump past slash

            c = toupper(*p);                // get char and upcase it
            slash_fnd = 0;                  // reset switch

            if (c == '?')                   // q. help request?
                quit_with(help);            // a. yes .. give help ..

            for (j = 0; j < n; j++)         // check each table entry
                if (c == t[j].cp_ltr)       // q. find match?
                    break;                  // a. yes .. exit loop

            if (j == n)                     // q. no matches?
                {
                if ((q = strchr(p, '/'))    // q. any more switches?
                         != 0)
                    *q = 0;                 // a. yes .. isolate bad one

                quit_with(bad_op, --p);     // give error message & quit
                }

            if (t[j].cp_flag)               // q. keyword parm w/data?
                {
                *(char **) t[j].cp_entry = ++p; // a. yes .. save token

                if (*(p += strcspn(p, "/")))// q. any switches left?
                    {
                    *p++ = 0;               // a. yes .. make a string
                    slash_fnd = 1;          // ..show nxt char a switch
                    }
                }

             else
                {
                (*t[j].cp_entry)++;         // show slash parm used
                p++;                        // ..and point at next one
                }
            }
         else
            {
            (*parms_array)[parms_fnd++] = p;// save positional string

            if (*(p += strcspn(p, "/")))    // q. any switches left?
                {
                *p++ = 0;                   // a. yes .. make a string
                slash_fnd = 1;              // ..show nxt char a switch
                }
            }
        }
    }

*parms_array = (char **) farrealloc(        // readjust array size
            *parms_array,                   // ..for what was found
            sizeof(char *) * parms_fnd);

return(parms_fnd);                          // rtn w/nbr of positionals

}



/* ******************************************************************** *
 *
 *  logical_sector() -- convert cluster number to sector number
 *
 * ******************************************************************** */

ULONG   logical_sector(UINT cluster,        // cluster number
                       IDRV *d)             // drive info block
{

return(cluster ? ((long)(cluster - 2)       // convert cluster to sector
        * d->sec_cluster) + d->data_sector  // ..for regular clusters
        : d->root_sector);                  // or give root dir sector

}



/* ******************************************************************** *
 *
 *  timer_count() -- get elapsed tick counts
 *
 * ******************************************************************** */

long    timer_count(void)
{
static
long    far *timer = (long far *)           // BIOS timer tick counter
                MK_FP(0x40, 0x6c),
        start = -1,                         // start tick count
        r;                                  // return value


if (start == -1)                            // q. first call?
    {
    start = *timer;                         // a. yes .. get start cnts
    r = 0;                                  // ..and return zero
    }
 else
    {
    r = *timer - start +                    // determine elapsed ticks
        ((start < *timer) ? 0 : 1573041L);  // ..handling midnights

    start = -1;                             // reset first cycle flag
    }

return(r);                                  // ..and rtn elapsed ticks

}



/* ******************************************************************** *
 *
 *  user_report() -- give user final report
 *
 * ******************************************************************** */

void    user_report(void)
{
char    s1[10], s2[10], s3[30];             // work strings
UINT    work;                               // ..and transfer rate

rc = 0;                                     // show successful operation

if (format_time)                            // q. zero seconds formatting?
    {                                       // a. no .. calc fmt rate
    work = (format_count * d->bytes_sector  // calculate transfer rate
            * d->sectors) /                 // ..in bytes per second
            (format_time / 18.2);

    large_fmt(work, s1);                    // format rate in kbytes

    sprintf(s3, format_timing, s1);         // then put into msg phrase
    }
 else
    s3[0] = 0;                              // else .. make fmt a null string

if (NOT read_time)                          // q. zero seconds reading?
    read_time = 1;                          // a. yes .. force one sec

work = (i_remain * d->bytes_sector)         // calculate transfer rate
        / (read_time / 18.2);               // ..in bytes per second

large_fmt(work, s1);                        // format read transfer rate

if (NOT move_time)                          // q. zero seconds writing?
    move_time = 1;                          // a. yes .. force one sec

work = (moved_count * d->bytes_sector)      // calculate transfer rate
        / (move_time / 18.2);               // ..in bytes per second

large_fmt(work, s2);                        // format write transfer rate

if (moved_count)                            // q. anything done?
    quit_with(all_done,                     // a. yes .. give summary
        moved_count,                        // ..total clusters moved
        s1, s2, s3,                         // ..also in bytes/sec
        copied, t_drv + '@');               // ..copies using drive x:
 else
    quit_with(nothing);                     // else .. say nothing

}



/* ******************************************************************** *
 *
 *  control_break() -- control break intercept routine
 *
 * ******************************************************************** */

int     control_break(void)
{

rc = 0;                                     // clear return code

quit_with(stop_here);                       // quit with abort msg
return(0);                                  // ..then return

}



/* ******************************************************************** *
 *
 *  critical_handler() -- DOS critical error handler
 *
 * ******************************************************************** */

#pragma argsused                    // hold unused argument messages
#pragma option -O2-b-e              // no global register allocation
                                    // ..or dead code elimination

void    interrupt far critical_handler(...)
{

   _AX = (_AX & 0xff00) | 2;                // else .. abort request

}



/* ******************************************************************** *
 *
 *  extended_check() -- check for EMS or XMS availability
 *
 *  This routine checks for XMS first then checks for EMS since
 *  some XMS managers support EMS from the XMS memory pool.
 *
 *  return:    0 = no extended memory managers
 *          ...1 = XMS available
 *          ..1. = EMS available
 *
 * ******************************************************************** */

int     extended_check(void)
{
char    far *ivec,                          // interrupt vector pointer
        rc = 0;                             // return code


ivec = (char far *) _dos_getvect(0x67);     // get address of int 67h
ivec = (char far *) MK_FP(FP_SEG(ivec), 10);// ..point to work string

if (NOT _fstrncmp(ivec, "EMMXXXX0", 8))     // q. EMS available?
    {
    rc = 2;                                 // a. yes .. set return flag

    r.h.ah = 0x41;                          // ah = get page frame segment
    int86(0x67, &r, &r);                    // call EMS driver

    ems_segment = r.x.bx;                   // save page frame segment
    }

r.x.ax = 0x4300;                            // ax = installation check
int86x(0x2f, &r, &r, &s);                   // call multiplex interrupt

if (r.h.al == 0x80)                         // q. XMS loaded?
    {
    rc |= 1;                                // a. yes .. set return flag

    r.x.ax = 0x4310;                        // ax = get XMS driver address
    int86x(0x2f, &r, &r, &s);               // call multiplex interrupt

    xms_driver = (int (far *)())            // save driver address
            MK_FP(s.es, r.x.bx);            // ..for later use
    }

return(rc);                                 // finally .. return w/flag

}



/* ******************************************************************** *
 *
 *  ems_alloc() -- allocate some EMS or XMS memory
 *
 *  This routine allocates extended memory from either XMS or EMS
 *  memory pools.
 *
 *  return: 0 = not enough extended memory available
 *          h = EMS handle to allocated memory
 *
 * ******************************************************************** */

int     ems_alloc(long amt)                 // requested amount
{

if (ems_flag & 1)                           // q. XMS available?
    {                                       // a. yes .. try to allocate
    _DX = (int)((amt + 1024) / 1024);       // dx = size in kb
    _AX = 0x900;                            // ah = allocate XMS space
    (xms_driver)();                         // call XMS driver

    if (_AX == 1)                           // q. everything ok?
        {
        ems_flag = 1;                       // a. yes .. show XMS used
        return(_DX);                        // ..then return w/XMS handle
        }
    }

if (ems_flag & 2)                           // q. EMS available?
    {                                       // a. yes .. try allocation
    r.h.ah = 0x43;                          // ah = allocate EMS pages
    r.x.bx = (int)((amt + 16383L) / 16384L);// bx = page count
    int86(0x67, &r, &r);                    // allocate EMS pages

    if (NOT r.h.ah)                         // q. enough EMS available?
        {
        ems_flag = 2;                       // a. yes .. show EMS used
        return(r.x.dx);                     // ..then return w/EMS handle
        }
    }

return(0);                                  // else .. return error condition

}



/* ******************************************************************** *
 *
 *  ems_free() -- release allocated EMS or XMS memory
 *
 *  This routine frees allocated extended memory from either XMS or EMS
 *  memory pools.
 *
 * ******************************************************************** */

void    ems_free(void)
{

switch (ems_flag)
    {
    case 1:                                 // XMS memory
        _AX = 0x0a00;                       // ax = free XMS memory
        _DX = ems_handle;                   // dx = XMS handle to use
        (xms_driver)();                     // call XMS driver to free pages
        break;                              // ..and return

    case 2:                                 // EMS memory
        r.h.ah = 0x45;                      // ah = release EMS pages
        r.x.dx = ems_handle;                // dx = EMS handle to use
        int86(0x67, &r, &r);                // free allocate EMS pages
        break;                              // ..and return
    }
}



/* ******************************************************************** *
 *
 *  xms_copy() -- do XMS copy using copy control block
 *
 * ******************************************************************** */

int     xms_copy(struct  copy_block *cb)    // XMS copy control block
{
int     (far *x_drv)();                     // XMS driver address

x_drv = xms_driver;                         // get local copy of address

asm     push ds                             // save registers
_DS = FP_SEG(cb);                           // ds = our segment
_SI = FP_OFF(cb);                           // ds:si -> control block
_AX = 0xB00;                                // ax = copy to/from XMS
(x_drv)();                                  // call XMS driver to copy pages
asm     pop ds                              // restore registers
return(_AX);                                // ..and return w/error code

}



/* ******************************************************************** *
 *
 *  ems_put() -- copy to EMS memory
 *
 *  This routine copies from convential memory to EMS memory.
 *
 * ******************************************************************** */

long    ems_put(HUGE *buf,                  // conventional memory buffer
                long n)                     // length to copy
{
static
UINT    page = 0,                           // working page number
        offset = 0;                         // ..and offset
static
long    rc,                                 // return code (offset in XMS)
        x_offset = 0;                       // working
int     cnt;                                // work count
struct  copy_block cb;                      // XMS copy block


if (ems_flag == 2)                          // q. EMS memory available?
    {                                       // a. yes .. send to EMS memory
    rc = (page << 16) + offset;             // save current EMS position

    while (n)                               // loop until out of data
        {
        r.x.ax = 0x4400;                    // ax = map page 0
        r.x.bx = page;                      // bx = page number
        r.x.dx = ems_handle;                // dx = handle number
        int86(0x67, &r, &r);                // make page be current

        if (r.h.ah)                         // q. ok?
            quit_with(ems_error);           // a. no .. give up and quit

        cnt = (int)(min(n, 16384L - offset));// get copy count

        _fmemcpy(MK_FP(ems_segment, offset),// copy data to EMS memory
                 buf, cnt);

        n -= (long) cnt;                    // get new remaining length
        buf += cnt;                         // ..and new source pointer

        offset += cnt;                      // set up new offset

        if (offset >= 16384)                // q. at end of EMS page?
            {
            page++;                         // bump page number
            offset = 0;                     // ..and start new page
            }
        }
    }

 else if (ems_flag == 1)                    // q. using XMS memory?
    {                                       // a. yes .. set up copy
    rc = x_offset;                          // rtn code is current offset

    cb.count = n;                           // set up length to copy
    cb.to_handle = ems_handle;              // XMS handle
    cb.to_offset = x_offset;                // offset in XMS area
    cb.from_handle = 0;                     // conventional mem handle
    (HUGE *) cb.from_offset = buf;          // ..for segment and offset
    if (xms_copy(&cb) != 1)                 // q. copy go ok?
        quit_with(ems_error);               // a. no .. give up and quit

    x_offset += n;                          // set up next offset in XMS
    }

return(rc);                                 // return w/page and offset

}



/* ******************************************************************** *
 *
 *  ems_get() -- copy from EMS memory
 *
 *  This routine copies from EMS to conventional memory.
 *
 * ******************************************************************** */

void    ems_get(HUGE *buf,                  // buffer address
                long loc,                   // page and offset
                long n)                     // length to copy
{
UINT    page,                               // working page number
        offset;                             // ..and offset
int     cnt;                                // work count
struct  copy_block cb;                      // XMS copy block


if (ems_flag == 2)                          // q. EMS memory available?
    {                                       // a. yes .. send to EMS memory
    page = (int) (loc >> 16);               // get page number
    offset = (int) (loc & 0xffff);          // ..and offset

    while (n)                               // loop until out of data
        {
        r.x.ax = 0x4400;                    // ax = map page 0
        r.x.bx = page;                      // bx = page number
        r.x.dx = ems_handle;                // dx = handle number
        int86(0x67, &r, &r);                // make page be current

        if (r.h.ah)                         // q. ok?
            quit_with(ems_error);           // a. no .. give up and quit

        cnt = (int)(min(n, 16384L - offset));   // get copy count

        _fmemcpy(buf,                       // copy data from EMS memory
                MK_FP(ems_segment, offset), // ..into conventional memory
                cnt);

        n -= (long) cnt;                    // get new remaining length
        buf += cnt;                         // ..and new source pointer

        offset += cnt;                      // set up new offset

        if (offset >= 16384)                // q. at end of EMS page?
            {
            page++;                         // bump page number
            offset = 0;                     // ..and start new page
            }
        }
    }
 else if (ems_flag == 1)                    // q. using XMS memory?
    {
    cb.count = n;                           // set up length to copy
    cb.from_handle = ems_handle;            // XMS handle
    cb.from_offset = loc;                   // offset in XMS area
    cb.to_handle = 0;                       // conventional mem handle
    (HUGE *) cb.to_offset = buf;            // ..for segment and offset
    if (xms_copy(&cb) != 1)                 // q. copy go ok?
        quit_with(ems_error);               // a. no .. give up and quit
    }
}



/* ******************************************************************** *
 *
 *  get_all() -- determine EMS requirements for source disk
 *
 *  This routine determines the size of the source disk.  Since the
 *  disk is not necessarily in DOS format, a FAT may not exist.  Instead
 *  the size is determined from testing the known diskette formats.
 *
 * ******************************************************************** */

long    get_all(void)
{
UINT    i, j,                               // loop counter
        in_row;                             // max copy in clusters
struct  copy_entry huge *c;                 // work pointer

d = get_disk_type(s_drv);                   // get drive information

if (d->dos_valid == -1)                     // q. need formatting?
    quit_with(unreadable);                  // a. yes .. quit w/err msg

i_remain = (long) d->heads * d->tracks      // set up initial remaining
        * d->sectors;                       // ..sector count

printf(disk_info, s_drv + '@',              // give standard information
        d->heads, d->tracks, d->sectors,    // ..about physical volume
        large_fmt(d->raw_disk, 0));         // ..volume size

a_mem = coreleft() - 1024;                  // start with almost everything

for (i = 1; i < i_remain; i++)              // determine maximum
    {                                       // memory utilization
    j = (int)((i_remain + (i - 1)) / i);    // number of copy entries needed

    if ((a_mem - (j * SS(copy_entry))) >    // q. enough memory available
            ((long)i * d->bytes_sector))    // ..to support required buffer?
        {}                                  // a. yes .. continue
     else
        break;                              // else .. exit loop
    }

in_row = i - 1;                             // nbr of sectors per buffer
a_mem = (long) in_row * d->bytes_sector;    // buffer size
copy_entries = (UINT) (i_remain +           // nbr of copy entries needed
        (in_row - 1)) / in_row;

if (NOT in_row)                             // q. enough memory for buffers?
    quit_with(no_mem);                      // a. no .. quit with error msg

c_table = c = (struct copy_entry huge *)    // allocate some memory for
        huge_malloc(copy_entries * SS(copy_entry)); // ..the copy entries

for(i = 0, j = 0; j < i_remain;             // loop thru building
        i++, j += in_row, c++)              // ..copy entry table
    {
    c->c_sector = j;                        // save sector number
    c->c_length = (i_remain - j) > in_row ? // compute entry size
            a_mem :                         // ..in bytes
            ((i_remain - j) * d->bytes_sector);
    }

return(d->raw_disk);                        // return w/disk size

}



/* ******************************************************************** *
 *
 *  get_mem_needed() -- determine EMS requirements for source DOS disk
 *
 *  This routine calculates the EMS requirements by reading the file
 *  allocation table, counting allocated clusters and multiplying by
 *  the number of bytes per cluster.  This routine also builds the
 *  copy_entry array which is a list of sectors to be copied.
 *
 *  returns:  nbr of bytes of EMS/XMS memory needed
 *
 *
 * ******************************************************************** */

long    get_mem_needed(void)
{
UINT    i,                                  // loop counter
        in_row,                             // in-a-row clusters
        max_row;                            // max copy in clusters
char    strings[3][10];                     // temporary strings
struct  copy_entry huge *c;                 // work pointer


d = get_disk_type(s_drv);                   // get drive information

if (d->dos_valid != 1)                      // q. DOS disk?
    {                                       // a. no ..
    sw_all = 1;                             // ..set proper switch
    return(get_all());                      // ..then do it
    }

get_fat(d);                                 // get FAT from source disk

i_remain = ((d->nclusters -                 // sectors needed
        d->avail_clusters) *                // ..to be moved
        d->sec_cluster) + d->root_length;   // ..and root directory

printf(drive_info, s_drv + '@',             // give standard information
        read_label(s_drv), d->nsectors,     // ..about logical volume
        d->nclusters, d->cluster_size,      // ..and format out disk
        large_fmt(d->raw_disk,              // ..volume size
                strings[0]),
        large_fmt(d->raw_disk -             // ..and total used
                d->avail_disk,
                strings[1]),
        large_fmt(d->avail_disk,            // ..and available size
                strings[2]));

c_table = c = (struct copy_entry huge *)    // allocate some memory
        huge_malloc(((i_remain + 2) /       // ..for starts
        d->sec_cluster) * SS(copy_entry));  // ..then re-alloc later

c->c_sector = d->root_sector;               // root directory sector nbr
c->c_length = d->root_size;                 // ..and root dir length
c++;                                        // get to next copy entry

a_mem = coreleft() - 40000L;                // start with almost everything
max_row = (int) (a_mem / d->cluster_size);  // get max nbr of clusters
a_mem = max_row * d->cluster_size;          // ..and final buffer size

if (NOT max_row)                            // q. at least one cluster buffer?
    quit_with(no_mem);                      // a. no .. quit with error msg

for(i = 2; i < d->nclusters; i++)           // loop thru FAT
    {
    if (cluster_status(i, d))               // q. allocated cluster?
        {
        c->c_sector = logical_sector(i, d); // a. yes .. save sector nbr

        for (in_row = 1, i++;               // now find how many in a row
                i < d->nclusters &&         // before the end of the FAT
                in_row < max_row;           // and before max transfer size
                i++, in_row++)
            if (NOT cluster_status(i, d))   // q. next cluster used?
                break;                      // a. no .. exit loop

        c->c_length = in_row *              // save length in bytes
                d->cluster_size;
        c++;                                // get to next copy entry
        copy_entries++;                     // ..and keep a good count
        }
    }

farrealloc(c_table, (long) copy_entries     // re-adjust table length
        * SS(copy_entry));                  // ..for used amount

return(i_remain * d->bytes_sector);         // return w/needed memory for
                                            // ..clusters and the root dir
}



/* ******************************************************************** *
 *
 *  get_source() -- read source diskette into EMS/XMS memory
 *
 * ******************************************************************** */

void    get_source(void)
{
int     i;                                  // loop control
struct  copy_entry huge *c;                 // work ptr into copy table


disk_buf = (HUGE *)huge_malloc(a_mem + 511);// allocate disk buffer
ADJUST(disk_buf);                           // ..and make on proper boundary

update_msg = read_source;                   // set up phase name
remaining = i_remain;                       // ..and remaining counter

enabled = 1;                                // enable update msg
timer_count();                              // start the clock

for (i = 0, c = c_table; i < copy_entries;  // loop thru copy entries
        c++, i++)                           // ..and read/copy to EMS
    {
    if (disk_io(0, c->c_sector,             // q. read ok?
            (UINT) (c->c_length / d->bytes_sector),
            disk_buf, d))
        quit_with(read_error);              // a. no .. quit with error

    c->c_ems = ems_put(disk_buf,            // ..then send to EMS memory
            c->c_length);                   // ..and save location in EMS
    }

read_time = timer_count();                  // count time to read diskette
printf(clean_up, "");                       // clean up from status msgs

}



/* ******************************************************************** *
 *
 *  format_disk() -- format target diskette
 *
 * ******************************************************************** */

int     format_disk(IDRV *d,                // source drive info block
                    IDRV *t)                // target drive info block
{
int     i, j, k,                            // work/loop variables
        rc = 0;                             // function return code
ULONG   i_hold;                             // save area for i_remain
union   REGS rs;                            // reset registers
union   REGS ro;                            // output registers
struct  format_block *f;                    // work pointer


update_msg = format_target;                 // set up phase name
i_hold = i_remain;                          // save old max counter
i_remain = remaining = d->tracks * d->heads;// ..and set up new

r.x.ax = 0x440d;                            // ax = ioctl call
r.x.bx = t->drive;                          // bx = drive number
r.x.cx = 0x0840;                            // cx = set device parms
r.x.dx = FP_OFF(&dp);                       // dx -> device parm block
s.ds = FP_SEG(&dp);                         // ds:dx -> ...
int86x(0x21, &r, &ro, &s);                  // set target device parms

if (ro.x.cflag)                             // q. formatable in target drv?
    quit_with(drv_wont);                    // a. no .. quit w/error msg

_dos_setvect(0x1e, new_1e);                 // install new disk parm ptr

rs.h.ah = 0;                                // reset function call
rs.h.dl = t->drive - 1;                     // drive number
int86(0x13, &rs, &rs);                      // call BIOS to reset drive

for (f = fmt, i = 1; i <= d->sectors;       // for each entry in table
            i++, f++)                       // ..build the constant info
    {
    f->sector = i;                          // set the sector number
    f->size = 2;                            // ..and the sector size
    }

r.h.ah = 5;                                 // format track function call
r.h.al = d->sectors;                        // sectors per track
r.h.dl = t->drive - 1;                      // drive number
r.x.bx = FP_OFF(fmt);                       // track address buffer
s.es = FP_SEG(fmt);                         // ..offset and segment

enabled = 1;                                // enable update msg
timer_count();                              // start the clock

for (i = 0; i < d->tracks && NOT rc; i++)   // for each track..
    for (j = 0; j < d->heads && NOT rc; j++)// ..and each head
        {
        update_progress();                  // update user with status

        for (f = fmt, k = 0;                // build the track
                k < d->sectors; k++, f++)   // ..layout buffer
            {
            f->track = i;                   // set up the track field
            f->head = j;                    // ..and the head field
            }

        r.h.ch = io_rw.track = i;           // prepare format call
        r.h.dh = io_rw.head = j;            // ..with track and head

        for (k = 5, ro.x.cflag = 1;         // give BIOS a couple of
                    k-- && ro.x.cflag;)     // ..cracks to ..
            {
            int86x(0x13, &r, &ro, &s);      // format track

            if (ro.x.cflag)                 // q. problems?
                int86(0x13, &rs, &rs);      // a. yes .. reset drive
            }

        if (k < 0)                          // q. fail track format?
            {
            printf(disk_wont);              // a. yes .. flush diskette
            rc = 1;                         // ..set return code
            break;                          // ..and exit loop
            }

        remaining--;                        // one more done..
        format_count++;                     // ..and one more tallyed

        }

format_time = timer_count();                // count time formatting
printf(clean_up, "");                       // clean up from status msgs
enabled = 0;                                // disable status messages

_dos_setvect(0x1e, old_1e);                 // since done, restore old
i_remain = i_hold;                          // fix up old max count

t->heads = d->heads;                        // formatted heads
t->tracks = d->tracks;                      // ..cylinders or tracks
t->sectors = d->sectors;                    // ..sectors per track

return(rc);                                 // return with return code

}



/* ******************************************************************** *
 *
 *  make_copies() -- copy from EMS memory to the diskettes
 *
 * ******************************************************************** */

void    make_copies(void)
{
long    fs;                                 // FAT sector number
int     i,                                  // loop control
        e_flag,                             // error flag
        cnt;                                // sector count
struct  copy_entry huge *c;                 // work pointer



for (;;)                                    // loop till done
    {
    if (disk_change())                      // q. change disks ok?
        break;                              // a. no .. exit loop

    enabled = 0;                            // disable update msg
    e_flag = 0;                             // clear error flag

    free_drive_block(t);                    // release old target drv blk
    t = get_disk_type(t_drv);               // get diskette type

    if (t->dos_valid == -1 &&               // q. need formatting?
            format_disk(d, t))              // ..and format go ok?
        continue;                           // a. no .. next diskette

    if (media_same_check())                 // q. same format?
        {
        printf(bad_target1);                // a. no .. give error msg
        continue;                           // ..then get next diskette
        }

    if (NOT sw_all)                         // q. doing DOS copy?
        {                                   // a. yes .. process source FAT
        if (t->dos_valid == 1)              // q. was target valid DOS format?
            {                               // a. yes .. process target disk
            get_fat(t);                     // get current info

            if (do_FAT_check())             // q. FATs ok?
                {
                printf(bad_target2);        // a. no .. give error msg
                continue;                   // ..then get next diskette
                }
            }
         else
            clean_source_FAT();             // else .. remove bad clusters
        }

    update_msg = write_target;              // set up phase name
    enabled = 1;                            // re-enable update msg
    remaining = i_remain;                   // set up remaining counter
    timer_count();                          // start the clock

    for (i = 0, c = c_table;                // loop thru copy entries
            i < copy_entries; c++, i++)     // ..and copy to diskettes
        {
        ems_get(disk_buf, c->c_ems,         // get a block from EMS
                c->c_length);               // ..and put in disk buffer

        if (disk_io(1, c->c_sector,         // q. write target ok?
                cnt = (UINT)(c->c_length
                / d->bytes_sector),
                disk_buf, t))
            {
            printf(clean_up, "");           // a. no .. clean up line
            printf(write_error);            // ..give error message
            e_flag = 1;                     // set error flag
            break;                          // ..and exit inner loop
            }

        moved_count += cnt;                 // count copied sectors
        }

    if (NOT sw_all && NOT e_flag)           // q. DOS copy?
        {                                   // a. yes .. write target FATs
        enabled = 0;                        // turn off graph display

        if (disk_io(1, 0, 1,                // q. write ok for
                (HUGE *) d->i_boot, t))     // ..boot record?
            {
            printf(boot_w_error);           // a. no .. give error msg
            e_flag = 1;                     // set error flag
            }

        fs = d->i_boot->res_sectors;        // first FAT sector number

        for (i = d->i_boot->fats;           // for each copy of the FAT
                i-- && NOT e_flag;          // ..write it to the target
                fs += d->i_boot->fat_sectors)
            if (disk_io(1, fs,              // q. write ok for new
                    d->i_boot->fat_sectors, // ..copy of the FAT?
                    d->i_fat, t))
                {
                printf(fat_w_error);        // a. no .. give error msg
                e_flag = 1;                 // set error flag
                }

        moved_count += d->i_boot->fat_sectors   // tally all sectors
                * d->i_boot->fats + 1;          // ..for all FATs
        }

    move_time += timer_count();             // count time to write diskette

    if (e_flag)                             // q. error during last disk?
        continue;                           // a. yes .. try next diskette

    printf(clean_up, "");                   // clean up from status msgs
    copied++;                               // ..and bump counter

    if (sw_one)                             // q. only one wanted?
        break;                              // a. yes .. exit loop
    }
}



/* ******************************************************************** *
 *
 *  initialization() -- initialize program and parse command line
 *
 * ******************************************************************** */

void    initialization(int  ac,             // DOS cmd line token count
                       char *av[])          // ..token strings
{
int     dos_ver;                            // dos version


ctrlbrk(control_break);                     // set up ctrl break and
_dos_setvect(0x24, critical_handler);       // ..DOS critical handlers

dos_ver = (_osmajor * 100) + _osminor;      // get current DOS version

if (dos_ver < 330)                          // q. pre-DOS 3.3?
    quit_with(dos_error);                   // a. yes .. can't run it

if ((pos_found = parse_parms(ac, av,        // parse switches and
        PARM_TABLE_CNT, parm_table,         // ..positional parameters
        &pos_parms)) > 2)                   // q. too may positionals?
    quit_with(too_many);                    // a. yes .. give err/quit

if (pos_found == 0)                         // q. enough parms given?
    quit_with(help);                        // a. no .. give error and quit

s_drv = drive_check(pos_parms[0]);          // check source drive

if (pos_found == 2)                         // q. target drive given?
    t_drv = drive_check(pos_parms[1]);      // a. yes .. check target
 else
    t_drv = s_drv;                          // else .. use same drive

if (s_drv == t_drv)                         // q. source and target the same?
    same_drive = 1;                         // a. yes .. set global flag

memset(block_1, 0xfa, 16);                  // set up string of blocks
memset(block_2, 0xdb, 16);                  // ..to show progress

if (NOT (ems_flag = extended_check()))      // q. EMS or XMS installed?
    quit_with(no_ems);                      // a. no .. quit with error msg

check_source_available(s_drv);              // make source available
get_drive_parms();                          // build device control block

ems_required = sw_all ?                     // determine minimum EMS
        get_all() : get_mem_needed();       // ..requirements


if (NOT (ems_handle = ems_alloc(            // q. able to allocate enough?
        ems_required)))
    quit_with(no_ems_space);                // a. no .. quit with error msg

printf(requirements, i_remain,              // tell user the minimum
        large_fmt(ems_required, 0),         // ..memory requirements
        (ems_flag & 2) ? "EMS" : "XMS");    // ..and kind of memory was used

}



/* ******************************************************************** *
 *
 *  main()
 *
 * ******************************************************************** */

void    main(int  ac,                       // DOS cmd line token count
             char *av[])                    // ..token strings
{

printf(copyright);                          // give copyright message
initialization(ac, av);                     // init and parse cmd line
get_source();                               // load source disk
make_copies();                              // ..and make disk copies
user_report();                              // give final report ..

}
