//===========================================================
// WCL2 - A second function library for the WinCmd language
// Copyright (C) 1994 Douglas Boling
//===========================================================
// Returns no. of elements
#define dim(x) (sizeof(x) / sizeof(x[0]))   
#define MAXREADLINE       1024   

#define DROPWINMAX        64
#define ERR_NOTWINDOW     -1
#define ERR_NODROPSPACE   -2

#include "windows.h"
#include "toolhelp.h"
#include "commdlg.h"
#include "shellapi.h"

#include "stdlib.h"
#include "string.h"
#include "direct.h"
#include "dos.h"
#include "time.h"

#include "WinCmd.h"
#include "wcl2.h"

typedef struct {
	HWND	hWnd;
	HWND	hWndOwner;
	UINT	fFlags;
	FARPROC lpfnOldWndProc;
	LPLINETOKEN far *lpLineToken;
	HGLOBAL hFileNames;
} ONDROPSTRUCT;
typedef ONDROPSTRUCT far *LPONDROPSTRUCT;

typedef struct {
	INT		sType;
	char		*szName;
	PLFUNC	Fxn;
} LIBFUNCSTRUCT;
typedef LIBFUNCSTRUCT *PLIBFUNCSTRUCT;

typedef	INT (*FSTATEMENTFUNC) (LPLINETOKEN far *, INT);

//Undocumented function prototypes
BOOL WINAPI IsWinOldApTask(HTASK);

INT LibSetWindowText (LPTOKEN, LPTOKEN);
INT LibCharVal (LPTOKEN, LPTOKEN);
INT LibCharChr (LPTOKEN, LPTOKEN);
INT LibStrInStr (LPTOKEN, LPTOKEN);
INT LibStrRev (LPTOKEN, LPTOKEN);
INT LibIsRunning (LPTOKEN, LPTOKEN);
INT LibGetKeyState (LPTOKEN, LPTOKEN);
INT LibGetKeyShift (LPTOKEN, LPTOKEN);
INT LibShowWin (LPTOKEN, LPTOKEN);
INT LibGetINIVal (LPTOKEN, LPTOKEN);
INT LibSetINIVal (LPTOKEN, LPTOKEN);
INT LibFullPath (LPTOKEN, LPTOKEN);
INT LibFileDrive (LPTOKEN, LPTOKEN);
INT LibFileDir (LPTOKEN, LPTOKEN);
INT LibFilePath (LPTOKEN, LPTOKEN);
INT LibFileName (LPTOKEN, LPTOKEN);
INT LibFileExt (LPTOKEN, LPTOKEN);
INT LibGetFileName (LPTOKEN, LPTOKEN);
INT LibGetSaveName (LPTOKEN, LPTOKEN);
INT LibGetDate (LPTOKEN, LPTOKEN);
INT LibGetTime (LPTOKEN, LPTOKEN);

//-----------------------------------------------------------
// Global data
//-----------------------------------------------------------
char     szDllName[] = "WCL2";
UINT		wWCL2Ver = 10;
FIND_T	fs;
char     szTemp[256];
char     szTemp1[256];
LONG 		lTime;

HANDLE	hInst;
HWND		hMain;
RECT		rect;

INT		sErrStart = 0;
LPCBTOKFUNC	lpCallbackTokenFunction;

char szFilter[256];

//
// Library function tokens
//
LIBFUNCSTRUCT FuncList[] = {
	{TTYPE_LIBFUNC,"SETWINDOWTEXT", LibSetWindowText},
	{TTYPE_LIBFUNC,"VAL", LibCharVal},
	{TTYPE_LIBFUNC,"CHR", LibCharChr},
	{TTYPE_LIBFUNC,"INSTR", LibStrInStr},
	{TTYPE_LIBFUNC,"REVERSE", LibStrRev},
	{TTYPE_LIBFUNC,"ISRUNNING", LibIsRunning},
	{TTYPE_LIBFUNC,"ISKEYDOWN", LibGetKeyState},
	{TTYPE_LIBFUNC,"ISKEYSHIFTED", LibGetKeyShift},
	{TTYPE_LIBFUNC,"SHOWWINDOW", LibShowWin},
	{TTYPE_LIBFUNC,"READINIDATA", LibGetINIVal},
	{TTYPE_LIBFUNC,"WRITEINIDATA", LibSetINIVal},
	{TTYPE_LIBFUNC,"FULLPATH", LibFullPath},
	{TTYPE_LIBFUNC,"GETFILEDRIVE", LibFileDrive},
	{TTYPE_LIBFUNC,"GETFILEDIR", LibFileDir},
	{TTYPE_LIBFUNC,"GETFILEPATH", LibFilePath},
	{TTYPE_LIBFUNC,"GETFILENAME", LibFileName},
	{TTYPE_LIBFUNC,"GETFILEEXT", LibFileExt},
	{TTYPE_LIBFUNC,"FILEOPENBOX", LibGetFileName},
	{TTYPE_LIBFUNC,"FILESAVEBOX", LibGetSaveName},
	{TTYPE_LIBFUNC,"GETDATE", LibGetDate},
	{TTYPE_LIBFUNC,"GETTIME", LibGetTime},
};
//
// Library contstant tokens
//
struct {
	char *szName;
	INT sVal;
} ConstList[] = {{"WINDOW_SHOW", SW_SHOW}, {"WINDOW_HIDE", SW_HIDE},
                };
struct {
   char	*szName;
   BYTE	ucKey;
} SKList[] = {
	{"add", VK_ADD}, 
	{"shift", VK_SHIFT}, {"control", VK_CONTROL}, {"alt", VK_MENU},
	{"bksp", VK_BACK}, {"break", VK_CANCEL}, {"capslock", VK_CAPITAL},
	{"clear",VK_CLEAR}, {"delete", VK_DELETE}, {"down", VK_DOWN},
	{"end" , VK_END}, {"enter", VK_RETURN}, {"esc", VK_ESCAPE},
	{"help", VK_HELP}, {"home", VK_HOME}, {"insert", VK_INSERT},
	{"left", VK_LEFT}, {"numlock", VK_NUMLOCK}, {"pgdn", VK_NEXT},
	{"pgup", VK_PRIOR}, {"prtsc", VK_SNAPSHOT}, {"pause", VK_PAUSE},
	{"right", VK_RIGHT}, {"tab", VK_TAB}, {"up", VK_UP}, {"scroll", VK_SCROLL},
	{"F1", VK_F1}, {"F2", VK_F2}, {"F3", VK_F3}, {"F4", VK_F4},
	{"F5", VK_F5}, {"F6", VK_F6}, {"F7", VK_F7}, {"F8", VK_F8},
	{"F9", VK_F9}, {"F10", VK_F10}, {"F11", VK_F11}, {"F12", VK_F12},
	{"F13", VK_F13}, {"F14", VK_F14}, {"F15", VK_F15}, {"F16", VK_F16},
	{"F17", VK_F17}, {"F18", VK_F18}, {"F19", VK_F19}, {"F20", VK_F20},
	{"F21", VK_F21}, {"F22", VK_F22}, {"F23", VK_F23}, {"F24", VK_F24},
};		
//===========================================================
// Library initialization and terminataion functions
//===========================================================
//-----------------------------------------------------------
// LibMain - DLL initialization routine
//-----------------------------------------------------------
INT CALLBACK LibMain (HANDLE hInstance, WORD wDataSeg, 
                      WORD wHeapSize, LPSTR lpszCmdLine) {
	hInst = hInstance;
  	return 1;
}
//-----------------------------------------------------------
// WEP - DLL termination routine
//-----------------------------------------------------------
INT CALLBACK WEP (int nParameter) {
	return 1;
}
//===========================================================
// Library exported functions
//===========================================================
//-----------------------------------------------------------
// WCLibLoad - Routine called by WinCmd at function lib load
//-----------------------------------------------------------
INT CALLBACK WCLibLoad (HWND hWinCmdWnd, INT sMaxErrNum, INT sLibErrNum) {
	sErrStart = sLibErrNum;
   return 0;
}   
//-----------------------------------------------------------
// WCLibReset - Routine called by WinCmd to reset lib
//-----------------------------------------------------------
INT CALLBACK WCLibReset (INT sLibID, FARPROC lpfnCBTokenFunc) {
   INT	i, sRC = 0;

	lpCallbackTokenFunction = (LPCBTOKFUNC) lpfnCBTokenFunc;


	i = (INT) (*lpCallbackTokenFunction) (CBF_GETWINCMDVER, 0, 0, 0, 0, 0);
	if (i < 12)
		return sErrStart + ERR_BADWINCMDVER;

	// Add Functions to token list
   for (i = 0; i < dim (FuncList); i++) {
		sRC = (INT) (*lpCallbackTokenFunction) (CBF_ADDTOKEN, 0, 
		                    (LPSTR) FuncList[i].szName, (BYTE)FuncList[i].sType,
		                    0, MAKELONG (i, sLibID));
		if (sRC) break;
	}
	// Add predefined tokens
   for (i = 0; i < dim (ConstList); i++) {
		sRC = (INT)(*lpCallbackTokenFunction) (CBF_ADDTOKEN, 0, 
		                    (LPSTR) ConstList[i].szName, 
		                    TTYPE_NUM, 0, (DWORD) ConstList[i].sVal);
		if (sRC) break;
	}
	return sRC;
}
//-----------------------------------------------------------
// WCLibFunc - Routine called by WinCmd at function dispatch
//-----------------------------------------------------------
INT CALLBACK WCLibFunc (LPLINETOKEN *lpltLine, LPTOKEN lptDest, UINT wFuncNum,
                        FARPROC lpfnCBTokenFunc, HWND hWin) {
	INT	sRC;
	WORD	wDSeg;
	LPLINETOKEN far *lpltKluge;	
	
	lpCallbackTokenFunction = (LPCBTOKFUNC) lpfnCBTokenFunc;
	hMain = hWin;
   lptDest->ucType = TTYPE_NUM;
	if (FuncList[wFuncNum].sType == TTYPE_LIBFUNC)
		sRC = (FuncList[wFuncNum].Fxn) (lptDest, (LPTOKEN)lptDest->dwData);
	else {	
		_asm mov	wDSeg,ss
		lpltKluge = MAKELP (wDSeg, lpltLine);
		sRC = ((FSTATEMENTFUNC)FuncList[wFuncNum].Fxn) (lpltKluge, 
		                                                (INT)lptDest);
	}	
   return sRC;
}
//-----------------------------------------------------------
// WCLibSFunc - Routine called by WinCmd at lib statement dispatch
//-----------------------------------------------------------
INT CALLBACK WCLibSFunc (LPLINETOKEN far *lpltLine, INT sFunc, 
                 UINT wFuncNum, FARPROC lpfnCBTokenFunc, HWND hWin) {
	INT	sRC;
	
	lpCallbackTokenFunction = (LPCBTOKFUNC) lpfnCBTokenFunc;
	hMain = hWin;
	sRC = ((FSTATEMENTFUNC)FuncList[wFuncNum].Fxn) (lpltLine, (INT)sFunc);
   return sRC;
}
//-----------------------------------------------------------
// WCGetErrStr - Routine that returns a string for an error
// code.
//-----------------------------------------------------------
INT CALLBACK WCGetErrStr (INT sError, LPSTR lpszStrOut, INT sStrSize) {

	lstrcpy (lpszStrOut, szDllName);
	lstrcat (lpszStrOut, " Error.  ");
	if (LoadString (hInst, sError, lpszStrOut+lstrlen (lpszStrOut), 
	                sStrSize - lstrlen (lpszStrOut)) == 0) {
		lstrcat (lpszStrOut, "Error Number: ");
		itoa (sError, szTemp, 10);
		lstrcat (lpszStrOut, szTemp);
	}
   return 1;
}   
//-----------------------------------------------------------
// WCGetHelp - Routine that returns a help string for a function.
//-----------------------------------------------------------
INT CALLBACK WCGetHelp (INT sLineNum, LPSTR lpszStrOut, INT sStrSize) {

	if (LoadString (hInst, sLineNum + HELPSTART, lpszStrOut, sStrSize))
		return 1;
	return 0;
}   
//-----------------------------------------------------------
// WCGetVersion - Routine returns lib name and version
//-----------------------------------------------------------
INT CALLBACK WCGetVersion (LPWORD lpwVersion, LPSTR lpszNameOut, INT sStrSize) {
   *lpwVersion = wWCL2Ver;
   if (strlen (szDllName) < (UINT) sStrSize)
      lstrcpy (lpszNameOut, szDllName);
   return 0;
}   
//===========================================================
// Local procedures
//===========================================================
//------------------------------------------------------------
// SetToken - Sets a token to a new value
//------------------------------------------------------------ 
INT SetToken (LPTOKEN lptDest, BYTE ucNewType, UINT wData, DWORD dwData) {
   return (INT) (*lpCallbackTokenFunction) (CBF_SETTOKEN, lptDest, 
                               0, ucNewType, wData, dwData);
}   
//------------------------------------------------------------
// AddToken - Adds a token to the token list
//------------------------------------------------------------ 
INT AddToken (LPSTR lpName, BYTE ucType, UINT wData, DWORD dwData) {
   return (INT) (*lpCallbackTokenFunction) (CBF_SETTOKEN, 0,
	                            lpName, ucType, wData, dwData);
}   
//------------------------------------------------------------
// DeleteToken - Deletes a token from the list
//------------------------------------------------------------ 
INT DeleteToken (LPTOKEN lptDest) {
   return (INT) (*lpCallbackTokenFunction) (CBF_DELTOKEN, lptDest, 
                                            0, 0, 0, 0);
}   
//------------------------------------------------------------
// FindToken - Finds a token in the list
//------------------------------------------------------------ 
LPTOKEN FindToken (LPSTR lpszName, INT sNameLen) {
   return (LPTOKEN)(*lpCallbackTokenFunction) (CBF_FINDTOKEN, 0, lpszName, 
                                             0, sNameLen, 0);
}   
//------------------------------------------------------------
// Execute - Executes part of a line 
//------------------------------------------------------------ 
INT Execute (LPLINETOKEN far *lpltStart, INT sFunc) {

   return (INT)(*lpCallbackTokenFunction) (CBF_EXECUTE, 0, 0, 0,
                                           sFunc, (DWORD)(LPVOID)lpltStart);
}   
//------------------------------------------------------------
// SetHoldFlag - Sets or clears the Hold Flag
//------------------------------------------------------------ 
INT SetHoldFlag (BOOL fHold) {

	return (INT) (*lpCallbackTokenFunction) (CBF_SETRUNNING, 0, 0, 0, 
	                                         fHold, 0);
}   
//------------------------------------------------------------
// GetWinCmdVer - Returns the version of WinCmd
//------------------------------------------------------------ 
INT GetWinCmdVer (void) {
	LONG lRC;

   lRC = (*lpCallbackTokenFunction) (CBF_GETWINCMDVER, 0, 0, 0, 0, 0);
	if (lRC = ERR_BAD_FUNC)
		lRC = 10;
	return (INT) lRC;	
}   
//-----------------------------------------------------------
// GetTokenString - Returns a pointer to the string value
// for a token.
//-----------------------------------------------------------
LPSTR GetTokenString (LPTOKEN lptDest) {
   return (LPSTR) (*lpCallbackTokenFunction)(CBF_GETTOKENSTRING, 
                                             lptDest, 0, 0, 0, 0);
}  	
//-----------------------------------------------------------
// GetTokenVal - Returns a pointer to the numberic value
// for a token.
//----------------------------------------------------------- 
LONG GetTokenVal (LPTOKEN lptDest) {
   return (LONG) (*lpCallbackTokenFunction)(CBF_GETTOKENVAL, 
                                            lptDest, 0, 0, 0, 0);
}  	
//-----------------------------------------------------------
// LaunchString - Launches the program indicated in the 
// string.
//----------------------------------------------------------- 
LONG LaunchString (LPSTR lpStr) {
   return (LONG) (*lpCallbackTokenFunction)(CBF_LAUNCHLINE, 
                                            0, lpStr, 0, 0, 0);
}  	
//----------------------------------------------------------- 
// CmpFilename - Compares a filename to a search template
//----------------------------------------------------------- 
BOOL CmpFilename (LPSTR lpTemplate, LPSTR lpName) {
	BOOL fMatch = TRUE;
	
	while (fMatch && (*lpTemplate != '\0')) {
		if (*lpTemplate == '?')
			;
		else if (*lpTemplate == '*') {
			while ((*lpName != '\0') && 
			       (toupper(*lpName) != toupper(*(lpTemplate+1))))
				lpName++;
			lpTemplate++;

		} else if (toupper(*lpName) != toupper(*lpTemplate))
			fMatch = FALSE;

		lpTemplate++;
		if (*lpName != '\0')
			lpName++;
	}
	return fMatch;
}	
//------------------------------------------------------------
// MySubClassWindow - Subclasses a window 
//------------------------------------------------------------
WNDPROC MySubClassWindow (HWND hWnd, WNDPROC lpfnNewProc) {
   WNDPROC lpfnOldProc;

	lpfnOldProc = (WNDPROC) GetWindowLong (hWnd, GWL_WNDPROC);
	SetWindowLong (hWnd, GWL_WNDPROC, (LONG) lpfnNewProc);
	return lpfnOldProc;				               
}				               
//------------------------------------------------------------
// GetVKey - returns a virtual key from a sendkey string
//------------------------------------------------------------
INT GetVKey (LPSTR pszIn) {
	INT i, sVKey;
	LPSTR pszEnd;
	BOOL bFound = FALSE;
	
	sVKey = *pszIn;
	if (sVKey == 0) return 0;
	if (sVKey == '{') {
		pszIn++;
   	if (*(pszIn+1) == '}')
			sVKey = VkKeyScan (*pszIn) & 0x00ff;
		else {
			pszEnd = pszIn;
			while ((*pszEnd != '\0') && (*pszEnd != '}'))
				pszEnd++;
		   if (pszEnd == '\0') return 0;
		   *pszEnd = '\0';
			for (i = 0; i < dim(SKList); i++)
				if (lstrcmpi (SKList[i].szName, pszIn) == 0) {
				   bFound = TRUE;
					break;
				}	
			*pszEnd = '}';
			if (!bFound) return 0;
			sVKey = SKList[i].ucKey;
		}   
	} else 
		sVKey = VkKeyScan (sVKey) & 0x00ff;
	return sVKey;
}		
//===========================================================
// WCL2 functions
//===========================================================
//-----------------------------------------------------------
// LibSetWindowText - Function to set the title text of a 
// Window
// Arg 1 - Handle of the window
// Arg 2 - New title text
//----------------------------------------------------------- 
INT LibSetWindowText (LPTOKEN lptDest, LPTOKEN lptArg) {
	LPSTR lpszStr;
	HWND hWnd;
	INT sRC;

	if (lptDest->wData <= 2)
		return ERR_TOO_FEW_PARMS;
	hWnd = (HWND)GetTokenVal (lptArg-1);
	lpszStr = GetTokenString (lptArg-2);

	if (IsWindow(hWnd)) {
		SetWindowText (hWnd, lpszStr);		
  		sRC = SetToken (lptDest, TTYPE_NUM, 0, (DWORD) MYTRUE);
	} else	
  		sRC = SetToken (lptDest, TTYPE_NUM, 0, (DWORD) MYFALSE);
  	return sRC;
}
//-----------------------------------------------------------
// LibShowWin - Function to show/hide a window
// Arg 1 - Window handle
// Arg 2 - Show parameter
//----------------------------------------------------------- 
INT LibShowWin (LPTOKEN lptDest, LPTOKEN lptArg) {
   INT		sSWParam;
   HWND     hwndTarg;
   
	if (lptDest->wData <= 2) 
		return ERR_TOO_FEW_PARMS;

   hwndTarg = (HWND) GetTokenVal (lptArg-1);
   sSWParam = (INT) GetTokenVal (lptArg-2);
	if (!IsWindow (hwndTarg)) 
		return SetToken (lptDest, TTYPE_NUM, 0, MYFALSE);

	return SetToken (lptDest, TTYPE_NUM, 0, 
	                 (LONG)ShowWindow (hwndTarg, sSWParam));
}
//-----------------------------------------------------------
// LibSetIconPos - Function to set the position of a window's
// icon.
// Arg 1 - Window handle
// Arg 2 - X position for icon
// Arg 3 - Y position for icon
//----------------------------------------------------------- 
INT LibSetIconPos (LPTOKEN lptDest, LPTOKEN lptArg) {
   INT		sXPos, sYPos;
   HWND     hwndTarg;
	WINDOWPLACEMENT wp;
   
	if (lptDest->wData <= 3) 
		return ERR_TOO_FEW_PARMS;

   hwndTarg = (HWND) GetTokenVal (lptArg-1);
   sXPos = (INT) GetTokenVal (lptArg-2);
   sYPos = (INT) GetTokenVal (lptArg-3);
	if (!IsWindow (hwndTarg)) 
		return SetToken (lptDest, TTYPE_NUM, 0, MYFALSE);
		
	if (!GetWindowPlacement (hwndTarg, &wp))
		return SetToken (lptDest, TTYPE_NUM, 0, MYFALSE);
	wp.ptMinPosition.x = sXPos;
	wp.ptMinPosition.y = sYPos;
	return SetToken (lptDest, TTYPE_NUM, 0, 
	                 (LONG)SetWindowPlacement (hwndTarg, &wp));
}
//-----------------------------------------------------------
// LibCharVal - Function that returns the ASCII value of a
// character
// Arg 1 - Number to convert
//----------------------------------------------------------- 
INT LibCharVal (LPTOKEN lptDest, LPTOKEN lptArg) {
	LPSTR lpszStr;

	if (lptDest->wData <= 1) 
		return ERR_TOO_FEW_PARMS;
	lpszStr = GetTokenString (lptArg-1);

  	return SetToken (lptDest, TTYPE_NUM, 0, (DWORD) *lpszStr);
}
//-----------------------------------------------------------
// LibCharChr - Function that returns the ASCII character for
// a number
// Arg 1 - ASCII character to convert
//----------------------------------------------------------- 
INT LibCharChr (LPTOKEN lptDest, LPTOKEN lptArg) {
	char szStr[2];

	if (lptDest->wData <= 1) 
		return ERR_TOO_FEW_PARMS;
	szStr[0] = (char) GetTokenVal (lptArg-1);
	szStr[1] = '\0';

  	return SetToken (lptDest, TTYPE_STR, 0, (DWORD)(LPSTR) szStr);
}
//-----------------------------------------------------------
// LibStrInStr - Function that scans a string for a character
// Arg 1 - String
// Arg 2 - Character to look for
//----------------------------------------------------------- 
INT LibStrInStr (LPTOKEN lptDest, LPTOKEN lptArg) {
	LPSTR lpszStart, lpszStr, lpszChr;
	INT i, sLen;
	BOOL fFound = FALSE;

	if (lptDest->wData <= 2) 
		return ERR_TOO_FEW_PARMS;
		
	lpszStr = GetTokenString (lptArg-1);
	lpszChr = GetTokenString (lptArg-2);
	AnsiUpper (lpszStr);
	AnsiUpper (lpszChr);
	lpszStart = lpszStr;
	sLen = lstrlen (lpszChr);
	while ((*lpszStr != '\0') && !fFound) {
		for (i = 0; i < sLen && *lpszStr != '\0'; i++)
			if (*(lpszStr+i) != *(lpszChr+i))
				break;
		if (i == sLen)
			fFound = TRUE;
		else	
			lpszStr++;
	}
	if (fFound)
	  	return SetToken (lptDest, TTYPE_NUM, 0, (DWORD)(lpszStr - lpszStart));
	else	
	  	return SetToken (lptDest, TTYPE_NUM, 0, (DWORD)-1);
}
//-----------------------------------------------------------
// LibStrRev - Function that scans a string for a character
// Arg 1 - String
// Arg 2 - Character to look for
//----------------------------------------------------------- 
INT LibStrRev (LPTOKEN lptDest, LPTOKEN lptArg) {
	LPSTR lpszStart, lpszStr, lpszEnd;
	char ch;

	if (lptDest->wData <= 1) 
		return ERR_TOO_FEW_PARMS;
		
	lpszStr = GetTokenString (lptArg-1);
	lpszStart = lpszStr;
	lpszEnd = lpszStr + lstrlen (lpszStr) - 1;
	while (lpszStr < lpszEnd) {
		ch = *lpszStr;
		*lpszStr++ = *lpszEnd;
		*lpszEnd-- = ch;
	}			
  	return SetToken (lptDest, TTYPE_STR, 0, (DWORD)(LPSTR) lpszStart);
}
//-----------------------------------------------------------
// LibIsRunning - Function determines if a program is running
// Arg 1 - Name of EXE file
//----------------------------------------------------------- 
INT LibIsRunning (LPTOKEN lptDest, LPTOKEN lptArg) {
	LPSTR lpszEXEName;
	LPSTR lpszPSPCmdLine;
	LPSTR lpszTemp;
	UINT hTask;
	
	char *pszName;
	HWND hwndNext;
	BOOL fRunning = FALSE;

	if (lptDest->wData <= 1) 
		return ERR_TOO_FEW_PARMS;
	lpszEXEName = GetTokenString (lptArg-1);
	//See if EXE template includes path.
	lpszTemp = lpszEXEName;
	while ((*lpszTemp != '\0') && (*lpszTemp != '\\'))
		lpszTemp++;
	
	hwndNext = GetWindow (GetDesktopWindow(), GW_CHILD);
	while (hwndNext && !fRunning) {
		//If WinOldApp, look in task DB for name of DOS program
		hTask = GetWindowTask (hwndNext);
		if (IsWinOldApTask (hTask)) {
			//Make pointer to PSP command line
			lpszPSPCmdLine = MAKELP (hTask, 0x180);
			//Skip past parameters to name of program			
			lpszPSPCmdLine += lstrlen (lpszPSPCmdLine)+1;
			lstrcpy (szTemp, lpszPSPCmdLine);
		} else
			GetModuleFileName (GetWindowWord (hwndNext, GWW_HINSTANCE),
			                   szTemp, sizeof (szTemp));
		// If passed prog name doesn't have path, strip path from module name.
		pszName = szTemp;
		if (*lpszTemp == '\0') {
			pszName = strrchr (szTemp, '\\');
			if (pszName == 0)
				pszName = szTemp;
			else 	
				pszName++;
		}		
		fRunning = CmpFilename (lpszEXEName, pszName);
		hwndNext = GetWindow (hwndNext, GW_HWNDNEXT);
	}	
  	return SetToken (lptDest, TTYPE_NUM, 0, (DWORD) fRunning);
}
//-----------------------------------------------------------
// LibGetKeyState - Function that returns a key state 
// Arg 1 - Virtual key value
//----------------------------------------------------------- 
INT LibGetKeyState (LPTOKEN lptDest, LPTOKEN lptArg) {
	INT sKey;

	if (lptDest->wData <= 1) 
		return ERR_TOO_FEW_PARMS;
	sKey = GetVKey (GetTokenString (lptArg-1));
	if (sKey == 0)
		return ERR_BAD_SENDKEY_STR;
	
  	return SetToken (lptDest, TTYPE_NUM, 0, 
	           (DWORD)GetKeyState (sKey) & 0x8000);
}
//-----------------------------------------------------------
// LibGetKeyShift - Function that returns a shift lock state
// Arg 1 - Virtual key value
//----------------------------------------------------------- 
INT LibGetKeyShift (LPTOKEN lptDest, LPTOKEN lptArg) {
	INT sKey;

	if (lptDest->wData <= 1) 
		return ERR_TOO_FEW_PARMS;

	sKey = GetVKey (GetTokenString (lptArg-1));
	if (sKey == 0)
		return ERR_BAD_SENDKEY_STR; 

  	return SetToken (lptDest, TTYPE_NUM, 0, (DWORD)GetKeyState (sKey) & 1);
}
//-----------------------------------------------------------
// LibGetINIVal - Function to read a value from an INI file
// Arg 1 - Section 
// Arg 2 - Key
// Arg 3 - Default
// Arg 4 - INI file name
//----------------------------------------------------------- 
INT LibGetINIVal (LPTOKEN lptDest, LPTOKEN lptArg) {
	LPSTR lpszSec;
	LPSTR lpszKey;
	LPSTR lpszDef;
	LPSTR lpszININame;
	BOOL fFlag = FALSE;
	char *pszTemp;
   
	if (lptDest->wData <= 4) 
		return ERR_TOO_FEW_PARMS;

	lpszSec = GetTokenString (lptArg-1);
	lpszKey = GetTokenString (lptArg-2);
	lpszDef = GetTokenString (lptArg-3);
	lpszININame = GetTokenString (lptArg-4);
	if (lstrlen (lpszKey) == 0) {
		lpszKey = 0;
		fFlag = TRUE;
	}
	GetPrivateProfileString (lpszSec, lpszKey, lpszDef, szTemp,
	                         sizeof (szTemp), lpszININame);
	//If returning section names, change zeros with ; chars
	if (fFlag) {
		pszTemp = szTemp;
		while (*pszTemp != '\0') {
			while (*pszTemp != '\0')
				pszTemp++;
			if (*(pszTemp+1) != '\0')
				*pszTemp = '|';
			pszTemp++;
		}
	}
  	return SetToken (lptDest, TTYPE_STR, 0, (DWORD)(LPSTR) szTemp);
}
//-----------------------------------------------------------
// LibSetINIVal - Function to read a value from an INI file
// Arg 1 - Section 
// Arg 2 - Key
// Arg 3 - Value
// Arg 4 - INI file name
//----------------------------------------------------------- 
INT LibSetINIVal (LPTOKEN lptDest, LPTOKEN lptArg) {
	INT sRC;
	LPSTR lpszSec;
	LPSTR lpszKey;
	LPSTR lpszVal;
	LPSTR lpszININame;

	if (lptDest->wData <= 4) 
		return ERR_TOO_FEW_PARMS;

	lpszSec = GetTokenString (lptArg-1);
	lpszKey = GetTokenString (lptArg-2);
	lpszVal = GetTokenString (lptArg-3);
	lpszININame = GetTokenString (lptArg-4);
	
	if (lstrlen (lpszSec) == 0)
		lpszSec = 0;	
	if (lstrlen (lpszKey) == 0)
		lpszKey = 0;	
	if (lstrlen (lpszVal) == 0)
		lpszVal = 0;	
	sRC = WritePrivateProfileString (lpszSec, lpszKey, lpszVal, lpszININame);

	return SetToken (lptDest, TTYPE_NUM, 0, (LONG) sRC);
}
//----------------------------------------------------------- 
// GetFullpath - Helper function for following routines.
//----------------------------------------------------------- 
char *GetFullpath (LPSTR lpIn) {
	lstrcpyn (szTemp1, lpIn, sizeof (szTemp1));
	strupr (szTemp1);
	_fullpath (szTemp, szTemp1, sizeof (szTemp));
	return szTemp;
}	
//-----------------------------------------------------------
// LibFullPath - Function to return a fully specified filename
// Arg 1 - relative filename
//----------------------------------------------------------- 
INT LibFullPath (LPTOKEN lptDest, LPTOKEN lptArg) {

	if (lptDest->wData <= 1) 
		return ERR_TOO_FEW_PARMS;

  	return SetToken (lptDest, TTYPE_STR, 0, (DWORD)(LPSTR)
  	                 GetFullpath (GetTokenString (lptArg-1)));
}
//-----------------------------------------------------------
// LibFileDrive - Function to return the drive segment of a 
// filename.
// Arg 1 - relative filename
//----------------------------------------------------------- 
INT LibFileDrive (LPTOKEN lptDest, LPTOKEN lptArg) {
	char *pszName;

	if (lptDest->wData <= 1) 
		return ERR_TOO_FEW_PARMS;

	pszName = GetFullpath (GetTokenString(lptArg-1));
	//Truncate at directory start
	*(pszName+2) = '\0';
  	return SetToken (lptDest, TTYPE_STR, 0, (DWORD)(LPSTR) pszName);
}
//-----------------------------------------------------------
// LibFileDir - Function to return the directory segment
// of a filename.
// Arg 1 - relative filename
//----------------------------------------------------------- 
INT LibFileDir (LPTOKEN lptDest, LPTOKEN lptArg) {
	char *pszName;

	if (lptDest->wData <= 1) 
		return ERR_TOO_FEW_PARMS;

	pszName = GetFullpath (GetTokenString(lptArg-1));
	pszName += 2;                        //Point to dir
	*strrchr (pszName, '\\') = '\0';     //Truncate at filename
	//If root dir, replace \.
	if (*pszName == '\0') {
		*pszName = '\\';
		*(pszName+1) = '\0';
	}
  	return SetToken (lptDest, TTYPE_STR, 0, (DWORD)(LPSTR) pszName);
}
//-----------------------------------------------------------
// LibFilePath - Function to return the drive and dir segment
// of a filename.
// Arg 1 - relative filename
//----------------------------------------------------------- 
INT LibFilePath (LPTOKEN lptDest, LPTOKEN lptArg) {
	char *pszName;

	if (lptDest->wData <= 1) 
		return ERR_TOO_FEW_PARMS;

	pszName = GetFullpath (GetTokenString(lptArg-1));
	*strrchr (pszName, '\\') = '\0';     //Truncate at filename
	//If root dir, replace \.
	if (*pszName == '\0') {
		*pszName = '\\';
		*(pszName+1) = '\0';
	}
  	return SetToken (lptDest, TTYPE_STR, 0, (DWORD)(LPSTR) pszName);
}
//-----------------------------------------------------------
// LibFileName - Function to return the name segment of a 
// filename.
// Arg 1 - relative filename
//----------------------------------------------------------- 
INT LibFileName (LPTOKEN lptDest, LPTOKEN lptArg) {
	char *pszName, *pszExt;

	if (lptDest->wData <= 1) 
		return ERR_TOO_FEW_PARMS;

	pszName = GetFullpath (GetTokenString(lptArg-1));
	//Start at filename
	pszName = strrchr (pszName, '\\') + 1;
	pszExt = strchr (pszName, '.');
	if (pszExt)
		*pszExt = '\0';
  	return SetToken (lptDest, TTYPE_STR, 0, (DWORD)(LPSTR) pszName);
}
//-----------------------------------------------------------
// LibFileExt - Function to return the file extension of a
// filename.
// Arg 1 - relative filename
//----------------------------------------------------------- 
INT LibFileExt (LPTOKEN lptDest, LPTOKEN lptArg) {
	char *pszName, *pszExt;

	if (lptDest->wData <= 1) 
		return ERR_TOO_FEW_PARMS;

	pszName = GetFullpath (GetTokenString(lptArg-1));
	//Start at filename
	pszName = strrchr (pszName, '\\');
	pszExt = strchr (pszName, '.');
	if (pszExt)
	  	return SetToken (lptDest, TTYPE_STR, 0, (DWORD)(LPSTR) pszExt+1);
	else	
	  	return SetToken (lptDest, TTYPE_STR, 0, (DWORD)(LPSTR) "");
}
//-----------------------------------------------------------
// LibGetFileName - Function to display a File Open dialog
// Arg 1 - Name of filter
// Arg 2 - Filename filter
//----------------------------------------------------------- 
INT LibGetFileName (LPTOKEN lptDest, LPTOKEN lptArg) {
   OPENFILENAME ofn;
	char *pszFilter;

	pszFilter = szFilter;
	if (lptDest->wData >= 2) {
		lstrcpy (szFilter, GetTokenString(lptArg-1));
		pszFilter += lstrlen (pszFilter)+1;
		lstrcpy (pszFilter, GetTokenString(lptArg-2));
		pszFilter += lstrlen (pszFilter)+1;
	}
	strcpy (pszFilter, "All Files (*.*)");
	pszFilter += lstrlen (pszFilter)+1;
	strcpy (pszFilter, "*.*");
	pszFilter += lstrlen (pszFilter)+1;
	*pszFilter = '\0';

	_fmemset (&ofn, 0, sizeof (ofn));
   szTemp[0] = '\0';

	ofn.lStructSize = sizeof (ofn);
	ofn.hwndOwner = hMain;
	ofn.hInstance = hInst;
	ofn.lpstrFilter = szFilter;
	ofn.nFilterIndex = 1;
	ofn.lpstrFile = szTemp;
	ofn.nMaxFile = sizeof (szTemp);
	ofn.Flags = OFN_HIDEREADONLY;

 	if (GetOpenFileName (&ofn))
	  	return SetToken (lptDest, TTYPE_STR, 0, (DWORD)(LPSTR) szTemp);
	else		
	  	return SetToken (lptDest, TTYPE_NUM, 0, (DWORD) -1);
}
//-----------------------------------------------------------
// LibGetSaveName - Function to display a File Save dialog
// Arg 1 - Name of filter
// Arg 2 - Filename filter
//----------------------------------------------------------- 
INT LibGetSaveName (LPTOKEN lptDest, LPTOKEN lptArg) {
   OPENFILENAME ofn;
	char *pszFilter;

	pszFilter = szFilter;
	if (lptDest->wData >= 2) {
		lstrcpy (szFilter, GetTokenString(lptArg-1));
		pszFilter += lstrlen (pszFilter)+1;
		lstrcpy (pszFilter, GetTokenString(lptArg-2));
		pszFilter += lstrlen (pszFilter)+1;
	}
	strcpy (pszFilter, "All Files (*.*)");
	pszFilter += lstrlen (pszFilter)+1;
	strcpy (pszFilter, "*.*");
	pszFilter += lstrlen (pszFilter)+1;
	*pszFilter = '\0';

	_fmemset (&ofn, 0, sizeof (ofn));
   szTemp[0] = '\0';

	ofn.lStructSize = sizeof (ofn);
	ofn.hwndOwner = hMain;
	ofn.hInstance = hInst;
	ofn.lpstrFilter = szFilter;
	ofn.nFilterIndex = 1;
	ofn.lpstrFile = szTemp;
	ofn.nMaxFile = sizeof (szTemp);
	ofn.Flags = OFN_HIDEREADONLY | OFN_OVERWRITEPROMPT;

 	if (GetSaveFileName (&ofn))
	  	return SetToken (lptDest, TTYPE_STR, 0, (DWORD)(LPSTR) szTemp);
	else		
	  	return SetToken (lptDest, TTYPE_NUM, 0, (DWORD) -1);
}
//------------------------------------------------------------
// WinGetDate - Predefined func that returns the system date.
//------------------------------------------------------------ 
INT LibGetDate (LPTOKEN lptDest, LPTOKEN lptArg) {

	_strdate (szTemp);
	return SetToken (lptDest, TTYPE_STR, 0, (DWORD)(LPSTR) szTemp);
}  	
//------------------------------------------------------------
// WinGetTime - Predefined func that returns the system Time.
//------------------------------------------------------------ 
INT LibGetTime (LPTOKEN lptDest, LPTOKEN lptArg) {

	_strtime (szTemp);
	return SetToken (lptDest, TTYPE_STR, 0, (DWORD)(LPSTR) szTemp);
}  	
