/*
 *  WinPrn prints text files one, two, and four pages sheet. First
 *  published in PC Magazine's June 14, 1994 Utilities column.  
 */

#include <windows.h>
#include <commdlg.h>
#include <shellapi.h>
#include <print.h>
#include <stdlib.h>
#include <memory.h>
#include <stdio.h>
#include <time.h>
#include "winprn.h"

#define MAXFONTS    64
#define IOSIZE      0x4000

long FAR PASCAL WndProc (HWND, WORD, WORD, LONG);
BOOL FAR PASCAL OptionsDlgProc (HWND, WORD, WORD, LONG);
BOOL FAR PASCAL PrintDlgProc (HWND, WORD, WORD, LONG);
BOOL FAR PASCAL AboutDlgProc (HWND, WORD, WORD, LONG);
BOOL FAR PASCAL EnumFontsProc (LPLOGFONT, LPNEWTEXTMETRIC, int, LPARAM);
BOOL FAR PASCAL AbortProc (HDC, int);

int PrintFile (HWND, HINSTANCE, char *);
HDC CreatePrinterDC (NPDEVMODE);
NPDEVMODE BuildDevMode (HWND, int);
int BuildFontList (HWND, HINSTANCE);
void PrintHeader (HDC, int, int, int, int, int, char *, char *);
DWORD PrintPage (HDC, int, int, int, int, int, DWORD, char FAR *);
void MakeDateString (char *);
void ODBDrawButton (DRAWITEMSTRUCT FAR *, HBITMAP, HBITMAP);
void ODBChangeSelection (DRAWITEMSTRUCT FAR *);
void ODBChangeFocus (DRAWITEMSTRUCT FAR *);
void DrawButtonBorder (HDC, RECT *);
void DrawPageImage (HDC, int, int);
void PaintPanel (HDC, int, int, HBRUSH);
void InitProgram (HWND);
void InitGroupRect (HWND, int, RECT *, BOOL);
void DrawGroupRect (HDC, RECT *);
void PositionWindow (HWND);
void CenterDialog (HWND);
void ReadIniFile (void);
void SaveSettings (HWND);

BOOL bAbort;                            // Abort printing flag
HWND hwndPrintDlg;                      // Print dialog box handle

int nLayout;                            // Layout number (0, 1, or 2)
int nPanelIndex;                        // Number of leftmost panel (0 or 1)
char szFileName[128];                   // Name of file to print
char aszFaces[MAXFONTS][LF_FACESIZE];   // Array of typeface names
int nCurrentFont;                       // Index of current font
int nNumFonts;                          // Number of fonts
int nTabLength;                         // Tab length (in spaces)
BOOL bPrintOutline;                     // TRUE = Print page outlines
BOOL bPrintHeader;                      // TRUE = Print page headers
int nPaperSize;                         // 0=Letter, 1=A4
int nTabChars;                          // Tab characters remaining
int xDialog, yDialog;                   // Dialog box coordinates

RECT rcGroup;                           // Group rectangle coordinates
RECT rcPanel[2];                        // Left and right panel coordinates
RECT rcPortrait, rcLandscape;           // Page image coordinates

char szIniFile[128];                    // INI file name
char szSection[] = "Settings";          // INI file section name
char *szEntry[] = {                     // INI file entry names
    "WindowPos",
    "Layout",
    "PanelIndex",
    "Typeface",
    "TabLength",
    "PrintOutline",
    "PrintHeader",
    "PaperSize"
};

char *szError[] = {
    "WinPrn Error",
    "Unable to create a device context for the default printer (CreateDC "\
        "failed or there is not a default printer specified in WIN.INI)",
    "Unable to create the printer font (CreateFontIndirect failed)",
    "Unable to start the print job (StartDoc failed)",
    "There must be at least one fixed-pitch TrueType font installed for "\
        "WinPrn to work. If you removed the Courier New font that came "\
        "with Windows, reinstall it and start WinPrn again.",
    "Unable to open ",
    "Error reading ",
    "There is not enough memory to print the file (LocalAlloc failed)",
    "There is not enough memory to print the file (GlobalAlloc failed)",
    "Unable to continue the print job (StartPage failed)",
    "Printing was terminated because Print Manager ran out of memory or "\
        "disk space",
    "Unable to set the paper orientation (ExtDeviceMode failed or is not "\
        "supported on the default printer, or there is not a default "\
        "printer specified in WIN.INI)"
};

char *szFilter[] = {
    "All Files (*.*)", "*.*",
    "Text Files (*.TXT)", "*.TXT", ""
};

//
// The following array of structures holds information describing the
// physical layout of the printed page. The variable nLayout is used as
// an index into the array to retrieve the parameters for a given layout.
// The term "logical page" refers to one of the pages printed on a sheet
// of paper (the "physical page"). The first layout prints one logical page
// per physical page, the second prints two, and the third prints four.
//
struct {
    int nOrientation;           // Page orientation (0=portrait, 1=landscape)
    int nPagesPerSheet;         // Number of logical pages per physical page
    int nCharHeight;            // Desired character height (in twips)
    int nPageWidth;             // Logical page width (in twips)
    int nLinesPerPage;          // Number of lines per logical page
    int dy;                     // Distance between lines (in twips)
    int x[4];                   // X coordinates of logical pages (letter size)
    int y[4];                   // Y coordinates of logical pages (letter size)
    int Ax[4];                  // X coordinates of logical pages (A4)
    int Ay[4];                  // Y coordinates of logical pages (A4)
} pl[3] = {
    0, 1, -160, 10080, 68, -200,  720,    0,   0,    0,  -720,     0,     0,     0,
//                                695,    0,   0,    0, -1224,     0,     0,     0,
                                  553,    0,   0,    0, -1224,     0,     0,     0,
    1, 2, -120,  6300, 66, -150,  720, 8100,   0,    0,  -720,  -720,     0,     0,
//                               1198, 8914,   0,    0,  -553,  -553,     0,     0,
                                 1056, 8772,   0,    0,  -553,  -553,     0,     0,
    0, 4,  -80,  4500, 63, -100,  720, 6300, 720, 6300,  -720,  -720, -8080, -8080,
//                                750, 6219, 750, 6219, -1056, -1056, -8752, -8752
                                  608, 6077, 608, 6077, -1056, -1056, -8752, -8752
};

/*
 *  Function WinMain.
 */

int PASCAL WinMain (HINSTANCE hInstance, HINSTANCE hPrevInstance,
                    LPSTR lpszCmdLine, int nCmdShow)
{
    static char szAppName[] = "WinPrn";
    WNDCLASS wndclass;
    HWND hwnd;
    MSG msg;

    if (!hPrevInstance) {
        wndclass.style = 0;
        wndclass.lpfnWndProc = (WNDPROC) WndProc;
        wndclass.cbClsExtra = 0;
        wndclass.cbWndExtra = DLGWINDOWEXTRA;
        wndclass.hInstance = hInstance;
        wndclass.hIcon = LoadIcon (hInstance, szAppName);
        wndclass.hCursor = LoadCursor (NULL, IDC_ARROW);
        wndclass.hbrBackground = NULL;
        wndclass.lpszMenuName = NULL;
        wndclass.lpszClassName = szAppName;

        RegisterClass (&wndclass);
    }

    hwnd = CreateDialog (hInstance, szAppName, 0, NULL);

    InitProgram (hwnd);

    ShowWindow (hwnd, nCmdShow == SW_SHOWMAXIMIZED ? SW_SHOW : nCmdShow);
    UpdateWindow (hwnd);

    while (GetMessage (&msg, NULL, 0, 0))
        if (!IsDialogMessage (hwnd, &msg))
            DispatchMessage (&msg);

    return msg.wParam;
}

/*
 *  WndProc processes messages to the main window
 */

long FAR PASCAL WndProc (HWND hwnd, WORD message, WORD wParam, LONG lParam)
{
    static HINSTANCE hInstance;
    static HBRUSH hPatternBrush;
    static HBITMAP hDisabledButton, hEnabledButton;
    static OPENFILENAME ofn;
    static BOOL bFirstTime = TRUE;

    FARPROC lpfnOptionsDlgProc, lpfnAboutDlgProc;
    int nResult, nFiles, i;
    char szFile[128];
    HBITMAP hBitmap;
    HBRUSH hBrush;
    HMENU hSysMenu;
    PAINTSTRUCT ps;
    POINT point;
    RECT rect;
    HDC hdc;

    switch (message) {

    case WM_CREATE:
        //
        // Get ready!
        //
        hInstance = ((LPCREATESTRUCT) lParam)->hInstance;
        i = GetModuleFileName (hInstance, szIniFile, sizeof (szIniFile));
        while (szIniFile[i--] != 0x2E);
        lstrcpy (&szIniFile[i+2], "INI");

        if (BuildFontList (hwnd, hInstance))
            return -1;

        hPatternBrush = CreatePatternBrush (hBitmap = LoadBitmap (hInstance,
            "Background"));
        DeleteObject (hBitmap);

        hDisabledButton = LoadBitmap (hInstance, "DisabledButton");
        hEnabledButton = LoadBitmap (hInstance, "EnabledButton");

        hSysMenu = GetSystemMenu (hwnd, FALSE);
        DeleteMenu (hSysMenu, SC_SIZE, MF_BYCOMMAND);
        DeleteMenu (hSysMenu, SC_MAXIMIZE, MF_BYCOMMAND);
        AppendMenu (hSysMenu, MF_SEPARATOR, 0, NULL);
        AppendMenu (hSysMenu, MF_STRING, IDM_ABOUT, "&About WinPrn...");

        DragAcceptFiles (hwnd, TRUE);
        PositionWindow (hwnd);
        ReadIniFile ();
        return 0;

    case WM_PAINT:
        //
        // Repaint the window.
        //
        hdc = BeginPaint (hwnd, &ps);
        DrawGroupRect (hdc, &rcGroup);

        DrawPageImage (hdc, 0, nPanelIndex);
        DrawPageImage (hdc, 1, nPanelIndex + 1);

        EndPaint (hwnd, &ps);
        return 0;

    case WM_ERASEBKGND:
        //
        // Paint the window background.
        //
        GetClientRect (hwnd, &rect);
        UnrealizeObject (hPatternBrush);
        point.x = point.y = 0;
        ClientToScreen (hwnd, &point);
        SetBrushOrg ((HDC) wParam, point.x, point.y); 
        FillRect ((HDC) wParam, &rect, hPatternBrush);
        return 1;

    case WM_CTLCOLOR:
        //
        // Paint the backgrounds of all the controls except the scroll bar
        // light gray.
        //
        if (LOWORD (lParam) == GetDlgItem (hwnd, IDD_SCROLLBAR))
            return NULL;
        SetBkColor ((HDC) wParam, RGB (192, 192, 192));
        return GetStockObject (LTGRAY_BRUSH);

    case WM_DRAWITEM:
        //
        // Paint the owner-draw Print button.
        //
        switch (((DRAWITEMSTRUCT FAR *) lParam)->itemAction) {

        case ODA_DRAWENTIRE:
            ODBDrawButton ((DRAWITEMSTRUCT FAR *) lParam, hDisabledButton,
                hEnabledButton);
            return TRUE;

        case ODA_SELECT:
            ODBChangeSelection ((DRAWITEMSTRUCT FAR *) lParam);
            return TRUE;

        case ODA_FOCUS:
            ODBChangeFocus ((DRAWITEMSTRUCT FAR *) lParam);
            return TRUE;
        }
        break;

    case WM_COMMAND:
        switch (wParam) {

        case IDD_SELECT:
            //
            // Select a file and print it.
            //
            if (bFirstTime) {
                bFirstTime = FALSE;
                ofn.nFilterIndex = 0;
            }
            szFile[0] = 0;

            ofn.lStructSize = sizeof (OPENFILENAME);
            ofn.hwndOwner = hwnd;
            ofn.hInstance = NULL;
            ofn.lpstrFilter = szFilter[0];
            ofn.lpstrCustomFilter = NULL;
            ofn.lpstrFile = szFile;
            ofn.nMaxFile = sizeof (szFile);
            ofn.lpstrFileTitle = NULL;
            ofn.lpstrInitialDir = NULL;
            ofn.lpstrTitle = NULL;
            ofn.Flags = OFN_FILEMUSTEXIST | OFN_PATHMUSTEXIST |
                OFN_HIDEREADONLY;
            ofn.lpstrDefExt = NULL;

            if (GetOpenFileName (&ofn)) {
                lstrcpy (szFileName, szFile);
                AnsiLower (szFileName);
                SetDlgItemText (hwnd, IDD_FILENAME, szFileName);
                EnableWindow (GetDlgItem (hwnd, IDD_PRINT), TRUE);
            }
            return 0;

        case IDD_PRINT:
            //
            // Print the file.
            //
            xDialog = yDialog = 0;
            PrintFile (hwnd, hInstance, szFileName);
            return 0;

        case IDD_OPTIONS:
            //
            // Allow the user to specify configuration options.
            //
            lpfnOptionsDlgProc = MakeProcInstance (OptionsDlgProc, hInstance);
            DialogBox (hInstance, "Options", hwnd, lpfnOptionsDlgProc);
            FreeProcInstance (lpfnOptionsDlgProc);
            return 0;

        case IDD_CLOSE:
            //
            // Send a WM_CLOSE message to terminate the program.
            //
            SendMessage (hwnd, WM_CLOSE, 0, 0L);
            return 0;

        case IDOK:
            //
            // Simulate a click of the Print button if that button has the
            // input focus.
            //
            if (GetFocus () == GetDlgItem (hwnd, IDD_PRINT)) {
                PostMessage (hwnd, WM_COMMAND, IDD_PRINT, 0L);
                return 0;
            }
            break;

        case IDCANCEL:
            //
            // Minimize the window when the Esc key is pressed.
            //
            ShowWindow (hwnd, SW_MINIMIZE);
            return 0;
        }
        break;

    case WM_DROPFILES:
        //
        // Print drag-and-drop files.
        //
        lstrcpy (szFile, szFileName);
        DragAcceptFiles (hwnd, FALSE);
        nFiles = DragQueryFile ((HDROP) wParam, -1, NULL, 0);

        if (nFiles > 0) {
            xDialog = yDialog = 0;
            for (i=0; i<nFiles; i++) {
                DragQueryFile ((HDROP) wParam, i, szFileName,
                    sizeof (szFileName));
                if (PrintFile (hwnd, hInstance, szFileName))
                    break;
            } 
        }

        DragFinish ((HDROP) wParam);
        DragAcceptFiles (hwnd, TRUE);
        lstrcpy (szFileName, szFile);
        return 0;

    case WM_HSCROLL:
        //
        // Process messages from the scroll bar.
        //
        rect.left = rcPanel[0].left;
        rect.top = rcPanel[0].top;
        rect.right = rcPanel[1].right;
        rect.bottom = rcPanel[1].bottom;

        switch (wParam) {

        case SB_LINEUP:
        case SB_PAGEUP:
            //
            // Scroll the page layout window left if the left arrow or any
            // part of the scroll bar is clicked and nPanelIndex == 1.
            //
            if (nPanelIndex == 1) {
                nPanelIndex--;
                SetScrollPos (HIWORD (lParam), SB_CTL, 0, TRUE);
                InvalidateRect (hwnd, &rect, TRUE);
            }
            return 0;

        case SB_LINEDOWN:
        case SB_PAGEDOWN:
            //
            // Scroll the page layout window right if the right arrow or any
            // part of the scroll bar is clicked and nPanelIndex == 0.
            //
            if (nPanelIndex == 0) {
                nPanelIndex++;
                SetScrollPos (HIWORD (lParam), SB_CTL, 100, TRUE);
                InvalidateRect (hwnd, &rect, TRUE);
            }
            return 0;

        case SB_THUMBPOSITION:
            //
            // Scroll the page layout window if the scroll bar thumb is moved
            // left or right.
            //
            if ((nPanelIndex == 0) && (LOWORD (lParam) > 0)) {
                nPanelIndex++;
                SetScrollPos (HIWORD (lParam), SB_CTL, 100, TRUE);
                InvalidateRect (hwnd, &rect, TRUE);
            }
            else if ((nPanelIndex == 1) && (LOWORD (lParam) < 100)){
                nPanelIndex--;
                SetScrollPos (HIWORD (lParam), SB_CTL, 0, TRUE);
                InvalidateRect (hwnd, &rect, TRUE);
            }
            return 0;
        }
        break;

    case WM_LBUTTONDOWN:
        //
        // Select a new page layout if an inactive layout is clicked.
        //
        if (PtInRect (&rcPanel[0], MAKEPOINT (lParam)) &&
            (nLayout != nPanelIndex)) {
            nLayout = nPanelIndex;
            hdc = GetDC (hwnd);
            PaintPanel (hdc, 1, nPanelIndex + 1, GetStockObject (LTGRAY_BRUSH));
            hBrush = CreateSolidBrush (RGB (255, 255, 0));
            PaintPanel (hdc, 0, nPanelIndex, hBrush);
            DeleteObject (hBrush);
            ReleaseDC (hwnd, hdc);
            return 0;
        }
        else if (PtInRect (&rcPanel[1], MAKEPOINT (lParam)) &&
            (nLayout != (nPanelIndex + 1))) {
            nLayout = nPanelIndex + 1;
            hdc = GetDC (hwnd);
            PaintPanel (hdc, 0, nPanelIndex, GetStockObject (LTGRAY_BRUSH));
            hBrush = CreateSolidBrush (RGB (255, 255, 0));
            PaintPanel (hdc, 1, nPanelIndex + 1, hBrush);
            DeleteObject (hBrush);
            ReleaseDC (hwnd, hdc);
            return 0;
        }
        break;

    case WM_SYSCOMMAND:
        //
        // Display the About WinPrn dialog box.
        //
        if (wParam == IDM_ABOUT) {
            lpfnAboutDlgProc = MakeProcInstance (AboutDlgProc, hInstance);
            DialogBox (hInstance, "AboutBox", hwnd, lpfnAboutDlgProc);
            FreeProcInstance (lpfnAboutDlgProc);
            return 0;
        }
        break;

    case WM_CLOSE:
        //
        // Save program settings and destroy the window.
        //
        SaveSettings (hwnd);
        DestroyWindow (hwnd);
        return 0;

    case WM_ENDSESSION:
        //
        // Save program settings before terminating.
        //
        if (wParam)
            SaveSettings (hwnd);
        return 0;

    case WM_DESTROY:
        //
        // Clean up before terminating.
        //
        DeleteObject (hEnabledButton);
        DeleteObject (hDisabledButton);
        DeleteObject (hPatternBrush);
        PostQuitMessage (0);
        return 0;
    }
    return DefDlgProc (hwnd, message, wParam, lParam);
}

/*
 *  PrintFile outputs a file to the printer.
 *
 *  Input parameters:
 *
 *    hwnd = Window handle
 *    hInstance = Instance handle
 *    szFileName = Pointer to file name
 *
 *  Returns:
 *
 *    0 = Printing completed
 *    Nonzero = An error occurred or printing was canceled by the user
 */

int PrintFile (HWND hwnd, HINSTANCE hInstance, char *szFileName)
{
    HFILE hFile;
    DWORD dwFileSize, dwBytesRemaining;
    HGLOBAL hGlobal;
    DOCINFO di;
    LOGFONT lf;
    HFONT hFont, hOldFont;
    HDC hdcPrn;
    char FAR *fpText;
    NPDEVMODE npDevMode;
    OUTLINETEXTMETRIC NEAR *npOtm;
    char szBuffer[128], szDate[16];
    FARPROC lpfnPrintDlgProc, lpfnAbortProc;
    int nResult, nPage, nSheet, nSize, i;
    int nCharWidth, nCharsPerLine;
    int x, y, x1, y1, x2, y2;

    //
    // Open the file and determine the file size.
    //
    if ((hFile = _lopen (szFileName, READ)) == HFILE_ERROR) {
        lstrcpy (szBuffer, szError[IDERR_LOPENFAILED]);
        lstrcat (szBuffer, szFileName);
        MessageBox (hwnd, szBuffer, szError[IDERR_TITLE],
            MB_ICONINFORMATION | MB_OK);
        return IDERR_LOPENFAILED;
    }        

    dwFileSize = _llseek (hFile, 0, 2); 
    _llseek (hFile, 0, 0);
    dwBytesRemaining = dwFileSize;

    //
    // Allocate a file I/O buffer in the global heap.
    //
    if ((hGlobal = GlobalAlloc (GMEM_FIXED, IOSIZE)) == NULL) {
        _lclose (hFile);        
        MessageBox (hwnd, szError[IDERR_GLOBALALLOCFAILED],
            szError[IDERR_TITLE], MB_ICONINFORMATION | MB_OK);
        return IDERR_GLOBALALLOCFAILED;
    }
    fpText = (char FAR *) MAKELONG (0, hGlobal);

    //
    // Create a device context for the default printer.
    //
    if ((npDevMode = BuildDevMode (hwnd, pl[nLayout].nOrientation)) == NULL) {
        GlobalFree (hGlobal);
        _lclose (hFile);
        MessageBox (hwnd, szError[IDERR_DEVMODEFAILED], szError[IDERR_TITLE],
            MB_ICONINFORMATION | MB_OK);
        return IDERR_DEVMODEFAILED;
    }

    if ((hdcPrn = CreatePrinterDC (npDevMode)) == NULL) {
        LocalFree ((HLOCAL) npDevMode);
        GlobalFree (hGlobal);
        _lclose (hFile);
        MessageBox (hwnd, szError[IDERR_CREATEPRINTERDCFAILED],
            szError[IDERR_TITLE], MB_ICONINFORMATION | MB_OK);
        return IDERR_CREATEPRINTERDCFAILED;
    }

    LocalFree ((HLOCAL) npDevMode);

    //
    // Set the mapping mode to MM_TWIPS.
    //
    SetMapMode (hdcPrn, MM_TWIPS);

    //
    // Create a logical font and select it into the DC.
    //
    memset (&lf, 0, sizeof (lf));
    lstrcpy (lf.lfFaceName, aszFaces[nCurrentFont]);
    lf.lfHeight = pl[nLayout].nCharHeight;
    lf.lfOutPrecision = OUT_TT_ONLY_PRECIS;

    if ((hFont = CreateFontIndirect ((LPLOGFONT) &lf)) == NULL) {
        DeleteDC (hdcPrn);
        GlobalFree (hGlobal);
        _lclose (hFile);
        MessageBox (hwnd, szError[IDERR_CREATEFONTINDIRECTFAILED],
            szError[IDERR_TITLE], MB_ICONINFORMATION | MB_OK);
        return IDERR_CREATEFONTINDIRECTFAILED;
    }

    hOldFont = SelectObject (hdcPrn, hFont);

    //
    // Compute the maximum number of characters that will fit on a line.
    //
    nSize = GetOutlineTextMetrics (hdcPrn, 0, NULL);

    if (nSize == 0)     // This statement fixes a bug in LaserJet IV printer
        nSize = 512;    // drivers, which return 0 in the previous statement

    if ((npOtm = (OUTLINETEXTMETRIC NEAR *) LocalAlloc (LMEM_FIXED, nSize)) ==
        NULL) {
        SelectObject (hdcPrn, hOldFont);
        DeleteObject (hFont);
        DeleteDC (hdcPrn);
        GlobalFree (hGlobal);
        _lclose (hFile);
        MessageBox (hwnd, szError[IDERR_LOCALALLOCFAILED],
            szError[IDERR_TITLE], MB_ICONINFORMATION | MB_OK);
        return IDERR_LOCALALLOCFAILED;
    }

    GetOutlineTextMetrics (hdcPrn, nSize, npOtm);
    nCharWidth = (npOtm->otmTextMetrics).tmAveCharWidth;
    nCharsPerLine = pl[nLayout].nPageWidth / nCharWidth;
    LocalFree ((HLOCAL) npOtm);

    //
    // Initialize a date string if page headers will be printed.
    //
    if (bPrintHeader)
        MakeDateString (szDate);    
    
    //
    // Prepare for printing.
    //
    EnableWindow (hwnd, FALSE);

    bAbort = FALSE;
    lpfnPrintDlgProc = MakeProcInstance (PrintDlgProc, hInstance);
    hwndPrintDlg = CreateDialog (hInstance, "PrintStatus", hwnd,
        lpfnPrintDlgProc); 

    lpfnAbortProc = MakeProcInstance (AbortProc, hInstance);
    SetAbortProc (hdcPrn, lpfnAbortProc);

    nPage = 1;
    nSheet = 1;
    nTabChars = 0;

    //
    // Call Windows's StartDoc function to start the print job.
    //
    di.cbSize = sizeof (DOCINFO);
    di.lpszDocName = szFileName;
    di.lpszOutput = (LPCSTR) NULL;

    if (StartDoc (hdcPrn, &di) <= 0) {
        EnableWindow (hwnd, TRUE);
        DestroyWindow (hwndPrintDlg);
        FreeProcInstance (lpfnAbortProc);
        FreeProcInstance (lpfnPrintDlgProc);
        SelectObject (hdcPrn, hOldFont);
        DeleteObject (hFont);
        DeleteDC (hdcPrn);
        GlobalFree (hGlobal);
        _lclose (hFile);
        MessageBox (hwnd, szError[IDERR_STARTDOCFAILED], szError[IDERR_TITLE],
            MB_ICONINFORMATION | MB_OK);
        return IDERR_STARTDOCFAILED;
    }

    //
    // Output the file to the printer one page at a time. This while loop
    // executes as long as there are bytes remaining to be output (dwBytes-
    // Remaining != 0).
    //
    while (dwBytesRemaining) {
        //
        // Reinitialize the DC.
        //
        SetMapMode (hdcPrn, MM_TWIPS);
        SelectObject (hdcPrn, hFont);

        //
        // Start a new page and abort the print job if StartPage returns
        // an error value.
        //
        if (StartPage (hdcPrn) <= 0) {
            EnableWindow (hwnd, TRUE);
            DestroyWindow (hwndPrintDlg);
            AbortDoc (hdcPrn);
            FreeProcInstance (lpfnAbortProc);
            FreeProcInstance (lpfnPrintDlgProc);
            SelectObject (hdcPrn, hOldFont);
            DeleteObject (hFont);
            DeleteDC (hdcPrn);
            GlobalFree (hGlobal);
            _lclose (hFile);
            MessageBox (hwnd, szError[IDERR_STARTPAGEFAILED],
                szError[IDERR_TITLE], MB_ICONINFORMATION | MB_OK);
            return IDERR_STARTPAGEFAILED;
        }

        //
        // Update the status display in the dialog box.
        //
        wsprintf (szBuffer, (pl[nLayout].nPagesPerSheet > 1) ? "Sheet %d" :
            "Page %d", nSheet);
        SetDlgItemText (hwndPrintDlg, IDD_PAGE, szBuffer);

        //
        // Print the physical page. This while loop executes once for every
        // logical page on the physical page. At the beginning of the loop, a
        // block of data is read from the file. The PrintPage function outputs
        // the logical page, and at the end of the loop the file pointer is
        // reset to the first character not processed so the next call to
        // _lread will read from the proper location.
        //
        // Important: This logic assumes that one logical page will never
        // exceed IOSIZE bytes in length. IOSIZE = 0x4000 should be plenty
        // sufficient to ensure that this is true.
        //
        i = 0;
        while ((i < pl[nLayout].nPagesPerSheet) && dwBytesRemaining) {
            if (_lread (hFile, fpText, IOSIZE) == HFILE_ERROR) {
                EnableWindow (hwnd, TRUE);
                DestroyWindow (hwndPrintDlg);
                AbortDoc (hdcPrn);
                FreeProcInstance (lpfnAbortProc);
                FreeProcInstance (lpfnPrintDlgProc);
                SelectObject (hdcPrn, hOldFont);
                DeleteObject (hFont);
                DeleteDC (hdcPrn);
                GlobalFree (hGlobal);
                _lclose (hFile);
                lstrcpy (szBuffer, szError[IDERR_LREADFAILED]);
                lstrcat (szBuffer, szFileName);
                MessageBox (hwnd, szBuffer, szError[IDERR_TITLE],
                    MB_ICONINFORMATION | MB_OK);
                return IDERR_LREADFAILED;
            }

            x = (nPaperSize == 0) ? pl[nLayout].x[i] : pl[nLayout].Ax[i];
            y = (nPaperSize == 0) ? pl[nLayout].y[i] : pl[nLayout].Ay[i];

            if (bPrintOutline) {
                x1 = x - 360;
                y1 = y + 360;
                x2 = x + pl[nLayout].nPageWidth + 360;
                y2 = y + ((pl[nLayout].nLinesPerPage - 1) *
                    pl[nLayout].dy) + pl[nLayout].nCharHeight - 360;
                Rectangle (hdcPrn, x1, y1, x2, y2);
            }

            if (bPrintHeader) {
                PrintHeader (hdcPrn, x, y, pl[nLayout].dy, nCharsPerLine,
                    nPage, szFileName, szDate);

                dwBytesRemaining = PrintPage (hdcPrn, x,
                    y + (pl[nLayout].dy << 1), pl[nLayout].dy, nCharsPerLine,
                    pl[nLayout].nLinesPerPage - 2, dwBytesRemaining, fpText);
            }
            else
                dwBytesRemaining = PrintPage (hdcPrn, x, y, pl[nLayout].dy,
                    nCharsPerLine, pl[nLayout].nLinesPerPage,
                    dwBytesRemaining, fpText);

            _llseek (hFile, dwFileSize - dwBytesRemaining, 0); 
            nPage++;
            i++;
        }

        //
        // Increment the physical page count and end the page. If EndPage
        // returns an error value, abort the print job. An error return could
        // mean that the user canceled the print job, or that Print Manager
        // ran out of memory or disk space. If the latter, display an error
        // message informing the user what happened.
        //
        nSheet++;
        nResult = EndPage (hdcPrn);

        if (nResult < 0) {
            if (hwndPrintDlg != 0) {
                EnableWindow (hwnd, TRUE);
                DestroyWindow (hwndPrintDlg);
            }

            AbortDoc (hdcPrn);
            FreeProcInstance (lpfnAbortProc);
            FreeProcInstance (lpfnPrintDlgProc);
            SelectObject (hdcPrn, hOldFont);
            DeleteObject (hFont);
            DeleteDC (hdcPrn);
            GlobalFree (hGlobal);
            _lclose (hFile);

            if ((nResult & SP_NOTREPORTED) && ((nResult == SP_OUTOFDISK) ||
                (nResult == SP_OUTOFMEMORY)))
                MessageBox (hwnd, szError[IDERR_ENDPAGEFAILED],
                    szError[IDERR_TITLE], MB_ICONINFORMATION | MB_OK);
            return IDERR_ENDPAGEFAILED;
        }
    }

    //
    // End the document, destroy the dialog box, clean up, and return.
    //
    EndDoc (hdcPrn);

    if (hwndPrintDlg != 0) {
        EnableWindow (hwnd, TRUE);
        DestroyWindow (hwndPrintDlg);
    }

    FreeProcInstance (lpfnPrintDlgProc);
    FreeProcInstance (lpfnAbortProc);
    SelectObject (hdcPrn, hOldFont);
    DeleteObject (hFont);
    DeleteDC (hdcPrn);
    GlobalFree (hGlobal);
    _lclose (hFile);
    return 0;
}

/*
 *  AbortProc is the callback function registered by SetAbortProc.
 *
 *  Input parameters:
 *
 *    hdcPrn = Printer device context handle
 *    nCode = Error code (0 means no error)
 *
 *  Returns:
 *
 *    TRUE if printing is to continue, FALSE if printing is to stop
 */

BOOL FAR PASCAL AbortProc (HDC hdcPrn, int nCode)
{
    MSG msg;

    while (!bAbort && PeekMessage (&msg, NULL, 0, 0, PM_REMOVE))
        if ((hwndPrintDlg == 0) || !(IsDialogMessage (hwndPrintDlg, &msg)))
            DispatchMessage (&msg);
    return !bAbort;
}

/*
 *  PrintHeader outputs a page header to the printer.
 *
 *  Input parameters:
 *
 *    hdcPrn = Printer device context handle
 *    x, y = Coordinates of page's upper-left corner (in twips)
 *    dy = Distance between consecutive lines (in twips)
 *    nCharsPerLine = Maximum number of characters per line
 *    nPageNumber = Logical page number
 *    szFileName = Pointer to the name of the file that is printing
 *    szDate = Pointer to string specifying the current date
 *
 *  Returns:
 *
 *    Nothing
 */

void PrintHeader (HDC hdcPrn, int x, int y, int dy, int nCharsPerLine,
                  int nPageNumber, char *szFileName, char *szDate)
{
    char szBuffer[256], szTemp[16];
    int nLength, i;

    memset (szBuffer, 0x20, sizeof (szBuffer));
    memcpy (szBuffer, "Date: ", 5);
    memcpy (&szBuffer[6], szDate, lstrlen (szDate));

    i = lstrlen (szFileName);
    while (szFileName[i--] != 0x5C);
    lstrcpy (szTemp, &szFileName[i + 2]);
    AnsiUpper (szTemp);

    nLength = lstrlen (szTemp);
    i = (nCharsPerLine - 11 - nLength) >> 1;
    memcpy (&szBuffer[i], "File name:", 10);
    memcpy (&szBuffer[i + 11], szTemp, nLength);

    wsprintf (szTemp, "Page %d", nPageNumber);
    nLength = lstrlen (szTemp);
    i = nCharsPerLine - nLength;
    memcpy (&szBuffer[i], szTemp, nLength);

    TextOut (hdcPrn, x, y, szBuffer, nCharsPerLine);
    MoveTo (hdcPrn, x, y + dy);
    LineTo (hdcPrn, x + LOWORD (GetTextExtent (hdcPrn, szBuffer,
        nCharsPerLine)), y + dy); 
}

/*
 *  PrintPage outputs one page of text to the printer.
 *
 *  Input parameters:
 *
 *    hdcPrn = Printer device context handle
 *    x, y = Coordinates of page's upper-left corner (in twips)
 *    dy = Distance between consecutive lines (in twips)
 *    nCharsPerLine = Maximum number of characters per line
 *    nLinesPerPage = Maximum number of lines per page (including header)
 *    dwBytesRemaining = Number of bytes in the file remaining to be printed
 *    fpText = Far pointer to text to print
 *
 *  Returns:
 *
 *    Number of bytes remaining to be printed after this page is printed.
 *    If 0, the end of the file was reached and printing should cease.
 *
 *  Notes:
 *
 *    It's a little ugly, but nTabChars, which holds a count of the number
 *    of characters remaining while a tab is expanded, is a global variable.
 *    PrintFile initializes nTabChars to 0 before calling PrintPage for the
 *    first time. The fact that nTabChars retains its value across calls to
 *    PrintPage allows tabs to be expanded across page boundaries.
 *
 *    This function assumes that one line will never hold more than 160
 *    characters. It this is not true, the size of the szBuffer array should
 *    be expanded to accommodate the longer line length.
 */

DWORD PrintPage (HDC hdcPrn, int x, int y, int dy, int nCharsPerLine,
                 int nLinesPerPage, DWORD dwBytesRemaining, char FAR *fpText)
{
    int xLeftMargin, nCharsThisLine, nLinesThisPage, i;
    unsigned char ch, szBuffer[160];

    i = 0;
    nCharsThisLine = 0;
    nLinesThisPage = 0;
    xLeftMargin = x;

    while ((dwBytesRemaining || nTabChars) &&
        (nLinesThisPage < nLinesPerPage)) {
        if (nTabChars == 0) {
            ch = fpText[i++];
            dwBytesRemaining--;
        }
        else {
            ch = 32;
            nTabChars--;
        }

        switch (ch) {

        case 13:
            //
            // Process a carriage return character.
            //
            if (nCharsThisLine) {
                TextOut (hdcPrn, x, y, szBuffer, nCharsThisLine);
                nCharsThisLine = 0;
            }
            x = xLeftMargin;
            break;

        case 10:
            //
            // Process a line feed character.
            //
            if (nCharsThisLine) {
                TextOut (hdcPrn, x, y, szBuffer, nCharsThisLine);
                x += GetTextExtent (hdcPrn, szBuffer, nCharsThisLine);
                nCharsThisLine = 0;
            }
            nLinesThisPage++;
            y += dy;
            break;

        case 9:
            //
            // Process a tab character.
            //
            nTabChars = nTabLength - (nCharsThisLine % nTabLength);
            break;

        case 12:
            //
            // Process a form feed character.
            //
            if (nCharsThisLine)
                TextOut (hdcPrn, x, y, szBuffer, nCharsThisLine);
            return dwBytesRemaining;

        case 26:
            //
            // Process an end-of-file character.
            //
            if (nCharsThisLine)
                TextOut (hdcPrn, x, y, szBuffer, nCharsThisLine);
            return 0;           

        default:
            //
            // Process any other character.
            //
            if (ch < 32)
                ch = 32;
            szBuffer[nCharsThisLine++] = ch;
            if ((nCharsThisLine == nCharsPerLine) ||
                ((dwBytesRemaining == 0) && (nTabChars == 0))) {
                TextOut (hdcPrn, x, y, szBuffer, nCharsThisLine);
                nLinesThisPage++;
                nCharsThisLine = 0;
                x = xLeftMargin;
                y += dy;
            }
            break;
        }
    }
    return dwBytesRemaining;
}

/*
 *  CreatePrinterDC creates a device context for the default printer.
 *
 *  Input parameters:
 *
 *    npDevMode = Pointer to DEVMODE structure containing printer parameters
 *        (NULL if default printer parameters are okay)
 *
 *  Returns:
 *
 *    Handle to printer device context handle if the call succeeds; NULL if
 *    the call to CreateDC fails, or if a default printer is not specified in
 *    WIN.INI
 */

HDC CreatePrinterDC (NPDEVMODE npDevMode)
{
    char szBuffer[128];
    char *szPrinterName, *szDriver, *szPort;
    int i;

    if (!GetProfileString ("windows", "device", "", szBuffer,
        sizeof (szBuffer)))
        return NULL; 

    szPrinterName = szBuffer;

    i = 0;
    while (szBuffer[i++] != 0x2C);
    szDriver = &szBuffer[i];
    szBuffer[i - 1] = 0;

    while (szBuffer[i++] != 0x2C);
    szPort = &szBuffer[i];
    szBuffer[i - 1] = 0;

    return CreateDC (szDriver, szPrinterName, szPort, npDevMode);
}

/*
 *  BuildDevMode builds a DEVMODE structure for the default printer.
 *
 *  Input parameters:
 *
 *    hwnd = Window handle
 *    nOrientation = Desired paper orientation (0=Portrait, 1=Landscape)
 *
 *  Returns:
 *
 *    Pointer to an intialized DEVMODE structure if the function succeeds;
 *    NULL if it fails
 *
 *  Note:
 *
 *    It is the caller's responsibility to free the local memory block
 *    allocated to hold the DEVMODE structure.
 */

NPDEVMODE BuildDevMode (HWND hwnd, int nOrientation)
{
    HINSTANCE hLib;
    LPFNDEVMODE lpfnEDM;
    char szBuffer[128], szDriver[80];
    char *szPrinterName, *szPort, *szTmp;
    NPDEVMODE npDevMode;
    int nSize, i;

    //
    // Read the default printer parameters from the [windows] section of
    // WIN.INI.
    //
    if (!GetProfileString ("windows", "device", "", szBuffer,
        sizeof (szBuffer)))
        return NULL; 

    szPrinterName = szBuffer;

    i = 0;
    while (szBuffer[i++] != 0x2C);
    szTmp = &szBuffer[i];
    szBuffer[i - 1] = 0;

    while (szBuffer[i++] != 0x2C);
    szPort = &szBuffer[i];
    szBuffer[i - 1] = 0;

    lstrcpy (szDriver, szTmp);
    lstrcat (szDriver, ".DRV");

    //
    // Load the printer driver and get the address of the ExtDeviceMode
    // function. Return NULL if the driver does not support ExtDeviceMode.
    //
    if ((hLib = LoadLibrary (szDriver)) < HINSTANCE_ERROR)
        return NULL;

    if ((lpfnEDM = (LPFNDEVMODE) GetProcAddress (hLib, "ExtDeviceMode")) ==
        NULL) {
        FreeLibrary (hLib);
        return NULL;
    }

    //
    // Call ExtDeviceMode once to determine the size of the DEVMODE structure.
    // Then allocate a local memory block large enough to hold the structure
    // and call ExtDeviceMode a second time to fill it in.
    //
    if ((nSize = (*lpfnEDM) (hwnd, (HMODULE) hLib, NULL, szPrinterName,
        szPort, NULL, NULL, 0)) < 0) {
        FreeLibrary (hLib);
        return NULL;
    }

    if ((npDevMode = (NPDEVMODE) LocalAlloc (LMEM_FIXED, nSize)) == NULL) {
        FreeLibrary (hLib);
        return NULL;
    }

    if ((*lpfnEDM) (hwnd, (HMODULE) hLib, npDevMode, szPrinterName, szPort,
        NULL, NULL, DM_OUT_BUFFER) < 0) {
        LocalFree ((HLOCAL) npDevMode);
        FreeLibrary (hLib);
        return NULL;
    }

    //
    // Call ExtDeviceMode a third time to modify DEVMODE's dmOrientation field.
    //
    npDevMode->dmFields = DM_ORIENTATION; 
    npDevMode->dmOrientation = (nOrientation == 0) ? DMORIENT_PORTRAIT :
        DMORIENT_LANDSCAPE;

    if ((*lpfnEDM) (hwnd, hLib, npDevMode, szPrinterName, szPort, npDevMode,
        NULL, DM_IN_BUFFER | DM_OUT_BUFFER) < 0) {
        LocalFree ((HLOCAL) npDevMode);
        FreeLibrary (hLib);
        return NULL;
    }

    FreeLibrary (hLib);
    return npDevMode;
}

/*
 *  BuildFontList builds a font list for the default printer.
 *
 *  Input parameters:
 *
 *    hwnd = Window handle
 *    hInstance = Instance handle
 *
 *  Returns:
 *
 *    0 = Font list built
 *    IDERR_CREATEPRINTERDCFAILED = Could not create a printer DC 
 *    IDERR_NOTRUETYPEFONTS = No fixed-pitch TrueType fonts installed
 */

int BuildFontList (HWND hwnd, HINSTANCE hInstance)
{
    FARPROC lpfnEnumFontsProc;
    HDC hdcPrn;

    if ((hdcPrn = CreatePrinterDC (NULL)) == NULL) {
        MessageBox (hwnd, szError[IDERR_CREATEPRINTERDCFAILED],
            szError[IDERR_TITLE], MB_ICONSTOP | MB_OK);
        return IDERR_CREATEPRINTERDCFAILED;
    }

    nNumFonts = 0;
    lpfnEnumFontsProc = MakeProcInstance (EnumFontsProc, hInstance);
    EnumFontFamilies (hdcPrn, NULL, lpfnEnumFontsProc, 0L);
    FreeProcInstance (lpfnEnumFontsProc);
    DeleteDC (hdcPrn);

    if (nNumFonts == 0) {
        MessageBox (hwnd, szError[IDERR_NOTRUETYPEFONTS],
            szError[IDERR_TITLE], MB_ICONSTOP | MB_OK);
        return IDERR_NOTRUETYPEFONTS;
    }
    return 0;
}

/*
 *  EnumFontsProc is the callback function for EnumFontFamilies.
 *
 *  Input parameters: 
 *
 *    lplf = Pointer to LOGFONT structure
 *    lpNTM = Pointer to NEWTEXTMETRIC structure
 *    nFontType = Font type (device, raster, TrueType)  
 *    lParam = Not used
 *
 *  Returns:
 *
 *    Nonzero if the aszFaces structure isn't full, 0 if it is (0 stops the
 *    font enumeration process).
 */

BOOL FAR PASCAL EnumFontsProc (LPLOGFONT lplf, LPNEWTEXTMETRIC lpntm,
                               int nFontType, LPARAM lParam)
{
    /* Despite what the SDK says, the TMPF_FIXED_PITCH bit is 0 instead of 1
       for fixed-pitch TrueType fonts, hence the ! prefixing the expression
       (lpntm->tmPitchAndFamily & TMPF_FIXED_PITCH) below. */

    if ((lpntm->tmPitchAndFamily & TMPF_TRUETYPE) &&
        (!(lpntm->tmPitchAndFamily & TMPF_FIXED_PITCH)))
        lstrcpy (aszFaces[nNumFonts++], lplf->lfFaceName);

    return (nNumFonts == MAXFONTS) ? 0 : 1;
}

/*
 *  MakeDateString creates a string representing today's date. The [intl]
 *  section of WIN.INI is consulted for the date format.
 *
 *  Input parameters:
 *
 *    szDate = Pointer to buffer to receive the text string
 *
 *  Returns:
 *
 *    Nothing
 */

 void MakeDateString (char *szDate)
 {
    int iDate;
    char ch[2];
            
    _strdate (szDate);
    iDate = GetProfileInt ("intl", "iDate", 0);
    if (iDate == 1) {
        ch[0] = szDate[0];
        ch[1] = szDate[1];
        szDate[0] = szDate[3];
        szDate[1] = szDate[4];
        szDate[3] = ch[0];
        szDate[4] = ch[1];
    }
    else if (iDate == 2) {
        ch[0] = szDate[6];
        ch[1] = szDate[7];
        szDate[7] = szDate[4];
        szDate[6] = szDate[3];
        szDate[4] = szDate[1];
        szDate[3] = szDate[0];
        szDate[1] = ch[1];
        szDate[0] = ch[0];
    }
        
    GetProfileString ("intl", "sDate", "/", ch, sizeof (ch));
    szDate[2] = ch[0];
    szDate[5] = ch[0];  
 }

/*
 *  DrawPageImage draws a page image in the left or right panel of the page
 *  layout window. 
 *
 *  Input parameters:
 *
 *    hdc = Device context handle
 *    nPanelNumber = Panel number (0=left, 1=right)
 *    nLayoutNumber = Layout number (0, 1, or 2)
 *
 *  Returns:
 *
 *    Nothing
 */

void DrawPageImage (HDC hdc, int nPanelNumber, int nLayoutNumber)
{
    HBRUSH hBrush;
    RECT rect, rcPage;
    int nMarginWidth, nPageWidth, nPageHeight;

    if (nLayoutNumber == nLayout) {
        hBrush = CreateSolidBrush (RGB (255, 255, 0));
        FillRect (hdc, &rcPanel[nPanelNumber], hBrush);
        DeleteObject (hBrush);
    }

    hBrush = CreateSolidBrush (RGB (0, 255, 255));
    SelectObject (hdc, GetStockObject (BLACK_PEN));
    nMarginWidth = (rcPortrait.right - rcPortrait.left) / 11;

    switch (nLayoutNumber) {

    case 0:
        //
        // Draw layout 0 (one page per sheet, portrait).
        //
        rect.left = rcPanel[nPanelNumber].left + rcPortrait.left;
        rect.top = rcPanel[nPanelNumber].top + rcPortrait.top;
        rect.right = rcPanel[nPanelNumber].left + rcPortrait.right;
        rect.bottom = rcPanel[nPanelNumber].top + rcPortrait.bottom;
        Rectangle (hdc, rect.left, rect.top, rect.right, rect.bottom);

        InflateRect (&rect, -nMarginWidth, -nMarginWidth);
        FillRect (hdc, &rect, hBrush);
        DrawFocusRect (hdc, &rect);
        break;

    case 1:
        //
        // Draw layout 1 (two pages per sheet, landscape).
        //
        rect.left = rcPanel[nPanelNumber].left + rcLandscape.left;
        rect.top = rcPanel[nPanelNumber].top + rcLandscape.top;
        rect.right = rcPanel[nPanelNumber].left + rcLandscape.right;
        rect.bottom = rcPanel[nPanelNumber].top + rcLandscape.bottom;
        Rectangle (hdc, rect.left, rect.top, rect.right, rect.bottom);

        nPageWidth = (rect.right - rect.left - (3 * nMarginWidth)) >> 1;

        CopyRect (&rcPage, &rect);
        rcPage.left = rect.left + nMarginWidth;
        rcPage.right = rcPage.left + nPageWidth;
        rcPage.top = rect.top + nMarginWidth;
        rcPage.bottom = rect.bottom - nMarginWidth;
        FillRect (hdc, &rcPage, hBrush);
        DrawFocusRect (hdc, &rcPage);

        rcPage.right = rect.right - nMarginWidth;
        rcPage.left = rcPage.right - nPageWidth;
        FillRect (hdc, &rcPage, hBrush);
        DrawFocusRect (hdc, &rcPage);
        break;

    case 2:
        //
        // Draw layout 2 (four pages per sheet, portrait).
        //
        rect.left = rcPanel[nPanelNumber].left + rcPortrait.left;
        rect.top = rcPanel[nPanelNumber].top + rcPortrait.top;
        rect.right = rcPanel[nPanelNumber].left + rcPortrait.right;
        rect.bottom = rcPanel[nPanelNumber].top + rcPortrait.bottom;
        Rectangle (hdc, rect.left, rect.top, rect.right, rect.bottom);

        nPageWidth = (rect.right - rect.left - (3 * nMarginWidth)) >> 1;
        nPageHeight = (rect.bottom - rect.top - (3 * nMarginWidth)) >> 1;

        CopyRect (&rcPage, &rect);
        rcPage.left = rect.left + nMarginWidth;
        rcPage.right = rcPage.left + nPageWidth;
        rcPage.top = rect.top + nMarginWidth;
        rcPage.bottom = rcPage.top + nPageHeight;
        FillRect (hdc, &rcPage, hBrush);
        DrawFocusRect (hdc, &rcPage);

        rcPage.bottom = rect.bottom - nMarginWidth;
        rcPage.top = rcPage.bottom - nPageHeight;
        FillRect (hdc, &rcPage, hBrush);
        DrawFocusRect (hdc, &rcPage);

        CopyRect (&rcPage, &rect);
        rcPage.right = rect.right - nMarginWidth;
        rcPage.left = rcPage.right - nPageWidth;
        rcPage.top = rect.top + nMarginWidth;
        rcPage.bottom = rcPage.top + nPageHeight;
        FillRect (hdc, &rcPage, hBrush);
        DrawFocusRect (hdc, &rcPage);

        rcPage.bottom = rect.bottom - nMarginWidth;
        rcPage.top = rcPage.bottom - nPageHeight;
        FillRect (hdc, &rcPage, hBrush);
        DrawFocusRect (hdc, &rcPage);
        break;
    }
    DeleteObject (hBrush);
}

/*
 *  PaintPanel paints the background of one of the two panels whose
 *  coordinates are stored in the rcPanel array using the specified brush.
 *
 *  Input parameters:
 *
 *    hdc = Device context handle
 *    nPanelNumber = Panel number (0=left, 1=right)
 *    nLayoutNumber = Layout number (0, 1, or 2)
 *    hBrush = Brush handle
 *
 *  Returns:
 *
 *    Nothing
 */

void PaintPanel (HDC hdc, int nPanelNumber, int nLayoutNumber, HBRUSH hBrush)
{
    HRGN hRgnDest, hRgnSrc1, hRgnSrc2;
    RECT rect;

    switch (nLayoutNumber) {

    case 0:
    case 2:
        rect.left = rcPanel[nPanelNumber].left + rcPortrait.left;
        rect.top = rcPanel[nPanelNumber].top + rcPortrait.top;
        rect.right = rcPanel[nPanelNumber].left + rcPortrait.right;
        rect.bottom = rcPanel[nPanelNumber].top + rcPortrait.bottom;
        break;

    case 1:
        rect.left = rcPanel[nPanelNumber].left + rcLandscape.left;
        rect.top = rcPanel[nPanelNumber].top + rcLandscape.top;
        rect.right = rcPanel[nPanelNumber].left + rcLandscape.right;
        rect.bottom = rcPanel[nPanelNumber].top + rcLandscape.bottom;
        break;
    }

    hRgnDest = CreateRectRgn (0, 0, 0, 0);
    hRgnSrc1 = CreateRectRgnIndirect (&rcPanel[nPanelNumber]);
    hRgnSrc2 = CreateRectRgnIndirect (&rect);
    CombineRgn (hRgnDest, hRgnSrc1, hRgnSrc2, RGN_DIFF);

    FillRgn (hdc, hRgnDest, hBrush);

    DeleteObject (hRgnSrc2);
    DeleteObject (hRgnSrc1);
    DeleteObject (hRgnDest);
}

/*
 *  OBDDrawButton draws the Print button.
 *
 *  Input parameters:
 *
 *    lpdis = Pointer to DRAWITEMSTRUCT structure
 *    hDisabledButton = Bitmap handle (disabled button)
 *    hEnabledButton = Bitmap handle (enabled button)
 *
 *  Returns:
 *
 *    Nothing
 */

void ODBDrawButton (DRAWITEMSTRUCT FAR *lpdis, HBITMAP hDisabledButton,
                    HBITMAP hEnabledButton)
{
    HDC hdc, hdcMem;
    HBITMAP hBitmap, hOldBitmap;
    BITMAP bm;
    RECT rect;
    int x, y;

    hdc = lpdis->hDC;
    CopyRect (&rect, &(lpdis->rcItem));

    //
    // Fill the button interior with light gray and border the button with
    // a thin black line.
    //
    FillRect (hdc, &rect, GetStockObject (LTGRAY_BRUSH));
    SelectObject (hdc, GetStockObject (BLACK_PEN));
    MoveTo (hdc, rect.left + 1, rect.top);
    LineTo (hdc, rect.right - 1, rect.top);
    MoveTo (hdc, rect.right - 1, rect.top + 1);
    LineTo (hdc, rect.right - 1, rect.bottom - 1);
    MoveTo (hdc, rect.right - 2, rect.bottom - 1);
    LineTo (hdc, rect.left, rect.bottom - 1);
    MoveTo (hdc, rect.left, rect.bottom - 2);
    LineTo (hdc, rect.left, rect.top);

    //
    // Draw the raised button border. If the button has the input focus,
    // double the thickness of the black border and draw a focus rectangle.
    //
    InflateRect (&rect, -1, -1);

    if (lpdis->itemState & ODS_FOCUS) {
        FrameRect (hdc, &rect, GetStockObject (BLACK_BRUSH));
        InflateRect (&rect, -7, -7);
        DrawFocusRect (hdc, &rect);
        InflateRect (&rect, 6, 6);
    }

    DrawButtonBorder (hdc, &rect);

    //
    // Draw the bitmap on the face of the button.
    //
    CopyRect (&rect, &(lpdis->rcItem));
    hdcMem = CreateCompatibleDC (hdc);
    hBitmap = (lpdis->itemState & ODS_DISABLED) ? hDisabledButton :
        hEnabledButton;
    hOldBitmap = SelectObject (hdcMem, hBitmap);
    GetObject (hBitmap, sizeof (bm), &bm);

    x = ((rect.right - rect.left) - bm.bmWidth) >>1 ;
    y = ((rect.bottom - rect.top) - bm.bmHeight) >> 1;
    BitBlt (hdc, x, y, bm.bmWidth, bm.bmHeight, hdcMem, 0, 0, SRCCOPY);

    SelectObject (hdcMem, hOldBitmap);
    DeleteDC (hdcMem);
}

/*
 *  OBDChangeSelection redraws the Print button to reflect the fact that
 *  it was just selected or deselected.
 *
 *  Input parameters:
 *
 *    lpdis = Pointer to DRAWITEMSTRUCT structure
 *
 *  Returns:
 *
 *    Nothing
 */

void ODBChangeSelection (DRAWITEMSTRUCT FAR *lpdis)
{
    HDC hdc;
    RECT rect;
    HPEN hPen, hOldPen;

    hdc = lpdis->hDC;
    CopyRect (&rect, &(lpdis->rcItem));

    if (lpdis->itemState & ODS_SELECTED) {
        //
        // Draw the button in its selected state.
        //
        InflateRect (&rect, -2, -2);
        FrameRect (hdc, &rect, GetStockObject (LTGRAY_BRUSH));
        InflateRect (&rect, -1, -1);
        FrameRect (hdc, &rect, GetStockObject (LTGRAY_BRUSH));

        InflateRect (&rect, 1, 1);
        hPen = CreatePen (PS_SOLID, 1, RGB (128, 128, 128));
        hOldPen = SelectObject (hdc, hPen);
        MoveTo (hdc, rect.left, rect.bottom);
        LineTo (hdc, rect.left, rect.top);
        LineTo (hdc, rect.right, rect.top);
        DeleteObject (SelectObject (hdc, hOldPen));

        InflateRect (&rect, -2, -2);
        BitBlt (hdc, rect.left + 1, rect.top + 1, rect.right -
            rect.left + 1, rect.bottom - rect.top + 1, hdc,
            rect.left, rect.top, SRCCOPY);
    }
    else {
        //
        // Draw the button in its unselected state.
        //
        InflateRect (&rect, -2, -2);
        DrawButtonBorder (hdc, &rect);

        InflateRect (&rect, -4, -4);
        BitBlt (hdc, rect.left, rect.top, rect.right -
            rect.left + 1, rect.bottom - rect.top + 1, hdc,
            rect.left + 1, rect.top + 1, SRCCOPY);
    }
}

/*
 *  OBDChangeFocus redraws the Print button to reflect the fact that it
 *  just lost or gained the input focus.
 *
 *  Input parameters:
 *
 *    lpdis = Pointer to DRAWITEMSTRUCT structure
 *
 *  Returns:
 *
 *    Nothing
 */

void ODBChangeFocus (DRAWITEMSTRUCT FAR *lpdis)
{
    HDC hdc;
    RECT rect;

    hdc = lpdis->hDC;
    CopyRect (&rect, &(lpdis->rcItem));
    InflateRect (&rect, -1, -1);

    if (lpdis->itemState & ODS_FOCUS) {
        FrameRect (hdc, &rect, GetStockObject (BLACK_BRUSH));
        InflateRect (&rect, -1, -1);
    }

    DrawButtonBorder (hdc, &rect);
    CopyRect (&rect, &(lpdis->rcItem));
    InflateRect (&rect, -8, -8);
    DrawFocusRect (hdc, &rect);
}

/*
 *  DrawButtonBorder draws the raised border of an owner-draw button.
 *
 *  Input parameters:
 *
 *    hdc = Device context handle
 *    npRect = Pointer to RECT structure
 *
 *  Returns:
 *
 *    Nothing
 */

void DrawButtonBorder (HDC hdc, RECT *npRect)
{
    SelectObject (hdc, CreatePen (PS_SOLID, 1, RGB (128, 128, 128)));
    MoveTo (hdc, npRect->left, npRect->bottom - 1);
    LineTo (hdc, npRect->right - 1, npRect->bottom - 1);
    LineTo (hdc, npRect->right - 1, npRect->top - 1);
    MoveTo (hdc, npRect->right - 2, npRect->top + 1);
    LineTo (hdc, npRect->right - 2, npRect->bottom - 2);
    LineTo (hdc, npRect->left, npRect->bottom - 2);

    DeleteObject (SelectObject (hdc, GetStockObject (WHITE_PEN)));
    MoveTo (hdc, npRect->left, npRect->bottom - 2);
    LineTo (hdc, npRect->left, npRect->top);
    LineTo (hdc, npRect->right - 1, npRect->top);
    MoveTo (hdc, npRect->right - 3, npRect->top + 1);
    LineTo (hdc, npRect->left + 1, npRect->top + 1);
    LineTo (hdc, npRect->left + 1, npRect->bottom - 2);
}

/*
 *  InitProgram initializes the program.
 *
 *  Input parameters:
 *
 *    hwnd = Window handle
 *
 *  Returns:
 *
 *    Nothing
 */

void InitProgram (HWND hwnd)
{
    RECT rect;
    HWND hwndScrollBar;

    //
    // Get the coordinates of the group rectangle and the rectangles
    // describing the page images.
    //
    InitGroupRect (hwnd, IDD_GROUP, &rcGroup, TRUE);
    InitGroupRect (hwnd, IDD_PORTRAIT, &rcPortrait, TRUE);
    InitGroupRect (hwnd, IDD_LANDSCAPE, &rcLandscape, TRUE);

    //
    // Reposition and resize the scroll bar so that it appears at the
    // bottom of the group rectangle. Also initialize the scroll bar range
    // and thumb position.
    //
    hwndScrollBar = GetDlgItem (hwnd, IDD_SCROLLBAR);

    GetWindowRect (hwndScrollBar, &rect);
    ScreenToClient (hwnd, (POINT FAR *) &rect.left);
    ScreenToClient (hwnd, (POINT FAR *) &rect.right);

    SetWindowPos (hwndScrollBar, NULL, rcGroup.left + 1,
        rcGroup.bottom - (rect.bottom - rect.top) - 1, rcGroup.right -
        rcGroup.left - 2, rect.bottom - rect.top, SWP_NOZORDER);

    SetScrollRange (hwndScrollBar, SB_CTL, 0, 100, FALSE);
    SetScrollPos (hwndScrollBar, SB_CTL, nPanelIndex * 100, FALSE);

    //
    // Compute the coordinates of the left and right panels.
    //
    rcPanel[0].left = rcGroup.left + 1;
    rcPanel[0].top = rcPanel[1].top = rcGroup.top + 1;
    rcPanel[0].right = rcGroup.left + ((rcGroup.right - rcGroup.left) >> 1);
    rcPanel[0].bottom = rcPanel[1].bottom = rcGroup.bottom -
        (rect.bottom - rect.top) - 1;

    rcPanel[1].left = rcPanel[0].right;
    rcPanel[1].right = rcGroup.right - 1;

    //
    // Modify the page image coordinates so that they're relative to the
    // upper-left corner of the left and right panels.
    //
    rcPortrait.left -= rcPanel[0].left;
    rcPortrait.top -= rcPanel[0].top;
    rcPortrait.right -= rcPanel[0].left;
    rcPortrait.bottom -= rcPanel[0].top;

    rcLandscape.left -= rcPanel[1].left;
    rcLandscape.top -= rcPanel[1].top;
    rcLandscape.right -= rcPanel[1].left;
    rcLandscape.bottom -= rcPanel[1].top;
}

/*
 *  InitGroupRect saves the coordinates of the control window in a RECT
 *  structure, and optionally destroys the control.
 *
 *  Input parameters:
 *
 *    hwnd = Handle of the control's parent
 *    nID = Control ID
 *    rect = Pointer to RECT structure to receive the coordinates
 *    bDestroy = Flag specifying whether the control should be destroyed
 *
 *  Returns:
 *
 *    Nothing
 */

void InitGroupRect (HWND hwnd, int nID, RECT *rect, BOOL bDestroy)
{
    GetWindowRect (GetDlgItem (hwnd, nID), rect);
    ScreenToClient (hwnd, (POINT FAR *) &rect->left);
    ScreenToClient (hwnd, (POINT FAR *) &rect->right);
    if (bDestroy)
        DestroyWindow (GetDlgItem (hwnd, nID));
}

/*
 *  DrawGroupRect draws a recessed group rectangle at the specified location.
 *
 *  Input parameters:
 *
 *    hdc = Device context handle
 *    rect = Pointer to RECT structure containing the rectangle's coordinates
 *
 *  Returns:
 *
 *    Nothing
 */

void DrawGroupRect (HDC hdc, RECT *rect)
{
    HPEN hPen;

    FillRect (hdc, rect, GetStockObject (LTGRAY_BRUSH)); 
    hPen = CreatePen (PS_SOLID, 1, RGB (128, 128, 128));
    SelectObject (hdc, hPen);
    MoveTo (hdc, rect->left, rect->bottom - 1);
    LineTo (hdc, rect->left, rect->top);
    LineTo (hdc, rect->right, rect->top);
    DeleteObject (SelectObject (hdc, GetStockObject (WHITE_PEN)));
    MoveTo (hdc, rect->right - 1, rect->top + 1);
    LineTo (hdc, rect->right - 1, rect->bottom - 1);
    LineTo (hdc, rect->left, rect->bottom - 1);
}

/*
 *  PositionWindow positions the window on the screen using the
 *  WindowPos entry stored in WINPRN.INI. If WINPRN.INI does not exist,
 *  if it does exist but does not contain a WindowPos entry, or if WindowPos
 *  specifies a position that is off (or nearly off) the screen,
 *  PositionWindows centers the window on the screen.
 *
 *  Input parameters:
 *
 *    hwnd = Window handle
 *
 *  Returns:
 *
 *    Nothing
 */

void PositionWindow (HWND hwnd)
{
    char szBuffer[32];
    static char szDefString[] = "WindowPos=32767,32767";
    int cxWindowPos, cyWindowPos, cxScreenSize, cyScreenSize;
    RECT rect;

    GetPrivateProfileString (szSection, szEntry[0], szDefString, szBuffer,
        sizeof (szBuffer), szIniFile);
    sscanf (szBuffer, "%u,%u", &cxWindowPos, &cyWindowPos);

    cxScreenSize = GetSystemMetrics (SM_CXSCREEN);
    cyScreenSize = GetSystemMetrics (SM_CYSCREEN);

    if ((cxWindowPos > (cxScreenSize - 32)) ||
        (cyWindowPos > (cyScreenSize - 32))) {
        GetWindowRect (hwnd, &rect);
        cxWindowPos = (cxScreenSize - (rect.right - rect.left)) >> 1;
        cyWindowPos = (cyScreenSize - (rect.bottom - rect.top)) >> 1;
    }

    SetWindowPos (hwnd, 0, cxWindowPos, cyWindowPos, 0, 0,
        SWP_NOSIZE | SWP_NOZORDER);
}

/*
 *  ReadIniFile reads program settings from WINPRN.INI.
 *
 *  Input parameters:
 *
 *    None
 *
 *  Returns:
 *
 *    Nothing
 */

void ReadIniFile (void)
{
    char szTypeface[LF_FACESIZE];
    int i;

    nLayout = min (2, GetPrivateProfileInt (szSection, szEntry[1], 0,
        szIniFile));

    nPanelIndex = min (1, GetPrivateProfileInt (szSection, szEntry[2], 0,
        szIniFile));

    GetPrivateProfileString (szSection, szEntry[3], "", szTypeface,
        sizeof (szTypeface), szIniFile);

    nCurrentFont = 0;
    for (i=0; i<nNumFonts; i++) {
        if (lstrcmp (szTypeface, aszFaces[i]) == 0) {
            nCurrentFont = i;
            break;
        }
    }
    
    nTabLength = min (32, GetPrivateProfileInt (szSection, szEntry[4], 8,
        szIniFile));

    bPrintOutline = GetPrivateProfileInt (szSection, szEntry[5], 1,
        szIniFile) ? TRUE : FALSE;

    bPrintHeader = GetPrivateProfileInt (szSection, szEntry[6], 1,
        szIniFile) ? TRUE : FALSE;

    nPaperSize = min (1, GetPrivateProfileInt (szSection, szEntry[7], 0,
        szIniFile));
}

/*
 *  SaveSettings saves program settings to WINPRN.INI.
 *
 *  Input parameters:
 *
 *    hwnd = Window handle
 *
 *  Returns:
 *
 *    Nothing
 */

void SaveSettings (HWND hwnd)
{
    WINDOWPLACEMENT wp;
    char szBuffer[32];

    wp.length = sizeof (wp);
    GetWindowPlacement (hwnd, &wp);
    wsprintf (szBuffer, "%u,%u", (wp.rcNormalPosition).left,
        (wp.rcNormalPosition).top);
    WritePrivateProfileString (szSection, szEntry[0], szBuffer, szIniFile);

    wsprintf (szBuffer, "%d", nLayout);
    WritePrivateProfileString (szSection, szEntry[1], szBuffer, szIniFile);

    wsprintf (szBuffer, "%d", nPanelIndex);
    WritePrivateProfileString (szSection, szEntry[2], szBuffer, szIniFile);

    WritePrivateProfileString (szSection, szEntry[3], aszFaces[nCurrentFont],
        szIniFile);

    wsprintf (szBuffer, "%d", nTabLength);
    WritePrivateProfileString (szSection, szEntry[4], szBuffer, szIniFile);

    WritePrivateProfileString (szSection, szEntry[5],
        bPrintOutline ? "1" : "0", szIniFile);

    WritePrivateProfileString (szSection, szEntry[6],
        bPrintHeader ? "1" : "0", szIniFile);

    WritePrivateProfileString (szSection, szEntry[7],
        nPaperSize ? "1" : "0", szIniFile);
}

/*
 *  CenterDialog centers a dialog box window (or any window, for that matter)
 *  on the screen.
 *
 *  Input parameters:
 *
 *    hwnd = Window handle
 *
 *  Returns:
 *
 *    Nothing
 */

void CenterDialog (HWND hwnd)
{
    RECT rect;

    GetWindowRect (hwnd, &rect);
    SetWindowPos (hwnd, 0,
        (GetSystemMetrics (SM_CXSCREEN) - (rect.right - rect.left)) >> 1,
        (GetSystemMetrics (SM_CYSCREEN) - (rect.bottom - rect.top)) >> 1,
        0, 0, SWP_NOSIZE | SWP_NOZORDER);
}

/*
 *  OptionsDlgProc processes messages to the Options dialog box.
 */

BOOL FAR PASCAL OptionsDlgProc (HWND hwnd, WORD message, WORD wParam,
                                LONG lParam)
{
    char szBuffer[LF_FACESIZE];
    int i;

    switch (message) {

    case WM_INITDIALOG:
        //
        // Initialize the dialog box controls.
        //
        for (i=0; i<nNumFonts; i++)
            SendDlgItemMessage (hwnd, IDD_TYPEFACE, CB_ADDSTRING, 0,
                (LPARAM) (LPCSTR) aszFaces[i]);

        SendDlgItemMessage (hwnd, IDD_TYPEFACE, CB_SETCURSEL, nCurrentFont,
            0L);

        SendDlgItemMessage (hwnd, IDD_TABLENGTH, CB_LIMITTEXT, 2, 0L);
        SendDlgItemMessage (hwnd, IDD_TABLENGTH, CB_ADDSTRING, 0,
            (LPARAM) (LPCSTR) "4");
        SendDlgItemMessage (hwnd, IDD_TABLENGTH, CB_ADDSTRING, 0,
            (LPARAM) (LPCSTR) "8");

        if (nTabLength == 4)
            SendDlgItemMessage (hwnd, IDD_TABLENGTH, CB_SETCURSEL, 0, 0L);
        else if (nTabLength == 8)
            SendDlgItemMessage (hwnd, IDD_TABLENGTH, CB_SETCURSEL, 1, 0L);
        else {
            wsprintf (szBuffer, "%d", nTabLength);
            SetDlgItemText (hwnd, IDD_TABLENGTH, szBuffer);
        }

        CheckDlgButton (hwnd, IDD_PRINTOUTLINE, bPrintOutline);
        CheckDlgButton (hwnd, IDD_PRINTHEADER, bPrintHeader);
        CheckRadioButton (hwnd, IDD_LETTER, IDD_A4, IDD_LETTER + nPaperSize);
        return TRUE;

    case WM_COMMAND:
        //
        // Process messages from the controls.
        //
        switch (wParam) {

        case IDD_LETTER:
        case IDD_A4:
            CheckRadioButton (hwnd, IDD_LETTER, IDD_A4, wParam);
            return TRUE;

        case IDOK:
            nCurrentFont = SendDlgItemMessage (hwnd, IDD_TYPEFACE,
                CB_GETCURSEL, 0, 0L);

            GetDlgItemText (hwnd, IDD_TABLENGTH, szBuffer, sizeof (szBuffer));
            nTabLength = atoi (szBuffer);
            if ((nTabLength < 1) || (nTabLength > 32))
                nTabLength = 8;

            bPrintOutline = IsDlgButtonChecked (hwnd, IDD_PRINTOUTLINE) ?
                TRUE : FALSE;
            bPrintHeader = IsDlgButtonChecked (hwnd, IDD_PRINTHEADER) ?
                TRUE : FALSE; 
            nPaperSize = IsDlgButtonChecked (hwnd, IDD_LETTER) ? 0 : 1;
            EndDialog (hwnd, 1);
            return TRUE;

        case IDCANCEL:
            EndDialog (hwnd, 0);
            return TRUE;
        }
        break;

    }
    return FALSE;
}

/*
 *  PrintDlgProc processes messages to the Printing dialog box.
 */

BOOL FAR PASCAL PrintDlgProc (HWND hwnd, WORD message, WORD wParam,
                              LONG lParam)
{
    RECT rect;

    switch (message) {

    case WM_INITDIALOG:
        //
        // Initialize.
        //
        if ((xDialog != 0) || (yDialog != 0))
            SetWindowPos (hwnd, NULL, xDialog, yDialog, 0, 0, SWP_NOSIZE |
                SWP_NOZORDER); 
        else if (IsIconic (GetWindow (hwnd, GW_OWNER)))
            CenterDialog (hwnd);

        SetDlgItemText (hwnd, IDD_FILE, szFileName);
        return TRUE;

    case WM_COMMAND:
        //
        // Destroy the dialog box when wParam == IDCANCEL.
        //
        if (wParam == IDCANCEL) {
            bAbort = TRUE;
            EnableWindow (GetWindow (hwnd, GW_OWNER), TRUE);
            DestroyWindow (hwnd); 
            hwndPrintDlg = 0;
            return TRUE;
        }
        break;

    case WM_DESTROY:
        //
        // Save the dialog box coordinates before ending.
        //
        GetWindowRect (hwnd, &rect);
        xDialog = rect.left;
        yDialog = rect.top;
        return TRUE;
    }
    return FALSE;
}

/*
 *  AboutDlgProc processes messages to the About WinPrn dialog box.
 */

BOOL FAR PASCAL AboutDlgProc (HWND hwnd, WORD message, WORD wParam,
                              LONG lParam)
{
    static HPEN hPen;
    static RECT rectFrame;
    static HFONT hDlgFont;
    PAINTSTRUCT ps;
    LOGFONT lf;
    RECT rect;
    HDC hdc;

    switch (message) {

    case WM_INITDIALOG:
        //
        // Initialize and create GDI objects.
        //
        InitGroupRect (hwnd, IDD_FRAME, &rectFrame, TRUE);

        if (IsIconic (GetWindow (hwnd, GW_OWNER)))
            CenterDialog (hwnd);

        GetObject ((HFONT) SendMessage (hwnd, WM_GETFONT, 0, 0L),
            sizeof (LOGFONT), &lf);
        lstrcpy (lf.lfFaceName, "Arial");
        lf.lfHeight *= 3;
        lf.lfWidth *= 3;
        lf.lfItalic = 1;
        lf.lfOutPrecision = OUT_TT_PRECIS;
        if ((hDlgFont = CreateFontIndirect ((LPLOGFONT) &lf)) != NULL)
            SendDlgItemMessage (hwnd, IDD_TITLE, WM_SETFONT, hDlgFont, 0L);

        hPen = CreatePen (PS_SOLID, 1, RGB (128, 128, 128));
        return TRUE;

    case WM_CTLCOLOR:
        //
        // Paint all control backgrounds light gray. Also change the title
        // color to bright red.
        //
        SetBkColor ((HDC) wParam, RGB (192, 192, 192));
        if (GetDlgCtrlID (LOWORD (lParam)) == IDD_TITLE)
            SetTextColor ((HDC) wParam, RGB (255, 0, 0));
        return GetStockObject (LTGRAY_BRUSH);

    case WM_ERASEBKGND:
        //
        // Paint the window background light gray.
        //
        GetClientRect (hwnd, &rect);
        FillRect ((HDC) wParam, &rect, GetStockObject (LTGRAY_BRUSH));
        return TRUE;

    case WM_PAINT:
        //
        // Draw the rectangle surrounding the dialog box text.
        //
        hdc = BeginPaint (hwnd, &ps);

        SelectObject (hdc, hPen);
        SelectObject (hdc, GetStockObject (NULL_BRUSH));
        Rectangle (hdc, rectFrame.left, rectFrame.top,
            rectFrame.right, rectFrame.bottom);

        SelectObject (hdc, GetStockObject (WHITE_PEN));
        Rectangle (hdc, rectFrame.left + 1, rectFrame.top + 1,
            rectFrame.right + 1, rectFrame.bottom + 1);

        EndPaint (hwnd, &ps);
        return TRUE;

    case WM_COMMAND:
        //
        // Dismiss the dialog box in response to an IDOK or IDCANCEL message.
        //
        switch (wParam) {

        case IDOK:
        case IDCANCEL:
            EndDialog (hwnd, 0);
            return TRUE;
        }
        break;

    case WM_DESTROY:
        //
        // Delete GDI objects.
        //
        if (hDlgFont != NULL)
            DeleteObject (hDlgFont);
        DeleteObject (hPen);
        return TRUE;
    }
    return FALSE;
}
