//===========================================================
// Switcher - Switch between different windows configurations
// Copyright (C) 1994 Douglas Boling
//
// Revision History:
//
// 0.0   Initial Release
//
// Todo: 
//
//===========================================================
// Returns no. of elements
#define dim(x) (sizeof(x) / sizeof(x[0]))   

//-----------------------------------------------------------
// Include files
//-----------------------------------------------------------
#include "windows.h"
#include "stdlib.h"
#include "string.h"

#include "switcher.h"
#include "statbar.h"
//-----------------------------------------------------------
// Global data
//-----------------------------------------------------------
// Message dispatch table for MainWindowProc
struct decodeUINT MainMessages[] = {
	WM_CREATE, DoCreateMain,
	WM_SIZE, DoSizeMain,
	WM_INITMENU, DoInitMenuMain,
	WM_PAINT, DoPaintMain,
	WM_COMMAND, DoCommandMain,
	WM_SYSCOMMAND, DoCommandMain,
	WM_CLOSE, DoCloseMain,
	WM_DESTROY, DoDestroyMain,
};
// Command Message dispatch for MainWindowProc
struct decodeCMD MainMenuItems[] = {
	IDM_EXIT, DoMainMenuExit,
	IDM_CONFIG, DoMainMenuConfig,
	IDM_BTNADD, DoMainMenuBtnAdd,
	IDM_BTNEDIT, DoMainMenuBtnEdit,
	IDM_BTNDEL, DoMainMenuBtnDel,
	IDM_ABOUT, DoMainMenuAbout,
	IDD_BUTTONS, DoMainBtnClick,
	IDD_BUTTONS+1, DoMainBtnClick,
	IDD_BUTTONS+2, DoMainBtnClick,
	IDD_BUTTONS+3, DoMainBtnClick,
	IDD_BUTTONS+4, DoMainBtnClick,
	IDD_BUTTONS+5, DoMainBtnClick,
	IDD_BUTTONS+6, DoMainBtnClick,
	IDD_BUTTONS+7, DoMainBtnClick,
	IDD_BUTTONS+8, DoMainBtnClick,
	IDD_BUTTONS+9, DoMainBtnClick,
	IDD_BUTTONS+10, DoMainBtnClick,
	IDD_BUTTONS+11, DoMainBtnClick,
	IDD_BUTTONS+12, DoMainBtnClick,
	IDD_BUTTONS+13, DoMainBtnClick,
	IDD_BUTTONS+14, DoMainBtnClick,
	IDD_BUTTONS+15, DoMainBtnClick,
};
HANDLE	hInst;
HWND		hMain;
UINT		wVersion = 10;
BOOL		fFirst = TRUE;

char	szAppName[] = "WinSwitcher";         // Application name
char	szTitleText[] = "Switcher";          // Application window title
char	szMenuName[] = "WinSwitcherMenu";    // Menu name
char	szIconName[] = "WinSwitcherIcon";    // Icon name
char	szDataFileName[] = "\\switcher.dat"; // Data file name (No INI file)
char	szVTag[8] = "SWDAT09";               // Data file verify tag
// INI section and key names 
char	*pszSProSec = "Sounds";

char	*pszMProGenSec = "mouse";
char	*pszMProHSen = "HorizontalSensitivity";
char	*pszMProVSen = "VerticalSensitivity";
char	*pszMProRot = "RotationAngle";
char	*pszMProPriB = "PrimaryButton";
char	*pszMProSecB = "SecondaryButton";
char	*pszMProAAP = "ActiveAccelerationProfile";
char	*pszMProAPSec = "AccelerationProfile";
char	*pszMProAPL = "Label";
char	*pszMProAPM = "Movement";
char	*pszMProAPF = "Factor";

char	szProgDir[128];                      // Directory of EXE file
HGLOBAL hSWData;
LPSWDATA lpSWData;
LPBTNDATA lpBtnData;
char *pszColorNames[21] = {"Background","AppWorkspace","Window",
	                        "WindowText","Menu","MenuText",
	                        "ActiveTitle","InactiveTitle",
	                        "TitleText","ActiveBorder",
	                        "InactiveBorder","WindowFrame",
	                        "Scrollbar","ButtonFace", "ButtonShadow",
	                        "ButtonText","GrayText","Hilight",
   	                     "HilightText","InactiveTitleText",
	                        "ButtonHilight"};

INT sColorIDs[21] = {COLOR_BACKGROUND, COLOR_APPWORKSPACE, COLOR_WINDOW,
                     COLOR_WINDOWTEXT, COLOR_MENU, COLOR_MENUTEXT,
                     COLOR_ACTIVECAPTION, COLOR_INACTIVECAPTION, 
                     COLOR_CAPTIONTEXT, COLOR_ACTIVEBORDER,
                     COLOR_INACTIVEBORDER, COLOR_WINDOWFRAME, 
                     COLOR_SCROLLBAR, COLOR_BTNFACE,COLOR_BTNSHADOW, 
                     COLOR_BTNTEXT, COLOR_GRAYTEXT,COLOR_HIGHLIGHT, 
                     COLOR_HIGHLIGHTTEXT, COLOR_INACTIVECAPTIONTEXT, 
                     COLOR_BTNHIGHLIGHT};
//============================================================
// WinMain -- entry point for this application from Windows.
//============================================================
INT APIENTRY WinMain(HANDLE hInstance, HANDLE hPrevInstance, 
                     LPSTR lpCmdLine, INT nCmdShow) {
	MSG	msg;
	INT	rc;

	hInst = hInstance;
	//
	// If first instance, perform any init processing
	//
   if(hPrevInstance == 0)
		if((rc = InitApp(hInstance)) != 0)
			return rc;
	//
	// Initialize this instance
	//
	if((rc = InitInstance(hInstance, lpCmdLine, nCmdShow)) != 0)
		return rc;
	//
	// Application message loop
	//
	while (GetMessage (&msg, NULL, 0, 0)) {
		TranslateMessage(&msg);
		DispatchMessage(&msg);
	}
	// Instance cleanup
	return TermInstance(hInstance, msg.wParam);
}
//-----------------------------------------------------------
// InitApp - Global initialization code for this application.
//-----------------------------------------------------------
INT InitApp(HANDLE hInstance) {
	WNDCLASS 	wc;
	//
	// Register App Main Window class
	//		
	wc.style = 0;                             // Class style
	wc.lpfnWndProc = MainWndProc;             // Callback function
	wc.cbClsExtra = 0;                        // Extra class data
	wc.cbWndExtra = 0;                        // Extra window data
	wc.hInstance = hInstance;                 // Owner handle
	wc.hIcon = LoadIcon(hInst, szIconName);   // Application icon
	wc.hCursor = LoadCursor(NULL, IDC_ARROW); // Default cursor
	wc.hbrBackground = COLOR_WINDOW + 1;
	wc.lpszMenuName =  szMenuName;            // Menu name
	wc.lpszClassName = szAppName;             // Window class name
	if (RegisterClass(&wc) == 0)
		return 1;
	
	StatusBarInit (hInstance);
	return 0;
}
//-----------------------------------------------------------
// InitInstance - Instance initialization code for this app.
//-----------------------------------------------------------
INT InitInstance(HANDLE hInstance, LPSTR lpCmdLine, INT nCmdShow) {
   INT i;
	INT sField[2];
	char szTemp[255];
	OFSTRUCT	of;
	HFILE hFile;

	//Get directory for data file.  Assume prog dir if not specified
	// on the command line.
	if (lstrlen (lpCmdLine)) {
		lstrcpy (szProgDir, lpCmdLine);
		strcat (szProgDir, "\\CON");
		if (OpenFile (szProgDir, &of, OF_EXIST) == HFILE_ERROR) {
			strcpy (szTemp, "Data File Directory:\n");
			lstrcat (szTemp, lpCmdLine);
			strcat (szTemp, "\ninvalid.  Will use program directory.");
			MessageBox (NULL, szTemp, szTitleText, MB_OK | MB_ICONEXCLAMATION);
			GetModuleFileName (hInstance, szProgDir, sizeof (szProgDir));
		} else
			lstrcpy (szProgDir, lpCmdLine);
	} else	
		GetModuleFileName (hInstance, szProgDir, sizeof (szProgDir));
	//Append data file name to path after truncating path from original name
	strcpy (strrchr (szProgDir,'\\'), szDataFileName);
		
	//Read Data file info
	hSWData = GlobalAlloc (GHND, sizeof (SWDATA));
	if (hSWData == 0)
		return ERR_OUTOFMEM;
	//Since Win 3.1 only, we can lock the block for the life
	//of the program. 		
	lpSWData = (LPSWDATA) GlobalLock (hSWData);		
	lpBtnData = lpSWData->bdArray;
	//Read button data from file and verify.
	hFile = OpenFile (szProgDir, &of, OF_READWRITE);
	if (hFile != HFILE_ERROR) {
		_lread (hFile, lpSWData, sizeof (SWDATA));
		_lclose (hFile);
	}
	//Verify data read.  If fail, reset data
	if (lstrcmp (lpSWData->szVerify, szVTag) != 0) {
		_fmemset (lpSWData, 0, sizeof (SWDATA));
		lstrcpy (lpSWData->szVerify, szVTag);
		lpSWData->sWinPosX = 100;
		lpSWData->sWinPosY = 100;
		lpSWData->sBtnRows = 1;
		lpSWData->sCurrSelected = -1;
		lpSWData->fFlags = SWF_SHOWMENU;
	}
	lpSWData->sBtnRows = max (lpSWData->sBtnRows ,1);
	// Create main window
	hMain = CreateWindow (szAppName, szTitleText, WS_POPUP | WS_CAPTION | 
	                      WS_SYSMENU | WS_BORDER | WS_MINIMIZEBOX,
	                      lpSWData->sWinPosX, lpSWData->sWinPosY, 
	                      CW_USEDEFAULT, CW_USEDEFAULT, NULL, NULL, hInstance, NULL);
	if(!hMain) return 0x10;

	//Create status bar
	sField[0] = 0;
	i = StatusBarCreate (hMain, 1, sField);
	if (i) return i;
	
	ShowWindow(hMain, nCmdShow | SW_SHOW);
	DisplayCurrStatus (hMain);
	SetMenuConfig (hMain, lpSWData->fFlags & SWF_SHOWMENU);
	UpdateWindow(hMain);              // force WM_PAINT message
	return 0;                         // return success flag
}
//------------------------------------------------------------
// TermInstance - Instance termination code for this app.
//------------------------------------------------------------
INT TermInstance(HANDLE hinstance, int sDefRC) {

	WriteBtnData ();
	if (hSWData) {
		GlobalUnlock (hSWData);		
		GlobalFree (hSWData);
	}	
	return sDefRC;
}
//============================================================
// Message handling procedures for MainWindow
//============================================================
//------------------------------------------------------------
// MainWndProc - Callback function for application window
//------------------------------------------------------------
LONG CALLBACK MainWndProc(HWND hWnd, UINT wMsg, UINT wParam, 
                          LONG lParam) {
	INT i;
	//
	// Search message list to see if we need to handle this
	// message.  If in list, call procedure.
	//
	for(i = 0; i < dim(MainMessages); i++) {
		if(wMsg == MainMessages[i].Code)
			return (*MainMessages[i].Fxn)(hWnd, wMsg, wParam, lParam);
	}
	return DefWindowProc(hWnd, wMsg, wParam, lParam);
}
//------------------------------------------------------------
// DoCreateMain - process WM_CREATE message for frame window.
//------------------------------------------------------------ 
LONG DoCreateMain (HWND hWnd, UINT wMsg, UINT wParam, LONG lParam) {

	CreateButtons (hWnd);
	return 0;
}
//------------------------------------------------------------
// DoSizeMain - process WM_SIZE message for frame window.
//------------------------------------------------------------ 
LONG DoSizeMain (HWND hWnd, UINT wMsg, UINT wParam, LONG lParam) {

	DisplayCurrStatus (hWnd);
	if (wParam != SIZE_MINIMIZED) {
	}	
	return 0;
}
//------------------------------------------------------------
// DoInitMenuMain - process WM_INITMENU message for frame window.
//------------------------------------------------------------ 
LONG DoInitMenuMain (HWND hWnd, UINT wMsg, UINT wParam, LONG lParam) {
	INT i;
	HMENU hMenu, hTop;
	LPBTNDATA lpBtnPtr;

	hTop = GetMenu (hWnd);
	if (hTop == 0)
		return 0;
	lpBtnPtr = lpBtnData;
	//Get menu for button list
	hMenu = GetSubMenu (hTop, MENU_BUTTONS);
	for (i = 0; i < BTNMAX; i++)
		if (!DeleteMenu (hMenu, IDD_BUTTONS + i, MF_BYCOMMAND))
			break;
			
	for (i = 0; i < lpSWData->sBtnCnt; i++) {
		AppendMenu (hMenu, MF_ENABLED, IDD_BUTTONS + i, 
		            lpBtnPtr->szName);
		lpBtnPtr++;
	}
	// Enable or gray edit and del menu items depending on number of buttons.
	if (lpSWData->sBtnCnt) {
		EnableMenuItem (hMenu, IDM_BTNEDIT, MF_ENABLED);
		EnableMenuItem (hMenu, IDM_BTNDEL, MF_ENABLED);
	} else {	
		EnableMenuItem (hMenu, IDM_BTNEDIT, MF_GRAYED);
		EnableMenuItem (hMenu, IDM_BTNDEL, MF_GRAYED);
	}	
	return 0;
}
//------------------------------------------------------------
// DoPaintMain - process WM_PAINT message for frame window 
//------------------------------------------------------------ 
LONG DoPaintMain (HWND hWnd, UINT wMsg, UINT wParam, LONG lParam) {
	PAINTSTRUCT ps;
	RECT rect;
	HDC hdc;
	
	// Classic Petzoldian example from chapter 1
	hdc = BeginPaint (hWnd, &ps);
	SetBkMode (hdc, TRANSPARENT);
	GetClientRect (hWnd, &rect);
	if (lpSWData->fFlags & SWF_SHOWMENU)
		ModifyClientRect (hWnd, &rect); //Needed for status bar
	DrawText (hdc,"Select \'Add New Button\' menu item to create button", 
	          -1, &rect, DT_CENTER | DT_VCENTER | DT_WORDBREAK);
	EndPaint (hWnd, &ps);
	return 0;
}
//------------------------------------------------------------
// DoCommandMain - process WM_COMMAND message for frame window 
// by decoding the menubar item with the menuitems[] array, 
// then running the corresponding function to process the command.
//------------------------------------------------------------ 
LONG DoCommandMain (HWND hWnd, UINT wMsg, UINT wParam, LONG lParam) {
	INT	i;
	UINT	idItem, wNotifyCode;
	HWND	hwndCtl;

	idItem = (UINT) wParam;                      // Parse Parameters
	hwndCtl = (HWND) LOWORD(lParam);
	wNotifyCode = (UINT) HIWORD(lParam);
	//
	// Call routine to handle control message
	//
	for(i = 0; i < dim(MainMenuItems); i++) {
		if(idItem == MainMenuItems[i].Code)
			return (*MainMenuItems[i].Fxn)(hWnd, idItem, hwndCtl, 
			                               wNotifyCode);
	}
	return DefWindowProc(hWnd, wMsg, wParam, lParam);
}
//------------------------------------------------------------
// DoCloseMain - process WM_CLOSE message for frame window.
//------------------------------------------------------------ 
LONG DoCloseMain (HWND hWnd, UINT wMsg, UINT wParam, LONG lParam) {
	DestroyWindow (hMain);
	return 0;
}
//------------------------------------------------------------
// DoDestroyMain - process WM_DESTROY message for frame window.
//------------------------------------------------------------ 
LONG DoDestroyMain (HWND hWnd, UINT wMsg, UINT wParam, LONG lParam) {
   RECT	rect;

	//Save Window position
	if	(!IsIconic (hWnd)) {
		GetWindowRect (hWnd, &rect);
		lpSWData->sWinPosX = rect.left;
		lpSWData->sWinPosY = rect.top;
	}
	PostQuitMessage (0);
	return DefWindowProc(hWnd, wMsg, wParam, lParam);
}
//============================================================
// Control handling procedures for MainWindow
//============================================================
//------------------------------------------------------------
// OutputDebugLong - Write a number to the debug console
//------------------------------------------------------------
void OutputDebugLong (LONG lNum) {
   char szStr[34];

	ltoa (lNum, szStr, 16);
	OutputDebugString (szStr);
	return;
}				               

//------------------------------------------------------------
// SetMenuConfig - Hides or displays the window menu and
// status bar.
//------------------------------------------------------------ 
void SetMenuConfig (HWND hWnd, BOOL fShow) {
	HMENU hMenu, hSysMenu;

	//Create or delete menu 
	hSysMenu = GetSystemMenu (hWnd, FALSE);
	if (fShow) {
		//Get menu from system menu then set as window menu
		if (GetMenu (hWnd) == 0) {
			//Load original menu for window
			SetMenu (hWnd, LoadMenu (hInst, szMenuName));
			//Delete menu and seperator from system menu.
			DeleteMenu (hSysMenu, GetMenuItemCount (hSysMenu) - 1, 
			             MF_BYPOSITION);
			DeleteMenu (hSysMenu, GetMenuItemCount (hSysMenu) - 1, 
			            MF_BYPOSITION);
			ShowWindow (GetDlgItem (hWnd, IDD_STATBAR), SW_SHOW);
		}	
	} else {			
		if ((hMenu = GetMenu (hWnd)) != 0) {
			//Delete menu from window
			SetMenu (hWnd, 0);
			//Append menu to system menu
			AppendMenu (hSysMenu, MF_SEPARATOR, 0, 0);
			AppendMenu (hSysMenu, MF_ENABLED | MF_POPUP, hMenu, "&Switcher");
			ShowWindow (GetDlgItem (hWnd, IDD_STATBAR), SW_HIDE);
		}
	}
	DrawMenuBar (hWnd);
	return;
}
//------------------------------------------------------------
// DoMainMenuConfig - Process Switcher|Configure menu item
//------------------------------------------------------------ 
LONG DoMainMenuConfig (HWND hWnd, UINT idItem, HWND hwndCtl, 
                       UINT wNotifyCode) {
	INT i; 

	if (MyDisplayDialog(hInst, "Configure", hWnd, 
   	             (WNDPROC) ConfigDlgProc, (LONG) wVersion)) {
		//Hide menu at statbar if necessary
		SetMenuConfig (hWnd, lpSWData->fFlags & SWF_SHOWMENU);
		//Delete old buttons 
		for (i = 0; i < lpSWData->sBtnCnt; i++) {
			DestroyWindow (GetDlgItem (hWnd, IDD_BUTTONS+i));
		}
		//Recreate buttons
		CreateButtons (hWnd);
		DisplayCurrStatus (hWnd);
	}						 	
	return 0;
}
//------------------------------------------------------------
// DoMainMenuExit - Process Exit menu item
//------------------------------------------------------------ 
LONG DoMainMenuExit (HWND hWnd, UINT idItem, HWND hwndCtl, 
                     UINT wNotifyCode) {

	SendMessage (hWnd, WM_CLOSE, 0, 0);
	return 0;
}
//------------------------------------------------------------
// DoMainMenuBtnAdd - Process Button Add menu item
//------------------------------------------------------------ 
LONG DoMainMenuBtnAdd (HWND hWnd, UINT idItem, HWND hwndCtl, 
                      UINT wNotifyCode) {
	BTNDATA BtnInit;
	INT i, rc;

	if (lpSWData->sBtnCnt == BTNMAX) {
		MyPrintError (hWnd, ERR_TOOMANYBUTTONS);
		return 0;
	}	
	memset (&BtnInit, 0, sizeof (BTNDATA));
	rc = MyDisplayDialog(hInst, "AddButton", hWnd, 
   	             (WNDPROC) AddBtnDlgProc, (LONG)(LPBTNDATA)&BtnInit);
	if (rc > 0) {
		//Save current state of the machine for the button
		SaveState (&BtnInit);
		//Delete old buttons
		for (i = 0; i < lpSWData->sBtnCnt; i++)
			DestroyWindow (GetDlgItem (hWnd, IDD_BUTTONS+i));

		*(lpBtnData + lpSWData->sBtnCnt) = BtnInit;
		//Update INI file with new button data
		lpSWData->sCurrSelected = lpSWData->sBtnCnt;
		lpSWData->sBtnCnt++;
		WriteBtnData ();
		//Create new button set
		CreateButtons (hWnd);
	} else if (rc < 0)
		MyPrintError (hWnd, rc);

	DisplayCurrStatus (hWnd);
	return 0;
}
//------------------------------------------------------------
// DoMainMenuBtnEdit - Process Button|Edit menu item
//------------------------------------------------------------ 
LONG DoMainMenuBtnEdit (HWND hWnd, UINT idItem, HWND hwndCtl, 
                           UINT wNotifyCode) {
	BTNDATA BtnInit;
	INT rc, i, sEditItem;

	sEditItem = MyDisplayDialog(hInst, "BtnDel", hWnd, 
	                           (WNDPROC) DelBtnDlgProc, 1);
	if (sEditItem == 0)
		return 0;										 
	sEditItem--;
	
	BtnInit = *(lpBtnData + sEditItem);
	rc = MyDisplayDialog(hInst, "AddButton", hWnd, 
   	             (WNDPROC) AddBtnDlgProc, (LONG)(LPBTNDATA)&BtnInit);
	if (rc > 0) {
		//Save new state of the machine for the button
		SaveState (&BtnInit);
		//Copy new config data back to array and save array.
		*(lpBtnData + sEditItem) = BtnInit;
		lpSWData->sCurrSelected = sEditItem;
		WriteBtnData ();
		//Delete old buttons
		for (i = 0; i < lpSWData->sBtnCnt; i++)
			DestroyWindow (GetDlgItem (hWnd, IDD_BUTTONS+i));
		//Create new button set
		CreateButtons (hWnd);
	} else if (rc < 0)
		MyPrintError (hWnd, rc);

	DisplayCurrStatus (hWnd);
	return 0;
}
//------------------------------------------------------------
// DoMainMenuBtnDel - Process Button|Del menu item
//------------------------------------------------------------ 
LONG DoMainMenuBtnDel (HWND hWnd, UINT idItem, HWND hwndCtl, 
                           UINT wNotifyCode) {
	LPBTNDATA lpBtnPtr;
	INT i, sDelItem;

	sDelItem = MyDisplayDialog(hInst, "BtnDel", hWnd, 
	                           (WNDPROC) DelBtnDlgProc, 0);
	if (sDelItem == 0)
		return 0;										 
	sDelItem--;
	//Delete item by overwriting others on top of it.
	lpBtnPtr = lpBtnData + sDelItem;
	for (i = sDelItem; i < lpSWData->sBtnCnt-1; i++) {
		*lpBtnPtr = *(lpBtnPtr+1);
		lpBtnPtr++;
	}
	//If this button currently active, remove active sel
	if (sDelItem == lpSWData->sCurrSelected)
		lpSWData->sCurrSelected = -1;
	else if (sDelItem < lpSWData->sCurrSelected)
		lpSWData->sCurrSelected--;
	//Delete old buttons 
	for (i = 0; i < lpSWData->sBtnCnt; i++) {
		DestroyWindow (GetDlgItem (hWnd, IDD_BUTTONS+i));
	}		
	lpSWData->sBtnCnt--;
	//Rewrite data file
	WriteBtnData ();
	//Recreate buttons
	CreateButtons (hWnd);
	DisplayCurrStatus (hWnd);
	return 0;
}
//------------------------------------------------------------
// DoMainMenuAbout - Process Help|About menu item
//------------------------------------------------------------ 
LONG DoMainMenuAbout (HWND hWnd, UINT idItem, HWND hwndCtl, 
                     UINT wNotifyCode) {
								
	MyDisplayDialog(hInst, "AboutBox", hWnd, 
   	             (WNDPROC) AboutDlgProc, (LONG) wVersion);
	return 0;
}
//------------------------------------------------------------
// DoMainBtnClick - Process button clicks
//------------------------------------------------------------ 
LONG DoMainBtnClick (HWND hWnd, UINT idItem, HWND hwndCtl, 
                     UINT wNotifyCode) {
	LPBTNDATA lpBtnPtr;
	char szTemp[256];
	INT rc;

	idItem -= IDD_BUTTONS;								
	if ((INT)idItem > lpSWData->sBtnCnt)
		return 0;

	//Run clean up program and save the current state
	if (lpSWData->sCurrSelected != -1) {
		lpBtnPtr = lpBtnData + lpSWData->sCurrSelected;
		if (lstrlen (lpBtnPtr->szEProg) > 0) {
			rc = WinExec (lpBtnPtr->szEProg, SW_SHOW);
			if (rc < 32) {
				strcpy (szTemp, "Couldn\'t launch cleanup program:\n");
				lstrcat (szTemp, lpBtnPtr->szEProg);
				MessageBox (hWnd, szTemp, szTitleText, MB_OK | MB_ICONHAND);
			}
		}
		if (lpBtnPtr->fFlags & FLAGS_RESAVE)
			SaveState (lpBtnPtr);
	}
	//Restore new state and launch startup program
	lpBtnPtr = lpBtnData + idItem;
	RestoreState (lpBtnPtr);
	if (lstrlen (lpBtnPtr->szSProg) > 0) {
		rc = WinExec (lpBtnPtr->szSProg, SW_SHOW);
		if (rc < 32) {
			strcpy (szTemp, "Couldn\'t launch startup program:\n");
			lstrcat (szTemp, lpBtnPtr->szEProg);
			MessageBox (hWnd, szTemp, szTitleText, MB_OK | MB_ICONHAND);
		}
	}
	lpSWData->sCurrSelected = idItem;
	WriteBtnData ();
	DisplayCurrStatus (hWnd);
	return 0;
}
//============================================================
// ConfigDlgProc - Configure dialog box dialog procedure
//============================================================
BOOL CALLBACK ConfigDlgProc (HWND hWnd, UINT wMsg, UINT wParam, 
                             LONG lParam) {
	INT sTemp;
	BOOL fTranslated;
	
	switch (wMsg) {
		case WM_INITDIALOG:

			if (lpSWData->fFlags & SWF_VERT) {
				CheckDlgButton (hWnd, IDD_BYCOLUMNS, TRUE);
			 	SetDlgItemText (hWnd, IDD_ROWTAG, "Number of columns:");
			} else {
				CheckDlgButton (hWnd, IDD_BYROWS, TRUE);
			 	SetDlgItemText (hWnd, IDD_ROWTAG, "Number of rows:");
			}	
			SetDlgItemInt (hWnd, IDD_NUMROWS, lpSWData->sBtnRows, FALSE);
			CheckDlgButton (hWnd, IDD_SHOWMENU, lpSWData->fFlags & SWF_SHOWMENU);
			return TRUE;

		case WM_COMMAND:
			switch (wParam) {

				case IDD_BYROWS:
					if (HIWORD (lParam) == BN_CLICKED) 
						SetDlgItemText (hWnd, IDD_ROWTAG, "Number of rows:");
					break;	

				case IDD_BYCOLUMNS:
					if (HIWORD (lParam) == BN_CLICKED) 
						SetDlgItemText (hWnd, IDD_ROWTAG, "Number of columns:");
					break;

				case IDOK:
					sTemp = GetDlgItemInt (hWnd, IDD_NUMROWS, &fTranslated, FALSE);
					lpSWData->sBtnRows = max (min (sTemp, lpSWData->sBtnCnt), 1);
					if (IsDlgButtonChecked (hWnd, IDD_BYCOLUMNS))
						lpSWData->fFlags |= SWF_VERT;
					else	
						lpSWData->fFlags &= ~SWF_VERT;
					if (IsDlgButtonChecked (hWnd, IDD_SHOWMENU))
						lpSWData->fFlags |= SWF_SHOWMENU;
					else	
						lpSWData->fFlags &= ~SWF_SHOWMENU;

					EndDialog(hWnd, 1);
					return TRUE;

				case IDCANCEL:
					EndDialog(hWnd, 0);
					return TRUE;
			}
	}	
	return FALSE;
}
//============================================================
// AddBtnDlgProc - Add Button dialog box dialog procedure
//============================================================
BOOL CALLBACK AddBtnDlgProc (HWND hWnd, UINT wMsg, UINT wParam, 
                             LONG lParam) {
	static LPBTNDATA lpBtnInit;

	switch (wMsg) {
		case WM_INITDIALOG:
			if (lParam == 0)
				EndDialog(hWnd, -1);

			lpBtnInit = (LPBTNDATA) lParam;
			SendDlgItemMessage (hWnd, IDD_BTNNAME, EM_LIMITTEXT, 
		                       sizeof (lpBtnInit->szName)-1, 0L);
			SendDlgItemMessage (hWnd, IDD_PROGLIST, EM_LIMITTEXT, 
		                       sizeof (lpBtnInit->szSProg)-1, 0L);
			SendDlgItemMessage (hWnd, IDD_TERMPROG, EM_LIMITTEXT, 
		                       sizeof (lpBtnInit->szEProg)-1, 0L);

			SetDlgItemText (hWnd, IDD_BTNNAME, lpBtnInit->szName);
			SetDlgItemText (hWnd, IDD_PROGLIST, lpBtnInit->szSProg);
			SetDlgItemText (hWnd, IDD_TERMPROG, lpBtnInit->szEProg);
			CheckDlgButton (hWnd, IDD_RESAVE, 
			                lpBtnInit->fFlags & FLAGS_RESAVE);
			return TRUE;

		case WM_COMMAND:
			switch (wParam) {

				case IDOK:
					GetDlgItemText (hWnd, IDD_BTNNAME, lpBtnInit->szName, 
					                sizeof (lpBtnInit->szName));
					if (lstrlen (lpBtnInit->szName) == 0) {
						MessageBox (hWnd, "A button name must be entered.", "Switcher",
						            MB_OK | MB_ICONSTOP);
						return TRUE;
					}							
					GetDlgItemText (hWnd, IDD_PROGLIST, lpBtnInit->szSProg, 
					                sizeof (lpBtnInit->szSProg));
					GetDlgItemText (hWnd, IDD_TERMPROG, lpBtnInit->szEProg, 
					                sizeof (lpBtnInit->szEProg));
					if (IsDlgButtonChecked (hWnd, IDD_RESAVE))
						lpBtnInit->fFlags |= FLAGS_RESAVE;
					else
						lpBtnInit->fFlags &= ~FLAGS_RESAVE;

					EndDialog(hWnd, 1);
					return TRUE;

				case IDCANCEL:
					EndDialog(hWnd, 0);
					return TRUE;
			}
	}	
	return FALSE;
}
//============================================================
// BtnDelDlgProc - Delete Set dialog box dialog procedure
//============================================================
BOOL CALLBACK DelBtnDlgProc (HWND hWnd, UINT wMsg, UINT wParam, 
                             LONG lParam) {
	INT i;
	LPBTNDATA lpBtnPtr;

	switch (wMsg) {
		case WM_INITDIALOG:
			lpBtnPtr = lpBtnData;
			SendDlgItemMessage (hWnd, IDD_DELLIST, LB_RESETCONTENT, 0, 0);
			for (i = 0; i < lpSWData->sBtnCnt; i++) {
				SendDlgItemMessage (hWnd, IDD_DELLIST, LB_ADDSTRING, 0,
				                    (LONG)(LPSTR)lpBtnPtr->szName);
				lpBtnPtr++;
			}	
			if (lParam) {
				SetDlgItemText (hWnd, IDD_DELTEXT, "Select button to edit");
				SetDlgItemText (hWnd, IDOK, "Edit");
				SetWindowText (hWnd, "Edit Button");
			}	
			return TRUE;
			
		case WM_COMMAND:
			switch (wParam) {
				case IDOK:
			 		i = (INT) SendDlgItemMessage (hWnd, IDD_DELLIST, 
					                              LB_GETCURSEL, 0, 0);
					if (i == LB_ERR) {
						EndDialog(hWnd, 0);
					} else	
						EndDialog(hWnd, i+1);
					return TRUE;
					
				case IDCANCEL:
					EndDialog(hWnd, 0);
					return TRUE;
			}		
	} 
	return FALSE;
}
//============================================================
// AboutDlgProc - About dialog box dialog procedure
//============================================================
BOOL CALLBACK AboutDlgProc (HWND hWnd, UINT wMsg, UINT wParam, 
                            LONG lParam) {
	char	szAboutStr[128];
                              
	if (wMsg == WM_INITDIALOG) {
		GetDlgItemText (hWnd, IDD_PROGSTR, szAboutStr, sizeof (szAboutStr));
		itoa ((INT)lParam/10, &szAboutStr[strlen (szAboutStr)], 10);
		strcat (szAboutStr, ".");
		itoa ((INT)lParam%10, &szAboutStr[strlen (szAboutStr)], 10);
		SetDlgItemText (hWnd, IDD_PROGSTR, szAboutStr);
		return TRUE;
	}   
	if (wMsg == WM_COMMAND) 
	   if ((wParam == IDOK) || (wParam == IDCANCEL)) {
			EndDialog(hWnd, 0);
			return TRUE;
		} 
	return FALSE;
}
//------------------------------------------------------------
// GetSoundInfo - Reads the sound assignments from WIN.INI
//------------------------------------------------------------ 
INT GetSoundInfo (LPSTR lpBuff, INT sMax, INT *sDataOff) {
	INT sLen, sOut;
	LPSTR lpIn;

	sLen = GetProfileString (pszSProSec, 0, "", lpBuff, sMax);
	sOut = sLen;
	*sDataOff = sLen;
	lpIn = lpBuff + sLen;
	while (*lpBuff) {
		sLen = GetProfileString (pszSProSec, lpBuff, "", lpIn, sMax - sOut);
		lpIn += sLen;
		lpBuff += lstrlen (lpBuff) + 1;
		sOut += sLen;
	}
	return sOut;
}
//------------------------------------------------------------
// SetSoundInfo - Reads the sound assignments from WIN.INI
//------------------------------------------------------------ 
void SetSoundInfo (LPSTR lpBuff, INT sDataOff) {
	LPSTR lpIn;

	lpIn = lpBuff + sDataOff;
	while (*lpBuff) {
		WriteProfileString (pszSProSec, lpBuff, lpIn);
		lpIn += lstrlen (lpIn) + 1;
		lpBuff += lstrlen (lpBuff) + 1;
	}
	return;
}
//------------------------------------------------------------
// GetMouseInfo - Calls the mouse driver to read config info
//------------------------------------------------------------ 
INT GetMouseInfo (LPBTNDATA lpData) {
	LPMOUSEFUNC lpfnGetSetMouseData;
	FARPROC lpfnMProc;
	HINSTANCE hMouseLib;
	MOUSEDATA mdData;
	char szProfName[128];
	char szSecName[22];
	INT i, rc = 0;
	
	hMouseLib = LoadLibrary ("MOUSE");
	if (hMouseLib > 32) {
		lpfnMProc = GetProcAddress (hMouseLib, "Inquire");
		lpfnMProc = GetProcAddress (hMouseLib, "Enable");
		lpfnMProc = GetProcAddress (hMouseLib, "Disable");
		lpfnMProc = GetProcAddress (hMouseLib, "PowerEventProc");
		lpfnMProc = GetProcAddress (hMouseLib, "MouseGetIntVect");

		lpfnGetSetMouseData = (LPMOUSEFUNC)GetProcAddress (hMouseLib, 
		                                                   "GetSetMouseData");
		if (lpfnGetSetMouseData) {
			//Get orientation                   0x0020
			(*lpfnGetSetMouseData)(&mdData, 0x0020);
			lpData->sMouseOrientation = mdData.sMouseOrientation;
			//Get primary / secondary buttons   0x0040
			(*lpfnGetSetMouseData)(&mdData, 0x0040);
			lpData->usMouseButtons = mdData.usMouseBtnAssign;

			//Get horz and vert sensitivity     0x0100
			(*lpfnGetSetMouseData)(&mdData, 0x0100);
			lpData->usMouseSensitivity = mdData.usMouseSpeed;
			//Get acceleration table data       0x0400
			mdData.lpAccelTbl = (LPBYTE)&lpData->bMAccelTbl;
			(*lpfnGetSetMouseData)(&mdData, 0x0400);
			//Get acceleration table names      0x0800
			mdData.lpAccelNames = (LPBYTE)&lpData->bMAccelNames;
			(*lpfnGetSetMouseData)(&mdData, 0x0800);
			//Get active acceleration table     0x1000
			(*lpfnGetSetMouseData)(&mdData, 0x1000);
			lpData->bMAccelActive = mdData.bAccelActive;

			//Get profile name of mouse driver
			szProfName[0] = (char) sizeof (szProfName);
			mdData.lpszProfileName = (LPSTR)&szProfName;
			(*lpfnGetSetMouseData)(&mdData, 0x4000);
			//Get profile strings from mouse.ini
			lpData->sMProHSen = (INT)GetPrivateProfileInt (pszMProGenSec,
			                                pszMProHSen, 0, &szProfName[1]);
			lpData->sMProVSen = (INT)GetPrivateProfileInt (pszMProGenSec,
			                                pszMProVSen, 0, &szProfName[1]);
			lpData->sMProRot = (INT)GetPrivateProfileInt (pszMProGenSec,
			                                pszMProRot, 0, &szProfName[1]);
			lpData->sMProPriB = (INT)GetPrivateProfileInt (pszMProGenSec,
			                                pszMProPriB, 0, &szProfName[1]);
			lpData->sMProSecB = (INT)GetPrivateProfileInt (pszMProGenSec,
			                                pszMProSecB, 0, &szProfName[1]);
			lpData->sMProAAP = (INT)GetPrivateProfileInt (pszMProGenSec,
			                                pszMProAAP, 0, &szProfName[1]);
			//Get acceleration data profile strings
			for (i = 0; i < 4; i++) {
				strcpy (szSecName, pszMProAPSec);
				itoa (i + 1, &szSecName[strlen(szSecName)], 10);
				GetPrivateProfileString (szSecName, pszMProAPL, "",
				                         lpData->szMProAPL[i], 
				                         sizeof (lpData->szMProAPL[i]),
												 &szProfName[1]);
				GetPrivateProfileString (szSecName, pszMProAPM, "",
				                         lpData->szMProAPM[i], 
				                         sizeof (lpData->szMProAPM[i]),
												 &szProfName[1]);
				GetPrivateProfileString (szSecName, pszMProAPF, "",
				                         lpData->szMProAPF[i], 
				                         sizeof (lpData->szMProAPF[i]),
												 &szProfName[1]);
			}
		} else
			rc = 1;
		FreeLibrary (hMouseLib);	
	} else			
		rc = 1;
	return rc;	
}	
//------------------------------------------------------------
// SetMouseInfo - Calls the mouse driver to read config info
//------------------------------------------------------------ 
INT SetMouseInfo (LPBTNDATA lpData) {
	LPMOUSEFUNC lpfnGetSetMouseData;
	HINSTANCE hMouseLib;
	MOUSEDATA mdData;
	char szProfName[128];
	char szSecName[22];
	INT i, rc = 0;
	
	hMouseLib = LoadLibrary ("MOUSE");
	if (hMouseLib > 32) {
		lpfnGetSetMouseData = (LPMOUSEFUNC)GetProcAddress (hMouseLib, 
		                                                   "GetSetMouseData");
		if (lpfnGetSetMouseData) {
			//Set orientation
			mdData.sMouseOrientation = lpData->sMouseOrientation;
			(*lpfnGetSetMouseData)(&mdData, 0x0021);
			//Set primary / secondary buttons
			mdData.usMouseBtnAssign = lpData->usMouseButtons;
			(*lpfnGetSetMouseData)(&mdData, 0x0041);
			//Set horz and vert sensitivity
			mdData.usMouseSpeed = lpData->usMouseSensitivity;
			(*lpfnGetSetMouseData)(&mdData, 0x0101);
			//Set acceleration table data
			mdData.lpAccelTbl = (LPBYTE)&lpData->bMAccelTbl;
			(*lpfnGetSetMouseData)(&mdData, 0x0401);
			//Set acceleration table names
			mdData.lpAccelNames = (LPBYTE)&lpData->bMAccelNames;
			(*lpfnGetSetMouseData)(&mdData, 0x0801);
			//Set active acceleration table
			mdData.bAccelActive = lpData->bMAccelActive;
			(*lpfnGetSetMouseData)(&mdData, 0x1001);

			//Get profile name of mouse driver
			szProfName[0] = (char) sizeof (szProfName);
			mdData.lpszProfileName = (LPSTR)&szProfName;
			(*lpfnGetSetMouseData)(&mdData, 0x4000);
			//Set profile strings from mouse.ini
			MyWritePrivateProfileInt (pszMProGenSec, pszMProHSen, 
			                          lpData->sMProHSen, &szProfName[1]);
			MyWritePrivateProfileInt (pszMProGenSec, pszMProVSen, 
			                          lpData->sMProVSen, &szProfName[1]);
			MyWritePrivateProfileInt (pszMProGenSec, pszMProRot, 
			                          lpData->sMProRot, &szProfName[1]);
			MyWritePrivateProfileInt (pszMProGenSec, pszMProPriB, 
			                          lpData->sMProPriB, &szProfName[1]);
			MyWritePrivateProfileInt (pszMProGenSec, pszMProSecB, 
			                          lpData->sMProSecB, &szProfName[1]);
			MyWritePrivateProfileInt (pszMProGenSec, pszMProAAP, 
			                          lpData->sMProAAP, &szProfName[1]);
			//Set acceleration data profile strings
			for (i = 0; i < 4; i++) {
				strcpy (szSecName, pszMProAPSec);
				itoa (i + 1, &szSecName[strlen(szSecName)], 10);
				WritePrivateProfileString (szSecName, pszMProAPL, 
				                         lpData->szMProAPL[i], &szProfName[1]);
				WritePrivateProfileString (szSecName, pszMProAPM, 
				                         lpData->szMProAPM[i], &szProfName[1]);
				WritePrivateProfileString (szSecName, pszMProAPF, 
				                         lpData->szMProAPF[i], &szProfName[1]);
			}
		} else
			rc = 1;
		FreeLibrary (hMouseLib);	
	} else			
		rc = 1;
	return rc;	
}	
//------------------------------------------------------------
// SaveState - Saves the current state of the machine
//------------------------------------------------------------ 
void SaveState (LPBTNDATA lpData) {
	INT i;
 
 	//Get beep flag
	SystemParametersInfo (SPI_GETBEEP, 0, &lpData->fBeep, 0);
	//Get border width
	SystemParametersInfo (SPI_GETBORDER, 0, &lpData->sBorder, 0);
	//Get desktop grid
	SystemParametersInfo (SPI_GETGRIDGRANULARITY, 0, &lpData->sDeskGrid, 0);
	//Get icon horizonal and vertical spacing
	SystemParametersInfo (SPI_GETICONTITLEWRAP, 0, &lpData->fIconTitleWrap, 0);
	SystemParametersInfo (SPI_ICONHORIZONTALSPACING, 0, 
	                      &lpData->sIconHSpacing, 0);
	SystemParametersInfo (SPI_ICONVERTICALSPACING, 0, 
	                      &lpData->sIconVSpacing, 0);

	//Get desktop pattern and wallpaper
	GetProfileString ("Desktop","Pattern", "(none)", lpData->szPattern,
	                  sizeof (lpData->szPattern));
	lpData->sWallpaperTile = GetProfileInt ("Desktop","TileWallpaper", 0);
	GetProfileString ("Desktop","wallpaper", "(none)", lpData->szWallpaper,
	                      sizeof (lpData->szWallpaper));

	//Get screen saver name, active flag and timeout value
	GetPrivateProfileString ("Boot","scrnsave.exe", "", lpData->szScrnSaveName,
	                         sizeof (lpData->szScrnSaveName),"system.ini");
	GetPrivateProfileString ("ScreenSaver","password", "", lpData->szScrnSavePW,
	                         sizeof (lpData->szScrnSavePW),"control.ini");
	SystemParametersInfo (SPI_GETSCREENSAVEACTIVE, 0, &lpData->fScrnSaveAct, 0);
	SystemParametersInfo (SPI_GETSCREENSAVETIMEOUT, 0, 
	                      &lpData->sScrnSaveTime, 0);

	//Get keyboard delay and repeat speed
	SystemParametersInfo (SPI_GETKEYBOARDDELAY, 0, &lpData->sKeyDelay, 0);
	SystemParametersInfo (SPI_GETKEYBOARDSPEED, 0, &lpData->sKeySpeed, 0);

	//Get mouse Settings.  speed, double click speed and spacing, and
	//  button swap.
	SystemParametersInfo (SPI_GETMOUSE, 0, lpData->sMouse, 0);
	lpData->sDblClkSpeed = GetProfileInt ("Windows","DoubleClickSpeed", 550);
	lpData->sDblClkWidth = GetSystemMetrics (SM_CXDOUBLECLK);
	lpData->sDblClkHeight = GetSystemMetrics (SM_CYDOUBLECLK);
	lpData->fBtnSwap = GetSystemMetrics (SM_SWAPBUTTON);

	//Save mouse data for new MS mouse driver
	if (GetMouseInfo (lpData) == 0)
		lpData->fFlags |= FLAGS_MDATAGOOD;
	else	
		lpData->fFlags &= ~FLAGS_MDATAGOOD;
	// Save Colors 
	for (i = 0; i < 21; i++)
		lpData->rgbColors[i] = GetSysColor(sColorIDs[i]);
	//Save color scheme name
	GetPrivateProfileString ("Current", "Color Schemes", "", 
	                         lpData->szColorScheme, 
	                         sizeof (lpData->szColorScheme), "CONTROL.INI");
	return; 
}
//------------------------------------------------------------
// RestoreState - Restores the current state of the machine
//------------------------------------------------------------ 
void RestoreState (LPBTNDATA lpData) {
	INT i;
	char szNewColor[64], szTemp[20];
	
	//Change color group name used by the control panel	
	WritePrivateProfileString ("Current", "Color Schemes", 
	                           lpData->szColorScheme, "CONTROL.INI");
	//Change the color settings in WIN.INI for next boot	
	for (i = 0; i < 21; i++) {
		itoa (GetRValue (lpData->rgbColors[i]), szNewColor, 10);
		strcat (szNewColor, " ");
		itoa (GetGValue (lpData->rgbColors[i]), szTemp, 10);
		strcat (szNewColor, szTemp);
		itoa (GetBValue (lpData->rgbColors[i]), szTemp, 10);
		strcat (szNewColor, " ");
		strcat (szNewColor, szTemp);
		WriteProfileString ("Colors", pszColorNames[i], szNewColor);
	}
	//Set system Colors
	SetSysColors (21, sColorIDs, lpData->rgbColors);
 	//Set beep flag
	SystemParametersInfo (SPI_SETBEEP, lpData->fBeep, 0, SPIF_UPDATEINIFILE);
	//Set border width
	SystemParametersInfo (SPI_SETBORDER, lpData->sBorder, 0, SPIF_UPDATEINIFILE);
	//Set desktop grid
	SystemParametersInfo (SPI_SETGRIDGRANULARITY, lpData->sDeskGrid, 0, 
	                      SPIF_UPDATEINIFILE);
	//Set icon horizonal and vertical spacing
	SystemParametersInfo (SPI_SETICONTITLEWRAP, lpData->fIconTitleWrap, 0, 
	                      SPIF_UPDATEINIFILE);
	SystemParametersInfo (SPI_ICONHORIZONTALSPACING, lpData->sIconHSpacing, 
	                      0, SPIF_UPDATEINIFILE);
	SystemParametersInfo (SPI_ICONVERTICALSPACING, lpData->sIconVSpacing, 
	                      0, SPIF_UPDATEINIFILE);
	//Set desktop pattern and wallpaper
	SystemParametersInfo (SPI_SETDESKPATTERN, 0, lpData->szPattern, 
	                      SPIF_UPDATEINIFILE);
	MyWritePrivateProfileInt ("Desktop","TileWallpaper", 
	                          lpData->sWallpaperTile, "WIN.INI");
	SystemParametersInfo (SPI_SETDESKWALLPAPER, 0, lpData->szWallpaper, 
	                      SPIF_UPDATEINIFILE);
	//Set screen saver name, active flag and timeout value
	WritePrivateProfileString ("Boot","scrnsave.exe", lpData->szScrnSaveName,
	                           "system.ini");
	WritePrivateProfileString ("ScreenSaver","password", lpData->szScrnSavePW,
	                           "control.ini");
	SystemParametersInfo (SPI_SETSCREENSAVEACTIVE, lpData->fScrnSaveAct, 0, 
	                      SPIF_UPDATEINIFILE);
	SystemParametersInfo (SPI_SETSCREENSAVETIMEOUT, lpData->sScrnSaveTime, 0,
	                      SPIF_UPDATEINIFILE);
	//Set keyboard delay and repeat speed
	SystemParametersInfo (SPI_SETKEYBOARDDELAY, lpData->sKeyDelay, 0, 
	                      SPIF_UPDATEINIFILE);
	SystemParametersInfo (SPI_SETKEYBOARDSPEED, lpData->sKeySpeed, 0, 
	                      SPIF_UPDATEINIFILE);
	//Set mouse Settings.  speed, double click speed and spacing, and
	//  button swap.
	SystemParametersInfo (SPI_SETMOUSE, 0, lpData->sMouse, SPIF_UPDATEINIFILE);
	SystemParametersInfo (SPI_SETDOUBLECLICKTIME, lpData->sDblClkSpeed, 0, 
	                      SPIF_UPDATEINIFILE);
	SystemParametersInfo (SPI_SETDOUBLECLKWIDTH, lpData->sDblClkWidth, 0, 
	                      SPIF_UPDATEINIFILE);
	SystemParametersInfo (SPI_SETDOUBLECLKHEIGHT, lpData->sDblClkHeight, 0, 
	                      SPIF_UPDATEINIFILE);

	if (lpData->fFlags & FLAGS_MDATAGOOD)
		SetMouseInfo (lpData);
								 
	//Since last change, use last param to notify system of all changes
	SystemParametersInfo (SPI_SETMOUSEBUTTONSWAP, lpData->fBtnSwap, 0,
	                      SPIF_UPDATEINIFILE);
								 
	SendMessage (HWND_BROADCAST, WM_WININICHANGE, 0, 0);								 
	return; 
}
//------------------------------------------------------------
// GetBtnDim - Compute button width and height depending on 
// the button text.
//------------------------------------------------------------ 
LONG GetBtnDim (LPBTNDATA lpBtnPtr, INT sCnt, INT sRows) {
	HDC hdc;
	INT i, sMaxWidth = 75, sMaxHeight = 50;
	DWORD dwResult;
	
	hdc = GetDC (NULL);
	for (i = 0; i < sCnt; i++) {
		dwResult = GetTextExtent (hdc, lpBtnPtr->szName, 
		                          lstrlen (lpBtnPtr->szName));
		if (sMaxWidth < (INT)LOWORD (dwResult) + 8)
			sMaxWidth = (INT)LOWORD (dwResult) + 8;

		if (sMaxHeight < (INT)HIWORD (dwResult) + 6)
			sMaxHeight = (INT)HIWORD (dwResult) + 6;
		lpBtnPtr++;			
	}
	ReleaseDC (NULL, hdc);
	//Compute minimum window width
	i = max ((sCnt + sRows - 1) / sRows, 1);
	if (sMaxWidth * i < 200 - GetSystemMetrics (SM_CXBORDER) * 2)
		sMaxWidth = 200 / i;
	
	return MAKELONG (sMaxWidth, sMaxHeight);
}
//------------------------------------------------------------
// CreateButtons - Create buttons on client window 
//------------------------------------------------------------ 
INT CreateButtons (HWND hWnd) {
	HWND hBtn;
	LPBTNDATA lpBtnPtr;
	INT x, y, i, j, sCnt, sRowCnt, sColCnt, sWidth, sHeight;
	INT sFrameHeight, sFrameWidth;
	LONG lResult;
	RECT rectWin;
	POINT pt;

	lpBtnPtr = lpBtnData;
	lResult = GetBtnDim (lpBtnPtr, lpSWData->sBtnCnt, lpSWData->sBtnRows);
	sWidth = LOWORD (lResult);
	sHeight = HIWORD (lResult);
	sCnt = 0;
	sRowCnt = (lpSWData->sBtnCnt + lpSWData->sBtnRows - 1) / lpSWData->sBtnRows;
	for (i = 0; i < lpSWData->sBtnRows; i++) {
		for (j = 0; j < sRowCnt && sCnt < lpSWData->sBtnCnt; j++) {
			if (lpSWData->fFlags & SWF_VERT) {
				x = i * sWidth;
				y = j * sHeight;
			} else {	
				x = j * sWidth;
				y = i * sHeight;
			}
			hBtn = CreateWindow ("button", lpBtnPtr->szName, 
			                     BS_PUSHBUTTON | WS_CHILD | WS_VISIBLE,
	      	                  x, y, sWidth, sHeight, 
	      	                  hWnd, IDD_BUTTONS+sCnt, hInst, NULL);
			if(!hBtn)
				return ERR_BTNWNDCREATE;
			lpBtnPtr++;
			sCnt++;
		}
	}
	//Compute size of frame controls
	sFrameHeight = GetSystemMetrics (SM_CYCAPTION) + 
	               GetSystemMetrics (SM_CYBORDER);
	if (lpSWData->fFlags & SWF_SHOWMENU)						
		sFrameHeight += GetSystemMetrics (SM_CYMENU) + GetStatusBarHeight();
	sFrameWidth = GetSystemMetrics (SM_CXBORDER) * 2;

	sRowCnt = max (sRowCnt, 1);
	sColCnt = max ((sCnt + sRowCnt - 1) / sRowCnt, 1);
	if (lpSWData->fFlags & SWF_VERT) {
		x = sFrameWidth + sWidth * sColCnt;
		y = sFrameHeight + sHeight * sRowCnt;
	} else {	
		x = sFrameWidth + sWidth * sRowCnt;
		y = sFrameHeight + sHeight * sColCnt;
	}
	SetWindowPos (hWnd, NULL, 0, 0, x, y, SWP_NOMOVE | SWP_NOZORDER);
	//Nasty hack to determine if the menu has wrapped to more than one line.
	pt.x = pt.y = 0;
	ClientToScreen (hWnd, &pt);
	GetWindowRect (hWnd, &rectWin);	
	//If menu + caption height bigger than we expect, use the new
	//height instead of the expected one.
	if ((lpSWData->fFlags & SWF_SHOWMENU) &&
	    (pt.y - rectWin.top > GetSystemMetrics (SM_CYCAPTION) + 
	                          GetSystemMetrics (SM_CYMENU))) {

		sFrameWidth = GetSystemMetrics (SM_CXBORDER) * 2;
		sFrameHeight = pt.y - rectWin.top + GetSystemMetrics (SM_CYBORDER) +
		               GetStatusBarHeight();
		if (lpSWData->fFlags & SWF_VERT) {
			x = sFrameWidth + sWidth * sColCnt;
			y = sFrameHeight + sHeight * sRowCnt;
		} else {	
			x = sFrameWidth + sWidth * sRowCnt;
			y = sFrameHeight + sHeight * sColCnt;
		}
		SetWindowPos (hWnd, NULL, 0, 0, x, y, SWP_NOMOVE | SWP_NOZORDER);
	}
	return 0;
}
//------------------------------------------------------------
// WriteBtnData - Write button data to the INI file
//------------------------------------------------------------ 
INT WriteBtnData (void) {
	OFSTRUCT	of;
	HFILE hFile;

	hFile = OpenFile (szProgDir, &of, OF_CREATE);
	if (hFile != HFILE_ERROR) {
		_lwrite (hFile, lpSWData, sizeof (SWDATA) - sizeof (lpSWData->bdArray) + 
		         lpSWData->sBtnCnt * sizeof (BTNDATA));
		_lclose (hFile);
	} else
		return ERR_CANTWRITEDATA;
	return 0;
}	 
//------------------------------------------------------------
// DisplayCurrStatus - Creates a text string describing the 
// current status, then displays it in the status bar.
//------------------------------------------------------------ 
void DisplayCurrStatus (HWND hWnd) {
	LPBTNDATA lpBtnPtr;
	char szTemp[80];
	
	if (lpSWData->sCurrSelected == -1) {
		SetStatusBarText (hWnd, "No setup active", 0);
		return;
	}		
	lpBtnPtr = lpBtnData + lpSWData->sCurrSelected;
	lstrcpy (szTemp, lpBtnPtr->szName);
	strcat (szTemp, " active");
	SetStatusBarText (hWnd, szTemp, 0);
	return;
}
//============================================================  
// General Helper Routines 
//============================================================  
//------------------------------------------------------------
// MyDisplayDialog - Display a dialog box
//------------------------------------------------------------ 
INT MyDisplayDialog (HINSTANCE hInstance, LPCSTR szDlgName,
                     HWND hWnd, WNDPROC lpDialogProc, 
                     LPARAM lParam) {
	WNDPROC lpDlgProcInst;
	INT rc;

	lpDlgProcInst = MakeProcInstance(lpDialogProc, hInst);
	rc = DialogBoxParam (hInstance, szDlgName, hWnd, 
	                     lpDlgProcInst, lParam);
	FreeProcInstance(lpDlgProcInst);
	return rc;                              
}
//------------------------------------------------------------
// MyWritePrivateProfileInt - Writes an integer to the profile
//------------------------------------------------------------
BOOL MyWritePrivateProfileInt (char *szSec, char *szEntry, 
                               int Num, char *szProfile) {
	char	szStr[33];
	                           
	itoa (Num, szStr, 10);
	return WritePrivateProfileString (szSec, szEntry, szStr, 
	                                  szProfile);
}
//------------------------------------------------------------
// MySubClassWindow - Subclasses a window 
//------------------------------------------------------------
WNDPROC MySubClassWindow (HWND hWnd, WNDPROC lpfnNewProc) {
   WNDPROC lpfnOldProc;

	lpfnOldProc = (WNDPROC) GetWindowLong (hWnd, GWL_WNDPROC);
	SetWindowLong (hWnd, GWL_WNDPROC, (LONG) lpfnNewProc);
	return lpfnOldProc;				               
}				               
//------------------------------------------------------------
// MyYield - Yields control to other programs, but returns
// if Windows is idle.
//------------------------------------------------------------
BOOL MyYield (void) {
   MSG	msg;
   BOOL	bCont;
   
   bCont = TRUE;
	while (PeekMessage (&msg, NULL, 0, 0, PM_REMOVE)) {
	   if (msg.message == WM_QUIT)
	      bCont = FALSE;
		TranslateMessage(&msg);
		DispatchMessage(&msg);
	}
	return bCont;
}
//------------------------------------------------------------
// MyPrintError - Displays a message box with an error code
//------------------------------------------------------------ 
void	MyPrintError (HWND hWnd, INT rc) {
	char szErrStr[80];
	char szTemp[12];
	
	if (rc > 0)
	   rc += MYDOSERR;
	else 
	   rc = abs (rc);
	if (LoadString (hInst, rc, szErrStr, sizeof (szErrStr)) == 0) {
		itoa (rc, szTemp, 10);
		strcpy (szErrStr, "Error number: ");
		strcat (szErrStr, szTemp);
	}
	MessageBox (hWnd, szErrStr, szTitleText, MB_OK | MB_ICONHAND);
	return;
}
