/* deflate.c -- compress data using the deflation algorithm
 * Copyright (C) 1995 Steve Benz
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License, see the file COPYING.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#if defined(WIN32) || defined(WIN16) || defined(_DOS)
#include <direct.h> /* For _mkdir */
#include <errno.h> /* For EEXIST */
#include <io.h>
#include <sys/types.h>
#include <sys/stat.h>
#endif

#if defined(WIN16) || defined(_DOS)
#include <dos.h>
#define _findfirst(FS,FD) ( ( 0 == _dos_findfirst( FS, _A_ARCH | _A_HIDDEN | _A_NORMAL | _A_RDONLY | _A_SUBDIR | _A_SYSTEM, FD ) ) ? 0 : -1 )
#define _findnext(JNK,FD) _dos_findnext( FD )
#define _finddata_t _find_t
#define _findclose(JNK) 0
#endif

#ifndef BOOL
#define BOOL int
#endif

#ifndef TRUE
#define TRUE (1==1)
#define FALSE (1==0)
#endif

#define TBLOCK 512
#define NAMSIZ 100

#define	LF_OLDNORMAL	'\0'		/* Normal disk file, Unix compat */
#define	LF_NORMAL	'0'		/* Normal disk file */
#define	LF_LINK		'1'		/* Link to previously dumped file */
#define	LF_SYMLINK	'2'		/* Symbolic link */
#define	LF_CHR		'3'		/* Character special file */
#define	LF_BLK		'4'		/* Block special file */
#define	LF_DIR		'5'		/* Directory */
#define	LF_FIFO		'6'		/* FIFO special file */
#define	LF_CONTIG	'7'		/* Contiguous file */

union hblock {
	char data[TBLOCK];
	struct Header {
       		char name[NAMSIZ];
		char mode[8];
        	char uid[8];
        	char gid[8];
        	char size[12];
        	char mtime[12];
        	char chksum[8];
        	char linkflag;
        	char linkname[NAMSIZ];
	} dbuf;
};

void delete_chars( char *s, unsigned int n )
{
	assert( strlen(s) >= n );
	while ( '\0' != (s[0] = s[n]) )
	  s++;
}

void insert_string( char *into, char *from )
{
	int n = strlen(from);
	int i = strlen( into )+1;
	while ( --i >= 0 )
	  into[i+n] = into[i];
	memcpy( into, from, n );
}

#if defined(_DOS) || defined(WIN16)
static void unix_name_to_dos_name( char *fn )
{
	static char legal_chars[] =
	  "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890_~$-";
	char *p = fn;
	while ( *p != '\0' ) {
		int goodchars = strspn( p, legal_chars );
		while ( p[goodchars] != '\\' && p[goodchars] != '.' &&
			       p[goodchars] != '\0' ) {
			p[goodchars] = '$';
			goodchars += 1 + strspn( p+goodchars+1, legal_chars );
		}
		if ( goodchars > 8 ) {
			delete_chars( p+8, goodchars-8 );
			goodchars = 8;
		}
		if ( p[goodchars] == '\\' ) {
			p += goodchars+1;
			continue;
		}
		if ( p[goodchars] == '\0' )
		  break;
		p += goodchars + 1;
		goodchars = strspn( p, legal_chars );
		while ( p[goodchars] != '\\' && p[goodchars] != '.' &&
				p[goodchars] != '\0' ) {
			p[goodchars] = '$';
			goodchars += 1 + strspn( p+goodchars+1, legal_chars );
		}
		if ( goodchars > 3 ) {
			delete_chars( p+3, goodchars-3 );
			goodchars = 3;
		}
		if ( p[goodchars] == '.' ) {
			delete_chars( p+goodchars, strcspn( p+goodchars, "\\" ) );
		}
		p += goodchars;
	}
}
#else
#define unix_name_to_dos_name(FN)
#endif


static BOOL check_header( hblock &hdr )
{
	if ( !( hdr.dbuf.chksum[7] == '\0' &&
	        hdr.dbuf.chksum[6] == ' ' ||
	        strspn( hdr.dbuf.chksum, " 0123456789" ) == 7 ) &&
	     !( hdr.dbuf.chksum[6] == '\0' &&
	        strspn( hdr.dbuf.chksum, " 0123456789" ) == 6 ) )
	  return( FALSE );

	unsigned long target_chksum = strtol( hdr.dbuf.chksum, 0, 8 );
	memcpy( hdr.dbuf.chksum, "        ", 8 );
	unsigned long chksum = 0;
	for ( register int i = 0; i < TBLOCK; i++ )
	  chksum += hdr.data[i];

	if ( (0xffff & chksum) != (0xffff & target_chksum) )
	  return( FALSE );

	for ( i = 0; i < NAMSIZ; i++ ) {
		if ( hdr.dbuf.name[i] == '/' )
		  hdr.dbuf.name[i] = '\\';
		if ( hdr.dbuf.linkname[i] == '/' )
		  hdr.dbuf.linkname[i] = '\\';
	}

	unix_name_to_dos_name( hdr.dbuf.name );
	unix_name_to_dos_name( hdr.dbuf.linkname );

	return( TRUE );
}

static unsigned long nDataBytesLeft;
static BOOL chmodrdonly;
static FILE *outfile;
static int eofblocks = 0;
BOOL errors = FALSE;
static BOOL listContents = FALSE;
static char topdir[NAMSIZ];
static BOOL no_discernable_topdir = FALSE;

static void get_file_topdir( char *filename )
{
	if ( no_discernable_topdir )
	  return;
	char *slash = strchr( filename, '\\' );
	if ( slash == 0 ) {
		no_discernable_topdir = TRUE;
		if ( topdir != 0 )
		  free( topdir );
	}
	if ( topdir[0] == '\0' ) {
		strncpy( topdir, filename, slash-filename );
		topdir[slash-filename] = '\0';
	}
	else {
		if ( strlen(topdir) != (size_t)(slash-filename) ||
		     0 != strncmp( topdir, filename, slash-filename ) ) {
			free( topdir );
			no_discernable_topdir = TRUE;
		}
	}
}

extern "C" int complain( const char *fmt, ... );
extern "C" int warning( const char *fmt, ... );
extern "C" void status_message( const char *fmt, ... );
extern "C" void zcat( FILE * );

static BOOL process_block( hblock &b )
{
	FILE *infile;
	unsigned long count;
	char block[TBLOCK];

	if ( nDataBytesLeft > 0 ) {
		if ( nDataBytesLeft > TBLOCK )
		  count = TBLOCK;
		else
		  count = nDataBytesLeft;
		if ( outfile != 0 &&
		     (size_t)count > fwrite( b.data, 1, (size_t)count, outfile ) ) {
			fclose( outfile );
			return( FALSE );
		}
		nDataBytesLeft -= count;
		if ( nDataBytesLeft == 0 ) {
			if ( outfile != 0 )
			  fclose( outfile );
			if ( chmodrdonly &&
			     0 != _chmod( b.dbuf.name, _S_IREAD ) &&
			     complain( "Unable to make %s read-only",
					    b.dbuf.name ) )
			  return( FALSE );
		}
		return( TRUE );
	}
	else if ( b.dbuf.name[0] == '\0' ) {
		hblock zero;
		memset( zero.data, '\0', TBLOCK );
		if ( 0 != memcmp( b.data, zero.data, TBLOCK ) )
		  if ( warning( "Bad EOF block" ) )
		    return( FALSE );
		eofblocks++;
		return( TRUE );
	}
	else {
		if ( !check_header( b ) &&
		     complain( "Invalid block (tar file is corrupted)" ) )
		  return( FALSE );

		unsigned long mode = strtol( b.dbuf.mode, 0, 8 );
		unsigned long uid = strtol( b.dbuf.uid, 0, 8 );
		unsigned long gid = strtol( b.dbuf.gid, 0, 8 );
		unsigned long size = strtol( b.dbuf.size, 0, 8 );
		unsigned long mtime = strtol( b.dbuf.mtime, 0, 8 );

	      curveball:
		switch( b.dbuf.linkflag ) {
		      case '0': // Normal file
		      case '\0': // Older tar files sometimes use this.
			/* Some tar's seem to want to make directories look like
			 *  files.  This heads that problem off.
			 */
			if ( b.dbuf.name[strlen(b.dbuf.name)-1] == '\\' ) {
				b.dbuf.linkflag = '5';
				goto curveball;
			}

			nDataBytesLeft = size;

			if ( listContents ) {
				printf( "%s\n", b.dbuf.name );
				outfile = 0;
			}
			else {
				get_file_topdir( b.dbuf.name );
				outfile = fopen( b.dbuf.name, "wb" );
				if ( outfile == 0 &&
				     complain( "Unable to write to %s", b.dbuf.name ) )
				  return( FALSE );
				status_message( "X %s\n", b.dbuf.name );
			}


			/* Change the permissions.  Note that DOS only has
			 *  read/write & read-only modes, so that's all we
			 *  deal with.  We also are taking the Unix user-
			 *  permissions as the final permissions
			 */
			chmodrdonly = (!listContents && (mode & 0200) == 0 );
			break;

		      case '2': // Symbolic link
			{
				char *slash = strrchr( b.dbuf.name, '\\' );
				if ( slash != 0 && b.dbuf.linkname[0] != '\\' &&
				     1+strlen(b.dbuf.linkname)+slash-b.dbuf.name < NAMSIZ ) {
					char old = slash[1];
					slash[1] = '\0';
					insert_string( b.dbuf.linkname, b.dbuf.name );
					slash[1] = old;
				}
			}
		      case '1': // Hard link
			if ( listContents ) {
				printf( "%s -> %s\n", b.dbuf.name, b.dbuf.linkname );
				break;
			}
			/* try to make a copy of the file */
			infile = fopen( b.dbuf.linkname, "rb" );
			if ( infile == 0 ) {
				if ( warning( "Unable to duplicate linked file (%s->%s)",
					      b.dbuf.name, b.dbuf.linkname ) )
				  return( FALSE );
				break;
			}
			status_message( "L %s -> %s\n", b.dbuf.name, b.dbuf.linkname );
			get_file_topdir( b.dbuf.name );
			outfile = fopen( b.dbuf.name, "wb" );
			if ( outfile == 0 ) {
				fclose( infile );
				if ( complain( "Unable to write to %s", b.dbuf.name ) )
				  return( FALSE );
				break;
			}

			while ( !feof( infile ) ) {
				count = fread( block, 1, TBLOCK, infile );
				if ( count <= 0 ) {
					fclose( infile );
					fclose( outfile );
					if ( complain( "Could not read from %s",
							  b.dbuf.linkname ) )
					  return( FALSE );
					return( TRUE );
				}
				if ( (size_t)count >
				     fwrite( block, 1, (size_t)count, outfile ) ) {
					fclose( infile );
					fclose( outfile );
					if ( complain( "Could not write to %s",
							  b.dbuf.name ) )
					  return( FALSE );
					return( TRUE );
				}
			}
			fclose( infile );
			fclose( outfile );
			/*
			 * TODO: Set the permissions like the other file.
			 */
			break;

		      case '3': // Character special file
		      case '4': // Block special file
		      case '6': // Pipe
		      case '7': // "contiguous" file - whatever that means
			if ( warning( "Unsupported file type ignored, file=%s\n",
				      b.dbuf.name ) )
			  return( FALSE );
			/*
			 * TODO: Warn about the fact we're ignoring this
			 *   nasty file.
			 */
		        break;
		      case '5': // Directory
			if ( listContents )
			  break;
			/* Get rid of trailing backslashes, if any */
			if ( b.dbuf.name[strlen(b.dbuf.name)-1] == '\\' )
			  b.dbuf.name[strlen(b.dbuf.name)-1] = '\0';
			if ( 0 != _mkdir( b.dbuf.name ) &&
			     errno != EEXIST && errno != EACCES &&
			     complain( "Unable to create directory, %s",
					    b.dbuf.name ) )
			  return( FALSE );
			break;
		      default:
			if ( complain( "Bad record in tar file\n" ) )
			  return( FALSE );
		}
	}
	return( TRUE );
}


hblock io_block;
static int partialFill = 0;

extern "C" int write_to_tar( void *buf, unsigned int count )
{
	if ( partialFill + count < TBLOCK ) {
		memcpy( &io_block.data[partialFill], buf, count );
		partialFill += count;
		return( count );
	}
	else {
		memcpy( &io_block.data[partialFill], buf,
			TBLOCK - partialFill );
		partialFill += 0;
		if ( !process_block( io_block ) )
		  return( -1 );
		return( TBLOCK - partialFill );
	}
}

static void test_u2d( char *s )
{
	char buf[500];
	strcpy( buf, s );
	unix_name_to_dos_name( buf );
	printf( " '%s'->'%s'\n", s, buf );
}

static void untar_init()
{
	eofblocks = 0;
	errors = FALSE;
	outfile = 0;
	nDataBytesLeft = 0;
	no_discernable_topdir = FALSE;
	strcpy( topdir, "" );
}

static BOOL untar_okay()
{
	return( !errors && ( eofblocks > 0 && partialFill == 0 ) );
}


size_t serious_fread( void *buffer, size_t size, size_t count, FILE *stream )
{
	size_t n = 0;
	size_t inc = 0;
	do {
		n += inc;
		inc = fread( ((char *)buffer) + n * count, size, count - n,
			     stream );
	} while ( inc > 0 && (n+inc) < count );
	return( n + inc );
}

extern "C" BOOL untar( FILE *tarfile, BOOL compressed, BOOL contents )
{
	hblock b;

	untar_init();
	listContents = contents;

	if ( compressed )
	  zcat( tarfile );
	else {
		while ( !feof( tarfile ) ) {
			int bytes = serious_fread( &b, 1, sizeof(b), tarfile );
			if ( bytes == 0 && feof(tarfile) )
			  break; // why did this happen?
			if ( bytes < sizeof(b) ) {
				int err = ferror( tarfile );
				int eof = feof( tarfile );
				int no = errno;
				if ( complain( "Error while reading tar file" ) )
				  return( FALSE );
			}

			if ( !process_block(b) )
			  return( FALSE );
		}
	}
	return( untar_okay() );
}

extern "C" void dotopdir()
{
	if ( no_discernable_topdir || topdir[0] == '\0' )
	  return;

	_finddata_t fd;
	char filespec[NAMSIZ+5];
	sprintf( filespec, "%s\\*.*", topdir );
	int h = _findfirst( filespec, &fd );
	if ( h < 0 )
	  return;
	do {
		if ( 0 == strcmp( fd.name, "." ) ||
		     0 == strcmp( fd.name, ".." ) )
		  continue;

		char oldname[NAMSIZ+15];
		sprintf( oldname, "%s\\%s", topdir, fd.name );
		rename( oldname, fd.name );
	} while ( 0 == _findnext( h, &fd ) );
	_findclose( h );
	_rmdir( topdir );
}
