/* deflate.c -- compress data using the deflation algorithm
 * Copyright (C) 1995 Steve Benz
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License, see the file COPYING.
 *
 * This module is the main routine for ungzptar.  In addition to defining
 *   main(), it attempts to define all the other routines that interact
 *   with the user.
 *
 * This is one of two routines written by Steve Benz - the other being untar.cpp,
 *   the rest comes from the main gzip distribution.  These files were modified
 *   slightly (well, actually gzip.c was mangled rather alot and the rest were
 *   barely touched).  The originals can be found by the name of gzipsrc.zip or
 *   maybe gzip.src on ftp.cica.indiana.edu or its mirrors.
 *
 * The code in here and in untar.cpp easily fits the description of "spagetti
 *   code" and isn't something I'm terribly proud to put my name on, but it does
 *   pretty much work.
 */


#include <stdio.h>
#include <string.h>
#include <conio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <fcntl.h>
#include <io.h>

int wait_on_err = 0;
int verbose = 0;

extern "C" int untar( FILE * tarfile, int compressed, int verbose );
extern "C" void dotopdir(void);

#ifdef WIN16
#define _cprintf
#define _getch getchar
#endif

#ifndef BOOL
#define BOOL int
#endif

#ifndef TRUE
#define TRUE (1==1)
#define FALSE (1==0)
#endif

extern BOOL errors;

extern "C" int complain( const char *format, ... )
{
	va_list args;

	errors = TRUE;

	va_start( args, format );
	vfprintf( stderr, format, args );
	if ( format[strlen(format)-1] != '\n' )
	  putchar( '\n' );
	if ( wait_on_err ) {
		_cprintf( " - Press escape to continue unpacking, anything else to cancel:" );
		char key = _getch();
		_cprintf( "\r                                                                   \r" );
		if ( 27 != key )
		  return( 0 ); // unpacking should continue
	}
	return( -1 ); // cease unpacking
}

extern "C" int warning( const char *format, ... )
{
	va_list args;

	va_start( args, format );
	vfprintf( stderr, format, args );
	if ( format[strlen(format)-1] != '\n' )
	  putchar( '\n' );
	if ( wait_on_err ) {
		_cprintf( " - Press escape to cancel the unpacking, anything else to continue:" );
		char key = _getch();
		_cprintf( "\r                                                                   \r" );
		if ( 27 == key ) {
			errors = TRUE;
			return( -1 ); // unpacking should cease
		}
	}
	return( 0 ); // cease unpacking
}

extern "C" void status_message( const char *format, ... )
{
	va_list args;

	if ( verbose ) {
		va_start( args, format );
		vprintf( format, args );
		if ( format[strlen(format)-1] != '\n' )
		  putchar( '\n' );
	}
}


static void usage()
{
	fprintf( stderr, "usage: ungzptar [/wait] [/verbose] [/contents] [ /unzip | /untar ]\n"
			 "                [/topdir] [ <filename> ]\n"
			 "\n"
			 "   /wait     - When an error or warning occurs, ask if the\n"
			 "               unpacking should continue\n"
			 "   /verbose  - Displays the names of the files being unpacked\n"
			 "   /contents - Does not unpack the archive, just displays the\n"
			 "               names of the files in the archive.\n"
			 "   /unzip    - Indicates that the input is a compressed tar file\n"
			 "   /untar    - Indicates that the input is a normal tar file\n"
			 "   /topdir   - If all the files in the archive are to be unpacked\n"
			 "               into a single directory, then all the files and\n"
			 "               directories in that top-level directory will be\n"
			 "               moved to the current directory\n"
			 "\n"
	                 "       NOTE: if no file name is given, it assumes that the\n"
	                 "             input is coming from a pipe or a file.  If the\n"
			 "             file name is not given or it does not end in\n"
			 "             .TAR or .TGZ, then one of the /unzip or /untar\n"
			 "             options must be given.\n" );
	exit( 1 );
}

int main( int argc, char **argv )
{
	int i = 1;
	char *filename = 0;
	int force_unzip = 0;
	int force_untar = 0;
	int listContent = 0;
	int topdir = 0;

	wait_on_err = 0;
	verbose = 0;
	FILE *input;

	for ( char **s = &argv[1]; *s != 0; s++ ) {
		if ( **s != '/' && **s != '-' ) {
			if ( filename == 0 )
			  filename = *s;
			else
			  usage();
		}
		else if ( 0 == _stricmp( (*s)+1, "wait" ) ) {
			if ( wait_on_err )
			  usage();
			else
			  wait_on_err = 1;
		}
		else if ( 0 == _stricmp( (*s)+1, "unzip" ) ) {
			if ( force_unzip || force_untar )
			  usage();
			else
			  force_unzip = 1;
		}
		else if ( 0 == _stricmp( (*s)+1, "untar" ) ) {
			if ( force_unzip || force_untar )
			  usage();
			else
			  force_untar = 1;
		}
		else if ( 0 == _stricmp( (*s)+1, "verbose" ) ||
			  0 == _stricmp( (*s)+1, "v" ) )
		  verbose = 1;
		else if ( 0 == _stricmp( (*s)+1, "contents" ) ||
			  0 == _stricmp( (*s)+1, "t" ) )
		  listContent = 1;
		else if ( 0 == _stricmp( (*s)+1, "topdir" ) )
		  topdir = 1;
		else
		  usage();
	}

	if ( filename == 0 ) {
		if ( !force_untar && !force_unzip )
		  usage();
	}
	else {
		char *ext = strrchr( filename, '.' );
		if ( ext == 0 )
		  usage();
		if ( 0 == _stricmp( ext, ".tgz" ) )
		  force_unzip = 1;
		else if ( 0 == _stricmp( ext, ".tar" ) )
		  force_untar = 1;
		else
		  usage();
	}

	if ( filename == 0 ) {
		input = stdin;
		if ( isatty(fileno(input)) ) {
			fprintf( stderr, "ungzptar: Input must come from a file or a pipe\n" );
			exit(1);
		}
		setmode( fileno(input), O_BINARY );
	}
	else {
		input = fopen( filename, "rb" );
		if ( input == 0 ) {
			char buf[200];
			sprintf( buf, "ungzptar: Unable to read %s", filename );
			perror( buf );
			exit( 1 );
		}
	}

	int rc = untar( input, force_unzip, listContent );
	if ( input != stdin )
	  fclose( input );

	if ( topdir )
	  dotopdir();

	return( rc ? 1 : 0 );
}
