// view.cpp : Defines the low level view classes.
//
// Copyright (c) 1994 Michael Taylor
//

#include <zapp.hpp>
extern "C" {
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
// the following is for regular expressions
#include "regexp.h"
}
#include "fileview.h"

#include "file.hpp"
#include "view.hpp"

FileView::FileView (zFrameWin * hWnd, zFontSpec *font, int vscroll, int hscroll, 
							char *fname) : 
	zMDIChildFrame (hWnd, ZNEW zSizer, zSTDFRAME, "[View]") 
{
	tWnd = ZNEW FileText (this, ZNEW zSizeWithParent, fname, vscroll, 
		hscroll);
        if (!tWnd->name ())
                return;
    	caption (tWnd->name ());
	setIcon (ZNEW zIcon (zResId (IDI_FileView)));
	deleteOnClose (1);
	if (!font)
		tWnd->canvas ()->pushFont (ZNEW zFont (SystemFixed)); 
	else
		tWnd->canvas ()->pushFont (ZNEW zFont (*font));
	tWnd->show ();
	setHandler(this, (NotifyProc)(KeyProc)&FileView::keyUp,  WM_KEYUP);
	setHandler(this, (NotifyProc)(KeyProc)&FileView::keyUp,  WM_SYSKEYUP);
	setHandler(this, (NotifyProc)(KeyProc)&FileView::keyDown,WM_KEYDOWN);
	setHandler(this, (NotifyProc)(KeyProc)&FileView::keyDown,WM_SYSKEYDOWN);
	setFocus();
}	  

FileView::~FileView () {
        if (tWnd)
                delete tWnd;
}	  

int FileView::MDIActivate(zActivateEvt *e) {
	if (e->active()) setFocus();

	return 0;
}

int FileView::size (zSizeEvt *szEvt) {
	tWnd->SetSize (szEvt->width (), szEvt->height ());

       	return zWindow::size (szEvt);
}

int FileView::ch(zKeyEvt*Ev) {
	return 1;
}

int FileView::keyDown(zKeyEvt*Ev) {
	return tWnd->keyDown (Ev);
}

int FileView::keyUp(zKeyEvt*Ev) {
	return tWnd->keyUp (Ev);
}

FileText::FileText (zMDIChildFrame *cWnd, zSizeWithParent *s, char *fname, 
	int vscroll, int hscroll) : 
			zPane (cWnd, s, vscroll*zSCROLLV | hscroll*zSCROLLH),
			FileRead (fname) {
	down = 0;
	horzOffset = 0;
	findoffset = 3;
        highlight = 0;		// no highlighted area yet!
        inPrint = 0;
        leftMargin = 0;
        rightMargin = 0;
        topMargin = 0;
        bottomMargin = 0;
        wordwrap = 0;
        noffset = woffset = 0;
        scrolling = 0;
        start_noffset = end_noffset = start_woffset = 
        	end_woffset = start_rlen = end_rlen = 0;
	if (vscroll || hscroll) {
	        scroller (ZNEW FVScroller (this));
		if (vscroll)
			addVertScrollBar ();
		if (hscroll)
			addHorzScrollBar ();
		scroller()->respondToSize();
	}
	
        clip = ZNEW zClipboard (app->rootWindow ());
	parent = cWnd;
	
	dline = (char HUGE *)malloc (cols+1);
	dlen  = cols + 1;
}

FileText::~FileText () {
        if (scroller ())
                delete scroller ();
	delete clip;
	if (dline)
	        free (dline);
}

void FileText::SetWordWrap (int i) {
	wordwrap = i;
	ShowCurrPage ();
}

void FileText::SetSize (int x, int y) {
	// get dimension of the font
	canvas()->lock();
	zDimension d = canvas()->getTextDim ("W", 1);
	canvas()->unlock();
        // calculate rows and columns
        rows = y / d.height ();
        cols = x / d.width  ();

	if (scrollBarVert()) {
		scrollBarVert()->limits (zRange (1, TotalLines ()));
		scrollBarVert()->pageDelta (Rows ());
		scrollBarVert()->moveDelta (1);
    	}
	if (scrollBarHoriz()) {
		scrollBarHoriz()->limits (zRange (1, MAXRIGHT));
		scrollBarHoriz()->pageDelta (10);
		scrollBarHoriz()->moveDelta (1);
    	}
}

void FileText::ResetScrollPos () {
	if (scrollBarVert ())
		scrollBarVert ()->pos (LineNo ());
	if (scrollBarHoriz ())
		scrollBarHoriz ()->pos (horzOffset > MAXRIGHT ? 
						MAXRIGHT : horzOffset);
}

void FileText::setpageStart (char HUGE *p) {
       	FileRead::setpageStart (p);
	start_woffset = woffset;
	start_noffset = noffset;
	start_rlen = rlen;
}

void FileText::setpageStart () {
       	FileRead::setpageStart ();
	start_woffset = woffset;
	start_noffset = noffset;
	start_rlen = rlen;
}

void FileText::setpageEnd () {
       	FileRead::setpageEnd ();
	end_woffset = woffset;
	end_noffset = noffset;
	end_rlen = rlen;	
}

void FileText::gotoPageStart () {
	moveToPageStart ();
	woffset = start_woffset;
	noffset = start_noffset;
	rlen = start_rlen;
}

void FileText::gotoPageEnd () {
	moveToPageEnd ();
	woffset = end_woffset;
	noffset = end_noffset;
	rlen = end_rlen;
}

void FileText::MarkEndLine (int x, int y) {
      	char *endline = (char *)malloc(cols+1);
       	memset (endline, '-', cols);
       	endline[cols] = '\0';
       	if (cols > 25)
       	    memcpy (endline + cols/2 - 7, " end of file ", 13);
	DisplayLine (endline, x, y, NEXT);
	free (endline);
}

void FileText::ShowPrevLine () {
  	if (atStart ())
  	        return;
  
	gotoPageEnd ();
	
        if (!GetNextLine (cols, PREV))
		return;

  	if (atStart ()) {
  	    	ShowFirstPage ();
  	        return;
  	}
       	setpageStart ();
	
	ShowPrevPage ();
}

void FileText::ShowNextLine () {
  	if (atEnd ())
  	        return;
  
	gotoPageStart ();

        if (!GetNextLine (cols, NEXT))
		return;
       	setpageEnd ();
	
       	ShowNextPage ();
}

void FileText::ShowPrevPage () {
        int i;
  
  	if (atStart ())
  	        return;
  
	gotoPageStart ();
	
  	if (atStart ()) {
  	    	ShowFirstPage ();
  	        return;
  	}
	setpageEnd ();
	SetMark ();
	
        for (i = rows - 1; i >= 0; --i) {
	       	if (!DisplayLine (0, 0, i, PREV)) {
	       	        ShowFirstPage ();
	       	        return;
	       	}
        }
        setpageStart ();
	setFindLine ();

        return;
}

void FileText::ShowNextPage () {
        int i, e=0;
  
	gotoPageEnd ();

  	if (atEnd ())
  	        return;
  
	setpageStart ();
	setFindLine ();
	SetMark ();
        
        for (i = 0; i < rows; i++) {
	       	if (!DisplayLine (0, 0, i, NEXT)) {
	       	        if (!e) {
	        		MarkEndLine (0, i);
	        		++e;
	        	} else
	       			DisplayLine ("", 0, i, 0);
        	}
        }
        setpageEnd ();
	
        return;
}

void FileText::ShowLastPage () {
  	toEnd ();
  
	ShowPrevPage ();
	return;
}

void FileText::ShowFirstPage () {
  	toStart ();
  
        setpageEnd ();
  	end_rlen = 0;		// forces a reread of the text file line

        ShowNextPage ();
        setFindLine (1);
        toStart ();
        
        return;
}

void FileText::ShowCurrPage () {
        int i, e=0;

	gotoPageStart ();
	setpageEnd ();
	ShowNextPage ();
	
        return;
}

void FileText::HiliteCurrPage () {
        int i;
  
	SetMark ();
	gotoPageStart ();

        for (i = 0; i < prevmarky; i++) {
	       	if (!DisplayLine (0, 0, i, NEXT))
	       	        break;
        }
        prevmarky = emarky;

        return;
}

void FileText::ShowFoundPage (int dir) {
        int i=0, e=0, fline;
        char *line;
  
	moveToLineNo (getFindLine ());
	if (NEXT==dir)
        	fline = LineNo()+1;
        else
        	fline = getFindLine ();

     	setpageStart ();

	gotoPageStart ();
	rlen = 0;

        startmline = getFindLine ();
	endmline = startmline + 1;
  	
  	for (i = 0; i < findoffset; i++) {
		if (!(line = GetNextLine (cols, PREV)))
			break;
	}
	setpageStart ();

	endmarkx = startmarkx = 0;
	startmarky = i; endmarky = startmarky+1;
	highlight = 1;
//char str1[256];
//sprintf (str1, "findlineno=%d", 
//(int)getFindLine ());
//MessageBox (0, str1, "FileRead - ShowFoundPage", MB_OK);	

	setpageEnd ();
	ShowNextPage ();
	
//	highlight = 0;		// turn off highlighting - disable copy!
        setFindLine (fline);	
	ResetMenu ();
        return;
}

// if the end selected point is earlier than the start selected point
// then swap the mark points
void FileText::SetMark () {
	prevmarky = emarky;		// save the previous bottom
	if ( endmarky < startmarky ||
	    (endmarky == startmarky && endmarkx < startmarkx)) {
        	emarkx = startmarkx;
        	emarky = startmarky;
	} else {
	        emarkx = endmarkx;
	        emarky = endmarky;
	}
	if (endmline < startmline) {
	        smline = endmline;
	        emline = startmline;
	} else {
	        smline = startmline;
	        emline = endmline;
	}
}

/*
	GetNextLine
	===========
	
	Calls the file class to get a line. Returns the part of the 
	line that should be displayed.
	
	Handles word wrap.
*/
char * FileText::GetNextLine (int width, int dir) {
	if (dlen != width + 1) {	// reallocate room for screen line
		dline = (char HUGE *)realloc (dline, width+1);
		if (!dline) {
			delete ZNEW zMessage (app->rootWindow(), 
			/* message= */	"Memory allocation error\nClose file and try again",  
			/* title=   */	"File Viewer: Fatal Error", 
			MB_OK|MB_ICONINFORMATION);

		        return NULL;
		}
		dlen  = width + 1;
	}
	
	// wrap lines
	if (rlen && wordwrap && woffset < rlen && dir == NEXT) {
		FormatLine (width, line, dline);

		return dline;
	} else if (rlen && wordwrap && dir == PREV) {
	        int	poffset;
        
		if (noffset == 0) {	// get another line
			line = GetLine (&rlen, dir);
			if (!line)
			        return NULL;
			noffset = rlen;
		}
		// to find the next section for dir==PREV
		// we move to the last remembered position in the
		// wrapped line and we remember the new last previous
		// position.
		//
	        poffset = noffset;
		woffset = noffset = 0;
		FormatLine (width, line, dline);
		while (woffset < poffset) {
		        noffset = woffset;
			FormatLine (width, line, dline);
			if (woffset >= rlen)
			        break;	// just in case!!!
		}

		return dline;
	} else {
	        woffset = 0;
		line = GetLine (&rlen, dir);
		if (!line)
		        return NULL;
		FormatLine (width, line, dline);
//char str1[256];
//sprintf (str1, "lineno=%d, width=%d, rlen=%d, line=%-80.80s\ndline=%-80.80s", 
//(int)LineNo (), (int)width, (int)rlen, (char *)line, (char *)dline);
//MessageBox (0, str1, "FileRead - GetLine", MB_OK);	
		
		return dline;
	}
	
        return NULL;
}

char * FileText::FormatLine (int width, char HUGE * lline, char * dline) {
        int	i;
        char	*q, HUGE *p, HUGE *r;
        
	if (0==rlen) {		// empty line
		memset (dline, ' ', width);
	dline[width] = '\0';
		return dline;
	}
        if (woffset >= rlen)	// error condition
                return NULL;
        
	// copy part of line to display line
	memset (dline, ' ', width);
	for (i = 0, q = dline, r = lline + rlen, p = lline + woffset;
	     p < r && i < width + horzOffset; ) {
        	if (*p == '\t') {	// translate tabs to spaces
        	        i += 8 - i % 8;
        	        if (i > horzOffset)
        	                q = dline + i - horzOffset;
        	        ++woffset;
                        ++p;
                        continue;
        	}
        	if (++i > horzOffset) {
        	// move 1 character. Do not overflow currline
	        	if (q < dline + width) {
#ifdef WINDOWS
				if (IsCharAlphaNumeric (*p) || isprint (*p))
#else	        	    
				if (isprint (*p))
#endif				
					*q++ = *p;
				else if (*p >= ' ') 
					*q++ = '.'; // non-printable character
			}
		}
       	        ++woffset;
	    	++p;
	}
	dline[width] = '\0';

	return dline;
}

/*
	Display Line will highlight the line if it is in the
	selected area
*/
#ifdef WINDOWS
#define NEWCOLOUR 	GetSysColor(COLOR_HIGHLIGHT)
#define TEXTCOLOUR 	GetSysColor(COLOR_HIGHLIGHTTEXT)
#else
#define NEWCOLOUR	BLUE
#define TEXTCOLOUR	WHITE
#endif
int FileText::DisplayLine (char * displayline, int x, int y, int dir) {
       	zColor old, oldtext, *newcolor, *textcolor;
       	int dohigh, width, f=0;
       	char * line;

	if (!displayline) {	// if line not supplied read one from file
		line = GetNextLine (cols, dir);
	    	if (!line)
	        	return 0;
		width = cols;
	} else {
		width = strlen (displayline);
		if (width > cols)
	        	width = cols;
		if (width < 1) {
		        line = (char *)malloc (cols+1);
	        	++f;
		        memset (line, ' ', cols);
		        line[cols] = '\0';
	        	width = cols;
		} else {
		    	line = displayline;
			width = strlen (line);
		}
    	}

	canvas()->lock();
	zDimension d = canvas()->getTextDim ("W", 1);
	canvas()->unlock();

       	dohigh = (highlight && !displayline && 
       		  LineNo () > smline && LineNo () <= emline);
       	if (dohigh) {
	       	old = canvas ()->backColor ();
	       	oldtext = canvas ()->textColor ();
       		newcolor = ZNEW zColor (NEWCOLOUR);
       		textcolor = ZNEW zColor (TEXTCOLOUR);
		canvas ()->backColor (*newcolor);
		canvas ()->textColor (*textcolor);
		canvas()->lock ();
		canvas()->text (x*d.width(), y*d.height(), line, width);
		canvas()->unlock ();
		canvas ()->backColor (old);
		canvas ()->textColor (oldtext);
		delete newcolor;
		delete textcolor;
	} else {
		canvas()->lock ();
		canvas()->text (x*d.width(), y*d.height(), line, width);
		canvas()->unlock ();
	}
	if (f)
	        free (line);
        
        return (1);
}
#undef NEWCOLOUR
#undef TEXTCOLOUR

// subroutine to print out errors from regexp regcomp()/regexp()
void regerror (char *str) {
	delete ZNEW zMessage (app->rootWindow(), 
	/* message= */	str,  
	/* title=   */	"File Viewer: Regular Expression Error", 
	MB_OK|MB_ICONINFORMATION);
}

/*
	FindString
	==========
	
	Find the string in the current file
	
	searchpos is the start of the line for the last successful find
	flags has the following values:
		Pos 0: direction of search
		    1: match case
		    2: use regular expressions
	next:
		0:	first
		1:	next (same direction)
*/
int FileText::FindString (char * str, unsigned long flags, int next) {
	int dir=NEXT, mcase=0, regexp=0, reflag=0, savewrap=0;
	char *p, *line;
	long	origline;
	struct regexp * re;

	dir    = (int)flags & 0x01;
	mcase  = (int)(flags & 0x02) ? TRUE : FALSE;
	regexp = (int)(flags & 0x04) ? TRUE : FALSE;
	
	savewrap = wordwrap;
	wordwrap = 0;

	if (next) { // find next...
		moveToLineNo (getFindLine ());
        } else {
                setFindLine ();
            	if (PREV == dir) {
    	    		gotoPageEnd ();
    	        } else {
			gotoPageStart ();
		}
    	}
    	origline = LineNo ();

	if (regexp) {			// call the regexp compiler
	        re = regcomp (str);	// if this fails it calls regerror()
		if (!re)
		        goto goodexit;  // actually failed but we have
				   	// already reported the reg-exp error
	}

        for (;;) {
             	setFindLine ();
                if ((line = GetNextLine (cols, dir))) {
                    if (regexp) {
			reflag = 0;
                        if (!regexec (re, line, reflag))
                                continue;	// try next line
		        // match found show page starting with 
		        // current line
		        if (PREV==dir)
		             	setFindLine ();
       	        	ShowFoundPage (dir);
       	        	free ((void *)re);
		        goto goodexit;		// match
                    } else {
			for (p = line; *p != '\0'; ++p) {
			    if (!mcase && !strnicmp (str, p, strlen (str)) ||
			         mcase && !strncmp  (str, p, strlen (str))) {			                
			        // match found so show page starting with 
			        // current line
			        if (PREV==dir)
			             	setFindLine ();
        	        	ShowFoundPage (dir);
			        goto goodexit;	// match
			    }
			}
		    }
	        } else 
	            break;
        }
// nomatch
	if (regexp)        
	       	free ((void *)re);
        moveToLineNo (origline);	// restore position if find failed
	wordwrap = savewrap;

        return 0;		// no match
goodexit:
	wordwrap = savewrap;
        return 1;		// match

}

int FileText::printSetup(zPrinterDisplay *pr,zPrintJob *pj) {
	inPrint=1;
  	toStart ();
        printPos = getPos ();
	return 1;
}

BOOL FileText::morePages() {
	if (!printPos) {
		return 0;
	}
	return 1;
}

int FileText::printFile (zPrinterDisplay *pr) {
	zPrJobDlg *prDlg =
		ZNEW zPrJobDlg (app->rootWindow(), zResId("ZDEFPRINT"));
	zPrintJob *pj = ZNEW zPrintJob (pr, prDlg);
	pj->setSetup (this, (zPrSetupProc)&FileText::printSetup);
	pj->setPrint (this, (zPrintProc)&FileText::print);
	pj->setMorePages (this, (zPrMorePagesProc)&FileText::morePages);
	pj->setJobName (name ());
	pj->go ();

	return 1;
}

void FileText::pageSetup (int left, int right, int top, int bottom, 
							zFontSpec *f) {
        leftMargin   = left;
        rightMargin  = right;
        topMargin    = top;
        bottomMargin = bottom;
        font = f;
        
}

int FileText::print (zPrinterDisplay *pd, zRect *rect) {
	char *line, HUGE *origp;
       	int width, lcols, lrows, lineno, loff;
	zRect r;

        origp = getPos ();
        setPos (printPos);

	pushDisplay(pd);

	canvas()->lock();
	canvas()->getVisible(r);	

	zDimension d = canvas()->getTextDim ("W", 1);
	lcols = r.width () / d.width () - rightMargin - leftMargin;
	lrows = r.height () / d.height () - bottomMargin;
	if (lcols <= 0) {
		delete ZNEW zMessage (app->rootWindow(), 
		/* message= */	"Print: right and left margin greater than page width",  
		/* title=   */	"File Viewer Error", 
		MB_OK|MB_ICONINFORMATION);
		
	        return (0);	// can't print to 0 length page!!!
	} else 	if (lrows <= 0) {
		delete ZNEW zMessage (app->rootWindow(), 
		/* message= */	"Print: page length of zero or bottom margin too great",  
		/* title=   */	"File Viewer Error", 
		MB_OK|MB_ICONINFORMATION);
		
	        return (0);	// can't print to 0 length page!!!
	} else if (lrows <= topMargin+bottomMargin ) {
		delete ZNEW zMessage (app->rootWindow(), 
		/* message= */	"Print: Top and Bottom margin greater than page length",  
		/* title=   */	"File Viewer Error", 
		MB_OK|MB_ICONINFORMATION);
		
	        return (0);	// can't print to page!!!
	}

	loff = leftMargin*d.width ();
	lineno = topMargin;
	while ((line = GetNextLine (lcols, NEXT))) {
		width = strlen (line);
		if (width > lcols)
	        	width = lcols;
		canvas()->text(loff, lineno*d.height(), 
			line, width);
	        if (++lineno >= lrows)
	                break;	// end of page
	}
	canvas()->unlock();
	popDisplay();
	if (!line)
	        printPos = 0;
	else
	        printPos = getPos ();	// remember position of print
	setPos (origp);
	return 0;
}

void FileText::clearRect () {
	canvas()->lock();
	canvas()->fill (sizer()->rect());
	canvas()->unlock();
}

int FileText::copyToClip() {
	int i;
	long startline;
	char *line, HUGE *startp, HUGE *endp, *c;

	highlight = 0;		// turns off highlighting
        clip->clear ();
	
	SetMark ();
	startp = moveToLineNo (smline+1);
	moveToLineNo (emline-1);
	endp = getPos ();
        ++endp;		// make sure we copy cr/lf or final character if at eof
#if 0
	if (!atEnd ())
		++endp;	// step past cr/lf pair
#endif

        c = (char *)malloc ((int)(endp-startp+1));
        memcpy (c, startp, (int)(endp-startp));
	c[endp-startp] = '\0';
	clip->add (c);

	return 1;
}

int FileText::draw(zDrawEvt*) {
	zRect zr;
	
	canvas()->getVisible (zr);
	SetSize (zr.width (), zr.height ());
	ShowCurrPage ();
	ResetMenu ();

	return 1;
}

/*
	The following events are used to control the highlighting
	of a selected area. 
	Selection is started by the mouseButtonDown method
	           and ended by the mouseButtonUp method
	The mouseMove method is the place to highlight whilst moving
*/

int FileText::mouseMove (zMouseMoveEvt *me) {
	if (down) {
		zPoint x = me->pos ();
		canvas()->lock();
		zDimension d = canvas()->getTextDim ("W", 1);
		canvas()->unlock();
		endmarkx = x.x()/d.width();
		endmarky = x.y()/d.height();
		endmline = StartLineNo () + endmarky;
	        highlight = 1;
//char str[256];
//sprintf (str, "x.y()=%d, Rows()=%d", (int)x.y(), (int) Rows());
//MessageBox (0, str, "FileView - 2", MB_OK);	
		// check to see if screen should be scrolled?
//	        if (x.y() > d.height()*Rows()) {	// scroll down?
//                	ShowNextLine ();
//	        } else if (x.y() <= 0) {		// scroll up?
//	                ShowPrevLine ();
//		} else {
			HiliteCurrPage ();	// highlight while dragging
//		}
	}
	return 1;
}

int FileText::mouseButtonDown (zMouseClickEvt *be) {
	if (!down && be->isButtonDown(1)) {
		captureMouse();
		zPoint x = be->pos ();
		highlight = 0;
		ShowCurrPage ();	// clear any old highlighting
		canvas()->lock();
		zDimension d = canvas()->getTextDim ("W", 1);
		canvas()->unlock();
		startmarkx = endmarkx = x.x()/d.width();
		startmarky = endmarky = x.y()/d.height();
		startmline = StartLineNo () + startmarky;
		down=1;
	}
	return 1;
}

int FileText::mouseButtonUp (zMouseClickEvt *be) {
	if (down && be->isButton(1)) {
		releaseMouse();
		zPoint x = be->pos ();
		canvas()->lock();
		zDimension d = canvas()->getTextDim ("W", 1);
		canvas()->unlock();
		endmarkx = x.x()/d.width();
		endmarky = x.y()/d.height();
		endmline = StartLineNo () + endmarky;
	        highlight = 1;
		ShowCurrPage ();	// display with highlighting
		down=0;
		ResetMenu ();
	}
	return 1;
}

int FileText::keyDown(zKeyEvt*Ev) {
        highlight = 0;	// any key press will clear highlight
    	switch (Ev->ch()) {
		case VK_LEFT:			// scroll left
		        if (Ev->ctrl())
		                horzOffset = 0;
		        else {
				horzOffset -= 1;
				if (horzOffset < 0)
				        horzOffset = 0;
			}
			ShowCurrPage ();
		        break;
		case VK_RIGHT:			// scroll right
			horzOffset += 1;
			ShowCurrPage ();
		        break;
		case VK_DOWN:			// scroll up
			ShowNextLine ();
		        break;
		case VK_UP:			// scroll down
			ShowPrevLine ();
		        break;
		case VK_PRIOR:			// page up
		        if (Ev->ctrl()) {
		        	ShowFirstPage ();
		        	break;
		        }
			ShowPrevPage ();
		        break;
		case VK_NEXT:			// page down
		        if (Ev->ctrl()) {
		        	ShowLastPage ();
		        	break;
		        }
		case ' ':			// space bar
			ShowNextPage ();
		        break;
		case VK_HOME:			// HOME (control-HOME)
		        if (Ev->ctrl()) {
		        	ShowFirstPage ();
		        	break;
		        }
		        break;
		case VK_END:			// END (control-END)
		        if (Ev->ctrl()) {
		        	ShowLastPage ();
		        	break;
		        }
		        break;
		default:
			return 0;
	}
       	ResetScrollPos ();
	ResetMenu ();
	return 1;
}

int FileText::keyUp(zKeyEvt*Ev) {
	return 1;
}

FVScroller::FVScroller (FileText *t) : zScrollerBase ((zWindow *)t) 
{
	parent = t;
}

FVScroller::~FVScroller () 
{
}

int FVScroller::vertBottom () 
{
        parent->highlight = 0;		// turn off highlighting
	parent->ShowLastPage ();
	ResetMenu ();

	return 1;
}

int FVScroller::vertTop () 
{
        parent->highlight = 0;		// turn off highlighting
        parent->ShowFirstPage ();
	ResetMenu ();

	return 1;
}

int FVScroller::vertLineMove () {
        parent->highlight = 0;		// turn off highlighting
	if (parent->scrollBarVert()->pos() > parent->scrollBarVert()->oldpos())
		parent->ShowNextLine ();
	else
		parent->ShowPrevLine ();
	ResetMenu ();

	return 1;
}

int FVScroller::vertPageMove () {
        parent->highlight = 0;		// turn off highlighting
	if (parent->scrollBarVert()->pos() > parent->scrollBarVert()->oldpos())
		parent->ShowNextPage ();
	else
		parent->ShowPrevPage ();
	ResetMenu ();

	return 1;
}

int FVScroller::vertThumbPos () 
{
	int  i;

	if (parent->scrollBarVert()->pos() >= 
	    parent->scrollBarVert()->limits().hi())
		return vertBottom ();
	else if (parent->scrollBarVert()->pos() <= 
	         parent->scrollBarVert()->limits().lo())
		return vertTop ();
	
	i = parent->scrollBarVert()->pos();
	parent->moveToLineNo (i);
	parent->setpageStart ();
        parent->highlight = 0;		// turn off highlighting
	parent->ShowCurrPage ();
	ResetMenu ();

	return 1;
}

int FVScroller::horzLineMove () 
{
        parent->SetOffset (parent->scrollBarHoriz()->pos() - 1);
        parent->highlight = 0;		// turn off highlighting
	parent->ShowCurrPage ();
	ResetMenu ();

	return 1;
}

int FVScroller::horzPageMove () 
{
        parent->SetOffset (parent->scrollBarHoriz()->pos() - 1);
        parent->highlight = 0;		// turn off highlighting
	parent->ShowCurrPage ();
	ResetMenu ();

	return 1;
}

int FVScroller::horzThumbPos () 
{
	parent->SetOffset (parent->scrollBarHoriz()->pos() - 1);
        parent->highlight = 0;		// turn off highlighting
        parent->ShowCurrPage ();
	ResetMenu ();
        
	return 1;
}

int FVScroller::horzLeft () 
{
	parent->SetOffset (parent->scrollBarHoriz()->limits().lo() - 1);
        parent->highlight = 0;		// turn off highlighting
	parent->ShowCurrPage ();
	ResetMenu ();

	return 1;
}

int FVScroller::horzRight () 
{
	parent->SetOffset (parent->scrollBarHoriz()->limits().hi() - 1);
        parent->highlight = 0;		// turn off highlighting
	parent->ShowCurrPage ();
	ResetMenu ();

	return 1;
}
