;;; SELSET.LSP
;;; Copyright 1993,94 by Mountain Software
;;;
;;; This program requires ELF, the Extended List Function library
;;;
;;; THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED
;;; WARRANTY.  ALL IMPLIED WARRANTIES OF FITNESS FOR ANY PARTICULAR
;;; PURPOSE AND OF MERCHANTABILITY ARE HEREBY DISCLAIMED.
;;;
;;;*===================================================================*
;;;
;;; This is a test program to demonstrate the use of the ELF functions
;;; ss_move, ss_scale, ss_rotate which implement the AutoCAD dragger ADS
;;; function ads_draggen; ss_xform which implements ads_xform. In addition
;;; the command SSU demonstrates the use of the EFL finctions ss_union,
;;; ss_inters, and ss_diff.
;;;
;;; The drag functions don't change the selection set but only provide a
;;; means for the user to visualize a user defined entity modification
;;; function. The AutoLISP programmer can then use ss_xform and / or
;;; combinations of AutoLISP functions and AutoCAD commands to modify
;;; the selection set.
;;;
;;; For instance, a COPY_SCALE command could be made that creats a new copy
;;; of a selection set scaled and translated leaving the original intact.
;;; the ss_scale function would be called to let the user drag the selection
;;; set and then the COPY command invoked followed by ss_xform to do the
;;; scaling.

(Princ "\nLoading SelSet.Lsp...")
(Load"ELF")                             ;load ELF.EXP, color and key symbols

;*------ supporting functions -------

;;; multiply vectors

(defun mult_vec (v1 v2)
  (mapcar '* v1 v2)
)

;;; subtract vectors

(defun sub_vec (v1 v2)
  (mapcar '- v1 v2)
)

;;; sum a vector

(defun sum_vec (v)
  (+ (car v)(cadr v)(caddr v))
)

;;; multiply a matrix and a vector

(defun mat_x_vec(mat pt)
  (mapcar '(lambda (m)
             (sum_vec(mult_vec m pt))
           )
           mat
  )
)

;;; rotate selection set in xy plane from a base point

(defun rotate(ss bp ang / ca sa nsa)
  (setq ca  (cos ang)
        sa  (sin ang)
        nsa (* -1 sa)
        mat (list(list ca nsa 0)              ; rotated matrix
                 (list sa ca  0)
                 (list 0  0   1))
        vec (mat_x_vec mat bp)                ; translation vector
        vec (sub_vec bp vec)                  ; subtract the vectors
        mat (list(list ca nsa 0 (car vec))    ; rotated and translated matrix
                 (list sa ca  0 (cadr vec))
                 (list 0  0   1 (caddr vec)))
  )
  (ss_xform ss mat)
)

;;; translate a selection set

(defun move(ss pt1 pt2)
  (setq v (sub_vec pt2 pt1)
        mat (list (list 1 0 0 (car v))
                  (list 0 1 0 (cadr v))
                  (list 0 0 1 (caddr v))))
  (ss_xform ss mat)
)

;;; scale a selection set relative to a base point

(defun scale(ss bp sf)
  (setq mat (list (list sf 0 0 (- (car bp)   (*(car bp) sf)))
                  (list 0 sf 0 (- (cadr bp)  (*(cadr bp) sf)))
                  (list 0 0 sf (- (caddr bp) (*(caddr bp) sf)))))
  (ss_xform ss mat)
)

;;; a work-alike scale command

(defun C:SC( / ss pt1 pt2 sf)
  (setq ss (ssget))
  (setq pt1 (getpoint "Base Point"))
  (initget 128)
  (setq pt2 (ss_scale ss pt1 "\n<Scale factor>/Reference " 5))
  (setq sf (if(=(type pt2) 'STR)
    (atof pt2)
    (distance pt1 pt2)))
  (if(scale ss pt1 sf)
    (printf "\nScale Factor is %f" sf)
    (princ "\nError"))
  (princ)
)

;;; a work-alike move command

(defun C:MV( / ss pt1 pt2)
  (setq ss (ssget))
  (setq pt1 (getpoint "Base point or displacement: "))
  (initget 128)
  (setq pt2 (ss_move ss pt1 "\nSecond point of displacement: " 5))
  (if (=(type pt2) 'STR)
    (setq pt2 (read (sprintf "(%s)" pt2))))
  (if(not(move ss pt1 pt2))
    (princ "\nError"))
  (princ)
)

;;; a work-alike rotate command

(defun C:RO( / ss pt1 pt2 ang)
  (setq ss (ssget))
  (setq pt1 (getpoint "Base Point: "))
  (initget 128)
  (setq pt2 (ss_rotate ss pt1 "\n<Rotation angle>/Reference: " 5))
  (setq ang (if (=(type pt2) 'STR)
    (angtof pt2)             ;convert to angle
    (angle pt1 pt2)
  ))
  (if(rotate ss pt1 ang)
    (printf "\nRotation angle is %f" (rtd ang))
    (princ "\nError"))
  (princ)
)

;;; a mirror command with no translation (mirrors around x or y axis)
;;;
;;; The rotation axis calculations and translation function are left
;;; as an exercise for the student (The instructor is too lazy to do it!)

(defun C:MIR()
  (setq ss (ssget) ang pi
        maty (list(list 1)
                  (list 0 -1)
                  (list 0 0 -1))     ; rotate in yz plane
        matx (list(list -1)
                  (list 0 1)
                  (list 0 0 -1))     ; rotate in xz plane
  )
  (initget "X Y")
  (if(= "X" (getkword "\nRotation Axis: [X/Y]"))
    (ss_xform ss matx)
    (ss_xform ss maty)
  )
  (princ)
)

(defun C:SSU()
  (prompt "\nSSU changes the color to red of the selection set created from logical")
  (prompt "\noperators on two selection set. Pick the First selection set.")
  (setq ss1 (ssget))
  (prompt "Second selection set")
  (setq ss2 (ssget))
  (initget "Union Intersection Difference")
  (setq kw (getkword "\nUnion/Intersection/Difference[Union]"))
  (setq ss
    (cond ((= kw "Intersection")  (ss_inters ss1 ss2))
          ((= kw "Difference")    (ss_diff   ss1 ss2))
          (T                      (ss_union  ss1 ss2))
  ))
  (if(> (sslength ss) 0)
    (progn
      (setvar "CMDECHO" 0)
      (command ".CHPROP" ss "" "C" "1" "")
    )
    (prompt "\nEmpty selection set")
  )
  (princ)
)

(DeFun C:SELSET( / mstr flst rslt fname key i)
  (Setq mstr '("MV  - move a selection set"
               "SC  - scale a selection set"
               "RO  - rotate a selection set"
               "MIR - mirror a selection set"
               "SSU - logical set manipulation")
        flst '(c:mv c:sc c:ro c:mir c:ssu)
        rslt (Wmenu mstr))
  (Cls 7)
  (If(/= (Cadr rslt) Esc_Key)
    (Eval(List(Nth (Car rslt) flst))))
  (princ)
)

(princ "\nSelSet.Lsp loaded...")
(princ)

