;;; DEMO.LSP
;;; Copyright 1992-94 by Mountain Software
;;;
;;; This program requires ELF, the Extended Lisp Function library
;;;
;;; THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED
;;; WARRANTY.  ALL IMPLIED WARRANTIES OF FITNESS FOR ANY PARTICULAR
;;; PURPOSE AND OF MERCHANTABILITY ARE HEREBY DISCLAIMED.
;;;
;;;*===================================================================*
;;;
;;; Demo.Lsp is a demonstration of the capabilities of the ELF
;;; library. Demo exercises many (but not all) ELF functions and
;;; commands and illustrates how the functions can be utilized.
;;; It also provides AutoLISP programmers with sample code that
;;; can included in their own programs.

(Princ "\nLoading Demo.Lsp")
(Load"ELF")

;;;*----- The ELF Demo

(DeFun C:DEMO( / mstr mfun i attr done video vcols vrows ans key helplst)
  (SetQ mstr   '("Introduction" "Window System" "Video Functions"
                 "Data Entry Form" "String Functions" "Math Functions"
                 "Directory Demo" "File Functions" "Low Level Functions"
                 "Look at an ASCII file" "Function List" "ELF Apps" "Quit")
        mfun   '(intro wdemo scrdemo edemo strdemo mdemo ddemo
                 fdemo ldemo look_demo (list () '(Set_Color 23) '(c:elf))
                 apps_demo (list () '(SetQ done T)))
        bcolor (| white lgrey_bg)
        logo    "\n\n\n\n"
        old_error *error*
        *error*   DemoError
        helplst (list "[ ELF Demo Help ]" ""
                      "Select one of the Menu Items or press <Esc> to Quit" ""
                      "This is a Demonstration of ELF"
                      "The Extended Lisp Function Library" ""
                      "ELF is available from:" ""
                      (eval name) (eval address)
                      (strcat city ", " state " " zip) ""
                      "Press <F1> for Menu help")
  )
  (TextScr)
  (SetQ attr (| lgrey blue_bg)
        video (Get_Video) vcols (Car video) vrows (Cadr video))
  (Set_Color attr)
  (Scr_Fill 0 1 vcols (- vrows 2) 176 bcolor)
  (GotoXY 0 1)
  (Puts logo bcolor)
  (Scr_Fill 0 0 vcols 1 32 attr)
  (Prts 19 0 "Extended Lisp Function [ELF] library Demo" (| white blue_bg))
  (Scr_Fill 0 (1- vrows) vcols 1 32 attr)
  (Prts 26 (1- vrows) "(c) 1992-94 Mountain Software" attr)
  (bloop)
  (While (Not done) (Progn
    (Set_menu_help helplst)
    (Wopen -1 -1 27 18 (| white cyan_bg) (| cyan cyan_bg) (| no_bd shadow_bd))
    (Wputcen "ELF Demo Menu")
    (WgotoXY 0 1)
    (Wputcen "Select" (| lcyan cyan_bg))
    (SetQ ans (Wmenu mstr -1 6 (| white cyan_bg) (| black cyan_bg)
                     (| white black_bg) (| single_bd tlhl_bd))
          key (Cadr ans)
          i   (Car ans))
    (Wclose)
    (Save_Screen)
    (If(= key Esc_Key)
      (SetQ done T)
    ;else
      (Eval(List(nth i mfun)))         ;Execute the selected function
    )
    (TextScr)
    (Restore_Screen)
  ))
  (WcloseAll)
  (SetQ *error* old_error)
  (Cls 7)
)

;;;*----- Introduction

(DeFun INTRO()
  (Wmenu '("ELF is a library of over 190 new functions for"
           "AutoLISP.  In addition, ELF also adds 17 file"
           "management and utility commands to AutoCAD. The"
           "library and commands are contained in a single"
           "EXP file, and expands AutoLISP into a \"rich\""
           "programming language. The range of functions"
           "include video and text window extentions, math"
           "and string handling, file and directory, list"
           "handling, keyboard, sound, and utility routines."
           "During this demonstration you may press <F1> for"
           "online help and <Esc> to exit or quit.")
           -1 -1 23 31 31 (| 5 32))
)

;;;*----- ELF Apps

(DeFun APPS_DEMO( / mstr flst rslt fname key i)
  (Setq mstr '("ELF Notepad - an ASCII editor in AutoLISP"
               "eTables     - view Block, Layer, etc tables"
               "SelSet      - selection set manipulation"
               "Template    - a generic ELF application")
        flst '("notepad" "etables" "selset" "template")
        rslt (Wmenu mstr))
  (Cls 7)
  (If(/= (Cadr rslt) Esc_Key)
    (run (Nth (Car rslt) flst)))
)

;;;*----- Load and run an AutoLISP file

(Defun RUN(funcname / func filename)
  (SetQ func     (Read(Strcat "C:" funcname))
        filename (Strcat funcname ".LSP"))

;*-----Load the function if not already loaded

  (If(Not(Cadr(Eval Func)))
    (If(Findfile filename)
      (Load funcname)
   ;else
      (Wmsg(Strcat "\nError: " filename "\nis not on the AutoCAD library path."))
    )
  )

;*----- Execute the function if it exists

  (if(= (Type (Eval func)) 'LIST)
    (Eval(list func))
  ;else
    (Wmsg(Strcat "\nError: " funcname "\nis not a valid AutoLISP function"))
  )
  (princ)
)

(SetQ contact "Jerry Workman"
      name    "Mountain Software"
      address "1579 Nottingham Road"
      city    "Charleston"
      state   "WV"
      zip     "25304-2453"
)

;;;*-----Data Entry Demo

(DeFun EDEMO( / fields tcolor dcolor)
  (SetQ tcolor (| dgrey lgrey_bg)
        dcolor (| white lgrey_bg)
        wcolor (| black lgrey_bg))

;;;*=========================================================*
;;;*              Data Entry Form Layout                     *
;;;*=============col=row=prompt=========col=row=symbol==width*
  (SetQ fields '((1  1   "Contact"       15 1   contact 40)
                 (1  3   "Company Name"  15 3   name    40)
                 (1  5   "Address"       15 5   address 40)
                 (1  7   "City"          15 7   city    15)
                 (33 7   "State"         40 7   state    2)
                 (43 7   "ZIP"           47 7   zip     10)))

  (Wopen 0 0 vcols vrows 7 7 0)         ;open a window to cover the screen
  (Wpopup 68 15 dcolor dcolor 0)        ;and another for the background
  (Wtitle "Data entry form" 1 wcolor)
  (Wtitle "<Ctrl><Enter> - done" 3 wcolor)
  (Wtitle "<F1> - help" 5 wcolor)
  (Wpopup 64 13 wcolor wcolor 10)       ;two more nested windows to make the
  (Wpopup 60 11 wcolor wcolor 18)       ;raised border effect
  (getdata fields tcolor dcolor)        ;process the form
  (WcloseAll)
)

;;;*----- Process the Data Entry form

(DeFun GETDATA(template pcolor dcolor / done fld i key rslt &symbol)
  (SetQ cnt (Length template) i 0)
  (Repeat cnt                          ;;; display form
    (SetQ fld   (Nth i template)
          i     (1+ i))
    (Wprts (Car fld) (Cadr fld) (Caddr fld) pcolor)
    (Wprts (Nth 3 fld) (Nth 4 fld) (Eval(Nth 5 fld)) dcolor)
  )
  (SetQ i 0 done nil)
  (While (Not done) (Progn
    (SetQ fld     (Nth i template)
          rslt    (getitem fld dcolor)
          &symbol (Nth 5 fld)               ;;; pointer to variable
          key     (Cadr rslt)
    )
    (Set  &symbol (Car rslt))               ;;; assign string to variable
  (Cond
    ((= key Esc_Key)   (SetQ done T))       ;;; escape pressed, quit
    ((= key C_Ent_Key) (SetQ done T))       ;;; <Ctrl>Enter
    ((= key Up_Key)    (SetQ i (up1 cnt)))  ;;; up arrow
    (T (SetQ i (Rem (1+ i) cnt))))
  ))
)

;;;*----- Move back one field

(DeFun UP1(cnt)                         ;;; guarantees no negative values
  (Rem (1- (+ cnt i)) cnt)
)

;;;*----- Fetch the String

(DeFun GETITEM(fld dcolor)
    (WgotoXY (Nth 3 fld) (Nth 4 fld))
    (StrGet (Eval (Nth 5 fld)) (Nth 6 fld) 0 "" dcolor)
)

;;;*----- Screen Demo

(DeFun SCRDEMO()
  (Scr_Fill 0 0 vcols vrows 178 (| blue lgrey_bg))
  (Wmsg (StrCat "(Scr_Fill) fills areas of the screen,"
                "\n(Save_Screen) takes a snapshot of the screen,"
                "\n(Restore_Screen) restores the snapshot,"
                "\nand (CLS) clears the screen. We will cycle"
                "\n100 screen redraws now.") 1 (| white brown_bg))
  (Repeat 100
    (Restore_Screen)
    (Save_Screen)
    (CLS 7)
  )
  (Repeat 1000
    (Prts (Fix(Rem (Rand) (- vcols 15))) (Fix(Rem (Rand) (1- vrows))) "Hello World" (Fix(Rem (Rand) 90)))
  )
  (Wmsg "1000 Strings using (Prts)\n(The hiccup was from AutoLISP)" 1 (| white red_bg) (| black red_bg))
)

;;;*----- Directory Demo

(DeFun DDEMO( / dstat drive_no drive_str sector cluster
                drive_bytes free_bytes)
  (SetQ   dstat       (GetDiskFree 0)
          drive_no    (GetDisk)
          drive_str   (Chr(+ drive_no (1-(ASCII "A"))))
          sector      (Cadddr dstat)
          cluster     (* sector (Caddr dstat))
          drive_bytes (* cluster (Car dstat))
          free_bytes  (* cluster (Cadr dstat))
  )
  (Wopen 0 0 vcols vrows attr attr 4)
  (Wtitle "Directory" 1)
  (Wprintf "\n(GetDir) returns\n\t%s" (GetDir))
  (Wprintf "\n(GetDisk) returns\n\t%d or drive %s:" drive_no drive_str)
  (Wprintf "\n\n(GetDiskFree 0) returns:\n\t(%.0f %.0f %d %d)"
                drive_bytes free_bytes sector cluster)
  (Wprintf "\nThe current drive has the following stats:")
  (Wprintf "\n\tsector size: %d, cluster size: %d " sector cluster)
  (Wprintf "\n\tTotal bytes: %.0f, Free bytes: %.0f " drive_bytes free_bytes)
  (wpause)
  (Wopen 2 7 40 8 32 32 (| 1 8))
  (Wputs "\n (WgetFile) Gets a filename...")
  (Wputs "\n\n  Press <AltD> to Select\n  another Disk Drive")
  (Wmsg (WgetFile "*.*" 33 50 -1 (| white cyan_bg)))
  (Wclose)
)

;;;*----- File system Demo

(DeFun FDEMO( / filename fl)
  (SetQ filename    (FindFile "ACAD.PGP")
        fl          (SplitPath filename)
  )
  (Wopen 0 0 vcols vrows attr attr 3)
  (Wtitle "File Demo" 1)
  (Wprintf "\n\n(CopyFile \"C:\\CONFIG.SYS\" \".\")")
    (CopyFile "C:\\CONFIG.SYS" ".")
  (Wprintf "\n(MkDir \"&TEMP&\")")
    (MkDir "&TEMP&")
  (Wprintf "\n(MoveFile \"CONFIG.SYS\" \"&TEMP&\")")
    (MoveFile "CONFIG.SYS" "&TEMP&")
  (Wprintf "\n\n(FullPath \"&TEMP&\\CONFIG.SYS\") returns\n%s"
    (FullPath "&TEMP&\\CONFIG.SYS"))
  (Wprintf "\n\n(EraseFile \"&TEMP&\\CONFIG.SYS\")")
    (EraseFile "&TEMP&\\CONFIG.SYS")
  (Wprintf "\n(RmDir \"&TEMP&\")")
    (RmDir "&TEMP&")
  (wpause)
  (Wprintf "\n\n(SplitPath %s) returns\n\t" filename)
  (Wprintf "\nDrive:     \"%s\"" (Car fl))
  (Wprintf "\nDirectory: \"%s\"" (Cadr fl))
  (Wprintf "\nName:      \"%s\"" (Caddr fl))
  (Wprintf "\nExt:       \"%s\"" (Cadddr fl))
  (wpause)
  (Wclose)
)

(DeFun BLOOP()
  (Beep 1600 0.1)
  (Beep 800 0.1)
  (Beep 1600 0.1)
)

;;;*----- Play Charge

(DeFun CHARGE( / c f a c2)
  (SetQ c 262 f 349 a 440 c2 523)
  (Beep c 0.1)
  (Beep f 0.1)
  (Beep a 0.1)
  (Beep c2 0.2)
  (Beep a 0.1)
  (Beep c2 0.3)
)

;;;*----- Low Level Function Demo

(DeFun LDEMO()
  (Wopen 0 0 vcols vrows attr attr 3)
  (Wtitle "Low Level Functions" 1)
  (Wprintf "\nbeeping the speaker...") (Beep) (Wait 0.5)
  (Wprintf "\nand custom sounds...\n") (charge)
  (Wprintf "\nThe current time is:\n\t%s on %s\n" (StrTime) (StrDate))
  (Wprintf "\nPress any letter key to test (GetKey)...")  (SetQ key (GetKey))
  (Wprintf "(GetKey) returns \%d\) or \"%s\"" key (Chr(LoByte key)))
  (Wprintf "\n\n(Key_Ready) returns immediately with any waiting keystroke")
  (Wprintf "\n(KbHit) checks for a waiting keystroke")
  (Wprintf "\n\teg (While(Not(KbHit)) (long_loop_process))")
  (Wprintf "\n\n(Key_Stuff) inserts keystrokes in the keyboard buffer")
  (Wprintf "\n(Key_Clear) removes any pending keystrokes")
  (Wprintf "\n(Key_Stat) returns the status of control keys (Ctrl/Alt/Shift)")
  (wpause)
  (Wclose)
)

;;;*----- Math Demo

(DeFun MDEMO( / radians val1 val2)
  (SetQ val1 123.456 val2 0.5)
  (Wopen 0 0 vcols vrows attr attr 5)
  (Wtitle "Math Functions" 1)
  (SRand)
  (Wprintf "\n\nrandom numbers:\n") (Repeat 5 (Wprintf "%.0f " (Rand)))
  (Wprintf "\n\nDegrees to Radians conversion:\n\t%f degrees is %f radians"
          val1 (SetQ radians (DtR val1)))
  (Wprintf "\nRadians to Degrees conversion:\n\t%f radians is %f degrees"
          radians (RtD radians))
  (Wprintf "\nTrig Functions:")
  (Wprintf "\n\t(Tan  %f) returns %f" val2 (Tan radians))
  (Wprintf "\n\t(Acos %f) returns %f" val2 (Acos val2))
  (Wprintf "\n\t(Asin %f) returns %f" val2 (Asin val2))
  (Wprintf "\n\t(SinH %f) returns %f" val2 (SinH val2))
  (Wprintf "\nAnd...")
  (Wprintf "\n\t(Round %f 1) returns %f" radians (Round radians 1))
  (Wprintf "\n\t(Floor %f) returns %f" radians (Floor radians))
  (Wprintf "\n\t(Ceil  %f) returns %f" radians (Ceil radians))
  (Wpause)
  (Wclose)
)

;;;*----- String Demo

(DeFun STRDEMO( / str1 str2 str3 str4 ulist slist fmt real1 int1)
  (SetQ str1        "AAA;BBB;CCC"
        str2        "    Hello    World    "
        str3        "Th;is: is /a; test"
        str4        ";:/"
        ulist       '("ZZZ" "SSS" "AAA")
        slist       (Qsort ulist)
        fmt         "%8.3g %4Xh"
        pos         2
        real1       1.23456
        int1        4321
  )
  (Wopen 0 0 vcols vrows attr attr 3)
  (Wtitle "String / List Demo" 1)
  (Wprintf "\n\n(Sprintf \"%s\" %f %d) returns\n\t\"%s\"" fmt real1 int1 (Sprintf fmt real1 int1))
  (Wprintf "\n\n(StrDela \"%s\" \"%s\") returns\n\t\"%s\"" str3 str4 (StrDela str3 str4))
  (Wprintf "\n(StrTrimL \"%s\") returns\n\t\"%s\"" str2 (StrTrimL str2))
  (Wprintf "\n(StrTrimR \"%s\") returns\n\t\"%s\"" str2 (StrTrimR str2))
  (Wprintf "\n(StrTrim \"%s\") returns\n\t\"%s\"" str2 (StrTrim  str2))
  (Wprintf "\n(StrRev \"%s\") returns\n\t\"%s\"" str1 (StrRev str1))
  (Wprintf "\n(Field \"%s\" \";\" 2) returns\n\t\"%s\"" str1 (Field str1 ";" 2))
  (Wprintf "\n(Qsort \'\(\"%s\" \"%s\" \"%s\"\)) returns\n\t\(\"%s\" \"%s\" \"%s\"\)" (Car ulist)(Cadr ulist)(Caddr ulist)
        (Car slist)(Cadr slist)(Caddr slist))
  (Wprintf "\n(Insert \'\(\(\"%s\" \"%s\" \"%s\"\) %d \"%s\"\)) returns\n\t\(\"%s\" \"%s\" \"%s\" \"%s\"\)"
        (Car slist)(Cadr slist)(Caddr slist) pos str1
        (Car (setq slist (Insert slist pos str1)))(Cadr slist)(Caddr slist)(nth 3 slist))
  (Wpause)
  (Wclose)
)

;;;*----- Window Demo

(DeFun WDEMO( / str edit_help cols rows)
  (Scr_Fill 0 0 vcols vrows 178 (| lgrey blue_bg))
  (Set_Color (SetQ attr (| white blue_bg)))
  (Setq cols 4 rows 4)
  (Set_Cursor 32 0)                     ; cursor off
  (While (Or(< cols Vcols)(< rows Vrows)) (progn
    (Setq cols (Min(+ cols 4) Vcols) rows (Min(+ rows 2) Vrows))
    (Wpopup cols rows attr attr 1)
  ))
  (Wopen 2 2 35 6 (| black cyan_bg) (| yellow cyan_bg) hdouble_bd)
  (Wtitle "Hidden Window")
  (Wgotoxy 0 3) (Wputcen "This was is hidden")
  (Wopen 4 4 35 8 (| dgrey lgrey_bg) (| black lgrey_bg) no_bd)
  (Wtitle "Multiple Overlapping Windows")
  (Wputcen "with cursor positioning," (| blue lgrey_bg))
  (Wait 0.5)  (WgotoXY 0 2)
  (Wputcen "window write functions," (| red lgrey_bg))
  (WgotoXY 0 4)
  (Wputcen "full color, shadows and" (| black lgrey_bg))
  (WgotoXY 0 5)
  (Wputcen "cursor control" (| black lgrey_bg))
  (Wait 0.5) (Set_Cursor 12 13)                    ; cursor on
  (Wshadow)  (Waiting)  (Wclose)
  (Wopen 2 16 40 7 (| white brown_bg) (| yellow brown_bg) vdouble_bd)
  (Wtitle "[ Editor ]" 1)
  (Wtitle "[ F1 - Help ]" 3) (Wshadow)
  (Wputcen "Using (getstr) to get input:")
  (Wputs "\n\nEdit this string: " (| yellow brown_bg))
  (Set_Edit_Help
    '("(StrGet) Function" ""
      "This is user defined help for the line editor (StrGet)" ""
      "It is defined by the function \"(Set_Edit_Help)\" and"
      "the symbol \"edit_help\" in release 12 and higher."))
  (SetQ str (Car(StrGet "ELF demo" 20 0 "" (| black lgrey_bg))))
  (Wputs "\nYou entered: ") (Wputs str (| black red_bg))
  (SetQ str "Mountain Software"
        str (WgetStr "using (WgetStr)" str 40 (| yellow black_bg)(| white black_bg)))
  (Wmsg (StrCat "Wmsg Displays a Message\nYou entered " str) 1 (| black lgrey_bg))
  (WcloseAll)
  (Wmsg (StrCat "The Window System contains\nfunctions for text window"
                "\nhandling, menus, screen painting\n"
                "with complete color and cursor\ncontrol") 1 (| white red_bg))
  (Set_Color (SetQ attr (| lgrey blue_bg)))
  (Wcloseall)
)

(DeFun WAITING()
  (Wtitle "Delaying 2 second..." 4 (| blink white red_bg))
  (Wait 2.0)
)

(DeFun LOOK_DEMO( / key dat i col)
  (Wopen 0 0 vcols vrows 48 48 0)
  (Set_Color 23)                        ;;; look will use this color
  (Wopen 2 -1 19 5 23 23 5)
  (Wputs "\n Select a file")
  (While (SetQ fn (WgetFile))
    (If fn (Progn
      (Save_Screen)
      (Look fn)
      (Restore_Screen)
    ))
  )
  (Wclose)
  (Wclose)
)

(DeFun WPAUSE()
  (Wputs "\n\npress any key...")
  (GetKey)
)

(DeFun C:TIMETEST( / a)
  (timeit '(List () (Line '(0.0 0.0) '(1.0 1.0))) 1000)
  (Pause)
  (timeit '(List () (Command "line" "0,0" "1,1" "")) 1000)
)

(DeFun TIMEIT(func times / start stop)
  (printf "\n\nTiming %d executions of function " times) (Princ func)
  (printf "\nClock start at %.2f (%s)" (SetQ start (Clock)) (StrTime))
  (Repeat times (Eval func))
  (SetQ stop (Clock))
  (printf "\n%s" func)
  (printf " finished...\nClock stop at %.2f, elapsed time is %.2f seconds"
     stop (SetQ seconds (Abs(- stop start))))
  (printf "\nor %f seconds per iteration" (/ seconds times))
  (Princ)
)

(DeFun C:TOP10( / t10 ans key)
  (SetQ t10 '("ZOOM W" "ZOOM P" "LINE" "ARC" "CIRCLE"
              "ERASE" "PLINE" "PEDIT" "TRIM" "BREAK"))
  (SetQ ans (Wmenu t10 -1 -1 (| white red_bg)))
   (If(/= (Cadr ans) Esc_Key)
     (Key_Stuff (StrCat(Nth (Car ans) t10)"\n")))
)

(DeFun DemoError(s)
  (Beep)
  (Wmsg (Strcat "Demo ERROR\n" s) 1 (| white red_bg))
  (WcloseAll)
  (Cls 7)
  (SetQ *error* old_error old_error nil)
  (Princ)
)

(Princ "\nDEMO.LSP loaded, enter \"DEMO\" to run...")
(Princ)

;;;*----- End of Demo.Lsp

