/***
*arrays.c - array handling
*
*for Public Domain, 1995, Gregg Jennings.  All wrongs reserved.
*   P O Box 200, Falmouth, MA 02541-0200
*
*Purpose:
*   General array manipulation routines.
*
*Notice:
*   This source file may be freely used and distributed without restriction.
*******************************************************************************/

/*
   Versions

   0.1   14-Mar-1995    added arraysub()
   0.0   15-Dec-1994    These routines were written to replace some
                        specific ones that in FILES.C.  They have
                        are very general purpose though.

   Note: The use of _huge arrays are for  DISKED's clusters[] array.
         I have not used these for other arrays yet but the huge-ness
         should not be a cause of any problems nor any performance
         decrease.
*/

#include <stdlib.h>           /* for itoa() */

#include "console.h"          /* for output() (you can use putch()) */
#include "limits.h"           /* my additions to <limits.h> */


/* NO globals referenced here */

/* NO globals defined here */


/***
*arraymax   -  get max number of array
*
****/

extern int arraymax(int _huge *array, size_t max, int n)
{
size_t i;

   for (i = 0; i < max; i++)
      if (array[i] > n)
         n = array[i];
   return n;
}

/***
*arraymin   -  get min number of array
*
****/

extern int arraymin(int _huge *array, size_t max, int n)
{
size_t i;

   for (i = 0; i < max; i++)
      if (array[i] < n)
         n = array[i];
   return n;
}

/***
*arraycnt   -  count occurences
*
****/

extern int arraycnt(int _huge *array, size_t max, int n)
{
size_t i;
int c;

   for (c = i = 0; i < max; i++)
      if (array[i] == n)
         ++c;
   return c;
}

/***
*arrayfirst -  return offset of first occurance of n
*              max if not found
****/

extern size_t arrayfirst(int _huge *array, size_t max, int n)
{
size_t i;

   for (i = 0; i < max; i++)
      if (array[i] == n)
         break;
   return i;
}

/***
*arraylast  -  return offset of last occurance of n
*              0 if not found
****/

extern size_t arraylast(int _huge *array, size_t max, int n)
{
size_t i;

   for (i = max-1; i > 0; i--)
      if (array[i] == n)
         break;
   return i;
}

/***
*arrayfind  -  return offset of occurance of n
*              max if not found
****/

extern size_t arrayfind(int _huge *array, size_t max, int n)
{
size_t i;

   for (i = 0; i < max; i++)
      if (array[i] == n)
         break;
   return i;
}

/***
*arraytrav  -  find next/prev occurance of number at offset n
*
****/

extern size_t arraytrav(int _huge *array, size_t max, size_t n, int dir)
{
int num;

   num = array[n];

   for (n += dir; ; n += dir)
   {
      if (dir == -1 && n == 0xFFFF)
         n = max-1;
      if (dir == 1 && n > max)
         n = 0;
      if (array[n] == num)
         return n;
   }
}

/***
*arraysub   -  retreive array offsets of number in array
*
****/

extern size_t arraysub(int _huge *array, int *sub, size_t max, int n)
{
size_t i,j;

   for (i = j = 0; i < max; i++)
   {
      if (array[i] == n)
         sub[j++] = i;
   }
   return j;
}

/***
*arraymap -
*
*     2 3 4 5 6 7 8 9 10 11 12 13 14 15
*     2 2 2 2 2 0 0 3 3  0  0  0  0  4
*
*     7 8 11-14 ...
*
****/

extern int arraymap(int _huge *array, size_t max, int n, int base)
{
size_t i;
int c = 0;
int count = 0;
int indash = 0;
char buf[INT_MAX_DIG+1];

   for (i = 2; i < max; i++)
   {
      if (array[i] != n)
      {
         c = 1;
         if (count == 2)
            output(' ');
         if (count > 1)
         {
            print(itoa(i-1,buf,base));
            output(' ');
         }
         if (count == 1)
            output(' ');
         count = indash = 0;
      }
      else
      {
         if (count == 1)
            count++;
         else if (count == 0)
         {
            print(itoa(i,buf,base));
            count++;
         }
         else if (!indash)
         {
            output('-');
            indash = 1;
            count++;
         }
      }
   }
   return c;
}

/* special for listing DISKED's tagged[] array */

extern void plist(long *array, size_t max, int base)
{
size_t i;
char buf[LONG_MAX_DIG+1];

   for (i = 1; i < max; i++)
      print("%d:%s ",i,ltoa(array[i],buf,base));
}

#if 0

#include <stdio.h>
#include <stdlib.h>

#define output putchar
#define print printf

extern void arraymap(int *array, int max, int n, int base);

void main(void)
{
int array[] = { 0, 0, 1, 1, 0, 0, 2, 2, 2, 2, 0, 2, 0, -1 };

   arraymap(array,sizeof(array)/sizeof(int),2,10);
}

#endif
