
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                         *
 * Module Name : BCMLIB.C                                                  *
 *                                                                         *
 * Description : C language software interface for the BCMDRV Driver.      *
 *                                                                         *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * BICOM (R) BCMLIB.C  Version 1.02                                        *
 * Copyright (c) BICOM 1990, 1991, 1992     All rights reserved.           *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                          REVISION HISTORY                               *
 *-------------------------------------------------------------------------*
 *  Date    New Rev.                    Changes                            *
 *                                                                         *
 * 9/16/92  1.01     Revised Bcm56_MonitorChan to add Gain parameter.      *
 *                                                                         *
 * 9/24/92  1.02     Added Bcm38_SetChanParam & Bcm39_Wink functions       *
 *                                                                         *
 * 10/1/92  1.03     Added Bcm61_DirectPort                                *
 *                                                                         *
 * 11/03/92 1.04     Added Bcm62_MakeBridge,Bcm65_DelBridgeChan            *
 *                   Bcm64_AddBridgeChan,Bcm63_BreakBridge                 *
 *                                                                         *
 * 11/24/92 1.05     Bcm66_GroupMonitor                                    *
 *                                                                         *
 * 2/16/93  1.06     Modified mode byte in Bcm61_DirectPort function       *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include <stdio.h>              
#include <string.h>             
#include <stdlib.h>
#include <dos.h>

#include "bcmlib.h"
                

    
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm_1StartSystem(HW_Int,SysMode,BufSeg,BufSize,pNumChan)
 * DESCRIPTION : Start the BICOM boards and device driver.
 *  PARAMETERS : HW_Int   : BICOM hardware interrupt level (2-7)
 *               SysMode  : system mode.
 *                            = 0 (SM_POLL)  : polling mode 
 *                            = 1 (SM_EVENT) : event driven mode 
 *               BufSeg   : buffer start address segment          
 *               BufSize  : buffer size in paragraphs               
 *                          16 bytes per paragraph
 *               pNumChan : pointer to number of channels in system
 *     RETURNS : Return code.
 *    CAUTIONS : Unpredictable results will occur if driver was not
 *               installed.  Use the Bcm_DrvInstalled() function to check
 *               whether the BICOM driver was installed or not.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm1_StartSystem(unsigned int HW_Int,
                     unsigned int SysMode,
                     unsigned int BufSeg,
                     unsigned int BufSize,
                     unsigned int *pNumChan)
{
   int ReturnCode;                     

   RegAL = (unsigned char) HW_Int;
   RegBX = (SysMode&1);
   RegCX = BufSize;
   RegDX = BufSeg;

   ReturnCode = BcmDrvCall(F_SSTART,pRegAL,pRegBX,pRegCX,pRegDX);
   
   *pNumChan = *pRegAL;   /* number of channels */

   /* set status flag if successful */
   if (!ReturnCode) 
      BCM_SystemStatus = 1;
   
   return(ReturnCode);
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm3_StopSystem()
 * DESCRIPTION : Stop all operations and deinstall hardware interrupt.
 *  PARAMETERS : None
 *     RETURNS : Return code.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm3_StopSystem()
{
   RegAL = 0;
   RegBX = 0;
   RegCX = 0;

   ReturnCode = BcmDrvCall(F_SSTOP,pRegAL,pRegBX,pRegCX,pRegDX);

   /* clear status flag if successful */
   if (!ReturnCode)
      BCM_SystemStatus = 0;

   return(ReturnCode);
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm4_StopChannel(ChanNum)
 * DESCRIPTION : Stop operations for the specified channel and generate a
 *               T_STOP event. 
 *  PARAMETERS : ChanNum : channel number
 *     RETURNS : Return code.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm4_StopChannel(unsigned int ChanNum)
{
   RegAL = (unsigned char)ChanNum;
   RegBX = 0;
   RegCX = 0;

   return(BcmDrvCall(F_CHSTOP,pRegAL,pRegBX,pRegCX,pRegDX));
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm5_GetChanStatus(ChanNum, pCSB)
 * DESCRIPTION : Copy channel status into the application's local CSB.
 *  PARAMETERS : ChanNum : channel number
 *               pCSB    : pointer to CSB
 *     RETURNS : Return code.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm5_GetChanStatus(unsigned int ChanNum,
                       CSB *pCSB)
{
   RegAL = (unsigned char)ChanNum;
   RegBX = 0;
   RegCX = 0;

   return(BcmDrvCallP(F_CHSTAT,pRegAL,pRegBX,pRegCX,(unsigned char *)pCSB));
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm6_SetHook(ChanNum, State)
 * DESCRIPTION : Set the specified channel onhook or offhook.
 *  PARAMETERS : ChanNum : channel number
 *               State = 0 (H_OFFH) : offhook
 *                       1 (H_ONH)  : onhook
 *     RETURNS : Return code.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm6_SetHook(unsigned int ChanNum,
                 unsigned int State)
{
   RegAL = (unsigned char)ChanNum;
   RegBX = State;
   RegCX = 0;

   return(BcmDrvCall(F_SETH,pRegAL,pRegBX,pRegCX,pRegDX));
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm7_SetEGMask(ChanNum, EGMask, Rings)
 * DESCRIPTION : Set the event generate mask for the specified channel.
 *  PARAMETERS : ChanNum : channel number
 *               EGMask  : bit mask of EG
 *               Rings   : number of rings to receive to generate event
 *     RETURNS : Return code.
 *    CAUTIONS : Only this function could modify EG mask in CSB.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm7_SetEGMask(unsigned int ChanNum,
                   unsigned int EGMask,
                   unsigned int Rings)
{
   RegAL = (unsigned char)ChanNum;
   RegBX = EGMask;
   RegCX = Rings;

   return(BcmDrvCall(F_SETCST,pRegAL,pRegBX,pRegCX,pRegDX));
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm8_ClearDTMFQ(ChanNum)
 * DESCRIPTION : Clear the DTMF buffer.
 *  PARAMETERS : ChanNum : channel number
 *     RETURNS : Return code.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm8_ClearDTMFQ(unsigned int ChanNum)
{
   RegAL = (unsigned char)ChanNum;
   RegBX = 0;
   RegCX = 0;

   return(BcmDrvCall(F_CLRDT,pRegAL,pRegBX,pRegCX,pRegDX));
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm9_GetDTMFDigit(ChanNum, pDTMFdigit)
 * DESCRIPTION : Read the next ONE DTMF digit from the DTMF buffer.
 *               The digits are received on the FIFO basis.
 *  PARAMETERS : ChanNum    : channel Number
 *               pDTMFdigit : pointer to an ascii digit
 *     RETURNS : Return code.
 *    CAUTIONS : Make sure that there is a DTMF digit in the buffer, then
 *               use this function to get it.   
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm9_GetDTMFDigit(unsigned int ChanNum,
                       unsigned char *pDTMFdigit)
{
   RegAL = (unsigned char)ChanNum;

   ReturnCode = BcmDrvCall(F_GETDT,pRegAL,pRegBX,pRegCX,pRegDX);

   if (!ReturnCode)
      *pDTMFdigit = (unsigned char)(RegBX&0xff);
   else
      *pDTMFdigit = 0;   /* no digit in the buffer */

   return(ReturnCode);      
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm12_RecordFile(ChanNum, pTCB, Mode)
 * DESCRIPTION : Record audio data from the specified channel to a
 *               file handle until a termiating condition is received.
 *  PARAMETERS : ChanNum : Channel number
 *               pTCB    : pointer to the TCB
 *               Mode    : record mode bit mask 
 *                           = 0 (RD_NORM)    : normal record
 *                           = 1 (RD_SILCOMP) : silence compressed record
 *     RETURNS : Return code.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm12_RecordFile(unsigned int ChanNum,
                     TCB *pTCB,
                     unsigned int Mode)
{
   RegAL = (unsigned char)ChanNum;
   RegBX = (int)(Mode & 0xff);
   RegCX = 0;

   return(BcmDrvCallP(F_RECF,pRegAL,pRegBX,pRegCX,(unsigned char *)pTCB));
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm13_PlayFile(ChanNum, pTCB)
 * DESCRIPTION : Play audio data from the given file handle to the specified
 *               channel  until a terminating condition is received.
 *  PARAMETERS : ChanNum : channel number
 *               pTCB    : pointer to the TCB
 *     RETURNS : Return code.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm13_PlayFile(unsigned int ChanNum,
                   TCB *pTCB)
{
   RegAL = (unsigned char)ChanNum;
   RegBX = 0;
   RegCX = 0;

   return(BcmDrvCallP(F_PLAYF,pRegAL,pRegBX,pRegCX,(unsigned char *)pTCB));
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm14_DialOut(ChanNum, pDialStrg)
 * DESCRIPTION : This will dial the given string of digits out for the
 *               specified channel.
 *  PARAMETERS : ChanNum   : Channel number
 *               pDialStrg : pointer to ASCII string of digits
 *                           The digits must be ascii format (0-9,a-d,*,#,&)
 *                           with ',' indicating a pause.  Capital 'T'and  
 *                           'P' in the index zero position of the string 
 *                           indicate tone and pulse dialing respectivly.
 *                           Default is tone dialing. 
 *     RETURNS : Return code.
 *    CAUTIONS : The ASCII string is null terminated.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm14_DialOut(unsigned int ChanNum,
                  unsigned char *pDialStrg)
{
   RegAL = (unsigned char)ChanNum;
   RegBX = 0;
   RegCX = 0;

   return(BcmDrvCallP(F_DIAL,pRegAL,pRegBX,pRegCX,pDialStrg));
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm15_GetDTMFString(ChanNum, pTCB)
 * DESCRIPTION : Get an ASCII digit string from the specified channel buffer.
 *  PARAMETERS : ChanNum : channel number.
 *               pTCB    : pointer to TCB
 *     RETURNS : Return code.
 *    CAUTIONS : The ASCII string is null terminated.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm15_GetDTMFString(unsigned int ChanNum,
                        TCB *pTCB)
{
   RegAL = (unsigned char)ChanNum;
   RegBX = 0;
   RegCX = 0;

   return(BcmDrvCallP(F_GETDTS,pRegAL,pRegBX,pRegCX,(unsigned char *)pTCB));
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm16_GetEvent(pChan, pCode, pData)
 * DESCRIPTION : Get next event.
 *  PARAMETERS : pChan : pointer to the channel number
 *               pCode : pointer to the event code
 *               pData : pointer to the event data
 *     RETURNS : -1 if an event is available
 *                0 if no event is available
 *               >0 if error (see return code)
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm16_GetEvent(unsigned int *pChan,
                   unsigned int *pCode,
                   unsigned int *pData)
{
   ReturnCode = BcmDrvCall(F_GETEVT,pRegAL,pRegBX,pRegCX,pRegDX);
   *pChan = *pRegAL;
   *pCode = *pRegBX;
   *pData = *pRegCX;

   return((*pChan) ?  -1 : ReturnCode);
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm18_Scheduler()
 * DESCRIPTION : Give BICOM driver a time slice to do the housekeeping
 *               and necessary DOS calls.   
 *  PARAMETERS : none.
 *     RETURNS : Return code.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm18_Scheduler()
{
   RegAL = 0;
   RegBX = 0;
   RegCX = 0;

   ReturnCode = BcmDrvCall(F_SCHED,pRegAL,pRegBX,pRegCX,pRegDX);
   return(ReturnCode);
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*    FUNCTION : Bcm19_PlayIndexedFile(ChanNum, Mode, pTCB)
* DESCRIPTION : This will play data from the given file until a 
*               terminating condition is received. Indexed play files
*               and multiple indexed files are supported with this
*               function, but remember that the table supporting
*               the indexing must not be changed until command terminates.
*  PARAMETERS : ChanNum : channel number
*               Mode    : record mode bit mask 
*                           = 0 (PB_NORM) : normal playfile
*                           = 1 (PB_IDX)  : indexed play
*                           = 8 (PB_FILE) : multiple indexed play files
*               pTCB    : pointer to the TCB
*     RETURNS : Return code.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm19_PlayIndexedFile(unsigned int ChanNum,
                          unsigned int Mode,
                          TCB *pTCB)
{
   RegAL = (unsigned char)ChanNum;
   RegBX = Mode;
   RegCX = 0;

   return(BcmDrvCallP(F_XPLAYF,pRegAL,pRegBX,pRegCX,(unsigned char *)pTCB));
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm27_GetCAResults(ChanNum, pCAR)
 * DESCRIPTION : This function gets the Call Analysis Results information
 *               and copies it to user application's local CAR.     
 *  PARAMETERS : ChanNum : channel number
 *               pCAR    : pointer to Call Analysis Block (CAR)
 *     RETURNS : Return code.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm27_GetCAResults(unsigned int ChanNum,
                       CAR *pCAR)
{
   RegAL = (unsigned char)ChanNum;
   RegBX = 0;
   RegCX = 0;

   return(BcmDrvCallP(F_GETCAR,pRegAL,pRegBX,pRegCX,(unsigned char *)pCAR));
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm28_SetSystemParams(pSCB)
 * DESCRIPTION : Set the parameters in SCB.
 *  PARAMETERS : pSCB : pointer to the SCB
 *     RETURNS : Return code.
 *    CAUTIONS : Call this function before Bcm1_StartSystem().
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm28_SetSystemParams(SCB *pSCB)
{
   RegAL = 0;
   RegBX = 0;
   RegCX = 0;

   return(BcmDrvCallP(F_SXPARM,pRegAL,pRegBX,pRegCX,(unsigned char *)pSCB));
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *        NAME : Bcm29_SetChanParams(ChanNum, pCPB)
 * DESCRIPTION : Set channel parameters for the specified channel.
 *  PARAMETERS : ChanNum : channel number
 *               pCPB    : pointer to the CPB
 *     RETURNS : Return code.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm29_SetChanParams(unsigned int ChanNum,
                       CPB *pCPB)
{
   RegAL = (unsigned char)ChanNum;
   RegBX = 0;
   RegCX = 0;

   return(BcmDrvCallP(F_SCPARM,pRegAL,pRegBX,pRegCX,(unsigned char *)pCPB));
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm30_CallProgress(ChanNum, pString)
 * DESCRIPTION : This function will go offhook wait for dialtone. Upon
 *               receiving dialtone the ascii string will be dialed out
 *               to the given channel. During this operation all call
 *               progress events will be evavuated and returned.
 *  PARAMETERS : ChanNum : Channel number
 *               pString : Pointer to string of characters
 *     RETURNS : Return code.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm30_CallProgress(unsigned int ChanNum,
                       unsigned char *pString)
{
   RegAL = (unsigned char)ChanNum;
   RegBX = 0;
   RegCX = 0;

   return(BcmDrvCallP(F_CALL,pRegAL,pRegBX,pRegCX,pString));
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm31_SetDTMFControl(ChanNum, DigitMask, Method)
 * DESCRIPTION : This function control the detection of DTMF.
 *  PARAMETERS : ChanNum   : channel number
 *               DigitMask : digit control mask
 *                             bit 1-7 : reserved 
 *                             bit 0   
 *                               = 0          : disable DTMF digit detection
 *                               = 1 (D_DTMF) : enable DTMF digit detection 
 *               Method    : method of operation
 *                             = 0            : do not flush buffer
 *                             = 1 (DT_FLUSH) : flush digit buffer
 *     RETURNS : Return code
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm31_SetDTMFControl(unsigned int ChanNum,
                         unsigned int DigitMask,
                         unsigned int Method)
{
   RegAL = (unsigned char)ChanNum;
   RegBX = DigitMask;
   RegCX = Method;

   return(BcmDrvCall(F_SETD,pRegAL,pRegBX,pRegCX,pRegDX));
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm32_GetDrvVersion(pMajorVer, pMinorVer)
 * DESCRIPTION : Get BICOM driver version number.
 *  PARAMETERS : *pMajorVer = major version number
 *               *pMinorVer = minor version number
 *     RETURNS : Return code
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm32_GetDrvVersion(unsigned int *pMajorVer,
                        unsigned int *pMinorVer)
{
   RegBX = 0;
   RegCX = 0;

   /* check if BICOM driver was installed */
   if (Bcm_DrvInstalled((unsigned char)BCM_SoftwareInt) == 1)
   {
      RegAL = 0;
      ReturnCode = BcmDrvCall(F_GETVER,pRegAL,pRegBX,pRegCX,pRegDX);
      *pMajorVer = RegBX & 0xFF;
      *pMinorVer = (RegBX>>8) & 0xFF;
   }
   else
   {
      printf("\nBICOM driver not installed!\n");
      exit(1);
   }

   return(ReturnCode);
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm38_SetChanParam(ChanNum, Code, Data)
 * DESCRIPTION : Sets an individual channel parameter
 *       INPUT : ChanNum = channel number
 *               Code  = parameter to change
 *               Data  = the value for the new parameter
 *      OUTPUT : none
 *     RETURNS : Return code.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm38_SetChanParam(ChanNum, Code, Data)
   unsigned int ChanNum;
   unsigned int Code;
   unsigned int Data;
   {
      RegAL = (unsigned char)ChanNum;
      RegBX = Code;
      RegCX = Data;

      return(BcmDrvCall(F_SICPARM,pRegAL,pRegBX,pRegCX,pRegDX));
   }

/********************************************************************
 *        NAME : Bcm39_Wink(ChanNum)
 * DESCRIPTION : Wink (hook-flash) the channel
 *       INPUT : ChanNum = channel number
 *      OUTPUT : none
 *     RETURNS : Return code.
 ********************************************************************/
int Bcm39_Wink(ChanNum)
   unsigned int ChanNum;
   {
      RegAL = (unsigned char)ChanNum;
      RegBX = 0;
      RegCX = 0;
      RegDX = 0;

      return(BcmDrvCall(F_WINK,pRegAL,pRegBX,pRegCX,pRegDX));
   }

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm70_GetEventBlock(pEvent)
 * DESCRIPTION : This function gets an event block from event queue and
 *               copies it to user application loical sturcture.
 *  PARAMETERS : pEvent : pointer to users event block structure (EB)
 *     RETURNS : -1 if there is an event in queue
 *                0 if there is no event in queue
 *               >0 if error (see return code)
 *    CAUTIONS : This function is preferred than Bcm16_GetEvent().
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm70_GetEventBlock(EB *pEvent)
{
   int ReturnCode;

   /* call the scheduler */
   Bcm18_Scheduler();

   RegAL = X_GETQ;
   RegBX = Q_STDEVT;
   RegCX = NULL;

   ReturnCode = BcmDrvCallP(F_XSRV,pRegAL,pRegBX,pRegCX,(unsigned char *)pEvent);
   
   if (ReturnCode == D_QEMPTY)  return (0);
   
   return((ReturnCode) ? ReturnCode:-1);
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm70_PutEventInQ(ChanNum, Code, Data)
 * DESCRIPTION : Put a user defined event onto the system event queue.
 *  PARAMETERS : ChanNum : channel number
 *               Code    : event code
 *               Data    : event data
 *     RETURNS : 0 if the event was placed onto the event queue
 *    CAUTIONS : Please use event codes which are DIFFERENT from BICOM
 *               system defined event codes.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm70_PutEventInQ(unsigned int ChanNum,
                      unsigned int Code,
                      unsigned int Data)
{
   EB UserEvent;

   RegAL = X_USREVT;
   RegBX = NULL;
   RegCX = NULL;

   /* build the event block data structure */
   UserEvent.DeviceType = U_DEV1;
   UserEvent.Chan = ChanNum;
   UserEvent.EventCode = Code;
   UserEvent.EventData = Data;

   return(BcmDrvCallP(F_XSRV,pRegAL,pRegBX,pRegCX,(unsigned char *)&UserEvent));
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm_ClearTCB(pTCB)
 * DESCRIPTION : Initialize the TCB  
 *  PARAMETERS : pTCB : pointer to the TCB
 *     RETURNS : void
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
void Bcm_ClearTCB(TCB *pTCB)
{
   memset(pTCB,0,sizeof(TCB));
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm_ClearSCB(pSCB)
 * DESCRIPTION : Initialize the SCB  
 *  PARAMETERS : pSCB : pointer to the SCB
 *     RETURNS : void
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
void Bcm_ClearSCB(SCB *pSCB)
{
    memset(pSCB, 0, sizeof(SCB));
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm_ClearCPB(pCPB)
 * DESCRIPTION : Initialize the CPB 
 *  PARAMETERS : pCPB : pointer to the CPB
 *     RETURNS : void
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
void Bcm_ClearCPB(CPB *pCPB)
{
   memset(pCPB,0,sizeof(CPB));
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*    FUNCTION : Bcm_DrvInstalled(SW_Int)
* DESCRIPTION : This function checks if the Bicom Driver was installed.
*  PARAMETERS : SW_Int : software interrupt 
*     RETURNS : = 0 : BCMDRV driver was NOT installed
*               = 1 : BCMDRV driver was installed
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm_DrvInstalled(unsigned char SW_Int)
{
   unsigned long int far *IntVctrAddr;
   unsigned char far *DrvAddr;

   /* convert software interrupt number to a vector address */
   IntVctrAddr = (unsigned long int far *)(unsigned long int)(SW_Int*4);
   /* find driver address */
   DrvAddr = (unsigned char far *)(*IntVctrAddr);

   if (*(DrvAddr-2)=='D' && *(DrvAddr-1)=='L')
      return(1);

   return(0);
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm_ShowErrorMsg(ErrorCode)
 * DESCRIPTION : Show error message according to the error code
 *  PARAMETERS : ErrorCode : error code of last function
 *     RETURNS : Pointer to the error message
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
char *Bcm_ShowErrorMsg(unsigned int ErrorCode)
{
   switch (ErrorCode)
   {
      case E_SELF_TEST:
         return("Board failed self test");
      case E_NO_DTMF:
         return("DTMF buffer empty");
      case E_SYS_ACT:
         return("System already active");
      case E_SYS_INACT:
         return("System not active");
      case E_BOARD:
         return("BICOM hardware not respond");
      case E_INV_CMD:
         return("Invalid function code requested");
      case E_INV_INT:
         return("Interrupt level not available");
      case E_INV_SCB:
         return("SCB parameter error");
      case E_INV_CHAN:
         return("Invalid channel number");
      case E_TASK_ACT:
         return("Channel busy");
      case E_TASK_INACT:
         return("Channel idle");
      case E_INV_DATA:
         return("Invalid parameter");
      case E_INV_VER:
         return("Feature not supported in current version of DSP software");
      case E_NOT_IMP:
         return("Option not available");
      case E_NO_TERM:
         return("Terminating condition not specified");
      case E_NO_MEM:
         return("Insufficent buffer memory");
      case E_DOS_ERR:
         return("DOS error (DOS error returned in AL register)");
      case E_DISK:
         return("Disk read/write error");
      case E_INV_VOX:
         return("Bad TCB start value");
      case E_NO_EMS:
         return("EMM not installed or corrupted");
      case E_EMS_ERR:
         return("EMM reported error");
      case E_NO_BUF:
         return("No buffer allocated for EMS ram disk");
   }
   return("");
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm_GetSWIntVector()
 * DESCRIPTION : The function gets the BICOM driver software interrupt 
 *               vector.
 *  PARAMETERS : none
 *     RETURNS : = software interrupt vector
 *               = 0 if BICOM driver not installed
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
unsigned int Bcm_GetSWIntVector()
{
   unsigned char vector;
   int type;

   /* first check the default interrupt vector */
   if (Bcm_DrvInstalled((unsigned char)BCM_SoftwareInt) != 0)
      return(BCM_SoftwareInt);

   /* search for voice driver software interrupt vector */
   for (vector=0x40; vector<=0xfe; vector++)
   {
      type = Bcm_DrvInstalled(vector);
      if (type!=0)
         break;
   }

   /* return 0 if no vector was found  */
   if (type==0)
      return(0);

   /* save BICOM software interrupt vector */
   BCM_SoftwareInt = (unsigned int)vector;

   return((unsigned int)vector);
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm50_GetNextDTMF(pEventChan, pDTMF)
 * DESCRIPTION : Get next DTMF off DTMF event queue.
 *  PARAMETERS : pEventChan : pointer to the channel number
 *               pDTMF      : pointer to the DTMF digit 
 *     RETURNS : Return code.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm50_GetNextDTMF(unsigned int *pEventChan,
                      char *pDTMF)
{
   int ReturnCode;

   ReturnCode = BcmDrvCall(F_GETDTMF, pRegAL, pRegBX, pRegCX, pRegDX);

   *pEventChan = *pRegBX;
   *pDTMF      = (char)*pRegCX;

   return(ReturnCode);
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm51_SetPlaySpeed(ChanNum, Speed)
 * DESCRIPTION : Change the playback speed.
 *  PARAMETERS : ChanNum : channel number
 *               Speed   : playback level
 *                           = 0 : slowest
 *                           = 5 : normal
 *                           = 9 : fastest
 *     RETURNS : Return code.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm51_SetPlaySpeed(unsigned int ChanNum,
                       unsigned int Speed)
{
   int ReturnCode;

   RegAL = (unsigned char)ChanNum;
   RegBX = Speed;
   RegCX = 0;

   ReturnCode = BcmDrvCall(F_SPEED, pRegAL, pRegBX, pRegCX, pRegDX);

   return(ReturnCode);
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm52_SetPlayVolume(ChanNum, Volume)
 * DESCRIPTION : Change the playback volume.
 *  PARAMETERS : ChanNum : channel number
 *               Volume  : playback volume
 *                           = 0 : lowest
 *                           = 5 : normal
 *                           = 9 : loudest
 *     RETURNS : Return code.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm52_SetPlayVolume(unsigned int ChanNum,
                        unsigned int Volume)
{
   int ReturnCode;

   RegAL = (unsigned char)ChanNum;
   RegBX = Volume;
   RegCX = 0;

   ReturnCode = BcmDrvCall(F_LEVEL, pRegAL, pRegBX, pRegCX, pRegDX);

   return(ReturnCode);
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm53_GenerateBeep(ChanNum, pTDB)
 * DESCRIPTION : Generate Beep.
 *  PARAMETERS : ChanNum : channel number
 *               pTDB    : pointer to the Tone Descriptor Block
 *     RETURNS : Return code.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm53_GenerateBeep(unsigned int ChanNum,
                       TDB *pTDB)
{
   RegAL = (unsigned char)ChanNum;
   RegBX = 0;
   RegCX = 0;

   return(BcmDrvCallP(F_GEN_BEEP,pRegAL,pRegBX,pRegCX,(unsigned char *)pTDB));
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm54_GenerateDTMF(ChanNum, pTDB)
 * DESCRIPTION : Generate DTMF.
 *  PARAMETERS : ChanNum : channel number
 *               pTDB    : pointer to the Tone Descriptor Block
 *     RETURNS : Return code.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm54_GenerateDTMF(unsigned int ChanNum,
                       TDB *pTDB)
{
   RegAL = (unsigned char)ChanNum;
   RegBX = 0;
   RegCX = 0;

   return(BcmDrvCallP(F_GEN_DTMF,pRegAL,pRegBX,pRegCX,(unsigned char *) pTDB));
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm56_MonitorChan(ChanNum, ChanToMon, Gain)
 * DESCRIPTION : Monitor the activity on the given channel.
 *  PARAMETERS : ChanNum   : channel number 
 *               ChanToMon : channel number to monitor   
 *               Gain      : Monitor gain in dB. Range is 0 to 16dB
 *     RETURNS : Return code.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm56_MonitorChan(unsigned int ChanNum,
                      unsigned int ChanToMon,
                      unsigned int Gain)
{
   int ReturnCode;

   RegAL = (unsigned char)ChanNum;
   RegBX = ChanToMon;
   RegCX = Gain;

   ReturnCode = BcmDrvCall(F_MONITOR, pRegAL, pRegBX, pRegCX, pRegDX);

   return(ReturnCode);
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm57_StopMonitorChan(ChanNum)
 * DESCRIPTION : Go back to normal mode.
 *  PARAMETERS : ChanNum : channel number to stop monitoring
 *     RETURNS : Return code.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm57_StopMonitorChan(unsigned int ChanNum)
{
   int ReturnCode;

   RegAL = (unsigned char)ChanNum;
   RegBX = 0;
   RegCX = 0;

   ReturnCode = BcmDrvCall(F_STOP_MON, pRegAL, pRegBX, pRegCX, pRegDX);
 
   return(ReturnCode);
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm58_GenerateTELCO(ChanNum, pTDB)
 * DESCRIPTION : Generate Telco Tone.
 *  PARAMETERS : ChanNum : channel number
 *               pTDB    : pointer to the Tone Descriptor Block
 *     RETURNS : Return code.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm58_GenerateTELCO(unsigned int ChanNum,
                        TDB *pTDB)
{
   RegAL = (unsigned char)ChanNum;
   RegBX = 0;
   RegCX = 0;

   return(BcmDrvCallP(F_GEN_TELCO,pRegAL,pRegBX,pRegCX,(unsigned char *) pTDB));
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm60_StopTask(ChanNum)
 * DESCRIPTION : Terminate a multitasking function on a given channel.
 *  PARAMETERS : ChanNum : channel number
 *     RETURNS : Return code.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm60_StopTask(unsigned int ChanNum)
{
   int ReturnCode;

   RegAL = (unsigned char)ChanNum;
   RegBX = 0;
   RegCX = 0;

   ReturnCode=BcmDrvCall(F_STOP_TASK, pRegAL, pRegBX, pRegCX, pRegDX);
 
   return(ReturnCode);
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm61_DirectPort(ChanNum, Mode)
 * DESCRIPTION : Controls the connection of the Telephone Set 
 *               and the RCA audio port to a channel or CO line.
 *  PARAMETERS : ChanNum : channel number
 *               Mode    : connection mode
 * 
 * 
 *        Mode                    Telephone                    RCA
 *
 *         0            Connect to ChanNum CO line       Disconnected                  
 * 
 *         1            Supplied with battery and        Connected to ChanNum's
 *                      connected into ChanNum's         direct audio path.
 *                      direct audio path.    
 *
 *         2 (default)  Disconnected                    Disconnected
 *
 * 
 *     RETURNS : Return code.
 * 
 *     CAUTIONS: This switching feature is available only 
 *               on the BICOM 2LS board. 
 * 
 * 
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm61_DirectPort(unsigned int ChanNum, unsigned int Mode)
{
   int ReturnCode;

   RegAL = (unsigned char)ChanNum;
   RegBX = Mode;
   RegCX = 0;

   ReturnCode=BcmDrvCall(F_DIRECT_PORT, pRegAL, pRegBX, pRegCX, pRegDX);
 
   return(ReturnCode);
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm62_MakeBridge(pChanList, Gain)
 * DESCRIPTION : Set up a channel bridge of four or less channels
 *  PARAMETERS : pChanList : list of channels that are to be used in bridge
 *               Gain      : Monitor gain in dB. Range is 0 to 6dB
 *     RETURNS : Return code error if -1 else bridge handle
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm62_MakeBridge(unsigned int *pChanList, unsigned int Gain)
{
   int status;

   RegAL = 0;
   RegBX = Gain;
   RegCX = 0;

   status = BcmDrvCallP(F_MAKE_BRIDGE,pRegAL,pRegBX,pRegCX,
                     (unsigned char *) pChanList);
   if(status != 0)
   {
     printf("ERROR = %d\n",status);
     return(-1);        // Error
   }
   else
     return(*pRegBX);   // Bridge Handle
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm63_BreakBridge(Handle)
 * DESCRIPTION : Remove the bridge
 *  PARAMETERS : Handle    : Handle for the bridge that was returned from the
 *                           MakeBridge command
 *     RETURNS : Return code
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm63_BreakBridge(unsigned int Handle)
{
   RegAL = 0;
   RegBX = Handle;
   RegCX = 0;
   return(BcmDrvCall(F_BREAK_BRIDGE, pRegAL, pRegBX, pRegCX, pRegDX) );

}
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm64_AddBridgeChan(Handle, ChanNum)
 * DESCRIPTION : Add channel to the bridge
 *  PARAMETERS : Handle    : Handle for the bridge that was returned from the
 *                           MakeBridge command
 *             : ChanNum   : Channel number to be added to bridge
 *     RETURNS : Return code
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm64_AddBridgeChan(unsigned int Handle, unsigned int ChanNum)
{
   RegAL = (unsigned char)ChanNum;
   RegBX = Handle;
   RegCX = 0;
   return(BcmDrvCall(F_ADD_BRIDGE_CH, pRegAL, pRegBX, pRegCX, pRegDX) );
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm65_DelBridgeChan(Handle, ChanNum)
 * DESCRIPTION : Add channel to the bridge
 *  PARAMETERS : Handle    : Handle for the bridge that was returned from the
 *                           MakeBridge command
 *             : ChanNum   : Channel number to be added to bridge
 *     RETURNS : Return code
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm65_DelBridgeChan(unsigned int Handle, unsigned int ChanNum)
{
   RegAL = (unsigned char)ChanNum;
   RegBX = Handle;
   RegCX = 0;
   return(BcmDrvCall(F_DEL_BRIDGE_CH, pRegAL, pRegBX, pRegCX, pRegDX) );
}
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm66_GroupMonitor(pChanList, Gain)
 * DESCRIPTION : Set up a channel bridge of four or less channels
 *  PARAMETERS : ChanNum   : Channel number that monitors the list
 *             : pChanList : list of channels that are to be used in bridge
 *             : Gain      : Monitor gain in dB. Range is 0 to 6dB
 *     RETURNS : Return code error 
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm66_GroupMonitor(unsigned int ChanNum, unsigned int *pChanList, unsigned int Gain)
{
   int status;

   RegAL = (unsigned char)ChanNum;
   RegBX = Gain;
   RegCX = 0;

   status = BcmDrvCallP(F_GRP_MONITOR,pRegAL,pRegBX,pRegCX,
                     (unsigned char *) pChanList);
   return(status);
}

/********************************************************************
        NAME : LoadCpmTbl
 DESCRIPTION : 
       INPUT : Channel Number, and Tone 
     RETURNS : Return code.
********************************************************************/
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm67_LoadDspTable(ChanNum, TableType, pDspTbl)
 * DESCRIPTION : This command will download one of the following tables to
 *               the DSP, CPM, GTD, or Tone Qualifications
 *  PARAMETERS : ChanNum   : Channel number that monitors the list
 *             : TableType : Either LD_CPM_TBL, LD_GTD_TBL, LD_QUAL_TBL
 *             : pDspTbl   : pointer to the data
 *     RETURNS : Return code error 
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm67_LoadDspTable(ChanNum, TableType, pDspTbl)
unsigned int ChanNum;
unsigned int TableType;
char	       *pDspTbl;
{
      int rc;

      RegAL = (unsigned char)ChanNum;
      RegBX = TableType;
      RegCX = 0;
      return (BcmDrvCallP(F_LOAD_DSP_TBL,pRegAL,pRegBX,pRegCX,(unsigned char *) pDspTbl));
}
/********************************************************************
        NAME : GTD_State
 DESCRIPTION : Enable / Disable GTD
       INPUT : Channel Number, State 1=Enable 0 = Disable
     RETURNS : Return code.
********************************************************************/
// int GTD_State(Chan, Mode)
// unsigned int Chan;
// unsigned int Mode;
// {
//      int rc;

//      RegAL = (unsigned char)Chan;
//      RegBX = Mode;
//      RegCX = 0;
//      rc = BcmDrvCall(F_ENA_DIS_GTD, pRegAL,pRegBX,pRegCX,pRegDX);
//      return (rc);
// }

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : BcmDrvCall(RegAH, pRegAL, pRegBX, pRegCX, pRegDX)
 * DESCRIPTION : Call the BICOM driver without a pointer parameter.
 *  PARAMETERS : RegAH  : value required in the ah register.
 *               pRegAL : pointer to the value required in register al 
 *               pRegBX : pointer to the value required in register bx 
 *               pRegCX : pointer to the value required in register cx 
 *               pRegDX : pointer to the value required in register dx 
 *     RETURNS : value returned in ah register by BICOM driver
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int BcmDrvCall(unsigned char   RegAH,
               unsigned char *pRegAL,
               unsigned int  *pRegBX,
               unsigned int  *pRegCX,
               unsigned int  *pRegDX)
{
   ireg.h.ah =   RegAH;
   ireg.h.al = *pRegAL;
   ireg.x.bx = *pRegBX;
   ireg.x.cx = *pRegCX;
   ireg.x.dx = *pRegDX;

   int86x(BCM_SoftwareInt, &ireg, &oreg, &sr);

   *pRegAL = oreg.h.al;
   *pRegBX = oreg.x.bx;
   *pRegCX = oreg.x.cx;
   *pRegDX = oreg.x.dx;

   return((int)oreg.h.ah);
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : BcmDrvCallP( RegAH, pRegAL, pRegBX, pRegCX, Pointer)
 * DESCRIPTION : Call the BICOM driver with a pointer parameter.
 *  PARAMETERS : RegAH   : value required in the ah register.
 *               pRegAL  : pointer to the value required in register al 
 *               pRegBX  : pointer to the value required in register bx 
 *               pRegCX  : pointer to the value required in register cx 
 *               Pointer : pointer required by the BICOM driver
 *     RETURNS : value returned in ah register by BICOM driver.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int BcmDrvCallP(unsigned char   RegAH,
                unsigned char *pRegAL,
                unsigned int  *pRegBX,
                unsigned int  *pRegCX,
                unsigned char *Pointer)
{
   ireg.h.ah = RegAH;
   ireg.h.al = *pRegAL;
   ireg.x.bx = *pRegBX;
   ireg.x.cx = *pRegCX;

   if (Pointer)
   {
      /* get segment and offset */
      ireg.x.dx = FindOffset(Pointer);
      sr.ds     = FindSegment(Pointer);
   }
   else
   {
      /* pointer not used */
      ireg.x.dx = 0;
      sr.ds     = 0;
   }

   int86x(BCM_SoftwareInt, &ireg, &oreg, &sr);

   *pRegAL = oreg.h.al;
   *pRegBX = oreg.x.bx;
   *pRegCX = oreg.x.cx;

   return((int)oreg.h.ah);
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : FindOffset(Pointer)
 * DESCRIPTION : Get the offset value of the pointer.
 *  PARAMETERS : Pointer : a pointer.
 *     RETURNS : offset value of the pointer.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
unsigned short int FindOffset(unsigned char *Pointer)
{
   lp.lptr = Pointer;
   return((unsigned short int) lp.so.off);
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : FindSegment(Pointer)
 * DESCRIPTION : Get the segment value of the pointer.
 *  PARAMETERS : Pointer : a pointer.
 *     RETURNS : segment value of the pointer.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
unsigned short int FindSegment(unsigned char *Pointer)
{
   lp.lptr = Pointer;
   return((unsigned short int) lp.so.seg);
}


