/*
Spingu.cpp --- SPINGU.EXE utility.
--
Snmp Ping Utility (Unicode)
--  
Copyright 1995 - 1997 by MG-SOFT d.o.o., Slovenia
All rights reserved.
E-mail: info@mg-soft.si
Web URL: http://www.mg-soft.si/
--
This source file is for demonstrational purpose only.
Check LICENSE.TXT for licensing information.
--
Use <alt>+<f8> for your favorite code aligning.
*/
// #define _UNICODE
// #undef _UNICODE

// Unicode defines for version string
#if defined(_UNICODE)
	#define IS_UNICODE TRUE
	#if !defined UNICODE
		#define UNICODE
	#endif
#else
	#define IS_UNICODE FALSE
	#undef UNICODE
#endif

#include <tchar.h>
#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <conio.h>
#include <windows.h>
#include <winsock.h>
//
#include <mgtypes.h>
#include <winsnmp.h>
#include <wsnmpmib.h>

// WinSNMP messages
#define WM_SNMP_MSG        WM_USER + 1
#define WM_SNMP_MSG_STOP   WM_USER + 2
// Default community
// #define PUBLIC_COMMUNITY   _TEXT("public")
#define PUBLIC_COMMUNITY_ANSI   "public"
// Output string flags
#define OUTPUT_OID			0x01
#define OUTPUT_OID_NAME		0x02
#define OUTPUT_VALUE		0x04
#define OUTPUT_SEPARATOR	0x08
#define OUTPUT_VALUE_TYPE	0x10
#define OUTPUT_NEWLINE		0x20

// Function definitions
class SnmpQuery;
void setDefaultParameters(void);
void printParameters(void);
void getArgs(int argc, TCHAR *argv[]);
void usage(void);
SNMPAPI_STATUS convertString2smiValue(smiLPVALUE value, TCHAR *buffer);
WINAPI_STATUS allocateNotificationHandle(void);
WINAPI_STATUS freeNotificationHandle(void);
WINAPI_STATUS startupWinsock(void);
WINAPI_STATUS shutdownWinsock(void);
LONG APIENTRY notificationSnmpWndProc( HWND hWnd, UINT message, UINT wParam, LONG lParam);
BOOL ReadFileVersion(TCHAR *lptszFileName,
					 TCHAR *lptszFileVersionBuf, 
					 DWORD dwBufLen);

// Snmp return values to string conversion fot translate mode
static TCHAR *g_szTranslateMode[] = {
	_TEXT("Translated"),
		_TEXT("Untranslated V1"),
		_TEXT("Untranslated V2"),
		_TEXT("")
};

// Max SNMP errors 
#define MAX_SNMP_ERRORS SNMP_ERROR_INCONSISTENTNAME
static TCHAR *g_szSnmpErrors[] = {
	_TEXT("no error"),
		_TEXT("too big"),
		_TEXT("no such name"),
		_TEXT("bad value"),
		_TEXT("read-only"),
		_TEXT("generic error"),
		_TEXT("no access"),
		_TEXT("wrong type"),
		_TEXT("wrong length"),
		_TEXT("wrong encoding"),
		_TEXT("wrong value"),
		_TEXT("no creation"),
		_TEXT("inconsistent value"),
		_TEXT("resource unavailable"),
		_TEXT("commit failed"),
		_TEXT("undo failed"),
		_TEXT("authorization error"),
		_TEXT("not writable"),
		_TEXT("inconsistent name"),
		_TEXT("")
};

// Snmp return values to string conversion fot retransmit mode
static TCHAR *g_szRetransmitMode[] = {
	_TEXT("Not executing the retransmission policy."),
		_TEXT("Executing the retransmission policy."),
		_TEXT("")
};

// WinSNMP notificaiton window class and handle
static WNDCLASS  g_wc;
static HWND      g_hWinSnmpNotification = 0;

// parameters structure
struct parameters {
	TCHAR m_szDestinationIpAddress[20]; // destination IP 
	char m_szLocalHostName[100];   // local host name
	TCHAR m_tszLocalHostName[200];   // local host name
	char m_szLocalHostIp[20];      // local IP
	TCHAR m_tszLocalHostIp[40];      // local IP
	DWORD m_dwRetransmit;
	DWORD m_dwTimeout;
	BOOL m_bRetransmitMode;
	BOOL m_bBrowseWholeTree;
	BOOL m_bBrowseSystemInfo;
	BOOL m_bBrowseDepth;
	BOOL m_bOidToStringConversion;
	BOOL m_bCustomStartStopConditions;
	BOOL m_bPingVersion;					
	BOOL m_bResolveAgentIp;
	BOOL m_bPrintValueType;
	BOOL m_bPrintLineNumber;
	BOOL m_bPrintParameters;
	BOOL m_bSetSnmpValue;
	BOOL m_bPingUntilInterrupted;
	DWORD m_dwPingInterval;
	DWORD m_dwCurrentLineNumber;
	char *m_lpszCommunity;
	TCHAR *m_lptszCommunity;
	char *m_lpszStartOid;
	TCHAR *m_lptszStartOid;
	char *m_lpszStopOid;
	TCHAR *m_lptszStopOid;
	char *m_lpszValueToSet;
	TCHAR *m_lptszValueToSet;
	smiUINT32 m_dwValueType;
} g_sp;

// SnmpQuery class
class SnmpQuery {
private:
	// General WinSnmp data
	static smiUINT32 m_nMajorVersion;
	static smiUINT32 m_nMinorVersion;
	static smiUINT32 m_nLevel;
	static smiUINT32 m_nTranslateMode;
	static smiUINT32 m_nRetransmitMode;
	static BOOL m_bSnmpInitialized;
	// Initialize depended data
	BOOL m_bInitialized;
	BOOL m_bWinSnmpStarted;
	// Session depended data
	char m_szIPManager[80];
	TCHAR m_tszIPManager[160];
	char m_szIPAgent[80];
	TCHAR m_tszIPAgent[160];
	char m_szCommunity[80];
	TCHAR m_tszCommunity[160];
	smiOCTETS m_smiCommunity;
	// Session values
	smiOID m_stopOid;
	smiUINT32 m_stopOidVal[MAXOBJIDSIZE];
	smiOID m_oid;
	smiUINT32 m_oidVal[MAXOBJIDSIZE];
	smiVALUE m_value;
	DWORD m_dwRequestId;
	// Session handles
	HSNMP_SESSION m_hSnmpSession;
	HSNMP_ENTITY m_hAgentEntity;
	HSNMP_ENTITY m_hManagerEntity;
	HSNMP_CONTEXT m_hViewContext;
	HSNMP_VBL m_hVbl;
	HSNMP_PDU m_hPdu;
	// WinSnmp notification
	HWND m_hNotifyHandle;
	UINT m_nNotifyMessage;
public:
	// functions
	SnmpQuery(HANDLE notifyHandle, UINT notifyMessage);
	~SnmpQuery(void);
	WINAPI_STATUS Startup(void);
	WINAPI_STATUS Cleanup(void);
	void NotificationCallback(void);
	WINAPI_STATUS doQuery(TCHAR *qName);
	WINAPI_STATUS SendGetRequest(void);
	WINAPI_STATUS SendGetNextRequest(void);
	WINAPI_STATUS SendSetRequest(void);
	WINAPI_STATUS SnmpOutputValue(smiOID oid, smiVALUE value, BYTE flags);
};
class SnmpQuery *g_sq;

// global class variables initialization
smiUINT32 SnmpQuery::m_nMajorVersion = 0;
smiUINT32 SnmpQuery::m_nMinorVersion = 0;
smiUINT32 SnmpQuery::m_nLevel = 0;
smiUINT32 SnmpQuery::m_nTranslateMode = 0;
smiUINT32 SnmpQuery::m_nRetransmitMode = 0;
BOOL SnmpQuery::m_bSnmpInitialized = FALSE;

/*
main
--
Application entry function
*/
extern "C"
int __cdecl _tmain(int argc, TCHAR *argv[])
{
	int rv = 1;
	TCHAR tszFileVersion[101];
	TCHAR tszFileName[_MAX_PATH];
	// Initialize parameters struct
	memset(&g_sp, 0, sizeof(struct parameters));
	// Read file version from versioninfo resources
	if ((0 == GetModuleFileName(NULL, tszFileName, _MAX_PATH)) ||
		(!ReadFileVersion(&tszFileName[0], &tszFileVersion[0], 100))) {
	   _tcscpy(&tszFileVersion[0], _TEXT("<Not Available>"));
	}
	// Print application name and version
	// Printf version information
	_ftprintf(stdout, _TEXT("\n%s Snmp Ping Utility %sVersion %s")
		__TEXT("\nCopyright 1996, 1997 by MG-SOFT Corporation, Slovenia. All rights reserved."), 
		IS_UNICODE ? _TEXT("SPINGU") : _TEXT("SNMPPING"),
		IS_UNICODE ? _TEXT("(Unicode) ") : _TEXT(""), 
		tszFileVersion);
	// Check if unicode environment
	#if defined (_UNICODE)
		if (!IsWindowUnicode(GetDesktopWindow())) {
			_ftprintf(stdout, _TEXT("\n\nSystem does not support unicode."));
			_ftprintf(stdout, _TEXT("\nTerminating."));
			exit(EXIT_FAILURE);
		}
	#endif
	// set default parameters
	setDefaultParameters();
	// parse arguments and override default parameters
	getArgs(argc, argv);
	
	// Allocate the hidden notification window handle
	if (allocateNotificationHandle() == WINAPI_FAILURE) {
		_ftprintf(stdout, _TEXT("\n\nCan not allocate a window handle."));
		_ftprintf(stdout, _TEXT("\nTerminating."));
		rv = 0;
		goto doExit;
	}
	
	// Startup winsock (for local host resolving)
	startupWinsock();
	
	// Initialize class and WInSNMP
	g_sq = new SnmpQuery(g_hWinSnmpNotification, WM_SNMP_MSG);
	// Startup winsnmp
	g_sq->Startup();
	// print parameters (after snmp startup)
	if (g_sp.m_bPrintParameters)
		printParameters();
	
	// Check if destination IP is assigned
	if (g_sp.m_szDestinationIpAddress[0] == 0) {
		rv = 0;
		goto doExit;
	}
	
	// Get remote host name
	struct hostent *he;
	unsigned long ia;
	char ansiTs[255];
#if defined(_UNICODE)
	int ansiTsLen;
	ansiTsLen = WideCharToMultiByte(
		CP_ACP,	// code page 
		NULL,	// performance and mapping flags 
		&g_sp.m_szDestinationIpAddress[0],	// address of wide-character string 
		-1,	// number of characters in string 
		ansiTs,	// address of buffer for new string 
		255,	// size of buffer 
		NULL,	// address of default for unmappable characters  
		NULL	// address of flag set when default char. used 
	);
#else
	_tcscpy(ansiTs, &g_sp.m_szDestinationIpAddress[0]);
#endif
	ia = htonl(inet_addr(&ansiTs[0]));
	he = (g_sp.m_bResolveAgentIp) ? gethostbyaddr((char *) &ia, 4, PF_INET) : NULL;
	if (he)
		_ftprintf(stdout, _TEXT("\n\nPinging %s [%s] with SNMP protocol"), he->h_name, g_sp.m_szDestinationIpAddress);
	else
		_ftprintf(stdout, _TEXT("\n\nPinging %s with SNMP protocol"), g_sp.m_szDestinationIpAddress);
	
	_ftprintf(stdout, _TEXT("\nCommunity: %s"), g_sp.m_lpszCommunity);
	
	// Create first query
	if (g_sq->doQuery(NULL) == SNMPAPI_FAILURE) {
		// Something is wrong, terminate application
		rv = 0;
		goto terminate;
	}
	
	// Loop until STOP message from SnmpQuery class
	MSG msg;
	while (GetMessage(&msg, NULL, 0, 0)) {
		if (msg.message == WM_SNMP_MSG_STOP)
			break;
		TranslateMessage(&msg);    
		DispatchMessage(&msg);     
	}
	
	// Cleanup WinSNMP and destroy class 
terminate:
	_ftprintf(stdout, _TEXT("\n"));
	g_sq->Cleanup();
	delete g_sq;
	
	// Free the notification window handle
	freeNotificationHandle();
	
	//	while (!kbhit())
	//		;
doExit:
	return rv;
}

/*
printParameters
--
Prints parameters
*/
void printParameters(void)
{
	smiOID tOid = {0};
	smiVALUE tValue = {0};

	_ftprintf(stdout, _TEXT("\n\nCurrent parameters:\n"));
	_ftprintf(stdout, _TEXT("\nRemote agent IP address: %s"), g_sp.m_szDestinationIpAddress);
	_ftprintf(stdout, _TEXT("\nPrint WinSNMP module version: %s"),  g_sp.m_bPingVersion ? "TRUE" : "FALSE");
	_ftprintf(stdout, _TEXT("\nRetransmits: %d"), g_sp.m_dwRetransmit);
	_ftprintf(stdout, _TEXT("\nTimeout (ms): %d"), g_sp.m_dwTimeout);
	_ftprintf(stdout, _TEXT("\nRetransmit mode: %s"), (g_sp.m_bRetransmitMode == SNMPAPI_ON) ? "ON" : "OFF");
	_ftprintf(stdout, _TEXT("\nCommunity: %s"), g_sp.m_lpszCommunity);
	_ftprintf(stdout, _TEXT("\nBrowse whole tree: %s"), g_sp.m_bBrowseWholeTree ? "TRUE" : "FALSE");
	_ftprintf(stdout, _TEXT("\nBrowse system info: %s"), g_sp.m_bBrowseSystemInfo ? "TRUE" : "FALSE");
	_ftprintf(stdout, _TEXT("\nBrowse subtrees: %s"), g_sp.m_bBrowseDepth ? "TRUE" : "FALSE");
	_ftprintf(stdout, _TEXT("\nOid to string conversion: %s"), g_sp.m_bOidToStringConversion ? "TRUE" : "FALSE");
	if (g_sp.m_lpszStartOid) {
		_ftprintf(stdout, _TEXT("\nStart OID: %s"), g_sp.m_lptszStartOid);
		if (SnmpStrToOid(g_sp.m_lpszStartOid, &tOid) == SNMPAPI_SUCCESS) {
			_ftprintf(stdout, _TEXT(" ["));
			g_sq->SnmpOutputValue(tOid, tValue, OUTPUT_OID_NAME);
			SnmpFreeDescriptor(SNMP_SYNTAX_OID, (smiLPOPAQUE) &tOid);
			_ftprintf(stdout, _TEXT("]"));
		}
	}
	if (g_sp.m_lpszStopOid) {
		_ftprintf(stdout, _TEXT("\nStop OID:  %s"), g_sp.m_lptszStopOid);
		if (SnmpStrToOid(g_sp.m_lpszStopOid, &tOid) == SNMPAPI_SUCCESS) {
			_ftprintf(stdout, _TEXT(" ["));
			g_sq->SnmpOutputValue(tOid, tValue, OUTPUT_OID_NAME);
			SnmpFreeDescriptor(SNMP_SYNTAX_OID, (smiLPOPAQUE) &tOid);
			_ftprintf(stdout, _TEXT("]"));
		}
	}
	_ftprintf(stdout, _TEXT("\nSet SNMP value: %s"), g_sp.m_bSetSnmpValue ? "TRUE" : "FALSE");
	if (g_sp.m_lpszValueToSet) {
		tValue.syntax = g_sp.m_dwValueType;
		_ftprintf(stdout, _TEXT(", ")); 
		if (g_sp.m_lpszStartOid) {
			_ftprintf(stdout, _TEXT(" %s"), g_sp.m_lpszStartOid);
			if (SnmpStrToOid(g_sp.m_lpszStartOid, &tOid) == SNMPAPI_SUCCESS) {
				_ftprintf(stdout, _TEXT(" ["));
				g_sq->SnmpOutputValue(tOid, tValue, OUTPUT_OID_NAME);
				SnmpFreeDescriptor(SNMP_SYNTAX_OID, (smiLPOPAQUE) &tOid);
				_ftprintf(stdout, _TEXT("]"));
			}
		}
		_ftprintf(stdout, _TEXT(", ")); 
		g_sq->SnmpOutputValue(tOid, tValue, OUTPUT_VALUE_TYPE);
		_ftprintf(stdout, _TEXT("'%s'"), g_sp.m_lpszValueToSet);
	}
	_ftprintf(stdout, _TEXT("\nResolve SNMP agent IP address: %s"), g_sp.m_bResolveAgentIp ? "TRUE" : "FALSE");
	_ftprintf(stdout, _TEXT("\nPrint value types: %s"), g_sp.m_bPrintValueType ? "TRUE" : "FALSE");
	_ftprintf(stdout, _TEXT("\nPrint line numbers: %s"), g_sp.m_bPrintLineNumber ? "TRUE" : "FALSE");
	_ftprintf(stdout, _TEXT("\nPrint parameters: %s"), g_sp.m_bPrintParameters ? "TRUE" : "FALSE");
	_ftprintf(stdout, _TEXT("\nPing g_specified agent until interrupted: %s"), g_sp.m_bPingUntilInterrupted ? "TRUE" : "FALSE");
	_ftprintf(stdout, _TEXT("\n"));
}


/*
setDefaultParameters
--
  Sets default parameters which are efficient when
  no arguments are set
*/
void setDefaultParameters(void)
{
	g_sp.m_szDestinationIpAddress[0] = 0;
	g_sp.m_bPingVersion = FALSE;             // Write a SNMP version?
	g_sp.m_dwRetransmit = 1;                  // times
	g_sp.m_dwTimeout = 5000;                  // (5s) miliseconds
	g_sp.m_bRetransmitMode = SNMPAPI_ON;     // Do a retransmitting
	g_sp.m_lpszCommunity = PUBLIC_COMMUNITY_ANSI;
	g_sp.m_bBrowseWholeTree = FALSE;
	g_sp.m_bBrowseSystemInfo = FALSE;
	g_sp.m_bBrowseDepth = FALSE;
	g_sp.m_bOidToStringConversion = TRUE;
	g_sp.m_bCustomStartStopConditions = FALSE;
	g_sp.m_lpszStartOid = NULL;
	g_sp.m_lpszStopOid = NULL;
	g_sp.m_lptszStartOid = NULL;
	g_sp.m_lptszStopOid = NULL;
	g_sp.m_bResolveAgentIp = FALSE;
	g_sp.m_bPrintValueType = FALSE;
	g_sp.m_bPrintLineNumber = FALSE;
	g_sp.m_dwCurrentLineNumber = 1;
	g_sp.m_bPrintParameters = FALSE;
	g_sp.m_bSetSnmpValue = FALSE;
	g_sp.m_lpszValueToSet = NULL;
	g_sp.m_dwValueType = SNMP_SYNTAX_NULL;
	g_sp.m_bPingUntilInterrupted = FALSE;
	g_sp.m_dwPingInterval = 1000;
}

/*
getArgs
--
Parses arguments and overrides default parameters.
No space between switch and parameter is allowed.
*/
void getArgs(int argc, TCHAR *argv[])
{
	int i, j;
	int dip[4];
	
	if (argc <= 1)
		usage();
	while (--argc) {
		register TCHAR *p = *++argv;
		// first argument byte should be the '-' or '/'
		if ((p[0] == TCHAR('-')) || (p[0] == TCHAR('/'))) {
			// second argument byte should be the switch name
			switch (_totlower(p[1])) {
			case TCHAR('v'):
				g_sp.m_bPingVersion = TRUE;
				break;
			case TCHAR('c'):
				g_sp.m_lptszCommunity = &p[2];
				break;
			case TCHAR('f'):
				g_sp.m_bBrowseWholeTree = TRUE;
				break;
			case TCHAR('i'):
				g_sp.m_bBrowseSystemInfo = TRUE;
				break;
			case TCHAR('b'):
				g_sp.m_bBrowseDepth = TRUE;
				break;
			case TCHAR('s'):
				g_sp.m_lptszStartOid = &p[2];
				g_sp.m_bCustomStartStopConditions = TRUE;
				break;
			case TCHAR('e'):
				g_sp.m_lptszStopOid = &p[2];
				g_sp.m_bCustomStartStopConditions = TRUE;
				break;
			case TCHAR('m'):
				{
					switch (p[2]) {
					case TCHAR('o'): g_sp.m_dwValueType = SNMP_SYNTAX_OCTETS; break;
					case TCHAR('i'): g_sp.m_dwValueType = SNMP_SYNTAX_INT; break;
					case TCHAR('u'): g_sp.m_dwValueType = SNMP_SYNTAX_INT32; break;
					case TCHAR('c'): g_sp.m_dwValueType = SNMP_SYNTAX_CNTR32; break;
					case TCHAR('d'): g_sp.m_dwValueType = SNMP_SYNTAX_OID; break;
					case TCHAR('t'): g_sp.m_dwValueType = SNMP_SYNTAX_TIMETICKS; break;
					case TCHAR('n'): g_sp.m_dwValueType = SNMP_SYNTAX_NULL; break;
					case TCHAR('a'): g_sp.m_dwValueType = SNMP_SYNTAX_IPADDR; break;
					case TCHAR('b'): g_sp.m_dwValueType = SNMP_SYNTAX_BITS; break;
					case TCHAR('p'): g_sp.m_dwValueType = SNMP_SYNTAX_NSAPADDR; break;
					case TCHAR('g'): g_sp.m_dwValueType = SNMP_SYNTAX_GAUGE32; break;
					default:
						break;
					}
					g_sp.m_lptszValueToSet = &p[3];
					g_sp.m_bSetSnmpValue = TRUE;
					break;
				}
			case TCHAR('a'):
				g_sp.m_bResolveAgentIp = TRUE;
				break;
			case TCHAR('o'):
				g_sp.m_bOidToStringConversion = FALSE;
				break;
			case TCHAR('t'):
				g_sp.m_bPrintValueType = TRUE;
				break;
			case TCHAR('l'):
				g_sp.m_bPrintLineNumber = TRUE;
				break;
			case TCHAR('u'):
				g_sp.m_dwTimeout = _ttoi(&p[2]);
				break;
			case TCHAR('r'):
				g_sp.m_dwRetransmit = _ttoi(&p[2]);
				break;
			case TCHAR('p'):
				g_sp.m_bPrintParameters = TRUE;
				break;
			case TCHAR('d'):
				g_sp.m_bPingUntilInterrupted = TRUE;
				if (p[2] != TCHAR('\0')) 
					g_sp.m_dwPingInterval = _ttoi(&p[2]);
				break;
			default :
				// print usage if switch is unknown
				usage();
			}
		}
		else {
			// Get destination IP address
			for (i = 0; i < 4; i++)
				dip[i] = -1;
			j = _stscanf(p, _TEXT("%d.%d.%d.%d"), &dip[0], &dip[1], &dip[2], &dip[3]);
			if (j != 4)
				usage();
			for (i = 0; i < 4; i++) {
				if ((dip[i] < 0) || (dip[i] > 255))
					usage();
			}
			_stprintf(&g_sp.m_szDestinationIpAddress[0], _TEXT("%d.%d.%d.%d"), dip[0], dip[1], dip[2], dip[3]);
		}
	}
}

/*
usage
--
Prints command line usage
*/
void usage(void)
{
	static TCHAR *msg[] = {
		_TEXT("\n"),
			_TEXT("\nUsage: snmpping [-v] [-c(community)] [-r] [-u] [-f] [-b] [-i] [-o]"),
			_TEXT("\n                [-a] [-s(start OID)] [-e(end OID)] [-m(type)(value)]"),
			_TEXT("\n                [-l] [-t] [-p] [-d(interval)] ip.ip.ip.ip"),
			_TEXT("\n"),
			_TEXT("\nOptions (No space between switch and parameter):"),
			_TEXT("\n"),
			_TEXT("\n   -v              Display WinSNMP module version."),
			_TEXT("\n   -c community    Community [default = public]."),
			_TEXT("\n   -r              Number of retransmits [default = 1]."),
			_TEXT("\n   -u              Timeout per retransmit in miliseconds [default = 5000]."),
			_TEXT("\n   -f              Browse whole MIB tree."),
			_TEXT("\n   -b              Browse subtrees (-s must be g_specified)."),
			_TEXT("\n   -i              Browse system info."),
			_TEXT("\n   -o              Don't resolve object ID names."),
			_TEXT("\n   -a              Resolve remote SNMP agent IP address."),
			_TEXT("\n   -s start-Oid    Start OID [default = 1.3.6.1.2.1.1.3 (sysUpTime)]."),
			_TEXT("\n   -e end-Oid      End OID [default = 1.3.6.1.2.1.1.4 (sysContact)]."),
			_TEXT("\n   -m type value   Set value (-s must be g_specified)."),
			_TEXT("\n                   Types: o octets, i integer, d oid, u unsigned integer"),
			_TEXT("\n                          g gauge, t timeticks, c counter, a IP address, "),
			_TEXT("\n                          p NSAP address, b bits, n null."),
			_TEXT("\n   -l              Print line numbers."),
			_TEXT("\n   -t              Print value types."),
			_TEXT("\n   -p              Print parameters."),
			_TEXT("\n   -d interval-ms  Ping util interrupted [default interval = 1000 miliseconds]."),
			_TEXT("\n"),
			_TEXT("\nExamples: \n"),
			_TEXT("\nPing: snmpping ip.ip.ip.ip"),
			_TEXT("\nPing: snmpping -d ip.ip.ip.ip"),
			_TEXT("\nInfo: snmpping -i -t ip.ip.ip.ip"),
			_TEXT("\nWalk: snmpping -f -t ip.ip.ip.ip"),
			_TEXT("\nWalk: snmpping -f -s1.3.6.1 ip.ip.ip.ip"),
			_TEXT("\nWalk: snmpping -b -s1.3.6.1.2.1.1.1 ip.ip.ip.ip"),
			_TEXT("\nWalk: snmpping -s1.3.6.1 -e1.3.6.1.2.1.2 ip.ip.ip.ip"),
			_TEXT("\nSet:  snmpping -mo\"Contact person name\" -s1.3.6.1.2.1.1.4.0 ip.ip.ip.ip"),
			_TEXT("\nSet:  snmpping -mi12 -s1.3.6.1.2.1.1.7.0 ip.ip.ip.ip"),
			_TEXT("\nSet:  snmpping -md1.3.6.1.4.1.1315 -s1.3.6.1.2.1.1.2.0 ip.ip.ip.ip"),
			_TEXT("\nPerformance: snmpping -d0 -l ip.ip.ip.ip"),
			_TEXT("\n"),
			_TEXT("\nUse 'snmpping > file' or 'snmpping | more' when output is larger than screen."),
			_TEXT("\n"),
			_TEXT("")
	};
	register TCHAR **pmsg = msg;
	
	while (**pmsg)
		_ftprintf(stdout, *pmsg++);
	exit (0);
}

/*
allocateNotificationHandle
--
Registers windows class and and creates a
window for WinSNMP messages.
*/
WINAPI_STATUS allocateNotificationHandle(void)
{
	// Create new window class
	// Register the window class for main window.                                                           */
	g_wc.style = 0;                       // Class style.
	g_wc.lpfnWndProc = (WNDPROC) notificationSnmpWndProc; // Window procedure for this class.
	g_wc.cbClsExtra = 0;                  // No per-class extra data.
	g_wc.cbWndExtra = 0;                  // No per-window extra data.
	g_wc.hInstance = 0;//hInstance;           // Application that owns the class.
	g_wc.hIcon = LoadIcon(NULL, IDI_APPLICATION);
	g_wc.hCursor = LoadCursor(NULL, IDC_ARROW);
	g_wc.hbrBackground = GetStockObject(GRAY_BRUSH);
	g_wc.lpszMenuName =  _TEXT("");   // Name of menu resource in .RC file.
	g_wc.lpszClassName = _TEXT("MG-SOFT_SNMPPING_NOTIFICATION_CLASS"); // Name used in call to CreateWindow.
	
	if (RegisterClass(&g_wc) == 0) 
		return WINAPI_FAILURE;
	
	// Create a main window for this application instance.
	g_hWinSnmpNotification = CreateWindow (
		_TEXT("MG-SOFT_SNMPPING_NOTIFICATION_CLASS"),
		_TEXT("MG-SOFT WinSNMP Ping Notification Window C-1996"),
		WS_OVERLAPPEDWINDOW,            // Window style.
		CW_USEDEFAULT,                  // Default horizontal position.
		CW_USEDEFAULT,                  // Default vertical position.
		CW_USEDEFAULT,                  // Default width.
		CW_USEDEFAULT,                  // Default height.
		NULL,                           // Overlapped windows have no parent.
		NULL,                           // Use the window class menu.
		NULL,                           // This instance owns this window.
		NULL                            // Pointer not needed.
		);
	return (g_hWinSnmpNotification == NULL) ? WINAPI_FAILURE : WINAPI_SUCCESS;
}

/*
freeNotificationHandle
--
Unregisters windows class and and destroyes a
window for WinSNMP messages.
*/
WINAPI_STATUS freeNotificationHandle(void)
{
	// Destroy hidden window
	DestroyWindow(g_hWinSnmpNotification);
	// Unregister class
	UnregisterClass((LPCTSTR) &g_wc.lpszClassName, NULL);
	return WINAPI_SUCCESS;
}

/*
startupWinsock
--
Starts winsock (needed for name resolving)
*/
WINAPI_STATUS startupWinsock(void)
{
	WORD wVersionRequested;
	WSAData  wsaData;
	int err;
	struct hostent *he;
	in_addr *(*heLocalIP);
	in_addr localHostIp;
	
	// Get local IP
	wVersionRequested = MAKEWORD(1, 1);
	err = WSAStartup(wVersionRequested, &wsaData);
	if (err == 0) {
		if ((LOBYTE(wsaData.wVersion) != 1) ||
			(HIBYTE(wsaData.wVersion) != 1)) {
			WSACleanup();
		}
		else {
			// Retreive local host IP address
			if (0 == gethostname(&g_sp.m_szLocalHostName[0], 200)) {
				he = gethostbyname(&g_sp.m_szLocalHostName[0]);
				if (he != NULL) {
					heLocalIP = (struct in_addr **) he->h_addr_list;
					while (heLocalIP != NULL) {
						localHostIp.S_un.S_addr = (*heLocalIP)[0].S_un.S_addr;
						// Convert host name to ANSI
						strncpy(&g_sp.m_szLocalHostIp[0], inet_ntoa(localHostIp), 20);
						// Convert ANSI string to UNICODE
#if defined(_UNICODE)
						int tLen;
						tLen = MultiByteToWideChar(
							CP_ACP,	// code page 
							NULL,	// character-type options 
							&g_sp.m_szLocalHostIp[0],	// address of string to map 
							-1,	// number of characters in string 
							&g_sp.m_tszLocalHostIp[0],	// address of wide-character buffer 
							40 	// size of buffer 
							);
#else
						_tcscpy(&g_sp.m_tszLocalHostIp[0], &g_sp.m_szLocalHostIp[0]);
#endif
						// heLocalIP++;
						break;
					}
				}
			}
		}
	}
	return WINAPI_SUCCESS;
}

/*
shutdownWinsock
--
Shutdowns winsock.
*/
WINAPI_STATUS shutdownWinsock(void)
{
	WSACleanup();
	return WINAPI_SUCCESS;
}



/*
notificationSnmpWndProc
--
This function is called by the system when
application has an message in the input message
que.
*/
LONG APIENTRY notificationSnmpWndProc( HWND hWnd, UINT message, UINT wParam, LONG lParam)
{
	LONG rv = 0;
	
	// _ftprintf(stdout, "\nmessage %d", message);
	switch (message) {
	case WM_CREATE:
		break;
	case WM_DESTROY:                  
		break;
	case WM_SNMP_MSG:
		if (g_sq)
			g_sq->NotificationCallback();
		break;
	default:
		return (DefWindowProc(hWnd, message, wParam, lParam));
	}
	return rv;
}

/*
SnmpQuery::SnmpQuery
--
SnmpQuery class constructor
*/
SnmpQuery::SnmpQuery(HANDLE notifyHandle, UINT notifyMessage)
{
	m_hNotifyHandle = notifyHandle;
	m_nNotifyMessage = notifyMessage;
	m_hSnmpSession = 0;
	m_bInitialized = FALSE;
	m_bWinSnmpStarted = FALSE;
}

/*
SnmpQuery::~SnmpQuery
--
SnmpQuery class destructor
*/
SnmpQuery::~SnmpQuery(void)
{
}

/*
SnmpQuery::Startup
--
Startup WinSNMP module
*/
WINAPI_STATUS SnmpQuery::Startup(void)
{
	SNMPAPI_STATUS s;
	WINAPI_STATUS rv = WINAPI_SUCCESS;
	
	if (!m_bSnmpInitialized) {
		s = SnmpStartup(&m_nMajorVersion, &m_nMinorVersion,
			&m_nLevel, &m_nTranslateMode,
			&m_nRetransmitMode);
		rv = (s == SNMPAPI_SUCCESS) ? WINAPI_SUCCESS : WINAPI_FAILURE;
		m_bSnmpInitialized = TRUE;
	}
	
	if (g_sp.m_bPingVersion) {
		_ftprintf(stdout, _TEXT("\n\nWinSNMP module version:\n"));
		_ftprintf(stdout, _TEXT("\nMajor version: %d"), m_nMajorVersion);
		_ftprintf(stdout, _TEXT("\nMinor version: %d"), m_nMinorVersion);
		_ftprintf(stdout, _TEXT("\nLevel: %d"), m_nLevel);
		_ftprintf(stdout, _TEXT("\nTranslate mode: %s"), g_szTranslateMode[(m_nTranslateMode < 3) ? m_nTranslateMode : 3]);
		_ftprintf(stdout, _TEXT("\nRetransmit mode: %s"), g_szRetransmitMode[(m_nRetransmitMode < 2) ? m_nRetransmitMode : 2]);
		_ftprintf(stdout, _TEXT("\n"));
	}
	return rv;
}

/*
SnmpQuery::Cleanup
--
Cleanup WinSNMP module
*/
WINAPI_STATUS SnmpQuery::Cleanup(void)
{
	SNMPAPI_STATUS s;
	WINAPI_STATUS rv;
	
	// Free descriptors
	if (m_oid.ptr)
		s = SnmpFreeDescriptor(SNMP_SYNTAX_OID, (smiLPOPAQUE) &m_oid);
	if (m_stopOid.ptr)
		s = SnmpFreeDescriptor(SNMP_SYNTAX_OID, (smiLPOPAQUE) &m_stopOid);
	
	s = SnmpCleanup();
	rv = (s == SNMPAPI_SUCCESS) ? WINAPI_SUCCESS : WINAPI_FAILURE;
	return rv;
}

/*
SnmpQuery::doQuery
--
Creates a first query. All other queries will be 
created in WinSNMP message handler function.
*/
WINAPI_STATUS SnmpQuery::doQuery(TCHAR *qName)
{
	SNMPAPI_STATUS s;
	smiVALUE tValue = {0};
	qName;
	
	if (m_hSnmpSession == 0) {
		m_hSnmpSession = SnmpOpen(m_hNotifyHandle, m_nNotifyMessage);
	}
	if (m_hSnmpSession == SNMPAPI_FAILURE) {
		return WINAPI_FAILURE;
	}
	// Set agent address
	_tcscpy(&m_tszIPAgent[0], g_sp.m_szDestinationIpAddress);
	// Set manager address
	_tcscpy(&m_tszIPManager[0], _TEXT("0.0.0.0"));
	// Set community
	m_smiCommunity.len = strlen(g_sp.m_lpszCommunity);
	m_smiCommunity.ptr = (smiLPBYTE) g_sp.m_lpszCommunity;
	// Set the WinSNMP internals
	// 
#if defined(_UNICODE)
	WideCharToMultiByte(
		CP_ACP,	// code page 
		0,	// performance and mapping flags 
		&m_tszIPAgent[0],	// address of wide-character string 
		-1,	// number of characters in string 
		&m_szIPAgent[0],	// address of buffer for new string 
		80,	// size of buffer 
		NULL,	// address of default for unmappable characters  
		NULL 	// address of flag set when default char. used 
		);
#else
	_tcscpy(&m_szIPAgent[0], &m_tszIPAgent[0]);
#endif
	m_hAgentEntity = SnmpStrToEntity(m_hSnmpSession, &m_szIPAgent[0]);
#if defined(_UNICODE)
	WideCharToMultiByte(
		CP_ACP,	// code page 
		0,	// performance and mapping flags 
		&m_tszIPManager[0],	// address of wide-character string 
		-1,	// number of characters in string 
		&m_szIPManager[0],	// address of buffer for new string 
		80,	// size of buffer 
		NULL,	// address of default for unmappable characters  
		NULL 	// address of flag set when default char. used 
		);
#else
	_tcscpy(&m_szIPManager[0], &m_tszIPManager[0]);
#endif
	m_hManagerEntity = SnmpStrToEntity(m_hSnmpSession, &m_szIPManager[0]);
	m_hViewContext = SnmpStrToContext(m_hSnmpSession, &m_smiCommunity);
	s = SnmpSetRetry(m_hManagerEntity, g_sp.m_dwRetransmit);
	s = SnmpSetTimeout(m_hManagerEntity, g_sp.m_dwTimeout / 10);
	s = SnmpSetRetry(m_hAgentEntity, g_sp.m_dwRetransmit);
	s = SnmpSetTimeout(m_hAgentEntity, g_sp.m_dwTimeout / 10);
	s = SnmpSetRetransmitMode(g_sp.m_bRetransmitMode ? SNMPAPI_ON : SNMPAPI_OFF);
	m_bInitialized = TRUE;
	
	// Initialize start OID and stop OID  
	// first: highest priority, last: lowest priority
	m_oid.len = 0;
	m_stopOid.len = 0;
	if (g_sp.m_bPingUntilInterrupted) {
		// Start conditions set to sysUpTime
		SnmpStrToOid("1.3.6.1.2.1.1.3.0", &m_oid);
		// Stop conditions set to sysContact
		SnmpStrToOid("1.3.6.1.2.1.1.4", &m_stopOid);
	}
	else if (g_sp.m_bSetSnmpValue) {
		// Start condition set by arguments
		if (g_sp.m_lpszStartOid) {
			s = SnmpStrToOid(g_sp.m_lpszStartOid, &m_oid);
			s = SnmpStrToOid(g_sp.m_lpszStartOid, &m_stopOid);
			g_sp.m_lpszStartOid = NULL;
		}
	}
	else if (g_sp.m_bCustomStartStopConditions) {
		// Start condition set by arguments
		if (g_sp.m_lpszStartOid) 
			s = SnmpStrToOid(g_sp.m_lpszStartOid, &m_oid);
		// Stop conditions set by arguments
		if (!g_sp.m_bBrowseWholeTree) {
			if (g_sp.m_lpszStopOid)
				s = SnmpStrToOid(g_sp.m_lpszStopOid, &m_stopOid);
			else {
				s = SnmpStrToOid(g_sp.m_lpszStartOid, &m_stopOid);
				(*(m_stopOid.ptr + m_stopOid.len - 1))++; 
			}
		}
		else {
			// no stop oid condition
			m_stopOid.len = 0;
			m_stopOid.ptr = NULL;
		}
	} 
	else if (g_sp.m_bBrowseWholeTree) {
		SnmpStrToOid("1.3.6.1.2.1.1.1", &m_oid);
		// no stop oid condition
		m_stopOid.len = 0;
		m_stopOid.ptr = NULL;
	}
	else if (g_sp.m_bBrowseSystemInfo) {
		// System info starts with sysDescriptor
		SnmpStrToOid("1.3.6.1.2.1.1.1", &m_oid);
		// Stop conditions set to interfaces
		SnmpStrToOid("1.3.6.1.2.1.2", &m_stopOid);
	}
	else {
		// Start conditions set to sysUpTime
		SnmpStrToOid("1.3.6.1.2.1.1.3", &m_oid);
		// Stop conditions set to sysContact
		SnmpStrToOid("1.3.6.1.2.1.1.4", &m_stopOid);
	}
	
	if (g_sp.m_bBrowseDepth) {
		// System info starts with sysDescriptor
		// Start condition set by arguments
		if (g_sp.m_lpszStartOid) {
			s = SnmpStrToOid(g_sp.m_lpszStartOid, &m_oid);
			s = SnmpStrToOid(g_sp.m_lpszStartOid, &m_stopOid);
			m_stopOid.len--;
			(*(m_stopOid.ptr + m_stopOid.len - 1))++; 
		}
	}
	
	
	// Print start and stop OID
	if (m_oid.ptr) {
		_ftprintf(stdout, _TEXT("\nStart OID: "));
		SnmpOutputValue(m_oid, tValue, OUTPUT_OID);
		_ftprintf(stdout, _TEXT(" ["));
		SnmpOutputValue(m_oid, tValue, OUTPUT_OID_NAME);
		_ftprintf(stdout, _TEXT("]"));
	}
	if (m_stopOid.ptr) {
		_ftprintf(stdout, _TEXT("\nStop OID:  "));
		SnmpOutputValue(m_stopOid, tValue, OUTPUT_OID);
		_ftprintf(stdout, _TEXT(" ["));
		SnmpOutputValue(m_stopOid, tValue, OUTPUT_OID_NAME);
		_ftprintf(stdout, _TEXT("]"));
	}
	_ftprintf(stdout, _TEXT("\n"));
	
	
	// Check start and stop OIDs before query
	if ((m_oid.len > 0) && (m_stopOid.len > 0)) {
		// check if stop condition matches current oid
		smiINT oidCompareResult;
		if (SNMPAPI_SUCCESS == SnmpOidCompare(&m_stopOid, &m_oid, 0, &oidCompareResult)) {
			if (oidCompareResult == -1) {
				_ftprintf(stdout, _TEXT("\nEnd OID is less than start OID."));
				_ftprintf(stdout, _TEXT("\nNothing to do."));
				return WINAPI_FAILURE;
			}
		}
	}
	
	if (g_sp.m_bSetSnmpValue) 
		s = SendSetRequest();
	else if (g_sp.m_bPingUntilInterrupted)
		s = SendGetRequest();
	else
		s = SendGetNextRequest();
	return s;
}

/*
SnmpQuery::SendGetRequest
--
Creates a first query. All other queries will be 
created in WinSNMP message handler function.
*/
WINAPI_STATUS SnmpQuery::SendGetRequest(void)
{
	SNMPAPI_STATUS s;
	
	// Create a variable binding list
	m_hVbl = SnmpCreateVbl(m_hSnmpSession, &m_oid, NULL);
	// Request ID should be increased every time a new message is sent
	m_dwRequestId = m_dwRequestId + 1;
	// Create a PDU
	m_hPdu = SnmpCreatePdu(m_hSnmpSession, SNMP_PDU_GET, m_dwRequestId, 0, 0, m_hVbl);
	// Send SNMP message
	s = SnmpSendMsg(m_hSnmpSession, m_hManagerEntity, m_hAgentEntity, m_hViewContext, m_hPdu);
	// Free resources
	s = SnmpFreeVbl(m_hVbl);
	s = SnmpFreePdu(m_hPdu);
	return SNMPAPI_SUCCESS;
}

/*
SnmpQuery::SendGetNextRequest
--
Creates a first query. All other queries will be 
created in WinSNMP message handler function.
*/
WINAPI_STATUS SnmpQuery::SendGetNextRequest(void)
{
	SNMPAPI_STATUS s;
	
	// Create a variable binding list
	m_hVbl = SnmpCreateVbl(m_hSnmpSession, &m_oid, NULL);
	// Request ID should be increased every time a new message is sent
	m_dwRequestId = m_dwRequestId + 1;
	// Create a PDU
	m_hPdu = SnmpCreatePdu(m_hSnmpSession, SNMP_PDU_GETNEXT, m_dwRequestId, 0, 0, m_hVbl);
	// Send SNMP message
	s = SnmpSendMsg(m_hSnmpSession, m_hManagerEntity, m_hAgentEntity, m_hViewContext, m_hPdu);
	// Free resources
	s = SnmpFreeVbl(m_hVbl);
	s = SnmpFreePdu(m_hPdu);
	return SNMPAPI_SUCCESS;
}
  
/*
SnmpQuery::SendSetRequest
--
Creates a SNMP set request.
*/
WINAPI_STATUS SnmpQuery::SendSetRequest(void)
{
	SNMPAPI_STATUS s;
	BOOL error;
	char ansiTs[200];
	
	// Stop oid is same as start oid
	// Create a variable binding list
	s = SnmpOidCopy(&m_oid, &m_stopOid);

	// Create one VB with value set
	switch (g_sp.m_dwValueType) {
	case SNMP_SYNTAX_OCTETS:
    case SNMP_SYNTAX_BITS:
    case SNMP_SYNTAX_OPAQUE:
    case SNMP_SYNTAX_NSAPADDR:
    case SNMP_SYNTAX_IPADDR:
		s = convertString2smiValue(&m_value, g_sp.m_lptszValueToSet);
		if (s == SNMPAPI_FAILURE) error = TRUE;
		break;
	case SNMP_SYNTAX_INT:
		// case SNMP_SYNTAX_INT32:
		m_value.value.sNumber = _ttoi(g_sp.m_lptszValueToSet);
		break;
    case SNMP_SYNTAX_UINT32:
    case SNMP_SYNTAX_CNTR32:
    case SNMP_SYNTAX_GAUGE32:
    case SNMP_SYNTAX_TIMETICKS:
		m_value.value.uNumber = _tcstoul(g_sp.m_lptszValueToSet, 
			NULL,
			// sp.m_lptszValueToSet + _tcsclen(sp.m_lptszValueToSet), 
			10);
		break;
	case SNMP_SYNTAX_OID:
		m_value.value.oid.len = 0;
#if defined(_UNICODE)
		int ansiTsLen;
		ansiTsLen = WideCharToMultiByte(
			CP_ACP,	// code page 
			NULL,	// performance and mapping flags 
			g_sp.m_lptszValueToSet,	// address of wide-character string 
			-1,	// number of characters in string 
			ansiTs,	// address of buffer for new string 
			255,	// size of buffer 
			NULL,	// address of default for unmappable characters  
			NULL	// address of flag set when default char. used 
			);
#else
		_tcsncpy(ansiTs, g_sp.m_lptszValueToSet, 255);
#endif
		s = SnmpStrToOid(ansiTs, &m_value.value.oid);
		if (s == SNMPAPI_FAILURE) error = TRUE;
		break;
    case SNMP_SYNTAX_NULL:
		m_value.value.empty = 0;
		break;
	default:
		error = TRUE;
		m_value.syntax = SNMP_SYNTAX_NULL;
	}
	if (error)
		return SNMPAPI_FAILURE;
	//
	m_hVbl = SnmpCreateVbl(m_hSnmpSession, &m_oid, &m_value);
	// Request ID should be increased every time a new message is sent
	m_dwRequestId = m_dwRequestId + 1;
	// Create a PDU
	m_hPdu = SnmpCreatePdu(m_hSnmpSession, SNMP_PDU_SET, m_dwRequestId, 0, 0, m_hVbl);
	// Send SNMP message
	s = SnmpSendMsg(m_hSnmpSession, m_hManagerEntity, m_hAgentEntity, m_hViewContext, m_hPdu);
	// Free resources
	s = SnmpFreeVbl(m_hVbl);
	s = SnmpFreePdu(m_hPdu);
	return SNMPAPI_SUCCESS;
}
  
  
/*
SnmpQuery::NotificationCallback
--
If WinSNMP message is in the application input que
this function ic called.
--	
At first it chechs the message reason, next it checks
if stopOid condition is set. If stop condition is not
set then it sends a new query. If stop condition is set
the it sends a STOP message to the application. STOP 
message is also sent if SnmpRecvMsg returns an error.
Application will break the message loop and exit on STOP
message.
*/
void SnmpQuery::NotificationCallback(void)
{
	SNMPAPI_STATUS s;
	HSNMP_ENTITY hRcvAgentEntity;
	HSNMP_ENTITY hRcvManagerEntity;
	HSNMP_CONTEXT hRcvViewContext;
	HSNMP_VBL hRcvVbl;
	HSNMP_PDU hRcvPdu;
	BYTE outputFlags;

	smiINT32 requestId;
	smiINT pduType, errorStatus, errorIndex;

	int oidc;
	smiOID oid;
	smiVALUE value;
	
	int i;
	smiINT oidCompareResult;
	BOOL stopCondition = FALSE;

	// _ftprintf(stdout, "\n Got SNMP reg_sponse");
	s = SnmpRecvMsg(m_hSnmpSession, &hRcvAgentEntity, &hRcvManagerEntity,
		&hRcvViewContext, &hRcvPdu);
	if (s == SNMPAPI_FAILURE) {
		s = SnmpGetLastError(m_hSnmpSession);
		if (s == SNMPAPI_TL_TIMEOUT) {
			stopCondition = TRUE;
			_ftprintf(stdout, _TEXT("\nRequest timed out."));
			while (!PostThreadMessage(GetCurrentThreadId(), WM_SNMP_MSG_STOP, 0, 0))
				;
			return;
		}
		if (s == SNMPAPI_NOOP) {
			// MyWriteLn('NOOP');
			return;
		}
		// MyWriteLn('ERROR');
		// no more queries
		_ftprintf(stdout, _TEXT("\nUnhandled WinSNMP error %d."), s);
		while (!PostThreadMessage(GetCurrentThreadId(), WM_SNMP_MSG_STOP, 0, 0))
			;
		return;
	}

	s = SnmpGetPduData(hRcvPdu, &pduType, &requestId, &errorStatus, &errorIndex, &hRcvVbl);
	// Did we reach the end of agents' MIB tree?
	if (errorStatus != 0) {
		if (!g_sp.m_bSetSnmpValue) {
			if (errorStatus == SNMP_ERROR_NOSUCHNAME) {
				stopCondition = TRUE;
				goto doExitCleanCallback;
			}
		}
		_ftprintf(stdout, _TEXT("\nError returned by remote agent: %s (%d)"), 
			((errorStatus < MAX_SNMP_ERRORS) ? g_szSnmpErrors[errorStatus] : _TEXT("unknown error")),
			errorStatus);
	}
	
	oidc = SnmpCountVbl(hRcvVbl);
	for (i = 1; i <= oidc; i++) {
		s = SnmpGetVb(hRcvVbl, i, &oid, &value);
		// remember first SNMP binding value for the next query
		if (i == 1) {
			// free oid descriptor from previous query
			if (m_oid.ptr) 
				SnmpFreeDescriptor(SNMP_SYNTAX_OID, (smiLPOPAQUE) &m_oid);
			// make oid for new query
			SnmpOidCopy(&oid, &m_oid);
			m_value.syntax = SNMP_SYNTAX_NULL;
		}
		// stop condition set ?
		if (m_stopOid.len > 0) {
			// check if stop condition matches current oid
			if (SNMPAPI_SUCCESS == SnmpOidCompare(&m_stopOid, &m_oid, 0, &oidCompareResult)) {
				if (oidCompareResult == -1) {
					// Stop condition is set. 
					stopCondition = TRUE;
				}
				else {
					outputFlags = OUTPUT_VALUE | OUTPUT_SEPARATOR | OUTPUT_NEWLINE;
					if (g_sp.m_bOidToStringConversion)
						outputFlags |= OUTPUT_OID_NAME;
					else
						outputFlags |= OUTPUT_OID;
					if (g_sp.m_bPrintValueType)
						outputFlags |= OUTPUT_VALUE_TYPE;
					if (g_sp.m_bPrintLineNumber) {
						_ftprintf(stdout, _TEXT("\n%d: "), g_sp.m_dwCurrentLineNumber++);
						outputFlags &= ~OUTPUT_NEWLINE;
					}
					SnmpOutputValue(oid, value, outputFlags);
				}
			}
		}
		else {
			outputFlags = OUTPUT_VALUE | OUTPUT_SEPARATOR | OUTPUT_NEWLINE;
			if (g_sp.m_bOidToStringConversion)
				outputFlags |= OUTPUT_OID_NAME;
			else
				outputFlags |= OUTPUT_OID;
			if (g_sp.m_bPrintValueType)
				outputFlags |= OUTPUT_VALUE_TYPE;
			if (g_sp.m_bPrintLineNumber) {
				_ftprintf(stdout, _TEXT("\n%d: "), g_sp.m_dwCurrentLineNumber++);
				outputFlags &= ~OUTPUT_NEWLINE;
			}
			SnmpOutputValue(oid, value, outputFlags);
		}
		//  Free receive descriptors
		s = SnmpFreeDescriptor(SNMP_SYNTAX_OID, (smiLPOCTETS) &oid);
		switch (value.syntax) {
		case SNMP_SYNTAX_OCTETS:
			s = SnmpFreeDescriptor(value.syntax, &value.value.string);
			break;
		case SNMP_SYNTAX_OID:
			s = SnmpFreeDescriptor(value.syntax, (smiLPOCTETS) &value.value.oid);
			break;
		case SNMP_SYNTAX_IPADDR:
			s = SnmpFreeDescriptor(value.syntax, &value.value.string);
			break;
		}
	}

doExitCleanCallback:
	// Free WinSNMP internal data
	SnmpFreePdu(hRcvPdu);
	SnmpFreeVbl(hRcvVbl);
	SnmpFreeEntity(hRcvAgentEntity);
	SnmpFreeEntity(hRcvManagerEntity);
	SnmpFreeContext(hRcvViewContext);


	// Decide what to do next...
	// Ping until interrupted
	if (g_sp.m_bPingUntilInterrupted) {
		stopCondition = TRUE;
		// Print remote SNMP agent IP address
		_ftprintf(stdout, _TEXT(" (%s)"), g_sp.m_szDestinationIpAddress);
		// Wait for interval
		Sleep(g_sp.m_dwPingInterval);
		// Send get-request to confirm set-request
		SnmpFreeVbl(m_hVbl);
		m_hVbl = SnmpCreateVbl(m_hSnmpSession, &m_oid, NULL);
		m_dwRequestId++;
		SnmpFreePdu(m_hPdu);
		m_hPdu = SnmpCreatePdu(m_hSnmpSession, SNMP_PDU_GET, m_dwRequestId, 0, 0, m_hVbl);
		s = SnmpSendMsg(m_hSnmpSession, m_hManagerEntity, m_hAgentEntity, m_hViewContext, m_hPdu);
		return;
	}
	// Set-value
	if (g_sp.m_bSetSnmpValue) {
		stopCondition = TRUE;
		g_sp.m_bSetSnmpValue = FALSE;
		_ftprintf(stdout, _TEXT("\nSet confirmation..."));
		// Send get-request to confirm set-request
		SnmpFreeVbl(m_hVbl);
		m_hVbl = SnmpCreateVbl(m_hSnmpSession, &m_oid, NULL);
		m_dwRequestId++;
		SnmpFreePdu(m_hPdu);
		m_hPdu = SnmpCreatePdu(m_hSnmpSession, SNMP_PDU_GET, m_dwRequestId, 0, 0, m_hVbl);
		s = SnmpSendMsg(m_hSnmpSession, m_hManagerEntity, m_hAgentEntity, m_hViewContext, m_hPdu);
		return;
	}
	// Mib-walk
	// Is mib-walk stop condition set?
	if (!stopCondition) {
		// yet another query
//		SnmpFreeVbl(m_hVbl);
		m_hVbl = SnmpCreateVbl(m_hSnmpSession, &m_oid, NULL);
		m_dwRequestId++;
//		SnmpFreePdu(m_hPdu);
		m_hPdu = SnmpCreatePdu(m_hSnmpSession, SNMP_PDU_GETNEXT, m_dwRequestId, 0, 0, m_hVbl);
		s = SnmpSendMsg(m_hSnmpSession, m_hManagerEntity, m_hAgentEntity, m_hViewContext, m_hPdu);
	}
	else {
		// Post finish message to thread main message handler
		_ftprintf(stdout, _TEXT("\n\nFinished."));
		while (!PostThreadMessage(GetCurrentThreadId(), WM_SNMP_MSG_STOP, 0, 0))
			;
	}
}

/*
SnmpQuery::SnmpOutputValue
--
Outputs the oid and value to stdout.
Flags define output style and contents.
*/
WINAPI_STATUS SnmpQuery::SnmpOutputValue(smiOID oid, smiVALUE value, BYTE flags)
{
	smiLPUINT32 oidp;
	long i;
	unsigned int ui;
	int j;
	int d,h,m,s,th;
	HWSM_NODE mibNode;
	int initialOidLen;
	char ansiTs[1028];
	TCHAR ts[2048];
	TCHAR ts1[2048];
	TCHAR ts2[5];
	smiBYTE *chp;
	BOOL printHex;
	
	ts[0] = TCHAR('\0');
	ts1[0] = TCHAR('\0');
	
	initialOidLen = oid.len;
	// append newline
	if (flags & OUTPUT_NEWLINE) {
		_tcscat(ts, _TEXT("\n"));
	}
	// append OID
	if (flags & OUTPUT_OID) {
		// append OID reminder to the string
		j = 0;
		oidp = oid.ptr;
		oidp += j;
		for (ui = j; ui < oid.len; ui++) {
			if (ui != 0)
				_tcscat(ts, _TEXT("."));
			_stprintf(ts1, _TEXT("%d"), *oidp);
			_tcscat(ts, ts1);
			oidp++;
		}
	}
	// append OID string name
	if (flags & OUTPUT_OID_NAME) {
		for (j = oid.len; j > 0; j--) {
			oid.len = j;
			mibNode = WsmMapOidToNode(&oid);
			if (mibNode != 0) {
				i = 500;
				WsmGetNodeString(mibNode, WSM_INFO_NAME, (long *) &i, &ansiTs[0]);
#if defined(_UNICODE)
				int len;
				len = MultiByteToWideChar(
					CP_ACP,	// code page 
					0,	// character-type options 
					ansiTs,	// address of string to map 
					-1,	// number of characters in string 
					ts1,	// address of wide-character buffer 
					2048 	// size of buffer 
					);
#else
				_tcscpy(ts1, ansiTs);
#endif
				_tcscat(ts, ts1);
				break;
			}
		}
		oid.len = initialOidLen;
		// append OID reminder to the string
		oidp = oid.ptr;
		oidp += j;
		for (ui = j; ui < oid.len; ui++) {
			if (ui != 0)
				_tcscat(ts, _TEXT("."));
			_stprintf(ts1, _TEXT("%d"), *oidp);
			_tcscat(ts, ts1);
			oidp++;
		}
	}
	// append separator
	if (flags & OUTPUT_SEPARATOR) {
		_tcscat(ts, _TEXT(" **** "));
	}
	if (flags & OUTPUT_VALUE_TYPE) {
		switch (value.syntax) {
		case SNMP_SYNTAX_INT:       _tcscat(ts, _TEXT("(int,int32)")); break;
			//case SNMP_SYNTAX_INT32:     _tcscat(ts, "(int32)"); break;
		case SNMP_SYNTAX_UINT32:    _tcscat(ts, _TEXT("(uint32)")); break;
		case SNMP_SYNTAX_CNTR32:    _tcscat(ts, _TEXT("(cntr32)")); break;
		case SNMP_SYNTAX_GAUGE32:   _tcscat(ts, _TEXT("(gauge32)")); break;
		case SNMP_SYNTAX_TIMETICKS: _tcscat(ts, _TEXT("(timeticks)")); break;
		case SNMP_SYNTAX_CNTR64:    _tcscat(ts, _TEXT("(cntr64)")); break;
		case SNMP_SYNTAX_OCTETS:    _tcscat(ts, _TEXT("(octets)")); break;
		case SNMP_SYNTAX_BITS:      _tcscat(ts, _TEXT("(bits)")); break;
		case SNMP_SYNTAX_OPAQUE:    _tcscat(ts, _TEXT("(opaque)")); break;
		case SNMP_SYNTAX_IPADDR:    _tcscat(ts, _TEXT("(ipaddr)")); break;
		case SNMP_SYNTAX_NSAPADDR:  _tcscat(ts, _TEXT("(nsapaddr)")); break;
		case SNMP_SYNTAX_OID:       _tcscat(ts, _TEXT("(oid)")); break;
		case SNMP_SYNTAX_NULL:      _tcscat(ts, _TEXT("(null)")); break;
		case SNMP_SYNTAX_NOSUCHOBJECT:    _tcscat(ts, _TEXT("(nosuchobject)")); break;
		case SNMP_SYNTAX_NOSUCHINSTANCE:  _tcscat(ts, _TEXT("(nosuchinstance)")); break;
		case SNMP_SYNTAX_ENDOFMIBVIEW:    _tcscat(ts, _TEXT("(endofmibview)")); break;
		}
		_tcscat(ts, _TEXT(" "));
	}
	// append value
	if (flags & OUTPUT_VALUE) {
		switch (value.syntax) {
		case SNMP_SYNTAX_INT:
			// case SNMP_SYNTAX_INT32:
			_stprintf(ts1, _TEXT("%d"), value.value.sNumber);
			_tcscat(ts, ts1);
			break;
		case SNMP_SYNTAX_UINT32:
		case SNMP_SYNTAX_CNTR32:
		case SNMP_SYNTAX_GAUGE32:
			_stprintf(ts1, _TEXT("%d"), value.value.uNumber);
			_tcscat(ts, ts1);
			break;
		case SNMP_SYNTAX_TIMETICKS:
			d = value.value.uNumber;
			th = d % 100; d = d / 100;
			s = d % 60;   d = d / 60;
			m = d % 60;   d = d / 60;
			h = d % 24;
			d = d / 24;
			if (d == 1)
				_stprintf(ts1, _TEXT("%d day %02.02dh:%02.02dm:%02.02ds.%02.02dth"), d, h, m, s, th);
			else
				_stprintf(ts1, _TEXT("%d days %02.02dh:%02.02dm:%02.02ds.%02.02dth"), d, h, m, s, th);
			_tcscat(ts, ts1);
			break;
		case SNMP_SYNTAX_CNTR64:
			_stprintf(ts1, _TEXT("%x.%x"), value.value.hNumber.hipart, value.value.hNumber.lopart);
			_tcscat(ts, ts1);
			_tcscat(ts, _TEXT(" (hex)"));
			break;
		case SNMP_SYNTAX_OCTETS:
			/*
			  Octets are in ANSI notation.
			  Have to be converted to Unicode
			*/
			if (value.value.string.len < 1) {
				_tcscat(ts, _TEXT("(zero-length)"));
				break;
			}
			printHex = FALSE;
			chp = value.value.string.ptr;
			_tcscpy(ts1, _TEXT(""));
			for (ui = 0; ui < value.value.string.len; ui++) {
				if (((isprint(*chp) == 0) && 
					(isspace(*chp) == 0)) &&
					((*chp == char('\0')) &&
					(ui != (value.value.string.len - 1)))) {
					// to print string only last char can be '\0'
					printHex = TRUE;
				}
				if (ui != 0)
					_tcscat(ts1, _TEXT("."));
				_stprintf(ts2, _TEXT("%02x"), *chp++);
				_tcscat(ts1, ts2);
			}
			if (printHex) {
				// print octets
				_tcscat(ts, ts1);
				_tcscat(ts, _TEXT(" (hex)"));
				break;
			}
			// print string
			memcpy(ansiTs, value.value.string.ptr, value.value.string.len);
			ansiTs[value.value.string.len] = char('\0');
#if defined(_UNICODE)
			MultiByteToWideChar(
				CP_ACP,	// code page 
				0,	// character-type options 
				ansiTs,	// address of string to map 
				-1,	// number of characters in string 
				ts1,	// address of wide-character buffer 
				2048 	// size of buffer 
				);
#else
				_tcscpy(ts1, ansiTs);
#endif
			_tcscat(ts, ts1);
			break;
		case SNMP_SYNTAX_BITS:
			chp = value.value.string.ptr;
			for (ui = 0; ui < value.value.string.len; ui++) {
				if (ui != 0)
					_tcscat(ts, _TEXT("."));
				_stprintf(ts1, _TEXT("%02x"), *chp++);
				_tcscat(ts, ts1);
			}
			_tcscat(ts, _TEXT(" (hex)"));
			break;
		case SNMP_SYNTAX_OPAQUE:
			chp = value.value.string.ptr;
			for (ui = 0; ui < value.value.string.len; ui++) {
				if (ui != 0)
					_tcscat(ts, _TEXT("."));
				_stprintf(ts1, _TEXT("%02x"), *chp++);
				_tcscat(ts, ts1);
			}
			_tcscat(ts, _TEXT(" (hex)"));
			break;
		case SNMP_SYNTAX_IPADDR:
			chp = value.value.string.ptr;
			for (ui = 0; ui < value.value.string.len; ui++) {
				if (ui != 0)
					_tcscat(ts, _TEXT("."));
				_stprintf(ts1, _TEXT("%d"), *chp++);
				_tcscat(ts, ts1);
			}
			break;
		case SNMP_SYNTAX_NSAPADDR:
			chp = value.value.string.ptr;
			for (ui = 0; ui < value.value.string.len; ui++) {
				if (ui != 0)
					_tcscat(ts, _TEXT("."));
				_stprintf(ts1, _TEXT("%02x"), *chp++);
				_tcscat(ts, ts1);
			}
			_tcscat(ts, _TEXT(" (hex)"));
			break;
		case SNMP_SYNTAX_OID:
			// check if OID is NULL-OID 0.0 
			if (value.value.oid.len == 2) {
				oidp = value.value.oid.ptr;
				if (*oidp++ == 0) 
					if (*oidp == 0)
					_tcscat(ts, _TEXT("(null-oid) "));
			}
			//  find a string alias for the OID
			initialOidLen = value.value.oid.len;
			for (j = value.value.oid.len; j > 0; j--) {
				value.value.oid.len = j;
				mibNode = WsmMapOidToNode(&value.value.oid);
				if (mibNode != 0) {
					i = 500;
					WsmGetNodeString(mibNode, WSM_INFO_NAME, (long *) &i, &ansiTs[0]);
#if defined(_UNICODE)
					int len;
					len = MultiByteToWideChar(
						CP_ACP,	// code page 
						0,	// character-type options 
						ansiTs,	// address of string to map 
						-1,	// number of characters in string 
						ts1,	// address of wide-character buffer 
						2048 	// size of buffer 
						);
#else
					_tcscpy(ts1, ansiTs);
#endif
					_tcscat(ts, ts1);
					break;
				}
			}
			value.value.oid.len = initialOidLen;
			// append OID reminder to the string
			oidp = value.value.oid.ptr;
			oidp += j;
			for (ui = j; ui < value.value.oid.len; ui++) {
				if (ui != 0)
					_tcscat(ts, _TEXT("."));
				_stprintf(ts1, _TEXT("%d"), *oidp);
				_tcscat(ts, ts1);
				oidp++;
			}
			break;
		case SNMP_SYNTAX_NOSUCHOBJECT:
			_tcscat(ts, _TEXT("NOSUCHOBJECT"));
			break;
		case SNMP_SYNTAX_NOSUCHINSTANCE:
			_tcscat(ts, _TEXT("NOSUCHINSTANCE"));
			break;
		case SNMP_SYNTAX_ENDOFMIBVIEW:
			_tcscat(ts, _TEXT("ENDOFMIBVIEW"));
			break;
		case SNMP_SYNTAX_NULL:
			_tcscat(ts, _TEXT("NULL"));
			break;
		default:
			return WINAPI_FAILURE;
		}
	}
	// Output result
	_ftprintf(stdout, _TEXT("%s"), ts);
	return WINAPI_SUCCESS;
}

SNMPAPI_STATUS convertString2smiValue(smiLPVALUE value, TCHAR *buffer)
{
	SNMPAPI_STATUS rv = SNMPAPI_SUCCESS;
	TCHAR *ts;
	int i, n;
	smiOID tOid;
	smiUINT32 tOidVal[MAXOBJIDSTRSIZE];
	smiVALUE tValue;
	smiBYTE tValueVal[500];
	
	switch (value->syntax) {
    case SNMP_SYNTAX_INT:
		// case SNMP_SYNTAX_INT32:
		value->value.sNumber = _ttol(buffer);
		break;
    case SNMP_SYNTAX_UINT32:
    case SNMP_SYNTAX_CNTR32:
    case SNMP_SYNTAX_GAUGE32:
		value->value.uNumber = _tcstoul(buffer, NULL, 10);
		break;
    case SNMP_SYNTAX_TIMETICKS:
		value->value.uNumber = _tcstoul(buffer, NULL, 10);
		break;
    case SNMP_SYNTAX_CNTR64:
		// To do!
		_stscanf(buffer, _TEXT("%08x.%08x"), 
			value->value.hNumber.hipart, 
			value->value.hNumber.lopart);
		break;
    case SNMP_SYNTAX_OCTETS:
    #if defined(__BORLANDC__)
		value->value.string.len = _tcslen(buffer);
    #else
		value->value.string.len = _tcsclen(buffer);
    #endif
		value->value.string.ptr = (smiBYTE *) malloc(value->value.string.len * sizeof(smiBYTE));
		if (value->value.string.ptr) {
			memcpy(value->value.string.ptr, buffer, value->value.string.len * sizeof(smiBYTE));
		}
		else {
			rv = SNMPAPI_FAILURE;
		}
		break;
    case SNMP_SYNTAX_BITS:
    case SNMP_SYNTAX_OPAQUE:
    case SNMP_SYNTAX_IPADDR:
		tValue.syntax = value->syntax;
		tValue.value.string.len = 0;
		tValue.value.string.ptr = tValueVal;
		ts = buffer;
		for (n = 0; n < 4; n++) {
			if (_stscanf(ts, _TEXT("%d"), &i) == 0) {
				rv = SNMPAPI_FAILURE;
				break;
			}
			tValue.value.string.len++;
			tValue.value.string.ptr[n] = (BYTE) i;
			while ((*ts != TCHAR('.')) && (*ts != TCHAR('\0')))
				ts++;
			if (*ts == TCHAR('\0')) break;
			ts++;         // One more for the first digit
			if (*ts == TCHAR('\0')) break;
		}
		if (n != 3) {
			rv = SNMPAPI_FAILURE;
			break;
		}
		// Allocate memory for value.oid value and copy oid
		// leave syntax unchanged
		value->value.string.ptr = (smiBYTE *) malloc(tValue.value.string.len * sizeof(smiBYTE));
		if (value->value.string.ptr) {
			value->value.string.len = tValue.value.string.len;
			memcpy(value->value.string.ptr, tValue.value.string.ptr, tValue.value.string.len * sizeof(smiBYTE));
		}
		else {
			rv = SNMPAPI_FAILURE;
		}
		break;
    case SNMP_SYNTAX_NSAPADDR:
		break;
    case SNMP_SYNTAX_OID:
		// Use SnmpStrToOid ...
		tOid.len = 0;
		tOid.ptr = &tOidVal[0];
		value->value.oid.len = 0;
		n = 0;
		ts = buffer;
		while (1) {
			if (_stscanf(ts, _TEXT("%d"), &i) == 0) {
				rv = SNMPAPI_FAILURE;
				break;
			}
			tOid.len++;
			tOid.ptr[n++] = i;
			while ((*ts != TCHAR('.')) && (*ts != TCHAR('\0')))
				ts++;
			if (*ts == TCHAR('\0')) break;
			ts++;         // One more for the first digit
			if (*ts == TCHAR('\0')) break;
		}
		if (rv != TRUE) 
			break;
		// Allocate memory for value.oid value and copy oid
		value->value.oid.ptr = (smiUINT32 *) malloc(tOid.len * sizeof(smiUINT32));
		if (value->value.oid.ptr) {
			value->value.oid.len = tOid.len;
			memcpy(value->value.oid.ptr, tOid.ptr, tOid.len * sizeof(smiUINT32));
		}
		else {
			rv = SNMPAPI_FAILURE;
		}
		break;
    case SNMP_SYNTAX_NULL:
    case SNMP_SYNTAX_NOSUCHOBJECT:
    case SNMP_SYNTAX_NOSUCHINSTANCE:
    case SNMP_SYNTAX_ENDOFMIBVIEW:
		value->value.empty = 0;
		break;
    default:
		rv = SNMPAPI_FAILURE;
		break;
	}
	return rv;
}


/*
ReadFileVersion (Unicode)
--
Reads the file version information from VersionInfo
resources
*/
BOOL ReadFileVersion(TCHAR *lptszFileName,
					 TCHAR *lptszFileVersionBuf, 
					 DWORD dwBufLen)
{
	BOOL brv;
	DWORD  handle;
	UINT  uiInfoSize;
	UINT  uiVerSize;
	UINT  uiSize;
 	BYTE *pbData;
	DWORD *lpBuffer;
	TCHAR tszName[512];

	// Get the size of the version information.
	uiInfoSize = ::GetFileVersionInfoSize(lptszFileName, &handle);
	if (uiInfoSize == 0)
		return FALSE;
	// Allocate a buffer for the version information.
	pbData = (BYTE *) malloc(uiInfoSize);
	if (pbData == NULL)
		return FALSE;
	// Fill the buffer with the version information.
	brv = ::GetFileVersionInfo(lptszFileName,
		handle,
		uiInfoSize,
		pbData);
	if (!brv) goto Failure;
	// Get the translation information.
	brv = ::VerQueryValue( pbData,
		_TEXT("\\VarFileInfo\\Translation"),
		(void**)&lpBuffer,
		&uiVerSize);
	if ((!brv) ||
		(uiVerSize == 0)) {
		goto Failure;
	}
	// Build the path to the FileVersion key
	// using the translation information.
	_stprintf(tszName,
		_TEXT("\\StringFileInfo\\%04hX%04hX\\FileVersion"),
		LOWORD(*lpBuffer),
		HIWORD(*lpBuffer));
	// Search for the key.
	brv = ::VerQueryValue(pbData, 
		tszName, 
		(void**)&lpBuffer, 
		&uiSize);
	if ((!brv) ||
		(uiSize == 0)) {
		goto Failure;
	}
	// Copy string to the output buffer
   #if defined(__BORLANDC__)
   	_tcsncpy(lptszFileVersionBuf, (TCHAR *) lpBuffer, dwBufLen);
   #else
   	_tcsnccpy(lptszFileVersionBuf, (TCHAR *) lpBuffer, dwBufLen);
   #endif
	// Set the NULL termination
	*(lptszFileVersionBuf + dwBufLen - sizeof(TCHAR)) = TCHAR('\0');
	// Finished

Failure:  
	free(pbData);
	return brv;
}
