/* A demo program for SimpleWin that displays a rendering of the Mandelbrot
set and allows the user to show the Julia set for a given point by
clicking with the left mouse button on that point.
Clicking with the right mouse button gives a x5 magnification of the
Mandelbrot set centred around the point clicked on.

This source may be freely used in any other programs that use SimpleWn.dll

*/

// Use one or the other of the following include files
//#include "..\..\SWLoad.h"	// Use if using SWLoad.c
#include "SimpleWn.h" // Use if using SimpleWn.lib


#include <stdio.h>
#include <math.h>
#include <conio.h>

#define WIN_SIZE 150

#define MANDEL 1
#define JULIA 2


// Function prototypes
void SaveImage(SWHandle hSWnd);
void DrawSet(int set, SWHandle hSWnd, double x0, double y0, double scale);
int CalcMandPt(double x, double y);
int CalcJulPt(double xd, double yd, double x0, double y0);
void PlotSetPoint(SWHandle hSWnd, int x, int y, int density);
double CalcCentre(double centre, double scale, int pt);


int Setup(void);	// Setup the SimpleWin library
void Finish(void);	// Close the library down



void main()
{
	int			setup;
	SWHandle	h1, h2;
	double		scale, xcentre, ycentre, xc, yc;
	int			quit;
	char		key;

	setup = Setup();	// Load the SimpleWn.dll library
	if(setup == 0)
		return;

	h1 = SWCreateWindow(20, 20, WIN_SIZE, WIN_SIZE, "Mandelbrot Set");  // Create the Mandelbrot window
	h2 = SW_NULL_HANDLE;  // Clear the handle for the Julia window

	xcentre = ycentre = 0.0;
	scale = 4.0;

	
	printf("\nFractal demonstration program for SimpleWin\n\n");
	printf("On Mandelbrot window:\n");
	printf("  Use the left mouse button to select a point to draw the corresponding Julia \n   set.\n");
	printf("  Use the right mouse button to zoom in on the Mandelbrot set.\n");
	printf("  Use the right mouse button + SHIFT to zoom out from the Mandelbrot set.\n\n");
	printf("On Julia window:\n");
	printf("  Use the left mouse button to close the window.\n\n");
	printf("With the appropriate window at the front:\n");
	printf("  'S' to save the image\n");
	printf("  'C' to copy the image to the clipboard\n\n");
	printf("With any of the application windows at the front:\n");
	printf("  'Q' to quit\n\n");
	
	quit = 0;
	DrawSet(MANDEL, h1, xcentre, ycentre, scale);
	while(!quit)
	{
		// Capture the last left mouse click
		// Calls to SWGetLastLeftMouseClickX, SWGetLastLeftMouseClickY, 
		// SWGetLastLeftMouseClickSHIFT and SWGetLastLeftMouseClickCTRL
		// will use this captured value until capture is called again
		// If the relevant mouse button was clicked, the function returns SW_OKAY, 
		// else SW_NO_MOUSE_CLICK
		
		if(SWCaptureLastLeftMouseClick(h1) == SW_OKAY)
		{
			if(h2 == 0)
				h2 = SWCreateWindow(200, 200, WIN_SIZE, WIN_SIZE, "Julia Set");
			xc = CalcCentre(xcentre, scale, SWGetLastLeftMouseClickX(h1));
			yc = CalcCentre(ycentre, scale, SWGetLastLeftMouseClickY(h1));
			DrawSet(JULIA, h2, xc, yc, scale);
		}
		if(SWCaptureLastRightMouseClick(h1) == SW_OKAY)
		{
			xcentre = CalcCentre(xcentre, scale, SWGetLastRightMouseClickX(h1));
			ycentre = CalcCentre(ycentre, scale, SWGetLastRightMouseClickY(h1));
			if(SWGetLastRightMouseClickSHIFT(h1))
				scale = scale * 5.0;
			else
				scale = scale / 5.0;
			DrawSet(MANDEL, h1, xcentre, ycentre, scale);
		}
		if((SWCaptureLastLeftMouseClick(h2) == SW_OKAY))
		{
			if(h2 != 0)
				SWDestroyWindow(h2);	// Close the window
			h2 = SW_NULL_HANDLE;
		}
		if(kbhit())
		{
			key = getch();
			switch(key)
			{
				case 'q':
				case 'Q':	quit = 1; break;
			}
		}
		if(SW_kbhit(h1) == 1)
		{
			key = SWReadChar(h1);
			switch(key)
			{

				case 'S':
				case 's':	printf("Save Mandelbrot - ");
							SaveImage(h1); // Save as BMP
							break; 
				case 'C':
				case 'c':	printf("Copy image to clipboard\n");
							SWCopyImageToClipboard(h1);
							break;
				case 'q':
				case 'Q':	quit = 1; 
							break;
			}
		}
		if(SW_kbhit(h2) == 1)
		{
			key = SWReadChar(h2);
			switch(key)
			{

				case 'S':
				case 's':	printf("Save Julia - ");
							SaveImage(h2); // Save as BMP
							break; 
				case 'C':
				case 'c':	printf("Copy image to clipboard\n");
							SWCopyImageToClipboard(h2);
							break;
				case 'q':
				case 'Q':	quit = 1; 
							break;
			}
		}
	}

}


// Save the image in the given window
void SaveImage(SWHandle hSWnd)
{
	char	filename[300];
	
	if(!SWIsWindow(hSWnd)) // Test to see if the window is valid - 1 if so, 0 if not
	{
		printf("Window not open\n");
		return;
	}
	printf("\nEnter filename: ");
	scanf("%s", filename);
	SWSaveImage(hSWnd, filename); // Save the image as a BMP file
	printf(" - Saved\n");
}



 // No SimpleWin functions here, just a function to draw either Mandelbrot or Julia set
void DrawSet(int set, SWHandle hSWnd, double x0, double y0, double scale)
{
	double		delta, xd, yd;
	int			xi, yi, density;

	if(set == JULIA)
		scale = 5.0;
	
	delta = scale/(double)WIN_SIZE;

	if(set == JULIA)
		yd = -scale/2.0;
	else
		yd = y0 - scale/2.0;
	for(yi = 0; yi < WIN_SIZE; ++yi)
	{
		if(set == JULIA)
			xd = -scale/2.0;
		else
			xd = x0 - scale/2.0;
		for(xi = 0; xi < WIN_SIZE; ++xi)
		{
			if(set == MANDEL)
				density = CalcMandPt(xd, yd);
			else
				density = CalcJulPt(xd, yd, x0, y0);
			PlotSetPoint(hSWnd, xi, yi, density);
			xd += delta;
		}
		yd += delta;
	}
}


// Calculate the value for the Mandelbrot image
int CalcMandPt(double x, double y)
{
	double	zr, zi, wr, mag;
	int		count;

	count = 0;
	zr = x;
	zi = y;
	mag = x * x + y * y;
	while((mag < 20.0)&&(count < 128))
	{
		wr = zr;
		zr = zr * zr - zi * zi + x;
		zi = 2.0 * wr * zi + y;
		mag = zr * zr + zi * zi;
		++count;
	}
	return(count);
}



// Calculate the value for the Julia image
int CalcJulPt(double xd, double yd, double x0, double y0)
{
	double	zr, zi, wr, mag;
	int		count;

	count = 0;
	zr = xd;
	zi = yd;
	mag = xd * xd + yd * yd;
	while((mag < 20.0)&&(count < 128))
	{
		wr = zr;
		zr = zr * zr - zi * zi + x0;
		zi = 2.0 * wr * zi + y0;
		mag = zr * zr + zi * zi;
		++count;
	}
	return(count);
}



// Calculate the colour to use then plot a point in that colour
void PlotSetPoint(SWHandle hSWnd, int x, int y, int density)
{
	int			mod;
	COLOURREF	colour;

	mod = density % 15;

	switch(mod)
	{
		case 0:		colour = SW_DKGREY; break;
		case 1:		colour = SW_DKRED; break;
		case 2:		colour = SW_DKGREEN; break;
		case 3:		colour = SW_DKBLUE; break;
		case 4:		colour = SW_DKYELLOW; break;
		case 5:		colour = SW_DKMAGENTA; break;
		case 6:		colour = SW_DKCYAN; break;
		case 7:		colour = SW_LTGREY; break;
		case 8:		colour = SW_RED; break;
		case 9:		colour = SW_GREEN; break;
		case 10:	colour = SW_BLUE; break;
		case 11:	colour = SW_YELLOW; break;
		case 12:	colour = SW_MAGENTA; break;
		case 13:	colour = SW_CYAN; break;
		case 14:	colour = SW_WHITE; break;
	}
	SWSetPixelFast(hSWnd, x, y, colour); // Plot the point in the window
}



// Calculate the new centre of an image
double CalcCentre(double centre, double scale, int pt)
{
	return(((double)pt/(double)WIN_SIZE - 0.5)*scale + centre);
}



// Load the library using the functions in SWLoad.c if necessary
// The function calls are missed if only SimpleWn.h is included - in which case
//  SimpleWn.lib must be linked in with the project.
int Setup(void)
{
	#ifdef SWLOAD_H
	int lib = SWLoadLibrary();	// Load the library

	if(lib != SWL_OKAY)
	{
		if(lib == SWL_LIB_NOT_FOUND)
			printf("Could not find SimpleWn.dll");
		else if(lib == SWL_FN_NOT_FOUND)
			printf("Could not find function in DLL - Old DLL probably");
		else
			printf("Unknown error");
		return(0);
	}
	#endif SWLOAD_H

	SWDoRegister();
	return(1);
}
